\name{aws}
\alias{aws}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Local polynomial Adaptive Weights Smoothing for regression with
additive errors  }
\description{ This function implements a local polynomial
              Adaptive Weights Smoothing procedure for
              regression problems with additive errors as
              described in Polzehl \& Spokoiny (2003).
              This function superseedes functions awsuni, awsbi and awstri.
}
\usage{
aws(y, x = NULL, p = 0, sigma2 = NULL, qlambda = NULL, eta = 0.5, 
    tau = NULL, lkern = "Triangle", hinit = NULL, hincr = NULL, 
    hmax = 10, NN = FALSE, u = NULL, graph = FALSE, demo = FALSE, 
    symmetric = NULL, wghts= NULL)
}
\arguments{
  \item{y}{ \code{y} contains the observed values (regression function plus errors).
           In case of \code{x=NULL} (second parameter) \code{y} is assumed to be
           observed on a one, two or three-dimensional grid. The dimension of
           \code{y} determines if one, two or three-dimensional AWS is used.}
  \item{x}{ \code{x} is either \code{NULL}, in this case \code{y} is assumed
           to be observed on a grid, or is a matrix, with rows corresponding to variables,
           containing the design points where \code{y} is observed. }
  \item{p}{ \code{p} is the degree of the polynomial model to use. For univariate
           regression \code{p} can be an nonnegative integer less or equal than 5, for
           bivariate regression on a grid \code{p} can be either \code{0}, \code{1} or
           \code{2}, for three dimensional AWS only \code{p=0} is implemented.  }
  \item{sigma2}{ \code{sigma2} can be used to provide an estimate for the error
           variance. If \code{is.null(sigma2)} a variance estimate is generated from
           the data.}
  \item{qlambda}{ \code{qlambda} determines the scale parameter \code{qlambda}
           for the stochastic penalty. The scaling parameter in the stochastic
           penalty \code{lambda} is choosen as the \code{qlambda}-quantile
           of a Chi-square-distribution with number of parameters in the polynomial
           model as degrees of freedom. If \code{is.null(qlambda)} a standard value
           depending on \code{p} is choosen.}
  \item{eta}{ \code{eta} is a memory parameter used to stabilize the procedure.
           \code{eta}  has to be between \code{0} and \code{1}, with
           \code{eta=.5} being the default. }
  \item{tau}{ \code{tau} is used in case of a the scale parameter polynomial
           degree \code{p!=0} only. It is the scale parameter in the extention
           penalty used to prevent from leverage problems. The default value
           for \code{tau} depends on \code{p} and on the number of parameters
           in the local model. }
  \item{lkern}{ \code{lkern} determines the location kernel to be used. Options
           are \code{"Uniform"}, \code{"Triangle"}, \code{"Quadratic"},
           \code{"Cubic"} and \code{"Exponential"}. Default is \code{"Triangle"}.
           The Kernel operates on the squared distance, so \code{"Triangle"}
           corresponds to the use of an Epanechnikov kernel in kernel smoothing.
           \code{"Exponential"} requires larger values of \code{hmax} and
           therefore more iterations to reach comparable results. }
  \item{hinit}{ \code{hinit} Initial bandwidth for the location penalty.
           Appropriate value is choosen in case of \code{hinit=NULL} }
  \item{hincr}{ \code{hincr} \code{hincr^(1/d)}, with \code{d} the
           dimensionality of the design, is used as a factor to increase the 
           bandwidth between iterations. Defauts to \code{hincr=1.25} }
  \item{hmax}{ \code{hmax} Maximal bandwidth to be used. Determines the
           number of iterations and is used as the stopping rule. }
   \item{NN}{ If \code{NN=TRUE} use nearest neighbor-rules instead of
           distances in the location term. }
  \item{u}{ \code{u} used to supply values of the true regression function 
           for test purposes to calculate  Mean Squared Error (MSE) and
           Mean Absolute Error (MAE) }
  \item{graph}{ \code{graph} if \code{TRUE} results are displayed after each
           iteration step. }
  \item{demo}{ \code{demo} if \code{TRUE} after each iteration step results
           are displayed and the process waits for user interaction. }
  \item{symmetric}{ If \code{symmetric==TRUE} the stochastic penalty is
           symmetrized, i.e. \code{(sij + sji)/lambda} is used instead of
           \code{sij/lambda}. See references for details.
           \code{symmetric==FALSE} is forced if \code{p!=0}}
   \item{wghts}{ Specifies wghts for distance evaluation on a bi- or trivariate grid.
                Allows for anisotropic local neighborhoods. If \code{wghts=NULL}
                isotropic neighborhoods are used. }
}
\details{This function implements an adaptive weights smoothing (AWS) procedure for
   local polynomial models with additive errors. The approach generalizes the original
   AWS procedure from Polzehl and Spokoiny (2000).

   Adaptive weights smoothing is an iterative data adaptive smoothing technique that
   is designed  for smoothing in regression problems with discontinuous regression
   function. The basic assumption is that the regression function can be approximated
   by a simple local,  e.g. local constant or local polynomial, model.
   The estimate of the regression function, i.e. the conditional expectation of \code{y}
   given \code{x} is computed as a weighted maximum likelihood estimate, with weights choosen
   in a completely data adaptive way. The procedure is edge preserving. If the assumed local
   model is globally valid, almost all weights used will be 1, i.e. the resulting estimate
   almost is the global estimate.


   Currently implemented are the following models (specified by parameter \code{p} and attributes
   of \code{x} and \code{y}) are implemented:

   \describe{
   \item{p=0, x=NULL}{local constant univariate, bivariate or three-variate AWS on a
             grid specified by the dimensionality of \code{y}. Measurement unit for \code{hinit}
             and \code{hmax} is 1 (number grid points within distance \code{h})}
   \item{p=0, !is.null(x)}{local constant AWS for arbitrary design given in \code{x}. If
             \code{x} is a matrix, then the rows correspond to components (variables).
             If \code{NN==TRUE} distances in the location penalty are replaced by nearest-
             neighbor-rules. }
   \item{p=1 or 2, x=NULL}{local linear univariate and bivariate AWS on a grid
             grid specified by the dimensionality of \code{y}. Measurement unit for \code{hinit}
             and \code{hmax} is 1 (number grid points within distance \code{h})}
   \item{p=1, !is.null(x)}{local linear AWS for arbitrary design given in \code{x}. If
             \code{x} is a matrix, then the rows correspond to components (variables).
             If \code{NN==TRUE} distances in the location penalty are replaced by nearest-
             neighbor-rules. }
   \item{\code{p>2}, univariate}{univariate local polynomial AWS for arbitrary univariate design
              given in \code{x}. \code{y} has to be a vector. If
             \code{x} is provided it has to be a vector of the same length as \code{y}.
             If \code{is.null(x)} an equidistant design (grid) with grid-step 1 is assumed.}
         }

   The essential parameter in the procedure is \code{qlambda}. This parameter has an
   interpretation as a significance level of a test for equivalence of two local
   parameter estimates. Optimal values mainly depend on the choosen \code{p}
   and the value of \code{symmetric} (determines the use of an asymmetric or a symmetrized
   test). The optimal values only slightly depend on the model parameters, i.e. the
   default parameters should work in most situations. Larger values of \code{qlambda}
   may lead to oversmoothing, small values of \code{qlambda} lead to a random segmentation
   of homogeneous regions. A good value of \code{qlambda} can be obtained by the propagation
   condition, requiring that in case of global validity of the local model the 
   estimate for large \code{hmax} should be equal to the global estimate.

   The numerical complexity of the procedure is mainly determined by \code{hmax}. The number
   of iterations is \code{d*log(hmax/hinit)/log(hincr)} with \code{d} being the dimension
   of \code{y}. Comlexity in each iteration step is \code{Const*hakt*n} with \code{hakt}
   being the actual bandwith in the iteration step and \code{n} the number of design points.
   \code{hmax} determines the maximal possible variance reduction.
}
\value{
  \item{theta }{Parameter estimates, first dimension corresponds to parameter
components}
  \item{y}{values provided in \code{y}}
  \item{x}{values provided in \code{x}}
  \item{call}{actual function call}
}
\references{ \item{ }{ Polzehl, J. and Spokoiny, V. (2003).
   \emph{Varying coefficient regression modeling
     by adaptive weights smoothing}, WIAS-Preprint 818. }
\item{ }{ Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing
     with applications to image restoration}, J.R.Statist.Soc. B, 62,
     Part 2, pp.335-354
 }
 }
\author{ Joerg Polzehl, \email{polzehl@wias-berlin.de},
\url{http://www.wias-berlin.de/project-areas/stat/projects/adaptive-image-processing.html}
 }
\note{ This function supersedes functions \code{\link{awsuni}}, \code{\link{awsbi}}, and \code{\link{awstri}} }


\seealso{ SEE ALSO \code{\link{laws}}, \code{\link{awsuni}}, \code{\link{awsbi}}, 
\code{\link{awstri}} }

\examples{
#######################################################
#                                                     
#                  univariate examples                
#                                                     
####################################################### 
#  
#    Blocks data (Example 6 from Fan & Gijbels (1996)
#
     mofx6 <- function(x){
     xj <- c(10,13,15,23,25,40,44,65,76,78,81)/100
     hj <- c(40,-50,30,-40,50,-42,21,43,-31,21,-42)*.37
     Kern <- function(x) (1-sign(x))/2
     apply(Kern(outer(xj,x,"-"))*hj,2,sum)
     }
#
#    sigma==1   step by step with graphics
#
     fx6 <- mofx6(seq(0,1,1/2047))
     y <- rnorm(fx6,fx6,1)
     tmp <- aws(y,p=0,hmax=100,graph=TRUE)
     par(mfrow=c(1,1),mar=c(3,3,2.5,.5),mgp=c(2,1,0))
     plot(seq(0,1,1/2047),y)
     lines(seq(0,1,1/2047),tmp$theta,col=3,lwd=2)
     lines(seq(0,1,1/2047),fx6,col=2)
#
#    sigma==3   without graphics (much faster)
#
     y <- rnorm(fx6,fx6,3)
     tmp <- aws(y,hmax=500)
     par(mfrow=c(1,1),mar=c(3,3,2.5,.5),mgp=c(2,1,0))
     plot(seq(0,1,1/2047),y)
     lines(seq(0,1,1/2047),tmp$theta,col=3,lwd=2)
     lines(seq(0,1,1/2047),fx6,col=2)
     rm(mofx6,fx6,y,tmp)
#  
#    second example from Polzehl and Spokoiny (2002)
#
     f2 <- function(x) sin(2*pi*1.2/(x+.2))
     n <- 1000
     x <- seq(0,1,length=n)
     fx2 <- f2(x)
     set.seed(1)
     sigma <- .25
     y <- rnorm(x,fx2,sigma)
#   increase hmax to 2 for good results
     ex1p0 <- aws(y,x,p=0,hmax=.1)$theta[1,]
     ex1p1 <- aws(y,x,p=1,hmax=.1)$theta[1,]
     ex1p2 <- aws(y,x,p=2,hmax=.1)$theta[1,]
     ex1p3 <- aws(y,x,p=3,hmax=.1)$theta[1,]
     par(mfrow=c(2,2),mar=c(2.5,2.5,2.5,.5),mgp=c(1.5,.5,0))
     plot(x,y)
     lines(x,fx2,col=2)
     lines(x,ex1p0,col=3,lwd=2)
     title("local constant AWS")
     plot(x,y)
     lines(x,fx2,col=2)
     lines(x,ex1p1,col=3,lwd=2)
     title("local linear AWS")
     plot(x,y)
     lines(x,fx2,col=2)
     lines(x,ex1p2,col=3,lwd=2)
     title("local quadratic AWS")
     plot(x,y)
     lines(x,fx2,col=2)
     title("local cubic AWS")
     lines(x,ex1p3,col=3,lwd=2)
     rm(f2,fx2,x,sigma,y,ex1p0,ex1p1,ex1p2,ex1p3)
#######################################################
#                                                     
#                  bivariate examples                 
#                                                     
####################################################### 
#  
#  Local constant image from Polzehl and Spokoiny (2000)
#
     xy <- rbind(rep(0:255,256),rep(0:255,rep(256,256)))
     indw <- c(1:12,29:48,73:100,133:168,209:256)
     w0 <- matrix(rep(FALSE,256*256),ncol=256)
     w0[indw,] <- TRUE
     w0[,indw] <- !w0[,indw]
     w0 <- w0-.5
     
     w0[((xy[1,]-129)^2+(xy[2,]-129)^2)<=10000&((xy[1,]-129)^2+(xy[2,]-129)^2)>=4900] <- 0
     w0[abs(xy[1,]-xy[2,])<=20&((xy[1,]-129)^2+(xy[2,]-129)^2)<4900] <- 0
     w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625] <- 0
     
     w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625&xy[2,]>27&xy[2,]<31] <- -.5
     
     w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625&xy[1,]>223&xy[1,]<227] <- .5
     w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625] <- 0
     
     w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625&xy[1,]>27&xy[1,]<31] <- -.5
     
     w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625&xy[2,]>223&xy[2,]<227] <- .5
     w0[((xy[2,]-225)^2+(xy[1,]-225)^2)+1*(xy[2,]-225)*(xy[1,]-225)<=400] <- 0
     w0[((xy[2,]-30)^2+(xy[1,]-30)^2)<=256] <- 0
     rm(xy,indw)
     set.seed(1)
     sigma <- .5
     y <- w0+rnorm(w0,0,sigma)
#    run some steps interactively  (increase hmax)
     tmp <- aws(y,graph=TRUE,hmax=2,demo=TRUE)
#    now without graphics and larger hmax
#   increase hmax for better results
     tmp <- aws(y,hmax=5)
     par(mfrow=c(1,3))
     image(y,col=gray((0:255)/255))
     image(tmp$theta,zlim=range(y),col=gray((0:255)/255))
     image(w0,zlim=range(y),col=gray((0:255)/255))
     rm(y,w0,tmp,sigma)
#
#   piecewise smooth image from Polzehl and Spokoiny (2003)
#
     x <- (0:99)/99
     fbi <- function(x,y) (x^2+y^3)*sign(x^2-1*x*y-.75*y^3)
     z0 <- outer(2*(x-.5),2*(x-.25),FUN=fbi)
     z <- z0+rnorm(z0,0,.5)
     par(mfrow=c(1,3),mar=c(3,3,3,.5),mgp=c(2,1,0))
     set.seed(1)
     persp(x,x,z0,phi=15,theta=150,d=5,r=2,expand=1.5,col="white")
     title("True function")
     persp(x,x,z,phi=15,theta=150,d=5,r=2,expand=1.5,col="white")
     title("Observed values")
     image(x,x,z,col=gray((0:255)/255))
     title("Observed values")
#
#   local constant
#
     ex1p0 <- aws(z,hmax=3,symmetric=FALSE)$theta # use hmax=5 or larger
#
#   local linear
#
     ex1p1 <- aws(z,p=1,hmax=3)$theta[1,,]  # use hmax=12 or larger
#
#   local quadratic
#
     ex1p2 <- aws(z,p=2,hmax=3)$theta[1,,]  # use hmax=20 or larger
#
#   display results
#
     par(mfrow=c(2,2),mar=c(0.25,.5,.5,.5),mgp=c(.5,.5,0))
     persp(x,x,z,phi=15,theta=150,d=5,r=2,expand=1,xlab="",ylab="",zlab="",box=FALSE)
     title("original data",line=-3)
     persp(x,x,ex1p0,phi=15,theta=150,d=5,r=2,expand=1,xlab="",ylab="",zlab="",box=FALSE)
     title("local constant AWS",line=-3)
     persp(x,x,ex1p1,phi=15,theta=150,d=5,r=2,expand=1,xlab="",ylab="",zlab="",box=FALSE)
     title("local linear AWS (small hmax)",line=-3)
     persp(x,x,ex1p2,phi=15,theta=150,d=5,r=2,expand=1,xlab="",ylab="",zlab="",box=FALSE)
     title("local quadratic AWS (small hmax)",line=-3)
     rm(x,fbi,z0,ex1p0,ex1p1,ex1p2)

#######################################################
#
#                  three-variate examples
#
#######################################################
#
#  Local constant image from Polzehl and Spokoiny (2000)
#
     xy <- rbind(rep(0:30,31),rep(0:30,rep(31,31)))
     w3 <- array(0,c(31,31,31))
     w3[4:28,4:28,4:28] <- 1
     dim(w3) <- c(961,31)
     w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=144,16] <- 0
     for(i in 1:12) {
        r2 <- 144-i*i
        w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=r2,16+c(-i,i)] <- 0
     }
     dim(w3) <- c(31,31,31)
     w3[10:22,10:22,10:22] <- 1
     dim(w3) <- c(961,31)
     w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=36,16] <- 0
     for(i in 1:6) {
        r2 <- 36-i*i
        w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=r2,16+c(-i,i)] <- 0
     }
     dim(w3) <- c(31,31,31)
     rm(xy)
     sigma <- .5
     set.seed(1)
     y <- w3+rnorm(w3,0,sigma)
#   increase hmax for reasonable results (hmax >=5)
     tmp <- aws(y,hmax=2)
     par(mfrow=c(1,3))
     for(i in 14:18){
     image(y[,,i],col=gray((0:255)/255))
     image(tmp$theta[,,i],zlim=range(y),col=gray((0:255)/255))
     image(w3[,,i],zlim=range(y),col=gray((0:255)/255))
#     readline()
     }
     rm(w3,y,tmp,sigma)
}
\keyword{ smooth }% at least one, from doc/KEYWORDS
\keyword{ nonparametric }% __ONLY ONE__ keyword per line
\keyword{ regression }% __ONLY ONE__ keyword per line
\keyword{ models }% __ONLY ONE__ keyword per line

\eof
\name{awsbi}
\alias{awsbi}
\title{Two-dimensional Adaptive Weights Smoothing}
\description{Performes two dimensional Adaptive Weigths Smoothing (depreciated version, use aws instead)}
\usage{awsbi(y, lambda=3, gamma=1.3, eta =4, s2hat = NULL, kstar = length(radii),
             rmax=max(radii), radii = c((1:8)/2,4.4,5.,(6:10),(6:10)*2), 
             graph = FALSE, u0 = NULL, control="dyadic", demomode=FALSE, 
             colors=gray((0:255)/255))}
\arguments{
\item{y}{matrix of observed values}
\item{lambda}{main smoothing parameter (should be approximately 3)}
\item{gamma}{allow for increase of variances during iteration by factor gamma 
(!! gamma >=1) }
\item{eta}{main control parameter (should be approximately 4)   }
\item{s2hat}{initial variance estimate (if available,
           can be either a number (homogeneous case), a matrix of same
           dimension as y (inhomogeneous variance) or NULL (a homogeneous 
           variance estimate will be generated in this case)}
\item{kstar}{maximal number of iterations to perform, actual number may be 
           smaller depending on parameters radii and rmax}
\item{radii}{radii of circular neighbourhoods used}
\item{rmax}{maximal radius of neighborhood to be used, may change kstar}
\item{graph}{logical, if TRUE progress (for each iteration) is illustrated 
           grahically, if FALSE the program runs until the final estimate is 
           obtained (much faster !!!)}
\item{u0}{allows for submission of "true" values for illustration and test 
           purposes; only if graph=TRUE,  MSE and MAE are reported for each 
           iteration step}
\item{control}{the control step is performed in either a dyadic sceme
           ("dyadic") or using all previous estimates (otherwise)}
\item{demomode}{if TRUE the function will wait for user input after each 
           iteration; only if graph=TRUE }
\item{colors}{color sceme to be used for images}
}
\value{A list with components
\item{yhat}{estimates of the regression function (matrix corresponding to the 
y's)}
\item{shat}{estimated standard deviations of yhat (conditional on the chosen 
weights)}
\item{nu}{maximal number of design points in neighborhood used}
\item{args}{main arguments supplied to awsbi}
}
\references{Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing 
with applications
to image restoration}, J.R.Statist.Soc. B, 62, Part 2, pp. 335-354}
\note{The function assumes that the data are given on a 2D-grid 
corresponding to the dimensionality of y. 
This function is superseded by function aws and will be 
removed in the next mayor version of the package.}
\author{Joerg Polzehl \email{polzehl@wias-berlin.de}}
\seealso{\code{\link{aws}}, \code{\link{awsuni}},\code{\link{awstri}}}

\examples{
xy<-rbind(rep(0:255,256),rep(0:255,rep(256,256)))
indw<-c(1:12,29:48,73:100,133:168,209:256)
w0<-matrix(rep(0,256*256),ncol=256)
w0[indw,]<-1
w0[,indw]<-!w0[,indw]
w0<-w0-.5
w0[((xy[1,]-129)^2+(xy[2,]-129)^2)<=10000&((xy[1,]-129)^2+(xy[2,]-129)^2)>=4900]<- 0
w0[abs(xy[1,]-xy[2,])<=20&((xy[1,]-129)^2+(xy[2,]-129)^2)<4900]<- 0
w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625]<- 0
w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625&xy[2,]>27&xy[2,]<31]<- -.5
w0[((xy[1,]-225)^2+2*(xy[2,]-30)^2)-(xy[1,]-225)*(xy[2,]-30)<=625&xy[1,]>223&xy[1,]<227]<- .5
w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625]<- 0
w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625&xy[1,]>27&xy[1,]<31]<- -.5
w0[((xy[2,]-225)^2+2*(xy[1,]-30)^2)+(xy[2,]-225)*(xy[1,]-30)<=625&xy[2,]>223&xy[2,]<227]<- .5
w0[((xy[2,]-225)^2+(xy[1,]-225)^2)+1*(xy[2,]-225)*(xy[1,]-225)<=400]<- 0
w0[((xy[2,]-30)^2+(xy[1,]-30)^2)<=256]<-0
sigma<-.25
y<-w0+rnorm(w0,0,sigma)
#  increase rmax for better results
yhat<-awsbi(y,rmax=3)
par(mfrow=c(1,3))
image(y,col=gray((0:255)/255))
title("Noisy image")
image(yhat$yhat,zlim=range(y),col=gray((0:255)/255))
title("AWS reconstruction")
image(w0,zlim=range(y),col=gray((0:255)/255))
title("Original image")
rm(y,w0,yhat,xy)
}
\keyword{regression}
\keyword{nonparametric}
\keyword{smooth}

\eof
\name{awsdens}
\alias{awsdens}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ local constant density estimation }
\description{
  This function uses adaptive weights smoothing for density estimation. The estimate
  is constructed using poisson regression and the structural assumption of a local
  constant density. Implementation is for 1D, 2D and 3D problems, although it 
  applications may be interesting only in 2D and 3D.
}
\usage{
awsdens(y, ngrid = NULL, nempty = NULL, qlambda = NULL, eta = 0.5, 
        lkern = "Triangle", fu = NULL, hinit = 1, hincr = 1.2, 
    hmax = NULL, graph = FALSE, demo = FALSE, symmetric = TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ \code{y} contains the observed random variables or vectors. 
       In case of random vectors \code{y} is supposed to be an
       observation matrix with columns containing the observed vectors}
  \item{ngrid}{ \code{ngrid} determines the size of the grid used for binning. 
       If \code{ngrid} is a vector its components are supposed to determine 
       the number of intervals for the corresponding component of \code{y};
       if it is a scalar \code{rep(ngrid,dim(y)[1])} is used.
       If \code{is.null(ngrid)} a default of \code{2*n^{1/d}}, with 
       \code{dim(y)=c(d,n)}, is chosen, i.e. a grid with \code{2^d*n} bins.}
  \item{nempty}{ determines the width of a boundary region of the grid containing only empty bins.
        \code{nempty} defaults to \code{0.1*ngrid}. The grid generated is equispaced with 
        \code{prod(ngrid)} bins and grid-coordinates \code{(nempty+1)[i]} and \code{(ngrid+1-nempty)[i]}
        determined by \code{min(y[i,])} and \code{max(y[i,])}.}
  \item{qlambda}{ \code{qlambda} determines the scale parameter \code{qlambda} 
           for the stochastic penalty. The scaling parameter in the stochastic 
           penalty \code{lambda} is choosen as the \code{qlambda}-quantile
           of a Chi-square-distribution with number of parameters in the polynomial 
           model as degrees of freedom. If \code{qlambda=NULL} a standard value 
           depending on \code{model} and \code{symmetric} is choosen. }
  \item{eta}{ \code{eta} is a memory parameter used to stabilize the procedure. 
           \code{eta}  has to be between \code{0} and \code{1}, with 
           \code{eta=.5} being the default. }
  \item{lkern}{ \code{lkern} determines the location kernel to be used. Options 
           are \code{"Uniform"}, \code{"Triangle"}, \code{"Quadratic"}, 
           \code{"Cubic"} and \code{"Exponential"}. Default is \code{"Triangle"}. 
           The Kernel operates on the squared distance, so \code{"Triangle"}
           corresponds to the use of an Epanechnikov kernel in kernel smoothing. 
           \code{"Exponential"} requires larger values of \code{hmax} and 
           therefore more iterations to reach comparable results. }
  \item{fu}{ \code{fu} can be used to specify a function to calculate the 
           values of the true density on the grid. This may be used
           for test purposes to calculate  Mean Squared Error (MSE) and 
           Mean Absolute Error (MAE) }
  \item{hinit}{ \code{hinit} Initial bandwidth for the location penalty. 
           Appropriate value is choosen in case of \code{hinit==NULL}  }
  \item{hincr}{ \code{hincr} \code{hincr^(1/d)}, with \code{d} the 
           dimensionality of the design, is used as a factor to increase the 
           bandwidth between iterations. Defauts to \code{hincr=1.2} }  
  \item{hmax}{ \code{hmax} Maximal bandwidth to be used. Determines the 
           number of iterations and is used as the stopping rule. }
  \item{graph}{ \code{graph} if \code{TRUE} results are displayed after each 
           iteration step. }
  \item{demo}{ \code{demo} if \code{TRUE} after each iteration step results 
           are displayed and the process waits for user interaction. }
  \item{symmetric}{ If \code{symmetric==TRUE} the stochastic penalty is
           symmetrized, i.e. \code{(sij + sji)/lambda} is used instead of 
           \code{sij/lambda}. See references for details.}
}
\details{
  The density estimate ist constructed using an approach proposed by 
  Lindsay (1974a,b). 1, 2 or 3-dimensional binning is used to produce
  counts, i.e. numbers of observations, per bin, with bins defined by
  a regular grid. THe bin-counts are then considered as poisson random
  variables with intensity depending on the location within the grid.
  Adaptive Weights Smoothing for poisson regression models, i.e. function 
  \code{\link{laws}} with parameter \code{model="Poisson"}, is used 
  to construct a location dependent intensity estimate which is then 
  transformed into a density estimate, see Section 6 in Polzehl and Spokoiny
  (2002b) for details.
}
\value{ The returned object is a list with components
  \item{bin }{Bin counts}
  \item{dens }{density values for the bin. Values correspond to the grid-centers
              determined by component \code{xgrid}}
  \item{xgrid }{list with components containing the coordinates of bin-centers.
            The \code{dim(y)[1]} components of \code{xgrid} correspond to the
            components of the grid-coordinates. }
  \item{call }{actual function call}
}
\references{ 
\item{ }{    Polzehl, J. and Spokoiny, V. (2002). 
\emph{Local likelihood modelling by adaptive weights smoothing}, 
WIAS-Preprint 787 } 
   \item{ }{  Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing
     with applications to image restoration}, J.R.Statist.Soc. B, 62,
     Part 2, pp.335-354}
\item{ }{
Lindsay, J. (1974a). \emph{Comparison of probabbility distributions}, J. Royal Statist. Soc. Ser. B
36, 38-47.}
\item{ }{
Lindsay, J. (1974b). \emph{Construction and comparison of statistical models}, J. Royal Statist. Soc. Ser. B
36, 418-425.
}
}
\author{ Joerg Polzehl, \email{polzehl@wias-berlin.de }}


\seealso{ SEE ALSO \code{\link{aws}}, \code{\link{laws}} }

\examples{
###
###   univariate density estimation
###
f0 <- function(x) 1.5*(x>=0)-(x>=.25)+(x>=.75)-1.5*(x>1)
set.seed(1)
m <- 1000
x1 <- runif(m)
ind <- (runif(m)<f0(x1)/1.5)
n <- 200
y <- x1[ind][1:n]
tmp0 <- awsdens(y,440,20,hmax=2000)
plot(tmp0$xgrid[[1]],tmp0$dens,type="l",lwd=2,
     ylim =range(0,1.5,tmp0$dens),xlab="x",ylab="Density")
lines(tmp0$xgrid[[1]],f0(tmp0$xgrid[[1]]),lty=3,col=2,lwd=2)
title("Estimated and true density (n=200)")
###
###   bivariate density estimation
###
f1 <- function(x,y) 7.5*pmax(x*(1-x^2-y^2),0)*(x>0)*(y>0)
set.seed(1)
m <- 10000
x1 <- runif(m)
x2 <- runif(m)
fx12 <- f1(x1,x2)
ind <- (runif(m)<f1(x1,x2)/.385/7.5)
n <- 2500
y <- rbind(x1[ind],x2[ind])[,1:n]
tmp <- awsdens(y,120,10,hmax=5)
image(tmp$xgrid[[1]],tmp$xgrid[[2]],tmp$dens,xlab="x1",ylab="x2",col=gray(.5+(255:0)/511),lwd=2)
#   thats the estimated density on the grid
lines(seq(0,1,.01),sqrt(1-seq(0,1,.01)^2),col=2,lty=2,lwd=2)
lines(c(0,1),c(0,0),col=2,lty=2,lwd=2)
lines(c(0,0),c(0,1),col=2,lty=2,lwd=2)
#    thats the boundary of the support
title("Estimated density (n=2500)")
#    now add contour lines of the estimated density
contour(tmp$xgrid[[1]],tmp$xgrid[[2]],tmp$dens,nlevels=20,add=TRUE,col=1,lty=1,labcex=.1)
rm(f0,m,x1,ind,n,y,tmp0,f1,x2,fx12,tmp)
}
\keyword{ smooth }% at least one, from doc/KEYWORDS
\keyword{ nonparametric }% __ONLY ONE__ keyword per line
\keyword{ models }% __ONLY ONE__ keyword per line

\eof
\name{awsh}
\alias{awsh}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Univariate local polynomial Adaptive Weights Smoothing for regression with
heteroscedastic additive errors }
\description{
  This function implements a univariate local polynomial
              Adaptive Weights Smoothing procedure for
              regression problems with heteroskedastic additive errors as
              described in Polzehl \& Spokoiny (2003).
}
\usage{
awsh(y, x = NULL, p = 0, sigma2 = NULL, qlambda = NULL, eta = 0.5, tau = NULL,
lkern = "Triangle", hinit = NULL, hincr = NULL, hmax = 100, hmaxs= 2*hmax, u = NULL,
graph = FALSE, demo = FALSE, symmetric = NULL, conf = FALSE, qconf = 0.95, alpha = 2)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{y}{ \code{y} contains the observed values (regression function plus errors).
           In case of \code{x=NULL} (second parameter) \code{y} is assumed to be
           observed on a one-dimensional grid. }
  \item{x}{ \code{x} is either \code{NULL}, in this case \code{y} is assumed
           to be observed on a grid, or is a vector determining the design. }
  \item{p}{ \code{p} is the degree of the polynomial model to use. For univariate
           regression \code{p} can be an nonnegative integer less or equal than 5.  }
  \item{sigma2}{ \code{sigma2} can be used to provide an estimate for the error
           variance. If \code{is.null(sigma2)} a homoskedastic model is assumed and
           a variance estimate is generated from the data. If \code{length(sigma2)==length(y)}
	   this parameter provides variances at the design points.}
  \item{qlambda}{ \code{qlambda} determines the scale parameter \code{qlambda}
           for the stochastic penalty. The scaling parameter in the stochastic
           penalty \code{lambda} is choosen as the \code{qlambda}-quantile
           of a Chi-square-distribution with number of parameters in the polynomial
           model as degrees of freedom. If \code{is.null(qlambda)} a standard value
           depending on \code{p} is choosen.}
  \item{eta}{ \code{eta} is a memory parameter used to stabilize the procedure.
           \code{eta}  has to be between \code{0} and \code{1}, with
           \code{eta=.5} being the default. }
  \item{tau}{ \code{tau} is used in case of a the scale parameter polynomial
           degree \code{p!=0} only. It is the scale parameter in the extention
           penalty used to prevent from leverage problems. The default value
           for \code{tau} depends on \code{p} and on the number of parameters
           in the local model. }
  \item{lkern}{ \code{lkern} determines the location kernel to be used. Options
           are \code{"Uniform"}, \code{"Triangle"}, \code{"Quadratic"},
           \code{"Cubic"} and \code{"Exponential"}. Default is \code{"Triangle"}.
           The Kernel operates on the squared distance, so \code{"Triangle"}
           corresponds to the use of an Epanechnikov kernel in kernel smoothing.
           \code{"Exponential"} requires larger values of \code{hmax} and
           therefore more iterations to reach comparable results. }
  \item{hinit}{ \code{hinit} Initial bandwidth for the location penalty.
           Appropriate value is choosen in case of \code{hinit=NULL} }
  \item{hincr}{ \code{hincr} \code{hincr} is used as a factor to increase the
           bandwidth between iterations. Defauts to \code{hincr=1.25} }
  \item{hmax}{ \code{hmax} Maximal bandwidth to be used. Determines the
           number of iterations and is used as the stopping rule. }
  \item{hmaxs}{ \code{hmaxs} Maximal bandwidth to be used when estimating the
           heterogenous variance from consequtive differences of \code{y} by function
           \code{laws}. Determines the number of iterations of \code{laws}. }
  \item{u}{ \code{u} used to supply values of the true regression function
           for test purposes to calculate  Mean Squared Error (MSE) and
           Mean Absolute Error (MAE) }
  \item{graph}{ \code{graph} if \code{TRUE} results are displayed after each
           iteration step. }
  \item{demo}{ \code{demo} if \code{TRUE} after each iteration step results
           are displayed and the process waits for user interaction. }
  \item{symmetric}{ If \code{symmetric==TRUE} the stochastic penalty is
           symmetrized, i.e. \code{(sij + sji)/lambda} is used instead of
           \code{sij/lambda}. See references for details.
           \code{symmetric==FALSE} is forced if \code{p!=0}}
  \item{conf}{ \code{conf} if \code{TRUE} conditional (on weights) confidence intervals are provided
           at each design point.}
  \item{qconf}{ \code{qconf} determines the level of the conditional (on weights) confidence intervals }
  \item{alpha}{ \code{alpha} Parameter used for a penalized MSE estimate for \code{p=0}. This is
                experimental to try to select \code{hmax}. }
}
\details{
   This function implements an local polynomial adaptive weights smoothing (AWS) procedure for a
   univariate heteroskedastic regression model. The approach generalizes the original
   AWS procedure from Polzehl and Spokoiny (2000).

   Adaptive weights smoothing is an iterative data adaptive smoothing technique that
   is designed  for smoothing in regression problems with discontinuous regression
   function. The basic assumption is that the regression function can be approximated
   by a simple local,  e.g. local constant or local polynomial, model.
   The estimate of the regression function, i.e. the conditional expectation of \code{y}
   given \code{x} is computed as a weighted maximum likelihood estimate, with weights choosen
   in a completely data adaptive way. The procedure is edge preserving. If the assumed local
   model is globally valid, almost all weights used will be 1, i.e. the resulting estimate
   almost is the global estimate.


   Currently implemented are the following models (specified by parameter \code{p} and attributes
   of \code{x} and \code{y}) are implemented:

   \describe{
   \item{p=0, x=NULL}{local constant univariate AWS on a grid. Measurement unit for \code{hinit}
             and \code{hmax} is 1 (number grid points within distance \code{h})}
   \item{p=0, !is.null(x)}{local constant AWS for arbitrary design given in \code{x}. }
   \item{\code{p>2}}{univariate local polynomial AWS for arbitrary univariate design
              given in \code{x}. \code{y} has to be a vector. If
             \code{x} is provided it has to be a vector of the same length as \code{y}.
             If \code{is.null(x)} an equidistant design (grid) with grid-step 1 is assumed.}
         }

   The essential parameter in the procedure is \code{qlambda}. This parameter has an
   interpretation as a significance level of a test for equivalence of two local
   parameter estimates. Optimal values mainly depend on the choosen \code{p}
   and the value of \code{symmetric} (determines the use of an asymmetric or a symmetrized
   test). The optimal values only slightly depend on the model parameters, i.e. the
   default parameters should work in most situations. Larger values of \code{qlambda}
   may lead to oversmoothing, small values of \code{qlambda} lead to a random segmentation
   of homogeneous regions. A good value of \code{qlambda} can be obtained by the propagation
   condition, requiring that in case of global validity of the local model the 
   estimate for large \code{hmax} should be equal to the global estimate.

   The numerical complexity of the procedure is mainly determined by \code{hmax}. The number
   of iterations is \code{d*log(hmax/hinit)/log(hincr)} with \code{d} being the dimension
   of \code{y}. Comlexity in each iteration step is \code{Const*hakt*n} with \code{hakt}
   being the actual bandwith in the iteration step and \code{n} the number of design points.
   \code{hmax} determines the maximal possible variance reduction.
}
\value{
  \item{theta }{Parameter estimates, first dimension corresponds to parameter components}
  \item{theta }{Standard deviations (conditional on weights) of parameter estimates,
  first dimension corresponds to parameter components}
  \item{y}{values provided in \code{y}}
  \item{x}{values provided in \code{x}}
  \item{call}{actual function call}
}
\references{ \item{ }{ Polzehl, J. and Spokoiny, V. (2003).
   \emph{Varying coefficient regression modeling
     by adaptive weights smoothing}, WIAS-Preprint 818. }
\item{ }{ Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing
     with applications to image restoration}, J.R.Statist.Soc. B, 62,
     Part 2, pp.335-354
 }
 }
\author{ Joerg Polzehl, \email{polzehl@wias-berlin.de},
\url{http://www.wias-berlin.de/project-areas/stat/projects/adaptive-image-processing.html}
}
\seealso{  \code{\link{aws}} }
\examples{
##---- Should be DIRECTLY executable !! ----
##-- ==>  Define data, use random,
##--	or do  help(data=index)  for the standard data sets.
}
\keyword{ smooth }% at least one, from doc/KEYWORDS
\keyword{ nonparametric }% __ONLY ONE__ keyword per line
\keyword{ regression }% __ONLY ONE__ keyword per line
\keyword{ models }% __ONLY ONE__ keyword per line

\eof
\name{awstindex}
\alias{awstindex}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Tail index estimation }
\description{
  The function finds a pareto-approximation of the tail of a univariate 
  distribution and estimates the parameter in this pareto-approximation.
  The construction is similar to the Hill-estimator. The number of
  largest observations used in the estimate is chosen adaptively.
}
\usage{
awstindex(y, qlambda = NULL, eta = 0.5, lkern = "Triangle", hinit = 1, 
          hincr = 1.25, hmax = 1000, graph = FALSE, symmetric = FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ \code{y} contains the observed values at location \code{x}. 
           In case of \code{x=NULL} (second parameter) \code{y} is assumed to be 
           observed on a one, two or three-dimensional grid. The dimension of 
           \code{y} determines if one, two or three-dimensional AWS is used.}
  \item{qlambda}{ \code{qlambda} determines the scale parameter \code{qlambda} 
           for the stochastic penalty. The scaling parameter in the stochastic 
           penalty \code{lambda} is choosen as the \code{qlambda}-quantile
           of a Chi-square-distribution with number of parameters in the polynomial 
           model as degrees of freedom. If \code{qlambda=NULL} a standard value 
           depending on \code{model} and \code{symmetric} is choosen. }
  \item{eta}{ \code{eta} is a memory parameter used to stabilize the procedure. 
           \code{eta}  has to be between \code{0} and \code{1}, with 
           \code{eta=.5} being the default. }
  \item{lkern}{ \code{lkern} determines the location kernel to be used. Options 
           are \code{"Uniform"}, \code{"Triangle"}, \code{"Quadratic"}, 
           \code{"Cubic"} and \code{"Exponential"}. Default is \code{"Triangle"}. 
           The Kernel operates on the squared distance, so \code{"Triangle"}
           corresponds to the use of an Epanechnikov kernel in kernel smoothing. 
           \code{"Exponential"} requires larger values of \code{hmax} and 
           therefore more iterations to reach comparable results. }
  \item{hinit}{ \code{hinit} Initial bandwidth for the location penalty. 
           Appropriate value is choosen in case of \code{hinit==NULL}  }
  \item{hincr}{ \code{hincr} \code{hincr^(1/d)}, with \code{d} the 
           dimensionality of the design, is used as a factor to increase the 
           bandwidth between iterations. Defauts to \code{hincr=1.2}}   
  \item{hmax}{ \code{hmax} Maximal bandwidth to be used. Determines the 
           number of iterations and is used as the stopping rule. }
  \item{graph}{ \code{graph} if \code{TRUE} results are displayed after each 
           iteration step. }
  \item{symmetric}{ If \code{symmetric==TRUE} the stochastic penalty is
           symmetrized, i.e. \code{(sij + sji)/lambda} is used instead of 
           \code{sij/lambda}. See references for details.}

}
\details{
  From the data \code{y} an descending order statistics \code{yn <- order(y)[n:1]} is computed 
  and transformed observations \code{x <- (1:(n-1))*yn[-n]/yn[-1]} are defined. The transformed
  observations are assumed to follow an inhomogenious exponential model. Adaptive Weights Smoothing,
  i.e. function \code{\link{laws}} with parameter \code{model="Exponential"}, is used 
  to construct an inhomogenious intensity estimate. The estimated tail index is the estimated
  intensity in the left-most point, corresponding to the largest observation in the sample.
  This estimate is similar to the Hill-estimate computed from the \code{k} largest observations
  with \code{k} approximately equal to the sum of weights used for estimating the tail index
  by AWS. See Section 8 in Polzehl and Spokoiny (2002) for details.
  
  }
\value{ The returned object is a list with components
  \item{tindex }{Estimated tail-index}
  \item{intensity }{Estimates of the intensity in the exponential model}
  \item{y }{Values of \code{y}}
  \item{call }{actual function call}
}
\references{ 
\item{ }{    Polzehl, J. and Spokoiny, V. (2002). 
\emph{Local likelihood modelling by adaptive weights smoothing}, 
WIAS-Preprint 787} 
\item{ }{ 
     Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing
     with applications to image restoration}, J.R.Statist.Soc. B, 62,
     Part 2, pp.335-354
 }
 }
\author{ Joerg Polzehl, \email{polzehl@wias-berlin.de }}


\seealso{ SEE ALSO \code{\link{aws}}, \code{\link{laws}} }

\examples{
###
###   Estimate the tail-index of a cauchy distribution
###   absolute values can be used because of the symmetry of centered cauchy
###
set.seed(1)
n <- 500
x <- rcauchy(n)
tmp <- awstindex(abs(x),hmax=n)
tmp$tindex
###
###   now show the segmentation generated by AWS 
###
plot(tmp$intensity[1:250],type="l")
}
\keyword{ smooth }% at least one, from doc/KEYWORDS
\keyword{ nonparametric }% at least one, from doc/KEYWORDS
\keyword{ models }% __ONLY ONE__ keyword per line

\eof
\name{awstri}
\alias{awstri}
\title{Three-dimensional Adaptive Weights Smoothing}
\description{Performes three-dimensional Adaptive Weigths Smoothing (depreciated version, use aws instead)}
\usage{awstri(y, lambda=3, gamma=1.3, eta =4, s2hat = NULL, kstar = length(radii),
       rmax=max(radii), weight = c(1,1,1), radii = 
       c((1:4)/2,2.3,(5:12)/2,7:9,10.5,12,13.5), control="dyadic"}
\arguments{
\item{y}{array of observed values}
\item{lambda}{main smoothing parameter (should be approximately 3)}
\item{gamma}{allow for increase of variances during iteration by factor gamma 
 (!!\code{gamma >=1}) }
\item{eta}{main control parameter (should be approximately 4)   }
\item{s2hat}{initial variance estimate (if available,
           can be either a number (homogeneous case), a matrix of same 
dimension 
           as y (inhomogeneous variance) or NULL (a homogeneous variance 
estimate
           will be generated in this case)}
\item{kstar}{maximal number of iterations to perform, actual number may be 
smaller depending
             on parameters radii and rmax}
\item{weight}{weights used for distances, determining elliptical neighborhoods}
\item{radii}{radii of circular neighbourhoods used}
\item{rmax}{maximal radius of neighborhood to be used, may change kstar}
\item{control}{the control step is performed in either a dyadic sceme
           ("dyadic") or using all previous estimates (otherwise)}
}

\value{
A list with components
\item{yhat}{estimates of the regression function (matrix corresponding to the 
y's)}
\item{shat}{estimated standard deviations of yhat (conditional on the chosen 
weights)}
\item{args}{Main arguments supplied to awstri}
}
\references{Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing 
with applications
to image restoration}, J.R.Statist.Soc. B, 62, Part 2, pp.335-354}
\note{The function assumes that the data are given on a 3D-grid 
corresponding to the dimensionality of y. 
This function is superseded by function aws and will be 
removed in the next mayor version of the package.}
\author{Joerg Polzehl \email{polzehl@wias-berlin.de}}
\seealso{\code{\link{aws}}, \code{\link{awsbi}},\code{\link{awsuni}}}

\examples{
xy <- rbind(rep(0:30,31),rep(0:30,rep(31,31)))
w3 <- array(0,c(31,31,31))
w3[4:28,4:28,4:28] <- 1
dim(w3) <- c(961,31)
w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=144,16] <- 0
for(i in 1:12) {
   r2 <- 144-i*i
   w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=r2,16+c(-i,i)] <- 0
}
dim(w3) <- c(31,31,31)
w3[10:22,10:22,10:22] <- 1
dim(w3) <- c(961,31)
w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=36,16] <- 0
for(i in 1:6) {
   r2 <- 36-i*i
   w3[((xy[1,]-15)^2+(xy[2,]-15)^2)<=r2,16+c(-i,i)] <- 0
}
dim(w3) <- c(31,31,31)
sigma <- .4
y <- w3+rnorm(w3,0,sigma)
#  increase rmax for better results
yhat <- awstri(y,rmax=2)
rm(y,yhat,w3,xy)
}
\keyword{regression}
\keyword{nonparametric}
\keyword{smooth}

\eof
\name{awsuni}
\alias{awsuni}
\title{One-dimensional Adaptive Weights Smoothing}
\description{Performes one-dimensional Adaptive Weigths Smoothing (depreciated version, use aws instead)}
\usage{awsuni(y, lambda=3, gamma=1.3, eta =4, s2hat = NULL, kstar = length(radii),
              radii = c(1:8,(5:12)*2,(7:12)*4,(7:12)*8,(7:10)*16,(6:8)*32,
                      (5:8)*64,(5:8)*128,(5:8)*256),
          rmax=max(radii),graph = FALSE,z0 = NULL, eps = 1e-08, 
          control="dyadic", demomode=FALSE)}
\arguments{
\item{y}{observed values (ordered by value of independent variable)}
\item{lambda}{main smoothing parameter (should be approximately 3)}
\item{gamma}{allow for increase of variances during iteration by factor gamma (!! gamma >=1) }
\item{eta}{main control parameter (should be approximately 4)   }
\item{s2hat}{initial variance estimate (if available,
           can be either a number (homogeneous case), a vector of same length 
           as y (inhomogeneous variance) or NULL (a homogeneous variance estimate
           will be generated in this case)}
\item{kstar}{maximal number of iterations to perform, actual number may be smaller depending
             on parameters radii, rmax and eps}
\item{radii}{radii of neighbourhoods used}
\item{rmax}{maximal radius of neighborhood to be used, may change kstar}
\item{graph}{logical, if TRUE progress (for each iteration) is illustrated grahically,
           if FALSE the program runs until the final estimate is obtained (much faster !!!)}
\item{z0}{allows for submission of "true" values for illustration and test purposes; only
               if graph=TRUE,  MSE and MAE are reported for each iteration step}
\item{eps}{stop iteration if $||(yhatnew - yhat)||^2 < eps * sum(s2hat)$}
\item{control}{the control step is performed in either a dyadic sceme
           ("dyadic") or using all previous estimates (otherwise)}
\item{demomode}{if TRUE the function will wait for user input after each iteration; 
                  only if graph=TRUE }
}

\value{
A list with components
\item{yhat}{estimates of the regression function (corresponding to the y's)}
\item{shat}{estimated standard deviations of yhat (conditional on the chosen weights)}
\item{args}{Main arguments supplied to awsuni}
}
\references{Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing with applications
to image restoration}, J.R.Statist.Soc. B, 62, Part 2, pp. 335-354}
\note{Although the algorithm evaluates a regression model the structure of
the regression function only depends on the ordering of the independent variable.
Therefore no independent variable is to be given as a parameter but the 
values of the dependent variable are required to be ordered by the value
of the independent variable.
This function is superseded by function aws and will be 
removed in the next mayor version of the package.}
\author{Joerg Polzehl \email{polzehl@wias-berlin.de}}
\seealso{\code{\link{aws}}, \code{\link{awsbi}}, \code{\link{awstri}}}

\examples{
#  Blocks data (from Donoho, Johnstone, Kerkyacharian and Picard (1995))
mofx6 <- function(x){
xj <- c(10,13,15,23,25,40,44,65,76,78,81)/100
hj <- c(40,-50,30,-40,50,-42,21,43,-31,21,-42)*.37
Kern <- function(x) (1-sign(x))/2
apply(Kern(outer(xj,x,"-"))*hj,2,sum)
}
x <- seq(0,1,1/2047)
fx6 <- mofx6(x)
#    sigma==3
y <- rnorm(fx6,fx6,3)
tmp <- awsuni(y)
par(mfrow=c(1,1))
plot(x,y)
lines(x,tmp$yhat,col=2)
lines(x,fx6,col=3)
title(expression(paste("AWS Reconstruction of blocks data  ",sigma==3)))
rm(x,y,fx6,mofx6,tmp)
}
\keyword{regression}
\keyword{nonparametric}
\keyword{smooth}

\eof
\name{getnubi}
\alias{getnubi}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Calculate number of grid points in a bivariate elliptical neighborhood}
\description{
  This function is for internal use only
}
\usage{
getnubi(radiusq, weights)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{radiusq}{ Square of radius }
  \item{weights}{ axis weights \code{w}}
}
\details{
}
\value{
  Returns number of grid points (i,j) with   \code{w[1] i^2 + w[2] j^2 <= radiusq }
}
\author{Joerg Polzehl \email{polzehl@wias-berlin.de}}
\seealso{\code{\link{awstri}}}
\examples{
}
\keyword{misc}% at least one, from doc/KEYWORDS

\eof
\name{getnutri}
\alias{getnutri}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Calculate number of grid points in a 3D ellipsoidal neighborhood}
\description{
  This function is for internal use only
}
\usage{
getnutri(radiusq, weights)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{radiusq}{ Square of radius }
  \item{weights}{ axis weights }
}
\details{
}
\value{
  Returns number of grid points (i,j,k) with  \code{ w[1] i^2 + w[2] j^2 + w[3] k^2 <= radiusq }
}
\references{}
\author{Joerg Polzehl \email{polzehl@wias-berlin.de}}


\seealso{\code{\link{awstri}}}

\examples{
}
\keyword{ misc }% at least one, from doc/KEYWORDS

\eof
\name{laws}
\alias{laws}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Likelihood based Adaptive Weights Smoothing }
\description{
  This function implements a Adaptive Weights Smoothing 
  procedure for local constant Poisson, Bernoulli, Exponential,
  Weibull, Volatility and Gaussian models
  as described in Polzehl \& Spokoiny (2002).
}
\usage{
laws(y, x = NULL, qlambda = NULL, eta = 0.5, lkern = "Triangle", 
     model = "Poisson", shape = NULL, hinit = NULL, hincr = NULL, 
     hmax = 10, NN = FALSE, u = NULL, graph = FALSE, demo = FALSE, 
     symmetric = FALSE, wghts=NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ \code{y} contains the observed values at location \code{x}. 
           In case of \code{x==NULL} (second parameter) \code{y} is assumed to be 
           observed on a one, two or three-dimensional grid. The dimension of 
           \code{y} determines if one, two or three-dimensional AWS is used.}
  \item{x}{ \code{x} is either \code{NULL}, in this case \code{y} is assumed 
           to be observed on a grid, or is a matrix, with rows corresponding to variables, 
           containing the design points where \code{y} is observed. }
  \item{qlambda}{ \code{qlambda} determines the scale parameter \code{qlambda} 
           for the stochastic penalty. The scaling parameter in the stochastic 
           penalty \code{lambda} is choosen as the \code{qlambda}-quantile
           of a Chi-square-distribution with number of parameters in the polynomial 
           model as degrees of freedom. If \code{qlambda==NULL} a standard value 
           depending on \code{model} and \code{symmetric} is choosen. }
  \item{eta}{ \code{eta} is a memory parameter used to stabilize the procedure. 
           \code{eta}  has to be between \code{0} and \code{1}, with 
           \code{eta==0.5} being the default. }
  \item{lkern}{ \code{lkern} determines the location kernel to be used. Options 
           are \code{"Uniform"}, \code{"Triangle"}, \code{"Quadratic"}, 
           \code{"Cubic"} and \code{"Exponential"}. Default is \code{"Triangle"}. 
           The Kernel operates on the squared distance, so \code{"Triangle"}
           corresponds to the use of an Epanechnikov kernel in kernel smoothing. 
           \code{"Exponential"} requires larger values of \code{hmax} and 
           therefore more iterations to reach comparable results. }
  \item{model}{ \code{model} determines the distribution type of \code{y}. 
           Currently implemented models are \code{"Poisson"}, \code{"Bernoulli"},
           \code{"Gaussian"}, \code{"Exponential"}, \code{"Weibull"}, \code{"Volatility"}
           (Estimation of the scale parameter of a Gaussian distribution).
           Defaults to \code{"Poisson"}.}
  \item{shape}{ used for additional parameters of the specified distribution if needed,
           i.e. variance if \code{model=="Gaussian"}
            }
  \item{hinit}{ \code{hinit} Initial bandwidth for the location penalty. 
           Appropriate value is choosen in case of \code{hinit==NULL}  }
  \item{hincr}{ \code{hincr} \code{hincr^(1/d)}, with \code{d} the 
           dimensionality of the design, is used as a factor to increase the 
           bandwidth between iterations. Defauts to \code{hincr==1.2}}   
  \item{hmax}{ \code{hmax} Maximal bandwidth to be used. Determines the 
           number of iterations and is used as the stopping rule. }
  \item{NN}{ If \code{NN==TRUE} use nearest neighbor-rules instead of 
           distances in the location term. }
  \item{u}{ \code{u} used to supply values of the true regression function 
           for test purposes to calculate  Mean Squared Error (MSE) and 
           Mean Absolute Error (MAE) }
  \item{graph}{ \code{graph} if \code{TRUE} results are displayed after each 
           iteration step. }
  \item{demo}{ \code{demo} if \code{TRUE} after each iteration step results 
           are displayed and the process waits for user interaction. }
  \item{symmetric}{ If \code{symmetric==TRUE} the stochastic penalty is
           symmetrized, i.e. \code{(sij + sji)/lambda} is used instead of 
           \code{sij/lambda}. See references for details.}
   \item{wghts}{ Specifies wghts for distance evaluation on a bi- or trivariate grid.
                Allows for anisotropic local neighborhoods. If \code{wghts=NULL}
                isotropic neighborhoods are used. }
}
\details{ This function implements an adaptive weights 
   smoothing (AWS) procedure for a several classes of distributions for the dependent
   variable in local constant regression models. The approach generalizes the original 
   AWS procedure from Polzehl and Spokoiny (2000).
   
   Adaptive weights smoothing is an iterative data adaptive smoothing technique that
   is designed  for smoothing in regression problems with discontinuous regression
   function. The basic assumption is that the regression function can be approximated 
   by a simple local,  e.g. local constant or local polynomial, model. 
   The estimate of the regression function, i.e. the conditional expectation of \code{y}
   given \code{x} is computed as a weighted maximum likelihood estimate, with weights choosen
   in a completely data adaptive way. The procedure is edge preserving. If the assumed local
   model is globally valid, almost all weights used will be 1, i.e. the resulting estimate
   almost is the global estimate.
    
      
   Currently implemented are the following models (specified by parameter \code{model}):

   \describe{
   \item{Binary response}{\code{model="Bernoulli"}}
   \item{Poisson regression}{\code{model="Poisson"} This model allows e.g. for 
   density estimation or for the analysis of poisson count data on a grid (e.g. Positron
   emission tomography (PET)).}
   \item{Exponential regression}{\code{model="Exponential"} Applications of this model
   include e.g. test for constant (over time) failure rates and estimation of tail indicies.}
   \item{Gaussian regression}{\code{model="Gaussian"} This essentially coincides with the
   local constant regression model with additive sub-gaussian errors as described in 
   Polzehl and Spokoiny (2000, 2003).}
   \item{Weibull regression}{\code{model="Weibull"} Applications in reliability analysis.}
   \item{Volatility model}{\code{model="Volatility"} .}
   }
   
   The essential parameter in the procedure is \code{qlambda}. This parameter has an 
   interpretation as a significance level of a test for equivalence of two local
   parameter estimates. Optimal values mainly depend on the choosen \code{model}
   and the value of \code{symmetric} (determines the use of an asymmetric or a symmetrized
   test). The optimal values only slightly depend on the model parameters, i.e. the
   default parameters should work in most situations. Larger values of \code{qlambda}
   may lead to oversmoothing, small values of \code{qlambda} lead to a random segmentation
   of homogeneous regions. A good value of \code{qlambda} can be obtained by the propagation
   condition, requiring that in case of global validity of the local model the 
   estimate for large \code{hmax} should be equal to the global estimate. 
   
   The numerical complexity of the procedure is mainly determined by \code{hmax}. The number
   of iterations is \code{d*log(hmax/hinit)/log(hincr)} with \code{d} being the dimension 
   of \code{y}. Comlexity in each iteration step is \code{Const*hakt*n} with \code{hakt}
   being the actual bandwith in the iteration step and \code{n} the number of design points.
   \code{hmax} determines the maximal possible variance reduction.
}
\value{
  \item{theta }{Parameter estimates, first dimension corresponds to parameter 
components}
  \item{y}{values provided in \code{y}}
  \item{x}{values provided in \code{x}}
  \item{call}{actual function call}
}
\references{ 
\item{ }{Polzehl, J. and Spokoiny, V. (2003). \emph{Varying coefficient 
regression modeling 
     by adaptive weights smoothing}, WIAS-Preprint 818.}
\item{ }{Polzehl, J. and Spokoiny, V. (2002). \emph{Local likelihood 
modelling by adaptive weights smoothing}, WIAS-Preprint 787.}
\item{ }{ Polzehl, J. and Spokoiny, V. (2000). \emph{Adaptive Weights Smoothing
     with applications to image restoration}, J.R.Statist.Soc. B, 62,
     Part 2, pp. 335-354}
 }
\author{ Joerg Polzehl, \email{polzehl@wias-berlin.de}, 
\url{http://www.wias-berlin.de/project-areas/stat/projects/adaptive-image-processing.html}}


\seealso{ SEE ALSO \code{\link{aws}}, \code{\link{awsdens}}, \code{\link{awstindex}} }

\examples{
###
###    Artificial PET data
###
x <- 1:128
f12 <- function(x,y){
2*((1.5*(x-64)^2+(y-64)^2<3025)) +
((x-64)^2+(y-104)^2<16)-((x-92)^2+(y-84)^2<25)+
((x-78)^2+(y-84)^2<25)-((x-64)^2+(y-84)^2<36)+
((x-50)^2+(y-84)^2<36)-((x-36)^2+(y-84)^2<25)+
((x-92)^2+(y-64)^2<25)-((x-78)^2+(y-64)^2<16)+
((x-64)^2+(y-64)^2<16)-((x-50)^2+(y-64)^2<25)+
((x-36)^2+(y-64)^2<25)-((x-92)^2+(y-44)^2<36)+
((x-78)^2+(y-44)^2<36)-((x-64)^2+(y-44)^2<25)+
((x-50)^2+(y-44)^2<25)-((x-36)^2+(y-44)^2<16)+
((x-64)^2+(y-24)^2<16)
}
u0 <- 2*outer(x,x,"f12")
set.seed(1)
y <- matrix(rpois(u0,u0),128,128)
# use larger hmax for good results
yhat <- laws(y,model="Poisson",hmax=4)$theta
par(mfrow=c(1,3),mar=c(3,3,3,.5),mgp=c(2,1,0))
image(y,col=gray((0:255)/255))
title("Observed image")
image(yhat,col=gray((0:255)/255))
title("AWS-Reconstruction")
image(u0,col=gray((0:255)/255))
title("True image")
rm(u0,y,yhat,x)
###
###   VOLATITILTY ESTIMATION
###
###   artificial example
###
sigma <- c(rep(1,125),rep(2,125),rep(.5,125),rep(1,125))
set.seed(1)
x <- rnorm(sigma,0,sigma)
tmpa <- laws(x,model="Volatility",u=sigma,graph=TRUE,hmax=250)
tmps <- laws(x,model="Volatility",u=sigma,hmax=250,symmetric=TRUE)
par(mfrow=c(1,1),mar=c(3,3,3,1))
plot(abs(x),col=3,xlab="time t",ylab=expression(abs( R[t] )))
lines(tmpa$theta,col=1,lwd=2)
lines(tmps$theta,col=1,lwd=2,lty=2)
lines(sigma,col=2,lwd=2,lty=3)
legend(350,5.5,c("asymmetric AWS","symmetric AWS","true sigma"),
                 lwd=c(2,2,2),lty=c(1,2,3),col=c(1,1,2))
title(expression(paste("Estimates of  ",sigma(t))))
rm(tmpa,tmps,sigma,x)
}
\keyword{ smooth }% at least one, from doc/KEYWORDS
\keyword{ nonparametric }% __ONLY ONE__ keyword per line
\keyword{ regression }% __ONLY ONE__ keyword per line
\keyword{ models }% __ONLY ONE__ keyword per line

\eof
