\name{aic}
\alias{aic}
\title{
Compute Akaike's Information Criterion.
}
\usage{
aic(x, \dots, pen=2)
}
\description{
  The calling sequence for \code{aic} matches those for the
  \code{\link{locfit}} or \code{\link{locfit.raw}} functions.
  The fit is not returned; instead, the returned object contains
  Akaike's information criterion for the fit.

  The definition of AIC used here is -2*log-likelihood + pen*(fitted d.f.).
  For quasi-likelihood, and local regression, this assumes the scale
  parameter is one. Other scale parameters can effectively be used
  by changing the penalty.

  The AIC score is exact (up to numerical roundoff) if the
  \code{ev="data"} argument is provided. Otherwise, the residual
  sum-of-squares and degrees of freedom are computed using locfit's
  standard interpolation based approximations.
}
\arguments{
  \item{x}{model formula}
  \item{...}{other arguments to locfit}
  \item{pen}{penalty for the degrees of freedom term}
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{aicplot}}
}

\keyword{smooth}
\keyword{htest}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{aicplot}
\alias{aicplot}
\title{
Compute an AIC plot.
}
\description{
  The \code{aicplot} function loops through calls to the \code{\link{aic}}
function (and hence to \code{link{locfit}}), using a different
smoothing parameter for each call.
The returned structure contains the AIC statistic for each fit, and can
be used to produce an AIC plot.
}
\usage{
aicplot(..., alpha)
}
\arguments{
  \item{...}{ arguments to the \code{\link{aic}}, \code{\link{locfit}}
     functions.}
  \item{alpha}{ Matrix of smoothing parameters. The \code{aicplot} function
     loops through calls to \code{\link{aic}}, using each row of
     \code{alpha} as the smoothing parameter in turn. If \code{alpha}
     is provided as a vector, it will be converted to a one-column
     matrix, thus interpreting each component as a nearest neighbor
    smoothing parameter.}
}

\value{
  An object with class \code{"gcvplot"}, containing the smoothing
  parameters and AIC scores. The actual plot is produced using
  \code{\link{plot.gcvplot}}.
}

\examples{
data(morths)
plot(aicplot(deaths~age,weights=n,data=morths,family="binomial",alpha=seq(0.2,1.0,by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{aic}},
  \code{\link{plot.gcvplot}}
}

\keyword{smooth}
\keyword{hplot}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{ais}
\alias{ais}
\title{Australian Institute of Sport Dataset}
\usage{data(ais)}
\format{
  A dataframe.
}
\description{
  The first two columns are the gender of the athlete
  and their sport. The remaining 11 columns are various measurements
  made on the athletes.
}
\source{
  Cook and Weisberg (1994).
}
\references{
  Cook and Weisberg (1994). An Introduction to Regression Graphics.
    Wiley, New York.
}
\keyword{datasets}

\eof
\name{ang}
\alias{ang}
\title{
Angular Term for a Locfit model.
}
\usage{
ang(x)
}
%locfit(y ~ ang(x), scale = 1/(2*pi))
\description{
  The \code{ang()} function is used in a locfit model formula
  to specify that a variable should be treated as an angular
  or periodic term. The \code{scale} argument of
  \code{\link{locfit}()} is used to set the period.
}

\arguments{
  \item{x}{numeric variable to be treated periodically.}
}
\note{
    The \code{scale} argument to \code{\link{locfit}} is used to specify
    the period divided by \eqn{2\pi} of the term. The default is
    \code{sacle=1}, giving a period of  \eqn{2\pi}.
}
\examples{
# generate an x variable, and a response with period 0.2
x <- seq(0, 1, length=200)
y <- sin(10*pi*x) + rnorm(200) / 5

# compute the periodic local fit. Note the scale argument is period/(2pi)
fit <- locfit(y ~ ang(x), scale=0.2/(2*pi))

# plot the fit over a single period
plot(fit)

# plot the fit over the full range of the data
plot(fit, xlim=c(0,1))
}

\references{
Loader, C. (1999). Local Regression and Likelihood. Springer, NY (Section 6.2).
}

\seealso{
  \code{\link{locfit}}.
}

\keyword{models}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{bad}
\alias{bad}
\title{Example dataset for bandwidth selection}
\usage{data(bad)}
\format{
Data Frame with x and y variables.
}
\description{
Example dataset from Loader (1999).
}
\references{
  Loader, C. (1999). Bandwidth Selection: Classical or Plug-in?
  Annals of Statistics 27.
}
\keyword{datasets}

\eof
\name{border}
\alias{border}
\title{Cricket Batting Dataset}
\usage{data(border)}
\format{
  A dataframe with day (decimalized); not out indicator and score.
  The not out indicator should be used as a censoring data.
}
\description{
  Scores in 265 innings for Australian batsman Allan Border.
}
\source{
  Compiled from the Cricinfo archives.
}
\references{CricInfo: The Home of Cricket on the Internet.
  \url{http://www.cricket.org}
}
\keyword{datasets}

\eof
\name{chemdiab}
\alias{chemdiab}
\title{Chemical Diabetes Dataset}
\usage{data(chemdiab)}
\format{
  Data frame with five numeric measurements and categroical response.
}
\description{
  Numeric variables are \code{rw}, \code{fpg},
  \code{ga}, \code{ina} and \code{sspg}. Classifier \code{cc} is the Diabetic
  type.
}
\source{
  Reaven and Miller (1979).
}
\references{
  Reaven, G. M. and Miller, R. G. (1979).
  An attempt to define the nature of chemical diabetes using a multidimensional
  analysis. Diabetologia 16, 17-24.
}
\keyword{datasets}

\eof
\name{claw54}
\alias{claw54}
\title{Claw Dataset}
\usage{data(claw54)}
\format{
  Numeric vector with length 54.
}
\description{
  A random sample of size 54 from the claw density of Marron and Wand
  (1992), as used in Figure 10.5 of Loader (1999).
}
\source{
  Randomly generated.
}
\references{
  Loader, C. (1999). Local Regression and Likelihood. Springer, New York.
  
  Marron, J. S. and Wand, M. P. (1992). Exact mean integrated squared error.
    Annals of Statistics 20, 712-736.
}
\keyword{datasets}

\eof
\name{cldem}
\alias{cldem}
\title{Example data set for classification}
\usage{data(cldem)}
\format{
Data Frame with x and y variables.
}
\description{
Observations from Figure 8.7 of Loader (1999).
}
\references{
  Loader, C. (1999). Local Regression and Likelihood. Springer, New York.
}
\keyword{datasets}

\eof
\name{cltest}
\alias{cltest}
\title{Test dataset for classification}
\usage{data(cltest)}
\format{
Data Frame. Three variables x1, x2 and y. The latter indicates
class membership.
}
\description{
200 observations from a 2 population model. Under population 0,
\eqn{x_{1,i}} has a standard normal distribution, and
\eqn{x_{2,i} = (2-x_{1,i}^2+z_i)/3}, where \eqn{z_i} is also standard normal.
Under population 1, \eqn{x_{2,i} = -(2-x_{1,i}^2+z_i)/3}.
The optimal classification regions form a checkerboard pattern,
with horizontal boundary at \eqn{x_2=0}, vertical boundaries at
\eqn{x_1 = \pm \sqrt{2}}.

This is the same model as the cltrain dataset.
}
\keyword{datasets}

\eof
\name{cltrain}
\alias{cltrain}
\title{Training dataset for classification}
\usage{data(cltrain)}
\format{
Data Frame. Three variables x1, x2 and y. The latter indicates
class membership.
}
\description{
200 observations from a 2 population model. Under population 0,
\eqn{x_{1,i}} has a standard normal distribution, and
\eqn{x_{2,i} = (2-x_{1,i}^2+z_i)/3}, where \eqn{z_i} is also standard normal.
Under population 1, \eqn{x_{2,i} = -(2-x_{1,i}^2+z_i)/3}.
The optimal classification regions form a checkerboard pattern,
with horizontal boundary at \eqn{x_2=0}, vertical boundaries at
\eqn{x_1 = \pm \sqrt{2}}.

This is the same model as the cltest dataset.
}
\keyword{datasets}

\eof
\name{co2}
\alias{co2}
\title{Carbon Dioxide Dataset}
\usage{data(co2)}
\format{
  Data frame with \code{year}, \code{month} and \code{co2} variables.
}
\description{
  Monthly time series of carbon dioxide measurements at Mauna Loa,
  Hawaii from 1959 to 1990.
}
\source{
  Boden, Sepanski and Stoss (1992).
}
\references{
  Boden, Sepanski and Stoss (1992).
  Trends '91: A compedium of data on global change - Highlights.
  Carbon Dioxide Information Analysis Center, Oak Ridge National Laboratory.
}
\keyword{datasets}

\eof
\name{cp}
\alias{cp}
\title{
Compute Mallows' Cp for local regression models.
}
\usage{
cp(x, \dots, sig2=1)
}
\description{
  The calling sequence for \code{cp} matches those for the
  \code{\link{locfit}} or \code{\link{locfit.raw}} functions.
  The fit is not returned; instead, the returned object contains
  Cp criterion for the fit.

  Cp is usually computed using a variance estimate from the largest
  model under consideration, rather than
  \eqn{\sigma^2=1}. This will be done
  automatically when the \code{\link{cpplot}} function is used.

  The Cp score is exact (up to numerical roundoff) if the
  \code{ev="data"} argument is provided. Otherwise, the residual
  sum-of-squares and degrees of freedom are computed using locfit's
  standard interpolation based approximations.
}
\arguments{
  \item{x}{model formula or numeric vector of the independent variable.}
  \item{...}{other arguments to \code{\link{locfit}} and/or
    \code{\link{locfit.raw}}.}
  \item{sig2}{residual variance estimate.}
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{cpplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{cpar}
\alias{cpar}
\title{
Conditionally parametric term for a Locfit model.
}
\usage{cpar(x)}
\description{
  A term entered in a \code{\link{locfit}} model formula using
  \code{cpar} will result in a fit that is conditionally parametric.
  If the formula is \code{y~x1+cpar(x2)}, the resulting surface,
  for fixed \code{x1}, will be quadratic (using the default \code{deg=2})
  in \code{x2}. Note that the coefficients of the quadratic may depend
  on \code{x1}; this distinguishes a conditionally parametric model
  from a semiparametric model.
}
\arguments{
  \item{x}{numeric variable.}
}
\examples{
data(ethanol)
# fit a conditionally parametric model
fit <- locfit(NOx~E+cpar(C),data=ethanol)
plot(fit)
# one way to force a parametric fit with locfit
fit <- locfit(NOx~cpar(E),data=ethanol)
}

\seealso{
  \code{\link{locfit}}
}

\keyword{models}

\eof
\name{cpplot}
\alias{cpplot}
\title{
Compute a Cp plot.
}
\usage{
cpplot(..., alpha, sig2)
}
\description{
  The \code{cpplot} function loops through calls to the \code{\link{cp}}
function (and hence to \code{link{locfit}}), using a different
smoothing parameter for each call.
The returned structure contains the Cp statistic for each fit, and can
be used to produce an AIC plot.
}
\arguments{
  \item{...}{ arguments to the \code{\link{cp}}, \code{\link{locfit}}
     functions.}
  \item{alpha}{ Matrix of smoothing parameters. The \code{cpplot} function
     loops through calls to \code{\link{cp}}, using each row of
     \code{alpha} as the smoothing parameter in turn. If \code{alpha}
     is provided as a vector, it will be converted to a one-column
     matrix, thus interpreting each component as a nearest neighbor
    smoothing parameter.}
  \item{sig2}{ Residual variance. If not specified, the residual variance
    is computed using the fitted model with the fewest residual degrees
    of freedom.}
}

\value{
  An object with class \code{"gcvplot"}, containing the smoothing
  parameters and CP scores. The actual plot is produced using
  \code{\link{plot.gcvplot}}.
}

\examples{
data(ethanol)
plot(cpplot(NOx~E,data=ethanol,alpha=seq(0.2,1.0,by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{aic}},
  \code{\link{plot.gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{crit}
\alias{crit}
\title{
Compute critical values for confidence intervals.
}
\usage{
crit(fit, const = c(0, 1), d = 1, cov = 0.95, rdf = 0)
}
\arguments{
\item{fit}{
  \code{"locfit"} object. This is optional; if a fit is provided, defaults
  for the other arguments are taken from the critical value currently
  stored on this fit, rather than the usual values above.
  \code{crit(fit)} with no other arguments will just return the current
  critical value.
}
\item{const}{
  Tube formula constants for simultaneous bands (the default, \code{c(0,1)},
  produces pointwise coverage). Usually this is generated by the
  \code{\link{kappa0}} function and should not be provided by the user.
}
\item{d}{
  Dimension of the fit. Again, users shouldn't usually provide it.
}
\item{cov}{
Coverage Probability for critical values.
}
\item{rdf}{
  Residual degrees of freedom. If non-zero, the critical values are based
  on the Student's t distribution. When \code{rdf=0}, the normal distribution
  is used.
}
}
\description{
  Every \code{"locfit"} object contains a critical value object to be used in
  computing and ploting confidence intervals. By default, a 95% pointwise
  confidence level is used. To change the confidence level, the critical
  value object must be substituted using \code{\link{crit}} and
  \code{\link{crit<-}}.
}
\value{
  Critical value object.
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}},
\code{\link{kappa0}}, \code{\link{crit<-}}.
}
\examples{
# compute and plot 99\% confidence intervals, with local variance estimate.
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
crit(fit) <- crit(fit,cov=0.99)
plot(fit,band="local")

# compute and plot 99\% simultaneous bands
crit(fit) <- kappa0(NOx~E,data=ethanol,cov=0.99)
plot(fit,band="local")
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{crita}
\alias{crit<-}
\title{
Substitute critical values on ``locfit'' object.
}
\usage{
crit(fit) <- crit(fit, cov=0.95)
}
\arguments{
\item{fit}{\code{locfit} object.}
}
\value{
  Critical value object generated by
  \code{\link{crit}} or \code{\link{kappa0}}.
}
\description{
  Every \code{locfit} object contains a critical value object to be used in
  computing and ploting confidence intervals. By default, a 95\% pointwise
  confidence level is used.  To change the confidence level, the critical
  value object must be substituted using \code{\link{crit}} and
  \code{\link{crit<-}}.
}
%\value{
%  Critical value object.
%}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}},
\code{\link{kappa0}}, \code{\link{crit}}.
}
\examples{
# compute and plot 99\% confidence intervals, with local variance estimate.
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
crit(fit) <- crit(fit,cov=0.99)
plot(fit,band="local")

# compute and plot 99\% simultaneous bands
crit(fit) <- kappa0(NOx~E,data=ethanol,cov=0.99)
plot(fit, band="local")
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{density.lf}
\alias{density.lf}
\title{
Density estimation using Locfit
}
\usage{
density.lf(x, n=50, window="gaussian", width, from, to, cut, ev="grid", deg=0, family="density", link="ident", ...)
}
\description{
  This function provides an interface to Locfit, in the syntax of
  the standard S-Plus \code{density} function. This can reproduce
  \code{density} results, but allows additional
  \code{\link{locfit.raw}} arguments, such as the degree of fit, to be given.

  It also works in double precision, whereas \code{density} only works
  in single precision.
}
\arguments{
\item{x}{numeric vector of observations whose density is to be estimated.}
\item{n}{number of evaluation points.
  Equivalent to the \code{\link{locfit.raw} mg} argument.}
\item{window}{Window type to use for estimation.
  Equivalent to the \code{\link{locfit.raw} kern} argument. This includes
  all the \code{density} windows except \code{cosine}.}
\item{width}{Window width. Following \code{density}, this is the full
  width; not the half-width usually used by Locfit and
  many other smoothers.}
\item{from}{Lower limit for estimation domain.}
\item{to}{Upper limit for estimation domain.}
\item{cut}{Controls default expansion of the domain.}
\item{ev}{Locfit evaluation structure -- default \code{"grid"}.}
\item{deg}{Fitting degree -- default 0 for kernel estimation.}
\item{family}{Fitting family -- default is \code{"density"}.}
\item{link}{Link function -- default is the \code{"identity"}.}
\item{...}{Additional arguments to \code{\link{locfit.raw}}, with standard
  defaults.}
}
\value{
  A list with components \code{x} (evaluation points) and \code{y}
  (estimated density).
}
\seealso{
\code{density},
\code{\link{locfit}},
\code{\link{locfit.raw}}
}
\examples{
data(geyser)
density.lf(geyser, window="tria")
# the same result with density, except less precision.
density(geyser, window="tria")
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{diab}
\alias{diab}
\title{Exhaust emissions}
\usage{data(ethanol)}
\format{
  Data frame with NOx, E and C variables.
}
\description{
  NOx exhaust emissions from a single cylinder engine.  Two predictor
  variables are E (the engine's equivalence ratio) and C (Compression
  ratio). 
}
\source{
  Brinkman (1981). Also studied extensively by Cleveland (1993).
}
\references{
  Brinkman, N. D. (1981). Ethanol fuel - a single-cylinder engine study
  of efficiency and exhaust emissions. SAE transactions 90, 1414-1424.
  
  Cleveland, W. S. (1993). Visualizing data. Hobart Press, Summit, NJ.
}
\keyword{datasets}

\eof
\name{ethanol}
\alias{ethanol}
\title{Exhaust emissions}
\usage{data(ethanol)}
\format{
  Data frame with NOx, E and C variables.
}
\description{
  NOx exhaust emissions from a single cylinder engine.  Two predictor
  variables are E (the engine's equivalence ratio) and C (Compression
  ratio). 
}
\source{
  Brinkman (1981). Also studied extensively by Cleveland (1993).
}
\references{
  Brinkman, N. D. (1981). Ethanol fuel - a single-cylinder engine study
  of efficiency and exhaust emissions. SAE transactions 90, 1414-1424.
  
  Cleveland, W. S. (1993). Visualizing data. Hobart Press, Summit, NJ.
}
\keyword{datasets}

\eof
\name{expit}
\alias{expit}
\title{
Inverse logistic link function
}
\usage{
expit(x)
}
\description{
  Computes \eqn{e^x/(1+e^x)}{\exp(x)/(1+\exp(x))}. 
  This is the inverse of the logistic link function,
  \eqn{\log(p/(1-p))}.
}
\arguments{
  \item{x}{numeric vector}
}

\keyword{math}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{fitted.locfit}
\alias{fitted.locfit}
\title{
Fitted values for a `"locfit"' object.
}
\usage{
\method{fitted}{locfit}(object, data, what="coef", cv=FALSE, studentize=FALSE,
    type="fit", tr, ...)
}
\description{Evaluates the fitted values (i.e. evaluates the surface
  at the original data points) for a Locfit object. This function works
  by reconstructing the model matrix from the original formula, and
  predicting at those points. The function may be fooled; for example,
  if the original data frame has changed since the fit, or if the
  model formula includes calls to random number generators.}
\arguments{
\item{object}{
\code{"locfit"} object.
}
\item{data}{
The data frame for the original fit. Usually, this shouldn't be needed,
especially when the function is called directly. It may be needed
when called inside another function.
}
\item{what}{
What to compute fitted values of. The default, \code{what="coef"}, works
  with the fitted curve itself. Other choices include \code{"nlx"} for the
  length of the weight diagram; \code{"infl"} for the influence function;
  \code{"band"} for the bandwidth; \code{"degr"} for the local polynomial
  degree; \code{"lik"} for the maximized local likelihood; \code{"rdf"}
  for the local residual degrees of freedom and \code{"vari"} for the
  variance function. The interpolation algorithm for some of these quantities
  is questionable.
}
\item{cv}{
If \code{TRUE}, leave-one-out cross validated fitted values are approximated.
Won't make much sense, unless \code{what="coef"}.
}
\item{studentize}{
If \code{TRUE}, residuals are studentized.
}
\item{type}{
  Type of fit or residuals to compute. The default is \code{"fit"} for
  \code{fitted.locfit}, and \code{"dev"} for \code{\link{residuals.locfit}}.
  Other choices include \code{"pear"} for Pearson residuals; \code{"raw"}
  for raw residuals, \code{"ldot"} for likelihood derivative;
  \code{"d2"} for the deviance residual squared; \code{lddot} for the
  likelihood second derivative. Generally, \code{type} should only be
  used when \code{what="coef"}.
}
\item{tr}{
Back transformation for likelihood models.
}
\item{...}{currently not used.}
}
\value{
A numeric vector of the fitted values.
}
\seealso{
  \code{locfit},
  \code{predict.locfit},
  \code{residuals.locfit}
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{formula.locfit}
\alias{formula.locfit}
\title{
Formula method for a Locfit object.
}
\usage{
\method{formula}{locfit}(x, ...)
}
\description{Extract the model formula from a locfit object.}
\arguments{
\item{x}{ \code{"locfit"} object.}
\item{...}{currently not used.}
}
\value{
Returns the formula from the locfit object.
}
\seealso{
\code{locfit}
}
\keyword{models}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{gam.lf}
\alias{gam.lf}
\title{
Locfit call for Generalized Additive Models
}
\usage{
gam.lf(x, y, w, xeval, ...)
}
\description{
  This is a locfit calling function used by
  \code{\link{lf}()} terms in additive models. It is
  not normally called directly by users.
}
\arguments{
  \item{x}{numeric predictor}
  \item{y}{numeric response}
  \item{w}{prior weights}
  \item{xeval}{evaluation points}
  \item{...}{other arguments to \code{\link{locfit.raw}()}}
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{lf}},
  \code{gam}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{gam.slist}
\alias{gam.slist}
\title{Vector of GAM special terms}
\format{
  Character vector.
}
\description{
  This vector adds \code{"lf"} to the default vector of special
  terms recognized by a \code{gam()} model formula.
  To ensure this is recognized, attach the Locfit library with
  \code{library(locfit,first=T)}.
}
\seealso{
  \code{\link{lf}},
  \code{gam}
}
\keyword{datasets}

\eof
\name{gcv}
\alias{gcv}
\title{
Compute generalized cross-validation statistic.
}
\usage{
gcv(x, \dots)
}
\description{
  The calling sequence for \code{gcv} matches those for the
  \code{\link{locfit}} or \code{\link{locfit.raw}} functions.
  The fit is not returned; instead, the returned object contains
  Wahba's generalized cross-validation score for the fit.

  The GCV score is exact (up to numerical roundoff) if the
  \code{ev="data"} argument is provided. Otherwise, the residual
  sum-of-squares and degrees of freedom are computed using locfit's
  standard interpolation based approximations.

  For likelihood models, GCV is computed uses the deviance
  in place of the residual sum of squares. This produces useful
  results but I do not know of any theory validating
  this extension.
}

\arguments{
   \item{x}{either a numeric vector or a data frame (same as the
   arguments to \code{\link{locfit}}).}
   \item{...}{other argument to \code{\link{locfit}}.}
}
\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{gcvplot}
\alias{gcvplot}
\title{
Compute a generalized cross-validation plot.
}
\usage{
gcvplot(..., alpha, df=2)
}
\description{
  The \code{gcvplot} function loops through calls to the \code{\link{gcv}}
function (and hence to \code{link{locfit}}), using a different
smoothing parameter for each call.
The returned structure contains the GCV statistic for each fit, and can
be used to produce an GCV plot.
}
\arguments{
  \item{...}{ arguments to the \code{\link{gcv}}, \code{\link{locfit}}
     functions.}
  \item{alpha}{ Matrix of smoothing parameters. The \code{gcvplot} function
     loops through calls to \code{\link{gcv}}, using each row of
     \code{alpha} as the smoothing parameter in turn. If \code{alpha}
     is provided as a vector, it will be converted to a one-column
     matrix, thus interpreting each component as a nearest neighbor
    smoothing parameter.}
  \item{df}{ Degrees of freedom to use as the x-axis. 2=trace(L),
    3=trace(L'L).}
}

\value{
  An object with class \code{"gcvplot"}, containing the smoothing
  parameters and GCV scores. The actual plot is produced using
  \code{\link{plot.gcvplot}}.
}

\examples{
data(ethanol)
plot(gcvplot(NOx~E,data=ethanol,alpha=seq(0.2,1.0,by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{plot.gcvplot}},
  \code{\link{summary.gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{geyser}
\alias{geyser}
\title{Old Faithful Geyser Dataset}
\usage{data(geyser)}
\format{
  A numeric vector of length 107.
}
\description{
  The durations of 107 eruptions of the Old Faithful Geyser.
}
\source{
  Scott (1992). Note that several different Old Faithful Geyser datasets
  (including the faithful dataset in R's base library) have been used
  in various places in the statistics literature. The version provided
  here has been used in density estimation and bandwidth selection work.
}
\references{
  Scott, D. W. (1992). Multivariate Density Estimation: Theory,
  Practice and Visualization. Wiley.
}
\keyword{datasets}

\eof
\name{geyser.round}
\alias{geyser.round}
\title{Discrete Old Faithful Geyser Dataset}
\usage{data(geyser.round)}
\format{
  Data Frame with variables \code{duration} and \code{count}.
}
\description{
  This is a variant of the \code{\link{geyser}} dataset, where
  each observation is rounded to the nearest 0.05 minutes, and the
  counts tallied.
}
\source{
  Scott (1992). Note that several different Old Faithful Geyser datasets
  (including the faithful dataset in R's base library) have been used
  in various places in the statistics literature. The version provided
  here has been used in density estimation and bandwidth selection work.
}
\references{
  Scott, D. W. (1992). Multivariate Density Estimation: Theory,
  Practice and Visualization. Wiley.
}
\keyword{datasets}

\eof
\name{hatmatrix}
\alias{hatmatrix}
\title{
Weight diagrams and the hat matrix for a local regression model.
}
\usage{
hatmatrix(formula, dc, ...)
}
\arguments{
\item{formula}{
model formula.
}
\item{dc}{ Derivative adjustment (to be passed to \code{\link{locfit.raw}}.}
\item{...}{
Other arguments to \code{\link{locfit}} and
\code{\link{locfit.raw}}.
}}
\description{
\code{hatmatrix()} computes the weight diagrams (also known as
equivalent or effective kernels) for a local regression smooth.
Essentially, \code{hatmatrix()} is a front-end to \code{\link{locfit}()},
setting a flag to compute and return weight diagrams, rather than the
fit.
}
\value{
A matrix with n rows and p columns; each column being the
weight diagram for the corresponding \code{locfit} fit point.
If \code{ev="data"}, this is the transpose of the hat matrix.
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit.1d}}, \code{\link{plot.locfit.2d}},
\code{\link{plot.locfit.3d}}, \code{\link{lines.locfit}}, \code{\link{predict.locfit}}
}
\keyword{smooth}

\eof
\name{heart}
\alias{heart}
\title{Survival Times of Heart Transplant Recipients}
\usage{data(heart)}
\format{
  Data frame with surv, cens and age variables.
}
\description{
  The survival times of 184 participants in the Stanford heart
  transplant program.
}
\source{
  Miller and Halperin (1982).
  The original dataset includes information on additional patients who
  never received a transplant. Other authors reported earlier versions
  of the data.
}
\references{
  Miller, R. G. and Halperin, J. (1982). Regression with censored data.
  Biometrika 69, 521-531.
}
\keyword{datasets}

\eof
\name{insect}
\alias{insect}
\title{Insect Dataset}
\usage{data(insect)}
\format{
  Data frame with \code{lconc} (dosage), \code{deaths}
  (number of deaths) and \code{nins}
  (number of insects) variables.
}
\description{
  An experiment measuring death rates for insects, with 30 insects at
  each of five treatment levels.
}
\source{
  Bliss (1935).
}
\references{
  Bliss (1935).
  The calculation of the dosage-mortality curve.
  Annals of Applied Biology 22, 134-167.
}
\keyword{datasets}

\eof
\name{iris}
\alias{iris}
\title{Fisher's Iris Data (subset)}
\usage{data(iris)}
\format{
  Data frame with species, petal.wid, petal.len, sepal.wid, sepal.len.
}
\description{
  Four measurements on each of fifty flowers of two species of iris
  (Versicolor and Virginica) -- A classification dataset. Fisher's
  original dataset contained a third species (Setosa) which is trivially
  seperable.
}
\source{
  Fisher (1936). Reproduced in Andrews and Herzberg (1985) Chapter 1.

}
\references{
  Andrews, D. F. and Herzberg, A. M. (1985). Data. Springer-Verlag.

  Fisher, R. A. (1936). The Use of Multiple Measurements in Taxonomic
  Problems. Annals of Eugenics 7, Part II. 179-188.
}
\keyword{datasets}

\eof
\name{kangaroo}
\alias{kangaroo}
\title{Kangaroo skull measurements dataset}
\usage{data(kangaroo)}
\format{
  Data frame with measurements on the skulls of 101 kangaroos.
  (number of insects) variables.
}
\description{
  Variables are \code{sex} (m/f), \code{spec} (giganteus, melanops,
  fuliginosus) and 18 numeric measurements.
}
\source{
  Andrews and Herzberg (1985) Chapter 53.
}
\references{
  Andrews, D. F. and Herzberg, A. M. (1985).
  Data.
  Springer-Verlag, New York.
}
\keyword{datasets}

\eof
\name{kappa0}
\alias{kappa0}
\title{
Critical Values for Simultaneous Confidence Bands.
}
\usage{
kappa0(formula, cov=0.95, \dots)
}
\arguments{
\item{formula}{
  Local regression model formula. A \code{"locfit"} object can also
  be provided; in this case the formula and other arguments are extracted
  from this object.
}
\item{cov}{ Coverage Probability for critical values.  }
\item{...}{ Other arguments to \code{\link{locfit}}. Important arguments
  include \code{flim} and \code{alpha}. }
}
\description{
  The geometric constants for simultaneous confidence bands are computed,
  as described in Sun and Loader (1994) (bias adjustment is not implemented
  here). These are then passed to the \code{\link{crit}} function, which
  computes the critical value for the confidence bands.

  The method requires both the weight diagrams l(x), the
  derivative l'(x) and (in 2 or more dimensions) the second
  derivatives l''(x).
  These are implemented exactly for a constant bandwidth; that
  is, \code{alpha=c(0,h)} for some \code{h}. For nearest
  neighbor bandwidths, the computations are approximate and a warning
  is produced.

  The theoretical justification for the bands uses normality of
  the random errors \eqn{e_1,\dots,e_n} in the regression model,
  and in particular the spherical symmetry of the error vector.
  For non-normal distributions, and likelihood models, one relies
  on central limit and related theorems.

  Computation uses the product Simpson's rule to evaluate the
  multidimensional integrals (The domain of integration, and
  hence the region of simultaneous coverage, is determined by
  the \code{flim} argument). Expect the integration to be slow in more
  than one dimension. The \code{mint} argument controls the
  precision.
}
\value{
  A list with components for the critical value, geometric constants,
  e.t.c. Can be passed directly to \code{\link{plot.locfit}} as the
  \code{crit} argument.
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}},
\code{\link{crit}}, \code{\link{crit<-}}.
}
\examples{
# compute and plot simultaneous confidence bands
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
crit(fit) <- kappa0(NOx~E,data=ethanol)
plot(fit,crit=crit,band="local")
}
\references{
Sun, J. and Loader, C. (1994). Simultaneous confidence bands for linear
  regression and smoothing. Annals of Statistics 22, 1328-1345.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{kdeb}
\alias{kdeb}
\title{
Bandwidth selectors for kernel density estimation.
}
\usage{
kdeb(x, h0 = 0.01 * sd, h1 = sd, meth = c("AIC", "LCV", "LSCV", "BCV",
  "SJPI", "GKK"), kern = "gauss", gf = 2.5)

}
\description{
Function to compute kernel density estimate bandwidths, as used in the
simulation results in Chapter 10 of Loader (1999).

  This function is included for comparative purposes only. Plug-in
selectors are based on flawed logic, make unreasonable and restrictive
assumptions
and do not use the full power of the estimates available in Locfit.
Any relation between the results produced by this function and
desirable estimates are entirely coincidental.
}
\arguments{
\item{x}{One dimensional data vector.}
\item{h0}{Lower limit for bandwidth selection. Can be fairly small,
  but h0=0 would cause problems.}
\item{h1}{Upper limit.}
\item{meth}{Required selection method(s).}
\item{kern}{Kernel. Most methods require \code{kern="gauss"}, the default
  for this function only.}
\item{gf}{Standard deviation for the gaussian kernel. Default 2.5, as
  Locfit's standard. Most papers use 1.
}
}
\value{
Vector of selected bandwidths.
}
\references{
Loader, C. (1999). Local Regression and Likelihood. Springer, New York.
}
\keyword{smooth}

\eof
\name{km.mrl}
\alias{km.mrl}
\title{
Mean Residual Life using Kaplan-Meier estimate
}
\usage{
km.mrl(times,cens)
}
\description{
  This function computes the mean residual life for censored data
  using the Kaplan-Meier estimate of the survival function. If
  \eqn{S(t)} is the K-M estimate, the MRL for a censored observation
  is computed as \eqn{(\int_t^{\infty} S(u)du)/S(t)}. We take
  \eqn{S(t)=0} when \eqn{t} is greater than the largest observation,
  regardless of whether that observation was censored.

  When there are ties between censored and uncensored observations,
  for definiteness our ordering places the censored observations
  before uncensored.

  This function is used by \code{\link{locfit.censor}} to compute
  censored regression estimates.
}
\arguments{
\item{times}{
  Obsereved survival times.
}
\item{cens}{
  Logical variable indicating censoring. The coding is \code{1}
  or \code{TRUE} for censored; \code{0} or \code{FALSE} for uncensored.
}
}
\value{
A vector of the estimated mean residual life. For uncensored observations,
the corresponding estimate is 0.
}
\examples{
# censored regression using the Kaplan-Meier estimate.
data(heart)
fit <- locfit.censor(log10(surv+0.5)~age, cens=cens, data=heart, km=TRUE)
plotbyfactor(heart$age, 0.5+heart$surv, heart$cens, ylim=c(0.5,16000), log="y")
lines(fit, tr=function(x)10^x)
}
\references{
Buckley, J. and James, I. (1979). Linear Regression with censored data.
  Biometrika 66, 429-436.

Loader, C. (1999). Local Regression and Likelihood. Springer, NY (Section 7.2).
}
\seealso{
\code{\link{locfit.censor}}
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{knots}
\alias{knots}
\title{
Extraction of fit-point information from a Locfit object.
}
\usage{
knots(x, tr, what, delete.pv, ...)
}
\description{
  Extracts information, such as fitted values, influence functions
  from a \code{"locfit"} object.
}
\arguments{
\item{x}{
Fitted object from \code{\link{locfit}()}.
}
\item{tr}{
Back transformation. Default is the invers link function from the Locfit
object.
}
\item{what}{
What to return; default is \code{c("x","coef","h","nlx")}.
Allowed fields are \code{x} (fit points); \code{coef} (fitted values);
\code{f1} (local slope); \code{nlx} (length of the weight diagram);
\code{nlx1} (estimated derivative of \code{nlx}); \code{se} (standard errors);
\code{infl} (influence function); \code{infla} (slope of influence function);
\code{lik} (maximixed local log-likelihood and local degrees of freedom);
\code{h} (bandwidth) and \code{deg} (degree of fit).
}
\item{delete.pv}{If \code{T}, pseudo-vertices are deleted.}
\item{...}{not used.}
}
\value{
A matrix with one row for each fit point. Columns correspond to
the specified \code{what} vector; some fields contribute multiple columns.
}
\keyword{smooth}

\eof
\name{lcv}
\alias{lcv}
\title{
Compute Likelihood Cross Validation Statistic.
}
\usage{
lcv(x, ...)
}
\description{
  The calling sequence for \code{lcv} matches those for the
  \code{\link{locfit}} or \code{\link{locfit.raw}} functions.
  The fit is not returned; instead, the returned object contains
  likelihood cross validation score for the fit.

  The LCV score is exact (up to numerical roundoff) if the
  \code{ev="cross"} argument is provided. Otherwise, the influence
  and cross validated residuals
  are computed using locfit's
  standard interpolation based approximations.
}
\arguments{
  \item{x}{model formula}
  \item{...}{other arguments to locfit}
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{lcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{lcvplot}
\alias{lcvplot}
\title{
Compute the likelihood cross-validation plot.
}
\usage{
lcvplot(..., alpha)
}
\description{
  The \code{lcvplot} function loops through calls to the \code{\link{lcv}}
function (and hence to \code{link{locfit}}), using a different
smoothing parameter for each call.
The returned structure contains the likelihood cross validation statistic
for each fit, and can be used to produce an LCV plot.
}
\arguments{
  \item{...}{ arguments to the \code{\link{lcv}}, \code{\link{locfit}}
     functions.}
  \item{alpha}{ Matrix of smoothing parameters. The \code{aicplot} function
     loops through calls to \code{\link{lcv}}, using each row of
     \code{alpha} as the smoothing parameter in turn. If \code{alpha}
     is provided as a vector, it will be converted to a one-column
     matrix, thus interpreting each component as a nearest neighbor
    smoothing parameter.}
}

\value{
  An object with class \code{"gcvplot"}, containing the smoothing
  parameters and LCV scores. The actual plot is produced using
  \code{\link{plot.gcvplot}}.
}

\examples{
data(ethanol)
plot(lcvplot(NOx~E,data=ethanol,alpha=seq(0.2,1.0,by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{lcv}},
  \code{\link{plot.gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{left}
\alias{left}
\title{
One-sided left smooth for a Locfit model.
}
\usage{
left(x)
%locfit(y~left(x), ev="data", deg=1)
}
\description{
  The \code{left()} function is used in a locfit model formula
  to specify a one-sided smooth: when fitting at a point \eqn{x},
  only data points with \eqn{x_i \le x} should be used.
  This can be useful in estimating points of discontinuity,
  and in cross-validation for forecasting a time series.

  When using this function, it will usually be necessary to specify an
  evaluation structure, since the fit is not smooth and locfit's
  interpolation methods are unreliable. Also, it is usually best
  to use \code{deg=0} or \code{deg=1}, otherwise the fits may be too
  variable. If nearest neighbor bandwidth specification is used,
  it does not recognize \code{left()}.

}

\arguments{
  \item{x}{numeric variable.}
}

\examples{
# compute left and right smooths
data(penny)
xev <- (1945:1988)+0.5
fitl <- locfit(thickness~left(year), alpha=c(0,10), deg=1, ev=xev, data=penny)
fitr <- locfit(thickness~right(year),alpha=c(0,10), deg=1, ev=xev, data=penny)
# plot the squared difference, to show the change points.
plot( xev, (predict(fitr,where="ev") - predict(fitl,where="ev"))^2 )
}

\seealso{
  \code{\link{lfbas}},
  \code{\link{locfit}},
  \code{\link{right}}
}

\keyword{models}

\eof
\name{lf}
\alias{lf}
\title{
Locfit term in Additive Model formula
}
\usage{
lf(..., alpha=0.7, deg=2, scale=1, kern="tcub", ev="tree", maxk=100)
}
\description{
  This function is used to specify a smooth term in a \code{gam()}
  model formula.

  At the time of writing, \code{gam} is not available in R.
}
\arguments{
  \item{...}{numeric predictor variable(s)}
  \item{alpha}{
  Smoothing parameter. A single number (e.g. \code{alpha=0.7})
  is interpreted as a nearest neighbor fraction. With two
  componentes (e.g. \code{alpha=c(0.7,1.2)}), the first component
  is a nearest neighbor fraction, and the second component is
  a fixed component. A third component is the penalty term in locally
  adaptive smoothing.}
  \item{deg}{degree of polynomials to be fitted locally.}
  \item{scale}{A scale to apply to each variable. This is especially important for
  multivariate fitting, where variables may be measured in
  non-comparable units. It is also used to specify the frequency
  for \code{\link{ang}} terms. If \code{scale=FALSE} (the default) no scaling
  is performed. If \code{scale=TRUE}, marginal standard deviations are used.
  Alternatively, a numeric vector can provide scales for the
  individual variables.}
  \item{kern}{Weight function, default = \code{"tcub"}.
  Other choices are \code{"rect"}, \code{"trwt"}, \code{"tria"},
  \code{"epan"}, \code{"bisq"} and \code{"gauss"}. Choices may be restricted
  when derivatives are required; e.g. for confidence bands and some
  bandwidth selectors.}
  \item{ev}{Evaluation Structure, default = \code{"tree"}. Also available are
  \code{"phull"}, \code{"data"}, \code{"grid"}, \code{"kdtree"},
  \code{"kdcenter"} and \code{"crossval"}. \code{ev="none"} gives no
  evaluation points, effectively producing the global parametric fit.
  A vector or matrix of evaluation points can also be provided.}
  \item{maxk}{Controls space assignment for evaluation structures.
  For the adaptive evaluation structures, it is impossible to be sure
  in advance how many vertices will be generated. If you get
  warnings about `Insufficient vertex space', Locfit's default assigment
  can be increased by increasing \code{maxk}. The default is \code{maxk=100}.}

}
\examples{
# fit an additive semiparametric model to the ethanol data.
if (exists("gam")) {  # shut up R's checks
fit <- gam(NOx~lf(E)+C,data=ethanol)
plot(fit)
}
}
\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gam.lf}},
  \code{gam}
}

\keyword{models}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{lfbas}
\alias{lfbas}
\title{
User-specified basis functions for Locfit.
}
%\usage{
%locfit(...,basis)
%}
\description{
  By default, Locfit uses a polynomial basis as the fitting functions.
  An alternative set of basis functions can be specified through
  the \code{basis} argument to \code{\link{locfit.raw}}.
  \code{lfbas} is a wrapper function used internally in Locfit's processing.

  To use the \code{basis} argument, you must write a function
  \code{f(x,t)} to evaluate the basis function at a fitting
  point \code{t} and data point(s) \code{x}. See below for an example.
  Note that the basis function will be called with multiple
  data points simultaneously, so should assume \code{x} is a matrix
  with \code{d} columns, where \code{d} is the number of independent
  variables. The fitting point \code{t} will always be a single point,
  in a vector of length \code{d}.

  The basis function should return a matrix, with each column being the
  evaluation of one fitting function.

  To ensure that the returned fit estimates the mean function, the
  first component of the basis should be 1; the remaining components
  should be 0 when \code{x=t}. To help ensure Locfit's interpolation
  routines are meaningful, the next \code{d} components should represent
  partial derivatives at \code{x=t}. Specifically, df(x,t)/dx[i],
  evaluated at \code{x=t}, should be the unit vector with 1 in the
  (i+1)st position.

  Violation of these rules can be useful, if functionals
  other than the mean are of interest. But this will require extreme
  care.

  Specifying a user basis results in extensive use of the
  \code{call_S} function, which may be slow. Speed may also
  be significantly affected by different ways of writing
  the basis.
}
\examples{
# Specify a bivariate linear with interaction basis.
data(ethanol)
my.basis <- function(x,t)
{
  u1 <- x[, 1] - t[1]
  u2 <- x[, 2] - t[2]
  cbind(1, u1, u2, u1 * u2)
}
fit <- locfit(NOx~E+C, data=ethanol, scale=0, basis=my.basis)
# With this basis, Locfit's standard interpolation and plot methods
# should be reasonable.
plot(fit,get.data=TRUE)

# Estimation of change points. This provides an alternative to using
# left() and right(), and can easily be modified to detecting
# a change in slopes or other parameters. Note that the first
# component is the indicator of x>t, so the coefficient estimates
# the size of the change, assuming the change occurs at t.
data(penny)
my.basis <- function(x,t) cbind(x>t,1,x-t)
xev <- (1945:1988) + 0.5
fit <- locfit(thickness~year, data=penny, alpha=c(0,10), ev=xev, basis=my.basis)
# The plot will show peaks where change points are most likely.
# in S4, S-Plus 5 etc,
# plot(preplot(fit,where="fitp")^2, type="b") is an alternative.
plot(xev, predict(fit,where="fitp")^2, type="b")

# Estimate the mean function using local linear regression, with
# discontinuities at 1958.5 and 1974.5.
# The basis functions must consist of the constant 1, the linear term
# x-t, and indicator functions for two of the three sections.
# Note the care taken to ensure my.basis(t,t) = c(1,0,0,0) for all t.
my.basis <- function(x,t)
{ ret <- NULL
  if (t<1958.5) ret <- cbind(1, x>=1958.5, x>1974.5, x-t)
  if (t>1974.5) ret <- cbind(1, x<=1974.5, x<1958.5, x-t)
  if (is.null(ret))
    ret <- cbind(1, x<1958.5, x>1974.5, x-t)
  ret
}
fit <- locfit(thickness~year, data=penny, alpha=c(0,10), ev=xev, basis=my.basis)
plot(preplot(fit,where="fitp", get.data=TRUE))
}
\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}}
}

\keyword{smooth}

\eof
\name{lfeval}
\alias{lfeval}
\title{
Extract Locfit Evaluation Structure.
}
\usage{
lfeval(object)
}
\description{
  Extracts the evaluation structure from a \code{"locfit"} object.
  This object has the class \code{"lfeval"}, and has its own set of
  methods for plotting e.t.c.
}
\arguments{
  \item{object}{\code{"locfit"} object}
}

\value{
  \code{"lfeval"} object.
}

\seealso{
  \code{\link{locfit}},
  \code{\link{plot.lfeval}},
  \code{\link{print.lfeval}}
}

\keyword{smooth}

\eof
\name{lflim}
\alias{lflim}
\title{
Construct Limit Vectors for Locfit fits.
}
\usage{
lflim(limits, nm, ret)
}
\description{
  This function is used internally to interpret \code{xlim} and \code{flim}
  arguments. It should not be called directly.
}
\arguments{
\item{limits}{
Limit argument.
}
\item{nm}{
Variable names.
}
\item{ret}{
Initial return vector.
}
}
\value{
Vector with length 2*dim.
}
\seealso{
\code{\link{locfit}}
}
\keyword{smooth}

\eof
\name{lfmarg}
\alias{lfmarg}
\title{
Generate grid margins.
}
\usage{
lfmarg(xlim, m = 40)
}
\arguments{
\item{xlim}{
Vector of limits for the grid. Should be of length 2*d;
the first d components represent the lower left corner,
and the next d components the upper right corner.
Can also be a \code{"locfit"} object.
}
\item{m}{
Number of points for each grid margin. Can be a vector of length d.
}
}
\value{
A list, whose components are the d grid margins.
}
\description{
This function is usually called by \code{\link{plot.locfit}}.
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}}
}
\keyword{smooth}

\eof
\name{lines.locfit}
\alias{lines.locfit}
\alias{llines.locfit}
\title{
Add `locfit' line to existing plot
}
\usage{
\method{lines}{locfit}(x, m, tr=x$trans, ...)
llines.locfit(x, m, tr=x$trans, ...)
}
\description{
Adds a Locfit line to an existing plot.
}
\arguments{
\item{x}{
\code{"locfit"} object. Should be a model with one predictor.
}
\item{m}{
Number of points to evaluate the line at. Default: 100.
}
\item{tr}{
Transformation function to use for plotting. Default is the inverse
link function, or the identity function if derivatives are requested.
}
\item{...}{
Other arguments to the default \code{lines} function.
}}
\note{\code{llines.locfit} is only used for Lattice/grid graphics.}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}}
}
\keyword{smooth}

\eof
\name{livmet}
\alias{livmet}
\title{liver Metastases dataset}
\usage{data(livmet)}
\format{
  Data frame with survival times (\code{t}), censoring indicator
  (\code{z}) and a number of covariates.
}
\description{
  Survival times for 622 patients diagnosed with Liver Metastases.

  Beware, the censoring variable
  is coded as 1 = uncensored, so use \code{cens=1-z} in
  \code{\link{locfit}()} calls.
}
\source{
  Haupt and Mansmann (1995)
}
\references{
  Haupt, G. and Mansmann, U. (1995)
  CART for Survival Data.
  Statlib Archive, \url{http://lib.stat.cmu.edu/S/survcart}.
}
\keyword{datasets}

\eof
\name{locfit}
\alias{locfit}
\title{
Local Regression, Likelihood and Density Estimation.
}
\usage{
locfit(formula, data, weights, cens, base, subset, geth, ..., lfproc)
}
\description{
  \code{locfit} is the model formula-based interface to the Locfit
  library for fitting local regression and likelihood models.

  \code{locfit} is implemented as a front-end to \code{\link{locfit.raw}}.
  See that function for options to control smoothing parameters,
  fitting family and other aspects of the fit.
}
\arguments{
\item{formula}{
Model Formula; e.g. \code{y~x} for a regression model; \code{~x} for a
density estimation model
}
\item{data}{
Data Frame.
}
\item{weights}{
Prior weights (or sample sizes) for individual observations. This is
typically used where observations have unequal variance.
}
\item{cens}{
Censoring indicator. \code{1} (or \code{TRUE}) denotes a censored observation.
\code{0} (or \code{FALSE}) denotes uncensored.
}
\item{base}{
Baseline for local fitting. For local regression models, specifying
a \code{base} is equivalent to using \code{y-base} as the reponse. But \code{base}
also works for local likelihood.
}
\item{subset}{
Subset observations in the data frame.
}
\item{geth}{
Don't use.
}
\item{...}{
Other arguments to \code{\link{locfit.raw}()} (or the \code{lfproc}).
}
\item{lfproc}{
A processing function to compute the local fit. Default is
\code{locfit.raw()}. Other choices include \code{locfit.robust()},
\code{locfit.censor()} and \code{locfit.quasi()}.
}}
\value{
An object with class \code{"locfit"}. A standard set of methods for printing,
ploting, etc. these objects is provided.
}
\seealso{
\code{\link{locfit.raw}}
}
\examples{
# fit and plot a univariate local regression
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
plot(fit,get.data=TRUE)

# a bivariate local regression with smaller smoothing parameter
fit <- locfit(NOx~E+C, data=ethanol, scale=0, alpha=0.5)
plot(fit)

# density estimation
data(geyser)
fit <- locfit(~geyser, alpha=c(0.1,0.8))
plot(fit,get.data=TRUE)
}
\references{
Loader, C. (1999). Local Regression and Likelihood. Springer, New York.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{locfit.censor}
\alias{locfit.censor}
\title{
Censored Local Regression
}
\usage{
locfit.censor(x, y, cens, ..., iter=3, km=FALSE)
%locfit(y~x, cens=cens, lfproc=locfit.censor)
%locfit.censor(y~x, cens, ..., iter=3, km=F)
}
\description{
  \code{locfit.censor} produces local regression estimates for censored
  data. The basic idea is to use an EM style algorithm, where one
  alternates between estimating the regression and the true values
  of censored observations.

  \code{locfit.censor} is designed as a front end
  to \code{\link{locfit.raw}} with data vectors, or as an intemediary
  between \code{\link{locfit}} and \code{\link{locfit.raw}} with a
  model formula. If you can stand the syntax, the second calling
  sequence above will be slightly more efficient than the third.
}
\arguments{
\item{x}{
  Either a \code{\link{locfit}} model formula or a numeric vector
  of the predictor variable.
}
\item{y}{
  If \code{x} is numeric, \code{y} gives the response variable.
}
\item{cens}{
  Logical variable indicating censoring. The coding is \code{1}
  or \code{TRUE} for censored; \code{0} or \code{FALSE} for uncensored.
}
\item{...}{
  Other arguments to \code{\link{locfit.raw}}
}
\item{iter}{Number of EM iterations to perform}
\item{km}{
  If \code{km=TRUE}, the estimation of censored observations uses
  the Kaplan-Meier estimate, leading to a local version of the
  Buckley-James estimate. If \code{km=F}, the estimation is based
  on a normal model (Schmee and Hahn). Beware of claims that B-J
  is nonparametric; it makes stronger assumptions on the upper tail
  of survival distributions than most authors care to admit.
}
}
\value{
\code{"locfit"} object.
}
\seealso{
\code{\link{km.mrl}},
\code{\link{locfit}},
\code{\link{locfit.raw}}
}
\examples{
data(heart)
fit <- locfit.censor(log10(surv+0.5)~age, cens=cens, data=heart)
plotbyfactor(heart$age, 0.5+heart$surv, heart$cens, ylim=c(0.5,16000), log="y")
lines(fit, tr=function(x)10^x)
}
\references{
Buckley, J. and James, I. (1979). Linear Regression with censored data.
  Biometrika 66, 429-436.

Loader, C. (1999). Local Regression and Likelihood. Springer, NY (Section 7.2).

Schmee, J. and Hahn, G. J. (1979). A simple method for linear regression
  analysis with censored data (with discussion). Technometrics 21, 417-434.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{locfit.matrix}
\alias{locfit.matrix}
\title{
Reconstruct a Locfit model matrix.
}
\usage{
locfit.matrix(fit, data)
}
\description{
  Reconstructs the model matrix, and associated variables such as
  the response, prior weights and censoring indicators, from a
  \code{locfit} object. This is used by functions such as
  \code{\link{fitted.locfit}}; it is not normally called directly.
  The function will only work properly if the data frame has not been
  changed since the fit was constructed.
}
\arguments{
\item{fit}{Locfit object}
\item{data}{ Data Frame.}
\item{...}{Other arguments to \code{\link{locfit.raw}()} (or the
  \code{lfproc}).}
}
\value{
A list with variables \code{x} (the model matrix); \code{y} (the response);
\code{w} (prior weights); \code{sc} (scales); \code{ce} (censoring indicator)
and \code{base} (baseline fit).
}
\seealso{
\code{\link{locfit}}, \code{\link{fitted.locfit}}, \code{\link{residuals.locfit}}
}
\keyword{models}

\eof
\name{locfit.quasi}
\alias{locfit.quasi}
\title{
Local Quasi-Likelihood with global reweighting.
}
\usage{
locfit.quasi(x, y, weights, ..., iter=3, var=function(mean)abs(mean))
%locfit(y~x, lfproc=locfit.quasi)
%locfit.quasi(y~x)
}
\description{
  \code{locfit.quasi} assumes a specified mean-variance relation,
  and performs iterartive reweighted local regression under this
  assumption. This is appropriate for local quasi-likelihood models,
  and is an alternative to specifying a family such as \code{"qpoisson"}.

  \code{locfit.quasi} is designed as a front end
  to \code{\link{locfit.raw}} with data vectors, or as an intemediary
  between \code{\link{locfit}} and \code{\link{locfit.raw}} with a
  model formula. If you can stand the syntax, the second calling
  sequence above will be slightly more efficient than the third.
}
\arguments{
\item{x}{
  Either a \code{\link{locfit}} model formula or a numeric vector
  of the predictor variable.
}
\item{y}{
  If \code{x} is numeric, \code{y} gives the response variable.
}
\item{weights}{Prior weights (or sample sizes) for individual observations. This is
typically used where observations have unequal variance.}
\item{...}{
  Other arguments to \code{\link{locfit.raw}}
}
\item{iter}{Number of EM iterations to perform}
\item{var}{
  Function specifying the assumed relation between the mean and variance.
}
}
\value{
\code{"locfit"} object.
}
\seealso{
\code{\link{locfit}},
\code{\link{locfit.raw}}
}
\keyword{smooth}

\eof
\name{locfit.raw}
\alias{locfit.raw}
\title{
Local Regression, Likelihood and Density Estimation.
}
\usage{
locfit.raw(x, y, weights=1, cens=NULL, base=0,
  scale=FALSE, alpha=0.7, deg=2, kern="tricube", kt="sph", acri="none",
  basis=list(NULL), deriv=numeric(0), dc=FALSE,
  family, link="default", xlim, renorm=FALSE,
  ev="tree", flim, mg=10, cut=0.8,
  maxk=100, itype="default", mint=20, maxit=20, debug=0,
  geth=FALSE, sty=rep(1,d))
}
\description{
  \code{locfit.raw} is an interface to Locfit using numeric vectors
  (for a model-formula based interface, use \code{\link{locfit}}).
  Although this function has a large number of arguments, most users
  are likely to need only a small subset.

  The first set of arguments (\code{x}, \code{y}, \code{weights},
  \code{cens}, and \code{base}) specify the regression
  variables and associated quantities.

  Another set (\code{scale}, \code{alpha}, \code{deg}, \code{kern},
    \code{kt}, \code{acri} and \code{basis}) control the amount of smoothing:
    bandwidth, smoothing weights and the local model.

  \code{deriv} and \code{dc} relate to derivative (or local slope)
    estimation.

  \code{family} and \code{link} specify the likelihood family.

  \code{xlim} and \code{renorm} may be used in density estimation.

  \code{ev}, \code{flim}, \code{mg} and \code{cut}
     control the set of evaluation points.

  \code{maxk},  \code{itype}, \code{mint}, \code{maxit} and \code{debug}
    control the Locfit algorithms, and will be rarely used.

  \code{geth} and \code{sty} are used by other functions calling
    \code{locfit.raw}, and should not be used directly.
}
\arguments{
\item{x}{
Vector (or matrix) of the independent variable(s).
}
\item{y}{
Response variable for regression models. For density families,
\code{y} can be omitted.
}
\item{weights}{
Prior weights for observations (reciprocal of variance, or sample size).
}
\item{cens}{
Censoring indicators for hazard rate or censored regression. The coding
is \code{1} (or \code{TRUE}) for a censored observation, and
\code{0} (or \code{FALSE}) for uncensored observations.
}
\item{base}{
Baseline parameter estimate. If provided, the local regression model is
fitted as \eqn{Y_i = b_i + m(x_i) + \epsilon_i}, with Locfit estimating
the \eqn{m(x)} term. For regression models, this effectively subtracts
\eqn{b_i} from \eqn{Y_i}. The advantage of the \code{base} formulation
is that it extends to likelihood regression models.
}

\item{scale}{
A scale to apply to each variable. This is especially important for
multivariate fitting, where variables may be measured in
non-comparable units. It is also used to specify the frequency
for \code{\link{ang}} terms. If \code{scale=F} (the default) no scaling
is performed. If \code{scale=T}, marginal standard deviations are used.
Alternatively, a numeric vector can provide scales for the
individual variables.
}
\item{alpha}{
Smoothing parameter. A single number (e.g. \code{alpha=0.7})
is interpreted as a nearest neighbor fraction. With two
componentes (e.g. \code{alpha=c(0.7,1.2)}), the first component
is a nearest neighbor fraction, and the second component is
a fixed component. A third component is the penalty term in locally
adaptive smoothing.
}
\item{deg}{
  Degree of local polynomial. Default: 2 (local quadratic). Degrees
  0 to 3 are supported by almost all parts of the Locfit code. Higher
  degrees may work in some cases.
}
\item{kern}{
  Weight function, default = \code{"tcub"}.
  Other choices are \code{"rect"}, \code{"trwt"}, \code{"tria"},
  \code{"epan"}, \code{"bisq"} and \code{"gauss"}. Choices may be restricted
  when derivatives are required; e.g. for confidence bands and some
  bandwidth selectors.
}
\item{kt}{
  Kernel type, \code{"sph"} (default); \code{"prod"}.
  In multivariate problems, \code{"prod"} uses a
  simplified product model which speeds up computations.
}
\item{acri}{Criterion for adaptive bandwidth selection.}
\item{basis}{User-specified basis functions. See
  \code{\link{lfbas}} for more details on this argument.}

\item{deriv}{
  Derivative estimation. If \code{deriv=1}, the returned fit will be
  estimating the derivative (or more correctly, an estimate of the
  local slope). If \code{deriv=c(1,1)} the second order derivative
  is estimated. \code{deriv=2} is for the partial derivative, with
  respect to the second variable, in multivariate settings.  }
\item{dc}{ Derivative adjustment.  }

\item{family}{
  Local likelihood family; \code{"gaussian"};
  \code{"binomial"}; \code{"poisson"}; \code{"gamma"} and \code{"geom"}.
  Density and rate estimation families are \code{"dens"}, \code{"rate"} and
  \code{"hazard"} (hazard rate). If the family is preceded by a \code{'q'}
  (for example, \code{family="qbinomial"}), quasi-likelihood variance
  estimates are used. Otherwise, the residual variance (\code{\link{rv}})
  is fixed at 1. The default family is \code{"qgauss"} if a response
  \code{y} is provided; \code{"density"} if no response is provided.
}
\item{link}{
  Link function for local likelihood fitting. Depending on the family,
  choices may be \code{"ident"}, \code{"log"}, \code{"logit"},
  \code{"inverse"}, \code{"sqrt"} and \code{"arcsin"}.
}

\item{xlim}{
  For density estimation, Locfit allows the density to be supported on
  a bounded interval (or rectangle, in more than one dimension).
  The format should be \code{c(ll,ul)} where \code{ll} is a vector of
  the lower bounds and \code{ur} the upper bounds. Bounds such as
  \eqn{[0,\infty)} are not supported, but can be effectively
  implemented by specifying a very large upper bound.
}
\item{renorm}{Local likelihood density estimates may not integrate
  exactly to 1. If \code{renorm=T}, the integral will be estimated
  numerically and the estimate rescaled. Presently this is implemented
  only in one dimension.
}

\item{ev}{
  Evaluation Structure, default = \code{"tree"}. Also available are
  \code{"phull"}, \code{"data"}, \code{"grid"}, \code{"kdtree"},
  \code{"kdcenter"} and \code{"crossval"}. \code{ev="none"} gives no
  evaluation points, effectively producing the global parametric fit.
  A vector or matrix of evaluation points can also be provided.
}
\item{flim}{
  A vector of lower and upper bounds for the evaluation structure,
  specified as \code{c(ll,ur)}. This should not be confused with
  \code{xlim}. It defaults to the data range.
}
\item{mg}{
  For the \code{"grid"} evaluation structure, \code{mg} specifies the
  number of points on each margin. Default 10. Can be either a single
  number or vector.
}
\item{cut}{
  Refinement parameter for adaptive partitions. Default 0.8; smaller
  values result in more refined partitions.
}

\item{maxk}{
  Controls space assignment for evaluation structures.
  For the adaptive evaluation structures, it is impossible to be sure
  in advance how many vertices will be generated. If you get
  warnings about `Insufficient vertex space', Locfit's default assigment
  can be increased by increasing \code{maxk}. The default is \code{maxk=100}.
}
\item{itype}{
  Integration type for density estimation. Available methods include
  \code{"prod"}, \code{"mult"} and \code{"mlin"}; and \code{"haz"} for
  hazard rate estimation problems. The available integration methods
  depend on model specification (e.g. dimension, degree of fit). By
  default, the best available method is used.
}
\item{mint}{
  Points for numerical integration rules. Default 20.
}
\item{maxit}{
  Maximum iterations for local likelihood estimation. Default 20.
}
\item{debug}{If > 0; prints out some debugging information.}

\item{geth}{ Don't use!  }
\item{sty}{Style for special terms (\code{\link{left}},
  \code{\link{ang}} e.t.c.). Do not try to set this directly;
  call \code{\link{locfit}} instead.
}
}
\value{
An object with class "locfit". A standard set of methods for printing, ploting, etc. these objects is provided.
}
\references{
Consult the Web page \url{http://cm.bell-labs.com/stat/project/locfit/}.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a4.

\eof
\name{locfit.robust}
\alias{locfit.robust}
\title{
Robust Local Regression
}
\usage{
locfit.robust(x, y, weights, ..., iter=3)
%locfit(y~x, lfproc=locfit.robust)
%locfit.robust(y~x, ..., iter=3)
}
\description{
  \code{locfit.robust} implements a robust local regression where
  outliers are iteratively identified and downweighted, similarly
  to the lowess method (Cleveland, 1979). The iterations and scale
  estimation are performed on a global basis.

  The scale estimate is 6 times the median absolute residual, while
  the robust downweighting uses the bisquare function. These are
  performed in the S code so easily changed.

  This can be interpreted as an extension of M estimation to local
  regression. An alternative extension (implemented in locfit via
  \code{family="qrgauss"}) performs the iteration and scale estimation
  on a local basis.
}
\arguments{
\item{x}{
  Either a \code{\link{locfit}} model formula or a numeric vector
  of the predictor variable.
}
\item{y}{
  If \code{x} is numeric, \code{y} gives the response variable.
}
\item{weights}{Prior weights (or sample sizes) for individual observations. This is
typically used where observations have unequal variance.}
\item{...}{
  Other arguments to \code{\link{locfit.raw}}
}
\item{iter}{Number of iterations to perform}
}
\value{
\code{"locfit"} object.
}
\seealso{
\code{\link{locfit}},
\code{\link{locfit.raw}}
}
\references{
Cleveland, W. S. (1979).
  Robust locally weighted regression and smoothing scatterplots.
  J. Amer. Statist. Assn. 74, 829-836.
}
\keyword{smooth}
\keyword{survival}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{lscv}
\alias{lscv}
\title{
Least Squares Cross Validation Statistic.
}
\usage{
lscv(x, ..., exact=FALSE)
}
\description{
  The calling sequence for \code{lscv} matches those for the
  \code{\link{locfit}} or \code{\link{locfit.raw}} functions.
  Note that this function is only designed for density estimation
  in one dimension. The returned object contains the
  least squares cross validation score for the fit.

  The computation of \eqn{\int \hat f(x)^2 dx} is performed numerically.
  For kernel density estimation, this is unlikely to agree exactly
  with other LSCV routines, which may perform the integration analytically.
}
\arguments{
  \item{x}{model formula (or numeric vector, if \code{exact=T})}
  \item{...}{other arguments to \code{\link{locfit}} or
    \code{\link{lscv.exact}} }
  \item{exact}{By default, the computation is approximate.
    If \code{exact=TRUE}, exact computation using
    \code{\link{lscv.exact}} is performed. This uses kernel density estimation
    with a constant bandwidth.}
}
\value{
  A vector consisting of the LSCV statistic and fitted degrees of freedom.
}
\examples{
# approximate calculation for a kernel density estimate
data(geyser)
lscv(~geyser, alpha=cbind(0,1), ev="grid", mg=100, deg=0,
  flim=c(1,6), kern="gauss")
# same computation, exact
lscv(geyser,alpha=1,exact=TRUE)
}
\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{lscv.exact}}
  \code{\link{lscvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{lscv.exact}
\alias{lscv.exact}
\title{
Exact LSCV Calculation
}
\usage{
lscv.exact(x, alpha, ...)
}
\description{
  This function performs the exact computation of the least squares
  cross validation statistic for one-dimensional kernel density estimation
  and a constant bandwidth.

  At the time of writing, it is implemented only for the Gaussian
  kernel (with the standard deviation of 0.4; Locfit's standard).
}
\arguments{
  \item{x}{Numeric data vector}
  \item{alpha}{The bandwidth. Note that this function only works with
    a constant bandwidth, and in accordance with Locfit's standard, this
    can be specified as \code{alpha=c(0,h)}. It can also be given as
    a single number, which will be interpreted as a constant
    (not NN) bandwidth. }
  \item{...}{not used.}
}
\value{
  A vector of the LSCV statistic and the fitted degrees of freedom.
}
\examples{
data(geyser)
lscv.exact(geyser,alpha=0.25)
# equivalent form using lscv
lscv(geyser,alpha=0.25,exact=TRUE)
}
\seealso{
  \code{\link{lscv}},
  \code{\link{lscvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{lscvplot}
\alias{lscvplot}
\title{
Compute the LSCV plot.
}
\usage{
lscvplot(..., alpha)
}
\description{
  The \code{lscvplot} function loops through calls to the \code{\link{lscv}}
function (and hence to \code{link{locfit}}), using a different
smoothing parameter for each call.
The returned structure contains the LSCV statistic for each density
estimate, and can be used to produce an LSCV plot.
}
\arguments{
  \item{...}{ arguments to the \code{\link{lscv}}, \code{\link{locfit}}
     functions.}
  \item{alpha}{ Matrix of smoothing parameters. The \code{lscvplot} function
     loops through calls to \code{\link{lscv}}, using each row of
     \code{alpha} as the smoothing parameter in turn. If \code{alpha}
     is provided as a vector, it will be converted to a one-column
     matrix, thus interpreting each component as a nearest neighbor
    smoothing parameter.}
}

\value{
  An object with class \code{"gcvplot"}, containing the smoothing
  parameters and LSCV scores. The actual plot is produced using
  \code{\link{plot.gcvplot}}.
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{lscv}},
  \code{\link{plot.gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{mcyc}
\alias{mcyc}
\title{Acc(De?)celeration of a Motorcycle Hitting a Wall}
\usage{data(mcyc)}
\format{
  Data frame with time and accel variables.
}
\description{
  Measurements of the acceleration of a motorcycle as it hits a wall.
  Actually, rumored to be a concatenation of several such datasets.
}
\source{
  H\"{a}rdle (1990).
}
\references{
  H\"{a}rdle, W. (1990). Applied Nonparametric Regression.
  Cambridge University Press.
}
\keyword{datasets}

\eof
\name{mine}
\alias{mine}
\title{Fracture Counts in Coal Mines}
\usage{data(mine)}
\format{
A dataframe with the response frac, and predictor variables
extrp, time, seamh and inb.
}
\description{
The number of fractures in the upper seam of coal mines, and four
predictor variables. This dataset can be modeled using Poisson regression.
}
\source{
Myers (1990).
}
\references{
Myers, R. H. (1990). Classical and Modern Regression with Applications
(Second edition). PWS-Kent Publishing, Boston.
}
\keyword{datasets}

\eof
\name{mmsamp}
\alias{mmsamp}
\title{Test dataset for minimax Local Regression}
\usage{data(cltest)}
\format{
Data Frame with x and y variables.
}
\description{
50 observations, as used in Figure 13.1 of Loader (1999).
}
\references{
  Loader, C. (1999). Local Regression and Likelihood. Springer, New York.
}
\keyword{datasets}

\eof
\name{morths}
\alias{morths}
\title{Henderson and Sheppard Mortality Dataset}
\usage{data(morths)}
\format{
  Data frame with age, n and number of deaths.
}
\description{
  Observed mortality for 55 to 99.
}
\source{
  Henderson and Sheppard (1919).
}
\references{
  Henderson, R. and Sheppard, H. N. (1919). Graduation of mortality and
  other tables. Actuarial Society of America, New York.
}
\keyword{datasets}

\eof
\name{panel.locfit}
\alias{panel.locfit}
\title{
Locfit panel function
}
\usage{
panel.locfit(x, y, subscripts, z, xyz.labs, xyz.axes, xyz.mid, xyz.minmax,
  xyz.range, col.regions, at, drape, contour, region, groups, ...)
%xyplot(formula, data, panel=panel.locfit, ...)
}
\description{
This panel function can be used to add \code{locfit} fits to plots generated
by trellis.

At the time of writing, it works with \code{xyplot} for 1-d fits
and crudely with \code{wireframe} for 2-d fits.
}
\arguments{
  \item{x, y}{x and y data.}
  \item{subscripts}{Subscripts of indices in the original data frame.}
  \item{z}{z data.}
  \item{xyz.labs, xyz.axes, xyz.mid, xyz.minmax, xyz.range, col.regions,
  at, drape, contour, region}{Arguments to be passed to 3D
  trellis panel functions.}
  \item{groups}{Currently not used.}
  \item{...}{Most Locfit arguments can be passed through \code{xyplot}.}
}
\examples{
library("lattice")

# a simple multi-panel display
data(ethanol)
xyplot(NOx~E|C,data=ethanol,panel=panel.locfit)

# The second example uses some Locfit optional arguments.
# Note we can pass the alpha (bandwidth) and family arguments directly to
# xyplot. The cens argument must be given in full; not as a data frame variable.
# The resulting plot does not (yet) distinguish the censored points, but
# the fit will correctly apply censoring.
data(border)
xyplot(runs~day,data=border,panel=panel.locfit,family="poisson",
 alpha=0.3,cens=border$no)
}
\seealso{
  \code{\link{locfit}}, \code{\link{plot.locfit.3d}}, \code{xyplot}.
}

\keyword{smooth}

\eof
\name{panel.xyplot.lf}
\alias{panel.xyplot.lf}
\title{
Locfit panel function
}
\usage{
panel.xyplot.lf(x, y, subscripts, clo, cup, wh, type, ...)
}
\description{
Panel function used by \code{\link{plot.locfit.3d}} for one dimensional
plots.
}
\arguments{
  \item{x}{x data}
  \item{y}{y data}
  \item{subscripts}{subscripts giving indices in the original data frame.}
  \item{clo}{lower limits of the confidence band.}
  \item{cup}{upper limits of the confidence band.}
  \item{wh}{\code{"f"} or \code{"d"}.}
  \item{type}{Plot type, to be passed to \code{panel.xyplot}.}
  \item{...}{Other arguments to be passed to \code{panel.xyplot}.}
}
  \seealso{
  \code{\link{plot.locfit.3d}}
}

\keyword{smooth}

\eof
\name{penny}
\alias{penny}
\title{Penny Thickness Dataset}
\usage{data(penny)}
\format{
  A dataframe.
}
\description{
  For each year, 1945 to 1989, the thickness of two U.S. pennies
  was recorded.
}
\source{
  Scott (1992).
}
\references{
  Scott (1992). Multivariate Density Estimation.
    Wiley, New York.
}
\keyword{datasets}

\eof
\name{plot.eval}
\alias{plot.eval}
\title{
Plot evaluation points from a 2-d locfit object.
}
\usage{
plot.eval(x, add=FALSE, text=FALSE, ...)
}
\description{
This function is used to plot the evaluation structure generated by
Locfit for a two dimensional fit. Vertices of the tree structure are
displayed as \code{O}; pseudo-vertices as \code{*}.
}
\arguments{
\item{x}{\code{"locfit"} object. }
\item{add}{If \code{TRUE}, add to existing plot.}
\item{text}{If \code{T}, numbers will be added indicating the order points
  were added.}
  \item{...}{other graphics parameters.}
}
\examples{
data(ethanol)
fit <- locfit(NOx~E+C,data=ethanol,scale=0)
plot.eval(fit)
}
\seealso{
\code{\link{locfit}}.
}
\keyword{smooth}

\eof
\name{plot.gcvplot}
\alias{plot.gcvplot}
\title{
Produce a cross-validation plot.
}
\usage{
plot.gcvplot(x, xlab = "Fitted DF", ylab = x$cri, ...)
}
\description{
Plots the value of the GCV (or other statistic) in a \code{gcvplot} object
against the degrees of freedom of the fit.
}
\arguments{
  \item{x}{ A "gcvplot" object, produced by \code{\link{gcvplot}},
     \code{\link{aicplot}} etc.}
  \item{xlab}{x axis label}
  \item{ylab}{y axis label}
  \item{...}{ Other arguments to \code{plot}.}
}

\examples{
data(ethanol)
plot(gcvplot(NOx~E,data=ethanol,alpha=seq(0.2,1.0,by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{gcv}},
  \code{\link{aicplot}},
  \code{\link{cpplot}},
  \code{\link{gcvplot}},
  \code{\link{lcvplot}}
}

\keyword{methods}

\eof
\name{plot.lfeval}
\alias{plot.lfeval}
\title{
Plot a Locfit Evaluation Structure.
}
\usage{
plot.lfeval(x, add=FALSE, txt=FALSE, ...)
}
\description{
  Plots the evaluation points from a \code{"locfit"} or \code{"lfeval"}
  structure, for one or two dimensional fits.
}
\arguments{
  \item{x}{A \code{"lfeval"} or \code{"locfit"} object}
  \item{add}{If \code{TRUE}, the points will be added to the existing plot.
    Otherwise, a new plot is created.}
  \item{txt}{If \code{TRUE}, the points are annotated with numbers in the
    order they were entered into the fit.}
  \item{...}{not used.}
}

\value{
  \code{"lfeval"} object.
}

\seealso{
  \code{\link{lfeval}},
  \code{\link{locfit}},
  \code{\link{print.lfeval}}
}

\keyword{smooth}

\eof
\name{plot.locfit.1d}
\alias{plot.locfit.1d}
\title{
Plot a one dimensional "preplot.locfit" object.
}
\usage{
plot.locfit.1d(x, pv, tv, main = "", xlab = "default", ylab = x$yname,
     type = "l", ylim, add = FALSE, lty = 1, col = 1, ...)
}
\arguments{
\item{x}{
One dimensional \code{"preplot.locfit"} object.
}
\item{pv}{Panel variables, to be varied within each panel of a plot.  May be
        specified as a character vector, or variable numbers.  There must be
        one or two panel variables; default is all variables in one or two
        dimensions; Variable 1 in three or more dimensions.
May by specified using either variable numbers or names.}
\item{tv}{Trellis variables, to be varied from panel to panel of the plot.}
\item{main}{Main title for the plot.}
\item{xlab}{label for the x-axis.}
\item{ylab}{label for the y-axis.}
\item{type}{type of plot (\code{"l"} for line, \code{"p"} for points,
            etc.)}
\item{ylim}{limit (min and max) for the y-axis.}
\item{add}{
If \code{TRUE}, the plot will be added to the existing plot.
}
\item{lty}{line type.}
\item{col}{color.}
\item{...}{
Additional arguments to the \code{plot} function.
}
}
\description{
This function is not usually called directly. It will be called automatically
when plotting one-dimensional \code{"locfit"} or \code{"preplot.locfit"}
objects.
}
\seealso{
  \code{\link{locfit}},
  \code{\link{plot.locfit}},
  \code{\link{preplot.locfit}}
}
\keyword{smooth}

\eof
\name{plot.locfit.2d}
\alias{plot.locfit.2d}
\title{
Plot a two-dimensional "preplot.locfit" object.
}
\usage{
plot.locfit.2d(x, pv, tv, main, type = "contour", xlab, ylab,
               zlab = x$yname, ...)
}
\arguments{
\item{x}{
Two dimensional \code{"preplot.locfit"} object.
}
\item{pv}{Panel variables, to be varied within each panel of a plot.
          May be specified as a character vector, or variable numbers.
          There must be one or two panel variables; default is all
          variables in one or two dimensions; Variable 1 in three
          or more dimensions.  May by specified using either variable
          numbers or names.}
\item{tv}{Trellis variables, to be varied from panel to panel of the
          plot.}
\item{main}{main title for the plot.}
\item{type}{
Type of plot. Can be \code{"contour"} (the default);
\code{"persp"} or \code{"image"}.
}
\item{xlab}{
Labels for the predictor variable axis; should be a character
vector of length 2.
}
\item{ylab}{Label for the y-axis.}
\item{zlab}{If \code{type="persp"}, the label for the z-axis.}
\item{...}{
Additional arguments to the \code{contour},
\code{persp} or \code{image} functions.
}
}

\description{
This function is not usually called directly. It will be called automatically
when plotting one-dimensional \code{"locfit"} or \code{"preplot.locfit"}
objects.
}

\seealso{
  \code{\link{locfit}},
  \code{\link{plot.locfit}},
  \code{\link{preplot.locfit}}
}
\keyword{smooth}

\eof
\name{plot.locfit.3d}
\alias{plot.locfit.3d}
\title{
Plot a high-dimensional "preplot.locfit" object using trellis displays.
}
\usage{
plot.locfit.3d(x, main="", pv, tv, type = "level", pred.lab = x$vnames,
               resp.lab = x$yname, crit = 1.96, ...)

}
\description{
This function plots cross-sections of a Locfit model (usually in three
or more dimensions) using trellis displays. It is not usually called
directly, but is invoked by \code{\link{plot.locfit}}.

The R libraries \code{lattice} and \code{grid} provide a partial
(at time of writing) implementation of trellis. Currently, this works
with one panel variable.
}
\arguments{
  \item{x}{\code{"preplot.locfit"} object.}
  \item{main}{main title for the plot.}
  \item{pv}{Panel variables. These are the variables (either one or two)
    that are varied within each panel of the display.}
  \item{tv}{Trellis variables. These are varied from panel to panel
    of the display.}
  \item{type}{Type of display. When there are two panel variables,
    the choices are \code{"contour"}, \code{"level"} and \code{"persp"}.}
  \item{pred.lab}{labels to use for predictors.}
  \item{resp.lab}{label to use for the response.}
  \item{crit}{critical value for plotting error band.}
  \item{...}{other graphical parameters.}
}

\seealso{
  \code{plot.locfit},
  \code{preplot.locfit}
}

\keyword{smooth}

\eof
\name{plot.locfit}
\alias{plot.locfit}
\title{
Plot an object of class locfit.
}
\usage{
\method{plot}{locfit}(x, xlim, pv, tv, m, mtv, band, tr,
        what, get.data, f3d, ...)
}
\arguments{
\item{x}{
locfit object.
}
\item{xlim}{
Plotting limits. Eg. \code{xlim=c(0,0,1,1)} plots over the unit square in
two dimensions. Default is bounding box of the data.
}
\item{pv}{
Panel variables, to be varied within each panel of a plot.  May be
        specified as a character vector, or variable numbers.  There must be
        one or two panel variables; default is all variables in one or two
        dimensions; Variable 1 in three or more dimensions.
May by specified using either variable numbers or names.
}
\item{tv}{
Trellis variables, to be varied from panel to panel of the plot.
}
\item{m}{
Controls the plot resolution (within panels, for trellis displays).
Default is 100 points in one dimension; 40 points (per dimension)
in two or more dimensions.
}
\item{mtv}{
Number of points for trellis variables; default 6.
}
\item{band}{
  Type of confidence bands to add to the plot. Default is \code{"none"}. Other
  choices include \code{"global"} for bands using a global variance estimate;
  \code{"local"} for bands using a local variance estimate and \code{"pred"}
  for prediction bands (at present, using a global variance estimate).
  To obtain the global variance estimate for a fit, use \code{\link{rv}}.
  This can be changed with \code{\link{rv<-}}. Confidence bands, by default,
  are 95%, based on normal approximations and neglecting bias.
  To change the critical value or confidence level, or to obtain
  simultaneous instead of pointwise confidence,
  the critical value stored on the fit must be changed. See the
  \code{\link{kappa0}} and \code{\link{crit}} functions.
}
%\item{deriv}{derivative estimation (see \code{\link{locfit.raw}}). }
%\item{se.fit}{If \code{TRUE}, standard errors are plotted along with the
%              fitted values.}
\item{tr}{
Transformation function to use for plotting. Default is the inverse
link function, or the identity function if derivatives are requested.
}
\item{what}{
What to plot. See \code{\link{predict.locfit}}.
}
\item{get.data}{
If \code{TRUE}, original data is added to the plot. Default: \code{FALSE}.
}
\item{f3d}{
Force the \code{locfit.3d} class on the prediction object, thereby generating
a trellis style plot. Default: \code{FALSE}, unless a \code{tv} argument is'
provided. Not available in R.
}
\item{...}{
Other arguments to \code{plot.locfit.1d}, \code{plot.locfit.2d} or
\code{plot.locfit.3d} as appropriate.
}}
\description{
The \code{plot.locfit} function generates grids of ploting points, followed
by a call to \code{preplot.locfit}. The returned object is then passed to
\code{plot.locfit.1d}, \code{plot.locfit.2d} or \code{plot.locfit.3d} as appropriate.
}
\examples{
x <- rnorm(100)
y <- dnorm(x)+rnorm(100)/5
plot(locfit(y ~ x), vband=TRUE)
x <- cbind(rnorm(100), rnorm(100))
plot(locfit(~x), type="persp")
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit.1d}},
\code{\link{plot.locfit.2d}}, \code{\link{plot.locfit.3d}},
\code{\link{lines.locfit}}, \code{\link{predict.locfit}},
\code{\link{preplot.locfit}}
}
\keyword{smooth}

\eof
\name{plot.preplot.locfit}
\alias{plot.preplot.locfit}
\title{
Plot a "preplot.locfit" object.
}
\usage{
\method{plot}{preplot.locfit}(x, ...)
}
\arguments{
\item{x}{
A \code{"preplot.locfit"} object, produced by
\code{\link{preplot.locfit}()}.
}
\item{...}{
Other arguments to \code{plot.locfit.1d}, \code{plot.locfit.2d} or
\code{plot.locfit.3d} as appropriate.
}}
\description{
The \code{\link{plot.locfit}()} function is implemented, roughly, as
a call to \code{\link{preplot.locfit}()}, followed by a call to
\code{plot.locfitpred()}. For most users, there will be little
need to call \code{plot.locfitpred()} directly.
}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}},
\code{\link{preplot.locfit}}, \code{\link{plot.locfit.1d}},
\code{\link{plot.locfit.2d}}, \code{\link{plot.locfit.3d}}.
}
\keyword{smooth}

\eof
\name{plot.scb}
\alias{plot.scb}
\title{
Plot method for simultaneous confidence bands
}
\usage{
plot.scb(x, add=FALSE, ...)
}
\description{
  Plot method for simultaneous confidence bands created by the
  \code{\link{scb}} function.
}
\arguments{
\item{x}{ \code{"scb"} object created by \code{\link{scb}()}.  }
\item{add}{If true, bands will be added to the existing plot.}
\item{...}{Other graphical parameters.}
}
\examples{
# corrected confidence bands for a linear logistic model
data(insect)
fit <- scb(deaths ~ lconc, type=4, w=nins, data=insect,
   deg=1, family="binomial", kern="parm")
plot(fit)
}
\seealso{
\code{\link{scb}}
}
\keyword{methods}

\eof
\name{plotbyfactor}
\alias{plotbyfactor}
\title{
x-y scatterplot, colored by levels of a factor.
}
\usage{
plotbyfactor(x, y, f, data, col, pch, add, lg,
             xlab = deparse(substitute(x)),
             ylab = deparse(substitute(y)), log = "", ...)
}
\description{
  Produces a scatter plot of x-y data, with different classes given
  by a factor f. The different classes are identified by different
  colours and/or symbols.
}
\arguments{
\item{x}{
Variable for x axis.
}
\item{y}{
Variable for y axis.
}
\item{f}{
Factor (or variable for which as.factor() works).
}
\item{data}{
data frame for variables x, y, f. Default: \code{sys.parent()}.
}
\item{col}{
Color numbers to use in plot. Will be replicated if shorter than the
number of levels of the factor f. Default: \code{1:10}.
}
\item{pch}{
Vector of plot characters. Replicated if necessary. Default: O.
}
\item{add}{
If TRUE, add to existing plot. Otherwise, create new plot.
Default: \code{FALSE}.
}
\item{lg}{
Coordinates to place a legend. Default: Missing (no legend).
}
\item{xlab}{label for x-axis.}
\item{ylab}{label for y-axis.}
\item{log}{which axis should be plotted in log scale (\code{"x"},
           \code{"y"}, or \code{"xy"}.)}
\item{...}{
Other graphical parameters, labels, titles e.t.c.}
}
\examples{
data(iris)
plotbyfactor(petal.wid,petal.len,species,data=iris)
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a3.

\eof
\name{points.locfit}
\alias{points.locfit}
\title{
Add `locfit' points to existing plot
}
\usage{
\method{points}{locfit}(x, tr, \dots)
}
\arguments{
\item{x}{
\code{locfit} object. Should be a model with one predictor.
}
\item{tr}{
Back transformation.
}
\item{...}{
Other arguments to the default \code{points} function.
}}
\description{This function shows the points at which the local fit
  was computed directly, rather than being interpolated. This can
  be useful if one is unsure of the validity of interpolation.}
\seealso{
\code{\link{locfit}}, \code{\link{plot.locfit}}
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{predict.locfit}
\alias{predict.locfit}
\title{
Prediction from a Locfit object.
}
\usage{
\method{predict}{locfit}(object, newdata, where="fitp", se.fit=FALSE,
               band="none", what="coef", ...)
}
\description{
  The \code{\link{locfit}} function computes a local fit at a selected set
of points (as defined by the \code{ev} argument). The \code{predict.locfit}
function is used to interpolate from these points to any other points.
The method is based on cubic hermite polynomial interpolation, using the
estimates and local slopes at each fit point.

  The motivation for this two-step procedure is computational speed.
Depending on the sample size, dimension and fitting procedure, the
local fitting method can be expensive, and it is desirable to keep the
number of points at which the direct fit is computed to a minimum.
The interpolation method used by \code{predict.locfit()} is usually
much faster, and can be computed at larger numbers of points.
}
\arguments{
\item{object}{
Fitted object from \code{\link{locfit}()}.
}
\item{newdata}{
Points to predict at. Can be given in several forms: vector/matrix;
list, data frame.
}
\item{where}{
An alternative to \texttt{newdata}. Choices include \code{"grid"}
for the grid \code{\link{lfmarg}(object)}; \code{"data"} for the original
data points and \code{"fitp"} for the direct fitting points (ie. no
interpolation).
}
\item{se.fit}{
If \code{TRUE}, standard errors are computed along with the fitted values.
}
\item{band}{
  Compute standard errors for the fit and include confidence bands on
  the returned object. Default is \code{"none"}. Other
  choices include \code{"global"} for bands using a global variance estimate;
  \code{"local"} for bands using a local variance estimate and \code{"pred"}
  for prediction bands (at present, using a global variance estimate).
  To obtain the global variance estimate for a fit, use \code{\link{rv}}.
  This can be changed with \code{\link{rv<-}}. Confidence bands, by default,
  are 95%, based on normal approximations and neglecting bias.
  To change the critical value or confidence level, or to obtain
  simultaneous instead of pointwise confidence,
  the critical value stored on the fit must be changed. See the
  \code{\link{kappa0}} and \code{\link{crit}} functions.
}
\item{what}{
What to compute predicted values of. The default, \code{what="coef"}, works
  with the fitted curve itself. Other choices include \code{"nlx"} for the
  length of the weight diagram; \code{"infl"} for the influence function;
  \code{"band"} for the bandwidth; \code{"degr"} for the local polynomial
  degree; \code{"lik"} for the maximized local likelihood; \code{"rdf"}
  for the local residual degrees of freedom and \code{"vari"} for the
  variance function. The interpolation algorithm for some of these quantities
  is questionable.
}
\item{...}{
Additional arguments to \code{\link{preplot.locfit}}.
}}
\value{
If \code{se.fit=F}, a numeric vector of predictors.
If \code{se.fit=T}, a list with components \code{fit}, \code{se.fit} and
\code{residual.scale}.
}
\examples{
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
predict(fit,c(0.6,0.8,1.0))
}
\keyword{smooth}

\eof
\name{preplot.locfit}
\alias{preplot.locfit}
\title{
Prediction from a Locfit object.
}
\usage{
\method{preplot}{locfit}(object, newdata, where, tr=NULL, what="coef",
  band="none", get.data=FALSE, f3d=FALSE, ...)
}
\arguments{
\item{object}{
Fitted object from \code{\link{locfit}()}.
}
\item{newdata}{
Points to predict at. Can be given in several forms: vector/matrix;
list, data frame.
}
\item{where}{
An alternative to \texttt{newdata}. Choices include \code{"grid"}
for the grid \code{\link{lfmarg}(object)}; \code{"data"} for the original
data points and \code{"fitp"} for the direct fitting points (ie. no
interpolation).
}
\item{tr}{
Transformation for likelihood models. Default is the inverse of
the link function.
}
\item{what}{
What to compute predicted values of. The default, \code{what="coef"}, works
  with the fitted curve itself. Other choices include \code{"nlx"} for the
  length of the weight diagram; \code{"infl"} for the influence function;
  \code{"band"} for the bandwidth; \code{"degr"} for the local polynomial
  degree; \code{"lik"} for the maximized local likelihood; \code{"rdf"}
  for the local residual degrees of freedom and \code{"vari"} for the
  variance function. The interpolation algorithm for some of these quantities
  is questionable.
}
\item{band}{
  Compute standard errors for the fit and include confidence bands on
  the returned object. Default is \code{"none"}. Other
  choices include \code{"global"} for bands using a global variance estimate;
  \code{"local"} for bands using a local variance estimate and \code{"pred"}
  for prediction bands (at present, using a global variance estimate).
  To obtain the global variance estimate for a fit, use \code{\link{rv}}.
  This can be changed with \code{\link{rv<-}}. Confidence bands, by default,
  are 95%, based on normal approximations and neglecting bias.
  To change the critical value or confidence level, or to obtain
  simultaneous instead of pointwise confidence,
  the critical value stored on the fit must be changed. See the
  \code{\link{kappa0}} and \code{\link{crit}} functions.
}
\item{get.data}{
If \code{TRUE}, the original data is attached to the returned object, and
added to the plot.
}
\item{f3d}{
If \code{TRUE}, sets a flag that forces ploting using the trellis style.
Not available in R.}
\item{...}{not used.}
}
\value{
An object with class \code{"preplot.locfit"}, containing the predicted
values and additional information used to construct the plot.
}
\description{
  \code{preplot.locfit} can be called directly, although it is more usual
  to call \code{\link{plot.locfit}} or \code{\link{predict.locfit}}.
  The advantage of \code{preplot.locfit} is in S-Plus 5, where arithmetic
  and transformations can be performed on the \code{"preplot.locfit"}
  object.

  \code{plot(preplot(fit))} is essentially synonymous with \code{plot(fit)}.
}
\seealso{
\code{\link{locfit}}, \code{\link{predict.locfit}}, \code{\link{plot.locfit}}.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{preplot.locfit.raw}
\alias{preplot.locfit.raw}
\title{
Prediction from a Locfit object.
}
\usage{
preplot.locfit.raw(object, newdata, where, what, band, ...)
}
\arguments{
\item{object}{
Fitted object from \code{\link{locfit}()}.
}
\item{newdata}{
New data points.
}
\item{where}{
Type of data provided in \code{newdata}.
}
\item{what}{
What to compute predicted values of.
}
\item{band}{
  Compute standard errors for the fit and include confidence bands on
  the returned object.
}
\item{...}{not used.}
}
\value{
A list containing raw output from the internal prediction routines.
}
\description{
  \code{preplot.locfit.raw} is an internal function used by
  \code{\link{predict.locfit}} and \code{\link{preplot.locfit}}.
  It should not normally be called directly.
}
\seealso{
\code{\link{locfit}}, \code{\link{predict.locfit}}, \code{\link{preplot.locfit}}.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{print.gcvplot}
\alias{print.gcvplot}
\title{
Print method for gcvplot objects
}
\usage{
print.gcvplot(x, ...)
}
\description{
  Print method for \code{"gcvplot"} objects. Actually, equivalent to
  \code{\link{plot.gcvplot}()}.
  \code{\link{scb}} function.
}
\arguments{
\item{x}{ \code{"gcvplot"} object.  }
\item{...}{other graphical parameters.}
}
\seealso{
\code{\link{gcvplot}},
\code{\link{plot.gcvplot}}
\code{\link{summary.gcvplot}}
}
\keyword{methods}

\eof
\name{print.lfeval}
\alias{print.lfeval}
\title{
Print the Locfit Evaluation Points.
}
\usage{
print.lfeval(x, ...)
}
\description{
  Prints a matrix of the evaluation points from a \code{"locfit"}
  or \code{"lfeval"} structure.
}
\arguments{
  \item{x}{A \code{"lfeval"} or \code{"locfit"} object}
  \item{...}{not used.}
}

\value{
  Matrix of the fit points.
}

\seealso{
  \code{\link{lfeval}},
  \code{\link{locfit}},
  \code{\link{plot.lfeval}}
}

\keyword{smooth}

\eof
\name{print.locfit}
\alias{print.locfit}
\title{
Print method for "locfit" object.
}
\usage{
\method{print}{locfit}(x, ...)
}
\description{
Prints a short summary of a \code{"locfit"} object.
}
\arguments{
\item{x}{
\code{"locfit"} object.
}
\item{...}{not used.}
}
\seealso{\code{\link{locfit}}}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{print.preplot.locfit}
\alias{print.preplot.locfit}
\title{
Print method for preplot.locfit objects.
}
\usage{
\method{print}{preplot.locfit}(x, ...)
}
\description{
  Print method for objects created by the
  \code{\link{preplot.locfit}} function.
}
\arguments{
\item{x}{ \code{"preplot.locfit"} object.  }
\item{...}{not used.}
}
\seealso{
\code{\link{preplot.locfit}},
\code{\link{predict.locfit}}
}
\keyword{methods}

\eof
\name{print.scb}
\alias{print.scb}
\title{
Print method for simultaneous confidence bands
}
\usage{
print.scb(x, ...)
}
\description{
  Print method for simultaneous confidence bands created by the
  \code{\link{scb}} function.
}
\arguments{
\item{x}{ \code{"scb"} object created by \code{\link{scb}()}.}
\item{...}{unused.}
}
\seealso{
\code{\link{scb}}
}
\keyword{methods}

\eof
\name{print.summary.locfit}
\alias{print.summary.locfit}
\title{
Print a Locfit summary object.
}
\usage{
\method{print}{summary.locfit}(x, ...)
}
\description{
  Print method for \code{"summary.locfit"} objects.
}
\arguments{
\item{x}{
Object from \code{\link{summary.locfit}()}.
}
\item{...}{not used.}
}
\seealso{
\code{\link{summary.locfit}()}
}
\keyword{methods}

\eof
\name{regband}
\alias{regband}
\title{
Bandwidth selectors for local regression.
}
\usage{
regband(formula, what = c("CP", "GCV", "GKK", "RSW"), deg=1, ...)
}
\description{
Function to compute local regression bandwidths for local linear regression,
implemented as a front end to \code{\link{locfit}()}.

  This function is included for comparative purposes only. Plug-in
selectors are based on flawed logic, make unreasonable and restrictive
assumptions
and do not use the full power of the estimates available in Locfit.
Any relation between the results produced by this function and
desirable estimates are entirely coincidental.
}
\arguments{
\item{formula}{Model Formula (one predictor).}
\item{what}{Methods to use.}
\item{deg}{Degree of fit.}
\item{...}{Other Locfit options.}
}
\value{
Vector of selected bandwidths.
}
\keyword{smooth}

\eof
\name{residuals.locfit}
\alias{residuals.locfit}
\title{
Fitted values for a `"locfit"' object.
}
\usage{
\method{residuals}{locfit}(object, data=NULL, type="deviance", ...)
}
\arguments{
\item{object}{
\code{"locfit"} object.
}
\item{data}{
The data frame for the original fit. Usually, shouldn't be needed.
}
\item{type}{
  Type of fit or residuals to compute. The default is \code{"fit"} for
  \code{fitted.locfit}, and \code{"dev"} for \code{\link{residuals.locfit}}.
  Other choices include \code{"pear"} for Pearson residuals; \code{"raw"}
  for raw residuals, \code{"ldot"} for likelihood derivative;
  \code{"d2"} for the deviance residual squared; \code{lddot} for the
  likelihood second derivative. Generally, \code{type} should only be
  used when \code{what="coef"}.
}
\item{...}{other arguments to \code{\link{fitted.locfit}}; e.g.,
           \code{cv} and \code{studentize}.}
}
\description{\code{residuals.locfit} is implemented as a front-end to
  \code{\link{fitted.locfit}}, with the \code{type} argument set.
}
\value{
A numeric vector of the residuals.
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{right}
\alias{right}
\title{
One-sided right smooth for a Locfit model.
}
\usage{
right(x)
%locfit(y~right(x), ev="data", deg=1)
}
\description{
  The \code{right()} function is used in a locfit model formula
  to specify a one-sided smooth: when fitting at a point \eqn{x},
  only data points with \eqn{x_i \le x} should be used.
  This can be useful in estimating points of discontinuity,
  and in cross-validation for forecasting a time series.

  When using this function, it will usually be necessary to specify an
  evaluation structure, since the fit is not smooth and locfit's
  interpolation methods are unreliable. Also, it is usually best
  to use \code{deg=0} or \code{deg=1}, otherwise the fits may be too
  variable. If nearest neighbor bandwidth specification is used,
  it does not recognize \code{right()}.

}

\arguments{
  \item{x}{numeric variable.}
}

\examples{
# compute left and right smooths
data(penny)
xev <- (1945:1988)+0.5
fitl <- locfit(thickness~left(year), alpha=c(0,10), deg=1, ev=xev, data=penny)
fitr <- locfit(thickness~right(year),alpha=c(0,10), deg=1, ev=xev, data=penny)
# plot the squared difference, to show the change points.
plot( xev, (predict(fitr,where="ev") - predict(fitl,where="ev"))^2 )
}

\seealso{
  \code{\link{lfbas}},
  \code{\link{locfit}},
  \code{\link{left}}
}

\keyword{models}

\eof
\name{rv}
\alias{rv}
\title{
Residual variance from a "locfit" object.
}
\usage{
rv(fit)
}
\description{
  As part of the \code{\link{locfit}} fitting procedure, an estimate
of the residual variance is computed; the \code{rv} function extracts
the variance from the \code{"locfit"} object.
The estimate used is the residual sum of squares
(or residual deviance, for quasi-likelihood models),
divided by the residual degrees of freedom.

For likelihood (not quasi-likelihood) models, the estimate is 1.0.
}
\arguments{
\item{fit}{
\code{"locfit"} object.
}}
\value{
  Returns the residual variance estimate from the \code{"locfit"} object.
}
\examples{
data(ethanol)
fit <- locfit(NOx~E,data=ethanol)
rv(fit)
}
\seealso{
\link{locfit}, \link{rv<-}
}
\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{rva}
\alias{rv<-}
\title{
Substitute variance estimate on a "locfit" object.
}
\description{
By default, Locfit uses the normalized residual sum of squares as the
variance estimate when constructing confidence intervals.
In some cases, the user may like to use alternative variance
estimates; this function allows the default value to be changed.
}
\usage{
rv(fit) <- 1.2345
}
\arguments{
\item{fit}{
\code{"locfit"} object.
}}
\seealso{
\link{locfit}(), \link{rv}(), \link{plot.locfit}()
}
\keyword{smooth}

\eof
\name{scb}
\alias{scb}
\title{
Simultaneous Confidence Bands
}
\usage{
scb(x, ..., type=1)
}
\description{
  \code{scb} is implemented as a front-end to \code{\link{locfit}},
  to compute simultaneous confidence bands using the tube formula
  method and extensions, based on Sun and Loader (1994).

  Some examples can also be found at
  \url{http://sun.cwru.edu/~jiayang/sci3.html}.
}
\arguments{
\item{x}{
A numeric vector or matrix of predictors (as in \code{\link{locfit.raw}}),
or a model formula (as in \code{\link{locfit}}).
}
\item{...}{Additional arguments to \code{\link{locfit.raw}}; e.g.,
  \code{mg}, \code{flim}, etc.}
\item{type}{
Type of confidence bands. \code{type=0} computes pointwise 95\% bands.
\code{type=1} computes basic simultaneous bands with no corrections.
\code{type=2,3,4} are the centered and corrected bands for parametric
regression models listed in Table 3 of Sun, Loader and McCormick (2000).
}
}
\value{
A list containing the evaluation points, fit, standard deviations and upper
and lower confidence bounds. The class is \code{"scb"}; methods for
printing and ploting are provided.
}
\seealso{
\code{\link{locfit}}, \code{\link{print.scb}}, \code{\link{plot.scb}}.
}
\examples{
# corrected confidence bands for a linear logistic model
data(insect)
fit <- scb(deaths ~ lconc, type=4, w=nins, data=insect,
   deg=1, family="binomial", kern="parm")
plot(fit)
}
\references{
Sun J. and Loader, C. (1994).
 Simultaneous confidence bands in linear regression and smoothing.
 \emph{The Annals of Statistics} 22, 1328-1345.

Sun, J., Loader, C. and McCormick, W. (2000).
  Confidence bands in generalized linear models.
 \emph{The Annals of Statistics} 28, 429-460.
}
\keyword{htest}
\keyword{smooth}

\eof
\name{sjpi}
\alias{sjpi}
\title{
Sheather-Jones Plug-in bandwidth criterion.
}
\usage{
sjpi(x, a)
}
\description{
  Given a dataset and set of pilot bandwidths, this function
  computes a bandwidth via the plug-in method, and the assumed
  `pilot' relationship of Sheather and Jones (1991).
  The S-J method chooses the bandwidth at which the two intersect.

  The purpose of this function is to demonstrate the sensitivity of
  plug-in methods to pilot bandwidths and assumptions. This function
  does not provide a reliable method of bandwidth selection.
}
\arguments{
  \item{x}{data vector}
  \item{a}{vector of pilot bandwidths}
}
\value{
  A matrix with four columns; the number of rows equals the length of \code{a}.
  The first column is the plug-in selected bandwidth. The second column
  is the pilot bandwidths \code{a}. The third column is the pilot bandwidth
  according to the assumed relationship of Sheather and Jones. The fourth
  column is an intermediate calculation.
}
\examples{
# Fig 10.2 (S-J parts) from Loader (1999).
data(geyser)
gf <- 2.5
a <- seq(0.05, 0.7, length=100)
z <- sjpi(geyser, a)

# the plug-in curve. Multiplying by gf=2.5 corresponds to Locfit's standard
# scaling for the Gaussian kernel.
plot(gf*z[, 2], gf*z[, 1], type = "l", xlab = "Pilot Bandwidth k", ylab
     = "Bandwidth h")

# Add the assumed curve.
lines(gf * z[, 3], gf * z[, 1], lty = 2)
legend(gf*0.05, gf*0.4, lty = 1:2, legend = c("Plug-in", "SJ assumed"))
}
\references{
  Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
  selection method for kernel density estimation. JRSS-B 53, 683-690.
}
\seealso{
  \code{\link{locfit}},
  \code{\link{locfit.raw}},
  \code{\link{lcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{spence.15}
\alias{spence.15}
\title{
Spencer's 15 point graduation rule.
}
\usage{
spence.15(y)
}
\description{
  Spencer's 15 point rule is a weighted moving average operation
  for a sequence of observations equally spaced in time. The
  average at time t depends on the observations at times t-7,...,t+7.

  Except for boundary effects, the function will reproduce
  polynomials up to degree 3.
}
\arguments{
  \item{y}{Data vector of observations at equally spaced points.}
}
\value{
  A vector with the same length as the input vector, representing
  the graduated (smoothed) values.
}
\examples{
  data(spencer)
  yy <- spence.15(spencer$mortality)
  plot(spencer$age, spencer$mortality)
  lines(spencer$age, yy)
}

\seealso{
  \code{\link{spence.21}},
  \code{\link{spencer}},
}

\references{
  Spencer, J. (1904).
  On the graduation of rates of sickness and mortality.
  Journal of the Institute of Actuaries 38, 334-343.
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{spence.21}
\alias{spence.21}
\title{
Spencer's 21 point graduation rule.
}
\usage{
spence.21(y)
}
\description{
  Spencer's 21 point rule is a weighted moving average operation
  for a sequence of observations equally spaced in time. The
  average at time t depends on the observations at times t-11,...,t+11.

  Except for boundary effects, the function will reproduce
  polynomials up to degree 3.
}
\arguments{
  \item{y}{Data vector of observations at equally spaced points.}
}
\value{
  A vector with the same length as the input vector, representing
  the graduated (smoothed) values.
}
\examples{
  data(spencer)
  yy <- spence.21(spencer$mortality)
  plot(spencer$age, spencer$mortality)
  lines(spencer$age, yy)
}

\seealso{
  \code{\link{spence.15}},
  \code{\link{spencer}},
}

\references{
  Spencer, J. (1904).
  On the graduation of rates of sickness and mortality.
  Journal of the Institute of Actuaries 38, 334-343.
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{spencer}
\alias{spencer}
\title{Spencer's Mortality Dataset}
\usage{data(spencer)}
\alias{spencer}
\format{
  Data frame with age and mortality variables.
}
\description{
  Observed mortality rates for ages 20 to 45.
}
\source{
  Spencer (1904).
}
\references{
  Spencer, J. (1904).
  On the graduation of rates of sickness and mortality.
  Journal of the Institute of Actuaries 38, 334-343.
}
\keyword{datasets}

\eof
\name{stamp}
\alias{stamp}
\title{Stamp Thickness Dataset}
\usage{data(stamp)}
\format{
  Data frame with \code{thick} (stamp thickness) and \code{count}
  (number of stamps) variables.
}
\description{
  Thicknesses of 482 postage stamps of the 1872 Hidalgo issue of Mexico.
}
\source{
  Izenman and Sommer (1988).
}
\references{
  Izenman, A. J. and Sommer, C. J. (1988).
  Philatelic mixtures and multimodal densities.
  Journal of the American Statistical Association 73, 602-606.
}
\keyword{datasets}

\eof
\name{store}
\alias{store}
\title{
Save S functions.
}
\usage{
store(data=FALSE, grand=FALSE)
}
\description{
  I've gotta keep track of this mess somehow!
}
\arguments{
  \item{data}{should the data sets be dumped to file?}
  \item{grand}{should the whole shebang be dumped to file?}
}
\keyword{internal}

\eof
\name{summary.gcvplot}
\alias{summary.gcvplot}
\title{
Summary method for a gcvplot structure.
}
\usage{
\method{summary}{gcvplot}(object, ...)
}
\description{
  Computes a short summary for a generalized cross-validation plot structure
}
\arguments{
  \item{object}{A \code{gcvplot} structure produced by a call to
    \code{\link{gcvplot}}, \code{\link{cpplot}} etc.
}
  \item{...}{currently ignored.}
}

\value{
  A matrix with two columns; one row for each fit computed in the
  \code{\link{gcvplot}} call.
  The first column is the fitted degrees
  of freedom; the second is the GCV or other criterion computed.
}

\examples{
data(ethanol)
summary(gcvplot(NOx ~ E, data=ethanol, alpha=seq(0.2, 1.0, by=0.05)))
}

\seealso{
  \code{\link{locfit}},
  \code{\link{gcv}},
  \code{\link{gcvplot}}
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{summary.locfit}
\alias{summary.locfit}
\title{
Print method for a locfit object.
}
\usage{
\method{summary}{locfit}(object, ...)
}
\description{
Prints a short summary of a \code{"locfit"} object.
}
\arguments{
\item{object}{
\code{"locfit"} object.
}
\item{...}{currently not used.}
}

\value{
A \code{"summary.locfit"} object, containg a short summary of the
\code{"locfit"} object.
}

\keyword{smooth}
% Converted by Sd2Rd version 0.2-a5.

\eof
\name{summary.preplot.locfit}
\alias{summary.preplot.locfit}
\title{
Summary method for a preplot.locfit object.
}
\usage{
\method{summary}{preplot.locfit}(object, ...)
}
\description{
Prints a short summary of a \code{"preplot.locfit"} object.
}
\arguments{
\item{object}{
\code{"preplot.locfit"} object.
}
\item{...}{currently not used.}
}

\value{
The fitted values from a
\code{"preplot.locfit"} object.
}

\keyword{smooth}

\eof
\name{trimod}
\alias{trimod}
\title{Generated sample from a bivariate trimodal normal mixture}
\usage{data(trimod)}
\format{
  Data frame with 225 observations and variables x0, x1.
}
\description{
  This is a random sample from a mixture of three bivariate standard
  normal components;
  the sample was used for the examples in Loader (1996).
}
\source{
  Randomly generated in S.
}
\references{
  Loader, C. R. (1996).
  Local Likelihood Density Estimation.
  Annals of Statistics 24, 1602-1618.
}
\keyword{datasets}

\eof
