\name{draw.clust}
\alias{draw.clust}
\title{Graph a Hierarchical Cluster Tree}
\description{
  Graph a hierarchical cluster tree of class \code{twins}
  or \code{hclust} using shaded square symbols at observations.
}
\usage{
  draw.clust (cluster, cex=par("cex"), pch=par("pch"), size=2.5*cex, 
      col=NULL, nodeinfo=FALSE, cases="obs", new=TRUE)
}
\arguments{
  \item{cluster}{object of class \code{hclust} or \code{twins}.}
  \item{cex}{size of text, par parameter.}
  \item{pch}{shape of symbol at leaves, par parameter.}
  \item{size}{size in cex units of symbol at leaves.}
  \item{col}{vector of colors from \code{\link{hsv}}, \code{\link{rgb}},
             etc, or if \code{NULL}, then use \code{\link{rainbow}}, or
             if \code{gray} or \code{grey}, then use \code{\link{gray}}.}
  \item{nodeinfo}{if \code{TRUE}, add a line at each node with number
                  of observations included in each leaf.}
  \item{cases}{label for type of observations.}
  \item{new}{if \code{TRUE}, call \code{\link{plot.new}}.}
}
\details{
  An alternative to \code{\link{pltree}} and \code{\link{plot.hclust}}.
}
\value{The vector of colors supplied or generated.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[cluster]{agnes}}, 
  \code{\link[cluster]{diana}}, 
  \code{\link[mva]{hclust}}, 
  \code{\link{draw.tree}}, 
  \code{\link{map.groups}}
}
\examples{
  library (cluster)
  data (oregon.bird.dist)

  draw.clust (prune.clust (agnes (oregon.bird.dist), k=6))
}
\keyword{hplot}
\keyword{cluster}

\eof
\name{draw.tree}
\alias{draw.tree}
\title{Graph a Classification or Regression Tree}
\description{
  Graph a classification or regression tree with a hierarchical
  tree diagram, optionally including shaded symbols at leaves 
  and additional info at intermediate nodes.
}
\usage{
  draw.tree (tree, cex=par("cex"), pch=par("pch"), size=2.5*cex, 
      col=NULL, nodeinfo=FALSE, units="", cases="obs", 
      digits=getOption("digits"), print.levels=TRUE, 
      new=TRUE)
}
\arguments{
  \item{tree}{object of class \code{rpart} or \code{tree}.}
  \item{cex}{size of text, par parameter.}
  \item{pch}{shape of symbol at leaves, par parameter.}
  \item{size}{if \code{size=0}, draw terminal symbol at leaves else a
              symbol of size in cex units.}
  \item{col}{vector of colors from \code{\link{hsv}}, \code{\link{rgb}},
             etc, or if \code{NULL}, then use \code{\link{rainbow}}, or
             if \code{gray} or \code{grey}, then use \code{\link{gray}}.}
  \item{nodeinfo}{if \code{TRUE}, add a line at each node with mean value
                  of response, number of observations, and percent
                  deviance explained (or classified correct).}
  \item{units}{label for units of mean value of response, if
               regression tree.}
  \item{cases}{label for type of observations.}
  \item{digits}{number of digits to round mean value of response, if
                regression tree.}
  \item{print.levels}{if \code{TRUE}, print levels of factors at splits,
                      otherwise only the factor name.}
  \item{new}{if \code{TRUE}, call \code{\link{plot.new}}.}
}
\details{
  As in \code{plot.rpart(,uniform=TRUE)}, each level has constant depth.  
  Specifying \code{nodeinfo=TRUE}, shows the deviance explained or the 
  classification rate at each node.

  A split is shown, for numerical variables, as 
  \code{variable <> value} when the cases with lower values go left, or as 
  \code{variable >< value} when the cases with lower values go right.
  When the splitting variable is a factor, and print.levels=TRUE, 
  the split is shown as \code{levels = factor = levels} with the cases 
  on the left having factor levels equal to those on the left of the 
  factor name, and correspondingly for the right.
}
\value{The vector of colors supplied or generated.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[rpart]{rpart}}, 
  \code{\link[tree]{tree}}, 
  \code{\link{draw.clust}}, 
  \code{\link{map.groups}}
}
\examples{
  library (rpart)
  data (oregon.env.vars)

  draw.tree (prune.Rpart (rpart (oregon.env.vars), best=7), 
      nodeinfo=TRUE, units="species", cases="cells", digits=0)
}
\keyword{hplot}
\keyword{tree}

\eof
\name{group.clust}
\alias{group.clust}
\title{Observation Groups for a Hierarchical Cluster Tree}
\description{
  Alternative to \code{\link{cutree}} that orders pruned groups from
  left to right in draw order.
}
\usage{
  group.clust (cluster, k=NULL, h=NULL)
}
\arguments{
  \item{cluster}{object of class \code{hclust} or \code{twins}.}
  \item{k}{desired number of groups.}
  \item{h}{height at which to prune for grouping.}

  At least one of \code{k} or \code{h} must be specified; \code{k} takes
  precedence if both are given.
}
\details{
  Normally used with \code{\link{map.groups}}.  See example.
}
\value{Vector of pruned cluster membership} 
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[mva]{hclust}}, 
  \code{\link[cluster]{twins.object}}, 
  \code{\link{cutree}}, 
  \code{\link{map.groups}}
}
\examples{
  library (mva)
  data (oregon.bird.dist, oregon.grid)

  group <- group.clust (hclust (dist (oregon.bird.dist)), k=6)
  names(group) <- row.names(oregon.bird.dist)
  map.groups (oregon.grid, group)
}
\keyword{manip}
\keyword{cluster}

\eof
\name{group.tree}
\alias{group.tree}
\title{Observation Groups for Classification or Regression Tree}
\description{
  Alternative to \code{tree[["where"]]} that orders groups from left
  to right in draw order.
}
\usage{
  group.tree (tree)
}
\arguments{
  \item{tree}{object of class \code{rpart} or \code{tree}.}
}
\details{
  Normally used with \code{\link{map.groups}}.  See example.
}
\value{Vector of rearranged \code{tree[["where"]]}} 
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[rpart]{rpart}}, 
  \code{\link[tree]{tree}}, 
  \code{\link{map.groups}}
}
\examples{
  library (rpart)
  data (oregon.env.vars, oregon.grid)

  group <- group.tree (prune.Rpart (rpart (oregon.env.vars), best=7))
  names(group) <- row.names(oregon.env.vars)
  map.groups (oregon.grid, group=group)
}
\keyword{manip}
\keyword{tree}

\eof
\name{kgs}
\alias{kgs}
\title{KGS Measure for Pruning Hierarchical Clusters}
\description{
  Computes the Kelley-Gardner-Sutcliffe penalty function for a 
  hierarchical cluster tree.
}
\usage{
  kgs (cluster, diss, alpha=1, maxclust=NULL)
}
\arguments{
  \item{cluster}{object of class \code{hclust} or \code{twins}.}
  \item{diss}{object of class \code{dissimilarity} or \code{dist}.}
  \item{alpha}{weight for number of clusters.}
  \item{maxclust}{maximum number of clusters for which to compute
                  measure.}
}
\details{
  Kelley et al. (see reference) proposed a method that can help decide 
  where to prune a hierarchical cluster tree.  At any level of the 
  tree the mean across all clusters of the mean within clusters of the 
  dissimilarity measure is calculated.  After normalizing, the number 
  of clusters times alpha is added.  The minimum of this function  
  corresponds to the suggested pruning size.

  The current implementation has complexity O(n*n*maxclust), thus 
  very slow with large n.  For improvements, at least it should only 
  calculate the spread for clusters that are split at each level, 
  rather than over again for all.

  Uses contributed function \code{\link[combinat]{combn}}.
}
\value{Vector of the penalty function for trees of size 2:maxclust. 
  The names of vector elements are the respective numbers of clusters.}
\references{
  Kelley, L.A., Gardner, S.P., Sutcliffe, M.J.  (1996)  An automated
  approach for clustering an ensemble of NMR-derived protein structures 
  into conformationally-related subfamilies, \emph{Protein Engineering}, 
  \bold{9}, 1063-1065.
}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[cluster]{twins.object}}, 
  \code{\link[cluster]{dissimilarity.object}}, 
  \code{\link[mva]{hclust}}, 
  \code{\link[mva]{dist}},
  \code{\link{prune.clust}},
  \code{\link[combinat]{combn}}
}
\examples{
  library (cluster)
  library (combinat)
  data (votes.repub)

  a <- agnes (votes.repub, method="ward")
  b <- kgs (a, a$diss, maxclust=20)
  plot (names (b), b, xlab="# clusters", ylab="penalty")
}
\keyword{manip}
\keyword{cluster}

\eof
\name{map.groups}
\alias{map.groups}
\title{Map Groups of Observations}
\description{
  Draws maps of groups of observations created by clustering,
  classification or regression trees, or some other type of
  classification.
}
\usage{
  map.groups (pts, group, pch=par("pch"), size=2, col=NULL, 
      border=NULL, new=TRUE)
}
\arguments{
  \item{pts}{matrix or data frame with components \code{"x"},
             and \code{"y"} for each observation (see details).}
  \item{group}{vector of integer class numbers corresponding to
               \code{pts} (see details), and indexing colors
               in \code{col}.}
  \item{pch}{symbol number from \code{par("pch")} if < 100, otherwise
             parameter \code{n} for ngon.}
  \item{size}{size in cex units of point symbol.}
  \item{col}{vector of fill colors from \code{\link{hsv}}, \code{\link{rgb}},
             etc, or if \code{NULL}, then use \code{\link{rainbow}}, or
             if \code{gray} or \code{grey}, then use \code{\link{gray}}.}
  \item{border}{vector of border colors from \code{\link{hsv}},
             \code{\link{rgb}},
             etc, or if \code{NULL}, then use \code{\link{rainbow}}, or
             if \code{gray} or \code{grey}, then use \code{\link{gray}}.}
  \item{new}{if \code{TRUE}, call \code{\link{plot.new}}.}
}
\details{
  If the number of rows of \code{pts} is not equal to the length
  of \code{group}, then (1) \code{pts} are assumed to represent
  polygons and \code{\link{polygon}} is used, (2) the identifiers in 
  \code{group} are matched to the polygons in \code{pts} through 
  \code{names(group)} and \code{pts$x[is.na(pts$y)]}, and (3) these
  identifiers are mapped to dense integers to reference colours.
  Otherwise, \code{group} is assumed to parallel \code{pts}, and,
  if \code{pch < 100}, then \code{\link{points}} is used, otherwise
  \code{\link{ngon}}, to draw shaded polygon symbols for each
  observation in pts.
}
\value{The vector of fill colors supplied or generated.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{ngon}}, 
  \code{\link{polygon}}, 
  \code{\link{group.clust}}, 
  \code{\link{group.tree}}, 
  \code{\link{map.key}}
}
\examples{
  data (oregon.bird.names, oregon.env.vars, oregon.bird.dist)
  data (oregon.border, oregon.grid)

  # range map for American Avocet
  spp <- match ("American avocet", oregon.bird.names[["common.name"]])
  group <- oregon.bird.dist[,spp] + 1
  names(group) <- row.names(oregon.bird.dist)
  map.groups (oregon.grid, group=group, col="gray")
  lines (oregon.border)

  # distribution of January temperatures
  cuts <- quantile (oregon.env.vars[["jan.temp"]], probs=seq(0,1,1/5))
  group <- cut (oregon.env.vars[["jan.temp"]], cuts, labels=FALSE, 
    include.lowest=TRUE)
  names(group) <- row.names(oregon.env.vars)
  map.groups (oregon.grid, group=group, col="gray")
  lines (oregon.border)

  # January temperatures using point symbols rather than polygons
  map.groups (oregon.env.vars, group, pch=19, col="gray")
  lines (oregon.border)
}
\keyword{hplot}
\keyword{cluster}
\keyword{tree}

\eof
\name{map.key}
\alias{map.key}
\title{Draw Key to accompany Map of Groups}
\description{
  Draws legends for maps of groups of observations.
}
\usage{
  map.key (x, y, labels=NULL, cex=par("cex"), pch=par("pch"),
      size=2.5*cex, col=NULL, head="", sep=0.25*cex, new=FALSE)
}
\arguments{
  \item{x, y}{coordinates of lower left position of key
              in proportional units (0-1) of plot.}
  \item{labels}{vector of labels for classes, or if \code{NULL},
                then integers \code{1:length(col)}, or \code{1}.}
  \item{size}{size in cex units of shaded key symbol.}
  \item{pch}{symbol number from \code{par("pch")} if < 100, otherwise
             parameter \code{n} for \code{\link{ngon}}.}
  \item{cex}{pointsize of text, \code{par} parameter.}
  \item{head}{text heading for key.}
  \item{sep}{separation in cex units between adjacent symbols in key.
             If \code{sep=0}, assume a continuous scale, use square
             symbols, and put labels at breaks between squares.}
  \item{col}{vector of colors from \code{\link{hsv}}, \code{\link{rgb}},
             etc, or if \code{NULL}, then use \code{\link{rainbow}}, or
             if \code{gray} or \code{grey}, then use \code{\link{gray}}.}
  \item{new}{if \code{TRUE}, call \code{\link{plot}}.}
}
\details{
  Uses \code{\link{points}} or \code{\link{ngon}}, depending on value of
  \code{pch}, to draw shaded polygon symbols for key.
}
\value{The vector of colors supplied or generated.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{ngon}}, 
  \code{\link{map.groups}}
}
\examples{
  data (oregon.env.vars)

  # key for examples in help(map.groups)
  # range map for American Avocet
  map.key (0.2, 0.2, labels=c("absent","present"), pch=106, 
    col="gray", head="key", new=TRUE)
  # distribution of January temperatures
  cuts <- quantile (oregon.env.vars[["jan.temp"]], probs=seq(0,1,1/5))
  map.key (0.2, 0.2, labels=as.character(round(cuts,0)), 
    col="gray", sep=0, head="key", new=TRUE)

  # key for example in \code{help(group.tree)}
  map.key (0.2, 0.2, labels=as.character(seq(6)), 
    pch=19, head="node", new=TRUE)
}
\keyword{hplot}
\keyword{aplot}

\eof
\name{ngon}
\alias{ngon}
\title{Outline or Fill a Regular Polygon}
\description{
  Draws a regular polygon at specified coordinates as an
  outline or shaded.
}
\usage{
  ngon (xydc, n=4, angle=0, type=1)
}
\arguments{
  \item{xydc}{four element vector with \code{x} and \code{y}
              coordinates of center, \code{d} diameter in mm, and
              \code{c} color.}
  \item{n}{number of sides for polygon (>8 => circle).}
  \item{angle}{rotation angle of figure, in degrees.}
  \item{type}{\code{type=1} => interior filled, \code{type=2} => edge,
              \code{type=3} => both.}
}
\details{
  Uses \code{\link{polygon}} to draw shaded polygons and
  \code{\link{lines}} for outline.  If n is odd, there is
  a vertex at (0, d/2), otherwise the midpoint of a side is
  at (0, d/2).
}
\value{Invisible.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{polygon}}, 
  \code{\link{lines}}, 
  \code{\link{map.key}}, 
  \code{\link{map.groups}}
}
\examples{
  plot (c(0,1), c(0,1), type="n")
  ngon (c(.5, .5, 10, "blue"), angle=30, n=3)
  apply (cbind (runif(8), runif(8), 6, 2), 1, ngon)
}
\keyword{aplot}

\eof
\name{oregon.bird.dist}
\alias{oregon.bird.dist}
\title{Presence/Absence of Bird Species in Oregon, USA}
\description{
  Binary matrix (1 = present) for distributions of 248 native
  breeding bird species for 389 grid cells in Oregon, USA.
}
\usage{data (oregon.bird.dist)}
\format{A data frame with 389 rows and 248 columns.}
\details{
  Row names are hexagon identifiers from White et al. (1992).
  Column names are species element codes developed by
  The Nature Conservancy (TNC),
  the Oregon Natural Heritage Program (ONHP), and
  NatureServe.
}
\source{Denis White, \email{white.denis@epa.gov} }
\references{
  Master, L. (1996) Predicting distributions for vertebrate
  species: some observations, \emph{Gap Analysis: A Landscape
  Approach to Biodiversity Planning}, Scott, J.M., Tear, T.H.,
  and Davis, F.W., editors, American Society for Photogrammetry
  and Remote Sensing, Bethesda, MD, pp. 171-176.

  White, D., Preston, E.M., Freemark, K.E., Kiester, A.R. (1999)
  A hierarchical framework for conserving biodiversity,
  \emph{Landscape ecological analysis: issues and applications},
  Klopatek, J.M., Gardner, R.H., editors, Springer-Verlag, pp.
  127-153.

  White, D., Kimerling, A.J., Overton, W.S.  (1992)  Cartographic
  and geometric components of a global sampling design for
  environmental monitoring, \emph{Cartography and Geographic
  Information Systems}, \bold{19}(1), 5-22.

  TNC, \url{http://nature.org/}

  ONHP, \url{http://natureserve.org/nhp/us/or/}

  NatureServe, \url{http://natureserve.org/}
}
\seealso{
  \code{\link{oregon.env.vars}}, 
  \code{\link{oregon.bird.names}}, 
  \code{\link{oregon.grid}}, 
  \code{\link{oregon.border}}
}
\keyword{datasets}

\eof
\name{oregon.bird.names}
\alias{oregon.bird.names}
\title{Names of Bird Species in Oregon, USA}
\description{
  Scientific and common names for 248 native breeding bird species
  in Oregon, USA.
}
\usage{data (oregon.bird.names)}
\format{A data frame with 248 rows and 2 columns.}
\details{
  Row names are species element codes.  Columns are
  \code{"scientific.name"} and \code{"common.name"}.
  Data are provided by The Nature Conservancy (TNC),
  the Oregon Natural Heritage Program (ONHP), and
  NatureServe.
}
\source{Denis White, \email{white.denis@epa.gov} }
\references{
  Master, L. (1996) Predicting distributions for vertebrate
  species: some observations, \emph{Gap Analysis: A Landscape
  Approach to Biodiversity Planning}, Scott, J.M., Tear, T.H.,
  and Davis, F.W., editors, American Society for Photogrammetry
  and Remote Sensing, Bethesda, MD, pp. 171-176.

  TNC, \url{http://nature.org/}

  ONHP, \url{http://natureserve.org/nhp/us/or/}

  NatureServe, \url{http://natureserve.org/}
}
\seealso{
  \code{\link{oregon.bird.dist}}
}
\keyword{datasets}

\eof
\name{oregon.border}
\alias{oregon.border}
\title{Boundary of Oregon, USA}
\description{
  The boundary of the state of Oregon, USA, in
  \code{\link{lines}} format.
}
\usage{data (oregon.border)}
\format{A data frame with 485 rows and 2 columns (the components
  \code{"x"} and \code{"y"}).}
\details{
  The map projection for this boundary, as well as the point
  coordinates in \code{\link{oregon.env.vars}}, is the Lambert
  Conformal Conic with standard parallels at 33 and 45
  degrees North latitude, with the longitude of the central
  meridian at 120 degrees, 30 minutes West longitude,
  and with the projection origin latitude at 41 degrees,
  45 minutes North latitude.
}
\source{Denis White, \email{white.denis@epa.gov} }
\keyword{datasets}

\eof
\name{oregon.env.vars}
\alias{oregon.env.vars}
\title{Environmental Variables for Oregon, USA}
\description{
  Distributions of 10 environmental variables for 389 grid
  cells in Oregon, USA.
}
\usage{data (oregon.env.vars)}
\format{A data frame with 389 rows and 10 columns.}
\details{
  Row names are hexagon identifiers from White et al. (1992).
  Variables (columns) are 

  \tabular{ll}{
    bird.spp \tab number of native breeding bird species\cr
    x \tab x coordinate of center of grid cell\cr
    y \tab y coordinate of center of grid cell\cr
    jan.temp \tab mean minimum January temperature (C)\cr
    jul.temp \tab mean maximum July temperature (C)\cr
    rng.temp \tab mean difference between July and January temperatures (C)\cr
    ann.ppt \tab mean annual precipitation (mm)\cr
    min.elev \tab minimum elevation (m)\cr
    rng.elev \tab range of elevation (m)\cr
    max.slope \tab maximum slope (percent)
  }}
}
\source{Denis White, \email{white.denis@epa.gov} }
\references{
  White, D., Preston, E.M., Freemark, K.E., Kiester, A.R. (1999)
  A hierarchical framework for conserving biodiversity,
  \emph{Landscape ecological analysis: issues and applications},
  Klopatek, J.M., Gardner, R.H., editors, Springer-Verlag, pp.
  127-153.

  White, D., Kimerling, A.J., Overton, W.S.  (1992)  Cartographic
  and geometric components of a global sampling design for
  environmental monitoring, \emph{Cartography and Geographic
  Information Systems}, \bold{19}(1), 5-22.
}
\seealso{
  \code{\link{oregon.bird.dist}}, 
  \code{\link{oregon.grid}}, 
  \code{\link{oregon.border}}
}
\keyword{datasets}

\eof
\name{oregon.grid}
\alias{oregon.grid}
\title{Hexagonal Grid Cell Polygons covering Oregon, USA}
\description{
  Polygon borders for 389 hexagonal grid cells covering Oregon, USA,
  in \code{\link{polygon}} format.
}
\usage{data (oregon.grid)}
\format{A data frame with 3112 rows and 2 columns (the components
  \code{"x"} and \code{"y"}).}
\details{
  The polygon format used for these grid cell boundaries is a slight
  variation from the standard R/S format.  Each cell polygon is
  described by seven coordinate pairs, the last repeating the first.
  Prior to the first coordinate pair of each cell is a row containing
  NA in the \code{"y"} column and, in the \code{"x"} column, an
  identifier for the cell.  The identifiers are the same as the
  row names in \code{\link{oregon.bird.dist}} and
  \code{\link{oregon.env.vars}}.  See \code{\link{map.groups}} for
  how the linkage is made in mapping.

  These grid cells are extracted from a larger set covering the
  conterminous United States and adjacent parts of Canada and
  Mexico, as described in White et al. (1992).  Only cells with
  at least 50 percent of their area contained within the state of
  Oregon are included.

  The map projection for the coordinates, as well as the point
  coordinates in \code{\link{oregon.env.vars}}, is the Lambert
  Conformal Conic with standard parallels at 33 and 45
  degrees North latitude, with the longitude of the central
  meridian at 120 degrees, 30 minutes West longitude,
  and with the projection origin latitude at 41 degrees,
  45 minutes North latitude.
}
\source{Denis White, \email{white.denis@epa.gov} }
\references{
  White, D., Kimerling, A.J., Overton, W.S.  (1992)  Cartographic
  and geometric components of a global sampling design for
  environmental monitoring, \emph{Cartography and Geographic
  Information Systems}, \bold{19}(1), 5-22.
}
\keyword{datasets}

\eof
\name{prune.Rpart}
\alias{prune.Rpart}
\title{Prunes an Rpart Classification or Regression Tree}
\description{
  Reduces a prediction tree produced by \code{\link{rpart}} to a
  smaller tree by specifying either a cost-complexity parameter,
  or a number of nodes to which to prune.
}
\usage{
  prune.Rpart (tree, cp=NULL, best=NULL)
}
\arguments{
  \item{tree}{object of class \code{rpart}.}
  \item{cp}{cost-complexity parameter.}
  \item{best}{number of nodes to which to prune.}

  If both \code{cp} and \code{best} are not \code{NULL}, then
  \code{cp} is used.
}
\details{
  A minor enhancement of the existing \code{\link{prune.rpart}} to
  incorporate the parameter \code{best} as it is used in
  \code{\link[tree]{prune.tree}}.  See example.
}
\value{Pruned tree object of class \code{rpart}.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[tree]{prune.tree}},
  \code{\link[rpart]{rpart}}, 
  \code{\link[rpart]{prune.rpart}}
}
\examples{
  library (rpart)
  data (oregon.env.vars, oregon.border, oregon.grid)

  draw.tree (prune.Rpart (rpart (oregon.env.vars), best=7), 
    nodeinfo=TRUE, units="species", cases="cells", digits=0)

  group <- group.tree (prune.Rpart (rpart (oregon.env.vars), best=7))
  names(group) <- row.names(oregon.env.vars)
  map.groups (oregon.grid, group)
  lines (oregon.border)
  map.key (0.05, 0.65, labels=as.character(seq(6)), 
    size=1, new=FALSE, sep=0.5, pch=19, head="node")
}
\keyword{manip}
\keyword{cluster}

\eof
\name{prune.clust}
\alias{prune.clust}
\title{Prunes a Hierarchical Cluster Tree}
\description{
  Reduces a hierarchical cluster tree to a smaller tree either
  by pruning until a given number of observation groups remain,
  or by pruning tree splits below a given height.
}
\usage{
  prune.clust (cluster, k=NULL, h=NULL)
}
\arguments{
  \item{cluster}{object of class \code{hclust} or \code{twins}.}
  \item{k}{desired number of groups.}
  \item{h}{height at which to prune for grouping.}

  At least one of \code{k} or \code{h} must be specified; \code{k} takes
  precedence if both are given.
}
\details{
  Used with \code{\link{draw.clust}}.  See example.
}
\value{Pruned cluster object of class \code{hclust}.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link[mva]{hclust}}, 
  \code{\link[cluster]{twins.object}}, 
  \code{\link{cutree}}, 
  \code{\link{draw.clust}}
}
\examples{
  library (cluster)
  data (oregon.bird.dist)

  draw.clust (prune.clust (agnes (oregon.bird.dist), k=6))
}
\keyword{manip}
\keyword{cluster}

\eof
