\name{snow-cluster}
\title{Cluster-Level SNOW Functions}
\alias{clusterSplit}
\alias{clusterCall}
\alias{clusterApply}
\alias{clusterApplyLB}
\alias{clusterEvalQ}
\alias{clusterExport}
\usage{
clusterSplit(cl, seq)
clusterCall(cl, fun, ...)
clusterApply(cl, x, fun, ...)
clusterApplyLB(cl, x, fun, ...)
clusterEvalQ(cl, expr)
clusterExport(cl, list)
}
\arguments{
  \item{cl}{cluster object}
  \item{fun}{function or character string naming a function}
  \item{expr}{expression to evaluate}
  \item{seq}{vector to split}
  \item{list}{character vector of variables to export}
  \item{x}{array}
  \item{...}{additional arguments to pass to standard function}
}
\description{
  Functions for computing on a SNOW cluster.
}
\details{
  These are the basic functions for computing on a cluster.  All
  evaluations on the slave nodes are done using \code{try}.  Currently
  the result lists returned by the cluster computing functions can a
  combination of proper results and \code{try} error objects.  More
  sophisticated approaches will be considered in the future.

  \code{clusterCall} calls a function \code{fun} with identical arguments 
  \code{...} on each node in the cluster \code{cl} and returns a list
  of the results.

  \code{clusterEvalQ} evaluates a literal expression on each cluster node.
  It a cluster version of \code{evalq}, and is a convenience function
  defined in terms of \code{clusterCall}.

  \code{clusterApply} calls \code{fun} on the first cluster node 
  with arguments \code{seq[[1]]} and \code{...}, on the second node with
  \code{seq[[2]]} and \code{...}, and so on.  The length of \code{seq}
  must be less than or equal to the number of nodes in the cluster.
  A list of the results is returned; the length of the result list
  will equal the length of \code{seq}.

  \code{clusterApplyLB} is a load balancing version of \code{clusterApply}.
  if the length \code{p} of \code{seq} is greater than the number of
  cluster nodes \code{n}, then the first \code{n} jobs are placed in
  order on the \code{n} nodes.  When the first job completes, the next
  job is placed on the available node; this continues until all jobs
  are complete.  Using \code{clusterApplyLB} can result in better
  cluster utilization than using \code{clusterApply}.  However,
  increased communication can reduce performance.  Furthermore, the
  node that executes a particular job is nondeterministic, which can
  complicate ensuring reproducibility in simulations.

  \code{clusterExport} assigns the global values on the master of the
  variables named in \code{list} to variables of the same names in the
  global environments of each node.

  \code{clusterSplit} splits \code{seq} into one consecutive piece for
  each cluster and returns the result as a list with length equal to the
  number of cluster nodes.  Currently the pieces are chosen to be close
  to equal in length.  Future releases will attempt to use relative
  performance information about nodes to choose split proportional to
  performance.

  For more details see
  \url{http://www.stat.uiowa.edu/~luke/R/cluster/cluster.html}.
}
\examples{
  \dontrun{
cl <- makeSOCKcluster(c("localhost","localhost"))

clusterApply(cl, 1:2, get("+"), 3)

clusterEvalQ(cl, library(boot))

x<-1
clusterExport(cl, "x")
clusterCall(cl, function(y) x + y, 2)

  }
}
\keyword{programming}

\eof
\name{snow-internal}
\title{Internal SNOW Objects}
\alias{runMPIslave}
\alias{setMPIcluster}
\alias{stopCluster.default}
\alias{stopCluster.MPIcluster}
\alias{stopCluster.spawnedMPIcluster}
\alias{defaultClusterOptions}
\alias{enquote}
\alias{docall}
\alias{slaveLoop}
\alias{newSOCKnode}
\alias{makeSOCKmaster}
\alias{closeNode.SOCKnode}
\alias{sendData.SOCKnode}
\alias{recvData.SOCKnode}
\alias{newPVMnode}
\alias{makePVMmaster}
\alias{closeNode.PVMnode}
\alias{sendData.PVMnode}
\alias{recvData.PVMnode}
\alias{newMPInode}
\alias{makeMPImaster}
\alias{closeNode.MPInode}
\alias{sendData.MPInode}
\alias{recvData.MPInode}
\alias{closeNode}
\alias{closeNode.default}
\alias{sendData}
\alias{recvData}
\alias{sendNode}
\alias{postNode}
\alias{stopNode}
\alias{sendCall}
\alias{recvResult}
\alias{initSprngNode}
\alias{initRNGstreamNode}
\alias{splitIndices}
\alias{splitList}
\alias{splitRows}
\alias{splitCols}
\alias{addClusterOptions}
\alias{initDefaultClusterOptions}
\alias{findRecvOneTag}
\alias{recvOneData}
\alias{recvOneData.MPIcluster}
\alias{recvOneData.PVMcluster}
\alias{recvOneData.SOCKcluster}
\alias{recvOneResult}
\alias{getClusterOption}
\usage{
stopCluster.default(cl)
stopCluster.MPIcluster(cl)
stopCluster.spawnedMPIcluster(cl)
defaultClusterOptions
addClusterOptions(options, new)
initDefaultClusterOptions()

enquote(x)
docall(fun, args)
slaveLoop(master)

newSOCKnode(machine = "localhost", ..., options = defaultClusterOptions)
makeSOCKmaster()
closeNode.SOCKnode(node)
sendData.SOCKnode(node, data)
recvData.SOCKnode(node)

newPVMnode(where = "", options = defaultClusterOptions)
makePVMmaster()
sendData.PVMnode(node, data)
recvData.PVMnode(node)

setMPIcluster(new)
runMPIslave()
newMPInode(rank, comm)
makeMPImaster(comm)
sendData.MPInode(node, data)
recvData.MPInode(node)

closeNode(node)
closeNode.default(node)
sendData(node, data)
recvData(node)

postNode(con, type, value = NULL, tag = NULL)
stopNode(n)

sendCall(con, fun, args, return = TRUE)
recvResult(con)

initSprngNode(streamno, nstream, seed, kind, para) 
initRNGstreamNode(stream)

splitIndices(nx, ncl)
splitList(x, ncl)
splitRows(x, ncl)
splitCols(x, ncl)

findRecvOneTag(cl, anytag)
recvOneData(cl)
recvOneResult(cl)
recvOneData.MPIcluster(cl)
recvOneData.PVMcluster(cl)
recvOneData.SOCKcluster(cl)
getClusterOption(name, options = defaultClusterOptions)
}
\description{
  Internal functions for the snow package.
}
\details{
  These are not to be called by the user.
}
\keyword{internal}

\eof
\name{snow-parallel}
\title{Higher Level SNOW Functions}
\alias{parLapply}
\alias{parRapply}
\alias{parCapply}
\alias{parApply}
\alias{parMM}
\alias{parSapply}
\usage{
parLapply(cl, x, fun, ...)
parSapply(cl, X, FUN, ..., simplify = TRUE, USE.NAMES = TRUE) 
parApply(cl, X, MARGIN, FUN, ...)
parRapply(cl, x, fun, ...)
parCapply(cl, x, fun, ...)
parMM(cl, A, B)
}
\arguments{
  \item{cl}{cluster object}
  \item{fun}{function or character string naming a function}
  \item{X}{array to be used}
  \item{x}{matrix to be used}
  \item{FUN}{function or character string naming a function}
  \item{MARGIN}{vector specifying the dimensions to use.}
  \item{simplify}{logical; see \code{sapply}}
  \item{USE.NAMES}{logical; see \code{sapply}}
  \item{...}{additional arguments to pass to standard function}
  \item{A}{matrix}
  \item{B}{matrix}
}
\description{
  Parallel versions of \code{apply} and related functions.
}
\details{
  \code{parLapply}, \code{parSapply}, and \code{parApply} are parallel
  versions of \code{lapply}, \code{sapply}, and \code{apply}.

  \code{parRapply} and \code{parCapply} are parallel row and column
  \code{apply} functions for a matrix \code{x}; they may be slightly
  more efficient than \code{parApply}.

  \code{parMM} is a very simple(minded) parallel matrix multiply;
  it is intended as an illustration.

  For more details see
  \url{http://www.stat.uiowa.edu/~luke/R/cluster/cluster.html}.
}
\examples{
  \dontrun{
cl <- makeSOCKcluster(c("localhost","localhost"))
parSapply(cl, 1:20, get("+"), 3)
  }
}
\keyword{programming}

\eof
\name{snow-rand}
\title{Uniform Random Number Generation in SNOW Clusters}
\alias{clusterSetupRNG}
\alias{clusterSetupSPRNG}
\alias{clusterSetupRNGstream}
\usage{
clusterSetupRNG (cl, type = "RNGstream", ...)

clusterSetupRNGstream (cl, seed=rep(12345,6), ...)
clusterSetupSPRNG (cl, seed = round(2^32 * runif(1)),
                   prngkind = "default", para = 0, ...) 

}
\arguments{
  \item{cl}{Cluster object.}
  \item{type}{\code{type="RNGstream"} (default) initializes the
    L'Ecuyer's RNG.  \code{type="SPRNG"} initializes the
    SPRNG generator.}
  \item{...}{Arguments passed to the underlying function (see details
    bellow).} 
  \item{seed}{Integer value (SPRNG) or a vector of six integer values
    (RNGstream) used as seed
    for the RNG.}
  \item{prngkind}{Character string naming generator type used with SPRNG.}
  \item{para}{Additional parameters for the generator.}
}
\description{
  Initialize independent uniform random number streams to be used in a
  SNOW cluster. It uses either the L'Ecuyer's random number generator
  (package rlecuyer required) or the SPRNG generator (package rsprng required).
}
\details{
  \code{clusterSetupRNG} calls (subject to its argument values) one of
  the other functions, passing arguments \code{(cl, ...)}.  If the
  "SPRNG" type is used, then the function \code{clusterSetupSPRNG} is
  called.  If the "RNGstream" type is used, then the function
  \code{clusterSetupRNGstream} is called.

  \code{clusterSetupSPRNG} loads the \code{rsprng} package and initializes
  separate streams  on each node. For further details see the documentation
  of \code{init.sprng}.  The generator on the master is not affected.

  \code{clusterSetupRNGstream} loads the \code{rlecuyer} package,
  creates one stream per node and distributes the stream states
  to the nodes.

  For more details see
  \url{http://www.stat.uiowa.edu/~luke/R/cluster/cluster.html}.
}
\examples{
  \dontrun{
clusterSetupSPRNG(cl)
clusterSetupSPRNG(cl, seed=1234)
clusterSetupRNG(cl, seed=rep(1,6))
  }
}
\keyword{programming}

\eof
\name{snow-startstop}
\title{Starting and Stopping SNOW Clusters}
\alias{getMPIcluster}
\alias{makeMPIcluster}
\alias{makePVMcluster}
\alias{makeSOCKcluster}
\alias{makeCluster}
\alias{stopCluster}
\alias{setDefaultClusterOptions}
\usage{
makeCluster(spec, type = getClusterOption("type"), ...)
stopCluster(cl)

setDefaultClusterOptions(...)

makeSOCKcluster(names, ..., options = defaultClusterOptions)
makePVMcluster(count, ..., options = defaultClusterOptions)
makeMPIcluster(count, ..., options = defaultClusterOptions)
getMPIcluster()
}
\arguments{
  \item{spec}{cluster specification}
  \item{count}{number of nodes to create}
  \item{names}{character vector of node names}
  \item{options}{cluster options object}
  \item{cl}{cluster object}
  \item{...}{cluster option specifications}
  \item{type}{character; specifies cluster type.}
}
\description{
  Functions to start and stop a SNOW cluster and to set default cluster
  options.
}
\details{
  \code{makeCluster} starts a cluster of the specified or default type
  and returns a reference to the cluster.  Supported cluster types are
  \code{"SOCK"}, \code{"PVM"}, and \code{"MPI"}.  For \code{"PVM"} and
  \code{"MPI"} clusters the \code{spec} argument should be an integer
  specifying the number of slave nodes to create.  For \code{"SOCK"}
  clusters \code{spec} should be a character vector naming the hosts on
  which slave nodes should be started; one node is started for each
  element in the vector.

  \code{stopCluster} should be called to properly shut down the cluster
  before exiting R.  If it is not called it may be necessary to use
  external means to ensure that all slave processes are shut down.

  \code{setDefaultClusterOptions} can be used to specify alternate
  values for default cluster options.  There are many options.  The
  most useful ones are \code{type} and \code{homogeneous}.  The
  default value of the \code{type} option is currently set to
  \code{"PVM"} if the \code{rpvm} package is available; otherwise, it
  is set to \code{"MPI"} if \code{Rmpi} is available, and it is set to
  \code{"SOCK"} if neither of these packages is found.

  The \code{homogeneous} option should be set to \code{FALSE} to
  specify that the startup procedure for inhomogeneous clusters is to
  be used; this requires some additional configuration. The default
  setting is \code{TRUE} unless the environment variable
  \code{R_HOME_LIB} is defined on the master host with a non-empty
  value.

  The option\code{outfile} can be used to specify the file to which
  slave node output is to be directed.  The default is
  \code{/dev/null}; during debugging of an installation it can be
  useful to set this to a proper file.

  The functions \code{makeSOCKcluster}, \code{makePVMcluster}, and
  \code{makeMPIcluster} can be used to start a cluster of the
  corresponding type.

  In MPI configurations where process spawning is not available and
  something like \code{mpirun} is used to start a master and a set of slaves
  the corresponding cluster will have been pre-constructed and can be obtained
  with \code{getMPIcluster}.  This interface is still experimental and
  subject to change.

  For more details see
  \url{http://www.stat.uiowa.edu/~luke/R/cluster/cluster.html}.
}
\examples{
  \dontrun{
cl <- makeCluster(c("localhost","localhost"), type = "SOCK")
clusterApply(cl, 1:2, get("+"), 3)
  }
}
\keyword{programming}

\eof
