\name{xcusum.ad}
\alias{xcusum.ad}
\title{Compute steady-state ARLs of CUSUM control charts}
\description{Computation of the steady-state Average Run Length (ARL)
for different types of CUSUM control charts monitoring normal mean.}
\usage{xcusum.ad(k, h, mu1, mu0 = 0, sided = "one", r = 30)}
\arguments{
\item{k}{reference value of the CUSUM control chart.}
\item{h}{decision interval (alarm limit, threshold) of the CUSUM control chart.}
\item{mu1}{out-of-control mean.}
\item{mu0}{in-control mean.}
\item{sided}{distinguish between one-, two-sided and Crosier's modified
two-sided CUSUM scheme by choosing \code{"one"}, \code{"two"},
and \code{"Crosier"}, respectively.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-, two-sided) or \code{2r+1}
(Crosier).}
}
\details{
\code{xcusum.ad} determines the steady-state Average Run Length (ARL)
by numerically solving the related ARL integral equation by means
of the Nystrm method based on Gauss-Legendre quadrature
and using the power method for deriving the largest in magnitude
eigenvalue and the related left eigenfunction.
}
\value{Returns a single value which resembles the steady-state ARL.}
\references{
R. B. Crosier (1986),
A new two-sided cumulative quality control scheme,
\emph{Technometrics 28}, 187-194.
}
\note{Be cautious in increasing the dimension parameter \code{r} for
two-sided CUSUM schemes. The resulting matrix dimension is \code{r^2} times
\code{r^2}. Thus, go beyond 30 only on fast machines. This is the only case,
were the package routines are based on the Markov chain approach. Moreover,
the two-sided CUSUM scheme needs a two-dimensional Markov chain.} 
\author{Sven Knoth}
\seealso{
\code{xcusum.arl} for zero-state ARL computation and
\code{xewma.ad} for the steady-state ARL of EWMA control charts.
}
\examples{
## comparison of zero-state (= worst case ) and steady-state performance
## for one-sided CUSUM control charts

k <- .5
h <- xcusum.crit(k,500)
mu <- c(0,.5,1,1.5,2)
arl <- sapply(mu,k=k,h=h,xcusum.arl)
ad <- sapply(mu,k=k,h=h,xcusum.ad)
round(cbind(mu,arl,ad),digits=2)

## Crosier (1986), Crosier's modified two-sided CUSUM
## He introduced the modification and evaluated it by means of
## Markov chain approximation

k <- .5
h2 <- 4
hC <- 3.73
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,4,5)
ad2 <- sapply(mu,k=k,h=h2,sided="two",r=20,xcusum.ad)
adC <- sapply(mu,k=k,h=hC,sided="Crosier",xcusum.ad)
round(cbind(mu,ad2,adC),digits=2)

## results in the original paper are (in Table 5)
## 0.00 163.   164.
## 0.25  71.6   69.0
## 0.50  25.2   24.3
## 0.75  12.3   12.1
## 1.00   7.68   7.69
## 1.50   4.31   4.39
## 2.00   3.03   3.12
## 2.50   2.38   2.46
## 3.00   2.00   2.07
## 4.00   1.55   1.60
## 5.00   1.22   1.29.
}
\keyword{ts}

\eof
\name{xcusum.arl}
\alias{xcusum.arl}
\title{Compute ARLs of CUSUM control charts}
\description{Computation of the (zero-state) Average Run Length (ARL)
for different types of CUSUM control charts monitoring normal mean.}
\usage{xcusum.arl(k, h, mu, hs = 0, sided = "one", r = 30)}
\arguments{
\item{k}{reference value of the CUSUM control chart.}
\item{h}{decision interval (alarm limit, threshold) of the CUSUM control chart.}
\item{mu}{true mean.}
\item{hs}{so-called headstart (give fast initial response).}
\item{sided}{distinguish between one-, two-sided and Crosier's modified
two-sided CUSUM scheme by choosing \code{"one"}, \code{"two"},
and \code{"Crosier"}, respectively.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-, two-sided) or \code{2r+1}
(Crosier).}
}
\details{
\code{xcusum.arl} determines the Average Run Length (ARL) by numerically
solving the related ARL integral equation by means of the Nystrm method
based on Gauss-Legendre quadrature.
}
\value{Returns a single value which resembles the ARL.}
\references{
A. L. Goel, S. M. Wu (1971),
Determination of A.R.L. and a contour nomogram for CUSUM charts to
control normal mean, \emph{Technometrics 13}, 221-230.

D. Brook, D. A. Evans (1972),
An approach to the probability distribution of cusum run length,
\emph{Biometrika 59}, 539-548.

J. M. Lucas, R. B. Crosier (1982),
Fast initial response for cusum quality-control schemes: 
Give your cusum a headstart, \emph{Technometrics 24}, 199-205.

L. C. Vance (1986), 
Average run lengths of cumulative sum control charts for controlling 
normal means, \emph{Journal of Quality Technology 18}, 189-193.

K.-H. Waldmann (1986),
Bounds for the distribution of the run length of one-sided and
two-sided CUSUM quality control schemes,
\emph{Technometrics 28}, 61-67.

R. B. Crosier (1986),
A new two-sided cumulative quality control scheme,
\emph{Technometrics 28}, 187-194.
}
\author{Sven Knoth}
\seealso{
\code{xewma.arl} for zero-state ARL computation of EWMA control charts
and \code{xcusum.ad} for the steady-state ARL.
}
\examples{
## Brook/Evans (1972), one-sided CUSUM
## Their results are based on the less accurate Markov chain approach.

k <- .5
h <- 3
round(c( xcusum.arl(k,h,0), xcusum.arl(k,h,1.5) ),digits=2)

## results in the original paper are L0 = 117.59, L1 = 3.75 (in Subsection 4.3).

## Lucas, Crosier (1982)
## (one- and) two-sided CUSUM with possible headstarts

k <- .5
h <- 4
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,4,5)
arl1 <- sapply(mu,k=k,h=h,sided="two",xcusum.arl)
arl2 <- sapply(mu,k=k,h=h,hs=h/2,sided="two",xcusum.arl)
round(cbind(mu,arl1,arl2),digits=2)

## results in the original paper are (in Table 1)
## 0.00 168.   149.
## 0.25  74.2   62.7
## 0.50  26.6   20.1
## 0.75  13.3    8.97
## 1.00   8.38   5.29
## 1.50   4.75   2.86
## 2.00   3.34   2.01
## 2.50   2.62   1.59
## 3.00   2.19   1.32
## 4.00   1.71   1.07
## 5.00   1.31   1.01.

## Vance (1986), one-sided CUSUM
## The first paper on using Nystrm method and Gauss-Legendre quadrature
## for solving the ARL integral equation (see as well Goel/Wu, 1971)

k <- 0
h <- 10
mu <- c(-.25,-.125,0,.125,.25,.5,.75,1)
round(cbind(mu,sapply(mu,k=k,h=h,xcusum.arl)),digits=2)

## results in the original paper are (in Table 1 incl. Goel/Wu (1971) results)
##  -0.25  2071.51
##  -0.125  400.28
##   0.0    124.66
##   0.125   59.30
##   0.25    36.71
##   0.50    20.37
##   0.75    14.06
##   1.00    10.75.

## Waldmann (1986),
## one- and two-sided CUSUM

## one-sided case

k <- .5
h <- 3
mu <- c(-.5,0,.5)
round(sapply(mu,k=k,h=h,xcusum.arl),digits=2)

## results in the original paper are 1963, 117.4, and 17.35, resp.
## (in Tables 3, 1, and 5, resp.).

## two-sided case

k <- .6
h <- 3
round(xcusum.arl(k,h,-.2,sided="two"),digits=1)  # fits to Waldmann's setup

## result in the original paper is 65.4 (in Table 6).

## Crosier (1986), Crosier's modified two-sided CUSUM
## He introduced the modification and evaluated it by means of
## Markov chain approximation

k <- .5
h <- 3.73
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,4,5)
round(cbind(mu,sapply(mu,k=k,h=h,sided="Crosier",xcusum.arl)),digits=2)

## results in the original paper are (in Table 3)
## 0.00 168.
## 0.25  70.7
## 0.50  25.1
## 0.75  12.5
## 1.00   7.92
## 1.50   4.49
## 2.00   3.17
## 2.50   2.49
## 3.00   2.09
## 4.00   1.60
## 5.00   1.22.

## SAS/QC manual 1999
## one- and two-sided CUSUM schemes

## one-sided

k <- .25
h <- 8
mu <- 2.5
print(xcusum.arl(k,h,mu),digits=12)
print(xcusum.arl(k,h,mu,hs=.1),digits=12)

## original results are 4.1500836225 and 4.1061588131.

## two-sided

print(xcusum.arl(k,h,mu,sided="two"),digits=12)

## original result is 4.1500826715.
}
\keyword{ts}

\eof
\name{xcusum.crit}
\alias{xcusum.crit}
\title{Compute decision intervals of CUSUM control charts}
\description{Computation of the decision intervals (alarm limits)
for different types of CUSUM control charts monitoring normal mean.}
\usage{xcusum.crit(k, L0, mu0 = 0, hs = 0, sided = "one", r = 30)}
\arguments{
\item{k}{reference value of the CUSUM control chart.}
\item{L0}{in-control ARL.}
\item{mu0}{in-control mean.}
\item{hs}{so-called headstart (give fast initial response).}
\item{sided}{distinguish between one-, two-sided and Crosier's modified
two-sided CUSUM schemoosing \code{"one"}, \code{"two"},
and \code{"Crosier"}, respectively.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-, two-sided) or \code{2r+1}
(Crosier).}
}
\details{
\code{xcusum.crit} determines the decision interval (alarm limit)
for given in-control ARL \code{L0}
by applying secant rule and using \code{xcusum.arl()}.
}
\value{Returns a single value which resembles the decision interval
\code{h}.}
%\references{Later...}
\author{Sven Knoth}
\seealso{\code{xcusum.arl} for zero-state ARL computation.}
\examples{
k <- .5
incontrolARL <- c(500,5000,50000)
sapply(incontrolARL,k=k,xcusum.crit,r=10) # accuracy with 10 nodes
sapply(incontrolARL,k=k,xcusum.crit,r=20) # accuracy with 20 nodes
sapply(incontrolARL,k=k,xcusum.crit)      # accuracy with 30 nodes
}
\keyword{ts}

\eof
\name{xewma.ad}
\alias{xewma.ad}
\title{Compute steady-state ARLs of EWMA control charts}
\description{Computation of the steady-state Average Run Length (ARL)
for different types of EWMA control charts monitoring normal mean.}
\usage{xewma.ad(l,c,mu1,mu0=0,zr=0,sided="one",limits="fix",r=40)}
\arguments{
\item{l}{smoothing parameter lambda of the EWMA control chart.}
\item{c}{critical value (similar to alarm limit) of the EWMA control chart.}
\item{mu1}{in-control mean.}
\item{mu0}{out-of-control mean.}
\item{zr}{reflection border for the one-sided chart.}
\item{sided}{distinguish between one- and two-sided
two-sided EWMA control chart by choosing \code{"one"} and \code{"two"},
respectively.}
\item{limits}{distinguish between different control limits behavior.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-sided) or \code{r}
(two-sided).}
}
\details{
\code{xewma.ad} determines the steady-state Average Run Length (ARL)
by numerically solving the related ARL integral equation by means
of the Nystrm method based on Gauss-Legendre quadrature
and using the power method for deriving the largest in magnitude
eigenvalue and the related left eigenfunction.
}
\value{Returns a single value which resembles the steady-state ARL.}
\references{
S. V. Crowder (1987),
A simple method for studying run-length distributions of
exponentially weighted moving average charts,
\emph{Technometrics 29}, 401-407.

J. M. Lucas, M. S. Saccucci (1990),
Exponentially weighted moving average control schemes: Properties
and enhancements, \emph{Technometrics 32}, 1-12.
}
\author{Sven Knoth}
\seealso{
\code{xewma.arl} for zero-state ARL computation and
\code{xcusum.ad} for the steady-state ARL of CUSUM control charts.
}
\examples{
## comparison of zero-state (= worst case ) and steady-state performance
## for two-sided EWMA control charts

l <- .1
c <- xewma.crit(l,500,sided="two")
mu <- c(0,.5,1,1.5,2)
arl <- sapply(mu,l=l,c=c,sided="two",xewma.arl)
ad <- sapply(mu,l=l,c=c,sided="two",xewma.ad)
round(cbind(mu,arl,ad),digits=2)

## Lucas/Saccucci (1990)
## Lucas/Saccucci (1990)
## two-sided EWMA

## with fixed limits
l1 <- .5
l2 <- .03
c1 <- 3.071
c2 <- 2.437
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,3.5,4,5)
ad1 <- sapply(mu,l=l1,c=c1,sided="two",xewma.ad)
ad2 <- sapply(mu,l=l2,c=c2,sided="two",xewma.ad)
round(cbind(mu,ad1,ad2),digits=2)

## original results are (in Table 3)
## 0.00 499.   480.  
## 0.25 254.    74.1
## 0.50  88.4   28.6
## 0.75  35.7   17.3
## 1.00  17.3   12.5
## 1.50   6.44   8.00
## 2.00   3.58   5.95
## 2.50   2.47   4.78
## 3.00   1.91   4.02
## 3.50   1.58   3.49
## 4.00   1.36   3.09
## 5.00   1.10   2.55.
}
\keyword{ts}

\eof
\name{xewma.arl}
\alias{xewma.arl}
\title{Compute ARLs of EWMA control charts}
\description{Computation of the (zero-state) Average Run Length (ARL)
for different types of EWMA control charts monitoring normal mean.}
\usage{xewma.arl(l,c,mu,zr=0,hs=0,sided="one",limits="fix",r=40)}
\arguments{
\item{l}{smoothing parameter lambda of the EWMA control chart.}
\item{c}{critical value (similar to alarm limit) of the EWMA control chart.}
\item{mu}{true mean.}
\item{zr}{reflection border for the one-sided chart.}
\item{hs}{so-called headstart (give fast initial response).}
\item{sided}{distinguish between one- and two-sided 
two-sided EWMA control chart by choosing \code{"one"} and \code{"two"},
respectively.}
\item{limits}{distinguish between different control limits behavior.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-sided) or \code{r}
(two-sided).}
}
\details{
In case of the two-sided chart with fixed control limits and
of the one-sided chart,
\code{xewma.arl} determines the Average Run Length (ARL) by numerically
solving the related ARL integral equation by means of the Nystrm method
based on Gauss-Legendre quadrature.
Otherwise, for the two-sided chart with varying limits the method
presented in Knoth (2003) is utilized.
}
\value{Returns a single value which resembles the ARL.}
\references{
K.-H. Waldmann (1986),
Bounds for the distribution of the run length of geometric moving 
average charts, \emph{Appl. Statist. 35}, 151-158.

S. V. Crowder (1987),
A simple method for studying run-length distributions of
exponentially weighted moving average charts,
\emph{Technometrics 29}, 401-407.

J. M. Lucas, M. S. Saccucci (1990),
Exponentially weighted moving average control schemes: Properties
and enhancements, \emph{Technometrics 32}, 1-12.

S. Chandrasekaran, J. R. English, R. L. Disney (1995), 
Modeling and analysis of EWMA control schemes with variance-adjusted 
control limits, \emph{IIE Transactions 277}, 282-290.

T. R. Rhoads, D. C. Montgomery, C. M. Mastrangelo (1996),
Fast initial response scheme for exponentially weighted moving average
control chart, \emph{Quality Engineering 9}, 317-327.

S. H. Steiner (1999), 
EWMA control charts with time-varying control limits and fast initial response, 
\emph{Journal of Quality Technology 31}, 75-86.

S. Knoth (2003),
EWMA schemes with non-homogeneous transition kernels,
\emph{Sequential Analysis 22}, 241-255.

S. Knoth (2004), 
Fast initial response features for EWMA Control Charts, 
\emph{forthcoming in Statistical Papers}.
}
\author{Sven Knoth}
\seealso{
\code{xcusum.arl} for zero-state ARL computation of CUSUM control charts
and \code{xewma.ad} for the steady-state ARL.
}
\examples{
## Waldmann (1986), one-sided EWMA
l <- .75
round(xewma.arl(l,2*sqrt((2-l)/l),0,zr=-4*sqrt((2-l)/l)),digits=1)
l <- .5
round(xewma.arl(l,2*sqrt((2-l)/l),0,zr=-4*sqrt((2-l)/l)),digits=1)
## original values are 209.3 and 3907.5 (in Table 2).

## Waldmann (1986), two-sided EWMA with fixed control limits
l <- .75
round(xewma.arl(l,2*sqrt((2-l)/l),0,sided="two"),digits=1)
l <- .5
round(xewma.arl(l,2*sqrt((2-l)/l),0,sided="two"),digits=1)
## original values are 104.0 and 1952 (in Table 1).

## Crowder (1987), two-sided EWMA with fixed control limits
l1 <- .5
l2 <- .05
c <- 2
mu <- (0:16)/4
arl1 <- sapply(mu,l=l1,c=c,sided="two",xewma.arl)
arl2 <- sapply(mu,l=l2,c=c,sided="two",xewma.arl)
round(cbind(mu,arl1,arl2),digits=2)

## original results are (in Table 1)
## 0.00 26.45 127.53
## 0.25 20.12  43.94
## 0.50 11.89  18.97
## 0.75  7.29  11.64
## 1.00  4.91   8.38
## 1.25  3.95*  6.56
## 1.50  2.80   5.41
## 1.75  2.29   4.62
## 2.00  1.94   4.04
## 2.25  1.70   3.61
## 2.50  1.51   3.26
## 2.75  1.37   2.99
## 3.00  1.26   2.76
## 3.25  1.18   2.56
## 3.50  1.12   2.39
## 3.75  1.08   2.26
## 4.00  1.05   2.15  (* -- in Crowder (1987) typo!?). 

## Lucas/Saccucci (1990)
## two-sided EWMA

## with fixed limits
l1 <- .5
l2 <- .03
c1 <- 3.071
c2 <- 2.437
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,3.5,4,5)
arl1 <- sapply(mu,l=l1,c=c1,sided="two",xewma.arl)
arl2 <- sapply(mu,l=l2,c=c2,sided="two",xewma.arl)
round(cbind(mu,arl1,arl2),digits=2)

## original results are (in Table 3)
## 0.00 500.   500.
## 0.25 255.    76.7
## 0.50  88.8   29.3
## 0.75  35.9   17.6
## 1.00  17.5   12.6
## 1.50   6.53   8.07
## 2.00   3.63   5.99
## 2.50   2.50   4.80
## 3.00   1.93   4.03
## 3.50   1.58   3.49
## 4.00   1.34   3.11
## 5.00   1.07   2.55.

## with fir feature
l1 <- .5
l2 <- .03
c1 <- 3.071
c2 <- 2.437
hs1 <- c1/2
hs2 <- c2/2
mu <- c(0,.5,1,2,3,5)
arl1 <- sapply(mu,l=l1,c=c1,hs=hs1,sided="two",limits="fir",xewma.arl)
arl2 <- sapply(mu,l=l2,c=c2,hs=hs2,sided="two",limits="fir",xewma.arl)
round(cbind(mu,arl1,arl2),digits=2)

## original results are (in Table 5)
## 0.0 487.   406.
## 0.5  86.1   18.4
## 1.0  15.9    7.36
## 2.0   2.87   3.43
## 3.0   1.45   2.34
## 5.0   1.01   1.57.

## Chandrasekaran, English, Disney (1995)
## two-sided EWMA with fixed and variance adjusted limits (vacl)

l1 <- .25
l2 <- .1
c1s <- 2.9985
c1n <- 3.0042
c2s <- 2.8159
c2n <- 2.8452
mu <- c(0,.25,.5,.75,1,2)
arl1s <- sapply(mu,l=l1,c=c1s,sided="two",xewma.arl)
arl1n <- sapply(mu,l=l1,c=c1n,sided="two",limits="vacl",xewma.arl)
arl2s <- sapply(mu,l=l2,c=c2s,sided="two",xewma.arl)
arl2n <- sapply(mu,l=l2,c=c2n,sided="two",limits="vacl",xewma.arl)
round(cbind(mu,arl1s,arl1n,arl2s,arl2n),digits=2)

## original results are (in Table 2)
## 0.00 500.   500.   500.   500.
## 0.25 170.09 167.54 105.90  96.6
## 0.50  48.14  45.65  31.08  24.35
## 0.75  20.02  19.72  15.71  10.74
## 1.00  11.07   9.37  10.23   6.35
## 2.00   3.59   2.64   4.32   2.73.

## The results in Chandrasekaran, English, Disney (1995) are not
## that accurate. Let us consider the more appropriate comparison

c1s <- xewma.crit(l1,500,sided="two")
c1n <- xewma.crit(l1,500,sided="two",limits="vacl")
c2s <- xewma.crit(l2,500,sided="two")
c2n <- xewma.crit(l2,500,sided="two",limits="vacl")
mu <- c(0,.25,.5,.75,1,2)
arl1s <- sapply(mu,l=l1,c=c1s,sided="two",xewma.arl)
arl1n <- sapply(mu,l=l1,c=c1n,sided="two",limits="vacl",xewma.arl)
arl2s <- sapply(mu,l=l2,c=c2s,sided="two",xewma.arl)
arl2n <- sapply(mu,l=l2,c=c2n,sided="two",limits="vacl",xewma.arl)
round(cbind(mu,arl1s,arl1n,arl2s,arl2n),digits=2)

## which demonstrate the abilities of the variance-adjusted limits
## scheme more explicitely.

## Rhoads, Montgomery, Mastrangelo (1996)
## two-sided EWMA with fixed and variance adjusted limits (vacl),
## with fir and both features

l <- .03
c <- 2.437
mu <- c(0,.5,1,1.5,2,3,4)
sl <- sqrt(l*(2-l))
arlfix  <- sapply(mu,l=l,c=c,sided="two",xewma.arl)
arlvacl <- sapply(mu,l=l,c=c,sided="two",limits="vacl",xewma.arl)
arlfir  <- sapply(mu,l=l,c=c,hs=c/2,sided="two",limits="fir",xewma.arl)
arlboth <- sapply(mu,l=l,c=c,hs=c/2*sl,sided="two",limits="both",xewma.arl)
round(cbind(mu,arlfix,arlvacl,arlfir,arlboth),digits=1)

## original results are (in Table 1)
## 0.0 477.3* 427.9* 383.4* 286.2*
## 0.5  29.7   20.0   18.6   12.8
## 1.0  12.5    6.5    7.4    3.6
## 1.5   8.1    3.3    4.6    1.9
## 2.0   6.0    2.2    3.4    1.4
## 3.0   4.0    1.3    2.4    1.0
## 4.0   3.1    1.1    1.9    1.0
## * -- the in-control values differ sustainably from the true values!

## Steiner (1999)
## two-sided EWMA control charts with various modifications

## fixed vs. variance adjusted limits

l <- .05
c <- 3
mu <- c(0,.25,.5,.75,1,1.5,2,2.5,3,3.5,4)
arlfix <- sapply(mu,l=l,c=c,sided="two",xewma.arl)
arlvacl <- sapply(mu,l=l,c=c,sided="two",limits="vacl",xewma.arl)
round(cbind(mu,arlfix,arlvacl),digits=1)

## original results are (in Table 2)
## 0.00 1379.0   1353.0
## 0.25  135.0    127.0
## 0.50   37.4     32.5 
## 0.75   20.0     15.6
## 1.00   13.5      9.0
## 1.50    8.3      4.5
## 2.00    6.0      2.8
## 2.50    4.8      2.0
## 3.00    4.0      1.6
## 3.50    3.4      1.3
## 4.00    3.0      1.1.

## fir, both, and Steiner's modification

l <- .03
cfir <- 2.44
cboth <- 2.54
cstein <- 2.55
hsfir <- cfir/2
hsboth <- cboth/2*sqrt(l*(2-l))
mu <- c(0,.5,1,1.5,2,3,4)
arlfir <- sapply(mu,l=l,c=cfir,hs=hsfir,sided="two",limits="fir",xewma.arl)
arlboth <- sapply(mu,l=l,c=cboth,hs=hsboth,sided="two",limits="both",xewma.arl)
arlstein <- sapply(mu,l=l,c=cstein,sided="two",limits="Steiner",xewma.arl)
round(cbind(mu,arlfir,arlboth,arlstein),digits=1)

## original values are (in Table 5)
## 0.0 383.0   384.0   391.0
## 0.5  18.6    14.9    13.8
## 1.0   7.4     3.9     3.6
## 1.5   4.6     2.0     1.8
## 2.0   3.4     1.4     1.3
## 3.0   2.4     1.1     1.0
## 4.0   1.9     1.0     1.0.

## SAS/QC manual 1999
## two-sided EWMA control charts with fixed limits

l <- .25
c <- 3
mu <- 1
print(xewma.arl(l,c,mu,sided="two"),digits=11)

# original value is 11.154267016.
}
\keyword{ts}

\eof
\name{xewma.crit}
\alias{xewma.crit}
\title{Compute critical values of EWMA control charts}
\description{Computation of the critical values (similar to alarm limits)
for different types of EWMA control charts monitoring normal mean.}
\usage{xewma.crit(l,L0,mu0=0,zr=0,hs=0,sided="one",limits="fix",r=40)}
\arguments{
\item{l}{smoothing parameter lambda of the EWMA control chart.}
\item{L0}{in-control ARL.}
\item{mu0}{in-control mean.}
\item{zr}{reflection border for the one-sided chart.}
\item{hs}{so-called headstart (give fast initial response).}
\item{sided}{distinguish between one- and two-sided
two-sided EWMA control chart by choosing \code{"one"} and \code{"two"},
respectively.}
\item{limits}{distinguish between different control limits behavior.}
\item{r}{number of quadrature nodes, dimension of the resulting linear
equation system is equal to \code{r+1} (one-sided) or \code{r}
(two-sided).}
}
\details{
\code{xewma.crit} determines the critical values (similar to alarm limits)
for given in-control ARL \code{L0}
by applying secant rule and using \code{xewma.arl()}.
}
\value{Returns a single value which resembles the critical value
\code{c}.}
\references{
S. V. Crowder (1989), Design of exponentially weighted moving average
schemes, \emph{Journal of Quality Technology 21}, 155-162.
}
\author{Sven Knoth}
\seealso{\code{xewma.arl} for zero-state ARL computation.}
\examples{
l <- .1
incontrolARL <- c(500,5000,50000)
sapply(incontrolARL,l=l,sided="two",xewma.crit,r=30) # accuracy with 30 nodes
sapply(incontrolARL,l=l,sided="two",xewma.crit)      # accuracy with 40 nodes
sapply(incontrolARL,l=l,sided="two",xewma.crit,r=50) # accuracy with 50 nodes

## Crowder (1989)
## two-sided EWMA control charts with fixed limits

l <- c(.05,.1,.15,.2,.25)
L0 <- 250
round(sapply(l,L0=L0,sided="two",xewma.crit),digits=2)

## original values are 2.32, 2.55, 2.65, 2.72, and 2.76.
}
\keyword{ts}

\eof
