\name{ExampleSonde}
\alias{ExampleSonde}
\title{Example Sonde/Balloon/Profiler Data}
\usage{
data(ExampleSonde)
}
\description{
This data was collected by a radiosonde which was carried aloft by a 
weather balloon. This data was collected as part of a special field
program, but the instrument and data formats are \bold{very} common 
in the atmospheric community.\cr

Fundamentally; temperature, pressure, and humidity sensors are bundled 
with a radio transmitter and are either sent aloft on a balloon 
(rawindsonde) or tied to a small parachute and dropped from a plane 
(dropsonde). Either way, the location of the sonde is observed 
(either visually or with GPS), the data from the radio transmitter 
recorded, and the result is an ascii file that contains a header 
(of varying length) with descriptive information followed by a 
table of information.\cr

For rawindsondes, the first entry for the table is actually the 
surface data from the weather station, but the remaining entries 
are from the sonde. Also, the wind is inferred from the position 
of the balloon. Since this is \bold{real} data, there are missing
values -- which arise \emph{all the time}. If you get a profile
without missing values, you can be sure it is synthetic or interpolated.
The datastreme ends when the operator calls it quits. This happens when
a) the balloon breaks, or b) the radio reception has so many dropouts that
is is not worth it anymore. Different fields have different \code{missing}
flags. Where possible, we replace the "standard" missing flags with 
\code{NaN}
}

\format{
  \code{ExampleSonde} is the nominal output and is a mixture of ancillary
data (contained in the first ???? lines of the file), and the instrument 
measures and some derived fields that follow in a fixed-format table
with an unknown number of lines.

\tabular{lllllll}{
Field \tab varname \tab width \tab format \tab description \tab units \tab missing value \cr
 [ 1] \tab time    \tab 6 \tab F6.1 \tab Time                       \tab Seconds          \tab 9999.0 \cr
 [ 2] \tab press   \tab 6 \tab F6.1 \tab Pressure                   \tab Millibars        \tab 9999.0 \cr
 [ 3] \tab temp    \tab 5 \tab F5.1 \tab Dry-bulb Temperature       \tab Degrees C        \tab 999.0 \cr
 [ 4] \tab dewpt   \tab 5 \tab F5.1 \tab Dew Point Temperature      \tab Degrees C        \tab 999.0 \cr
 [ 5] \tab rhum    \tab 5 \tab F5.1 \tab Relative Humidity          \tab Percent          \tab 999.0 \cr
 [ 6] \tab uwind   \tab 6 \tab F6.1 \tab U Wind Component           \tab Meters/Second    \tab 9999.0 \cr
 [ 7] \tab vwind   \tab 6 \tab F6.1 \tab V Wind Component           \tab Meters/Second    \tab 9999.0 \cr
 [ 8] \tab wspd    \tab 5 \tab F5.1 \tab Wind Speed                 \tab Meters/Second    \tab 999.0 \cr
 [ 9] \tab wdir    \tab 5 \tab F5.1 \tab Wind Direction             \tab Degrees          \tab 999.0 \cr
 [10] \tab dz      \tab 5 \tab F5.1 \tab Ascension Rate             \tab Meters/Second    \tab 999.0 \cr
 [11] \tab lon     \tab 8 \tab F8.3 \tab Longitude                  \tab Degrees          \tab 9999.0 \cr
 [12] \tab lat     \tab 7 \tab F7.3 \tab Latitude                   \tab Degrees          \tab 999.0 \cr
 [13] \tab range   \tab 5 \tab F5.1 \tab Variable (see below)       \tab                  \tab 999.0 \cr
 [14] \tab angle   \tab 5 \tab F5.1 \tab Variable (see below)       \tab                  \tab 999.0 \cr
 [15] \tab alt     \tab 7 \tab F7.1 \tab Altitude                   \tab Meters           \tab 99999.0 \cr
 [16] \tab qp      \tab 4 \tab F4.1 \tab QC flag for Pressure       \tab Code (see below) \tab 99.0 \cr
 [17] \tab qt      \tab 4 \tab F4.1 \tab QC flag for Temperature    \tab Code (see below) \tab 99.0 \cr
 [18] \tab qh      \tab 4 \tab F4.1 \tab QC flag for Humidity       \tab Code (see below) \tab 99.0 \cr
 [19] \tab qu      \tab 4 \tab F4.1 \tab QC flag for U Component    \tab Code (see below) \tab 99.0 \cr
 [20] \tab qv      \tab 4 \tab F4.1 \tab QC flag for V Component    \tab Code (see below) \tab 99.0 \cr
 [21] \tab quv     \tab 4 \tab F4.1 \tab QC flag for Ascension Rate \tab Code (see below) \tab 99.0 \cr
}

The \code{missing value} flags are in brackets [].
Fields 13 and 14 are `variable' because depending on the sounding
system the variables used in these positions can vary.\cr

Fields 16 through 21 contain the Quality Control information (flags) 
generated at JOSS. These flags are based on the automated or 
visual checks made.\cr

The JOSS QC flag codes are as follows:\cr
\item{99.0}{Unchecked (QC information is `missing.')  (`UNCHECKED')}
\item{1.0}{Checked, datum seems physically reasonable.  (`GOOD')}
\item{2.0}{Checked, datum seems questionable on physical basis.  (`MAYBE')}
\item{3.0}{Checked, datum seems to be in error.  (`BAD')}
\item{4.0}{Checked, datum is interpolated.  (`ESTIMATED')}
\item{9.0}{Checked, datum was missing in original file.  (`MISSING')}

}

\source{
  \url{http://www.joss.ucar.edu/cgi-bin/codiac/dss?2.151}
}

\seealso{ \code{\link{getsonde}},
          \code{\link{plotsonde}},
          \code{\link{skewt.lines}},
          \code{\link{skewt.points}} }

\examples{
filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)
}

\keyword{datasets}
\eof
\name{PWV}
\alias{PWV}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Precipitable Water Vapor }
\description{
Calculates precipitable water (in mm) up to \code{minp} (minimum pressure)
using dew-point temperature (\code{td}) and temperature (\code{temp}) (both
in degrees C).
}
\usage{
PWV(sonde = NULL, press, td, temp, minp = 400)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{sonde}{ Sonde data read in using \code{getsonde} function. }
  \item{press}{ Pressure (only used if \code{sonde} is NULL) }
  \item{td}{ Dew-point temperature (in deg C) (only used if \code{sonde}
		is NULL) }
  \item{temp}{ Temperature (in deg C) (only used if \code{sonde} is NULL)}
  \item{minp}{ Minimum Pressure }
}
\value{
	single numeric value is returned.
}
\author{ Junhong Wang }

\seealso{ \code{\link{getsonde}}, \code{td.to.q} }

\examples{
# Example 1
filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)

######
# OK, now find precipitable water vapor.

PWV( sample.sonde)

}
\keyword{ manip }% at least one, from doc/KEYWORDS
\eof
\name{RadioSonde internal }

\alias{satlft}
\alias{skewt.data}
\alias{skewtx}
\alias{skewty}
\alias{sonde.data}
\alias{td.to.q}
\alias{wobf}

\title{
  RadioSonde internal and secondary functions
}
\description{
Listed below are supporting functions for the major methods in RadioSonde.
}
\usage{

satlft(thw, p)

skewt.data 

skewtx(temp, ycoord)

skewty(pres) 

sonde.data

td.to.q(temp, td, p)

wobf

}

\keyword{internal}
\keyword{misc}
\eof
\name{getsonde}
\alias{getsonde}
\title{read a radiosonde file to an R dataframe.}
\description{ \code{getsonde} takes a common radiosonde data file and reads it in as an R dataframe. }
\usage{ getsonde(filename, datakey="------", varkey=" Time", unitkey="  sec") }
\arguments{
  \item{filename}{Full path and file name of radiosonde data.}
  \item{datakey}{character string identifying the line \bold{preceeding} the datastream. 
                 The string gets compared to the first N characters of each line, so 
                 all whitespace is important!}
  \item{varkey}{character string identifying the line containing the variable names.
                 The string gets compared to the first N characters of each line, so 
                 all whitespace is important!}
  \item{unitkey}{character string identifying the line containing the units of the variables.
                 The string gets compared to the first N characters of each line, so 
                 all whitespace is important!}
}

\details{
   \code{getsonde} assumes the data is in a very common format for radiosonde data.
   The files are typically ASCII files with a header of unknown length stating such
   things as the location and time of the data collection etc. Appended to this is
   a table of data where each column is a particular quantity and each row corresponds
   to the time of the observation. This typically has a header identifying the
   quantity in each of the columns and is separated from the data by a particular
   character string. The location of this string in the file is crucial in deciphering
   the start of the datastream from the metadata. For much more about the file format,
   go to \url{http://www.cgd.ucar.edu/stats/RadioSonde} or look at the example in 
   \file{RadioSonde/data/ExampleSonde.txt}
}

\value{
   Returns a dataframe with the following items, if present in the input file.
   Note that each of these items can be present in any order in the input file.\cr

  \item{time }{time in seconds since weather balloon launched or recording begins. }
  \item{press}{Pressure in mb. }
  \item{temp }{Dry-bulb Temperature in degrees C. }
  \item{dewpt}{Dew point temperature in degrees C. }
  \item{rhum }{Relative Humidity (Percent).  }
  \item{uwind}{East-West wind component (m/s).  }
  \item{vwind}{North-South wind component (m/s). }
  \item{wspd }{Wind speed (m/s).  }
  \item{wdir }{Wind direction (deg.).  }
  \item{dz   }{Ascension rate of weather balloon (m/s).  }
  \item{lon  }{Longitude of weather balloon.   }
  \item{lat  }{Latitude of weather balloon .  }
  \item{rng  }{Range (see \bold{warning} below)}
  \item{az   }{Azimuth of weather balloon from originating station (see \bold{warning} below) }
  \item{alt  }{Altitude of weather balloon (m).  }
  \item{qp   }{Quality Control (QC) flag for pressure (see note below)}
  \item{qt   }{QC flag for temperature (see note below)}
  \item{qh   }{QC flag for humidity (see note below)}
  \item{qu   }{QC flag for U Component (see note below)}
  \item{qv   }{QC flag for V Component (see note below)}
  \item{quv  }{QC flag for Ascension rate (see note below)}

   If the \code{units} are avialable in the datafile, they are included in 
   the dataframe as the attribute \code{units}.\cr
   
   Rather than throw away the original header information, which usually contains
   valuable metadata, it is included as another attribute: \code{metadata}.\cr
}

\author{ Tim Hoar, Eric Gilleland, Doug Nychka }

\note{
   The Quality Control information (flags): qp, qt, qh, qu, qv, and quv 
   are generated at JOSS are based on the automated or visual checks made.  
   The JOSS QC flags are as follows:\cr
   99.0 means it is unchecked,\cr
   1.0 implies datum seems physically reasonable (good),\cr
   2.0 implies datum seems questionable on physical basis (maybe),\cr
   3.0 implies datum seems to be in error (bad),\cr
   4.0 implies datum is interpolated (estimated), and\cr
   9.0 implies datum was missing in original file (missing).\cr
} % End of note

\section{Warning}{The connotation of fields is not exactly standard. Different
recording systems use these fields in different ways. Hence, they cannot be
automatically interpreted without knowledge of the recording system.}

\seealso{ \code{\link{plotsonde}} , 
          \code{\link{skewt.points}} , 
          \code{\link{skewt.lines}} }

\examples{
# Read a typical radiosonde (ASCII) datafile, headers and all.
# The datakey,varkey,and unitkey arguments come from examining 
# the datafile. The whitespace is important.

filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
datakey  <- "------"
varkey   <- " Time"
unitkey  <- "  sec"
sample.sonde <- getsonde(filename,datakey,varkey,unitkey)
attr(sample.sonde,"units")
attr(sample.sonde,"metadata")
plotsonde(sample.sonde,title=filename)
#
# Read the same radiosonde dataset, but ignore the units ... 
#
datakey  <- "------"
varkey   <- " Time"
sample.sonde <- getsonde(filename,datakey,varkey)
attr(sample.sonde,"metadata")
}
\keyword{ manip }        % at least one, from doc/KEYWORDS
\eof
\name{plotsonde}
\alias{plotsonde}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Creates a SKEW-T, log p diagram for a radiosonde dataframe.}
\description{
   Creates a SKEW-T, log p diagram with dry-bulb temperature and 
   dewpoint temperature traces versus (log) pressure. Optionally plots
   the vertical wind profile using wind barbs.
}
\usage{ plotsonde(dataframe, skewT=T, winds=T, site="", title="",
           windplot=NULL, s=3, col=c(1, 2), ... ) }

\arguments{
  \item{dataframe}{ R dataframe of a radiosonde dataset 
                (as created by \code{getsonde}) }
  \item{skewT}{ Logical value.  If false, will not plot the 
	        SKEW-T, log p diagram. }
  \item{winds}{ Logical value.  If false, will not plot the
		winds profile. }
  \item{site}{ Optional character to add to plot title. }
  \item{title}{ Title for plot (must be a character value). }
  \item{windplot}{ 4X1 numeric vector that gives the position of wind profile 
		   plot (next to SKEW-T diagram).  Only used if it is desired
		   to have this plot in a different position.  Generally, the 
		   default NULL is used and the placement is chosen 
		   automatically.  }
  \item{s}{ Size of winds profile plot.  }
  \item{col}{ Usual plotting parameter. }
  \item{...}{ Other optional plotting parameters. }
}
\value{
  No value returned.  Creates a plot.
}
\references{ 
	1.  Department of Defense, 1969, "USAF SKEW-T, log p DIAGRAM,"
	DOD-WPC-9-16-1, Aeronautical Chart and Information Center,
	United States Air Force, St. Louis, Missouri 63118.

	2.  List, R.J. (editor), 1958, \emph{Smithsonian Meteorological Tables},
	Smithsonian Institute, Washington, D.C.

	3.  Nordquist, W.S., 1973, "Numerical Approximations of Selected
	Meteorological Parameters for Cloud Physics Problems," ECOM-5475,
	Atmospheric Sciences Laboratory, US Army Electronics Command,
	White Sands Missile Range, New Mexico 88002.

	4.  Stipanuk, G.S., 1973, "Algorithms for Generating a SKEW-T, log p
	Diagram and Computing Selected Meteorological Quantities," American
	Sciences Laboratory, US Army Electronics Command, White Sands Missile
	Range, New Mexico 88002.

	5. \url{http://www.atd.ucar.edu/dir_off/tc_corr/index.html}

	6. \url{http://weather.unisys.com/upper_air/skew/details.html}
}
\author{ Tim Hoar, Eric Gilleland, Doug Nychka}
\seealso{ \code{\link{getsonde}}, 
          \code{\link{skewt.axis}} ,
          \code{\link{skewt.lines}} ,
          \code{\link{skewt.points}} }

\examples{
# Example 1
filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)
plotsonde(sample.sonde)
}
\keyword{ hplot }% at least one, from doc/KEYWORDS
\keyword{ aplot }% __ONLY ONE__ keyword per line
\eof
\name{plotwind}
\alias{plotwind}
\title{ Winds Profile Plot }
\description{ Creates a wind profile in the standard atmospheric 
              notation. Each full barb = 10 m/s, half barb = 5 m/s, 
              and a filled triangle for 50 m/s.  The maximum wind 
              speed that can be plotted without amending the program 
              is 65 m/s. }

\usage{ plotwind(dataframe, size=5, ylim=c(1050, 100), legend=FALSE) }

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{dataframe}{ Data frame for sounding data, must have components for 
                    wind speed \code{wspd}, wind direction \code{dir}, 
                    and pressure \code{press}.}
  \item{size}{change the thickness of the plotted lines}
  \item{ylim}{plot extent, in units of pressure}
  \item{legend}{explains wind barb strategy}
}

\details{ the dataframe must have components for wind speed \code{wspd}, 
          wind direction \code{dir}, and pressure \code{press}.  
          Missing values may be coded as either NA or 999. and are 
          not plotted.\cr
          The standard atmospheric wind sybmol is a vector of fixed length 
          with barbs proportional to wind speed. A full barb for each 10 m/s,  
          half barbs for 5 m/s and a triangular barb for 50 m/s. }

\value{None -- creates a plot.}

\author{ Eric Gilleland, Tim Hoar, Doug Nychka }

\seealso{ \code{\link{getsonde}}, \code{\link{plotsonde}} }

\examples{
# Example 1:
filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)
plotwind(sample.sonde)
# Example 2:
plotwind(sample.sonde, size = 7, legend=TRUE)
}
\keyword{ hplot }          % at least one, from doc/KEYWORDS
\eof
\name{sample.sonde}
\alias{sample.sonde}
\non_function{}
\title{Example R dataframe for radiosonde observations}
\usage{data("sample.sonde")}
\description{
The \code{sample.sonde} data frame has 461 rows and 21 columns.
The columns are different physical measurements made during the balloon's
ascent.
}
\format{
    This data frame contains the following named components:

    \tabular{ll}{
         time   \tab Time in seconds from the ballon's release, a numeric vector. \cr
         press  \tab Pressure (mb), a numeric vector \cr
         temp   \tab Dry-bulb Temperature (deg. C), a numeric vector \cr
         dewpt  \tab Dew point temperaure (deg. C), a numeric vector \cr
         rh     \tab Relative humidity (\%), a numeric vector \cr
         uwind  \tab East-West component of the wind, a numeric vector \cr
         vwind  \tab North-South component of the wind, a numeric vector \cr
         wspd   \tab wind speed  (m/s), a numeric vector \cr
         dir    \tab Wind direction, a numeric vector \cr
         dz     \tab Ascension rate (m/s), a numeric vector \cr
         lon    \tab Longitude, a numeric vector \cr
         lat    \tab Latitude, a numeric vector \cr
         rng    \tab range, a numeric vector \cr
         az     \tab azimuth (angle along horizon), a numeric vector \cr
         alt    \tab altitude (m), a numeric vector \cr
         qp     \tab QC flag for pressure, a numeric vector \cr
         qt     \tab QC flag for temperature, a numeric vector \cr
         qh     \tab QC flag for humidity, a numeric vector \cr
         qu     \tab QC flag for U component, a numeric vector \cr
         qv     \tab QC flag for V component, a numeric vector \cr
         quv    \tab QC flag for ascension rate, a numeric vector \cr
    }
}
\note{Quality Control (QC) flags are generated locally at JOSS based on either
automated or visual checks made.\cr
999.0 implies the QC information is missing, \cr
1.0 implies datum seems physically reasonable (good),\cr
2.0 implies datum questionable on a physical basis (maybe),\cr
3.0 implies datum seems to be in error (bad), \cr
4.0 implies datum interpolated (estimated), \cr
9.0 implies datum missing in original file.  }
\seealso{\code{\link{getsonde}} ,
         \code{\link{skewt.lines}},
         \code{\link{skewt.points}},
         \code{\link{plotsonde}} }
\keyword{datasets}
\eof
\name{skewt.axis}
\alias{skewt.axis}
\title{ Draws a SKEW-T, log p axis. }
\description{
Draws a SKEW-T, log p axis. This is the standard axis for interpreting atmospheric
sounding profiles like those collected by the instrument carried aloft by a 
weather balloon (\emph{radiosonde}). Use \code{skewt.lines} or \code{skewt.points} 
to layer information on top of the skewt axis.
}
\usage{
skewt.axis(BROWN="brown", GREEN="green", redo=F, ... )
}
\arguments{
  \item{BROWN}{ Color of lines for both Temperature (solid skewed) and 
                Pressure (dashed horizontal)}
  \item{GREEN}{ Color of lines for dry adiabats (solid) and constant mixing
ratio (dashed) }
  \item{redo}{ flag to generate the adiabats, should \code{skewt.data} become
               corrupt. The adiabats are the result of an iterative process
               and to make smooth curves, you need a lot of points. Hence this
               is time-consuming, so [FALSE] is the preferred value.}
  \item{\dots}{ The usual \code{plot} parameters. }
}
\details{ Radiosondes record temperature, humidity and winds. 
          They can be lifted by weather balloons, dropped from aircraft, 
          there is even something called a glidersonde. The data collected by 
          radiosondes are plotted versus pressure/height to give details on the 
          vertical structure of the atmosphere.  The type of plot is called a 
          SKEW-T, log p diagram.\cr
          \file{skewt.axis} creates the traditional axis for a SKEW-T, log p 
          diagram, including moist and dry adiabats, etc ...\cr
          Generating the necessary plot data is time-consuming, so there is an
          option to read from a specific dataset that Doug will change.
}
\value{ Returns the \code{par()$plt} values, which are used by 
        \code{plotsonde} if you choose to plot the wind profile 
        adjacent to the skewt axis. }

\author{ Tim Hoar }
\seealso{ \code{\link{getsonde}}, 
          \code{\link{plotsonde}}, 
          \code{\link{skewt.lines}}, 
          \code{\link{skewt.points}}  }

\examples{
# draw a background, then
#    draw the temperature (with a solid line) in color 6
#    draw the dewpoint in color 7
#    overlay the temperature observations in a different color
#    you get the point ...
#
filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)
skewt.axis( mar=c(5.1, 1.1, 2.1, 5.1) )
skewt.lines( sample.sonde$temp,  sample.sonde$press, col = 6)
skewt.lines( sample.sonde$dewpt, sample.sonde$press, col = 7)
skewt.points(sample.sonde$temp,  sample.sonde$press, col = 3)
skewt.points(sample.sonde$dewpt, sample.sonde$press, col = 4)
#
# Changing the moist adiabats: you must edit the \code{skewt.axis} function
# directly and then capture the output in \code{skewt.data} to be used in 
# subsequent calls.
skewt.data <- skewt.axis(redo=TRUE)
skewt.axis() 
skewt.axis() 
}
\keyword{ hplot }
\eof
\name{skewt.lines}
\alias{skewt.lines}
\title{Overlays data on a SKEW-T, log p axis}
\description{Overlays observations as lines on a SKEW-T, log p axis
             (as created by \code{skewt.axis}). }
\usage{ skewt.lines(temp, pressure, ...) }
\arguments{
  \item{temp}{ Temperature in degrees C. }
  \item{pressure}{ Pressure in millibars }
  \item{...}{ Any graphical arguments }
}
\details{\code{skewt.lines} overlays observations on a SKEW-T, log p axis}

\seealso{ \code{\link{skewt.points}}, 
          \code{\link{skewt.axis}}, 
          \code{\link{plotsonde}} }

\examples{
# draw a background, then
#    draw the temperature (with a solid line) in color 6
#    draw the dewpoint in color 7
#    overlay the temperature observations in a different color
#    you get the point ...

filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
print(filename)
sample.sonde <- getsonde(filename)
skewt.axis()
skewt.lines( sample.sonde$temp,  sample.sonde$press, col = 6)
skewt.lines( sample.sonde$dewpt, sample.sonde$press, col = 7)
skewt.points(sample.sonde$temp,  sample.sonde$press, col = 3)
skewt.points(sample.sonde$dewpt, sample.sonde$press, col = 4)
}

\keyword{ aplot }
\eof
\name{skewt.points}
\alias{skewt.points}
\title{Overlays data on a SKEW-T, log p axis}
\description{Overlays observations on a SKEW-T, log p axis 
             (as created by \code{skewt.axis}). }

\usage{ skewt.points(temp, pressure, ...) }

\arguments{
  \item{temp}{ Temperature in degrees C. }
  \item{pressure}{ Pressure in millibars }
  \item{...}{ Any graphical arguments }
}

\details{\code{skewt.points} overlays observations on a SKEW-T, log p axis}

\seealso{ \code{\link{skewt.axis}}, 
          \code{\link{skewt.lines}}, 
          \code{\link{plotsonde}} }

\examples{
# draw a background, then
#    draw the temperature (with a solid line) in color 6
#    draw the dewpoint in color 7
#    overlay the temperature observations in a different color
#    you get the point ...

filename <- system.file("exampleData", "ExampleSonde.txt", package = "RadioSonde")   
sample.sonde <- getsonde(filename)
skewt.axis()
skewt.lines( sample.sonde$temp,  sample.sonde$press, col = 6)
skewt.lines( sample.sonde$dewpt, sample.sonde$press, col = 7)
skewt.points(sample.sonde$temp,  sample.sonde$press, col = 3)
skewt.points(sample.sonde$dewpt, sample.sonde$press, col = 4)
}

\keyword{ aplot }
\eof
\name{station.symbol}
\alias{station.symbol}
\title{ Adds a meteorological surface station annotation to a plot. }
\description{ Adds a meteorological surface station annotation at the given 
              coordinates. The annotation includes speed and direction of the 
              wind, temperature, pressure, dewpoint ...  to a given plot. }

\usage{
station.symbol(cx, cy, direction = 0, speed = 0, fill = 0, 
		color = "green", temp = NULL, press = NULL, 
		dewpt = NULL, circle=T, cex = 2)
}

\arguments{
  \item{cx}{x coordinate for location of the annotation.}
  \item{cy}{y coordinate for the annotation.}
  \item{direction}{Wind direction.}
  \item{speed}{Wind speed.}
  \item{fill}{Fill for visibility: 0 = clear skies, 1 = 25\% obscured, 
              2 = 50\% obscured, 3 = 75\% obscured, 4 = no visibility }
  \item{color}{Fill color. Used only when fill > 0}
  \item{temp}{Temperature value to plot symbol (must be a scalar).
              If NULL then no temperature value will be annotated.}
  \item{press}{Pressure value to plot symbol (must be a scalar).
              If NULL then no pressure value will be annotated.}
  \item{dewpt}{Dewpoint Temperature value to plot symbol (must be a scalar).
               If NULL then no dewpoint value will be annotated.}
  \item{circle}{If TRUE, will plot the usual station symbol with a circle 
                at its base.}
  \item{cex}{Usual plotting parameter. }
}

\value{ Adds to a plot. }

\author{ Doug Nychka, Eric Gilleland }

\seealso{ \code{\link{plotsonde}}, 
          \code{\link{plotwind}}, 
          \code{\link{skewt.axis}}
}

\examples{
   plot(0:1, 0:1, type="n")
   station.symbol(0.5, 0.5, direction=75, speed=30, fill=3, 
                  temp=31, press=987, dewpt=26)
}

\keyword{ aplot }
\eof
