\name{BIC}
\docType{genericFunction}
\alias{BIC}
\alias{BIC,logLik-method}
\title{Bayesian Information Criterion}
\description{
  This generic function calculates the Bayesian information criterion,
  also known as Schwarz's Bayesian criterion (SBC), for one or several
  fitted model objects for which a log-likelihood value can be obtained,
  according to the formula \eqn{-2 \mbox{log-likelihood} + n_{par}
    \log(n_{obs})}{-2*log-likelihood + npar*log(nobs)}, where
  \eqn{n_{par}}{npar}  represents the
  number of parameters and \eqn{n_{obs}}{nobs} the number of
  observations in the fitted model. 
}
\usage{
BIC(object, \dots)
}
\arguments{
  \item{object}{An object of a suitable class for the BIC to be
    calculated - usually a \code{\link[base:logLik]{logLik}} object
    created by a call to the \code{\link[base:logLik]{logLik}} generic.
  }
  \item{\dots}{Some methods for this generic function may take
    additional, optional arguments.  At present none do.}
}
\value{
  if just one object is provided, returns a numeric value with the
  corresponding BIC; if more than one object are provided, returns a
  \code{data.frame} with rows corresponding to the objects and columns
  representing the number of parameters in the model (\code{df}) and the
  BIC.
}
\references{
  Schwarz, G. (1978) "Estimating the Dimension of a Model", Annals of
  Statistics, 6, 461-464.
}
%\author{}
%\note{}
\seealso{\code{\link[base:logLik]{logLik}}, \code{\link{AIC}}}
%\examples{}
\keyword{models}

\eof
\name{EMstepsGets}
\docType{genericFunction}
\alias{EMsteps<-}
\alias{EMsteps<-,reStruct,list-method}
\title{Perform EM iterations}
\description{
  The object \code{x} is updated through EM or ECME iterations.  The
  number of iterations and whether verbose output should be produced are
  specified in the list passed as \code{value}.
}
\usage{
EMsteps(x) <- value
}
\arguments{
  \item{x}{An object of a suitable class to be updated according to the
    EM or ECME algorithm.  Generally an \code{reStruct} object.}
  \item{value}{A list that contains components \code{EMiter}, a
    non-negative integer specifying the number of iterations, and
    \code{EMverbose}, a logical value that, when \code{TRUE}, requests
    verbose output.}
}
\value{
  The object \code{x}, updated.
}
%\references{}
%\author{}
%\note{}
\seealso{\code{\link{reStruct-class}}, \code{\link{lmeControl}}}
\examples{}
\keyword{models}% at least one, from doc/KEYWORDS
\eof
\name{EMupdateGets}
\docType{genericFunction}
\alias{EMupdate<-}
\alias{EMupdate<--methods}
\alias{EMupdate<-,pdDiag,numeric,matrix-method}
\alias{EMupdate<-,pdIdent,numeric,matrix-method}
\alias{EMupdate<-,pdLogChol,numeric,matrix-method}
%\alias{EMupdate<-,pdScalar,numeric,matrix-method}
\alias{EMupdate<-,lmeLevel,missing,matrix-method}
\title{Update an object in the EM algorithm}
\description{
  A generic replacement function used in the EM and
  ECME optimization algorithms for linear mixed-effects models.
}
\usage{
EMupdate(x, nlev) <- value
}
\arguments{
  \item{x}{an \code{lmeStruct} or \code{pdMat} object}
  \item{nlev}{integer: the number of levels of the grouping factor
    corresponding to the random-effects structure}
  \item{value}{a matrix with the triangular factor from an
    orthogonal-triangular decomposition of the modal values of the
    random-effects and their precision matrices}
}
%\details{}
\value{
  an object of the same class as \code{x}
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
\examples{
library(lme4)
data(Oxboys, package = "nlme")
m3 <- pdLogChol(~ age)
m3
as(m3, 'pdmatrix') <- crossprod(model.matrix(formula(m3), Oxboys))
show(m3)
EMupdate(m3, length(levels(Oxboys$Subject))) <- diag(2)
show(m3)
}
\keyword{models}
\eof
\name{GLMM}
\alias{GLMM}
\alias{GLMM,ANY,ANY,missing,ANY-method}
\alias{GLMM,missing,ANY,groupedData,ANY-method}
\alias{GLMM,formula,ANY,groupedData,missing-method}
\alias{GLMM,formula,ANY,ANY,formula-method}
\alias{GLMM,formula,ANY,ANY,list-method}
\title{
Fit Generalized Linear Mixed Models via PQL
}
\description{
Fit a GLMM model with multivariate normal random effects, using
Penalized Quasi-Likelihood.
}
\usage{
GLMM(formula, family, data, random, control, niter, method, verbose, \dots)
}
\arguments{
  \item{formula}{
    a two-sided linear model formula giving fixed-effects part of the model.
  }
  \item{family}{
    a GLM family, see \code{\link{glm}}.
  }
  \item{random}{
    A formula or list of formulae describing the random effects.
  }
  \item{data}{
    an optional data frame used as the first place to find
    variables in the formulae.
  }
  \item{control}{
    an optional argument to be passed to \code{lme}.
  }
  \item{niter}{
    maximum number of PQL iterations.  Default is 20.
  }
  \item{method}{
    character: Estimation method to be used.  Possible values are
    \code{"PQL"}, the default, or \code{"Laplace"}.
    \code{"PQL"} provides penalized quasi-likelihood estimates.
    \code{"Laplace"} provides PQL
    estimation followed by optimization of the second-order Laplacian
    approximation to the marginal log-likelihood.
  }
  \item{verbose}{
    logical: print out record of iterations?  Default is \code{FALSE}.
  }
  \item{\dots}{
    Optional further arguments such as \code{subset} and \code{na.action}.
  }
}
\value{
A object of class \code{"glmm"}: see \code{\link{reStruct-class}}.
}
\references{
Schall, R. (1991) Estimation in generalized linear models with
random effects.
\emph{Biometrika}
\bold{78}, 719--727.


Breslow, N. E. and Clayton, D. G. (1993) Approximate inference in
generalized linear mixed models.
\emph{Journal of the American Statistical Association}
\bold{88}, 9--25.


Wolfinger, R. and O'Connell, M. (1993) Generalized linear mixed models: a
pseudo-likelihood approach.
\emph{Journal of Statistical Computation and Simulation}
\bold{48}, 233--243.
}
\seealso{
  \code{\link{lme}}
}
\examples{
library(lme4)
data(guImmun)
fm1 = GLMM(immun ~ kid2p + mom25p + ord + ethn +
                  momEd + husEd + momWork + rural + pcInd81,
          data = guImmun, family = binomial,
          random = ~1|comm/mom)
summary(fm1)
fm2 = GLMM(immun ~ kid2p + mom25p + ord + ethn +
                  momEd + husEd + momWork + rural + pcInd81,
          data = guImmun, family = binomial,
          random = ~1|comm, method = 'Laplace')
summary(fm2)
}
\keyword{models}
\eof
\name{Hsb82}
\alias{Hsb82}
\docType{data}
\title{High School and Beyond - 1982}
\description{
  Data from the 1982 study ``High School and Beyond''.
}
\usage{data(Hsb82)}
\format{
  A data frame with 7185 observations on students including the
  following 8 variables.
  \describe{
    \item{school}{an ordered factor designating the school that the
      student attends.}
    \item{minrty}{a factor with levels}
    \item{sx}{a factor with levels \code{Male} and \code{Female}}
    \item{ses}{a numeric vector of socio-economic scores}
    \item{mAch}{a numeric vector of Mathematics achievement scores}
    \item{meanses}{a numeric vector of mean \code{ses} for the school}
    \item{sector}{a factor with levels \code{Public} and \code{Catholic}}
    \item{cses}{a numeric vector of centered \code{ses} values where the
    centering is with respect to the \code{meanses} for the school.}
  }
}
\details{
  Each row in this data frame contains the data for one student.
}
%\source{}
\references{
  Raudenbush, Stephen and Bryk, Anthony (2002), \emph{Hierarchical
    Linear Models: Applications and Data Analysis Methods}, Sage
  (chapter 4).
}
\examples{
data(Hsb82)
summary(Hsb82)
}
\keyword{datasets}
\eof
\name{LMEgradient}
\docType{genericFunction}
\alias{LMEgradient}
\alias{LMEgradient-methods}
\alias{LMEgradient,pdDiag,matrix,numeric-method}
\alias{LMEgradient,pdIdent,matrix,numeric-method}
\alias{LMEgradient,pdLogChol,matrix,numeric-method}
\alias{LMEgradient,pdNatural,matrix,numeric-method}
\alias{LMEgradient,lmeLevel,matrix,missing-method}
\alias{LMEgradient,lmeLevel,missing,missing-method}
\alias{LMEhessian}
\alias{LMEhessian-methods}
\alias{LMEhessian,pdDiag,matrix,array,numeric-method}
\alias{LMEhessian,pdLogChol,matrix,array,numeric-method}
\alias{LMEhessian,lmeLevel,missing,missing,missing-method}
\alias{LMEhessian,reStruct,missing,missing,missing-method}
\title{The gradient and Hessian in lme optimization-method}
\description{
  The LMEgradient and LMEhessian generic functions return the gradient
  and the Hessian of the
  log-likelihood or log-restricted-likelihood in an lme model with
  respect to the parameters of the object represented by \code{x}.
}
\usage{
LMEgradient(x, A, nlev)
LMEhessian(x, A, H, nlev)
}
\arguments{
  \item{x}{a parameterized component of an lme model, usually the
    precision matrix of an \code{lmeLevel}.  Such precision matrices
    inherit from the \code{pdMat} class.}
  \item{A}{an upper triangular matrix with the same number of columns as
    the matrix represented by \code{x}}
  \item{H}{an array of four dimensions with each dimension same as the
    number of columns of the matrix represented by \code{x}}
  \item{nlev}{integer: the number of levels of the grouping factor
    corresponding to the random-effects structure}
}
%\details{}
\value{
  LMEgradient returns
  a numeric vector of length \code{length(coef(x))}.

  LMEhessian returns
  a symmetric matrix with number of columns \code{length(coef(x))}.
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdMat-class}}, \code{\link{lmeLevel-class}}}
\examples{
library(lme4)
data(Oxboys, package = "nlme")
m3 <- pdLogChol(~ age)
as(m3, 'pdmatrix') <- crossprod(model.matrix(formula(m3), Oxboys))
LMEgradient(m3, diag(2), 19)
}
\keyword{models}
\eof
\name{LMEoptimizeGets}
\docType{genericFunction}
\alias{LMEoptimize<-}
\alias{LMEoptimize<-,reStruct,list-method}
\alias{LMEoptimize<-,glmm,list-method}
\title{Optimize linear mixed-effects}
\description{
  Update the parameters in an object by using an optimizer such as
  \code{optim} or \code{nlm} to maximize the log-likelihood.
}
\usage{
LMEoptimize(x) <- value
}
\arguments{
  \item{x}{An object of a suitable class, such as \code{reStruct} to be
    optimized}
  \item{value}{A list that contains components \code{msMaxIter}, a
    non-negative integer specifying the maximum number of iterations, 
    \code{msVerbose}, a logical value that, when \code{TRUE}, requests
    verbose output, \code{msTol}, a numeric tolerance for declaring
    convergence, and \code{optimizer}, a character vector whose first
    element is \code{"optim"} or \code{"nlm"}}
}
%\details{}
\value{
  The object \code{x}, updated.
}
%\references{}
%\author{}
%\note{}
\seealso{\code{\link{reStruct-class}}, \code{\link{lmeControl}}}
%\examples{}
\keyword{models}
\eof
\name{Names}
\docType{genericFunction}
\alias{Names}
\alias{Names<-}
\alias{Names-methods}
\alias{Names<--methods}
\alias{Names,ANY-method}
\alias{Names<-,ANY-method}
\title{Extract or assign Names}
\description{
  Generic functions to extract or assign names of an object, usually an
  object of a class that inherits from the \code{pdMat} class.  This
  function is for back-compatibility only.
}
\usage{
Names(object, ...)
Names(object) <- value
}
\arguments{
  \item{object}{An object with names - usually an object that inherits
    from the \code{pdMat} class.}
  \item{value}{A character vector of the same length as \code{Names(object)}}
  \item{\dots}{Some methods for this generic function may use
    additional, optional arguments.}
}
\value{
  The names assigned to \code{object}.
}
\author{Douglas Bates \email{bates@stat.wisc.edu} and
  Saikat DebRoy \email{saikat@stat.wisc.edu}}
\seealso{\code{\link{names}}, \code{\link{names<-}}}
\examples{
}
\keyword{attribute}

\eof
\name{VarCorr-class}
\docType{class}
\alias{VarCorr-class}
\alias{show,VarCorr-method}
\title{Class "VarCorr"}
\description{The variance and correlation information for a
  random-effects model.}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("VarCorr", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{scale}:}{The estimated standard deviation of the lowest
      level noise term in the model.}
    \item{\code{reSumry}:}{Object of class \code{"list"}, a list
      containing the summary of the positive-definte matrices for the
      random-effects levels.}
    \item{\code{useScale}:}{Object of class \code{"logical"}, if TRUE
      the slot \code{scale} is the estimated within-groups standard deviation.}
  }
}
\section{Methods}{
  \describe{
    \item{\link[methods:show]{show}}{\code{signature(object =
	"VarCorr")}: show the object.}
  }
}
\keyword{classes}
\eof
\name{VarCorr}
\docType{genericFunction}
\alias{VarCorr}
\title{Extract variance and correlation components}
% \arguments{
%  \item{x}{a fitted model object, usually an object inheriting from
%      class \code{lme}. 
%  }
% }
\description{
  This function calculates the estimated variances, standard
  deviations, and correlations between the random-effects terms in a
  linear mixed-effects model, of class \code{lme}, or a generalized
  linear mixed-effects model, of class \code{glmm}. When appropriate,
  the within-group error variance and standard deviation are also
  calculated.
}
\value{
  an object of class \code{VarCorr}.
}
\author{Saikat DebRoy \email{saikat@stat.wisc.edu} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{GLMM}}, \code{\link{VarCorr-class}}}
\examples{
data(bdf, package = "nlme")
fm <- lme(langPOST ~ IQ.ver.cen + avg.IQ.ver.cen, data = bdf,
          random = ~ IQ.ver.cen | schoolNR)
VarCorr(fm)
}
\keyword{models}
\eof
\name{coefGets}
\docType{genericFunction}
\alias{coef<-}
\alias{coef<--methods}
\alias{coef<-,lmeLevel,numeric-method}
\alias{coef<-,pdCompSymm,numeric-method}
\alias{coef<-,pdDiag,numeric-method}
\alias{coef<-,pdIdent,numeric-method}
\alias{coef<-,pdLogChol,numeric-method}
\alias{coef<-,pdNatural,numeric-method}
%\alias{coef<-,pdScalar,numeric-method}
\alias{coef<-,reStruct,numeric-method}
\title{Assign parameters}
\description{
  A generic function to assign the parameter vector in an object.  For
  historical reasons parameters are accessed with \code{coef} and
  assigned with \code{coef<-}.
}
\usage{
coef(object) <- value
}
\arguments{
  \item{object}{An object that has a numeric parameter vector accessed
    by \code{coef}.}
  \item{value}{A numeric vector of the same length as \code{coef(object)}.}
}
%\details{}
\value{
  \code{object} with an updated parameter vector
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
\examples{
m1 <- pdLogChol(~ age)
coef(m1) <- rnorm(3)
m1
}
\keyword{manip}
\eof
\name{coerce-methods}
\docType{methods}
\alias{coerce-methods}
\alias{coerce,lme,reStruct-method}
\alias{coerce,matrix,pdmatrix-method}
\alias{coerce,pdCompSymm,pdmatrix-method}
\alias{coerce,pdDiag,pdmatrix-method}
\alias{coerce,pdIdent,corrmatrix-method}
\alias{coerce,pdIdent,pdmatrix-method}
\alias{coerce,pdLogChol,pdmatrix-method}
\alias{coerce,pdMat,corrmatrix-method}
\alias{coerce,pdMat,matrix-method}
\alias{coerce,pdNatural,corrmatrix-method}
\alias{coerce,pdNatural,pdmatrix-method}
\alias{coerce,pdfactor,pdmatrix-method}
\alias{coerce,pdmatrix,corrmatrix-method}
\alias{coerce,pdmatrix,pdfactor-method}
\alias{coerce<-,lme,reStruct-method}
\alias{coerce<-,pdDiag,pdmatrix-method}
\alias{coerce<-,pdIdent,pdmatrix-method}
\alias{coerce<-,pdLogChol,pdmatrix-method}
\alias{coerce<-,pdMat,matrix-method}
\alias{coerce<-,pdMat,pdfactor-method}
\alias{coerce<-,pdNatural,corrmatrix-method}
\alias{coerce<-,pdNatural,pdmatrix-method}
\alias{coerce<-,pdmatrix,pdfactor-method}
\title{Coercion methods and assignments}
\description{Methods for \code{coerce} or \code{"coerce<-"} are
  generated from \code{setAs} relationships between classes.  Typically
  these methods are invoked indirectly.}
\keyword{manip}
\keyword{methods}
\eof
\name{corFactor}
\docType{genericFunction}
\alias{corFactor}
\title{Extract the correlation factor}
\description{
  A generic function to extract the factor of the correlation matrix of
  the object represented by \code{object}.
}
\usage{
corFactor(object, ...)
}
\arguments{
  \item{object}{an object that can be coerced to a correlation matrix factor}
  \item{\dots}{some methods for this generic may take additional,
    optional arguments}
}
%\details{}
\value{
  An object of class \code{corrfactor}.
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
\examples{
}
\keyword{models}
\eof
\name{corMatrix}
\docType{genericFunction}
\alias{corMatrix}
\alias{corMatrix,pdMat-method}
\title{Generate the correlation matrix}
\description{
  This generic function generates a correlation matrix from an object of
  a suitable class.

  This function is deprecated.  Use coersion to the \code{corrmatrix}
  class instead.
}
\usage{
corMatrix(object, ...)
} 
\arguments{
  \item{object}{An object of a suitable class}
  \item{\dots}{Some methods for this generic take additional, optional
    arguments.}
}
%\details{}
\value{
  An object of class \code{\link[lme4:corrmatrix-class]{corrmatrix}}.
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{corrmatrix-class}}}
%\examples{}
\keyword{models}
\eof
\name{corrmatrix-class}
\docType{class}
\alias{corrmatrix-class}
\title{Class "corrmatrix"}
\description{A class of correlation matrices}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("corrmatrix", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{"matrix"} - the actual
      correlation matrix}
    \item{\code{stdDev}:}{Object of class \code{"numeric"} giving the
      corresponding standard deviations }
  }
}
\section{Extends}{
Class \code{"matrix"}, from data part.
Class \code{"structure"}, by class "matrix".
Class \code{"array"}, by class "matrix".
Class \code{"vector"}, by class "matrix".
}
\section{Methods}{
  \describe{
    \item{coerce}{\code{signature(from = "pdIdent", to = "corrmatrix")}: ... }
  }
}
\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\examples{
}
\keyword{classes}
\eof
\name{family-class}
\docType{class}
\alias{family-class}
\title{Class "family"}
\description{
  Family objects describe the error distributions and link functions
  used in generalized linear models and its extensions. It is used by
  functions such as \code{\link[base:glm]{glm}} and 
  \code{\link{GLMM}}.
}
\section{Objects from the Class}{This is an S3 style class. No objects
  may be created from it using \code{new}.}

\section{Extends}{
Class \code{"oldClass"}, directly.
}
\section{Methods}{
  No methods defined with class "family" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
% $Id: fixed.effects.Rd,v 1.3 2003/07/20 14:55:53 bates Exp $
\name{fixed.effects}
\docType{genericFunction}
\alias{fixed.effects}
\alias{fixef}
\alias{fixef<-}
\alias{fixef,ANY-method}
\alias{fixef,reStruct-method}
\title{Extract Fixed Effects}
\description{
  A generic function to extract the fixed effects.
}
\usage{
fixed.effects(object, \dots)
fixef(object, \dots)
fixef(object) <- value
}
\arguments{
 \item{object}{any fitted model object from which fixed effects
   estimates can be extracted.}
 \item{value}{A numeric vector of the same length as \code{fixef(object)}.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
%\seealso{}
\examples{
## see the method function documentation
}
\keyword{models}
\eof
\name{formulas}
\title{Utilities for formulas}
\alias{getCovariate}
\alias{getCovariateFormula}
\alias{getGroups}
\alias{getGroups,data.frame,formula,ANY,ANY,ANY-method}
\alias{getGroupsFormula}
\alias{getGroupsFormula,ANY-method}
\alias{getResponse}
\alias{getResponseFormula}
\alias{subFormula}
\alias{splitFormula}
\description{
  These generic functions extract sub-formulas or values of components
  from a formula/data pair.
}
\usage{
getCovariate(object, form, data)
getCovariateFormula(object)
getGroups(object, form, level, data, sep)
getGroupsFormula(object, asList, sep)
getResponse(object, form)
getResponseFormula(object)
subFormula(form, pos)
splitFormula(form, sep)
}
\arguments{
  \item{object}{a formula or an object that can provide a formula}
  \item{form}{an optional formula with a conditioning expression on its
    right hand side (i.e. an expression involving the \code{|}
    operator). Defaults to \code{formula(object)}.}
  \item{level}{a positive integer vector with the level(s) of grouping to
    be used when multiple nested levels of grouping are present. This
    argument is optional for most methods of this generic function and
    defaults to all levels of nesting.}
  \item{data}{a data frame in which to interpret the variables named in
    \code{form}. Optional for most methods.}
  \item{asList}{an optional logical value. If \code{TRUE} the returned
    value with be a list of formulas; else, if \code{FALSE} the returned
    value will be a one-sided formula. Defaults to \code{FALSE}.}
  \item{sep}{character, the separator to use between group levels when
    multiple levels are collapsed.  Defaults to \code{'/'}.}
  \item{pos}{the integer index of the component of the formula to
    return.  Defaults to \code{2}.}
}
\value{
  \code{getGroupsFormula} returns a one-sided formula, or a list of
  one-sided formulas, giving the grouping structure associated with
  \code{formula(object)}. If no conditioning expression is present in
  \code{formula(object)} a \code{NULL} value is returned.
}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\seealso{}
\examples{
form = resp ~ cov1 + cov2 | grp1/grp2
getResponseFormula(form)
getCovariateFormula(form)
getGroupsFormula(form)
getGroupsFormula(form, asList = TRUE)
}
\keyword{models}
\eof
\name{glmm-class}
\docType{class}
\alias{glmm-class}
\alias{VarCorr,glmm-method}
\alias{fixef<-,glmm,numeric-method}
\alias{getResponse,glmm-method}
\alias{logLik,glmm-method}
\alias{show,glmm-method}
\alias{summary,glmm-method}
\title{Class "glmm"}
\description{A generalized linear mixed-effects model.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("glmm", ...)}
  but more commonly they are created by calls to \code{\link{GLMM}}.
}
\section{Slots}{
  \describe{
    \item{\code{family}:}{Object of class \code{"family"}, specifying
      the error distribution and the link function used in the model.}
    \item{\code{origy}:}{Object of class \code{"numeric"}, the original
      response.}
    \item{\code{n}:}{Object of class \code{"numeric"}, if the family is
      \code{binomial}, the number of observations for each data point.}
    \item{\code{prior.weights}:}{Object of class \code{"numeric"},
      weights used when creating the model.frame.}
    \item{\code{init.weights}:}{Object of class \code{"numeric"},
      weights from a fixed effects generalized linear model for the data.}
    \item{\code{init.y}:}{Object of class \code{"numeric"}, the linear
      predictor from a fixed effects generalized linear model for the data.}
    \item{\code{method}:}{Object of class \code{"character"}, the method
      used to fit the generalized linear mixed model.}
    \item{\code{reStruct}:}{Object of class \code{"reStruct"}, from class
	\code{"lme"}, the random-effects structure for the model.}
    \item{\code{frame}:}{Object of class \code{"data.frame"}, from class
	\code{"lme"}, the model.frame used to fit the model.}
    \item{\code{na.action}:}{Object of class \code{"ANY"}, from class
	\code{"lme"}, the \code{na.action} argument used when creating the
      \code{model.frame}.}
    \item{\code{fitted}:}{Object of class \code{"numeric"}, from class
      \code{"lme"}, the fitted values in the linear predictor scale.}
    \item{\code{call}:}{Object of class \code{"call"}, from class
      \code{"lme"}, the function call used to create the object.}
  }
}
\section{Extends}{
Class \code{"lme"}, directly.
}
\section{Methods}{
  \describe{
    \item{LMEoptimize<-}{\code{signature(x = "glmm", value = "list")}:
      optimize the PQL approximation to the log-likelihood.}
    \item{VarCorr}{\code{signature(x = "glmm")}: Extract variance and correlation components.}
    \item{fixef<-}{\code{signature(object = "glmm", value = "numeric")}:
      assign the fixed effects (used for \code{method = "Laplace"}).}
    \item{getResponse}{\code{signature(object = "glmm")}: extract the response.}
    \item{logLik}{\code{signature(object = "glmm")}: extract the
      (approximate) log-likelihood.}
    \item{show}{\code{signature(object = "glmm")}: show the object.}
    \item{summary}{\code{signature(object = "glmm")}: summarize the object.}
  }
}
\author{Saikat DebRoy \email{saikat@stat.wisc.edu}
  and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
library(lme4)
data(guImmun)
# This returns an object of class glmm
fm = GLMM(immun ~ kid2p + mom25p + ord + ethn +
                  momEd + husEd + momWork + rural + pcInd81,
          data = guImmun, family = binomial,
          random = ~1|comm/mom)
fm
}
\keyword{classes}
\eof
\name{glmmStruct}
\docType{methods}
\docType{genericFunction}
\alias{glmmStruct}
\alias{glmmStruct,formula,list-method}
\title{Create a glmm object}
\description{
  Create a \code{glmm} object from values for its slots.
}
\usage{
glmmStruct(formula, random, family, data, nextraCols, method, ...)
}
\arguments{
  \item{formula}{a formula of the form \code{response ~ terms} where
    \code{response} is an expression for the response and \code{terms}
    are one or more terms that define the fixed-effects.}
  \item{random}{a named list of one-sided formulas.  The names in the
    list evaluate to the grouping factors and the right-hand side of the
    formula is an expression for a \code{pdMat} object that itself
    contains a formula.  The order of the components of the list
    determines the nesting structure in that the second component is
    nested within the first, the third within the second, and so on.}
  \item{family}{a description of the error distribution and link
    function to be used in the model.
    See \code{\link[base:family]{family}} for details.}
  \item{data}{a \code{data.frame} or \code{model.frame} in which the
    \code{fixed} and \code{random} arguments can be evaluated.}
  \item{nextraCols}{an \code{integer} giving the number of extra columns
    to be added to the stored matrices.}
  \item{method}{a \code{character} giving the method used to fit the
    model.}
  \item{\dots}{optional arguments passed to
    \code{\link[base:model.frame]{model.frame}}}
}
%\details{}
\value{
  A \code{glmm} object.
}
%\references{}
%\author{}
%\note{}
\seealso{\code{\link{glmm-class}}, \code{\link{lmeLevel-class}}}
%\examples{}
\keyword{models}
\eof
\name{groupedData-class}
\docType{class}
\alias{groupedData-class}
\alias{formula,groupedData-method}
\title{Class "groupedData"}
\description{A \code{groupedData} object is at least a data.frame and a
  display  formula that provides a grouping structure.  At present we
  are using an old-style class for this.}
\section{Objects from the Class}{A virtual Class: No objects may be
  created from it.}
\section{Extends}{
Class \code{"data.frame"}, directly.
Class \code{"oldClass"}, by class "data.frame".
}
\section{Methods}{
  No methods defined with class "groupedData" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{guImmun}
\alias{guImmun}
\docType{data}
\title{Immunization in Guatemala}
\description{
  Immunizations received by children in Guatemala.
}
\usage{data(guImmun)}
\format{
  A data frame with 2159 observations on the following 13 variables.
  \describe{
    \item{kid}{a factor identifying the child}
    \item{mom}{a factor identifying the family.}
    \item{comm}{a factor identifying the community.}
    \item{immun}{a factor indicating if the child received a
      complete set of immunizations. All children in this data frame
      received at least one immunization.}
    \item{kid2p}{a factor indicating if the child was 2 years or older
      at the time of the survey.}
    \item{mom25p}{a factor indicating if the mother was 25 years or older.}
    \item{ord}{an factor indicating the child's birth's order within the
      family.  Levels are \code{01} - first child, \code{23} - second or
      third child, \code{46} - fourth to sixth child, \code{7p} -
      seventh or later child.}
    \item{ethn}{a factor indicating the mother's ethnicity.  Levels are
      \code{L} - Ladino, \code{N} - indigenous not speaking Spanish, and
      \code{S} - indigenous speaking Spanish.}
    \item{momEd}{a factor describing the mother's level of eduation.
      Levels are \code{N} - not finished primary, \code{P} - finished
      primary, \code{S} - finished secondary}
    \item{husEd}{a factor describing the husband's level of education.
      Levels are the same as for \code{momEd} plus \code{U} - unknown.}
    \item{momWork}{a factor indicating if the mother had ever
      worked outside the home.}
    \item{rural}{a factor indicating if the family's location is
      considered rural or urban.}
    \item{pcInd81}{the percentage of indigenous population in the
      community at the 1981 census.}
  }
}
%\details{}
\source{
  These data are available at
  \url{http://data.princeton.edu/multilevel/guImmun.dat}.  Multiple
  indicator columns in the original data table have been collapsed to
  factors for this data frame.
}
\references{
  Rodriguez, Germn and Goldman, Noreen (1995), "Improved estimation
  procedures for multilevel models with binary response: a case-study",
  \emph{Journal of the Royal Statistical Society, Series A}, \bold{164},
  339-355.
}
\examples{
data(guImmun)
summary(guImmun)
}
\keyword{datasets}
\eof
\name{guPrenat}
\alias{guPrenat}
\docType{data}
\title{Prenatal care in Guatemala}
\description{
  Data on the prenatal care received by mothers in Guatemala.
}
\usage{data(guPrenat)}
\format{
  A data frame with 2449 observations on the following 15 variables.
  \describe{
    \item{kid}{a factor identifying the birth}
    \item{mom}{a factor identifying the mother or family}
    \item{cluster}{a factor identifying the community}
    \item{prenat}{a factor indicating if traditional or modern prenatal
      care was provided for the birth.}
    \item{childAge}{an ordered factor of the child's age at the time of
      the survey. }
    \item{motherAge}{a factor indicating if the mother was older or
      younger.  The cut-off age is 25 years.}
    \item{birthOrd}{an ordered factor for the birth's order within the family.}
    \item{indig}{a factor indicating if the mother is Ladino, or
      indigenous not speaking Spanish, or indigenous speaking Spanish.}
    \item{momEd}{a factor describing the mother's level of eduation.}
    \item{husEd}{a factor describing the husband's level of education.}
    \item{husEmpl}{a factor describing the husband's employment status.}
    \item{toilet}{a factor indicating if there is a modern toilet in the
      house.}
    \item{TV}{a factor indicating if there is a TV in the house and, if
      so, the frequency with which it is used.}
    \item{pcInd81}{the percentage of indigenous population in the
      community at the 1981 census.}
    \item{ssDist}{distance from the community to the nearest clinic.}
  }
}
%\details{}
\source{
  These data are available at
  \url{http://data.princeton.edu/multilevel/guPrenat.dat}.  Multiple
  indicator columns in the original data table have been collapsed to
  factors for this data frame.
}
\references{
  Rodriguez, Germn and Goldman, Noreen (1995), "Improved estimation
  procedures for multilevel models with binary response: a case-study",
  \emph{Journal of the Royal Statistical Society, Series A}, \bold{164},
  339-355.
}
\examples{
data(guPrenat)
summary(guPrenat)
}
\keyword{datasets}
\eof
% $Id: intervals.Rd,v 1.1 2003/06/03 18:31:33 bates Exp $
\name{intervals}
\title{Confidence Intervals on Coefficients}
\usage{
intervals(object, level, \dots)
}
\alias{intervals}
\arguments{
 \item{object}{a fitted model object from which parameter estimates can
   be extracted.}
 \item{level}{an optional numeric value for the interval confidence
   level. Defaults to 0.95.}
 \item{\dots}{some methods for the generic may require additional
   arguments.}
}
\description{
  Confidence intervals on the parameters associated with the model
  represented by \code{object} are obtained. This function is generic;
  method functions can be written to handle specific classes of
  objects. Classes which already have methods for this function include:
  \code{gls}, \code{lme}, and \code{lmList}. 
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\seealso{\code{\link{confint}}}
} 
%\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
%\examples{}
\keyword{models}
\eof
\name{isInitialized}
\docType{genericFunction}
\alias{isInitialized}
\alias{isInitialized-methods}
\alias{isInitialized,pdBlocked-method}
\alias{isInitialized,pdCompSymm-method}
\alias{isInitialized,pdMat-method}
\alias{isInitialized,pdNatural-method}
%\alias{isInitialized,pdScalar-method}
\alias{isInitialized,lmeLevelList-method}
\title{Check for initialization}
\description{
  A generic function.  Methods for specific classes should return
  \code{TRUE} if \code{object} is initialized, otherwise \code{FALSE}.
}
\usage{
isInitialized(object)
}
\arguments{
  \item{object}{An object that may or may not be initialized.}
}
\value{
  A logical value indicating if the object is initialized.
}
\note{The name of this generic was changed from the original name,
  \code{isinitialized}, because that name is used in the \code{methods}
  package. }
\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\examples{
}
\keyword{manip}
\eof
\name{lmList-class}
\docType{class}
\alias{lmList-class}
\alias{coef,lmList-method}
\alias{formula,lmList-method}
\alias{intervals,lmList-method}
\alias{plot,lmList-method}
\alias{pooledSD,lmList-method}
\alias{show,lmList-method}
\alias{update,lmList-method}
\title{Class "lmList"}
\description{A list of objects of class \code{lm} with a common model.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("lmList", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{"list"}, a list of
      objects of class \code{lm}}
    \item{\code{call}:}{Object of class \code{"call"}, the function call
      used to create the \code{lmList} object.}
    \item{\code{pool}:}{Object of class \code{"logical"}, if TRUE then
      calculate the pooled standard deviation estimate when displaying
      the object.}
  }
}
\section{Extends}{
Class \code{"list"}, from data part.
Class \code{"vector"}, by class "list".
}
\section{Methods}{
  \describe{
    \item{coef}{\code{signature(object = "lmList")}: extract the
      coefficients for the linear models.}
    \item{formula}{\code{signature(x = "lmList")}: extract the formula
      used when creating the lmList object.}
    \item{intervals}{\code{signature(object = "lmList")}: confidence
      intervals for the fitted linear models.}
    \item{plot}{\code{signature(x = "lmList")}: plot the fitted linear
      models.}
    \item{pooledSD}{\code{signature(object = "lmList")}: the pooled
      standard deviation estimate from the fitted linear models.}
    \item{\link[methods:show]{show}}{\code{signature(object =
	"lmList")}: show the object.}
    \item{\link[base:update]{update}}{\code{signature(object =
	"lmList")}: update the fit.}
    
  }
}
\keyword{classes}
\eof
% $Id: lmList.Rd,v 1.3 2003/08/11 18:39:01 bates Exp $
\name{lmList}
\alias{lmList}
\alias{lmList,formula,data.frame-method}
\title{List of lm Objects with a Common Model}
\usage{
lmList(formula, data, level, subset, na.action, pool)
}
\arguments{
  \item{formula}{For \code{lmList},
    a linear formula object of the form \code{y ~ x1+...+xn | g}.
    In the formula object, \code{y}
    represents the response, \code{x1,...,xn} the covariates, and
    \code{g} the grouping factor specifying the partitioning of the data
    according to which different \code{lm} fits should be performed. The
    grouping factor \code{g} may be omitted from the formula, in which
    case the grouping structure will be obtained from \code{data}, which
    must inherit from class \code{groupedData}.
  }
  \item{data}{
    a data frame in which to interpret the variables named in
    \code{object}. 
  }
  \item{level}{
    an optional integer specifying the level of grouping to be used when 
    multiple nested levels of grouping are present.
  }
 \item{subset}{an optional expression indicating which subset of the rows of
   \code{data} should  be  used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lmList} to print an error message and terminate if there are any
   incomplete observations.
 }
 \item{pool}{
   an optional logical value that is preserved as an attribute of the
   returned value.  This will be used as the default for \code{pool} in
   calculations of standard deviations or standard errors for summaries.
 }
}
\description{
  \code{Data} is partitioned according to the levels of the grouping
  factor \code{g} and individual \code{lm} fits are obtained for each
  \code{data} partition, using the model defined in \code{object}.
}

\value{
  an object of class lmList which is a list of \code{lm} objects with as
  many components as the number of groups defined by the grouping factor.
}

\seealso{
\code{\link[base:lm]{lm}}
}
\examples{
data(Orthodont, package = 'nlme')
fm1 <- lmList(distance ~ age | Subject, Orthodont)
fm1
}
\keyword{models}

\eof
\name{lme-class}
\docType{class}
\alias{lme-class}
\alias{VarCorr,lme-method}
\alias{anova,lme-method}
\alias{coef,lme-method}
\alias{fitted,lme-method}
\alias{fixef,lme-method}
\alias{formula,lme-method}
\alias{getResponse,lme-method}
\alias{intervals,lme-method}
\alias{logLik,lme-method}
\alias{plot,lme-method}
\alias{ranef,lme-method}
\alias{residuals,lme-method}
\alias{show,lme-method}
\alias{summary,lme-method}
\alias{update,lme-method}
\title{Class "lme"}
\description{A linear mixed-effects model.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("lme", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{reStruct}:}{Object of class \code{"reStruct"},
      the random-effects structure for the model.}
    \item{\code{frame}:}{Object of class \code{"data.frame"},
      the model.frame used to fit the model.}
    \item{\code{na.action}:}{Object of class \code{"ANY"},
      the \code{na.action} argument used when creating the
      \code{model.frame}.}
    \item{\code{fitted}:}{Object of class \code{"numeric"}, the current
      fitted values.}
    \item{\code{call}:}{Object of class \code{"call"},
      the function call used to create the object.}
  }
}
\section{Extends}{
Class \code{"lme"}, directly.
}
\section{Methods}{
  \describe{
    \item{VarCorr}{\code{signature(x = "lme")}: extract variance and correlation components.}
    \item{anova}{\code{signature(object = "lme")}: calculate analysis of
      variance tables for linear mixed effects models.}
    \item{coef}{\code{signature(object = "lme")}: extract the
      random effects parameters for the model.}
    \item{fitted}{\code{signature(object = "lme")}: extract the fitted values.}
    \item{fixef}{\code{signature(object = "lme")}: extract the fixed
      effects parameters.}
    \item{formula}{\code{signature(x = "lme")}: extract the fixed
      effects formula.}
    \item{getResponse}{\code{signature(object = "lme")}: extract the response.}
    \item{intervals}{\code{signature(object = "lme")}: calculate
      confidence intervals.}
    \item{logLik}{\code{signature(object = "lme")}: calculate the
      log-likelihood or log-restricted-likelihood for the model}
    \item{plot}{\code{signature(x = "lme")}: plot the fitted model.}
    \item{ranef}{\code{signature(object = "lme")}: calculate the
      random-effects coefficients}
    \item{residuals}{\code{signature(object = "lme")}: calculate the
      residuals from the model fit.}
    \item{show}{\code{signature(object = "lme")}: show the object.}
    \item{summary}{\code{signature(object = "lme")}: summarize the object.}
    \item{update}{\code{signature(object = "lme")}: update the model fit.}
  }
}
\author{Saikat DebRoy \email{saikat@stat.wisc.edu}
  and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
library(lme4)
data(bdf, package = "nlme")
# This returns an object of class lme
fm <- lme(langPOST ~ IQ.ver.cen + avg.IQ.ver.cen, data = bdf,
          random = ~ IQ.ver.cen | schoolNR)
fm
}
\keyword{classes}
\eof
\name{lme}
\docType{genericFunction}
\alias{lme}
\alias{lme,formula,data.frame,list-method}
\alias{lme,ANY,missing,ANY-method}
\alias{lme,missing,groupedData,ANY-method}
\alias{lme,formula,groupedData,missing-method}
\alias{lme,formula,ANY,formula-method}
\alias{lme,formula,ANY,list-method}
\title{Fit linear mixed-effects models}
\description{
  This generic function fits a linear mixed-effects model in the
  formulation described in Laird and Ware (1982) but allowing for nested
  random effects. The within-group errors are allowed to be correlated
  and/or have unequal variances. 
}
\usage{
lme(formula, data, random, correlation, weights, subset,
    method, na.action, control, model, x)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{formula}{a two-sided linear formula object describing the
    fixed-effects part of the model, with the response on the left of a
    \code{~} operator and the terms, separated by \code{+} operators, on
    the right.}
  \item{data}{an optional data frame containing the variables named in
    \code{formula}, \code{random}, \code{correlation}, \code{weights}, and
    \code{subset}.  By default the variables are taken from the
    environment from which \code{lme} is called.}
 \item{random}{optionally, any of the following: (i) a one-sided formula
   of the form \code{~x1+...+xn | g1/.../gm}, with \code{x1+...+xn}
   specifying the model for the random effects and \code{g1/.../gm} the
   grouping structure (\code{m} may be equal to 1, in which case no
   \code{/} is required). The random effects formula will be repeated
   for all levels of grouping, in the case of multiple levels of
   grouping; (ii) a list of one-sided formulas of the form
   \code{~x1+...+xn | g}, with possibly different random effects models
   for each grouping level. The order of nesting will be assumed the
   same as the order of the elements in the list; (iii) a one-sided
   formula of the form \code{~x1+...+xn}, or a \code{pdMat} object with
   a formula (i.e. a non-\code{NULL} value for \code{formula(object)}),
   or a list of such formulas or \code{pdMat} objects. In this case, the
   grouping structure formula will be derived from the data used to
   fit the linear mixed-effects model, which should inherit from class
   \code{groupedData}; (iv) a named list of formulas or \code{pdMat}
   objects as in (iii), with the grouping factors as names. The order of
   nesting will be assumed the same as the order of the order of the
   elements in the list; (v) an \code{reStruct} object. See the
   \code{\link{pdMat-class}} documentation for a description of the available
   \code{pdMat} classes. Defaults to a formula consisting of the right
   hand side of \code{formula}.}  
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL},
   corresponding to no within-group correlations.}
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homocesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"REML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{lmeControl}.
   Defaults to an empty list.}
 \item{model, x}{logicals.  If \code{TRUE} the corresponding
   components of the fit (the model frame, the model matrices)
   are returned.}
}
\details{
  Many of the options are not yet implemented.
}
\value{
  An lme object.%\code{\link{lme-class}{lme}} object.
}
%\references{ ~put references to the literature/web site here ~ }
%\author{}
%\note{}
%\seealso{\code{\link{lme-class}}}
\examples{
data(bdf, package = "nlme")
fm <- lme(langPOST ~ IQ.ver.cen + avg.IQ.ver.cen, data = bdf,
          random = ~ IQ.ver.cen | schoolNR)
summary(fm)
}
\keyword{models}
\keyword{methods}
\eof
% $Id: lmeControl.Rd,v 1.5 2003/09/20 11:54:43 saikat Exp $
\name{lmeControl}
\alias{lmeControl}
\title{Control values for lme}
\description{
  The values supplied in the function call replace the defaults and a
  list with all possible arguments is returned. The returned list is
  used as the \code{control} argument in the \code{lme} function.
}
\usage{
lmeControl(maxIter, msMaxIter, tolerance, niterEM, msTol,
           msScale, msVerbose, returnObject, gradHess, apVar,
           .relStep, minAbsParApVar, nlmStepMax, natural, optimizer,
           EMverbose, analyticGradient, analyticHessian)
}
\arguments{
 \item{maxIter}{maximum number of iterations for the \code{lme}
   optimization algorithm. Default is 50.} 
 \item{msMaxIter}{maximum number of iterations
   for the \code{nlm} optimization step inside the \code{lme}
   optimization. Default is 50.}
 \item{tolerance}{tolerance for the convergence criterion in the
   \code{lme} algorithm. Default is 1e-6.}
 \item{niterEM}{number of iterations for the EM algorithm used to refine
   the initial estimates of the random effects variance-covariance
   coefficients. Default is 25.}
 \item{msTol}{tolerance for the convergence criterion in \code{nlm},
   passed as the \code{rel.tolerance} argument to the function (see
   documentation on \code{nlm}). Default is 1e-7. }
 \item{msScale}{scale function passed as the \code{scale} argument to
   the \code{nlm} function (see documentation on that function). Default
   is \code{lmeScale}.}
 \item{msVerbose}{a logical value passed as the \code{trace} argument to
   \code{nlm} (see documentation on that function). Default is
   \code{FALSE}.} 
 \item{returnObject}{a logical value indicating whether the fitted
   object should be returned when the maximum number of iterations is
   reached without convergence of the algorithm. Default is
   \code{FALSE}.} 
 \item{gradHess}{a logical value indicating whether numerical gradient
   vectors and Hessian matrices of the log-likelihood function should
   be used in the \code{nlm} optimization. This option is only available
   when the correlation structure (\code{corStruct}) and the variance
   function structure (\code{varFunc}) have no "varying" parameters and
   the \code{pdMat} classes used in the random effects structure are
   \code{pdLogChol} (general positive-definite), \code{pdDiag} (diagonal),
   \code{pdIdent} (multiple of the identity),  or
   \code{pdCompSymm} (compound symmetry). Default is \code{TRUE}.}
 \item{apVar}{a logical value indicating whether the approximate
   covariance matrix of the variance-covariance parameters should be
   calculated. Default is \code{TRUE}.}
 \item{.relStep}{relative step for numerical derivatives
   calculations. Default is \code{.Machine$double.eps^(1/3)}.}
 \item{minAbsParApVar}{numeric value - minimum absolute parameter value
   in the approximate variance calculation.  The default is \code{0.05}.}
 \item{nlmStepMax}{stepmax value to be passed to nlm. See
     \code{\link{nlm}} for details. Default is 100.0}
 \item{natural}{a logical value indicating whether the \code{pdNatural}
   parametrization should be used for general positive-definite matrices
   (\code{pdLogChol}) in \code{reStruct}, when the approximate covariance
   matrix of the estimators is calculated. Default is \code{TRUE}.}
 \item{optimizer}{the optimizer to be used - either \code{"optim"}, the
   default, or \code{"nlm"}}
 \item{EMverbose}{a logical value indicating if verbose output should be
   produced during the EM iterations.  Default is \code{FALSE}.}
 \item{analyticGradient}{a logical value indicating if the analytic
   gradient of the objective should be used.  This option is for testing
   purposes and would not normally be changed from the default.  Default
   is \code{TRUE}.}
 \item{analyticHessian}{a logical value indicating if the analytic
   hessian of the objective should be calculated. This is an
   experimental feature and at present the default is \code{FALSE}. In
   future we may use the analytic Hessian in the optimization.}
}
\value{
  a list with a component for each of the possible arguments.
}
%\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{nlm}}, \code{\link{optim}},
  \code{\link{lmeScale}}}
\examples{
# decrease the maximum number iterations in the ms call and
# request that information on the evolution of the ms iterations be printed
lmeControl(msMaxIter = 20, msVerbose = TRUE)
}
\keyword{models}
\eof
\name{lmeLevel-class}
\docType{class}
\alias{lmeLevel-class}
\alias{coef,lmeLevel-method}
\title{Class "lmeLevel"}
\description{An object representing a single level of random effects in
  a linear mixed-effects model.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("lmeLevel",
    ...)} or by calls to the generic constructor \code{lmeLevel}.
}
\section{Slots}{
  \describe{
    \item{\code{precision}:}{A precision matrix of class \code{"pdMat"}}
    \item{\code{groups}:}{The grouping factor (of class \code{"factor"}) for the level.}
    \item{\code{columns}:}{An \code{"integer"} vector of the columns in
      the model matrix that correspond to this level.}
    \item{\code{parsInd}:}{An \code{"integer"} vector giving the
      subscripts in the overall parameter vector that apply to this level.}
    \item{\code{originalRows}:}{A list of integer vectors of row
      numbers in the original model matrix that correspond to each of the
      levels of the grouping factor.}
    \item{\code{decomposedRows}:}{A list of integer vectors of row
      numbers in the decomposed model matrix that correspond to each of the
      levels of the grouping factor.}
    \item{\code{storedRows}:}{A list of integer vectors of row
      numbers in the stored data matrix that correspond to each of the
      levels of the grouping factor.}
    \item{\code{nrow}:}{\code{integer}: the number of rows in the
      precision matrix.}
    \item{\code{updateFactor}:}{Object of class \code{"matrix"} - the
      matrix \code{A} used to evaluate the gradient of the
      log-likelihood and to formulate the ECME step.}
    \item{\code{hessianArray}:}{Object of class \code{"array"} - the
      array \code{H} used to evaluate the Hessian of the
      log-likelihood.}
    \item{\code{nlev}:}{\code{integer}: the number of levels in the
      grouping factor}
  }
}
\section{Methods}{
  \describe{
    \item{EMupdate<-}{\code{signature(x = "lmeLevel", nlev = "missing",
	value = "matrix")}: update the level within the EM or ECME algorithm}
    \item{LMEgradient}{\code{signature(x = "lmeLevel", A = "matrix",
	nlev = "missing")}: calculate the gradient of the log-likelihood
      with respect to the parameters from the decomposition of the scaled
      BLUPs and their variances.}
    \item{coef}{\code{signature(object = "lmeLevel")}: extract the
      parameters (from the precision matrix)}
    \item{coef<-}{\code{signature(object = "lmeLevel", value =
	"numeric")}: set the parameters.}
  }
}
\note{In \code{reStruct} objects the random-effects terms and the
  fixed-effects terms and the response are all represented by
  \code{lmeLevel} objects.  The \code{precision} slot in the
  fixed-effects and response objects is an uninitialized \code{pdMat}
  objects and the \code{nrow} slot is zero.}
%\references{}
\author{Saikat DebRoy \email{saikat@stat.wisc.edu} and Douglas Bates
  \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lmeLevel}}}
%\examples{}
\keyword{classes}
\eof
\name{lmeLevel}
\docType{genericFunction}
\alias{lmeLevel}
\alias{lmeLevel,pdMat,factor,integer,matrix-method}
\alias{lmeLevel,missing,missing,integer,matrix-method}
\title{Construct an lmeLevel object}
\description{
  Create an \code{lmeLevel} object from its components}
}
\usage{
lmeLevel(precision, groups, columns, modelMatrix)
}
\arguments{
  \item{precision}{A \code{pdMat} object representing the relative
    precision matrix for the random effects.  This argument is missing
    in calls to generate the \code{lmeLevel} for the fixed-effects terms
    and for the response.}
  \item{groups}{The grouping factor.  This argument is missing
    in calls to generate the \code{lmeLevel} for the fixed-effects terms
    and for the response.}
  \item{columns}{integer vector: the columns in \code{modelMatrix}
    corresponding to this level of random or fixed effects.}
  \item{modelMatrix}{a model matrix: the \code{original} slot of the
    containing \code{reStruct} object}
}
\details{
  An \code{lmeLevel} object contains information on one level of
  random-effects terms or the fixed-effects terms or the response in a
  linear mixed-effects structure.
}
\value{
  An \code{lmeLevel} object.
}
%\references{}
%\author{}
%\note{}
\seealso{\code{\link{lmeLevel-class}}, \code{\link{pdMat-class}},
  \code{\link{reStruct-class}}}
%\examples{}
\keyword{models}
\eof
\name{logDet}
\docType{genericFunction}
\docType{methods}
\alias{logDet}
\alias{logDet-methods}
\alias{logDet,pdBlocked,missing-method}
\alias{logDet,pdCompSymm,missing-method}
\alias{logDet,pdDiag,missing-method}
\alias{logDet,pdIdent,missing-method}
\alias{logDet,pdLogChol,missing-method}
\alias{logDet,pdMat-method}
\alias{logDet,pdNatural,missing-method}
%\alias{logDet,pdScalar,missing-method}
\title{Logarithm of the determinant}
\description{
  Return the logarithm of the determinant of an object of a suitable
  class.
}
\usage{
logDet(object, covariate = getCovariate(object), ...)
}
\arguments{
  \item{object}{An object of a suitable class.}
  \item{covariate}{Optionally, a covariate vector}
  \item{\dots}{Some methods may have optional arguments.}
}
%\details{}
\value{
  A numeric value: the logarithm of the determinant of the matrix
  represented by \code{object}. 
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
%\examples{}
\keyword{algebra}
\keyword{array}
\eof
\name{matrixGets}
\docType{genericFunction}
\alias{matrix<-}
\alias{matrix<--methods}
%\alias{matrix<-,pdBlocked,matrix-method}
\alias{matrix<-,pdMat,matrix-method}
\title{Assign the matrix}
\description{
  This generic function assigns the matrix to an object of a suitable
  class, such as the \code{pdMat} class.

  This function is deprecated.  Use a coerced assignment instead.
}
\usage{
matrix(object) <- value
}
\arguments{
  \item{object}{An object of a suitable class}
  \item{value}{A matrix of a suitable size and family.}
}
\value{
  An object of the same class as \code{object}.
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
%\examples{}
\keyword{manip}
\keyword{algebra}
\keyword{array}
\keyword{methods}
\eof
\name{model.matrixGets}
\docType{methods}
\docType{genericFunction}
\alias{model.matrix<-}
\alias{model.matrix<-,reStruct,matrix-method}
\title{Assign the model.matrix}
\description{
  Assign the model.matrix in an object of a suitable class. This function is
  generic.
}
\usage{
model.matrix(x) <- value
}
\arguments{
  \item{x}{An object of a suitable class.}
  \item{value}{A model matrix to assign to the object \code{x}.}
}
%\details{}
\value{
  The object \code{x} with the model.matrix updated.
}
\section{Methods}{
  \describe{
    \item{x = reStruct, value = matrix}{assign the model.matrix in an
      \code{reStruct} object}
  }
}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{reStruct-class}}}
%\examples{}
\keyword{manip}
\keyword{models}
\eof
\name{nfGroupedData-class}
\docType{class}
\alias{nfGroupedData-class}
\title{Class "nfGroupedData"}
\description{A \code{groupedData} object is at least a data.frame and a
  display  formula that provides a grouping structure.  In the
  \code{nfGroupedData} class there is only one level of grouping factors
  and the response in the display formula is numeric.  At present we
  use old-style classes for \code{groupedData} and the classes
  that inherit from it.}
\section{Objects from the Class}{A virtual Class: No objects may be
  created from it.}
\section{Extends}{
Class \code{"groupedData"}, directly.
Class \code{"data.frame"}, by class "groupedData".
Class \code{"oldClass"}, by class "groupedData".
}
\section{Methods}{
  No methods defined with class "groupedData" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{nffGroupedData-class}
\docType{class}
\alias{nffGroupedData-class}
\title{Class "nffGroupedData"}
\description{A \code{groupedData} object is at least a data.frame and a
  display  formula that provides a grouping structure.  In the
  \code{nfGroupedData} class there is only one level of grouping factors
  and the response in the display formula is numeric.  The
  \code{nffGroupedData} class is a further refinement that specifies
  that the primary covariate is a factor. At present we
  are using old-style classes for \code{groupedData} and the classes
  that extend it.}
\section{Objects from the Class}{A virtual Class: No objects may be
  created from it.}
\section{Extends}{
Class \code{"nfGroupedData"}, directly.
Class \code{"groupedData"}, by class "nfGroupedData".
Class \code{"data.frame"}, by class "nfGroupedData".
Class \code{"oldClass"}, by class "nfGroupedData".
}
\section{Methods}{
  No methods defined with class "nfnGroupedData" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{nfnGroupedData-class}
\docType{class}
\alias{nfnGroupedData-class}
\title{Class "nfnGroupedData"}
\description{A \code{groupedData} object is at least a data.frame and a
  display  formula that provides a grouping structure.  In the
  \code{nfGroupedData} class there is only one level of grouping factors
  and the response in the display formula is numeric.  The
  \code{nfnGroupedData} class is a further refinement that specifies
  that the primary covariate is numeric. At present we
  are using old-style classes for \code{groupedData} and the classes
  that extend it.}
\section{Objects from the Class}{A virtual Class: No objects may be
  created from it.}
\section{Extends}{
Class \code{"nfGroupedData"}, directly.
Class \code{"groupedData"}, by class "nfGroupedData".
Class \code{"data.frame"}, by class "nfGroupedData".
Class \code{"oldClass"}, by class "nfGroupedData".
}
\section{Methods}{
  No methods defined with class "nfnGroupedData" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{nlme-internal}
%\alias{allVarsRec}
%\alias{createConLin}
\alias{getFixDF}
\alias{glmmLa2LogLikComp}
\alias{glmmLa2RespWt}
\alias{glmmLa2Wt2}
%\alias{getParsGnls}
%\alias{getParsNlme}
%\alias{glsApVar}
%\alias{glsEstimate}
%\alias{gnlsApVar}
%\alias{lmeApVar}
%\alias{nonlinModel}
%\alias{print.correlation}
\title{Internal nlme functions}
\description{
  Internal nlme functions.
}
\usage{
%allVarsRec(object)
%createConLin(fixed,data,random,\dots)
getFixDF(object)
glmmLa2LogLikComp(x, eta)
glmmLa2RespWt(fam, eta, origy, w, off)
glmmLa2Wt2(fam, eta, w)
%getParsGnls(plist, pmap, beta, N)
%getParsNlme(plist, fmap, rmapRel, bmap, groups, beta, bvec, b, level, N)
%glsApVar(glsSt, sigma, conLin, .relStep, minAbsPar, natural)
%glsEstimate(object, conLin, control)
%gnlsApVar(gnlsSt, lsigma, conLin, .relStep, minAbsPar, natural)
%lmeApVar(lmeSt, sigma, conLin, .relStep, minAbsPar, natural)
%nonlinModel(modelExpression, env, paramNames)
%\method{print}{correlation}(x, title, rdig, \dots)
}
\arguments{
  \item{object}{An object of a suitable class.}
  \item{x}{An object of class glmm}
  \item{eta}{The linear predictors}
  \item{fam}{A family object}
  \item{origy}{The original response}
  \item{w}{The initial weights}
  \item{off}{The initial offset}
}
\details{
  These functions are internal to the package and are not intended to be
  called by the user. They will be hidden by a namespace when namespaces
  and S4 classes can work together.}
\keyword{models}
\eof
\name{nlmeInternal-class}
\docType{class}
\alias{lmeLevelList-class}
\alias{summary.glmm-class}
\alias{show,summary.glmm-method}
\alias{summary.lme-class}
\alias{show,summary.lme-method}
\alias{summary.pdMat-class}
\title{Some internal classes}
\description{These classes are used for internal use only.  When
  namespaces are available for packages that use S4 classes and methods
  these classes will be hidden inside the namespace}
%\references{}
%\author{}
%\note{}
%\examples{}
\keyword{classes}
\eof
\name{nmGroupedData-class}
\docType{class}
\alias{nmGroupedData-class}
\title{Class "nmGroupedData"}
\description{A \code{groupedData} object is at least a data.frame and a
  display  formula that provides a grouping structure.  In the
  \code{nmGroupedData} class there are multiple level of grouping factors
  and the response in the display formula is numeric.  At present we
  use old-style classes for \code{groupedData} and the classes
  that inherit from it.}
\section{Objects from the Class}{A virtual Class: No objects may be
  created from it.}
\section{Extends}{
Class \code{"groupedData"}, directly.
Class \code{"data.frame"}, by class "groupedData".
Class \code{"oldClass"}, by class "groupedData".
}
\section{Methods}{
  No methods defined with class "groupedData" in the signature.
}
%\references{}
%\author{}
%\note{}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{pdBlocked-class}
\docType{class}
\alias{pdBlocked-class}
\alias{pdBlocked}
\title{Class "pdBlocked", blocked pd matrices}
\description{
  Block-diagonal positive definite matrices.
}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pdBlocked", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{param}:}{Object of class \code{"list"} }
    \item{\code{form}:}{Object of class \code{"formula", from class "pdMat"}} \item{\code{Names}:}{Object of class \code{"character", from class "pdMat"} }
  }
}
\section{Extends}{
  Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{Names}{\code{signature(object = "pdBlocked")}:}
    \item{Names<-}{\code{signature(object = "pdBlocked", value = "character")}:}
    \item{[}{\code{signature(x = "pdBlocked")}: }
    \item{coef}{\code{signature(object = "pdBlocked")}: }
    \item{coef<-}{\code{signature(object = "pdBlocked", value = "ANY")}: }
    \item{corMatrix}{\code{signature(object = "pdBlocked")}:}
    \item{formula}{\code{signature(x = "pdBlocked")}:}
    \item{isInitialized}{\code{signature(object = "pdBlocked")}:}
    \item{logDet}{\code{signature(object = "pdBlocked", covariate = "missing")}:}
    \item{matrix<-}{\code{signature(object = "pdBlocked", value = "matrix")}: }
    \item{pdFactor}{\code{signature(object = "pdBlocked")}: }
    \item{solve}{\code{signature(a = "pdBlocked", b = "missing")}:}
    \item{summary}{\code{signature(object = "pdBlocked")}:}
  }
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\seealso{}
%\examples{}
\keyword{classes}
\eof
\name{pdCompSymm-class}
\docType{class}
\alias{pdCompSymm-class}
\alias{pdCompSymm}
\alias{pdCompSymm,formula,missing,missing,missing-method}
\title{Class "pdCompSymm"}
\description{A class of parameterized positive-definite symmetric
  matrices that have compound symmetry.  That is, the diagonal entries
  are constant and the off-diagonal entries are constant.}
\section{Objects from the Class}{
  Objects can be created by calls of the form
  \code{new("pdCompSymm", ...)} or
  by calls to the constructor \code{pdCompSymm(...)}
}
\section{Slots}{
  \describe{
    \item{\code{param}:}{Object of class \code{"numeric"} - a parameter
      vector of length 2}
    \item{\code{ncol}:}{Object of class \code{"integer"} - the number of
      columns (and rows) in the matrix.}
    \item{\code{form}:}{Object of class \code{"formula", from class
	"pdMat"} - an optional formula relating the object to a model matrix. }
    \item{\code{Names}:}{Object of class \code{"character", from class
	"pdMat"} - dimnames for the matrix}
  }
}
\section{Extends}{
Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{coef}{\code{signature(x = "pdCompSymm")}: get the parameter vector }
    \item{coef<-}{\code{signature(object = "pdCompSymm", value =
	"numeric")}: set the parameter vector}
    \item{coef<-}{\code{signature(object = "pdCompSymm", value =
	"matrix")}: set the parameter vector from a model matrix}
    \item{coerce}{\code{signature(from = "pdCompSymm", to = "matrix")}:
      get the positive-definite symmetric matrix represented by the object.}
    \item{corMatrix}{\code{signature(object = "pdCompSymm")}: get the
      matrix represented by the object as a correlation matrix (with
      standard deviations)}
    \item{dim}{\code{signature(x = "pdCompSymm")}: get the dimensions of
      the matrix represented by the object.}
    \item{isInitialized}{\code{signature(object = "pdCompSymm")}: check
      if the object has been initialized.}
    \item{logDet}{\code{signature(object = "pdCompSymm", covariate =
	"missing")}: get the logarithm of the determinant of the matrix}
    \item{pdFactor}{\code{signature(object = "pdCompSymm")}: get a
      square root factor of the matrix represented by the object.}
    \item{pdMatrix}{\code{signature(object = "pdCompSymm", factor =
	"logical")}: get the positive-definite symmetric matrix
	represented by the object.}
    \item{summary}{\code{signature(object = "pdCompSymm")}:}
  }
}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\seealso{}
\examples{
}
\keyword{classes}
\eof
\name{pdDiag-class}
\docType{class}
\alias{pdDiag-class}
\alias{pdDiag}
\alias{pdDiag,formula,missing,missing,missing-method}
\title{Class "pdDiag"}
\description{A class of parameterized diagonal positive-definite
  matrices.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pdDiag", ...)} or
  by calls to the constructor function \code{pdDiag}.
}
\section{Slots}{
  \describe{
    \item{\code{form}:}{Object of class \code{"formula", from class
	"pdMat"} an optional formula used to generate the object.}
    \item{\code{Names}:}{Object of class \code{"character", from class
	"pdMat"} row (and column) names for the matrix.}
    \item{\code{param}:}{Object of class \code{"numeric", from class
	"pdMat"} the unconstrained, numeric parameter vector of length
      \code{Ncol}.  The elements of the parameter vector are the
      logarithms of the square roots of the diagonal elements of the matrix. }
    \item{\code{Ncol}:}{Object of class \code{"integer", from class
	"pdMat"} the number of rows and columns in the matrix. }
    \item{\code{factor}:}{Object of class \code{"matrix", from class
	"pdMat"} a square-root factor of the matrix - also a diagonal
      matrix.}
    \item{\code{logDet}:}{Object of class \code{"numeric", from class
	"pdMat"} the logarithm of the absolute value of the determinant
      of the factor.}
  }
}
\section{Extends}{
Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{coef}{\code{signature(x = "pdDiag")}: extract the parameters.}
    \item{coef<-}{\code{signature(object = "pdDiag", value =
	"numeric")}: assign the parameters.}
    \item{coerce}{\code{signature(from = "pdDiag", to = "matrix")}:
      extract the matrix.}
    \item{corMatrix}{\code{signature(object = "pdDiag")}: extract the
      correlation matrix. }
    \item{dim}{\code{signature(x = "pdDiag")}: extract the dimensions.}
    \item{isInitialized}{\code{signature(object = "pdDiag")}: check if initialized.}
    \item{logDet}{\code{signature(object = "pdDiag", covariate =
	"missing")}: return the logarithm of the determinant of the matrix.}
    \item{solve}{\code{signature(a = "pdDiag", b = "missing")}: invert
      the matrix returning a \code{pdDiag} object.}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdMat-class}}}
\examples{
}
\keyword{classes}
\eof
\name{pdFactor}
\docType{methods}
\docType{genericFunction}
\alias{pdFactor}
\alias{pdFactor-methods}
\alias{pdFactor,pdMat-method}
\title{Square-root factor}
\description{This generic function extracts the square-root factor of
  the positive-definite matrix represented by \code{object}. Letting
  \eqn{\Sigma}{S} denote a positive-definite matrix, a square-root
  factor of \eqn{\Sigma}{S} is any square matrix \eqn{L}{L} such that
  \eqn{\Sigma = L'L}{S = L'L}. This function extracts \eqn{L}.

  This function is deprecated.  Use coersion to the \code{pdfactor}
  class instead.
}
\usage{
pdFactor(object)
}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix, which must have been initialized
   (i.e. \code{length(coef(object)) > 0}).} 
}
\value{
  A \code{pdfactor} object.
}
\seealso{\code{\link{pdMat-class}}, \code{\link{pdfactor-class}}}
%\examples{}
\keyword{models}
\eof
\name{pdIdent-class}
\docType{class}
\alias{pdIdent-class}
\alias{pdIdent}
\alias{pdIdent,formula,missing,missing,missing-method}
\title{Class "pdIdent"}
\description{A class of parameterized positive-definite
  matrices that are multiples of the identity matrix.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pdIdent", ...)} or
  by calls to the constructor function \code{pdIdent}.
}
\section{Slots}{
  \describe{
    \item{\code{form}:}{Object of class \code{"formula", from class
	"pdMat"} an optional formula used to generate the object.}
    \item{\code{Names}:}{Object of class \code{"character", from class
	"pdMat"} row (and column) names for the matrix.}
    \item{\code{param}:}{Object of class \code{"numeric", from class
	"pdMat"} the unconstrained, numeric parameter vector of length
      \code{Ncol}.  The elements of the parameter vector are the
      logarithms of the square roots of the diagonal elements of the matrix. }
    \item{\code{Ncol}:}{Object of class \code{"integer", from class
	"pdMat"} the number of rows and columns in the matrix. }
    \item{\code{factor}:}{Object of class \code{"matrix", from class
	"pdMat"} a square-root factor of the matrix - also a diagonal
      matrix.}
    \item{\code{logDet}:}{Object of class \code{"numeric", from class
	"pdMat"} the logarithm of the absolute value of the determinant
      of the factor.}
  }
}
\section{Extends}{
Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{coef}{\code{signature(x = "pdIdent")}: extract the parameters.}
    \item{coef<-}{\code{signature(object = "pdIdent", value =
	"numeric")}: assign the parameters.}
    \item{coerce}{\code{signature(from = "pdIdent", to = "matrix")}:
      extract the matrix.}
    \item{corMatrix}{\code{signature(object = "pdIdent")}: extract the
      correlation matrix. }
    \item{dim}{\code{signature(x = "pdIdent")}: extract the dimensions.}
    \item{isInitialized}{\code{signature(object = "pdIdent")}: check if initialized.}
    \item{logDet}{\code{signature(object = "pdIdent", covariate =
	"missing")}: return the logarithm of the determinant of the matrix.}
    \item{solve}{\code{signature(a = "pdIdent", b = "missing")}: invert
      the matrix returning a \code{pdIdent} object.}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdMat-class}}}
\examples{
}
\keyword{classes}
\eof
\name{pdLogChol-class}
\docType{class}
\alias{pdLogChol-class}
\alias{pdLogChol}
\alias{pdLogChol,formula,missing,missing,missing-method}
\title{Class "pdLogChol", positive-definite matrices}
\description{A class of general, positive-definite
  symmetric matrices parameterized by the non-zero elements in the
  Cholesky decomposition.  The diagonal elements are represented by
  their logarithms in the first \code{q} positions of the parameter
  vector.  The strict upper triangle of the factor is in the last
  \code{q(q-1)/2} positions.}
\section{Objects from the Class}{
  Objects of class \code{pdLogChol} can be created by calls of the form
  \code{new("pdLogChol", ...)} or by the generic constructor function
  \code{pdLogChol}.  Frequently the constructor is given a formula only,
  creating an uninitialized \code{pdLogChol} object which is later
  assigned a value.

  \code{pdLogChol} objects are primarily used to represent the
  variance-covariance matrix or the precision matrix of random-effects
  terms in mixed-effects models.
}
\section{Slots}{
  \describe{
    \item{\code{form}:}{Object of class \code{"formula", from class
	"pdMat"}, a formula for the object}
    \item{\code{Names}:}{Object of class \code{"character", from class
	"pdMat"}, names for the rows (and columns) of the
      positive-definite matrix.}
    \item{\code{param}:}{Object of class \code{"numeric", from class
	"pdMat"}, a parameter vector of length
      \eqn{[q(q+1)]/2}{[q(q+1)]/2} where \eqn{q}{q} is \code{Ncol}, the
      number of columns (and rows) in the positive-definite matrix.}
    \item{\code{Ncol}:}{Object of class \code{"integer", from class
	"pdMat"}, number of columns (and rows) in the positive-definite
      matrix.}
    \item{\code{factor}:}{Object of class \code{"matrix", from class
	"pdMat"}, a square root factor of the positive-definite matrix.}
    \item{\code{logDet}:}{Object of class \code{"numeric", from class
	"pdMat"} the logarithm of the absolute value of the determinant
      of the square root factor or, equivalently, half the logarithm of
      the determinant of the positive-definite matrix.}
  }
}
\section{Extends}{
Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{EMupdate<-}{\code{signature(x = "pdLogChol", nlev = "numeric",
	value = "matrix")}: update the \code{pdLogChol} object in the EM
      algorithm for a mixed-effects model.}
    \item{LMEgradient}{\code{signature(x = "pdLogChol", A = "matrix",
	nlev = "numeric")}: evaluate the gradient of the log-likelihood
      in a linear mixed-effects model.}
    \item{coef<-}{\code{signature(object = "pdLogChol", value =
	"numeric")}: assign the parameter.}
    \item{coerce}{\code{signature(from = "pdLogChol", to = "pdmatrix")}:
      extract the positive-definite matrix represented by the object.}
    \item{pdgradient}{\code{signature(x = "pdLogChol")}: the gradient of
      the positive definite matrix with respect to the parameter vector.}
    \item{solve}{\code{signature(a = "pdLogChol", b = "missing")}: a
      \code{pdLogChol} object representing the inverse of the
      positive-definite matrix represented by this object.}
    \item{summary}{\code{signature(object = "pdLogChol")}: summarize the
      object.}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\seealso{\code{\link{pdMat-class}}}
\examples{
m1 <- pdLogChol(~ age)
coef(m1) <- rnorm(3)
print(m1)
solve(m1)
}
\keyword{classes}
\eof
\name{pdMat-class}
\docType{class}
\alias{pdMat-class}
\alias{coef,pdMat-method}
\alias{coerce,pdMat,pdfactor-method}
\alias{dim,pdMat-method}
\alias{formula,pdMat-method}
\alias{names,pdMat-method}
\alias{names<-,pdMat-method}
\alias{show,pdMat-method}
\alias{solve,pdMat-method}
\alias{summary,pdMat-method}
\title{Class pdMat, positive-definite matrices}
\description{A virtual class of parameterized positive-definite
  symmetric matrices.  This class describes the slots and methods that
  actual classes of positive-definite matrices are expected to
  incorporate.  Some classes that inherit from \code{pdMat} have
  additional slots and methods.}
\section{Objects from the Class}{
  Objects of class \code{pdMat} are not constructed directly; only
  objects from classes that inherit from \code{pdMat} are constructed.
}
\section{Slots}{
  \describe{
    \item{\code{form}:}{Object of class \code{"formula"}, holds the
      formula for the object }
    \item{\code{Names}:}{Object of class \code{"character"} holding the
      names of the rows (and columns) of the positive-definite symmetric
      matrix represented by the object.}
    \item{\code{param}:}{\code{"numeric"} - the parameter vector.}
    \item{\code{Ncol}:}{\code{"integer"} - the number of columns (and
      rows) in the matrix.}
    \item{\code{factor}:}{\code{"matrix"} - a square-root factor of the
      matrix.}
    \item{\code{logDet}:}{\code{"numeric"} - the logarithm of the
      determinant of the factor.}
  }
}

\section{Methods}{
  \describe{
    \item{coerce}{\code{signature(from = "pdMat", to = "pdfactor")}:
      extract a square-root factor of the matrix represented by the
      object.  This factor has a \code{logDet} slot giving the
      logarithm of its determinant.  In the case of \code{pdLogChol}
      these are both scalars and the \code{logDet} attribute is the
      logarithm of the absolute value of the factor.}
    \item{corMatrix}{\code{signature(object = "pdMat")}: Extract the
      correlation matrix corresponding to the positive-definite matrix
      represented by the object.  This method is present for back
      compatibility only.  The preferred way of extracting the
      correlation matrix is to coerce the object to the
      \code{"corrmatrix"} class.}
    \item{dim}{\code{signature(x = "pdMat")}: the dimensions of the
      positive-definite matrix represented by the object.} 
    \item{formula}{\code{signature(x = "pdMat")}: extract the formula }
    \item{isInitialized}{\code{signature(object = "pdMat")}:
      \code{TRUE} if the object has been initialized, otherwise \code{FALSE}.}
    \item{logDet}{\code{signature(object = "pdMat", covariate =
	"missing")}: the logarithm of the determinant of the factor of
      the positive-definite matrix represented by the object.} 
    \item{names}{\code{signature(x = "pdMat")}: extract a vector of
      names, which are both the column names and the row names of the
      positive-definite matrix represented by the object.}
    \item{names<-}{\code{signature(x = "pdMat")}: assign the names,}
    \item{pdFactor}{\code{signature(object = "pdMat")}: Extract the
      square root factor of positive-definite symmetric matrix
      represented by the object.  This method is present for back
      compatibility only.  The preferred way of extracting the factor is
      to coerce the object to the \code{"pdfactor"} class.}
    \item{pdMatrix}{\code{signature(object = "pdMat")}: Extract the
      positive-definite symmetric matrix represented by the object.
      This method is present for back compatibility only.  The preferred
      way of extracting the positive-definite symmetric matrix is to
      coerce the object to the \code{"pdmatrix"} class.}
    \item{show}{\code{signature(x = "pdMat")}: show the object.}
    \item{solve}{\code{signature(a = "pdMat", b = "missing")}: Create an
      object of the same class representing the inverse of the
      positive-definite matrix.}
    \item{summary}{\code{signature(object = "pdMat")}:}
  }
}
\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\note{
  \code{pdMat} objects are primarily used to represent the variance-covariance
  matrix or the precision matrix of random-effects terms in
  mixed-effects models.  Frequently they are constructed from a formula
  only in the call to the mixed-effects modelling function then assigned
  a value as part of the initialization of the model.
}
\seealso{\code{\link{pdCompSymm-class}}, \code{\link{pdDiag-class}},
  \code{\link{pdLogChol-class}}, \code{\link{pdIdent-class}}
}
%\examples{}
\keyword{classes}
\eof
\name{pdMatrix}
\docType{methods}
\docType{genericFunction}
\alias{pdMatrix}
\alias{pdMatrix-methods}
\alias{pdMatrix,pdMat-method}
\title{Return the positive-definite matrix}
\description{This generic function extracts the positive-definite matrix
  represented by \code{object}.

  This function is deprecated.  Use coersion to the \code{pdmatrix}
  class instead.
}
\usage{
pdMatrix(object)
}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix, which must have been initialized
   (i.e. \code{length(coef(object)) > 0}).} 
}
\value{
  A \code{pdmatrix} object.
}
\seealso{\code{\link{pdmatrix-class}}, \code{\link{pdMat-class}}}
%\examples{}
\keyword{models}
\eof
\name{pdNatural-class}
\docType{class}
\alias{pdNatural-class}
\alias{pdNatural}
\alias{pdNatural,formula,missing,missing,missing-method}
\alias{pdNatural,pdMat,missing,missing,missing-method}
\title{Class "pdNatural", positive-definite matrices}
\description{A class of general, positive-definite
  symmetric matrices parameterized by the logarithm of the diagonal
  elements and Fisher's z transformation of the correlations.}
\section{Objects from the Class}{

  Objects can be created by calls of the form \code{new("pdNatural",
    ...)} or by the generic constructor function \code{pdNatural}.
  Frequently the constructor is given a formula only, creating an
  uninitialized \code{pdNatural} object which is later assigned a
  value.

  \code{pdNatural} objects are primarily used to represent the
  variance-covariance matrix or the precision matrix of random-effects
  terms in mixed-effects models.
}
\section{Slots}{
  \describe{
    \item{\code{param}:}{Object of class \code{"numeric", from class
	"pdMat"}, a parameter vector of length
      \eqn{[q(q+1)]/2}{[q(q+1)]/2} where \eqn{q}{q} is the number of
      rows (and columns) in the positive-definite matrix.}
    \item{\code{form}:}{Object of class \code{"formula", from class
	"pdMat"}, a formula for the object}
    \item{\code{Names}:}{Object of class \code{"character", from class
	"pdMat"}, names for the rows (and columns) of the
      positive-definite matrix.}
  }
}
\section{Extends}{
Class \code{"pdMat"}, directly.
}
\section{Methods}{
  \describe{
    \item{EMupdate<-}{\code{signature(x = "pdNatural", nlev = "numeric",
	value = "matrix")}: update the \code{pdNatural} object in the EM
      algorithm for a mixed-effects model.}
    \item{LMEgradient}{\code{signature(x = "pdNatural", A = "matrix",
	nlev = "numeric")}: evaluate the gradient of the log-likelihood
      in a linear mixed-effects model.}
    \item{coef}{\code{signature(object = "pdNatural")}: extract the
      parameter.}
    \item{coef<-}{\code{signature(object = "pdNatural", value =
	"numeric")}: assign the parameter.}
    \item{coerce}{\code{signature(from = "pdNatural", to = "pdmatrix")}:
      extract the positive-definite matrix represented by the object.}
    \item{coerce}{\code{signature(from = "pdNatural", to = "pdfactor")}:
      extract a square-root factor of the matrix represented by the
      object.  This factor has a \code{logDet} attribute giving the
      logarithm of its determinant.  In the case of \code{pdNatural}
      these are both scalars and the \code{logDet} attribute is the
      logarithm of the absolute value of the factor.}
    \item{dim}{\code{signature(x = "pdNatural")}: the dimensions of the
      positive-definite matrix represented by the object.} 
    \item{isInitialized}{\code{signature(object = "pdNatural")}: a
      logical scalar indicating if the object is initialized. }
    \item{logDet}{\code{signature(object = "pdNatural", covariate =
	"missing")}: the logarithm of the determinant of the factor of
      the positive-definite matrix represented by the object.} 
    \item{pdgradient}{\code{signature(x = "pdNatural")}: the gradient of
      the positive definite matrix with respect to the parameter vector.}
    \item{solve}{\code{signature(a = "pdNatural", b = "missing")}: a
      \code{pdNatural} object representing the inverse of the
      positive-definite matrix represented by this object.}
    \item{summary}{\code{signature(object = "pdNatural")}: summarize the
      object.}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
\seealso{\code{\link{pdMat-class}}}
%\examples{}
\keyword{classes}
\eof
\name{pdfactor-class}
\docType{class}
\alias{pdfactor-class}
\title{Factors of positive-definite matrices}
\description{If \eqn{\Sigma}{S} is a positive-definite matrix then any
  square matrix \eqn{L}{L} such that \eqn{\Sigma = L'L}{S = L'L} is said
  to be a square-root factor of \eqn{\Sigma}{S}.  The \code{pdfactor}
  class is the class of such matrices.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pdfactor",
    ...)}.  More frequently they are created by coercing a \code{pdMat}
  or \code{pdmatrix} object to the \code{pdfactor} class.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{\code{"matrix"}: the factor itself, a numeric,
      square matrix}
    \item{\code{logDet}:}{\code{"numeric"}: the logarithm of the
      absolute value of the determinant of the factor.}
  }
}
\section{Extends}{
Class \code{"matrix"}, from data part.
Class \code{"structure"}, by class "matrix".
Class \code{"array"}, by class "matrix".
Class \code{"vector"}, by class "matrix".
}
\section{Methods}{
  \describe{
    \item{coerce}{\code{signature(from = "pdIdent", to = "pdfactor")}:
      extract the pdfactor from a \code{pdIdent} object.}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdMat-class}}, \code{\link{pdmatrix-class}}}
%\examples{}
\keyword{classes}
\eof
\name{pdgradNumeric}
\alias{pdgradNumeric}
\title{Evaluate the pdmatrix gradient numerically}
\description{
  Evaluate the pdmatrix gradient via finite differences.  This function
  is used to validate the \code{pdgradient} methods for \code{pdMat} classes.
}
\usage{
pdgradNumeric(x)
}
\arguments{
  \item{x}{A \code{pdMat} object}
}
%\details{}
\value{
  A finite-difference approximation to the gradient of the
  \code{pdmatrix} represented by \code{x} with respect to the parameters
  of \code{x}.
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
%\seealso{}
\examples{
}
\keyword{array}
\keyword{algebra}
\keyword{models}

\eof
\name{pdgradient}
\docType{methods}
\alias{pdgradient}
\alias{pdgradient-methods}
\alias{pdgradient,pdIdent-method}
\alias{pdgradient,pdLogChol-method}
\alias{pdgradient,pdMat-method}
\title{Evaluated the pdmatrix gradient}
\description{
  This function is generic.  It returns the gradient of the positive
  definite matrix represented by \code{x} with respect to the parameters
  of \code{x}.
}
\usage{
pdgradient(x)
}
\arguments{
  \item{x}{An object of a suitable class, usually one of
  the classes that inherit from \code{pdMat}.}
}
\section{Methods}{
  \describe{
    \item{x = pdIdent}{gradient for the \code{pdIdent} class}
    \item{x = pdLogChol}{gradient for the \code{pdLogChol} class}
    \item{x = pdMat}{gradient for the \code{pdMat} class, implemented
      using \code{pdgradNumeric}}
%    \item{x = pdScalar}{gradient for the \code{pdScalar} class}
  }
}
%\details{}
\value{
  A numeric array of dimension \code{c(dim(x), length(coef(x)))} that
  is the gradient of the \code{pdmatrix} represented by \code{x} with
  respect to the parameters of \code{x}.
}
%\references{}
\author{Douglas Bates \email{bates@stat.wisc.edu}
  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdgradNumeric}}}
\examples{
}
\keyword{array}
\keyword{algebra}
\keyword{models}
\keyword{methods}
\eof
\name{pdmatrix-class}
\docType{class}
\alias{pdmatrix-class}
\title{Positive-definite matrices }
\description{The class of positive-definite, symmetric matrices.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pdmatrix", ...)}.
  More frequently they are created by coercing a \code{pdMat}
  object to the \code{pdmatrix} class.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{"matrix"}, which should
      be a numeric matrix.}
  }
}
\section{Extends}{
  Class \code{"matrix"}, from data part.
  Class \code{"structure"}, by class "matrix".
  Class \code{"array"}, by class "matrix".
  Class \code{"vector"}, by class "matrix".
}
\section{Methods}{
  \describe{
    \item{coerce}{\code{signature(from = "pdDiag", to = "pdmatrix")}: extract the positive-definite matrix }
    \item{coerce}{\code{signature(from = "pdIdent", to = "pdmatrix")}:
      extract the positive-definite matrix}
    \item{coerce}{\code{signature(from = "pdLogChol", to = "pdmatrix")}:
      extract the positive-definite matrix}
    \item{coerce}{\code{signature(from = "pdNatural", to = "pdmatrix")}:
      extract the positive-definite matrix}
%    \item{coerce}{\code{signature(from = "pdScalar", to = "pdmatrix")}:
%      extract the positive-definite matrix}
    \item{coerce}{\code{signature(from = "matrix", to = "pdmatrix")}:
      create a \code{pdmatrix} object from a matrix (does not check for
      symmetry or positive-definiteness)}
    \item{coerce}{\code{signature(from = "pdfactor", to = "pdmatrix")}:
      create a positive-definite matrix as the crossproduct of the factor.}
    \item{coerce}{\code{signature(from = "pdmatrix", to = "pdfactor")}:
      convert a \code{pdmatrix} object to its Cholesky factor}
  }
}
%\references{}
%\author{Douglas Bates \email{bates@stat.wisc.edu} and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{pdfactor-class}}, \code{\link{pdMat-class}}}
%\examples{}
\keyword{classes}
\eof
% $Id: pooledSD.Rd,v 1.1 2003/06/03 18:31:34 bates Exp $
\name{pooledSD}
\title{Extract pooled standard deviation}
\usage{
pooledSD(object)
}
\alias{pooledSD}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}.}
}
\description{
  The pooled estimated standard deviation is obtained by adding together
  the residual sum of squares for each non-null element of
  \code{object}, dividing by the sum of the corresponding residual
  degrees-of-freedom, and taking the square-root.
}
\value{
  the pooled standard deviation for the non-null elements of
  \code{object}, with an attribute \code{df} with the number of
  degrees-of-freedom used in the estimation.
}
%\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lmList}}, \code{\link{lm}}}
\examples{
data(Orthodont, package = "nlme")
fm = lmList(distance ~ age | Subject, Orthodont)
pooledSD(fm)
}
\keyword{models}
\eof
% $Id: random.effects.Rd,v 1.1 2002/11/27 22:49:45 bates Exp $
\name{random.effects}
\docType{genericFunction}
\docType{methods}
\title{Extract Random Effects}
\alias{random.effects}
\alias{ranef}
\alias{ranef-methods}
\alias{ranef,ANY-method}
\alias{ranef,reStruct-method}
\usage{
random.effects(object, \dots)
ranef(object, \dots)
}
\description{
  A generic function to extract the random effects.
}
\arguments{
 \item{object}{an object of a class from which random effects
   estimates can be extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\value{
  a list of matrices.
}
%\seealso{}
\examples{
## see the method function documentation
}
\keyword{models}
\keyword{methods}
\eof
\name{reStruct-class}
\docType{class}
\alias{reStruct-class}
\alias{LMEgradient,reStruct,missing,missing-method}
\alias{VarCorr,reStruct-method}
\alias{coef,reStruct-method}
\alias{fitted,reStruct-method}
\alias{fixef<-,reStruct,numeric-method}
\alias{getFixDF,reStruct-method}
\alias{getGroups,reStruct,missing,ANY,missing,missing-method}
\alias{getResponse,reStruct-method}
\alias{logLik,reStruct-method}
\alias{model.matrix,reStruct-method}
\alias{model.matrix<-,reStruct-method}
\alias{response<-,reStruct,numeric-method}
\alias{summary,reStruct-method}
\alias{weighted<-,reStruct,numeric-method}
\title{Class "reStruct"}
\description{The random-effects model structure in a linear
  mixed-effects model or a generalized linear mixed-effects model or a
  nonlinear mixed-effects model.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("reStruct", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{random}:}{A \code{"list"} of \code{lmeLevel} objects
      giving the levels of random effects in the model, the
      fixed-effects, and the response or the working residual.}
    \item{\code{fixed}:}{The \code{"formula"} of the response and
      fixed-effects terms.}
    \item{\code{offset}:}{\code{"numeric"}: the offset in the model, if present.}
    \item{\code{dirtyDecomposed}:}{\code{"logical"}: if \code{TRUE} the
      decomposed matrix must be recalculated.}
    \item{\code{useWeighted}:}{\code{"logical"}: if \code{TRUE}
      calculate and use the weighted model matrix to form the
      decompositions; otherwise use the original model matrix.}
    \item{\code{dirtyStored}:}{\code{"logical"}: if \code{TRUE} the
      matrix of stored decompositions must be recalculated.}
    \item{\code{dirtyBbetas}:}{\code{"logical"}: if \code{TRUE} the
      BLUP's and the conditional estimates of the fixed-effects must be
      recalculated.}
    \item{\code{logLik}:}{\code{"numeric"}: the log-likelihood at the
      current parameter values.}
    \item{\code{REML}:}{\code{"logical"}: if \code{TRUE} the parameters
      will be estimated according to the REML criterion}
    \item{\code{analyticHessian}:}{\code{"logical"}: if \code{TRUE} the
      \code{hessianArray} slot of the lmeLevel objects are calculated
      during optimization.}
    \item{\code{reverseOrder}:}{\code{"integer"}: the permutation of the
      rows that provides the original order.}
    \item{\code{origOrder}:}{\code{"integer"}: the inverse of the
      \code{reverseOrder} permutation.}
    \item{\code{original}:}{\code{"matrix"}: the original model matrix,
      including the original response in the last column.}
    \item{\code{weighted}:}{\code{"matrix"}: the model matrix and
      responses after applying weights.}
    \item{\code{stored}:}{\code{"matrix"}: a model matrix of
      intermediate decomposition results needed for evaluating the BLUPs
      and the EM or ECME iterations.}
    \item{\code{decomposed}:}{\code{"matrix"}: the model matrix after
      predecomposition.  This generally has many few rows than
      \code{original}.}
    \item{\code{bbetas}:}{\code{"numeric"}: the BLUPs and the conditional
       estimates of the fixed-effects parameters at the current values
       of the relative precision matrices.}
    \item{\code{dontCopy}:}{\code{"logical"}: if \code{TRUE} it indicates
      that this object has just been created and is only assigned to one
      name.  In these circumstances changes are made directly on the
      object without copying. This is dangerous.  You probably don't
      want to modify this setting.}
    \item{\code{assign.X}:}{Object of class \code{"ANY"}: the
      \code{assign} attribute from the model matrix for the fixed effects.}
  }
}
\section{Methods}{
  \describe{
    \item{EMsteps<-}{\code{signature(x = "reStruct", value = "list")}:
      perform the EM iterations}
    \item{LMEgradient}{\code{signature(x = "reStruct", A = "missing",
	nlev = "missing")}: evaluate the gradient of the linear
      mixed-effects profiled log-likelihood.}
    \item{LMEoptimize<-}{\code{signature(x = "reStruct", value =
	"list")}: Optimize with \code{optim} or \code{nlm}.}
    \item{VarCorr}{\code{signature(x = "reStruct")}: Extract the
      variances and covariances of the random effects.}
    \item{coef}{\code{signature(object = "reStruct")}: return the
      current parameters in the object.}
    \item{coef<-}{\code{signature(object = "reStruct", value =
	"numeric")}: assign the parameters to the object.}
    \item{fitted}{\code{signature(object = "reStruct")}: extract the
      fitted values.}
    \item{fixef}{\code{signature(object = "reStruct")}: extract the
      fixed effects.}
    \item{fixef<-}{\code{signature(object = "reStruct", value =
	"numeric")}: assign the fixed effects.}
    \item{getFixDF}{\code{signature(object = "reStruct")}: extract the
      denominator degrees of freedom for the fixed-effects terms.}
    \item{getGroups}{\code{signature(object = "reStruct", form =
	"missing", level = "ANY", data = "missing", sep = "missing")}:
      return the grouping factors for the random effects}
    \item{getResponse}{\code{signature(object = "reStruct")}: extract
      the response.}
    \item{logLik}{\code{signature(object = "reStruct")}: return the
      log-likelihood at the current parameter values. }
    \item{model.matrix}{\code{signature(object = "reStruct")}: return the model
      matrix for the object.}
    \item{model.matrix<-}{\code{signature(x = "reStruct", value =
	"matrix")}: assign the model matrix to the object.}
    \item{ranef}{\code{signature(object = "reStruct")}: extract the
      random effects predictors.}
    \item{response<-}{\code{signature(x = "reStruct", value =
	"numeric")}: set the working response (used in GLMM models).}
    \item{summary}{\code{signature(object = "reStruct")}: summarize the object.}
    \item{weighted<-}{\code{signature(x = "reStruct", value =
	"matrix")}: update the weights.}
  }
}
%\references{}
%\author{Saikat DebRoy \email{saikat@stat.wisc.edu} and Douglas Bates \email{bates@stat.wisc.edu}}
%\note{}
%\seealso{}
\examples{}
\keyword{classes}
\eof
\name{reStruct}
\docType{methods}
\docType{genericFunction}
\alias{reStruct}
\alias{reStruct,formula,list,data.frame,missing,logical-method}
\alias{reStruct,formula,list,data.frame,numeric,logical-method}
\title{Create an reStruct object}
\description{
  Create an \code{reStruct} object from values for its slots.
}
\usage{
reStruct(fixed, random, data, weights, REML, nextraCols, analyticHessian)
}
\arguments{
  \item{fixed}{a formula of the form \code{response ~ terms} where
    \code{response} is an expression for the response and \code{terms}
    are one or more terms that define the fixed-effects.}
  \item{random}{a named list of one-sided formulas.  The names in the
    list evaluate to the grouping factors and the right-hand side of the
    formula is an expression for a \code{pdMat} object that itself
    contains a formula.  The order of the components of the list
    determines the nesting structure in that the second component is
    nested within the first, the third within the second, and so on.}
  \item{data}{a \code{data.frame} or \code{model.frame} in which the
    \code{fixed} and \code{random} arguments can be evaluated.}
  \item{weights}{a numeric vector of prior case weights for the
    observations. Defaults to equal weights.}
  \item{REML}{a \code{logical} value: if \code{TRUE} the REML
    criterion will be used for parameter estimation, otherwise the ML
    criterion will be used.}
  \item{nextraCols}{an \code{integer} giving the number of extra columns
    to be added to the stored matrices.}
  \item{analyticHessian}{a \code{ilogical} value: if \code{TRUE} the
    analytic Hessian of the objective can be calculated using the
    LMEhessian function.}
}
%\details{}
\value{
  An \code{\link[lme4:reStruct-class]{reStruct}} object.
}
%\references{}
%\author{}
%\note{}
\seealso{\code{\link{reStruct-class}}, \code{\link{lmeLevel-class}}}
%\examples{}
\keyword{models}
\eof
\name{responseGets}
\docType{methods}
\docType{genericFunction}
\alias{response<-}
\alias{response<-,reStruct,matrix-method}
\title{Modify the response}
\description{
  Modify the reponse in an object of a suitable class. This function is
  generic.
}
\usage{
response(x) <- value
}
\arguments{
  \item{x}{An object of a suitable class.}
  \item{value}{updated response vector to assign to the object \code{x}.}
}
%\details{}
\value{
  The object \code{x} with the response updated.
}
\section{Methods}{
  \describe{
    \item{x = reStruct, value = matrix}{modify the response in an
      \code{reStruct} object}
  }
}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{reStruct-class}}}
%\examples{}
\keyword{manip}
\keyword{models}
\eof
\name{s3bbx}
\alias{s3bbx}
\non_function{}
\title{Covariates in the Rodriguez and Goldman simulation}
\usage{data(s3bbx)}
\description{
The \code{s3bbx} data frame has 2449 rows and 6 columns of the
covariates in the simulation by Rodriguez and Goldman of multilevel
dichotomous data.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{child}{a numeric vector identifying the child}
    \item{family}{a numeric vector identifying the family}
    \item{community}{a numeric vector identifying the community}
    \item{chldcov}{a numeric vector of the child-level covariate}
    \item{famcov}{a numeric vector of the family-level covariate}
    \item{commcov}{a numeric vector of the community-level covariate}
  }
}
\source{
  \url{http://data.princeton.edu/multilevel/simul.htm}
}
\references{
  Rodriguez, Germn and Goldman, Noreen (1995), "An assessment of
  estimation procedures for multilevel models with binary responses",
  \emph{Journal of the Royal Statistical Society, Series A}, \bold{158},
  73-89.
}
\examples{
data(s3bbx)
str(s3bbx)
}
\keyword{datasets}
\eof
\name{s3bby}
\alias{s3bby}
\non_function{}
\title{Responses simulated by Rodriguez and Goldman}
\description{
  A matrix of the results of 100 simulations of dichotomous multilevel
  data. The rows correspond to the 2449 births for which the covariates
  are given in \code{\link{s3bbx}}.  The elements of the matrix are all
  0, indicating no modern prenatal care, or 1, indicating model prenatal
  care.  These were simulated with "large" variances for both the family
  and the community random effects.
}
\usage{data(s3bby)}
\format{
  An integer matrix with 2449 rows and 100 columns.
}
\source{
  \url{http://data.princeton.edu/multilevel/simul.htm}
}
\references{
  Rodriguez, Germn and Goldman, Noreen (1995), "An assessment of
  estimation procedures for multilevel models with binary responses",
  \emph{Journal of the Royal Statistical Society, Series A}, \bold{158},
  73-89.
}
\examples{
data(s3bby)
str(s3bby)
}
\keyword{datasets}
\eof
\name{solve-methods}
\docType{methods}
\alias{solve-methods}
\alias{solve,pdBlocked,missing-method}
\alias{solve,pdDiag,missing-method}
\alias{solve,pdIdent,missing-method}
\alias{solve,pdLogChol,missing-method}
\alias{solve,pdMat,missing-method}
\alias{solve,pdNatural,missing-method}
%\alias{solve,pdScalar,missing-method}
\title{Methods for Function solve}
\description{The \code{solve} methods for \code{pdMat} classes all use
  a single argument call.  They return a matrix of the same class as
  \code{a}.
  }
\section{Methods}{
  \describe{
    \item{a = pdBlocked, b = missing}{invert a \code{pdBlocked} object}
    \item{a = pdDiag, b = missing}{invert a \code{pdDiag} object}
    \item{a = pdIdent, b = missing}{invert a \code{pdIdent} object}
    \item{a = pdLogChol, b = missing}{invert a \code{pdLogChol} object}
    \item{a = pdMat, b = missing}{invert a \code{pdMat} object}
    \item{a = pdNatural, b = missing}{invert a \code{pdNatural} object}
  }
}
\keyword{manip}
\keyword{algebra}
\keyword{array}
\keyword{methods}
\eof
\name{summary.reStruct-class}
\docType{class}
\alias{summary.reStruct-class}
\alias{show,summary.reStruct-method}
\title{Class "summary.reStruct"}
\description{A summary of an \code{\link[lme4:reStruct-class]{reStruct}}
  object.  The summary contains information derived from the original
  object.} 
\section{Objects from the Class}{
  Objects from the class are usually constructed by the
  \code{\link[base]{summary}} generic function applied to an
  \code{\link[lme4:reStruct-class]{reStruct}} object.
}
\section{Slots}{
  \describe{
    \item{\code{fixed}:}{\code{"formula"}: the formula defining the
      fixed-effects.}
    \item{\code{coefficients}:}{A matrix containing the fixed-effects
      coefficients, their standard errors and the t statistics.}
    \item{\code{scale}:}{\code{"numeric"}.  The estimate of the scale
      parameter in the distribution of the per-observation random
      variation, if it is estimated.  In the case of a \code{"lme"}
      model this is the estimated standard deviation of the lowest
      level noise term in the model.}
    \item{\code{denomDF}:}{\code{"integer"}: vector of denominator
      degrees of freedom.}
    \item{\code{REML}:}{\code{"logical"}: was the REML criterion used to
      estimate the parameters?}
    \item{\code{random}:}{\code{"lmeLevelList"}: as in the
      \code{\link[lme4:reStruct-class]{reStruct}} object from which this
      was derived.}
    \item{\code{ngrps}:}{\code{"integer"}: number of groups at each
      level of random effects.}
    \item{\code{nobs}:}{\code{"integer"}: total number of observations.}
    \item{\code{corFixed}:}{\code{"corrmatrix"}: Correlation of the
      estimates of the fixed-effects parameters.}
    \item{\code{reSumry}:}{\code{"list"}: a summary of the random
      effects parameter estimates}
    \item{\code{useScale}:}{\code{"logical"}: should the \code{"scale"}
      slot be used when calculating the variances and covariances of the
      random effects?}
    \item{\code{showCorrelation}:}{\code{"logical"}: should the
      \code{"corFixed"} slot be shown in the summary.}
  }
}

\section{Methods}{
  \describe{
    \item{show}{\code{signature(x = "summary.reStruct")}: show the object}
  }
}
\seealso{\code{\link{reStruct-class}}, \code{\link[base]{summary}}}
%\examples{}
\keyword{classes}
\eof
\name{weightedGets}
\docType{methods}
\docType{genericFunction}
\alias{weighted<-}
\alias{weighted<-,reStruct,matrix-method}
\title{Modify the weights}
\description{
  Modify the weights in an object of a suitable class. This function is
  generic.
}
\usage{
weighted(x) <- value
}
\arguments{
  \item{x}{An object of a suitable class.}
  \item{value}{weights to assign to the object \code{x}.}
}
%\details{}
\value{
  The object \code{x} with the weights updated.
}
\section{Methods}{
  \describe{
    \item{x = reStruct, value = matrix}{modify the weights in an
      \code{reStruct} object}
  }
}
%\author{Douglas Bates \email{bates@stat.wisc.edu}
%  and Saikat DebRoy \email{saikat@stat.wisc.edu}}
%\note{}
\seealso{\code{\link{reStruct-class}}}
%\examples{}
\keyword{manip}
\keyword{models}
\eof
