\name{AutoD2}
\alias{AutoD2}
\alias{CrossD2}
\alias{CenterD2}
\title{ AutoD2, CrossD2 or CenterD2 analysis of a multiple time-series }
\description{
  Compute and plot multiple autocorrelation using Mahalanobis generalized distance D2. AutoD2 uses the same multiple time-series. CrossD2 compares two sets of multiple time-series having same size (same number of descriptors). CenterD2 compares subsamples issued from a single multivariate time-series, aiming to detect discontinuities.
}
\usage{
AutoD2(series, lags=c(1, nrow(series)/3), step=1, plotit=TRUE,
        add=FALSE, ...)
CrossD2(series, series2, lags=c(1, nrow(series)/3), step=1,
        plotit=TRUE, add=FALSE, ...)
CenterD2(series, window=nrow(series)/5, plotit=TRUE, add=FALSE,
        type="l", level=0.05, lhorz=TRUE, lcol=2, llty=2, ...)
}
\arguments{
  \item{series}{ regularized multiple time-series }
  \item{series2}{ a second set of regularized multiple time-series }
  \item{lags}{ minimal and maximal lag to use. By default, 1 and a third of the number of observations in the series respectively }
  \item{step}{ step between successive lags. By default, 1 }
  \item{window}{ the window to use for CenterD2. By default, a fifth of the total number of observations in the series }
  \item{plotit}{ if \code{TRUE} then also plot the graph }
  \item{add}{ if \code{TRUE} then the graph is added to the current figure }
  \item{type}{ The type of line to draw in the CenterD2 graph. By default, a line without points }
  \item{level}{ The significance level to consider in the CenterD2 analysis. By default 5\% }
  \item{lhorz}{ Do we have to plot also the horizontal line representing the significance level on the graph? }
  \item{lcol}{ The color of the significance level line. By default, color 2 is used }
  \item{llty}{ The style for the significance level line. By default: \code{llty=2}, a dashed line is drawn}
  \item{\dots}{ additional graph parameters }
}
\value{
  An object of class 'D2' which contains:
  \item{lag }{The vector of lags}
  \item{D2 }{The D2 value for this lag}
  \item{call }{The command invoked when this function was called}
  \item{data }{The series used}
  \item{type }{The type of 'D2' analysis: 'AutoD2', 'CrossD2' or 'CenterD2'}
  \item{window }{The size of the window used in the CenterD2 analysis}
  \item{level }{The significance level for CenterD2}
  \item{chisq }{The chi-square value corresponding to the significance level in the CenterD2 analysis}
  \item{units.text }{Time units of the series, nicely formatted for graphs}
}
\references{
Cooley, W.W. & P.R. Lohnes, 1962. \emph{Multivariate procedures for the behavioural sciences.} Whiley & sons.

Dagnlie, P., 1975. \emph{Analyse statistique  plusieurs variables.} Presses Agronomiques de Gembloux.

Ibanez, F., 1975. \emph{Contribution  l'analyse mathmatique des vnements en cologie planctonique: optimisations mthodologiques; tude exprimentale en continu  petite chelle du plancton ctier.} Thse d'tat, Paris VI.

Ibanez, F., 1976. \emph{Contribution  l'analyse mathmatique des vnements en cologie planctonique. Optimisations mthodologiques.} Bull. Inst. Ocanogr. Monaco, 72:1-96.

Ibanez, F., 1981. \emph{Immediate detection of heterogeneities in continuous multivariate oceanographic recordings. Application to time series analysis of changes in the bay of Villefranche sur mer.} Limnol. Oceanogr., 26:336-349.

Ibanez, F., 1991. \emph{Treatment of the data deriving from the COST 647 project on coastal benthic ecology: The within-site analysis.} In: B. Keegan (ed), \emph{Space and time series data analysis in coastal benthic ecology}, p 5-43.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\section{WARNING }{If data are too heterogeneous, results could be biased (a singularity matrix appears in the calculations).} 

\seealso{ \code{\link[base]{mahalanobis}}, \code{\link[ts]{acf}} }

\examples{
data(marphy)
marphy.ts <- as.ts(as.matrix(marphy[, 1:3]))
AutoD2(marphy.ts)
marphy.ts2 <- as.ts(as.matrix(marphy[, c(1, 4, 3)]))
CrossD2(marphy.ts, marphy.ts2)
# This is not identical to:
CrossD2(marphy.ts2, marphy.ts)
marphy.d2 <- CenterD2(marphy.ts, window=16)
lines(c(17, 17), c(-1, 15), col=4, lty=2)
lines(c(25, 25), c(-1, 15), col=4, lty=2)
lines(c(30, 30), c(-1, 15), col=4, lty=2)
lines(c(41, 41), c(-1, 15), col=4, lty=2)
lines(c(46, 46), c(-1, 15), col=4, lty=2)
text(c(8.5, 21, 27.5, 35, 43.5, 57), 11, labels=c("Peripheral Zone", "D1",
        "C", "Front", "D2", "Central Zone")) # Labels
time(marphy.ts)[marphy.d2$D2 > marphy.d2$chisq]
}

\keyword{ ts }
\keyword{ multivariate }
\keyword{ htest }
\eof
\name{GetUnitText}
\alias{GetUnitText}
\title{ Format a nice time units for labels in graphs }
\description{
  This is an internal function called by some \code{plot()} methods. Considering the time series 'units' attribute and the frequency of the observations in the series, the function returns a string with a pertinent time unit. For instance, if the unit is 'years' and the frequency is 12, then data are monthly sampled and \code{GetUnitText()} returns the string "months"
}
\usage{
GetUnitText(series)
}
\arguments{
  \item{series}{ a regular time series (a 'rts' object in Splus, or a 'ts' object in \R) }
}
\value{
  a string with the best time unit for graphs
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{daystoyears}}, \code{\link{yearstodays}} }

\examples{
timeser <- ts(1:24, frequency=12)           # 12 observations per year
if (exists("is.R") && is.function(is.R) && is.R()) {  # We are in R
attr(timeser, "units") <- "years"           # time in years for 'ts' object
} else {                                              # We are in Splus
attr(attr(timeser, "tspar"), "units") <- "years" # Idem for Splus 'rts' object
}
GetUnitText(timeser)                        # formats unit (1/12 year=months)
}
\keyword{ ts }
\eof
\name{abund}
\alias{abund}
\alias{extract.abund}
\alias{identify.abund}
\alias{lines.abund}
\alias{plot.abund}
\alias{print.abund}
\alias{print.summary.abund}
\alias{summary.abund}
\title{ Sort variables by abundance }
\description{
  Sort variables (usually species in a species x stations matrix) in function of their abundance, either in number of non-null values, or in number of individuals (in log). The \code{f} coefficient allows adjusting weight given to each of these two criteria.
}
\synopsis{
abund(x, f=0.2) 
extract.abund(e, n, left=TRUE, ...)
identify.abund(x, label.pts=FALSE, lvert=TRUE, lvars=TRUE, col=2, lty=2, ...)
lines.abund(x, n=x$n, lvert=TRUE, lvars=TRUE, col=2, lty=2, ...)
plot.abund(x, n=x$n, lvert=TRUE, lvars=TRUE, lcol=2, llty=2, all=TRUE, dlab=c("cumsum", "\% log(ind.)", "\% non-zero"), dcol=c(1,2,4), dlty=c(par("lty"), par("lty"), par("lty")), dpos=c(1.5, 20), type="l", xlab="variables", ylab="abundance", main=paste("Abundance sorting for:",x$data, "with f =", round(x$f, 4)), ...)
print.abund(x, ...)
print.summary.abund(x, ...)
summary.abund(object, ...)
}
\usage{
abund(x, f=0.2)
\method{summary}{abund}(abd)
\method{plot}{abund}(abd, n=abd$n, lvert=TRUE, lvars=TRUE, lcol=2, llty=2,
        all=TRUE, dlab=c("cumsum", "\% log(ind.)", "\% non-zero"),
        dcol=c(1, 2, 4), dlty, dpos=c(1.5, 20), \dots)
\method{lines}{abund}(abd, n=abd$n, lvert=TRUE, lvars=TRUE, \dots)
\method{identify}{abund}(abd, label.pts=FALSE, lvert=TRUE, lvars=TRUE, \dots)
\method{extract}{abund}(abd, n=abd$n, left=TRUE)
}
\arguments{
  \item{x}{ A data frame containing the variables to sort according to their abundance in columns }
  \item{f}{ Weight given to the number of individuals criterium (strictly included between 0 and 1; weight for the non-null values is \code{1-f}. The default value, \code{f=0.2}, gives enough weight to the number of non-null values to get abundant species according to this criterium first, but allowing to get at the other extreme rare, but locally abundant species }
  \item{abd}{ An 'abund' object returned by \code{abund} }
  \item{n}{ The number of variables selected at left }
  \item{lvert}{ If \code{TRUE} then a vertical line separate the n variables at left from the others }
  \item{lvars}{ If \code{TRUE} then the x-axis labels of the n left variables are printed in a different color to emphasize them }
  \item{lcol}{ The color to use to draw the vertical line (\code{lvert=TRUE}) and the variables labels (\code{lvars=TRUE}) at left af the nth variable. By default, color 2 is used }
  \item{llty}{ The style used to draw the vertical line (\code{lvert=TRUE}). By default, a dashed line is used }
  \item{all}{ If \code{TRUE} then all lines are drawn (cumsum, \%log(ind.) and \%non-null). If \code{FALSE}, only the cumsum line is drawn }
  \item{dlab}{ The legend labels }
  \item{dcol}{ Colors to use for drawing the various curves on the graph }
  \item{dlty}{ The line style to use for drawing the various curves on the graph }
  \item{dpos}{ The position of the legend box on the graph (coordinates of its top-left corner). A legend box is drawn only if \code{all=TRUE} }
  \item{\dots}{ additional graph parameters }
  \item{label.pts}{ Do we have to label points on the graph or to chose an extraction level with the \code{identify()} method? }
  \item{left}{ If \code{TRUE}, the n variables at left are extracted. Otherwise, the total-n variables at right are extracted }
}
\details{
  Successive sorts can be applied. For instance, a first sort with \code{f = 0.2}, followed by an extraction of rare species and another sort with \code{f = 1} allows to collect only rare but locally abundant species.
}
\value{
  An object of type 'abund' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{extract()}.
}
\references{
Ibanez, F., J.-C. Dauvin & M. Etienne, 1993. \emph{Comparaison des volutions  long terme (1977-1990) de deux peuplements macrobenthiques de la baie de Morlaix (Manche occidentale): relations avec les facteurs hydroclimatiques.} J. Exp. Mar. Biol. Ecol., 169:181-214.
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{escouf}} }

\examples{
data(bnr)
bnr.abd <- abund(bnr)
summary(bnr.abd)
plot(bnr.abd, dpos=c(105, 100))
bnr.abd$n <- 26
# To identify a point on the graph, use: bnr.abd$n <- identify(bnr.abd)
lines(bnr.abd)
bnr2 <- extract(bnr.abd)
names(bnr2)
}

\keyword{ multivariate }
\eof
\name{bnr}
\alias{bnr}
\non_function{}
\title{ A data frame of 163 benthic species measured across a transect }
\usage{data(bnr)}
\description{
The \code{bnr} data frame has 103 rows and 163 columns.
Each column is a separate species observed at least once in one of all stations. Several species are very abundant, other are very rare.
}
\source{
 Unpublished dataset.
}
\keyword{datasets}
\eof
\name{buysbal}
\alias{buysbal}
\title{ Buys-Ballot table }
\description{
  Calculate a Buys-Ballot table from a time-series
}
\usage{
buysbal(x, y=NULL, frequency=12, units="years", datemin=NULL,
        dateformat="m/d/Y", count=FALSE)
}
\arguments{
  \item{x}{ Either a vector with time values (in this case, \code{y} must be defined), or a regular time-series }
  \item{y}{ If \code{x} is a vector with time values, \code{y} must contain corresponding observations }
  \item{frequency}{ The frequency of observations per year to use in the Buys-Ballot table. By default \code{frequency=12} which corresponds to monthly samples, and the resulting table has 12 column, one per month }
  \item{units}{ either \code{"years"} (by default), and time is not rescaled, or \code{"days"}, and the time is rescaled to "years" with the function \code{daystoyears()} }
  \item{datemin}{ A character string representing the first date, using a format corresponding to \code{dateformat}. For instance, with \code{datemin="04/23/1998"} and \code{dateformat="m/d/Y"}, the first observation is assumed to be made the 23th April 1998. In \R, it can also be a POSIXt date (see \code{?DataTimeClasses}). In this case, \code{dateformat} is not required and is ignored. By default, \code{datemin=NULL} }
  \item{dateformat}{ The format used for the date in \code{datemin}. For instance, \code{"d/m/Y"} or \code{"m/d/Y"} (by default). The distinction between "Y" and "y" is not important in Splus, but it is vital in \R to use "y" for two-digit years (ex: 89) and "Y" for four-digits years (ex: 1989), or the date will not be correctly converted! In \R, you can also use a POSIXt format specification like "\%d-\%m\%Y" for instance (see \code{?strptime} for a complete description of POSIXt format specification. In both Splus and \R, you can also use "mon" for abbreviated months like "mon d Y" for "Apr 20 1965", and "month" for fully-spelled months like "d month Y" for "24 September 2003" }
  \item{count}{ If \code{FALSE} (by default), the Buys-Ballot table is calculated. If \code{TRUE}, the function returns only the number of observations that are used in each cell of the Buys-Ballot table }
}
\details{
  A Buys-Ballot table summarizes data to highlight seasonal variations. Each line is one year and each column is a period of the year (12 months, 4 quarters, 52 weeks,...). A cell ij of this table contain the mean value for all observations made during the year i at the period j.
}
\value{
  A matrix containing either the Buys-Ballot table (\code{count=FALSE}), or the number of observations used to build the table (\code{count=TRUE})
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{daystoyears}}, \code{\link{tsd}} }

\examples{
data(releve)
buysbal(releve$Day, releve$Melosul, frequency=4, units="days",
        datemin="21/03/1989", dateformat="d/m/Y")
buysbal(releve$Day, releve$Melosul, frequency=4, units="days",
        datemin="21/03/1989", dateformat="d/m/Y", count=TRUE)
}
\keyword{ ts }
\eof
\name{daystoyears}
\alias{daystoyears}
\alias{yearstodays}
\title{ Convert time units from "days" to "years" or back }
\description{
  Convert time scales. The time scale "days" corresponds to 1 unit per day. The time scale "years" uses 1 unit for 1 year. It is used in any analysis that requires seasonal decomposition and/or elimination. 
}
\synopsis{
daystoyears(x, datemin=NULL, dateformat="m/d/Y")
yearstodays(x, xmin=NULL)
}
\usage{
daystoyears(x, datemin=NULL, dateformat="m/d/Y")
yearstodays(x, xmin=NULL)
}
\arguments{
  \item{x}{ A vector of time values }
  \item{datemin}{ A character string representing the first date, using a format corresponding to \code{dateformat}. For instance, with \code{datemin="04/23/1998"} and \code{dateformat="m/d/Y"}, the first observation is assumed to be made the 23th April 1998. In \R, it can also be a POSIXt date (see \code{?DataTimeClasses}). In this case, \code{dateformat} is not required and is ignored. By default, \code{datemin=NULL} }
  \item{dateformat}{ The format used for the date in \code{datemin}. For instance, \code{"d/m/Y"} or \code{"m/d/y"}. The distinction between "Y" and "y" is not important in Splus, but it is vital in \R to use "y" for two-digit years (ex: 89) and "Y" for four-digits years (ex: 1989), or the date will not be correctly converted! In \R, you can also use a POSIXt format specification like "\%d-\%m\%Y" for instance (see \code{?strptime} for a complete description of POSIXt format specification. In both Splus and \R, you can also use "mon" for abbreviated months like "mon d Y" for "Apr 20 1965", and "month" for fully-spelled months like "d month Y" for "24 September 2003" }
  \item{xmin}{ The minimum value for the "days" time-scale }
}
\details{
  The "years" time-scale uses one unit for each year. We deliberately "linearized" time in this time-scale and each year has 365.25 days. There is thus no adjustment for bissextile years. Similarly, one month is considered to be 1/12 year, no mather if it has 28, 29, 30 or 31 days. This representation simplifies further calculations, especially regarding seasonal effects (a quarter is exactly 0.25 units for instance), but introduces some shifts in time (of up to one day, which is not significant when working on long-term series with years as units). However, converting it back to "days", using \code{yearstodays()} restablishes correct initial days without errors. 
}
\value{
  A numerical vector of the same length as \code{x} with the converted time-scale
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{buysbal}} }

\examples{
# A vector with a "days" time-scale (25 values every 30 days)
A <- (1:25)*30
# Convert it to a "years" time-scale, using 23/05/2001 (d/m/Y) as first value
B <- daystoyears(A, datemin="23/05/2001", dateformat="d/m/Y")
B
# Convert it back to "days" time-scale
yearstodays(B, xmin=30)
}
\keyword{ ts }
\eof
\name{decaverage}
\alias{decaverage}
\title{ Time series decomposition using a moving average }
\description{
  Decompose a single regular time series with a moving average filtering. Return a 'tsd' object. To decompose several time series at once, use \code{tsd()} with the argument \code{method="average"}
}
\usage{
decaverage(x, type="additive", order=1, times=1, sides=2, ends="fill",
        weights=NULL)
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{type}{ the type of model, either \code{type="additive"} (by default), or \code{type="multiplicative"} }
  \item{order}{ the order of the moving average (the window of the average being 2*order+1), centered around the current observation or at left of this observation depending upon the value of the \code{sides} argument. Weights are the same for all observations within the window. However, if the argument \code{weights} is provided, it supersedes \code{order}. One can also use \code{order="periodic"}. In this case, a deseasoning filter is calculated according to the value of \code{frequency} }
  \item{times}{ The number of times to apply the method (by default, once) }
  \item{sides}{ If 2 (by default), the window is centered around the current observation. If 1, the window is at left of the current observation (including it) }
  \item{ends}{ either "NAs" (fill first and last values that are not calculable with NAs), or "fill" (fill them with the average of observations before applying the filter, by default), or "circular" (use last values for estimating first ones and vice versa), or "periodic" (use entire periods of contiguous cycles, deseasoning) }
  \item{weights}{ a vector indicating weight to give to all observations in the window. This argument has the priority over \code{order} }
}
\details{
  This function is a wrapper around the \code{filter()} function and returns a 'tsd' object. However, it offers more methods to handle ends.
}
\value{
  A 'tsd' object
}
\references{
Kendall, M., 1976. \emph{Time-series.} Charles Griffin & Co Ltd. 197 pp.

Laloire, J.C., 1972. \emph{Mthodes du traitement des chroniques.} Dunod, Paris, 194 pp.

Malinvaud, E., 1978. \emph{Mthodes statistiques de l'conomtrie.} Dunod, Paris. 846 pp.

Philips, L. & R. Blomme, 1973. \emph{Analyse chronologique.} Universit Catholique de Louvain. Vander ed. 339 pp.
 }
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{deccensus}}, \code{\link{decdiff}}, \code{\link{decmedian}},  \code{\link{decevf}}, \code{\link{decreg}}, \code{\link{decloess}} }

\examples{
data(marbio)
ClausoB.ts <- ts(log(marbio$ClausocalanusB + 1))
ClausoB.dec <- decaverage(ClausoB.ts, order=2, times=10, sides=2, ends="fill")
plot(ClausoB.dec, col=c(1, 3, 2), xlab="stations")
# A stacked graph is more representative in this case
plot(ClausoB.dec, col=c(1, 3), xlab="stations", stack=FALSE, resid=FALSE,
        lpos=c(53, 4.3))
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{deccensus}
\alias{deccensus}
\title{ Time decomposition using the CENSUS II method }
\description{
  The CENSUS II method allows to decompose a regular time series into a trend, a seasonal component and residuals, according to a multiplicative model
}
\usage{
deccensus(x, type="multiplicative", trend=FALSE)
}
\arguments{
  \item{x}{ A single regular time serie (a 'rts' object under S+ and a 'ts' object under R) with a "years" time scale (one unit = one year) and a complete number of cycles (at least 3 complete years) }
  \item{type}{ The type of model. This is for compatibility with other \code{decxxx()} functions, but only a multiplicative model is allowed here }
  \item{trend}{ If \code{trend=TRUE} a trend component is also calculated, otherwise, the decomposition gives only a seasonal component and residuals }
}
\details{
  The trend component contains both a general trend and long-term oscillations. The seasonal trend may vary from year to year. For a seasonal decomposition using an additive model, use \code{decloess()} instead
}
\value{
  a 'tsd' object
}
\references{ 
Bthoux, N., M. Etienne, F. Ibanez & J.L. Rapaire, 1980. \emph{Spcificits hydrologiques des zones littorales. Analyse chronologique par la mthode CENSUS II et estimation des changes ocan-atmosphre appliqus  la baie de Villefranche sur mer.} Ann. Inst. Ocanogr. Paris, 56:81-95.

Fromentin, J.M. & F. Ibanez, 1994. \emph{Year to year changes in meteorological features on the French coast area during the last half-century. Examples of two biological responses.} Oceanologica Acta, 17:285-296.

Institut National de Statistique de Belgique, 1965. \emph{Dcomposition des sries chronologiques en leurs composantes suivant diffrentes mthodes. Etudes statistiques et conomtriques.} Bull. Stat. INS, 10:1449-1524.

Philips, J. & R. Blomme, 1973. \emph{Analyse chronologique.} Universit Catholique de Louvain, Vander ed. 339 pp.

Rosenblatt, H.M., 1968. \emph{Spectral evaluation of BLS and CENSUS revised seasonal adjustment procedures.} J. Amer. Stat. Assoc., 68:472-501.

Shiskin, J. & H. Eisenpress, 1957. \emph{Seasonal adjustment by electronic computer methods.} J. Amer. Stat. Assoc., 52:415-449.

}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{decdiff}}, \code{\link{decmedian}}, \code{\link{decevf}}, \code{\link{decreg}}, \code{\link{decloess}} }

\examples{
data(releve)
# Get regulated time series with a 'years' time-scale
rel.regy <- regul(releve$Day, releve[3:8], xmin=6, n=87, units="daystoyears", frequency=24, tol=2.2, methods="linear", datemin="21/03/1989", dateformat="d/m/Y")
rel.ts <- tseries(rel.regy)
# We must have complete cycles to allow using deccensus()
start(rel.ts)
end(rel.ts)
rel.ts2 <- window(rel.ts, end=c(1992,5))
rel.dec2 <- deccensus(rel.ts2[, "Melosul"], trend=TRUE)
plot(rel.dec2, col=c(1,4,3,2))
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{decdiff}
\alias{decdiff}
\title{ Time series decomposition using differences (trend elimination) }
\description{
  A filtering using X(t+lag) - X(t) has the property to eliminate the general trend from the series, whatever its shape
}
\usage{
decdiff(x, type="additive", lag=1, order=1, ends="fill")
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{type}{ the type of model, either \code{type="additive"} (by default), or \code{type="multiplicative"} }
  \item{lag}{ The lag between the two observations used to calculate differences. By default, \code{lag=1} }
  \item{order}{ The order of the difference corresponds to the number of times it is applied, by default \code{order=1} }
  \item{ends}{ either "NAs" (fill first values that are not calculable with NAs), or "fill" (fill them with the average of following observations before applying the filter, by default), or "drop" (do not fill them). If \code{ends="drop"}, the filtered series will be shorter than the initial one by lag*order. In all other cases, the filtered series is as large as the initial one }
}
\details{
  This function is a wrapper around the \code{diff()} function to create a 'tsd' object. It also manages initial values through the \code{ends} argument.
}
\value{
  a 'tsd' object
}
\references{ 
Kendall, M., 1976. \emph{Time-series.} Charles Griffin & Co Ltd. 197 pp.

Laloire, J.C., 1972. \emph{Mthodes du traitement des chroniques.} Dunod, Paris, 194 pp.

Philips, L. & R. Blomme, 1973. \emph{Analyse chronologique.} Universit Catholique de Louvain. Vander ed. 339 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{deccensus}}, \code{\link{decmedian}},  \code{\link{decevf}}, \code{\link{decreg}}, \code{\link{decloess}} }

\examples{
data(marbio)
ClausoB.ts <- ts(log(marbio$ClausocalanusB + 1))
ClausoB.dec <- decdiff(ClausoB.ts, lag=1, order=2, ends="fill")
plot(ClausoB.dec, col=c(1, 4, 2), xlab="stations")
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{decevf}
\alias{decevf}
\title{ Time series decomposition using eigenvector filtering (EVF) }
\description{
  The eigenvector filtering decomposes the signal by applying a principal component analysis (PCA) on the original signal and a certain number of copies of it incrementally lagged, collected in a multivariate matrix. Reconstructing the signal using only the most representative eigenvectors allows filtering it.
}
\usage{
decevf(x, type="additive", lag=5, axes=1:2)
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{type}{ the type of model, either \code{type="additive"} (by default), or \code{type="multiplicative"} }
  \item{lag}{ The maximum lag used. A PCA is run on the matrix constituted by vectors lagged from 0 to \code{lag}. The defaulf value is 5, but a value corresponding to no significant autocorrelation, on basis of examination of the autocorrelation plot obtained by \code{acf} in the library 'ts' should be used (Lag at first time the autocorrelation curve crosses significance lines multiplied by the frequency of the series). }
  \item{axes}{ The principal axes to use to reconstruct the filtered signal. For instance, to use axes 2 and 4, use \code{axes=c(2,4)}. By default, the first two axes are considered (\code{axes=1:2}) }
}
\value{
  a 'tsd' object
}
\references{ 
Colebrook, J.M., 1978. \emph{Continuous plankton records: zooplankton and environment, North-East Atlantic and North Sea 1948-1975.} Oceanologica Acta, 1:9-23.

Ibanez, F. & J.C. Dauvin, 1988. \emph{Long-term changes (1977-1987) on a muddy fine sand Abra alba - Melinna palmate population community from the Western English Channel.} J. Mar. Prog. Ser., 49:65-81.

Ibanez, F., 1991. \emph{Treatment of data deriving from the COST 647 project on coastal benthic ecology: The within-site analysis.} In: B. Keegan (ed.) \emph{Space and time series data analysis in coastal benthic ecology.} Pp. 5-43.

Ibanez, F. & M. Etienne, 1992. \emph{Le filtrage des sries chronologiques par l'analyse en composantes principales de processus (ACPP).} J. Rech. Ocanogr., 16:27-33.

Ibanez, F., J.C. Dauvin & M. Etienne, 1993. \emph{Comparaison des volutions  long-terme (1977-1990) de deux peuplements macrobenthiques de la Baie de Morlaix (Manche Occidentale): relations avec les facteurs hydroclimatiques.} J. Exp. Mar. Biol. Ecol., 169:181-214.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{deccensus}}, \code{\link{decmedian}},  \code{\link{decdiff}}, \code{\link{decreg}}, \code{\link{decloess}} }

\examples{
data(releve)
melo.regy <- regul(releve$Day, releve$Melosul, xmin=9, n=87,
        units="daystoyears", frequency=24, tol=2.2, methods="linear",
        datemin="21/03/1989", dateformat="d/m/Y")
melo.ts <- tseries(melo.regy)
library(ts)
acf(melo.ts)
# Autocorrelation is not significant after 0.16
# That corresponds to a lag of 0.16*24=4 (frequency=24)
melo.evf <- decevf(melo.ts, lag=4, axes=1)
plot(melo.evf, col=c(1, 4, 2))
# A superposed graph is better in the present case
plot(melo.evf, col=c(1, 4), xlab="stations", stack=FALSE, resid=FALSE,
        lpos=c(0, 60000))
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{decloess}
\alias{decloess}
\title{ Time series decomposition by the LOESS method }
\description{
  Compute a seasonal decomposition of a regular time series using a LOESS method (local polynomial regression)
}
\usage{
decloess(x, type="additive", s.window=NULL, s.degree=0, t.window=NULL,
        t.degree=2, robust=FALSE, trend=FALSE)
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{type}{ the type of model. This is for compatibility purpose. The only model type that is accepted for this method is \code{type="additive"}. For a multiplicative model, use \code{deccensus()} instead }
  \item{s.window}{ the width of the window used to extract the seasonal component. Use an odd value equal or just larger than the number of annual values (frequency of the time series). Use another value to extract other cycles (circadian, lunar,...). Using \code{s.window="periodic"} ensures a correct value for extracting a seasonal component when the time scale is in years units }
  \item{s.degree}{ the order of the polynome to use to extract the seasonal component (0 or 1). By default \code{s.degree=0} }
  \item{t.window}{ the width of the window to use to extract the general trend when \code{trend=TRUE} (indicate an odd value). If this parameter is not provided, a reasonable value is first calculated, and then used by the algorithm. }
  \item{t.degree}{ the order of the polynome to use to extract the general trend (0, 1 or 2). By default \code{t.degree=2} }
  \item{robust}{ if \code{TRUE} a robust regression method is used. Otherwise (\code{FALSE}), by default, a classical least-square regression is used }
  \item{trend}{ If \code{TRUE} a trend is calculated (under R only). Otherwise, the series is decomposed into a seasonal component and residuals only }
}
\details{
  This function uses the \code{stl()} function for the decomposition. It is a wrapper that create a 'tsd' object
}
\value{
  a 'tsd' object
}
\references{ 
Cleveland, W.S., E. Grosse & W.M. Shyu, 1992. \emph{Local regression models. Chapter 8 of Statistical Models in S.} J.M. Chambers & T.J. Hastie (eds). Wadsworth & Brook/Cole.

Cleveland, R.B.,  W.S. Cleveland, J.E. McRae, & I. Terpenning, 1990. \emph{STL: A seasonal-trend  decomposition  procedure based on loess.} J. Official Stat., 6:3-73.
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{deccensus}}, \code{\link{decmedian}}, \code{\link{decdiff}}, \code{\link{decevf}}, \code{\link{decreg}} }

\examples{
data(releve)
melo.regy <- regul(releve$Day, releve$Melosul, xmin=9, n=87,
        units="daystoyears", frequency=24, tol=2.2, methods="linear",
        datemin="21/03/1989", dateformat="d/m/Y")
melo.ts <- tseries(melo.regy)
melo.dec <- decloess(melo.ts, s.window="periodic")
plot(melo.dec, col=1:3)
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{decmedian}
\alias{decmedian}
\title{ Time series decomposition using a running median }
\description{
  This is a nonlinear filtering method used to smooth, but also to segment a time series. The isolated peaks and pits are leveraged by this method.
}
\usage{
decmedian(x, type="additive", order=1, times=1, ends="fill")
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{type}{ the type of model, either \code{type="additive"} (by default), or \code{type="multiplicative"} }
  \item{order}{ the window used for the running median corresponds to 2*order + 1  }
  \item{times}{ the number of times the running median is applied. By default, 1 }
  \item{ends}{ the method used to calculate ends. Either "NAs" (fill extremes, non-calculable values with NAs), or "fill" (fill these extremes with the closest calculable median) }
}

\value{
  a 'tsd' object
}
\references{ 
Gebski, V.J., 1985. \emph{Some properties of splicing when applied to non-linear smoothers.} Comp. Stat. Data Anal., 3:151-157.

Philips, L. & R. Blomme, 1973. \emph{Analyse chronologique. Universit Catholique de Louvain.} Vander ed. 339 pp.

Tukey, J.W., 1977. \emph{Exploratory Data Analysis.} Reading Massachusetts: Addison-Wesley.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{deccensus}}, \code{\link{decdiff}}, \code{\link{decevf}}, \code{\link{decreg}}, \code{\link{decloess}} }

\examples{
data(marbio)
ClausoB.ts <- ts(log(marbio$ClausocalanusB + 1))
ClausoB.dec <- decmedian(ClausoB.ts, order=2, times=10, ends="fill")
plot(ClausoB.dec, col=c(1, 4, 2), xlab="stations")
# This is a transect across a frontal zone:
plot(ClausoB.dec, col=c(0, 2), xlab="stations", stack=FALSE, resid=FALSE)
lines(c(17, 17), c(0, 10), col=4, lty=2)
lines(c(25, 25), c(0, 10), col=4, lty=2)
lines(c(30, 30), c(0, 10), col=4, lty=2)
lines(c(41, 41), c(0, 10), col=4, lty=2)
lines(c(46, 46), c(0, 10), col=4, lty=2)
text(c(8.5, 21, 27.5, 35, 43.5, 57), 8.7, labels=c("Peripheral Zone", "D1",
        "C", "Front", "D2", "Central Zone"))
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{decreg}
\alias{decreg}
\title{ Time series decomposition using a regression model }
\description{
  Providing values coming from a regression on the original series, a \code{tsd} object is created using the original series, the regression model and the residuals
}
\usage{
decreg(x, xreg, type="additive")
}
\arguments{
  \item{x}{ a regular time series ('rts' under S+ and 'ts' under R) }
  \item{xreg}{ a second regular time series or a vector of the same length as \code{x} with corresponding values from the regression model }
  \item{type}{ the type of model, either \code{type="additive"} (by default), or \code{type="multiplicative"} }
}
\value{
  a 'tsd' object
}
\references{ 
Frontier, S., 1981. \emph{Mthodes statistiques.} Masson, Paris. 246 pp.

Kendall, M., 1976. \emph{Time-series.} Charles Griffin & Co Ltd. 197 pp.

Legendre, L. & P. Legendre, 1984. \emph{Ecologie numrique. Tome 2: La structure des donnes cologiques.} Masson, Paris. 335 pp.

Malinvaud, E., 1978. \emph{Mthodes statistiques de l'conomtrie.} Dunod, Paris. 846 pp.

Sokal, R.R. & F.J. Rohlf, 1981. \emph{Biometry.} Freeman & Co, San Francisco. 860 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{tsd}}, \code{\link{tseries}}, \code{\link{decaverage}}, \code{\link{deccensus}}, \code{\link{decdiff}}, \code{\link{decevf}}, \code{\link{decmedian}}, \code{\link{decloess}} }

\examples{
data(marphy)
density <- ts(marphy[, "Density"])
plot(density)
Time <- time(density)

# Linear model to represent trend
density.lin <- lm(density ~ Time)
summary(density.lin)
xreg <- predict(density.lin)
lines(xreg, col=3)
density.dec <- decreg(density, xreg)
plot(density.dec, col=c(1, 3, 2), xlab="stations")

# Order 2 polynomial to represent trend
density.poly <- lm(density ~ Time + I(Time^2))
summary(density.poly)
xreg2 <- predict(density.poly)
plot(density)
lines(xreg2, col=3)
density.dec2 <- decreg(density, xreg2)
plot(density.dec2, col=c(1, 3, 2), xlab="stations")

# Fit a sinusoidal model on seasonal (artificial) data
tser <- ts(sin((1:100)/12*pi)+rnorm(100, sd=0.3), start=c(1998, 4),
        frequency=24)
Time <- time(tser)
tser.sin <- lm(tser ~ I(cos(2*pi*Time)) + I(sin(2*pi*Time)))
summary(tser.sin)
tser.reg <- predict(tser.sin)
tser.dec <- decreg(tser, tser.reg)
plot(tser.dec, col=c(1, 4), xlab="stations", stack=FALSE, resid=FALSE,
        lpos=c(0, 4))
plot(tser.dec, col=c(1, 4, 2), xlab="stations")

# One can also use nonlinear models (see 'nls')
# or autoregressive models (see 'ar' and others in 'ts' library)
}
\keyword{ ts }
\keyword{ smooth }
\eof
\name{disjoin}
\alias{disjoin}
\title{ Complete disjoined coded data (binary coding) }
\description{
  Transform a factor in separate variables (one per level) with a binary code (0 for absent, 1 for present) in each variable
}
\usage{
disjoin(x)
}
\arguments{
  \item{x}{ a vector containing a factor data }
}
\details{
  Use \code{cut()} to transform a numerical variable into a factor variable
}
\value{
  a matrix containing the data with binary coding
}
\references{ 
Fromentin J.-M., F. Ibanez & P. Legendre, 1993. \emph{A phytosociological method for interpreting plankton data.} Mar. Ecol. Prog. Ser., 93:285-306.

Gebski, V.J., 1985. \emph{Some properties of splicing when applied to non-linear smoothers.} Comput. Stat. Data Anal., 3:151-157.

Grandjouan, G., 1982. \emph{Une mthode de comparaison statistique entre les rpartitions des plantes et des climats.} Thse d'Etat, Universit Louis Pasteur, Strasbourg.

Ibanez, F., 1976. \emph{Contribution  l'analyse mathmatique des vnements en Ecologie planctonique. Optimisations mthodologiques.} Bull. Inst. Ocanogr. Monaco, 72:1-96.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{buysbal}}, \code{\link[base]{cut}} }

\examples{
# Artificial data with 1/5 of zeros
Z <- c(abs(rnorm(8000)), rep(0, 2000))
# Let the program chose cuts
table(cut(Z, breaks=5))
# Create one class for zeros, and 4 classes for the other observations
Z2 <- Z[Z != 0]
cuts <- c(-1e-10, 1e-10, quantile(Z2, 1:5/5, na.rm=TRUE))
cuts
table(cut(Z, breaks=cuts))
# Binary coding of these data
disjoin(cut(Z, breaks=cuts))[1:10, ]
}
\keyword{ manip }
\eof
\name{disto}
\alias{disto}
\title{ Compute and plot a distogram }
\description{
  A distogram is an extension of the variogram to a multivariate time-series. It computes, for each observation (with a constant interval h between each observation), the euclidean distance normated to one (chord distance)
}
\usage{
disto(x, max.dist=nrow(x)/4, plotit=TRUE, disto.data=NULL)
}
\arguments{
  \item{x}{ a matrix, a data frame or a multiple time-series }
  \item{max.dist}{ the maximum distance to calculate. By default, it is the third of the number of observations (that is, the number of rows in the matrix) }
  \item{plotit}{ If \code{plotit=TRUE} then the graph of the distogram is plotted }
  \item{disto.data}{ data coming from a previous call to \code{disto()}. Call the function again with these data to plot the corresponding graph }
}
\value{
  A data frame containing distance and distogram values
}
\references{ 
Dauvin, J.C. & F. Ibanez, 1986. \emph{Variations  long-terme (1977-1985) du peuplement des sables fins de la Pierre Noire (baie de Morlaix, Manche Occidentale): analyse statistique de l'volution structurale.} Hydrobiologia, 142:171-186.

Ibanez, F. & J.C. Dauvin, 1988. \emph{Long-term changes (1977-1987) in a muddy fine sand Abra alba - Melinna palmate community from the Western English Channel: multivariate time-series analysis.} Mar. Ecol. Prog. Ser., 49:65-81.

Mackas, D.L., 1984. \emph{Spatial autocorrelation of plankton community composition in a continental shelf ecosystem.} Limnol. Ecol., 20:451-471.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{vario}} }

\examples{
data(bnr)
disto(bnr)
}
\keyword{ multivariate }
\keyword{ ts }
\eof
\name{escouf}
\alias{escouf}
\alias{extract.escouf}
\alias{identify.escouf}
\alias{lines.escouf}
\alias{plot.escouf}
\alias{print.escouf}
\alias{print.summary.escouf}
\alias{summary.escouf}
\title{ Choose variables using the Escoufier's equivalent vectors method }
\description{
  Calculate equivalent vectors sensu Escoufier, that is, most significant variables from a multivariate data frame according to a principal component analysis (variables that are most correlated with the principal axes). This method is useful mainly for physical or chemical data where simply summarizing them with a PCA does not always gives easily interpretable principal axes.
}
\synopsis{
escouf(x, level=1, verbose=TRUE)
extract.escouf(e, n, level=e$level, ...)
identify.escouf(x, lhorz=TRUE, lvert=TRUE, lvars=TRUE, col=2, lty=2, ...)
lines.escouf(x, level=x$level, lhorz=TRUE, lvert=TRUE, lvars=TRUE, col=2, lty=2, ...)
plot.escouf(x, level=x$level, lhorz=TRUE, lvert=TRUE, lvars=TRUE, lcol=2, llty=2, diff=TRUE, dlab="RV' (units not shown)", dcol=4, dlty=par("lty"), dpos=0.8, type="s", xlab="variables", ylab="RV", main=paste("Escoufier's equivalent vectors for:",x$data), ...)
print.escouf(x, ...)
print.summary.escouf(x, ...)
summary.escouf(object, ...)  
}
\usage{
escouf(x, level=1, verbose=TRUE)
\method{summary}{escouf}(esc)
\method{plot}{escouf}(esc, level=x$level, lhorz=TRUE, lvert=TRUE, lvars=TRUE,
        lcol=2, llty=2, diff=TRUE, dlab="RV' (units not shown)", dcol=4,
        dlty=par("lty"), dpos=0.8, \dots)
\method{lines}{escouf}(esc, level=x$level, lhorz=TRUE, lvert=TRUE, lvars=TRUE,
        lcol=2, llty=2, \dots)
\method{identify}{escouf}(esc, lhorz=TRUE, lvert=TRUE, lvars=TRUE, lcol=2,
        llty=2, \dots)
\method{extract}{escouf}(esc, n=NULL, level=e$level)
}
\arguments{
  \item{x}{ A data frame containing the variables to sort according to the Escoufier's method }
  \item{level}{ The level of correlation at which to stop calculation. By default \code{level=1}, the highest value, and all variables are sorted. Specify a value lower than one to speed up calculation. If you specify a too low values you will not be able to extract all significant variables (extraction level must be lower than calculation level). We advise you keep 0.95 < level < 1 }
  \item{verbose}{ Print calculation steps. This allows to control the percentage of calculation already achieved when computation takes a long time (that is, with many variables to sort) }
  \item{esc}{ An 'escouf' object returned by \code{escouf}}
  \item{lhorz}{ If \code{TRUE} then an horizontal line indicating the extraction level is drawn }
  \item{lvert}{ If \code{TRUE} then a vertical line separate the n extracted variables at left from the rest }
  \item{lvars}{ If \code{TRUE} then the x-axis labels of the n extracted variables at left are printed in a different color to emphasize them }
  \item{lcol}{ The color to use to draw the lines (\code{lhorz=TRUE} and \code{lvert=TRUE}) and the variables labels (\code{lvars=TRUE}) of the n extracted variables. By default, color 2 is used }
  \item{llty}{ The style used to draw the lines (\code{lhorz=TRUE} and \code{lvert=TRUE}). By default, lines are dashed }
  \item{diff}{ If \code{TRUE} then the RV' curve is also plotted (by default) }
  \item{dlab}{ The label to use for the RV' curve. By default: \code{"RV' (units not shown)"} }
  \item{dcol}{ The color to use for the RV' curve (by default, color 4 is used) }
  \item{dlty}{ The style for the RV' curve }
  \item{dpos}{ The relative horizontal position of the label for the RV' curve. The default value of 0.8 means that the label is placed at 80\% of the horizontal axis.Vertical position of the label is automatically determined }
  \item{\dots}{ additional graph parameters }
  \item{n}{ The number of variables to extract. If a value is given, it has the priority on \code{level} }
}

\value{
  An object of type 'escouf' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{extract()}.
}
\references{ 
Cambon, J., 1974. \emph{Vecteur quivalent  un autre au sens des composantes principales.} Application hydrologique. DEA de Mathmatiques Appliques, Universit de Montpellier.

Escoufier, Y., 1970. \emph{Echantillonnage dans une population de variables alatoires relles.} Pub. Inst. Stat. Univ. Paris, 19:1-47.

Jabaud, A., 1996. \emph{Cadre climatique et hydrobiologique du lac Lman.} DEA d'Ocanologie Biologique Paris.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}), Benjamin Planque (\email{Benjamin.Planque@ifremer.fr}),  Jean-Marc Fromentin (\email{Jean.Marc.Fromentin@ifremer.fr}) }
\section{WARNING }{Since a large number of iterations is done, this function is slow with a large number of variables (more than 25-30)!}

\seealso{ \code{\link{abund}} }

\examples{
data(marbio)
marbio.esc <- escouf(marbio)
summary(marbio.esc)
plot(marbio.esc)
# The x-axis has short labels. For more info., enter: 
marbio.esc$vr
# Define a level at which to extract most significant variables
marbio.esc$level <- 0.90
# Show it on the graph
lines(marbio.esc)
# This can also be done interactively on the plot using:
# marbio.esc$level <- identify(marbio.esc)
# Finally, extract most significant variables
marbio2 <- extract(marbio.esc)
names(marbio2)
}
\keyword{ multivariate }
\eof
\name{extract}
\alias{extract}
\title{ Extract a subset of the original dataset }
\description{
  `extract' is a generic function for extracting a part of the original dataset according to an analysis...)
}
\usage{
extract(e, n, ...)
}
\arguments{
  \item{e}{ An object from where extraction is performed }
  \item{n}{ The number of elements to extract }
  \item{\dots}{ Additional arguments affecting the extraction }
}
\value{
  A subset of the original dataset contained in the extracted object
}
\seealso{ \code{\link{abund}}, \code{\link{regul}}, \code{\link{tsd}}, \code{\link{turnogram}}, \code{\link{turnpoints}} }

\keyword{ methods }
\eof
\name{first}
\alias{first}
\title{ Get the first element of a vector }
\description{
  Extract the first element of a vector. Useful for the \code{turnogram()} function
}
\usage{
first(x, na.rm=FALSE)
}
\arguments{
  \item{x}{ a numerical vector }
  \item{na.rm}{ if \code{na.rm=TRUE}, then the first non-missing value (if any) is returned. By default, it is FALSE and the first element (whatever its value) is returned }
}
\value{
  a numerical value
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{last}}, \code{\link{turnogram}} }

\examples{
a <- c(NA, 1, 2, NA, 3, 4, NA)
first(a)
first(a, na.rm=TRUE)
}
\keyword{ manip }
\eof
\name{.gleissberg.table}
\alias{.gleissberg.table}
\non_function{}
\title{Table of probabilities according to the Gleissberg distribution}
%\usage{data(gleissberg.table)}
%\arguments{
%  \item{gleissberg.table}{ the dataset to load. Note that this dataset is loaded into the hidden \code{.gleissberg.table} variable (starting with a dot)! }
%}
\description{
The table of probabilities to have 0 to n-2 extrema in a series, for n=3 to 50 (for n > 50, the Gleissberg distribution is approximated with a normal distribution
}
\note{
This dataset is used internally by \code{pgleissberg()}. You do not have to access it directly. See \code{pgleissberg()} for more information }
\seealso{ \code{\link{pgleissberg}} }
\keyword{ datasets }
\keyword{ distribution }
\eof
\name{is.tseries}
\alias{is.tseries}
\title{ Is this object a time series? }
\description{
  This is equivalent to \code{is.rts()} in Splus and to \code{is.ts()} in \R. \code{is.tseries()} recognizes both 'rts' and 'ts' objects whatever the environment (Splus or R)
}
\usage{
is.tseries(x)
}
\arguments{
  \item{x}{ an object }
}
\value{
  a boolean value
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{tseries}} }

\examples{
tser <- ts(sin((1:100)/6*pi)+rnorm(100, sd=0.5), start=c(1998, 4), frequency=12)
is.tseries(tser)      # TRUE
not.a.ts <- c(1,2,3)
is.tseries(not.a.ts)  # FALSE
}
\keyword{ ts }
\eof
\name{last}
\alias{last}
\title{ Get the last element of a vector }
\description{
  Extract the last element of a vector. Useful for the \code{turnogram()} function
}
\usage{
last(x, na.rm=FALSE)
}
\arguments{
  \item{x}{ a numerical vector }
  \item{na.rm}{ if \code{na.rm=TRUE}, then the last non-missing value (if any) is returned. By default, it is FALSE and the last element (whatever its value) is returned }
}
\value{
  a numerical value
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{first}}, \code{\link{turnogram}} }

\examples{
a <- c(NA, 1, 2, NA, 3, 4, NA)
last(a)
last(a, na.rm=TRUE)
}
\keyword{ manip }
\eof
\name{local.trend}
\alias{local.trend}
\alias{identify.local.trend}
\title{ Calculate local trends using cumsum }
\description{
  A simple method using cumulated sums that allows to detect changes in the tendency in a time series
}
\synopsis{
  local.trend(x, k=mean(x), plotit=TRUE, type="l", cols=1:2, ltys=2:1, xlab="Time", ylab="cusum", ...)
  identify.local.trend(x, ...)
}
\usage{
local.trend(x, k=mean(x), plotit=TRUE, \dots)
identify.local.trend(loctrd)
}
\arguments{
  \item{x}{ a regular time series (a 'rts' object under S+ or a 'ts' object under \R) }
  \item{k}{ the reference value to substract from cumulated sums. By default, it is the mean of all observations in the series }
  \item{plotit}{ if \code{plotit=TRUE} (by default), a graph with the cumsum curve superposed to the original series is plotted }
  \item{\dots}{ additional arguments for the graph }
  \item{loctrd}{ a 'local.trend' object, as returned by the function \code{local.trend()} }
}
\details{
  With \code{local.trend()}, you can:
  
  - detect changes in the mean value of a time series
  
  - determine the date of occurence of such changes
  
  - estimate the mean values on homogeneous intervals
}
\value{
  a 'local.trend' object is returned. It has the method \code{identify()}
}
\references{ 
Ibanez, F., J.M. Fromentin & J. Castel, 1993. \emph{Application de la mthode des sommes cumules  l'analyse des sries chronologiques ocanographiques.} C. R. Acad. Sci. Paris, Life Sciences, 316:745-748.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\note{ Once transitions are identified with this method, you can use \code{stat.slide()} to get more detailed information on each phase. A smoothing of the series using running medians (see \code{decmedian()}) allows also to detect various levels in a time series, but according to the median statistic. Under \R, see also the 'strucchange' package for a more complete, but more complex, implementation of cumsum applied to time series. }

\seealso{ \code{\link{trend.test}}, \code{\link{stat.slide}}, \code{\link{decmedian}} }

\examples{
data(bnr)
# Calculate and plot cumsum for the 8th series
bnr8.lt <- local.trend(bnr[,8])
# To identify local trends, use:
# identify(bnr8.lt)
# and click points between which you want to compute local linear trends...
}
\keyword{ ts }
\eof
\name{marbio}
\alias{marbio}
\non_function{}
\title{ Several zooplankton taxa measured across a transect }
\usage{data(marbio)}
\description{
The \code{marbio} data frame has 68 rows (stations) and 24 columns (taxonomic zooplankton groups).
Abundances are expressed in no per cubic meter of seawater.
}
\format{
  This data frame contains the following columns giving abundance of:
  \describe{
    \item{Acartia}{Acartia clausi - herbivorous}
    \item{AdultsOfCalanus}{Calanus helgolandicus (adults) - herbivorous}
    \item{Copepodits1}{Idem (copepodits 1) - omnivorous}
    \item{Copepodits2}{Idem (copepodits 2) - omnivorous}
    \item{Copepodits3}{Idem (copepodits 3) - omnivorous}
    \item{Copepodits4}{Idem (copepodits 4) - omnivorous}
    \item{Copepodits5}{Idem (copepodits 5) - omnivorous}
    \item{ClausocalanusA}{Clausocalanus size class A - herbivorous}
    \item{ClausocalanusB}{Clausocalanus size class B - herbivorous}
    \item{ClausocalanusC}{Clausocalanus size class C - herbivorous}
    \item{AdultsOfCentropages}{Centropages tipicus (adults) - omnivorous}
    \item{JuvenilesOfCentropages}{Centropages typicus (juv.) - omnivorous}
    \item{Nauplii}{Nauplii of copepods - filter feeders}
    \item{Oithona}{Oithona sp. - carnivorous}
    \item{Acanthaires}{Various species of acanthaires - misc}
    \item{Cladocerans}{Various species of cladocerans - carnivorous}
    \item{EchinodermsLarvae}{Larvae of echinoderms - filter feeders}
    \item{DecapodsLarvae}{Larvae of decapods - omnivorous}
    \item{GasteropodsLarvae}{Larvae of gastropods - filter feeders}
    \item{EggsOfCrustaceans}{Eggs of crustaceans - misc}
    \item{Ostracods}{Various species of ostracods - omnivorous}
    \item{Pteropods}{Various species of pteropods - herbivorous}
    \item{Siphonophores}{Various species of siphonophores - carnivorous}
    \item{BellsOfCalycophores}{Bells of calycophores - misc}
  }
}
\details{
This dataset corresponds to a single transect sampled with a plankton net across the Ligurian Sea front in the Mediterranean Sea, between Nice (France) and Calvi (Corsica). The transect extends from the Cap Ferrat (close to Nice) to about 65 km offshore in the direction of Calvi (along a bearing of 123). 68 regularly spaced samples where collected on this transect. For more information about the water masses and their physico-chemical characteristics, see the marphy dataset.
}
\source{
Boucher, J., F. Ibanez & L. Prieur (1987) \emph{Daily and seasonal variations in the spatial distribution of zooplankton populations in relation to the physical structure in the Ligurian Sea Front.} Journal of Marine Research, 45:133-173.
}
\references{
Fromentin, J.-M., F. Ibanez & P. Legendre (1993) \emph{A phytosociological method for interpreting plankton data.} Marine Ecology Progress Series, 93:285-306.
}

\seealso{ \code{\link{marphy}} }

\keyword{datasets}
\eof
\name{marphy}
\alias{marphy}
\non_function{}
\title{ Physico-chemical records at the same stations as for marbio }
\usage{data(marphy)}
\description{
The \code{marphy} data frame has 68 rows (stations) and 4 columns.
They are seawater measurements at a deep of 3 to 7 m at the same 68 stations as \code{marbio}.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Temperature}{Seawater temperature in C}
    \item{Salinity}{Salinity in g/kg}
    \item{Fluorescence}{Fluorescence of chlorophyll a}
    \item{Density}{Excess of volumic mass of the seawater in g/l}
  }
}
\details{
This dataset corresponds to a single transect sampled across the Ligurian Sea front in the Mediterranean Sea, between Nice (France) and Calvi (Corsica). The transect extends from the Cap Ferrat (close to Nice) to about 65 km offshore in the direction of Calvi (along a bearing of 123). 68 regularly spaced measurements where recorded. They correspond to the stations where zooplankton was collected in the \code{marbio} dataset. Water masses in the transect across the front where identified as:
\describe{
    \item{Stations 1 to 17}{Peripheral zone}
    \item{Stations 17 to 25}{D1 (divergence) zone}
    \item{Stations 25 to 30}{C (convergence) zone}
    \item{Stations 30 to 41}{Frontal zone}
    \item{Stations 41 to 46}{D2 (divergence) zone}
    \item{Stations 46 to 68}{Central zone}
  }
}
\source{
Boucher, J., F. Ibanez & L. Prieur (1987) \emph{Daily and seasonal variations in the spatial distribution of zooplankton populations in relation to the physical structure in the Ligurian Sea Front.} Journal of Marine Research, 45:133-173.
}
\references{
Fromentin, J.-M., F. Ibanez & P. Legendre (1993) \emph{A phytosociological method for interpreting plankton data.} Marine Ecology Progress Series, 93:285-306.
}

\seealso{ \code{\link{marbio}} }

\keyword{datasets}
\eof
\name{match.tol}
\alias{match.tol}
\title{ Determine matching observation with a tolerance in time-scale }
\description{
  Determine which observations in a regular time series match observation in an original irregular one, accepting a given tolerance window in the time-scale. This function is internally used for regulation (functions \code{regul()}, \code{regul.screen()} and \code{regul.adj()}
}
\usage{
match.tol(x, table, nomatch=NA, tol.type="both", tol=0)
}
\arguments{
  \item{x}{ a numerical vector containing seeked values (time-scale of the regular series) }
  \item{table}{ a numerical vector containing initial times to look for match in \code{x} }
  \item{nomatch}{ the symbol tu use to flag an entry where no match is found. By default, \code{nomatch=NA} }
  \item{tol.type}{ the type of adjustment to use for the time-tolerance: \code{"left"}, \code{"right"}, \code{"both"} (by default) or \code{"none"}. If \code{tol.type="left"}, corresponding \code{x} values are seeked in a window ]table-tol, table]. If \code{tol.type="right"}, they are seeked in the window [table, table+tol[. If \code{tol.type="both"}, then they are seeked in the window ]table-tol, table+tol]. If several observations are in this window, the closest one is used. Finally, if \code{tol.type="none"}, then the function returns the \code{nomatch} symbol for all entries }
  \item{tol}{ the tolerance in the time-scale to determine if a value in \code{x} matches a value in \code{table}. If \code{tol=0}, observations in each respective series must match exactly, otherwise observations in the regulated series are interpolated. \code{tol} must be a round fraction of the interval between observations in \code{x} (x[i+1] - x[i], (x[i+1] - x[i])/2, (x[i+1] - x[i])/3, etc...), and cannot be larger than it, otherwise, \code{tol} is automatically adjusted to the closest allowed value. By default, \code{tol=NULL}. This is equivalent to \code{tol=0}. Warning! }
}
\value{
  a vector of the same length of \code{x}, indicating the position of the matching observations in \code{table}
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{regul}}, \code{\link{regul.screen}}, \code{\link{regul.adj}} }

\examples{
X <- 1:5
Table <- c(1, 3.1, 3.8, 4.4, 5.1, 6)
match.tol(X, Table)
match.tol(X, Table, tol=0.2)
match.tol(X, Table, tol=0.55)
}
\keyword{ ts }
\keyword{ manip }
\eof
\name{pennington}
\alias{pennington}
\title{ Calculate Pennington statistics }
\description{
  Calculate the mean, the variance and the variance of the mean for a single series, according to Pennington (correction for zero/missing values for abundance of species collected with a net)
}
\usage{
pennington(x, calc="all", na.rm=FALSE)
}
\arguments{
  \item{x}{ a numerical vector }
  \item{calc}{ indicate which estimator(s) should be calculated. Use: \code{"mean"}, \code{"var"}, \code{"mean.var"} or \code{"all"} (by default) for the mean, the variance, the variance of the mean or all these three statitics, respectively }
  \item{na.rm}{ if \code{na.rm=TRUE}, missing data are eliminated first. If it is FALSE (by default), any missing value in the series leads to NA as the result for the statistic }
}
\details{
  A complex problem in marine ecology is the notion of zero. In fact, the random sampling of a fish population often leads to a table with many null values. Do these null values indicate that the fish was absent or do we have to consider these null values as missing data, that is, that the fish was rare but present and was not caught by the net? For instance, for 100 net trails giving 80 null values, how to estimate the mean? Do we have to divide the sum of fishes caught by 100 or by 20?
  
  Pennington (1983) applied in this case the probabilistic theory of Aitchison (1955). The later established a method to estimate mean and variance of a random variable with a non-null probability to present zero values and whose the conditional distribution corresponding to its non-null values follows a Gaussian curve. In practice, a lognormal distribution is found most of the time for non-null values in fishes population. It is also often the case for the plankton, after our own experience. \code{pennington()} calculates thus statistics for such conditional lognormal distributions.
}
\value{
    a single value, or a vector with "mean", "var" and "mean.var" if the argument \code{calc="all"} 
}
\references{ 
Aitchison, J., 1955. \emph{On the distribution of a positive random variable having a discrete probability mass at the origin.} J. Amer. Stat. Ass., 50:901-908.

Pennington, M., 1983. \emph{Efficient estimations of abundance for fish and plankton surveys.} Biometrics, 39:281-286.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\note{
For multiple series, or for getting more statistics on the series, use \code{stat.pen()}. Use \code{stat.slide()} for obtaining statistics calculated separately for various intervals along the time for a time series
}

\seealso{ \code{\link{stat.pen}}, \code{\link{stat.slide}} }

\examples{
data(marbio)
pennington(marbio[, "Copepodits2"])
pennington(marbio[, "Copepodits2"], calc="mean", na.rm=TRUE)
}
\keyword{ univar }
\eof
\name{pgleissberg}
\alias{pgleissberg}
\title{ Gleissberg distribution probability }
\description{
  The Gleissberg distribution gives the probability to have k extrema in a series of n observations. This distribution is used in the turnogram to determine if monotony indices are significant (see \code{turnogram()})
}
\usage{
pgleissberg(n, k, lower.tail=TRUE, two.tailed=FALSE)
}
\arguments{
  \item{n}{ the number of observations in the series }
  \item{k}{ the number of extrema in the series, as calculated by \code{turnpoints()} }
  \item{lower.tail}{ if \code{lower.tail=TRUE} (by default) and \code{two.tailed=FALSE}, the left-side probability is returned. If it is FALSE, the right-side probability is returned }
  \item{two.tailed}{ if \code{two.tailed=TRUE}, the two-sided probability is returned. By default, it is FALSE and a one-sided probability is returned (left or right, depending on the value of \code{lower.tail} }
}
\value{
  a value giving the probability to have \code{k} extrema in a series of \code{n} observations
}
\references{ 
Dallot, S. & M. Etienne, 1990. \emph{Une mthode non paramtrique d'analyse des sries en ocanographie biologique: les tournogrammes.} Biomtrie et ocanographie - Socit de biomtrie, 6, Lille, 26-28 mai 1986. IFREMER, Actes de colloques, 10:13-31.

Johnson, N.L. & Kotz, S., 1969. \emph{Discrete distributions.} J. Wiley & sons, New York, 328 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ The Gleissberg distribution is asymptotically normal. For \code{n} > 50, the distribution is approximated by a Gaussian curve. For lower \code{n} values, the exact probability is returned (using data in the variable \code{.gleissberg.table} }

\seealso{ \code{\link{.gleissberg.table}}, \code{\link{turnpoints}}, \code{\link{turnogram}}}

\examples{
# Until n=50, the exact probability is returned
pgleissberg(20, 10, lower.tail=TRUE, two.tailed=FALSE)
# For higher n values, it is approximated by a normal distribution
pgleissberg(60, 33, lower.tail=TRUE, two.tailed=FALSE)
}
\keyword{ distribution }
\eof
\name{regarea}
\alias{regarea}
\title{ Regulate a series using the area method }
\description{
  Transform an irregular time series in a regular time series, or fill gaps in regular time series using the area method
}
\usage{
regarea(x, y=NULL, xmin=min(x), n=length(x),
        deltat=(max(x) - min(x))/(n - 1), rule=1,
        window=deltat, interp=FALSE, split=100)
}
\arguments{
  \item{x}{ a vector with time in the irregular series. Missing values are allowed }
  \item{y}{ a vector of same length as \code{x} and holding observations at corresponding times }
  \item{xmin}{ allows to respecify the origin of time in the calculated regular time series. By default, the origin is not redefined and it is equivalent to the smallest value in \code{x} }
  \item{n}{ the number of observations in the regular time series. By default, it is the same number than in the original irregular time series (i.e., \code{length(x)} }
  \item{deltat}{ the time interval between two observations in the regulated time series }
  \item{rule}{ the rule to use for extrapolated values (outside of the range in the initial irregular time series) in the regular time series. With \code{rule=1} (by default), these entries are not calculated and get \code{NA}; with \code{rule=2}, these entries are extrapolated }
  \item{window}{ size of the window to use for interpolation. By default, adjacent windows are used (\code{window=deltat}) }
  \item{interp}{ indicates if matching observations in both series must be calculated (\code{interp=TRUE}), or if initial observations are used "as is" in the final regular series (\code{interp=FALSE}, by default) }
  \item{split}{ a parameter to optimise calculation time and to avoid saturation of the memory. Very long time series are splitted into smaller subunits. This is transparent for the user. The default value of \code{split=100} should be rarely changed. Give a lower value if the program fails and reports a memory problem during calculation }
}
\details{
  This is a linear interpolation method described by Fox (1972). It takes into account all observations located in a given time window around the missing observation. On the contrary to many other interpolations (constant, linear, spline), the interpolated curve does not pass by the initial observations. Indeed, a smoothing is also operated simultaneously by this method. The importance of the smoothing is dependent on the size of the window (the largest it is, the smoothest will be the calculated regular time series) 
}
\value{
  An object of type 'regul' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{hist()}, \code{extract()} and \code{specs()}.
}
\references{ 
Fox, W.T. & J.A. Brown, 1965. \emph{The use of time-trend analysis for environmental interpretation of limestones.} J. Geol., 73:510-518.

Ibanez, F., 1991. \emph{Treatment of the data deriving from the COST 647 project on coastal benthic ecology: The within-site  analysis.} In: B. Keegan (ed). Space and Time Series Data Analysis in Coastal Benthic Ecology. Pp 5-43.

Ibanez, F. & J.C. Dauvin, 1988. \emph{Long-term changes (1977-1987) on a muddy fine sand Abra alba - Melinna palmata population community from the Western English Channel.} J. Mar. Ecol. Prog. Ser., 49:65-81.
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{regul}}, \code{\link{regconst}}, \code{\link{reglin}}, \code{\link{regspline}}, \code{\link{regul.screen}}, \code{\link{regul.adj}}, \code{\link{tseries}}, \code{\link{is.tseries}} }

\examples{
data(releve)
# The 'Melosul' series is regulated with a 25-days window
reg <- regarea(releve$Day, releve$Melosul, window=25)
# Then with a 50-days window
reg2 <- regarea(releve$Day, releve$Melosul, window=50)
# The initial and both regulated series are shown on the graph for comparison
plot(releve$Day, releve$Melosul, type="l")
lines(reg$x, reg$y, col=2)
lines(reg2$x, reg2$y, col=4)
}
\keyword{ ts }
\keyword{ manip }
\keyword{ chron }
\keyword{ smooth }
\eof
\name{regconst}
\alias{regconst}
\title{ Regulate a series using the constant value method }
\description{
  Transform an irregular time series in a regular time series, or fill gaps in regular time series using the constant value method
}
\usage{
regconst(x, y=NULL, xmin=min(x), n=length(x),
        deltat=(max(x) - min(x))/(n - 1), rule=1, f=0)
}
\arguments{
  \item{x}{ a vector with time in the irregular series. Missing values are allowed }
  \item{y}{ a vector of same length as \code{x} and holding observations at corresponding times }
  \item{xmin}{ allows to respecify the origin of time in the calculated regular time series. By default, the origin is not redefined and it is equivalent to the smallest value in \code{x} }
  \item{n}{ the number of observations in the regular time series. By default, it is the same number than in the original irregular time series (i.e., \code{length(x)} }
  \item{deltat}{ the time interval between two observations in the regulated time series }
  \item{rule}{ the rule to use for extrapolated values (outside of the range in the initial irregular time series) in the regular time series. With \code{rule=1} (by default), these entries are not calculated and get \code{NA}; with \code{rule=2}, these entries are extrapolated }
  \item{f}{ coefficient giving more weight to the left value (\code{f=0}, by default), to the right value (\code{f=}) or to a combination of these two observations (0 < f <1) }
}
\details{
  This is the simplest, but the less powerful regulation method. Interpolated values are calculated according to existing observations at left and at right as: x[reg] = x[right]*f + x[left]*(f-1), with 0 < f < 1.
}
\value{
  An object of type 'regul' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{hist()}, \code{extract()} and \code{specs()}.  
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ This function uses \code{approx()} for internal calculations }

\seealso{ \code{\link{regul}}, \code{\link{regarea}}, \code{\link{reglin}}, \code{\link{regspline}}, \code{\link{regul.screen}}, \code{\link{regul.adj}}, \code{\link{tseries}}, \code{\link{is.tseries}}, \code{\link[base]{approxfun}} }

\examples{
data(releve)
reg <- regconst(releve$Day, releve$Melosul)
plot(releve$Day, releve$Melosul, type="l")
lines(reg$x, reg$y, col=2)
}
\keyword{ ts }
\keyword{ manip }
\keyword{ chron }
\keyword{ smooth }

\eof
\name{reglin}
\alias{reglin}
\title{ Regulation of a series using a linear interpolation }
\description{
  Transform an irregular time series in a regular time series, or fill gaps in regular time series using a linear interpolation
}
\usage{
reglin(x, y=NULL, xmin=min(x), n=length(x),
        deltat=(max(x) - min(x))/(n - 1), rule=1)
}
\arguments{
  \item{x}{ a vector with time in the irregular series. Missing values are allowed }
  \item{y}{ a vector of same length as \code{x} and holding observations at corresponding times }
  \item{xmin}{ allows to respecify the origin of time in the calculated regular time series. By default, the origin is not redefined and it is equivalent to the smallest value in \code{x} }
  \item{n}{ the number of observations in the regular time series. By default, it is the same number than in the original irregular time series (i.e., \code{length(x)} }
  \item{deltat}{ the time interval between two observations in the regulated time series }
  \item{rule}{ the rule to use for extrapolated values (outside of the range in the initial irregular time series) in the regular time series. With \code{rule=1} (by default), these entries are not calculated and get \code{NA}; with \code{rule=2}, these entries are extrapolated }
}
\details{
  Observed values are connected by lines and interpolated values are obtained from this "polyline".
}
\value{
  An object of type 'regul' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{hist()}, \code{extract()} and \code{specs()}.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ This function uses \code{approx()} for internal calculations }

\seealso{ \code{\link{regul}}, \code{\link{regarea}}, \code{\link{regconst}}, \code{\link{regspline}}, \code{\link{regul.screen}}, \code{\link{regul.adj}}, \code{\link{tseries}}, \code{\link{is.tseries}}, \code{\link[base]{approxfun}} }

\examples{
data(releve)
reg <- reglin(releve$Day, releve$Melosul)
plot(releve$Day, releve$Melosul, type="l")
lines(reg$x, reg$y, col=2)
}
\keyword{ ts }
\keyword{ manip }
\keyword{ chron }
\keyword{ smooth }
\eof
\name{regspline}
\alias{regspline}
\title{ Regulation of a time series using splines }
\description{
  Transform an irregular time series in a regular time series, or fill gaps in regular time series using splines
}
\usage{
regspline(x, y=NULL, xmin=min(x), n=length(x),
        deltat=(max(x) - min(x))/(n - 1), rule=1, periodic=FALSE)
}
\arguments{
  \item{x}{ a vector with time in the irregular series. Missing values are allowed }
  \item{y}{ a vector of same length as \code{x} and holding observations at corresponding times }
  \item{xmin}{ allows to respecify the origin of time in the calculated regular time series. By default, the origin is not redefined and it is equivalent to the smallest value in \code{x} }
  \item{n}{ the number of observations in the regular time series. By default, it is the same number than in the original irregular time series (i.e., \code{length(x)} }
  \item{deltat}{ the time interval between two observations in the regulated time series }
  \item{rule}{ the rule to use for extrapolated values (outside of the range in the initial irregular time series) in the regular time series. With \code{rule=1} (by default), these entries are not calculated and get \code{NA}; with \code{rule=2}, these entries are extrapolated }
  \item{periodic}{ indicates if the time series should be considered as periodic (\code{periodic=TRUE}, first value must be equal to the last one). If this is the case, first and second derivates used to calculate spline segments around first and last observations use data in the other extreme of the series. In the other case (\code{periodic=FALSE} (by default), derivates for extremes observations are considered to be equal to zero }
}
\details{
  Missing values are interpolated using cubic splines between observed values.
}
\value{
  An object of type 'regul' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{hist()}, \code{extract()} and \code{specs()}.
}
\references{ 
Lancaster, P. & K. Salkauskas, 1986. \emph{Curve and surface fitting.} Academic Press, England, 280 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ This function uses \code{spline()} for internal calculations. However, interpolated values are not allowed to be higher than the largest initial observation or lower than the smallest one. }

\seealso{ \code{\link{regul}}, \code{\link{regarea}}, \code{\link{regconst}}, \code{\link{reglin}}, \code{\link{regul.screen}}, \code{\link{regul.adj}}, \code{\link{tseries}}, \code{\link{is.tseries}}, \code{\link[base]{splinefun}} }

\examples{
data(releve)
reg <- regspline(releve$Day, releve$Melosul)
plot(releve$Day, releve$Melosul, type="l")
lines(reg$x, reg$y, col=2)
}
\keyword{ ts }
\keyword{ manip }
\keyword{ chron }
\keyword{ smooth }
\eof
\name{regul}
\alias{regul}
\alias{extract.regul}
\alias{hist.regul}
\alias{identify.regul}
\alias{lines.regul}
\alias{plot.regul}
\alias{print.regul}
\alias{print.specs.regul}
\alias{print.summary.regul}
\alias{specs.regul}
\alias{summary.regul}
\title{ Regulation of one or several time series using various methods }
\description{
  Regulate irregular time series or regular time series with gaps. Create a \code{regul} object from whose one or several regular time series can be extracted using \code{extract()} or \code{tseries()}. This is the function to apply most of the time to create regular time series ('rts' objects in Splus or 'ts' objects in \R) that will be further analyzed by other functions that apply to regular time series.
}
\synopsis{
regul(x, y=NULL, xmin=min(x), n=length(x), units="days", frequency=NULL, deltat=1/frequency, datemin=NULL, dateformat="m/d/Y", tol=NULL, tol.type="both", methods="linear", rule=1, f=0, periodic=FALSE, window=(max(x) - min(x))/(n - 1), split=100, specs=NULL) 
extract.regul(e, n, series=NULL, ...)
hist.regul(x, nclass=30, col=c(4, 5, 2), xlab=paste("Time distance in", x$units, "with start =", min(x$x), , ylab=paste("Frequency, tol =", x$specs$tol), main="Number of matching observations", plotit=TRUE, ...)
identify.regul(x, series=1, col=3, label="#", ...)
lines.regul(x, series=1, col=3, lty=1, plot.pts=TRUE, ...)
plot.regul(x, series=1, col=c(1, 2), lty=c(par("lty"), par("lty")), plot.pts=TRUE, leg=FALSE, llab=c("initial", x$specs$methods[series]), lpos=c(1.5, 10), xlab=paste("Time (", x$units, ")", sep = ""), ylab="Series", main=paste("Regulation of", names(x$y)[series]), ...)
print.regul(x, ...)
print.specs.regul(x, ...)
print.summary.regul(x, ...)
specs.regul(x, ...)
summary.regul(object, ...)
}
\usage{
regul(x, y=NULL, xmin=min(x), n=length(x), units="days", frequency=NULL,
        deltat=1/frequency, datemin=NULL, dateformat="m/d/Y", tol=NULL,
        tol.type="both", methods="linear", rule=1, f=0, periodic=FALSE,
        window=(max(x) - min(x))/(n - 1), split=100, specs=NULL)
\method{summary}{regul}(reg)
\method{plot}{regul}(reg, series=1, col=c(1, 2), lty, plot.pts=TRUE,
        leg=FALSE, llab=c("initial", x$specs$methods[series]),
        lpos=c(1.5, 10), \dots)
\method{lines}{regul}(reg, series=1, col=3, lty=1, plot.pts=TRUE, \dots)
\method{identify}{regul}(reg, series=1, col=3, label="#", \dots)
\method{hist}{regul}(reg, nclass=30, col=c(4, 5, 2), plotit=TRUE, \dots)
\method{extract}{regul}(reg, n=ncol(reg$y), series=NULL)
\method{specs}{regul}(reg)
}
\arguments{
  \item{x}{ a vector containing times at which observations are sampled in the initial irregular time series. It can be expressed in any unit ("years", "days", "weeks", "hours", "min", "sec",...) as defined by the argument \code{units}. It is often expressed in "days" and the decimal part represents the part of the day, that is the time in hour:min:sec (dates coming from Excel, or even standard dates in S+ or \R are expressed like that) }
  \item{y}{ a vector (single series) or a matrix/data frame whose columns correspond to the various irregular time series to regulate. Rows are observations made at corresponding times in \code{x}. The number of rows must thus match the length of vector \code{x} }
  \item{xmin}{ allows to respecify the origin of time in \code{x}. By default, the origin is not redefined and thus, the smallest value in \code{x} is used }
  \item{n}{ the number of observations in the regular time series. By default, it is the same number than in the original irregular time series (i.e., \code{length(x)} }
  \item{units}{ the time unit for the \code{x} vector. By default \code{units="days"}. A special value, \code{units="daystoyears"} indicates that \code{x} is expressed in "days" (1 unit = 1 day) but that we want to obtain the final regular time series expressed in "years" (1 unit = 1 year). Give a correct value for \code{datemin} to make sure the right fraction of the year is computed for each observation (see example hereunder) }
  \item{frequency}{ the frequency of the regulated time series in the corresponding time unit. For instance, \code{frequency=12} with \code{units="years"} means montly sampled observations. Warning! When using \code{units="daystoyears"}, specify \code{frequency} (or \code{deltat}) in years! }
  \item{deltat}{ the interval between two observations in the regulated time series. It is the inverse of \code{frequency}. If both \code{frequency} and \code{deltat} are provided, then \code{frequency} supersedes \code{deltat} }
  \item{datemin}{ this is mostly useful for converting "days" in "years" time-scales (\code{units="daystoyears"}). If the \code{x} vector contains: 1, 3, 6,... (day \#1, day \#3, day \#6,... of the experiment), one can give here the exact date of the first observation, allowing to define a correct origin in the "years" time scale. Provide a string in a format compatible with \code{dateformat}. For instance, if day \#1 is the 21th March 1998, give \code{datemin="03/21/1998"} with \code{dateformat="m/d/Y"} }
  \item{dateformat}{ indicate how \code{datemin} is formated. For instance: \code{"d/m/Y"}, or \code{m/d/Y"} (by default), see \code{daystoyears()} for more info on date formatting }
  \item{tol}{ the tolerance in the time-scale to determine if a measured value is used to approximate a regulated value. If \code{tol=0}, observations in each respective series must match exactly, otherwise observations in the regulated series are interpolated. \code{tol} must be a round fraction of \code{deltat} (deltat, deltat/2, deltat/3, etc...), and cannot be larger than it, otherwise, \code{tol} is automatically adjusted to the closest allowed value. By default, \code{tol=NULL}. This is equivalent to \code{tol=0}. Warning! In the particular case of \code{units="daystoyears"}, \code{tol} must be expressed in the original time-scale, that is "days", while \code{deltat} must be expressed in the fimal time-scale, that is "years"! }
  \item{tol.type}{ the type of adjustment to use for the time-tolerance: \code{"left"}, \code{"right"}, \code{"both"} (by default) or \code{"none"}. If \code{tol.type="left"}, corresponding \code{x} values are seeked in a window ]xregul-tol, xregul]. If \code{tol.type="right"}, they are seeked in the window [xregul, xregul+tol[. If \code{tol.type="both"}, then they are seeked in the window ]xregul-tol, xregul+tol]. If several observations are in this window, the closest one is used. Finally, if \code{tol.type="none"}, then \emph{all} observations in the regulated time series are interpolated (even if exactly matching observations exist!) }
  \item{methods}{ the method(s) to use to regulate the time series. Currently, it can be: \code{"constant"}, \code{"linear"}, \code{"spline"} or \code{"area"} (or a unique abbreviation of them). If several time series are provided (\code{y} is a matrix or a data frame), it is possible to define methods individually for each series. For instance, \code{methods=c("l", "a", "s")} defines the "linear" method for the first series, the "area" method for the second one, the "spline" method for the third one,... and again the "linear" for the fourth, the "area" for the fifth one, etc. (recycling rule). By default, the "linear" method is selected for all series }
  \item{rule}{ the rule to use for extrapolated values (observations in the final regular time series that are outside the range of observed values in the initial time series). With \code{rule=1} (by default), these entries are not calculated and get NA; with \code{rule=2}, these entries are extrapolated (avoid using this option, or use with extreme care!!!) }
  \item{f}{ parameter for the \code{"constant"} regulation method. Coefficient giving more weight to the observation at left (\code{f=0}, by default), to the observation at right (\code{f=1}), or give an intermediate weight to both of these observations (0 < f < 1) during the interpolation (see \code{reglin()} }
  \item{periodic}{ parameter for the \code{"spline"} regulation method. Indicate if the time series should be considered as periodic (\code{periodic=TRUE}, first value must be equal to the last one). If this is the case, first and second derivates used to calculate spline segments around first and last observations use data in the other extreme of the series. In the other case (\code{periodic=FALSE}, by default), derivates for extremes observations are considered to be equal to zero }
  \item{window}{ parameter for the \code{"area"} regulation method. Size of the window to consider (see \code{regarea()}). By default, the mean interval between observations in the initial irregular time series is used. Give the same value as for deltat for working with adjacent windows }
  \item{split}{ other parameter for the \code{"area"} method. To optimise calculation time and to avoid to saturate memory, very long time series are splitted into smaller subunits (see \code{regarea()}). This is transparent for the user. The default value of \code{split=100} should be rarely changed. Give a lower value if the program fails and reports a memory problem during calculation }
  \item{specs}{ a \code{specs.regul} object returned by the function \code{specs()} applied to a \code{regul} object. Allows to collect parameterization of the \code{regul()} function and to apply them to another regulation }
  \item{reg}{ A \code{regul} object as obtained after using the \code{regul()} function }
  \item{series}{ the series to plot. By default, \code{series=1}, corresponding to the first (or possibly the unique) series in the \code{regul} object }
  \item{col}{ (1) for \code{plot()}: the two colors to use to draw respectively the initial irregular series and the final regulated series. \code{col=c(1,2)} by default. (2) for \code{hist()}: the three colors to use to represent respectively the fist bar (exact coincidence), the middle bars (coincidence in a certain tolerance window) and the last bar (values always interpolated). By default, \code{col=c(4,5,2)} }
  \item{lty}{ the style to use to draw lines for the initial series and the regulated series, respectively. The default style is used for both lines if this argument is not provided }
  \item{plot.pts}{ if \code{plot.pts=TRUE} (by default) then points are also drawn for the regulated series (+). Those points that match observations in the initial irregular series, and are not interpolated, are further marked with a circle }
  \item{leg}{ do we add a legend to the graph? By default, \code{leg=FALSE}, no legend is added }
  \item{llab}{ the labels to use for the initial irregular and the final regulated series, respectively. By default, it is \code{"initial"} for the first one and the name of the regulation method used for the second one (see \code{methods} argument) }
  \item{lpos}{ the position of the top-left corner of the legend box (x,y), in the graph coordinates }
  \item{\dots}{ additional graph parameters }
  \item{label}{ the character to use to mark points interactively selected on the graph. By default, \code{label="#"} }
  \item{nclass}{ the number of classes to calculate in the histogram. This is indicative and this value is automatically adjusted to obtain a nicely-formatted histogram. By default, \code{nclass=30} }
  \item{plotit}{ If \code{plotit=TRUE} then the histogram is plotted. Otherwise, it is only calculated }
}
\details{
  Several irregular time series (for instance, contained in a data frame) can be treated at once. Specify a vector with \code{"constant"}, \code{"linear"}, \code{"spline"} or \code{"area"} for the argument \code{methods} to use a different regulation method for each series. See corresponding fonctions (\code{regconst()}, \code{reglin()}, \code{regspline()} and \code{regarea()}), respectively, for more details on these methods. Arguments can be saved in a \code{specs} object and reused for other similar regulation processes. Functions \code{regul.screen()} and \code{regul.adj()} are useful to chose best time interval in the computed regular time series. If you want to work on seasonal effects in the time series, you will better use a "years" time-scale (1 unit = 1 year), or convert into such a scale. If initial time unit is "days" (1 unit = 1 day), a conversion can be operated at the same time as the regulation by specifying \code{units="daystoyears"}.
}
\value{
  An object of type 'regul' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()}, \code{identify()}, \code{hist()}, \code{extract()} and \code{specs()}.
}
\references{ 
Lancaster, P. & K. Salkauskas, 1986. \emph{Curve and surface fitting.} Academic Press, England, 280 pp.

Fox, W.T. & J.A. Brown, 1965. \emph{The use of time-trend analysis for environmental interpretation of limestones.} J. Geol., 73:510-518.

Ibanez, F., 1991. \emph{Treatment of the data deriving from the COST 647 project on coastal benthic ecology: The within-site  analysis.} In: B. Keegan (ed). Space and Time Series Data Analysis in Coastal Benthic Ecology. Pp 5-43.

Ibanez, F. & J.C. Dauvin, 1988. \emph{Long-term changes (1977-1987) on a muddy fine sand Abra alba - Melinna palmata population community from the Western English Channel.} J. Mar. Ecol. Prog. Ser., 49:65-81.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\seealso{ \code{\link{regul.screen}}, \code{\link{regul.adj}}, \code{\link{tseries}}, \code{\link{is.tseries}}, \code{\link{regconst}}, \code{\link{reglin}}, \code{\link{regspline}}, \code{\link{regarea}}, \code{\link{daystoyears}} }

\examples{
data(releve)
# The series in this data frame are very irregularly sampled in time:
releve$Day
length(releve$Day)
intervals <- releve$Day[2:61]-releve$Day[1:60]
intervals
range(intervals)
mean(intervals)
# The series must be regulated to be converted in a 'rts' or 'ts object
rel.reg <- regul(releve$Day, releve[3:8], xmin=9, n=63, deltat=21,
        tol=1.05, methods=c("s","c","l","a","s","a"), window=21)
rel.reg
plot(rel.reg, 5)
specs(rel.reg)
# Now we can extract one or several regular time series
melo.ts <- extract(rel.reg, series="Melosul")
is.tseries(melo.ts)

# One can convert time-scale from "days" to "years" during regulation
# This is most useful for analyzing seasonal cycles in a second step
melo.regy <- regul(releve$Day, releve$Melosul, xmin=6, n=87,
        units="daystoyears", frequency=24, tol=2.2, methods="linear",
        datemin="21/03/1989", dateformat="d/m/Y")
melo.regy
plot(melo.regy, main="Regulation of Melosul")
# In this case, we have only one series in 'melo.regy'
# We can use also 'tseries()' instead of 'extract()'
melo.tsy <- tseries(melo.regy)
is.tseries(melo.tsy)
}
\keyword{ ts }
\keyword{ manip }
\keyword{ chron }
\keyword{ smooth }
\eof
\name{regul.adj}
\alias{regul.adj}
\title{ Adjust regulation parameters }
\description{
  Calculate and plot an histogram of the distances between interpolated observations in a regulated time series and closest observations in the initial irregular time series. This allows to optimise the \code{tol} parameter
}
\synopsis{
regul.adj(x, xmin=min(x), frequency=NULL, deltat=(max(x, na.rm = TRUE) - min(x, na.rm = TRUE))/(length(x) - 1), tol=deltat, tol.type="both", nclass=50, col=c(4, 5, 2), xlab=paste("Time distance"), ylab=paste("Frequency"), main="Number of matching observations", plotit=TRUE, ...)
}
\usage{
regul.adj(x, xmin=min(x), frequency=NULL, deltat, tol=deltat,
        tol.type="both", nclass=50, col=c(4, 5, 2), plotit=TRUE, \dots)
}
\arguments{
  \item{x}{ a vector with times corresponding to the observations in the irregular initial time series }
  \item{xmin}{ the time corresponding to the first observation in the regular time series }
  \item{frequency}{ the frequency of observations in the regular time series }
  \item{deltat}{ the interval between two successive observations in the regular time series. This is the inverse of \code{frequency}. Only one of both parameters need to be given. If both are provided, \code{frequency} supersedes \code{deltat} }
  \item{tol}{ the tolerance in the difference between two matching observations (in the original irregular series and in the regulated series). If \code{tol=0} both values must be strictly identical; a higher value for \code{tol} allows some fuzzy matching. \code{tol} must be a round fraction of \code{deltat} and cannot be higher than it, otherwise, it is adjusted to the closest acceptable value. By default, \code{tol=deltat} }
  \item{tol.type}{ the type of window to use for the time-tolerance: \code{"left"}, \code{"right"}, \code{"both"} (by default) or \code{"none"}. If \code{tol.type="left"}, corresponding \code{x} values are seeked in a window ]xregul-tol, xregul]. If \code{tol.type="right"}, they are seeked in the window [xregul, xregul+tol[. If \code{tol.type="both"}, then they are seeked in the window ]xregul-tol, xregul+tol]. If several observations are in this window, the closest one is used. Finally, if \code{tol.type="none"}, then \emph{all} observations in the regulated time series are interpolated (even if exactly matching observations exist!) }
  \item{nclass}{ the number of classes to compute in the histogram. This is indicative, and will be adjusted by the algorithm to produce a nicely-formatted histogram. The default value is \code{nclass=50}. It is acceptable in many cases, but if the histogram is not correct, try a larger value }
  \item{col}{ the three colors to use to represent respectively the fist bar (exact coincidence), the middle bars (coincidence in a certain tolerance window) and the last bar (values always interpolated). By default, \code{col=c(4,5,2)} }
  \item{plotit}{ if \code{plotit=TRUE} then the histogram is plotted. Otherwise, it is only calculated }
  \item{\dots}{ additional graph parameters for the histogram }
}
\details{
  This function is complementary to \code{regul.screen()}. While the later look for the best combination of the number of observations, the interval between observations and the position of the first observation on the time-scale for the regular time series, \code{regul.adj()} look for the optimal value for \code{tol}, the tolerance window.
}
\value{
  A list with components:
  \item{params}{ the parameters used for the regular time-scale }
  \item{match}{ the number of matching observations in the tolerance window }
  \item{exact.match}{ the number of exact matching observations }
  \item{match.counts}{ a vector with the number of matching observations for increasing values of \code{tol} }
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{regul.screen}}, \code{\link{regul}} }

\examples{
# This example follows the example for regul.screen()
# where we determined that xmin=9, deltat=21, n=63, with tol=1.05
# is a good choice to regulate the irregular time series in 'releve' 
data(releve)
regul.adj(releve$Day, xmin=9, deltat=21)
# The histogram indicates that it is not useful to increase tol
# more than 1.05, because few observations will be added
# except if we increase it to 5-7, but this value could be
# considered to be too large in comparison with deltat=22
# On the other hand, with tol <= 1, the number of matching
# observations will be almost divided by two!
}
\keyword{ ts }
\keyword{ chron }
\eof
\name{regul.screen}
\alias{regul.screen}
\title{ Test various regulation parameters }
\description{
  Seek for the best combination of the number of observation, the interval between two successive observation and the position of the first observation in the regulated time series to match as much observations of the initial series as possible
}
\synopsis{
regul.screen(x, weight=NULL, xmin=min(x), frequency=NULL, deltat=(max(x, na.rm = TRUE) - min(x, na.rm = TRUE))/(length(x) - 1), tol=deltat/5, tol.type="both")
}
\usage{
regul.screen(x, weight=NULL, xmin=min(x), frequency=NULL, deltat,
        tol=deltat/5, tol.type="both")
}
\arguments{
  \item{x}{ a vector with times corresponding to the observations in the irregular initial time series }
  \item{weight}{ a vector of the same length as \code{x}, with the weight to give to each observation. A value of 0 indicates to ignore an observation. A value of 1 gives a normal weight to an observation. A higher value gives more importance to the corresponding observation. You can increase weight of observations around major peaks and pits, to make sure they are not lost in the regulated time series. If \code{weight=NULL} (by default), then a weight of 1 is used for all observations }
  \item{xmin}{ a vector with all time values for the first observation in the regulated time series to be tested }
  \item{frequency}{ a vector with all the frequencies to be screened }
  \item{deltat}{ a vector with all time intervals to screen. \code{deltat} is the inverse of \code{frequency}. Only one of these two arguments is required. If both are provided, \code{frequency} supersedes \code{deltat} }
  \item{tol}{ it is possible to tolerate some differences in the time between two matching observations (in the original irregular series and in the regulated series). If \code{tol=0} both values must be strictly identical; a higher value allows some fuzzy matching. \code{tol} must be a round fraction of \code{deltat} and cannot be higher than it, otherwise, it is adjusted to the closest acceptable value. By default, \code{tol=deltat/5} }
  \item{tol.type}{ the type of window to use for the time-tolerance: \code{"left"}, \code{"right"}, \code{"both"} (by default) or \code{"none"}. If \code{tol.type="left"}, corresponding \code{x} values are seeked in a window ]xregul-tol, xregul]. If \code{tol.type="right"}, they are seeked in the window [xregul, xregul+tol[. If \code{tol.type="both"}, then they are seeked in the window ]xregul-tol, xregul+tol]. If several observations are in this window, the closest one is used. Finally, if \code{tol.type="none"}, then \emph{all} observations in the regulated time series are interpolated (even if exactly matching observations exist!) }
}
\details{
  Whatever the efficiency of the interpolation procedure used to regulate an irregular time series, a matching, non-interpolated observation is always better than an interpolated one! With very irregular time series, it is often difficult to decide which is the better regular time-scale in order to interpolate as less observations as possible. \code{regul.screen()} tests various combinations of number of observation, interval between two observations and position of the first observation and allows to choose the combination that best matches the original irregular time series. To choose also an optimal value for \code{tol}, use \code{regul.adj()} concurrently.
}
\value{
  A list containing:
  \item{tol}{ a vector with the adjusted values of \code{tol} for the various values of \code{deltat} }
  \item{n}{ a table indicating the maximum value of \code{n} for all combinations of \code{deltat} and \code{xmin} to avoid any extrapolation }
  \item{nbr.match}{ a table indicating the number of matching observations (in the tolerance window) for all combinations of \code{deltat} and \code{xmin} }
  \item{nbr.exact.match}{ a table indicating the number of exactly matching observations (with a tolerance window equal to zero) for all combinations of \code{deltat} and \code{xmin} }
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\seealso{ \code{\link{regul.adj}}, \code{\link{regul}} }

\examples{
data(releve)
# This series is very irregular, and it is difficult
# to choose the best regular time-scale
releve$Day
length(releve$Day)
intervals <- releve$Day[2:61]-releve$Day[1:60]
intervals
range(intervals)
mean(intervals)
# A combination of xmin=1, deltat=22 and n=61 seems correct
# But is it the best one?
regul.screen(releve$Day, xmin=0:11, deltat=16:27, tol=1.05)
# Now we can tell that xmin=9, deltat=21, n=63, with tol=1.05
# is a much better choice! 
}
\keyword{ ts }
\keyword{ chron }
\eof
\name{releve}
\alias{releve}
\non_function{}
\title{ A data frame of six phytoplankton taxa followed in time at one station }
\usage{data(releve)}
\description{
The \code{releve} data frame has 61 rows and 8 columns. Several phytoplankton taxa were numbered in a single station from 1989 to 1992, but at irregular times. 
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Day}{days number, first observation being day 1}
    \item{Date}{strings indicating the date of the observations in "dd/mm/yyyy" format}
    \item{Astegla}{the abundance of Asterionella glacialis}
    \item{Chae}{the abundance of Chaetoceros sp}
    \item{Dity}{the abundance of Ditylum sp}
    \item{Gymn}{the abundance of Gymnodinium sp}
    \item{Melosul}{the abundance of Melosira sulcata + Paralia sulcata}
    \item{Navi}{the abundance of Navicula sp}
  }
}
\source{
 Belin, C. & B. Raffin, 1998. \emph{Les espces phytoplanctoniques toxiques et nuisibles sur le littoral franais de 1984  1995, rsultats du REPHY (rseau de surveillance du phytoplancton et des phycotoxines)}. Rapport IFREMER, RST.DEL/MP-AO-98-16. IFREMER, France.
}
\keyword{datasets}
\eof
\name{specs}
\alias{specs}
\title{ Collect parameters ("specifications") from one object to use them in another analysis }
\description{
  `specs' is a generic function for reusing specifications included in an object and applying them in another similar analysis 
}
\usage{
specs(x, ...)
}
\arguments{
  \item{x}{ An object that has "specs" data }
  \item{\dots}{ Additional arguments (redefinition of one or several parameters) }
}
\value{
  A `specs' object that has the \code{print} method and that can be entered as an argument to functions using similar "specifications"
}
\seealso{ \code{\link{regul}}, \code{\link{tsd}} }

\keyword{ methods }
\eof
\name{stat.desc}
\alias{stat.desc}
\title{ Descriptive statistics on a data frame or time series }
\description{
  Compute a table giving various descriptive statistics about the series in a data frame or in a single/multiple time series
}
\usage{
stat.desc(x, basic=TRUE, desc=TRUE, norm=FALSE, p=0.95)
}
\arguments{
  \item{x}{ a data frame or a time series }
  \item{basic}{ do we have to return basic statistics (by default, it is TRUE)? These are: the number of values (nbr.val), the number of null values (nbr.null), the number of missing values (nbr.na), the minimal value (min), the maximal value (max), the range (range, that is, max-min) and the sum of all non-missing values (sum) }
  \item{desc}{ do we have to return various descriptive statistics (by default, it is TRUE)? These are: the median (median), the mean (mean), the standard error on the mean (SE.mean), the confidence interval of the mean (CI.mean) at the \code{p} level, the variance (var), the standard deviation (std.dev) and the variation coefficient (coef.var) defined as the standard deviation divided by the mean }
  \item{norm}{ do we have to return normal distribution statistics (by default, it is FALSE)? the skewness coefficient g1 (skewness), its significant criterium (skew.2SE, that is, g1/2.SEg1; if skew.2SE > 1, then skewness is significantly different than zero), kurtosis coefficient g2 (kurtosis), its significant criterium (kurt.2SE, same remark than for skew.2SE), the statistic of a Shapiro-Wilk test of normality (normtest.W) and its associated probability (normtest.p) }
  \item{p}{ the probability level to use to calculate the confidence interval on the mean (CI.mean). By default, \code{p=0.95} }
}
\value{
  a data frame with the various statistics in rows and with each column correponding to a variable in the data frame, or to a separate time series
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ The Shapiro-Wilk test of normality is not available yet in Splus and it returns 'NA' in this environment. If you prefer to get separate statistics for various time intervals in your time series, use \code{stat.slide()}. If your data are fish or plankton sampled with a net, consider using the Pennington statistics (see \code{stat.pen()}) }

\seealso{ \code{\link{stat.slide}}, \code{\link{stat.pen}} }

\examples{
data(marbio)
stat.desc(marbio[,13:16], basic=TRUE, desc=TRUE, norm=TRUE, p=0.95)
}
\keyword{ ts }
\eof
\name{stat.pen}
\alias{stat.pen}
\title{ Pennington statistics on a data frame or time series }
\description{
  Compute a table giving various descriptive statistics, including Pennington's estimators of the mean, the variance and the variance of the mean, about the series in a data frame or in a single/multiple time series
}
\usage{
stat.pen(x, basic=FALSE, desc=FALSE)
}
\arguments{
  \item{x}{ a data frame or a time series }
  \item{basic}{ do we have to return also basic statistics (by default, it is FALSE)? These are: the number of values (nbr.val), the number of null values (nbr.null), the number of missing values (nbr.na), the minimal value (min), the maximal value (max), the range (range, that is, max-min) and the sum of all non-missing values (sum) }
  \item{desc}{ do we have to return also various descriptive statistics (by default, it is FALSE)? These are: the median (median), the mean (mean), the standard error on the mean (SE.mean), the confidence interval of the mean (CI.mean) at the \code{p} level, the variance (var), the standard deviation (std.dev) and the variation coefficient (coef.var) defined as the standard deviation divided by the mean }
}
\value{
  a data frame with the various statistics in rows and with each column correponding to a variable in the data frame, or to a separate time series
}
\references{ Aitchison, J., 1955. \emph{On the distribution of a positive random variable having a discrete probability mass at the origin.} J. Amer. Stat. Ass., 50:901-908.

Pennington, M., 1983. \emph{Efficient estimations of abundance for fish and plankton surveys.} Biometrics, 39:281-286.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ 
If you prefer to get separate statistics for various time intervals in your time series, use \code{stat.slide()}. Various other descriptive statistics, including test of the normal distribution are also available in \code{stat.desc()}
}

\seealso{ \code{\link{stat.slide}}, \code{\link{stat.desc}} }

\examples{
data(marbio)
stat.pen(marbio[,c(4, 14:16)], basic=TRUE, desc=TRUE)
}
\keyword{ ts }
\eof
\name{stat.slide}
\alias{stat.slide}
\alias{lines.stat.slide}
\alias{plot.stat.slide}
\alias{print.stat.slide}

\title{ Sliding statistics }
\description{
Statistical parameters are not constant along a time series: mean or variance can vary each year, or during particular intervals (radical or smooth changes due to a pollution, a very cold winter, a shift in the system behaviour, etc. Sliding statistics offer the potential to describe series on successive blocs defined along the space-time axis
}
\synopsis{
stat.slide(x, y, xcut=NULL, xmin=min(x), n=NULL, frequency=NULL, deltat=1/frequency, basic=FALSE, desc=FALSE, norm=FALSE, pen=FALSE, p=0.95)
lines.stat.slide(x, stat="mean", col=3, lty=1, ...)
plot.stat.slide(x, stat="mean", col=c(1, 2), lty=c(par("lty"), par("lty")), leg=FALSE, llab=c("series", stat), lpos=c(1.5, 10), xlab="time", ylab="y", main=paste("Sliding statistics"), ...)
print.stat.slide(x, ...)
}
\usage{
stat.slide(x, y, xcut=NULL, xmin=min(x), n=NULL, frequency=NULL,
        deltat=1/frequency, basic=FALSE, desc=FALSE, norm=FALSE,
        pen=FALSE, p=0.95)
plot.stat.slide(statsl, stat="mean", col=c(1, 2), lty=c(par("lty"), par("lty")),
        leg=FALSE, llab=c("series", stat), lpos=c(1.5, 10), \dots)
lines.stat.slide(statsl, stat="mean", col=3, lty=1, \dots)
}
\arguments{
  \item{x}{ a vector with time data }
  \item{y}{ a vector with observation at corresponding times }
  \item{xcut}{ a vector with the position in time of the breaks between successive blocs. \code{xcut=NULL} by default. In the later case, a vector with equally spaced blocs is constructed using \code{xmin}, \code{n} and \code{frequency} or \code{deltat}. If a value is provided for \code{xcut}, then it supersedes all these other parameters }
  \item{xmin}{ the minimal value in the time-scale to use for constructing a vector of equally spaced breaks }
  \item{n}{ the number of breaks to use }
  \item{frequency}{ the frequency of the breaks in the time-scale }
  \item{deltat}{ the bloc interval touse for constructing an equally-spaced break vector. \code{deltat} is 1/\code{frequency} }
  \item{basic}{ do we have to return basic statistics (by default, it is FALSE)? These are: the number of values (nbr.val), the number of null values (nbr.null), the number of missing values (nbr.na), the minimal value (min), the maximal value (max), the range (range, that is, max-min) and the sum of all non-missing values (sum) }
  \item{desc}{ do we have to return descriptive statistics (by default, it is FALSE)? These are: the median (median), the mean (mean), the standard error on the mean (SE.mean), the confidence interval of the mean (CI.mean) at the \code{p} level, the variance (var), the standard deviation (std.dev) and the variation coefficient (coef.var) defined as the standard deviation divided by the mean }
  \item{norm}{ do we have to return normal distribution statistics (by default, it is FALSE)? the skewness coefficient g1 (skewness), its significant criterium (skew.2SE, that is, g1/2.SEg1; if skew.2SE > 1, then skewness is significantly different than zero), kurtosis coefficient g2 (kurtosis), its significant criterium (kurt.2SE, same remark than for skew.2SE), the statistic of a Shapiro-Wilk test of normality (normtest.W) and its associated probability (normtest.p) }
  \item{pen}{ do we have to return Pennington and other associated statistics (by default, it is FALSE)? pos.median, pos.mean, pos.var, pos.std.dev, respectively the median, the mean, the standard deviation and the variance, considering only non-null values; geo.mean, the geometric mean that is, the exponential of the mean of the logarithm of the observations, excluding null values. pen.mean, pen.var, pen.std.dev, pen.mean.var, respectively the mean, the variance, the standard deviation and the variance of the mean after Pennington's estimators (see \code{pennington()}) }
  \item{p}{ the probability level to use to calculate the confidence interval on the mean (CI.mean). By default, \code{p=0.95} }
  \item{statsl}{ a 'stat.slide' object, as returned by the function \code{stat.slide()} }
  \item{stat}{ the statistic to plot on the graph. You can use "min", "max", "median", "mean" (by default), "pos.median", "pos.mean", "geo.mean" and "pen.mean". The other statistics cannot be superposed on the graph of the series in the current version of the function }
  \item{col}{ the colors to use to plot the initial series and the statistics, respectively. By default, \code{col=c(1,2)} }
  \item{lty}{ the style to use to draw the original series and the statistics. The default style is used if this argument is not provided }
  \item{leg}{ if \code{leg=TRUE}, a legend box is drawn on the graph }
  \item{llab}{ the labels to use for the legend. By default, it is "series" and the corresponding statistics provided in \code{stat}, respectively }
  \item{lpos}{ the position of the top-left corner (x,y) of the legend box in the graph coordinates. By default \code{lpos=c(1.5,10)} }
  \item{\dots}{ additional graph parameters }
}
\details{
  Available statistics are the same as for \code{stat.desc()} and \code{stat.pen()}. The Shapiro-Wilk test of normality is not available yet in Splus and it returns 'NA' in this environment. If not a priori known, successive blocs can be identified using either \code{local.trend()} or \code{decmedian()} (see respective functions for further details)
}
\value{
  An object of type 'stat.slide' is returned. It has methods \code{print()}, \code{plot()} and \code{lines()}.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{stat.desc}}, \code{\link{stat.pen}}, \code{\link{pennington}}, \code{\link{local.trend}}, \code{\link{decmedian}} }

\examples{
data(marbio)
# Sliding statistics with fixed-length blocs
statsl <- stat.slide(1:68, marbio[, "ClausocalanusA"], xmin=0, n=7, deltat=10)
statsl
plot(statsl, stat="mean", leg=TRUE, lpos=c(55, 2500), xlab="Station",
        ylab="ClausocalanusA")

# More information on the series, with predefined blocs
statsl2 <- stat.slide(1:68, marbio[, "ClausocalanusA"],
        xcut=c(0, 17, 25, 30, 41, 46, 70), basic=TRUE, desc=TRUE, norm=TRUE,
        pen=TRUE, p=0.95)
statsl2
plot(statsl2, stat="median", xlab="Stations", ylab="Counts",
        main="Clausocalanus A")              # Median
lines(statsl2, stat="min")                   # Minimum
lines(statsl2, stat="max")                   # Maximum
lines(c(17, 17), c(-50, 2600), col=4, lty=2) # Cuts
lines(c(25, 25), c(-50, 2600), col=4, lty=2)
lines(c(30, 30), c(-50, 2600), col=4, lty=2)
lines(c(41, 41), c(-50, 2600), col=4, lty=2)
lines(c(46, 46), c(-50, 2600), col=4, lty=2)
text(c(8.5, 21, 27.5, 35, 43.5, 57), 2300, labels=c("Peripheral Zone", "D1",
        "C", "Front", "D2", "Central Zone")) # Labels
legend(0, 1900, c("series", "median", "range"), col=1:3, lty=1)
# Get cuts back from the object
statsl2$xcut
}
\keyword{ ts }
\eof
\name{trend.test}
\alias{trend.test}
\title{ Test if an increasing or decreasing trend exists in a time series }
\description{
Test if the series has an increasing or decreasing trend, using a non-parametric Spearman test between the observations and time
}
\usage{
trend.test(tseries, R=1)
}
\arguments{
  \item{tseries}{ a univariate or multivariate time series (a 'rts' object in Splus or a 'ts' object in \R) }
  \item{R}{ The number of time the series is/are resampled for a bootstrap test. If \code{R1} (by default), an usual Spearman test is performed. If \code{R} > 1 then a bootstrap test is run }
}
\value{
A 'htest' object if \code{R=1}, a 'boot' object with an added \code{boot$p.value} item otherwise 
}
\references{ 
Siegel, S. & N.J. Castellan, 1988. \emph{Non-parametric statistics.} McGraw-Hill, New York. 399  pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\note{ 
In both cases (normal test with \code{R=1} and bootstrap test), the p-value can be obtained from \code{obj$p.value} (see examples)
}

\seealso{ \code{\link{local.trend}} }

\examples{
data(marbio)
trend.test(marbio[, 8])
# Run a bootstrap test on the same series
marbio8.trend.test <- trend.test(marbio[, 8], R=99)
# R=999 is a better value... but it is very slow!
marbio8.trend.test  
plot(marbio8.trend.test)
marbio8.trend.test$p.value
}
\keyword{ ts }
\keyword{ htest }
\eof
\name{tsd}
\alias{tsd}
\alias{extract.tsd}
\alias{plot.tsd}
\alias{print.specs.tsd}
\alias{print.summary.tsd}
\alias{print.tsd}
\alias{specs.tsd}
\alias{summary.tsd}
\title{ Decomposition of one or several regular time series using various methods }
\description{
  Use a decomposition method to split the series into two or more components. Decomposition methods are either series filtering/smoothing (difference, average, median, evf), deseasoning (loess) or model-based decomposition (reg, i.e., regression).
}
\synopsis{
tsd(x, specs=NULL, method="loess", type=if (method == "census") "multiplicative" else "additive", lag=1, axes=1:5, order=1, times=1, sides=2, ends="fill", weights=NULL, s.window=NULL, s.degree=0, t.window=NULL, t.degree=2, robust=FALSE, trend=FALSE, xreg=NULL) 
extract.tsd(e, n, series=NULL, components=NULL, ...)
plot.tsd(x, series=1, stack=TRUE, resid=TRUE, col=par("col"), lty=par("lty"), labels=dimnames(X)[[2]], leg=TRUE, lpos=c(0, 0), xlab="time", ylab="series", main=paste("Series decomposition by", x$specs$method, "-", x$specs$type), ...)
print.specs.tsd(x, ...)
print.summary.tsd(x, ...)
print.tsd(x, ...)
specs.tsd(x, ...)
summary.tsd(object, ...)
}
\usage{
tsd(x, specs=NULL, method="loess", type="additive", \dots)
\method{summary}{tsd}(tsdobj)
\method{plot}{tsd}(tsdobj, series=1, stack=TRUE, resid=TRUE,
        labels, leg=TRUE, lpos=c(0, 0), \dots)
\method{extract}{tsd}(tsdobj, n, series=NULL, components=NULL)
\method{specs}{tsd}(tsdobj)
}
\arguments{
  \item{x}{ an univariate or multivariate regular time series ('rts' in Splus or 'ts' in \R) to be decomposed }
  \item{specs}{ specifications are collected from a 'tsd' object, using the \code{specs} method. This allows for reusing parameters issued from a previous similar analysis }
  \item{method}{ the method to use to decompose the time series. Currently, possible values are: \code{"diff"}, \code{"average"}, \code{"median"}, \code{"evf"}, \code{"reg"}, \code{"loess"} (by default) or \code{"census"}. The corresponding function \code{decXXXX()} is applied to each of the series in \code{x} }
  \item{type}{ the type of model to use: either \code{"additive"} (by default) or \code{"multiplicative"}. In the additive model, all components must be added to reconstruct the initial series. In the multiplicative model, they must be multiplied (one components has the same unit as the original series, and the other ones are dimensionless multiplicative factors) }
  \item{\dots}{ (1) for \code{tsd()}: further arguments to pass to the corresponding \code{decXXXX()} function. (2) for \code{plot()}: further graphical arguments }
  \item{tsdobj}{ a 'tsd' object as returned by the function \code{tsd()}, or any of the \code{decXXXX()} functions }
  \item{series}{ (1) for \code{plot()}: the series to plot. By default, \code{series=1}, the first (or possibly unique) series in the 'tsd' object is plotted. (2) for \code{extract}: the name or the index of the series to extract. If \code{series} is provided, then \code{n} is ignored. By default, \code{series=NULL}. It is also possible to use negative indices. In this case, all series are extracted, except those ones }
  \item{stack}{ graphs of each component are either stacked (\code{stack=TRUE}, by default), or superposed on the same graph \code{stack=FALSE} }
  \item{resid}{ do we have to plot also the "residuals" components (\code{resid=TRUE}, by default) or not? Usually, in a stacked graph, you would like to plot the residuals, while in a superposed graph, you would not }
  \item{labels}{ the labels to use for all y-axes in a stacked graph, or in the legend for a superposed graph. By default, the names of the components ("trend", "seasonal", "deseasoned", "filtered", "residuals", ...) are used }
  \item{leg}{ only used when \code{stack=FALSE}. Do we plot a legend (\code{leg=TRUE} or not? }
  \item{lpos}{ position of the upper-left corner of the legend box in the graph coordinates (x,y). By default, \code{leg=c(0,0)} }
  \item{n}{ the number of series to extract (from series 1 to series n). By default, n equals the number of series in the 'tsd' object. If both \code{series} and \code{components} arguments are NULL, all series and components are extracted and this method has exactly the same effect as \code{tseries} }
  \item{components}{ the names or indices of the components to extract. If \code{components=NULL} (by default), then all components of the selected series are extracted. It is also possible to specify negative indices. In this case, all components are extracted, except those ones }
}
\details{
  To eliminate trend from a series, use "diff" or use "loess" with \code{trend=TRUE}. If you know the shape of the trend (linear, exponential, periodic, etc.), you can also use it with the "reg" (regression) method. To eliminate or extract seasonal components, you can use "loess" if the seasonal component is additive, or "census" if it is multiplicative. You can also use "average" with argument \code{order="periodic"} and with either an additive or a multiplicative model, although the later method is often less powerful than "loess" or "census". If you want to extract a seasonal cycle with a given shape (for instance, a sinusoid), use the "reg" method with a fitted sinusoidal equation. If you want to identify levels in the series, use the "median" method. To smooth the series, you can use preferably the "evf" (eigenvector filtering), or the "average" methods, but you can also use "median". To extract most important components from the series (no matter if they are cycles -seasonal or not-, or long-term trends), you should use the "evf" method. For more information on each of these methods, see online help of the corresponding \code{decXXXX()} functions.
}
\value{
  An object of type 'tsd' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{extract()} and \code{specs()}.
}
\references{ 
Kendall, M., 1976. \emph{Time-series.} Charles Griffin & Co Ltd. 197 pp.

Laloire, J.C., 1972. \emph{Mthodes du traitement des chroniques.} Dunod, Paris, 194 pp.

Legendre, L. & P. Legendre, 1984. \emph{Ecologie numrique. Tome 2: La structure des donnes cologiques.} Masson, Paris. 335 pp.

Malinvaud, E., 1978. \emph{Mthodes statistiques de l'conomtrie.} Dunod, Paris. 846 pp.

Philips, L. & R. Blomme, 1973. \emph{Analyse chronologique.} Universit Catholique de Louvain. Vander ed. 339 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }
\note{ If you have to decompose a single time series, you could also use the corresponding \code{decXXXX()} function directly. In the case of a multivariate regular time series, \code{tsd()} is more convenient because it decompose all times series of a set at once! }

\seealso{ \code{\link{tseries}}, \code{\link{decdiff}}, \code{\link{decaverage}}, \code{\link{decmedian}}, \code{\link{decevf}}, \code{\link{decreg}}, \code{\link{decloess}}, \code{\link{deccensus}} }

\examples{
data(releve)
# Regulate the series and extract them as a time series object
rel.regy <- regul(releve$Day, releve[3:8], xmin=6, n=87, units="daystoyears",
        frequency=24, tol=2.2, methods="linear", datemin="21/03/1989",
        dateformat="d/m/Y")
rel.ts <- tseries(rel.regy)

# Decompose all series in the set with the "loess" method
rel.dec <- tsd(rel.ts, method="loess", s.window=13, trend=FALSE)
rel.dec
plot(rel.dec, series=5, col=1:3)    # An plot series 5

# Extract "deseasoned" components
rel.des <- extract(rel.dec, series=3:6, components="deseasoned")
rel.des[1:10,]

# Further decompose these components with a moving average
rel.des.dec <- tsd(rel.des, method="average", order=2, times=10)
plot(rel.des.dec, series=3, col=c(2, 4, 6))
# In this case, a superposed graph is more appropriate:
plot(rel.des.dec, series=3, col=c(2,4), stack=FALSE, resid=FALSE,
        labels=c("without season cycle", "trend"), lpos=c(0, 55000))
# Extract residuals from the latter decomposition
rel.res2 <- extract(rel.des.dec, components="residuals")
}
\keyword{ ts }
\keyword{ smooth }
\keyword{ loess }
\keyword{ nonparametric }
\eof
\name{tseries}
\alias{tseries}
\title{ Convert a 'regul' or a 'tsd'  object into a time series }
\description{
  Regulated series contained in a 'regul' object or components issued from a time series decomposition with 'tsd' are extracted from their respective object and converted into uni- or multivariate regular time series ('rts' objects in Splus and 'ts' objects in \R)
}
\usage{
tseries(x)
}
\arguments{
  \item{x}{ A 'regul' or 'tsd' object }
}
\value{
  an uni- or multivariate regular time series
}
\author{ Philippe Grosjean (\email{phgrosjean@sciviews.org}), Frdric Ibanez (\email{ibanez@obs-vlfr.fr}) }

\note{
To extract some of the time series contained in the 'regul' or 'tsd' objects, use the \code{extract()} method
}

\seealso{ \code{\link{is.tseries}}, \code{\link{regul}}, \code{\link{tsd}} }

\examples{
data(releve)
rel.regy <- regul(releve$Day, releve[3:8], xmin=6, n=87, units="daystoyears",
        frequency=24, tol=2.2, methods="linear", datemin="21/03/1989",
        dateformat="d/m/Y")
# This object is not a time series
is.tseries(rel.regy)     # FALSE
# Extract all time series contained in the 'regul' object
rel.ts <- tseries(rel.regy)
# Now this is a time series
is.tseries(rel.ts)       # TRUE
}
\keyword{ ts }
\keyword{ manip }
\eof
\name{turnogram}
\alias{turnogram}
\alias{extract.turnogram}
\alias{identify.turnogram}
\alias{plot.turnogram}
\alias{print.summary.turnogram}
\alias{print.turnogram}
\alias{summary.turnogram}
\title{ Calculate and plot a turnogram for a regular time series }
\description{
The turnogram is the variation of a monotony index with the observation scale (the number of data per time unit). A monotony index indicates if the series has more or less erratic variations than a pure random succession of independent observations. Since a time series almost always has autocorrelation, it is expected to be more monotonous than a purely random series. The monotony index is a way to quantify the density of information beared by a time series. The turnogram determines at which observation scale this density of information is maximum. It is also the scale that optimize the sampling effort (best compromise between less samples versus more information).
}
\synopsis{
turnogram(series, intervals=c(1, length(series)/5), step=1, complete=FALSE, two.tailed=TRUE, FUN=mean, plotit=TRUE, level=0.05, lhorz=TRUE, lvert=FALSE, xlog=TRUE)
extract.turnogram(e, n, level=e$level, FUN=e$fun, drop=0, ...)
identify.turnogram(x, lvert=TRUE, col=2, lty=2, ...)
plot.turnogram(x, level=0.05, lhorz=TRUE, lvert=TRUE, lcol=2, llty=2, xlog=TRUE, xlab=paste("interval (", x$units.text, ")", sep=""), ylab="I (bits)", main=paste(x$type, "turnogram for:", x$data), sub=paste(x$fun, "/", x$proba), ...)
print.summary.turnogram(x, ...)
print.turnogram(x, ...)
summary.turnogram(object, ...)
}
\usage{
turnogram(series, intervals=c(1, length(series)/5), step=1, complete=FALSE,
        two.tailed=TRUE, FUN=mean, plotit=TRUE, level=0.05, lhorz=TRUE,
        lvert=FALSE, xlog=TRUE)
\method{summary}{turnogram}(turno)
\method{plot}{turnogram}(turno, level=0.05, lhorz=TRUE, lvert=TRUE, lcol=2,
        llty=2, xlog=TRUE, \dots)
\method{identify}{turnogram}(turno, lvert=TRUE, \dots)
\method{extract}{turnogram}(turno, n, level=turno$level, FUN=turno$fun, drop=0)
}
\arguments{
  \item{series}{ a single regular time series ('rts' object in Splus or 'ts' object in \R) }
  \item{intervals}{ the range (mini, maxi) of the intervals to calculate, i.e., to take one obervation every 'interval' one. By default, \code{intervals} ranges from 1 to the fifth of the total number of observations }
  \item{step}{ the increment used for the intervals. By defaults \code{step=1}. To limit calculation or for a first screenning with a large range in the intervals, use a higher value for \code{step} }
  \item{complete}{ if \code{complete=TRUE}, a complete turnogram is calculated, showing mean, minimal and maximal curves. If it is FALSE (by default), only a simple turnogram always starting from the first observation is calculated }
  \item{two.tailed}{ if \code{two.tailed=TRUE} (by default), the monotony index is tested with a bilateral test, otherwise, a left-sided test is used }
  \item{FUN}{ a function to apply to aggregate data in the intervals. It is a function of the type \code{FUN(x, na.rm, \dots)}. The most used function is \code{mean()} (by default), but it is also possible to keep only the first value with \code{first()}, the last value with \code{last()}, the median or the sum of values in the interval. The later function is useful for cumulative observations, like pluviometry. It should be noted that the turnograms with \code{FUN=mean} and with \code{FUN=sum} are the same, but that extraction of final series are different for levels > 1 }
  \item{plotit}{ if \code{plotit=TRUE} (by default), the graph of the turnogram is also plotted }
  \item{level}{ the significant level to draw on the graph. By default \code{level=0.05}, corresponding to a test with P = 5\% }
  \item{lhorz}{ if \code{lhorz=TRUE} (by default) then one (left-sided test), or two (two-sided test) horizontal lines are drawn on the graph, indicating the significant level of the test given by the argument \code{level}. Any point above the single line, or outside the interval defined by the two lines is significant }
  \item{lvert}{ if \code{lvert=TRUE} (by default, except for \code{turnogram()} function), a vertical line is drawn, indicating the time interval that corresponds to the maximum information and it is also the automatic level of extraction unless this value is changed }
  \item{lcol}{ the color to use to draw supplemental lines: the horizontal line indicating where the test is significant (if \code{lhorz=TRUE}) and the vertical line indicating the extraction level (if \code{lvert=TRUE}). By default, color 2 is used }
  \item{llty}{ the style for the supplemental lines. By default, style 2 is used (dashed lines) }
  \item{xlog}{ if \code{xlog=TRUE} (by default), then the x-axis is expressed in logarithms. Otherwise, a linear scale is used }
  \item{\dots}{ additional optional graphic arguments }
  \item{turno}{ a 'turnogram' object as returned by the function \code{turnogram()} }
  \item{n}{ the number of observations to take into account in the initial series. Use \code{n=NULL} (by default) to use all observations of the series }
  \item{drop}{ the number of observations to drop at the beginning of the series before proceeding with the aggregation of the data for the extracted series. By default, \code{drop=0}: no observations are dropped }
}
\details{
The turnogram is a generalisation of the information theory (see \code{turnpoints()}). If a series has a lot of erratic peaks and pits that alternate with a high frequency, it is more difficult to interpret than a more monotonous series. These erratic fluctuations can be eliminated by changing the scale of observation (keeping one observation every two, three, four,... from the original series). The turnogram resample the original series this way, and calculate a monotony index for each resampled subseries. This monotony index quantifies the number of peaks and pits presents in the series, compared to the total number of observations. The Gleissberg distribution (see \code{pgleissberg()}) indicates the probability to have such a number of extrema in a series given it is purely random. It is possible to test monotony indices: is it a random series or not (two-sided test), or is more monotonous than a random series (left-sided test) thanks to a Chi-2 test proposed by Wallis & Moore (1941). 

There are various turnograms depending on the way the observations are aggregated inside each time interval. For instance, if one consider one observation every three from the original series, each group of three observations can be aggregated in several different ways. One can take the mean of the three observation, or the median value, or the sum,... One can also decide not to aggregate observations, but to drop some of them. Hence, one can take only the first or the last observation of the group. All these options can be choosen by defining the argument \code{FUN=...}. A simple turnogram correspond to the change of the monotony index with the scale of observation, stating always from the first observation. One could also decide to start from the second, or the third observation for an aggregation of the observations three by three... and result could be somewhat different. A complete turnogram investigates all possibles combinations (observation scale versus starting point for the aggregation) and trace the maximal, minimal and mean curves for the change of the monotony index. It is thus more informative than the simple turnogram. However, it takes much more time to compute.

The most obvious use of the turnogram is for the pretreatment of continuously sampled data. It helps in deciding which is the optimal sampling interval for the series to bear as most information as possible while keeping the dataset as small as possible. It is also interesting to compare the turnogram with other functions like the variogram (see \code{vario()}) or the spectrogram (see \code{spectrum()}).
}
\value{
  An object of type 'turnogram' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{identify()} and \code{extract()}.
}
\references{ 
Dallot, S. & M. Etienne, 1990. \emph{Une mthode non paramtrique d'analyse des series en ocanographie biologique: les tournogrammes.} Biomtrie et ocanographie - Socit de biomtrie, 6, Lille, 26-28 mai 1986. IFREMER, Actes de colloques, 10:13-31.

Johnson, N.L. & Kotz, S., 1969. \emph{Discrete distributions.} J. Wiley & sons, New York, 328 pp.

Kendall, M.G., 1976. \emph{Time-series, 2nd ed.} Charles Griffin & co, London.

Wallis, W.A. & G.H. Moore, 1941. \emph{A significance test for time series.} National Bureau of Economic Research, tech. paper n1.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{pgleissberg}}, \code{\link{turnpoints}}, \code{\link{first}}, \code{\link{last}}, \code{\link{vario}}, \code{\link[ts]{spectrum}} }

\examples{
data(bnr)
# Let's transform series 4 into a time series (supposing it is regular)
bnr4 <- as.ts(bnr[, 4])
plot(bnr4, type="l", main="bnr4: raw data", xlab="Time")
# A simple turnogram is calculated
bnr4.turno <- turnogram(bnr4)
summary(bnr4.turno)
# A complete turnogram confirms that "level=3" is a good value: 
turnogram(bnr4, complete=TRUE)
# Data with maximum info. are extracted (thus taking 1 every 3 observations)
bnr4.interv3 <- extract(bnr4.turno)
plot(bnr4, type="l", lty=2, xlab="Time")
lines(bnr4.interv3, col=2)
title("Original bnr4 (dotted) versus max. info. curve (plain)")
# Choose another level (for instance, 6) and extract the corresponding series
bnr4.turno$level <- 6
bnr4.interv6 <- extract(bnr4.turno)
# plot both extracted series on top of the original one
plot(bnr4, type="l", lty=2, xlab="Time")
lines(bnr4.interv3, col=2)
lines(bnr4.interv6, col=3)
legend(70, 580, c("original", "interval=3", "interval=6"), col=1:3, lty=c(2, 1, 1))
# It is hard to tell on the graph which series contains more information
# The turnogram shows us that it is the "interval=3" one!
}
\keyword{ ts }
\keyword{ htest }
\eof
\name{turnpoints}
\alias{turnpoints}
\alias{extract.turnpoints}
\alias{lines.turnpoints}
\alias{plot.turnpoints}
\alias{print.summary.turnpoints}
\alias{print.turnpoints}
\alias{summary.turnpoints}
\title{ Analyze turning points (peaks or pits) }
\description{
  Determine the number and the position of extrema (turning points, either peaks or pits) in a regular time series. Calculate the quantity of information associated to the observations in this series, according to Kendall's information theory
}
\synopsis{
turnpoints(x)
extract.turnpoints(e, n, no.tp=0, peak=1, pit=-1, ...)
lines.turnpoints(x, max=TRUE, min=TRUE, median=TRUE, col=c(4, 4, 2), lty=c(2, 2, 1), ...)
plot.turnpoints(x, level=0.05, lhorz=TRUE, lcol=2, llty=2, type="l", xlab="data number", ylab=paste("I (bits), level = ", level * 100, "\%", sep = ""), main=paste("Information (turning points) for:", x$data), ...)
print.summary.turnpoints(x, ...)
print.turnpoints(x, ...)
summary.turnpoints(object, ...)
}
\usage{
turnpoints(x)
\method{summary}{turnpoints}(turnp)
\method{plot}{turnpoints}(turnp, level=0.05, lhorz=TRUE, lcol=2, llty=2, \dots)
\method{lines}{turnpoints}(turnp, max=TRUE, min=TRUE, median=TRUE,
        col=c(4, 4, 2), lty=c(2, 2, 1), \dots)
\method{extract}{turnpoints}(turnp, n, no.tp=0, peak=1, pit=-1)
}
\arguments{
  \item{x}{ a vector or a time series }
  \item{turnp}{ a 'turnpoints' object, as returned by the function \code{turnpoints()} }
  \item{level}{ the significant level to draw on the graph if \code{lhorz=TRUE}. By default, \code{level=0.05}, which corresponds to a 5\% p-value for the test }
  \item{lhorz}{ if \code{lhorz=TRUE} (by default), an horizontal line indicating significant level is drawn on the graph }
  \item{lcol}{ the color to use to draw the significant level line, by default, color 2 is used }
  \item{llty}{ the style to use for the significant level line. By default, style 2 is used (dashed line) }
  \item{\dots}{ Additional graph parameters }
  \item{max}{ do we plot the maximum envelope line (by default, yes) }
  \item{min}{ do we plot the minimum envelope line (by default, yes) }
  \item{median}{ do we plot the median line inside the envelope (by default, yes) }
  \item{col}{ a vector of three values for the color of the max, min, median lines, respectively. By default \code{col=c(4,4,2)} }
  \item{lty}{ a vector of three values for the style of the max, min, median lines, respectively. By default \code{lty=c(2,2,1)}, that is: dashed, dashed and plain lines }
  \item{n}{ the number of points to extract. By default \code{n=length(turnp)}, all points are extracted }
  \item{no.tp}{ extract gives a vector representing the position of extrema in the original series. \code{no.tp} represents the code to use for points that are not an extremum, by default '0' }
  \item{peak}{ the code to use to flag a peak, by default '1' }
  \item{pit}{ the code to use to flag a pit, by default '-1' }
}
\details{
  This function tests if the time series is purely random or not. Kendall (1976) proposed a series of tests for this. Moreover, graphical methods using the position of the turning points to draw automatically envelopes around the data are implemented, and also the drawing of median points between these envelopes
}
\value{
  An object of type 'turnpoints' is returned. It has methods \code{print()}, \code{summary()}, \code{plot()}, \code{lines()} and \code{extract()}.
}
\references{ 
Kendall, M.G., 1976. \emph{Time-series, 2nd ed.} Charles Griffin & Co, London.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\section{WARNING }{ the \code{lines()} method should be used to draw lines on the graph of the \emph{original} dataset (\code{plot(data, type="l")} for instance), \emph{not} on the graph of turning points (\code{plot(turnp)})! }

\seealso{ \code{\link{turnogram}}, \code{\link{stat.slide}} }

\examples{
data(marbio)
plot(marbio[, "Nauplii"], type="l")
# Calculate turning points for this series
Nauplii.tp <- turnpoints(marbio[, "Nauplii"])
summary(Nauplii.tp)
plot(Nauplii.tp)
# Add envelope and median line to original data
plot(marbio[, "Nauplii"], type="l")
lines(Nauplii.tp)
# Note that lines() applies to the graph of original dataset!!!
title("Raw data, envelope maxi., mini. and median line")
}
\keyword{ ts }
\eof
\name{vario}
\alias{vario}
\title{ Compute and plot a semi-variogram }
\description{
  Compute a classical semi-variogram for a single regular time series
}
\usage{
vario(x, max.dist=length(x)/3, plotit=TRUE, vario.data=NULL)
}
\arguments{
  \item{x}{ a vector or an univariate time series }
  \item{max.dist}{ the maximum distance to calculate. By default, it is the third of the number of observations }
  \item{plotit}{ If \code{plotit=TRUE} then the graph of the semi-variogram is plotted }
  \item{vario.data}{ data coming from a previous call to \code{vario()}. Call the function again with these data to plot the corresponding graph }
}
\value{
  A data frame containing distance and semi-variogram values
}
\references{ 
David, M., 1977. \emph{Developments in geomathematics. Tome 2: Geostatistical or reserve estimation.} Elsevier Scientific, Amsterdam. 364 pp.

Delhomme, J.P., 1978. \emph{Applications de la thorie des variables rgionalises dans les sciences de l'eau.} Bull. BRGM, section 3 n4:341-375.

Matheron, G., 1971. \emph{La thorie des variables rgionalises et ses applications.} Cahiers du Centre de Morphologie Mathmatique de Fontainebleau. Fasc. 5 ENSMP, Paris. 212 pp.
}
\author{ Frdric Ibanez (\email{ibanez@obs-vlfr.fr}), Philippe Grosjean (\email{phgrosjean@sciviews.org}) }

\seealso{ \code{\link{disto}} }

\examples{
data(bnr)
vario(bnr[, 4])
}
\keyword{ ts }
\eof
