\name{A1}
\title{Ratio of First and Zeroth Order Bessel Functions}
\alias{A1}

\description{Evaluates the first and zeroth order Bessel functions of the first kind at a specified non-negative real number, and returns the ratio.} 

\usage{
A1(kappa)
}

\arguments{
\item{kappa}{non-negative numeric value at which to evaluate the Bessel functions.}
}

\value{
If I1(kappa) is the first order Bessel function and I0(kappa) is the zeroth order Bessel function, then A1(kappa) returns I1(kappa)/I0(kappa).
}

\details{
The function use \code{\link{besselI}}.
}

\seealso{
 \code{\link{besselI}}, \code{\link{A1inv}}.
}


\keyword{math}


\eof
\name{A1inv}
\title{Inverse of A1}
\alias{A1inv}

\description{
Inverse function of the ratio of the first and zeroth order Bessel functions of the first kind.  This function is used to compute the maximum likelihood estimate of the concentration parameter of a von Mises distribution.
}

\usage{
A1inv(x)
}

\arguments{
\item{x}{numeric value in the interval between 0 and 1.}
}

\value{
Returns the value k, such that A1inv(x) = k, i.e. A1(k) = x.
}

\details{
A1inv(0) = 0 and A1inv(1) = inf.  This function is useful in estimating the concentration parameter of data from a von Mises distribution.
}

\seealso{
\code{\link{est.kappa}}, \code{\link{A1}}, \code{\link{besselI}}.
}

\examples{
#Generate data from a von Mises distribution
data <- rvm(50, pi, 4)
#Estimate the concentration parameter
s <- sum(sin(data))
c <- sum(cos(data))
mean.dir <- atan(s, c)
kappa <- A1inv(mean(cos(data - mean.dir)))
}

\keyword{math}


\eof
\name{I.0}
\title{Zeroth Order Bessel Function of the First Kind}
\alias{I.0}

\description{
An alias of besselI(x, nu=0, expon.scaled = FALSE) used for compatibility with old version package code.
}

\usage{
I.0(x)
}

\arguments{
\item{x}{non-negative numerical value at which to evaluate the Bessel function.}
}

\value{
Returns the zeroth order Bessel function of the first kind evaluated at a specified real number.
}

\details{
The use of this function is deprecated. Please use \code{\link{besselI}}.
}

\seealso{
\code{\link{besselI}}.
}

\keyword{math}


\eof
\name{I.1}
\title{First Order Bessel Function of the First Kind}
\alias{I.1}

\description{
An alias of besselI(x, nu=1, expon.scaled = FALSE) used for compatibility with old version package code.
}

\usage{
I.1(x)
}

\arguments{
\item{x}{non-negative numerical value at which to evaluate the Bessel function.}
}

\value{
Returns the first order Bessel function of the first kind, evaluated at a specified real number.
}

\details{
The use of this function is deprecated. Please use \code{\link{besselI}}.
}

\seealso{
\code{\link{besselI}}. 
}

\keyword{math}


\eof
\name{I.p}
\title{P-th Order Bessel Function of the First Kind}
\alias{I.p}

\description{
An alias of besselI(x, nu=p, expon.scaled = FALSE) used for compatibility with old version package code.
}

\usage{
I.p(p, x)
}

\arguments{
\item{p}{positive integer order of the Bessel function.}
\item{x}{non-negative numerical value at which to evaluate the Bessel function.}
}

\value{
Returns the p-th order Bessel function of the first kind, evaluated at a specified real number.
}

\details{
The use of this function is deprecated. Please use \code{\link{besselI}}.
}


\seealso{
\code{\link{besselI}}.
}

\keyword{math}


\eof
\name{change.pt}
\title{Change Point Test}
\alias{change.pt}

\description{
Tests for a change in mean direction, concentration, or both, given a set of directional data points. 
}

\usage{
change.pt(x)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
}

\value{
Returns a data frame with variables n, rho=r/n, rmax, k.r, rave, tmax, k.t, and tave. The first of these is the sample size, followed by the overall mean resultant length. Both of these are needed to enter any of the tables or nomograms(see under Details). The other values represent the change point test statistics.  While rmax and rave test for a change in mean direction (with unknown concentration), tmax and tave are useful in the context of testing more generally, for a change in mean direction and/or concentration. k.r and k.t are the observation numbers  for which rmax and tmax attain their maximum value and indicate the observation at which the change is most likely to have occurred, when the tables or nomograms indicate significance. 
}

\details{
In either context, the user can choose which statistic (max or ave) to use, and then consult the appropriate table provided in the book referenced below. The critical values for these 4 statistics are to be found in Table 11.3 (or Figure 11.3) for rmax, Table 11.4 (or Figure 11.4) for rave, Figure 11.5 for tmax and Figure 11.6 for tave.
}

\seealso{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Chapter 11, World Scientific Press, Singapore.  
}

\keyword{htest}


\eof
\name{circ.cor}
\title{Correlation Coefficient for Angular Variables}
\alias{circ.cor}

\description{
Computes a circular version of the Pearson's product moment correlation, and performs a significance test if requested. 
}

\usage{
circ.cor(alpha, beta, test=FALSE)
}

\arguments{
\item{alpha}{vector of circular data measured in radians.}
\item{beta}{vector of circular data measured in radians.}
\item{test}{if test = TRUE, then a significance test for the correlation coefficient is computed.}
}

\value{
Returns a data frame with variables r, a circular version of the Pearson's product moment correlation, test.stat and p.value, the test statistic and p-value respectively, for testing significance of the correlation coefficient. test.stat and p.value are by default not produced, but are given when test=TRUE is specified in the function call.
}

\details{
The correlation coefficient is computed like Pearson's product moment correlation for two linear variables X and Y.  In the computational formula,  however, (xi - xbar) and (yi - ybar) are replaced by sin(xi - xbar) and sin(yi - ybar), where xbar and ybar in the second two expressions are the mean directions of the samples.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 8.2, World Scientific Press, Singapore.

Jammalamadaka, S. and Sarma, Y. (1988).  A correlation coefficient for angular variables.  Statistical Theory and Data Analysis 2. North Holland:  New York.
}

\examples{
# Generate two circular data sets, and compute their correlation.
data1 <- rvm(50, 0, 3)
data2 <- data1 + pi + rvm(50, 0, 10)
circ.cor(data1, data2, test=TRUE)
}

\keyword{univar}


\eof
\name{circ.disp}
\title{Circular Dispersion}
\alias{circ.disp}

\description{
Computes measures of dispersion for a directional data set. 
}

\usage{
circ.disp(x)
}

\arguments{
\item{x}{vector of circular data measured in radians.}
}

\values{
Returns a dataframe with the following components. The sample size, n; the resultant length, r; the mean resultant length, rbar= r/n; and the circular variance, var=(1-r/n).
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 1.3, World Scientific Press, Singapore.
}

\seealso{
\link{circ.mean}, \link{circ.summary}, \link{est.kappa}, \link{est.rho}.
}

\keyword{univar}


\eof
\name{circ.mean}
\title{Mean Direction}
\alias{circ.mean}

\description{
Returns the mean direction of a vector of circular data. 
}

\usage{
circ.mean(x)
}

\arguments{
\item{x}{vector of data points measured in radians.}
}

\value{
Returns the mean direction of the data.
}

\details{
Each observation is treated as a unit vector, or point  on the unit circle.  The resultant vector of the observations is found, and the direction of the resultant vector is returned.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 1.3, World Scientific Press, Singapore.
}

\seealso{
\link{circ.disp}, \link{circ.summary}, \link{est.kappa}, \link{est.rho}.
}

\examples{
# Compute the mean direction of a random sample of observations.
data <- runif(50, 0, pi)
mean.dir <- circ.mean(data)
}

\keyword{univar}


\eof
\name{circ.plot}
\title{Circular Data Plot}
\alias{circ.plot}

\description{
Creates a plot of circular data points on the current graphics device.  Data points are either plotted as points on the unit circle, or the range of the circle is divided into a specified number of bins, and points are stacked in the bins corresponding to the number of observations in each bin. 
}

\usage{
circ.plot(x, main="", pch=16, stack=FALSE, bins=0, cex=1, dotsep=40, shrink=1)
}

\arguments{
\item{x}{vector of observations to be plotted, measured in radians.}
\item{main}{title of plot.}
\item{pch}{point character to use.  See help on par.}
\item{stack}{logical flag: if TRUE, points are stacked on the perimeter of the circle.  Otherwise, all points are plotted on the perimeter of the circle.  Default is FALSE.}
\item{bins}{if stack = TRUE, bins is the number of arcs to partition the circle with.}
\item{cex}{point character size.  See help on par.}
\item{dotsep}{constant used to specify the distance between stacked points, if stack = TRUE.  Default is 40; larger values will create smaller spaces, while smaller values create larger spaces.  This option can be useful when pch is other than 1, or when shrink is other than 1.}
\item{shrink}{parameter that controls the size of the plotted circle.  Default is 1.  Larger values shrink the circle, while smaller values enlarge the circle.  This option is useful when stack = TRUE.}
}

\details{
When there are many closely distributed observations, stacking is recommended.  Otherwise, much information can be lost due to overplotting of points.  When stacking the points, if there are many points in a particular bin, it may be necessary to shrink the plot of the circle so that all points fit.  This is controlled with the parameter shrink.  Generally the parameter dotsep does not need adjustment, however, when shrinking the plot, or for a very large number of observations, it may be helpful.
}

\note{this function use function \code{\link{eqscplot}} from package MASS.}

\seealso{
\link{rose.diag}.
}

\examples{
# Generate 100 observations from a von Mises distribution.
# with mean direction 0 and concentration 3.
data.vm <- rvm(100, 0, 3) 
# Plot data set. All points do not fit on plot.
circ.plot(data.vm, stack=TRUE, bins=150) 
# Shrink the plot so that all points fit.
circ.plot(data.vm, stack=TRUE, bins=150, shrink=1.5) 
}

\keyword{hplot}


\eof
\name{circ.range}
\title{Circular Range}
\alias{circ.range}
\alias{nCk}

\description{
Computes the circular range of a data set and performs a test of uniformity if specified.
}

\usage{
circ.range(x, test=FALSE)
}

\arguments{
\item{x}{vector of directional data measured in radians.}
\item{test}{logical flag:  if TRUE then the test of uniformity is performed; otherwise the test is not performed.  Default is FALSE.}
}

\value{
Returns a dataframe with the circular range, r.  If the significance test is requested the p-value of the test is returned as p.value.
}

\details{
The circular range is the shortest arc on the circle containing the entire set of data.  The p-value is computed using the exact distribution of the circular range under the hypothesis of uniformity.
}

\note{nCk(n,k) evaluate combinatoric, it should not be called by the users.}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.4, World Scientific Press, Singapore.
}

\seealso{
\link{kuiper}, \link{rao.spacing}, \link{r.test}, \link{v0.test}, \link{watson}.
}

\examples{
data <- rvm(50, 0, 2)
circ.range(data, test=TRUE)
data <- runif(50, 0, 2*pi)
circ.range(data, test=TRUE)
}

\keyword{univar}
\keyword{htest}



\eof
\name{circ.reg}    
\title{Circular-Circular Regression}
\alias{circ.reg}

\description{
Fits a regression model for a circular dependent and circular independent variable.  
}

\usage{
circ.reg(alpha, theta, order=1, level=0.05)
}

\arguments{
\item{alpha}{vector of data for the independent circular variable, measured in radians.}
\item{theta}{vector of data for the dependent circular variable, measured in radians.}
\item{order}{order of trigonometric polynomial to be fit.  order must be an integer value.  By default, order=1.}
\item{level}{level of the test for the significance of higher order trigonometric terms.}
}

\value{
A data frame is returned with the following components. 

\item{rho}{square root of the average of the squares of the estimated conditional concentration parameters of theta given alpha.}
\item{fitted}{fitted values of the model.}
\item{data}{matrix whose columns correspond to alpha and theta.}
\item{residuals}{circular residuals of the model.}
\item{coef}{matrix whose entries are the estimated coefficients of the model. The first column corresponds to the coefficients of the model predicting the cosine of theta, while the second column contains the estimates for the model predicting the sine of theta.  The rows of the matrix correspond to the coefficients according to increasing trigonometric order.}
\item{pvalues}{p-values testing whether the (order + 1) trigonometric terms are significantly different from zero.}  
\item{A.k}{is mean of the cosines of the circular residuals.}
\item{kappa}{assuming the circular residuals come from a von Mises distribution, kappa is the MLE of the concentration parameter.}
}

\details{
A trigonometric polynomial of alpha is fit against the cosine and sine of theta.  The order of trigonometric polynomial is specified by order.  Fitted values of theta are obtained by taking the inverse tangent of the predicted values of sin(theta) devided by the predicted values of cos(theta).  Details of the regression model can be found in Sarma and Jammalamadaka (1993).
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 8.3, World Scientific Press, Singapore.

Sarma, Y. and Jammalamadaka, S. (1993).  Circular Regression.  Statistical Science and Data Analysis, 109128.  Proceeding of the Thrid Pacific Area Statistical Conference.  VSP: Utrecht, Netherlands.
}

\examples{
# Generate a data set of dependent circular variables.
data1 <- runif(50, 0, 2*pi)
data2 <- atan(0.15*cos(data1) + 0.25*sin(data1), 0.35*sin(data1)) + rvm(50, 0, 5)

# Fit a circular regression model.
circ.lm <- circ.reg(data1, data2, order=1)
# Obtain a crude plot a data and fitted regression line.
plot(data1, data2)
circ.lm$fitted[circ.lm$fitted>pi] <- circ.lm$fitted[circ.lm$fitted>pi] - 2*pi 

points(data1[order(data1)], circ.lm$fitted[order(data1)], type='l')
}

\keyword{models}


\eof
\name{circ.summary}
\title{Circular Summary Statistics}
\alias{circ.summary}

\description{
Computes circular summary statistics including the sample size, mean direction and mean resultant length.
}

\usage{
circ.summary(x)
}

\arguments{
\item{x}{vector of data points measured in radians.}
}

\value{
Returns a data frame with variables n, the sample size; mean.dir, the sample mean direction; and rho, the sample mean resultant length.
}

\details{
Each observation is  treated  as  a  unit  vector or a point on the unit circle.  The resultant vector of the observations is found, and the direction of the resultant vector is returned as well as its length divided by the sample size.  
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 1.3, World Scientific Press, Singapore.
}

\seealso{
\link{circ.mean}, \link{circ.disp}, \link{est.kappa}, \link{est.rho}.
}

\examples{
# Compute summary statistics of a random sample of observations. 
data <- runif(50, 0, pi)
circ.summary(data)
}

\keyword{misc}


\eof
\name{dcard}   
\title{Cardioid Density Function}
\alias{dcard}

\description{
Returns the cardoid density function evaluated at a particular value. 
}

\usage{
dcard(theta, mu, r)
}

\arguments{
\item{theta}{vector of angles measured in radians at which to evaluate the density function.}
\item{mu}{mean direction of the distribution.}
\item{r}{concentration parameter of the distribution.  Absolute value of r must be less than 0.5.}
}

\value{
Returns the cardoid density function evaluated at theta.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.2, World Scientific Press, Singapore.
}

\seealso{
\link{rcard}
}

\keyword{distribution}


\eof
\name{deg}    
\title{Degrees}
\alias{deg}

\description{
Converts radians to degrees. 
}

\usage{
deg(radian)
}

\arguments{
\item{radian}{vector of radian measurements.}
}

\value{
Returns a vector of degree measurements corresponding to the data in vector radian.
}

\seealso{
\link{rad}
}

\keyword{math}

\eof
\name{dmixedvm}    
\title{Mixture of von Mises Distributions}
\alias{dmixedvm}

\description{
Evaluates the density function of a mixture of two von Mises distributions. 
}

\usage{
dmixedvm(theta, mu1, mu2, kappa1, kappa2, p)
}

\arguments{
\item{theta}{vector of values at which to evaluate the density function.}
\item{mu1}{mean direction in radians of one of the two von Mises distributions.}
\item{mu2}{mean direction in radians of the other von Mises distribution.}
\item{kappa1}{concentration parameter of one of the two von Mises distributions.}
\item{kappa2}{concentration parameter of the other von Mises distribution.}
\item{p}{mixing proportion.}
}

\value{
Evaluates the density function of a mixture of two von Mises distributions at a given vector of values, theta.
}

\details{
Evaluates the density function \eqn{p*VM(mu1, kappa1) + (1-p)*VM(mu2, kappa2)}, where  VM is the von Mises density function.
}

\seealso{
\link{rmixedvm}
}

\keyword{distribution}


\eof
\name{dtri}    
\title{Triangular Density Function}
\alias{dtri}

\description{
Returns the triangular density function evaluated at a particular value. 
}

\usage{
dtri(theta, r)
}

\arguments{
\item{theta}{vector of angles measured in radians at which to evaluate the density function.}
\item{r}{concentration parameter of the distribution.  r must be between 0 and \eqn{4/pi^2}.}
}

\value{
Returns the triangular density function evaluated at theta.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.3, World Scientific Press, Singapore.
}

\seealso{
\link{rtri}
}

\keyword{distribution}


\eof
\name{dvm}
\title{von Mises Density Function}
\alias{dvm}

\description{
Returns the von Mises density function evaluated at a particular value. 
}

\usage{
dvm(theta, mu, kappa)
}

\arguments{
\item{theta}{vector of angles measured in radians at which to evaluate the density	function.}
\item{mu}{mean direction of the distribution measured in radians.}
\item{kappa}{non-negative numeric value for the concentration parameter of the distribution.}
}

\value{
Returns the von Mises density function evaluated at theta.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.4, World Scientific Press, Singapore.
}

\seealso{
\link{rvm}
}

\keyword{distribution}


\eof
\name{dwrpcauchy}
\title{Wrapped Cauchy Density Function}
\alias{dwrpcauchy}

\description{
Returns the wrapped Cauchy density function evaluated at a particular value. 
}

\usage{
dwrpcauchy(theta, mu, rho)
}

\arguments{
\item{theta}{vector of angles measured in radians at which to evaluate the density	function.}
\item{mu}{mean direction of the distribution measured in radians.}
\item{rho}{concentration parameter of the distribution.  rho must be in the interval from 0 to 1.}
}

\value{
Returns the wrapped Cauchy density function evaluated at theta.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.7, World Scientific Press, Singapore.
}

\seealso{
\link{rwrpcauchy}
}

\keyword{distribution}


\eof
\name{dwrpnorm}    
\title{Wrapped Normal Density Function}
\alias{dwrpnorm}

\description{
Estimate of the wrapped normal density function.
}

\usage{
dwrpnorm(theta, mu, rho, sd=1, acc=1e-5, tol=acc)
}

\arguments{
\item{theta}{value at which to evaluate the density function, measured in radians.}
\item{mu}{mean direction of distribution, measured in radians.}
\item{rho}{mean resultant length of distribution.}
\item{sd}{different way of select \code{rho}, see details below.}
\item{acc}{parameter defining the accuracy of the estimation of the
  density.  Terms are added to the infinite summation that defines the
  density function until successive estimates are within \code{acc} of each
  other.}
\item{tol}{the same as \code{acc}.}
}

\value{
Returns an estimate of the wrapped normal density function.
}

\details{
The form of the wrapped normal density function is an infinite series
with index going from negative infinity to positive infinity.  This
function begins with the zeroth term and adds terms to the series,
corresponding to both the positive and negative index, until the
summation changes by less than the parameter value of \code{acc}. You
can set \code{rho} by using \code{sd} with the following equivalence:
\deqn{
\rho = \exp{- \sigma^2/2}
}
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.6, World Scientific Press, Singapore.
}

\seealso{
\link{rwrpnorm}
}

\examples{
# Values for which to evaluate density
theta <- c(1:500)*2*pi/500
#Compute wrapped normal density function
density <- c(1:500)
for(i in 1:500) density[i] <- dwrpnorm(theta[i], pi, .75)
plot(theta, density)
#Approximate area under density curve
sum(density*2*pi/500)
}

\keyword{distribution}


\eof
\name{est.kappa}
\title{Estimate Kappa}
\alias{est.kappa}

\description{
Computes the maximum likelihood estimate of kappa, the concentration parameter of a von Mises distribution, given a set of angular measurements. 
}

\usage{
est.kappa(x, bias=FALSE)
}

\arguments{
\item{x}{vector of circular data measured in radians.}
\item{bias}{logical parameter determining whether a bias correction is used in the computation of the MLE.  Default for bias is FALSE - the bias correction is not used.}
}

\value{
Maximum likelihood estimate of kappa, the concentration parameter of a von Mises distribution, given a set of angular measurements.
}

\details{
Best and Fisher (1981) show that the MLE of kappa is seriously biased when both sample size and mean resultant length are small.  They suggest a bias-corrected estimate for kappa when n < 16.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 4.2.1, World Scientific Press, Singapore.

Best, D. and Fisher N. (1981).  The bias of the maximum likelihood estimators of the von Mises-Fisher concentration parameters.  Communications in Statistics - Simulation and Computation, B10(5), 493502.
}

\seealso{
\link{circ.mean}, \link{circ.disp}, \link{circ.summary}, \link{est.rho}
}

\examples{
data <- rvm(15, 0, 3)
est.kappa(data)
est.kappa(data, bias=TRUE)
}

\keyword{htest}


\eof
\name{est.rho}
\title{Mean Resultant Length}
\alias{est.rho}

\description{
Returns the mean resultant length of a vector of circular data. 
}

\usage{
est.rho(x)
}

\arguments{
\item{x}{vector of data points measured in radians.}
}


\value{
Returns the mean resultant length of data.
}

\details{
Each observation is treated as a unit vector, or point  on the unit circle.  The resultant vector of the observations is found, and the length of the resultant vector divided by the sample size is returned.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 1.3, World Scientific Press, Singapore.
}

\seealso{
\link{circ.mean}, \link{circ.disp}, \link{circ.summary}, \link{est.kappa}.
}

\examples{
# Compute the mean resultant length of a random sample of observations.
data <- runif(100, 0, 2*pi)
est.rho(data)
}

\keyword{htest}



\eof
\name{kuiper}
\title{Kuiper's Test}
\alias{kuiper}

\description{
Performs Kuiper's one-sample test of uniformity on the circle. 
}

\usage{
kuiper(x, alpha=0)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{alpha}{significance level of the test.  Possible levels are 0.15, 0.1, 0.05, 0.025, 0.01.  Alpha may be omitted or set to zero, in which case a range for the p-value of the test will be returned.}
}

\value{
NULL
}

\note{
Kuiper's one-sample test of uniformity is performed, and the results are printed to the screen.  If alpha is specified and non-zero, the test statistic is printed along with the critical value and decision.  If alpha is omitted, the test statistic is printed and a range for the p-value of the test is given.
}

\details{
Kuiper's test statistic is a rotation-invariant Kolmogorov-type test statistic.  The critical values of a modified Kuiper's test statistic are used	according to the tabulation given in Stephens (1970). 
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.2, World Scientific Press, Singapore.

Stephens, M. (1970).  Use of the Kolmogorov-Smirnov, Cramer-von Mises and related statistics without extensive tables.  Journal of the Royal Statistical Society, B32, 115122.
}

\seealso{
\link{circ.range}, \link{rao.spacing}, \link{r.test}, \link{v0.test}, \link{watson}
}

\exammples{
# Generate data from the uniform distribution on the circle.
data <- runif(100, 0, 2*pi)
kuiper(data)
# Generate data from the von Mises distribution.
data <- rvm(100, 0, 3)
kuiper(data, alpha=0.01)
}

\keyword{htest}


\eof
\name{plot.edf}
\title{Plot Empirical Distribution Function}
\alias{plot.edf}

\description{
Plots the empirical distribution function of a data set. 
}

\usage{
\method{plot}{edf}(x, ...)
}

\arguments{
\item{x}{vector of circular data measured in radians.}
\item{...}{optional graphical parameters.  See help section on \link{par}.}
}

\note{
Creates a plot of the empirical distribution function of the vector data.
}

\details{
The vector data is taken modulo 2*pi, and then the linear ranks are used to generate an empirical distribution function.
}

\seealso{
\link{par}
}

\examples{
# Compare the edf's of two simulated sets of data.
data1 <- rvm(10, 0, 3)
data2 <- rvm(10, 0, 1)
plot.edf(data1, xlab="Data", ylab="EDF", main="Plots of Two EDF's")
par(new=TRUE)
plot.edf(data2, axes=FALSE, xlab="", ylab="", lty=2)
}

\keyword{hplot}


\eof
\name{pp.plot}    
\title{von Mises Probability-Probability Plot}
\alias{pp.plot}

\description{
Plots the empirical distribution of a data set against the best fitting von Mises distribution function.
}

\usage{
pp.plot(x, ref.line=TRUE)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{ref.line}{logical flag: if TRUE a 45 degree reference line is added to the plot. Default is TRUE.}
}

\value{
NULL
}

\note{
A plot is created on the current graphics device.
}

\details{
The maximum likelihood estimates of the parameters of the von Mises distribution are computed from the given data set.  The empirical distribution function is plotted against a von Mises distribution function with parameters given by the MLEs computed. 
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 10.2, World Scientific Press, Singapore.
}

\seealso{
\link{vm.ml}
}

\examples{
data <- rvm(25, 0, 3)
pp.plot(data)
data <- c(rvm(20, 0, 7), rvm(20, pi, 7))
pp.plot(data)
}

\keyword{hplot}


\eof
\name{pvm}
\title{Cummulative Probability for the von Mises Distribution}
\alias{pvm}

\description{
Estimates the cummulative probability for a von Mises distribution.} 

\usage{
pvm(theta, mu, kappa, acc=1e-020)
}

\arguments{
\item{theta}{angular value in radians.}
\item{mu}{mean direction of the von Mises distribution.}
\item{kappa}{concentration parameter of the von Mises distribution.}
\item{acc}{parameter relating to the accuracy of the estimated cummulative probability.  See details below.  Default value is 1e-020.}
}

\value{
Returns the probability that a von Mises random variable falls between 0 and theta.
}

\details{
Cummulative probabilities are computed according to the expression for the von Mises cdf given by Gumbel et al. (1953), which gives the cdf as a function of an infinite sum.  The parameter acc specifies the accuracy with which this sum is approximated. Terms greater than acc are included in the summation.  The cummulative probabilities given by pvm coincide with those tabulated	by Mardia (1972), which are given to five decimal values.
}

\references{
Gumbel, E., Greenwood, J. and Durand, D. (1953).  The circular normal distribution: theory and tables.  Journal of the American Statistical Association, 48, 131152.

Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.4, World Scientific Press, Singapore.
}

\seealso{
\link{dvm}, \link{rvm}
}

\keyword{distribution}


\eof
\name{r.test}
\title{Rayleigh Test of Uniformity:  General Unimodal Alternative}
\alias{r.test}

\description{
Performs a Rayleigh test of uniformity, assessing the significance of the mean resultant length.  The alternative hypothesis is a unimodal distribution with unknown mean direction and unknown mean resultant length.  
}

\usage{
r.test(x, degree=FALSE)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{degree}{logical flag:  if TRUE, data is assumed to be measured in degrees rather than radians. Default is FALSE.}
}

\value{
Returns a list with two components:  the mean resultant length, r.bar, and the p-value of the test statistic, p.value.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Sections 3.3.2 and 3.4.1, World Scientific Press, Singapore.
}

\seealso{
\link{circ.range}, \link{kuiper}, \link{rao.spacing}, \link{v0.test}, \link{watson}
}


\examples{
data <- rvm(25, pi, 2)
r.test(data)
}

\keyword{htest}


\eof
\name{rad}    
\title{Radians}
\alias{rad}

\description{
Converts degrees to radians. 
}

\usage{
rad(degree)
}

\arguments{
\item{degree}{vector of degree measurements.}
}

\value{
Returns a vector of radian measurements corresponding to the data in vector degree.
}

\seealso{
\link{deg}
}

\keyword{math}



\eof
\name{rao.homogeneity}
\title{Rao's Tests for Homogeneity}
\alias{rao.homogeneity}

\description{
Performs Rao's test for homogeneity on k populations of angular data. 
}

\usage{
rao.homogeneity(x, alpha=0)
}

\arguments{
\item{x}{a list of angular variables measured in radians.  Each component of the list should be a vector corresponding to one of the k samples. The data must be in the form of a list, to allow for samples of varying sizes.}
\item{alpha}{numeric value specifying the significance level of the test. Default is 0, in which case p-values for the test statistic are returned.}
}

\value{
NULL
}

\note{
The test is performed, and the results are written to the screen. Test results are given for both the test of equality of polar vectors, and of dispersions.  If alpha is specified, the test statistic is returned, along with the level alpha critical value.  If alpha is not specified, a p-value for the test is computed.  
}

\details{
Critical values and p-values are determined according to the chi-squared approximation of the the test statistic.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.6.1, World Scientific Press, Singapore.

Rao, J.S. (1967). Large sample tests for the homogeneity of angular data, Sankhya, Ser, B., 28.
}

\keyword{htest}


\eof
\name{rao.spacing}
\title{Rao's Spacing Test of Uniformity}
\alias{rao.spacing}

\description{
Performs Rao's Spacing Test of uniformity. 
}

\usage{
rao.spacing(x, alpha=0, rad=TRUE)
}

\arguments{
\item{x}{numeric vector of angular data measured in degrees.}
\item{alpha}{numeric value specifying the significance level of the test. The default value is 0, in which case, a range for the p-value will be returned.  Valid significance levels are 0.10, 0.05, 0.01 and 0.001.}
\item{rad}{logical value.  If true, data will be assumed to be measured radians.  If false, data will be assumed to be measured in degrees.}
}

\value{
NULL
}

\note{
The Rao's Spacing Test is performed, and the results are written to the screen.
}

\details{
If alpha is specified, critical values are determined from a table of simulated critical points (see reference below).  If alpha is not specified, a range for the p-value is determined using the table of simulated critical points.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.4, World Scientific Press, Singapore.

Rao, J.S. (1976). Some tests based on arc-lengths for the circle. Sankhya, The Indian Journal of Statistics, Serial B(4), 38, 329338.

Russell, G.S. and Levitin, D.J. (1995).  An expanded table of probability values for Rao's Spacing Test.  Communications in Statistics - Simulation and Computation, 24, 4, 879888.
}

\seealso{
\link{circ.range}, \link{kuiper}, \link{r.test}, \link{v0.test}, \link{watson}
}

\keyword{htest}



\eof
\name{rao.table}
\title{Table for Rao spacing function}
\alias{rao.table}

\description{Table for Rao spacing function}

\usage{data(rao.table)}

\keyword{datasets}


\eof
\name{rcard}
\title{Random Generation from the Cardoid Distribution}
\alias{rcard}

\description{
Generates pseudo-random numbers from a cardoid distribution. 
}

\usage{
rcard(n, mu, r)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{mu}{mean direction of the distribution in radians.}
\item{r}{concentration parameter of the distribution.  The absolute value of r must be less than 0.5.}
}

\value{
Returns a vector of n independent random variables generated from a cardoid distribution.
}

\details{
An acceptance/rejection method of simulation is employed.
}

\seealso{
\link{dcard}
}

\keyword{distribution}



\eof
\name{rmixedvm}
\title{Random Generation from the Mixed von Mises Distribution}
\alias{rmixedvm}

\description{
Generates pseudo-random numbers from a mixture of two von Mises distributions.
}

\usage{
rmixedvm(n, mu1, mu2, kappa1, kappa2, p)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{mu1}{mean direction in radians of one of the two von Mises distributions.}
\item{mu2}{mean direction in radians of the other von Mises distribution.}
\item{kappa1}{concentration parameter of one of the two von Mises distributions.}
\item{kappa2}{concentration parameter of the other von Mises distribution.}
\item{p}{mixing proportion.}
}

\value{
Returns a vector of n independent random variables generated from a mixture of two von Mises distributions.
}

\details{
Simulates random variables with the density function p*VM(mu1, kappa1) + (1-p)*VM(mu2, kappa2), where  VM is the von Mises density function.  

A uniform random variable on (0,1) is generated.  If it is less than p, then a variable is generated from VM(mu1, kappa1), else a variable is generated from VM(mu2, kappa2).  Simulation from the von Mises distribution is done via the algorithm due to Best and Fisher (1979).
}

\references{
Best, D. and Fisher, N. (1979).  Efficient simulation of the von Mises distribution.  Applied Statistics, 24, 152157.
}

\seealso{
\link{dmixedvm}, \link{rvm}
}

\keyword{distribution}



\eof
\name{rose.diag}
\title{Rose Diagram}
\alias{rose.diag}

\description{
Creates a rose diagram of a circular data set on the current graphics device.    
}

\usage{
rose.diag(x, bins, main="", prop=1, pts=FALSE, cex=1, pch=16, dotsep=40, shrink=1)
}


\arguments{
\item{x}{vector of directional data measured in radians.}
\item{bins}{number of groups to partition the circle with.  This will be the number of petals or sectors in the rose diagram.}
\item{main}{title of plot.  Default is no title.}
\item{prop}{numerical constant determining the radii of the sectors.  By default, prop = 1.}
\item{pts}{logical value.  If TRUE, points will be stacked on the circumference of the circle, according to bins - one point for each observation. The default value is FALSE, no points plotted.}
\item{cex}{size of points, if pts = TRUE.  By default,  cex = 1.  See help on cex in help section for par.}
\item{pch}{if pts = TRUE, pch determines the character used.  See help on pch in help section for par.}
\item{dotsep}{constant used to specify the distance between stacked points, if pts = TRUE.  Default is 40; larger values will create smaller spaces, while smaller values create larger spaces.  This option can be useful when pch is other than 1, or when shrink is other than 1 (see below).}
\item{shrink}{parameter that controls the size of the plotted circle.  Default is 1.  Larger values shrink the circle, while smaller values enlarge the circle.  This option can be useful when pts = TRUE.}
}

\value{
NULL
}

\note{
A rose diagram is plotted on the current graphics device.
}

\details{
The circumference of the circle is split into groups, the number of groups specified by bins.  For each group, a sector is drawn.  The radii of the sectors are by default equal to the square root of the relative frequencies of observations in each group.  This ensures that the area of the sector is proportional to the group frequency.  The length of the radii can be controlled by varying the parameter prop. 
}

\seealso{
\link{circ.plot}
}

\examples{
# Generate uniform data and create several rose diagrams.  
# Some optional parameters may be needed to optimize plots.
data <- runif(50, 0, 2*pi)
rose.diag(data, bins = 18, main = 'Uniform Data')
rose.diag(data, bins = 18, main = 'Stacked Points', pts=TRUE)
# Generate von Mises data and create several rose diagrams.
data <- rvm(25, 0, 5)
rose.diag(data, bins=18, pts=TRUE) # Points fall out of bounds.
rose.diag(data, bins=36, prop=1.5, pts=TRUE, shrink=1.5)
# Adjust optional parameters to fit all points on plot.
}

\keyword{hplot}



\eof
\name{rstable}
\title{Random Generator for Stable Family of Distributions}
\alias{rstable}

\description{
Returns random deviates from the stable family of probability distributions.
}

\usage{
rstable(n, scale = 1, index = stop("no index arg"), skewness = 0)
}

\arguments{
\item{n}{sample size}
\item{index}{number from the interval (0, 2]. An index of 2 corresponds to the normal, 1 to the Cauchy.  Smaller values mean longer tails.}
\item{skewness}{number giving the modified skewness (see Chambers et al., 1976).  Negative values correspond to skewness to the left (the median is smaller than the mean, if it exists), and positive values correspond to skewness to the right (the median is larger than the mean). The absolute value of skewness should not exceed 1.}
\item{scale}{the scale of the distribution.}
}

\value{
random sample from the specified stable distribution.
}

\details{
This function returns a random variate from the Levy skew stable
distribution with \code{index}=$\alpha$, \code{scale}=$c$ and \code{skewness}=$\beta$.  The \code{skewness} parameter must lie in the range [-1,1] while the \code{index} parameter must lie in the range (0,2].  The Levy skew stable probability distribution is defined by a fourier transform,
\eqn{
p(x) = {1 \over 2 \pi} \int_{-\infty}^{+\infty} dt \exp(-it x - |c t|^\alpha (1-i \beta sign(t) \tan(\pi\alpha/2)))
}

When $\alpha = 1$ the term $\tan(\pi \alpha/2)$ is replaced by $-(2/\pi)\log|t|$.   For $\alpha = 2$ the distribution reduces to a Gaussian distribution with $\sigma = \sqrt{2} scale$ and the skewness parameter has no effect.  
For $\alpha < 1$ the tails of the distribution become extremely
wide.  The symmetric distribution corresponds to $\beta =0$.

The Levy alpha-stable distributions have the property that if $N$
alpha-stable variates are drawn from the distribution $p(c, \alpha,
\beta)$ then the sum $Y = X_1 + X_2 + \dots + X_N$ will also be
distributed as an alpha-stable variate, $p(N^{1/\alpha} c, \alpha, \beta)$.

There is no explicit solution for the form of $p(x)$ and there are no density, probability or quantile functions supplied for this distribution.
}

\references{
Chambers, J. M., Mallows, C. L. and Stuck, B. W. (1976). A Method for Simulating Stable Random Variables. Journal of the American Statistical Association 71, 340-344.

Lo\"{g}ae\"{v}e, M. (1977). Probability Theory I. (fourth edition) Springer-Verlag, New York.
}

\seealso{
\link{rnorm}, \link{rcauchy}.
}

\examples{
hist(rstable(200, 1.5, .5)) #fairly long tails, skewed right
}

\keyword{distribution}





\eof
\name{rtri}
\title{Random Generation from the Triangular Distribution}
\alias{rtri}

\description{
Generates pseudo-random numbers from a symmetric triangular distribution.
}

\usage{
rtri(n, r)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{r}{concentration parameter of the distribution.  r must be between 0 and \eqn{4/pi^2}.}
}

\value{
Returns a vector of n independent random variables generated from a symmetric triangular distribution with mean direction 0 and concentration parameter r.
}

\details{
The triangular cdf is inverted to obtain the random numbers.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.3.3, World Scientific Press, Singapore.
}

\seealso{
\link{dtri}
}

\keyword{distribution}



\eof
\name{rvm}
\title{Random Generation from the von Mises Distribution}
\alias{rvm}

\description{
Generates pseudo-random numbers from a von Mises distribution. 
}

\usage{
rvm(n, mean, k)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{mean}{mean direction in radians of the von Mises distribution.}
\item{k}{concentration parameter of the von Mises distribution.}
}

\value{
Returns a vector of n independent random variables generated from a von Mises distribution.  Values are between 0 and 2 pi.
}

\details{
Simulation from the von Mises distribution is done via the algorithm due to Best and Fisher (1979).
}

\references{
Best, D. and Fisher, N. (1979).  Efficient simulation of the von Mises distribution.  Applied Statistics, 24, 152157.
}

\seealso{
\link{dmixedvm}, \link{dvm}, \link{pvm}, \link{rmixedvm}
}

\keyword{distribution}



\eof
\name{rwrpcauchy}
\title{Random Generation from the Wrapped Cauchy Distribution}
\alias{rwrpcauchy}

\description{
Generates pseudo-random numbers from a wrapped Cauchy distribution.}

\usage{
rwrpcauchy(n, location=0, rho=exp(-1))
}

\arguments{
\item{n}{number of random variables to generate.}
\item{location}{modal angle in radians of the wrapped Cauchy distribution.}
\item{rho}{concentration parameter.  rho must be between 0 and 1.}
}

\value{
Returns a vector of n independent random variables generated from a wrapped Cauchy distribution.
}

\details{
n random variables are generated from a Cauchy distribution with location parameter location, and concentration parameter -log(rho).  The function returns these values modulo 2*pi.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.6, World Scientific Press, Singapore.
}

\seealso{
\link{dwrpcauchy}
}

\keyword{distribution}


\eof
\name{rwrpnorm}
\title{Random Generation from the Wrapped Normal Distribution}
\alias{rwrpnorm}

\description{
Generates pseudo-random numbers from a wrapped normal distribution.}

\usage{
rwrpnorm(n, mu, rho, sd=1)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{mu}{mean direction in radians of the wrapped normal distribution.}
\item{rho}{concentration parameter.  rho must be between 0 and 1.}
\item{sd}{different way of select \code{rho}, see details below.}
}

\value{
Returns a vector of n independent random variables generated from a wrapped normal distribution.
}

\details{
n random variables are generated from a normal distribution with mean
direction mu, and variance -2*log(rho).  The function returns these
values modulo 2*pi. You can set \code{rho} by using \code{sd} with the following equivalence:
\deqn{
\rho = \exp{- \sigma^2/2}
}
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.5, World Scientific Press, Singapore.
}

\seealso{
\link{dwrpnorm}
}

\keyword{distribution}



\eof
\name{rwrpstab}
\title{Random Generation from the Wrapped Stable Distribution}
\alias{rwrpstab}

\description{
Generates pseudo-random numbers from a wrapped stable distribution.
}

\usage{
rwrpstab(n, index, skewness, scale=1)
}

\arguments{
\item{n}{number of random variables to generate.}
\item{index}{number from the interval (0, 2]. An index of 2 corresponds to the normal, 1 to the Cauchy.  Smaller values mean longer tails.}
\item{skewness}{number giving the modified skewness. Negative values correspond to skewness to the left (the median is smaller than the mean, if it exists), and positive values correspond to skewness to the right (the median is larger than the mean). The absolute value of skewness should not exceed 1.}
\item{scale}{the scale of the distribution}
}

\value{
Returns a vector of n independent random variables generated from a wrapped stable distribution.
}

\details{
n random variables are generated from a stable distribution with with parameters index, skewness and scale. The function returns these values modulo 2*pi.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 2.2.8, World Scientific Press, Singapore.
}

\keyword{distribution}



\eof
\name{trig.moment}    
\title{Trigonometric Moments}
\alias{trig.moment}  

\description{
Computes the specified order trigonometric moment for a set of directional data points. 
}

\usage{
trig.moment(x, p=1, center=FALSE)
}

\arguments{
\item{x}{directional data set measured in radians.}
\item{p}{order of trigonometric moment to be computed. Default is for a first order trigonometric moment.}
\item{center}{logical flag whether to compute centered moments or not.  Default is to compute an uncentered moment.}
}

\value{
Returns a data frame with variables mu.p, rho.p, cos.p, and sin.p, respectively the pth trigonometric moment's direction, resultant length, and real and imaginary components.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 1.3, World Scientific Press, Singapore.
}

\seealso{
\link{circ.disp}, \link{circ.mean}, \link{circ.summary}, \link{est.kappa}, \link{est.rho}
}

\keyword{math}


\eof
\name{v0.test}
\title{Rayleigh Test of Uniformity:  Alternative with Specified Mean Direction}
\alias{v0.test}

\description{
Performs a Rayleigh test of uniformity, assessing the significance of the mean resultant length.  The alternative hypothesis is a unimodal distribution with a specified mean direction and unknown mean resultant length.  
}

\usage{
v0.test(x, mu0=0, degree=FALSE)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{mu0}{Specified mean direction in alternative hypothesis, measured in the same units (degrees or radians) as the data.}
\item{degree}{logical flag:  if TRUE, data is assumed to be measured in degrees rather than radians. Default is FALSE.}
}

\value{
Returns a list with two components:  the mean resultant length, r0.bar, and the p-value of the test statistic, p.value.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Sections 3.3.3 and 3.4.1, World Scientific Press, Singapore.
}

\seealso{
\link{circ.range}, \link{kuiper}, \link{rao.spacing}, \link{r.test}, \link{watson}
}

\examples{
data <- rvm(25, pi, 2)
v0.test(data, mu0=pi)
}

\keyword{htest}


\eof
\name{vm.bootstrap.ci}
\title{Bootstrap Confidence Intervals}
\alias{vm.bootstrap.ci}

\description{
Generates simple bootstrap confidence intervals for the parameters of a von Mises distribtution: the mean direction mu, and the concentration parameter kappa.
}


\usage{
vm.bootstrap.ci(x, bias=FALSE, alpha=0.05, reps=1000, print=TRUE)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{bias}{logical flag: if TRUE, the replication estimates for kappa are computed with a bias corrected method.  See est.kappa.  Default is FALSE, i.e. no bias correction.}
\item{alpha}{parameter determining level of confidence intervals. 1-alpha confidence intervals for mu and kappa are computed.  By default, 95\% confidence intervals are generated.}
\item{reps}{number of resampled data sets to use.  Default is 1000.}
\item{print}{logical flag indicating whether the algorithm should print a message indicating which set of replicates is currently being drawn.  Default is TRUE.}
}



\value{
A list is returned with the following components:  mu.ci and kappa.ci contain the limits of the confidence intervals for mu and kappa respectively.  mu.reps and kappa.reps contain the estimates of mu and kappa for each resampled data set.
}

\note{
The confidence intervals are printed to the screen.
}

\details{
Percentile confidence intervals are computed by resampling from the original data set B times.  For each resampled data set, the MLE's of mu and kappa are computed.  The bootstrap confidence intervals are the alpha/2 and 1-alpha/2 percentiles of the B MLE's computed for each resampled data set.
}

\seealso{
\link{vm.ml}, \link{est.kappa}
}

\examples{
x <- rvm(25, 0, 3)
x.bs <- vm.bootstrap.ci(x, alpha=.10)
hist(x.bs$kappa.reps)
}



\keyword{htest}


\eof
\name{vm.ml}
\title{von Mises Maximum Likelihood Estimates}
\alias{vm.ml}

\description{
Computes the maximum likelihood estimates for the parameters of a von Mises distribution:  the mean direction and the concentration parameter.
}

\usage{
vm.ml(x, bias=FALSE)
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{bias}{logical flag: if TRUE, the estimate for kappa is computed with a bias corrected method.  See est.kappa.  Default is FALSE, i.e. no bias correction.}
}

\value{
Returns a data frame with two components:  mu and kappa, the MLES of the respective parameters.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 4.2.1, World Scientific Press, Singapore.
}

\seealso{
\link{circ.mean}, \link{est.kappa}, \link{vm.bootstrap.ci}
}

\keyword{htest}



\eof
\name{watson}
\title{Watson's Test }
\alias{watson}

\description{
Performs a Watson's goodness of fit test for the von Mises or circular uniform distribution. 
}

\usage{
watson(x, alpha=0, dist='uniform')
}

\arguments{
\item{x}{vector of angular measurements in radians.}
\item{alpha}{significance level of the test.  Valid levels are 0.01, 0.05, 0.1.  This argument may be ommited, in which case, a range for the p-value will be returned.}
\item{dist}{distribution to test for.  The default is the uniform distribution. To test for the von Mises distribution, set dist = 'vm'.}
}

\value{
NULL
}

\note{
If dist = 'uniform', Watson's one-sample test for the circular uniform distribution is performed, and the results are printed to the screen.  If alpha is specified and non-zero, the test statistic is printed along with the critical value and decision.  If alpha is omitted, the test statistic is printed and a range for the p-value of the test is given.

If dist = 'vm', estimates of the population parameters are used to evaluate the von Mises distribution function at all data points, thereby arriving at a sample of approximately uniformly distributed data, if the original observations have a von Mises distribution. The one-sample Watson test is then applied to the transformed data as above.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.2, World Scientific Press, Singapore.

Stephens, M. (1970).  Use of the Kolmogorov-Smirnov, Cramer-von Mises and related statistics without extensive tables.  Journal of the Royal Statistical Society, B32, 115122.
}

\seealso{
\link{circ.range}, \link{kuiper}, \link{rao.spacing}, \link{r.test}, \link{v0.test}
}

\examples{
# Generate data from the uniform distribution on the circle.
data <- runif(100, 0, 2*pi)  
watson(data)
# Generate data from a von Mises distribution.
data <- rvm(50, 0, 4) 
watson(data, 0.05, dist='vm')
}

\keyword{htest}



\eof
\name{watson.two}
\title{Watson's Two-Sample Test of Homogeneity}
\alias{watson.two}

\description{
Performs Watson's test for homogeneity on two samples of circular data. 
}

\usage{
watson.two(x, y, alpha=0, plot=FALSE)
}

\arguments{
\item{x}{vector of circular data measured in radians.}
\item{y}{vector of circular data measured in radians.}
\item{alpha}{significance level of the test.  Valid levels are 0.001, 0.01, 0.05, 0.1.  This argument may be ommited, in which case, a range for the p-value will be returned.}
\item{plot}{logical value.  If TRUE, an overlayed plot of both empirical distribution functions will be sent to the current graphics device.  The default value if FALSE.}
}

\value{
NULL
}

\note{
Watson's two-sample test of homogeneity is performed, and the results are printed to the screen.  If alpha is specified and non-zero, the test statistic is printed along with the critical value and decision.  If alpha is omitted, the test statistic is printed and a range for the p-value of the test is given.  If plot=TRUE, an overlayed plot of both empirical distribution functions will be sent to the current graphics device.
}

\details{
Critical values for the test statistic are obtained using the asymptotic distribution of the test statistic.  It is recommended to use the obtained critical values and ranges for p-values only for combined sample sizes in excess of 17.  Tables are available for smaller sample sizes and can be found in Mardia (1972) for instance.
}

\references{
Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 7.5, World Scientific Press, Singapore.
}

\examples{
# Perform a two-sample test of homogeneity on two
# simulated data sets.
data1 <- rvm(20, 0, 3)
data2 <- rvm(20, pi, 2)
watson.two(data1, data2, alpha=0.05, plot=TRUE)
watson.two(data1, data2)
}

\keyword{htest}



\eof
\name{wrpcauchy.ml}
\title{Wrapped Cauchy Maximum Likelihood Estimates}
\alias{wrpcauchy.ml}

\description{
Computes the maximum likelihood estimates of the location and scale parameters for a wrapped Cauchy distribution.
}

\usage{
wrpcauchy.ml(x, mu0, rho0, acc=1e-015)
}

\arguments{
\item{x}{vector of angular data measured in radians.}
\item{mu0}{initial estimate of the location parameter.}
\item{rho0}{initial estimate of the scale parameter.  rho0 should be between in [0,1).}
\item{acc}{degree of accuracy in the approximation of the estimates.  The default value is 1e-15.  See below for details.}
}

\value{
A dataframe is returned with the variables mu and rho, where mu and rho are the approximations of the MLE of the location  and scale parameter, respectively.
}

\details{
An iterative algorithm due to Kent and Tyler (1988) is used.  Initial values of the MLE's are required.  In the estimation process, estimates of quantities mu1 and mu2 (see Kent and Tyler) are updated iteratively.  When both mu1 and mu2 change by less than acc from one iteration to the next, the process ends, and mu and rho are computed from the final estimates of mu1 and mu2.
}

\references{
Kent, J. and Tyler, D. (1988).  Maximum likelihood estimation for the wrapped Cauchy distribution.  Journal of Applied Statistics, 15, 2, 247254.

Jammalamadaka, S. Rao and SenGupta, A. (2001). Topics in Circular Statistics, Section 4.4, World Scientific Press, Singapore.
}

\seealso{
\link{dwrpcauchy}, \link{rwrpcauchy}
}

\examples{
# Generate data from a wrapped Cauchy distribution.
data <- rwrpcauchy(50, 0, 0.75)
# Compute the sample mean direction and resultant length.
mu0 <- circ.mean(data)
rho0 <- est.rho(data)
# Estimate the MLE's of the Cauchy distribution parameters.
wrpcauchy.ml(data, mu0, rho0)
}

\keyword{htest}



\eof
