\name{MCMCbaselineEI}
\alias{MCMCbaselineEI}
\title{Markov chain Monte Carlo for Wakefield's Baseline Ecological
  Inference Model}
\description{
  MCMCbaselineEI is used to fit Wakefield's baseline ecological
  inference model for partially observed 2 x 2 contingency tables.
}
  
\usage{
MCMCbaselineEI(r0, r1, c0, c1, burnin=1000, mcmc=50000, thin=10,
               tune=2.65316, verbose=FALSE, seed=0, alpha0=1, beta0=1,
               alpha1=1, beta1=1, method="NA", ...)
   }

\arguments{
  \item{r0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 0.}

  \item{r1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 1.}

  \item{c0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 0.}

  \item{c1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 1.}

  \item{burnin}{The number of burn-in scans for the sampler.}

  \item{mcmc}{The number of mcmc scans to be saved.}

  \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

  \item{tune}{Tuning parameter for the Metropolis-Hasting sampling.}

  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  Information is printed
    if TRUE.}

  \item{seed}{The seed for the random number generator.  The code uses the
  Mersenne Twister, which requires an integer as an input.  If nothing
  is provided, the Scythe default seed is used.}

  \item{alpha0}{alpha parameter for the beta prior on \eqn{p_0}{p0}.}

  \item{beta0}{beta parameter for the beta prior on \eqn{p_0}{p0}.}

  \item{alpha1}{alpha parameter for the beta prior on \eqn{p_1}{p1}.}

  \item{beta1}{beta parameter for the beta prior on \eqn{p_1}{p1}.}

  \item{method}{Parameter determining whether a data augmentation
    algorithm should be used on the exact posterior (``DA"), or a
    Metropolis-Hastings algorithm should be used on
    Wakefield's normal approximation to the posterior (``NA"). For tables
    with large row and column sums, the preferred method is ``NA."}

  \item{...}{further arguments to be passed}     
}

\value{
  An mcmc object that contains the posterior density sample.
  This object can  be summarized by functions provided by the coda package.
}

\details{
  Consider the following partially observed 2 by 2 contingency table:\cr
  \cr
  \tabular{llll}{
               \tab | \eqn{Y=0} \tab | \eqn{Y=1} \tab |   \cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=0} \tab | \eqn{Y_0}{Y0} \tab |  \tab | \eqn{r_0}{r0}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=1} \tab | \eqn{Y_1}{Y1} \tab |  \tab | \eqn{r_1}{r1}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
              \tab | \eqn{c_0}{c0} \tab | \eqn{c_1}{c1} \tab | \eqn{N}\cr    
  }

  where \eqn{r_0}{r0}, \eqn{r_1}{r1}, \eqn{c_0}{c0}, \eqn{c_1}{c1}, and
  \eqn{N}  are non-negative integers that are
  observed. The interior cell entries are not observed. It is
  assumed that \eqn{Y_0|r_0 \sim \mathcal{B}inomial(r_0,
    p_0)}{Y0|r0 ~ Binomial(r0, p0)} and
  \eqn{Y_1|r_1 \sim \mathcal{B}inomial(r_1, p_1)}{Y1|r1 ~
    Binomial(r1,p1)}. Inference centers on
  \eqn{p_0}{p0} and \eqn{p_1}{p1}.
  Wakefield's baseline model starts with the
  assumption that a priori
  \eqn{p_0 \sim \mathcal{B}eta(\alpha_0, \beta_0)}{p0 ~
    Beta(alpha0, beta0)} and
  \eqn{p_1 \sim \mathcal{B}eta(\alpha_1, \beta_1)}{p1 ~
    Beta(alpha1, beta1)}.        
  }
  
\references{
   Jonathan Wakefield. 2001. ``Ecological Inference for 2 x 2 Tables,"
   Center for Statistics and the Social Sciences Working Paper
   no. 12. University of Washington. 

   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.   
   \emph{Scythe Statistical 
   Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}

\examples{
   \dontrun{
   posterior <- MCMCbaselineEI(300, 200, 100, 400)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link{MCMChierEI}}, \code{\link{MCMCdynamicEI}},
  \code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}}


\eof
\name{MCMCdynamicEI}
\alias{MCMCdynamicEI}
\title{Markov chain Monte Carlo for Quinn's Dynamic Ecological
  Inference Model}
\description{
  MCMCdynamicEI is used to fit Quinn's dynamic ecological inference
  model for partially observed 2 x 2 contingency tables.
  }
  
\usage{
MCMCdynamicEI(r0, r1, c0, c1, burnin=5000, mcmc=200000, thin=200,
              tune=2.65316, verbose=FALSE, seed=0, W=0, nu0=1,
              delta0=0.03, nu1=1, delta1=0.03, ...)
   }

\arguments{
  \item{r0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 0.}

  \item{r1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 1.}

  \item{c0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 0.}

  \item{c1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 1.}

  \item{burnin}{The number of burn-in scans for the sampler.}

  \item{mcmc}{The number of mcmc scans to be saved.}

  \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

  \item{tune}{Tuning parameter for the Metropolis-Hasting sampling.}


  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  Information is printed
    if TRUE.}

  \item{seed}{The seed for the random number generator.  The code uses the
   Mersenne Twister, which requires an integer as an input.  If nothing
   is provided, the Scythe default seed is used.}
    
  \item{W}{Weight (\emph{not precision}) matrix structuring the temporal
    dependence among elements of  \eqn{\theta_{0}}{theta0} and
    \eqn{\theta_{1}}{theta1}. The default value of 0 will construct a
    weight matrix that corresponds to random walk priors for
    \eqn{\theta_{0}}{theta0} and \eqn{\theta_{1}}{theta1}. The default
    assumes that the tables are equally spaced throughout time and that
    the elements of \eqn{r0}, \eqn{r1}, \eqn{c0}, and \eqn{c1} are
    temporally ordered.}

  \item{nu0}{Shape parameter for the inverse-gamma prior on the
     \eqn{\sigma^2_0}{sigma^2_0} parameter.}
  
  \item{delta0}{Scale parameter for the inverse-gamma prior on the
     \eqn{\sigma^2_0}{sigma^2_0} parameter.}
             
  \item{nu1}{Shape parameter for the inverse-gamma prior on the
     \eqn{\sigma^2_1}{sigma^2_1} parameter.}
  
  \item{delta1}{Scale parameter for the inverse-gamma prior on the
     \eqn{\sigma^2_1}{sigma^2_1} parameter.}

  \item{...}{further arguments to be passed}     
}

\value{
  An mcmc object that contains the posterior density sample.
  This object can  be summarized by functions provided by the coda package.
}

\details{
  Consider the following partially observed 2 by 2 contingency table for
  unit \eqn{t} where \eqn{t=1,\ldots,ntables}{t=1,...,ntables}:\cr
  \cr
  \tabular{llll}{
               \tab | \eqn{Y=0} \tab | \eqn{Y=1} \tab |   \cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=0} \tab | \eqn{Y_{0t}}{Y0[t]} \tab |  \tab | \eqn{r_{0t}}{r0[t]}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=1} \tab | \eqn{Y_{1t}}{Y1[t]} \tab |  \tab | \eqn{r_{1t}}{r1[t]}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \tab | \eqn{c_{0t}}{c0[t]} \tab | \eqn{c_{1t}}{c1[t]} \tab | \eqn{N_t}{N[t]}\cr    
  }
  Where \eqn{r_{0t}}{r0-t}, \eqn{r_{1t}}{r1[t]},
  \eqn{c_{0t}}{c0[t]}, \eqn{c_{1t}}{c1[t]}, and
  \eqn{N_t}{N[t]}  are non-negative integers that are
  observed. The interior cell entries are not observed. It is
  assumed that \eqn{Y_{0t}|r_{0t} \sim \mathcal{B}inomial(r_{0t},
    p_{0t})}{Y0[t]|r0[t] ~ Binomial(r0[t], p0[t])} and 
  \eqn{Y_{1t}|r_{1t} \sim \mathcal{B}inomial(r_{1t}, p_{1t})}{Y1[t]|r1[t] ~
    Binomial(r1[t],p1[t])}. Let \eqn{\theta_{0t} =
    log(p_{0t}/(1-p_{0t}))}{theta0[t] = log(p0[t]/(1-p0[t]))},
  and  \eqn{\theta_{1t} = log(p_{1t}/(1-p_{1t}))}{theta1[t] =
    log(p1[t]/(1-p1[t]))}.

  The following prior distributions are
  assumed:
  \deqn{p(\theta_0|\sigma^2_0) \propto \sigma_0^{-ntables}
    \exp \left(-\frac{1}{2\sigma^2_0}
    \theta'_{0} P \theta_{0}\right)}{p(theta0|sigma^2_0) propto
    sigma^(-ntables)_0 exp(-1/(2*sigma^2_0) theta0' * P * theta0)}
  and
  \deqn{p(\theta_1|\sigma^2_1) \propto \sigma_1^{-ntables}
    \exp \left(-\frac{1}{2\sigma^2_1}
    \theta'_{1} P \theta_{1}\right)}{p(theta1|sigma^2_1) propto
    sigma^(-ntables)_1 exp(-1/(2*sigma^2_1) theta1' * P * theta1)}
  where \eqn{P_{ts}}{P[t,s]} = \eqn{-W_{ts}}{-W[t,s]} for \eqn{t} not
  equal to \eqn{s} and \eqn{P_{tt}}{P[t,t]} =
  \eqn{\sum_{s \ne t}W_{ts}}{sum(W[t,])}.
  The \eqn{\theta_{0t}}{theta0[t]} is assumed to be a priori independent of
  \eqn{\theta_{1t}}{theta1[t]} for all t.
  In addition, the
  following hyperpriors are assumed:
  \eqn{\sigma^2_0 \sim \mathcal{IG}(\nu_0/2, \delta_0/2)}{\sigma^2_0 ~
    InvGamma(nu0/2, delta0/2)}, and
  \eqn{\sigma^2_1 \sim \mathcal{IG}(\nu_1/2, \delta_1/2)}{\sigma^2_1 ~
    InvGamma(nu1/2, delta1/2)}.

  Inference centers on \eqn{p_0}{p0}, \eqn{p_1}{p1},
  \eqn{\sigma^2_0}{sigma^2_0}, and \eqn{\sigma^2_1}{sigma^2_1}.
 The Metropolis-Hastings algorithm is used to sample from the posterior
  density. 
  }
  
\references{
  Jonathan Wakefield. 2001. ``Ecological Inference for 2 x 2 Tables."
  Center for Statistics and the Social Sciences Working Paper
  no. 12. University of Washington. 

  Kevin M. Quinn. 2002. ``Ecological Inference in the Presence of
  Temporal Dependence." Paper prepared for Ecological Inference
  Conference, Harvard University, June 17-18, 2002.

  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003. 
  \emph{Scythe Statistical 
    Library 0.4.} \url{http://scythe.wustl.edu}.
  
  Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
  \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
  \url{http://www-fis.iarc.fr/coda/}.
}

\examples{
   \dontrun{
   r0 <- rpois(10, 300)
   r1 <- rpois(10, 200)
   c0 <- 100 + 1:10*7 + rpois(10, 30)
   c1 <- (r0+r1) - c0
   posterior <- MCMCdynamicEI(r0, r1, c0, c1,
      burnin=10000, mcmc=5000000, thin=1000)
   plot(posterior)
   summary(posterior) 
   }
}

\keyword{models}

\seealso{\code{\link{MCMCbaselineEI}}, \code{\link{MCMChierEI}},
  \code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}}

\eof
\name{MCMCfactanal}
\alias{MCMCfactanal}
\title{Markov chain Monte Carlo for Normal Theory Factor Analysis Model}
\description{
  This function generates a posterior density sample from Normal theory
  factor analysis model. Normal priors are assumed on the factor
  loadings and factor scores while inverse Gamma priors are assumed for
  the uniquenesses. The user supplies data and parameters for the prior
  distributions, and a sample from the posterior density is returned as
  an mcmc object, which can be subsequently analyzed with
  functions provided in the coda package.
}
  
\usage{
MCMCfactanal(x, factors, lambda.constraints=list(),
             data=list(), burnin = 1000, mcmc = 10000,
             thin=5, verbose = FALSE, seed = 0,
             lambda.start = NA, psi.start = NA,
             l0=0, L0=0, a0=0.001, b0=0.001,
             store.scores = FALSE, std.var=TRUE, ... )
 }

\arguments{
  \item{x}{Either a formula or a numeric matrix containing the
    manifest variables.}

  \item{factors}{The number of factors to be fitted.}
  
  \item{lambda.constraints}{List of lists specifying possible simple equality
    or inequality constraints on the factor loadings. A typical
    entry in the list has one of three forms: \code{varname=list(d,c)} which
    will constrain the dth loading for the variable named varname to
    be equal to c, \code{varname=list(d,"+")} which will constrain the dth
    loading for the variable named varname to be positive, and
    \code{varname=list(d, "-")} which will constrain the dth loading for the
    variable named varname to be negative. If x is a matrix without
    column names defaults names of ``V1",``V2", ... , etc will be
    used.}
  
  \item{data}{A data frame.}
  
  \item{burnin}{The number of burn-in iterations for the sampler.}
  
  \item{mcmc}{The number of iterations for the sampler.}
  
  \item{thin}{The thinning interval used in the simulation.  The number of
    iterations must be divisible by this value.}
  
  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the factor loadings and uniquenesses are printed to the screen.}
  
  \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}
  
  \item{lambda.start}{Starting values for the factor loading matrix
    Lambda. If \code{lambda.start} is set to a scalar the starting value for
    all unconstrained loadings will be set to that scalar. If
    \code{lambda.start} is a matrix of the same dimensions as Lambda then the
    \code{lambda.start} matrix is used as the starting values (except
    for equality-constrained elements). If \code{lambda.start} is set to
    \code{NA} (the default) then starting values for unconstrained
    elements are set to 0, and starting values for inequality
    constrained elements are set to either 0.5 or -0.5 depending on the
    nature of the constraints.}
  
  \item{psi.start}{Starting values for the uniquenesses. If
    \code{psi.start} is set to a scalar then the starting value for all
    diagonal elements of \code{Psi} are set to this value. If
    \code{psi.start} is a \eqn{k}{k}-vector (where \eqn{k}{k} is the
    number of manifest variables) then the staring value of \code{Psi}
    has \code{psi.start} on the main diagonal. If \code{psi.start} is
    set to \code{NA} (the default) the starting values of all the
    uniquenesses are set to 0.5.}

  \item{l0}{The means of the independent Normal prior on the factor
    loadings. Can be either a scalar or a matrix with the same
    dimensions as \code{Lambda}.}

  \item{L0}{The precisions (inverse variances) of the independent Normal
  prior on the factor loadings. Can be either a scalar or a matrix with
  the same dimensions as \code{Lambda}.}

  \item{a0}{Controls the shape of the inverse Gamma prior on the
    uniqueness. The actual shape parameter is set to \code{a0/2}. Can be
    either a scalar or a \eqn{k}{k}-vector.} 

  \item{b0}{Controls the scale of the inverse Gamma prior on the
    uniquenesses. The actual scale parameter is set to \code{b0/2}. Can
    be either a scalar or a \eqn{k}{k}-vector.} 
   
  \item{store.scores}{A switch that determines whether or not to
    store the factor scores for posterior analysis. 
    \emph{NOTE: This takes an enormous amount of memory, so
      should only be used if the chain is thinned heavily, or for
      applications with a small number of observations}.  By default, the
    factor scores are not stored.}

  \item{std.var}{If \code{TRUE} (the default) the manifest variables are
  rescaled to have zero mean and unit variance. Otherwise, the manifest
  variables are rescaled to have zero mean but retain their observed
  variances.}
  
  \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{The model takes the following form:

  \deqn{x_i = \Lambda \phi_i + \epsilon_i}{x_i = Lambda phi_i + epsilon_i}
  \deqn{\epsilon_i \sim \mathcal{N}(0,\Psi)}{epsilon_i ~ N(0, Psi)}

  where \eqn{x_i}{x_i} is the \eqn{k}{k}-vector of observed variables
  specific to observation \eqn{i}{i}, \eqn{\Lambda}{Lambda} is the
  \eqn{k \times d}{k by d} matrix of factor loadings, \eqn{\phi_i}{phi_i} is
  the \eqn{d}{d}-vector of latent factor scores, and \eqn{\Psi}{Psi} is
  a diagonal, positive definite matrix. Traditional factor analysis
  texts refer to the diagonal elements of \eqn{\Psi}{Psi} as
  uniquenesses.  

  The implementation used here assumes independent conjugate priors for
  each element of \eqn{\Lambda}{Lambda}, each \eqn{\phi_i}{phi_i}, and
  each diagonal element of \eqn{\Psi}{Psi}. More specifically we assume:

  \deqn{\Lambda_{ij} \sim \mathcal{N}(l_{0_{ij}}, L_{0_{ij}}^{-1}),
    i=1,\ldots,k,  j=1,\ldots,d}{Lambda_ij ~ N(l0_ij,  L0_ij^-1),
    i=1,...,k, j=1,...,d} 

    \deqn{\phi_i \sim \mathcal{N}(0, I),  i=1,\dots,n}{phi_i ~ N(0, I),
      i=1,...,n} 

    \deqn{\Psi_{ii} \sim \mathcal{IG}(a_{0_i}/2, b_{0_i}/2), 
      i=1,\ldots,k}{Psi_ii ~ IG(a0_i/2, b0_i/2), i=1,...,k}
    
  \code{MCMCfactanal} simulates from the posterior density using
  standard Gibbs sampling. The simulation proper is done in
  compiled C++ code to maximize efficiency.  Please consult the
  coda documentation for a comprehensive list of functions that
  can be used to analyze the posterior density sample.     
  }
}

\references{
   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   ### An example using the formula interface
   data(swiss)
   posterior <- MCMCfactanal(~Agriculture+Examination+Education+Catholic
                    +Infant.Mortality, factors=2,
                    lambda.constraints=list(Examination=list(1,"+"),
                       Examination=list(2,"-"), Education=c(2,0),
                       Infant.Mortality=c(1,0)),
                    verbose=FALSE, store.scores=FALSE, a0=1, b0=0.15,
                    data=swiss, burnin=5000, mcmc=50000, thin=20)
   plot(posterior)
   summary(posterior)

   ### An example using the matrix interface
   Lambda <- matrix(runif(45,-.5,.5), 15, 3)
   Psi <- diag(1 - apply(Lambda ^2, 1, sum))
   Sigma <- Lambda \%*\% t(Lambda) + Psi 
   Y <- t(t(chol(Sigma)) \%*\% matrix(rnorm(500*15), 15, 500))

   posterior <- MCMCfactanal(Y, factors=3,
                    lambda.constraints=list(V1=c(2,0),
                       V1=c(3,0), V2=c(3,0), V3=list(1,"+"),
                       V3=list(2,"+"), V3=list(3,"+")),
                    verbose=FALSE)
   plot(posterior)
   summary(posterior)        
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},\code{\link[mva]{factanal}}}


\eof
\name{MCMChierEI}
\alias{MCMChierEI}
\title{Markov chain Monte Carlo for Wakefield's Hierarchial Ecological
  Inference Model}
\description{
  `MCMChierEI' is used to fit Wakefield's hierarchical ecological inference
  model for partially observed 2 x 2 contingency tables.
  }
  
\usage{
MCMChierEI(r0, r1, c0, c1, burnin=1000, mcmc=50000, thin=1,
           m0=0, M0=10, m1=0, M1=10, nu0=1, delta0=0.5, nu1=1,
           delta1=0.5, verbose=FALSE, tune=2.65316, seed=0, ...)
   }

\arguments{
  \item{r0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 0.}

  \item{r1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of row
  sums from row 1.}

  \item{c0}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 0.}

  \item{c1}{\eqn{(ntables \times 1)}{(ntables * 1)} vector of
  column sums from column 1.}

  \item{burnin}{The number of burn-in scans for the sampler.}

  \item{mcmc}{The number of mcmc scans to be saved.}

  \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

  \item{tune}{Tuning parameter for the Metropolis-Hasting sampling.}


  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  Information is printed
    if TRUE.}

  \item{seed}{The seed for the random number generator.  The code uses the
  Mersenne Twister, which requires an integer as an input.  If nothing
  is provided, the Scythe default seed is used.}

  \item{m0}{Prior mean of the \eqn{\mu_0}{mu0} parameter.}

  \item{M0}{Prior variance of the \eqn{\mu_0}{mu0} parameter.}

  \item{m1}{Prior mean of the \eqn{\mu_1}{mu1} parameter.}

  \item{M1}{Prior variance of the \eqn{\mu_1}{mu1} parameter.}

  \item{nu0}{Shape parameter for the inverse-gamma prior on the
    \eqn{\sigma^2_0}{sigma^2_0} parameter.}
  
  \item{delta0}{Scale parameter for the inverse-gamma prior on the
    \eqn{\sigma^2_0}{sigma^2_0} parameter.}
             
  \item{nu1}{Shape parameter for the inverse-gamma prior on the
    \eqn{\sigma^2_1}{sigma^2_1} parameter.}
  
  \item{delta1}{Scale parameter for the inverse-gamma prior on the
    \eqn{\sigma^2_1}{sigma^2_1} parameter.}


  \item{...}{further arguments to be passed}     
}

\value{
  An mcmc object that contains the posterior density sample.
  This object can  be summarized by functions provided by the coda package.
}

\details{
  Consider the following partially observed 2 by 2 contingency table for
  unit \eqn{t} where \eqn{t=1,\ldots,ntables}:\cr
  \cr
  \tabular{llll}{
               \tab | \eqn{Y=0} \tab | \eqn{Y=1} \tab |   \cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=0} \tab | \eqn{Y_{0t}}{Y0[t]} \tab |  \tab |\eqn{r_{0t}}{r0[t]}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=1} \tab | \eqn{Y_{1t}}{Y1[t]} \tab |  \tab | \eqn{r_{1t}}{r1[t]}\cr
   - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
   \tab | \eqn{c_{0t}}{c0[t]} \tab | \eqn{c_{1t}}{c1[t]} \tab | \eqn{N_t}{N[t]}\cr    
  }
  Where \eqn{r_{0t}}{r0[t]}, \eqn{r_{1t}}{r1[t]},
  \eqn{c_{0t}}{c0[t]}, \eqn{c_{1t}}{c1[t]}, and
  \eqn{N_t}{N[t]}  are non-negative integers that are
  observed. The interior cell entries are not observed. It is
  assumed that \eqn{Y_{0t}|r_{0t} \sim \mathcal{B}inomial(r_{0t},
    p_{0t})}{Y0[t]|r0[t] ~ Binomial(r0[t], p0[t])} and 
  \eqn{Y_{1t}|r_{1t} \sim \mathcal{B}inomial(r_{1t}, p_{1t})}{Y1[t]|r1[t] ~
    Binomial(r1[t],p1[t])}. Let \eqn{\theta_{0t} =
    log(p_{0t}/(1-p_{0t}))}{theta0[t] = log(p0[t]/(1-p0[t]))},
  and  \eqn{\theta_{1t} = log(p_{1t}/(1-p_{1t}))}{theta1[t] =
    log(p1[t]/(1-p1[t]))}.

   The following prior distributions are
  assumed: \eqn{\theta_{0t} \sim \mathcal{N}(\mu_0,
    \sigma^2_0)}{\theta0[t] ~ Normal(mu0, sigma^2_0)},
  \eqn{\theta_{1t} \sim \mathcal{N}(\mu_1,
    \sigma^2_1)}{\theta1[t] ~ Normal(mu1, sigma^2_1)}.
  \eqn{\theta_{0t}}{theta0[t]} is assumed to be a priori independent of
  \eqn{\theta_{1t}}{theta1[t]} for all t.
  In addition, we assume the
  following hyperpriors:
  \eqn{\mu_0 \sim \mathcal{N}(m_0,
    M_0)}{mu0 ~ Normal(m0, M0)},
  \eqn{\mu_1 \sim \mathcal{N}(m_1,
    M_1)}{mu1 ~ Normal(m1,
    M1)},
  \eqn{\sigma^2_0 \sim \mathcal{IG}(\nu_0/2, \delta_0/2)}{\sigma^2_0 ~
    InvGamma(nu0/2, delta0/2)}, and
  \eqn{\sigma^2_1 \sim \mathcal{IG}(\nu_1/2, \delta_1/2)}{\sigma^2_1 ~
    InvGamma(nu1/2, delta1/2)}.

   Inference centers on \eqn{p_0}{p0}, \eqn{p_1}{p1}, \eqn{\mu_0}{mu0},
   \eqn{\mu_1}{mu1}, \eqn{\sigma^2_0}{sigma^2_0}, and
   \eqn{\sigma^2_1}{sigma^2_1}.
   The Metropolis-Hastings algorithm is used to sample from the posterior
   density. 
}
  
\references{
  Jonathan Wakefield. 2001. ``Ecological Inference for 2 x 2 Tables."
   Center for Statistics and the Social Sciences Working Paper
   no. 12. University of Washington. 

   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003. 
    \emph{Scythe Statistical 
   Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}

\examples{
   \dontrun{
   c0 <- rpois(5, 500)
   c1 <- c(200, 140, 250, 190, 75)
   r0 <- rpois(5, 400)
   r1 <- (c0 + c1) - r0
   posterior <- MCMChierEI(r0,r1,c0,c1, mcmc=200000, thin=50)
   plot(posterior)
   summary(posterior) 
   }
}

\keyword{models}

\seealso{\code{\link{MCMCbaselineEI}}, \code{\link{MCMCdynamicEI}},
  \code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}}


\eof
\name{MCMCirt1d}
\alias{MCMCirt1d}
\title{Markov chain Monte Carlo for One Dimensional Item Response Theory
   Model}
\description{
  This function generates a posterior density sample from a one
  dimentional item response theory (IRT) model, with Normal
  priors on the subject abilities (ideal points), and
  multivariate Normal priors on the item parameters.  The user
  supplies data and priors, and a sample from the posterior
  density is returned as an mcmc object, which can be
  subsequently analyzed with functions provided in the coda
  package.

  \emph{NOTE:} This implementation of this model
  may be deprecated in future releases.  We have implemented
  a general K-dimensional item response theory model that
  allows the user to place arbitrary constraints on item and
  subject parameters.  
  }
  
\usage{
MCMCirt1d(datamatrix, theta.fixed = 1, burnin = 500, mcmc = 1000,
   thin=5, verbose = FALSE, seed = 0, theta.start = NA, 
   alpha.start = NA, beta.start = NA, t0 = 0, T0 = 1, b0.alpha = 0,
   b0.beta = 0, B0.alpha = 1, B0.beta = 1, B0.corr = 0,
   store.item = FALSE, ... )  }

\arguments{
    \item{datamatrix}{The matrix of data.  Must be 0, 1, or missing values.  
    It is of dimensionality items by subjects.}

    \item{theta.fixed}{Identifying restriction.  This is the
    subject whose subject ability (ideal point) is constrained to
    be negative.  It makes most sense to choose someone who is
    extreme on the latent scale.  Make sure to check the
    posterior density sample to ensure the sampler is sampling
    from only one posterior mode.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Gibbs iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    Gibbs iterations must be divisible by this value.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the subject abilities (ideal points) are printed to the screen.}
    
    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{theta.start}{The starting values for the subject
    abilities (ideal points).   This can either be a scalar or a
    column vector with dimension equal to the number of voters.  
    If this takes a scalar value, then that value will serve as
    the starting value for all of the thetas.  The default value
    of NA will choose the starting values randomly.}

    \item{alpha.start}{The starting values for the
    \eqn{\alpha}{alpha} difficulty parameters. This can either be
    a scalar or a column vector with dimension equal to the
    number of items.   If this takes a scalar value, then that
    value will serve as the starting value for all of the alphas.
     The default value of NA will choose the starting values
    randomly.}

    \item{beta.start}{The starting values for the
    \eqn{\beta}{beta} discrimination parameters. This can either
    be a scalar or a column vector with dimension equal to the
    number of items.   If this takes a scalar value, then that
    value will serve as the starting value for all of the betas. 
    The default value of NA will choose the starting values
    randomly.}
    
    \item{t0}{The prior means of the subject abilities (ideal
    points), stacked for all subjects.  This can either be a
    scalar or a column vector with dimension equal to the number
    of thetas. If this takes a scalar value, then that value will
    serve as the prior mean for all of the betas.}
    
    \item{T0}{The prior variances of the subject abilities (ideal
    points), stacked for all subjects.   This can either be a
    scalar or a column vector with dimension equal to the number
    of thetas. If this takes a scalar  value, then that value
    will serve as the prior variance for all of the thetas.}
    
    \item{b0.alpha}{The prior means of the difficulty parameters,
    stacked for all items. This can either be a scalar or a
    column vector with dimension equal to the number of alphas.
    If this takes a scalar value, then that value will serve as
    the prior mean for all of the alphas.}

    \item{b0.beta}{The prior means of the discrimination
    parameters, stacked for all items. This can either be a
    scalar or a column vector with dimension equal to the number
    of betas. If this takes a scalar value, then that value will
    serve as the prior mean for all of the betas.}
    
    \item{B0.alpha}{The prior variances of the difficulty
    parameters, stacked for all items. This can either be a
    scalar or a column vector with dimension equal to the number
    of alphas. If this takes a scalar  value, then that value
    will serve as the prior variance for all of the alphas.}
    
    \item{B0.beta}{The prior variances of the discrimination
    parameters, stacked for all items. This can either be a
    scalar or a column vector with dimension equal to the number
    of betas. If this takes a scalar  value, then that value will
    serve as the prior variance for all of the betas.}
    
    \item{B0.corr}{The prior correlations of the difficulty and
    discrimination parameters, stacked for all items.  These are
    converted into covariances to complete the multivariate
    Normal prior.  This can either be a scalar or a column vector
    with dimension equal to the number of items. If this takes a
    scalar  value, then that value will serve as the prior
    correlation for all of the items.}
    
    \item{store.item}{A switch that determines whether or not to
    store the item parameters for posterior analysis. 
    \emph{NOTE: This takes an enormous amount of memory, so
    should only be used if the chain is thinned heavily, or for
    applications with a small number of items}.  By default, the
    item parameters are not stored.}
    
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
  \code{MCMCirt1d} simulates from the posterior density using
  standard Gibbs sampling using data augmentation (a Normal draw
  for the subject abilities, a multivariate Normal
  draw for the item parameters, and a truncated Normal draw for
  the latent utilities). The simulation proper is done in
  compiled C++ code to maximize efficiency.  Please consult the
  coda documentation for a comprehensive list of functions that
  can be used to analyze the posterior density sample.
  
  The default number of burnin and mcmc iterations is much
  smaller than the typical default values in MCMCpack.  This is
  because fitting this model is extremely computationally
  expensive.  It does not mean that this small of a number of
  scans will yield good estimates. If the verbose option is
  chosen, output will be printed to the screen every fifty
  iterations.  The priors of this model need to be proper for
  identification purposes.  The user is asked to provide prior
  means and variances \emph{(not precisions)} for the item
  parameters and the subject parameters.
  
  The model takes the following form.  We assume that each
  subject has an subject ability (ideal point) denoted
  \eqn{\theta_j}{theta_j} and that each item has a difficulty
  parameter \eqn{\alpha_i}{alpha_i} and discrimination parameter
  \eqn{\beta_i}{beta_i}.  The observed choice by subject
  \eqn{j}{j} on item \eqn{i}{i} is the observed data matrix which
  is \eqn{(I \times J)}{(I * J)}.  We assume that the choice is
  dictated by an unobserved utility: \deqn{z_{i,j} = \alpha_i +
  \beta_i \theta_j + \varepsilon_{i,j}}{z_ij = alpha_i +
  beta_i*theta_j + epsilon_ij} Where the errors are assumed to be
  distributed standard Normal.  The parameters of interest are
  the subject abilities (ideal points) and the item parameters.

  We assume the following priors.  For the subject abilities (ideal points):
  \deqn{\theta_j \sim \mathcal{N}(t_{0,j},T_{0,j})}{theta_j ~ N(t_0j, T0j)}
  Note that this implies a separate prior mean and variance for each
  subject.  For the item parameters, the prior is:
  \deqn{\left[\alpha_i, \beta_i \right]' \sim \mathcal{N}_2 
  (b_{0,i},B_{0,i})}{[alpha_i beta_i]' ~ N_2 (b_0,i, B_0,i)}
  Again, there is a separate prior for each item parameter.  When 
  supplying priors of the item parameters to the function, the user provides 
  each element of the mean vector and the covariance matrix (taking 
  correlations instead of covariances for convenience).
  
  The model is identified by the proper priors on the subject
  abilities (ideal points) and item parameters.  The
  \code{theta.fixed} solves the rotational invariance problem by
  constraining the sampler to one of the two identical posterior
  modes.  This function differs from the \code{MCMCirtKd} function
  in the manner in which the model is identified and in the assumed
  priors.
  }
  
\references{
   James H. Albert. 1992. ``Bayesian Estimation of Normal Ogive Item Response 
   Curves Using Gibbs Sampling." \emph{Journal of Educational Statistics}.  
   17: 251-269.
   
   Joshua Clinton, Simon Jackman, and Douglas Rivers. 2000. ``The Statistical 
   Analysis of Legislative Behavior: A Unified Approach." Paper presented at 
   the Annual Meeting of the Political Methodology Society.
   
   Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
   Springer: New York.

   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   data(SupremeCourt)
   posterior1 <- MCMCirt1d(SupremeCourt, burnin=10000, mcmc=50000)
   plot(posterior1)
   summary(posterior1)
   
   data(Senate)
   posterior2 <- MCMCirt1d(t(Senate[,6:677]), theta.fixed = 9,
      burnin=1000, mcmc=5000)
   plot(posterior2)
   summary(posterior2)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},
\code{\link[MCMCpack]{MCMCirtKd}}}


\eof
\name{MCMCirtKd}
\alias{MCMCirtKd}
\title{Markov chain Monte Carlo for K-Dimensional Item Response Theory
   Model}
\description{
  This function generates a posterior density sample from a
  K-dimensional item response theory (IRT) model, with standard
  Normal priors on the subject abilities (ideal points), and
  Normal priors on the item parameters.  The user
  supplies data and priors, and a sample from the posterior
  density is returned as an mcmc object, which can be
  subsequently analyzed with functions provided in the coda
  package.
}
  
\usage{
MCMCirtKd(datamatrix, dimensions, item.constraints=list(),
   burnin = 1000, mcmc = 10000, thin=5, verbose = FALSE, seed = 0,
   alphabeta.start = NA, b0 = 0, B0=0, store.item = FALSE,
   store.ability=TRUE, drop.constantvars=TRUE, ... )  }

\arguments{
    \item{datamatrix}{The matrix of data.  Must be 0, 1, or missing values.  
    It is of dimensionality items by subjects.}

    \item{dimensions}{The number of dimensions in the latent space.}
    
    \item{item.constraints}{List of lists specifying possible equality
    or simple inequality constraints on the item parameters. A typical
    entry in the list has one of three forms: \code{rowname=list(d,c)}
    which will constrain the dth item parameter for the item named
    rowname to be equal to c, \code{rowname=list(d,"+")} which will
    constrain the dth item parameter for the item named rowname to be
    positive, and\code{rowname=list(d, "-")} which will constrain the dth
    item parameter for the item named varname to be negative. If x is a
    matrix without row names defaults names of ``V1", ``V2", ... , etc
    will be used. In a d dimensional model, the first item parameter for
    item \eqn{i}{i} is the difficulty parameter (\eqn{\alpha_i}{alpha_i}),
    the second item parameter is the discrimation parameter on dimension
    1 (\eqn{\beta_{i,1}{beta_{i,1}}}), the third item parameter is the
    discrimation parameter on dimension 2
    (\eqn{\beta_{i,2}{beta_{i,2}}}), ...,  and the (d+1)th item parameter
    is the discrimation parameter on dimension d
    (\eqn{\beta_{i,1}{beta_{i,1}}}). 
    The item difficulty parameters (\eqn{\alpha}{alpha}) should
    generally not be constrained. 
    }

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    iterations must be divisible by this value.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the subject abilities (ideal points) are printed to the screen.}
    
    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{alphabeta.start}{The starting values for the
    \eqn{\alpha}{alpha} and \eqn{\beta}{beta} difficulty and
    discrimination parameters. If \code{alphabeta.start} is set to a
    scalar the starting value for all unconstrained item parameters will
    be set to that scalar. If \code{alphabeta.start} is a matrix of
    dimension \eqn{(K+1) \times items}{(K+1) x items} then the
    \code{alphabeta.start} matrix is used as the starting values (except
    for equality-constrained elements). If \code{alphabeta.start} is set
    to \code{NA} (the default) then starting values for unconstrained
    elements are set to values generated from a series of proportional
    odds logistic regression fits, and starting values for inequality
    constrained elements are set to either 1.0 or -1.0 depending on the
    nature of the constraints. }

    \item{b0}{The prior means of the
    \eqn{\alpha}{alpha} and \eqn{\beta}{beta} difficulty and
    discrimination parameters, stacked for all items.
    If a scalar is passed, it
    is used as the prior mean for all items.}
    
    \item{B0}{The prior precisions (inverse variances) of the
    independent Normal prior on the item parameters.
    Can be either a scalar or a matrix of dimension
    \eqn{(K+1) \times items}{(K+1) x items}.}

    \item{store.item}{A switch that determines whether or not to
    store the item parameters for posterior analysis. 
    \emph{NOTE: This takes an enormous amount of memory, so
    should only be used if the chain is thinned heavily, or for
    applications with a small number of items}.  By default, the
    item parameters are not stored.}
 
   \item{store.ability}{A switch that determines whether or not to store
    the subject abilities for posterior analysis.  By default, the
    item parameters are all stored.}

  \item{drop.constantvars}{A switch that determines whether or not
        items and subjects that have no variation
	should be deleted before fitting the model. Default = TRUE.}
  
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
  \code{MCMCirtKd} simulates from the posterior density using
  standard Gibbs sampling using data augmentation (a Normal draw
  for the subject abilities, a multivariate Normal
  draw for the item parameters, and a truncated Normal draw for
  the latent utilities). The simulation proper is done in
  compiled C++ code to maximize efficiency.  Please consult the
  coda documentation for a comprehensive list of functions that
  can be used to analyze the posterior density sample.
  
  The default number of burnin and mcmc iterations is much
  smaller than the typical default values in MCMCpack.  This is
  because fitting this model is extremely computationally
  expensive.  It does not mean that this small of a number of
  scans will yield good estimates. If the verbose option is
  chosen, output will be printed to the screen every fifty
  iterations.  The priors of this model need to be proper for
  identification purposes.  The user is asked to provide prior
  means and precisions \emph{(not variances)} for the item
  parameters and the subject parameters.
  
  The model takes the following form.  We assume that each
  subject has an subject ability (ideal point) denoted
  \eqn{\theta_j}{theta_j} \eqn{(K \times 1)}{(K x 1)},
  and that each item has a difficulty
  parameter \eqn{\alpha_i}{alpha_i} and discrimination parameter
  \eqn{\beta_i}{beta_i} \eqn{(K \times 1)}{(K x 1)}.
  The observed choice by subject
  \eqn{j}{j} on item \eqn{i}{i} is the observed data matrix which
  is \eqn{(I \times J)}{(I * J)}.  We assume that the choice is
  dictated by an unobserved utility: \deqn{z_{i,j} = \alpha_i +
  \beta_i' \theta_j + \varepsilon_{i,j}}{z_ij = alpha_i +
  beta_i'*theta_j + epsilon_ij} Where the errors are assumed to be
  distributed standard Normal.  The parameters of interest are
  the subject abilities (ideal points) and the item parameters.

  We assume the following priors.  For the subject abilities (ideal points)
  we assume independent standard Normal priors:
  \deqn{\theta_{j,k} \sim \mathcal{N}(0,1)}{theta_j,k ~ N(0, 1)}
  These cannot be changed by the user.
  For each item parameter, we assume independent Normal priors:
  \deqn{\left[\alpha_i, \beta_i \right]' \sim \mathcal{N}_{(K+1)} 
  (b_{0,i},B_{0,i})}{[alpha_i beta_i]' ~ N_(K+1) (b_0,i, B_0,i)}
  Where \eqn{B_{0,i}}{B_0,i} is a diagonal matrix.
  One can specify a separate prior mean and precision
  for each item parameter.
  
  The model is identified by the constraints on the item parameters
  (see Jackman 2001).  The user cannot place constraints on the subect
  abilities.  This identification scheme differs from that in
  \code{MCMCirt1d}, which uses a single directional constraint on
  one subject ability.  However, in our experience, using subject 
  ability constraints for models in greater than one dimension does not work 
  particularly well.
  }
  
\references{
   James H. Albert. 1992. ``Bayesian Estimation of Normal Ogive Item Response 
   Curves Using Gibbs Sampling." \emph{Journal of Educational Statistics}.  
   17: 251-269.
   
   Joshua Clinton, Simon Jackman, and Douglas Rivers. 2000. ``The Statistical 
   Analysis of Legislative Behavior: A Unified Approach." Paper presented at 
   the Annual Meeting of the Political Methodology Society.
   
   Simon Jackman. 2001. ``Multidimensional Analysis of Roll Call Data
   via Bayesian Simulation.'' \emph{Political Analysis.} 9: 227-241.
   
   Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
   Springer: New York.

   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   data(SupremeCourt)
   # note that the rownames (the item names) are "1", "2", etc
   posterior1 <- MCMCirtKd(SupremeCourt, dimensions=1,
                   burnin=5000, mcmc=50000, thin=10,
                   B0=.25, store.item=TRUE,
                   item.constraints=list("1"=list(2,"-")))
   plot(posterior1)
   summary(posterior1)


   data(Senate)
   rownames(Senate) <- Senate$member
   # note that we need to transpose the data to get
   # the bills on the rows
   posterior2 <- MCMCirtKd(t(Senate[,6:677]), dimensions=2,
                   burnin=5000, mcmc=50000, thin=10,
                   item.constraints=list(rc2=list(2,"-"), rc2=c(3,0),
                                         rc3=list(3,"-")),
                   B0=.25)
   plot(posterior2)
   summary(posterior2)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},
\code{\link[MCMCpack]{MCMCirt1d}}, \code{\link[MCMCpack]{MCMCordfactanal}}
}


\eof
\name{MCMClogit}
\alias{MCMClogit}
\title{Markov chain Monte Carlo for Logistic Regression}
\description{
  This function generates a posterior density sample
  from a logistic regression model using a random walk Metropolis
  algorithm. The user supplies data and priors,
  and a sample from the posterior density is returned as an mcmc
  object, which can be subsequently analyzed with functions 
  provided in the coda package.
  }
  
\usage{
MCMClogit(formula, data = list(), burnin = 1000, mcmc = 10000,
   thin=5, tune=1.1, verbose = FALSE, seed = 0,  beta.start = NA,
   b0 = 0, B0 = 0.001, ...) }

\arguments{
    \item{formula}{Model formula.}

    \item{data}{Data frame.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Metropolis iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

    \item{tune}{Metropolis tuning parameter.  Make sure that the
    acceptance rate is satisfactory before using the posterior density sample
    for inference.}
    
    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the betas are printed to the screen every 500 iterations.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{beta.start}{The starting value for the \eqn{\beta}{beta} vector.
    This can either 
    be a scalar or a column vector with dimension equal to the number of 
    betas. If this takes a scalar value, then that value will serve as the 
    starting value for all of the betas.  The default value of NA will
    use the maximum likelihood estimate of \eqn{\beta}{beta} as the starting 
    value.}

    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a column      
    vector with dimension equal to the number of betas. If this takes a scalar
    value, then that value will serve as the prior mean for all of the
    betas.}
    
    \item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a
    scalar
    or a square matrix with dimensions equal to the number of betas.  If this
    takes a scalar value, then that value times an identity matrix serves
    as the prior precision of \eqn{\beta}{beta}.}
    
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{\code{MCMClogit} simulates from the posterior density of a logistic
  regression model using a random walk Metropolis algorithm. The simulation
  proper is done in compiled C++ code to maximize efficiency.  Please consult
  the coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample.
  
  The model takes the following form:
  \deqn{y_i \sim \mathcal{B}ernoulli(\pi_i)}{y_i ~ Bernoulli(pi_i)}
  Where the inverse link function:
  \deqn{\pi_i = \frac{\exp(x_i'\beta)}{1 + \exp(x_i'\beta)}}{pi_i = 
    exp(x_i'beta) / [1 + exp(x_i'beta)]}
  We assume a multivariate Normal prior on \eqn{\beta}{beta}:
    \deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}{beta ~ N(b0,B0^(-1))}
  The candidate generating density is a multivariate Normal density centered
  at the current value of \eqn{\beta}{beta} with variance-covariance matrix
  that is an approximation of the posterior based on the maximum likelihood 
  estimates and the prior precision multiplied by the tuning parameter  
   squared.
  }
  
\references{
      
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   data(birthwt)
   posterior <- MCMClogit(low~age+as.factor(race)+smoke, data=birthwt)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}, \code{\link[base]{glm}}}


\eof
\name{MCMCmixfactanal}
\alias{MCMCmixfactanal}
\title{Markov chain Monte Carlo for Mixed Data Factor Analysis Model}
\description{
  This function generates a posterior density sample from a mixed data
  (both continuous and ordinal) factor analysis model. Normal priors are
  assumed on the factor loadings and factor scores, improper
  uniform priors are assumed on the cutpoints, and inverse gamma priors
  are assumed for the error variances (uniquenesses). The user supplies
  data and parameters for the prior distributions, and a sample from the
  posterior density is returned as an mcmc object, which can be
  subsequently analyzed with functions provided in the coda package.
}
  
\usage{
MCMCmixfactanal(x, factors, lambda.constraints=list(),
                data=list(), burnin = 1000, mcmc = 10000,
                thin=5, tune=NA, verbose = FALSE, seed = 0,
                lambda.start = NA, psi.start=NA,
                l0=0, L0=0, a0=0.001, b0=0.001,
                store.lambda=TRUE, store.scores=FALSE,
                std.mean=TRUE, std.var=TRUE, ... )
 }

\arguments{
  \item{x}{A one-sided formula containing the
    manifest variables. Ordinal (including dichotomous) variables must
    be coded as ordered factors. NOTE: data input is different in
    \code{MCMCmixfactanal} than in either \code{MCMCfactanal} or
    \code{MCMCordfactanal}.}

  \item{factors}{The number of factors to be fitted.}
  
  \item{lambda.constraints}{List of lists specifying possible equality
    or simple inequality constraints on the factor loadings. A typical
    entry in the list has one of three forms: \code{varname=list(d,c)} which
    will constrain the dth loading for the variable named varname to
    be equal to c, \code{varname=list(d,"+")} which will constrain the dth
    loading for the variable named varname to be positive, and
    \code{varname=list(d, "-")} which will constrain the dth loading for the
    variable named varname to be negative. If x is a matrix without
    column names defaults names of ``V1", ``V2", ... , etc will be
    used. Note that, unlike \code{MCMCfactanal}, the
    \eqn{\Lambda}{Lambda} matrix used here has \code{factors}+1
    columns. The first column of \eqn{\Lambda}{Lambda} corresponds to
    negative item difficulty parameters for ordinal manifest variables
    and mean parameters for continuous manifest variables and should
    generally not be constrained directly by the user.  
    }
  
  \item{data}{A data frame.}
  
  \item{burnin}{The number of burn-in iterations for the sampler.}
  
  \item{mcmc}{The number of iterations for the sampler.}
  
  \item{thin}{The thinning interval used in the simulation.  The number of
    iterations must be divisible by this value.}

  \item{tune}{The tuning parameter for the Metropolis-Hastings
    sampling. Can be either a scalar or a \eqn{k}{k}-vector (where
    \eqn{k}{k} is the number of manifest variables). \code{tune} must be
    strictly positive.}
  
  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the Metropolis-Hastings acceptance rate are printed to the screen.}
  
  \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}
  
  \item{lambda.start}{Starting values for the factor loading matrix
    Lambda. If \code{lambda.start} is set to a scalar the starting value for
    all unconstrained loadings will be set to that scalar. If
    \code{lambda.start} is a matrix of the same dimensions as Lambda then the
    \code{lambda.start} matrix is used as the starting values (except
    for equality-constrained elements). If \code{lambda.start} is set to
    \code{NA} (the default) then starting values for unconstrained
    elements in the first column of Lambda are based on the observed
    response pattern, the remaining unconstrained elements of Lambda are
    set to 0, and starting values for inequality constrained elements
    are set to either 1.0 or -1.0 depending on the  nature of the
    constraints.} 

  \item{psi.start}{Starting values for the error variance (uniqueness)
    matrix. If  \code{psi.start} is set to a scalar then the starting
    value for all diagonal elements of \code{Psi} that represent error
    variances for continuous variables are set to this value. If
    \code{psi.start} is a \eqn{k}{k}-vector (where \eqn{k}{k} is the
    number of manifest variables) then the staring value of \code{Psi}
    has \code{psi.start} on the main diagonal with the exception that
    entries corresponding to error variances for ordinal variables are
    set to 1.. If \code{psi.start} is set to \code{NA} (the default) the
    starting values of all the continuous variable uniquenesses are set
    to 0.5. Error variances for ordinal response variables are always
    constrained (regardless of the value of \code{psi.start} to have an
    error variance of 1 in order to achieve identification.} 
  
  \item{l0}{The means of the independent Normal prior on the factor
    loadings. Can be either a scalar or a matrix with the same
    dimensions as \code{Lambda}.}

  \item{L0}{The precisions (inverse variances) of the independent Normal
    prior on the factor loadings. Can be either a scalar or a matrix with
    the same dimensions as \code{Lambda}.}

  \item{a0}{Controls the shape of the inverse Gamma prior on the
    uniqueness. The actual shape parameter is set to \code{a0/2}. Can be
    either a scalar or a \eqn{k}{k}-vector.} 

  \item{b0}{Controls the scale of the inverse Gamma prior on the
    uniquenesses. The actual scale parameter is set to \code{b0/2}. Can
    be either a scalar or a \eqn{k}{k}-vector.} 

  \item{store.lambda}{A switch that determines whether or not to store
    the factor loadings for posterior analysis. By default, the factor
    loadings are all stored.}
  
  \item{store.scores}{A switch that determines whether or not to
    store the factor scores for posterior analysis. 
    \emph{NOTE: This takes an enormous amount of memory, so
      should only be used if the chain is thinned heavily, or for
      applications with a small number of observations}.  By default, the
    factor scores are not stored.}

  \item{std.mean}{If \code{TRUE} (the default) the continuous manifest
    variables are rescaled to have zero mean.}
    
  \item{std.var}{If \code{TRUE} (the default) the continuous manifest
    variables are rescaled to have unit variance.}
    
  \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{The model takes the following form:

  Let \eqn{i=1,\ldots,N}{1=1,...,n} index observations and
  \eqn{j=1,\ldots,K}{j=1,...,K} index response variables within an
  observation. An observed
  variable \eqn{x_{ij}}{x_ij} can be either ordinal with a
  total of \eqn{C_j}{C_j}  
  categories or continuous.
  The distribution of \eqn{X}{X} is governed by a \eqn{N
    \times K}{N by K} matrix of latent variables \eqn{X^*}{Xstar} and a
  series of cutpoints \eqn{\gamma}{gamma}. \eqn{X^*}{Xstar} is assumed
  to be generated according to:
  
  \deqn{x^*_i = \Lambda \phi_i + \epsilon_i}{xstar_i = Lambda phi_i +
    epsilon_i}
  \deqn{\epsilon_i \sim \mathcal{N}(0,\Psi)}{epsilon_i ~ N(0, Psi)}

  where \eqn{x^*_i}{xstar_i} is the \eqn{k}{k}-vector of latent variables
  specific to observation \eqn{i}{i}, \eqn{\Lambda}{Lambda} is the
  \eqn{k \times d}{k by d} matrix of factor loadings, and
  \eqn{\phi_i}{phi_i} is
  the \eqn{d}{d}-vector of latent factor scores. It is assumed that the
  first element of \eqn{\phi_i}{phi_i} is equal to 1 for all
  \eqn{i}{i}. 

  If the \eqn{j}{j}th variable is ordinal, the probability that it takes
  the value \eqn{c}{c} in observation \eqn{i}{i} is:

   \deqn{
     \pi_{ijc} = \Phi(\gamma_{jc} - \Lambda'_j\phi_i) -
     \Phi(\gamma_{j(c-1)} - \Lambda'_j\phi_i)
   }{
     pi_ijc = pnorm(gamma_jc - Lambda'_j phi_i) -
     pnorm(gamma_j(c-1) - Lambda'_j phi_i)
   }

  If the \eqn{j}{j}th variable is continuous, it is assumed that
  \eqn{x^*_{ij} = x_{ij}}{xstar_{ij} = x_{ij}} for all \eqn{i}{i}. 
   
  The implementation used here assumes independent conjugate priors for
  each element of \eqn{\Lambda}{Lambda} and each
  \eqn{\phi_i}{phi_i}. More specifically we assume:

  \deqn{\Lambda_{ij} \sim \mathcal{N}(l_{0_{ij}}, L_{0_{ij}}^{-1}),
    i=1,\ldots,k,  j=1,\ldots,d}{Lambda_ij ~ N(l0_ij,  L0_ij^-1),
    i=1,...,k, j=1,...,d} 

  \deqn{\phi_{i(2:d)} \sim \mathcal{N}(0, I),
    i=1,\dots,n}{phi_i(2:d) ~ N(0, I),
      i=1,...,n} 

  \code{MCMCmixfactanal} simulates from the posterior density using
  a Metropolis-Hastings within Gibbs sampling algorithm. The algorithm
  employed is based on work by Cowles (1996).  Note that
  the first element of \eqn{\phi_i}{phi_i} is a 1. As a result, the
  first column of \eqn{\Lambda}{Lambda} can be interpretated as negative
  item difficulty parameters.  Further, the first
  element  \eqn{\gamma_1}{gamma_1} is normalized to zero, and thus not 
  returned in the mcmc object.
  The simulation proper is done in compiled C++ code to maximize
  efficiency.  Please consult the coda documentation for a comprehensive
  list of functions that can be used to analyze the posterior density
  sample. 
  }
}

\references{
  Kevin M. Quinn. N.D. ``Bayesian Factor Analysis for Mixed Ordinal and
  Continuous Responses". Typescript Harvard University.
  
  M. K. Cowles. 1996. ``Accelerating Monte Carlo Markov Chain Convergence for
  Cumulative-link Generalized Linear Models." \emph{Statistics and Computing.}
  6: 101-110.
     
  Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
  Springer: New York.
  
   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
\dontrun{
data(Cars93)
attach(Cars93)
new.cars <- data.frame(Price, MPG.city, MPG.highway,
                 Cylinders, EngineSize, Horsepower,
                 RPM, Length, Wheelbase, Width, Weight, Origin)
rownames(new.cars) <- paste(Manufacturer, Model)
detach(Cars93)

# drop obs 57 (Mazda RX 7) b/c it has a rotary engine
new.cars <- new.cars[-57,]
# drop 3 cylinder cars
new.cars <- new.cars[new.cars$Cylinders!=3,]
# drop 5 cylinder cars
new.cars <- new.cars[new.cars$Cylinders!=5,]

new.cars$log.Price <- log(new.cars$Price)
new.cars$log.MPG.city <- log(new.cars$MPG.city)
new.cars$log.MPG.highway <- log(new.cars$MPG.highway)
new.cars$log.EngineSize <- log(new.cars$EngineSize)
new.cars$log.Horsepower <- log(new.cars$Horsepower)

new.cars$Cylinders <- ordered(new.cars$Cylinders)
new.cars$Origin    <- ordered(new.cars$Origin)



posterior <- MCMCmixfactanal(~log.Price+log.MPG.city+
                 log.MPG.highway+Cylinders+log.EngineSize+
                 log.Horsepower+RPM+Length+
                 Wheelbase+Width+Weight+Origin, data=new.cars,
                 lambda.constraints=list(log.Horsepower=list(2,"+"),
                 log.Horsepower=c(3,0), weight=list(3,"+")),
                 factors=2,
                 burnin=5000, mcmc=500000, thin=100, verbose=TRUE,
                 L0=.25, tune=1.5)


}
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}}, \code{\link[coda]{summary.mcmc}},
  \code{\link[mva]{factanal}}, \code{\link[MCMCpack]{MCMCfactanal}},
  \code{\link[MCMCpack]{MCMCordfactanal}},
  \code{\link[MCMCpack]{MCMCirt1d}}, \code{\link[MCMCpack]{MCMCirtKd}}}


\eof
\name{MCMCoprobit}
\alias{MCMCoprobit}
\title{Markov chain Monte Carlo for Ordered Probit Regression}
\description{
  This function generates a posterior density sample
  from an ordered probit regression model using the data augmentation 
  approach of Cowles (1996). The user supplies data and priors,
  and a sample from the posterior density is returned as an mcmc
  object, which can be subsequently analyzed with functions 
  provided in the coda package.
  }
  
\usage{
MCMCoprobit(formula, data = list(), burnin = 1000, mcmc = 10000,
   thin=5, tune = NA, verbose = FALSE, seed = 0, beta.start = NA,
   b0 = 0, B0 = 0.001, ...) }

\arguments{
    \item{formula}{Model formula.}

    \item{data}{Data frame.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of MCMC iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    Gibbs iterations must be divisible by this value.}

    \item{tune}{The tuning parameter for the Metropolis-Hastings
      step. Default of NA corresponds to a choice of 0.05 divided by the
      number of categories in the response variable.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the betas are printed to the screen every 500 iterations.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}
  
    \item{beta.start}{The starting value for the \eqn{\beta}{beta} vector.   
    This can either be a scalar or a column vector with dimension equal
    to the number of betas. If this takes a scalar value, then that
    value will serve as the starting value for all of the betas. The
    default value of NA will use rescaled estimates from an ordered
    logit model.}

    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a column vector with dimension equal to the number of
    betas. If this takes a scalar value, then that value will serve as
    the prior mean for all of the betas.}
    
    \item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a 
    scalar or a square matrix with dimensions equal to the number of
    betas.  If this takes a scalar value, then that value times an
    identity matrix serves as the prior precision of \eqn{\beta}{beta}.}
    
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
\code{MCMCoprobit} simulates from the posterior density of a ordered probit
  regression model using data augmentation. The simulation proper is
  done in compiled C++ code to maximize efficiency.  Please consult the
  coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample. 
  
  The observed variable \eqn{y_i}{y_i} is ordinal with a total of \eqn{C}{C} 
  categories, with distribution
  governed by a latent variable:
  \deqn{z_i = x_i'\beta + \varepsilon_i}{z_i = x_i'beta + epsilon_i}
  The errors are assumed to be from a standard Normal distribution.  The 
  probabilities of observing each outcome is governed by this latent
  variable and \eqn{C-1}{C-1} estimable cutpoints, which are denoted
  \eqn{\gamma_c}{gamma_c}.  The probability that individual \eqn{i}{i}
  is in category \eqn{c}{c} is computed by:
  \deqn{
   \pi_{ic} = \Phi(\gamma_c - x_i'\beta) - \Phi(\gamma_{c-1} - x_i'\beta)
   }{
   pi_ic = Phi(gamma_c - x_i'beta) - Phi(gamma_(c-1) - x_i'beta)
   }
   These probabilities are used to form the multinomial distribution
   that defines the likelihoods.
   
   The algorithm employed is discussed in depth by Cowles (1996).  Note that 
   the model does include a constant in the data matrix.  Thus, the first
   element  \eqn{\gamma_1}{gamma_1} is normalized to zero, and is not 
   returned in the mcmc object.
}
  
\references{
  M. K. Cowles. 1996. ``Accelerating Monte Carlo Markov Chain Convergence for
  Cumulative-link Generalized Linear Models." \emph{Statistics and Computing.}
  6: 101-110.
     
  Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
  Springer: New York.
      
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
  Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}
}


\examples{
   \dontrun{
   x1 <- rnorm(100); x2 <- rnorm(100);
   z <- 1.0 + x1*0.1 - x2*0.5 + rnorm(100);
   y <- z; y[z < 0] <- 0; y[z >= 0 & z < 1] <- 1;
   y[z >= 1 & z < 1.5] <- 2; y[z >= 1.5] <- 3;
   posterior <- MCMCoprobit(y ~ x1 + x2, tune=0.3, mcmc=20000)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}}

\eof
\name{MCMCordfactanal}
\alias{MCMCordfactanal}
\title{Markov chain Monte Carlo for Ordinal Data Factor Analysis Model}
\description{
  This function generates a posterior density sample from an ordinal data
  factor analysis model. Normal priors are assumed on the factor
  loadings and factor scores while improper uniform priors are assumed
  on the cutpoints. The user supplies data and parameters for the prior
  distributions, and a sample from the posterior density is returned as
  an mcmc object, which can be subsequently analyzed with
  functions provided in the coda package.
}
  
\usage{
MCMCordfactanal(x, factors, lambda.constraints=list(),
                data=list(), burnin = 1000, mcmc = 10000,
                thin=5, tune=NA, verbose = FALSE, seed = 0,
                lambda.start = NA, l0=0, L0=0,
                store.lambda=TRUE, store.scores=FALSE,
                drop.constantvars=TRUE,
                drop.constantcases=FALSE, ... )
 }

\arguments{
  \item{x}{Either a formula or a numeric matrix containing the
    manifest variables.}

  \item{factors}{The number of factors to be fitted.}
  
  \item{lambda.constraints}{List of lists specifying possible equality
    or simple inequality constraints on the factor loadings. A typical
    entry in the list has one of three forms: \code{varname=list(d,c)} which
    will constrain the dth loading for the variable named varname to
    be equal to c, \code{varname=list(d,"+")} which will constrain the dth
    loading for the variable named varname to be positive, and
    \code{varname=list(d, "-")} which will constrain the dth loading for the
    variable named varname to be negative. If x is a matrix without
    column names defaults names of ``V1", ``V2", ... , etc will be
    used. Note that, unlike \code{MCMCfactanal}, the
    \eqn{\Lambda}{Lambda} matrix used here has \code{factors}+1
    columns. The first column of \eqn{\Lambda}{Lambda} corresponds to
    negative item difficulty parameters and should generally not be
    constrained. 
    }
  
  \item{data}{A data frame.}
  
  \item{burnin}{The number of burn-in iterations for the sampler.}
  
  \item{mcmc}{The number of iterations for the sampler.}
  
  \item{thin}{The thinning interval used in the simulation.  The number of
    iterations must be divisible by this value.}

  \item{tune}{The tuning parameter for the Metropolis-Hastings
    sampling. Can be either a scalar or a \eqn{k}{k}-vector. Must be
    strictly positive.}
  
  \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the Metropolis-Hastings acceptance rate are printed to the screen.}
  
  \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}
  
  \item{lambda.start}{Starting values for the factor loading matrix
    Lambda. If \code{lambda.start} is set to a scalar the starting value for
    all unconstrained loadings will be set to that scalar. If
    \code{lambda.start} is a matrix of the same dimensions as Lambda then the
    \code{lambda.start} matrix is used as the starting values (except
    for equality-constrained elements). If \code{lambda.start} is set to
    \code{NA} (the default) then starting values for unconstrained
    elements in the first column of Lambda are based on the observed
    response pattern, the remaining unconstrained elements of Lambda are
    set to , and starting values for inequality constrained elements are
    set to either 1.0 or -1.0 depending on the nature of the constraints.}
  
  \item{l0}{The means of the independent Normal prior on the factor
    loadings. Can be either a scalar or a matrix with the same
    dimensions as \code{Lambda}.}

  \item{L0}{The precisions (inverse variances) of the independent Normal
    prior on the factor loadings. Can be either a scalar or a matrix with
    the same dimensions as \code{Lambda}.}

  \item{store.lambda}{A switch that determines whether or not to store
    the factor loadings for posterior analysis. By default, the factor
    loadings are all stored.}
  
  \item{store.scores}{A switch that determines whether or not to
    store the factor scores for posterior analysis. 
    \emph{NOTE: This takes an enormous amount of memory, so
      should only be used if the chain is thinned heavily, or for
      applications with a small number of observations}.  By default, the
    factor scores are not stored.}

  \item{drop.constantvars}{A switch that determines whether or not
        manifest variables that have no variation should be deleted
	before fitting the model. Default = TRUE.}

  \item{drop.constantcases}{A switch that determines whether or not
        observations that have no variation should be deleted
	before fitting the model. Only useful when manifest variables
	are all dichotomous. Default = FALSE.}
  
  \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{The model takes the following form:

  Let \eqn{i=1,\ldots,N}{1=1,...,n} index observations and
  \eqn{j=1,\ldots,K}{j=1,...,K} index response variables within an
  observation. The typical observed
  variable \eqn{x_{ij}}{x_ij} is ordinal with a total of \eqn{C_j}{C_j}  
  categories. The distribution of \eqn{X}{X} is governed by a \eqn{N
    \times K}{N by K} matrix of latent variables \eqn{X^*}{Xstar} and a
  series of cutpoints \eqn{\gamma}{gamma}. \eqn{X^*}{Xstar} is assumed
  to be generated according to:
  
  \deqn{x^*_i = \Lambda \phi_i + \epsilon_i}{xstar_i = Lambda phi_i +
    epsilon_i}
  \deqn{\epsilon_i \sim \mathcal{N}(0,I)}{epsilon_i ~ N(0, I)}

  where \eqn{x^*_i}{xstar_i} is the \eqn{k}{k}-vector of latent variables
  specific to observation \eqn{i}{i}, \eqn{\Lambda}{Lambda} is the
  \eqn{k \times d}{k by d} matrix of factor loadings, and
  \eqn{\phi_i}{phi_i} is
  the \eqn{d}{d}-vector of latent factor scores. It is assumed that the
  first element of \eqn{\phi_i}{phi_i} is equal to 1 for all
  \eqn{i}{i}. 

  The probability that the \eqn{j}{j}th variable in observation
  \eqn{i}{i} takes the value \eqn{c}{c} is:

   \deqn{
     \pi_{ijc} = \Phi(\gamma_{jc} - \Lambda'_j\phi_i) -
     \Phi(\gamma_{j(c-1)} - \Lambda'_j\phi_i)
   }{
     pi_ijc = pnorm(gamma_jc - Lambda'_j phi_i) -
     pnorm(gamma_j(c-1) - Lambda'_j phi_i)
   }
  
  The implementation used here assumes independent conjugate priors for
  each element of \eqn{\Lambda}{Lambda} and each
  \eqn{\phi_i}{phi_i}. More specifically we assume:

  \deqn{\Lambda_{ij} \sim \mathcal{N}(l_{0_{ij}}, L_{0_{ij}}^{-1}),
    i=1,\ldots,k,  j=1,\ldots,d}{Lambda_ij ~ N(l0_ij,  L0_ij^-1),
    i=1,...,k, j=1,...,d} 

  \deqn{\phi_{i(2:d)} \sim \mathcal{N}(0, I),
    i=1,\dots,n}{phi_i(2:d) ~ N(0, I),
      i=1,...,n} 


    The standard two-parameter item response theory model with probit
    link is a special case of the model sketched above. 
    
  \code{MCMCordfactanal} simulates from the posterior density using
  a Metropolis-Hastings within Gibbs sampling algorithm. The algorithm
  employed is based on work by Cowles (1996).  Note that
  the first element of \eqn{\phi_i}{phi_i} is a 1. As a result, the
  first column of \eqn{\Lambda}{Lambda} can be interpretated as item
  difficulty parameters.  Further, the first
  element  \eqn{\gamma_1}{gamma_1} is normalized to zero, and thus not 
  returned in the mcmc object.
  The simulation proper is done in compiled C++ code to maximize
  efficiency.  Please consult the coda documentation for a comprehensive
  list of functions that can be used to analyze the posterior density
  sample. 
  }
}

\references{
  Shawn Treier and Simon Jackman. 2003. ``Democracy as a Latent Variable." 
  Paper presented at the Midwest Political Science Association Annual Meeting.

  M. K. Cowles. 1996. ``Accelerating Monte Carlo Markov Chain Convergence for
  Cumulative-link Generalized Linear Models." \emph{Statistics and Computing.}
  6: 101-110.
     
  Valen E. Johnson and James H. Albert. 1999. ``Ordinal Data Modeling." 
  Springer: New York.
  
   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   data(painters)
   new.painters <- painters[,1:4]
   cuts <- apply(new.painters, 2, quantile, c(.25, .50, .75))
   for (i in 1:4){
      new.painters[new.painters[,i]<cuts[1,i],i] <- 100
     new.painters[new.painters[,i]<cuts[2,i],i] <- 200
     new.painters[new.painters[,i]<cuts[3,i],i] <- 300
     new.painters[new.painters[,i]<100,i] <- 400
   }

   posterior <- MCMCordfactanal(~Composition+Drawing+Colour+Expression,
                        data=new.painters, factors=1,
                        lambda.constraints=list(Drawing=list(2,"+")),
                        burnin=5000, mcmc=500000, thin=200, verbose=TRUE,
                        L0=0.5, store.lambda=TRUE,
                        store.scores=TRUE, tune=.6)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}}, \code{\link[coda]{summary.mcmc}},
  \code{\link[mva]{factanal}}, \code{\link[MCMCpack]{MCMCfactanal}},
  \code{\link[MCMCpack]{MCMCirt1d}}, \code{\link[MCMCpack]{MCMCirtKd}}}


\eof
\name{MCMCpanel}
\alias{MCMCpanel}
\title{Markov chain Monte Carlo for the General Linear Panel Model}
\description{
  MCMCpanel generates a posterior density sample from a General
  Linear Panel Model using Algorithm 2 of Chib and Carlin (1999).
  This model uses a multivariate Normal prior for the fixed
  effects parameters, a Wishart prior on the random effects
  precision matrix, and a Gamma prior on the conditional error
  precision. The user supplies data and priors, and a sample from
  the posterior density is returned as an mcmc object,
  which can be subsequently analyzed with functions provided in
  the coda package.
  }
  
\usage{
MCMCpanel(obs, Y, X, W, burnin = 1000, mcmc = 10000, thin = 5, 
    verbose = FALSE, seed = 0, sigma2.start = NA,
    D.start = NA, b0 = 0, B0 = 1, eta0, R0, nu0 = 0.001,
    delta0 = 0.001, ...)
   }

\arguments{
    \item{obs}{An \eqn{(nk \times 1)}{(nk * 1)} vector that contains unique
    observation numbers for each subject.}

    \item{Y}{An \eqn{(nk \times 1)}{(nk * 1)} vector of response variables, 
    stacked across all subjects.}

    \item{X}{An \eqn{(nk \times p)}{(nk * p)} matrix of fixed effects
    covariates, stacked across all subjects.}

    \item{W}{An \eqn{(nk \times q)}{(nk * q)} matrix of random effects
    covariates, stacked across all subjects.}

    \item{burnin}{The number of burnin iterations for the sampler.}
    
    \item{mcmc}{The number of Gibbs iterations for the sampler.}
    
    \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number
    and parameters are printed to the screen.}

    \item{sigma2.start}{The starting value for the conditional error
    variance. Default value of NA uses the least squares estimates.}

    \item{D.start}{The starting value for precision matrix of the random 
    effects.  This can either be a scalar or square matrix with dimension
    equal to the number of random effects. If this takes a scalar value, then
    that value multiplied by an identity matrix will be the starting
    value. Default value of NA uses an identity matrix multiplied by 0.5
    the OLS \eqn{\sigma^2}{sigma2} estimate. }
    
    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a
    column vector with dimension equal to the number of betas. If this takes
    a scalar value, then that value will serve as the prior mean for all
    of the betas.} 
  
    \item{B0}{The prior precision of \eqn{\beta}{beta}.
    This can either be a scalar
    or a square matrix with dimensions equal to the number of betas.  If
    this takes  a scalar value, then that value times an identity matrix 
    serves as the prior precision of beta. Default value of 0 is equivalent
    to an improper uniform prior for beta.} 
    
    \item{eta0}{The shape parameter for the Wishart prior on
    precision matrix for the random effects.}
    
    \item{R0}{The scale matrix for the Wishart prior on precision matrix for
    the random effects.}
    
    \item{nu0}{The shape parameter for the Gamma prior on the
    conditional error precision.} 

    \item{delta0}{The scale  parameter for the Gamma prior on
    the conditional error precision.}

    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
  \code{MCMCpanel} simulates from the posterior density sample using 
  the blocked Gibbs sampler of Chib and Carlin (1999), Algorithm 2. 
  The simulation proper
  is done in compiled C++ code to maximize efficiency.  Please consult
  the coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample.
  
  The model takes the following form:
  \deqn{y_i = X_i \beta + W_i b_i + \varepsilon_i}{y_i = X_i * beta + W_i *
   b_i + epsilon_i}
  Where the random effects:
  \deqn{b_i \sim \mathcal{N}_q(0,D)}{b_i ~ N_q(0,D)}
  And the errors:
  \deqn{\varepsilon_i \sim \mathcal{N}(0, \sigma^2 I_k)}{epsilon_i ~ N(0,
   sigma^2 I_k)}
  We assume standard, conjugate priors:
  \deqn{\beta \sim \mathcal{N}(b0,B0^{-1})}{beta ~ N(b0,B0^(-1))}
  And:
  \deqn{\sigma^{-2} \sim \mathcal{G}amma(\nu_0/2, \delta_0/2)}{sigma^(-2) ~ 
  Gamma(nu0/2, delta0/2)}
  And:
  \deqn{D^{-1} \sim \mathcal{W}ishart(\eta_0, R_0^{-1})}{D^-1 ~ Wishart(eta0, 
   R0^-1)}
   See Chib and Carlin (1999) or Martin and Saunders (2002) for more details.
   
   \emph{NOTE: Unlike most models in MCMCpack, we do not provide default 
   parameters for the priors on the precision matrix for the random effects.}
   When fitting one of these models, it is of utmost importance to choose a 
   prior that reflects your prior beliefs about the random effects.  Using the
   \code{dwish} and \code{rwish} functions might be useful in choosing these
   values.  Also, the user is not allowed to specify a starting value
   for the \eqn{\beta}{beta} parameters, as they are simulated in the
   first block of the sampler.
   }
 
\references{
   Siddhartha Chib and Bradley P. Carlin. 1999. ``On MCMC Sampling in 
   Hierarchical Longitudinal Models." \emph{Statistics and Computing.} 9: 
   17-26.

   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  \emph{Scythe 
   Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Andrew D. Martin and Kyle L. Saunders. 2002. ``Bayesian Inference for 
   Political Science Panel Data.'' Paper presented at the 2002 Annual Meeting 
   of the American Political Science Association.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}}


\eof
\name{MCMCpoisson}
\alias{MCMCpoisson}
\title{Markov chain Monte Carlo for Poisson Regression}
\description{
  This function generates a posterior density sample
  from a Poisson regression model using a random walk Metropolis
  algorithm. The user supplies data and priors,
  and a sample from the posterior density is returned as an mcmc
  object, which can be subsequently analyzed with functions 
  provided in the coda package.
  }
  
\usage{
MCMCpoisson(formula, data = list(), burnin = 1000, mcmc = 10000,
   thin = 5, tune = 1.1, verbose = FALSE, seed = 0,  beta.start = NA,
   b0 = 0, B0 = 0.001, ...) }

\arguments{
    \item{formula}{Model formula.}

    \item{data}{Data frame.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Metropolis iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    mcmc iterations must be divisible by this value.}

    \item{tune}{Metropolis tuning parameter.  Make sure that the
    acceptance rate is satisfactory before using the posterior density sample
    for inference.}
    
    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the \eqn{\beta}{beta} is printed to the screen every 500 iterations.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{beta.start}{The starting value for the \eqn{\beta}{beta} vector.
    This can either 
    be a scalar or a column vector with dimension equal to the number of 
    betas. If this takes a scalar value, then that value will serve as the 
    starting value for all of the betas.  The default value of NA will
    use the maximum likelihood estimate of \eqn{\beta}{beta} as the starting 
    value.}

    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a column      
    vector with dimension equal to the number of betas. If this takes a scalar
    value, then that value will serve as the prior mean for all of the
    betas.}
    
    \item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a
    scalar
    or a square matrix with dimensions equal to the number of betas.  If this
    takes a scalar value, then that value times an identity matrix serves
    as the prior precision of \eqn{\beta}{beta}.}
    
    \item{...}{further arguments to be passed}       
}}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{\code{MCMCpoisson} simulates from the posterior density of a Poisson
  regression model using a random walk Metropolis algorithm. The simulation
  proper is done in compiled C++ code to maximize efficiency.  Please consult
  the coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample.
  
  The model takes the following form:
  \deqn{y_i \sim \mathcal{P}oisson(\mu_i)}{y_i ~ Poisson(mu_i)}
  Where the inverse link function:
  \deqn{\mu_i = \exp(x_i'\beta)}{mu_i = exp(x_i'beta)}
  We assume a multivariate Normal prior on \eqn{\beta}{beta}:
    \deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}{beta ~ N(b0,B0^(-1))}
  The candidate generating density is a multivariate Normal density centered
  at the current value of \eqn{\beta}{beta} with variance-covariance matrix
  that is an approximation of the posterior based on the maximum likelihood 
  estimates and the prior precision multiplied by the tuning parameter  
   squared.
  }}
  
\references{
      
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   counts <- c(18,17,15,20,10,20,25,13,12)
   outcome <- gl(3,1,9)
   treatment <- gl(3,3)
   posterior <- MCMCpoisson(counts ~ outcome + treatment)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}, \code{\link[base]{glm}}}


\eof
\name{MCMCprobit}
\alias{MCMCprobit}
\title{Markov chain Monte Carlo for Probit Regression}
\description{
  This function generates a posterior density sample
  from a probit regression model using the data augmentation
  approach of Albert and Chib (1993). The user supplies data and priors,
  and a sample from the posterior density is returned as an mcmc
  object, which can be subsequently analyzed with functions 
  provided in the coda package.
  }
  
\usage{
MCMCprobit(formula, data = list(), burnin = 1000, mcmc = 10000,
   thin = 5, verbose = FALSE, seed = 0, beta.start = NA,
   b0 = 0, B0 = 0, bayes.resid = FALSE, ...) }

\arguments{
    \item{formula}{Model formula.}

    \item{data}{Data frame.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Gibbs iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    Gibbs iterations must be divisible by this value.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number and
    the betas are printed to the screen.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}
  
    \item{beta.start}{The starting value for the \eqn{\beta}{beta} vector.    
    This can either 
    be a scalar or a column vector with dimension equal to the number of 
    betas.  If this takes a scalar value, then that value will serve as 
    the 
    starting value for all of the betas. The default value of NA will
    use the maximum likelihood estimate of \eqn{\beta}{beta} as the starting 
    value.}

    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a column       
    vector with dimension equal to the number of betas. If this takes a scalar
    value, then that value will serve as the prior mean for all of the
    betas.}
    
    \item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a     
    scalar
    or a square matrix with dimensions equal to the number of betas.  If this
    takes a scalar value, then that value times an identity matrix serves
    as the prior precision of \eqn{\beta}{beta}. Default value of 0 is 
    equivalent to
    an improper uniform prior on \eqn{\beta}{beta}.}

   \item{bayes.resid}{Should latent Bayesian residuals (Albert and Chib,
    1995) be returned? Default is FALSE meaning no residuals should be
    returned. Alternatively, the user can specify an array of integers
    giving the observation numbers for which latent residuals should be
    calculated and returned. TRUE will return draws of
    latent residuals for all observations.}
    
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
\code{MCMCprobit} simulates from the posterior density of a probit
  regression model using data augmentation. The simulation
  proper is done in compiled C++ code to maximize efficiency.  Please consult
  the coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample.

  The model takes the following form:
  \deqn{y_i \sim \mathcal{B}ernoulli(\pi_i)}{y_i ~ Bernoulli(pi_i)}
  Where the inverse link function:
  \deqn{\pi_i = \Phi(x_i'\beta)}{pi_i = Phi(x_i'beta)}
  We assume a multivariate Normal prior on \eqn{\beta}{beta}:
    \deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}{beta ~ N(b0,B0^(-1))}
  See Albert and Chib (1993) for estimation details.
  
 }
  
\references{
  Albert, J. H. and S. Chib. 1993. ``Bayesian Analysis of Binary and
  Polychotomous Response Data.'' \emph{J. Amer. Statist. Assoc.} 88, 669-679

  Albert, J. H. and S. Chib. 1995. ``Bayesian Residual Analysis for
  Binary Response Regression Models.'' \emph{Biometrika.} 82, 747-759.
      
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   data(birthwt)
   posterior <- MCMCprobit(low~age+as.factor(race)+smoke, data=birthwt)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}, \code{\link[base]{glm}}}


\eof
\name{MCMCregress}
\alias{MCMCregress}
\title{Markov chain Monte Carlo for Gaussian Linear Regression}
\description{
  This function generates a posterior density sample
  from a linear regression model with Gaussian errors using
  Gibbs sampling (with a multivariate Gaussian prior on the
  beta vector, and an inverse-Gamma prior on the conditional
  error variance).  The user supplies data and priors, and 
  a sample from the posterior density is returned as an mcmc
  object, which can be subsequently analyzed with functions 
  provided in the coda package.
  }
  
\usage{
MCMCregress(formula, data = list(), burnin = 1000, mcmc = 10000,
   thin = 5, verbose = FALSE, seed = 0, sigma2.start = NA,
   b0 = 0, B0 = 0, nu = 0.001, delta = 0.001, ...) }

\arguments{
    \item{formula}{Model formula.}

    \item{data}{Data frame.}

    \item{burnin}{The number of burn-in iterations for the sampler.}

    \item{mcmc}{The number of Gibbs iterations for the sampler.}

    \item{thin}{The thinning interval used in the simulation.  The number of
    Gibbs iterations must be divisible by this value.}

    \item{verbose}{A switch which determines whether or not the progress of
    the sampler is printed to the screen.  If TRUE, the iteration number, the
    \eqn{\beta}{beta} vector, and the conditional error variance is printed to 
    the screen 
    every 500 iterations.}

    \item{seed}{The seed for the random number generator.  The code uses the
    Mersenne Twister, which requires an integer as an input.  If nothing
    is provided, the Scythe default seed is used.}

    \item{sigma2.start}{The starting value for the conditional error
    variance. The default value of of NA will use the maximum
    likelihood estimate of \eqn{\sigma^2}{sigma2} as the starting value.}

    \item{b0}{The prior mean of \eqn{\beta}{beta}.  This can either be a 
    scalar or a
    column vector with dimension equal to the number of betas. If this
    takes a scalar  value, then that value will serve as the prior
    mean for all of the betas.} 
    
    \item{B0}{The prior precision of \eqn{\beta}{beta}.  This can either be a 
    scalar or a square matrix with dimensions equal to the number of betas.  
    If this
    takes a scalar value, then that value times an identity matrix serves
    as the prior precision of beta. Default value of 0 is equivalent to
    an improper uniform prior for beta.}
    
    \item{nu}{\eqn{\nu/2}{nu/2} is the shape parameter for inverse-Gamma prior 
     on the conditional error  variance.}
    
    \item{delta}{\eqn{\delta/2}{delta/2} is the scale parameter for 
    inverse-Gamma prior on the conditional error variance.}
    
    \item{...}{further arguments to be passed}       
}

\value{
   An mcmc object that contains the posterior density sample.  This 
   object can be summarized by functions provided by the coda package.
}

\details{
  \code{MCMCregress} simulates from the posterior density using 
  standard Gibbs sampling (a multivariate Normal draw for the betas, and an
  inverse-Gamma draw for the conditional error variance).  The simulation
  proper is done in compiled C++ code to maximize efficiency.  Please consult
  the coda documentation for a comprehensive list of functions that can be
  used to analyze the posterior density sample.
  
  The model takes the following form:
  \deqn{y_i = x_i ' \beta + \varepsilon_{i}}{y_i = x_i'beta + epsilon_i}
  Where the errors are assumed to be Gaussian:
  \deqn{\varepsilon_{i} \sim \mathcal{N}(0, \sigma^2)}{epsilon_i ~ N(0,
    sigma^2)} 
  We assume standard, conjugate priors:
  \deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}{beta ~ N(b0,B0^(-1))}
  And:
  \deqn{\sigma^{-2} \sim \mathcal{G}amma(\nu/2, \delta/2)}{sigma^(-2) ~
    Gamma(nu/2, delta/2)}
  Where \eqn{\beta}{beta} and \eqn{\sigma^{-2}}{sigma^(-2)} are assumed 
  \emph{a priori} independent.  Note that only starting values for
  the conditional error variance are allowed because
  \eqn{\beta}{beta} is the first block in the sampler.
  }
  
\references{
   Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
   \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
   
   Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2002.
   \emph{Output Analysis and Diagnostics for MCMC (CODA)}.
   \url{http://www-fis.iarc.fr/coda/}.
}


\examples{
   \dontrun{
   line   <- list(X = c(1,2,3,4,5), Y = c(1,3,3,3,5))
   line$X <- line$X - mean(line$X)
   posterior  <- MCMCregress(Y~X, data=line)
   plot(posterior)
   summary(posterior)
   }
}

\keyword{models}

\seealso{\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}}, \code{\link[base]{lm}}}


\eof
\name{Senate}
\alias{Senate}

\title{
  106th U.S. Senate Roll Call Vote Matrix
}
\description{
  This dataframe contains a matrix of votes cast by U.S. Senators
  in the 106th Congress.
}
\usage{
data(Senate)
}
\format{
  The dataframe contains roll call data for all Senators in the 106th 
  Senate.  The first column (id) is the ICPSR member ID number, the
  second column (statecode) is the ICPSR state code, the third column (party)
  is the member's state name, and the fourth column (member) is the member's
  name.  This is followed by all roll call votes (including unanimous ones)
  in the 106th.   Nay votes are coded 0, yea votes are coded 1,
  and NAs are missing votes.
}

\source{
  Keith Poole. 2003. \emph{106th Roll Call Vote Data}.
  \url{http://voteview.uh.edu/}.
}

\keyword{datasets}

\eof
\name{SupremeCourt}
\alias{SupremeCourt}
\title{
  U.S. Supreme Court Vote Matrix
}
\description{
  This dataframe contains a matrix votes cast by U.S. Supreme
  Court justices in all cases in the 2000 term.
}
\usage{
data(SupremeCourt)
}
\format{
  The dataframe has contains data for justices Rehnquist, Stevens,
  O'Connor, Scalia, Kennedy, Souter, Thomas, Ginsburg, and Breyer
  for the 2000 term of the U.S. Supreme Court.  It contains data
  from 43 non-unanimous cases. The votes are coded liberal (1)
  and conservative
  (0) using the protocol of Spaeth (2003).   The unit of analysis
  is the case citation (ANALU=0).  We are concerned with formally
  decided cases issued with written opinions, after full oral
  argument and cases decided by an equally divided vote
  (DECTYPE=1,5,6,7).
}

\source{
  Harold J. Spaeth. 2003. \emph{Original United States Supreme Court Database: 
    1953-2001 Terms.}  \url{http://polisci.msu.edu/pljp/}.
}

\keyword{datasets}

\eof
\name{check.parameters}
\alias{check.parameters}
\title{MCMCpack Helper Function: Check Simulation Parameters}
\description{
  Check the simulation parameters to make sure they are
  appropriate. This is called by the MCMC* model functions.
}

\usage{
  check.parameters(burnin, mcmc, thin, fcn, tune=NA)
}

\arguments{
  \item{burnin}{The number of burn-in iterations for the sampler.}
  
  \item{mcmc}{The number of MCMC iterations for the sampler.}
  
  \item{thin}{The thinning interval used in the simulation.  The
    number of MCMC iterations must be divisible by this value.}
  
  \item{fcn}{String containing function name (with no parentheses).
    It is used to throw informative errors.}
  
  \item{tune}{(Optional) tuning parameter for models that have a
    single Metropolis-Hastings tuning parameter.}
}
    

\value{
  Returns zero if exits cleanly.
}

\examples{
  check.parameters(100,1000,10,"MCMCregress")
}

\keyword{programming}


\eof
\name{ddirichlet}
\alias{ddirichlet}
\title{Evaluate Density of Dirichlet Distribution}
\description{
  Functions to compute the density of the Dirichlet distribution.
}
\usage{
  ddirichlet(x, alpha)
}
\arguments{
  \item{x}{A vector containing a single deviate or matrix
    containing one random deviate per row.} 
  \item{alpha}{Vector of shape parameters, or matrix of shape
    parameters corresponding to each row above. }
}
\details{
  The Dirichlet distribution is the multidimensional generalization of
  the beta distribution. 
}
\value{
  \code{ddirichlet} returns a vector containing the evaluated Dirichlet
  density for the corresponding rows of \code{x}.
}

\author{
  Code is taken from Greg's Miscellaneous Functions (gregmisc).  His
  code was based on code posted by Ben Bolker to R-News on Fri Dec 15
  2000.
}

\seealso{
  \code{\link[base]{Beta}},
  \code{\link{rdirichlet}}
}

\examples{
  ddirichlet(c(.1,.2,.7), c(1,1,1))
}
\keyword{distribution}

\eof
\name{dinvgamma}
\alias{dinvgamma}
\title{Evaluate the Density of the Inverse Gamma Distribution}
\description{
  This evaluates the density of the Inverse Gamma Distribution at the
  value x.
}

\usage{
   dinvgamma(x, shape, rate=1)
}

\arguments{
  \item{x}{Scalar location to evaluate density.}
  \item{shape}{Scalar shape parameter.}
  \item{rate}{Scalar rate parameter (default value one).}
}

\value{
  The density evaluated at \code{x}.  The parameterization is
  consistent with the Gamma Distribution in the base package.
}

\examples{
density <- dinvgamma(4.2, 1.1)
}

\keyword{distribution}

\seealso{
  \code{\link{rinvgamma}},
  \code{\link[base]{GammaDist}}
  }

\eof
\name{diwish}
\alias{diwish}
\title{Evaluate the Density of the Inverse Wishart Distribution}

\description{
  This evaluates the density of the Inverse Wishart Distribution at the 
  positive definite matrix W.
}
  
\usage{
   diwish(W, v, S)
}

\arguments{
  \item{W}{Positive definite matrix W \eqn{(p \times p)}{(p x p)}.}
  \item{v}{Inverse Wishart degrees of freedom (scalar).}
  \item{S}{Inverse Wishart scale matrix \eqn{(p \times p)}{(p x p)}.}}

\value{
  The density evaluated at W.
}

\examples{
x <- diwish(matrix(c(2,-.3,-.3,4),2,2), 3, matrix(c(1,.3,.3,1),2,2))
}


\keyword{distribution}

\seealso{\code{\link{riwish}}}

\eof
\name{dnoncenhypergeom}
\alias{dnoncenhypergeom}
\title{Evaluate Density of Noncentral Hypergeometric Distribution}
\description{
  This function evaluates the density of the the Noncentral  
  Hypergeometric distribution at a single point or all points.
}

\usage{
dnoncenhypergeom(x=NA, n1, n2, m1, psi)
}

\arguments{
  \item{x}{The location to evaluate the density.  If \code{x} is NA,
    then a matrix is returned with the density evaluated at all possible
    points.}
  \item{n1}{The size of group one.}
  \item{n2}{The size of group two.}
  \item{m1}{The observed number of positive outcomes (in both groups).}
  \item{psi}{Odds ratio.}
}

\value{
  The density evaluated at point \code{x}, or a matrix with
  the first column containing the possible values of the random
  variable, and the second column containing the probabilities.
}

\details{
  The Noncentral Hypergeometric is particularly useful for conditional
  inference for \eqn{(2 \times 2)}{(2 x 2)} tables.  We use the
  parameterization and algorithms of Liao and Rosen (2001).  The
  underlying R code is based on their published code.  See their
  article for details of the parameterization.
}

\source{
  L. G. Liao and Ori Rosen. 2001. ``Fast and Stable Algorithms for Computing
  and Sampling From the Noncentral Hypergeometric Distribution." \emph{The
  American Statistician.} 55: 366-369.
}

\examples{
  dnoncenhypergeom(NA, 500, 500, 500, 6.0)
}

\keyword{distribution}

\seealso{
   \code{\link{rnoncenhypergeom}}
}

\eof
\name{dtomogplot}
\alias{dtomogplot}
\title{Dynamic Tomography Plot}
\description{
  dtomogplot is used to produce a tomography plot (see King, 1997) for a
  series of temporally ordered, partially observed 2 x 2 contingency
  tables.
}
  
\usage{
dtomogplot(r0, r1, c0, c1, time.vec=NA,
           xlab="fraction of r0 in c0 (p0)",
           ylab="fraction of r1 in c0 (p1)",
           color.palette=heat.colors, bgcol="black", ...)
}

\arguments{
  \item{r0}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of row sums from row 0.}

  \item{r1}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of row sums from row 1.}

  \item{c0}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of column sums from column 0.}

  \item{c1}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of column sums from column 1.}

  \item{time.vec}{Vector of time periods that correspond to the elements
    of \eqn{r_0}{r0}, \eqn{r_1}{r1}, \eqn{c_0}{c0}, and \eqn{c_1}{c1}.}

  \item{xlab}{The x axis label for the plot.}

  \item{ylab}{The y axis label for the plot.}

  \item{color.palette}{Color palette to be used to encode temporal patterns.}
  
  \item{bgcol}{The background color for the plot.}
  
  \item{...}{further arguments to be passed}     
}


\details{
  Consider the following partially observed 2 by 2 contingency table:\cr
  \cr
  \tabular{llll}{
               \tab | \eqn{Y=0} \tab | \eqn{Y=1} \tab |   \cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=0} \tab | \eqn{Y_0}{Y0} \tab |  \tab | \eqn{r_0}{r0}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=1} \tab | \eqn{Y_1}{Y1} \tab |  \tab | \eqn{r_1}{r1}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
              \tab | \eqn{c_0}{c0} \tab | \eqn{c_1}{c1} \tab | \eqn{N}\cr    
  }

  where \eqn{r_0}{r0}, \eqn{r_1}{r1}, \eqn{c_0}{c0}, \eqn{c_1}{c1}, and
  \eqn{N}  are non-negative integers that are
  observed. The interior cell entries are not observed. It is
  assumed that \eqn{Y_0|r_0 \sim \mathcal{B}inomial(r_0,
    p_0)}{Y0|r0 ~ Binomial(r0, p0)} and
  \eqn{Y_1|r_1 \sim \mathcal{B}inomial(r_1, p_1)}{Y1|r1 ~
    Binomial(r1,p1)}.

  This function plots the bounds on the maximum likelihood
  estimates for (p0, p1) and color codes them by the elements of
  time.vec. 
}

\keyword{hplot}
  
\references{
  Gary King, 1997. \emph{A Solution to the Ecological Inference Problem}.
  Princeton: Princeton University Press.
  
  Jonathan Wakefield. 2001. ``Ecological Inference for 2 x 2 Tables,''
  Center for Statistics and the Social Sciences Working Paper 
  no. 12. University of Washington.

  Kevin M. Quinn. 2002. ``Ecological Inference in the Presence of
  Temporal Dependence.'' Paper prepared for Ecological Inference
  Conference, Harvard University, June 17-18, 2002.
}

\examples{
r0 <- rpois(20, 300)
r1 <- rpois(20, 200)
c0 <- 100 + 1:20*7 + rpois(20, 30)
c1 <- (r0+r1) - c0
dtomogplot(r0, r1, c0, c1, time.vec=1901:1920) 
}

\seealso{\code{\link{MCMCbaselineEI}}, \code{\link{MCMChierEI}},
  \code{\link{MCMCdynamicEI}},\code{\link{tomogplot}}
}

\eof
\name{dwish}
\alias{dwish}
\title{Evaluate the Density of the Wishart Distribution}
\description{
  This evaluates the density of the Wishart Distribution at the 
  positive definite matrix W.
}
  
\usage{
   dwish(W, v, S)
}

\arguments{
    \item{W}{Positive definite matrix W \eqn{(p \times p)}{(p x p)}.}
    \item{v}{Wishart degrees of freedom (scalar).}
    \item{S}{Wishart scale matrix \eqn{(p \times p)}{(p x p)}.}}

\value{
  The density evaluated at W.
}

\examples{
x <- dwish(matrix(c(2,-.3,-.3,4),2,2), 3, matrix(c(1,.3,.3,1),2,2))
}

\keyword{distribution}

\seealso{\code{\link{rwish}}}



\eof
\name{mcmc2}
\alias{mcmc2}
\title{Create Markov Chain Monte Carlo Objects}

\usage{
mcmc2(data= NA, start = 1, end = numeric(0), thin = 1)
}

\arguments{
  \item{data}{A vector or  matrix of MCMC output.}

  \item{start}{The iteration number of the first observation.}

  \item{end}{The iteration number of the last observation.}

  \item{thin}{The thinning interval between consecutive observations.}
}

\description{

   The function mcmc2 is used to create a Markov chain Monte Carlo object.
   The data are taken to be a vector, or a matrix with one column per
   variable.  This works slightly differently than the \code{coda} function
   to make more efficient use of memory.

   An mcmc object may be summarized by the \code{summary} function
   and visualized with the \code{plot} function.

   MCMC objects resemble time series (\code{ts}) objects and have
   methods for the generic functions \code{time}, \code{start},
   \code{end}, \code{frequency}, and \code{window}.
}

\author{Martyn Plummer (slightly modified by Kevin Quinn)}

\note{
   The format of the mcmc class has changed between coda version 0.3
   and 0.4.  Older mcmc objects will now cause \code{is.mcmc} to
   fail with an appropriate warning message.  Obsolete mcmc objects can
   be upgraded with the \code{mcmcUpgrade} function, which is part of the
   \code{coda} package.
}

\keyword{models}


\seealso{
   \code{\link{mcmc2dataframe}}
}

\eof
\name{mcmc2dataframe}
\alias{mcmc2dataframe}
\title{Convert Markov Chain Monte Carlo Objects to Dataframes}

\usage{
mcmc2dataframe(obj)
}

\arguments{
  \item{obj}{An mcmc object.}
}

\description{
  The function `mcmc2dataframe' is used to convert an mcmc object to a
  dataframe.
}
 
\value{
  A dataframe built from obj.
}

\keyword{models}

\seealso{
   \code{\link{mcmc2}}
}

\eof
\name{rdirichlet}
\alias{rdirichlet}
\title{Generate Random Draws from the Dirichlet Distribution}
\description{
  Generate random deviates from the Dirichlet distribution.
}
\usage{
rdirichlet(n, alpha)
}
\arguments{
  \item{n}{Number of random vectors to generate. }
  \item{alpha}{Vector of shape parameters, or matrix of shape
    parameters corresponding to draw.}
}
\details{
  The Dirichlet distribution is the multidimensional generalization of
  the beta distribution. 
}

\value{
  \code{rdirichlet} returns a matrix with \code{n} rows, each containing
  a single Dirichlet random deviate.
}
\author{
  Code is taken from Greg's Miscellaneous Functions (gregmisc).  His
  code was based on code posted by Ben Bolker to R-News on Fri Dec 15
  2000.
}
  
\seealso{ \code{\link[base]{Beta}}, \code{\link{ddirichlet}} }

\examples{
  x <- rdirichlet(20, c(1,1,1) )
}

\keyword{distribution}

\eof
\name{read.Scythe}
\alias{read.Scythe}
\title{Read a Matrix from a File written by Scythe}
\description{
  This function reads a matrix from an ASCII file in the form produced
  by the Scythe Statistical Library.  Scythe
  output files contain the number of rows and columns in the first row,
  followed by the data.
}

\usage{
   read.Scythe(infile=NA)
}

\arguments{
    \item{infile}{The file to be read. This can include path information.}
}

\value{
  A matrix containing the data stored in the read file.
}

\examples{
  \dontrun{mymatrix <- read.Scythe("myfile.txt")}
}

\references{
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.
  \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
}

\keyword{file}

\seealso{\code{\link{write.Scythe}}}

\eof
\name{rinvgamma}
\alias{rinvgamma}
\title{Generate Random Draw from Inverse Gamma Distribution}
\description{
  This function generates a random draw from the Inverse Gamma distribution.
}
  
\usage{
   rinvgamma(n, shape, rate = 1)
}

\arguments{
  \item{n}{Number of draws from the distribution.}
  \item{shape}{Scalar shape parameter.}
  \item{rate}{Scalar rate parameter (default value one).}
}

\value{
  \code{n} draws from the Inverse Gamma distribution.  The
  parameterization is consistent with the Gamma Distribution in the
  base package.
}

\examples{
  draws <- rinvgamma(10, 3.2)
}

\keyword{distribution}

\seealso{
  \code{\link{dinvgamma}},
  \code{\link[base]{GammaDist}}
}



\eof
\name{riwish}
\alias{riwish}
\title{Generate Random Draw from Inverse Wishart Distribution}
\description{
  This function generates a random draw from the Inverse Wishart
  distribution.
}
  
\usage{
   riwish(v, S)
}

\arguments{
  \item{v}{Inverse Wishart degrees of freedom (scalar).}
  \item{S}{Inverse Wishart scale matrix \eqn{(p \times p)}{(p x p)}.}}

\value{
  A \eqn{(p \times p)}{(p x p)} matrix.
}

\examples{
A <- riwish(3, matrix(c(1,.3,.3,1),2,2))
}

\keyword{distribution}

\seealso{\code{\link{diwish}}}

\eof
\name{rnoncenhypergeom}
\alias{rnoncenhypergeom}
\title{Generate Random Draw from Noncentral Hypergeometric Distribution}
\description{
  This function generates random draws from the Noncentral  
  Hypergeometric distribution.
}

\usage{
  rnoncenhypergeom(n, n1, n2, m1, psi)
}

\arguments{
  \item{n}{The number of draws to make from the distribution.}
  \item{n1}{The size of group one.}
  \item{n2}{The size of group two.}
  \item{m1}{The observed number of positive outcomes (in both groups).}
  \item{psi}{Odds ratio.}
}

\value{
  An list of \code{n} random draws from the distribution.
}

\details{
   The Noncentral Hypergeometric is particularly useful for conditional
  inference for \eqn{(2 \times 2)}{(2 x 2)} tables.  We use the
  parameterization and algorithms of Liao and Rosen (2001).  The
  underlying R code is based on their published code.  See their
  article for details of the parameterization.
}

\source{
  L. G. Liao and Ori Rosen. 2001. ``Fast and Stable Algorithms for Computing
  and Sampling From the Noncentral Hypergeometric Distribution." \emph{The
  American Statistician.} 55: 366-369.
}

\examples{
  rnoncenhypergeom(1, 500, 500, 500, 6.0)
}

\keyword{distribution}

\seealso{
   \code{\link{dnoncenhypergeom}}
}

\eof
\name{rwish}
\alias{rwish}
\title{Generate Random Draw from Wishart Distribution}
\description{
  This function generates a random draw from the Wishart distribution.
  }
  
\usage{
   rwish(v, S)
   }

\arguments{
    \item{v}{Wishart degrees of freedom (scalar).}
    \item{S}{Wishart scale matrix \eqn{(p \times p)}{(p x p)}.}}

\value{
   A \eqn{(p \times p)}{(p x p)} matrix.
}

\examples{
A <- rwish(3, matrix(c(1,.3,.3,1),2,2))
}

\keyword{distribution}

\seealso{\code{\link{dwish}}}

\eof
\name{tomogplot}
\alias{tomogplot}
\title{Tomography Plot}
\description{
  tomogplot is used to produce a tomography plot (see King, 1997) for a
  series of partially observed 2 x 2 contingency tables.
}
  
\usage{
tomogplot(r0, r1, c0, c1, xlab="fraction of r0 in c0 (p0)",
          ylab="fraction of r1 in c0 (p1)", bgcol="white", ...)
   }

\arguments{
  \item{r0}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of row sums from row 0.}
  
  \item{r1}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of row sums from row 1.}

  \item{c0}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of column sums from column 0.}

  \item{c1}{An \eqn{(ntables \times 1)}{(ntables * 1)} vector of column sums from column 1.}

  \item{xlab}{The x axis label for the plot.}

  \item{ylab}{The y axis label for the plot.}

  \item{bgcol}{The background color for the plot.}
  
  \item{...}{further arguments to be passed}     
}


\details{
  Consider the following partially observed 2 by 2 contingency table:\cr
  \cr
  \tabular{llll}{
               \tab | \eqn{Y=0} \tab | \eqn{Y=1} \tab |   \cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=0} \tab | \eqn{Y_0}{Y0} \tab |  \tab | \eqn{r_0}{r0}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
    \eqn{X=1} \tab | \eqn{Y_1}{Y1} \tab |  \tab | \eqn{r_1}{r1}\cr
    - - - - - \tab - - - - - \tab - - - - - \tab - - - - - \cr
              \tab | \eqn{c_0}{c0} \tab | \eqn{c_1}{c1} \tab | \eqn{N}\cr    
  }

  where \eqn{r_0}{r0}, \eqn{r_1}{r1}, \eqn{c_0}{c0}, \eqn{c_1}{c1}, and
  \eqn{N}  are non-negative integers that are
  observed. The interior cell entries are not observed. It is
  assumed that \eqn{Y_0|r_0 \sim \mathcal{B}inomial(r_0,
    p_0)}{Y0|r0 ~ Binomial(r0, p0)} and
  \eqn{Y_1|r_1 \sim \mathcal{B}inomial(r_1, p_1)}{Y1|r1 ~
    Binomial(r1,p1)}.

  This function  plots the bounds on the maximum likelihood
  estimatess for (p0, p1). 
  }

  \keyword{hplot}
  
  \references{
    Gary King, 1997. \emph{A Solution to the Ecological Inference Problem}.
    Princeton: Princeton University Press.
    
    Jonathan Wakefield. 2001. ``Ecological Inference for 2 x 2 Tables,''
    Center for Statistics and the Social Sciences Working Paper 
    no. 12. University of Washington.  
}

\examples{
r0 <- rpois(100, 500) 
r1 <- rpois(100, 200) 
c0 <- rpois(100, 100) 
c1 <- (r0 + r1) - c0 
tomogplot(r0, r1, c0, c1) 
}

\seealso{\code{\link{MCMCbaselineEI}}, \code{\link{MCMChierEI}},
  \code{\link{MCMCdynamicEI}}, \code{\link{dtomogplot}}
}



\eof
\name{vech}
\alias{vech}
\title{Extract Lower Triangular Elements from a Symmetric Matrix}

\description{
  This function takes a symmetric matrix and extracts
  a list of all lower triangular elements.
}
  
\usage{
  vech(x)
}

\arguments{
  \item{x}{A symmetric matrix.}
}

\value{
  A list of the lower triangular elements. 
}

\details{
  This function checks to make sure the matrix is square, but it does
  not check for symmetry (it just pulls the lower
  triangular elements).  The elements are stored in column major order. 
  The original matrix can be restored using the \code{xpnd} 
  command.
}


\examples{
   symmat <- matrix(c(1,2,3,4,2,4,5,6,3,5,7,8,4,6,8,9),4,4) 
   vech(symmat)
}

\keyword{manip}

\seealso{\code{\link{xpnd}}}

\eof
\name{write.Scythe}
\alias{write.Scythe}
\title{Write a Matrix to a File to be Read by Scythe}
\description{
  This function writes a matrix to an ASCII file that can be 
  read by the Sycthe Statistical Library.  Scythe requires that
  input files contain the number of rows and columns in the first
  row, followed by the data.
}
  
\usage{
  write.Scythe(outmatrix, outfile=NA, overwrite=FALSE)
}

\arguments{
  \item{outmatrix}{The matrix to be written to a file.}
  \item{outfile}{The file to be written. This can include path information.}
  \item{overwrite}{A logical that determines whether an existing file
    should be over-written.  By default, it protects the user from
    over-writing existing files.}
}

\value{
  A zero if the file is properly written.
}

\examples{
  \dontrun{write.Scythe(mymatrix, "myfile.txt")}
}

\references{
  Andrew D. Martin, Kevin M. Quinn, and Daniel Pemstein.  2003.  
  \emph{Scythe Statistical Library 0.4.} \url{http://scythe.wustl.edu}.
}

\keyword{file}

\seealso{\code{\link{write.Scythe}}}

  

\eof
\name{xpnd}
\alias{xpnd}
\title{Expand a Vector into a Symmetric Matrix}

\description{
  This function takes a vector of appropriate length (typically
  created using \code{vech}) and creates a symmetric matrix.
}
  
\usage{
   xpnd(x, nrow)
}

\arguments{
  \item{x}{A list of elements to expand into symmetric matrix.}
  \item{nrow}{The number of rows (and columns) in the returned matrix.}
}

\value{
  An \eqn{(nrows \times nrows)}{(nrows * nrows)} symmetric matrix.
}

\details{
  This function is particularly useful when dealing with variance
  covariance matrices. Note that R stores matrices in column major
  order, and that the items in \code{x} will be recycled to fill the
  matrix if need be.
}


\examples{
  xpnd(c(1,2,3,4,4,5,6,7,8,9),4)
}

\keyword{manip}

\seealso{\code{\link{vech}}}

\eof
