\name{attach.all}
\alias{attach.all}
\alias{detach.all}
\title{Attach / detach elements of a bugs objects to search path}
\description{The database is attached to the \R search path. See \code{\link{attach}} for details.}
\usage{
    attach.all(.a, overwrite = FALSE)
    detach.all()
}
\arguments{
  \item{.a}{A \code{bugs} object}
  \item{overwrite}{If \code{TRUE}, overwrites variables if necessary.
    (The usual "attach" function in R only attaches the variables that are not yet defined.)}
}
\note{
    Without detaching, do not use \code{attach.all} on another \code{bugs} object,
    because instead of the given name, an object called \code{.a} is attached.
    Therefore strange things may happen...
}
\value{
  \code{attach.all} invisibly returns the \code{\link{environment}}.\cr
  \code{detach.all} detaches the \code{environment} named \code{.a} created by \code{attach.all}.
}
\seealso{\code{\link{bugs}}, \code{\link{attach}}, \code{\link{detach}}}
\examples{
# An example model file is given in:
model.file <- file.path(.path.package("R2WinBUGS"), "model", "schools.txt")
# Some example data (see ?schools for details):
data(schools)  
J <- nrow(schools)
y <- schools$estimate
sigma.y <- schools$sd
data <- list ("J", "y", "sigma.y")
inits <- function(){
    list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100), 
        sigma.theta = runif(1, 0, 100))
}
parameters <- c("theta", "mu.theta", "sigma.theta")      
\dontrun{
## You may need to edit "bugs.directory",
## also you need write access in the working directory:
schools.sim <- bugs(data, inits, parameters, model.file, 
    n.chains = 3, n.iter = 1000, 
    bugs.directory = "c:/Program Files/WinBUGS14/",
    working.directory = NULL)

# Do some inferential summaries
attach.all(schools.sim)
# posterior probability that the coaching program in school A 
# is better than in school C:
print(mean(theta[,1] > theta[,3]))
# 50% posterior interval for the difference between school A's 
# and school C's program:
print(quantile(theta[,1] - theta[,3], c(.25, .75)))  
plot(theta[,1], theta[,3]) 
}
}
\keyword{data}

\eof
\name{bugs}
\alias{bugs}
\title{Run WinBUGS from R}
\description{The \code{bugs} function takes data and starting values as input.
It automatically writes a WinBUGS script, calls the model,
and saves the simulations for easy access in R.}
\usage{
bugs(data, inits, parameters.to.save, model.file = "model.txt",
    n.chains = 3, n.iter = 2000, n.burnin = floor(n.iter/2),
    n.thin = max(1, floor(n.chains * (n.iter - n.burnin)/1000)),
    debug = FALSE, DIC = TRUE, digits = 5, codaPkg = FALSE,
    bugs.directory = "c:/Program Files/WinBUGS14/",
    working.directory = NULL)
}
\arguments{
  \item{data}{either a named list (names corresponding to variable names in the \code{model.file}) 
    of the data for the WinBUGS model, \emph{or}
    a vector or list of the names of the data objects used by the model}
  \item{inits}{a list with \code{n.chains} elements; each element of the list is
    itself a list of starting values for the WinBUGS model, \emph{or}
    a function creating (possibly random) initial values.
    Alternatively, if \code{inits = NULL}, initial values are generated by WinBUGS}
  \item{parameters.to.save}{character vector of the names of the parameters to save which should be monitored}
  \item{model.file}{file containing the model written in WinBUGS code.
    The extension can be either \file{.bug} or \file{.txt}.
    If \file{.bug}, a copy of the file with extension \file{.txt} will be created
    in the \code{bugs()} call and removed afterwards.
    Note that similarly named \file{.txt} files will be overwritten.}
  \item{n.chains}{number of Markov chains (default: 3)}
  \item{n.iter}{number of total iterations per chain (including burn in; default: 2000)}
  \item{n.burnin}{length of burn in, i.e. number of iterations to discard at the beginning.
      Default is \code{n.iter/2}, that is, discarding the first half of the simulations.}
  \item{n.thin}{thinning rate.  Must be a positive integer.
   Set \code{n.thin} > 1 to save memory and computation time if \code{n.iter} is large.
   Default is \code{max(1, floor(n.chains * (n.iter-n.burnin) / 1000))}
   which will only thin if there are at least 2000 simulations.}
  \item{debug}{if \code{FALSE} (default), WinBUGS is closed automatically
  when the script has finished running, otherwise WinBUGS remains open for further investigation}
  \item{DIC}{logical; if \code{TRUE} (default), compute deviance, pD, and DIC}
  \item{digits}{number of significant digits used for WinBUGS input, see \code{\link{formatC}}}
  \item{codaPkg}{logical; if \code{FALSE} (default) a \code{bugs} object is returned,
    if \code{TRUE} file names of WinBUGS output are returned for easy access by the \pkg{coda} package.}
  \item{bugs.directory}{directory that contains the WinBUGS executable}
  \item{working.directory}{sets working directory during execution of this function;
    WinBUGS' in- and output will be stored in this directory;
    if \code{NULL}, the current working directory is chosen.}
}
\details{
To run:
\enumerate{
\item Write a WinBUGS model in a ASCII file.
\item Go into R.
\item Prepare the inputs to the \code{bugs} function and run it (see Example).
\item A WinBUGS window will pop up amd R will freeze up. The model
    will now run in WinBUGS. It might take awhile. You will see
    things happening in the Log window within WinBUGS. When WinBugs
    is done, its window will close and R will work again.
\item If an error message appears, re-run with \code{debug = TRUE}.
}}
\value{
 If \code{codaPkg = TRUE} the returned values are the names
 (without file extension) of files written by WinBUGS containing
 the Markov Chain Monte Carlo output in the CODA format and corresponding index files.
 This is useful for direct access with \code{\link[coda]{read.bugs}} from package \sQuote{coda}.

 If \code{codaPkg = FALSE}, the following values are returned:
  \item{n.chains}{see Section \sQuote{Arguments}}
  \item{n.iter}{see Section \sQuote{Arguments}}
  \item{n.burnin}{see Section \sQuote{Arguments}}
  \item{n.thin}{see Section \sQuote{Arguments}}
  \item{n.keep}{number of iterations kept per chain (equal to \code{(n.iter-n.burnin) / n.thin})}
  \item{n.sims}{number of posterior simulations (equal to \code{n.chains * n.keep})}
  \item{sims.array}{3-way array of simulation output, with dimensions
    n.keep, n.chains, and length of combined parameter vector}
  \item{sims.list}{list of simulated parameters:\cr
    for each scalar parameter, a vector of length n.sims\cr
    for each vector parameter, a 2-way array of simulations,\cr
    for each matrix parameter, a 3-way array of simulations, etc.}
  \item{sims.matrix}{matrix of simulation output, with \code{n.chains * n.keep} rows and
    one column for each element of each saved parameter
    (for convenience, the \code{n.keep * n.chains} simulations in
     sims.array and sims.list have been randomly permuted)}
  \item{summary}{summary statistics and convergence information for each
    element of each saved parameter.}
  \item{mean}{a list of the estimated parameter means}
  \item{sd}{a list of the estimated parameter standard deviations}
  \item{median}{a list of the estimated parameter medians}
  \item{root.short}{names of argument \code{parameters.to.save} and \dQuote{deviance}}
  \item{long.short}{indexes; programming stuff}
  \item{dimension.short}{dimension of \code{indexes.short}}
  \item{indexes.short}{indexes of \code{root.short}}
  \item{last.values}{list of simulations from the most recent iteration; they
    can be used as starting points if you wish to run WinBUGS for further iterations}
  \item{pD}{\code{var(deviance)/2}, an estimate of the effective number of parameters
    (the variance is computed as the average of the within-chain variances,
    which gives a more reasonable estimate when convergence has not been reached)}
  \item{DIC}{\code{mean(deviance) + pD}}
}
\references{Gelman, A., Carlin, J.B., Stern, H.S., Rubin, D.B. (2003):
    \emph{Bayesian Data Analysis}, 2nd edition, CRC Press.}
\author{Andrew Gelman, \email{gelman@stat.columbia.edu}, \url{http:/www.stat.columbia.edu/~gelman/bugsR/};
    modifications and packaged by Sibylle Sturtz, \email{sturtz@statistik.uni-dortmund.de}, and Uwe Ligges.}
\seealso{\code{\link{print.bugs}}, \code{\link{plot.bugs}}, and the \pkg{coda} package}
\examples{
# An example model file is given in:
model.file <- file.path(.path.package("R2WinBUGS"), "model", "schools.txt")
# Let's take a look:
file.show(model.file)

# Some example data (see ?schools for details):
data(schools)
schools

J <- nrow(schools)
y <- schools$estimate
sigma.y <- schools$sd
data <- list ("J", "y", "sigma.y")
inits <- function(){
    list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100),
        sigma.theta = runif(1, 0, 100))
}
parameters <- c("theta", "mu.theta", "sigma.theta")

\dontrun{
## You may need to edit "bugs.directory",
## also you need write access in the working directory:
schools.sim <- bugs(data, inits, parameters, model.file,
    n.chains = 3, n.iter = 1000,
    bugs.directory = "c:/Program Files/WinBUGS14/",
    working.directory = NULL)
print(schools.sim)
plot(schools.sim)
}
}
\keyword{interface}
\keyword{models}


\eof
\name{bugs.data.inits}
\alias{bugs.data.inits}
\title{Writing input for WinBUGS - intended for internal use only}
\description{Write files \file{data.txt} and \file{inits1.txt}, \file{inits2.txt}, etc., in the working directory 
for WinBUGS to read. For internal use only.}
\usage{
bugs.data.inits(data, inits, n.chains, digits)
}
\arguments{
  \item{data}{list of the data for the WinBUGS model, \emph{or}
    a vector of the names of the data objects used by the model}
  \item{inits}{a list with \code{n.chains} elements; each element of the list is
    itself a list of starting values for the WinBUGS model, \emph{or} 
    a function creating (possibly random) initial values}
  \item{n.chains}{number of Markov chains}
  \item{digits}{number of significant digits used for WinBUGS input, see \code{\link{formatC}}}
}
\value{
    Does not return anything.
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}

\eof
\name{bugs.plot}
\alias{bugs.plot.summary}
\alias{bugs.plot.inferences}
\title{Plotting summary information - intended for internal use only}
\description{Plotting summary information - intended for internal use only}
\usage{
bugs.plot.summary(sims, ...)
bugs.plot.inferences(sims, display.parallel, ...)
}
\arguments{
  \item{sims}{an object of class `bugs', see \code{\link{bugs}} for details}
  \item{display.parallel}{display parallel intervals in both halves of the summary plots;
    this is a convergence-monitoring tool and is not necessary once you have approximate convergence
    (default is \code{FALSE})}
  \item{...}{further arguments to be passed to low-level plot functions}
}
\details{
\code{bugs.plot.summary} (left hand side of plot) and \code{bugs.plot.inferences} (right hand side of plot).
}
\value{
Does not return anything, but prints and plots as side-effects.
}
\seealso{The main function to be called by the user is \code{plot}, see \code{\link{plot.bugs}} for details.}
\keyword{internal}

\eof
\name{bugs.run}
\alias{bugs.run}
\title{Calling WinBUGS - intended for internal use only}
\description{
Updates the lengths of the adaptive phases in the WinBUGS registry (using \code{\link{bugs.update.settings}}),
calls WinBUGS and runs it with \file{script.txt}. Intended for internal use only}
\usage{
bugs.run(n.burnin, bugs.directory)
}
\arguments{
  \item{n.burnin}{length of burn in}
  \item{bugs.directory}{directory that contains the WinBUGS executable}  
}
\value{
    Does not return anything.
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}

\eof
\name{bugs.script}
\alias{bugs.script}
\title{Writes script for running WinBUGS - intended for internal use only}
\description{Write file \file{script.txt} for WinBUGS to read - intended for internal use only}
\usage{
bugs.script(parameters.to.save, n.chains, n.iter, n.burnin, n.thin, 
    bugs.directory, model.file, debug = FALSE, is.inits)
}
\arguments{
  \item{parameters.to.save}{parameters that should be monitored}
  \item{n.chains}{number of Markov chains}
  \item{n.iter}{number of total iterations (including burn in)}
  \item{n.burnin}{length of burn in}
  \item{n.thin}{size of thinning parameter}
  \item{bugs.directory}{directory that contains the WinBUGS executable}  
  \item{model.file}{file containing the model written in WinBUGS code}
  \item{debug}{if \code{FALSE}, WinBUGS is closed automatically, 
    otherwise WinBUGS remains open for further investigation}
  \item{is.inits}{logical; whether initial values are given by the user (\code{TRUE}) 
    or have to be generated by WinBUGS}
}
\value{
    Does not return anything.
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}


\eof
\name{bugs.sims}
\alias{bugs.sims}
\title{WinBUGS output reader - intended for internal use only}
\description{
Reads simulations from WinBUGS into R, formats them, monitors convergence,
performs convergence checks, and computes medians and quantiles. Intended for internal use only.}
\usage{
bugs.sims(parameters.to.save, n.chains, n.iter, n.burnin, n.thin, 
    DIC = TRUE)
}
\arguments{
  \item{parameters.to.save}{parameters that should be monitored}
  \item{n.chains}{number of Markov chains}
  \item{n.iter}{number of total iterations (including burn in)}
  \item{n.burnin}{length of burn in}
  \item{n.thin}{size of thinning parameter}
  \item{DIC}{calculation of DIC}
}
\value{
Returns the same values as \code{\link{bugs}}.
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}

\eof
\name{bugs.update.settings}
\alias{bugs.update.settings}
\title{Adjusting WinBUGS registry - intended for internal use only}
\description{Adjusts WinBUGS registry - intended for internal use only
}
\usage{
bugs.update.settings(n.burnin, bugs.directory)
}
\arguments{
  \item{n.burnin}{length of burn in}
  \item{bugs.directory}{directory that contains the WinBUGS executable}
}
\details{
Alter the adaptive phases in the WinBUGS updaters (in the \file{Registry.odc} file)
so that all adaptive updating is done during the burnin stage. Otherwise
WinBUGS will not report results if the burnin is less than a minimum value
such as 500 or 4000 (depending on the model).
}
\value{
Does not return anything.
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}


\eof
\name{decode.parameter.name}
\alias{decode.parameter.name}
\title{Decodes WinBUGS parameter names - intended for internal use only}
\description{Decodes WinBUGS parameter names (e.g., \code{beta[3,14]} becomes \code{beta} with 2 indexes: 3 and 14)
 for use by the \code{\link{bugs.sims}} function.}
\usage{
decode.parameter.name(a)
}
\arguments{
  \item{a}{one element of the first column of \file{codaIndex.txt}}
}
\value{
list with elements
  \item{root}{name of parameter, e.g. \code{beta}}
  \item{dimension}{number of \code{indexes}, e.g. 2}
  \item{indexes}{indexes, e.g. 3 and 14}
given \code{a == "beta[3, 14]"}
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}

\eof
\name{monitor}
\alias{monitor}
\alias{conv.par}
\title{Special summary statistics - intended for internal use only}
\description{Special summary statistics of the WinBUGS output - intended for internal use only}
\usage{
monitor(a, n.chains, trans = NULL, keep.all = FALSE, Rupper.keep = FALSE)
conv.par(x, n.chains, Rupper.keep = TRUE)
}
\arguments{
  \item{x}{}
  \item{a}{a \code{n * m * k} array: \code{m} sequences of length \code{n}, \code{k} variables measured}
  \item{n.chains}{number of Markov chains}
  \item{trans}{a vector of length \code{k}:  "" if no transformation, or "log" or "logit"
    (If \code{trans} is \code{NULL}, it will be set to "log" for parameters that
    are all-positive and 0 otherwise.)}
  \item{keep.all}{if \code{FALSE} (default), first half of \code{a} will be discarded}
  \item{Rupper.keep}{if \code{FALSE}, don't return \code{Rupper}}
}
\details{
See the well documented code for details.
}

\value{
for \code{monitor}:
    \item{output}{
     list of "mean","sd", quantiles ("2.5\%","25\%","50\%","75\%","97.5\%"), "Rhat" if \code{n.chains > 1}, 
        "Rupper" if \code{(Rupper.keep == TRUE) && (n.chains > 1)}, and "n.eff" if \code{n.chains > 1}}
for \code{conv.par} a list with elements:
    \item{quantiles}{emipirical quantiles of simulated sequences}
    \item{confshrink}{estimated potential scale reduction (that would be achieved by
        continuing simulations forever) has two components:  an estimate and
        an approx. 97.5\% upper bound}
    \item{n.eff}{effective sample size: \code{m*n*min(sigma.hat^2/B,1)}.
        This is a crude measure of sample size because it relies on the between
        variance, \code{B}, which can only be estimated with m degrees of freedom.}
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}


\eof
\name{plot.bugs}
\alias{plot.bugs}
\title{Plotting a bugs object}
\description{Plotting a \code{bugs} object}
\usage{
plot.bugs(x, display.parallel = FALSE, ...)
}
\arguments{
  \item{x}{an object of class `bugs', see \code{\link{bugs}} for details}
  \item{display.parallel}{display parallel intervals in both halves of the summary plots;
    this is a convergence-monitoring tool and is not necessary once you have approximate convergence
    (default is \code{FALSE})}
  \item{...}{further arguments to \code{\link{plot}}}
}
\seealso{\code{\link{bugs}}}
\keyword{hplot}

\eof
\name{print.bugs}
\alias{print.bugs}
\title{Printing a bugs object}
\description{Printing a \code{bugs} object}
\usage{
print.bugs(x, digits.summary = 1, ...)
}
\arguments{
  \item{x}{an object of class `bugs', see \code{\link{bugs}} for details}
  \item{digits.summary}{rounding for tabular output on the console (default is to round to 1 decimal place)}
  \item{...}{further arguments to \code{\link{print}}}
}
\seealso{\code{\link{bugs}}}
\keyword{print}

\eof
\name{schools}
\alias{schools}
\docType{data}
\title{8 schools analysis}
\description{8 schools analysis}
\usage{data(schools)}
\format{
  A data frame with 8 observations on the following 3 variables.
  \describe{
    \item{school}{See Source.}
    \item{estimate}{See Source.}
    \item{sd}{See Source.}
  }
}
\source{
Rubin, D.B. (1981): Estimation in Parallel Randomized Experiments.
\emph{Journal of Educational Statistics} 6(4), 377-400.

Section 5.5 of
Gelman, A., Carlin, J.B., Stern, H.S., Rubin, D.B. (2003): 
\emph{Bayesian Data Analysis}, 2nd edition, CRC Press.
}
\keyword{datasets}

\eof
\name{write.datafile}
\alias{write.datafile}
\alias{formatdata}
\title{Write data for WinBUGS - intended for internal use only}
\description{Write data in files that can be read by WinBUGS - intended for internal use only}
\usage{
write.datafile(datalist, towhere, fill = TRUE)
formatdata(datalist)
}
\arguments{
  \item{datalist}{a \emph{list} to be written into an appropriate structure}
  \item{towhere}{the name of the file which the data are to be written to}
  \item{fill}{see \code{\link{cat}}, defaults to \code{TRUE}}
}
\value{
  \item{datalist.tofile}{A structure appropriate to be read in by WinBUGS.}
}
\seealso{The main function to be called by the user is \code{\link{bugs}}.}
\keyword{internal}

\eof
