\name{OZrain}
\alias{OZrain}
\title{Daily Rainfall in Melbourne, Australia, 1981-1990}
\description{Amount of daily rainfall in Melbourne, Australia,
  1981-1990, measured in millimeters.  The amounts are integers with
  many zeros and three days of more than 500mm rain.
}
\usage{data(OZrain)}
\format{
  A time-series of length 3653 with the amount of daily rainfall in mm.
  Because of the two leap years 1984 and '88, we have constructed it
  with \code{\link{ts}(*, start=1981, frequency=365.25,
    end = 1981+ (3653 - 1)/365.25)}.
}
\source{
  \file{rainfall.dat} in Rob J. Hyndman's \emph{Time Series Data Library},
  \url{http://www-personal.buseco.monash.edu.au/~hyndman/TSDL/}

  originally, Australian Bureau of Meteorology, \url{http://www.abs.gov.au}.
}
\note{
  There must be one extra observation since for the ten years with two
  leap years, there are only 3652 days.
  In 61 out of 100 days, there's no rain.
}
\examples{
data(OZrain)
(n <- length(OZrain)) ## should be 1 more than
ISOdate(1990,12,31) - ISOdate(1981, 1,1)## but it's 2 ..

has.rain <- OZrain > 0

summary(OZrain[has.rain])# Median = 18,  Q3 = 50
table(rain01 <- as.integer(has.rain))
table(rain4c <- cut(OZrain, c(-.1, 0.5, 18.5, 50.1, 1000)))

\testonly{
  table(rain4c <- as.integer(rain4c) - 1:1)
  draw(v4 <- vlmc(rain4c))
  v4 # cutoff 3.907; MC = 9, context = 39
  vlmc(rain4c, cut = 0  )$size # MC = 25; context = 2355, #{leaves}= 1062
  vlmc(rain4c, cut = 0.1)$size # MC = 25; context = 2344
  vlmc(rain4c, cut = 0.3)$size # MC = 25; context = 1990, #{leaves}=  967
%% ../tests/iid-cutoff.R finds an "iid-correct cutoff" of ~ 6
  vlmc(rain4c, cut = 6.0)$size # MC =  7; context =   10, #{leaves}=    4
  lLv <- function(co) {
    l <- logLik(vlmc(rain4c, cut = co)); c(Ll = l, df = attr(l,"df"))
  }
  cuts4 <- c(0,0.1,0.3,0.5,0.7, seq(1,7, by = .25), 9, 12,13)
  ll4 <- sapply(cuts4, lLv)
  par(mfcol = c(2,2))
  plot(cuts4, AIC4 <- c(c(-2,2) \%*\% ll4), type = 'b')
  plot(cuts4, BIC4 <- c(c(-2,log(n)) \%*\% ll4), type = 'b')
  ii <- cuts4 > 2.
  plot(cuts4[ii], AIC4[ii], type = 'b')
  plot(cuts4[ii], BIC4[ii], type = 'b')
}% testonly

AIC(v1  <- vlmc(rain01))# cutoff = 1.92
AIC(v00 <- vlmc(rain01, cut = 1.4))
AIC(v0  <- vlmc(rain01, cut = 1.5))

\testonly{
  lLv <- function(co) {
    cat(co,"")
    l <- logLik(vlmc(rain01, cut = co))
    c(Ll = l, df = attr(l,"df"))
  }
  cc <- seq(1., 3, len = 401)
  system.time(vll <- sapply(cc, lLv))# 21 & 49 sec. (on P III 900 MHz)
  llv <- vll["Ll",] ; dfv <- vll["df",]
  par(mfrow=c(2,2))
  ## very interesting:  quite a few local minima!!
  plot  (cc, -2*llv + 2*dfv, type='l', main="AIC(vlmc(rain01, cutoff = cc))")
  plot  (cc, dfv, type='l', main="Complexity(vlmc(*, cutoff = cc))")
  plot  (cc, llv, type='l', main="Log.Likelihood(vlmc(*, cutoff = cc))")

  ## BIC is all falling:
  ##  par(new=T)
  plot  (cc, -2*llv + log(3653)*dfv, type='l', main="BIC(vlmc(*, cutoff = cc))")
}

hist(OZrain)
hist(OZrain, breaks = c(0,1,5,10,50,1000), xlim = c(0,100))

plot(OZrain, main = "Rainfall 1981-1990 in Melbourne")
## work around bug in plot.ts() for R <= 1.3.1 :
newer1.4 <- {v <- R.version; v$major > 1 || as.numeric(v$minor) >= 4.2}
if(newer1.4) {
  plot(OZrain, log="y", main = "Non-0 Rainfall [LOG scale]")
} else
  plot.ts(time(OZrain)[has.rain],
          OZrain[has.rain], log="y", main = "Non-0 Rainfall [LOG scale]")
lOZ <- lowess(log10(OZrain[has.rain]), f= .05)
lines(time(OZrain)[has.rain], 10^lOZ$y, col = 2, lwd = 2)
}
\keyword{datasets}

\eof
\name{RCplot}
\alias{RCplot}
\title{Residuals vs Context plot}
\description{
  Plots the residuals of a fitted VLMC model against the contexts, i.e.,
  produces a boxplot of residuals for all contexts used in the model fit.

  This has proven to be useful function, and the many optional arguments
  allow quite a bit of customization.  However, the current
  implementation is somewhat experimental and the defaults have been
  chosen from only a few examples.
}
\usage{
RCplot(x, r2 = residuals(x, "deviance")^2,
       alphabet = x$alpha, lab.horiz = k <= 20,
       do.call = TRUE,
       cex.axis = if (k <= 20) 1 else if (k <= 40) 0.8 else 0.6,
       y.fact = if (.Device == "postscript") 1.2 else 0.75,
       col = "gray70", xlab = "Context", main = NULL,
       med.pars = list(col = "red", pch = 12, cex = 1.25 * cex.axis),
       ylim = range(0, r2, finite = TRUE),
       \dots)
}
\arguments{
  \item{x}{an \R object of class \code{vlmc}.}
  \item{r2}{numeric vector, by default of squared deviance residuals of
    \code{x}, but conceptually any (typically non-negative) vector of
    the appropriate length.}
  \item{alphabet}{the alphabet to use for labeling the contexts,
    via \code{\link{id2ctxt}}.}
  \item{lab.horiz}{logical indicating if the context labels should be
    written \bold{horiz}ontally or vertically.}
  \item{do.call}{logical indicating if the \code{\link{vlmc}} call
    should be put as subtitle.}
  \item{cex.axis}{the character expansion for axis labeling, see also
    \code{\link{par}}.  The default is only approximately good.}
  \item{y.fact}{numeric factor for expanding the space to use for the
    context labels (when \code{lab.horiz} is false).}
  \item{col}{color used for filling the boxes.}
  \item{xlab}{x axis label (with default).}
  \item{main}{main title to be used, \code{NULL} entailing a sensible default.}
  \item{med.pars}{graphical parameters to be used for coding of medians
    that are almost 0.}
  \item{ylim}{y range limits for plotting.}
  \item{\dots}{further arguments to be passed to \code{plot()}.}
}
\value{
  \emph{Invisibly}, a list with components
  \item{k}{the number of contexts (and hence box plots) used.}
  \item{fID}{a factor (as used in the interncal call to
    \code{\link{plot.factor}}).}
  \item{rp}{a list as resulting from the above call to \code{plot.factor()}.}
}
\references{
  Mchler M. and Bhlmann P. (2003)
  Variable Length Markov Chains: Methodology, Computing and Software.
  accepted for publication in \emph{J\. Computational and Graphical Statistics}.
}
\author{Martin Maechler}
\seealso{\code{\link{summary.vlmc}} for other properties of a VLMC model.}
\examples{
example(vlmc)
RCplot(vlmc.pres)
RCplot(vlmc.dt1c01)## << almost perfect fit (0 resid.)
}
\keyword{hplot}
\keyword{utilities}

\eof
\name{alpha2int}
\alias{alpha2int}
\alias{int2alpha}
\title{`Single Character' <--> Integer Conversion for Discrete Data}
\description{
  Simple conversion functions for discrete data (e.g., time series),
  between \code{0:k} integers and \emph{single} letter characters.
}
\usage{
alpha2int(x, alpha)
int2alpha(i, alpha)
}
\arguments{
  \item{x}{character vector of single letters.}
  \item{alpha}{the alphabet, as one character string.}
  \item{i}{integer vector of numbers in \code{0:k}.}
}
\value{
  \code{alpha2int(x,*)} returns an \code{\link{integer}} vector of the same
  length as \code{x}, consisting of values from \code{0:k} where
  \code{k + 1} is the length of the alphabet, \code{\link{nchar}(alpha)}.

  \code{int2alpha(i,*)} returns a vector of \emph{single letter}
  \code{\link{character}} of the same length as \code{i}.
}
\seealso{\code{\link{vlmc}}, and
  \code{\link{int2char}()} and its inverse, \code{\link{char2int}()},
  both working with multi-character strings instead of vectors of single
  characters; further, \code{\link{alphabet}}.
}
\examples{
alphabet <- "abcdefghijk"
(ch <- sample(letters[1:10], 30, replace = TRUE))
(ic <- alpha2int(ch, alphabet))
stopifnot(int2alpha(ic, alphabet) == ch)
}
\keyword{character}
\keyword{utilities}

\eof
\name{alphabet}
\alias{alphabet}
\alias{alphabet.vlmc}
\title{The Alphabet in Use}
\description{
  Return the alphabet in use, as a vector of \dQuote{characters}.
}
\usage{
alphabet(x, \dots)
\method{alphabet}{vlmc}(x, \dots)
}
\arguments{
  \item{x}{any \R object, currently only available for
    \code{\link{vlmc}} ones.}
  \item{\dots}{potential further arguments passed to and from methods.}
}
% \details{
%   ~~ If necessary, more details than the __description__  above ~~
% }
\value{
  a \code{\link{character}} vector, say \code{r}, with length equal to
  the alphabet size.  Currently, typically all \code{r[i]} are strings
  of just one character.
}
\seealso{\code{\link{alpha2int}} for conversion to and from integer
  codings.
}
\examples{
data(bnrf1)
vb <- vlmc(bnrf1EB, cutoff = 5)
alphabet(vb) # |--> "a" "c" "g" "t"
}
\keyword{character}
\keyword{utilities}

\eof
\name{as.dendrogram.vlmc}
\alias{as.dendrogram.vlmc}
\title{Dendrogram Construction from VLMCs}
\description{
  This is a method for the \code{as.dendrogram} generic function
}
\usage{
as.dendrogram.vlmc(object, \dots)
}
\arguments{
  \item{object}{a \code{\link{vlmc}} object.}
  \item{\dots}{further arguments passed to and from methods.}
}
% \details{
%   ~~ If necessary, more details than the __description__  above ~~
% }
\value{
  An object of class \code{\link{dendrogram}}, i.e. a nested list
  described on that page.
 %% FIXME add ??
}
\seealso{\code{\link[mva]{as.dendrogram}}, \code{\link[mva]{plot.dendrogram}}.}

\examples{
data(presidents)
dpr <- factor(cut(presidents, c(0,45,70,100)), exclude=NULL)# NA = 4th level
(vlmc.pres <- vlmc(dpr))
draw(vlmc.pres)
(dv.dpr <- as.dendrogram(vlmc.pres))
str(dv.dpr)
str(unclass(dv.dpr))

%% type = "tr" and these two `pch' should be *default*
plot(dv.dpr, type ="tr", nodePar = list(pch=c(1,16), cex = 1.5))

## Artificial example
f1 <- c(1,0,0,0) ; f2 <- rep(1:0, 2)
(dt1 <- c(f1,f1,f2,f1,f2,f2,f1))
(vlmc.dt1c01 <- vlmc(dts = dt1, cutoff.prune = 0.1))
(dvlmc <- as.dendrogram(vlmc.dt1c01))
%% ALSO: Should allow (and use by default!) a "hang = +1"
str(dvlmc)
%%% FIXME !!! ---
## not so useful:
plot(dvlmc, nodePar= list(pch=c(1,16)))
## complete disaster:
plot(dvlmc, type ="tr", nodePar= list(pch=c(1,16)))

## but this is not (yet) so much better (want the same angles to left
## and right!!
plot(dvlmc, type ="tr", nodePar = list(pch=c(1,16)), center=TRUE,
     main = format(vlmc.dt1c01$call))
mtext(paste("dt1 =", gsub(" ","",deparse(dt1,width=100))))
}
\keyword{graphs}% <<< INVALID for now!
\keyword{iplot}% (since that's one main use..)

\eof
\name{bnrf1}
\title{BNRF1 Gene DNA sequences: Epstein-Barr and Herpes}
\description{
  Two gene DNA data ``discrete time series'',
  \describe{
    \item{\code{bnrf1EB}}{the BNRF1 gene from the Epstein-Barr virus,}
    \item{\code{bnrf1HV}}{the BNRF1 gene from the herpes virus.}
  }
}
\alias{bnrf1EB}
\alias{bnrf1HV}
\usage{data(bnrf1)}
\format{
  The EB sequence is of \code{\link{length}} 3954, whereas the HV has
  3741 nucleotides.
  Both are \R \code{\link{factor}}s with the four levels
  \code{c("a","c","g","t")}.
}
\source{
  See the references, data are online at
  \url{http://anson.ucdavis.edu/~shumway/tsa.html}
}
\references{
  Shumway, R. and Stoffer, D. (2000)
  \emph{Time Series Analysis and its Applications}.
  Springer Texts in Statistics.
}
\author{Martin Maechler (packaging for \R).}
\examples{
data(bnrf1)
bnrf1EB[1:500]
table(bnrf1EB)
table(bnrf1HV)
n <- length(bnrf1HV)
table(t = bnrf1HV[-1], "t-1" = bnrf1HV[-n])

require(ts)
plot(as.integer(bnrf1EB[1:500]), type = "b")
\testonly{
 ftable(table( t = bnrf1HV[-(1:2)],
              "t-1" = bnrf1HV[-c(1,n)],
              "t-2" = bnrf1HV[-c(n-1,n)]))
 require(ts)
 lag.plot(jitter(as.ts(bnrf1HV)),lag = 4, pch = ".")
}

## Simplistic gene matching:
percent.eq <- sapply(0:200,
           function(i) 100 * sum(bnrf1EB[(1+i):(n+i)] ==  bnrf1HV))/n
plot.ts(percent.eq)
}
\keyword{datasets}

\eof
\name{deviance.vlmc}
\alias{deviance.vlmc}
\title{Compute the Deviance of a Fitted VLMC Object}
\description{
  Compute the Deviance, i.e., - 2 log[likelihood(*)] of a fitted VLMC
  object.  The log-likelihood is also known as ``entropy''.
}
\usage{
deviance.vlmc(object, ...)
}
\arguments{
 \item{object}{typically the result of \code{\link{vlmc}(..)}.}
 \item{\dots}{possibly further arguments (none at the moment).}
}
\value{
  A number, the deviance, i.e., \eqn{- 2 log.likelihood(*)}.
  where the  log.likelihood is really what we currently have
  as \code{\link{entropy}()}.

%% FIXME : need more here ???

}
\author{Martin Maechler}
\seealso{\code{\link{entropy}}, \code{\link{vlmc}},\code{\link{residuals.vlmc}}}
\examples{
example(vlmc)
deviance(vlmc.pres)

devianceR <- function(object)
{
    dn <- dimnames(pr <- predict(object))
    -2 * sum(log(pr[cbind(2:nrow(pr), match(dn[[1]][-1], dn[[2]]))]))
}
all.equal(deviance(vlmc.pres), devianceR(vlmc.pres), tol = 1e-14)
}
\keyword{ts}
\keyword{models}

\eof
% $Id: draw.vlmc.Rd,v 1.8 2003/10/02 10:13:04 maechler Exp $
\name{draw.vlmc}
\alias{draw.vlmc}
\alias{draw}
\title{Draw a "VLMC" Object (in ASCII) as tree.}
\description{
  Draws a \code{vlmc} object, typically the result of
  \code{\link{vlmc}(.)}, to the \R console,
  using one line per node.
}
\usage{
draw(x, \dots)
draw.vlmc(x, kind=3, flag = TRUE, show.hidden = 0,
          cumulative = TRUE, delta = cumulative, debug = FALSE, \dots)
}
\arguments{
  \item{x}{typically the result of \code{\link{vlmc}(..)}.}
  \item{kind}{integer code for the ``kind of drawing'', in \{0,1,2,3\}.}
  \item{flag}{logical; ..}
  \item{show.hidden}{integer code; if not \code{0}, give some
    indications about hidden (final) nodes .....}
  \item{cumulative}{logical indicating if the cumulative counts should
    be shown for nonterminal nodes; the `delta's can only be computed
    from the cumulative counts, i.e., \code{cumulative = FALSE} should
    be used only by the knowing one.}
  \item{delta}{logical indicating if `delta',
    i.e. \eqn{\delta}{delta}\eqn{(n, p(n))} should be computed and printed for
    each (non-root) node \eqn{n} with parent \eqn{p(n)}.  Note that this
  does not really make sense when \code{cumulative = FALSE}.}
  \item{debug}{logical; if \code{TRUE}, some extraneous progress
    information is printed to the \R console.}
  \item{\dots}{(potentially more arguments)}
}
\details{
 .............

%% FIXME -- DO THIS ---

 .............

 Note that the counts internally are stored ``non-cumulatively'',
 i.e. as \emph{difference} counts which is useful for likelihood (ratio)
 computations.  In the internal C code, the \emph{difference} counts are
 originally computed by the \code{comp_difference()} function after tree
 generation.  \code{draw(*, cumulative = TRUE)} internally calls the C
 function \code{cumulate()} for the cumulative sums.
}
\value{
  nothing is returned.}
\author{Martin Maechler}
\seealso{\code{\link{vlmc}}.}
\examples{
  example(vlmc)
  draw(vlmc.dt1c01)
  draw(vlmc.dt1c01, flag = FALSE)
  draw(vlmc.dt1c01, kind = 1)
  draw(vlmc.dt1)
  draw(vlmc.dt1, show = 3)
  draw(vlmc.dt1, cumulative = FALSE)
}
\keyword{ts}
\keyword{models}

\eof
\name{id2ctxt}
\alias{id2ctxt}
\title{VLMC Context ID Conversion}
\description{
  Utility for converting a \code{\link{vlmc}} state ID to the
  corresponding context.  Of rare interest to the average user.
}
\usage{
id2ctxt(id, m=nchar(alpha), alpha=NULL)
}
\arguments{
  \item{id}{integer, a context ID such as optionally returned by
    \code{\link{predict.vlmc}}.}
  \item{m}{integer, the alphabet length.  Defaults to
    \code{\link{nchar}(alpha)}, the alphabet size if that is given.}
  \item{alpha}{alphabet string}
}
\value{
  a list (if \code{alpha} is not specified) or character vector of the
  same length as \code{id}, giving the 
  context (as integer vector or single string) of the corresponding id
}
\seealso{\code{\link{predict.vlmc}(*, type = "ID")}.}
\examples{
    id2ctxt(c(2,3,5,9), alpha = "Ab")
str(id2ctxt(c(2,3,5,9), 2))
}
\keyword{utilities}

\eof
\name{int2char}
\alias{int2char}
\alias{char2int}
\title{Character - Integer Conversion}
\description{
  Simple conversion utilities for character to integer conversion and
  vice versa.
}
\usage{
int2char(i, alpha)
char2int(x, alpha)
}
\arguments{
  \item{i}{integer vectors, typically in \code{0:m} when \code{alpha}
    has \eqn{m + 1} letters.}
  \item{alpha}{character string with several letters, representing the
    alphabet.}
  \item{x}{character string, typically with letters from \code{alpha}.}
}
\value{
  \code{int2char()} gives a string (length 1 character) with as many
  characters as \code{length(i)}, by 0-indexing into the alphabet
  \code{alpha}.

  \code{char2int()} gives an integer vector of length \code{nchar(x)}
  of integer codes according to \code{alpha} (starting at 0 !).

}
\seealso{\code{\link{int2alpha}()} (which is used by \code{int2char})
  and its inverse, \code{\link{int2alpha}()}, both working with vectors
  of \emph{single} characters instead of multi-character strings.
}
\examples{
char2int("vlmc", paste(letters, collapse=""))

int2char(c(0:3, 3:1), "abcd")
int2char(c(1:0,3,3), "abc") # to eat ;-)
}
\keyword{character}
\keyword{utilities}

\eof
\name{logLik}
\alias{logLik.vlmc}
\alias{entropy}
\alias{entropy2}
\title{Log Likelihood of and between VLMC objects}
\description{
  Compute the log-likelihood or ``entropy'' of a fitted
  \code{\link{vlmc}} object.  This is a method for the
  generic \code{\link{logLik}}.
}
\usage{
entropy(object)
\method{logLik}{vlmc}(object, \dots)
entropy2(ivlmc1, ivlmc2, alpha.len = ivlmc1[1])
}
\arguments{
  \item{object}{typically the result of \code{\link{vlmc}(..)}.}
%
  \item{ivlmc1,ivlmc2}{two \code{vlmc} (sub) trees, see \code{\link{vlmc}}.}
  \item{alpha.len}{positive integer specifying the alphabet length.}
  \item{\dots}{(potentially more arguments; required by generic)}
}
\details{
  The \code{logLik.vlmc()} method computes the log likelihood for a fitted
  \code{\link{vlmc}} object.  \code{entropy} is an alias for
  \code{logLik} for reasons of back compatibility.

  \code{entropy2} is less clear ... ... [[[ FIXME ]]] ... ...
}
\value{
  a negative number, in some contexts typically further divided by
  \code{log(x$alpha.len)}.

  Note that the \code{logLik} method is used by the default method of
  the \code{\link{AIC}} generic function (from R version 1.4.x), and
  hence provides \code{AIC(object)} for vlmc objects.
}
\author{Martin Maechler}
\seealso{\code{\link{deviance.vlmc}},
  \code{\link{vlmc}}, \code{\link{draw.vlmc}}.}
\examples{
dd <- cumsum(rpois(999, 1.5)) \%\% 10
(vd <- vlmc(dd))
logLik(vd)

## AIC model selection:
f1 <- c(1,0,0,0)  # as in example(vlmc)
f2 <- rep(1:0,2)
(dt1 <- c(f1,f1,f2,f1,f2,f2,f1))
AIC(print(vlmc(dt1)))
AIC(print(vlmc(dt1, cutoff = 2.6)))
AIC(print(vlmc(dt1, cutoff = 0.4)))# these two differ ``not really''
AIC(print(vlmc(dt1, cutoff = 0.1)))

## Show how to compute it from the fitted conditional probabilities :
logLikR <- function(x) {
    dn <- dimnames(pr <- predict(x))
    sum(log(pr[cbind(2:nrow(pr), match(dn[[1]][-1], dn[[2]]))]))
}

all.equal(  logLikR(vd),
          c(logLik (vd)), tol=1e-10) # TRUE, they do the same

## Compare different ones:  [cheap example]:
example(draw)
for(n in ls())
  if(is.vlmc(get(n))) {
       vv <- get(n)
       cat(n,":",formatC(logLik(vv) / log(vv$alpha.len),
                         format= "f", wid=10),"\n")
  }
}
\keyword{ts}
\keyword{models}

\eof
% $Id: predict.vlmc.Rd,v 1.10 2003/10/27 18:00:43 maechler Exp $
\name{predict.vlmc}
\title{Prediction of VLMC for (new) Series}
\description{
  Compute predictions on a fitted VLMC object
  for each (but the first) element of another discrete time series.
  Computes by default a matrix of prediction probabilities.  The argument
  \code{type} allows other predictions such as the most probable
  \code{"class"} or \code{"response"}, the context length (tree
  \code{"depth"}), or an \code{"ID"} of the corresponding context.
}
\alias{print.predict.vlmc}
\alias{predict.vlmc}
\alias{fitted.vlmc}
\usage{
\method{predict}{vlmc}(object, newdata,
         type = c("probs", "class","response", "id.node", "depth", "ALL"),
         se.fit = FALSE,%% dispersion = NULL, terms=NULL,
         allow.subset = TRUE, check.alphabet=TRUE,
         \dots)
\method{fitted}{vlmc}(object, \dots)
}
\arguments{
  \item{object}{typically the result of \code{\link{vlmc}(..)}.}
  \item{newdata}{a discrete ``time series'', a numeric, character or
    factor, as the \code{dts} argument of \code{\link{vlmc}(.)}.}

  \item{type}{character indicating the type of prediction required,
    options given in the \emph{Usage} secion above, see also the
    \emph{Value} section below.  The default \code{"probs"}
    returns a matrix of prediction probabilties, whereas \code{"class"}
    or \code{"response"} give the corresponding most
    probable class.  The value of this argument can be abbreviated.
  }
  \item{se.fit}{a switch indicating if standard errors are required.
    \cr --- NOT YET supported --- .}
%   \item{dispersion}{the dispersion of the GLM fit to be assumed in
%     computing the standard errors.  If omitted, that returned by
%     \code{summary} applied to the object is used.}
%   \item{terms}{with \code{type="terms"} by default all terms are returned.
%     A vector of strings specifies which terms are to be returned}

  \item{allow.subset}{logical; if \code{TRUE}, \code{newdata} may not
    have all different ``alphabet letters'' used in \code{x}.}
  \item{check.alphabet}{logical; if \code{TRUE}, consistency of
    \code{newdata}'s alphabet with those of \code{x} is checked.}
  \item{\dots}{(potentially further arguments) required by generic.}
}
\note{
  The \code{predict} method and its possible arguments are still in active
  development, and we are considering to return the marginal
  probabilities instead of \code{NA} for the first value(s).
}
\value{
  Depending on the \code{type} argument,
  \item{"probs"}{an \eqn{n \times m}{n * m} matrix \code{pm} of (prediction)
    probabilities, i.e., all the rows of \code{pm} sum to 1.

    \code{pm[i,k]} is
%%-- TODO : first row, instead of NA, use marginals  Pr[ Y[i] = k ] !!
    \eqn{\hat P[Y_i = k | Y_{i-1},\dots]}{Pr(Y[i]=k | Y[i-1,...])} (and
    is therefore \code{\link{NA}} for \code{i=1}).
    The \code{\link{dimnames}} of \code{pm} are the values of
    \code{newdata[]} and the alphabet letters \code{k}.}
  \item{"class", "response"}{the corresponding most probable value of Y[];
    as \code{factor} for \code{"class"} and as integer in
    \code{0:(m-1)} for \code{type = "response"}. If there is more than
    one most probable value, the first one is chosen.}
  \item{"id.node"}{an (integer) ``ID'' of the current context (= node
    of the tree represented VLMC).}
  \item{"depth"}{the context length, i.e., the depth of the
    Markov chain, at the current observation (of \code{newdata}).}

  \item{"ALL"}{an object of class \code{"predict.vlmc"}, a list with the
    following components,
    \describe{
      \item{ID}{integer vector as for \code{type = "id.node"},}
      \item{probs}{prediction probability matrix, as above,}
      \item{flags}{integer vector, non-zero for particular states only,
	rather for debugging.}

      \item{ctxt}{character, \code{ctxt[i]} a string giving the
	context (backwards) for \code{newdata[i]}, using alphabet
	letters.}
      \item{fitted}{character with fitted values, i.e., the alphabet letter
	with the highest probability, using \code{\link{max.col}} where
	ties are broken at random.}
      \item{alpha, alpha.len}{the alphabet (single string) and its
	length.}
    }
    which has its own print method (\code{print.predict.vlmc}).}
}
\seealso{\code{\link{vlmc}} and \code{\link{residuals.vlmc}}.  For
    simulation, \code{\link{simulate.vlmc}}.}
\examples{
f1 <- c(1,0,0,0)
f2 <- rep(1:0,2)
(dt2 <- rep(c(f1,f1,f2,f1,f2,f2,f1),2))

(vlmc.dt2c15  <- vlmc(dt2, cutoff = 1.5))
draw(vlmc.dt2c15)

## Fitted Values:
all.equal(predict(vlmc.dt2c15, dt2), predict(vlmc.dt2c15))
(pa2c15 <- predict(vlmc.dt2c15, type = "ALL"))

## Depth = context length  ([1] : NA) :
stopifnot(nchar(pa2c15 $ ctxt)[-1] ==
          predict(vlmc.dt2c15, type = "depth")[-1])

same <- (ff1 <- pa2c15 $ fitted) ==
        (ff2 <- int2alpha(predict(vlmc.dt2c15, type ="response"), alpha="01"))
which(!same) #-> some are different, since max.col() breaks ties at random!

ndt2 <- c(rep(0,6),f1,f1,f2)
predict(vlmc.dt2c15, ndt2, "ALL")

(newdt2 <- sample(dt2, 17))
pm <- predict(vlmc.dt2c15, newdt2, allow.subset = TRUE)
summary(apply(pm, 1, sum))# all 1

predict(vlmc.dt2c15, newdt2, type = "ALL")

data(bnrf1)
(vbnrf <- vlmc(bnrf1EB))
(pA <- predict(vbnrf, bnrf1EB[1:24], type = "ALL"))
 pc <- predict(vbnrf, bnrf1EB[1:24], type = "class")
 pr <- predict(vbnrf, bnrf1EB[1:24], type = "resp")
stopifnot(as.integer  (pc[-1])   == 1 + pr[-1],
          as.character(pc[-1]) == strsplit(vbnrf$alpha,NULL)[[1]][1 + pr[-1]])

##-- Example of a "perfect" fit -- just for illustration:
##			    the default, thresh = 2 doesn't fit perfectly(i=38)
(vlmc.dt2c0th1 <- vlmc(dt2, cutoff = 0, thresh = 1))

## "Fitted" = "Data" (but the first which can't be predicted):
stopifnot(dt2[-1] == predict(vlmc.dt2c0th1,type = "response")[-1])
}
\keyword{ts}
\keyword{models}

\eof
\name{prt.vvec}
\alias{prt.vvec}
\title{Recursively Print the VLMC Result Vector}
\description{
  This is an auxiliary function which recursively displays (prints) the
  integer result vector of a \code{\link{vlmc}} fit.
}
\usage{
prt.vvec(v, nalph, pad=" ")
}
\arguments{
  \item{v}{typically \code{x $ vlmc.vec[ -1]} where \code{x} is the
    result of \code{\link{vlmc}(*)}.}
  \item{nalph}{alphabet size; typically \code{x $ vlmc.vec[1]}.}
  \item{pad}{character, to be used for padding \code{paste(*, collapse=pad)}.}
}
\seealso{\code{\link{summary.vlmc}} which uses \code{prt.vvec}.}
\examples{
  example(vlmc)
  str(vv <- vlmc.dt1$vlmc)
  prt.vvec(vv[-1], n = 2)
  prt.vvec(vv[-1], n = 2, pad = " | ")
}
\keyword{utilities}



\eof
% $Id: residuals.vlmc.Rd,v 1.10 2002/11/25 17:29:50 maechler Exp $
\name{residuals.vlmc}
\alias{residuals.vlmc}
\title{Compute Residuals of a Fitted VLMC Object}
\description{
  Compute residuals of a fitted \code{\link{vlmc}} object.

  This is yet a matter of research and may change in the future.
}
\usage{
\method{residuals}{vlmc}(object,
        type = c("classwise",
                 "deviance", "pearson", "working", "response", "partial"),
        y = object$y, \dots)
}
\arguments{
 \item{object}{typically the result of \code{\link{vlmc}(..)}.}
 \item{type}{The type of residuals to compute, defaults to
   \code{"classwise"} which returns an \eqn{n \times m}{n x m} matrix,
   see below.  The other types only make sense when the discrete
   values of \code{y} are ordered which always includes the binary case
   (\eqn{m=2}).
   \cr
   The \code{"deviance"} residuals \eqn{r} are defined similarly as for logistic
   regression, see below.
   \cr
   "pearson", "working" and "response" are currently
   identical and give the difference of the underlying integer code (of
   the discrete data).
   %%
   \bold{Note} that \code{"partial"} residuals are not yet defined!
 }
 \item{y}{discrete time series with respect to which the residuals are
   to be computed.}
 \item{\dots}{possibly further arguments (none at the moment).}
}
\value{
  If \code{type = "classwise"} (the default), a numeric matrix of dimension
  \eqn{n \times m}{n x m} of values \eqn{I_{i,j} - p_{i,j}}{I[i,j] -
    p[i,j]} where the indicator \eqn{I_{i,j}}{I[i,j]} is 1 iff
  \code{y[i] == a[j]} and \code{a} is the alphabet (or levels) of
  \code{y}, and \eqn{p_{i,j}}{p[i,j]} are the elements of the estimated (1-step
  ahead) predicted probabilities, \code{p <- predict(object)}.
  Hence, for each \eqn{i}, the only positive residual stands for the
  observed class.

  For all other \code{type}s, the result is
  a numeric vector of the length of the original time-series (with first
  element \code{NA}).\cr
  For \code{type = "deviance"},
  \eqn{r_i = \pm\sqrt{-2\log(P_i)}}{r[i] = +- sqrt(-2 log(P[i]))}
  where \eqn{P_i}{P[i]} is the predicted probability for the i-th
  observation which is the same as \eqn{p_{i,y_i}}{p[i,y[i]]} above (now
  assuming \eqn{y_i \in \{1,2,\dots,m}{y[i] in \{1,2,...,m\}}).
  The sum of the squared deviance residuals \emph{is} the deviance of
  the fitted model.
}
\author{Martin Maechler}
\seealso{\code{\link{vlmc}},\code{\link{deviance.vlmc}}, and
  \code{\link{RCplot}} for a novel residual plot.}
\examples{
example(vlmc)
rp <- residuals(vlmc.pres)
stopifnot(all(abs(apply(rp[-1,],1,sum)) < 1e-15))
matplot(seq(presidents), rp, ylab = "residuals", type="l")
## ``Tukey-Anscombe'' (the following is first stab at plot method):
matplot(fitted(vlmc.pres), rp, ylab = "residuals", xaxt = "n",
        type="b", pch=vlmc.pres$alpha)
axis(1, at = 0:(vlmc.pres$alpha.len-1),
     labels = strsplit(vlmc.pres$alpha,"")[[1]])


summary(rd <- residuals(vlmc.pres, type = "dev"))
rd[1:7]
## sum of squared dev.residuals === deviance :
all.equal(sum(rd[-1] ^ 2),
          deviance(vlmc.pres))
\testonly{stopifnot(all.equal(sum(rd[-1] ^ 2),
                    deviance(vlmc.pres),tol=1e-12))
}
}
\keyword{ts}
\keyword{models}

\eof
\name{simulate.vlmc}
\alias{simulate.vlmc}
\title{Simulate a Discrete Time Series from fitted VLMC model}
\description{Simulate from fitted VLMC model -- basis of the VLMC bootstrap}
\usage{
simulate.vlmc(x, n, n.start = 64 * x$size["context"], integer.return=FALSE)
}
\arguments{
  \item{x}{typically the result of \code{\link{vlmc}(..)}.}
  \item{n}{non-negative integer, giving the length of the result.}
  \item{n.start}{the number of initial values to be
    discarded (because of initial effects).}
  \item{integer.return}{logical; if \code{TRUE}, the result will be an
    \code{\link{integer}} vector with values in \code{0:(k-1)};
    otherwise the resulting vector consists of letters from the alphabet
    \code{x$alpha}.}
}
\details{
  The \code{\link{.Random.seed}} is used and updated as with other random
  number generation routines such as \code{\link{rbinom}}.

  Note that if you want to simulate from a given start sequence
  \code{x0}, you'd use \code{\link{predict.vlmc}(x, x0, type= "response")}
  --- actually not quite yet.
}
\value{
  A vector of length \code{n}.  Either \code{\link{integer}} or
  \code{\link{character}}, depending on the \code{integer.return}
  argument, see above.
}
\author{Martin Maechler}
\seealso{\code{\link{vlmc}} and \code{\link{predict.vlmc}}.}
\examples{
example(vlmc)

simulate.vlmc(vlmc.dt1, 100)
simulate.vlmc(vlmc.dt1c01, 100, int = TRUE)
# n.start = 0: 1st few observations will resemble the data
simulate.vlmc(vlmc.dt1c01, 20, n.start=0, int = TRUE)

}
\keyword{ts}
\keyword{models}

\eof
\name{summary.vlmc}
\title{Summary of Fitted Variable Length Markov Chain (VLMC)}
\alias{summary.vlmc}
\alias{print.summary.vlmc}
\description{
  Compute (and print) a summary of a \code{vlmc} object which is
  typically the result of \code{\link{vlmc}(..)}.
}
\usage{
\method{summary}{vlmc}(object, \dots)
\method{print}{summary.vlmc}(x, digits = getOption("digits"),
         vvec.printing = FALSE, \dots)
}
\arguments{
  \item{object}{an \R object of class \code{vlmc}.}
  \item{x}{an \R object of class \code{summary.vlmc}.}
  \item{digits}{integer giving the number of significant digits for
    printing numbers.}
  \item{vvec.printing}{logical indicating if the \code{vvec} component
    should be printed recursively via \code{\link{prt.vvec}()}.}
  \item{\dots}{potentially further arguments [Generic].}
}
\value{
  \code{summary.vlmc()} returns an object of class \code{"summary.vlmc"}
  for which there's a print method. It is basically a list containing
  all of \code{object}, plus additionally
  \item{confusion.table}{the symmetric contingency table of data vs fitted.}
  \item{depth.stats}{statistics of Markov chain depth along the data;
    currently just \code{summary(predict(object, type="depth"))}.}
  \item{R2}{the \eqn{R^2} statistic, i.e. the percentage (in [0,1]) of
    correctly predicted data.}
}
\seealso{\code{\link{vlmc}}, \code{\link{draw.vlmc}}.}
\examples{
data(bnrf1)
vb <- vlmc(bnrf1EB)
svb <- summary(vb)
svb
\testonly{% needs options(expression = <larger>) :
try(## since it currently fails (".. nested too deeply") -- FIXME -- !
  print(svb, vvec.print = TRUE)
)
}
}
\keyword{ts}
\keyword{models}

\eof
% $Id: vlmc.Rd,v 1.10 2003/07/09 08:16:05 maechler Exp $
\name{vlmc}
\title{Fit a Variable Length Markov Chain (VLMC)}
\alias{vlmc}
\alias{print.vlmc}
\alias{is.vlmc}
\description{
  Fit a Variable Length Markov Chain (VLMC) to a discrete time series,
  in basically two steps:\cr
  First a large Markov Chain is generated containing (all if
  \code{threshold.gen = 1}) the context states of the time series.  In
  the second step, many states of the MC are collapsed by \emph{pruning}
  the corresponding context tree.
}
\usage{
vlmc(dts,
     cutoff.prune = qchisq(alpha.c, df=max(.1,alpha.len-1),lower.tail=FALSE)/2,
     alpha.c = 0.05,
     threshold.gen = 2,
     code1char = TRUE, y = TRUE, debug = FALSE, quiet = FALSE,
     dump = 0, ctl.dump = c(width.ct = 1+log10(n), nmax.set = -1) )

is.vlmc(x)
\method{print}{vlmc}(x, digits = max(3, getOption("digits") - 3), \dots)
}
\arguments{
  \item{dts}{a discrete ``time series''; can be a numeric, character or factor.}
  \item{cutoff.prune}{non-negative number; the cutoff used for pruning;
    defaults to half the \eqn{\alpha}-quantile of a chisq distribution,
    where \eqn{\alpha =} \code{alpha.c}, the following argument:}
  \item{alpha.c}{number in (0,1) used to specify \code{cutoff.prune} in
    the more intuitive \eqn{\chi^2} quantile scale; defaulting to 5\%.}
  \item{threshold.gen}{integer \code{>= 1} (usually left at 2).  When
    \emph{generating} the initial large tree, only generate nodes with
    \code{count >= threshold.gen}.}
  \item{code1char}{logical; if true (default), the data \code{dts} will
    be ..........FIXME...........}
  \item{y}{logical; if true (default), the data \code{dts} will be
    returned.  This allows to ensure that residuals
    (\code{\link{residuals.vlmc}}) and ``k-step ahead'' predictions can
    be computed from the result.}
  \item{debug}{logical; should debugging info be printed to stderr.}
  \item{quiet}{logical; if true, don't print some warnings.}
%%\item{check.memory}{logical; for finding memory leak -- really not official.}
  \item{dump}{integer in \code{0:2}.  If positive, the pruned tree is
    dumped to stderr; if 2, the initial \bold{un}pruned tree is dumped as well.}
  \item{ctl.dump}{integer of length 2, say \code{ctl[1:2]} controlling
    the above dump when \code{dump > 0}.  \code{ctl[1]} is the width
    (number of characters) for the ``counts'', \code{ctl[2]} the maximal
    number of set elements that are printed per node; when the latter is
    not positive (by default), currently \code{max(6, 15 - log10(n))} is used.}
%
  \item{x}{a fitted \code{"vlmc"} object.}
  \item{digits}{integer giving the number of significant digits for
    printing numbers.}
  \item{\dots}{potentially further arguments [Generic].}
}
\value{
  A \code{"vlmc"} object, basically a list with components
  \item{n}{length of data series when fit.}
  \item{threshold.gen, cutoff.prune}{the arguments (or their defaults).}
  \item{alpha.len}{the alphabet size.}
  \item{alpha}{the alphabet used, as one string.}
  \item{size}{a named integer vector of length (>=) 4, giving
    characteristic sizes of the fitted VLMC.  Its named components are
    \describe{
      \item{"ord.MC"}{the (maximal) order of the Markov chain,}
      \item{"context"}{the ``context tree size'', i.e., the number of
	leaves plus number of ``hidden nodes'',}
      %---> reference to the Annals paper !,
      \item{"nr.leaves"}{is the number of leaves, and}
      \item{"total"}{the number of integers needed to encode the VLMC
	tree, i.e., \code{length(vlmc.vec)} (see below).}
    }
  }
  \item{vlmc.vec}{integer vector, containing (an encoding of) the fitted
    VLMC tree.}
  \item{y}{if \code{y = TRUE}, the data \code{dts}, as
    \code{\link{character}}, using the letters from \code{alpha}.}
  \item{call}{the \code{\link{call}} \code{vlmc(..)} used.}
}
\note{
  Set \code{cutoff = 0, thresh = 1} for getting a ``perfect fit'',
  i.e. a VLMC which perfectly re-predicts the data (apart from the first
  observation).  Note that even with \code{cutoff = 0} some pruning may
  happen, for all (terminal) nodes with \eqn{\delta}{delta}=0.
}
\references{
  Buhlmann P. and Wyner A. (1998)
  Variable Length Markov Chains.
  \emph{Annals of Statistics} \bold{27}, 480--513.

  Mchler M. and Bhlmann P. (2003)
  Variable Length Markov Chains: Methodology, Computing and Software.
  accepted for publication in \emph{J. Computational and Graphical Statistics}.

  Mchler M. (2003)
  VLMC --- Implementation and \R interface; working paper.
  %%Journal of Statistical Software, \emph{not yet submitted}.
}
\author{Martin Maechler}
\seealso{\code{\link{draw.vlmc}},
  \code{\link{entropy}}, \code{\link{simulate.vlmc}} for ``VLMC bootstrapping''.
}
\examples{
%--------------- don't change this example too quickly; used by many others!----
f1 <- c(1,0,0,0)
f2 <- rep(1:0,2)
(dt1 <- c(f1,f1,f2,f1,f2,f2,f1))

(vlmc.dt1  <- vlmc(dt1))
 vlmc(dt1, dump = 1,
      ctl.dump = c(wid = 3, nmax = 20), debug = TRUE)
(vlmc.dt1c01 <- vlmc(dts = dt1, cutoff.prune = .1, dump=1))

data(presidents)
dpres <- cut(presidents, c(0,45,70, 100)) # three values + NA
table(dpres <- factor(dpres, exclude = NULL)) # NA as 4th level
vlmc.pres <- vlmc(dpres, debug = TRUE)
vlmc.pres

## alphabet & and its length:
vlmc.pres$alpha
stopifnot(
  length(print(strsplit(vlmc.pres$alpha,NULL)[[1]])) == vlmc.pres$ alpha.len
)
}
\keyword{ts}
\keyword{models}

\eof
\name{vlmc.version}
\alias{vlmc.version}
\title{Version of VLMC Package}
\description{
  Character string, giving the version number (and date)
  of the VLMC package.
}
\examples{
vlmc.version
\dontrun{
[1] "VLMC 1.3-7;  after $Date: 2004/05/05 16:41:47 $ UTC"
}
}
\keyword{data}

\eof
\name{vlmctree}
\alias{vlmctree}
\alias{.vvec2tree}
\alias{str.vtree}
\title{Compute the tree structure of a "vlmc" object}
\description{
  Compute the tree representation of a \code{"\link{vlmc}"} object as \R
  \code{\link{list}}.
}
\usage{
vlmctree(x)

\method{str}{vtree}(object, \dots)
.vvec2tree(vv, k, chk.lev)
}
\arguments{
  \item{x,object}{typically the result of \code{\link{vlmc}(..)}.}
  \item{vv}{integer vector encoding the fitted vlmc, typically
    \code{x$vlmc.vec[-1]}.}
  \item{k}{integer, the alphabet size.}
  \item{chk.lev}{integer internally used for consistency checking.}
  \item{\dots}{further arguments passed to or from methods.}
}
\details{
  \code{.vvec2tree} is the internal (recursive) function building up the
  tree.

  \code{str.vtree} is a method for the generic \code{\link{str}}
  function and typically for the output of \code{vlmctree()}. For each
  node, it gives the ``parenting level'' in braces and the counts.
}
\value{
  A \code{\link{list}} of \code{\link{class}} \code{"vtree"}
  representing the tree structure recursively.

  Each ``node'' of the tree is itself a list with
  components
  \item{level}{length-2 integer giving the level in \{0,1,\dots\},
    counted from the root (which is \code{0}) and the parenting level,
    i.e the longest branch.}
  \item{count}{integer vector of length \code{k} where \code{k} is the
    number of ``letters'' in the alphabet.}
  \item{total}{equals to \code{sum(* $ count)}.}
  \item{child}{a list (of length \code{k}) of child nodes or
    \code{\link{NULL}} (i.e. not there).}
}
\author{Martin Maechler}
\seealso{\code{\link{vlmc}}.}
\examples{
data(presidents)
dpres <- cut(presidents, c(0,45,70, 100)) # three values + NA
table(dpres <- factor(dpres, exclude = NULL)) # NA as 4th level

(vlmc.prc1 <- vlmc(dpres, cut = 1, debug = TRUE))
str(vv.prc1 <- vlmctree(vlmc.prc1))
}
\keyword{ts}
\keyword{models}

\eof
