\name{betareg}
\alias{betareg}
\title{Fitting beta regression models}
\description{
'betareg' is used to fit a regression model where the response is beta distributed using a parameterization of the beta law that is indexed by mean and dispersion parameters. The model is useful for situations where the variable of interest is continuous, restricted to the standard unit interval (0,1), and related to other variables through a regression structure. The regression parameters of the beta regression model are interpretable in terms of the mean of the response and, when the logit link is used, of an odds ratio, unlike the parameters of a linear regression that employs a transformed response. Estimation is performed by maximum likelihood. The log-likelihood function is maximized using the quasi-Newton BFGS algorithm with analytical first derivatives; the choice of initial values follows the proposal made by Ferrari and Cribari-Neto (2004). 
  }
\usage{
betareg(formula, link = "logit", data)
}
\arguments{
  \item{formula}{ a symbolic description of the model to be fit: response ~ covariates. }
  \item{link}{ a link function; the following links are available: logit, probit and cloglog. The default link function is logit. }
  \item{data}{ the 'data argument': a data frame containing the data. }
}
\value{
The function returns an object of class 'betareg' or 'lm'. The function 'summary' is used to obtain an estimate of the precision parameter (phi), and the pseudo R2 value.
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{br.fit}}, \code{\link{summary.betareg}}, \code{\link{predict.betareg}}, \code{\link{residuals.betareg}} }
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit1 <- betareg(y ~ X)
fit2 <- betareg(y ~ X, link = "probit")
summary(fit1)
par(mfrow=c(2,2))
plot(fit1)
}
\keyword{regression}

\eof
\name{br.fit}
\alias{br.fit}
\title{ Function to estimate the coefficients of the Beta Regression. }
\description{
  Function to estimate the coefficients of the Beta Regression.
}
\usage{
br.fit(x, y, link)
}
\arguments{
  \item{x}{ the terms matrix }
  \item{y}{ the response matrix }
  \item{link}{ an object returned by the make.link function. }
}
\value{
   The function returns the estimated coefficients of the regression, the estimation of the precision parameter phi, the fitted values, the residuals,the pseudo R2, the standard errors, the critical values of the normal distribution and the p-values of each coefficient and the phi parameter.
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{betareg}}, \code{\link{summary.betareg}}, \code{\link{predict.betareg}}, \code{\link{residuals.betareg}}  }
\examples{
x <- cbind(rnorm(10),rnorm(10))
y <- cbind(rbeta(10,1,1))
link = betareg(y ~ x)$funlink
br.fit(x,y,link)
}
\keyword{regression}

\eof
\name{plot.betareg}
\alias{plot.betareg}
\alias{plot}
\title{ Plot Diagnostics for an betareg Object }
\description{
  This function returns four plots: a plot of residuals against fitted values, a plot of standardized residuals against fitted values, a generalized leverage plot against fitted values and a plot of Cook's distances versus row labels.}
\usage{
plot.betareg(x, which = 1:4, caption = c("Deviance residuals vs indices of obs.", 
    "Standardized residuals vs indices of obs.", "Generalized leverage vs. Predicted values", "Cook's distance plot"), 
    panel = points, sub.caption = deparse(x$call), main = "", 
    ask = prod(par("mfcol")) < length(which) && dev.interactive(), 
    ..., id.n = 3, labels.id = names(residuals(x)), cex.id = 0.75)
}
\arguments{
  \item{x}{ Fitted model by betareg. }
  \item{which}{If a subset of the plots is required, specify a subset of the numbers '1:4'.}
  \item{caption}{Captions to appear above the plots.}
  \item{panel}{Panel function.  A useful alternative to 'points' is 'panel.smooth'.}
  \item{sub.caption}{common title-above figures if there are multiple; used as 'sub' (s.'title') otherwise.}
  \item{main}{title to each plot-in addition to the above 'caption'.}
  \item{ask}{logical; if `TRUE', the user is \emph{asked} before each plot, see `par(ask=.)'.}
  \item{...}{optional arguments.}
  \item{id.n}{number of points to be labelled in each plot, starting with the most extreme.}
  \item{labels.id}{vector of labels, from which the labels for extreme points will be chosen.  'NULL' uses observation numbers.}
  \item{cex.id}{magnification of point labels.}
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{betareg}}, \code{\link{br.fit}}, \code{\link{predict.betareg}}, \code{\link{residuals.betareg}} }
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit <- betareg(y ~ X)
par(mfrow=c(2,2))
plot(fit)
  }
}
\keyword{regression}

\eof
\name{prater}
\alias{prater}
\docType{data}
\title{Prater's gasoline data}
\description{
This dataset was collected by Prater (1956), its dependent variable is the proportion of crude oil after distilation and fractionation. This dataset was analyzed by Atkunson (1985), who used the linear regression model and noted that there is "indication that the error distribution is not quite symmetrical, giving rise to some unduly large and small residuals" (p. 60).
}
\usage{data(prater)}
\format{
  A data frame with 32 observations on 5 variables.
}
\source{
Prater, N.H. (1956). Estimate gasoline yields from crudes. New York: Springer-Verlag.
}
\references{
Atkinson, A.C. (1985). \emph{Plots, Transformations and Regression: An Introduction to Graphical Methods of Diagnostic Regression Analysis.} New York: Oxford University Press.
}
\examples{
data(prater)
fit <- betareg(V5/100 ~ V1 + V2 + V3 + V4, data=prater)
plot(fit)
}
\keyword{datasets}

\eof
\name{pratergrouped}
\alias{pratergrouped}
\docType{data}
\title{Grouped Prater's gasoline data}
\description{
  This dataset was collected by Prater (1956), its dependent variable is the proportion of crude oil after distilation and fractionation. This dataset was analyzed by Atkunson (1985), who used the linear regression model and noted that there is "indication that the error distribution is not quite symmetrical, giving rise to some unduly large and small residuals" (p. 60).  
}
\usage{data(prater)}
\format{
  A data frame with 32 observations on 11 variables.
}
\source{
  Prater, N.H. (1956). Estimate gasoline yields from crudes. New York: Springer-Verlag.
}
\references{
  Atkinson, A.C. (1985). \emph{Plots, Transformations and Regression: An Introduction to Graphical Methods of Diagnostic Regression Analysis.} New York: Oxford University Press.
}
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit1 <- betareg(y ~ X)
fit2 <- betareg(y ~ X, link = "probit")
plot(fit1)
}
\keyword{datasets}

\eof
\name{predict.betareg}
\alias{predict.betareg}
\alias{predict}
\title{ Predicted values from beta regression model. }
\description{
  This function returns the predicted values based on beta regression models.}
\usage{
predict.betareg(object, terms,...)
}
\arguments{
  \item{object}{ Fitted model using betareg }
  \item{terms}{ A matrix or a vector containing the values that you want to have the predict. }
  \item{...}{Optional arguments}
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{betareg}}, \code{\link{br.fit}}, \code{\link{summary.betareg}}, \code{\link{residuals.betareg}} }
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit <- betareg(y ~ X)
predict(fit)
predict(fit,c(2,3,4,5,6,7,8,9,10,11))
  }
}
\keyword{regression}

\eof
\name{residuals.betareg}
\alias{residuals.betareg}
\alias{residuals}
\title{ Residuals function for beta regression models. }
\description{
  This function returns the standardized residuals from beta regression models, deviance residuals or the usual residuals. 
}
\usage{
residuals.betareg(object, type=c("standardized", "usual", "deviance",...)
}
\arguments{
  \item{object}{ Fitted model using betareg. }
  \item{type}{ The desired type of residuals. This function returns by default the standardized residuals, also returns the deviance residuals and may return the usual residuals (which is the value minus estimated mean). }
  \item{...}{Optional arguments}
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{betareg}}, \code{\link{br.fit}}, \code{\link{summary.betareg}}, \code{\link{predict.betareg}} }
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit <- betareg(y ~ X)
residuals(fit)
residuals(fit,type="dev")
  }
}
\keyword{regression}

\eof
\name{summary.betareg}
\alias{summary.betareg}
\title{Summary method for Beta Regression}
\description{
  This function returns the coefficient estimates, the precision parameter phi estimate and the Pseudo R2.}
\usage{
summary.betareg(object,...)
}
\arguments{
  \item{object}{ Fitted model using betareg }
  \item{...}{Optional arguments}
}
\value{
  \item{coefficients}{a p by 4 matrix with the columns for the estimated coefficients, the std. errors, the critical values of a normal distribution and the corresponded p-values.}
  \item{estimated phi}{the estimated value of the precision parameter.}
  \item{Pseudo R2}{the value of the pseudo R2 which is described by Ferrari and Cribari-Neto(2004).}
}
\author{ Alexandre de Bustamante Simas, Department of Statistics, Federal University of Pernambuco (alexandredbs@cox.de.ufpe.br). }
\references{
FERRARI, S.L.P., CRIBARI-NETO, F. (2004). Beta regression for modeling rates and proportions. Journal of Applied Statistics, forthcoming.
}
\seealso{\code{\link{betareg}}, \code{\link{br.fit}}, \code{\link{predict.betareg}}, \code{\link{residuals.betareg}} }
\examples{
data(pratergrouped)
attach(pratergrouped)
y <- V11/100
X <- cbind(V1,V2,V3,V4,V5,V6,V7,V8,V9,V10)
fit <- betareg(y ~ X)
summary(fit)
  }
}
\keyword{regression}

\eof
