\name{SimulVals}
\alias{SimulVals}
\title{Pre-simulated Joint Binary Probabilities}
\description{
  This dataset provides a pre-fabricated result from
  \code{\link{simul.commonprob}} such that it is normally not necessary
  to use this (time consuming) function. The table is automatically
  loaded wby \code{\link{rmvbin}} when necessary.
}
\usage{
data(SimulVals)
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Friedrich Leisch}
\seealso{
  \code{\link{simul.commonprob}}, \code{\link{rmvbin}}
}
\keyword{datasets}

\eof
\name{bincorr2commonprob}
\alias{bincorr2commonprob}
\title{Convert Binary Correlation Matrix to Matrix of Joint
  Probabilities}
\description{
  Compute a matrix of common probabilites for a binary random vector
  from given marginal probabilities and correlations.
}
\usage{
bincorr2commonprob(margprob, bincorr)
}
\arguments{
  \item{margprob}{vector of marginal probabilities.}
  \item{bincorr}{matrix of binary correlations.}
}
\value{
  The matrix of common probabilities.  This has the probabilities that
  variable \eqn{i} equals 1 in element \eqn{(i,i)}, and the joint
  probability that variables \eqn{i} and \eqn{j} both equal 1 in element
  \eqn{(i,j)} (if \eqn{i \ne j}{i != j}).
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Friedrich Leisch}
\seealso{
  \code{\link{commonprob2sigma}},
  \code{\link{simul.commonprob}}.
}
\keyword{distribution}
\keyword{multivariate}

\eof
\name{check.commonprob}
\title{Check Joint Binary Probabilities}
\alias{check.commonprob}
\usage{
check.commonprob(commonprob)
}
\arguments{
  \item{commonprob}{Matrix of pairwise probabilities.}
}
\description{
  The main diagonal elements \code{commonprob[i,i]} are interpreted as
  probabilities \eqn{p_{A_i}}{p(A_i)} that a binary variable \eqn{A_i}
  equals 1. The 
  off-diagonal elements \code{commonprob[i,j]} are the probabilites
  \eqn{p_{A_iA_j}}{p(A_iA_j)} that both \eqn{A_i} and \eqn{A_j} are 1.

  This programs checks some necessary conditions on these probabilities
  which must be fulfilled in order that a joint distribution of the
  \eqn{A_i} with the given probabilities can exist.

  The conditions checked are
  \deqn{0 \leq p_{A_i} \leq 1}{0 <= p(A_i) <= 1}
  
  \deqn{\max(0, p_{A_i} + p_{A_j} - 1) \leq p_{A_iA_j} \leq
    \min(p_{A_i}, p_{A_j}), i \neq j}{
    max(0, p(A_i)+p(A_j)-1) <= p(A_iA_j) <= min(p(A_i), p(A_j)), i != j}

   \deqn{p_{A_i} + p_{A_j} + p_{A_k} - p_{A_iA_j} -p_{A_iA_k} - p_{A_jA_k}
     \leq 1, i \neq j, i \neq k, j \neq k}{
     p(A_i)+p(A_j)+p(A_k)-p(A_iA_j)-p(A_iA_k)-p(A_jA_k) <= 1,
     i != j, i != k, j != k}
  
}
\value{
  \code{check.commonprob} returns \code{TRUE}, if all conditions are
  fulfilled. The attribute \code{"message"} of the return value contains
  some information on the errors that were found.
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Andreas Weingessel}
\seealso{
  \code{\link{simul.commonprob}},
  \code{\link{commonprob2sigma}}}
\examples{
check.commonprob(cbind(c(0.5, 0.4), c(0.4, 0.8)))

check.commonprob(cbind(c(0.5, 0.25), c(0.25, 0.8)))

check.commonprob(cbind(c(0.5, 0, 0), c(0, 0.5, 0), c(0, 0, 0.5)))
}
\keyword{distribution}
\keyword{multivariate}

\eof
\name{commonprob2sigma}
\alias{commonprob2sigma}
\title{Calculate a Covariance Matrix for the Normal Distribution out of
  a Matrix of Joint Probabilites}
\usage{
commonprob2sigma(commonprob, simulvals)
}
\arguments{
  \item{commonprob}{matrix of pairwise probabilities.}
  \item{simulvals}{array received by \code{simul.commonprob}.}
}
\description{
  Computes a covariance matrix for a normal distribution which
  corresponds to a binary distribution with marginal probabilites given
  by \code{diag(commonprob)} and pairwise probabilities given by
  \code{commonprob}. 

  For the simulations the values of \code{simulvals} are used.

  If a non-valid covariance matrix is the result, the program stops with
  an error in the case of NA arguments and yields are warning message if
  the matrix is not positive definite.
}
\value{
  A covariance matrix is returned with the same dimensions as
  \code{commonprob}. 
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Friedrich Leisch}
\seealso{
  \code{\link{simul.commonprob}}
}
\examples{
m <- cbind(c(1/2,1/5,1/6),c(1/5,1/2,1/6),c(1/6,1/6,1/2))
sigma <- commonprob2sigma(m)
}
\keyword{distribution}
\keyword{multivariate}

\eof
\name{condprob}
\alias{condprob}
\title{Conditional Probabilities of Binary Data}
\usage{
condprob(x)
}
\arguments{
  \item{x}{matrix of binary data with rows corresponding to cases and
    columns corresponding to variables.}
}
\description{
  Returns a matrix containing the conditional probabilities
  \eqn{P(x_i=1|x_j=1)} where \eqn{x_i} corresponds to the \code{i}-th
  column of \code{x}. 
}
\author{Friedrich Leisch}
\keyword{distribution}

\eof
\name{ra2ba}
\alias{ra2ba}
\title{Convert Real Valued Array to Binary Array}
\usage{
ra2ba(x)
}
\arguments{
  \item{x}{array of arbitrary dimension}
}
\description{
  Converts all values of the real valued array \code{x} to binary values
  by thresholding at 0.
}
\author{Friedrich Leisch}
\examples{
x <- array(rnorm(10), dim=c(2,5))
ra2ba(x)
}
\keyword{array}

\eof
\name{rmvbin}
\alias{rmvbin}
\title{Multivariate Binary Random Variates}
\description{
  Creates correlated multivariate binary random variables by
  thresholding a normal distribution.  The correlations of the
  components can be specified either as common probabilities,
  correlation matrix of the binary distribution, or covariance matrix of
  the normal distribution.
}
\usage{
rmvbin(n, margprob, commonprob=diag(margprob),
       bincorr=diag(length(margprob)),
       sigma=diag(length(margprob)),
       colnames=NULL, simulvals=NULL)
}
\arguments{
  \item{n}{number of observations.}
  \item{margprob}{margin probabilities that the components are 1.}
  \item{commonprob}{matrix of probabilities that components \code{i} and
    \code{j} are simultaneously 1.}
  \item{bincorr}{matrix of binary correlations.}
  \item{sigma}{covariance matrix for the normal distribution.}
  \item{colnames}{vector of column names for the resulting observation
    matrix.}
  \item{simulvals}{result from \code{simul.commonprob}, a default data
    array is automatically loaded if this argument is omitted.}
}
\details{
  Only one of the arguments \code{commonprob}, \code{bincorr} and
  \code{sigma} may be specified.  Default are uncorrelated components.
  
  \code{n} samples from a multivariate normal distribution with mean and
  variance chosen in order to get the desired margin and common
  probabilities are sampled.  Negative values are converted to 0,
  positive values to 1.
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Friedrich Leisch}
\seealso{
  \code{\link{commonprob2sigma}},
  \code{\link{check.commonprob}},
  \code{\link{simul.commonprob}}
}
\examples{
## uncorrelated columns:
rmvbin(10, margprob=c(0.3,0.9))

## correlated columns
m <- cbind(c(1/2,1/5,1/6),c(1/5,1/2,1/6),c(1/6,1/6,1/2))
rmvbin(10,commonprob=m)

## same as the second example, but faster if the same probabilities are
## used repeatedly (commonprob2sigma rather slow)
sigma <- commonprob2sigma(m)
rmvbin(10,margprob=diag(m),sigma=sigma)
}
\keyword{multivariate}
\keyword{distribution}

\eof
\name{simul.commonprob}
\alias{simul.commonprob}
\title{Simulate Joint Binary Probabilities}
\description{
  Compute common probabilities of binary random variates generated by
  thresholding normal variates at 0.
}
\usage{
simul.commonprob(margprob, corr=0, method="integrate", n1=10^5, n2=10)
}
\arguments{
  \item{margprob}{vector of marginal probabilities.}
  \item{corr}{vector of correlation values for normal distribution.}
  \item{method}{either \code{"integrate"} or \code{"monte carlo"}.}
  \item{n1}{number of normal variates if method is \code{"monte carlo"}.}
  \item{n2}{number of repetitions if method is \code{"monte carlo"}.}
}
\details{
  The output of this function is used by \code{\link{rmvbin}}.  For all
  combinations of \code{marginprob[i]}, \code{marginprob[j]} and
  \code{corr[k]}, the probability that both components of a normal
  random variable with mean \code{qnorm(marginprob[c(i,j)])} and
  correlation \code{corr[k]} are larger than zero is computed.

  The probabilities are either computed by numerical integration of the
  multivariate normal density, or by Monte Carlo simulation.

  For normal usage of \code{\link{rmvbin}} it is not necessary to use
  this function, one simulation result is provided as variable
  \code{\link{SimulVals}} in this package and loaded by default.
}
\value{
  \code{simul.commonprob} returns an array of dimension
  \code{c(length(margprob), length(margprob), length(corr))}. 
}
\references{Friedrich Leisch, Andreas Weingessel and Kurt Hornik
  (1998). On the generation of correlated artificial binary
  data. Working Paper Series, SFB ``Adaptive Information Systems and
  Modelling in Economics and Management Science'', Vienna University of
  Economics, \url{http://www.wu-wien.ac.at/am}
}
\author{Friedrich Leisch}
\seealso{
  \code{\link{rmvbin}}
}
\examples{
simul.commonprob(seq(0,1,0.5), seq(-1,1,0.5), meth="mo", n1=10^4)

data(SimulVals)

}
\keyword{distribution}
\keyword{multivariate}

\eof
