\name{boa-internal}
\alias{boa.menu.analysis}
\alias{boa.menu.chains}
\alias{boa.menu.coda}
\alias{boa.menu.data}
\alias{boa.menu.file}
\alias{boa.menu.import}
\alias{boa.menu.par}
\alias{boa.menu.parms}
\alias{boa.menu.plot}
\alias{boa.menu.plotcoda}
\alias{boa.menu.plotdesc}
\alias{boa.menu.setpar}
\alias{boa.menu.stats}
\alias{boa.menu.window}
\alias{key}

\title{Internal BOA Functions}
\description{Internal BOA functions}

\usage{
boa.menu.analysis()
boa.menu.chains()
boa.menu.coda()
boa.menu.data()
boa.menu.file()
boa.menu.import()
boa.menu.par()
boa.menu.parms()
boa.menu.plot()
boa.menu.plotcoda()
boa.menu.plotdesc()
boa.menu.setpar(group)
boa.menu.stats()
boa.menu.window()
key(x = c(par("usr")[1:2] %*% c(0.01, 0.99)),
y = c(par("usr")[3:4] %*% c(0.01, 0.99)), ..., title = "", align = TRUE,
background = 0, border = 0, between = 1, corner = c(missing(x), 1), divide = 3,
transparent = FALSE, cex = par("cex"), cex.title = max(cex), col = par("col"),
lty = par("lty"), lwd = par("lwd"), font = par("font"), pch = par("pch"),
adj = 0, type = "l", size = 5, columns = 1, between.columns = 3, angle = 0,
density = -1, plot = TRUE, text.width.multiplier = 1)
}

\details{These are not to be called by the user}
\keyword{internal}

\eof
\name{boa.acf}
\alias{boa.acf}

\title{Autocorrelation Function}
\description{
  Computes lag autocorrelations for the parameters in an MCMC sequence.
}
\usage{
boa.acf(link, lags)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}}
  \item{lags}{Vector of lags at which to estimate the autocorrelation function.}
}

\value{
  A matrix whose columns and rows contain the estimated autocorrelation
  functions at the specified lags and the monitored parameters, respectively.}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot}}, \code{\link{boa.plot.acf}},
  \code{\link{boa.print.acf}} }

\keyword{internal}

\eof
\name{boa.batchMeans}
\alias{boa.batchMeans}

\title{Batch Means}
\description{
  Computes batch means for the parameters in an MCMC sequence.
}
\usage{
boa.batchMeans(link, size)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{size}{Number of iterations to include in each batch.}
}

\value{
  A matrix whose columns and rows contain the means within sequential batches
  and the monitored parameters, respectively.
}

\author{Brian J. Smith}

\keyword{internal}

\eof
\name{boa.chain}
\alias{boa.chain}

\title{Session Lists of MCMC Sequences}
\description{
  Extracts and sets the MCMC sequence(s) stored internally for use during a BOA
  session. Users can safely use this function to extract data. However, it
  should not be called directly to modify the stored MCMC sequence(s). Instead,
  data management should accomplished with the libboa.chain functions.
}
\usage{
boa.chain(...)
}

\arguments{
  \item{\dots}{A list may be given as the only argument, or a character string
  given as the only argument, or any number of arguments may be in the
  \code{<name> = <value>} form, or no argument at all may be given. See the
  Value and Side Effects sections for further details.}
}

\section{Parameter Names and Default Values}{
  \describe{
  \item{master = list()}{Master list containing all added MCMC sequences.
  Sequences may only be added to or deleted from the master list. The sequences
  in this list are never modified or used in any of the analyses. It serves as
  a template for the working list described below. The parameters and
  iterations may differ between the sequences. Each element in the list is a
  matrix whose columns and rows contain the monitored parameters and the MCMC
  iterations, respectively. The iteration numbers and parameter names are
  stored in the dimnames.}
  \item{master.support = list()}{List containing the support for each of the
  parameters in 'master'. Each element in the list is a matrix whose columns
  and rows contain the monitored parameters and the support (lower and upper
  limits), respectively. The default for each parameter is c(-Inf, Inf).}
  \item{work = master}{Working list used in all analyses. This list is
  essentially a modifiable copy of 'master'. All data management functions
  operate on the working list.}
  \item{work.support = master.support}{List containing the support for each of
  the parameters in 'work'.}
  \item{work.sync = TRUE}{Logical value indicating that the working list is an
  identical copy of the master list. This variable is automatically set to F
  after a successful call to \code{\link{boa.chain.subset}} or 
  \code{\link{boa.chain.collapse}}. While \code{work.sync = F}, only the master
  list will be updated if MCMC sequences are added or deleted. A call to
  \code{\link{boa.chain.reset}} will reset this variable to T and copy the master
  list to the working list.}
}}

\value{
If no arguments are supplied, a list of the current values is returned. If a
character string is given, the current value of the named variable is returned.
Otherwise, no values are returned.
}

\section{Side Effects}{
When variables are set, boa.chain() modifies the internal list \code{.boa.chain}.
If boa.chain() is called with either a list as the single argument, or with one
or more arguments in the \code{<name> = <value>} form, the variables specified
by the names in the arguments are modified.
}

\author{Brian J. Smith}

\keyword{internal}

\eof
\name{boa.chain.add}
\alias{boa.chain.add}

\title{Add MCMC Sequence}
\description{
  Adds an MCMC sequence to the session lists of sequences. Most users should
  not call this function directly. Passing improperly formatted data to this
  function will most likely break the functions in this library. The preferred
  method of adding a new sequence is to call the function \code{\link{boa.chain.import}}.
}
\usage{
boa.chain.add(link, lname)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{lname}{Character string giving the list name to use for the newly added
  MCMC sequence.}
}

\value{
  A logical value indicating that 'link' has been successfully added.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.import}} }

\keyword{utilities}

\eof
\name{boa.chain.collapse}
\alias{boa.chain.collapse}

\title{Concatenate MCMC Sequences}
\description{
  Concatenates together all of the MCMC sequences in the working session list
  of sequences. Only those parameters common to all sequences are kept.
}
\usage{
boa.chain.collapse()
}

\value{
  A logical value indicating that the MCMC sequences have been collapsed. If
  the sequences share no common parameters, no modifications are made to the
  working list and FALSE is returned.}

\author{Brian J. Smith}

\seealso{ code{\link{boa.chain.reset}}, code{\link{boa.chain.subset}} }

\keyword{utilities}

\eof
\name{boa.chain.del}
\alias{boa.chain.del}

\title{Delete MCMC Sequences}
\description{
  Delete MCMC sequences from the session list of sequences.
}
\usage{
boa.chain.del(lnames, pnames)
}

\arguments{
  \item{lnames}{Character vector giving the names of the MCMC sequences in the
  session list of sequences to be deleted. If omitted, no sequences are deleted.}
  \item{pnames}{Character vector giving the names of the parameters in the MCMC
  sequences to be deleted. If omitted, no parameters are deleted.}
}

\section{Side Effects}{
  The specified MCMC sequences are deleted from the session lists of sequences.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.chain.eval}
\alias{boa.chain.eval}

\title{Formulate New Parameter}
\description{
  Evaluates a user-specified expression to formulate a new parameter in each of
  the system lists of MCMC sequences.
}
\usage{
boa.chain.eval(expr, pname)
}

\arguments{
  \item{expr}{S expression object used to formulate the new parameter.
  Typically, the new parameter is a function of the existing parameters.}
  \item{pname}{Name given to the new parameter.}
}

\section{Side Effects}{
  The specified parameter is added to the session lists of MCMC sequences.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.chain.gandr}
\alias{boa.chain.gandr}

\title{Gelman and Rubin Convergence Diagnostics}
\description{
  Computes the Gelman and Rubin convergence diagnostics for a list of MCMC
  sequences. Estimates are calculated from the second half of each sequence.
}
\usage{
boa.chain.gandr(chain, chain.support, alpha, pnames, window, to)
}

\arguments{
  \item{chain}{List of matrices whose columns and rows contain the monitored
  parameters and the MCMC iterations, respectively. The iteration numbers and
  parameter names must be assigned to the dimnames.}
  \item{chain.support}{List of matrices whose columns and rows contain the
  monitored parameters and the support (lower and upper limits), respectively.}
  \item{alpha}{Quantile (1 - alpha / 2) at which to estimate the upper limit of
  the shrink factor.}
  \item{pnames}{Character vector giving the names of the parameters to use in
  the analysis. If omitted, all parameters are used.}
  \item{window}{Proportion of interations to include in the analysis. If
  omitted, 50\% are included.}
  \item{to}{Largest iteration to include in the analysis. If omitted, no upper
  bound is set.}
}

\value{
  \item{psrf}{A vector containing the Gelman and Rubin (uncorrected) potential
  scale reduction factors for the monitored parameters.}
  \item{csrf}{A matrix whose columns and rows are the Gelman and Rubin corrected
  scale reduction factors (i.e. shrink factor estimates at the median and
  specified quantile of the sampling distribution) and the monitored parameters,
  respectively. A correction of (df + 3) / (df + 1) is applied to the scale
  reduction factors.}
  \item{mpsrf}{A numeric value giving the multivariate potential scale reduction
  factor proposed by Brooks and Gelman.}
  \item{window}{A numeric vector with two elements giving the range of the
  iterations used in the analysis.}
}

\references{
  \enumerate{
  \item Brooks, S. and Gelman, A. (1998). General methods for monitoring
  convergence of iterative simulations. Journal of Computational and Graphical
  Statistics, 7(4), 434-55.
  \item Gelman, A. and Rubin, D. B. (1992). Inference from iterative
  simulation using multiple sequences. Statistical Science, 7, 457-72.
  }
}

\author{Brian J. Smith, Nicky Best, Kate Cowles}

\seealso{ \code{\link{boa.plot}}, \code{\link{boa.plot.bandg}},
  \code{\link{boa.plot.gandr}}, \code{\link{boa.print.gandr}} }

\keyword{internal}

\eof
\name{boa.chain.import}
\alias{boa.chain.import}

\title{Import MCMC Sequence}
\description{
  Reads an MCMC sequence from an external file and adds it to the session lists
  of sequences.
}
\usage{
boa.chain.import(prefix, path = boa.par("path"), type = "ASCII")
}

\arguments{
  \item{prefix}{Character string giving the prefix for the file(s) in which the
  MCMC sequence is stored.}
  \item{path}{Character string giving the directory path in which the file(s)
  subsides.}
  \item{type}{Character string specifying the type of data to be imported.
  Currently, the supported types are:
  \itemize{
  \item "ASCII" : ASCII text file. Calls \code{\link{boa.importASCII}} to import
  data from "prefix.txt".
  \item "BUGS" : BUGS output files. Calls \code{\link{boa.importBUGS}} to import
  data from "prefix.ind" and "prefix.out".
  \item "S" : S data frame or matrix.
  } }
}

\value{
A logical value indicating that the specified file(s) has been successfully imported.
}

\section{Side Effects}{
  The imported MCMC sequence is automatically added to the session lists of
  sequences via a call to \code{\link{boa.chain.add}}.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.add}}, \code{\link{boa.importASCII}},
  \code{\link{boa.importBUGS}} }

\keyword{utilities}

\eof
\name{boa.chain.info}
\alias{boa.chain.info}

\title{Information about MCMC Sequences}
\description{
  Returns summary information from a list of MCMC sequences.
}
\usage{
boa.chain.info(chain, chain.support)
}

\arguments{
  \item{chain}{List of matrices whose columns and rows contain the monitored
  parameters and the MCMC iterations, respectively. The iteration numbers and
  parameter names must be assigned to the dimnames.}
  \item{chain.support}{List of matrices whose columns and rows contain the
  monitored parameters and the support (lower and upper limits), respectively.}
}

\value{
  \item{lnames}{Character vector giving the names of the MCMC sequences in the
  session list of sequences.}
  \item{pnames}{List of character vectors giving the parameter names in each
  of the MCMC sequences.}
  \item{iter}{List of numeric vectors giving the iterations from each MCMC
  sequence.}
  \item{iter.range}{Matrix whose columns give the range of the iterations for
  the MCMC sequences named in the rows.}
  \item{support}{List of numeric vectors giving the support for the parameters
  in each of the MCMC sequences.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.print.info}} }

\keyword{internal}

\eof
\name{boa.chain.reset}
\alias{boa.chain.reset}

\title{Reset MCMC Sequences}
\description{
  Returns the data in the working list of MCMC sequences to the state it was in
  when originally imported. This function undoes any subsetting or collapsing
  that was done via the \code{\link{boa.chain.collapse}} or 
  \code{\link{boa.chain.subset}} functions.
}
\usage{
boa.chain.reset()
}

\section{Side Effects}{
  Returns the data in the working list of MCMC sequences to the state it was in
  when originally imported. This function undoes any subsetting or collapsing
  that was done via the \code{\link{boa.chain.collapse}} or 
  \code{\link{boa.chain.subset}} functions.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.collapse}}, \code{\link{boa.chain.subset}} }

\keyword{utilities}

\eof
\name{boa.chain.subset}
\alias{boa.chain.subset}

\title{Subset MCMC Sequences}
\description{
  Selects a subset of the MCMC sequences stored in the working session list
  of sequences.
}
\usage{
boa.chain.subset(lnames, pnames, iter)
}

\arguments{
  \item{lnames}{Character vector giving the names of the MCMC sequences to
  include in the subset. If omitted, all sequences are included.}
  \item{pnames}{Character vector giving the names of the parameters to include
  in the subset. If omitted, all parameters are included.}
  \item{iter}{Character vector giving the names of the parameters to include
  in the subset. If omitted, all parameters are included.}
}

\value{
  A logical value indicating that the data has been successfully subsetted. If
  the requested subset contains no data, the working session list is not
  modified and FALSE is returned.
}

\section{Side Effects}{
  The subsetted data, if not an empty set, is copied to the working session
  list of MCMC sequences used in all analyses.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.collapse}}, \code{\link{boa.chain.reset}} }

\keyword{utilities}

\eof
\name{boa.chain.support}
\alias{boa.chain.support}

\title{Set Support for Parameters}
\description{
  Modifies the support (range of possible values) for the parameters in the
  session lists of MCMC sequences.
}
\usage{
boa.chain.support(lnames, pnames, limits)
}

\arguments{
  \item{lnames}{Character vector giving the names of the MCMC sequences over
  which to apply the changes. If omitted, changes are applied to all sequences.}
  \item{pnames}{Character vector giving the names of the parameters whose
  support is to be modified.}
  \item{limits}{Numeric vector with two elements giving the lower and upper
  limits, respectively, of the support for the specified parameters. Unbounded
  lower or upper limits should be specified as -Inf or Inf, respectively.}
}

\value{
  A logical vector with two elements indicating that the support has been
  changed for variables in the master list and working list of MCMC sequences,
  respectively.
}

\section{Side Effects}{
  The support for each of the specified parameters is changed in the session
  lists of MCMC sequences.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.getiter}
\alias{boa.getiter}

\title{Extract Iterations}
\description{
  Extracts a subset of the iterations in an MCMC sequence.
}
\usage{
boa.getiter(link, iter)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{iter}{Numeric vector containing the iterations to extract from 'link'.}
}

\value{
  A subset of 'link' containing all prior parameters and the iterations specified
  in 'iter'.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.iter}}, \code{\link{boa.getparms}} }

\keyword{internal}

\eof
\name{boa.getparms}
\alias{boa.getparms}

\title{Extract Parameters}
\description{
  Extracts a subset of the parameters in an MCMC sequence.
}
\usage{
boa.getparms(link, pnames)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{pnames}{Character vector containing the names of the parameters to
  extract from 'link'.}
}

\value{
  A subset of 'link' containing the specified parameters in 'pnames' and all
  prior iterations.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.getiter}}, \code{\link{boa.pnames}} }

\keyword{internal}

\eof
\name{boa.geweke}
\alias{boa.geweke}

\title{Geweke Convergence Diagnostics}
\description{
  Computes the Geweke convergence diagnostics for the parameters in an MCMC
  sequence.
}
\usage{
boa.geweke(link, p.first, p.last)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{p.first}{Proportion of iterations to include in the first window.}
  \item{p.last}{Proportion of iterations to include in the last window.}
}

\value{
  A matrix whose columns contain the Z-Scores and associated p-values and whose
  rows contain the monitored parameters.
}

\references{Geweke, J. (1992). Evaluating the accuracy of sampling-based
approaches to calculating posterior moments. In Bayesian Statistics 4, (ed. J.
M. Bernardo, J. O. Berger, A. P. Dawid, and A. F. M. Smith). Clarendon Press,
Oxford, UK.}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot}}, \code{\link{boa.plot.geweke}},
  \code{\link{boa.print.geweke}} }

\keyword{internal}

\eof
\name{boa.gewekePwr}
\alias{boa.gewekePwr}

\title{Geweke Spectral Density Estimation}
\description{
  Estimates the spectrum frequency for the parameters in an MCMC sequence.
}
\usage{
boa.gewekePwr(link)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
}

\value{
A vector of spectrum frequencies for the monitored parameters.
}

\author{Brian J. Smith}

\keyword{internal}

\eof
\name{boa.handw}
\alias{boa.handw}

\title{Heidelberger and Welch Convergence Diagnostics }
\description{
  Computes the Heidleberger and Welch convergence diagnostics for the
  parameters in an MCMC sequence.
}
\usage{
boa.handw(link, error, alpha)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters
  and the MCMC iterations, respectively. The iteration numbers and parameter
  names must be assigned to \code{dimnames(link)}.}
  \item{error}{Accuracy of the posterior estimates for the parameters.}
  \item{alpha}{Alpha level for the confidence in the sample mean of the
  retained iterations.}
}

\value{
A matrix whose columns and rows are the Heidleberger and Welch convergence
diagnostics (i.e. stationarity test, number of iterations to keep and to drop,
Cramer-von-Mises statistic, halfwidth test, mean, and halfwidth) and the
monitored parameters, respectively.
}
\references{Heidelberger, P. and Welch, P. (1983). Simulation run length control
in the presence of an initial transient. Operations Research, 31, 1109-44.}

\author{Brian J. Smith, Nicky Best, Kate Cowles}

\seealso{ \code{\link{boa.print.handw}} }

\keyword{internal}

\eof
\name{boa.hpd}
\alias{boa.hpd}

\title{Highest Probability Density Interval}
\description{
  Estimates the highest probability density (HPD) interval for the given
  parameter draws. Uses the Chen and Shao algorithm assuming a unimodal
  marginal posterior distribution.
}
\usage{
boa.hpd(x, alpha)
}

\arguments{
  \item{x}{MCMC draws from the marginal posterior to use in computing the HPD.}
  \item{alpha}{Specifies the 100*(1 - alpha)\% interal to compute.}
}

\value{
A vector containing the lower and upper bound of the HPD interval, labeled
"Lower Bound" and "Upper Bound", respectively.
}
\references{Chen, M-H. and Shao, Q-M. (1999). Monte Carlo estimation of Bayesian
credible and HPD intervals. Journal of Computational and Graphical Statistics,
8(1), 69-92.}
\author{Brian J. Smith}

\keyword{internal}

\eof
\name{boa.importASCII}
\alias{boa.importASCII}

\title{Import ASCII Data}
\description{
  Import data from an ASCII text file. The variables and iterations should
  appear in the columns and rows, respectively, of the file. The variable names
  must be given in the first row. The iteration numbers are taken from the the
  column entitled "iter" (case-sensitive), if it exists; otherwise, the numbers 
  1..n are used, where n is the number of row. The columns may be separated by 
  white space or tabs.
}
\usage{
boa.importASCII(prefix, path = NULL)
}

\arguments{
  \item{prefix}{Character string giving the prefix for the files in which the 
  ASCII data is stored. boa.importASCII() looks for the file "prefix.txt".}
  \item{path}{Character string giving the directory path in which the file
  subsides. This argument may be omitted if the file is located in the current
  working directory.  The specified path should not end with a slash(es).}
}

\value{
If the data is successfully imported, a matrix is returned whose columns and
rows contain the monitored parameters and the MCMC iterations, respectively.
The iteration numbers and parameter names are stored in the dimnames of the 
returned matrix. Otherwise, NULL is returned.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.import}}, \code{\link{boa.importBUGS}} }

\keyword{internal}

\eof
\name{boa.importBUGS}
\alias{boa.importBUGS}

\title{Import BUGS Data}
\description{
  Import data from BUGS output files.
}
\usage{
boa.importBUGS(prefix, path = NULL)
}

\arguments{
  \item{prefix}{Character string giving the prefix for the files in which the 
  BUGS output is stored. boa.importBUGS() looks for the two files "prefix.ind" 
  and "prefix.out".}
  \item{path}{Character string giving the directory path in which the file
  subsides. This argument may be omitted if the file is located in the current
  working directory.  The specified path should not end with a slash(es).}
}

\value{
If the data is successfully imported, a matrix is returned whose columns and 
rows contain the monitored parameters and the MCMC iterations, respectively. 
The iteration numbers and parameter names are stored in the dimnames of the 
returned matrix. Otherwise, NULL is returned.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.import}}, \code{\link{boa.importASCII}} }

\keyword{internal}

\eof
\name{boa.init}
\alias{boa.init}

\title{Start Session}
\description{
  boa.init() is the first function called to start a BOA session when using the
  command-line. It sets up the internal data structures and initializes them
  with the appropriate default values.
}
\usage{
boa.init()
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.load}}, \code{\link{boa.quit}} }

\keyword{utilities}

\eof
\name{boa.iter}
\alias{boa.iter}

\title{Extract Iteration Numbers}
\description{
  Extracts the iteration numbers from an MCMC sequence.
}
\usage{
boa.iter(link)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
}

\value{
A numeric vector containing the iteration numbers in 'link'.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.getiter}}, \code{\link{boa.pnames}} }

\keyword{internal}

\eof
\name{boa.license}
\alias{boa.license}

\title{Licensing Information}
\description{
  Prints the licensing information for the Bayesian Output Analysis Program
  (BOA).
}
\usage{
boa.license()
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.version}} }

\keyword{utilities}

\eof
\name{boa.load}
\alias{boa.load}

\title{Load Session}
\description{
  Loads the data and global parameter settings from a previously saved BOA
  session.
}
\usage{
boa.load(name, envir = globalenv())
}

\arguments{
  \item{name}{Character string giving the name of the object containing the
  session to be loaded.}
  \item{envir}{The 'environment' from which the object should come. For more
  information, consult the help documentation in R on the \code{get} function.}
}

\value{
A logical value indicating that the object was successfully loaded.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.menu}
\alias{boa.menu}

\title{Interactive BOA Menu}
\description{
  Starts the interactive menu for BOA.  Provides a menu-driven interface to
  all of the convergence diagnostics and statistical and graphical analysis
  tools.
}
\usage{
boa.menu(recover = FALSE)
}
\arguments{
  \item{recover}{
  a logical value for use if the previous BOA menu session
  terminated unexpectedly.  It may be possible to crash the menu system by
  supplying the wrong type of data.  In the event of a crash, \code{recover = TRUE}
  may be used to ensure that no data is lost.}
}

\note{
  \code{boa.menu} is the only function in the BOA library that need be called in
  order to use the menu-driven interface.
}

\section{Warning}{Do not use the \code{recover} argument unless starting the menu
  immediately after a crash.}

\author{Brian J. Smith}

\keyword{interface}

\eof
\name{boa.par}
\alias{boa.par}

\title{Global Parameters}
\description{
  Displays and sets the global parameters stored internally for use during a
  BOA session.
}
\usage{
boa.par(...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{\dots}{A list may be given as the only argument, or a character string 
  given as the only argument, or any number of arguments may be in the 
  \code{<name> = <value>} form, or no argument at all may be given.}
}

\section{Parameter Names and Default Values}{
  \describe{
  \item{acf.lags = c(1, 5, 10, 50)}{Numeric vector of lags at which to estimate 
  the autocorrelation function.}
  \item{alpha = 0.05}{Type I error rate used for all statistical tests and 
  confidence intervals.}
  \item{ASCIIext = ".txt"}{Character string giving the filename extension used 
  when importing ASCII files.}
  \item{bandwidth = function(x) 0.5 * diff(range(x)) / (log(length(x)) + 1)}{
  Function for computing the bandwidth used in estimating the density functions 
  for parameters. This should take one argument which is a numeric vector of 
  data on which density estimation is to be performed. A constant bandwidth may 
  be specified by having this function return the desired constant.}
  \item{batch.size = 50}{Number of iterations to include in each batch when 
  computing batch means and lag-1 batch autocorrelations. The batch size has 
  the single largest impact on the time required to compute summary statistics. 
  The computation time is decreased dramatically as the batch size increases. 
  Consequently, users may want to increase the value of this variable for long 
  MCMC sequences.}
  \item{dev = <<see below>>}{Character string giving the name of the function 
  that creates graphics windows on the current platform. For Unix systems this 
  is either "motif", "openlook", or "X11". The default is "motif" for the UNIX 
  S-PLUS, "win.graph" for Windows S-PLUS, "X11" for UNIX R, and "windows" for Windows R.}
  \item{dev.list = numeric(0)}{Numeric vector containing the active graphics 
  windows in use by the program. This is automatically maintained by the 
  program, user should not directly modify this variable.}
  \item{gandr.bins = 20}{Number of line segments within the MCMC sequence at 
  which to plot the Gelman and Rubin shrink factors.}
  \item{gandr.win = 0.50}{Proportion of iterations to include in the Brooks, 
  Gelman, and Rubin Statistics.}
  \item{geweke.bins = 10}{Number of line segments within the MCMC sequence at 
  which to plot the Geweke convergence diagnostics.}
  \item{geweke.first = 0.1}{Proportion of iterations to include in the first 
  window when computing the Geweke convergence diagnostics.}
  \item{geweke.last = 0.5}{Proportion of iterations to include in the last 
  window when computing the Geweke convergence diagnostics.}
  \item{handw.error = 0.1}{Accuracy of the posterior estimates when computing 
  the Heidleberger and Welch convergence diagnostics.}
  \item{kernel = "gaussian"}{Character string giving the type of window used in 
  estimating the density functions for parameters. Other choices are "cosine", 
  "rectangular", or "triangular".}
  \item{legend = TRUE}{Logical value indicating that a legend be included in 
  the plots.}
  \item{path = ""}{Character string giving the directory path in which the raw 
  data files are stored. The default may be used if the files are located in 
  the current working directory.  The specified path should not end with a
  slash(es).}
  \item{plot.mfdim = c(3, 2)}{Numeric vector giving the maximum number of rows 
  and columns, respectively, of plots to include in a single graphics window.}
  \item{plot.new = F}{Logical value indicating that a new graphics window be 
  automatically opened upon successive calls to boa.plot(). Otherwise, previous 
  graphics windows will be closed.}
  \item{plot.onelink = FALSE}{Logical value indicating that each plot should include 
  only one MCMC sequence. Otherwise, all sequences are displayed on the same plot.}
  \item{quantiles = c(0.025, 0.5, 0.975)}{Vector of probabilities at which to 
  compute the quantiles. Values must be between 0 and 1.}
  \item{randl.error = 0.005}{Desired amount of error in estimating the quantile 
  specified in the Raftery and Lewis convergence diagnostics.}
  \item{randl.delta = 0.001}{Delta valued used in computing the Raftery and Lewis 
  convergence diagnostics.}
  \item{randl.q = 0.025}{Quantile to be estimated in computing the Raftery and 
  Lewis convergence diagnostics.}
  \item{title = TRUE}{Logical value indicating that a title be added to the
  plots.}
}}

\value{
If no arguments are supplied, a list of the current values is returned. If a 
character string is given, the current value of the named variable is returned. 
Otherwise, a list of the named variables and their old values is returned, but 
not printed.
}

\section{Side Effects}{
When variables are set, boa.par() modifies the internal list .boa.par. If 
boa.par() is called with either a list as the single argument, or with one or 
more arguments in the <name> = <value> form, the variables specified by the 
names in the arguments are modified.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.pardesc}
\alias{boa.pardesc}

\title{Global Parameters Descriptions}
\description{
  Returns descriptive information on the global parameters.
}
\usage{
boa.pardesc()
}

\value{
A character matrix whose rows and columns ("group", "method", "desc", "par", 
"note") contain the global parameters and the corresponding descriptors 
(group = "Analysis", "Data", or "Plot"; method = subgroup classification;
desc = parameter description; par = parameter name name, note = information 
concerning the possible values for the parameter).}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.par}} }

\keyword{internal}

\eof
\name{boa.plot}
\alias{boa.plot}

\title{Plot Parameters in MCMC Sequences}
\description{
  Automatically generates the specified plot type for all parameters in the 
  working session list of MCMC sequences. This function takes care of all the 
  plotting tasks - opening windows, setting the number of plots per page, and 
  adding titles.
}
\usage{
boa.plot(type, dev = boa.par("dev"), mfdim = boa.par("plot.mfdim"),
newplot = boa.par("plot.new"), onelink = boa.par("plot.onelink"),
title = boa.par("title"))
}

\arguments{
  \item{type}{The type of plots to generate. The supported types are:
  \itemize{
  \item "acf" = autocorrelation functions
  \item "bandg" = Brooks and Gelman multivariate shrink factors
  \item "density" = density functions
  \item "gandr" = Gelman and Rubin shrink factors
  \item "geweke" = Geweke convergence diagnostics
  \item "history" = running means
  \item "trace" = trace histories
  }}
  \item{dev}{Character string giving the name of the function that creates 
  graphics windows on the current platform. For Unix systems this is either 
  "motif", "openlook", or "X11". For windows, this should be set to "win.graph"}
  \item{mfdim}{Numeric vector giving the maximum number of rows and columns, 
  respectively, of plots to include in a single graphics window. If only one 
  graphics window is opened, 'mfdim' is proportionately scaled down so as to 
  minimize the number of empty frames within that window.}
  \item{newplot}{Logical value indicating that a new graphics window be 
  automatically opened. Otherwise, previous graphics windows will be closed.}
  \item{onelink}{Logical value indicating that each plot should include only 
  one MCMC sequence. Otherwise, all sequences are displayed on the same plot.}
  \item{title}{Logical value indicating that a title be added to the plot.}
}

\value{
A logical value indicating that the plots were successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot.acf}}, \code{\link{boa.plot.bandg}},
  \code{\link{boa.plot.density}}, \code{\link{boa.plot.gandr}},
  \code{\link{boa.plot.geweke}}, \code{\link{boa.plot.history}},
  \code{\link{boa.plot.trace}} }

\keyword{hplot}

\eof
\name{boa.plot.acf}
\alias{boa.plot.acf}

\title{Plot Autocorrelation Function}
\description{
  Creates a single plot of the lag autocorrelations for a specified parameter.
}
\usage{
boa.plot.acf(lname, pname, legend = boa.par("legend"))
}

\arguments{
  \item{lname}{Character string giving the name of the desired MCMC sequence in 
  the working session list of sequences.}
  \item{pname}{Character string giving the name of the parameter to be plotted.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.acf}}, \code{\link{boa.print.acf}} }

\keyword{hplot}

\eof
\name{boa.plot.bandg}
\alias{boa.plot.bandg}

\title{Plot Brooks and Gelman Multivariate Shrink Factors}
\description{
  Plots the Brooks and Gelman multivariate shrink factors within different line 
  segments across the MCMC sequences. This diagnostic is a multivariate 
  extension to the Gelman and Rubin shrink factors.
}
\usage{
boa.plot.bandg(bins = boa.par("gandr.bins"), win = boa.par("gandr.win"),
legend = boa.par("legend"))
}

\arguments{
  \item{bins}{Number of line segments within the MCMC sequence at which to plot 
  the Gelman and Rubin shrink factors. The first segment contains the first 50 
  iterations; the remaining iterations are partitioned into equal bins and added 
  incrementally to construct the remaining line segments. The shrink factors are 
  plotted against the maximum iteration number for the segment. Cubic splines 
  are used to interpolate through the point estimates for each segment.}
  \item{win}{Proportion of iterations to include in the Brooks, Gelman, and 
  Rubin Statistics.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.gandr}}, \code{\link{boa.plot}},
  \code{\link{boa.plot.gandr}}, \code{\link{boa.print.gandr}} }

\keyword{hplot}

\eof
\name{boa.plot.close}
\alias{boa.plot.close}

\title{Close Graphics Windows}
\description{
  Closes graphics windows opened with boa.plot.open().
}
\usage{
boa.plot.close(which = dev.cur())
}

\arguments{
  \item{which}{Integer vector specifying the graphics windows to be closed.}
}

\value{
  If the specified graphics windows are successfully closed, the number of the 
  current graphics window is returned. Otherwise NULL is returned.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot.open}} }

\keyword{utilities}

\eof
\name{boa.plot.density}
\alias{boa.plot.density}

\title{Plot Density Functions}
\description{
  Estimates and displays, in a single plot, the density function(s) for the 
  specified parameter(s).
}
\usage{
boa.plot.density(lnames, pname, bandwidth = boa.par("bandwidth"),
window = boa.par("kernel"), legend = boa.par("legend"))
}

\arguments{
  \item{lnames}{Character vector giving the names of the desired MCMC sequence 
  in the working session list of sequences.}
  \item{pname}{Character string giving the name of the parameter to be plotted.}
  \item{bandwidth}{Function for computing the bandwidth used in estimating the 
  density functions for parameters. This should take one argument which is a 
  numeric vector of data on which density estimation is to be performed. A 
  constant bandwidth may be specified by having this function return the desired 
  constant.}
  \item{window}{Character string giving the type of window used in estimating 
  the density functions for the parameters. Available choices are "cosine", 
  "gaussian", "rectangular", or "triangular".}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot}} }

\keyword{hplot}

\eof
\name{boa.plot.gandr}
\alias{boa.plot.gandr}

\title{Plot Gelman and Rubin Shrink Factors}
\description{
  Creates a single plot of the Gelman and Rubin shrink factors within different 
  line segments for a parameter in an MCMC sequence.
}
\usage{
boa.plot.gandr(pname, bins = boa.par("gandr.bins"), alpha = boa.par("alpha"),
win = boa.par("gandr.win"), legend = boa.par("legend"))
}

\arguments{
  \item{pname}{Character string giving the name of the parameter in the working 
  session list of MCMC sequences to be plotted.}
  \item{bins}{Number of line segments within the MCMC sequence at which to plot 
  the Gelman and Rubin shrink factors. The first segment contains the first 50 
  iterations; the remaining iterations are partitioned into equal bins and 
  added incrementally to construct the remaining line segments. The shrink 
  factors are plotted against the maximum iteration number for the segment. 
  Cubic splines are used to interpolate through the point estimates for each 
  segment.}
  \item{alpha}{Quantile (1 - alpha / 2) at which to estimate the upper limit of 
  the shrink factor.}
  \item{win}{Proportion of iterations to include in the Brooks, Gelman, and 
  Rubin Statistics.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.gandr}}, \code{\link{boa.plot}},
  \code{\link{boa.plot.bandg}}, \code{\link{boa.print.gandr}} }

\keyword{hplot}

\eof
\name{boa.plot.geweke}
\alias{boa.plot.geweke}

\title{Plot Geweke Convergence Diagnostics}
\description{
  Creates a single plot of the Geweke convergence diagnostics within different 
  line segments for a parameter in an MCMC sequence.
}
\usage{
boa.plot.geweke(lname, pname, bins = boa.par("geweke.bins"), p.first = boa.par("geweke.first"), p.last = boa.par("geweke.last"), alpha = boa.par("alpha"), legend = boa.par("legend"))
}

\arguments{
  \item{lname}{Character string giving the name of the desired MCMC sequence in 
  the working session list of sequences.}
  \item{pname}{Character string giving the name of the parameter to be plotted.}
  \item{bins}{Number of line segments within the MCMC sequence at which to plot 
  the Geweke convergence diagnostics. The ith line segment contains the last 
  ((bins - i + 1) / bins)*100% of the iterations in the sequence, for i = 1 to 
  bins. This may lead to segments in which there are too few iterations to 
  compute the convergence diagnostics. Such segments, if they exist, are 
  automatically omitted from the calculations.}
  \item{p.first}{Proportion of iterations to include in the first window.}
  \item{p.last}{Proportion of iterations to include in the last window.}
  \item{alpha}{Alpha level for the rejection region lines drawn on the graph.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith, Nicky Best, Kate Cowles}

\seealso{ \code{\link{boa.geweke}}, \code{\link{boa.plot}},
  \code{\link{boa.print.geweke}} }

\keyword{hplot}

\eof
\name{boa.plot.history}
\alias{boa.plot.history}

\title{Plot Parameter Estimation Histories}
\description{
  Computes and displays, in a single plot, the running mean(s) for the 
  specified parameter(s).
}
\usage{
boa.plot.history(lnames, pname, legend = boa.par("legend"))
}

\arguments{
  \item{lnames}{Character vector giving the name of the desired MCMC sequence 
  in the working session list of sequences.}
  \item{pname}{Character vector giving the names of the parameters to be plotted.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot}} }

\keyword{hplot}

\eof
\name{boa.plot.open}
\alias{boa.plot.open}

\title{Open Graphics Windows}
\description{
  Opens a graphics window.
}
\usage{
boa.plot.open(which = boa.par("dev"))
}

\arguments{
  \item{which}{Character string giving the name of the function that creates 
  graphics windows on the current platform. For Unix systems this is either 
  "motif", "openlook", or "X11". For windows, "win.graph" should be specified.}
}

\value{
  If a graphics windows is successfully opened, the number of the current 
  graphics window is returned.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot.close}} }

\keyword{utilities}

\eof
\name{boa.plot.par}
\alias{boa.plot.par}

\title{Set Plotting Parameters}
\description{
  Set Plotting Parameters
}
\usage{
boa.plot.par(mfdim = c(1, 1), title = TRUE)
}

\arguments{
  \item{mfdim}{Numeric vector with two elements giving the number of rows and 
  column, respectively, of plots to display on the page. Plots will be drawn 
  row-by-row.}
  \item{title}{Logical value indicating that a title be added to the plot.}
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.plot.title}
\alias{boa.plot.title}

\title{Add Plot Title}
\description{
  Adds a title to the graphics window. This function should be called after all 
  plots have been added to the target graphics window.
}
\usage{
boa.plot.title(text)
}

\arguments{
  \item{text}{Character string giving the title to be centered and displayed 
  across the top of the graphics window.}
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.plot.trace}
\alias{boa.plot.trace}

\title{Plot Trace Histories}
\description{
  Displays, in a single plot, the trace histories for the specified parameter(s).
}
\usage{
boa.plot.trace(lnames, pname, legend = boa.par("legend"))
}

\arguments{
  \item{lnames}{Character vector giving the name of the desired MCMC sequence 
  in the working session list of sequences.}
  \item{pname}{Character string giving the name of the parameters to be plotted.}
  \item{legend}{Logical value indicating that a legend be included in the plot.}
}

\value{
  A logical value indicating that the plot was successfully created.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.plot}} }

\keyword{hplot}

\eof
\name{boa.pnames}
\alias{boa.pnames}

\title{Extract Parameter Names }
\description{
  Extracts the parameter names in an MCMC sequence.
}
\usage{
boa.pnames(link)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
}

\value{
A character vector containing the parameter names in 'link'.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.getparms}}, \code{\link{boa.iter}} }

\keyword{internal}

\eof
\name{boa.print.acf}
\alias{boa.print.acf}

\title{Print Autocorrelation Functions}
\description{
  Iteratively calls boa.acf() to display the lag autocorrelations for the 
  parameters in each of the MCMC sequences in the working session list of 
  sequences.
}
\usage{
boa.print.acf(lags = boa.par("acf.lags"))
}

\arguments{
  \item{lags}{Numeric vector of lags at which to estimate the autocorrelation 
  functions.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.acf}}, \code{\link{boa.plot}},
  \code{\link{boa.plot.acf}} }

\keyword{print}

\eof
\name{boa.print.cor}
\alias{boa.print.cor}

\title{Print Correlation Matrices}
\description{
  Iteratively computes and displays the correlation matrices for the parameters 
  in each of the MCMC sequences in the working session list of sequences.
}
\usage{
boa.print.cor()
}

\author{Brian J. Smith}

\keyword{print}

\eof
\name{boa.print.gandr}
\alias{boa.print.gandr}

\title{Print Gelman and Rubin Convergence Diagnostics}
\description{
  Calls boa.chain.gandr() and displays the Gelman and Rubin convergence 
  diagnostics for the MCMC sequences in the working session list of sequences.
}
\usage{
boa.print.gandr(alpha = boa.par("alpha"), win = boa.par("gandr.win"))
}

\arguments{
  \item{alpha}{Quantile (1 - alpha / 2) at which to estimate the upper limit of 
  the corrected shrink factor.}
  \item{win}{Proportion of iterations to include in the Brooks, Gelman, and 
  Rubin Statistics.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.gandr}}, \code{\link{boa.plot}},
  \code{\link{boa.plot.bandg}}, \code{\link{boa.plot.gandr}} }

\keyword{print}

\eof
\name{boa.print.geweke}
\alias{boa.print.geweke}

\title{Print Geweke Convergence Diagnostics}
\description{
  Iteratively calls boa.geweke() to display the Geweke convergence diagnostics 
  for the parameters in each of the MCMC sequences in the working session list 
  of sequences.
}
\usage{
boa.print.geweke(p.first = boa.par("geweke.first"), p.last = boa.par("geweke.last"))
}

\arguments{
  \item{p.first}{Proportion of iterations to include in the first window.}
  \item{p.last}{Proportion of iterations to include in the last window.}
}

\author{Brian J. Smiht}

\seealso{ \code{\link{boa.geweke}}, \code{\link{boa.plot}},
  \code{\link{boa.plot.geweke}} }

\keyword{print}

\eof
\name{boa.print.handw}
\alias{boa.print.handw}

\title{Print Heidelberger and Welch Convergence Diagnostics}
\description{
  Iteratively calls boa.handw() to display the Heidelberger and Welch 
  convergence diagnostics for the parameters in each of the MCMC sequences 
  in the working session list of sequences.
}
\usage{
boa.print.handw(error = boa.par("handw.error"), alpha = boa.par("alpha"))
}

\arguments{
  \item{error}{Accuracy of the posterior estimates for the monitored parameters.}
  \item{alpha}{Alpha level for the confidence in the sample mean of the retained 
  iterations.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.handw}} }

\keyword{print}

\eof
\name{boa.print.hpd}
\alias{boa.print.hpd}

\title{Print Highest Probability Density Intervals}
\description{
  Iteratively calls boa.hpd() to display the highest probability density (HPD) 
  intervals for the parameters in each of the MCMC sequences in the working 
  session list of sequences.
}
\usage{
boa.print.hpd(alpha = boa.par("alpha"))
}

\arguments{
  \item{alpha}{Specifies the 100*(1-alpha)\% HPD intervals to be computed.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.hpd}} }

\keyword{print}

\eof
\name{boa.print.info}
\alias{boa.print.info}

\title{Print Information about MCMC Sequences}
\description{
  Iteratively calls boa.chain.info() to display information from the MCMC 
  sequences in the specified session list of sequences.
}
\usage{
boa.print.info(which = "work")
}

\arguments{
  \item{which}{Character string specifying which session list of MCMC sequences 
  for which to print information. The two choices are "work" (used in all 
  analyses) or "data" (the template for "work").}
}

\value{
  The full list obtained from boa.chain.info() is returned, but not printed.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.chain.info}} }

\keyword{print}

\eof
\name{boa.print.par}
\alias{boa.print.par}

\title{Print Global Parameters' Descriptions and Current Values}
\description{
  Displays a description of and the current value for all global parameters.
}
\usage{
boa.print.par(group)
}

\arguments{
  \item{group}{Character string specifying which parameter group to display.
  The groups are "Analysis", "Data", and "Plot". If omitted, information on 
  all global parameters is displayed.}
}

\value{
  The character matrix obtained from boa.pardesc() is returned, but not printed.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.pardesc}} }

\keyword{print}

\eof
\name{boa.print.randl}
\alias{boa.print.randl}

\title{Print Raftery and Lewis Convergence Diagnostics}
\description{
  Iteratively calls boa.randl() to display the Raftery and Lewis convergence 
  diagnostics for the parameters in each of the MCMC sequences in the working 
  session list of sequences.
}
\usage{
boa.print.randl(q = boa.par("randl.q"), error = boa.par("randl.error"), prob = 1 - boa.par("alpha"), delta = boa.par("randl.delta"))
}

\arguments{
  \item{q}{Quantile to be estimated.}
  \item{error}{Desired amount of error in estimating the specified quantile 'q'.}
  \item{prob}{Probability of attaining the desired degree of 'error'.}
  \item{delta}{Delta value used in computing the convergence diagnostic.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.randl}} }

\keyword{print}

\eof
\name{boa.print.stats}
\alias{boa.print.stats}

\title{Print Summary Statistics}
\description{
  Iteratively calls boa.stats() to display summary statistics for the 
  parameters in each of the MCMC sequences in the working session list of 
  sequences.
}
\usage{
boa.print.stats(probs = boa.par("quantiles"), batch.size = boa.par("batch.size"))
}

\arguments{
  \item{probs}{Vector of probabilities at which to compute the quantiles. 
  Values must be between 0 and 1.}
  \item{batch.size}{Number of iterations to include in each batch when computing 
  batch means and lag-1 batch autocorrelation function.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.stats}} }

\keyword{print}

\eof
\name{boa.quit}
\alias{boa.quit}

\title{Quit Session}
\description{
  boa.quit() is called to end a BOA session. It removes the objects used to 
  store the session lists of MCMC sequences and the global parameters. 
  Failure to do this may result in lost or lingering data.
}
\usage{
boa.quit()
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.save}}, \code{\link{boa.init}} }

\keyword{utilities}

\eof
\name{boa.randl}
\alias{boa.randl}

\title{Raftery and Lewis Convergence Diagnostics }
\description{
  Computes the Raftery and Lewis convergence diagnostics for the parameters in 
  an MCMC sequence.
}
\usage{
boa.randl(link, q, error, prob, delta)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
  \item{q}{Quantile to be estimated.}
  \item{error}{Desired amount of error in estimating the specified quantile 'q'.}
  \item{prob}{Probability of attaining the desired degree of error - 'error'.}
  \item{delta}{Delta value used in computing the convergence diagnostics.}
}
\value{
A matrix whose columns and rows are the Raftery and Lewis convergence diagnostics 
(i.e. thin, burn-in, total, lower bound, and dependence factor) and the monitored 
parameters, respectively.
}
\references{
  \enumerate{
  \item Raftery, A. L. and Lewis, S. (1992a). Comment: One long run with diagnostics: 
  Implementation strategies for Markov chain Monte Carlo. Statistical Science, 7, 493-7.
  \item Raftery, A. L. and Lewis, S. (1992b). How many iterations in the Gibbs sampler? 
  In Bayesian Statistics 4, (ed. J. M. Bernardo, J. O. Berger, A. P. Dawid, and A. F. M. 
  Smith), pp. 763-74. Oxford University Press.
  }
}

\author{Brian J. Smith, Nicky Best, Kate Cowles}

\seealso{ \code{\link{boa.print.randl}} }

\keyword{internal}

\eof
\name{boa.save}
\alias{boa.save}

\title{Save Session}
\description{
  Save the current state of the session lists of MCMC sequences and the global 
  parameters to a database object.
}
\usage{
boa.save(name, envir = globalenv())
}

\arguments{
  \item{name}{Character string giving the name of the object to which the 
  current session should be saved.}
  \item{envir}{The 'environment' to which the object should be saved. For more 
  information, consult the help documentation in R on the assign() function.}
}

\value{
A logical value indicating that the session was successfully saved to the specified object.
}

\author{Brian J. Smith}

\keyword{utilities}

\eof
\name{boa.sortiter}
\alias{boa.sortiter}

\title{Sort by Iteration Numbers }
\description{
  Sorts an MCMC sequence by increasing iteration numbers.
}
\usage{
boa.sortiter(link)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
}

\value{
A copy of 'link' with the rows sorted by increasing iteration numbers.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.sortparms}} }

\keyword{internal}

\eof
\name{boa.sortparms}
\alias{boa.sortparms}

\title{Sort by Parameter Names }
\description{
  Sorts the parameters of an MCMC sequence alphabetically by parameter names.
}
\usage{
boa.sortparms(link)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
}

\value{
A copy of 'link' with the columns sorted alphabetically by parameter names.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.sortiter}} }

\keyword{internal}

\eof
\name{boa.stats}
\alias{boa.stats}

\title{Summary Statistics }
\description{
  Computes summary statistics for the parameters in an MCMC sequence.
}
\usage{
boa.stats(link, probs, batch.size)
}

\arguments{
  \item{link}{Matrix whose columns and rows contain the monitored parameters 
  and the MCMC iterations, respectively. The iteration numbers and parameter 
  names must be assigned to \code{dimnames(link)}.}
  \item{probs}{Vector of probabilities at which to compute the quantiles. 
  Values must be between 0 and 1.}
  \item{batch.size}{Number of iterations to include in each batch when 
  computing batch means and lag-1 batch autocorrelations.}
}

\value{
A matrix whose columns and rows contain the summary statistics (i.e. sample 
mean, standard deviation, naive standard error, MC error, batch standard error, 
lag-1 batch autocorrelation, specified quantiles, minimum and maximum iteration 
numbers, and total iterations in the sample) and the monitored parameters, 
respectively.
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.print.stats}} }

\keyword{internal}

\eof
\name{boa.transform}
\alias{boa.transform}

\title{Transform Variables}
\description{
  Applies a log or logit transformation to map bounded variables to the entire 
  real line.
}
\usage{
boa.transform(x, support, inv = FALSE)
}

\arguments{
  \item{x}{Numeric object to be transformed.}
  \item{support}{Vector giving the lower and upper limits of the support for 
  the untransformed data. Unbounded lower or upper limits should be specified 
  as -Inf or Inf, respectively.}
  \item{inv}{Logical value indicating that the inverse transformation should be 
  applied.}
}

\value{
Transformed or inverse-transformed analogue of 'x'.
}

\author{Brian J. Smith}

\keyword{internal}

\eof
\name{boa.version}
\alias{boa.version}

\title{Version Information}
\description{
  Information about this release of BOA.
}
\usage{
boa.version()
}

\value{
  \item{name}{Character string giving the program name.}
  \item{major}{Major number of this release.}
  \item{minor}{Minor number of this release.}
  \item{revision}{Revision number for this version.}
  \item{system}{Character string giving the platform for which this version 
  of BOA was designed.}
}

\author{Brian J. Smith}

\seealso{ \code{\link{boa.license}} }

\keyword{utilities}

\eof
\name{colVars}
\alias{colVars}

\title{Column Variances}
\description{
  Computes variances by column.
}
\usage{
colVars(x, na.rm = FALSE, unbiased = TRUE, SumSquares = FALSE)
}

\arguments{
  \item{x}{A matrix, array, or data frame}
  \item{na.rm}{Logical value indicating that missing values be omitted from the
  calculations.}
  \item{unbiased}{Not implemented}
  \item{SumSquares}{Not implemented}
}

\keyword{internal}

\eof
\name{line}
\docType{data}
\alias{line}
\alias{line1}
\alias{line2}

\title{BUGS Line Example}
\description{
The Line example involves a linear regression analysis of the data points (1,1),
(2,3), (3,3), (4,3), and (5,5).  The proposed Bayesian model is
\deqn{y[i] ~ N(mu[i], tau)}{y[i] ~ N(mu[i], tau)}
\deqn{mu[i] = alpha + beta * (x[i] - mean(x[]))}{mu[i] = alpha + beta * (x[i] - mean(x[]))}
with the following priors:
\deqn{alpha ~ N(0, 0.0001)}{alpha ~ N(0, 0.0001)}
\deqn{beta ~ N(0, 0.0001)}{beta ~ N(0, 0.0001)}
\deqn{tau ~ Gamma(0.001, 0.001)}{tau ~ Gamma(0.001, 0.001)}
Two parallel chains from the MCMC sampler were generated and the output saved in the 
S data frames line1 and line2. 
}
\usage{data(line)}
\format{The data frames line1 and line2 each containing 200 observations.}
\references{
Spiegelhalter, D., Thomas, A. and Best, N., (2000). WinBugs Version 1.4 User
Manual.
}
\keyword{datasets}

\eof
