.packageName <- "clim.pact"
# Computes approximate cartesian coordinates in km
# centered on 0E 65N.
# R.E. Benestad, DNMI, 04.01.2001
#
COn0E65N <- function(lon, lat) {
  a<-6357 # km
  if (length(lon) != length(lat)) {
    stop("lon and lat must have same length")
  }
  COn0E65N<-list(y=a * pi*(lat-65)/180,
                 x=a*pi*(lon)*cos(pi*lat/180)/180)
  COn0E65N
  }
EOF2field <- function(eof,anomalies=FALSE) {
  x <-t(eof$EOF) %*% diag(eof$W) %*% t(eof$PC); x <- t(x)
  dim(x) <- eof$size
  clim <- t(matrix(eof$clim,eof$size[3],eof$size[2]))
  if (!anomalies) { 
    for (i in 1:eof$size[1]) x[i,,] <- x[i,,] + clim
  }
  retrieve.nc  <- list(dat=x,lon=eof$lon,lat=eof$lat,tim=eof$tim,lev=NULL,
                       v.name=eof$v.name,id.x=eof$id.x,id.t=eof$id.t,
                       yy=eof$yy,mm=eof$mm,dd=eof$dd,n.fld=eof$n.fld,
                       id.lon=eof$id.lon,id.lat=eof$id.lat,
                       attributes=eof$attributes)
  class(retrieve.nc) <- class(eof)[-1]
  invisible(retrieve.nc)
}
# Writes an R/climpaxct-object as a CDF file that can
# be converted to netCDF through ncgen -b -o netcdf.file cdf.file
# R.E. Benestad

r2cdf <- function(filename,x,missing=-999.99,cleanup=TRUE,
                  ofs=NULL,scal=NULL) {

  if (class(x)=="field") {
    nt <- length(x$tim)
    ny <- length(x$lat)
    nx <- length(x$lon)
    neof <- NULL
  } else if (class(x)=="map") {
    nt <- 1
    ny <- length(x$lat)
    nx <- length(x$lon)
    x$dat <- x$map
    if (is.null(x$v.name)) x$v.name <- "map"
    if (is.null(x$attributes)) {
      x$attributes <- list(time.unit="unknown",
                           time.origin="unknown",
                           unit="unknown",
                           longname="unknown")
    }
    neof <- NULL
    x$map[!is.finite(x$map)] <- missing
    if (is.null(x$tim)) x$tim <- 0
  } else if (class(x)=="eof") {
    nt <- length(x$tim)
    ny <- length(x$lat)
    nx <- length(x$lon)
    x$dat <- x$EOF
    id <- row.names(table(x$id.x))
    neof <- length(x$W)
  } else {
    print("x has none of the classes: field, map, or EOF")
    return()
  }
  
  if ((is.null(ofs)) & (min(x$dat,na.rm=TRUE) > 0) & (class(x)=="field")) {
    ofs <- 10^(round(log(min(x$dat,na.rm=TRUE))/log(10)))
  } else ofs <- 0
  if ((is.null(scal)) & (class(x)=="field")) {
    max.dev <- range(x$dat[is.finite(x$dat)])
    scal <- 10^(-round(log(32000/(max.dev[2]-max.dev[1]))/log(10)))
  } else scal <- 1

  x$dat[!is.finite(x$dat)] <- missing
  cdf <- file(paste(filename,".cdf",sep=""),"w")
  cat("netcdf DNMI_slp {","dimensions:",file=cdf,sep = "\n")
  if (class(x)!="eof") {
    cat(paste("        Lon =",nx,";"),file=cdf,sep = "\n")
    cat(paste("        Lat =",ny,";"),file=cdf,sep = "\n")
  } else {
    for (i in 1:x$n.fld) {
      i.lon <- x$id.lon == id[i]
      i.lat <- x$id.lat == id[i]
      lon.x <- x$lon[i.lon]
      lat.x <- x$lat[i.lat]
      nx <- length(lon.x)
      ny <- length(lat.x)
      cat(paste("        Lon",i," = ",nx," ;",sep=""),file=cdf,sep = "\n")
      cat(paste("        Lat",i," = ",ny," ;",sep=""),file=cdf,sep = "\n")
    }
  }
  cat(paste("        Time =",nt,";"),file=cdf,sep = "\n")
  if (!is.null(neof)) {
    cat(paste("        eof =",neof,";"),file=cdf,sep = "\n")
  }

  cat("variables:",file=cdf,sep = "\n")
  if (class(x)!="eof") {
    cat("        float Lon(Lon) ;",file=cdf,sep = "\n")
    cat('                Lon:units = "degrees_east" ;',file=cdf,sep = "\n")
    cat('                Lon:modulo = " " ;',file=cdf,sep = "\n")
    cat('                Lon:long_name = "longitude" ;',file=cdf,sep = "\n")
    cat("        float Lat(Lat) ;",file=cdf,sep = "\n")
    cat('                Lat:units = "degrees_north" ;',file=cdf,sep = "\n")
    cat('                Lat:long_name = "latitude" ;',file=cdf,sep = "\n")
  } else {
    for (i in 1:x$n.fld) {
      i.lon <- x$id.lon == id[i]
      i.lat <- x$id.lat == id[i]
      lon.x <- x$lon[i.lon]
      lat.x <- x$lat[i.lat]
      nx <- length(lon.x)
      ny <- length(lat.x)
      cat(paste("        float Lon",i,"(Lon",i,") ;",sep=""),
          file=cdf,sep = "\n")
      cat(paste('                Lon',i,':units = "degrees_east" ;',sep=""),
          file=cdf,sep = "\n")
      cat(paste('                Lon',i,':modulo = " " ;',sep=""),file=cdf,
          sep = "\n")
      cat(paste('                Lon',i,':long_name = "longitude" ;',sep=""),
          file=cdf,sep = "\n")
      cat(paste("        float Lat",i,"(Lat",i,") ;",sep=""),file=cdf,
          sep = "\n")
      cat(paste('                Lat',i,':units = "degrees_north" ;',sep=""),
          file=cdf,sep = "\n")
      cat(paste('                Lat',i,':long_name = "latitude" ;',sep=""),
          file=cdf,sep = "\n")
    }
  }
  
  cat("        float Time(Time) ;",file=cdf,sep = "\n")
  cat(paste('                Time:units = "',x$attributes$time.unit,'" ;',
          sep=""),file=cdf,sep = "\n")
  cat(paste('                Time:time_origin = "',x$attributes$time.origin,
          '" ;',sep=""),file=cdf,sep = "\n")
  if (!is.null(neof)) {
    cat("        short eof(eof) ;",file=cdf,sep = "\n")
    cat('                eof:units = " " ;',file=cdf,sep = "\n")
    cat('                eof:long_name = "mode" ;',file=cdf,sep = "\n")
  }

  if (is.null(neof)) {
    cat(paste("        short ",x$v.name,"(Time, Lat, Lon) ;",sep=""),
        file=cdf,sep = "\n")
    cat(paste('              ',x$v.name,':units = "',x$attributes$unit,'" ;',sep=""),
        file=cdf,sep = "\n")
    cat(paste('              ',x$v.name,':longname = "',
              x$attributes$longname,'" ;',sep=""),
        file=cdf,sep = "\n")
    cat(paste('              ',x$v.name,':missing_value = ',missing,' ;',sep=""),
        file=cdf,sep = "\n")
    cat(paste('              ',x$v.name,':add_offset = ',ofs,' ;',sep=""),
        file=cdf,sep = "\n")
    cat(paste('              ',x$v.name,':scale_factor = ',scal,' ;',sep=""),
        file=cdf,sep = "\n")
  } else {
    for (i in 1:x$n.fld) {
      cat(paste("        float EOF",i,"(eof,Lat",i,",Lon",i,") ;",sep=""),
          file=cdf,sep = "\n")
      cat(paste('                EOF',i,':units = " " ;',sep=""),
          file=cdf,sep = "\n")
      cat(paste('                EOF',i,
                ':long_name = "Empirical Orthogonal Functions" ;',sep=""),
        file=cdf,sep = "\n")
    }
    cat("        float PC(Time,eof) ;",file=cdf,sep = "\n")
    cat('                PC:units = " " ;',file=cdf,sep = "\n")
    cat('                PC:long_name = "Principal Components" ;',
        file=cdf,sep = "\n")
    cat("        float lambda(eof) ;",file=cdf,sep = "\n")
    cat('                lambda:units = " " ;',file=cdf,sep = "\n")
    cat('                lambda:long_name = "Eigenvalues" ;',
        file=cdf,sep = "\n")
  }
    
  cat(" ",file=cdf,sep = "\n")
  cat("// global attributes:",file=cdf,sep = "\n")
  cat('                   :history = "Saved by r2cdf.R (clim.pact)" ;',
      file=cdf,sep = "\n")
  cat('                   :URL = "http://cran.r-project.org/" ;',
      file=cdf,sep = "\n")
      
  cat("data:",file=cdf,sep = "\n")
  cat(" ",file=cdf,sep = "\n")

  if (class(x)!="eof") {
    cat(" Lon = ",file=cdf,sep = " ")
    cat(as.character(round(x$lon,6)),file=cdf,sep = ", ")
    cat(";",file=cdf,sep = "\n")
    cat(" ",file=cdf,sep = "\n")
  
    cat(" Lat = ",file=cdf,sep = " ")
    cat(as.character(round(x$lat,6)),file=cdf,sep = ", ")
    cat(";",file=cdf,sep = "\n") 
    cat(" ",file=cdf,sep = "\n")
  } else {
    for (i in 1:x$n.fld) {
      i.lon <- x$id.lon == id[i]
      i.lat <- x$id.lat == id[i]
      lon.x <- x$lon[i.lon]
      lat.x <- x$lat[i.lat]
      nx <- length(lon.x)
      ny <- length(lat.x)
      cat(paste(" Lon",i," = ",sep=""),file=cdf,sep = " ")
      cat(as.character(round(lon.x,6)),file=cdf,sep = ", ")
      cat(";",file=cdf,sep = "\n")
      cat(" ",file=cdf,sep = "\n")
  
      cat(paste(" Lat",i," = ",sep=""),file=cdf,sep = " ")
      cat(as.character(round(lat.x,6)),file=cdf,sep = ", ")
      cat(";",file=cdf,sep = "\n") 
      cat(" ",file=cdf,sep = "\n")
    }
}
  cat(" Time = ",file=cdf,sep = " ")
  if (class(x)!="map"){
    cat("0",file=cdf,sep = ", ")
    cat(";",file=cdf,sep = "\n") 
    cat(" ",file=cdf,sep = "\n")
  } else {
    cat(as.character(x$tim),";",file=cdf,sep = "\n")
    cat(" ",file=cdf,sep = "\n")
  }
  if (!is.null(neof)) {
    cat(" eof = ",file=cdf,sep = " ")
    cat(as.character(1:neof),file=cdf,sep = ", ")
    cat(";",file=cdf,sep = "\n")
    cat(" ",file=cdf,sep = "\n")
  }

  if (class(x)=="map"){
    cat(paste(" ",x$v.name,sep=""),file=cdf,sep = " ")
    cat("= ",file=cdf,sep = "\n")
    for (j in 1:ny) {
      cat(as.character(round((x$map[j,]-ofs)/scal)),file=cdf,sep = ", ")
      if (j < ny) cat(", ",file=cdf,sep = "\n") else
                  cat("; ",file=cdf,sep = "\n")
    }
  } else   if (class(x)=="field"){
    cat(paste(" ",x$v.name,sep=""),file=cdf,sep = " ")
    cat("= ",file=cdf,sep = "\n")
    for (it in 1:nt) {
      for (j in 1:ny) {
        cat(as.character(round((x$dat[it,j,]-ofs)/scal)),file=cdf,sep = ", ")
        if ((it < nt) | (j < ny)) cat(", ",file=cdf,sep = "\n") else
                                  cat("; ",file=cdf,sep = "\n")
      }
    }
  } else {
    i.last <- 0
    for (i in 1:x$n.fld) {
#      print(i)
#      print(x$size[,i])
      i.fld <- seq(i.last+1,i.last+x$size[2,i]*x$size[3,i],by=1)
      i.last <- max(i.fld)
      EOF.1 <- x$EOF[,i.fld]
#      print(dim(EOF.1))
#      print(c(neof,ny,nx))
      
      dim(EOF.1)<-c(neof,x$size[2,i],x$size[3,i])
      EOF.1[!is.finite(EOF.1)] <- missing
      i.lon <- x$id.lon == id[i]
      i.lat <- x$id.lat == id[i]
      lon.x <- x$lon[i.lon]
      lat.x <- x$lat[i.lat]
      nx <- length(lon.x)
      ny <- length(lat.x)
      
      cat(paste(" EOF",i," = ",sep=""),file=cdf,sep = "\n")
      for (it in 1:neof) {
        for (j in 1:ny) {
          cat(as.character(round(EOF.1[it,j,],3)),file=cdf,sep = ",  ")
          if ((it < neof) | (j < ny)) cat(", ",file=cdf,sep = "\n") else
                                      cat("; ",file=cdf,sep = "\n")
        }
      }
      cat(" ",file=cdf,sep = "\n")
      
      cat(" PC = ",file=cdf,sep = "\n")
      for (it in 1:neof) {
        cat(as.character(round(x$PC[,it],3)),file=cdf,sep = ",  ")
        if (it < neof) cat(", ",file=cdf,sep = "\n") else
                       cat("; ",file=cdf,sep = "\n")
      }
      cat(" ",file=cdf,sep = "\n")
      
      cat(" lambda = ",file=cdf,sep = "\n")
      cat(as.character(round(x$W,4)),file=cdf,sep = ",  ")
      cat("; ",file=cdf,sep = "\n")
    }
  }
  cat("}",file=cdf,sep = "\n")
  close(cdf)
  
  system(paste("ncgen -b  -o ",filename,".nc ",filename,".cdf",sep=""),intern=T)

  if (cleanup) system(paste("rm -f ",filename,sep=""),intern=T)
}
# Overlays coast lines on contour plots/image plots. A mapping tool.
# Assumes that the x-axis and y-axis are given as degrees lon, lat.
#
# Reference: R.E. Benestad et al. (2002),
#            Empirically downscaled temperature scenarios for Svalbard,
#            submitted to Atm. Sci. Lett.
#
#            R.E. Benestad (2001),
#            A comparison between two empirical downscaling strategies,
#            Int. J. Climatology, 1645-1668, vol. 21, DOI 10.1002/joc.703
#
# R.E. Benestad, met.no, Oslo, Norway 16.04.2002
# rasmus.benestad@met.no
#------------------------------------------------------------------------


addland<-function(col="grey50",lwd=1) {

data("addland")
lines(lon.cont,lat.cont,type="l",col=col,lwd=lwd)
lon.cont<-lon.cont+360
lines(lon.cont,lat.cont,type="l",col=col,lwd=lwd)

}
# R.E. Benestad

anomaly.field <- function(x,period=NULL) {
  if ((class(x)[2]!="monthly.field.object") & (class(x)[2]!="field.object") &
      (class(x)[2]!="daily.field.object") & (class(x)[1]!="field")) {
      stop("Need a field.object") }
  nx <- length(x$lon)
  ny <- length(x$lat)
  nt <- length(x$tim)
  i.yy <- is.finite(x$yy)
  if (!is.null(period)) {
    i.yy <- (x$yy >= period[1]) & (x$yy <= period[2])
  }
  dd.rng <- range(x$dd)
  if ( (lower.case(substr(attr(x$tim,"unit"),1,5))=="month") |
       ((dd.rng[2]-dd.rng[1]<4) & (x$mm[2]-x$mm[1]>0)) ) {
    clim <- matrix(x$dat[1,,]*NA,ny,nx)
    for (im in 1:12) {
      it <- mod(1:nt,12)==mod(im,12) & i.yy
      for (j in 1:ny) {
        for (i in 1:nx) {
          clim[j,i] <- mean(x$dat[it,j,i],na.rm=TRUE)
          x$dat[it,j,i] <- x$dat[it,j,i] - clim[j,i]
        }
      }
    }
  } else {
    nt <- sum(i.yy)
#    time <- julian(x$mm,x$dd,x$yy)
    time <- julday(x$mm,x$dd,x$yy)
    x.1<-cos(2*pi*time/365.25)
    x.2<-sin(2*pi*time/365.25)
    x.3<-cos(4*pi*time/365.25)
    x.4<-sin(4*pi*time/365.25)
    x.5<-cos(6*pi*time/365.25)
    x.6<-sin(6*pi*time/365.25)
    dim(x$dat) <- c(nt,ny*nx)
    for (ip in seq(1,ny*nx,by=1)) {
      if (sum(is.finite(x$dat[,ip])) > 0) {
        calibrate <- data.frame(y=x$dat[i.yy,ip],x1=x.1[i.yy],x2=x.2[i.yy],
                       x3=x.3[i.yy],x4=x.4[i.yy],x5=x.5[i.yy],x6=x.6[i.yy])
        ac.fit<-lm(y ~ x1 + x2 + x3 + x4 + x5 + x6, data=calibrate)
        ac <- data.frame(x1=x.1,x2=x.2,x3=x.3,x4=x.4,x5=x.5,x6=x.6)
        x$dat[,ip]<- x$dat[,ip] - predict(ac.fit,newdata=ac)
      } else x$dat[,ip]<- rep(NA,nt)
    }
  dim(x$dat) <- c(nt,ny,nx)
  }

  attr(x$dat,'description') <- 'anomaly'
  invisible(x)  
}
# R.E. Benestad, met.no, Oslo, Norway 22.05.2002
# rasmus.benestad@met.no
#-------------------------------------------------------------------
# Estimate anomalies

anomaly.station <- function(obs,period=c(1961,1990)) {


cmon<-c("Jan","Feb","Mar","Apr","May","Jun",
        "Jul","Aug","Sep","Oct","Nov","Dec")

if (lower.case(class(obs)[2])=="monthly.station.record") {
  ny <- length(obs$yy)
  value <- t(obs$val)
  if (!is.null(period)) ii <- ((obs$yy>=period[1]) & (obs$yy<=period[2])) else
                        ii <- is.finite(obs$yy)
  for (im in 1:12) {
        value[im,] <- value[im,] - mean(value[im,ii],na.rm=TRUE)
      }
  obs$val <- t(value)
  obs$obs.name  <-  paste(obs$obs.name,"anomaly")
  invisible(obs)
  }  
}
avail.ds <- function(direc="output") {
  dir.0 <- getwd()
  if (file.exists(direc)) setwd(direc)
  avail.ds <- list.files(pattern=".Rdata")
  avail.ds <- avail.ds[grep("ds",avail.ds)]
  setwd(dir.0)
  avail.ds
}
# R.E. Benestad, met.no, Oslo, Norway 04.06.2002
# rasmus.benestad@met.no
#-------------------------------------------------------------------
# NORDKLIMstations.

avail.elem <- function() {

ele <- c(101,111,112,113,121,122,123,401,601,602,701,801,911)
ele.c <- c("TAM","TAX","Th","Thd","TAN","Tl","Tld","SLP","RR","RRX",
           "DSC","CLOUD","SDM")
nam <- c('mean T(2m)','mean maximum T(2m)','highest maximum T(2m)',
         'day of Th date Thd','mean minimum T(2m)','lowest minimum T(2m)',
         'day of Tl date Tld','mean SLP','monthly accum. precip.',
         'maximum precip.',
         'Number of days with snow cover (> 50% covered) days dsc',
         'Mean cloud cover % N',
         'mean snow depth')


avail.elem <- list(data.set="Nordklim",ele=ele,name=nam)
avail.elem
}
avail.eofs <- function(direc="data") {
  dir.0 <- getwd()
  if (file.exists(direc)) setwd(direc)
  avail.preds <- list.files(pattern=".Rdata")
  avail.preds <- avail.preds[grep("eof",avail.preds)]
  setwd(dir.0)
  avail.preds
}
# R.E. Benestad, met.no, Oslo, Norway 22.05.2002
# rasmus.benestad@met.no
#-------------------------------------------------------------------
# Selection of NACD and NORDKLIMstations.

avail.locs <- function(ele=101) {

#  source("strip.R")

  nacd.meta<-read.table('data/appendix.2')
  nordklim.meta<-read.fwf( 'data/nordklim_station_catalogue_v1_0.prn',
                 skip=1,as.is=TRUE,fill=TRUE,
                 width=c(2,30,12,11,11,4,3,2,4,3,2,
                         9,rep(6,23)),
                  col.names=c("i","location","height","country",
                              "number","Lat.deg","Lat.min","N.S",
                              "Lon.deg","Lon.min","E.W",
                              "ele101","ele101E","ele111","ele111E","ele112","ele112E",
                              "ele113","ele113E","ele121","ele121E","ele122","ele122E",
                              "ele123","ele123E","ele401","ele401E","ele601","ele601E",
                              "ele602","ele602E","ele701","ele701E","ele801","ele801E"))
  nacd <- length(nacd.meta$V5[is.element(nacd.meta$V14,ele)])
  iele <- eval(parse(text=paste("!is.na(nordklim.meta$ele",ele,")",sep="")))
  nnordklim <- sum(iele)
  loc.list <- c(as.character(nacd.meta$V5[is.element(nacd.meta$V14,ele)]),
                     strip(as.character(nordklim.meta$location[iele])))
  lat.list <- c(nacd.meta$V6[is.element(nacd.meta$V14,ele)] +
                1/60*nacd.meta$V7[is.element(nacd.meta$V14,ele)],
                nordklim.meta$Lat.deg[iele] + 1/60 * nordklim.meta$Lat.min[iele])
  lon.list <- c(nacd.meta$V9[is.element(nacd.meta$V14,ele)] +
                1/60*nacd.meta$V10[is.element(nacd.meta$V14,ele)],
                nordklim.meta$Lon.deg[iele] + 1/60 * nordklim.meta$Lon.min[iele])
  ew.list <- c(abbreviate(nacd.meta$V11[is.element(nacd.meta$V14,ele)]),
               abbreviate(nordklim.meta$E.W[iele]))
  lon.list[ew.list=="W"] <- lon.list[ew.list=="W"] * -1
  con.list <- c(as.character(nacd.meta$V3[is.element(nacd.meta$V14,ele)]),
                as.character(nordklim.meta$country[iele]))
  avail.locs<-list(name=loc.list,
                   lons=lon.list,
                   lats=lat.list,
                   country=factor(strip(abbreviate(con.list))),
                   nacd=nacd,nnordklim=nnordklim,
                   ident=c(rep("NACD",nacd),rep("NORDKLIM",nnordklim))) 
  avail.locs
}
avail.preds <- function(direc="data") {
  dir.0 <- getwd()
  if (file.exists(direc)) setwd(direc)
  avail.preds <- list.files(pattern=".Rdata")
  avail.preds <- avail.preds[grep("eof",avail.preds)]
  setwd(dir.0)
  avail.preds
}
# This routine computes the month, day, and year, given a Julian day.
# The algorithm is taken from Press et al. (1989), "Numerical Recipes 
# in Pascal", Cambridge, p. 13.
#
# This function removes the dependency to outdated packages 'chron' and
# 'date'.
#
# R.E. Benestad, met.no, Oslo, Norway 04.09.2003
# rasmus.benestad@met.no
#------------------------------------------------------------------------

caldat <- function(julian) {

  igreg=2299161
  julian <- trunc(julian)
  jalpha <- julian*0; ja <- julian*0
  im <-  (julian >= igreg)
  if (sum(im)>0) {
    jalpha[im] <- trunc(((julian-1867216) - 0.25)/36524.25) # Cross-over to Gregorian Calendar
                                                        # produces this correction.
    ja[im] <- julian+1+jalpha-trunc(0.25*jalpha)
  }
  im <- (julian < igreg)
  if (sum(im)>0) ja[im] <- julian[im]
  jb <- ja + 1524
  jc <- trunc(6680 + ((jb-2439870)-122.1)/365.25)
  jd <- 365*jc+trunc(0.25*jc)
  je <- trunc((jb-jd)/30.6001)
  id <- jb-jd-trunc(30.6001*je)
  mm <- je-1
  im  <- (mm > 12)
  if (sum(im)>0) mm[im] <- mm[im] - 12
  iyyy <- jc-4715
  im <-  (mm > 2)
  if (sum(im)>0) iyyy[im] <- iyyy[im] - 1
  im <-  (iyyy <= 0)
  if (sum(im)>0) iyyy <- iyyy - 1
  
  caldat <- list(month=mm,day=id,year=iyyy)
  invisible(caldat)
}
catFields <- function(field.1,field.2=NULL,lat=NULL,lon=NULL,
                       plot.interp=FALSE,interval.1=NULL,
                       interval.2=NULL,mon=NULL,demean=TRUE) {
  library(akima)
  l.one=FALSE
  l.newgrid <- FALSE
  if (is.null(field.2)) {
    l.one <- TRUE
    field.2 <- field.1
  }
  
  if (length(class(field.1))==2) {
    if (class(field.1)[1]=="mix.fields") {
      stop("Call mix.fields after catFields")
    }
  }
  if (class(field.1)[1] != class(field.2)[1]) {
    print(class(field.1)[1])
    print(class(field.2)[1])
    stop("The objects must have the same class")
  }

  tim.unit1 <- attr(field.1$tim,"unit")
  tim.torg1 <- attr(field.1$tim,"time_origin")
  tim.unit2 <- attr(field.2$tim,"unit")
  tim.torg2 <- attr(field.2$tim,"time_origin")
  if (is.null(tim.unit1)) tim.unit1<- "month"
  if (is.null(tim.unit2)) tim.unit2<- "month"
  if (lower.case(substr(tim.unit1,1,3)) != lower.case(substr(tim.unit2,1,3))) {
    print(c(tim.unit1,tim.unit2))
    stop('The time units must match')
  }
  
  if (!is.null(interval.1)) {
    print(interval.1)
    i1 <- ( (field.1$yy>=interval.1[1]) & (field.1$yy<=interval.1[2]))
    field.1$dat <- field.1$dat[i1,,]
    field.1$tim <- field.1$tim[i1]
    field.1$id.t <- field.1$id.t[i1]
    field.1$yy <- field.1$yy[i1]
    field.1$mm <- field.1$mm[i1]
    field.1$dd <- field.1$dd[i1]
    
  }
  if (!is.null(interval.2)) {
    print(interval.2)
    i2 <- ( (field.2$yy>=interval.2[1]) & (field.2$yy<=interval.2[2]))
    field.2$dat <- field.2$dat[i2,,]
    field.2$tim <- field.2$tim[i2]
    field.2$id.t <- field.2$id.t[i2]
    field.2$yy <- field.2$yy[i2]
    field.2$mm <- field.2$mm[i2]
    field.2$dd <- field.2$dd[i2]
  }
  if (!is.null(mon)) {
    cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
            'Jul','Aug','Sep','Oct','Nov','Dec')
    print(paste("Extract",cmon[mon]))
    i1 <- is.element(field.1$mm,mon)
    i2 <- is.element(field.2$mm,mon)
    field.1$dat <- field.1$dat[i1,,]
    field.1$tim <- field.1$tim[i1]
    field.1$id.t <- field.1$id.t[i1]
    field.1$yy <- field.1$yy[i1]
    field.1$mm <- field.1$mm[i1]
    field.1$dd <- field.1$dd[i1]
    field.2$dat <- field.2$dat[i2,,]
    field.2$tim <- field.2$tim[i2]
    field.2$id.t <- field.2$id.t[i2]
    field.2$yy <- field.2$yy[i2]
    field.2$mm <- field.2$mm[i2]
    field.2$dd <- field.2$dd[i2]
  }
  nt.1 <- length(field.1$tim)
  nx.1 <- length(field.1$lon)
  ny.1 <- length(field.1$lat)
  nt.2 <- length(field.2$tim)
  nx.2 <- length(field.2$lon)
  ny.2 <- length(field.2$lat)
  
#  print(paste("Field1: ",nt.1,nx.1,ny.1,"   Field2: ",nt.2,nx.2,ny.2))
  
  if (xor(min(field.1$lon)<0,min(field.2$lon)<0)) {
    if (min(field.2$lon)<0) {
      field.1$lon[field.1$lon > 180] <- field.1$lon[field.1$lon > 180]-360
      x.srt <- order(field.1$lon)
      field.1$lon <- field.1$lon[x.srt]
      field.1$dat <- field.1$dat[,,x.srt]
    } else {
      field.2$lon[field.2$lon > 180] <- field.2$lon[field.2$lon > 180]-360
      x.srt <- order(field.2$lon)
      field.2$lon <- field.2$lon[x.srt]
      field.2$dat <- field.2$dat[,,x.srt]
    }
  }
  field.1$dat[!is.finite(field.1$dat)] <- 0
  field.2$dat[!is.finite(field.2$dat)] <- 0

  if (demean) {
    for (j in 1:ny.1) {
      for (i in 1:nx.1) {
        field.1$dat[,j,i] <- field.1$dat[,j,i]-mean(field.1$dat[,j,i])
      }
    }
    for (j in 1:ny.2) {
      for (i in 1:nx.2) {
        field.2$dat[,j,i] <- field.2$dat[,j,i]-mean(field.2$dat[,j,i])
      }
    } 
  }
  if (!is.null(lat) & !is.null(lon)) {

    if (length(lat)==2) lat <- field.1$lat[field.1$lat >= min(lat) & field.1$lat <= max(lat)]
    if (length(lon)==2) lon <- field.1$lon[field.1$lon >= min(lon) & field.1$lon <= max(lon)]
    ny.1 <- length(lat); nx.1 <- length(lon)
    field.1$id.x <-  matrix(rep(field.1$v.nam,ny.1*nx.1),ny.1,nx.1)
    field.1$id.lon <- rep(field.1$v.nam,nx.1); field.1$id.lat <- rep(field.1$v.nam,ny.1)
    print("interpolate 1st field - please be patient :-)")
    l.newgrid <- TRUE
    lat.x<-rep(field.1$lat,length(field.1$lon))
    lon.x<-sort(rep(field.1$lon,length(field.1$lat)))
    dat.1<-matrix(nrow=nt.1,ncol=ny.1*nx.1)
    dim(dat.1)<-c(nt.1,ny.1,nx.1)
    for (it in 1:nt.1) {
      Z.in<-as.matrix(field.1$dat[it,,])
      Z.out<-interp(lat.x,lon.x,Z.in,lat,lon)
      dat.1[it,,]<-as.matrix(Z.out$z)
      if (plot.interp) {
        contour(field.1$lon,field.1$lat,t(round(Z.in,2)),col="blue",lwd=2,
                main=paste("Field 1: ",it,"/",nt.1),
                sub=paste(field.1$mm[it],"-",field.1$yy[it]))
        contour(lon,lat,t(round(Z.out$z,2)),add=TRUE,col="red",lty=2)
        addland()
        grid()
      }
    }
    
  } else {
    lat <- field.1$lat
    lon <- field.1$lon
    dat.1 <- field.1$dat
    nx.1 <- length(lon)
    ny.1 <- length(lat)
  }
  
  lat.x<-rep(field.2$lat,length(field.2$lon))
  lon.x<-sort(rep(field.2$lon,length(field.2$lat)))
  dat.2<-matrix(nrow=nt.2,ncol=ny.1*nx.1)
  dim(dat.2)<-c(nt.2,ny.1,nx.1)
  l.different <- TRUE
  if ( (ny.1==ny.2) & (nx.1==nx.2) ) {
    if ( (sum(field.1$lat==field.2$lat)==ny.1) &
         (sum(field.1$lon==field.2$lon)==nx.1) ) l.different <- FALSE
  }

#  print(c(l.one,l.different,l.newgrid))
  
  if (l.one | l.different | l.newgrid) {
    
    print("Interpolate 2nd field - please be patient :-)")
    for (it in 1:nt.2) {
      Z.in<-as.matrix(field.2$dat[it,,])
      Z.out<-interp(lat.x,lon.x,Z.in,lat,lon)
      dat.2[it,,]<-as.matrix(Z.out$z)
      if (plot.interp) {
        contour(field.2$lon,field.2$lat,t(round(Z.in,2)),col="blue",lwd=2,
                main=paste("Field 1: ",it,"/",nt.2),
                sub=paste(field.2$mm[it],"-",field.2$yy[it]))
        contour(lon,lat,t(round(Z.out$z,2)),add=TRUE,col="red")
        addland()
        grid()
      }
    }
  }

#  print(dim(dat.1));  print(dim(dat.2))

  dim(dat.1)<-c(nt.1,ny.1*nx.1)
  dim(dat.2)<-c(nt.2,ny.1*nx.1)
  if (!l.one) {
    dat<-rbind(dat.1,dat.2)
    dim(dat)<-c(nt.1+nt.2,ny.1,nx.1)
    tim <- c(field.1$tim,field.2$tim)
    yy <- c(field.1$yy,field.2$yy)
    mm <- c(field.1$mm,field.2$mm)
    dd <- c(field.1$dd,field.2$dd)
    id.t <- c(field.1$id.t,field.2$id.t)
  } else {
    dat<-dat.1
    dim(dat)<-c(nt.1,ny.1,nx.1)
    tim <- field.1$tim
    yy <- field.1$yy
    mm <- field.1$mm
    dd <- field.1$dd
    id.t <- field.1$id.t
  }
  id.x <- matrix(rep(field.1$id.x[1],ny.1*nx.1),ny.1,nx.1)
  attr(tim,"unit") <- tim.unit1
  attr(tim,"time_origin") <- tim.torg1
  if (field.1$v.name==field.2$v.name) var.name <- field.1$v.name else
  var.name <- paste(field.1$v.name,"&",field.2$v.name,sep="")
  result  <- list(dat=dat,lon=lon,lat=lat,tim=tim,v.name=var.name,
                  id.t=id.t,id.x=id.x,yy=yy,mm=mm,dd=dd,n.fld=field.1$n.fld,
                  id.lon=field.1$id.lon,id.lat=field.1$id.lat,attributes=field.1$attributes)
  class(result) <- c(class(field.1),"cat.fields")
  invisible(result)
}
# Obtains the name of variables in a netcdf File
#
# R.E. Benestad, 23.09.2003

cdfcont <- function(filename,path="") {

  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')

  system(paste(path,"ncdump -h  ",filename," > cdfcont.txt",sep=""),intern=T)
  cdfhead <- readLines("cdfcont.txt")
  cdfvars <- cdfhead[c(grep("float",lower.case(cdfhead)),
                       grep("short",lower.case(cdfhead)),
                       grep("double",lower.case(cdfhead)))]
  cdfdims <- cdfvars
  for (i in 1:length(cdfvars)) {
    i1 <- instring(" ",cdfvars[i])
    i2 <- instring("(",cdfvars[i])
    i3 <- instring(")",cdfvars[i])
    cdfdims[i] <- substr(cdfvars[i],i2[1]+1,i3[1]-1)
    cdfvars[i] <- substr(cdfvars[i],i1[1]+1,i2[1]-1)
  }
#  print(cdfvars)
#  print(cdfdims)
  torg <- cdfhead[grep("time_origin",lower.case(cdfhead))]
  if (length(torg)==0) {
    torg <- cdfhead[grep("since",lower.case(cdfhead))]
    t.org.pos <- regexpr("since",lower.case(torg))
    s<- instring('\"',torg)
    torg  <- substr(torg,t.org.pos+6,s[2]-1)
    dash <- instring("-",torg)
    spc <- instring(" ",torg)
    if (spc[1]==0) spc <- nchar(torg)+1
    yy0 <- as.numeric(substr(torg,1,dash[1]-1))
    while (nchar(yy0) < 4) yy0 <- paste("0",yy0,sep="")
    mm0 <- as.numeric(substr(torg,dash[1]+1,dash[2]-1))
    dd0 <- as.numeric(substr(torg,dash[2]+1,spc[1]-1))
    while (nchar(dd0) < 2) dd0 <- paste("0",dd0,sep="")
    torg <- paste(dd0,cmon[mm0],yy0)
 #   print(paste("time.origin=",torg))
    if (is.na(dd0[1])) dd0  <- 15
  } else {
    s<- instring('\"',torg)
    if (length(s)==2) torg<-substr(torg,s[1]+1,s[2]-1)
  }
  tunit<- cdfhead[grep("time:unit",lower.case(cdfhead))]
  if (length(tunit)>0) {
     s<- instring('\"',tunit)
     if (length(s)==2) tunit<- strip(substr(tunit,s[1]+1,s[2]-1))
  } else tunit<-NULL

  offs <- cdfhead[grep("add_offset",lower.case(cdfhead))]
  if (length(offs)>0) {
    e <- instring('=',offs);  f <- regexpr("f;",offs) # f <- instring('f',offs)
    if (f[1] <= 0) f <- regexpr("f ;",offs)
    if (f[1]>0) f<-f[length(f)] else f<-nchar(offs)
    yes <- (nchar(offs)>0) & (length(e)>0) & (length(f)>0)
    if (yes) offs<-substr(offs,e+1,f-1) else offs<-"0"
  } else offs <- "0"

  scal <- cdfhead[grep("scale_factor",lower.case(cdfhead))]
  if (length(offs)>0) {
    e <- instring('=',scal);  f <- regexpr("f;",scal) # f <- instring('f',scal)
    if (f[1] <= 0) f <- regexpr("f ;",scal)
    if (f[1]>0) f<-f[length(f)] else f<-nchar(scal)
    yes <- (nchar(scal)>0) & (length(e)>0) & (length(f)>0)
    if (yes) scal<-substr(scal,e+1,f-1) else scal<-1
  } else scal<-1

  miss <- cdfhead[grep("missing_value",lower.case(cdfhead))]
  if (length(miss)>0) {
    if (length(miss) > 1) miss <- miss[1]
    e <- instring('=',miss);   f <- regexpr("f;",miss) #f <- instring('f',miss)
    if (f[1] <= 0) f <- regexpr("f ;",miss)
    if (f[1]>0) f<-f[length(f)] else f<-nchar(miss)
    yes <- (nchar(miss)>0) & (length(e)>0) & (length(f)>0)
    if (yes) miss<-substr(miss,e+1,f-1) else miss<-NULL
  } else miss <- NA
  system("rm -f cdfcont.txt",intern=T)
  content <- list(vars=cdfvars,dims=cdfdims,time.origin=torg,time.unit=tunit,
                  add.offset=as.numeric(offs),scale.factor=as.numeric(scal),
       missing.value=as.numeric(miss))
  invisible(content)
}

# Extracts a variable and a subregion from a netcdf File
# Assumes a netCDF file with the structure:
# field(tim,lat,lon)
# R.E. Benestad, 23.09.2003
#
# Modified 27.04.2004 to base the IO on the ncdf package instead of
# netCDF (which is being phased out).
#

cdfextract <- function(filename,varname,x.rng=NULL,y.rng=NULL,t.rng=NULL,
                       greenwich=TRUE,x.nam="lon",y.nam="lat",t.nam="tim",
                       plot=TRUE,l.scale=TRUE) {

#  library(netCDF)
  library(ncdf)

  CDF <- cdfcont(filename)
  imatch <- (CDF$vars == varname)
  if (sum(imatch)>0) {
    lon <- NULL; lat <- NULL; tim <- NULL; lev <- NULL
    dat.att <- cdfcont(filename)
    ncid1 <- open.ncdf(filename)
    v1 <- ncid1$var[[1]]
    vars <- v1$name
    nvars <-  ncid1$nvars
    print(vars)
    dims <- names(ncid1$dim)
    vars <- names(ncid1$dim)
    n.dim <- ncid1$ndims
    d <- rep(0,nvars)
    dat <- NULL
    dat.att$unit <-v1$units

    eval(parse(text=paste("lon <- ncid1$dim$",names(ncid1$dim)[1],"$vals",sep="")))
    eval(parse(text=paste("lat <- ncid1$dim$",names(ncid1$dim)[2],"$vals",sep="")))
    eval(parse(text=paste("tim <- ncid1$dim$",names(ncid1$dim)[3],"$vals",sep="")))
    attr(lon,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[1],"$units",sep="")))
    attr(lat,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[2],"$units",sep="")))
    attr(tim,"time_origin") <- dat.att$torg
    if (!is.null(dat.att$time.unit)) attr(tim,"unit") <- dat.att$time.unit else 
     attr(tim,"unit") <-eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[itim],"$units",sep="")))     
    if (n.dim==4) {
      eval(parse(text=paste("lev <- ncid1$dim$",names(ncid1$dim)[3],"$vals",sep="")))
      eval(parse(text=paste("tim <- ncid1$dim$",names(ncid1$dim)[4],"$vals",sep="")))
      attr(lev,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[3],"$units",sep="")))
      attr(tim,"time_origin") <- dat.att$torg
      attr(tim,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[4],"$units",sep="")))
    }

    print("Time information:")
    if (!is.null(dat.att$time.origin)) {
      torg <-  dat.att$time.origin
    } else torg <- NULL
 
  t.unit <- attr(tim,"unit")
  if (!is.null(torg)) {
    yy0 <- as.numeric(substr(torg,8,11))
    dd0 <- as.numeric(substr(torg,1,2))
    mm0 <- switch(lower.case(substr(torg,4,6)),
                  "jan"=1,"feb"=2,"mar"=3,"apr"=4,"may"=5,"jun"=6,
                  "jul"=7,"aug"=8,"sep"=9,"oct"=10,"nov"=11,"dec"=12)
  } else if (grep("since",lower.case(t.unit))) {
    # Format: time:units = "hours since 1-1-1 00:00:0.0" (NCEP reanalysis)
    t.org.pos <- regexpr("since",lower.case(t.unit))
    torg  <- substr(t.unit,t.org.pos+6,nchar(t.unit))
    print(paste("torg=",torg))
    dash <- instring("-",torg)
    spc <- instring(" ",torg)
    yy0 <- as.numeric(substr(torg,1,dash[1]-1))
    mm0 <- as.numeric(substr(torg,dash[1]+1,dash[2]-1))
    dd0 <- as.numeric(substr(torg,dash[2]+1,spc[1]-1))
    if (is.na(dd0)) dd0  <- 15
  }
  print(paste("Time origin: (year-month-day)",yy0,"-",mm0,"-",dd0))
  if (yy0==0) {
    print('There is no year zero (Press et al., Numerical recipies)')
    print("'> print(julday(1,1,1)-julday(1,1,-1))' gives 365")
    print('julday wont work unless the time is fixed')
    print("year0 is set to 1, and 365 days is subtracted from tim")
    if (substr(lower.case(t.unit),1,4)=="hour") tim <- tim - 365*24
    if (substr(lower.case(t.unit),1,3)=="day") tim <- tim - 365
    if (substr(lower.case(t.unit),1,5)=="month") tim <- tim - 12
    if (substr(lower.case(t.unit),1,5)=="year") tim <- tim - 1
    yy0 <- 1
  }
  print(c(mm0,dd0,yy0))
    
  print(paste("Time unit:",lower.case(t.unit)))
  if (substr(lower.case(t.unit),1,5)=="month") {
    tim <- floor(tim)
    mm <- mod(mm0 + tim - 1,12)+1
    yy  <- yy0 + floor((tim+mm0-1)/12)
    dd <- rep(15,length(tim))
    obj.type <- "monthly.field.object"
  } else if (substr(lower.case(t.unit),1,3)=="day") {
    mmddyy <- caldat(tim + julday(mm0,dd0,yy0))
    mm <- mmddyy$month
    yy <- mmddyy$year
    dd <- mmddyy$day
    obj.type <- "daily.field.object"
  } else if (substr(lower.case(t.unit),1,4)=="hour") {
    mmddyy <- caldat(tim/24 + julday(mm0,dd0,yy0))
    mm <- mmddyy$month
    yy <- mmddyy$year
    dd <- mmddyy$day
    t.unit <- "day"
    obj.type <- "field.object"
  }
    
    if (greenwich) {
      lon[lon > 180] <- lon[lon > 180]-360
    }
    isrtx <- order(lon)
    print(range(lon)); print(range(lat)); print(range(tim)); 
    
    ix <- is.finite(lon); iy <- is.finite(lat); it <- is.finite(tim);
    if (!is.null(x.rng)) {ix <- ((lon>=x.rng[1]) & (lon<=x.rng[2])); print(range(lon[ix]))}
    if (!is.null(y.rng)) {iy <- ((lat>=y.rng[1]) & (lat<=y.rng[2])); print(range(lat[iy]))}
    if (!is.null(t.rng)) {it <- ((tim>=t.rng[1]) & (tim<=t.rng[2])); print(range(tim[it]))}
    iisrtx <- order(lon[ix])
    lonx <- lon[ix][iisrtx]
    nt <- length(tim); ny <- length(lat); nx <- length(lon); nz <- length(lev)
    
    x1 <- min((1:nx)[ix]); x2 <- max((1:nx)[ix])
    y1 <- min((1:ny)[iy]); y2 <- max((1:ny)[iy])
    t1 <- min((1:nt)[it]); t2 <- max((1:nt)[it])
    print(paste("netCDF dimensions:",nx,ny,nt))
    print(paste("Indeces start: ",x1,y1,t1,"& stop:",x2,y2,t2))
    print(paste("Coordinates start: ",lon[x1],lat[y1],tim[t1],
                "& stop:",lon[x2],lat[y2],tim[t2]))
    if (nz==0) {
      print(paste("Data size:",sum(it),sum(iy),sum(ix)))
      dat <- rep(0,(sum(it))*sum(iy)*sum(ix))
      dim(dat) <- c(sum(it),sum(iy),sum(ix))
      print(paste("Please be patient while reading data map by map..."))
      iit <- 0
      for (it in t1:t2) {
        iit <- iit+1
        datIN <- get.var.ncdf(ncid1,v1,,start=c(1,1,it),count=c(nx,ny,1))
        #print(c(dim(datIN),NA,nx,ny))
        dim(datIN) <- c(nx,ny)
        dat[iit,,] <- t(as.matrix(datIN[ix,iy]))
        if (plot) {
          image(lon[isrtx],lat,datIN[isrtx,],
                main=paste(yy[it],"-",mm[it],"-",dd[it]),sub=filename)
          contour(lonx,lat[iy],t(dat[iit,,iisrtx]),add=TRUE)
          lines(c(min(lonx),rep(max(lonx),2),rep(min(lonx),2)),
                c(rep(max(lat[iy]),2),rep(min(lat[iy]),2),max(lat[iy])),
                lwd=3,lty=2,col="grey95")
          addland()
          grid()
        }
       }
    } else {
       print("cdfextract does not yet know how to handle 4 dimensions ... Sorry!")
       return()
    }   
    close.ncdf(ncid1)

#print("<-------- Scale and determine time stamp...")
#print(dat.att)

    if ((l.scale) & !is.null(dat.att$scale.factor)) {
    dat <- dat * dat.att$scale.factor
  }
  # Have included a sanity test to detect an old 'bug': offset 273 and
  # units of deg C..
print("Old bug fix")

  if ( ((l.scale) & !is.null(dat.att$add.offset))) {
      if ( (dat.att$add.offset!=273) &
           (dat.att$unit=="deg C")) {
        a <- readline(prompt="Correct an old bug? (y/n)")
        if (lower.case(a)=="y") dat <- dat + dat.att$add.offset} else
        dat <- dat + dat.att$add.offset
  }

print("Set ID")

  eos <- nchar(varname)
  if (instring("-",varname)> 0) {
    eos <- instring("-",varname)-1
  } else if (instring("_",varname)> 0) {
    eos <- instring("_",varname)-1
  }
  varname <- substr(varname,1,eos)
  slash <- instring("/",filename)
  dot <- instring(".",filename)
  lon <- lon[ix]; lat <- lat[iy]; tim <- tim[t1:t2]
  isrtx <- order(lon)
  lon <- lon[isrtx]; dat <- dat[,,isrtx]
  nt <- length(tim); ny <- length(lat); nx <- length(lon); nz <- length(lev)
  id.x <- matrix(rep(varname,ny*nx),ny,nx)
  id.t <- rep(substr(filename,slash[length(slash)]+1,
                     dot[length(dot)]-1),nt)              
    
###    
print("Set coordinates")

    if (length(tim[t1:t2])>1) {
      results  <- list(dat=dat,lon=lon,lat=lat,tim=tim,lev=lev,
                       v.name=varname,id.x=id.x,id.t=id.t,
                       yy=yy[t1:t2],mm=mm[t1:t2],dd=dd[t1:t2],n.fld=1,
                       id.lon=rep(varname,nx),id.lat=rep(varname,ny),
                       attributes=dat.att)
      class(results) <- c("field",obj.type)
    } else if (length(tim[t1:t2])==1) {
      results  <- list(map=t(dat),lon=lon,lat=lat,tim=tim,
                       date=paste(yy[it],"-",mm[it],"-",dd[it]),
                       description=filename,v.name=varname)
      class(results) <- "map"
    }
#    print("Saving the extracted data in cdfextract.nc")
#    r2cdf("cdfextract.nc",results)
    invisible(results)
  } else {
    if (!file.exists(filename)) print("Cannot find filename") else {
      print(paste("Cannot find",varname,"in",filename))
      print("The netCDF file contains the following:")
      print(CDF$vars)
    }
  }
}
composite.field <- function(x,y,lsig.mask=TRUE,sig.lev=0.05,s=0.42,mon=NULL,
                            lty=1,col="black",lwd=1,main=NULL,sub=NULL) {
  library(ctest)

if ((class(x)[1]!="field") & (class(x)[1]!="monthly.field.object") &
    (class(x)[1]!="daily.field.object")){
  stop("x must be a 'field' object.")
}

 cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
         'Jul','Aug','Sep','Oct','Nov','Dec') 
  descr <- 'Composite:'
  date <- ""
  if (!is.null(mon)) {
    im <- x$mm== mon
    x$dat <- x$dat[im,,]
    x$yy <- x$yy[im]
    x$mm <- x$mm[im]
    x$dd <- x$dd[im]
    x$tim <- x$tim[im]
    x$id.t <- x$id.t[im]
    date <- cmon[mon]
  }

if (is.null(class(y))) class(y) <- 'vector'  
if (class(y)[1]=="station") {
  y.ts <- as.vector(t(y$val))
  yy <- sort(rep(y$yy,12))
  mm <- rep(1:12,length(y$yy))
  dd <- rep(15,length(yy))
  i1<-is.element(yy*10000+mm*100+dd,
                 x$yy*10000+x$mm*100+x$dd)
  i2<-is.element(x$yy*10000+x$mm*100+x$dd,
                 yy*10000+mm*100+dd)
  i.plus <- (y.ts[i1] >= mean(y.ts[i1],na.rm=TRUE)+s*sd(y.ts[i1],na.rm=TRUE))
  i.minus<- (y.ts[i1] <= mean(y.ts[i1],na.rm=TRUE)-s*sd(y.ts[i1],na.rm=TRUE))
} else if (length(y)==length(x$tim) & !is.logical(y)) {
    i1 <- seq(1,length(x$tim),by=1)
    i2 <- seq(1,length(x$tim),by=1)
    y.ts <- rep(0,length(i2))
    i.plus <- y > 0
    i.minus <- y < 0
} else if (length(y)==length(x$tim) & is.logical(y)) {
    i1 <- seq(1,length(x$tim),by=1)
    i2 <- seq(1,length(x$tim),by=1)
    y.ts <- rep(0,length(i2))
    i.plus <- y 
    i.minus <- !y    
} else if ((min(abs(y))>= min(x$yy)) & (max(abs(y)) <= max(x$yy))) {
    i1 <- seq(1,length(x$tim),by=1)    
    i2 <- seq(1,length(x$tim),by=1)
    i.plus <- is.element(x$yy,y[y > 0])
    i.minus <- is.element(x$yy,-y[y < 0])
} else {
  stop("Sorry - don't know how to interpret y; use station or vector of years")
}
  ni <- length(x$lon)
  nj <- length(x$lat)
  map <- matrix(rep(NA,ni*nj),nj,ni)
  p.val <- matrix(rep(NA,ni*nj),nj,ni)
#  print(range(yy))
#  print(range(x$yy))
#  print(range(mm))
#  print(range(x$mm))
#  print(range(dd))
#  print(range(x$dd))
#  print(c(sum(i1),sum(i2)))
#  print(c(sum(i.plus),sum(i.minus)))
  for (j in 1:nj) {
    for (i in 1:ni) {
      vec1 <- x$dat[i2,j,i]
      if (sum(is.finite(vec1)) > 10) {
        yy <- x$yy[i2]
        yy.plus <- yy[i.plus]
        yy.minus <- yy[i.minus]
        plus  <- vec1[i.plus]
        minus  <- vec1[i.minus]
        map[j,i] <- mean(plus,na.rm=TRUE) - mean(minus,na.rm=TRUE)
        if ((length(plus) > 0) & (length(minus) > 0)) p.val[j,i] <- t.test(plus,minus)$p.value
      } 
    }
  }

  z.levs <- seq(-max(abs(as.vector(map)),na.rm=TRUE),
                 max(abs(as.vector(map)),na.rm=TRUE),length=41)
  my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                c(abs(sin((0:40)*pi/40))),
                c(c(rep(1,21),seq(1,0,length=20))))
  if (lsig.mask) map[p.val > 0.05] <- NA
  if (sum(is.finite(map))==0) stop('No region with significance')
  if ( (is.null(main)) & (class(y)[1]=="station") ) {
    main <- paste(descr,attributes(x$dat)$"long_name",
                  "using",y$ele,"at",y$location)
  } else if (is.null(main)) main <- paste(descr,attributes(x$dat)$"long_name")
  if (is.null(sub)) sub <- date
  filled.contour(x$lon,x$lat,t(map),
                 col = my.col,levels=z.levs,
                 main=main,sub=sub,xlab="Longitude",ylab="Latitude")

# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = 1)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)
  contour(x$lon,x$lat,t(map),add=TRUE,col=col,lwd=lwd,lty=lty)
  addland()
  if (class(y)[1]=="station") {
    points(y$lon,y$lat,pch=20,col="white",cex=1.2)
    points(y$lon,y$lat,pch=20,col="black",cex=0.9)
  }
  results <- list(map=t(map),lon=x$lon,lat=x$lat,tim=NULL,
                  date=date,description=descr,v.name=,x$v.name,
                  yy.plus=yy.plus,yy.mins=yy.minus)
  class(results) <- "map"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- "Composite map"
  invisible(results)
}
compositeField <- function(x,y,lsig.mask=TRUE,sig.lev=0.05,s=0.42,mon=NULL,
                      lty=1,col="black",lwd=1,main=NULL,sub=NULL) {
  results <- composite.field(x,y,lsig.mask=lsig.mask,sig.lev=sig.lev,s=s,mon=mon,
                      lty=lty,col=col,lwd=lwd,main=main,sub=sub)
  invisible(results)
}
corEOF <- function(x,y,lsig.mask=TRUE,sig.lev=0.05,neofs=20,
                      lty=1,col="black",lwd=1) {
  library(ctest)

if (class(x)[1]!="eof") {
  stop("x must be an 'eof' object.")
}

if (class(y)[1]!="station") {
  stop(paste("y must be a 'monthly.station.record'",
             "object - Use  station.obj()"))
}

  descr <- 'Correlation:'
  date <- "(EOF)"
  y.ts <- as.vector(t(y$val))
  yy <- sort(rep(y$yy,12))
  mm <- rep(1:12,length(y$yy))
  dd <- rep(15,length(yy))
  i1<-is.element(yy*10000+mm*100+dd,
                 x$yy*10000+x$mm*100+x$dd)
  i2<-is.element(x$yy*10000+x$mm*100+x$dd,
                 yy*10000+mm*100+dd)
#  print(range(yy))
#  print(range(x$yy))
#  print(range(mm))
#  print(range(x$mm))
#  print(range(dd))
#  print(range(x$dd))
#  print(c(sum(i1),sum(i2)))
  ni <- length(x$lon)
  nj <- length(x$lat)
  U <- x$EOF
  dims <- dim(U)
  if (length(dims)==3) dim(U) <- c(dims[1],dims[2]*dims[3])
  neofs <- min(c(neofs,dims[1]))
#  print('UU <- t(U %*% U)')
#  print(dim(U))
#  UU <- U^2
#  WW <- x$W^2

  ya <- (y.ts[i1] - mean(y.ts[i1],na.rm=TRUE))/sd(y.ts[i1],na.rm=TRUE)

#  Va <- x$PC[i2,]
#  for (i in 1:neofs) Va[,i] <- (Va[,i] - mean(Va[,i],na.rm=TRUE))/
#                                 sd(Va[,i],na.rm=TRUE)
#  print('WWUU')
#   if (length(dims)==2) WWUU<-matrix(rep(0,neofs*dims[2]),dims[2],neofs) else
#             WWUU<-matrix(rep(0,dims[2]*dims[3]),dims[2]*dims[3],neofs)
#  WWUU.tsum <- rep(NA,neofs)
#  for (i in 1:neofs) {
#    print(c(length(WWUU[i,]),length(UU[,i]*WW[i])))
#    
#    WWUU[i,] <- UU[,i]*WW[i]
#    WWUU.tsum[i] <- sum(WWUU[i,],na.rm=TRUE)
#  }
#  denom <- sqrt(WWUU.tsum * sum(ya^2,na.rm=TRUE))
#  denom <- sqrt(sum(x$W^2) * sum(ya^2,na.rm=TRUE))
#  denom <- sqrt(x$tot.var * sum(ya^2,na.rm=TRUE))

 # Calculate the map:
  rmap <- matrix(rep(0,ni*nj),nj,ni)
  p.val <- matrix(rep(NA,ni*nj),nj,ni)
#  for (i in 1:neofs) {
#    map <- U[i,]*x$W[i]*sum(Va[,i]*ya,na.rm=TRUE)/denom[i]
#    dim(map) <- c(nj,ni)
#    rmap[is.finite(map)] <- rmap[is.finite(map)] + map[is.finite(map)]
#  }
  X.re <- t(t(U) %*% diag(x$W) %*% t(x$PC[i2,]))
  nt <- sum(i2)
#  print(sum(is.finite(ya)))
  dim(X.re) <- c(nt,nj,ni)
    for (j in 1:nj) {
      for (i in 1:ni) {
        if (sum(is.finite(X.re[,j,i]))>25) {
          a <- cor.test(X.re[,j,i],ya)
          rmap[j,i] <- as.numeric(a$estimate)
          p.val[j,i] <- as.numeric(a$p.value)
        } else {
          rmap[j,i] <- NA
          p.val[j,i] <-NA
        }
      }
    }

  # Plot
  z.levs <- seq(-max(abs(as.vector(rmap)),na.rm=TRUE),
                 max(abs(as.vector(rmap)),na.rm=TRUE),length=41)
#  print(range(z.levs))
#  print(range(x$lon))
#  print(range(x$lat))
#  print(denom)
  my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                c(abs(sin((0:40)*pi/40))),
                c(c(rep(1,21),seq(1,0,length=20))))
  if (lsig.mask) rmap[p.val > 0.05] <- NA


  filled.contour(x$lon,x$lat,t(rmap),
                 col = my.col,levels=z.levs,
                 main=paste(descr,attributes(x$dat)$"long_name","&",
                            y$ele,"at",y$location),
                 sub=date,xlab="Longitude",ylab="Latitude")

# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = 1)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)
  contour(x$lon,x$lat,t(rmap),add=TRUE,col=col,lwd=lwd,lty=lty)
  addland()
  points(y$lon,y$lat,pch=20,col="white",cex=1.2)
  points(y$lon,y$lat,pch=20,col="black",cex=0.9) 
  results <- list(map=t(rmap),lon=x$lon,lat=x$lat,tim=x$tim,
                  date=date,description=descr)
  class(results) <- "map"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- "Correlation map"
  invisible(results)
}
corField <- function(x,y,lsig.mask=TRUE,sig.lev=0.05,mon=NULL,
                     lty=1,col="black",lwd=1,main=NULL,z.levs=NULL,my.col=NULL) {
  library(ctest)
  library(akima)
  
if ((class(x)[1]!="field") & (class(x)[1]!="monthly.field.object") &
    (class(x)[1]!="daily.field.object")){
  stop("x must be a 'field' object.")
}

if ((class(y)[1]!="station") & (class(y)[1]!="field") &
    (class(y)[1]!="monthly.field.object")) {
  stop(paste("y must be a 'monthly.station.record' or a 'field'",
             "object - Use  station.obj()"))
}

 cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
         'Jul','Aug','Sep','Oct','Nov','Dec') 
  descr <- 'Correlation:'
  date <- ""
  if (!is.null(mon)) {
    im <- is.element(x$mm,mon)
    x$dat <- x$dat[im,,]
    x$yy <- x$yy[im]
    x$mm <- x$mm[im]
    x$dd <- x$dd[im]
    x$tim <- x$tim[im]
    x$id.t <- x$id.t[im]
    date <- cmon[mon]
  }

  if (class(y)[1]=="station") y.ts <- as.vector(t(y$val)) else
  {
    l.diffgrid <- TRUE
    if ( (length(x$lon)==length(y$lon)) & (length(x$lat)==length(y$lat)) ) {
      if ( (sum(x$lon==y$lon)==length(x$lon)) &
           (sum(x$lat==y$lat)==length(x$lat)) ) l.diffgrid <- FALSE
    }
    y.ts <- rep(NA,length(y$tim)*length(x$lat)*length(x$lon))
    dim(y.ts) <- c(length(y$tim),length(x$lat),length(x$lon))
    if (l.diffgrid) {
      print("Interpolate 2nd field - please be patient :-)")
      lat.x<-rep(x$lat,length(x$lon))
      lon.x<-sort(rep(x$lon,length(x$lat)))
      for (i in 1:length(y$tim)) {
         map <- as.matrix(y$dat[i,,])
         y.ts[i,,] <- interp(lat.x,lon.x,map,y$lat,y$lon)$z
      }
    } else {
      for (i in 1:length(y$tim)) {
         y.ts[i,,] <- as.matrix(y$dat[i,,])
      }
    }
  }
    
  if (class(y)[1]=="station") {
    yy <- sort(rep(y$yy,12))
    mm <- rep(1:12,length(y$yy))
    dd <- rep(15,length(yy))
  } else {
    yy <- y$yy; mm <- y$mm;  dd <- y$dd
  }
  
  i1<-is.element(yy*10000+mm*100+dd,
                 x$yy*10000+x$mm*100+x$dd)
  i2<-is.element(x$yy*10000+x$mm*100+x$dd,
                 yy*10000+mm*100+dd)
  ni <- length(x$lon)
  nj <- length(x$lat)
  map <- matrix(rep(NA,ni*nj),nj,ni)
  p.val <- matrix(rep(NA,ni*nj),nj,ni)
#  print(range(yy))
#  print(range(x$yy))
#  print(range(mm))
#  print(range(x$mm))
#  print(range(dd))
#  print(range(x$dd))
#  print(c(sum(i1),sum(i2)))
#  print(dim(x$dat[i2,,]))
#  print(dim(y.ts[i1,,]))
#  print(class(y)[1])
  for (j in 1:nj) {
    for (i in 1:ni) {
      if (class(y)[1]=="station") r.test <- cor.test(x$dat[i2,j,i],y.ts[i1]) else
                                  r.test <- cor.test(x$dat[i2,j,i],y.ts[i1,j,i])
      map[j,i] <- r.test$estimate
      p.val[j,i] <- r.test$p.value
    }
  }

  if (is.null(z.levs)) {
    z.levs <- seq(-max(abs(as.vector(map)),na.rm=TRUE),
                   max(abs(as.vector(map)),na.rm=TRUE),length=41)
  }
  if (is.null(my.col)) {
    my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                  c(abs(sin((0:40)*pi/40))),
                  c(c(rep(1,21),seq(1,0,length=20))))
  }
  if (lsig.mask) map[p.val > 0.05] <- NA
  
#  print(dim(map))
#  print(c(length(x$lon),length(x$lat)))

  if (is.null(attributes(x$dat)$"long_name")) attr(x$dat,"long_name") <- x$v.name
  if (is.null(attributes(y$dat)$"long_name")) attr(y$dat,"long_name") <- y$v.name
  
  if (is.null(main)) {
    if (class(y)[1]=="station") main <- paste(descr,attributes(x$dat)$"long_name","&",
                                              y$ele,"at",y$location) else
                                main <- paste(descr,attributes(x$dat)$"long_name","&",
                                              attributes(y$dat)$"long_name")
  }
  filled.contour(x$lon,x$lat,t(map),
                 col = my.col,levels=z.levs,
                 main=main,
                 sub=date,xlab="Longitude",ylab="Latitude")

# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = 1)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)
  contour(x$lon,x$lat,t(map),add=TRUE,col=col,lwd=lwd,lty=lty)
  addland()
  if (class(y)[1]=="station") {
    points(y$lon,y$lat,pch=20,col="white",cex=1.2)
    points(y$lon,y$lat,pch=20,col="black",cex=0.9)
  }
  results <- list(map=t(map),lon=x$lon,lat=x$lat,tim=x$tim,
                  date=date,description=descr)
  class(results) <- "map"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- "Correlation map"
  invisible(results)
}
# Transform to normal distribution 

dist2norm <- function(x,plot=FALSE,exclude=NULL,
                      sd=1,mean=0,force.zero=TRUE) {
  xT <- rep(NA,length(x))
  good <- is.finite(x)
  if (!is.null(exclude)) good <- good & (x!=exclude)
  breaks <- seq(min(x[good])-0.5*IQR(x[good]),
                max(x[good])+0.5*IQR(x[good]),length=15)
  h <- hist(x[good],breaks=breaks,plot=plot)
  q <- h$mids
  q.i <- seq(-3*sd+mean,3*sd+mean,length=100)
  edf <- cumsum(h$density)/sum(h$density)
  if (force.zero) {
    edf <- (edf-edf[1])
    edf <- edf/max(edf)
  }
  edf.i <- spline(q,edf,n=100)
  cdf.i <- pnorm(q.i,mean=mean,sd=sd)
  if (plot) {
    x11()
    plot(c(min(c(q.i,edf.i$x)),max(c(q.i,edf.i$x))),c(0,1),type="n",
         main="dist2normal",xlab="Value",ylab="Probability")
    grid()
    lines(edf.i$x,edf.i$y,type="l",lwd=2)
    lines(q.i,cdf.i,col="red")
  }

    for (i in 1:length(x)) {
    if (is.finite(x[i])) {
      y.i <- approx(edf.i$x,edf.i$y,x[i])$y
      xT[i] <- approx(cdf.i,q.i,y.i)$y
#      print(c(i,round(x[i],2),round(xT[i],2),round(y.i,2)))
      if ((mod(i,round(length(x)/10))==0) & (plot)){
        lines(rep(x[i],2),c(0,y.i),col="blue",lty=2)
        lines(c(x[i],xT[i]),rep(y.i,2),col="blue",lty=2)
        arrows(xT[i],y.i,xT[i],0,col="blue",lty=2,length=0.05)
      }
    }
  }

  dist2norm <- list(xT=xT,x=x,edf.i=edf.i,cdf.i=cdf.i,q.i=q.i)
  class(dist2norm) <- "dist2norm"
  invisible(dist2norm)
}


norm2dist <- function(x,plot=FALSE,exclude=NULL,
                      sd=1,mean=0,force.zero=TRUE) {
  if (class(x)!="dist2norm") {
    print("Needs a 'dist2norm'-type object")
    return()
  }
  if (plot) {
    x11()
    plot(c(min(c(x$q.i,x$edf.i$x)),max(c(x$q.i,x$edf.i$x))),c(0,1),type="n",
         main="dist2normal",xlab="Value",ylab="Probability")
    grid()
    lines(x$edf.i$x,x$edf.i$y,type="l",lwd=2)
    lines(x$q.i,x$cdf.i,col="red")
  }
  for (i in 1:length(x$xT)) {
    if (is.finite(x$xT[i]))
      y.i <- approx(x$q.i,x$cdf.i$y,x$xT[i])$y
      x$x[i] <- approx(x$edf.i,x$cdf.i$x,y.i)$y
      if ((mod(i,round(length(x$x)/10))==0) & (plot)){
        lines(rep(x$xT[i],2),c(0,y.i),col="blue",lty=2)
        lines(c(x$x[i],x$xT[i]),rep(y.i,2),col="blue",lty=2)
        arrows(x$x[i],y.i,x$x[i],0,col="blue",lty=2,length=0.05)
      }
  }
  norm2dist <- x$x
  invisible(norm2dist)
}
distAB <- function(lon,lat,lons,lats,a=6.378e06) {
  theta <- pi*lon/180
  phi <- pi*lat/180
  dist <- rep(NA,length(lons))
  r1 <- c(cos(phi)*cos(theta),
          sin(phi),
          cos(phi)*sin(theta))
  for (i in 1:length(lons)) {
    theta <- pi*lons[i]/180
    phi <- pi*lats[i]/180
    r2 <- c(cos(phi)*cos(theta),
            sin(phi),
            cos(phi)*sin(theta))
#    angle <- acos( sum(r1*r2)/(sqrt(sum(r1*r1)) * sqrt(sum(r2*r2))) )
    angle <- acos( sum(r1*r2) )
    dist[i] <- a* angle
  }
  dist
}
                   
# Empirical downscaling using EOFs of monthly values from eof.R
# Predictand is a time series of monthly values from NACD or climate station.
#
# Reference: R.E. Benestad et al. (2002),
#            Empirically downscaled temperature scenarios for Svalbard,
#            doi.10.1006/asle.2002.005, September 18.
#
#            R.E. Benestad (2001),
#            A comparison between two empirical downscaling strategies,
#            Int. J. Climatology, 1645-1668, vol. 21, DOI 10.1002/joc.703
#
# R.E. Benestad, met.no, Oslo, Norway 16.04.2002
# rasmus.benestad@met.no
#------------------------------------------------------------------------

DS <- function(dat,preds,mon=NULL,direc="output/",cal.id=NULL,
               ldetrnd=TRUE,i.eofs=seq(1,8,by=1),ex.tag="",
               method="lm",plot=TRUE,leps=FALSE,param="t2m",
               plot.res=FALSE,plot.rate=FALSE,xtr.args="",
               swsm="step",predm="predict",lsave=FALSE,rmac=TRUE,
               silent=FALSE) {
library(ts)
library(ctest)
#library(chron)
#library(date)
#library(xtable)

dir.0<-getwd()
if (!file.exists(direc)){
  if (!silent) print(paste("The directory",direc,"does not exists.. Creates it.."))
  dir.create(direc)
}
if (class(preds)[1]!="eof") {
  stop("The predictor must be an 'eof' object.")
}

if (class(dat)[1]!="station") {
  stop(paste("The predictand must be a 'monthly.station.record'",
             "object - Use  station.obj()"))
}

anm.weight <- FALSE
if (method=="anm.weight") {
  method <- "anm"
  anm.weight <- TRUE
}
if (method=="anm") {
  swsm <- "none"
#  predm <- "predictANM"
  ldetrnd <- FALSE
  rmac <- FALSE
}

if (class(preds)[2]=="daily.field.object") {
  good <- eval(parse(text=paste("is.finite(dat$",param,")",sep="")))
  eval(parse(text=paste("dat$",param," <- dat$",param,"[good]",sep="")))
  dat$mm <- dat$mm[good]; dat$yy <- dat$yy[good]
  dat$dd <- dat$dd[good]; dat$tim <- dat$tim[good]
} 

cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
        'Jul','Aug','Sep','Oct','Nov','Dec')
season.c<-c("","DJF","MAM","JJA","SON")
season<-cbind(c(12,1,2),c(3,4,5),c(6,7,8),c(9,10,11))
lon <- preds$lon
lat <- preds$lat
if (min(lon) < 0) deg.lon1.c<-"W" else deg.lon1.c<-"E"
if (max(lon) < 0) deg.lon2.c<-"W" else deg.lon2.c<-"E"
if (min(lat) < 0) deg.lat1.c<-"S" else deg.lat1.c<-"N"
if (max(lat) < 0) deg.lat2.c<-"S" else deg.lat2.c<-"N"
region<-paste(as.character(abs(round(min(lon)))),deg.lon1.c,
              as.character(abs(round(max(lon)))),deg.lon2.c,"-",
              as.character(abs(round(min(lat)))),deg.lat1.c,
              as.character(abs(round(max(lat)))),deg.lat2.c,sep="")

month <-cmon[mon]
if ((class(preds)[2]=="daily.field.object") & !is.null(mon)) {
  mon <- mod(mon-1,4)+1
  month <- season.c[mon+1]
  mon <- season[mon]
}

if (!is.null(mon) & !is.null(preds$mon)) {    
  if (is.null(mon)) mon <- preds$mon
  if (!silent) print(paste("Extract",cmon[mon],"-> # of data points=",
              sum(is.element(preds$mm,mon))))
  if (  sum(is.element(preds$mm,mon))==0  ) {
    if (!silent) print(paste(">>> ",cmon[mon],
                             " is not found in the PCA product! <<<"))
    months <- row.names(table(preds$mm))
    if (!silent) print(paste(" Available months are:",cmon[months]))
    mon<-months
  }
} else {
  if (max(preds$mm) > min(preds$mm)) {
    month <- paste(cmon[min(preds$mm)],"-",cmon[max(preds$mm)],sep="")
  } else month  <- cmon[mean(preds$mm)]
  months <- row.names(table(preds$mm))
  mon<-months
}

preds.names <- row.names(table(preds$id.t))
preds.id <- ""
for (i.pred in 1:length(preds.names)) {
  eos <- nchar(preds.names[i.pred])
  if (instring("-",preds.names[i.pred])> 0) {
    eos <- instring("-",preds.names[i.pred])-1
  } else if (instring("_",preds.names[i.pred])> 0) {
    eos <- instring("_",preds.names[i.pred])-1
  }
  preds.id  <- paste(preds.id,substr(preds.names[i.pred],1,eos),
                     "+",sep="")
}
if (is.null(attr(preds$tim,"unit"))) {
  if (preds$dd[2]-preds$dd[1]==0) attr(preds$tim,"unit")<-"mon" else
                                 attr(preds$tim,"unit")<-"day"
#  if (!silent) print(attr(preds$tim,"unit"))
}

eos <- instring(" ",dat$location)[1]-1
if ((is.null(eos)) | (eos <= 0)) eos <- nchar(dat$location)
preds.id <- substr(preds.id,1,nchar(preds.id)-1)
fname<-paste(direc,"ds_",preds.id,"_",region,"_",
             substr(dat$location,1,eos),"_",dat$ele,"_",preds$c.mon,'_',
             substr(attr(preds$tim,"unit"),1,3),"_",method,
             ex.tag,".Rdata",sep="")

# Get the predictand

loc <- dat$location
if (class(dat)[2]=="monthly.station.record"){
  v.name  <- abbreviate(dat$obs.name)
} else if (class(dat)[2]=="daily.station.record") {
  v.name  <- param
}

if (v.name=="mT(2") v.name <- "T"
ny<-length(dat$yy)

if ( (ny < 20) | (sum(is.na(dat$yy))>0) ) {
  if (!silent) print("ds: WARNING: ... SENSING POSSIBLE PROBLEMS!...")
  if (!silent) print(paste("For predictor, you selected",preds$f.name))
  if (!silent) print(paste("for predictand, you selected",loc))
  if (!silent) print(paste("Number of valid data points from station",
              sum(!is.na(dat$val)),"- Length of data record=",ny))
  if (!silent) print("Years of station observations:")
  if (!silent) print(range(dat$yy))
  if (!silent) print("Years of predictor (PCs):")
  if (!silent) print(range(preds$yy))
  if (!silent) print("You may want to try another location or different dataset")
}

if (class(dat)[2]=="monthly.station.record"){
  yy.o<-sort(rep(dat$yy,12))
  mm.o<-rep(seq(1,12,by=1),ny)
  dd.o <- rep(15,length(yy.o))
  y.o<-t(dat$val)
  dim(y.o)<-c(12*ny,1)
  ds.unit <- dat$unit
} else if (class(dat)[2]=="daily.station.record") {
  yy.o <- dat$yy
  mm.o <- dat$mm
  dd.o <- dat$dd
  if (eval(parse(text=paste("is.null(dat$",param,")",sep="")))) {
    if (!silent) print(summary(dat))
    param <- readline("Select object field:")
    ds.unit <- readline("unit:")
  } else {
    if (!silent) print(paste("y.o<-dat$",param,sep=""))
    eval(parse(text=paste("y.o<-dat$",param,sep="")))
    ds.unit <- dat$unit[1]
  }
#  tim.o <- julian(mm.o,dd.o,yy.o,origin.=c(1,1,1970))
#  tim.o <- mdy.date(mm.o, dd.o, yy.o)
  tim.o <- julday(mm.o, dd.o, yy.o) - julday(1,1,1970)
  nt <- length(tim.o)
  ac.mod<-matrix(rep(NA,nt*6),nt,6)
  ac.mod[,1]<-cos(2*pi*tim.o/365.25); ac.mod[,2]<-sin(2*pi*tim.o/365.25)
  ac.mod[,3]<-cos(4*pi*tim.o/365.25); ac.mod[,4]<-sin(4*pi*tim.o/365.25)
  ac.mod[,5]<-cos(6*pi*tim.o/365.25); ac.mod[,6]<-sin(6*pi*tim.o/365.25)
  ac.obs <- data.frame(y=y.o, X=ac.mod)
  ac.fit<-lm(y ~ X.1 + X.2 + X.3 + X.4 + X.5 + X.6,data=ac.obs)
  if (rmac) y.o <- ac.fit$residual
}

y.o[y.o < -99] <-NA
mm.o <- mm.o[!is.na(y.o)]
yy.o <- yy.o[!is.na(y.o)]
dd.o <- dd.o[!is.na(y.o)]
y.o <- y.o[!is.na(y.o)]

# Give the correct file names.
n.fld <- preds$n.fld
if (n.fld>1) {
  reg <- paste(preds$v.name[1],preds$v.name[2],"_",
               as.character(-1*min(preds$lon)),"W-",
               as.character(max(preds$lon)),"E_",
               as.character(min(preds$lat)) ,"N-",
               as.character(max(preds$lat)),"N",sep="")
} else {
  reg <- paste(preds$v.name[1],"_",
               as.character(-1*min(preds$lon)),"W-",
               as.character(max(preds$lon)),"E_",
               as.character(min(preds$lat)) ,"N-",
               as.character(max(preds$lat)),"N",sep="")
}

y.o<-  y.o[is.element(mm.o,mon)]
yy.o<- yy.o[is.element(mm.o,mon)]
dd.o<- dd.o[is.element(mm.o,mon)]
mm.o<- mm.o[is.element(mm.o,mon)]

if (is.null(cal.id)) cal.id<- preds$id.t[1]
#print(paste("Calibration predictors:",cal.id))
#print(sum(preds$id.t==cal.id & !is.na(preds$PC[,1])))
#print(range(preds$yy[preds$id.t==cal.id & !is.na(preds$PC[,1])]))
X.cal<-  preds$PC[preds$id.t==cal.id & !is.na(preds$PC[,1]),]
yy.cal<- preds$yy[preds$id.t==cal.id & !is.na(preds$PC[,1])]
mm.cal<- preds$mm[preds$id.t==cal.id & !is.na(preds$PC[,1])]
dd.cal<- preds$dd[preds$id.t==cal.id & !is.na(preds$PC[,1])]

if (!silent) print("------------Match times---------------- ")
#print(range(yy.cal))
X.cal<-  X.cal[is.element(mm.cal,mon),]
yy.cal<- yy.cal[is.element(mm.cal,mon)]
dd.cal<- dd.cal[is.element(mm.cal,mon)]
mm.cal<- mm.cal[is.element(mm.cal,mon)]
#print(c(length(y.o),length(yy.o),length(X.cal[,1]),length(yy.cal)))

if (sum((preds$id.t!=cal.id) & !is.na(preds$PC[,1]))>0) {
  X.gcm<-preds$PC[preds$id.t!=cal.id  & !is.na(preds$PC[,1]),]
  yy.gcm<-as.vector(preds$yy[preds$id.t!=cal.id & !is.na(preds$PC[,1])])
  mm.gcm<-as.vector(preds$mm[preds$id.t!=cal.id & !is.na(preds$PC[,1])])
  dd.gcm<-as.vector(preds$dd[preds$id.t!=cal.id & !is.na(preds$PC[,1])])
  
#print("The scenarios:")
  X.gcm<-X.gcm[is.element(mm.gcm,mon),]
  yy.gcm<-yy.gcm[is.element(mm.gcm,mon)]
  dd.gcm<-dd.gcm[is.element(mm.gcm,mon)]
  mm.gcm<-mm.gcm[is.element(mm.gcm,mon)]
} else {
  X.gcm<-X.cal
  yy.gcm <- yy.cal
  mm.gcm <- mm.cal
  dd.gcm <- dd.cal
}

# Find the common period:

if ((class(preds)[2]=="monthly.field.object") |
    (class(preds)[3]=="monthly.field.object")) {
  i1<-is.element(yy.o,yy.cal)
  i2<-is.element(yy.cal,yy.o)
} else {
  i1<-is.element(yy.o*10000+mm.o*100+dd.o,
                 yy.cal*10000+mm.cal*100+dd.cal)
  i2<-is.element(yy.cal*10000+mm.cal*100+dd.cal,
                 yy.o*10000+mm.o*100+dd.o)
}

# Extract the predictand & predictors:

if (!silent) print(paste("Number of coinciding obs:",sum(i1),",",sum(i2)))
if (!silent) print(summary(yy.o))
if (!silent) print(summary(yy.cal))

y.o<-y.o[i1] ; mm.o<-mm.o[i1]; yy.o<-yy.o[i1]; dd.o<-dd.o[i1]
X.cal<-X.cal[i2,]; mm.cal<-mm.cal[i2]; yy.cal<-yy.cal[i2]; dd.cal<-dd.cal[i2]

# Remove missing values:
i3 <- is.finite(y.o)
#print(c(length(y.o),length(yy.o),length(X.cal[,1]),length(yy.cal)))
y.o<-y.o[i3]; mm.o<-mm.o[i3]; yy.o<-yy.o[i3]; dd.o<-dd.o[i3]
X.cal<-X.cal[i3,]; mm.cal<-mm.cal[i3]; yy.cal<-yy.cal[i3]; dd.cal<-dd.cal[i3]

if (!silent) print("Common times:")
if (!silent) print(range(yy.o))
if (!silent) print(range(y.o))

#--------------------------------------------------------
# De-trend the data used for model calibration:
print("de-trend:")

if (ldetrnd) {
  for (i in 1:length(preds$var.eof)) {
    trnd<-seq(-1,1,length=length(X.cal[,i]))
    dtrnd<-lm(X.cal[,i] ~trnd)
    X.cal[,i]<-dtrnd$residual   
  }
}
if (method=="anm") y <- y.o   # Analog model
              else y <- y.o - mean(y.o,na.rm=TRUE)
trnd<-seq(-1,1,length=length(y))
dtrnd<-lm(y ~ trnd)
if (ldetrnd) {
  y<-dtrnd$residual
}


# Stepwise regression
#scen.gcm.str <- "data.frame("
#calibrate.str <- "data.frame(y=y,"
#for (ipre in 1:length(preds$var.eof)) {
# if (weight) scen.gcm.str <-
#paste(scen.gcm.str,"X",ipre,"=X.gcm[,",ipre,
#                                "]* preds$W[",ipre,"],",sep="")
#else scen.gcm.str <-
#paste(scen.gcm.str,"X",ipre,"=X.gcm[,",ipre,"],",sep="")
 
# Stepwise regression
print("stepwise regression:")
n.eofs<- min(c(length(preds$var.eof),length(i.eofs)))    
scen.gcm.str <- "data.frame("
calibrate.str <- "data.frame(y=y,"
for (ipre in 1:n.eofs) {
  scen.gcm.str <- paste(scen.gcm.str,"X",ipre,"=X.gcm[,",ipre,
                        "]* preds$W[",ipre,"],",sep="")
  if (method=="anm") {   # Analog model
    if (anm.weight) calibrate.str <- paste(calibrate.str,"X",ipre,"=X.cal[,",
                           ipre,"]* preds$W[",ipre,"],",sep="") else
                    calibrate.str <- paste(calibrate.str,"X",ipre,"=X.cal[,",
                           ipre,"],",sep="") 
  } else calibrate.str <- paste(calibrate.str,"X",ipre,"=X.cal[,",ipre,
                           "]* preds$W[",ipre,"],",sep="")
}
scen.gcm.str <- paste(scen.gcm.str,"yy=as.vector(yy.gcm),mm=as.vector(mm.gcm),dd=as.vector(dd.gcm))",sep="")
#print("GCM:")
#print(scen.gcm.str)
scen.gcm <- eval(parse(text=scen.gcm.str))

calibrate.str <- paste(calibrate.str,"yy=as.vector(yy.cal),mm=as.vector(mm.cal),dd=as.vector(dd.cal))",sep="")
#print("Calibration:")
#print(calibrate.str)
calibrate <- eval(parse(text=calibrate.str))

print(summary(calibrate))
# Due to a bug in step, 'attatch' cannot be used, so it's done
# in a more complicated way.
attach(calibrate)
exprn <- paste(method,"(y ~ 1",sep="")
for (i.eof in 1:n.eofs) {  
  eval(parse(text=
             paste("X",i.eofs[i.eof]," <- calibrate$X",i.eofs[i.eof],sep="")))
}
for (i.eof in 1:n.eofs) {  
  exprn <- paste(exprn," + X",i.eofs[i.eof],sep="")
}
#if (method!="anm") exprn <- paste(exprn,xtr.args,")",sep="") else 
#                   exprn <- paste(exprn,",","data=calibrate",xtr.args,")",sep="") 
if (method!="anm") exprn <- paste(exprn,xtr.args,",data=calibrate)",sep="") else 
                   exprn <- paste(exprn,",","data=calibrate",xtr.args,")",sep="") 

if (!silent) print(paste("Model: ",exprn))
#print(c(length(y),length(X1),length(X2),length(X3)))
lm.mod <- eval(parse(text=exprn))
#print(summary(lm.mod))
lm.mod$coefficients[!is.finite(lm.mod$coefficients)] <- 0
meths <- methods(class(lm.mod))
if (!silent) print(paste("Stepwise:   ",swsm,"(lm.mod,trace=0)",sep=""))
if ((swsm!="none") & !is.null(swsm)) {
  step.wise <- eval(parse(text=paste(swsm,"(lm.mod,trace=0)",sep="")))
}  else step.wise<-lm.mod
step.wise$coefficients[!is.finite(step.wise$coefficients)] <- 0

print("ANOVA from step-wise regression:")

stat <- summary(step.wise)
if (length(step.wise$coefficients)>1) {
  if (!is.null(stat$r.squared)) {
    r2 <- round(stat$r.squared*100)
    p.val <- round(100*(1-pf(stat$fstatistic[1],
                           stat$fstatistic[2],
                           stat$fstatistic[3])))
  } else if (method=="anm") {
    cor.test(y,eval(parse(text=paste(predm,"(lm.mod)",sep=""))))
    r2.stat <- cor.test(y,predict.anm(lm.mod))
    r2 <- round(100*r2.stat$estimate^2,2)
    p.val <- round(100*r2.stat$p.value,2)
  } else {
    r2.stat <- eval(parse(text=paste("r2.stat <- cor.test(y,",
                            predm,"(lm.mod))",sep="")))
    r2 <- round(100*r2.stat$estimate^2,2)
    p.val <- round(100*r2.stat$p.value,2)
  }
  fit.p<-as.character(p.val)
} else {
  if (!silent) print("-----------Step failed:----------")
  if (!silent) print(paste("---------",method,":"))
  if (!silent) print(summary(lm.mod))
  if (!silent) print("-----------Step:")
  if (!silent) print(stat)
  r2 <- 0
  p.val <- 100
  fit.p <- "100"
}

# Downscale predictions

#pre.y  <-predict(step.wise)
pre.y  <- eval(parse(text=paste(predm,"(step.wise)",sep="")))
for (i.eof in 1:20) {
  eval(parse(text=paste("rm (X",i.eofs[i.eof],")",sep="")))
}
detach(calibrate)
attach(scen.gcm)
#pre.gcm<-predict(step.wise,newdata=scen.gcm)
pre.gcm <- eval(parse(text=paste(predm,"(step.wise,newdata=scen.gcm)",sep="")))

if (!silent) print("Downscaled anomalies:")
if (!silent) print(summary(pre.gcm))

detach(scen.gcm)
if (!silent) print(summary(step.wise))

# A "fudge" to avoid problems when stepwise rejects all the predictors
# (i.e. only returns an intercept)
if (length(pre.gcm)==1) {
  if (!silent) print(c(length(pre.gcm),c(length(yy.gcm))))
  pre.gcm <- rep(pre.gcm,length(yy.gcm))
}

#print(summary(pre.y))
#print(summary(pre.gcm))
#print(c(mean(pre.gcm[yy.gcm<2010],na.rm=TRUE),
#        mean(y.o[yy.o>1980],na.rm=TRUE)))

if ((class(dat)[2]=="daily.station.record") & (rmac)) {
#  tim.cal <- julian(mm.cal,dd.cal,yy.cal,origin.=c(1,1,1970))
#  tim.cal <- mdy.date(mm.cal, dd.cal, yy.cal)
  tim.cal <- julday(mm.cal, dd.cal, yy.cal) - julday(1,1,1970)
  rm(ac.mod)
  nt.cal <- length(tim.cal)
  ac.mod<-matrix(rep(NA,nt.cal*6),nt.cal,6)
  ac.mod[,1]<-cos(2*pi*tim.cal/365.25); ac.mod[,2]<-sin(2*pi*tim.cal/365.25)
  ac.mod[,3]<-cos(4*pi*tim.cal/365.25); ac.mod[,4]<-sin(4*pi*tim.cal/365.25)
  ac.mod[,5]<-cos(6*pi*tim.cal/365.25); ac.mod[,6]<-sin(6*pi*tim.cal/365.25)
  ac.cal <- data.frame(X=ac.mod)
  rm(ac.mod)
#  tim.gcm <- julian(mm.gcm,dd.gcm,yy.gcm,origin.=c(1,1,1970))
#  tim.gcm <- mdy.date(mm.gcm, dd.gcm, yy.gcm)
  tim.gcm <- julday(mm.gcm, dd.gcm, yy.gcm) - julday(1,1,1970)
  nt.gcm <- length(tim.gcm)
  ac.mod<-matrix(rep(NA,nt.gcm*6),nt.gcm,6)
  ac.mod[,1]<-cos(2*pi*tim.gcm/365.25); ac.mod[,2]<-sin(2*pi*tim.gcm/365.25)
  ac.mod[,3]<-cos(4*pi*tim.gcm/365.25); ac.mod[,4]<-sin(4*pi*tim.gcm/365.25)
  ac.mod[,5]<-cos(6*pi*tim.gcm/365.25); ac.mod[,6]<-sin(6*pi*tim.gcm/365.25)
  ac.gcm <- data.frame(X=ac.mod)
  rm(ac.mod)
#  tim.o <- julian(mm.o,dd.o,yy.o,origin.=c(1,1,1970))
#  tim.o <- mdy.date(mm.o, dd.o, yy.o)
  tim.o <- julday(mm.o, dd.o, yy.o) - julday(1,1,1970)
  nt <- length(tim.o)
  ac.mod<-matrix(rep(NA,nt*6),nt,6)
  ac.mod[,1]<-cos(2*pi*tim.o/365.25); ac.mod[,2]<-sin(2*pi*tim.o/365.25)
  ac.mod[,3]<-cos(4*pi*tim.o/365.25); ac.mod[,4]<-sin(4*pi*tim.o/365.25)
  ac.mod[,5]<-cos(6*pi*tim.o/365.25); ac.mod[,6]<-sin(6*pi*tim.o/365.25)
  ac.obs <- data.frame(X=ac.mod)
  y.o <- y.o + predict(ac.fit,newdata=ac.obs)
  pre.y <- pre.y + predict(ac.fit,newdata=ac.cal)
  pre.gcm <- pre.gcm + predict(ac.fit,newdata=ac.gcm)  
}

if (method!="anm") {
  ii1 <- is.element(yy.gcm,yy.o)
  ii2 <- is.element(yy.o,yy.gcm)
  cal.mean <- mean(pre.y,na.rm=TRUE)
  gcm.mean <- mean(pre.gcm[ii1],na.rm=TRUE)
  obs.mean <- mean(y.o[ii2],na.rm=TRUE)
  obs.mean2 <- mean(y.o,na.rm=TRUE)
  if (!is.finite(gcm.mean)) gcm.mean  <- 0
  if (!is.finite(obs.mean)) obs.mean  <- 0 
  if (!is.finite(cal.mean)) cal.mean  <- 0
  if (!is.finite(obs.mean2)) obs.mean2  <- 0 
  pre.y   <- pre.y   - cal.mean + obs.mean2
  pre.gcm <- pre.gcm - gcm.mean + obs.mean
                       
}
#print("Check#3:")
#print(summary(pre.gcm))  # TEST

if ( (regexpr("precip",lower.case(dat$obs.name)) > 0) |
     (regexpr("rain",lower.case(dat$obs.name)) > 0) ) {
  pre.y[pre.y < 0] <-  0
  pre.gcm[pre.gcm < 0] <-  0
}

# Predictions: GCM
# Determine which PCs were selected in the step procedure
# Note, some of the higher modes are truncated

c<-as.character(step.wise$call[2])
c<-unlist(strsplit(c," \\~ "))
c<-c[2]
c<-paste(unlist(strsplit(c," \\+ ")),' ')
#print(c)
incl<- rep(FALSE,length(preds$var.eof))
for (i in 1:length(i.eofs)) {
  if (!is.na( charmatch(paste('X',as.character(i),' ',sep=""),c ) ))  {
    incl[i]<-TRUE
  }
}
#print(incl)

# Note that the intercept is included in lm.coe, but not in the
# coefficients held by c.

lm.coe <- coef(step.wise)

# Find the predictor patterns

preds2D<-preds$EOF
dims <- dim(preds2D)
if (length(dims) > 2) dim(preds2D)<-c(dims[1],dims[2]*dims[3])

if (!silent) print("Reconstruct the spatial patterns")
if (!silent) print(lm.coe)
i.last <- 0
list.expr <- "list("
id <- row.names(table(preds$id.x))
#print(id)
#print(table(preds$id.lon))
#print(table(preds$id.lat))
for (i in 1:n.fld) {
#  print(id[i])
  i.lon <- preds$id.lon == id[i]
  i.lat <- preds$id.lat == id[i]
  ny<-preds$size[2,i]
  nx<-preds$size[3,i]
  i.fld <- seq(i.last+1,i.last+ny*nx,by=1)
  i.last <- max(i.fld)
#  print(paste("Dimension of field ",i))
#  print(dim(preds2D))
#  print(c(sum(incl),sum(i.fld)))
  EOF.1 <- t(preds2D[,i.fld])
  EOF.1 <-  EOF.1[,incl]
#  print(dim(EOF.1))
  expr <- paste("X.",i," <- cbind(0,EOF.1) %*% lm.coe[1:(sum(incl)+1)]",sep="")
  eval(parse(text=expr))
#  print(paste("2D -> 3D: nx=",nx," ny=",ny))
#  print(eval(parse(text=paste("dim(X.",i,")",sep=""))))
  expr <- paste("dim(X.",i,") <- c(ny,nx)",sep="")
  eval(parse(text=expr))
  eval(parse(text=paste("lon.",i," <- preds$lon[i.lon]",sep="")))
  eval(parse(text=paste("lat.",i," <- preds$lat[i.lat]",sep="")))
  list.expr <- paste(list.expr,"X.",i,"=X.",i,
                     ", lon.",i,"=lon.",i,
                     ", lat.",i,"=lat.",i,", ",sep="")
#  print(length(lon.1))
#  print(length(lat.1))
}

# Linear trend:

if (!silent) print("Linear trend for GCM (deg C/decade)")
x.ind <- seq(0,1,length=length(yy.gcm))
tr.dat<-data.frame(y=pre.gcm, x=x.ind)
nt <- length(yy.gcm)
lm.tr <- lm(y ~ x, data=tr.dat)
stat.tr.fit <- summary(lm.tr)
coef.fit<-stat.tr.fit$coefficients
rate.ds <- round(as.real(round(coef.fit[2]*10,2))*(x.ind[2]-x.ind[1]),2)
rate.err  <- round(as.real(round(coef.fit[4]*10,2))*(x.ind[2]-x.ind[1]),2)

#print(coef.fit)
if (!silent) print("Slope and its uncertainty")
if (!silent) print(c(rate.ds,rate.err))
pre.fit<-predict(lm.tr,data= yy)

# Polinomial trend

#print("Polinomial trend")
lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3) + I(x^4) + I(x^5), data=tr.dat)
pre.p.fit<-predict(lm.tr.p,data=tr.dat)
coef.p.fit<-lm.tr.p$coefficients
coef.p.fit[is.na(coef.p.fit)] <- 0
der.p.fit<-c(coef.p.fit[2],2*coef.p.fit[3],3*coef.p.fit[4],
             4*coef.p.fit[5],5*coef.p.fit[6])*(x.ind[2]-x.ind[1])
tr.est.p.fit<-(der.p.fit[1] + der.p.fit[2]*x.ind + der.p.fit[3]*x.ind^2 +
               der.p.fit[4]*x.ind^3 + der.p.fit[5]*x.ind^4)*10
gcm.stat <- summary(lm.tr)

# Estimate the P-values associated with the trends

if (!is.null(gcm.stat$fstatistic)) {
  gcm.trnd.p<-as.character(round(100*(1-pf(gcm.stat$fstatistic[1],
                                           gcm.stat$fstatistic[2],
                                           gcm.stat$fstatistic[3]))))
} else {gcm.trnd.p<-"100"}
gcm.trnd.r2 <- gcm.stat$r.squared

if (!silent) print(paste("P-value of fit=",fit.p))
if (!silent) print(paste("P-value of trend-fit for downscaled scenario",gcm.trnd.p))

#---------------------------------------------------

print("Dignosis:")
print(direc)
print(preds.id)
print(region)
print(dat$location)
print(dat$ele)
print(preds$c.mon)
print(attr(preds$tim,"unit"))
print(method)

#if ((method!="nnet") & (method!= "anm") & lsave) {
#
#print("Make LaTeX & HTML tables of model")  
#mod.name<-paste(direc,"ds.mod_",preds.id,"_",region,"_",
#             substr(dat$location,1,eos),"_",dat$ele,"_",preds$c.mon,'_',
#             substr(attr(preds$tim,"unit"),1,3),"_",method,
#             ex.tag,sep="")
#mod.tab <- xtable(step.wise,
#                  caption=paste("Calibration period: ",month,
#                  " ",range(yy.cal)[1],"-",range(yy.cal)[2],
#                    " using ",preds$id.t[cal.id],sep=""))
#print("Save LaTeX & HTML tables of model")  
#print.xtable(mod.tab,type="latex",
#             file=paste(mod.name,".tex",sep=""))
#print.xtable(mod.tab,type="html",
#             file=paste(mod.name,".html",sep=""))
#}
#
#print("Make LaTeX & HTML tables of downscaled results")
sce.name<-paste(direc,"ds.res_",preds.id,"_",region,"_",
             substr(dat$location,1,eos),"_",dat$ele,"_",preds$c.mon,'_',
             substr(attr(preds$tim,"unit"),1,3),"_",method,
             ex.tag,sep="")
#
#scen.table<-xtable(data.frame(year=as.vector(yy.gcm),
#                              downscaled=round(pre.gcm,2)),
#                   caption=paste("Linear trend=",rate.ds,
#                     ds.unit,"/decade over ",month," ",
#                     range(yy.gcm)[1],"-",range(yy.gcm)[2],
#                     " using",preds$id.t[preds$id.t!=cal.id][1],
#                     "; p-value for linear trend-fit=",
#                     gcm.trnd.p,"%.",sep=""))

#if (lsave) {
#  print("Save LaTeX & HTML tables of downscaled results")
#  print.xtable(scen.table,type="html",
#           file=paste(sce.name,".html",sep=""))
#}

pred.name <- row.names(table(preds$id.x))
list.expr <- paste(list.expr,
         "lon.loc=dat$lon,lat.loc=dat$lat,alt.loc=dat$alt,",
         "step.wise=step.wise,location=loc,",
         "yy.gcm=yy.gcm, mm.gcm=mm.gcm, dd.gcm=dd.gcm, ",
         "yy.cal=yy.cal, dd.cal=dd.cal,","mm.cal=mm.cal,",
         "n.fld=n.fld,unit=ds.unit,",
         "rate.ds=rate.ds,rate.err=rate.err,gcm.trnd.p=gcm.trnd.p,",
         "y.o=y.o,mm.o=mm.o,yy.o=yy.o,dd.o=dd.o,",
         "fit.p=fit.p,fit.r2=r2,pre.p.fit=pre.p.fit,",
         "pre.gcm=pre.gcm,pre.y=pre.y,gcm.stat=gcm.stat,",
         "month=month,v.name=v.name, region=preds$region,",
         "id.1=cal.id,id.2=preds$id.t[preds$id.t!=cal.id][1],",
         "pre.fit=pre.fit,tr.est.p.fit=tr.est.p.fit,ex.tag=ex.tag,",
         "pred.name=pred.name,sce.name=sce.name,preds.name=preds$f.name)",sep="")       
#print(list.expr)
ds<-eval(parse(text=list.expr))
if (!silent) print(paste("File name:",fname))
class(ds) <- "ds"
if (lsave) save(file=fname,ds,ascii=FALSE) 
#print("Plotting...")
#print(preds$region)
if (plot) plotDS(ds,leps)
invisible(ds)
}
# Computes Empirical Orthogonal Functions (EOFs)
#
# R.E. Benestad, met.no, Oslo, Norway 7.10.2002
# rasmus.benestad@met.no
#
#------------------------------------------------------------------------

EOF<-function(fields,l.wght=TRUE,lc180e=FALSE,direc="data/",
              lon=NULL,lat=NULL,l.stndrd=TRUE,las=1,
              mon=NULL,plot=TRUE,neofs=20,l.rm.ac=TRUE,lsave=TRUE,
              LINPACK=TRUE) {

#=========================================================================
library(ts)

if ((class(fields)[2]!="monthly.field.object") &
    (class(fields)[2]!="daily.field.object") &
    (class(fields)[1]!="field")) {
      print("class(fields) gives:")
      print(class(fields))
      stop("Need a 'field.object'")
    }

dir.0<-getwd()
if (!file.exists(direc)){
  print(paste("The directory",direc,"does not exists.. Creates it.."))
  dir.create(direc)
}

if (is.null(attr(fields$tim,"units"))) attr(fields$tim,"units") <- fields$attributes$time.unit 
tunit <- attr(fields$tim,"unit")
if (is.null(attr(fields$tim,"time_origin"))) attr(fields$tim,"time_origin") <- fields$attributes$time.origin
tim.torg <- attr(fields$tim,"time_origin")
if (!is.null(attr(fields$tim,"unit"))) {
  tunit <- lower.case(substr(attr(fields$tim,"unit"),1,3))
} else tunit <- "mon"

dims <- dim(fields$dat) 
if (length(dims)==3) dim(fields$dat) <- c(dims[1],dims[2]*dims[3])
clim <- rep(NA,dims[2]*dims[3])

# For naming the files containing the results

if (is.null(lon)) lon <- fields$lon
if (is.null(lat)) lat <- fields$lat
if (min(lon) < 0) deg.lon1.c<-"W" else deg.lon1.c<-"E"
if (max(lon) < 0) deg.lon2.c<-"W" else deg.lon2.c<-"E"
if (min(lat) < 0) deg.lat1.c<-"S" else deg.lat1.c<-"N"
if (max(lat) < 0) deg.lat2.c<-"S" else deg.lat2.c<-"N"
region<-paste(as.character(abs(round(min(lon)))),deg.lon1.c,
              as.character(abs(round(max(lon)))),deg.lon2.c,"-",
              as.character(abs(round(min(lat)))),deg.lat1.c,
              as.character(abs(round(max(lat)))),deg.lat2.c,sep="")
months<-c("Jan","Feb","Mar","Apr","May","Jun",
          "Jul","Aug","Sep","Oct","Nov","Dec")
season<-cbind(c(12,1,2),c(3,4,5),c(6,7,8),c(9,10,11))
season.c<-c("","DJF","MAM","JJA","SON")

id <- row.names(table(fields$id.x))
mm <- fields$mm
yy <- fields$yy
dd <- fields$dd
dat <- fields$dat
id.t <- fields$id.t
tim <- fields$tim
dims <- dim(dat)
nt <- dims[1]
np <- dims[2]
c.mon <- ""
if (class(fields)[2]=="monthly.field.object") {
  print("monthly.field.object")
  if (is.null(mon))  {
      if (min(mm)==max(mm)) c.mon <- months[mm[1]]  else
               c.mon<-paste(months[min(mm)],"-",months[max(mm)],sep="")
      i.mm <- is.finite(mm)      
  } else {
      c.mon<-months[mon]
      i.mm <- is.element(mm,mon)
      dat <- dat[i.mm,]
      yy <- yy[i.mm]
      mm <- mm[i.mm]
      dd <- dd[i.mm]
      id.t <- id.t[i.mm]
      tim <- tim[i.mm]
    }
} else if (class(fields)[2]=="daily.field.object") {
  print("daily.field.object")
  ac.mod<-matrix(rep(NA,nt*6),nt,6)
  ac.mod[,1]<-cos(2*pi*fields$tim/365.25)
  ac.mod[,2]<-sin(2*pi*fields$tim/365.25)
  ac.mod[,3]<-cos(4*pi*fields$tim/365.25)
  ac.mod[,4]<-sin(4*pi*fields$tim/365.25)
  ac.mod[,5]<-cos(6*pi*fields$tim/365.25)
  ac.mod[,6]<-sin(6*pi*fields$tim/365.25)
  if (l.rm.ac) {
    for (ip in seq(1,np,by=1)) {
      ac.fit<-lm(dat[,ip] ~ ac.mod)
      dat[!is.na(dat[,ip]),ip]<-ac.fit$residual
    }
  }
  if (is.null(mon))  {
    if (min(mm)==max(mm)) {
      c.mon <- months[mm[1]]
    } else { 
      c.mon<-paste(months[min(mm)],"-",months[max(mm)],sep="")
    }
    i.mm <- is.finite(mm) 
  } else {
    mon <- mod(mon-1,4)+1
    c.mon<-season.c[mon+1]
    mon <- season[,mon]
    i.mm <- is.element(mm,mon)
    dat <- dat[i.mm,]
    yy <- yy[i.mm]
    mm <- mm[i.mm]
    dd <- dd[i.mm]
    id.t <- id.t[i.mm]
    tim <- fields$tim[i.mm]
#    print(season[,mon])
  }
#  print("Months:")
#  print(table(mm))
#  print(paste("Season:",mon))
#  print(season)
} else if (tunit=="mon") {
     print("Field with unspecified time unit - set to month")
     c.mon<-months[as.numeric(row.names(table(mm)))]
     i.mm <- is.finite(mm) 
} else {
  print(class(fields))
  print(tunit)
  stop('Error, did not know what to do with the class')
}

# Decide the name of the file containting EOF
#print("file name...")

preds.names <- row.names(table(lower.case(fields$id.t)))

preds.id <- ""; scen <- ""
if (sum(grep("gsdio",preds.names))>0) scen <- "-gsdio"
if (sum(grep("is92a",preds.names))>0) scen <- "-is92a"
if (sum(grep("b1",preds.names))>0) scen <- "-b1"
if (sum(grep("a1",preds.names))>0) scen <- "-a1"
if (sum(grep("b2",preds.names))>0) scen <- "-b2"
if (sum(grep("a2",preds.names))>0) scen <- "-a2"

for (i.pred in 1:length(preds.names)) {
  eos <- nchar(preds.names[i.pred])

  if (instring("_",preds.names[i.pred])[1]> 0) {
    eos <- instring("_",preds.names[i.pred])-1
    if (length(eos) > 1) eos <- eos[1]
  } else if (instring("-",preds.names[i.pred])[1]> 0) {
    eos <- instring("-",preds.names[i.pred])-1
    if (length(eos) > 1) eos <- eos[1]
  } else if (instring(".",preds.names[i.pred])[1]> 0) {
    eos <- instring(".",preds.names[i.pred])-1
    if (length(eos) > 1) eos <- eos[1]
  } else eos <- nchar(preds.names[i.pred])
  preds.id  <- paste(preds.id,substr(preds.names[i.pred],1,eos),
                     "+",sep="")
}

vnames <- substr(fields$v.name[1],1,min(nchar(fields$v.name[1]),4))
#if (length(fields$v.name)>1) {
#  for (i in 2:length(fields$v.name)) vnames <- paste(vnames,"+",strip(fields$v.name[i]),sep="")
#}
print(c("pred.names=",preds.names,"scen=",scen,"preds.id=",preds.id,"vnames=",vnames))
fname<-paste(direc,"eof_", preds.id,scen,"_",vnames,"_",region,"_",
       c.mon,'_',tunit,".Rdata",sep="")
print(paste("File name:",fname,"sum(i.mm)=",sum(i.mm)))
#print(dim(fields$dat))

#-------------------------------------------------------------------------

id <- row.names(table(fields$id.x))
size <- matrix(rep(0,3*fields$n.fld),3,fields$n.fld)
stdv <- rep(0,fields$n.fld)

#print(table(fields$id.lon))
#print(table(fields$id.lat))
#print(table(fields$id.x))
#print(id)

ixy <- 0
for (i in 1:fields$n.fld) {
  ii <- fields$id.x == id[i]
#  print(paste("id.x: ",sum(ii)))
  id.lon <- fields$id.lon
  id.lat <- fields$id.lat
#  print("i.lon/i.lat:")
  i.lon <- fields$id.lon == id[i]
  i.lat <- fields$id.lat == id[i]
#  print("lonx/latx:")
  lon.x <- fields$lon[i.lon]
  lat.x <- fields$lat[i.lat]
#  print("id.lon/id.lat:")
  id.lon <- fields$id.lon[i.lon]
  id.lat <- fields$id.lat[i.lat]
#  print("nx/ny:")
  nx <- length(lon.x)
  ny <- length(lat.x)
#  print("dat:")
#  print(dim(fields$dat))
  dat.x <- fields$dat[,ii]
#  print("ix/iy:")
  ix <- ((lon.x >= min(lon)) & (lon.x <= max(lon)))
  iy <- ((lat.x >= min(lat)) & (lat.x <= max(lat)))
#  print("new lonx/latx:")
  lon.x <- lon.x[ix]
  lat.x <- lat.x[iy]
#  print("new id.lon/id.lat:")
  id.lon  <- id.lon[ix]
  id.lat  <- id.lat[iy]
#  print(dim(dat.x))
#  print(c(length(yy),ny,nx,sum(ix),sum(iy)))

  dim(dat.x) <- c(length(yy),ny,nx)
  dat.x <- dat.x[,iy,ix]

#  print("Stdv[i]")  
  ny <- length(lat.x)
  nx <- length(lon.x)
  nt <- length(yy)
  stdv[i] <- sd(dat.x,na.rm=TRUE)

#  print("Remove mean values at each grid point")
  for (j.y in 1:ny) {
    for (i.x in 1:nx) {
      ixy <- ixy + 1
      clim[ixy] <- mean(dat.x[,j.y,i.x],na.rm=TRUE)
      dat.x[,j.y,i.x] <- dat.x[,j.y,i.x] -  clim[ixy]
    }
  }

#  print("Add geographical weighting")
  if (l.wght) {
    print(paste("Weighting according to area. Field",i))
    Wght <-matrix(nrow=ny,ncol=nx)
    for (j in 1:nx)  Wght[,j]<-sqrt(abs(cos(pi*lat.x/180)))
    Wght[Wght < 0.01]<-NA     
    for (it in 1:nt) dat.x[it,,] <- dat.x[it,,]*Wght/stdv[i]
#    print(paste("Wght.",i,"<-Wght",sep=""))
    eval(parse(text=paste("Wght.",i,"<-Wght",sep="")))
  }
  
  # reshape 3-D matrices to 2-D matrices

#print("Reshape 3-D matrices to 2-D matrices")
#print(dim(dat.x))
#print(c(nt,ny,nx,ny*nx))
  if (i == 1) {
    dat.d2 <- dat.x
    dim(dat.d2) <- c(nt,ny*nx)
  } else {
    dim(dat.x) <- c(nt,ny*nx)
    dat.d2 <- cbind(dat.d2,dat.x)
  }
  size[,i] <- c(nt,ny,nx)
  if (i==1) {
    id.x <- id[i]
    lons <- lon.x
    lats <- lat.x
    id.lons <- id.lon
    id.lats <- id.lat
#    print(paste("id.lons: ",length(id.lons),length(id.lon)))
#    print(paste("id.lats: ",length(id.lats),length(id.lat)))
  } else {
#    print(paste("Appending lons & lans, i=",i))
    id.x <- c(id.x,id[i])
    lons <- c(lons,lon.x)
    lats <- c(lats,lat.x)
    id.lons <- c(id.lons,id.lon)
    id.lats <- c(id.lats,id.lat)
#    print(paste("id.lons: ",length(id.lons),length(id.lon)))
#    print(paste("id.lats: ",length(id.lats),length(id.lat)))
  }
}

if (sum(is.na(dat.d2))>0) print(paste(sum(is.na(dat.d2)),
                            ' missing values of ',nx*ny*nt))
aver <- 0
# print(paste("Find max autocorr in",ny*nx,"grid boxes."))
for (i in 1:(ny*nx)) {
  vec <- as.vector(dat.d2[,i])
  i.bad <- is.na(vec)
  if (sum(i.bad) == 0) {
    ar1 <- acf(vec[],plot=FALSE)
    aver <- max(c(aver,ar1$acf[2,1,1]),na.rm=TRUE)
  }
}

n.eff <- round(nt * (1.0-aver)/(1.0+aver))  
print(paste("mean AR(1) =",aver, "n.eff=",n.eff))

# Apply the PCA:       
print(paste("Singular Value Decomposition: ",sum(is.na(dat.d2)),
            ' NA-values -> set to zero of ',length(dat.d2)))
dat.d2[!is.finite(dat.d2)]<-0
print(paste("Data range:",min(dat.d2),"-",max(dat.d2)," dimensions=",
            dim(dat.d2)[1],"x",dim(dat.d2)[2],"  std=",stdv))

if (LINPACK) pca<-svd(t(dat.d2)) else 
             pca<-La.svd(t(dat.d2))
PC<-pca$v[,1:neofs]
EOF<-t(pca$u[,1:neofs])
W<-pca$d[1:neofs]
tot.var <- sum(pca$d^2)
Var.eof<-100*pca$d[1:neofs]^2/tot.var

dW <- W*sqrt(2.0/n.eff)

# 2D->3D transform, invert weighting

#print("2D->3D transform")
i.last <- 0
x.last <- 0
y.last <- 0
for (i in 1:fields$n.fld) {
  i.fld <- seq(i.last+1,i.last+size[2,i]*size[3,i],by=1)
  i.last <- max(i.fld)
#  print(i.last)
  EOF.1 <- EOF[,i.fld]
  dim(EOF.1)<-c(neofs,size[2,i],size[3,i])
#  print(l.wght)
  if (l.wght) for (ieof in 1:neofs) EOF.1[ieof,,]<-
             EOF.1[ieof,,]*stdv[i]/eval(parse(text=paste("Wght.",i,sep="")))
#  print('eof.patt<-t(EOF.1[1,,])')
  eof.patt<-t(EOF.1[1,,])
  EOF[,i.fld] <- EOF.1
#  print('lonx,latx,...')
  lon.x <- lons[id.lons==id[i]]
  lat.x <- lats[id.lats==id[i]]
#  print('plot settings..')
  my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                c(abs(sin((0:40)*pi/40))),
                c(c(rep(1,21),seq(1,0,length=20))))
  z.levs <- seq(-max(abs(as.vector(eof.patt)),na.rm=TRUE),
                max(abs(as.vector(eof.patt)),na.rm=TRUE),length=41)
#----------------------------------------------------------------
       
  if (plot) {
    filled.contour(lon.x,lat.x,eof.patt,levels=z.levs,
       main=paste("1st EOF for",id[i]),col=my.col,
       sub=paste(fields$f.name," (",c.mon,")"),
       xlab="Longitude",ylab="Latitude")
# From filled.contour in base
    mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
    on.exit(par(par.orig))

    w <- (3 + mar.orig[2]) * par('csi') * 2.54
    layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
    par(las = las)
    mar <- mar.orig
    mar[4] <- 1
    par(mar=mar)
    contour(lon.x,lat.x,eof.patt,nlevels=7,add=TRUE,lwd=1,col="black")
    addland()
    grid()
    dev.copy2eps(file=paste("eof_",i,".eps",sep=""))
  }
}

attr(tim,"unit") <- fields$attributes$time.unit
attr(tim,"time_origin") <- fields$attributes$time.origin

#print("Construct list object")
eof<-list(EOF=EOF,W=W,PC=PC,id=preds.id,n.fld=fields$n.fld,tot.var=tot.var,
          id.t=id.t,id.x=fields$id.x,size=size,dW=dW,mon=mon,l.wght=l.wght,
          id.lon=id.lons,id.lat=id.lats,region=region,tim=tim,
          lon=lons,lat=lats,var.eof=Var.eof,yy=yy,mm=mm,dd=dd,
          v.name=fields$v.name,c.mon=c.mon,f.name=fname,clim=clim,
          attributes=fields$attributes)
class(eof) <- c("eof",class(fields))
save(file='data/ceof.Rdata',eof,ascii=FALSE)
if (lsave) save(file=fname,eof,ascii=FALSE) 

invisible(eof)
}
lagStation <- function(x,lag=0) {
  nt <- length(x$val)
  dims <- dim(x$val)
  y <- as.vector(t(x$val))
  yshift <- rep(NA,nt)
  if (lag >= 0) yshift[1:(nt-lag)] <- y[(1+lag):nt] else
                yshift[(1-lag):nt] <- y[1:(nt+lag)]
  x$val <- t(matrix(yshift,dims[2],dims[1]))
  if (!is.null(attr(x,'lagStation'))) attr(x,'lagStation')<- paste(attr(x,'lagStation'),lag) else
                                      attr(x,'lagStation') <- lag
  invisible(x)
}

ds2station <- function(x,what="scenario") {
   ele <- switch(lower.case(x$v.name),
                 "t" = 101, "temp" = 101, "t2m" = 101,
                 "p" = 601, "precip" = 601, "rain" = 601, "precipitation" = 601)
   if (what=="scenario") {
     y <- station.obj(x=x$pre.gcm,yy=x$yy.gcm,mm=x$mm.gcm,obs.name=x$v.name,
                      unit=x$unit,ele=ele,lat=x$lat.loc,lon=x$lon.loc,alt=x$alt.loc,
                      location=x$location,ref="ds2station")
   } else {
     y <- station.obj(x=x$pre.cal,yy=x$yy.cal,mm=x$mm.cal,obs.name=x$v.name,
                      unit=x$unit,ele=ele,lat=x$lat.loc,lon=x$lon.loc,alt=x$alt.loc,
                      location=x$location,ref="ds2station")
 } 
}

mergeEOF <- function(eof1,eof2,plot=TRUE,silent=FALSE,method="lm",
                     match="time",cut.off=8,adjust=TRUE) {
  
  if ((class(eof1)[1]!="eof") | (class(eof2)[1]!="eof")) {
    print(class(eof1)); print(class(eof2))
    stop('Need two eof objects')
  }
   
  if (match=="space") method <- "project"
  i1 <- is.element(eof1$yy+(eof1$mm-0.5)/12+(eof1$dd-0.5)/365, 
                   eof2$yy+(eof2$mm-0.5)/12+(eof2$dd-0.5)/365)
  i2 <- is.element(eof2$yy+(eof2$mm-0.5)/12+(eof2$dd-0.5)/365, 
                   eof1$yy+(eof1$mm-0.5)/12+(eof1$dd-0.5)/365)
  if (((sum(i1)==0) | (sum(i2)==0)) & method=="lm") {
    print(c(range(eof1$yy),NA,range(eof1$mm),NA,range(eof1$dd)))
    print(c(range(eof2$yy),NA,range(eof2$mm),NA,range(eof2$dd)))
    print("Use the projection method and set argument match='space'.")
    stop('No overlapping times')
  } else {
    if (!silent) print(paste("Common points: N=",sum(i1)))
    if (!silent) print(c(range(eof2$yy[i2]),NA,range(eof2$mm[i2]),NA,range(eof2$dd[i2])))
  }
  eof.match <- eof1
  eof.match$PC <- eof1$PC[!i1,]
  X <- as.matrix(eof1$PC[i1,] %*% diag(eof1$W))

  if (method=="lm") {                                       # Method section: "lm"
    lm.str <- "lm(y ~ X.1"
    for (i in 2:length(eof1$W)) {
      lm.str <- paste(lm.str,' + X.',i,sep="")
    }
    lm.str <- paste(lm.str,',data=cal)',sep="")
    #print(lm.str)
    r.squared <- eof2$W + NA
    cut.off <- min(c(cut.off,length(eof2$W)))
    #print(paste("cut-off=",cut.off))

    for (i in 1:cut.off) {
      #print(paste("<<< i=",i,">>>"))
      y <- as.vector(eof2$PC[i2,i] * eof2$W[i])
      cal <- data.frame(y = y, X = X)
      idp <- data.frame(X = as.matrix(eof1$PC[!i1,] %*% diag(eof1$W)))
      #print(summary(cal))

      match <- eval(parse(text=lm.str))
      step.wise <- step(match,trace=0)
      stats <- summary(step.wise)

# Determine which PCs were selected in the step procedure
# Note, some of the higher modes are truncated

      c<-as.character(step.wise$call[2])
      c<-unlist(strsplit(c," \\~ "))
      c<-c[2]
      c<-paste(unlist(strsplit(c," \\+ ")),' ')
      #print(c)
      incl<- rep(FALSE,length(eof1$var.eof))
      for (iv in 1:length(eof1$var.eof)) {
        if ( (!is.na( charmatch(paste('X',as.character(iv),' ',sep=""),c) )) |
             (!is.na( charmatch(paste('X.',as.character(iv),' ',sep=""),c) )) )
        {
          incl[iv]<-TRUE
        }
      }
 }
  # print(incl)

# Note that the intercept is included in lm.coe, but not in the
# coefficients held by c.

      lm.coe <- coef(step.wise)

# Find the predictor patterns

      preds2D<-eof1$EOF
      dims <- dim(preds2D)
      if (length(dims) > 2) dim(preds2D)<-c(dims[1],dims[2]*dims[3])

      if (!silent) print(paste("Reconstruct the spatial patterns",i))
      if (!silent) print(lm.coe)
      if (length(lm.coe)<2) {
        stop('Poor match!')
      }
     if (length(step.wise$coefficients) > 1) { 
      i.last <- 0
      id <- row.names(table(eof1$id.x))
      ordr <- rep(NA,length(id))
      for (iii in 1:length(id)) {
        ordr[iii] <- min((1:length(eof1$id.x))[is.element(eof1$id.x,id[iii])])
      }
      id<-id[order(ordr)]
 
      y.hat <- predict(step.wise, newdata=idp)
      y.fit <- predict(step.wise)
      #W <- as.numeric(step.wise$coefficients[-1])*eof1$W[incl[1:length(eof1$W)]]
      W <- as.numeric(step.wise$coefficients[-1])
       for (ii in 1:eof1$n.fld) {
        print(paste("field",ii))
        i.lon <- eof1$id.lon == id[ii]
        i.lat <- eof1$id.lat == id[ii]
        ny<-eof1$size[2,ii]
        nx<-eof1$size[3,ii]
        i.fld <- seq(i.last+1,i.last+ny*nx,by=1)
        i.last <- max(i.fld)
 
        EOF.1 <- t(preds2D[,i.fld])
        EOF.1 <-  EOF.1[,incl[1:length(eof1$W)]]
 
        expr <- paste("X.",ii," <- EOF.1 %*% W + step.wise$coefficients[1]",sep="")
        eval(parse(text=expr))
 
        expr <- paste("dim(X.",ii,") <- c(ny,nx)",sep="")
        eval(parse(text=expr))
        eval(parse(text=paste("lon.",ii," <- eof1$lon[i.lon]",sep="")))
        eval(parse(text=paste("lat.",ii," <- eof1$lat[i.lat]",sep="")))
 
        r.squared[i] <- stats$r.squared
        if ((stats$r.squared < 0.45) & (cut.off > i)) {
          cut.off <- i
          print(paste(">>> i=",i,"cut.off=",cut.off," R^2=",round(r.squared[i],2)))
        }
         if (!silent) print(summary(match))
  
        eof.match$EOF[i,i.fld] <- eval(parse(text=paste("X.",ii,sep="")))
      }                                      # end ii-loop
      #print("Store predicted PC")
      eof.match$PC[,i] <- y.hat/eof2$W[i]

    } else print(paste("Bad match for field",i))
                                                                        # end method section: "lm"
  } else {

# Use projection from G. Strang (1988), "Linear algrebra and its
#     applications", Hartcourt Brace & Company, 3rd ed. (p.147):
# psi <- t(X) %/% ( t(X)%*%(X) )
# y.hat <- t(psi %*% t(X)) %*% t(Y)
    print("Use projection from G. Strang (1988):")
  
    if (match=="time") {
      X <- eof2$PC[i2,1:cut.off]
    } else {
      X <- eof2$EOF[1:cut.off,]
    }
    psi <- t(X) %/% ( t(X)%*%(X) )
    eof.match$PC <- t(psi %*% t(X)) %*% t(eof1$PC[!i1,1:cut.off]) 
    eof.match$EOF <-  t(psi %*% t(X)) %*% eof1$EOF[1:cut.off,] 
  }

# Arrange the data
print("Arrange the data")
  eof <- eof2
  print(paste("cut-off=",cut.off))
  #print(dim(eof1$EOF))
  #print(dim(eof.match$EOF))

  nt1 <- sum(!i1); nt2 <- length(eof2$id.t)
  np1<- length(eof$EOF[1,]); np2 <- length(eof.match$EOF[1,])
  eof$EOF <- cbind(matrix(eof$EOF[1:cut.off,],cut.off,np1),
                   matrix(eof.match$EOF[1:cut.off,],cut.off,np2)) 
  eof$PC <- rbind(matrix(eof.match$PC[,1:cut.off],nt1,cut.off),
                  matrix(eof2$PC[,1:cut.off],nt2,cut.off))
  eof$r.squared <- cbind(matrix(rep(r.squared[1:cut.off],nt1),cut.off,nt1),
                         matrix(rep(1,cut.off*nt2),cut.off,nt2))

  id1 <- paste(eof2$id.x[1],".1",sep="")
  id.match <- paste(eof1$id.x[1],".merge",sep="")
  eof$id.x <- c(eof2$id.x,
                paste(eof1$id.x,".merge",sep=""))
  eof$id.lon <- c(eof2$id.lon,
                  paste(eof1$id.lon,".merge",sep=""))
  eof$id.lat <- c(eof2$id.lat,
                  paste(eof1$id.lat,".merge",sep=""))
  eof$var <- eof$var[1:cut.off]
  eof$W <- eof$W[1:cut.off]
  eof$dW <- eof$dW[1:cut.off]
  eof$n.fld <- eof$n.fld + 1
  eof$lon <- c(eof2$lon,eof.match$lon)
  eof$lat <- c(eof2$lat,eof.match$lat)
  eof$size <- cbind(c(eof2$size[1:3]),c(sum(!i1),eof1$size[2:3]))
  eof$yy <- c(eof1$yy[!i1],eof2$yy)
  eof$mm <- c(eof1$mm[!i1],eof2$mm)
  eof$dd <- c(eof1$mm[!i1],eof2$dd)
  eof$tim <- c(eof1$tim[!i1],eof2$tim) 
  eof$id.t <- c(eof1$id.t[!i1],eof2$id.t)
  class(eof) <- class(eof2)
  if (adjust) eof <- adjust.eof(eof)
  
  #print(table(eof$id.x))
  #print(table(eof$id.lon))
  #print(table(eof$id.lat))
  #print(dim(eof$EOF))
  #print(eof$size)

  if (plot) {
     plotEOF(eof)
 #   newFig()
 #   plot(eof$yy + (eof$mm-0.5)/12,eof$PC[,1],lwd=2,type="l")
 #   grid()
 #   lines(eof1$yy + (eof1$mm-0.5)/12,eof1$PC[,1],lwd=2,lty=2,col="blue")
 #   lines(eof2$yy + (eof2$mm-0.5)/12,eof2$PC[,1],lwd=2,lty=2,col="red")
  }

#  print("mergeEOF: setting class of eof explicitly")
#  print(class(eof))
  invisible(eof)
}

  mapEOF <- function(x,i.eof=1,nlevs=5,add=FALSE,
                   col=c("red","blue","darkgreen","steelblue"),lwd=2,lty=1) {
  map <- map.eof(x,i.eof=i.eof,nlevs=nlevs,add=add,col=col,lwd=lwd,lty=lty)
  invisible(map)
} 


adjust.eof <- function(x) {
  rn <- row.names(table(x$id.t))
  ordr <- rep(NA,length(rn))
  for (i in 1:length(rn)) {
    ordr[i] <- min((1:length(x$id.t))[is.element(x$id.t,rn[i])])
  }
  #print(ordr)
  rn<-rn[order(ordr)]
  #print(rn)

  dims <- dim(x$PC)
  for (i in 1:dims[2]) {
    mu <- mean(x$PC[x$id.t==x$id.t[1],i],na.rm=TRUE)
    si <- sd(x$PC[x$id.t==x$id.t[1],i],na.rm=TRUE)
     for (ii in 2:length(rn)) {
      i.cal <- (x$id.t==rn[ii])
      mu.gcm <- mean(x$PC[i.cal,i],na.rm=TRUE)
      si.gcm <- sd(x$PC[i.cal,i],na.rm=TRUE)
       x$PC[x$id.t==rn[ii],i] <- (x$PC[x$id.t==rn[ii],i] -
             mu.gcm)/si.gcm * si + mu
    }
  }
  invisible(x)
}

getgiss <- function(stnr=NULL,location=NULL,lon=NULL,lat=NULL,stations=NULL,silent=FALSE) {
  if (!silent) print("Retrieving the data from URL http://www.giss.nasa.gov/")
  if (!silent) print("Please be patient")
  if (is.null(stations)) {
    if (!silent) print("Looking up station meta-data on the station on URL")
    stations<- read.fwf("http://www.giss.nasa.gov/data/update/gistemp/station_list.txt",
               width=c(9,20,18,17,5,4,2,2,4,3,5),skip=1,as.is=TRUE,comment.char = "%",header=FALSE,
               col.names=c("number","location","Country","fill1","lat","lon",
                           "type","brighness","fill2","Contry code","Brightness index"))
    stations$lon <- as.numeric(stations$lon)/10
    stations$lat <- as.numeric(stations$lat)/10
    stations$type[stations$type=="R"] <- "Rural"
    stations$type[stations$type=="S"] <- "Surburbian"
    stations$type[stations$type=="U"] <- "Urban"
    stations$type <- as.factor(stations$type)
    ivalcont <- !is.element(stations$Country,"                  ")
    valcont <- seq(1,length(stations$Country),by=1)[ivalcont]
    for (i in 1:sum(ivalcont)) {
      imatchcont <- is.element(stations$Contry.code,stations$Contry.code[valcont[i]])
      stations$Country[imatchcont] <- stations$Country[valcont[i]]
    }
  }

  if (!silent) print("Got the station meta-data!")
  if (sum(is.element(stations$number,stnr))==0) stnr <- NULL else {
    locmatch <- is.element(stations$number,stnr)
     print(paste("Found",stations$location[locmatch],"stnr=",stnr," lon=",
                 stations$lon[locmatch]," lat=",stations$lat[locmatch],
                 "  type=",as.character(stations$type[locmatch]),
                 " country=",stations$Country[locmatch]))
  }
  if (is.null(stnr) & !is.null(location)) {
     locmatch <- is.element(substr(lower.case(stations$location),2,nchar(location)+1),
                    lower.case(location))
     if (sum(locmatch)>0) {
       stnr <- stations$number[locmatch]
       print(paste("Found",stations$location[locmatch],"stnr=",stnr," lon=",
                   stations$lon[locmatch]," lat=",stations$lat[locmatch],
                   "  type=",as.character(stations$type[locmatch]),
                   " country=",stations$Country[locmatch]))
     } else {
       print(paste("Did not find '",location,"'",sep=""))
       print(substr(lower.case(stations$location),2,nchar(location)+1))
     }
     if (length(stnr)>1) {
       i <- as.numeric(readline(paste("Which of these ( 1 -",length(stnr),")? ")))
       stnr <- stnr[i]
     }
  }
  if (!is.null(lon) & !is.null(lat) & is.null(stnr)) {
     if (!silent) print(paste("Find the nearest station to ",lon,
                              "E and ",lat,"N.",sep=""))
     dist <- distAB(lon,lat,stations$lon,stations$lat)
     distmatch <- dist == min(dist,na.rm=TRUE)
     distmatch[is.na(distmatch)] <- FALSE
     if (sum(distmatch,na.rm=TRUE)>0) {
       stnr <- stations$number[distmatch]
       locmatch <- distmatch
       print(paste("Found",stations$location[locmatch],"stnr=",stnr," lon=",
                   stations$lon[locmatch]," lat=",stations$lat[locmatch],
                   "  type=",as.character(stations$type[locmatch]),
                   " country=",stations$Country[locmatch]))
     }
  }

  if (!is.null(stnr)) {
    contcode<- stations$Contry.code[is.element(stations$number,stnr)]
    lat<- stations$lat[is.element(stations$number,stnr)]
    lon<- stations$lon[is.element(stations$number,stnr)]
    location<- stations$location[is.element(stations$number,stnr)]
    country<- stations$Country[is.element(stations$number,stnr)]
    type<- stations$type[is.element(stations$number,stnr)]
    if (nchar(stnr)==8) stnr <- paste("0",stnr,sep="")
    fname<-paste("http://www.giss.nasa.gov/data/update/gistemp/TMPDIR/tmp.",
                 contcode,stnr,".1.1/",contcode,stnr,".1.1.txt",sep="")
    if (!silent) print(fname)
    data<-read.table(fname,header=TRUE)
    yy <- data[,1]
    data[abs(data) > 99] <- NA

    t2m <- station.obj(x=as.matrix(data[,2:13]),yy=yy,
                       ele=101,station=stnr,lat=lat,lon=lon,alt=NA,unit="deg C",
                       location=location,wmo.no=NA,country=country,obs.name="Temperature",
                       ref="Hansen, J. et al. 1999. J.Geophys.Res. 104, 30997-31022")
    t2m$type <- as.character(type)
    t2m$fname <- fname
    invisible(t2m)    
  } else {
    print("Did not find a station!")
    print("Returning a list of available stations instead")
    invisible(stations)
  }
  
}



# This R routine reads the DNMI/met.no data. The code
# will not work with the original NACD files: a space
# must be inserted between the December value and the
# country tag, and the missing values must be changed
# from '-9999' to ' -999'.
#
# Arguments:
# 'location' determines the time series.
# 'ele.c' determines the element (default=T2m).
#
# R.E. Benestad

getdnmi <- function(location,ele.c='101',silent = FALSE,
                    direc="data/stations/") {

if (!silent) print(paste("GETDNMI:",location,ele.c))
row<-switch(as.character(ele.c),
              't2m'='V5','rr'='V4','slp'='V9',
              't2'='V5','precip'='V4','temp'='V5',
              'snow depth'='V6','snowy days'='V7',
              'humidity'='V8','101'='V5','601'='V4',
              '112'='V10','122'='V11',
              'tax'='V10','tan'='V11')

descr <- switch(as.character(ele.c),
              't2m'='Temperature',
              'rr'='Precipitation',
              'slp'='Sea level pressure',
              't2'='Temperature',
              'precip'='Precipitation',
              'temp'='Temperature',
              'snow depth'='Snow depth',
              'snowy days'='Snowy days',
              'humidity'='Relative humidity',
              '101'='Temperature',
              '601'='Precipitation',
              '112'='Abs. max monthly Temp.',
              '122'='Abs. min monthly Temp.',
              'tax'='Abs. max monthly Temp.',
              'tan'='Abs. min monthly Temp.')
unit <- switch(as.character(ele.c),
              't2m'='deg C',
              'rr'='mm',
              'slp'='hPa',
              't2'='deg C',
              'precip'='mm',
              'temp'='deg C',
              'snow depth'='cm',
              'snowy days'='days',
              'humidity'='%',
              '101'='deg C',
              '601'='mm',
              '112'='deg C',
              '122'='deg C',
              'tax'='deg C',
              'tan'='deg C')

f.name<-paste(lower.case(location),".dat",sep="")
if (file.exists(paste(direc,f.name,sep=""))) {
  no.find <- FALSE 

  obs<-read.table(paste(direc,f.name,sep=""))
#  dnmi.meta <- read.table(paste(direc,"dnmi.meta",sep=""))
  dnmi.meta <- read.table("data/dvh.station.list",header=TRUE,as.is=TRUE)
  station<-obs$V1[1]
#  alt<- dnmi.meta$alt[dnmi.meta$dnmi.no==station]
#  lon<- dnmi.meta$lon[dnmi.meta$dnmi.no==station]
#  lat<- dnmi.meta$lat[dnmi.meta$dnmi.no==station]
#  location <- dnmi.meta$location[dnmi.meta$dnmi.no==station]
  alt<- as.numeric(dnmi.meta$Hoh[dnmi.meta$Stnr==station][1])
  lon<- dnmi.meta$Lon[dnmi.meta$Stnr==station][1]
  lat<- dnmi.meta$Lat[dnmi.meta$Stnr==station][1]
  location <- dnmi.meta$Navn[dnmi.meta$Stnr==station][1]
  yy <- obs$V2
  mm <- obs$V3
  YY<-as.numeric(row.names(table(obs$V2)))
  ny <- length(YY)
  x <- rep(NA,ny*12)
  eval(parse(text=paste("dat <- obs$",row,sep="")))
  for (i in 1:(ny*12)) {
    i.yy <- yy[1] + floor(i/12)
    i.mm <- mod(i-1,12)+1
    ii <- (yy == i.yy) & (mm == i.mm)
    if (sum(ii)>0) {
       x[i] <- dat[ii]
    } else x[i] <- NA
  }
  x[x<= -999] <- NA
  dim(x) <- c(12,ny)
  x <- t(x)
  
  country<-"Norway"
  xy<-COn0E65N(lon,lat)
  yy<-row.names(table(obs$V2))
} else {
  x <- NULL
  station <- NA
  YY <- NULL
  lat <- NULL
  lon <- NULL
  alt <- NULL
  xy <- list(x=NULL,y=NULL)
  unit <- NULL
  country <- NULL
  descr <- NULL
  no.find <- TRUE
}
getdnmi<-list(val=x,station=station,yy=YY,
                lat=lat,lon=lon,alt=alt,
                x.0E65N=xy$x,y.0E65N=xy$y,
                location=location,
                wmo.no=NA,
                start=NA,yy0=NA,ele=NA,
                obs.name=descr, unit=unit,country=country,
                quality=NA,found=!no.find,
                ref=paste('The Norwegian Meteorological Institute',
                  'climatological archive'))
class(getdnmi) <- c("station","monthly.station.record")
getdnmi
}
# This R routine reads the NACD data. The code
# will not work with the original NACD files: a space
# must be inserted between the December value and the
# country tag, and the missing values must be changed
# from '-9999' to ' -999'.
#
# Arguments:
# 'location' determines the time series.
# 'ele.c' determines the element (default=T2m).
#
# R.E. Benestad

getnacd <- function(location="prompt",ele.c='101',ascii=FALSE,silent=FALSE,
                    direc="data") {

if (location=="prompt") {
  locs <- avail.locs(as.integer(ele.c))
  print(length(locs))
  locs.name <- locs$name[locs$ident=="NACD"]
  print(locs.name)
  i.loc <- readline(prompt="Please enter the number of desired location: ")
  i.loc <- as.integer(i.loc)
  location <- locs.name[i.loc]
} 


if (is.character(ele.c)) {
  ele.c<-lower.case(ele.c)
  ele.c<-switch(ele.c,
                't2m'='101','rr'='601','slp'='401','cloud'='801',
                't2'='101','precip'='601','101'='101','401'='401',
                '601'='601','801'='801')
} else {
  ele.c<-as.character(ele.c)
}

#print(location)
#print(ele.c)
fr.name<-paste(direc,'/getnacd_',ele.c,'.Rdata',sep="")
ascii<- ascii | !file.exists(fr.name)

if (ascii) {
# Read the original ASCII files - slow
  obs<-read.fwf(
     paste('data/nacd_v1.',ele.c,sep=""),
     width=c(5,3,4,rep(5,12),3))

# Read the information about the stations: Metadata

  data(meta.nacd)

# Save as R-data-file
  save(obs,meta.nacd,file=fr.name)
}


# Load R-data-file - FAST!

#print(fr.name)
load(fr.name)

station<-obs$V1
ele<-obs$V2

yy<-obs$V3
country<-obs$V16

scale<-10
if (ele[1]==801) scale<-1

val<-as.matrix(obs[,4:15])/scale
val[val <= -99.9] <- NA

#print(obs[1,])
#print(meta.nacd[1,])
#print(c(as.character(meta.nacd$V1[1]),as.character(meta.nacd$V2[1]),
#        as.character(meta.nacd$V3[1]),as.character(meta.nacd$V4[1]),
#        as.character(meta.nacd$V5[1]),as.character(meta.nacd$V6[1]),
#        as.character(meta.nacd$V7[1]),as.character(meta.nacd$V8[1]),
#        as.character(meta.nacd$V9[1]),as.character(meta.nacd$V10[1]),
#        as.character(meta.nacd$V11[1]),as.character(meta.nacd$V12[1]),
#        as.character(meta.nacd$V13[1]),as.character(meta.nacd$V14[1]),
#        as.character(meta.nacd$V15[1]),as.character(meta.nacd$V16[1])))

if (is.character(location)) {
  nc<-nchar(location)
  location<-paste(upper.case(location),
                paste(rep(" ",21-nc),sep="",collapse=""),sep="")

  no.find<-FALSE
  if ((sum(is.element(meta.nacd$location,location) &
            is.element(as.numeric(as.character(meta.nacd$element)),ele))==0) &
      !(silent)) no.find<-TRUE
  meta<-meta.nacd[is.element(meta.nacd$location,location) &
                  is.element(as.numeric(as.character(meta.nacd$element)),ele),]

} else if (is.numeric(location)){
  if ((sum(is.element(meta.nacd$station.number,location) &
            is.element(as.numeric(as.character(meta.nacd$element)),ele))==0) &
      !(silent)) no.find<-TRUE
  meta<-meta.nacd[is.element(meta.nacd$station.number,location) &
                  is.element(as.numeric(as.character(meta.nacd$element)),ele),]
}

if (no.find) {
  print("getnacd: ERROR - cannot find the right record!")
  print(sum(is.element(meta.nacd$location,location) &
              is.element(meta.nacd$element,ele)))

  print("location")
  print(location)
  print("levels(meta.nacd$location)")
  print(levels(meta.nacd$location))

  print("ele")
  print(table(ele))
  print("levels(meta.nacd$element)")
  print(table(as.numeric(as.character(meta.nacd$element))))

  print(paste("sum(is.element(meta.nacd$location,location))=",
              sum(is.element(meta.nacd$location,location))))
  print(paste("sum(is.element(meta.nacd$element,ele))=",
              sum(is.element(meta.nacd$element,ele)))) 
  print("meta:")
  print(meta)
  print("station:")
  print(summary(station))

  print("country:")
  print(levels(country))
  print(meta$country)
}
 
iloc<-is.element(station,meta$station.number) &
                (country == meta$country) 

if (sum(iloc)==0) {
 print("summary(iloc)")
 print(summary(iloc))
 print("sum(iloc)")
 print(sum(iloc))
 print("sum(is.element(station,meta$station.number))")
 print(sum(is.element(station,meta$station.number)))
 print("sum(country == meta$country)")
 print(sum(country == meta$country))
}

obs.name<-switch(as.character(ele[1]),
                     '101'='monthly mean T(2m)',
                     '401'='monthly mean SLP',
                     '601'='monthly precipitation sum',
                     '801'='monthly mean cloud cover')
unit<-switch(as.character(ele[1]),
                     '101'='degree Celsius',
                     '401'='hPa',
                     '601'='mm',
                     '801'='%')
#print(as.character(meta$V16))
quality<-switch(as.character(meta$quality),
                ' H'='Homogenous, rigorously tested & adjusted',
                'H'='Homogenous, rigorously tested & adjusted',
                ' T'='Tested, maybe adjusted but not perfectly H.',
                'T'='Tested, maybe adjusted but not perfectly H.',
                ' N'='Not tested for inhomogenouity',
                'N'='Not tested for inhomogenouity',
                ' E'='Environm. changes prevents clim.change studies',
                'E'='Environm. changes prevents clim.change studies',
                ' I'='Inhomogenous series which presently are unadjustable',
                'I'='Inhomogenous series which presently are unadjustable')

lat<-meta$degN + meta$minN/60
lon<-meta$degE + meta$minE/60
lat[meta$N.S==" S"]<-lat[meta$N.S==" S"]*-1
lon[meta$E.W==" W"]<-lon[meta$E.W==" W"]*-1
#print(levels(meta$V8))
#print(levels(meta$V11))

xy<-COn0E65N(lon,lat)

getnacd<-list(val=val[iloc,],station=meta$station.number,yy=yy[iloc],
              lat=lat,lon=lon,alt=meta$alt,
              x.0E65N=xy$x,y.0E65N=xy$y,
              location=location, wmo.no=meta$wmo.number,
              start=meta$start,yy0=meta$year.1,ele=ele[1],
              obs.name=obs.name, unit=unit,country=meta$country,
              quality=quality,found=!no.find,
              ref='Frich et al. (1996), DMI scientific report 96-1')
class(getnacd) <- c("station","monthly.station.record")
getnacd
}
# This R routine reads the NACD data. The code
# will not work with the original NACD files: a space
# must be inserted between the December value and the
# country tag, and the missing values must be changed
# from '-9999' to ' -999'.
#
# Arguments:
# 'location' determines the time series.
# 'ele.c' determines the element (default=T2m).
#
# R.E. Benestad

getnordklim <- function(location="prompt",ele.c='101',
                        ascii=FALSE,silent=FALSE,direc="data") {


#library(stringfun)
#source("COn0E65N.R")
#source("avail.locs.R")

#if (location=="prompt") {
#  locs <- avail.locs(as.integer(ele.c))
#  print(length(locs))
#  locs.name <- locs$name[locs$ident=="NORDKLIM"]
#  print(locs.name)
#  i.loc <- readline(prompt="Please enter the number of desired location: ")
#  i.loc <- as.integer(i.loc)
#  location <- locs.name[i.loc]
#}
ele <- c(101,111,112,113,121,122,123,401,601,602,701,801,911)
  
if (is.character(ele.c)) {
  ele.c<-lower.case(ele.c)
  ele.c<-switch(ele.c,
                't2m'='101','rr'='601','slp'='401','cloud'='801',
                't2'='101','precip'='601','101'='101','401'='401','111'='111',
                '112'='112','113'='113','121'='121','122'='122','123'='123',
                '601'='601','602'='602','801'='801','911'='911')
} else {
  ele.c<-as.character(ele.c)
}

#print(location)
#print(ele.c)
fr.name<-paste(direc,'/nordklim_',ele.c,'.Rdata',sep="")
ascii<- ascii | !file.exists(fr.name)

if (ascii) {
# Read the original ASCII files - slow
  obs<-read.table(paste('data/p',ele.c,'.prn',sep=""))

# Read the information about the stations: Metadata
# 

  data(nordklim.meta)

# Save as R-data-file
  save(obs,meta,file=fr.name)
}

# Load R-data-file - FAST!

load(fr.name)

station<-as.integer(obs$V1)
ele<-obs$V2
yy<-obs$V3
country<-strip(obs$V16)

obs.name<-switch(as.character(ele[1]),
              '101'='mean T(2m)',
              '111'='mean maximum T(2m)',
              '112'='highest maximum T(2m)',
              '113'='day of Th date Thd',
              '121'='mean minimum T(2m)',
              '122'='lowest minimum T(2m)',
              '123'='day of Tl date Tld',
              '401'='mean SLP',
              '601'='monthly accum. precip.',
              '602'='maximum precip.',
              '701'='Number of days with snow cover (> 50% covered) days dsc',
              '801'='Mean cloud cover % N',
              '911'='mean snow depth')

unit <-switch(as.character(ele[1]),
              '101'='deg C',
              '111'='deg C',
              '112'='deg C',
              '113'='date',
              '121'='deg C',
              '122'='deg C',
              '123'='date',
              '401'='hPa',
              '601'='mm/month',
              '602'='mm/day',
              '701'='days',
              '801'='%',
              '911'='cm')
#print(paste("obs name=",obs.name,"; unit=",unit))

scale <- switch(as.character(ele[1]),
               '101'=0.1,
               '111'=0.1,
               '112'=0.1,
               '113'=1,
               '121'=0.1,
               '122'=0.1,
               '123'=1,
               '401'=0.1,
               '601'=0.1,
               '602'=0.1,
               '701'=1,
               '801'=1,
               '911'=1)
#print("scale")
#print(scale)
#print(ele[1])
#print(dim(obs))
#print(summary(obs))

val<-as.matrix(obs[,4:15])*scale
val[val <= -99.9] <- NA

#print(obs[1,])
#print(meta[1,])
#print(c(as.character(meta$V1[1]),as.character(meta$V2[1]),
#        as.character(meta$V3[1]),as.character(meta$V4[1]),
#        as.character(meta$V5[1]),as.character(meta$V6[1]),
#        as.character(meta$V7[1]),as.character(meta$V8[1]),
#        as.character(meta$V9[1]),as.character(meta$V10[1]),
#        as.character(meta$V11[1]),as.character(meta$V12[1]),
#        as.character(meta$V13[1]),as.character(meta$V14[1]),
#        as.character(meta$V15[1]),as.character(meta$V16[1])))

#print(paste("GETNORDKLIM: call strip for ",location))
#location<-upper.case(strip(abbreviate(location)))
location<-upper.case(strip(location))
#print("GETNORDKLIM: call strip for meta$location")
#meta$location<-upper.case(strip(abbreviate(meta$location)))
meta$location<-upper.case(strip(meta$location))

#print("search appendix:")
#print(as.character(meta$V5))
in.app<-is.element(meta$location,location) 
#print("in.app=")
#print(sum(in.app))

no.find<-FALSE
if ((sum(in.app)==0) & !(silent)) {
  print("getnordklim: ERROR - cannot find the right record!")
  print(sum(is.element(meta$location,location)))
  
  print("ele.c")
  print(ele.c)
  print("location")
  print(location)
  print("table(ele)")
  print(table(ele))
  print(as.character(meta$location))
  print("sum(is.element(meta$location,location))")
  print(sum(is.element(meta$location,location)))
  print("meta:")
  print(meta)
  print("station:")
  print(summary(station))

  print("country:")
  print(levels(country))
  print(meta$V3)
  no.find<-TRUE

  print("Available locations:")
  print(meta$location)

  
#if (no.find) {
# print("summary(iloc)")
# print(summary(iloc))
# print("sum(iloc)")
# print(sum(iloc))
# print("sum(is.element(station,meta$V2))")
# print(sum(is.element(station,meta$V2)))
# print("sum(country == meta$V3)")
# print(sum(country == meta$V3))
#}

}

if (!no.find) {
  meta<-meta[in.app,]
#print(meta)
#print(table(station))
#print(meta$number)

  iloc<-is.element(station,meta$number)

#print(sum(iloc))
#print(as.character(meta$V16))

#  print(" Test 1:")
#  print(meta[1,])
#  print(c(meta$Lat.deg,meta$Lat.min,meta$Lon.deg,meta$Lon.min))

  lat<-meta$Lat.deg + meta$Lat.min/60
  lon<-meta$Lon.deg + meta$Lon.min/60
#  print(paste("GETNORDKLIM: call strip for N.S & E.W","'",
#              meta$N.S,"', '",meta$E.W,"'",sep=""))
  meta$N.S<-strip(meta$N.S)
  meta$E.W<-strip(meta$E.W)
  lat[(meta$N.S=="S") | (meta$N.S==" S")]<-lat[(meta$N.S=="S") | (meta$N.S==" S")]*-1
  lon[(meta$E.W=="W") | (meta$E.W==" W")]<-lon[(meta$E.W=="W") | (meta$E.W==" W")]*-1
  
#print(levels(meta$V8))
#print(levels(meta$V11))

  xy<-COn0E65N(lon,lat)
} else {
  lat<-NA
  lon<-NA
  xy<-list(x=NA,y=NA)
  location<-"Not found"
}
#print(" Test 2: dim(val)")
#print(length(station))
#print(length(yy))
#print(dim(val))
#print(dim(val[iloc,]))
#print(length(yy[iloc]))
#print(" Test 3:")
#print(c(lon,lat,meta$altitude,ele[1]))

getnordklim<-list(val=val[iloc,],station=meta$number,yy=yy[iloc],
              lat=lat,lon=lon,alt=meta$height,
              x.0E65N=xy$x,y.0E65N=xy$y,
              location=location, wmo.no=NA,
              start=NA,yy0=NA,ele=ele[1],
              obs.name=obs.name, unit=unit,country=meta$country,
              found=!no.find,
              ref=paste('Rissanen et al., (2000), DNMI KLIMA 10/00,',
                        'Norwegian Meteololog. Inst., met.no'))
class(getnordklim) <- c("station","monthly.station.record")
getnordklim
}
grd.box.ts <- function(x,lon,lat,what="abs",greenwich=TRUE,mon=NULL,
                       col="grey10",lwd=1,lty=1,pch=26,add=FALSE,
                       filter=NULL,type="s",main=NULL,sub=NULL,xlab=NULL,ylab=NULL) {

  library(akima)
  library(ts)
  
  if ((class(x)[1]!="field") & (class(x)[2]!="monthly.field.object") &
      (class(x)[2]!="daily.field.object") ) stop("Need a field.object")
  
  if (greenwich) {
    x$lon[x$lon > 180] <- x$lon[x$lon > 180]-360
    x.srt <- order(x$lon)
    x$lon <- x$lon[x.srt]
    x$dat <- x$dat[,,x.srt]
  }

  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')
  descr <- "Interpolated value"
  date <- " "
  if (!is.null(mon)) {
    im <- x$mm== mon
    x$dat <- x$dat[im,,]
    x$yy <- x$yy[im]
    x$mm <- x$mm[im]
    x$dd <- x$dd[im]
    x$id.t <- x$id.t[im]
    date <- cmon[mon]
  }
  dx <- x$lon[2] - x$lon[1]
  dy <- x$lat[2] - x$lat[1]
  x.keep <- (x$lon - 3*dx <= lon) & (x$lon + 3*dx >= lon)
  y.keep <- (x$lat - 3*dy <= lat) & (x$lat + 3*dx >= lat)
  x$lon <- x$lon[x.keep]
  x$lat <- x$lat[y.keep]
  x$dat <- x$dat[,y.keep,x.keep]
  if (sum(!is.finite(x$dat))>0) x$dat[!is.finite(x$dat)] <- 0
  lat.x<-rep(x$lat,length(x$lon))
  lon.x<-sort(rep(x$lon,length(x$lat)))
  nt <- length(x$yy)
  y <- rep(NA,nt)
  for (it in 1:nt) {
    Z.in<-as.matrix(x$dat[it,,])
    Z.out<-interp(lat.x,lon.x,Z.in,lat,lon)
    y[it] <- Z.out$z
  }

#  print("time unit")
  
if (!is.null(attributes(x$tim)$unit)) {
  attr(x$tim,"units") <- attributes(x$tim)$unit
}
  #print(attributes(x$tim)$units)
  #print(attributes(x$tim)$unit)

  tunit <- attributes(x$tim)$units
  if (!is.null(tunit)) tunit <- lower.case(substr(tunit,1,3)) else
                       tunit <- "mon"
                       
  if (tunit== "mon") {
    clim <- y
    for (im in 1:12) {
      ii <- mod((1:nt)-1,12)+1 == im
      clim[ii] <- mean(y[ii],na.rm=T)
    }
  } else {
    ac.mod<-matrix(rep(NA,nt*6),nt,6)
    if (tunit=="day") jtime <- x$tim
    if (tunit=="hou")  jtime <- x$tim/24
    ac.mod[,1]<-cos(2*pi*jtime/365.25); ac.mod[,2]<-sin(2*pi*jtime/365.25)
    ac.mod[,3]<-cos(4*pi*jtime/365.25); ac.mod[,4]<-sin(4*pi*jtime/365.25)
    ac.mod[,5]<-cos(6*pi*jtime/365.25); ac.mod[,6]<-sin(6*pi*jtime/365.25)
    ac.fit<-lm(y ~ ac.mod); clim <- ac.fit$fit
  } 

#  print("what?")
  ts <- switch(lower.case(substr(what,1,3)),
                "ano"=y - clim,
                "cli"=clim,
                "abs"=y)
  descr <- switch(lower.case(substr(what,1,3)),
                "ano"="anomaly",
                "cli"="climatological",
                "abs"="absolute value")

  if (!is.null(filter)) ts <- filter(ts,filter)
  if (is.null(main)) main <-  x$v.name
  if (is.null(sub)) sub <- paste("Interpolated at ",lon,"E, ",lat,"N ",date,sep="")
  if (is.null(xlab)) xlab <- "Time"
  if (is.null(ylab)) ylab <- attributes(x$dat)$unit


  if (!add) {

     plot(x$yy+(x$mm-0.5)/12,ts,type=type,pch=pch,
       main=main,sub=sub,xlab=xlab,ylab=ylab,col=col,lwd=lwd,lty=lty)

     points(x$yy+(x$mm-0.5)/12,ts,pch=pch,col=col)
   } else {
     if (type!='p') lines(x$yy+(x$mm-0.5)/12,ts,type=type,col=col,lwd=lwd,lty=lty)
     points(x$yy+(x$mm-0.5)/12,ts,pch=pch,col=col)
   }
  grid()
#  print("plotted")
  
  dd.rng <- range(x$dd)
  if (is.null(attr(x$tim,"units"))) attr(x$tim,"units") <- "unknown"
  if ( (tunit=="mon") |
       ((dd.rng[2]-dd.rng[1]<4) & (x$mm[2]-x$mm[1]>0)) ) {
#    print("Monthly")
    results <- station.obj(ts,yy=x$yy,obs.name=x$v.name,unit=attr(x$dat,"unit"),
                           ele=NA,mm=x$mm,
                           station=NA,lat=round(lat,4),lon==round(lon,4),alt=NA,
                           location="interpolated",wmo.no=NA,
                           start=min(x$yy),yy0=attr(x$tim,"time_origin"),country=NA,
                           ref="grd.box.ts.R (clim.pact)")
  } else {
    results <- station.obj.dm(t2m=ts,precip=rep(NA,length(ts)),
                              dd=x$dd,mm=x$mm,yy=x$yy,
                              obs.name=x$v.name,unit=attr(x$dat,"unit"),ele=NA,
                              station=NA,lat=round(lat,4),lon=round(lon,4),alt=NA,
                              location="interpolated",wmo.no=NA,
                              start=min(x$yy),yy0=attr(x$tim,"time_origin"),country=NA,
                              ref="grd.box.ts.R (clim.pact)")
  }

#  print("exit grd.box.ts()")
  invisible(results)
}


instring <- function(c,target,case.match=TRUE) {
  l <- nchar(target)
  if (!case.match) {
    c <- lower.case(c)
    target <- lower.case(target)
  }
  pos <- 0
  for (i in 1:l) {
    tst <- substr(target,i,i)
   if (tst==c) pos <- c(pos,i)
   }
  if (length(pos) > 1) pos <- pos[-1]
  pos
}
# This routine computes the Julian day given a month, day, and year.
# The algorithm is taken from Press et al. (1989), "Numerical Recipes 
# in Pascal", Cambridge, p. 10.
#
# This function removes the dependency to outdated packages 'chron' and
# 'date'.
#
# R.E. Benestad, met.no, Oslo, Norway 04.09.2003
# rasmus.benestad@met.no
#------------------------------------------------------------------------

julday <- function(mm,id,iyyy) {
  igreg <- 588829
  mm <- trunc(mm)
  id <- trunc(id)
  iyyy <- trunc(iyyy)
  im <-  (iyyy == 0)
  if (sum(im)>0) return("There is no year zero!")
  if ((length(mm) != length(id)) | (length(mm) != length(iyyy)) |
      (length(iyyy) != length(id))) return("The vectors must have same length!")
  im <-  (iyyy < 0)
  if (sum(im)>0) iyyy[im] <- iyyy[im]+1
  jy <- mm*0; jm <- mm*0; ja <- mm*0; 
  im <-  (mm > 2)
  if (sum(im)>0) {
    jy[im] <- iyyy
    jm[im] <- mm+1
  }
  im <-  (mm <= 2)
  if (sum(im)>0) {
    jy[im] <- iyyy-1
    jm[im] <- mm+13
  }
  jul <- trunc(365.25*jy) + trunc(30.6001*jm) + id + 1720995
  im <- (id+31*(mm+12*iyyy)>= igreg)
  if (sum(im)>0) {
    ja[im] <- trunc(0.01*jy)
    jul[im] <- jul+2-ja[im]+trunc(0.25*ja[im])
  }
  julday <- jul
  invisible(julday)
}
# Computes approximate cartesian coordinates in km
# centered on 0E 65N.
# R.E. Benestad, DNMI, 04.01.2001
#
km2lat <- function(x, y, x.centre=0, y.centre=65) {
  a<-6357 # km
  km2lat<-180/pi * y /a + y.centre
  km2lat
  }
# Computes approximate cartesian coordinates in km
# centered on 0E 65N.
# R.E. Benestad, DNMI, 04.01.2001
#
km2lon <- function(x, y, x.centre=0, y.centre=65) {
  a<-6357 # km
  lat<-180/pi * y /a + y.centre
  km2lon<-180/pi * x/(a*cos(pi*lat/180)) + x.centre
  km2lon
  }
# Returns the lower case of a string;
# Test:
# > lower.case("ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890") gives
# [1] "abcdefghijklmnopqrstuvwxyz1234567890"
# > lower.case("abcdefghijklemnoprstuvwxyz1234567890")
# [1] "abcdefghijklemnoprstuvwxyz1234567890"
# R.E. Benestad (REB), DNMI (met.no), 08.01.2002
#                REB:  03.05.2002 - modified to hande string arrays

lower.case <- function(u.case) {

  lfac<-FALSE                           # Set flag if we are dealing with a factor
                                        # object. Then the output is converted to
                                        # factor.
  
  if (is.factor(u.case)) { lfac <- TRUE }
  
  str<-as.character(u.case)
#  print(str)
  lower.case<-str

  for (is in 1:length(str)) {
    nc<-nchar(str[is])
    lower.case[is]<-""
    for (ic in 1:nc) {
      sstr<-substr(str[is],ic,ic)
      if (sstr=="E") sstr<-"e"     # Fudge - E didn't work in switch..
      u.l<-switch(as.character(sstr),
                      A="a",B="b",C="c",D="d",F="f",G="g",H="h",I="i",
                      J="j",K="k",L="l",M="m",N="n",O="o",P="p",Q="q",R="r",
                      S="s",T="t",U="u",V="v",W="w",X="x",Y="y",Z="z")
      if (length(u.l) == 0) u.l<-sstr
#    print(c(sstr,u.l,lower.case))
      lower.case[is]<-paste(lower.case[is],u.l,sep="")
    }
  }
  if (lfac) {
    lower.case<-factor(lower.case)
  }
  lower.case
}
  
map <- function(x,y=NULL,col="black",lwd=1,lty=1,sym=TRUE,
                    plot=TRUE,inv.col=FALSE,add=FALSE,las = 1) {

  if (!is.null(y)) map <- x$map - y$map else  map <- x$map

  if (sym) {
    z.levs <- seq(-max(abs(as.vector(map)),na.rm=TRUE),
                   max(abs(as.vector(map)),na.rm=TRUE),length=41)
  } else {
    z.levs <- seq(min(as.vector(map),na.rm=TRUE),
                  max(as.vector(map),na.rm=TRUE),length=41)
  }
  if (plot) {
    my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                  c(abs(sin((0:40)*pi/40))),
                  c(c(rep(1,21),seq(1,0,length=20))))
    if (inv.col) my.col <- reverse(my.col)
    if (!add) { filled.contour(x$lon,x$lat,map,
                               col = my.col,levels=z.levs,
                               main=paste(attributes(x$dat)$"long_name"),
                               sub=x$date,xlab="Longitude",ylab="Latitude")
              }
    # From filled.contour in base
    mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
    on.exit(par(par.orig))

    w <- (3 + mar.orig[2]) * par('csi') * 2.54
    layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
    par(las = las)
    mar <- mar.orig
    mar[4] <- 1
    par(mar=mar)
    contour(x$lon,x$lat,map,add=TRUE,col=col,lwd=lwd,lty=lty)
    addland()
  }
  results <- list(lon=x$lon,lat=x$lat,map=map,v.name=,x$v.name,
                  tim=NULL,date=NULL,attributes=x$attributes)
  class(results) <- "map"
#  attr(results) <- attr(x)
  attr(results,"descr") <- "Mean values"
  invisible(results) 
}
# Plots the spatial EOF patterns 
#
# Reference: R.E. Benestad et al. (2002),
#            Empirically downscaled temperature scenarios for Svalbard,
#            submitted to Atm. Sci. Lett.
#
#            R.E. Benestad (2001),
#            A comparison between two empirical downscaling strategies,
#            Int. J. Climatology, 1645-1668, vol. 21, DOI 10.1002/joc.703
#
# R.E. Benestad, met.no, Oslo, Norway 10.05.2002
# rasmus.benestad@met.no
#

#------------------------------------------------------------------------

map.eof <- function(x,i.eof=1,nlevs=9,add=FALSE,
            col=c("red","blue","darkgreen","steelblue"),lwd=2,lty=1) {

if (class(x)[1]!= "eof") stop ("The argument must be an 'eof' object") 

attach(x)

dims <- dim(x$EOF) 
if (length(dims)==3) dim(x$EOF) <- c(dims[1],dims[2]*dims[3])

title.1 <- paste("EOF pattern #",i.eof,"(",class(x)[2],")",sep="")
i.last <- 0
id <- row.names(table(x$id.x))
if (!add) {
  plot(c(floor(min(lon)),ceiling(max(x$lon))),
     c(floor(min(lat)),ceiling(max(x$lat))),
     type="n",main=title.1,
     sub=paste(x$f.name," (",c.mon,")"),
     xlab="Longitude",ylab="Latitude")
}

if (range(x$lon)[2]-range(x$lon)[1] > 360) {
  xy.cont <- COn0E65N(lon.cont, lat.cont)
  addland(lon=xy.cont$x,lat=xy.cont$y)
} else addland()
grid()

col.tab <- col[1:length(id)]
neofs <- length(x$var)
i.last <- 0
for (i in 1:x$n.fld) {
  #print(c(i,NA,x$size[,i],NA,i.last,NA,i.last+1,i.last+x$size[2,i]*x$size[3,i]))
  i.fld <- seq(i.last+1,i.last+x$size[2,i]*x$size[3,i],by=1)
  i.last <- max(i.fld)
  EOF.1 <- x$EOF[,i.fld]
  dim(EOF.1)<-c(neofs,x$size[2,i],x$size[3,i])
  eof.patt<-t(EOF.1[i.eof,,])
  i.lon <- x$id.lon == id[i]
  i.lat <- x$id.lat == id[i]
  lon.x <- x$lon[i.lon]
  lat.x <- x$lat[i.lat]
  #print(summary(as.vector(eof.patt)))
  #print(lon.x)
  #print(lat.x)
  contour(lon.x,lat.x,eof.patt,
          nlevels=nlevs,add=TRUE,lwd=2,col=col.tab[i])
}
if ((x$n.fld>1) & (!add)) legend(min(x$lon),max(x$lat),id,
             col=c(col.tab),lty=1,
             lwd=2,merge=TRUE, bg='gray95')

  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')

  results <- list(map=eof.patt,lon=x$lon,lat=x$lat,tim=x$tim,
                  date=c.mon,description=id[i])

  class(results) <- c("map","eof")
}


# R.E. Benestad

mapField <- function(x,l=NULL,greenwich=TRUE,what="ano",method="nice",
                       col="black",lwd=2,lty=1,add=FALSE,las = 1) {
  if ((class(x)!="monthly.field.object") & (class(x)!="field.object") &
      (class(x)!="daily.field.object") & (class(x)!="field")) {
      stop("Need a field.object") }
  if (is.null(l)) l <- length(x$tim)
  nx <- length(x$lon)
  ny <- length(x$lat)
  nt <- length(x$tim)
  clim <- x$dat[l,,]
  if ( (x$mm[2]-x$mm[1]>=1) & (x$dd[2]==x$dd[1]) ) {
    it <- mod(1:nt,12)==mod(l,12)
    for (j in 1:ny) {
      for (i in 1:nx) {
        clim[j,i] <- mean(x$dat[it,j,i],na.rm=TRUE)
      }
    }
  } else {
      ac.mod<-matrix(rep(NA,nt*6),nt,6)
      ac.mod[,1]<-cos(2*pi*x$tim/365.25)
      ac.mod[,2]<-sin(2*pi*x$tim/365.25)
      ac.mod[,3]<-cos(4*pi*x$tim/365.25)
      ac.mod[,4]<-sin(4*pi*x$tim/365.25)
      ac.mod[,5]<-cos(6*pi*x$tim/365.25)
      ac.mod[,6]<-sin(6*pi*x$tim/365.25)
      dim(x$dat) <- c(nt,ny*nx)
      dim(clim) <- c(ny*nx)
      for (ip in seq(1,ny*nx,by=1)) {
        if (sum(is.finite(x$dat[,ip])) > 0) {
          ac.fit<-lm(x$dat[,ip] ~ ac.mod)
          clim[ip]<-ac.fit$fit[l]
        } else clim[ip]<- NA
      }
      dim(x$dat) <- c(nt,ny,nx)
      dim(clim) <- c(ny,nx)
    }

  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')
  
  if ( (x$mm[2]-x$mm[1]>=1) & (x$dd[2]==x$dd[1]) ) {
    date <- switch(lower.case(substr(what,1,3)),
                   "ano"=paste(cmon[x$mm[l]],x$yy[l]),
                   "cli"=cmon[x$mm[l]],
                   "abs"=paste(cmon[x$mm[l]],x$yy[l]))
  } else {
    date <- switch(lower.case(substr(what,1,3)),
                   "ano"=paste(x$dd[l],cmon[x$mm[l]],x$yy[l]),
                   "cli"=paste(x$dd[l],cmon[x$mm[l]]),
                   "abs"=paste(x$dd[l],cmon[x$mm[l]],x$yy[l]))
  }
  if (greenwich) {
    x$lon[x$lon > 180] <- x$lon[x$lon > 180]-360
    x.srt <- order(x$lon)
    x$lon <- x$lon[x.srt]
    clim <- clim[,x.srt]
    x$dat <- x$dat[,,x.srt]
  }
  anom <- x$dat[l,,]-clim
  map <- switch(lower.case(substr(what,1,3)),
                "ano"=anom,
                "cli"=clim,
                "abs"=x$dat[l,,])
  descr <- switch(lower.case(substr(what,1,3)),
                "ano"="anomaly",
                "cli"="climatological",
                "abs"="absolute value")
  z.levs <- seq(-max(abs(as.vector(map)),na.rm=TRUE),
                 max(abs(as.vector(map)),na.rm=TRUE),length=41)
  my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                c(abs(sin((0:40)*pi/40))),
                c(c(rep(1,21),seq(1,0,length=20))))
  if ((!add) & (method!="nice")) {
        image(x$lon,x$lat,t(map),levels=z.levs,
        main=paste(attributes(x$dat)$"long_name",descr),
        sub=date,xlab="Longitude",ylab="Latitude")
       } else {
         filled.contour(x$lon,x$lat,t(map),
                        col = my.col,levels=z.levs,
                        main=paste(attributes(x$dat)$"long_name",descr),
                        sub=date,xlab="Longitude",ylab="Latitude")
       }

# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = las)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)
  contour(x$lon,x$lat,t(map),add=TRUE,col=col,lwd=lwd,lty=lty)
  addland()
  results <- list(map=t(map),lon=x$lon,lat=x$lat,tim=x$tim,
                  date=date,description=descr)
  class(results) <- "map"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- descr
  invisible(results)

}
# R.E. Benestad

mapField <- function(x,l=NULL,greenwich=TRUE,
                     what="ano",method="nice",val.rng=NULL,
                     col="black",col.coast="grey",lwd=2,lty=1,
                     add=FALSE,las = 1) {
  if ((class(x)[2]!="monthly.field.object") & (class(x)[2]!="field.object") &
      (class(x)[2]!="daily.field.object") & (class(x)[1]!="field")) {
      stop("Need a field.object") }
  if (is.null(l)) l <- length(x$tim)
#  print("here")
  nx <- length(x$lon)
  ny <- length(x$lat)
  nt <- length(x$tim)
  clim <- x$dat[l,,]
  dd.rng <- range(x$dd)
  if (is.null(attr(x$tim,"units"))) attr(x$tim,"units") <- x$attributes$time.unit
  if ( (lower.case(substr(attr(x$tim,"units"),1,5))=="month") |
       ((dd.rng[2]-dd.rng[1]<4) & (x$mm[2]-x$mm[1]>0)) ) {
    it <- mod(1:nt,12)==mod(l,12)
    for (j in 1:ny) {
      for (i in 1:nx) {
        clim[j,i] <- mean(x$dat[it,j,i],na.rm=TRUE)
      }
    }
  } else {
      ac.mod<-matrix(rep(NA,nt*6),nt,6)
      if (substr(lower.case(attributes(x$tim)$units),1,3)=="day") jtime <- x$tim
      if (substr(lower.case(attributes(x$tim)$units),1,4)=="hour")  jtime <- x$tim/24
      ac.mod[,1]<-cos(2*pi*jtime/365.25); ac.mod[,2]<-sin(2*pi*jtime/365.25)
      ac.mod[,3]<-cos(4*pi*jtime/365.25); ac.mod[,4]<-sin(4*pi*jtime/365.25)
      ac.mod[,5]<-cos(6*pi*jtime/365.25); ac.mod[,6]<-sin(6*pi*jtime/365.25)               
      dim(x$dat) <- c(nt,ny*nx)
      dim(clim) <- c(ny*nx)
      for (ip in seq(1,ny*nx,by=1)) {
        if (sum(is.finite(x$dat[,ip])) > 0) {
          ac.fit<-lm(x$dat[,ip] ~ ac.mod)
          clim[ip]<-ac.fit$fit[l]
        } else clim[ip]<- NA
      }
      dim(x$dat) <- c(nt,ny,nx)
      dim(clim) <- c(ny,nx)
    }

#  print("here")
  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')
  
  if ( (x$mm[2]-x$mm[1]>=1) & (x$dd[2]==x$dd[1]) ) {
    date <- switch(lower.case(substr(what,1,3)),
                   "ano"=paste(cmon[x$mm[l]],x$yy[l]),
                   "cli"=cmon[x$mm[l]],
                   "abs"=paste(cmon[x$mm[l]],x$yy[l]))
  } else {
    date <- switch(lower.case(substr(what,1,3)),
                   "ano"=paste(x$dd[l],cmon[x$mm[l]],x$yy[l]),
                   "cli"=paste(x$dd[l],cmon[x$mm[l]]),
                   "abs"=paste(x$dd[l],cmon[x$mm[l]],x$yy[l]))
  }
  if (greenwich) {
    x$lon[x$lon > 180] <- x$lon[x$lon > 180]-360
    x.srt <- order(x$lon)
    x$lon <- x$lon[x.srt]
    clim <- clim[,x.srt]
    x$dat <- x$dat[,,x.srt]
  }
  anom <- x$dat[l,,]-clim
  map <- switch(lower.case(substr(what,1,3)),
                "ano"=anom,
                "cli"=clim,
                "abs"=x$dat[l,,])
  descr <- switch(lower.case(substr(what,1,3)),
                "ano"="anomaly",
                "cli"="climatological",
                "abs"="absolute value")
  if (is.null(val.rng)) {
    print("set range")
    nn <- floor(-max(abs(as.vector(map)),na.rm=TRUE))
    xx <- ceiling(max(abs(as.vector(map)),na.rm=TRUE))
    nl <- xx-nn
    while (nl > 20) {
      nl <- nl/10
    }
    while (nl < 5) {
      nl <- nl*2
    }
    scl <- 10^floor(log(max(abs(as.vector(map)),na.rm=TRUE))/log(10))
#    print(scl)
    z.levs <- round(seq(nn,xx,length=nl)/scl,2)*scl
#    print(z.levs)
    my.col <- rgb(c(seq(0,1,length=floor(nl/2)),rep(1,ceiling(nl/2))),
                  c(abs(sin((0:(nl-1))*pi/(nl-1)))),
                  c(c(rep(1,ceiling(nl/2)),seq(1,0,length=floor(nl/2)))))
#    print(nl)
  } else {
    z.levs <- seq(val.rng[1],val.rng[2],length=41)
    my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                  c(abs(sin((0:40)*pi/40))),
                  c(c(rep(1,21),seq(1,0,length=20))))
  }
  if ((!add) & (method!="nice")) {
        newFig()
        image(x$lon,x$lat,t(map),levels=seq(nn,xx,length=101),
        main=paste(attributes(x$dat)$"long_name",descr),
        sub=date,xlab="Longitude",ylab="Latitude")
       } else if (!add) {
         newFig()
         filled.contour(x$lon,x$lat,t(map),
                        col = my.col,levels=z.levs,
                        main=paste(attributes(x$dat)$"long_name",descr),
                        sub=date,xlab="Longitude",ylab="Latitude")
       }

# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = las)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)
  contour(x$lon,x$lat,t(map),add=TRUE,col=col,lwd=lwd,lty=lty,levels=z.levs)
  addland(col=col.coast)
  results <- list(map=t(map),lon=x$lon,lat=x$lat,tim=x$tim,
                  date=date,description=descr,attributes=x$attributes)
  class(results) <- "map"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- descr
  invisible(results)

}
meanField <- function(x,lon.rng=NULL,lat.rng=NULL,t.rng=NULL,mon=NULL) {

  season<-cbind(c(12,1,2),c(3,4,5),c(6,7,8),c(9,10,11))
  months<-c("Jan","Feb","Mar","Apr","May","Jun",
          "Jul","Aug","Sep","Oct","Nov","Dec")
  season.c<-c("","DJF","MAM","JJA","SON")
  dat <- x$dat
  lons <- x$lon
  lats <- x$lat
  mm <- x$mm
  month <- NULL
  if (!is.null(lon.rng)) {
    ix <- (lons >= lon.rng[1]) & (lons <= lon.rng[2])
    dat <- dat[,,ix]
    lons <- lons[ix]
  }
  if (!is.null(lat.rng)) {
    iy <- (lats >= lat.rng[1]) & (lats <= lat.rng[2])
    dat <- dat[,iy,]
    lats <- lats[iy]
  }
  if (!is.null(t.rng)) {
    it <- (x$yy >= t.rng[1]) & (x$yy <= t.rng[2])
    dat <- dat[it,,]
  }
  if (!is.null(mon)) {
    if (class(x)[2]=="monthly.field.object") {
      i.mm <- is.element(mm,mon)
      dat <- dat[i.mm,,]
      mm <- mm[i.mm]
      month <- months[mon+1]
    } else if (class(x)[2]=="daily.field.object") {
      mon <- mod(mon-1,4)+1
      mon <- season[,mon]
      i.mm <- is.element(mm,mon)
      dat <- dat[i.mm,,]
      mm <- mm[i.mm]
      month <- season.c[mon+1]
    }
  }
  nj <- length(lats)
  ni <- length(lons)
  map <- matrix(rep(NA,ni*nj),ni,nj)
  for (j in 1:nj) {
    for (i in 1:ni) {
      map[i,j] <- mean(dat[,j,i],na.rm=T)
    }
  }
  
  results <- list(lon=lons,lat=lats,map=map,v.name=,x$v.name,
                  tim=month,date=t.rng)
  class(results) <- "map"
#  attr(results) <- attr(x)
  attr(results,"long_name")<- paste("Mean",x$v.name)   
  attr(results,"descr") <- "Mean values"
  invisible(results) 
}
# Merges two station series of the same variable but from different sources to
# produce a long, updated series.
# R.E. Benestad

mergeStation <- function(x.1,x.2,plot=FALSE) {

if ( (class(x.1)[2]!="monthly.station.record") |
     (class(x.2)[2]!="monthly.station.record")) {
  stop(paste("The predictand must be a 'monthly.station.record'",
             "object - Use  station.obj()"))
}
print("Time intervals:")
print(range(x.1$yy))
print(range(x.2$yy))

if (min(x.1$yy) > min(x.2$yy)) {
  XX <- x.1
  x.1 <-  x.2
  x.2 <- XX
  rm(XX)
}
ny.1 <- length(x.1$yy)
ny.2 <- length(x.2$yy)
y.1 <- t(x.1$val)
y.2 <- t(x.2$val)
dim(y.1) <- c(12*ny.1); y.1 <- as.vector(y.1)
dim(y.2) <- c(12*ny.2); y.2 <- as.vector(y.2)
y.1[y.1 <= -999] <- NA
y.2[y.2 <= -999] <- NA
yymm.1 <- sort(rep(x.1$yy,12)) + (rep(1:12,ny.1)-0.5)/12
yymm.2 <- sort(rep(x.2$yy,12)) + (rep(1:12,ny.2)-0.5)/12
i.1 <- is.element(yymm.1,yymm.2)
i.2 <- is.element(yymm.2,yymm.1)
print(range(yymm.1[i.1]))
ovrlp <- data.frame(y=y.1[i.1],x=y.2[i.2])
new.dat <- data.frame(x=y.2[!i.2])

Y.1 <- y.1[i.1]
Y.2 <- y.2[i.2]
ii <- is.finite(Y.1) & is.finite(Y.2)


print(paste("RMSE: ",round(sqrt(sum( (Y.1[ii]-Y.2[ii])^2 ))/sum(i.1),2)))

agree <- lm(y ~ 1 + x, data=ovrlp)
print(summary(agree))
coefs <- agree$coefficients

#print("New series")
y <- c(y.1,coefs[1] + coefs[2]* y.2[!i.2])
#print("Years")
yy <- c(x.1$yy,as.numeric(row.names(table(floor(yymm.2[!i.2])))))
ny <- length(yy)
yymm <- sort(rep(yy,12)) + (rep(1:12,ny) - 0.5)/12

#print("Plot?")
if (plot) {
  plot(yymm.1,y.1,type="s",lwd=3,col="darkblue")
  lines(yymm.2,y.2,type="s",col="steelblue",lty=3,lwd=2)
  lines(yymm,y,type="s",col="wheat",lwd=2)
  grid()
}

#print("Change dimensions")
dim(y) <- c(12,ny)

x<- x.1
x$val <- t(y)
x$yy <- yy
x
}

mixFields <- function(field.1,field.2,mon=NULL,
                       interval=NULL) {

  if (class(field.1)[1] != class(field.2)[1]) {
    print(class(field.1)[1])
    print(class(field.2)[1])
    stop("The objects must have the same class")
  }
  if (!is.null(interval)) {
    if (length(interval)!=2) stop('interval must be a vector of length 2')
    if ( (interval[1] > max(c(field.1$yy,field.2$yy))) |
         (interval[2] < min(c(field.1$yy,field.2$yy))) ) {
      print('Time intervals covered by the fields:')
      print(range(field.1$yy))
      print(range(field.2$yy))
      print(interval)
      stop('Check the interval: interval=c(YYYY.1, YYYY.2)')
    }
  }

  tim.unit1 <- attr(field.1$tim,"unit")
  tim.torg1 <- attr(field.1$tim,"time_origin")
  tim.unit2 <- attr(field.2$tim,"unit")
  tim.torg2 <- attr(field.2$tim,"time_origin")
  if (lower.case(substr(tim.unit1,1,3)) != lower.case(substr(tim.unit2,1,3))) {
    print("mixFields: Time units=")
    print(c(tim.unit1,tim.unit2))
    stop('The time units must match')
  }
  
  dims.1 <- dim(field.1$dat)
  if (length(dims.1)>2) dim(field.1$dat) <- c(dims.1[1],dims.1[2]*dims.1[3])
  dims.2 <- dim(field.2$dat)
  if (length(dims.2)>2) dim(field.2$dat) <- c(dims.2[1],dims.2[2]*dims.2[3])
  
  if (!is.null(mon)) {
    if (is.null(interval)) i.mm <- is.element(field.1$mm,mon) else
      i.mm <- is.element(field.1$mm,mon) & (field.1$yy >= interval[1]) &
              (field.1$yy <= interval[2])
    field.1$dat <- field.1$dat[i.mm,]
    field.1$tim <- field.1$tim[i.mm]
    field.1$yy <- field.1$yy[i.mm]
    field.1$mm <- field.1$mm[i.mm]
    field.1$dd <- field.1$dd[i.mm]
    field.1$id.t <- field.1$id.t[i.mm]
    if (is.null(interval)) i.mm <- is.element(field.2$mm,mon) else
      i.mm <- is.element(field.2$mm,mon) & (field.2$yy >= interval[1]) &
              (field.2$yy <= interval[2])    
    field.2$dat <- field.2$dat[i.mm,]
    field.2$tim <- field.2$tim[i.mm]
    field.2$yy <- field.2$yy[i.mm]
    field.2$mm <- field.2$mm[i.mm]
    field.2$dd <- field.2$dd[i.mm]
    field.2$id.t <- field.2$id.t[i.mm]
  }

  # Match the times:
  
  i1<-is.element(field.1$yy*10000+field.1$mm*100+field.1$dd,
                 field.2$yy*10000+field.2$mm*100+field.2$dd)
  i2<-is.element(field.2$yy*10000+field.2$mm*100+field.2$dd,
                 field.1$yy*10000+field.1$mm*100+field.1$dd)
  
  field.1$dat <- field.1$dat[i1,]
  field.1$tim <- field.1$tim[i1]
  field.1$yy <- field.1$yy[i1]
  field.1$mm <- field.1$mm[i1]
  field.1$dd <- field.1$dd[i1]
  field.1$id.t <- field.1$id.t[i1]
  field.2$dat <- field.2$dat[i2,]
  field.2$tim <- field.2$tim[i2]
  field.2$yy <- field.2$yy[i2]
  field.2$mm <- field.2$mm[i2]
  field.2$dd <- field.2$dd[i2]
  field.2$id.t <- field.2$id.t[i2] 
  
  nt.1 <- length(field.1$tim)
  nt.2 <- length(field.2$tim)
  nx.1 <- length(field.1$lon)
  nx.2 <- length(field.2$lon)
  ny.1 <- length(field.1$lat)
  ny.2 <- length(field.2$lat)

# This should never occur!  
  if (nt.1!=nt.2) {
    print("SOMETHING WENT VERY WRONG!")
    print(table(field.1$id.t))
    print(table(field.1$yy))
    print(table(field.1$mm))
    print(table(field.1$dd))
    print(table(field.2$id.t))
    print(table(field.2$yy))
    print(table(field.2$mm))
    print(table(field.2$dd))
    stop(paste("The fields must have the same time dimension: nt1=",
               nt.1," nt2=", nt.2))
  }

  dim(field.1$dat)<-c(nt.1,ny.1*nx.1)
  dim(field.2$dat)<-c(nt.2,ny.2*nx.2)

#  print(dim(field.1$dat))
#  print(dim(field.2$dat))
  
  field.mxf<-cbind(field.1$dat,field.2$dat)
  tim <- field.1$tim
  attr(tim,"unit") <- tim.unit1
  attr(tim,"time_origin") <- tim.torg1
  yy <- field.1$yy
  mm <- field.1$mm
  dd <- field.1$dd
  lon <- c(field.1$lon,field.2$lon)
  lat <- c(field.1$lat,field.2$lat)
  dim(field.1$id.x) <- c(ny.1*nx.1)
  dim(field.2$id.x) <- c(ny.2*nx.2)
  id.x <- as.vector(c(field.1$id.x,field.2$id.x))
  id.lon <- c(rep(field.1$id.x[1],nx.1),rep(field.2$id.x[1],nx.2))
  id.lat <- c(rep(field.1$id.x[1],ny.1),rep(field.2$id.x[1],ny.2))
  id.t <- paste(field.1$id.t,"+",field.2$id.t,sep="")
  var.name <- c(field.1$v.name,field.2$v.name)
  result  <- list(dat=field.mxf,lon=lon,lat=lat,tim=tim,v.name=var.name,
                  id.t=id.t,id.x=id.x,yy=yy,mm=mm,dd=dd,
                  n.fld=field.1$n.fld+field.2$n.fld,
                  id.lon=id.lon,id.lat=id.lat,,attributes=field.1$attributes)
  class(result) <- c(class(field.1),"mix.fields")
  invisible(result)
}
# Estimates the modulo of two numbers: mod(y1,y2);
# Eg. mod(3,12)=3; mod(20,10)=0; mod(13,6)=1;
# R.E. Benestad, DNMI, 04.01.2001
#
mod <- function(x, y) {
  x1<-trunc( trunc(x/y)*y )
  z<-trunc(x)-x1
  z
  }
# Converts a number to string and formats with requested
# decimal points
# R.E. Benestad

num2str <- function(x,dec=2,f.width=NULL,d.point=".") {
  num2str <- rep(" ",length(x))
  for (i in 1:length(x)) {
    x.r <- as.character(round(x[i],dec))
    dot <- instring(".",x.r)
    if (dot==0) {
      x.r <- paste(x.r,".",sep="")
      dot <- instring(".",x.r)
    }
    zeros <- ""
#    print(x.r)
#    print(c(nchar(x.r),dot,NA,nchar(x.r)-dot+1,dec))
    if (nchar(x.r)-dot+1 <= dec) {
      for (ii in (nchar(x.r)-dot+1):dec) {
        zeros <- paste(zeros,"0",sep="")
      }
    }
    y <- paste(x.r,zeros,sep="")
    num2str[i] <- y
  }
  num2str
}
plotDSobj <- function(result,outdir="output",figs=c(1,2,3,4)) {

if (class(result)!="objDS") {
  stop("The argument is not an 'objDS' object!")
}

months<-c("Jan","Feb","Mar","Apr","May","Jun",
          "Jul","Aug","Sep","Oct","Nov","Dec")

for (mon in months) {
   var.n <- paste("result$",mon,"$pre.gcm",sep="")
   eval(parse(text=paste(var.n,"<-",var.n," - mean(",var.n,",na.rm=TRUE)",sep="")))
   var.n <- paste("result$",mon,"$pre.y",sep="")
   eval(parse(text=paste(var.n,"<-",var.n," - mean(",var.n,",na.rm=TRUE)",sep="")))
   var.n <- paste("result$",mon,"$y.o",sep="")
   eval(parse(text=paste(var.n,"<-",var.n," - mean(",var.n,",na.rm=TRUE)",sep="")))

}
# Plotting and diagnostics:

# Construct a time series for the whole year:

  ds.all.gcm <-cbind(
           result$Jan$pre.gcm,result$Feb$pre.gcm,result$Mar$pre.gcm,
           result$Apr$pre.gcm,result$May$pre.gcm,result$Jun$pre.gcm,
           result$Jul$pre.gcm,result$Aug$pre.gcm,result$Sep$pre.gcm,
           result$Oct$pre.gcm,result$Nov$pre.gcm,result$Dec$pre.gcm)
  yymm.all.gcm <-cbind(
           result$Jan$yy.gcm + (result$Jan$mm.gcm - 0.5)/12,
           result$Feb$yy.gcm + (result$Feb$mm.gcm - 0.5)/12,
           result$Mar$yy.gcm + (result$Mar$mm.gcm - 0.5)/12,
           result$Apr$yy.gcm + (result$Apr$mm.gcm - 0.5)/12,
           result$May$yy.gcm + (result$May$mm.gcm - 0.5)/12,
           result$Jun$yy.gcm + (result$Jun$mm.gcm - 0.5)/12,
           result$Jul$yy.gcm + (result$Jul$mm.gcm - 0.5)/12,
           result$Aug$yy.gcm + (result$Aug$mm.gcm - 0.5)/12,
           result$Sep$yy.gcm + (result$Sep$mm.gcm - 0.5)/12,
           result$Oct$yy.gcm + (result$Oct$mm.gcm - 0.5)/12,
           result$Nov$yy.gcm + (result$Nov$mm.gcm - 0.5)/12,
           result$Dec$yy.gcm + (result$Dec$mm.gcm - 0.5)/12)
  y.gcm <- as.vector(t(ds.all.gcm))
  yymm.gcm <-  as.vector(t(yymm.all.gcm))

  ds.all.cal <-cbind(
           result$Jan$pre.y,result$Feb$pre.y,result$Mar$pre.y,
           result$Apr$pre.y,result$May$pre.y,result$Jun$pre.y,
           result$Jul$pre.y,result$Aug$pre.y,result$Sep$pre.y,
           result$Oct$pre.y,result$Nov$pre.y,result$Dec$pre.y)
  yymm.all.cal <-cbind(
           result$Jan$yy.cal + (result$Jan$mm.cal - 0.5)/12,
           result$Feb$yy.cal + (result$Feb$mm.cal - 0.5)/12,
           result$Mar$yy.cal + (result$Mar$mm.cal - 0.5)/12,
           result$Apr$yy.cal + (result$Apr$mm.cal - 0.5)/12,
           result$May$yy.cal + (result$May$mm.cal - 0.5)/12,
           result$Jun$yy.cal + (result$Jun$mm.cal - 0.5)/12,
           result$Jul$yy.cal + (result$Jul$mm.cal - 0.5)/12,
           result$Aug$yy.cal + (result$Aug$mm.cal - 0.5)/12,
           result$Sep$yy.cal + (result$Sep$mm.cal - 0.5)/12,
           result$Oct$yy.cal + (result$Oct$mm.cal - 0.5)/12,
           result$Nov$yy.cal + (result$Nov$mm.cal - 0.5)/12,
           result$Dec$yy.cal + (result$Dec$mm.cal - 0.5)/12)
  y.cal <- as.vector(t(ds.all.cal))
  yymm.cal <-  as.vector(t(yymm.all.cal))

  obs.all <-cbind(
           result$Jan$y.o,result$Feb$y.o,result$Mar$y.o,
           result$Apr$y.o,result$May$y.o,result$Jun$y.o,
           result$Jul$y.o,result$Aug$y.o,result$Sep$y.o,
           result$Oct$y.o,result$Nov$y.o,result$Dec$y.o)
  yymm.all.obs <-cbind(
           result$Jan$yy.o + (result$Jan$mm.o - 0.5)/12,
           result$Feb$yy.o + (result$Feb$mm.o - 0.5)/12,
           result$Mar$yy.o + (result$Mar$mm.o - 0.5)/12,
           result$Apr$yy.o + (result$Apr$mm.o - 0.5)/12,
           result$May$yy.o + (result$May$mm.o - 0.5)/12,
           result$Jun$yy.o + (result$Jun$mm.o - 0.5)/12,
           result$Jul$yy.o + (result$Jul$mm.o - 0.5)/12,
           result$Aug$yy.o + (result$Aug$mm.o - 0.5)/12,
           result$Sep$yy.o + (result$Sep$mm.o - 0.5)/12,
           result$Oct$yy.o + (result$Oct$mm.o - 0.5)/12,
           result$Nov$yy.o + (result$Nov$mm.o - 0.5)/12,
           result$Dec$yy.o + (result$Dec$mm.o - 0.5)/12)
  y.obs <- as.vector(t(obs.all))
  yymm.obs <-  as.vector(t(yymm.all.obs))

if (!is.null(result$Jan$f.name)) {
  slash<-instring("/",result$Jan$f.name)
  uscr<-instring("_",result$Jan$f.name)
  subtitle <- substr(result$Jan$f.name,slash+1,uscr[2]-1)
} else subtitle <- " "


if (sum(is.element(figs,1))>0) {newFig()
plot(range(yymm.obs,yymm.gcm,na.rm=TRUE),
     range(y.obs,y.gcm,y.cal,na.rm=TRUE),type="n",
     main=paste("Downscaled ",result$Jan$v.name," anomalies at ",result$Jan$location,
                "     (",round(result$Jan$lat.loc,2),"N/",round(result$Jan$lon.loc,2),"E)",sep=""),
     sub=subtitle,xlab="Time",ylab=result$Jan$unit)
grid()
points(yymm.obs+0.025,y.obs,pch=20,cex=1.2,col="grey60")
points(yymm.obs,y.obs,pch=20,cex=1.2,col="black")
lines(yymm.obs,y.obs,lty=3)
lines(yymm.cal+0.0025,y.cal,lty=2,lwd=2,col="grey60")
lines(yymm.cal,y.cal,lty=2,lwd=2,col="grey30")
lines(yymm.gcm+0.0025,y.gcm,lty=1,lwd=2,col="darkblue")
lines(yymm.gcm,y.gcm,lty=1,lwd=1,col="blue")
legend(min(yymm.obs,yymm.gcm,na.rm=TRUE),
       max(y.obs,y.gcm,y.cal,na.rm=TRUE),
       c("Obs","Calibr.","Scenario"),col=c("black","grey30","blue"),
       pch=c(20,26,26),lty=c(3,2,1),lwd=c(1,2,1),bg="wheat",cex=0.8)
if (lower.case(options()$device)=="x11") dev.copy2eps(file=paste(outdir,"/plotDSobj_1.eps",sep=""))
}
# Residuals:

res.rng <- range(result$Jan$step.wise$residual,result$Feb$step.wise$residual,
                 result$Mar$step.wise$residual,result$Apr$step.wise$residual,
                 result$May$step.wise$residual,result$Jun$step.wise$residual,
                 result$Jul$step.wise$residual,result$Aug$step.wise$residual,
                 result$Sep$step.wise$residual,result$Oct$step.wise$residual,
                 result$Nov$step.wise$residual,result$Dec$step.wise$residual, na.rm=TRUE)
if (sum(is.element(figs,2))>0) {newFig()
plot(c(0,length(result$Jan$yy.o)),res.rng,     
     type="n", main=paste("Residuals ",result$Jan$v.name," anomalies at ",result$Jan$location,
                          "     (",round(result$Jan$lat.loc,2),"N/",round(result$Jan$lon.loc,2),"E)",sep=""),
     sub=subtitle,xlab="Time",ylab=result$Jan$unit)
grid()
points(result$Jan$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Feb$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Mar$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Apr$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$May$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Jun$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Jul$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Aug$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Sep$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Oct$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Nov$step.wise$residual,col="black",pch=20,cex=0.5)
points(result$Dec$step.wise$residual,col="black",pch=20,cex=0.5)
lines(result$Jan$step.wise$residual,col="black")
lines(result$Feb$step.wise$residual,col="grey40")
lines(result$Mar$step.wise$residual,col="red")
lines(result$Apr$step.wise$residual,col="darkred")
lines(result$May$step.wise$residual,col="blue")
lines(result$Jun$step.wise$residual,col="darkblue")
lines(result$Jul$step.wise$residual,col="green")
lines(result$Aug$step.wise$residual,col="darkgreen")
lines(result$Sep$step.wise$residual,col="magenta")
lines(result$Oct$step.wise$residual,col="cyan")
lines(result$Nov$step.wise$residual,col="wheat")
lines(result$Dec$step.wise$residual,col="brown")
dev.copy2eps(file=paste(outdir,"/plotDSobj_2.eps",sep="")) 
}

if (sum(is.element(figs,3))>0) { newFig()
brks <- seq(res.rng[1]-1,res.rng[2]+1,length=25)
h.jan<-hist(result$Jan$step.wise$residual,breaks=brks)$density
h.feb<-hist(result$Feb$step.wise$residual,breaks=brks)$density
h.mar<-hist(result$Mar$step.wise$residual,breaks=brks)$density
h.apr<-hist(result$Apr$step.wise$residual,breaks=brks)$density
h.may<-hist(result$May$step.wise$residual,breaks=brks)$density
h.jun<-hist(result$Jun$step.wise$residual,breaks=brks)$density
h.jul<-hist(result$Jul$step.wise$residual,breaks=brks)$density
h.aug<-hist(result$Aug$step.wise$residual,breaks=brks)$density
h.sep<-hist(result$Sep$step.wise$residual,breaks=brks)$density
h.oct<-hist(result$Oct$step.wise$residual,breaks=brks)$density
h.nov<-hist(result$Nov$step.wise$residual,breaks=brks)$density
h.dec<-hist(result$Dec$step.wise$residual,breaks=brks)$density
brks <- hist(result$Dec$step.wise$residual,breaks=brks)$mids
plot(range(brks),range(c(h.jan,h.feb,h.mar,h.apr,h.may,h.jun,
     h.jul,h.aug,h.sep,h.oct,h.nov,h.dec)),type="n",
     main=paste("Residuals ",result$Jan$v.name," anomalies at ",result$Jan$location,
                          "     (",round(result$Jan$lat.loc,2),"N/",round(result$Jan$lon.loc,2),"E)",sep=""),
     sub=subtitle,ylab="Density",xlab=result$Jan$unit)
grid()
lines(brks,h.jan,col="black")
lines(brks,h.feb,col="grey40")
lines(brks,h.mar,col="red")
lines(brks,h.apr,col="darkred")
lines(brks,h.may,col="blue")
lines(brks,h.jun,col="darkblue")
lines(brks,h.jul,col="green")
lines(brks,h.aug,col="darkgreen")
lines(brks,h.sep,col="magenta")
lines(brks,h.oct,col="cyan")
lines(brks,h.nov,col="wheat")
lines(brks,h.dec,col="brown")
if (lower.case(options()$device)=="x11") dev.copy2eps(file=paste(outdir,"/plotDSobj_3.eps",sep="")) 
}

rates <- c(result$Jan$rate.ds,result$Feb$rate.ds,result$Mar$rate.ds,
           result$Apr$rate.ds,result$May$rate.ds,result$Jun$rate.ds,
           result$Jul$rate.ds,result$Aug$rate.ds,result$Sep$rate.ds,
           result$Oct$rate.ds,result$Nov$rate.ds,result$Dec$rate.ds)
err <- c(result$Jan$rate.err,result$Feb$rate.err,result$Mar$rate.err,
           result$Apr$rate.err,result$May$rate.err,result$Jun$rate.err,
           result$Jul$rate.err,result$Aug$rate.err,result$Sep$rate.err,
           result$Oct$rate.err,result$Nov$rate.err,result$Dec$rate.err)
r2 <- c(result$Jan$fit.r2,result$Feb$fit.r2,result$Mar$fit.r2,
           result$Apr$fit.r2,result$May$fit.r2,result$Jun$fit.r2,
           result$Jul$fit.r2,result$Aug$fit.r2,result$Sep$fit.r2,
           result$Oct$fit.r2,result$Nov$fit.r2,result$Dec$fit.r2)
p.val <- as.numeric(c(result$Jan$gcm.trnd.p,result$Feb$gcm.trnd.p,result$Mar$gcm.trnd.p,
           result$Apr$gcm.trnd.p,result$May$gcm.trnd.p,result$Jun$gcm.trnd.p,
           result$Jul$gcm.trnd.p,result$Aug$gcm.trnd.p,result$Sep$gcm.trnd.p,
           result$Oct$gcm.trnd.p,result$Nov$gcm.trnd.p,result$Dec$gcm.trnd.p))

if (sum(is.element(figs,4))>0) {newFig()
plot(c(0,13),range(c(rates+err,rates-err),na.rm=TRUE),type="n",
     main=paste("Linear trend rates ",result$Jan$v.name," derived ",result$Jan$location,
                          "     (",round(result$Jan$lat.loc,2),"N/",round(result$Jan$lon.loc,2),"E)",sep=""),
     sub=subtitle,ylab=paste(result$Jan$unit,"/ decade"),xlab="Month")
grid()

polygon(c(1:12,reverse(1:12)),c(rates+err,reverse(rates-err)),
        col="wheat",border="grey",lwd=2)
lines(0:12+0.5,c(r2[1],r2)/100*max(rates-err,na.rm=TRUE)+min(rates-err,na.rm=TRUE),
      type="S",col="steelblue")
lines(rates,lwd=2)
points((1:12)[p.val < 5],rates[p.val < 5],pch=20,cex=1.5)
points((1:12)[p.val >= 5],rates[p.val >= 5],pch=21,cex=1.5)
text((1:12)-0.33,rates-0.01*diff(range(c(rates+err,rates-err),na.rm=TRUE)),rates,
       cex=0.8,col="grey45")

for (i in 0:10) {
  lines(c(11.8,12),rep(i/10*max(rates-err,na.rm=TRUE)+min(rates-err,na.rm=TRUE),2),col="steelblue")
  lines(c(0,11),rep(i/10*max(rates-err,na.rm=TRUE)+min(rates-err,na.rm=TRUE),2),lty=3,col="steelblue")
  text(11.5,i/10*max(rates-err,na.rm=TRUE)+min(rates-err,na.rm=TRUE),paste(i*10,'%',sep=""),
        cex=0.8,col="steelblue")
  }
mtext("R-squared (%) from calibration regression",side=4,col="steelblue",cex=0.80)
points(1,max(rates+err),pch=20); text(3,max(rates+err),"5% sign.level")
points(7,max(rates+err),pch=21); text(8,max(rates+err),"not sign.")

if (lower.case(options()$device)=="x11") dev.copy2eps(file=paste(outdir,"/plotDSobj_4.eps",sep="")) 
}
}


objDS <- function(field.obs,field.gcm,station,plot=TRUE,positive=NULL,
                  mon=NULL,direc="output/",cal.id=NULL,
                  ldetrnd=TRUE,i.eofs=seq(1,8,by=1),ex.tag="",
                  method="lm",leps=FALSE,param="t2m",
                  plot.res=FALSE,plot.rate=FALSE,xtr.args="",
                  swsm="step",predm="predict",lsave=TRUE,rmac=TRUE,
                  silent=FALSE) {

  cmon<-c("Jan","Feb","Mar","Apr","May","Jun",
          "Jul","Aug","Sep","Oct","Nov","Dec")
  
  dims <- dim(field.obs$dat); ny <- dims[2]; nx <- dims[3]
#  print(dims)
  wy <- 2*pi*seq(0,ny-1,by=1)/(ny-1)
  x.mod<-matrix(rep(NA,ny*nx),ny,nx)
  x.mod[,1]<-cos(wy);   x.mod[,2]<-sin(wy)
  x.mod[,3]<-cos(2*wy); x.mod[,4]<-sin(2*wy)
  x.mod[,5]<-cos(3*wy); x.mod[,6]<-sin(3*wy)
  x.mod[,7]<-cos(4*wy); x.mod[,8]<-sin(4*wy)
  wx <- 2*pi*seq(0,nx-1,by=1)/(nx-1)
  y.mod<-matrix(rep(NA,ny*nx),ny,nx)
  y.mod[1,]<-cos(wx);   y.mod[2,]<-sin(wx)
  y.mod[3,]<-cos(2*wx); y.mod[4,]<-sin(2*wx)
  y.mod[5,]<-cos(3*wx); y.mod[6,]<-sin(3*wx)
  y.mod[7,]<-cos(4*wx); y.mod[8,]<-sin(4*wx)
  if (is.null(mon)) mon  <-  1:12
  
  result <- list(station=station)
  if (is.null(positive) &
      sum(is.element(c("t2m","tem"),lower.case(substr(field.obs$v.name,1,3))))> 0) {
    positive <- TRUE
  }
  for (imon in mon) {
    print(imon)
    cormap <- corField(field.obs,station,mon=imon)
    if (lower.case(options()$device)=="x11") dev.copy2eps(file=paste(direc,"/cormap_",cmon[imon],".eps",sep=""))

    # Find optimal longitudes & latitudes:
    
    latx <- 0.5*(field.obs$lat[2:ny]+field.obs$lat[1:(ny-1)])
    lonx <- 0.5*(field.obs$lon[2:nx]+field.obs$lon[1:(nx-1)])
    iy <- min( (1:ny)[station$lat <= field.obs$lat], na.rm=TRUE)
    ix <- min( (1:nx)[station$lon <= field.obs$lon], na.rm=TRUE)
    yprof <- as.vector(cormap$map[ix,]); yprof[is.na(yprof)] <- 0
    xprof <- as.vector(cormap$map[,iy]); xprof[is.na(xprof)] <- 0
    largescale <- data.frame(y=yprof, X=x.mod)      
    y.fit<-lm(y ~ X.1 + X.2 + X.3 + X.4 + X.5 + X.6 + X.7 + X.8,data=largescale)
    largescale <- data.frame(y=xprof, X=t(y.mod))
    lsX <- data.frame(X=x.mod);  lsY <- data.frame(X=t(y.mod));
    x.fit<-lm(y ~ X.1 + X.2 + X.3 + X.4 + X.5 + X.6 + X.7 + X.8,data=largescale)
    yhat <- predict(y.fit,newdata=lsX); xhat <- predict(x.fit,newdata=lsY);
    yzero <- yhat[2:ny]*yhat[1:(ny-1)]; xzero <- xhat[2:nx]*xhat[1:(nx-1)]
    lonx <- lonx[xzero < 0]; latx <- latx[yzero < 0]
    x.rng <- c(max(c(min(field.obs$lon),max(lonx[lonx < station$lon])), na.rm=TRUE),
               min(c(max(field.obs$lon),min(lonx[lonx > station$lon])), na.rm=TRUE))
    y.rng <- c(max(c(min(field.obs$lat),max(latx[latx < station$lat])), na.rm=TRUE),
               min(c(max(field.obs$lat),min(latx[latx > station$lat])), na.rm=TRUE))
    if (plot) {
      plot(range(c(field.obs$lat,field.obs$lon)),range(cormap$map,na.rm=TRUE),type="n",
           main=paste("Finding optimal domain for",cmon[imon]),xlab="deg N & deg E",
           sub=paste(round(field.obs$lon[ix],2),"E/",round(field.obs$lat[iy],2),"N",sep=""))
      grid()
      lines(range(c(field.obs$lat,field.obs$lon)),rep(0,2),lty=3)
      points(field.obs$lat,yprof)
      lines(field.obs$lat,as.numeric(yhat),lwd=2);
      lines(rep(y.rng[1],2),range(cormap$map,na.rm=TRUE),lty=2)
      lines(rep(y.rng[2],2),range(cormap$map,na.rm=TRUE),lty=2)

      points(field.obs$lon,xprof,col="red",pch=20)
      lines(field.obs$lon,as.numeric(xhat),col="red",lwd=2)
      lines(rep(x.rng[1],2),range(cormap$map,na.rm=TRUE),lty=2,col="red")
      lines(rep(x.rng[2],2),range(cormap$map,na.rm=TRUE),lty=2,col="red")
      if (lower.case(options()$device)=="x11") dev.copy2eps(file=paste(direc,"/objDS_",cmon[imon],"_1.eps",sep=""))
    }
    print("catFields:")
#    print(">>> Check REB 11.02.2004!")
#    print(x.rng)
#    print(c(sum(!is.finite(field.obs$dat)),sum(!is.finite(field.gcm$dat))))
#    print(summary(field.obs$lon)); print(summary(field.obs$lat))
#    print(summary(field.gcm$lon)); print(summary(field.gcm$lat))
    field.2 <- catFields(field.obs,field.gcm,lon=x.rng,lat=y.rng,mon=imon)
    print("EOF:")
    eof <- EOF(field.2)
    print("DS:")
    ds <- DS(preds=eof,dat=station,direc=direc,cal.id=cal.id,
                  ldetrnd=ldetrnd,i.eofs=i.eofs,ex.tag=ex.tag,
                  method=method,plot=FALSE,leps=leps,param=param,
                  plot.res=plot.res,plot.rate=plot.rate,xtr.args=xtr.args,
                  swsm=swsm,predm=predm,lsave=lsave,rmac=rmac,
                  silent=silent)
    print(paste("result$",cmon[imon]," <- ds",sep=""))
    eval(parse(text=paste("result$",cmon[imon]," <- ds",sep="")))
  }

  class(result) <- "objDS"
  if (plot) plotDSobj(result)

  invisible(result)
}
# Optimal Interpolation procedure:
#
# After Reynolds and Smith (1994), J. Clim., June, pp 929--948
#
# R.E. Benestad

delta <- function(i,j) {
  if (i==j) delta <- 1 else delta <- 0
}

optint <- function(lon,lat,obs,
                   lon.grd,lat.grd,fguess,
                   eps,lambda=50,M=NULL,piipij=NULL,w=NULL,
                   tim=NULL,date=NULL) {

#  print('OPTINT: Optical Interpolation')
  a <- 6370  # Earth's radius
  np <- length(obs)
  nx <- length(lon.grd)
  ny <- length(lat.grd)
  dx <- 0.5*mean(diff(lon.grd))
  dy <- 0.5*mean(diff(lat.grd))
  obs.grd <- matrix(rep(NA,nx*ny),nx,ny)
  for (ip in 1:np) {
    ix <- (lon.grd >= lon[ip] - dx) &  (lon.grd <= lon[ip] + dx)
    iy <- (lat.grd >= lat[ip] - dy) &  (lat.grd <= lat[ip] + dy)
#    print(c(ip,lon.grd[ix],lat.grd[iy],obs[ip]))
    obs.grd[ix,iy] <- obs[ip]
  }

# Transform the coordinates from degrees to radians:
 
  if (is.null(piipij)) {  
    lon <- lon * pi/180
    lat <- lat * pi/180
    lons <- lon.grd
    lats <- lat.grd
    lon.grd <- lon.grd * pi/180
    lat.grd <- lat.grd * pi/180
#    print(paste('optint: Estimate the first-guess correlation-error',
#                '<pi pi>, dims=',ny*nx,'x',ny*nx,'=',ny^2*nx^2))
    piipij <- matrix(rep(0,ny^2*nx^2),ny*nx,ny*nx)
    for (j0 in 1:ny) {
      for (i0 in 1:nx) {
        ii <- (j0-1)*nx + i0
        for (j in 1:ny) {
          for (i in 1:nx) {
            jj <- (j-1)*nx + i
            piipij[ii,jj] <- exp( -( a^2*
              (lon.grd[i]*cos(lat.grd[j]) - lon.grd[i0]*cos(lat.grd[j0]))^2 +
              (lat.grd[j] - lat.grd[j0])^2 )/lambda^2 )
          }
        }
      }
    }
  }

#  x11()
#  image(piipij,main="piipij")
  
  if (is.null(M)) {
    dim(eps) <- c(nx*ny,1)
    M <- matrix(rep(0,ny^2*nx^2),ny*nx,ny*nx)
    for (j in 1:(ny*nx)) {
      for (i in 1:(ny*nx)) {
        M[i,j] <- piipij[i,j] + eps[i]*eps[j]*delta(i,j)
      }
    }
  }

#  x11()  
#  image(M,main="M")
#  x11()
#  image(M-piipij,main="M-piipij")
  
# Incomplete lower and upper triangular matrix decomposition:
# Solves the equation \sum_j M_ij w_ik = <pi_j pi_k> 

#  print("Solve 'sum_j{M_ij w_ik} = <pi_j pi_k>'")
  if (is.null(w)) {
    w <- matrix(rep(0,ny^2*nx^2),ny*nx,ny*nx)
    for (i in 1:ny*nx) {
      w[i,] <- qr.solve(M,piipij[,i]);
    }
  }
  
# Estimating the analysis increments: eq 1

  dim(obs.grd) <- c(nx*ny,1)
  dim(fguess) <- c(nx*ny,1)
  good <- !is.na(obs.grd)
  
  q <-  obs.grd - fguess
  r <- rep(0,ny*nx)
#  print("Solve 'r = w_i q'")

  for (i in 1:(ny*nx)) {
    r[i] <- w[i,good]*q[good]
  }
#  print(summary(as.vector(w)))
#  print(summary(q))
#  print(summary(r))
  dim(r) <- c(nx,ny)
  dim(fguess) <- c(nx,ny)

  results <- list(lon=lons,lat=lats,map=fguess + r,
                  tim=tim,date=date,M=M,piipij=piipij,w=w)
  class(results) <- "map"
#  attr(results) <- attr(obs)
  attr(results,"descr") <- "Optimal interpolation"
  invisible(results)
}
# R.E. Benestad, met.no, Oslo, Norway 27.08.2003
# rasmus.benestad@met.no
#------------------------------------------------------------------------

patternIndex <- function(map,field,anomaly=TRUE) {

  library(akima)
  
  size.map <- dim(map$map)
  size.field <- dim(field$dat)
  if ( (size.map[2] != size.field[2]) |
       (size.map[1] != size.field[3]) ) {
    print("Interpolate: different grids!")
    lat.x<-rep(field$lat,length(field$lon))
    lon.x<-sort(rep(field$lon,length(field$lat)))
    Z.in<-t(as.matrix(map$map))
    Z.out<-interp(lat.x,lon.x,Z.in,map$lat,map$lon)
  } else Z.out <- t(as.matrix(map$map))
  nx <- size.field[3]; ny <- size.field[2]; nt <- size.field[1]
  
  ind <- rep(NA,nt)
  X <- as.vector(Z.out)
  Y <- field$dat
  dim(Y) <- c(nt,ny*nx)
  
  if (anomaly) {
    if ( (field$mm[2]-field$mm[1]>=1) & (field$dd[2]==field$dd[1]) ) {
      print("Months")
      for (l in as.numeric(rownames(table(field$mm)))) {
        it <- is.element(field$mm,l)
        clim <- rep(colMeans(Y[it,],na.rm=TRUE),sum(it))
#        dim(clim) <- c(sum(it),ny*nx)
        dim(clim) <- c(ny*nx,sum(it))
        clim <- t(clim)
#        plot(clim[1,],type="l",lwd=2)
#        print(dim(clim))
#        print(dim(Y[it,]))
#        points(Y[it,][1,],pch=20,col="red")
        Y[it,] <- Y[it,] - clim
#        points(Y[it,][1,],pch=21,col="blue")    # Testing OK...
      }
    } else {
      print("Other time units")
      ac.mod<-matrix(rep(NA,nt*6),nt,6)
      if (substr(lower.case(attributes(field$tim)$units),1,3)=="day") jtime <- field$tim
      if (substr(lower.case(attributes(field$tim)$units),1,4)=="hour") jtime <- field$tim/24
      ac.mod[,1]<-cos(2*pi*jtime/365.25); ac.mod[,2]<-sin(2*pi*jtime/365.25)
      ac.mod[,3]<-cos(4*pi*jtime/365.25); ac.mod[,4]<-sin(4*pi*jtime/365.25)
      ac.mod[,5]<-cos(6*pi*jtime/365.25); ac.mod[,6]<-sin(6*pi*jtime/365.25)               
      for (ip in seq(1,ny*nx,by=1)) {
        if (sum(is.finite(Y[,ip])) > 0) {
          ac.fit<-lm(Y[,ip] ~ ac.mod)
          Y[,ip]<-ac.fit$residual
        } else Y[,ip]<- NA
      }
    }
  }

  for (i in 1:nt) {
    good <- is.finite(X) & is.finite(Y[i,])
    ind[i] <- cor(as.vector(Y[i,good]),X[good])
  }

  newFig()
  dx <- (max(field$yy)-min(field$yy))/200
  plot(range(field$yy + field$mm/12 + field$dd/365.25),
       c(-1,1),type="n",lwd=3,main="Pattern Index",
       sub=field$v.name,xlab="Time",ylab="Pattern Index")
  for (ii in 1:sum(iext)) text(field$yy[iext][ii],0.975*ind[iext][ii]/abs(ind[iext][ii]),
                               as.character(field$yy[iext][ii]),cex=0.6,col="grey20",srt=90)

  lines(field$yy + field$mm/12 + field$dd/365.25+dx,ind+0.01,col="grey80",lwd=2)
  lines(field$yy + field$mm/12 + field$dd/365.25,ind,lwd=2)
  lines(c(min(field$yy),max(field$yy)+1),rep(2*sd(ind,na.rm=TRUE),2),
        lty=2,col="grey50",lwd=2)
  lines(c(min(field$yy),max(field$yy)+1),rep(-2*sd(ind,na.rm=TRUE),2),
        lty=2,col="grey50",lwd=2)
  grid()
  iext <- abs(ind) > 2* sd(ind,na.rm=TRUE)
  
  pInd <- list(index=ind,yy=field$yy,mm=field$mm,dd=field$dd,map=map)
  invisible(pInd)
}
# R.E. Benestad, met.no, Oslo, Norway 09.10.2002
# rasmus.benestad@met.no
#------------------------------------------------------------------------

newFig <- function() {
   dev <- paste(options()$device,"()",sep="")
#   print(paste("newFig: options()$device=",dev))
   if ((dev!="none()") & (dev!="bitmap()")) eval(parse(text=dev))
   if (dev=="bitmap()") bitmap(file="newFig.jpg",type="jpeg")
 }


plotDS <- function(ds.obj,leps=FALSE,plot.map=TRUE, plot.res=FALSE,
                   plot.rate=FALSE,add=FALSE,col="darkred",lwd=2,lty=1,
                   direc="output/",main=NULL,sub=NULL,xlab=NULL,ylab=NULL) {

if (class(ds.obj)!="ds") stop("Need a 'ds' object!")
attach(ds.obj)

# Plotting: -----------------------------------------------
  
pred.descr <- paste("Empirical Downscaling (",id.1,"[")

lons <- lon.loc
lats <- lat.loc
if (!is.finite(lons)) lons <- mean(ds.obj$lon,na.rm=TRUE)
if (!is.finite(lats)) lats <- mean(ds.obj$lat,na.rm=TRUE)

for (i in 1:n.fld) {
  eval(parse(text=paste("x.srt<-order(ds.obj$lon.",i,")",sep="")))
  eval(parse(text=paste("y.srt<-order(ds.obj$lat.",i,")",sep="")))
  eval(parse(text=paste("ds.obj$lons<-ds.obj$lon.",i,"[x.srt]",sep="")))
  eval(parse(text=paste("ds.obj$lats<-ds.obj$lat.",i,"[y.srt]",sep="")))
  ds.obj$X.1 <- ds.obj$X.1[y.srt,x.srt]
  eval(parse(text=paste("ds.obj$X.",i,"<-ds.obj$X.",i,"[y.srt,x.srt]",sep="")))
  lons <- eval(parse(text=paste("c(lons,lon.",i,")",sep="")))
  lats <- eval(parse(text=paste("c(lats,lat.",i,")",sep="")))
}

if (is.null(main)) main <-  paste(pred.descr,region,"] ->",v.name,")")       
if (is.null(sub)) sub <- paste("Calibration: ",month," ",v.name," at ",ds.obj$location,
                  " using ",id.1,": R2=",fit.r2,
                  "%, p-value=",fit.p,"%.",sep="")
if (is.null(xlab)) xlab <- "Time"
if (is.null(ylab)) ylab <- paste(v.name,"(",unit,")")

#print(paste("subtitle:",subtitle))

y.lim.tr <- range(c(y.o,pre.y,pre.gcm),na.rm=TRUE)
yymm.o<-yy.o + (mm.o-0.5)/12 + (dd.o-0.5)/365.25
yymm.gcm<-yy.gcm + (mm.gcm-0.5)/12 + (dd.gcm-0.5)/365.25

#if (!leps) par(ask=TRUE)
if ((!add) & (plot.map)) {
  if (leps) {
    figname<- paste("predictor_",v.name,"_",location,"_",region,"_",
                  month,ex.tag,".eps",sep="")
    postscript(file = figname,onefile=TRUE,horizontal=FALSE,paper="a4")
  } else eval(parse(text=paste(lower.case(options()$device),"()",sep="")))
  par(ps=16,cex.sub=0.7,cex.main=0.9)
  plot(c(floor(min(lons,na.rm=TRUE)),ceiling(max(lons,na.rm=TRUE))),
       c(floor(min(lats,na.rm=TRUE)),ceiling(max(lats,na.rm=TRUE))),type="n",
       main=main,sub=sub,xlab=xlab,ylab=ylab)

  col.tab=c("darkblue","darkred","darkgreen","brown")
  t.rng <- paste(range(ds.obj$yy.cal)[1],"-",range(ds.obj$yy.cal)[2])
  ds.map <- list(tim=NULL,date=NULL,n.maps=NULL)
  ds.map$tim<-month; ds.map$date<-t.rng; ds.map$n.maps=n.fld
  for (i in 1:n.fld) {
    eval(parse(text=paste("lines(c(min(ds.obj$lon.",i,"),max(ds.obj$lon.",i,")),",
                              "c(min(ds.obj$lat.",i,"),min(ds.obj$lat.",i,")),",
                              "col=col.tab[i],lty=2)",sep="")))
    eval(parse(text=paste("lines(c(min(ds.obj$lon.",i,"),max(ds.obj$lon.",i,")),",
                              "c(max(ds.obj$lat.",i,"),max(ds.obj$lat.",i,")),",
                              "col=col.tab[i],lty=2)",sep="")))
    eval(parse(text=paste("lines(c(min(ds.obj$lon.",i,"),min(ds.obj$lon.",i,")),",
                              "c(min(ds.obj$lat.",i,"),max(ds.obj$lat.",i,")),",
                              "col=col.tab[i],lty=2)",sep="")))
    eval(parse(text=paste("lines(c(max(ds.obj$lon.",i,"),max(ds.obj$lon.",i,")),",
                              "c(min(ds.obj$lat.",i,"),max(ds.obj$lat.",i,")),",
                              "col=col.tab[i],lty=2)",sep="")))
    eval(parse(text=paste("contour(ds.obj$lon.",i,",ds.obj$lat.",i,",t(ds.obj$X.",i,
               "),nlevels=7,add=TRUE,lwd=2,col=col.tab[i])",sep="")))
    eval(parse(text=paste("ds.map$lon.",i,"<-ds.obj$lon.",i,sep="")))
    eval(parse(text=paste("ds.map$lat.",i,"<-ds.obj$lat.",i,sep="")))
    eval(parse(text=paste("ds.map$map.",i,"<-t(ds.obj$X.",i,")",sep="")))
  }
  class(ds.map) <- "map"; attr(ds.map,"descr") <- "ds: large-scale pattern"

#  print("plotDS: HERE")

  if (!is.null(lon.loc) & !is.null(lat.loc))
    points(lon.loc,lat.loc,pch=20,col="wheat",cex=1.5)
    points(lon.loc,lat.loc,pch=20,col="black",cex=0.9)
  addland()
  grid()


  if (n.fld > 1) {
    legend(min(c(lons,lon.loc,na.rm=TRUE)),
         max(c(lats,lat.loc,na.rm=TRUE)),
         c(pred.name[1:n.fld]),
         col=c(col.tab[1:n.fld]),
         lwd=2,lty=1,merge=TRUE,bg="grey95")
  }

  if (leps) { 
    dev.off()
    if (!file.exists(direc)){
      print(paste("The directory",direc,"does not exists.. Creates it.."))
      dir.create(direc)
    } 
    file.copy(figname,direc)
    file.remove(figname)
  } 
}

if (!add) {
  if (leps) {
    figname<- paste("scen_",v.name,"_",location,"_",region,"_",
                    month,ex.tag,".eps",sep="")
    postscript(file = figname,onefile=TRUE,horizontal=FALSE,paper="a4")
  } else newFig()
  par(ps=16,cex.sub=0.7,cex.main=0.9)

  plot(c(min(yymm.o[1],yymm.gcm[1]),yymm.gcm[length(yymm.gcm)]),
       y.lim.tr,type="n",
       main=main,sub=sub,xlab=xlab,ylab=ylab)
  grid()

}

lines(yymm.o,y.o,col="darkblue",lwd=3);
lines(yymm.o,pre.y,col="grey40",lty=2,lwd=2);
lines(yymm.gcm,pre.gcm,col=col,lwd=lwd,lty=lty);
lines(yymm.gcm,pre.fit, col = "red",lwd=1,lty=2) 
lines(yymm.gcm,pre.p.fit, col = "red",lwd=1,lty=2)
points(yymm.o,y.o,col="darkblue",pch=20);
points(yymm.o,pre.y,col="grey40",pch=21);
points(yymm.gcm,pre.gcm,col="darkred",pch=21);

if (!add) legend(quantile(c(yymm.o,yymm.gcm),0.01),
                 max(c(y.o,pre.y,pre.gcm)),
                 c("Obs.","Fit","GCM","Trends"),cex=0.75,
                 col=c("darkblue","grey40","darkred","red"),
                 lwd=c(3,2,2,1),lty=c(1,2,1,2),pch=c(20,21,21,26,26),
                 merge=TRUE,bg="grey95")

text(quantile(c(yymm.o,yymm.gcm),0.01),
     min(c(y.o,pre.y,pre.gcm)),pos=4,cex=0.6,
     paste(month,": Trend fit: P-value=",gcm.trnd.p,"%; ",
           "Projected trend= ",rate.ds,"+-",rate.err," ",
           unit,"/decade",sep=""))

if (leps) { 
  dev.off()
  file.copy(figname,direc)
  file.remove(figname)
}

# Plot the rate of change:

if ((plot.rate) & !(add)) {
if (leps) { 
  figname<- paste("tendency_",v.name,"_",location,"_",region,"_",
                month,ex.tag,".eps",sep="")
  postscript(file = figname,onefile=TRUE,horizontal=FALSE,paper="a4")
} else newFig()
par(ps=16,cex.sub=0.7,cex.main=0.9)

plot(c(min(yymm.gcm),max(yymm.gcm)),y.lim.tr,type="n",
     main=main,sub=sub,xlab=xlab,ylab=paste("rate of change in",ylab))
grid()
lines(yymm.gcm,tr.est.p.fit, col = "blue",lwd=3)
lines(c(min(yymm.gcm),max(yymm.gcm)),c(rate.ds,rate.ds),col = "red",lwd=2)

legend(min(yymm.gcm),-1.5,c("Polinomial fit","Linear fit"),
       lwd=c(3,2),col=c("blue","red"),bg="grey95")
if (leps) { 
  dev.off()
  file.copy(figname,direc)
  file.remove(figname)
}
}

# Plot the residuals:

if ((plot.res)  & !(add)) {
if (leps) { 
  figname<- paste("residual_",v.name,"_",location,"_",region,"_",
                month,ex.tag,".eps",sep="")
  postscript(file = figname,onefile=TRUE,horizontal=FALSE,paper="a4")
} else newFig()
par(ps=16,cex.sub=0.9,cex.main=0.7)
plot(yymm.o,step.wise$residual,type="l",lwd=3,
     main=paste("Residual",
                region,"] ->",v.name,")"),sub=sub,xlab=xlab,ylab=ylab)
lines(yymm.o,pre.y-mean(pre.y,na.rm=TRUE),col="grey",lty=3); 
grid()


if (leps) { 
  dev.off()
  file.copy(figname,direc)
  file.remove(figname)
  figname<- paste("qq-residual_",v.name,"_",location,"_",region,"_",
                month,".eps",ex.tag,sep="")
  postscript(file = figname,onefile=TRUE,horizontal=FALSE,paper="a4")
} else newFig()
par(ps=16,cex.sub=0.8,cex.main=0.85)
qqnorm((step.wise$residual-mean(step.wise$residual,na.rm=TRUE))/
       sd(step.wise$residual,na.rm=TRUE))
lines(c(-5,5),c(-5,5),col="grey",lty=2)
grid()

if (leps) { 
  dev.off()
  file.copy(figname,direc)
  file.remove(figname)
}
}

invisible(ds.map)
}
# Plots EOF products
# Monthly mean values.
#
# Reference: R.E. Benestad et al. (2002),
#            Empirically downscaled temperature scenarios for Svalbard,
#            submitted to Atm. Sci. Lett.
#
#            R.E. Benestad (2001),
#            A comparison between two empirical downscaling strategies,
#            Int. J. Climatology, 1645-1668, vol. 21, DOI 10.1002/joc.703
#
# R.E. Benestad, met.no, Oslo, Norway 10.05.2002
# rasmus.benestad@met.no
#

#------------------------------------------------------------------------

plotEOF<-function(x,i.eof=1,nlevs=5,
                   col=c("red","blue","darkgreen","steelblue"),
                   main=NULL,sub=NULL) {

if (class(x)[1]!= "eof") stop ("The argument must be an 'eof' object") 
ok.eps <- (lower.case(options()$device)=="x11") | (lower.case(options()$device)=="windows")
attach(x)

dims <- dim(x$EOF) 
if (length(dims)==3) dim(EOF) <- c(dims[1],dims[2]*dims[3])

title.1 <- paste("EOF pattern #",i.eof,"(",class(x)[2],")",sep="")
title.2 <- "The fraction of variance accounted by the EOFs"
title.3 <- paste("Principal component (",class(x)[2],")",sep="")

vnames <- x$v.name[1]
if (!is.null(main)) {
   title.1  <-  main; title.2 <- main; title.3 <- main}       
if (is.null(sub)) sub <- paste(vnames," (",c.mon,")")

for (i in 2:length(vnames)) vnames <- paste(vnames,"+",x$v.name[i])
i.last <- 0
id <- row.names(table(id.x))
ordr <- rep(NA,length(id))
for (i in 1:length(id)) {
  ordr[i] <- min((1:length(id.x))[is.element(id.x,id[i])])
}
#print(ordr)
id<-id[order(ordr)]
#print(id)

#par(ask=TRUE)
newFig()
plot(c(floor(min(x$lon)),ceiling(max(x$lon))),
     c(floor(min(x$lat)),ceiling(max(x$lat))),
     type="n",main=title.1,sub=sub,
     xlab="Longitude",ylab="Latitude")
if (range(x$lon)[2]-range(x$lon)[1] > 360) {
  xy.cont <- COn0E65N(lon.cont, lat.cont)
  addland(lon=xy.cont$x,lat=xy.cont$y)
} else addland()
grid()
col.tab <- col[1:length(id)]
neofs <- length(x$var)
i.last <- 0
print(paste("plotEOF: n.fld=",n.fld))
for (i in 1:n.fld) {
  i.fld <- seq(i.last+1,i.last+x$size[2,i]*x$size[3,i],by=1)
  i.last <- max(i.fld)
  #print(c(i,NA,dim(x$EOF),NA,size[,i],NA,range(i.fld)))
  EOF.1 <- x$EOF[,i.fld]
  dim(EOF.1)<-c(dim(x$EOF)[1],size[2,i],size[3,i])
  eof.patt<-t(EOF.1[i.eof,,])
  i.lon <- x$id.lon == id[i]
  i.lat <- x$id.lat == id[i]
  lon.x <- x$lon[i.lon]
  lat.x <- x$lat[i.lat]
  #print(c(size[,i],NA,length(lon.x),length(lat.x),NA,dim(eof.patt),id[i]))
  contour(lon.x,lat.x,eof.patt,
          nlevels=nlevs,add=TRUE,lwd=2,col=col.tab[i])
}

if (n.fld>1) legend(min(x$lon),max(x$lat),id,
             col=c(col.tab),lty=1,
             lwd=2,merge=TRUE, bg='gray95')

if (ok.eps) dev.copy2eps(file=paste("plotEOF_1.eps",sep=""))

newFig()
plot(100*(W+dW)^2/tot.var,main=title.2,type="n",
     ylab="Variance (%)",xlab="EOF order",sub=sub)
lines(var.eof,lty=3)
for (i in 1:length(var.eof)) {
  lines(rep(i,2),100*c((W[i]+dW[i])^2/tot.var,(W[i]-dW[i])^2/tot.var),
        lty=2,col="darkgrey")
  lines(c(i-0.25,i+0.25),100*rep((W[i]+dW[i])^2/tot.var,2),
        lwd=2,col="darkgrey")
  lines(c(i-0.25,i+0.25),100*rep((W[i]-dW[i])^2/tot.var,2),
        lwd=2,col="darkgrey")
}
points(var.eof)
points(var.eof,pch=20,cex=0.8,col="darkgrey")
grid()
if (ok.eps) dev.copy2eps(file=paste("plotEOF_2.eps",sep=""))

newFig()
yymm<-x$yy + (x$mm-0.5)/12 + (x$dd-0.5)/365.25
#print(c(length(yy),length(mm),length(dd),length(yymm),length(PC[,i.eof])))
plot(yymm,PC[,i.eof],pch=20,cex=0.7,
     main=title.3,,col="grey70",sub=sub)

lines(yymm[id.t==id.t[1]],PC[id.t==id.t[1],i.eof],col="red",lty=2,lwd=2)
if (sum(id.t!=id.t[1])>0) lines(yymm[id.t!=id.t[1]],
          PC[id.t!=id.t[1],i.eof],col="blue",lty=2,lwd=2)
grid()
if (ok.eps) dev.copy2eps(file=paste("plotEOF_3.eps",sep=""))

detach(x)
}

plotField <- function(x,lon=NULL,lat=NULL,tim=NULL,mon=NULL,val.rng=NULL,
                      col="black",col.coast="grey",lty=1,lwd=1,what="ano",
                      type="s",pch=26,my.col=NULL,add=FALSE,
                      main=NULL,sub=NULL,xlab=NULL,ylab=NULL) {

  if ((class(x)[1]!="field") & (class(x)[1]!="monthly.field.object") &
      (class(x)[1]!="daily.field.object")){
    stop("x must be a 'field' object.")
  }

  if (is.null(lon) & is.null(lat) & is.null(tim)) {
    stop("At least one of lon/lat/tim must be specified for 1D or 2D plots.")
  }
  
 cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
         'Jul','Aug','Sep','Oct','Nov','Dec')
  
  if (!is.null(mon)) {
    im <- x$mm== mon
    x$dat <- x$dat[im,,]
    x$yy <- x$yy[im]
    x$mm <- x$mm[im]
    x$dd <- x$dd[im]
    x$id.t <- x$id.t[im]
    date <- cmon[mon]
  }
  ind <- 1:length(x$tim)

  # Hovmuller diagrams
  dims <- dim(x$dat)
  nt <- dims[1]
  lon.tim=FALSE
  tim.lat=FALSE
  lon.lat=FALSE
  time.ts=FALSE

  if (!is.null(lon) & is.null(lat) & is.null(tim)) {
#    print("Time-lat")
    dx <- mean(diff(x$lon),na.rm=TRUE)
    ii <- (x$lon >= min(lon)) & (x$lon < max(lon) + dx)
    if (sum(ii)==1) Z <- x$dat[,,ii] else if (sum(ii)>1) {
      Z <- x$dat[,,1]*0
      for (j in 1:dims[2]) {
        for (i in 1:dims[1]) {
          Z[i,j] <- mean(x$dat[i,j,ii],na.rm=TRUE)
        }
      }
    }
    xlab <- "Time"
    ylab <- "Latitude (deg N)"
    X <- x$yy + (x$mm - 0.5)/12
    Y <- x$lat
    tim.lat=TRUE
    np <- dims[2]
    dim(Z) <- c(dims[1],dims[2])
  }
  if (is.null(lon) & !is.null(lat) & is.null(tim)) {
#    print("Time-lon")
    dy <- mean(diff(x$lat),na.rm=TRUE)
    ii <- (x$lat >= min(lat)) & (x$lat < max(lat) + dy)
    if (sum(ii)==1) Z <- t(x$dat[,ii,]) else if (sum(ii)>1) {
      Z <- x$dat[,,1]*0
      for (j in 1:dims[1]) {
        for (i in 1:dims[3]) {
          Z[i,j] <- mean(x$dat[j,ii,i],na.rm=TRUE)
        }
      }
    }
    ylab <- "Time"
    xlab <- "Longitude (deg E)"
    Y <- x$yy + (x$mm - 0.5)/12
    X <- x$lon
    lon.tim=TRUE
    np <- dims[3]
    dim(Z) <- c(dims[3],dims[1])
  }
  # Map - call lower level plot function:
  if (is.null(lon) & is.null(lat) & !is.null(tim)) {
#    print("Map")
    ii <- (ind >= min(tim)) & (ind < max(tim) + 1)
    if (sum(ii) == 0) {
      ii <- rep(FALSE,length(x$tim))
      ii[tim] <- TRUE
    }
    if (sum(ii) == 1) {
      l <- seq(1,dims[1],by=1)[ii]
#      print(paste("l=",l,"date=",x$yy[ii],x$mm[ii],x$dd[ii]))
      results <- mapField(x,l=l,what=what,col=col,col.coast=col.coast,lty=lty,
                          lwd=lwd,val.rng=val.rng)
    } else if (sum(ii) > 1) {
      map <- meanField(x,t.rng=range(x$yy[ii]))
      results <- mapField(x,col=col,col.coast=col.coast,lty=lty,lwd=lwd,val.rng=val.rng)
    }       
    lon.lat <- TRUE
    invisible(results)
    return()
  }
  # Time series - call lower level plot function:
  if (!is.null(lon) & !is.null(lat)) {
#    print("plotField: Time-series")
    results <- grd.box.ts(x,lon,lat,what=what,col=col,
                          lty=lty,lwd=lwd,pch=pch,type=type,add=add,
                          main=main,sub=sub,xlab=xlab,ylab=ylab)
#    Z <- results$t2m
    time.ts <- TRUE
#    print("Now, try to exit this...")
  } 

  if ((tim.lat) | (lon.tim)) {
    print("Hovmuller diagrams")
    clim <- Z*0
    if (is.null(mon) & (what=="ano")) {
      if ( (x$mm[2]-x$mm[1]>=1) & (x$dd[2]==x$dd[1]) ) {
      for (im in 1:12) {
        it <- x$mm==im
           for (ip in 1:np) {
           if (tim.lat) clim[it,ip] <- mean(Z[it,ip],na.rm=TRUE)
           if (lon.tim) clim[ip,it] <- mean(Z[ip,it],na.rm=TRUE)
        }
      }
    } else if (lon.lat) {
      print("Longitude-latitude map")
        ac.mod<-matrix(rep(NA,nt*6),nt,6)
        ac.mod[,1]<-cos(2*pi*x$tim/365.25); ac.mod[,2]<-sin(2*pi*x$tim/365.25)
        ac.mod[,3]<-cos(4*pi*x$tim/365.25); ac.mod[,4]<-sin(4*pi*x$tim/365.25)
        ac.mod[,5]<-cos(6*pi*x$tim/365.25); ac.mod[,6]<-sin(6*pi*x$tim/365.25)
        dim(x$dat) <- c(nt,np)
        for (ip in seq(1,np,by=1)) {
          if (tim.lat) vec <- Z[,ip] else vec <- Z[ip,]
          if (sum(is.finite(vec)) > 0) {
            ac.fit<-lm(vec ~ ac.mod)
            if (tim.lat) clim[,ip] <- ac.fit$fit else clim[ip,] <- ac.fit$fit
          } else {
            if (tim.lat) clim[,ip] <- NA else clim[ip,] <- NA
          }
        }
      }
    }
  }

  if (!time.ts) {
    print("2D-plots")
    Z <- switch(lower.case(substr(what,1,3)),
                              "ano"=Z - clim,
                              "cli"=clim,
                              "abs"=Z)
 
   if (is.null(main)) main <-  paste(attributes(x$dat)$"long_name")
   if (is.null(sub)) sub <- date
    
    if (is.null(val.rng)) {
      z.levs <- seq(min(abs(as.vector(Z)),na.rm=TRUE),
                    max(abs(as.vector(Z)),na.rm=TRUE),length=21)
    } else z.levs <- seq(val.rng[1],val.rng[2],length=21)
    
    if (is.null(my.col)) my.col <- rgb(c(seq(0,1,length=10),rep(1,11)),
                                       c(abs(sin((0:20)*pi/20))),
                                       c(c(rep(1,11),seq(1,0,length=10))))
    filled.contour(X,Y,Z,
                   col = my.col,levels=z.levs,
                   main=main,sub=sub,xlab=xlab,ylab=ylab)

# From filled.contour in base
    mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
    on.exit(par(par.orig))

    w <- (3 + mar.orig[2]) * par('csi') * 2.54
    layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
    par(las = 1)
    mar <- mar.orig
    mar[4] <- 1
    par(mar=mar)
#    contour(X,Y,Z,add=TRUE,col=col,lwd=lwd,lty=lty,levels=z.levs)
    results <- list(Z=Z,x=X,y=Y,xlab=xlab,ylab=ylab,
                    descr=paste(attributes(x$dat)$"long_name"))

#  print("Finished")
  class(results) <- "2D field"
  attr(results,"long_name") <- attr(x$dat,"long_name")
  attr(results,"descr") <- "plotField"
  }
invisible(results)
}

# R.E. Benestad, met.no, Oslo, Norway 22.05.2002
# rasmus.benestad@met.no
#-------------------------------------------------------------------
# PLot data from NORDKLIMstations.

plotStation <- function(obs,l.anom=TRUE,mon=NULL,
                        leps=FALSE,out.dir="output",what="b",trend=TRUE,
                        type="l",pch=26,col="black",lwd=3,lty=3,add=FALSE,
                        main=NULL,sub=NULL,xlab=NULL,ylab=NULL) {

if (sum(is.element(c("b","t","d"),what))==0) stop("Argumet 'what' must be 'b','t' or 'd'!")
if ( (class(obs)[2]!="monthly.station.record") &
     (class(obs)[2]!="daily.station.record") ){
  stop(paste("The predictand must be a 'monthly.station.record'",
             "object - Use  station.obj()"))
}

if (class(obs)[2]=="daily.station.record") {
  newFig()
  plot(obs$yy + obs$mm/12 + obs$dd/365.25, obs$t2m,pch=20,cex=0.5,
       main=obs$location,sub="met.no Klima DataVareHus",
       xlab="Time",ylab="Temperature (deg C)")
  grid()
  lines(obs$yy + obs$mm/12 + obs$dd/365.25, obs$t2m,lty=3,col="grey")

  newFig()
  plot(obs$yy + obs$mm/12 + obs$dd/365.25, obs$precip,pch=20,cex=0.5,
       main=obs$location,sub="met.no Klima DataVareHus",
       xlab="Time",ylab="Precipitation (mm)")
  grid()
  lines(obs$yy + obs$mm/12 + obs$dd/365.25, obs$precip,lty=3,col="grey")
  plotStation <- obs
} else if (class(obs)[2]=="monthly.station.record") {
  
if ((!obs$found) | (sum(is.finite(obs$val))==0)) stop("No valid data!")

cmon<-c("Jan","Feb","Mar","Apr","May","Jun",
        "Jul","Aug","Sep","Oct","Nov","Dec")

if (!is.null(mon)) {
  if (((length(mon)== 1)) & (mon>0)) season <- cmon[mon]
  if (((length(mon)== 1)) & (mon==0)) season <- ""
  if ((length(mon)> 1)) season <- paste(cmon[mon[1]],'-',
                                      cmon[mon[length(mon)]],sep="")
} else {
  season <-"Dec-Jan"
}

  loc <- obs$location
  
  if (is.null(mon)) {
    ny <- length(obs$yy)
    mm <- rep(1:12,ny)
    value <- t(obs$val)
    if (l.anom) {
      for (im in 1:12) {
        value[im,] <- value[im,] - mean(value[im,],na.rm=TRUE)
        
        if ((!is.null(obs$alt)) & (!is.null(obs$lon)) & (!is.null(obs$lat))) {
          sub.tit <- paste("Anomaly",round(obs$alt,2),"m a.sl.",
                         round(obs$lon,2),"degE",
                         round(obs$lat,2),"degN")
        } else sub.tit <- paste("Anomaly:",loc)
          
      }
    } else {
      if ((!is.null(obs$alt)) & (!is.null(obs$lon)) & (!is.null(obs$lat))) {
         sub.tit <- paste("Absolute",round(obs$alt,2),"m a.sl.",
                          round(obs$lon,2),"degE",
                          round(obs$lat,2),"degN")
       } else sub.tit <- paste("Absolute:",loc)
    }

    dims <- dim(value)
    dim(value) <- c(dims[1]*dims[2],1)
    yy <- sort(rep(obs$yy,12)) + (rep(seq(1,12,by=1),ny)-0.5)/12
  } else {
    yy <- obs$yy
    ny <- length(obs$yy)
    mm <- rep(mon[1],ny)
    value <- obs$val[,mon[1]]
    if (mon==c(12,1,2)) {
      obs$val[2:ny,12] <- obs$val[1:(ny-1),12]
      obs$val[1,12] <- NA
    }

    for (i in 1:ny) value[i] <- mean(obs$val[i,mon],na.rm=TRUE)
    
    if (is.null(obs$ele)) {
          for (i in 1:ny) value[i] <- mean(obs$val[i,mon],na.rm=TRUE)
          obs$ele <- 0
    }
    if (is.element(obs$ele,c(101,111,121,401,601,701,801,911)))
          for (i in 1:ny) value[i] <- mean(obs$val[i,mon],na.rm=TRUE)
    if (is.element(obs$ele,c(112,602)))
          for (i in 1:ny) value[i] <- max(obs$val[i,mon],na.rm=TRUE)
    if (is.element(obs$ele,c(122)))
          for (i in 1:ny) value[i] <- max(obs$val[i,mon],na.rm=TRUE)
    
    if ((!is.null(obs$alt)) & (!is.null(obs$lon)) & (!is.null(obs$lat))) {
       sub.tit <- paste(season," - ",round(obs$alt,2),"m a.sl.",
                     round(obs$lon,2),"degE",
                     round(obs$lat,2),"degN")
    } else sub.tit <- paste(season,": ",loc,sep="")
  }


  # Polinomial trend

  y <- value
  x <- yy
  X <- data.frame(y=value,x = yy)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3) + I(x^4) + I(x^5))
  pre.p.fit<-predict(lm.tr.p,newdata=X)
  coef.p.fit<-lm.tr.p$coefficients
  coef.p.fit[is.na(coef.p.fit)] <- 0
  der.p.fit<-c(coef.p.fit[2],2*coef.p.fit[3],3*coef.p.fit[4],
             4*coef.p.fit[5],5*coef.p.fit[6])
  tr.est.p.fit<-(der.p.fit[1] + der.p.fit[2]*yy + der.p.fit[3]*yy^2 +
                 der.p.fit[4]*yy^3 + der.p.fit[5]*yy^4)*10


  good <- !is.na(value)
  yy <- yy[good]
  pre.p.fit <- pre.p.fit[good]
  value <- value[good]

  if (!leps) {
    
#  par(ask=TRUE)
    if ((what=="t") | (what=="b")) {
      if (!add) {
        newFig()
        par(cex.sub=0.8)
        plot(yy,value,type="l",lwd=lwd,col=col,pch=pch,lty=lty,
                     main=paste(obs$location,obs$obs.name),
                     sub=sub.tit,xlab="Time",ylab=obs$unit)
      } else lines(yy,value,lwd=lwd,col=col,pch=pch,lty=lty)
      if (trend) lines(yy,pre.p.fit,col="red") 
      lines(c(min(yy),max(yy)),rep(mean(value,na.rm=TRUE)+
                                   1.96*sd(value,na.rm=TRUE),2),
                                   lty=2,col="grey")
      lines(c(min(yy),max(yy)),rep(mean(value,na.rm=TRUE)-
                                   1.96*sd(value,na.rm=TRUE),2),
                                   lty=2,col="grey")
      grid()
    }

    if ((what=="d") | (what=="b")) {
      newFig()
      par(cex.sub=0.8)
      histo <- hist(value[!is.na(value)],breaks=15,lwd=3,freq=FALSE,
         main=paste(obs$location,obs$obs.name),
         sub=paste(min(round(yy,2)),"--",max(round(yy,2)),
           ":",sub.tit,xlab=obs$unit))

      x.dist <- seq(min(histo$mids),max(histo$mids),length=101)
      y.dist <- dnorm(x.dist,
                      mean=mean(value,na.rm=TRUE),
                      sd=sd(value,na.rm=TRUE))
      if (trend) lines(x.dist,y.dist,col="red")
      lines(x.dist,dgamma(x.dist-min(x.dist),
            shape=mean((value-min(x.dist))^2,na.rm=TRUE)/sd(value^2,na.rm=TRUE),
            scale=sd(value^2,na.rm=TRUE)/mean(value-min(x.dist),na.rm=TRUE)),
            col="blue",lty=3)
      grid()
    } else  {
      histo <- hist(value[!is.na(value)],breaks=15,lwd=3,freq=FALSE,plot=FALSE)
      x.dist <- seq(min(histo$mids),max(histo$mids),length=101)
      y.dist <- dnorm(x.dist,
                      mean=mean(value,na.rm=TRUE),
                      sd=sd(value,na.rm=TRUE))
    }

  } else  {
    
    figname1 <- paste(obs$location,'_',abbreviate(obs$obs.name),
                      '_',season,'1.eps',sep="")
    figname2 <- paste(obs$location,'_',abbreviate(obs$obs.name),
                      '_',season,'2.eps',sep="")
    postscript(file = figname1,onefile=TRUE,horizontal=FALSE,paper="a4")
    par(ps=14,cex.sub=0.8)
    plot(yy,value,type="l",lwd=3,
         main=paste(obs$location,obs$obs.name),
         sub=sub.tit,xlab="Time",ylab=obs$unit)
    if (trend) lines(yy[!is.na(y)],pre.p.fit,col="red")
    lines(c(min(yy),max(yy)),rep(mean(value,na.rm=TRUE)+
                                 1.96*sd(value,na.rm=TRUE),2),
          lty=2,col="grey")
    lines(c(min(yy),max(yy)),rep(mean(value,na.rm=TRUE)-
                                 1.96*sd(value,na.rm=TRUE),2),
          lty=2,col="grey")
    grid()
    dev.off()

    postscript(file = figname2,onefile=TRUE,horizontal=FALSE,paper="a4")
    par(ps=14,cex.sub=0.8)
    histo <- hist(value,breaks=15,lwd=3,freq=FALSE,
         main=paste(obs$location,obs$obs.name),
         sub=paste(min(round(yy,2)),"--",max(round(yy,2)),
           ":",sub.tit,xlab=obs$unit))

    x.dist <- seq(min(histo$mids),max(histo$mids),length=101)
    y.dist <- dnorm(x.dist,
                       mean=mean(value,na.rm=TRUE),
                       sd=sd(value,na.rm=TRUE))
    
    if (trend) lines(x.dist,y.dist,col="red")
    lines(x.dist,dgamma(x.dist-min(x.dist),
          shape=mean((value-min(x.dist))^2,na.rm=TRUE)/sd(value^2,na.rm=TRUE),
          scale=sd(value^2,na.rm=TRUE)/mean(value-min(x.dist),na.rm=TRUE)),
          col="blue",lty=3)
    grid()
    dev.off()
    file.copy(c(figname1,figname2),out.dir)
    file.remove(c(figname1,figname2))
  }

  plotStation <- list(yy=yy,mm=mm,value=value,loc=obs$location,
                      histo=histo,x.dist=x.dist,y.dist=y.dist)
}
  invisible(plotStation)

}
# R.E. Benestad, met.no, Oslo, Norway 09.10.2002
# rasmus.benestad@met.no
#------------------------------------------------------------------------

plumePlot <- function(ds.name.list=NULL,location,mon,direc="output",
                         t.rng=c(1850,2074),r2.th=50,p.th=0.05,
                         col="darkred",lwd=2,lty=1) {

  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
        'Jul','Aug','Sep','Oct','Nov','Dec')
  
  if (is.null(ds.name.list)) ds.name.list <- avail.ds(direc=direc)

  yy <- seq(t.rng[1],t.rng[2],by=1)
  nt <- length(yy)
  nds <- length(ds.name.list)
  sce <- matrix(rep(NA,nt*nds),nt,nds)
  i.sce <- rep(FALSE,nds)
  yy.min <- NA
  yy.max <- NA
  
#  x11()
#  load(paste(direc,"/",ds.name.list[1],sep=""))
#  plot(ds$yy.gcm,ds$pre.gcm,type="n")
  
  for (i.ds in 1:nds) {
    load(paste(direc,"/",ds.name.list[i.ds],sep=""))
#    print(summary(ds))
#    print(c(strip(lower.case(ds$location)),lower.case(location)))
#    print(ds$mm.gcm[1])
#    print(c(ds$fit.r2,ds$fit.p))
    if ( (strip(lower.case(ds$location))==lower.case(location)) &
         (mon==ds$mm.gcm[1]) & (ds$fit.r2 >= r2.th) &
         (ds$fit.p <= p.th) ) {
      i1 <- is.element(yy,ds$yy.gcm)
      i2 <- is.element(ds$yy.gcm,yy)
      sce[i1,i.ds] <- ds$pre.gcm[i2]
      i.sce[i.ds] <- TRUE
#      print(range(ds$yy.gcm[i2]))
#      lines(ds$yy.gcm[ii],ds$pre.gcm[ii])
      if ((is.na(yy.min)) | (min(ds$yy.gcm)<yy.min)) yy.min<-min(ds$yy.gcm) 
      if ((is.na(yy.max)) | (max(ds$yy.gcm)>yy.max)) yy.max<-max(ds$yy.gcm) 
    }
  }
  sce <- sce[,i.sce]
  q975 <- rep(NA,nt)
  q025 <- rep(NA,nt)
  q500 <- rep(NA,nt)
  q250 <- rep(NA,nt)
  q750 <- rep(NA,nt)
  for (it in 1:nt) {
    if (sum(is.finite(sce[it,]))>0) {
      q975[it] <- quantile(sce[it,is.finite(sce[it,])],0.975)
      q025[it] <- quantile(sce[it,is.finite(sce[it,])],0.025)
      q500[it] <- quantile(sce[it,is.finite(sce[it,])],0.500)
      q250[it] <- quantile(sce[it,is.finite(sce[it,])],0.250)
      q750[it] <- quantile(sce[it,is.finite(sce[it,])],0.750)
    }
  }
#  print(summary(sce))
#  print(i.sce)
  x.ind <- seq(0,1,length=nt)
  tr.dat<-data.frame(y=q975, x=x.ind)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3), data=tr.dat)
  p975<-predict(lm.tr.p,newdata=tr.dat)
  tr.dat<-data.frame(y=q025, x=x.ind)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3), data=tr.dat)
  p025<-predict(lm.tr.p,newdata=tr.dat)
  tr.dat<-data.frame(y=q500, x=x.ind)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3), data=tr.dat)
  p500<-predict(lm.tr.p,newdata=tr.dat)
  tr.dat<-data.frame(y=q250, x=x.ind)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3), data=tr.dat)
  p250<-predict(lm.tr.p,newdata=tr.dat)
  tr.dat<-data.frame(y=q750, x=x.ind)
  lm.tr.p<-lm(y ~ x + I(x^2) +I(x^3), data=tr.dat)
  p750<-predict(lm.tr.p,newdata=tr.dat)       

  subtitle <- paste("Based on",sum(i.sce),"scenarios.",
                    "Using a cubic-fit to trend,",
                    "R^2 fit >",r2.th,"% & p-val fit <",p.th)
  yy.o <- ds$yy.o
  obs <- ds$y.o
  ii <- is.element(yy,seq(yy.min,yy.max,by=1))
  y.lim.tr <- range(c(p975[ii],p025[ii]))

  print(summary(p025))
  plot(c(min(c(yy.o,yy.min)),yy.max),
       y.lim.tr,type="n",
       main=location,sub=subtitle,
       xlab="Time",
       ylab=paste(cmon[mon],ds$v.name,"(",ds$unit,")"))
  grid()

  polygon(c(yy,reverse(yy)),
          c(p975,reverse(p025)),col="grey75")
  polygon(c(yy,reverse(yy)),
          c(p250,reverse(p750)),col="wheat",density=10,lwd=5)
  lines(yy,p500,col="black",lwd=3)
  lines(yy,q975,col="grey20",type="s",lty=3)
  lines(yy,q025,col="grey20",type="s",lty=3)
  lines(yy,q250,col="grey20",type="s",lty=3)
  lines(yy,q750,col="grey20",type="s",lty=3)
  lines(yy,q500,col="black",type="s",lty=3)
  lines(range(yy),rep(min(p500[ii]),2),lty=2)
  lines(range(yy),rep(max(p500[ii]),2),lty=2)
  points(yy.o,obs,pch=20)
  points(yy.o,obs,pch=20,cex=0.6,col="red")

  legend(quantile(yy,0.95),
         quantile(c(p025[ii]),0.75),
         c("2.5%--97.5%","25%--75%","Median","Obs"),
         col=c("grey75","wheat","black","red"),
         lwd=c(5,5,3,0),lty=c(rep(1,3),0),
         pch=c(rep(26,3),20),bg="grey95",cex=0.6)
}
# R.E. Benestad, met.no, Oslo, Norway 16.04.2002
# rasmus.benestad@met.no
#
# Modified 27.04.2004 to base the IO on the ncdf package instead of
# netCDF (which is being phased out).
#------------------------------------------------------------------------


retrieve.nc <- function(filename=file.path("data","ncep_t2m.nc"),v.nam="AUTO",
                        l.scale=TRUE,greenwich=TRUE,
                        x.nam="lon",y.nam="lat",z.nam="lev",t.nam="tim",
                        x.rng=NULL,y.rng=NULL,t.rng=NULL,force.chron=TRUE) {
  library(ncdf)
  if (!file.exists(filename)) {
    stop(paste("Sorry,",filename," does not exist!"))
  }

  if (lower.case(Sys.info()[1])=="windows") {
   stop("Sorry, this function currently only works on Linux systems!")
  }  
  
  dat.att <- cdfcont(filename)
  ncid1 <- open.ncdf(filename)
  if (v.nam=="AUTO") {
    v1 <- ncid1$var
    n.vars <- length(v1)
    if (n.vars==1) v1 <- v1[[1]] else {
      ipick<-0
      for (i in seq(n.vars,1,by=-1)) {
        if (v1[[i]]$ndim==3) ipick <- i
      }
      if (ipick > 0) v1 <- v1[[ipick]] else {
         print("Tip: use open.ncdf to read the data manually")
         print(names(v1))
         print(dat.att)
         stop("Error: did'n find a variable with 3D")
      }
    }
  } else {
    v1 <- ncid1$var
    vars <- names(v1)
    ipick <- grep(v.nam,vars)
    if (length(ipick)==0) {
      print(vars)
      ipick <- as.numeric(readline(paste("Choose variable (1 - ",length(vars),"): ",sep="")))
    }
    v1 <- ncid1$var[[ipick]]
  }
  data <- get.var.ncdf(ncid1,v1)
  close.ncdf(ncid1)
  vars <- v1$name
  nvars <-  ncid1$nvars
  print(paste("Reading",vars))
  v.nam <- v1$longname
  dims <- names(ncid1$dim)
  vars <- names(ncid1$dim)
  n.dim <- ncid1$ndims
  d <- rep(0,nvars)
  dat <- NULL

  ilon <- grep("lon",lower.case(names(ncid1$dim)))
  ilat <- grep("lat",lower.case(names(ncid1$dim)))
  itim <- grep("tim",lower.case(names(ncid1$dim)))
  ilev <- grep("lev",lower.case(names(ncid1$dim)))
  #print(c(ilon,ilat,itim))

  eval(parse(text=paste("lon <- ncid1$dim$",names(ncid1$dim)[ilon],"$vals",sep="")))
  eval(parse(text=paste("lat <- ncid1$dim$",names(ncid1$dim)[ilat],"$vals",sep="")))
  eval(parse(text=paste("tim <- ncid1$dim$",names(ncid1$dim)[itim],"$vals",sep="")))
  attr(lon,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[ilon],"$units",sep="")))
  attr(lat,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[ilat],"$units",sep="")))
  attr(tim,"time_origin") <- dat.att$time.origin
  if (!is.null(dat.att$time.unit)) attr(tim,"unit") <- dat.att$time.unit else 
    attr(tim,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[itim],"$units",sep="")))
   
  if (length(ilev)>0) {
    eval(parse(text=paste("lev <- ncid1$dim$",names(ncid1$dim)[ilev],"$vals",sep="")))
    attr(lev,"unit") <- eval(parse(text=paste("ncid1$dim$",names(ncid1$dim)[ilev],"$units",sep="")))
    n.dim <- 4
  } else {
    lev <- NULL
    n.dim <- 3
  }
 # Re-order the data: (old convention)
  nt <- length(tim); ny <- length(lat); nx <- length(lon)
  #print(c(nt,ny,nx,NA,dim(data)))
  dat <- data*NA; dim(dat) <- c(nt,ny,nx)
  for (i in 1:nt) dat[i,,] <- t(as.matrix(data[,,i]))
  
  cmon<-c('Jan','Feb','Mar','Apr','May','Jun',
          'Jul','Aug','Sep','Oct','Nov','Dec')
  season<-cbind(c(12,1,2),c(3,4,5),c(6,7,8),c(9,10,11))
  season.c<-c("","DJF","MAM","JJA","SON")

  dtim <- diff(tim)
  if ( sum(dtim<=0) > 0) {
    print(paste("Warning! Test of chonological order finds",sum(dtim<=0),"jump(s)"))
    print(paste("median(dtim)=",median(dtim)))
    if (force.chron) {
      nt <- length(tim)
      tim.att <- attributes(tim)
      dtims <- as.numeric(row.names(table(dtim)))
      if (length(dtims < 4)) {
        print(paste("Force correction: assume tim[1] is correct,",
                    median(dtim),"is correct time step, and length=",nt))
        tim <- seq(tim[1],tim[1]+nt-1,by=median(dtim))
      } else {
        dt <- readline("What is the correct time step? (0 leaves tim unchanged)")
        if (dt != 0) tim <- seq(tim[1],tim[1]+nt-1,by=dt)
      }
    }
    print(paste("length(tim)=",length(tim),"nt=",nt))
#    print("set new attributes for tim")
    attributes(tim) <- tim.att
#    print("continue...")
  }

  t.unit <- attr(tim,"unit")
  dat.att$unit <-v1$units

  if (!is.null(dat.att$time.origin)) {
    torg <-  dat.att$time.origin
  } else torg <- NULL

  if (!is.null(torg)) {
    yy0 <- as.numeric(substr(torg,8,11))
    dd0 <- as.numeric(substr(torg,1,2))
    mm0 <- switch(lower.case(substr(torg,4,6)),
                  "jan"=1,"feb"=2,"mar"=3,"apr"=4,"may"=5,"jun"=6,
                  "jul"=7,"aug"=8,"sep"=9,"oct"=10,"nov"=11,"dec"=12)
  } else if (grep("since",lower.case(t.unit))) {
    # Format: time:units = "hours since 1-1-1 00:00:0.0" (NCEP reanalysis)
    t.org.pos <- regexpr("since",lower.case(t.unit))
    torg  <- substr(t.unit,t.org.pos+6,nchar(t.unit))
    print(paste("torg=",torg))
    dash <- instring("-",torg)
    spc <- instring(" ",torg)
    yy0 <- as.numeric(substr(torg,1,dash[1]-1))
    mm0 <- as.numeric(substr(torg,dash[1]+1,dash[2]-1))
    dd0 <- as.numeric(substr(torg,dash[2]+1,spc[1]-1))
    if (is.na(dd0[1])) dd0  <- 15
  }
  print(paste("Time origin: (year-month-day)",yy0,"-",mm0,"-",dd0))
  if (yy0[1]==0) {
    print('There is no year zero (Press et al., Numerical recipies)')
    print("'> print(julday(1,1,1)-julday(1,1,-1))' gives 365")
    print('julday wont work unless the time is fixed')
    print("year0 is set to 1, and 365 days is subtracted from tim")
    if (substr(lower.case(t.unit),1,4)=="hour") tim <- tim - 365*24
    if (substr(lower.case(t.unit),1,3)=="day") tim <- tim - 365
    if (substr(lower.case(t.unit),1,3)=="mon") tim <- tim - 12
    if (substr(lower.case(t.unit),1,5)=="year") tim <- tim - 1
    yy0 <- 1
  }

  print(paste("Time unit:",lower.case(t.unit)))
  if (substr(lower.case(t.unit),1,3)=="mon") {
    tim <- floor(tim)
    mm <- mod(mm0 + tim - 1,12)+1
    yy  <- yy0 + floor((tim+mm0-1)/12)
    dd <- rep(15,length(tim))
    obj.type <- "monthly.field.object"
  } else if (substr(lower.case(t.unit),1,3)=="day") {
    mmddyy <- caldat(tim + julday(mm0,dd0,yy0))
    mm <- mmddyy$month
    yy <- mmddyy$year
    dd <- mmddyy$day
    obj.type <- "daily.field.object"
  } else if (substr(lower.case(t.unit),1,4)=="hour") {
    mmddyy <- caldat(tim/24 + julday(mm0,dd0,yy0))
    mm <- mmddyy$month
    yy <- mmddyy$year
    dd <- mmddyy$day
    t.unit <- "day"
    obj.type <- "field.object"
  } 
  
# Extra processing for NCEP files e.g. with Time unit: hours since 1-1-1 00:00:0.0.
  if ( ((substr(lower.case(t.unit),1,4)=="hour") |
        (substr(lower.case(t.unit),1,3)=="day")) &
       (max(diff(dd)) == 0) ) {
    print("Monthly data, but time unit set to 'hour'/'day'")
    print("Set time unit to month")
    obj.type <- "monthly.field.object"
    dd[] <- 15
    t.unit <- "month"
}


#  print("Latitude:")
  if (attributes(lat)$"unit"=="degrees_south") lat <- lat * -1
  if (attributes(lon)$"unit"=="degrees_west") lon <- lon * -1
  if (!is.null(y.rng)) {
    print(range(lat))
    print("Extract latitudes:")
    print(y.rng)
    y.keep <- (lat >= min(y.rng)) & (lat <= max(y.rng))
    if (n.dim==3) dat <- dat[,y.keep,] else
                  dat <- dat[,,y.keep,] 
    lat <- lat[y.keep]
  }
  if (greenwich) {
    lon[lon > 180] <- lon[lon > 180]-360
  }
#  print("Sort longs and lats")
  x.srt <- order(lon)
  y.srt <- order(lat)
  lon <- lon[x.srt]
  lat <- lat[y.srt]
  if (n.dim==3) dat <- dat[,y.srt,x.srt] else
                dat <- dat[,,y.srt,x.srt]
  
  if (!is.null(x.rng)) {
    print(range(lon))
    print("Extract longitudes:")
    print(x.rng)
    x.keep <- (lon >= min(x.rng)) & (lon <= max(x.rng))
    if (n.dim==3) dat <- dat[,,x.keep] else
                  dat <- dat[,,,x.keep]
    lon <- lon[x.keep]
  }
  if (!is.null(t.rng)) {
    print(range(yy))
    print("Extract times:")
    print(t.rng)
    t.keep <- (yy >= min(t.rng)) & (yy <= max(t.rng))
    if (n.dim==3) dat <- dat[t.keep,,] else
                  dat <- dat[t.keep,,,]
    torg <- attr(tim,"time.origin")
    tunit <- attr(tim,"unit")
    tim <- tim[t.keep]
    attr(tim,"time.origin") <- torg
    attr(tim,"unit") <- tunit
    yy <- yy[t.keep]
    mm <- mm[t.keep]
    dd <- dd[t.keep]
    nt <- length(tim)
  }
  print(paste("First & last records:",yy[1],mm[1],dd[1],
              "&",yy[length(yy)],mm[length(mm)],dd[length(dd)]))
  
#  print(dat.att$scale.factor)
#  print(dat.att$add.offset)
#  print(dat.att$unit)

#  print(dat.att)
  if ((l.scale) & !is.null(dat.att$scale.factor)) {
     if (is.finite(dat.att$scale.factor)) dat <- dat * dat.att$scale.factor
  }
  # Have included a sanity test to detect an old 'bug': offset 273 and
  # units of deg C..
  if ( ((l.scale) & !is.null(dat.att$add.offset))) {
      if ( (dat.att$add.offset!=273) &
           (dat.att$unit=="deg C")) {
        a <- readline(prompt="Correct an old bug? (y/n)")
        if (lower.case(a)=="y") dat <- dat + dat.att$add.offset} else
        if (is.finite(dat.att$add.offset)) dat <- dat + dat.att$add.offset
  }
  if (l.scale) {   
    print("BEFORE scale adjustment & weeding")
    print(summary(as.vector(dat)))
    dat[dat == dat.att$missing.value] <- NA
    if (sum(is.na(dat))>0) print(paste(sum(is.na(dat)),"of",length(dat),
                                 " are set to 'NA'"))
      print("AFTER scale adjustment & weeding")
  }

  if (!is.null(dat.att$units)) {
     if (is.finite(dat.att$units)) dat.att$unit <- dat.att$units
  } 
  if ((dat.att$unit=="K") | (dat.att$unit=="Kelvin") |
      (dat.att$unit=="degrees Kelvin") |
      (dat.att$unit=="deg K") | (dat.att$unit=="degK")) {
    dat <- dat - 273
    dat.att$unit <- "deg C"
  }
    if ((dat.att$unit=="Pa") | (dat.att$unit=="Pascal") |
      (dat.att$unit=="N/m^2") |
      (dat.att$unit=="N m^{-1}")) {
    dat <- dat/100
    dat.att$unit <- "hPa"
  }
  print(summary(as.vector(dat)))
  nx <- length(lon)
  ny <- length(lat)
  print(c(nt,ny,nx))
  eos <- nchar(v.nam)
  if (instring("-",v.nam)> 0) {
    eos <- instring("-",v.nam)-1
  } else if (instring("_",v.nam)> 0) {
    eos <- instring("_",v.nam)-1
  }
  v.nam <- substr(v.nam,1,eos)
  id.x <- matrix(rep(v.nam,ny*nx),ny,nx)
  slash <- instring("/",filename)
  dot <- instring(".",filename)

  id.t <- rep(substr(filename,slash[length(slash)]+1,
                     dot[length(dot)]-1),nt)              
  dat.att$time.unit <- t.unit
  dat.att$time.origin <- torg
  retrieve.nc  <- list(dat=dat,lon=lon,lat=lat,tim=tim,lev=lev,
                       v.name=v.nam,id.x=id.x,id.t=id.t,
                       yy=yy,mm=mm,dd=dd,n.fld=1,
                       id.lon=rep(v.nam,nx),id.lat=rep(v.nam,ny),
                       attributes=dat.att)
  class(retrieve.nc) <- c("field",obj.type)
  invisible(retrieve.nc)
}

# Produces a vector with the reversed order to that of a sort
# call
# R.E. Benestad
reverse.sort <- function(x) {
  reverse <- -1*sort(-1*x)
  reverse
}

reverse <- function(x) {
  reverse  <- x
  for (i in 1:length(x)) reverse[i]  <-  x[length(x)-i+1]
  reverse
}
rotate <- function(lons,lats,lon.0=NULL,lat.0=NULL) {

  if (is.null(lon.0)) lon.0 <- mean(lons)
  if (is.null(lat.0)) lat.0 <- mean(lats)
  
  theta0 <- pi*lon.0/180
  phi0 <- pi*lat.0/180
  r0 <- c(cos(phi0)*cos(theta0),
          sin(phi0),
          cos(phi0)*sin(theta0))

  theta <- rep(NA,length(lons)); phi <- theta
  for (i in 1:length(lons)) {
    thetaA <- pi*lons[i]/180
    phiA <- pi*lats[i]/180
    r <- c(cos(phiA)*cos(thetaA),
           sin(phiA),
           cos(phiA)*sin(thetaA))
    thetaA <- pi*lon.0/180
    r1 <- c(cos(phiA)*cos(thetaA),
            sin(phiA),
            cos(phiA)*sin(thetaA))
    a <- r - r0
    b <- r1 - r0
    phi[i] <- acos( sum(r*r0) )
    theta[i] <- acos(sum(a*b) / (sqrt(sum(a*a)) * sqrt(sum(b*b))) )
 }
  
#  thetaA <- pi*thetaA/180
#  dtheta <- pi*dtheta/180
#  phiA <- pi*phiA/180
#  dphi <- pi*dphi/180
#    
#  phi <- rep(NA,length(thetaA))
#  theta <- rep(NA,length(thetaA))
#
#  for (i in 1:length(lons)) {
#    tA <- thetaA[i]
#    R <- rbind(
#     c(cos(dphi)*cos(dtheta),-cos(dphi)*sin(dtheta),sin(dphi)*cos(tA+dtheta)),
#     c(cos(dphi)*sin(dtheta),cos(dphi)*cos(dtheta),sin(dphi)*sin(tA+dtheta)),
#     c(-sin(dphi-tA),cos(dphi-tA),cos(dphi)))
#
#
#    x <- c(cos(phiA[i])*cos(thetaA[i]),
#           sin(phiA[i]),
#           cos(phiA[i])*sin(thetaA[i]))
#    y <- R %*% x
#    print(R)
#    print(cbind(x,y))
#    phi[i] <- acos(y[3])
#    theta[i] <- acos(y[1]/sin(phi[i]))
#   }
  result <- list(phi=180*phi/pi,theta=180*theta/pi,)
  result
}
satellite <- function(map.obj,col="black",lwd=2,lty=1,add=FALSE,
                      las = 1,lon.0=NULL,lat.0=NULL,method="normal",
                      ni=100,nj=100, n.nearest=4,max.dist=3,landdata="addland2") {
  if (class(map.obj)!="map")  stop("Need a map object (map.field)")

  if (!is.null(map$map.1)) n.maps <- map$n.maps else n.maps <- 1
  
  map.xy <- matrix(rep(NA,ni*nj*n.maps),ni,nj*n.maps)
  dim(map.xy) <- c(ni,nj,n.maps)
  for (i.map in 1:n.maps) {
    if (!is.null(map$n.maps)) {
      expression <- paste("map$map.",i.map,",[i.near[1]]",sep="")
      lon <- eval(parse(text=paste("map.obj$lon.",i.map,sep="")))
      lat <- eval(parse(text=paste("map.obj$lat.",i.map,sep="")))
    } else {
      lon <- map.obj$lon; lat <- map.obj$lat
      expression <- "map$map[i.near[1]]"
    }
 
    if (is.null(lon.0)) lon.0 <- mean(lon)
    if (is.null(lat.0)) lat.0 <- mean(lat)
  
    nx <- length(lon)
    ny <- length(lat)
    np <- nx*ny
    theta <- pi*lon.0/180
    phi <- pi*lat.0/180
    phi.min <- pi*min(lat[is.finite(lat)])/180
    r0 <- c(cos(phi)*cos(theta),
            sin(phi),
            cos(phi)*sin(theta))
    lats <- rep(lat,nx)
    lons <- sort(rep(lon,ny))

#    print(c(lon.0,lat.0))
#    print(range(lons))
#    print(range(lats))
#    print(r0)
  
    x <- rep(NA,np)
    y <- rep(NA,np)
    if (method=="polarstereo") {
      r <- sin( pi*(90-lats)/180 )
      x <- r*sin(pi*lons/180)
      y <- -r*cos(pi*lons/180)
    } else if (method=="distance") {
    for (i in 1:np) {
        theta <- pi*lons[i]/180
        phi <- pi*lats[i]/180
        r <- c(cos(phi)*cos(theta),
               sin(phi),
               cos(phi)*sin(theta))
        theta <- pi*lon.0/180
        r1 <- c(cos(phi)*cos(theta),
                sin(phi),
                cos(phi)*sin(theta))
        if (is.finite(r0*r1)) y[i] <- acos(sum(r0*r1)) else y[i] <- NA
        if (is.finite(r*r1)) x[i] <- acos(sum(r*r1)) else x[i] <- NA
      }
      y[y>0.5*pi] <- NA; x[x>0.5*pi] <- NA
      x <- sin(x); y <- sin(y)
      x[lons<lon.0] <- x[lons<lon.0]*-1
      y[lats<lat.0] <- y[lats<lat.0]*-1
      x[!is.finite(x)] <- 0
      y[!is.finite(y)] <- 0
    } else {
        for (i in 1:np) {
        theta <- pi*lons[i]/180
        phi <- pi*lats[i]/180
        r <- c(cos(phi)*cos(theta),
               sin(phi),
               cos(phi)*sin(theta))
        theta <- pi*lon.0/180
        phi <- phi.min
        r1 <- c(cos(phi)*cos(theta),
                sin(phi),
                cos(phi)*sin(theta))
        a <- r - r0
        b <- r1 - r0
        if (is.finite(r*r0)) newphi <- acos( sum(r*r0) ) else newphi<-NA 
        newtheta <- acos(sum(a*b) / (sqrt(sum(a*a)) * sqrt(sum(b*b))) )
        if (lons[i]<lon.0) newtheta <- -newtheta
        d <- sin(newphi)
        x[i] <- d*sin(newtheta)
        y[i] <- -d*cos(newtheta)
      }
    }

  
    map <- t(map.obj$map)
    dim(map) <- c(np,1)

    x.grd <- seq(-1,1,length=ni)
    y.grd <- seq(-1,1,length=nj)
    dx <- sqrt(mean(diff(x.grd))^2 + mean(diff(y.grd))^2)

#  print("gridding...")
#  x11(); plot(x,y)

    for (j in 1:nj) {
      for (i in 1:ni) {
        dist <- sqrt( (x.grd[i]-x)^2 + (y.grd[j]-y)^2 )
        i.near <- order(dist)
        i.near <- i.near[1:n.nearest]
        i.ok <- (dist[i.near] <= dx*max.dist) & (is.finite(map[i.near]))
        if ((sum(i.ok) > 0) & (x.grd[i]^2 + y.grd[j]^2 <= 1)) {
          i.near <- i.near[i.ok] 
          if (min(dist[i.near]) > 0) {
            map.xy[i,j,i.map] <-  sum(map[i.near]/dist[i.near])/sum(1/dist[i.near])
          } else map.xy[i,j,i.map] <- eval(parse(text=expression))
        } else map.xy[i,j,i.map] <- NA
      }
    }
  }

  z.levs <- seq(-max(abs(as.vector(map.xy)),na.rm=T),
                 max(abs(as.vector(map.xy)),na.rm=T),length=41)
  my.col <- rgb(c(seq(0,1,length=20),rep(1,21)),
                c(abs(sin((0:40)*pi/40))),
                c(c(rep(1,21),seq(1,0,length=20))))

  if (!add) {
    par(col.lab="white")
    filled.contour(x.grd,y.grd,map.xy[,,1],
                   col = my.col,levels=z.levs,
                   main=paste(attributes(map.obj)$"long_name",
                              attributes(map.obj)$"descr"),
                   sub=map.obj$date,xlab="",ylab="")
    par(col.lab="black")
  }
# From filled.contour in base
  mar.orig <- (par.orig <- par(c("mar","las","mfrow")))$mar
  on.exit(par(par.orig))

  w <- (3 + mar.orig[2]) * par('csi') * 2.54
  layout(matrix(c(2, 1), nc=2), widths=c(1, lcm(w)))
    
  par(las = las)
  mar <- mar.orig
  mar[4] <- 1
  par(mar=mar)

#  points(x,y,pch=".")

# Circumpherence of the Earth:
#  

  if (!add) {
    eval(parse(text=paste("data(",landdata,")",sep="")))
    lon.cont[lon.cont > 180] <- lon.cont[lon.cont > 180] - 360

    iwest <- lon.cont < lon.0
    isouth <- lat.cont < lat.0
    np.cont <- length(lon.cont)
    y.cont <- rep(NA,np.cont)
    x.cont <- y.cont
    if (method=="polarstereo") {
      x <- seq(-1,1,length=100)
      y <- sqrt(1 - x^2)
      lines(x,y,type="l")
      lines(x,-y,type="l")

      r <- sin( pi*(90-lat.cont)/180 )
      x.cont <- r*sin(pi*(lon.cont)/180)
      y.cont <- -r*cos(pi*(lon.cont)/180)
      y.cont[lat.cont < 0] <- NA
    } else if (method=="distance") {
      for (i in 1:np.cont) {
        theta <- pi*lon.cont[i]/180
        phi <- pi*lat.cont[i]/180
        r <- c(cos(phi)*cos(theta),
               sin(phi),
               cos(phi)*sin(theta))
        theta <- pi*lon.0/180
        r1 <- c(cos(phi)*cos(theta),
                sin(phi),
                cos(phi)*sin(theta))
        y.cont[i] <- acos(sum(r0*r1))
        x.cont[i] <- acos(sum(r*r1))
      }
      y.cont[abs(y.cont)>0.5*pi] <- NA
      x.cont[abs(x.cont)>0.5*pi] <- NA
      x.cont <- sin(x.cont); y.cont <- sin(y.cont)
      x.cont[iwest] <- -x.cont[iwest]
      y.cont[isouth] <- -y.cont[isouth]
      x.cont[!is.finite(x.cont)] <- NA
      y.cont[!is.finite(y.cont)] <- NA

  } else {
      x <- seq(-1,1,length=100)
      y <- sqrt(1 - x^2)
      lines(x,y,type="l")
      lines(x,-y,type="l")
      for (i in 1:np.cont) {
        if (is.finite(lon.cont[i])) {

          theta <- pi*lon.cont[i]/180
          phi <- pi*lat.cont[i]/180
          r <- c(cos(phi)*cos(theta),
                 sin(phi),
                 cos(phi)*sin(theta))
          theta <- pi*lon.0/180
          r1 <- c(cos(phi)*cos(theta),
                  sin(phi),
                  cos(phi)*sin(theta))
          a <- r - r0
          b <- r1 - r0

          if (is.finite(r*r0)) newphi <- acos( sum(r*r0) ) else newphi<-NA  
          newtheta <- acos(sum(a*b) / (sqrt(sum(a*a)) * sqrt(sum(b*b))) )

          if (!is.finite(newphi))  newphi<- NA
          if (!is.finite(newtheta))  newtheta<- NA
          if (newphi > 0.5*pi) newphi<- NA
          if ((lon.cont[i]<lon.0) & (phi<pi*lat.0/180)) newtheta <- -newtheta
          if ((lon.cont[i]>lon.0) & (phi>pi*lat.0/180)) newtheta <- pi - newtheta
          if ((lon.cont[i]<lon.0) & (phi>pi*lat.0/180)) newtheta <- newtheta - pi
          d <- sin(newphi)
          x.cont[i] <- d*sin(newtheta)
          y.cont[i] <- -d*cos(newtheta)
        }     
      }
    }
  }
  lines(x.cont,y.cont,col="grey30")
  for (i.map in 1:n.maps){
    contour(x.grd,y.grd,as.matrix(map.xy[,,i.map]),add=T,lwd=lwd,lty=lty,col=col)
  }
} 
# This function converts a time series to the same format as getnacd
# and getnordklim - compatible with ds.R
#
#
# R.E. Benestad 04.06.2002

station.obj <- function(x,yy,obs.name,unit,ele=NULL,mm=NULL,
                        station=NULL,lat=NULL,lon=NULL,alt=NULL,
                        location="unspecified",wmo.no=NULL,
                        start=NULL,yy0=NULL,country=NULL,ref=NULL) {

  if ((!is.null(lat)) & (!is.null(lon))) {
    xy <- COn0E65N(lon,lat)
 } else xy <- list(x=NULL,y=NULL)

  x[x <= -999] <-  NA
  yrs <- table(yy)
  ny <- length(row.names(yrs))
  yrs <- as.numeric(row.names(yrs))
#  print(table(yy))
#  print(table(mm))
#  print(c(mm[1],mm[length(mm)],ny))  
#  print(yrs)

#x11()
#plot(c(1,12),c(260,295),type="n")
#grid()
#col<-c("black","red","blue","green","darkred","darkblue","darkgreen","grey30")
#lwd <- c(2,rep(1,7),2)

#if (!is.null(mm)) print(paste("First month:",mm[1]," - last month:",mm[length(mm)]))  
  if (is.vector(x)) {
#    print("Vector")
    if (length(x)==length(yy)) {
      x.2D <- matrix(rep(NA,ny*12),ny,12)
      for (i in 1:ny) {
        x.i <- x[yy==yrs[i]]
        m.i <- mm[yy==yrs[i]]
 #       print(rbind(c(i,NA,m.i),c(yrs[i],NA,x.i)))
        x.2D[i,m.i] <- x.i
 #       lines(as.vector(x.2D[i,]),col=col[i],lwd=lwd[i])
 #       points(as.vector(x.2D[i,]),col=col[i],pch=20)
      }
    } else {
      print(paste("Different lengths: x->",length(x),
                  "    yy->",length(yy)))
      stop("Error: time stamp doesn't agree with data")
    }
    x <- x.2D
    yy <- yrs
    rm(x.2D,yrs)
  }

  #print("Make list")
  station.obj<-list(val=x,yy=yy,station=station,
                    lat=lat,lon=lon,alt=alt,
                    x.0E65N=xy$x,y.0E65N=xy$y,
                    location=location, wmo.no=wmo.no,
                    start=start,yy0=NULL,ele=ele,
                    obs.name=obs.name, unit=unit,country=country,
                    found=TRUE,
                    ref=ref)
  class(station.obj) <- c("station","monthly.station.record")
  invisible(station.obj )
}
# This function converts a time series to the "daily.station.record"
# class - compatible with ds.dm.R
#
#
# R.E. Benestad 04.06.2002

station.obj.dm <- function(t2m,precip,dd,mm,yy,
                        obs.name=NULL,unit=NULL,ele=NULL,
                        station=NULL,lat=NULL,lon=NULL,alt=NULL,
                        location="unspecified",wmo.no=NULL,
                        start=NULL,yy0=NULL,country=NULL,ref=NULL) {

#  source("COn0E65N.R")
  xy<-COn0E65N(lon,lat)
#  if ((lat != NULL) & (lon != NULL)) xy<-COn0E65N(lon,lat) else
#     xy <- list(x=NULL,y=NULL)

  station.obj.dm<-list(t2m=t2m,precip=precip,
                    dd=dd,mm=mm,yy=yy,
                    obs.name=obs.name,unit=unit,ele=ele,
                    station=station,
                    lat=lat,lon=lon,alt=alt,
                    x.0E65N=xy$x,y.0E65N=xy$y,
                    location=location, wmo.no=wmo.no,
                    start=start,yy0=yy0,country=country,
                    found=TRUE,ref=ref)
  class(station.obj.dm) <- c("station","daily.station.record")
  invisible(station.obj.dm)
}
# Empirical downscaling using "mixed common EOFs" from ceof.R
# Predictand is a time series from NACD or climate station.
# Monthly mean values.
#
# Reference: R.E. Benestad et al. (2002),
#            Empirically downscaled temperature scenarios for Svalbard,
#            submitted to Atm. Sci. Lett.
#
#            R.E. Benestad (2001),
#            A comparison between two empirical downscaling strategies,
#            Int. J. Climatology, 1645-1668, vol. 21, DOI 10.1002/joc.703
#
# R.E. Benestad, met.no, Oslo, Norway 16.04.2002
# rasmus.benestad@met.no
#------------------------------------------------------------------------


stationmap <- function(ele=101,NORDKLIM=TRUE,NACD=TRUE,silent=TRUE) {

# Load libraries, and compile function:
  
#source("getnordklim.R")
#source("getnacd.R")
#source("addland.R")
#source("strip.R")

ele.c<-switch(as.character(ele),
              '101'='mean T(2m)',
              '111'='mean maximum T(2m)',
              '112'='highest maximum T(2m)',
              '113'='day of Th date Thd',
              '121'='mean minimum T(2m)',
              '122'='lowest minimum T(2m)',
              '123'='day of Tl date Tld',
              '401'='mean SLP',
              '601'='monthly accum. precip.',
              '602'='maximum precip.',
              '701'='Number of days with snow cover (> 50% covered) days dsc',
              '801'='Mean cloud cover % N',
              '901'='mean snow depth')

#nacd.meta<-read.table('data/appendix.2')
#nordklim.meta<-read.fwf( 'data/nordklim_station_catalogue_v1_0.prn',
#                        skip=1,width=c(2,30,12,11,11,4,3,2,4,3,2),
#                        col.names=c("i","location","height","country",
#                                    "number","Lat.deg","Lat.min","N.S",
#                                    "Lon.deg","Lon.min","E.W"))
data(nacd.meta)
data(nordklim.meta)

#-------------------------------------------------------------------
# Selection of NACD and NORDKLIMstations.

if ((NORDKLIM) & (NACD)) { 
 locs<-c(as.character(nacd.meta$V5[is.element(nacd.meta$V14,ele)]),
         strip(as.character(meta$location)))
} else if (NORDKLIM) locs<-as.character(strip(as.character(meta$location))) else
       if (NACD) locs<-as.character(nacd.meta$V5[is.element(nacd.meta$V14,ele)])

plot(c(-80,40),c(50,82),type="n",
     main=ele.c,xlab="Longitude",ylab="Latitude")
addland()
if (!silent) print(locs)

for (loc in locs) {
#  print("NACD:")
  if ((NORDKLIM) & (NACD)) { 
    obs.nacd<-getnacd(loc,silent=TRUE)
    if (obs.nacd$found) {
      points(obs.nacd$lon,obs.nacd$lat,col="blue",pch=20,cex=1.25)
    }
#     print("NordKlim")
    obs.nork<-getnordklim(loc,silent=TRUE)
    if (obs.nork$found) {
        points(obs.nork$lon,obs.nork$lat,col="red",pch=20,cex=0.8)
    }
  } else if (NORDKLIM) {
    obs.nork<-getnordklim(loc,silent=TRUE)
      if (obs.nork$found) {
        points(obs.nork$lon,obs.nork$lat,col="red",pch=20,cex=0.8)
      }
  } else if (NACD) {
    obs.nacd<-getnacd(loc,silent=TRUE)
    if (obs.nacd$found) {
      points(obs.nacd$lon,obs.nacd$lat,col="blue",pch=20,cex=1.25)
    }
  }
}
grid()
legend(-75,55,c('NACD','NordKlim'),pch=c(20,20),
       col=c('blue','red'),bg="grey95")

}
# Strips the strings by cutting off at the first space
# R.E. Benestad, Oslo, Norway, April 19.04.2002.
# met.no

strip<-function(in.str) {
  
  lfac<-FALSE                           # Set flag if we are dealing with a factor
                                        # object. Then the output is converted to
                                        # factor.
  if (is.factor(in.str)) { lfac <- TRUE }
  in.str<-as.character(in.str)
  
#  print(in.str)
  out.str<-in.str

# Go through list of a string array and remove the remainder of the string
# starting at the first space character...
  
  for (is in 1:length(in.str)) {
    c.str<-paste(unlist(strsplit(in.str[is],"")),sep="")
#    print(c.str)
    ispc<-pmatch(" ",c.str)
#    print(ispc)
    if (!is.na(ispc) & (ispc > 1)) {
      out.str[is]<-substr(in.str[is],1,ispc-1)
    } 
  }
  if (lfac) {
    out.str<-factor(out.str)
  }
  strip<-out.str
  strip
}
# Returns the upper case of a string;
# Test:
# > upper.case("abcdefghijklemnoprstuvwxyz1234567890")
# [1] "ABCDEFGHIJKLEMNOPRSTUVWXYZ1234567890"
# > upper.case("ABCDEFGHIJKLEMNOPRSTUVWXYZ1234567890")
# [1] "ABCDEFGHIJKLEMNOPRSTUVWXYZ1234567890"
# R.E. Benestad (REB), DNMI (met.no), 08.01.2002
#                REB:  03.05.2002 - modified to hande string arrays

upper.case <- function(u.case) {

  lfac<-FALSE                           # Set flag if we are dealing with a factor
                                        # object. Then the output is converted to
                                        # factor.
  
  if (is.factor(u.case)) { lfac <- TRUE }
  
  str<-as.character(u.case)
#  print(str)  
  upper.case<-str

  for (is in 1:length(str)) {
    nc<-nchar(str[is])
    upper.case[is]<-""
    for (ic in 1:nc) {
      sstr<-substr(str[is],ic,ic)
      u.l<-switch(as.character(sstr),
                      a="A",b="B",c="C",d="D",f="F",e="E",g="G",h="H",i="I",
                      j="J",k="K",l="L",m="M",n="N",o="O",p="P",q="Q",r="R",
                      s="S",t="T",u="U",v="V",w="W",x="X",y="Y",z="Z")
      if (length(u.l) == 0) u.l<-sstr
#    print(c(sstr,u.l,upper.case))
      upper.case[is]<-paste(upper.case[is],u.l,sep="")
    }
  }
  if (lfac) {
    upper.case<-factor(upper.case)
  }
  upper.case
}
  
what.data <- function() {
  print("NACD: www.dmi.dk/f+u/publikation/tekrap/2001/Tr01-11.pdf")
  print("Nordklim: http://www.smhi.se/hfa_coord/nordklim/")
  print("NCEP reanalysis: http://www.cdc.noaa.gov/")
}
