\name{diamond.base}
\alias{diamond.base}
\title{Create a base diamond}
\description{
  Generates vertices of a portrait format diamond with center at
  \code{(x, y)} and short axis \code{d}.
}
\usage{
  diamond.base (x=0, y=0, a=1)
}
\arguments{
  \item{x}{x coordinate of diamond center}
  \item{y}{y coordinate of diamond center}
  \item{a}{length of short axis of diamond}
}
\details{
  This function is normally called first to establish a base diamond
  from which recursive partitions are derived.  The partitions are
  created from 4-fold subdivision into \code{4^(d-1)} congruent
  sub-diamonds.

  A "diamond" in this package means the figure formed by two equilateral
  triangles connected across a common edge, or, two adjacent faces of
  an octahedron or icosahedron.

  Recursion in this package limits the useful size of problem.  A version
  with C routines is in development.
}
\value{
  A list with four components, each a pair of \code{(x, y)} coordinates
  describing a vertex of the diamond.
}
\references{
  White, D., (2000) Global grids from recursive diamond subdivisions
  of the surface of an octahedron or icosahedron, \emph{Environmental
  Monitoring and Assessment}, \bold{64}(1), 93-103.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()

  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4))

  # kites for levels 2-4
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4), lwd=1, col="blue")
  lines (diamond.triedges (base, 4), lwd=1, col="blue")
  lines (diamond.dualedges (base, 4), lwd=1, col="blue")
  lines (diamond.edges (base, 3), lwd=2, col="green")
  lines (diamond.triedges (base, 3), lwd=2, col="green")
  lines (diamond.dualedges (base, 3), lwd=2, col="green")
  lines (diamond.edges (base, 2), lwd=3, col="red")
  lines (diamond.triedges (base, 2), lwd=3, col="red")
  lines (diamond.dualedges (base, 2), lwd=3, col="red")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("'Kites' for Levels 2-4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.centers}
\alias{diamond.centers}
\title{Coordinates of centers of diamonds}
\description{
  Creates \code{x} and \code{y} coordinates of center points of
  a diamond partition, suitable for \code{\link{points}}.
}
\usage{
  diamond.centers (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  points (diamond.centers (base, 4), pch=19, col="red")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("Diamond Centers at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.dualcents}
\alias{diamond.dualcents}
\title{Coordinates of centers of hexagons}
\description{
  Creates \code{x} and \code{y} coordinates of center points of
  the hexagon duals of a diamond partition, suitable for
  \code{\link{points}}.
}
\usage{
  diamond.dualcents (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  Centers are generated for "left-side" hexagons, assuming that
  diamonds are joined as in octahedron or icosahedron.  See reference.
}
\value{
  List with \code{x} and \code{y} components.
}
\references{
  White, D., (2000) Global grids from recursive diamond subdivisions
  of the surface of an octahedron or icosahedron, \emph{Environmental
  Monitoring and Assessment}, \bold{64}(1), 93-103.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.dualedges (base, 4), col="blue")
  points (diamond.dualcents (base, 4), pch=19, col="red")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("Centers of Dual Hexagons at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.dualedges}
\alias{diamond.dualedges}
\title{Coordinates of edges of hexagons}
\description{
  Creates \code{x} and \code{y} coordinates of edges of
  the hexagon duals of a diamond partition, suitable for
  \code{\link{lines}}.
}
\usage{
  diamond.dualedges (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.dualedges (base, 4), col="blue")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("Dual Hexagons at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.edges}
\alias{diamond.edges}
\title{Coordinates of edges of diamonds}
\description{
  Creates \code{x} and \code{y} coordinates of edges of
  a diamond partition, suitable for \code{\link{lines}}.
}
\usage{
  diamond.edges (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4))
  lines (diamond.edges (base, 1), lwd=2)
  title ("Diamond Partition at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.hierwalk}
\alias{diamond.hierwalk}
\title{Coordinates of hierarchical walk}
\description{
  Creates \code{x} and \code{y} coordinates of paths connecting
  centers, in hierarchical order, of a diamond partition, suitable
  for \code{\link{lines}}.
}
\usage{
  diamond.hierwalk (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  The walk connects diamonds in the partition in the order of the
  labels generated by \code{\link{diamond.labels}}.  That is, the
  walk describes the Morton or Peano sequence of traverse across the
  partition.
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{diamond.labels}},
  \code{\link{diamond.randwalk}}
}
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  points (diamond.centers (base, 4), pch=19, col="red")
  lines (diamond.hierwalk (base, 4), lwd=1, col="blue")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("Hierarchical Walk at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.labels}
\alias{diamond.labels}
\title{Labels for a diamond partition}
\description{
  Creates a vector of character labels for diamonds in a
  partition.
}
\usage{
  diamond.labels (d)
}
\arguments{
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  The labels are generated from a quadrant-recursive ordering,
  see reference.
}
\value{
  A vector of character labels for each diamond in the partition.
}
\references{
  White, D., (2000) Global grids from recursive diamond subdivisions
  of the surface of an octahedron or icosahedron, \emph{Environmental
  Monitoring and Assessment}, \bold{64}(1), 93-103.
}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{diamond.hierwalk}},
  \code{\link{diamond.randlabels}}
}
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4), lwd=1, col="blue")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  text (diamond.centers (base, 4), diamond.labels (4))
  title ("Morton/Peano Numbers at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.plot}
\alias{diamond.plot}
\title{Define a fixed aspect ratio plot}
\description{
  Calls \code{\link{plot}} to create a plot space whose dimensions
  are defined by the parameters, and whose aspect ratio is fixed
  with respect to them.
}
\usage{
  diamond.plot (cl)
}
\arguments{
  \item{cl}{a list of coordinates with components \code{x} and
    \code{y}, such as returned by \code{\link{diamond.edges}}}
}
\details{
  This function calls \code{\link{plot}} with \code{type="n"}, and it
  maintains the aspect ratio of the input coordinates in the output
  graphic by setting \code{par(pin=)}, thus the map does not scale
  dynamically in an on-screen window.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4))
  lines (diamond.edges (base, 1), lwd=2)
  title ("Diamond Partition at Level 4", line=0) }
\keyword{hplot}
\eof
\name{diamond.randlabels}
\alias{diamond.randlabels}
\title{Labels for a hierarchically randomized partition}
\description{
  Creates a vector of character labels for diamonds in a
  partition, where the labels are randomly permuted.
}
\usage{
  diamond.randlabels (d)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  The labels are generated from a quadrant-recursive ordering,
  then randomly permuted at each level in the recursive hierarchy,
  see reference.
}
\value{
  A vector of the hierarchically randomized character labels for
  each diamond in the partition.
}
\references{
  White, D., (2000) Global grids from recursive diamond subdivisions
  of the surface of an octahedron or icosahedron, \emph{Environmental
  Monitoring and Assessment}, \bold{64}(1), 93-103.
}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{diamond.randwalk}},
  \code{\link{diamond.labels}}
}
\examples{
  diamond.randlabels (2)
}
\keyword{dplot}
\eof
\name{diamond.randpts}
\alias{diamond.randpts}
\title{Coordinates of random points}
\description{
  Creates \code{x} and \code{y} coordinates of random points in
  each diamond element of a diamond partition, suitable
  for \code{\link{points}}.
}
\usage{
  diamond.randpts (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4), lwd=1, col="blue")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  points (diamond.randpts (base, 4), pch=19, col="red")
  title ("Random Points at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.randwalk}
\alias{diamond.randwalk}
\title{Coordinates of hierarchical random walk}
\description{
  Creates \code{x} and \code{y} coordinates of paths connecting
  centers, in random hierarchical order, of a diamond partition,
  suitable for \code{\link{lines}}.
}
\usage{
  diamond.randwalk (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  The walk connects diamonds in the partition in the order of the
  labels generated by \code{\link{diamond.labels}}, except that,
  at each hierarchical level, the four elements are randomly permuted.
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{
  \code{\link{diamond.randlabels}},
  \code{\link{diamond.hierwalk}}
}
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  points (diamond.centers (base, 4), pch=19, col="red")
  lines (diamond.randwalk (base, 4), lwd=1, col="blue")
  lines (diamond.edges (base, 1), lwd=2, col="black")
  title ("Hierarchical Random Walk at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamond.triedges}
\alias{diamond.triedges}
\title{Coordinates of edges of triangles}
\description{
  Creates \code{x} and \code{y} coordinates of bisecting edges of
  diamonds in partition, providing for two equilateral triangles,
  with output suitable for \code{\link{lines}}.
}
\usage{
  diamond.triedges (b, d)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond, as from 
    \code{\link{diamond.base}}}
  \item{d}{depth of recursion of partition; if \code{d=1},
    then no partition}
}
\details{
  Both this function and \code{\link{diamond.edges}} are necessary
  to show triangle partitions.  See example.
}
\value{
  List with \code{x} and \code{y} components.
}
\author{Denis White, \email{white.denis@epa.gov} }
\examples{
  base <- diamond.base ()
  diamond.plot (diamond.edges (base, 1))
  lines (diamond.edges (base, 4))
  lines (diamond.triedges (base, 4))
  title ("Triangles at Level 4", line=0)
}
\keyword{dplot}
\eof
\name{diamonds}
\alias{diamonds}
\alias{subdiamonds}
\alias{hexlines}
\alias{hiercenters}
\alias{subcenters}
\alias{hierlabels}
\alias{deep}
\alias{labeldeep}
\alias{centerdeep}
\alias{randlevel}
\alias{ranptdia}
\title{Internal functions for diamonds package}
\description{
  Various functions for internal use in the diamonds package.
}
\usage{
  diamonds(b, d)
  subdiamonds (b)
  hexlines (b, d)
  hiercenters (b, d)
  subcenters (b) 
  hierlabels (parent, level, d)
  deep (b, level, d)
  labeldeep (dt, parent, level, d)
  centerdeep (b, level, d)
  randlevel (tl, level, d)
  ranptdia (b)
}
\arguments{
  \item{b}{list of coordinates of vertices of a diamond}
  \item{d}{final depth of recursion of partition}
  \item{level}{current depth of recursion}
  \item{parent}{label of next higher level diamond}
  \item{dt}{list of labels being build recursively}
  \item{tl}{list of diamonds}
}
\details{See R codes.}
\author{Denis White, \email{white.denis@epa.gov} }
\seealso{\code{\link{diamond.base}}}
\keyword{dplot}
\eof
