\name{00Intro.dse1}
\alias{00Intro.dse1}
\title{DSE}
\center{Dynamic Systems Estimation - Multivariate Time Series Package}
\synopsis{}

\section{Usage}{
   library("dse1") 
   library("dse2") 
}


\description{
This is a library of functions for time series modeling.  The library 
works with the S (Splus) and R languages. A "Brief User's Guide" is available 
at \url{www.bank-banque-canada.ca/pgilbert}  
This help and other information is also available at that web location.

The library implements an object oriented approach to time series
modeling. This means that different
model and data representations can be implemented with fairly simple 
extensions to the library. 

The library includes multi-variate state 
space and ARMA (including VAR) models. The library also implements 
Troll models as an example of another class of model. (Troll models are run by completely 
separate software, Troll from Intex Inc, as if they were an integral part 
of the library. Models and data are passed back and forth to Troll and the
results can be analyzed with already existing functions in the
library. Note: the Troll interface is broken in the current version.)

The library includes methods for simulating, estimating, and converting 
among different
model representations. These form the basic part of the library. Methods for
studying estimation techniques and for examining the forecasting properties
of models are also documented in the User's Guide.

There are also functions for forecasting and for
evaluating the performance of forecasting models as well as 
functions for evaluating model estimation techniques.

DSE requires libraries tframe and setRNG. The first provides a kernel of 
functions for 
programming time series methods relatively independently of the representation 
of time. The second provides some utilities for setting and resetting the random
number generator and normal transformations, and for generating the same random
numbers in S and R differences. For programmers, these libraries may of interest 
by themselves. Tframe is intended to make it easier to write code which can use any new/better 
time representations when they appear. It also provides plotting, time 
windowing, and some other utility functions which are specifically intended 
for time series.

Relative to commercial packages the library is especially useful for
time series research (such as studying estimation methods). 
For usual time series applications there may be commercial
packages which are preferable.

The functions described in the
Brief User's Guide should work fairly reliably, however, many of the functions
described in the help facility are still under development and may not
work. In addition, there may be functions described in the help
facility for which the code is not yet distributed. This is a compromise which 
allows me to make the software
available with minimum effort. This software is not a commercial
product. It is the by-product of an ongoing research effort.
Constructive suggestions and comments are welcomed. I can be reached at
\email{pgilbert@bank-banque-canada.ca} or by phone at (613) 782-7346.

There is also a mailing list available which I hope will let users help 
one another, as I cannot always be available. Please subscibe if you use
the library extensively, especially if you can help other users.
To subscribe send mail to 
\email{boc\_list@bank-banque-canada.ca}
with the body line \code{subscribe boc\_dse Your Name}
}

\section{MainObjects}{
The main objects are classes "TSdata", "TSmodel", and  "TSestModel". 
These are, respectively,  time series data objects, models, and objects with 
a model, data and some estimation information. For each of these there are 
several sub classes.
}

\seealso{
\code{\link{TSdata}},
\code{\link{TSmodel}},
\code{\link{TSestModel.object}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{ARMA}
\alias{ARMA}
\alias{is.ARMA}

\title{ARMA Model Constructor}
\description{Constructs an ARMA TSmodel object as used by the DSE package.}
\usage{
    ARMA(A=NULL, B=NULL, C=NULL, TREND=NULL, description=NULL,
          names=NULL, input.names=NULL, output.names=NULL)
    is.ARMA(obj)
}
\arguments{
    \item{A}{The auto-regressive polynomial, an axpxp array.}
    \item{B}{The moving-average polynomial, an bxpxp array.}
    \item{C}{The  input polynomial, an cxpxm array. C should be NULL if 
    there is no input}
    \item{TREND}{A vector of length a.}
    \item{description}{An arbitrary string.}
    \item{names}{A list with elements input and output, each a vector of 
       strings. Arguments input.names and output.names should not be used if
       argument names is used.}
    \item{input.names}{A vector of strings. }
    \item{output.names}{A vector of strings.}
    \item{obj}{Any object.}
}
\value{An ARMA TSmodel}
\details{ 
The ARMA model is defined by:

      Ay =  Bw + Cu
      
sometimes written

      A(L)y(t) =  B(L)w(t) + C(L)u(t)

where
    \item{A}{(axpxp) is the auto-regressive polynomial array.}
    \item{B}{(bxpxp) is the moving-average polynomial array.}
    \item{C}{
    (cxpxm) is the  input polynomial array. C should be NULL if 
    there is no input}
    \item{y}{is the p dimensional output data.}
    \item{u}{is the m dimensional control (input) data.}

}

\seealso{
    \code{\link{TSmodel}}
}
\examples{
    mod1 <- ARMA(A=array(c(1,-.25,-.05), c(3,1,1)), B=array(1,c(1,1,1)))
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{DSECOMPILED}
\alias{.DSECOMPILED}
\alias{.DSEDUP}
\title{Use Compiled or S/R Code}
\description{
    Determines if compiled code should be used or not.
    }
\usage{
    .DSECOMPILED <- TRUE
    .DSECOMPILED <- FALSE
    .DSEDUP <- TRUE
    .DSEDUP <- FALSE
}

\sideffects{
    Several DSE functions which call compiled fortran or C code will use
    the equivalent S/R version if the .DSECOMPILED global variable is set 
    to FALSE. Its main purpose is for debugging. .DSEDUP is passed to
    fortran calls and in R controls whether R duplicates arguments passed
    to the fortran code. The safe setting is TRUE. Setting FALSE saves some
    memory but does not seem to give much speed gain. 
}
\examples{
    .DSECOMPILED <- TRUE
    .DSEDUP <- TRUE
}
%\keyword{DSE}
\keyword{programming}



\eof
\name{DSEutilities}
\alias{DSE.ar}
\alias{criteria.table.heading}
\alias{criteria.table.legend}
\alias{criteria.table.nheading}
\alias{dseclass}
\alias{dseclass<-}
\alias{estVARXmean.correction}
\alias{fake.TSestModel.missing.data}
\alias{printTestValue}
\alias{dsescan}
\alias{read.int}
\alias{svd.criteria}

\title{DSE Utilities}
\description{
    These functions are used by other functions in DSE and are not
    typically called directly by the user.}
\usage{
   estVARXmean.correction(X, y, bbar, fuzz=sqrt(.Machine$double.eps), warn=TRUE)
   fake.TSestModel.missing.data(model,data, residual, max.lag)   
   printTestValue(x, digits=16)
   read.int(prmt)
   svd.criteria(sv)
   criteria.table.heading()
   criteria.table.legend()
   criteria.table.nheading()
   DSE.ar(data, ...)
}
%\keyword{DSE}
\keyword{internal}

\eof
\name{DSEversion}
\alias{DSEversion}
\title{Print Version Information}
\description{Print version information.}
\usage{
    DSEversion()
}
\examples{DSEversion()}
%\keyword{DSE}
\keyword{ts}



\eof
\name{McMillanDegree}
\alias{McMillanDegree}
\alias{McMillanDegree.ARMA}
\alias{McMillanDegree.SS}
\alias{McMillanDegree.TSestModel}

\title{Calculate McMillan Degree}
\description{

Calculate the McMillan degree of an ARMA TSmodel.
}
\usage{
    McMillanDegree(model,  ...)
    \method{McMillanDegree}{ARMA}(model, fuzz=1e-4, verbose=TRUE, warn=TRUE, ...)
    \method{McMillanDegree}{SS}(model, fuzz=1e-4, ...)
    \method{McMillanDegree}{TSestModel}(model,  ...)
}
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{fuzz}{Roots within fuzz distance are counted as equivalent.}
    \item{verbose}{If TRUE roots are printed.}
    \item{warn}{If FALSE then warnings about unit roots added for TREND are not printed.}
    \item{...}{arguments to be passed to other methods.}
}
\value{
A list with elements gross and distinct containing all roots 
and distinct roots.
}
\sideffects{
The number of roots and distinct roots is printed if verbose is TRUE.
}
\seealso{

\code{\link{stability}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
McMillanDegree(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{MittnikReducedModels}
\alias{MittnikReducedModels}
\title{Reduced Models via Mittnik SVD balancing}
\description{Reduced Models via Mittnik SVD balancing.}
\usage{
    MittnikReducedModels(largeModel)
}
\arguments{
    \item{largeModel}{An SS TSmodel.}
}
\value{A list of state space TSmodels with smaller state dimensions.}
\details{
The largeModel is balanced by the SVD method promoted by Mittnik (see 
MittnikReduction) and then models for every state dimension up to the 
state dimension of the largeModel are return. Note that this procedure 
does not result in smaller models which are balanced.
}
\seealso{
\code{\link{MittnikReduction}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    z <-  MittnikReducedModels(toSS(estVARXls(eg1.DSE.data.diff))) 
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{MittnikReduction}
\alias{MittnikReduction}
\alias{MittnikReduction.from.Hankel}

\title{Balance and Reduce a Model}
\description{
Balance and reduce the state dimension of a state space model a la Mittnik.
}
\usage{
    MittnikReduction(model, data=NULL, criterion=NULL, verbose=TRUE,warn=TRUE)
    MittnikReduction.from.Hankel(M, data=NULL, nMax=NULL, 
       criterion=NULL, verbose=TRUE, warn=TRUE, 
       Spawn=if (exists(".SPAWN")) .SPAWN else FALSE)
}
\arguments{
    \item{model}{An object of class TSmodel or TSestModel.}
    \item{data}{
      If the supplied model is of class TSestModel and data is not supplied
      then it is taken from the model. If the model is of class TSmodel
      then data must be supplied.}
    \item{criterion}{Criterion to be used for model 
       selection. see \code{informationTestsCalculations}.}
    \item{verbose}{logical indicating if information should be printed 
       during estimation.}
    \item{warn}{logical indicating if some warning messages should be suppressed.}
    \item{M}{a matrix. See details.}
    \item{nMax}{integer indicating the state dimension of the largest model
       considered.}
    \item{Spawn}{logical indicating if Splus For loops should be used.}
}
\value{
A state space model balance a la Mittnik in an object of class TSestModel.
}
\details{
\code{MittnikReduction} gives nested-balanced state space model using
reduction by svd of the Hankel matrix generated from a model.
If a state space model is supplied the max. state dimension for the result is
taken from the model. If an ARMA model is supplied then singular values 
will be printed and the program prompts for the max. state dimension.
criterion should be the name of one of the values returned by informationTests,
that is, one of ("port","like","aic","bic","gvc","rice","fpe","taic", 
"tbic","tgvc","trice","tfpe"). If criteria is not specified then the 
program prompts for the state dimension (n) to use for the returned model.
The program requires data to calculate selection criteria. (The program
balanceMittnik calculates svd criteria only and can be used for reduction
without data.) 

The function \code{MittnikReduction.from.Hankel} is called by
\code{MittnikReduction} and typically not by the user, but there are situations
when the former might be called directly. It selects a reduced state 
space model by svd a la Mittnik. Models and several criteria for all 
state dimensions up to the max. state dim. specified are calculated. 
(If nMax is not supplied then svd criteria are printed and the program 
prompts for nMax). The output dimension p is taken from nrow(M).
M is a matrix with p x (m+p)  blocks giving the markov parameters,
that is, the first row of the Hankel matrix. It can be generated from the
model as in the function markovParms, or from the data, as in the function
estSSMittnik.

\code{data} is necessary only if criteria (AIC,etc) are to be calculated.
}
\references{
    Gilbert, P. D. (1993) State space and ARMA models: An overview of
    the equivalence. Working paper 93-4, Bank of Canada. Available at
    <www.bank-banque-canada.ca/pgilbert>

    Gilbert, P. D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" \emph{J. of Forecasting: 
    Special Issue on VAR Modelling}. 14:229-250.

    Mittnik, S. (1989), Multivariate Time Series Analysis With State 
    Space Models, \emph{Computers Math Appl.} Vol 17, No 8/9, pp1189-1201.
   
    Mittnik, S. (1990), Macroeconomic Forecasting Experience With 
    Balance State Space Models,  
    \emph{International Journal Of Forecasting}, Vol 6, pp337-348.
  
    Mittnik, S. (1990), Forecasting With Balanced State Space 
    Representations of Multivariate Distributed Lag Models. 
    \emph{J. of Forecasting}, Vol.9, 207-218.  
}
\seealso{
    \code{\link{estVARXls}}
    \code{\link{bft}}
    \code{\link{balanceMittnik}}
    \code{\link{informationTests}}
    \code{\link[dse1]{informationTestsCalculations}}
}
\examples{
    if(is.R()) data("egJofF.1dec93.data", package="dse1")
    model <- toSS(estVARXls(egJofF.1dec93.data))
    newmodel <-MittnikReduction(model, criterion="taic")
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{Polynomials}
\alias{characteristicPoly}
\alias{companionMatrix}
\alias{polyvalue}
\alias{polydet}
\alias{polyprod}
\alias{polysum}
\alias{polyrootDet}
        
\title{Polynomial Utilities}
\description{Polynomial utility functions used by DSE.}
\usage{
    characteristicPoly(a)
    companionMatrix(a)
    polyvalue(coef, z)
    polydet(a)
    polyprod(a, b)
    polysum(a, b)
    polyrootDet(a)
}
\arguments{
    \item{a}{An array representing a matrix polynomial.}
    \item{b}{An array representing a matrix polynomial.}
    \item{coef}{Coefficients of a polynomial.}
    \item{z}{Value at which the polynomial is to be evaluated.}
}
\value{depends}
\details{
    These are utility functions used in some ARMA model calculations
    such as root and stability calculations.}
\seealso{
    \code{\link{polyroot}}
    \code{\link{roots}}
    \code{\link{stability}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{Portmanteau}
\alias{Portmanteau}
\title{Calculate Portmanteau statistic}
\description{Calculate Portmanteau statistic.}
\usage{
    Portmanteau(res)}
\arguments{
    \item{res}{A matrix with time-series residuals in columns.}
}
\seealso{
\code{\link{informationTests}}
}
\examples{
if(is.R()) require("ts")
Portmanteau(matrix(rnorm(200), 100,2)) # but typically with a residual
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{Riccati}
\alias{Riccati}
\title{Riccati Equation}
\description{Solve a Matrix Riccati Equation}
\usage{
    Riccati(A, B, fuzz=1e-10, iterative=FALSE)
}
\arguments{
    \item{A}{A matrix.}
    \item{B}{A matrix.}
    \item{fuzz}{The tolerance used for testing convergence.}
    \item{iterative}{If TRUE an iterative solution technique is used.}
}
\value{xxx}
\details{
    Solve  Riccati equation  P = APA' + B 
    by eigenvalue decompostion of a symplectic matrix or by iteration.
}
\references{
    R. J. Vaccaro and T.Vukina(1993),
      "A Solution to the Positivity Problem in the State-Space Approach 
      to Modeling Vector-Valued Time Series." 
      \emph{Journal of Economic Dynamics and Control}, 17,
    Anderson & Moore, (1979) sec 6.7,  
    D. Vaughan (1970) \emph{IEEE Tr AC}.
    A. Laub (1983), \emph{Proc IEEE conf Decision and Control}. 
    Gudmundsson, Kenney and Laub (1992) \emph{IEEE Tr AC}.
}
\seealso{
    \code{\link{eigen}}
}
\note{This procedure has not been tested.}

%\keyword{DSE}
\keyword{ts}
\keyword{algebra}


\eof
\name{SS}
\alias{SS}
\alias{is.SS}
\alias{is.innov.SS}
\alias{is.nonInnov.SS}

\title{State Space Models}
\description{Construct a }
\usage{
    SS(F.=NULL, G=NULL, H=NULL, K=NULL, Q=NULL, R=NULL, z0=NULL, P0=NULL,
             description=NULL, names=NULL, input.names=NULL, output.names=NULL)
    is.SS(obj)
    is.innov.SS(obj)
    is.nonInnov.SS(obj)
}
\arguments{
    \item{F.}{(nxn) state transition matrix.}
    \item{H}{(pxn) output matrix.}
    \item{Q}{(nxn) matrix specifying the system noise distribution.}
    \item{R}{(pxp) matrix specifying the output (measurement) noise distribution.}
    \item{G}{(nxp) input (control) matrix. G should be NULL if there is no input.}
    \item{K}{(nxp) matrix specifying the Kalman gain.}
    \item{z0}{vector indicating estimate of the state at time 0.
       Set to zero if not supplied.}
    \item{P0}{matrix indicating initial tracking error P(t=1|t=0).
       Set to I if not supplied.}
    \item{description}{String. An arbitrary description.}
    \item{names}{A list with elements input and output, each a vector of 
       strings. Arguments input.names and output.names should not be used if
       argument names is used.}
    \item{input.names}{
      A vector of character strings indicating input variable names.
    }
    \item{output.names}{
      A vector of character strings indicating output variable names.
    }
    \item{obj}{an object.}
}
\value{An SS TSmodel}
\details{
State space models have a
further sub-class: innov or non-innov, indicating an innovations form
or a non-innovations form.  

The state space (SS) model is defined by:

      z(t) =Fz(t-1) + Gu(t) + Qe(t)
      
      y(t) = Hz(t)  + Rw(t)

or the innovations model:

       z(t) =Fz(t-1) + Gu(t) + Kw(t-1)
       
       y(t) = Hz(t)  + w(t)

Matrices are as specified above in the aguments, and 
    \item{y}{is the p dimensional output data.}
    \item{u}{is the m dimensional exogenous (input) data.}
    \item{z}{is the n dimensional (estimated) state at time t,  
    E[z(t)|y(t-1), u(t)] denoted E[z(t)|t-1]. An initial value for z can 
    be specified as z0 and an initial one step ahead state tracking 
    error (for non-innovations models) as P0.}
    \item{z0}{An initial value for z can be specified as z0.}
    \item{P0}{An initial one step ahead state tracking error (for 
    non-innovations models) can be specified as P0.}  
    \item{K, Q, R}{
    For sub-class innov the Kalman gain K is specified but not Q and R.
    For sub-class non-innov Q and R are specified but not the Kalman gain K.}
    \item{e and w}{are typically assumed to be white noise in the 
    non-innovations form, in which case
    the covariance of the system noise is QQ' and the covariance of 
    the measurement noise is RR'. The covariance of e and w  can be specified 
    otherwise in the simulate 
    method \code{simulate.SS} for this class of model, but the assumption is
    usually maintained when estimating models of this form (although, not by all
    authors).
    
    Typically, an non-innovations form is harder to identify than an innovations
    form. Non-innovations form would typically be choosen when there is
    considerable theoretical or physical knowledge of the system (e.g. the
    system was built from known components with measured physical values).
    
    By default, elements in parameter matrices are treated as constants if they
    are exactly 1.0 or 0.0, and as parameters otherwise. A value of 1.001 would
    be treated as a parameter, and this is the easiest way to initialize an
    element which is not to be treated as a constant of value 1.0. Any matrix
    elements can be fixed to constants of any value using fixConstants.  
    }
}

\reference{B. D. O.Anderson & J. B.Moore \emph{Optimal Filtering} p.39,44.}

\seealso{
    \code{\link{TSmodel}}
    \code{\link{ARMA}}
    \code{\link{simulate.SS}}
    \code{\link{l.SS}}
    \code{\link{fixConstants}}
}
\examples{
    f <- array(c(.5,.3,.2,.4),c(2,2))
    h <- array(c(1,0,0,1),c(2,2))
    k <- array(c(.5,.3,.2,.4),c(2,2))
    ss <- SS(F=f,G=NULL,H=h,K=k)
    is.SS(ss)
    ss
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{TSdata}
\alias{TSdata}
\alias{TSdata.default}
\alias{TSdata.TSdata}
\alias{TSdata.TSestModel}
\alias{is.TSdata}
\alias{as.TSdata}

\title{Construct TSdata time series object}
\description{
    Constructor for generating or extracting a "TSdata" object 
    containing data for use by TSmodels.
}
\usage{
    TSdata(data=NULL, ...)
    \method{TSdata}{default}(data=NULL, input=NULL, output=NULL, ...)
    \method{TSdata}{TSdata}(data, ...)
    \method{TSdata}{TSestModel}(data, ...)
    is.TSdata(obj)
    as.TSdata(d)
}
\arguments{
    \item{data}{object of class TSdata, TSestModel, matrix, list with input
       and output matrices, or another object for which a constructor or 
       TSdata extraction method has been defined.}
    \item{input}{a matrix of time series data.}
    \item{output}{a matrix of time series data.}
    \item{...}{arguments to be passed to other methods.}
    \item{obj}{an object.}
    \item{d}{an object from which a TSdata object can be extracted. See below.}
}
\details{
    Generic method to construct or extract a TSdata object. The default 
    method constructs a TSdata object. Specific methods extract the TSdata
    from other objects (which must contain TSdata).
    The function \code{is.TSdata(data)} returns TRUE is data inherits 
    from "TSdata" and FALSE otherwise.

    The function \code{as.TSdata} uses the elements input and output 
    directly and
    strips away other class information and parts of the object (and does 
    not make use of inputData(data) or outputData(data) which may 
    do something special for certain classes.
}
\examples{
   rain  <- matrix(rnorm(86*67), 86,17)
   radar <- matrix(rnorm(86*5),  86,5)
   mydata <- TSdata(input=radar, output=rain)
}
\seealso{
\code{\link{TSdata.object}},
\code{\link[tframe]{freeze}},
\code{\link[dse2]{freeze.TSPADIdata}},
\code{\link{TSmodel}},
\code{\link{TSestModel.object}}
\code{\link[dse2]{setTSPADIdata}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{TSdata.object}
\alias{TSdata.object}
\title{time series data object}
\description{
Class "TSdata" of time series data objects for use with TSmodels.
}

\section{Generation}{
This class of objects is returned by specific methods of the 
function TSdata or can be built according to the description below.
}

\section{Methods}{
The \code{TSdata} class of objects has methods for the generic
functions \code{print, plot, periods, start, end, ..., testEqual, seriesNames,
seriesNamesInput, seriesNamesOutput}
Also, the function \code{is.TSdata} is supported.
}

\section{Inheritance}{
Other data classes inherit from the class \code{TSdata}.}

\section{Structure}{
Objects are a list with class the most general class "TSdata". The
native form for this library has elements \code{input} and
\code{output}. Any other elements are ignored. \code{input} and
\code{output} are matrices (or tframe or time series matrices) of the
input and output data, with each series in a column.  \code{TSPADIdata}
objects inherit from this class but have a somewhat different
structure.  \code{TSPADIdata} makes it possible to retrieve data from
an external database when it is needed. These subclass objects do not
contain the actual data, but only the names of the series and the data
base where they are located. The function setTSPADIdata can be used to set
up an object of class c("TSPADIdata", "TSdata").
}

\seealso{
\code{\link{TSdata}}
\code{\link{TSmodel}}
\code{\link{TSestModel.object}}
\code{\link[dsepadi]{TSPADIdata}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{TSestModel}
\alias{TSestModel}
\alias{TSestModel.TSestModel}
\alias{TSestModel.object}
\alias{is.TSestModel}

\title{Estimated Time Series Model}
\description{Object containing a time series model, data, and
estimation information.}
\usage{
    TSestModel(obj)
    \method{TSestModel}{TSestModel}(obj)
    is.TSestModel(obj)
}
\arguments{
    \item{obj}{in the first usage an object from which a TSestModel 
       object can be extracted (or constructed).}
}
\details{
The TSestModel class of objects are generated by estimation methods. See, for
example, \code{estVARXls}. They contains a time series model (TSmodel), 
data (TSdata), and information obtained by evaluating the model 
with the data in an element called \code{estimates} containing:
 
    \item{like}{
    The negative log likelihood function value (a vector of the total, 
    constant, the det part, and the cov part)}
    \item{cov}{The estimated residual covariance.}
    \item{pred}{
    The one step ahead predictions (see predictT below). These are 
aligned with output data so that 
residuals are pred[1:sampleT,] - output[1:sampleT,]}
    \item{sampleT }{
    The end of the period (starting from 1) for which output
is used for calculating one step ahead predictions.}
    \item{predictT }{
    The end of the period for which the model is simulated.
sampleT must be less than or equal predictT. If predictT is
greater than sampleT then each step ahead beyond sampleT is
based on the prediction of the previous step and not 
corrected by the prediction error.}

The element \code{estimates} may optionally also contain and 
element \code{filter} which may have
    \item{state}{
    The one step ahead (filter) estimate of the state E[z(t)|y(t-1), u(t)].
   Note: In the case where there is no input u this corresponds to what
   would usually be called the predicted state - not the filtered state.}
    \item{track}{
    The estimated state tracking error P(t|t-1). Again note, this corresponds
to the predicted tracking eror not the filtered tracking error.
This is NULL for innovations models.}
    \item{smooth}{a list of:}
    \item{state}{
    The smoother (two sided filter) estimate of the state E[z(t)| sampleT].}
    \item{track}{
    The smoothed estimate of the state tracking error P(t|sampleT). 
    This is NULL for innovations models.}
}

\seealso{
    \code{\link{estVARXls}},
    \code{\link{TSmodel}},
    \code{\link{TSdata}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{TSmodel}
\alias{TSmodel}
\alias{TSmodel.TSmodel}
\alias{TSmodel.TSestModel}
\alias{is.TSmodel}

\title{Time Series Models}
\description{Construct or extract a "TSmodel" from objects.}
\usage{
    TSmodel(obj, ...)
    \method{TSmodel}{TSmodel}(obj, ...)
    \method{TSmodel}{TSestModel}(obj, ...)
    is.TSmodel(obj)
}
\arguments{
    \item{obj}{ 
      An object containing an object of class TSmodel or a list containing the 
      information necessary to build an object of class "TSmodel".}
    \item{...}{arguments passed to other methods.}
}
\details{
This is a generic method which will extract a "TSmodel" from an object (e.g. a
TSestModel. The default method will try to build an ARMA or state-space 
"TSmodel" from a list, which must contain the necessary information.

This class of objects is returned by estimation methods or can be built 
according to the description for specific sub-classes (eg "ARMA", "SS").

The \code{TSmodel} class of objects has methods for the generic functions
\code{print, testEqual, seriesNames, seriesNamesInput,
seriesNamesOutput, l, roots, stability,  
forecast, featherForecasts, horizonForecasts,
simulate, MonteCarloSimulations}

Also, the function \code{is.TSmodel} and the functions \code{toSS, toARMA, to.troll} are supported.
Other model classes inherit from the class \code{TSmodel}.

This class of objects contains a time series model. It is the class of
objects expected by many of the functions in this library.  

Sub-class (e.g. ARMA and SS for linear,
time-invariant ARMA and state space models.) are documented individually.
Many of the functions in this library are
designed for estimating and converting among various representations of
these types of models.
}

\seealso{
    \code{\link{ARMA}},
    \code{\link{SS}},
    \code{\link{TSestModel}},
    \code{\link{TSdata}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{acfM}
\alias{acfM}
\alias{acfM.TSdata}
\alias{acfM.TSmodel}
\alias{acfM.TSestModel}

\title{Calculate Auto-Covariance}
\description{
    Calculate a matrix with partitions [M0|...|Mi|...|Ml] giving 
    the auto-covariance.
}
\usage{
    acfM(obj, ...)
    \method{acfM}{TSdata}(obj, lag=round(6*log(periods(obj))), 
        type ="covariance", sub.mean=TRUE, ...)
    \method{acfM}{TSmodel}(obj, lag=NULL, type ="covariance", Psi=NULL, ...)
    \method{acfM}{TSestModel}(obj, ...)
}
\arguments{
    \item{obj}{An object of class TSdata or TSmodel.}
    \item{lag}{Number of lags for which to calculate the autocorrelations.}
    \item{type}{
        With the defaults the blocks are auto-covariances. 
        If type == 'correlation' the result is scaled to give 
        autocorrelations.}
    \item{sub.mean}{
        Only valid if object is of class TSdata. If FALSE then means 
        are not subtracted.}
    \item{Psi}{
        A matrix of innovation covariance. Only valid if object 
        is of class TSmodel.}
    \item{...}{arguments passed to other methods.}
}

\value{
    A matrix with partitions [M0|...|Mi|...|Ml] giving the covariance 
    or correlation, including the that between the output and input 
    series (as in the first block row of a Hankel matrix).
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    z <- acfM(eg1.DSE.data.diff)
    model <- TSmodel(toSS(estVARXls(eg1.DSE.data.diff)))
    #  z <- acfM(model) not working
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{addPlotRoots}
\alias{addPlotRoots}
\title{Add Model Roots to a plot}
\description{

Calculate and plot roots of a model.
}
\usage{
    addPlotRoots(v, pch='*', fuzz=0)
}
\arguments{

    \item{v}{An object containing a TSmodel.}
    \item{pch}{Character to use for plotting.}
    \item{fuzz}{If non-zero then roots within fuzz distance are considered equal.}
}
\sideffects{
The roots are addeded to an existing plot.
}
\value{
The eigenvalues of the state transition matrix or the inverse of the roots of the
determinant of the AR polynomial are returned invisibly.
}
\seealso{
    \code{\link{plot.roots}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
plot(roots(model))
addPlotRoots(toSS(model))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{balanceMittnik}
\alias{balanceMittnik}
\alias{SVDbalanceMittnik}

\title{Balance a state space model}
\description{Balance a state space model a la Mittnik.}

\usage{
    balanceMittnik(model, n=NULL)
    SVDbalanceMittnik(M, m, n=NULL)
}
\arguments{
    \item{model}{An TSmodel object.}
    \item{M}{a matrix. See details in \code{MittnikReduction}.}
    \item{m}{an integer indicating the number of input series in the model.}
    \item{n}{see details}
}
\value{A state space model in a TSestModel object.}
\details{
\code{balanceMittnik} calculate a state space model balance a la Mittnik.
n is intended primarily for producing a state space model from the markov
parameters of an ARMA model, but if it is supplied with an SS model the
result will be a model with state dimension n based on the n largest
singular values of the svd of a Hankel matrix of markov parameters generated 
by the original model. If n is not supplied then the singular values are
printed and the program prompts for n. \code{balanceMittnik} calls
\code{SVDbalanceMittnik}

\code{SVDbalanceMittnik} calculates a nested-balanced state space 
model by svd a la Mittnik. If state dim n is supplied then svd 
criteria are not calculated and the given n is used. Otherwise, 
the singular values are printed and the program prompts for n.
M is a matrix with p x (m+p)  blocks giving the markov parameters,
that is, the first row of the Hankel matrix. It can be generated from the
model as in the function markovParms, or from the data, as in the function
estSSMittnik.
m is the dimension of input series, which is needed to decompose M.
The output dimension p is taken from nrow(M).

See also \code{MittnikReduction} and references.
}
\references{
    See references for \code{\link{MittnikReduction}}. 
}
\seealso{
    \code{\link{estVARXls}},
    \code{\link{estVARXar}}
    \code{\link{MittnikReduction}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toSS(TSmodel(estVARXls(eg1.DSE.data.diff)))
    \dontrun{newmodel <-balanceMittnik(model)}
    # this prints information about singular values and prompts with
    #Enter the number of singular values to use for balanced model:
    # 18 might be a good choice in this example. 
    newmodel <-balanceMittnik(model, n=18)

}
%\keyword{DSE}
\keyword{ts}



\eof
\name{bestTSestModel}
\alias{bestTSestModel}
\title{Select Best Model}
\description{Select the best model.}
\usage{
    bestTSestModel(models, sample.start=10, sample.end=NULL,
     criterion='aic', verbose=TRUE)
}
\arguments{
    \item{models}{a list of TSestModels.}
    \item{sample.start}{the starting point to use for  calculating 
       information criteria.}
    \item{sample.end}{the end point to use for  calculating 
       information criteria.}
    \item{criterion}{Criterion to be used for model 
       selection. see \code{informationTestsCalculations}. 'taic' would 
       be a better default
       but this is not available for VAR and ARMA models.}
    \item{verbose}{if TRUE then additional information is printed.}
}
\value{A TSestModel}
\details{Information criteria are calculated and the best model returned.}
\seealso{
\code{\link{estBlackBox1}},
\code{\link{estBlackBox2}}
\code{\link{estBlackBox3}}
\code{\link{estBlackBox4}}
\code{\link[dse1]{informationTestsCalculations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
models <- list(estVARXls(eg1.DSE.data.diff), estVARXar(eg1.DSE.data.diff))
z <-  bestTSestModel(models)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{checkBalance}
\alias{checkBalance}
\alias{checkBalance.SS}
\alias{checkBalance.ARMA}
\alias{checkBalance.TSestModel}

\title{Check Balance of a TSmodel}
\description{
Calculate the difference between observability and reachability gramians.}
\usage{
    checkBalance(model)
    \method{checkBalance}{SS}(model)
    \method{checkBalance}{ARMA}(model)
    \method{checkBalance}{TSestModel}(model)
   }
\arguments{
    \item{model}{A TSmodel object.}
}
\value{No value is returned.}
\sideffects{
Differences  between the observability and reachability gramians are printed.
}
\details{
Balanced models should have equal observability and reachability gramians.
}
\seealso{
    \code{\link{checkBalanceMittnik}}
    \code{\link[dse2]{MittnikReduction}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toSS(estVARXls(eg1.DSE.data.diff))
    checkBalance(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{checkBalanceMittnik}
\alias{checkBalanceMittnik}
\alias{checkBalanceMittnik.ARMA}
\alias{checkBalanceMittnik.SS}
\alias{checkBalanceMittnik.TSestModel}

\title{Check Balance of a TSmodel}
\description{
Calculate the difference between observability and reachability gramians of the model transformed to Mittnik's form.
}
\usage{
    checkBalanceMittnik(model)
    \method{checkBalanceMittnik}{ARMA}(model)
    \method{checkBalanceMittnik}{SS}(model)
    \method{checkBalanceMittnik}{TSestModel}(model)
}
\arguments{
    \item{model}{An object of class TSmodel.}
}
\value{No value is returned.}
\sideffects{
Differences  between the observability and reachability gramians are printed.}
\details{
Balanced models should have equal observability and reachability gramians.
}
\seealso{
\code{\link{checkBalance}}
\code{\link[dse2]{MittnikReduction}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toSS(estVARXls(eg1.DSE.data.diff))
    checkBalanceMittnik(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{checkConsistentDimensions}
\alias{checkConsistentDimensions}
\alias{checkConsistentDimensions.default}
\alias{checkConsistentDimensions.ARMA}
\alias{checkConsistentDimensions.SS}
\alias{checkConsistentDimensions.TSdata}
\alias{checkConsistentDimensions.TSestModel}

\title{Check Consistent Dimensions}
\description{Check that dimensions of a model and data agree.}
\usage{
    checkConsistentDimensions(obj1, obj2=NULL)
    \method{checkConsistentDimensions}{default}(obj1, obj2=NULL)
    \method{checkConsistentDimensions}{ARMA}(obj1, obj2=NULL)
    \method{checkConsistentDimensions}{SS}(obj1, obj2=NULL)
    \method{checkConsistentDimensions}{TSdata}(obj1, obj2=NULL)
    \method{checkConsistentDimensions}{TSestModel}(obj1, obj2=NULL)
    }
\arguments{
    \item{obj1}{An object containing a TSmodel, TSdata, or TSestModel,
       depending on the method}
    \item{obj2}{Another object containing TSdata corresponding to the TSmodel in
       obj1, or a TSmodel corresponding to the TSdata in obj1.}
}
\value{logical}
\details{
    Check that dimensions of a model and data agree. If \code{obj1} is a
    \code{TSestModel} then if \code{obj2} is \code{NULL, TSdata} is 
    taken from \code{obj1}. 
}

\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    checkConsistentDimensions(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{checkResiduals}
\alias{checkResiduals}
\alias{checkResiduals.default}
\alias{checkResiduals.TSdata}
\alias{checkResiduals.TSestModel}

\title{Autocorrelations Diagnostics}
\description{Calculate autocorrelation diagnostics of a time series 
matrix or TSdata or residuals of a TSestModel}
\usage{
    checkResiduals(obj, ...)
    \method{checkResiduals}{default}(obj, ac=TRUE, pac=TRUE, select=seq(nseries(obj)), 
                   drop=NULL, plot.=TRUE, graphs.per.page=5, verbose=FALSE, ...)
    \method{checkResiduals}{TSdata}(obj, ...)
    \method{checkResiduals}{TSestModel}(obj, ...)
}
\arguments{
    \item{obj}{An TSestModel or TSdata object.}
    \item{ac}{If TRUE the auto-correlation function is plotted.}
    \item{pac}{If TRUE the partial auto-correlation function is plotted.}
    \item{select}{
      Is used to indicate a subset of the residual series. By default all
      residuals are used.}
    \item{drop}{
      Is used to indicate a subset of the residual time periods to drop. 
      All residuals are used with the default (NULL).Typically this can be used 
      to get rid of bad initial conditions (eg. drop=seq(10) ) or outliers.}
    \item{plot.}{If FALSE then plots are not produced.}
    \item{graphs.per.page}{Integer indicating number of graphs to place on a
    page.}
    \item{verbose}{
        If TRUE then the auto-correlations and partial auto-correlations 
        are printed if they are calculated.}
    \item{...}{arguments passed to other methods.}
}
\value{
    A list with residual diagnostic information: residuals, mean, cov, 
    acf= autocorrelations, pacf= partial autocorrelations.
}
\details{
This is a generic function. The default method works for a time series
matrix which is treated as if it were a matrix of residuals. However, in a
Box-Jenkins type of analysis the matrix may be data which is being evaluated to
determine a model. The method for a TSestModel evaluates the residuals
calculated by subtracting the output data from the model predictions.}

\sideffects{
    Diagnostic information is printed and plotted if a device is available.
Output graphics can be paused between pages by setting par(ask=TRUE).
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    checkResiduals(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{coef.TSmodel}
\alias{coef.TSmodel}
\alias{coef.TSestModel}

\alias{coef<-}
\alias{coef<-.default}

\title{Extract Model Parameters}
\description{
    Set or extract coefficients (parameter values) of model objects.
}
\usage{
    \method{coef}{TSmodel}(object, ...)
    \method{coef}{TSestModel}(object, ...)
    coef(object) <- value
    \method{coef}{default}(object) <- value
    }
\arguments{
    \item{object}{An object of class TSmodel or TSestModel.}
    \item{value}{value to be assigned to object.}
    \item{...}{(further arguments, currently disregarded).}
    }
\value{A vector of parameter values.}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    coef(model)
    coef(model) <- 0.1 + coef(model)
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{combine}
\alias{combine}
\alias{combine.default}

\title{Combine two objects.}
\description{This is a generic method to
combine two objects of the same class to make a single object of that class.
}
\usage{
    combine(e1, e2)
    \method{combine}{default}(e1, e2)
    }
\arguments{
    \item{e1, e2}{TSdata objects.}
}
\value{An object of the same class as the argument but containing both e1 and e2.}
\seealso{
tbind, combine.TSdata, combine.forecastCov 
}
\examples{
if(is.R()) {data("eg1.DSE.data.diff", package="dse1")
            data("eg1.DSE.data", package="dse1") }
new.data.set <- combine(eg1.DSE.data.diff, eg1.DSE.data) 
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{combine.TSdata}
\alias{combine.TSdata}
\title{Combine series from two TSdata objects.}
\description{Combine series from two TSdata objects.}
\usage{
    \method{combine}{TSdata}(e1, e2)
    }
\arguments{
    \item{e1, e2}{TSdata objects.}
}
\value{
An object of class TSdata which includes series from both e1 and e2.
}
\seealso{
tbind 
}
\examples{
if(is.R()) {data("eg1.DSE.data.diff", package="dse1")
            data("eg1.DSE.data", package="dse1") }
new.data.set <- combine(eg1.DSE.data.diff, eg1.DSE.data) 
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{diffLog}
\alias{diffLog}
\title{Calculate the difference of log data}
\description{Calculate the difference from lag periods prior for log of data.}
\usage{
    diffLog(x, lag=1, base = exp(1), ...)}
\arguments{
    \item{x}{A time series.}
    \item{lag}{The difference is calculated relative to lag periods prior.}
    \item{base}{Base to use when calculating logrithms.}
    \item{...}{(further arguments, currently disregarded).}
}
\value{
A time series of the difference relative to lag periods prior for the log of 
the data. lag data points are lost from the beginning of the series. Negative
values will result in NAs.
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
z <- diffLog(outputData(eg1.DSE.data))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{eg1.DSE.data}
\docType{data}
\alias{eg1.DSE.data}
\alias{eg1.DSE.data.diff}
\alias{eg1.dat}

\title{Four Time Series used in Gilbert (1993)}

\description{

	Data is for Canada. 
	The series start in March 1961 (April 1961 for eg1.DSE.data.diff)
	and end in June 1991, giving 364  
	observations on each variable (363 for eg1.DSE.data.diff).

	The input series is 90-day interest rates (R90) in both eg1.DSE.data and
	eg1.DSE.data.diff.

	The output series are M1, GDP lagged two months, and CPI. 
	M1, GDP and CPI were all seasonally adjusted data. 
	These are not transformed in eg1.DSE.data and first difference of
	logs in eg1.DSE.data.diff.

	GDP is lagged because it is not available on as timely a basis. 
	(The data was used in an example where the intent was to build 
	a model for timely monitoring.)

	The Statistics Canada series identifiers are B14017, B1627, I37026,
	and B820200.
	
	The data for M1 (B1627) were taken prior to revisions made in 
	December 1993.
	
	The file eg1.dat contains the same data as eg1.DSE.data in a simple 
	ASCII file. 
}


\usage{
	data(eg1.DSE.data)
	data(eg1.DSE.data.diff)
}

\format{
	The objects eg1.DSE.data and eg1.DSE.data.diff are TSdata objects.
	The file eg1.dat is an ASCII file with 5 columns, the first enumerating
	the observations, the second giving the input series, and the third to
	fifth giving the output series. 
	The input series name  is  "R90" and the output series names are
	"M1", "GDPl2" and "CPI". GDPl2 is GDP lagged two months
}

\source{
	\emph{Statistics Canada,} 
	\emph{Bank of Canada.}
}

\references{
	Gilbert, P.D. 1993. "State Space and ARMA Models: An Overview of the 
	Equivalence." Working Paper 93-4, Bank of Canada. Also available at 
	\url{www.bank-banque-canada.ca/pgilbert}.

}

\seealso{
	\code{\link{TSdata}}
}
\keyword{datasets}

\eof
\name{egJofF.1dec93.data}
\docType{data}
\alias{egJofF.1dec93.data}
\title{Eleven Time Series used in Gilbert (1995)}

\usage{
	data(egJofF.1dec93.data)
}

\format{
	This data is a TSdata object.
	The input series name  is  "R90" and the output series names are
	"CPI", "GDP", "M1", "RL", "TSE300", "employment", "PFX", 
	"commod.price index", "US ind.prod." and  "US CPI"
}
\description{

	Data is for Canada unless otherwise indicated. 
	The series start in February 1974 and end in September 1993 (236 
	observations on each variable).

	The input series is 90 day interest rates (R90) and the
	ten output variables are CPI, GDP, M1, long run interest rates (RL), 
	the Toronto stock exchange 300 index (TSE300), employment, 
	the Canada/US exchange rate (PFX), a commodity price index in US dollars, 
	US industrial production, and US CPI.

	R90, RL and TSE are differenced. All other variables are in terms of 
	percent change.

	R90 is the 3 month prime corporate paper rate. While it is not set 
	directly by the Bank of Canada, Bank policy influences it directly 
	and it is often thought of as a proxy "policy variable."
	
	The Statistics Canada identifiers are B14017 (R90), P484549 (CPI), 
	I37026 (GDP), B1627 (M1), B14013 (RL), B4237 (TSE300), 
	D767608 (employment), B3400 (PFX). 
	
	M.BCPI (commodity price index) is published by the Bank of Canada. 
	JQIND (US industrial production), and CUSA0 (US CPI) 
	are DRI identifiers.
	
	The data for M1 (B1627) were taken prior to revisions made in 
	December 1993. 	
}


\source{
	\emph{Statistics Canada,} 
	\emph{Bank of Canada,}
	\emph{DRI.}
}

\references{
	Gilbert, P.D. 1995 "Combining VAR Estimation and State Space Model 
	Reduction for Simple Good Predictions" \emph{J. of Forecasting: Special 
	Issue on VAR Modelling.} 14:229-250
}

\seealso{
	\code{\link{TSdata}}
}
\keyword{datasets}

\eof
\name{estBlackBox}
\alias{estBlackBox}
\title{Estimate a TSmodel}
\description{Estimate a TSmodel.}
\usage{
    estBlackBox(data,...)
}
\arguments{
    \item{data}{Data in an object of class TSdata.}
    \item{...}{
    Optional arguments depent on the function which is eventually called.}
}
\value{A state space model in an object of class TSestModel.}
\details{
The function makes a call the most promising procedure currently available.
These tend to have names like estBlackBox1, estBlackBox2, estBlackBox4.
This is an active area of ongoing research and so the actual routine called
will probably change with new versions.
}
\examples{
if(is.R()) data("egJofF.1dec93.data", package="dse1")
goodmodel <- estBlackBox(egJofF.1dec93.data)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estBlackBox1}
\alias{estBlackBox1}
\title{Estimate a TSmodel}
\description{Estimate a TSmodel.}
\usage{
    estBlackBox1(data, estimation="estVARXls", 
        reduction="MittnikReduction", 
        criterion="taic", trend=FALSE, subtract.means=FALSE, 
	verbose=TRUE, max.lag=6)
}
\arguments{
    \item{data}{Data in an object of class TSdata.}
    \item{estimation}{Initial estimation method to be used.}
    \item{reduction}{Reduction method to be used.}
    \item{criterion}{Criterion to be used for model 
       selection. see \code{informationTestsCalculations}.}
    \item{trend}{logical indicating if a trend should be estimated.}
    \item{subtract.means}{logical indicating if the mean should be subtracted
       from data before estimation.}
    \item{verbose}{logical indicating if information should be printed 
       during estimation.}
    \item{max.lag}{integer indicating the maximum number of lags to consider.}
}
\value{A state space model in an object of class TSestModel.}
\sideffects{
If verbose is TRUE then estimation information is printed and checkResiduals is run, 
which gives plots of information about the residuals.
}
\seealso{
    \code{\link[dse1]{informationTestsCalculations}}
}

\examples{
if(is.R()) data("egJofF.1dec93.data", package="dse1")
goodmodel <- estBlackBox1(egJofF.1dec93.data)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estBlackBox2}
\alias{estBlackBox2}
\title{Estimate a TSmodel}
\description{Estimate a TSmodel.}
\usage{
    estBlackBox2(data, estimation='estVARXls', 
          lag.weight=.9, 
          reduction='MittnikReduction', 
          criterion='taic', 
          trend=FALSE, 
          subtract.means=FALSE,  re.add.means=TRUE, 
          standardize=FALSE, verbose=TRUE, max.lag=12)
}
\arguments{
    \item{data}{a TSdata object.}
    \item{estimation}{a character string indicating the estimation method to use.}
    \item{lag.weight}{weighting to apply to lagged observations. }
    \item{reduction}{character string indicating reduction procedure to use. }
    \item{criterion}{criterion to be used for model 
       selection. see \code{informationTestsCalculations}.}
    \item{trend}{if TRUE include a trend in the model. }
    \item{subtract.means}{
    if TRUE the mean is subtracted from the data before estimation.}
    \item{re.add.means}{
    if subtract.means is TRUE then if re.add.means is TRUE 
the estimated model is
converted back to a model for data without the mean subtracted.}
    \item{standardize}{
    if TRUE the data is transformed so that all variables have the same variance.}
    \item{verbose}{if TRUE then additional information from the estimation and 
reduction procedures is printed.}
    \item{max.lag}{The number of lags to include in the VAR estimation.}
}
\value{A TSestModel.}
\details{
A model is estimated and then a reduction procedure applied. The
default estimation procedure is least squares estimation of
a VAR model with lagged values weighted. This procedure is discussed in
Gilbert (1995).
}
\references{
    Gilbert, P.D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" \emph{J. of Forecasting: 
    Special Issue on VAR Modelling}. 14:229-250.
}
\seealso{
\code{\link{estBlackBox1}},
\code{\link{estBlackBox3}}
\code{\link{estBlackBox4}}
\code{\link[dse1]{informationTestsCalculations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  estBlackBox2(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estBlackBox3}
\alias{estBlackBox3}
\title{Estimate a TSmodel}
\description{Estimate a TSmodel.}
\usage{
    estBlackBox3(data, estimation='estVARXls', 
          lag.weight=1.0, 
          reduction='MittnikReduction', 
          criterion='aic', 
          trend=FALSE, 
          subtract.means=FALSE,  re.add.means=TRUE, 
          standardize=FALSE, verbose=TRUE, max.lag=12, sample.start=10)
}
\arguments{
    \item{data}{A TSdata object.}
    \item{estimation}{A character string indicating the estimation method to use.}
    \item{lag.weight}{Weighting to apply to lagged observations. }
    \item{reduction}{Character string indicating reduction procedure to use. }
    \item{criterion}{Criterion to be used for model 
       selection. see \code{informationTestsCalculations}. taic might be a
       better default selection criteria but it is not available for ARMA
       models.}
    \item{trend}{If TRUE include a trend in the model. }
    \item{subtract.means}{
    If TRUE the mean is subtracted from the data before estimation.}
    \item{re.add.means}{
    If subtract.means is TRUE then if re.add.means is T the estimated model is
converted back to a model for data without the mean subtracted.}
    \item{standardize}{
    If TRUE the data is transformed so that all variables have the same variance.}
    \item{verbose}{
    If TRUE then additional information from the estimation and reduction 
procedures is printed.}
    \item{max.lag}{The number of lags to include in the VAR estimation.}
    \item{sample.start}{The starting point to use for calculating information criteria.}
}
\value{A TSestModel.}
\details{
VAR models are estimated for each lag up to the specified max.lag. From 
these the best is selected according to the specified criteria. The 
reduction procedure is then applied to this best model and the best 
reduced model selected. The default estimation procedure is least 
squares estimation of a VAR model.
}
\seealso{
\code{\link{estBlackBox1}},
\code{\link{estBlackBox2}}
\code{\link{estBlackBox4}}
\code{\link[dse1]{informationTestsCalculations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  estBlackBox3(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estBlackBox4}
\alias{estBlackBox4}
\alias{bft}
\title{Estimate a TSmodel}
\description{Estimate a TSmodel with Brute Force Technique.}
\usage{
    estBlackBox4(data, estimation="estVARXls", 
                lag.weight=1.0,  variable.weights=1, 
                reduction="MittnikReduction", 
                criterion="taic", 
                trend=FALSE, subtract.means=FALSE,  re.add.means=TRUE, 
                standardize=FALSE, verbose=TRUE, max.lag=12, sample.start=10, warn=TRUE)
    bft(data, ... )
}
\arguments{
    \item{data}{A TSdata object.}
    \item{estimation}{a character string indicating the estimation method to use.}
    \item{lag.weight}{weighting to apply to lagged observations. }
    \item{variable.weights}{
       weighting to apply to series if estimation method is estWtVariables.}
    \item{reduction}{character string indicating reduction procedure to use.}
    \item{criterion}{criterion to be used for model 
       selection. see \code{informationTestsCalculations}.}
    \item{trend}{if TRUE include a trend in the model.}
    \item{subtract.means}{
       if TRUE the mean is subtracted from the data before estimation.}
    \item{re.add.means}{
       if subtract.means is TRUE then if re.add.means is T the estimated model is
       converted back to a model for data without the mean subtracted.}
    \item{standardize}{
     if TRUE the data is transformed so that all variables have the same variance.}
    \item{verbose}{
       if TRUE then additional information from the estimation and reduction 
       procedures is printed.}
    \item{max.lag}{VAR estimation is done for each lag up to max.lag.}
    \item{sample.start}{
       the starting point to use for calculating information criteria in the 
       final selection.}
    \item{warn}{logical indicating if warning messages should be suppressed.}
    \item{...}{arguments passed to estBlackBox4.}
}
\value{A TSestModel.}
\details{
For each lag up to max.lag a VAR model is estimated and then a
reduction procedure applied to select the best reduced model. Finally
the best of the best reduced models is selected. The default estimation
procedure is least squares estimation of the VAR models. This procedure
is described as the brute force technique (bft) in Gilbert (1995).
}
\references{
    Gilbert, P.D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" \emph{J. of Forecasting: 
    Special Issue on VAR Modelling}. 14:229-250.
}
\seealso{
\code{\link{estBlackBox1}},
\code{\link{estBlackBox2}}
\code{\link{estBlackBox3}}
\code{\link[dse1]{informationTestsCalculations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  bft(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{estMaxLik}
\alias{estMaxLik}
\alias{estMaxLik.TSdata}
\alias{estMaxLik.TSmodel}
\alias{estMaxLik.TSestModel}
\title{Maximum Likelihood Estimation}
\description{Maximum likelihood estimation.}
\usage{
    estMaxLik(obj1, obj2=NULL, ...) 
    \method{estMaxLik}{TSmodel}(obj1, obj2, algorithm="optim",
	algorithm.args=list(method="BFGS", upper=Inf, lower=-Inf, hessian=TRUE),
	...)
    \method{estMaxLik}{TSestModel}(obj1, obj2=TSdata(obj1), ...)
    \method{estMaxLik}{TSdata}(obj1, obj2, ...) 
}
\arguments{
    \item{obj1}{an object of class TSmodel, TSdata or TSestModel}
    \item{obj2}{TSdata or a TSmodel to be fitted with obj1.}
    \item{algorithm}{the algorithm ('optim', 'nlm' or 'nlmin') to use for maximization.}
    \item{algorithm.args}{arguments for the optimization algorithm.}
    \item{...}{arguments passed on to other methods.}
}
\value{
The value returned is an object of class TSestModel with  additional
elements \code{est$converged}, which is TRUE or FALSE indicating convergence, 
\code{est$converceCode}, which is the code returned by the estimation algorithm, 
and \code{est$results}, which are detailed results returned by the estimation 
algorithm. The hessian and gradient in results could potentially 
be used for restarting in the case of non-convergence, but that has not 
yet been implemented.
}
\details{
  One of \code{obj1} or \code{obj2} should specify a \code{TSmodel} and 
  the other \code{TSdata}. If \code{obj1} is a \code{TSestModel} and 
  \code{obj2} is NULL, then the data is extracted from \code{obj1}. 
  The \code{TSmodel} object is used to specify both the initial parameter 
  values and the model structure (the placement of the parameters
  in the various arrays of the TSmodel). Estimation attempts to minimimize the
  negative log likelihood (as returned by \code{l} ) of the given model 
  structure by adjusting the
  parameter values. A \code{TSmodel} can also have constant values in 
  some array elements, and these are not changed.
}
\warning{
  Maximum likelihood estimation of multivariate time series models tends to be
  problematic, even when a good structure and good starting parameter values
  are known. This is especially true for state space models. Also, it seems that
  in-sample fit is often obtained at the expense of out-of-sample forecasting
  ability. If a prior model structure is not important then the \code{bft}
  estimation method may be preferable.
}
\seealso{
\code{\link{optim}}
\code{\link{nlm}}
\code{\link{estVARXls}}
\code{\link{bft}}
\code{\link{TSmodel}}
\code{\link{l}}
}
\examples{
  true.model <- ARMA(A=c(1, 0.5), B=1)
  est.model <-  estMaxLik(true.model,  simulate(true.model))
  summary(est.model)
  est.model
  tfplot(est.model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estSSMittnik}
\alias{estSSMittnik}
\title{Estimate a State Space Model}
\description{Estimate a state space model using Mittnik's markov parameter estimation.
}
\usage{
    estSSMittnik(data, max.lag=6, n=NULL, subtract.means=FALSE, normalize=FALSE)
}
\arguments{
    \item{data}{A TSdata object.}
    \item{max.lag}{The number of markov parameters to estimate.}
    \item{n}{The state dimension.}
    \item{subtract.means}{If TRUE subtract the means from the data before estimation.}
    \item{normalize}{If TRUE normalize the data before estimation.}
}
\value{A state space model in an object of class TSestModel.}
\details{
Estimate a nested-balanced state space model by svd from least squares
estimate of markov parameters a la Mittnik p1195 Comp.Math Appl.v17,1989.
The quality of the estimate seems to be quite sensitive to max.lag, 
and this is not properly resolved yet.
If n is not supplied the svd criteria will be printed and n prompted for.
If subtract.means=T then the sample mean is subtracted. 
If normalize is T the lsfit estimation is done with outputs normalize to cov=I
(There still seems to be something wrong here!!).
The model is then re-transformed to the original scale.

See MittnikReduction and references cited there. If the state
dimension is not specified then the singular values of the Hankel matrix are
printed and the user is prompted for the state dimension.
}
\references{
    See references for \code{\link{MittnikReduction}}. 
}
\seealso{
    \code{\link{MittnikReduction}}
    \code{\link{estVARXls}}
    \code{\link{bft}}
}
\examples{
    if(is.R()) data("egJofF.1dec93.data", package="dse1")
    \dontrun{model <- estSSMittnik(egJofF.1dec93.data)}
    # this prints information about singular values and prompts with
    #Enter the number of singular values to use for balanced model:
    # the choice is difficult in this example. 
    model <- estSSMittnik(egJofF.1dec93.data, n=3)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estSSfromVARX}
\alias{estSSfromVARX}

\title{Estimate a state space TSmodel using VAR estimation}
\description{
    Estimate a VAR TSmodel with (optionally) an exogenous input 
    and convert to state space.
}
\usage{
    estSSfromVARX(data, warn=TRUE, ...)}
\arguments{
    \item{data}{
    An object with the structure of an object of class TSdata (see TSdata).}
    \item{warn}{Logical indicating if warnings should be printed (TRUE) or
    suppressed (FALSE).}
    \item{...}{See arguements to estVARXls}}

\value{A state space model in an object of class TSestModel.}
\details{This function uses the functions estVARXls and toSS.}
\references{
    Gilbert, P. D. (1993) State space and ARMA models: An overview of
    the equivalence. Working paper 93-4, Bank of Canada. Available at
    <www.bank-banque-canada.ca/pgilbert>

    Gilbert, P. D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" \emph{J. of Forecasting: 
    Special Issue on VAR Modelling}. 14:229-250.
}
\seealso{
    \code{\link{toSS}}
    \code{\link[dse2]{estSSMittnik}}
    \code{\link[dse2]{bft}}
    \code{\link{estVARXls}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <-estSSfromVARX(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{estVARXar}
\alias{estVARXar}
\alias{old.estVARXar}

\title{Estimate a VAR TSmodel}
\description{Estimate a VAR TSmodel with (optionally) an exogenous input.}
\usage{
    estVARXar(data, subtract.means=FALSE,  re.add.means=TRUE, standardize=FALSE, 
         unstandardize=TRUE, aic=TRUE, max.lag=NULL, method="yule-walker", warn=TRUE)
}
\arguments{
    \item{data}{A TSdata object.}
    \item{subtract.means}{
        If TRUE subtract the means from the data before estimation.}
    \item{re.add.means}{
      If TRUE the model is adjusted for the non-zero mean data when returned. 
      If subtract.means is also TRUE then the mean is added back to the data.}
    \item{standardize}{
      Note that the mean is not subtracted unless subtract.means is TRUE.
      A VAR model in an object of class TSestModel.}
    \item{unstandardize}{
      If TRUE and standardize is TRUE then the returned model is adjusted to 
      correspond to the original data.}
    \item{aic}{Passed to function ar.}
    \item{max.lag}{The maximum number of lags that should be considered.}
    \item{method}{Passed to function ar.}
    \item{warn}{If TRUE certain warning message are suppressed.}
}
\value{A TSestModel object containing an ARMA TSmodel object. The model has no
MA portion so it is a VAR model.}

\details{
This function estimates a VAR model with exogenous variable using ar().
Residuals,etc, are calculated by evaluating the estimated model with ARMA.
The procedure ar is used by combine exogeneous variables and endogenous variable
and estimating as if all variables were endogenous. The estVARXar 
method does not support trend estimation (as in estVARXls).

If aic=TRUE the number of lags is determined by an AIC statistic (see ar). 
If an exogenous (input) 
variable is supplied the input and output are combined (i.e.- both
treated as outputs) for estimation, and the resulting model is
converted back by transposing the exogenous variable part of the 
polynomial and discarding inappropriate blocks.
Residuals,etc, are calculated by evaluating the estimated model as a 
TSmodel/ARMA with the data (ie. residuals are not the residuals from the
regression).

Note: ar uses a Yule-Walker approach (uses autocorrelations) so effectively the 
  model is for data with means removed. Thus subtract.means does not make much
  difference and re.add.means must be TRUE to get back to a model for the 
  original data.

The conventon for AR(0)  and sign are changed to ARMA format.
Data should be of class TSdata.
The exog. variable is shifted so contemporaneous effects enter.
the model for the exog. variable (as estimated by ar() is  discarded.
}
\references{
    Gilbert, P. D. (1993) State space and ARMA models: An overview of
    the equivalence. Working paper 93-4, Bank of Canada. Available at
    <www.bank-banque-canada.ca/pgilbert>

    Gilbert, P. D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" \emph{J. of Forecasting: 
    Special Issue on VAR Modelling.} 14:229-250.
}
\seealso{
    \code{\link{estSSfromVARX}}
    \code{\link[dse2]{estSSMittnik}}
    \code{\link[dse2]{bft}}
    \code{\link{estVARXls}}
    \code{\link[ts]{ar}}
    \code{\link{DSE.ar}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXar(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estVARXls}
\alias{estVARXls}

\title{Estimate a VAR TSmodel}
\description{Estimate a VAR TSmodel with (optionally) an exogenous input and 
(optionally) a trend.
}
\usage{
    estVARXls(data, subtract.means=FALSE, re.add.means=TRUE, standardize=FALSE, 
     unstandardize=TRUE, max.lag=NULL, trend=FALSE, lag.weight=1.0, warn=TRUE) 
}
\arguments{
    \item{data}{A TSdata object.}
    \item{subtract.means}{If TRUE subtract the means from the data before estimation.}
    \item{re.add.means}{
    If TRUE and subtract.means is TRUE then the mean is added back to the data and 
    the model is adjusted for the non-zero mean data when returned.}
    \item{standardize}{
    If TRUE divide each series by its sample standard 
    deviation before estimation.
    Note that the mean is not subtracted unless subtract.means is TRUE.}
    \item{unstandardize}{
    If TRUE and standardize is TRUE then the returned 
    model is adjusted to correspond to the original data.}
    \item{trend}{If TRUE a trend is estimated.}
    \item{max.lag}{Number of lags to be used.}
    \item{lag.weight}{
    Weight between 0 and 1 to be applied to lagged data. 
    Lower weights mean lagged data is less important (more noisy).}
    \item{warn}{
    If TRUE a warning message is issued when missing data (NA) is
    detected and the model predictions are reconstructed from the lsfit residuals.}
}
\value{A TSestModel object containing a TSmodel object which is a VAR model.}
\details{
A VAR model is fitted by least squares regression using lsfit. The
argument max.lag determines the number of lags. If a trend is not
estimated the function estVARXar may be preferred. Missing data is
allowed in lsfit, but not (yet) by ARMA which generates the model
predictions, etc., based on the estimated model and the data. (This is
done to ensure the result is consistent with other estimation
techniques.) In the case of missing data ARMA is not used and the model
predictions, etc., are generated by adding the data and the lsfit
residual. This is slightly different from using ARMA, especially with
respect to initial conditions.
}
\references{
    Gilbert, P. D. (1993) State space and ARMA models: An overview of
    the equivalence. Working paper 93-4, Bank of Canada. Available at
    <www.bank-banque-canada.ca/pgilbert>

    Gilbert, P. D. (1995) "Combining VAR Estimation and State Space 
    Model Reduction for Simple Good Predictions" J. of Forecasting: 
    Special Issue on VAR Modelling. 14:229-250.
}
\seealso{
    \code{\link{estSSfromVARX}}
    \code{\link[dse2]{estSSMittnik}}
    \code{\link[dse2]{bft}}
    \code{\link{estVARXar}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estWtVariables}
\alias{estWtVariables}
\title{Weighted Estimation}
\description{estWtVariables}
\usage{
    estWtVariables(data, variable.weights,
                        estimation="estVARXls", estimation.args=NULL)
}
\arguments{
    \item{data}{A TSdata object.}
    \item{variable.weights}{weights to use for each output series.}
    \item{estimation}{An estimation method.}
    \item{estimation.args}{An arguments for the estimation method.}
}
\value{A TSestModel.}
\details{
    Weight series so that some series residuals are more 
    important than others. Each output variable is scaled according to 
    variable.weights, estimate is done, and then the estimated model
    unscaled. Estmation is done the method specified by estimate and any
    arguments specified by estimation.args.
    estimation.args should be NULL if no args are needed.
}
\seealso{
    \code{\link{estVARXls}}
    \code{\link{estBlackBox}}
    \code{\link{bft}}
    \code{\link{estMaxLik}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{findg}
\alias{findg}
\title{Find Equivalence Transformation}
\description{

Try to find a matrix g which makes two models equivalent.
}
\usage{
    findg(model1, model2, minf=nlmin)
}
\arguments{

    \item{model1, model2}{Objects of class TSmodel.}
    \item{minf}{Algorithm used to minimize the difference.}
}
\value{A matrix which converts one model to the other.}
\details{
This is set up as a minimization problem with the objective 
to reduce the squared difference between parameters.
}
\note{WARNING: This program does not work very well.}
\seealso{
    \code{\link{gmap}}
}
\examples{
# findg(model1, model2)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{fixConstants}
\alias{fixConstants}
\title{Fix TSmodel Coefficients (Parameters) to Constants}
\description{
    Fix any coefficients within fuzz of 0.0 or 1.0 to exactly 0.0 or 1.0.
    This will not change the model much but will affect some estimation 
    techniques and information criteria results, as these are considered to 
    be constants rather than coefficients.
}
\usage{
    fixConstants(model, fuzz=1e-5, constants=NULL)
}
\arguments{
    \item{model}{an object of class TSmodel.}
    \item{fuzz}{absolute difference to be considered equivalent.}
    \item{constants}{NULL or a list of logical arrays.}
}
\value{
    An object of class 'SS' 'TSmodel' with some array entries set to
    constants 0.0 or 1.0.
}
\details{If constants is not NULL then parameters within fuzz of 0.0 or 1.0 set 
    as constants 0.0 or 1.0. If constants is not NULL then it should be a list 
       with logical arrays named F, G ..., with TRUE corresponding to any 
       array elements which are to  be treated as constant.
}
\examples{
    f <- array(c(.5,.3,.2,.4),c(2,2))
    h <- array(c(1,0,0,1),c(2,2))
    k <- array(c(.5,.3,.2,.4),c(2,2))
    ss <- SS(F=f,G=NULL,H=h,K=k)
    ss
    coef(ss)
    ss <- fixConstants(ss, constants=list(
              F = matrix(c(TRUE, FALSE, FALSE, FALSE), 2,2)))
    ss
    coef(ss)
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toARMA(toSS(estVARXls(eg1.DSE.data.diff)))
    model <- fixConstants(model)
}
\seealso{
    \code{\link{fixF}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{fixF}
\alias{fixF}
\title{Set SS Model F Matrix to Constants}
\description{
    Set any parameters of the F matrix to constants. The same values are
    retained but they are considered to be constants rather than parameters.
    This will not change the model but will affect some estimation 
    techniques and information criteria results.
}
\usage{
    fixF(model)
}
\arguments{
    \item{model}{An object of class TSmodel.}

}
\value{
    An SS TSmodel object.
}

\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toSS(estVARXls(eg1.DSE.data.diff))
    model <- fixF(model)
}
\seealso{
    \code{\link{fixConstants}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{gmap}
\alias{gmap}
\title{Basis Transformation of a Model.}
\description{
Transform the basis for the state by a given invertible matrix.
}
\usage{
    gmap(g, model)
}
\arguments{
    \item{g}{An invertible matrix}
    \item{model}{An object of class TSmodel.}
}
\value{

An equivalent model transformed using g.
}
\details{

If the input model is in state space form g is a change of basis 
for the state. If the input model is in ARMA form then the 
polynomials are premultiplied by g. If g is a scalar it is treated 
as a diagonal matrix.
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- toSS(estVARXls(eg1.DSE.data.diff))
gmap(2, model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{informationTests}
\alias{informationTests}
\title{Tabulates selection criteria}
\description{Tabulates several model selection criteria.}
\usage{
    informationTests(..., sample.start=1,sample.end=NULL, Print=TRUE, warn=TRUE)}
\arguments{
    \item{...}{At least one object of class TSestModel.}
    \item{sample.start}{The start of the period to use for criteria calculations.}
    \item{sample.end}{
    The end of the period to use for criteria calculations. If omitted
    the end of the sample is used.}
    \item{Print}{If FALSE then printing suppressed.}
    \item{warn}{If FALSE then some warning messages are suppressed.}
}

\value{A matrix of the value for each model on each test returned invisibly.}
\sideffects{Criteria are tabulated for all models in the list.}
\seealso{
    \code{\link{informationTestsCalculations}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model1 <- estVARXls(eg1.DSE.data.diff)
    model2 <- estVARXar(eg1.DSE.data.diff)
    informationTests(model1, model2)}
%\keyword{DSE}
\keyword{ts}


\eof
\name{informationTestsCalculations}
\alias{informationTestsCalculations}

\title{Calculate selection criteria}
\description{Calculates several model selection criteria.}
\usage{
    informationTestsCalculations(lst, sample.start=1,sample.end=NULL, warn=TRUE)
}
\arguments{
    \item{lst}{One or more objects of class TSestModel.}
    \item{sample.start}{
        The start of the period to use for criteria calculations.
    }
    \item{sample.end}{
        The end of the period to use for criteria calculations. 
        If omitted the end of the sample is used.
    }
    \item{warn}{If FALSE then some warning messages are suppressed.}
}
\value{
   The calculated values are returned in a vector with names: port, like, 
   aic, bic, gvc, rice, fpe, taic, tbic, tgvc, trice and tfpe. These
   correspond to values for the  Portmanteau test, likelihood, 
   Akaike Information Criterion, Bayes  Information Criterion, 
   Generalized Cross Validation, Rice Criterion, and Final Prediction Error. 
   The preceeding 't' indicates that the theoretical parameter space 
   dimension has been used, rather than the number of 
   coefficient (parameter) values. Methods which select a model based on some
   information criterion calculated by \code{informationTestsCalculations}
   should use the name of the vector element to specify the test value which
   is to be used.
}
\seealso{
    \code{\link{informationTests}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    informationTestsCalculations(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{inputData}
\alias{inputData}
\alias{inputData.default}
\alias{inputData.TSdata}
\alias{inputData.TSestModel}
\alias{inputData<-}
\alias{inputData<-.default}
\alias{inputData<-.TSdata}
\alias{inputData<-.TSestModel}

\alias{outputData}
\alias{outputData.default}
\alias{outputData.TSdata}
\alias{outputData.TSestModel}
\alias{outputData<-}
\alias{outputData<-.default}
\alias{outputData<-.TSdata}
\alias{outputData<-.TSestModel}

\title{TSdata Series}
\description{Extract or set input or output series in a TSdata object.}
\usage{
    inputData(x, series=seqN(nseriesInput(x)))
    \method{inputData}{default}(x, series=seqN(nseriesInput(x)))
    \method{inputData}{TSdata}(x,  series=seqN(nseriesInput(x)))
    \method{inputData}{TSestModel}(x, series=seqN(nseriesInput(x)))

    outputData(x, series=seqN(nseriesOutput(x)))
    \method{outputData}{default}(x, series=seqN(nseriesOutput(x)))
    \method{outputData}{TSdata}(x,  series=seqN(nseriesOutput(x)))
    \method{outputData}{TSestModel}(x, series=seqN(nseriesOutput(x)))

    inputData(x)  <- value
    outputData(x) <- value
}
\arguments{
    \item{x}{object of class TSdata.}
    \item{value}{a time series matrix.}
    \item{series}{vector of strings or integers indicating the series to select.}
}
\value{
The first usages returns the input or output series.
The second usages assigns the input or output series.
}
\seealso{
    \code{\link{TSdata}}
    \code{\link[tframe]{selectSeries}}
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
outputData(eg1.DSE.data) 
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{l.ARMA}
\alias{l.ARMA}
\title{Evaluate an ARMA TSmodel}
\description{Evaluate an ARMA TSmodel.}
\usage{
    \method{l}{ARMA}(obj1, obj2, sampleT=NULL, predictT=NULL,result=NULL,
       error.weights=0,  compiled=.DSECOMPILED, warn=TRUE, 
       return.debug.info=FALSE, ...)
}
\arguments{
    \item{obj1}{an 'ARMA' 'TSmodel' object.}
    \item{obj2}{a TSdata object.}
    \item{sampleT}{an integer indicating the number of periods of data to use.}
    \item{predictT}{an integer to what period forecasts should be extrapolated.}
    \item{result}{
      if non-NULL then the returned value is only the sub-element indicated by 
      result. result can be a character string or integer.}
    \item{error.weights}{a vector of weights to be applied to the 
       squared prediction errors.}
    \item{compiled}{indicates if a call should be made to the compiled 
      code for computation. A FALSE value is mainly for testing purposes.}
    \item{warn}{if FALSE then certain warning messages are turned off.}
    \item{return.debug.info}{logical indicating if additional debugging
       information should be returned.}
    \item{...}{(further arguments, currently disregarded).}
}

\value{an object of class TSestModel (see TSestModel).}

\details{
This function is called by the function l() when the argument to l is an ARMA
model. Using l() is usually preferable to calling l.ARMA directly.
l.ARMA calls a compiled program unless compiled=FALSE. The compiled version is much 
faster. 

Output data must be at least as long as sampleT. If sampleT is not supplied it
is taken to be periods(data).

Input data must be at least as long as predictT. predictT must be at least as
large as sampleT. If predictT is not supplied it
is taken to be sampleT.

If \code{error.weights} is greater than zero then weighted prediction 
errors are calculated up to the horizon indicated
by the length of error.weights. The weights are applied to the squared
error at each period ahead.

}
\seealso{
\code{\link{l}},
\code{\link{l.SS}}
\code{\link{TSmodel}}
\code{\link{TSestModel.object}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- TSmodel(estVARXls(eg1.DSE.data.diff))
    evalutated.model <- l.ARMA(model,eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{l}
\alias{l}
\alias{l.TSdata}
\alias{l.TSestModel}

\title{Evaluate a TSmodel}
\description{Evaluate a model with data.}
\usage{
    l(obj1, obj2, ...)
    \method{l}{TSdata}(obj1, obj2, ...)
    \method{l}{TSestModel}(obj1, obj2, ...)
}
\arguments{
    \item{obj1}{a TSmodel, TSdata, or TSestModel object.}
    \item{obj2}{a TSmodel or TSdata object.}
    \item{...}{arguments to be passed to other methods.}
}
\value{Usually a TSestModel object is returned. Some methods allow an
  argument \code{result} which specifies that a certain part of the object is
  returned. (For example, the likelihood can be returned. This is useful for
  optimization routines.)}
\details{
This function determines whether the model is in state space or 
ARMA form and calls l.SS or l.ARMA.
}
\seealso{
\code{\link{l.SS}},
\code{\link{l.ARMA}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- toSS(TSmodel(estVARXls(eg1.DSE.data.diff)))
evalutated.model <- l(model, eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{l.SS}
\alias{l.SS}
\title{Evaluate a state space TSmodel}
\description{Evaluate a state space TSmodel.
}
\usage{
    \method{l}{SS}(obj1, obj2, sampleT=NULL, predictT=NULL, error.weights=0,
         return.state=FALSE, return.track=FALSE, result=NULL, 
	 compiled=.DSECOMPILED,
         warn=TRUE, return.debug.info=FALSE, ...)
}
\arguments{
    \item{obj1}{An 'SS' 'TSmodel' object.}
    \item{obj2}{A TSdata object.}
    \item{sampleT}{an integer indicating the last data point to use 
       for one step ahead filter estimation. If NULL all available data is used. }
    \item{predictT}{an integer indicating how far past the end of the 
       sample predictions should be made. For models with an input, 
       input data must be provided up to predictT. Output data is 
       necessary only to sampleT. If NULL predictT is set to sampleT.}
    \item{error.weights}{a vector of weights to be applied to the 
       squared prediction errors.}
    \item{return.state}{if TRUE the element \code{filter$state} 
       containing E[z(t)|y(t-1), u(t)] is returned as part of the 
       result. This can be a fairly large matrix.}
    \item{return.track}{if TRUE the element \code{filter$track} containing
       the expectation of the tracking error given y(t-1) and u(t) is 
       returned as part of the result. This can be an very large array. }
    \item{result}{if result is not specified an object of class 
       TSestModel is returned. Otherwise, the specified element 
       of \code{TSestModel$estimates} is returned.}
    \item{compiled}{if TRUE the compiled version of the code is used. 
       Otherwise the S/R version is used.}
    \item{warn}{if FALSE then certain warning messages are turned off.}
    \item{return.debug.info}{logical indicating if additional debugging
       information should be returned.}
    \item{...}{(further arguments, currently disregarded).}
}
\value{
Usually an object of class TSestModel (see TSestModel), but see result above.}

\details{
This function is called by the function l() when the argument to l is a state
space model. Using l() is usually preferable to calling l.SS directly.
l.SS calls a compiled program unless compiled=FALSE. The compiled version is much faster
than the S version.

Output data must be at least as long as sampleT. If sampleT is not supplied it
is taken to be periods(data).

Input data must be at least as long as predictT. predictT must be at least as
large as sampleT. If predictT is not supplied it
is taken to be sampleT.

If \code{error.weights} is greater than zero then weighted prediction 
errors are calculated up to the horizon indicated
by the length of error.weights. The weights are applied to the squared
error at each period ahead.

}
\seealso{
\code{\link{SS}}
\code{\link{l}}
\code{\link{l.ARMA}}
\code{\link{TSmodel}}
\code{\link{TSestModel.object}}
\code{\link{smoother}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- toSS(TSmodel(estVARXls(eg1.DSE.data.diff)))
lmodel <- l.SS(model,eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{markovParms}
\alias{markovParms}
\title{Markov Parameters}
\description{Construct a Matrix of the Markov Parameters}
\usage{
    markovParms(model, blocks=NULL)
}
\arguments{
    \item{model}{An ARMA or SS TSmodel.}
    \item{blocks}{Number of blocks to calculate.}
}
\value{A matrix}
\details{
    Construct a matrix with partitions [M0|...|Mi] giving the Markov
    parameters Mi, i+1 = blocks
    where each Mi is a p by (m+p) matrix, (m is the dimension of the exogeneous 
    series and p is the dimension of endogeneous series)
    ie.  y(t) = e(t) + M [u'(t)|y'(t-1) | u'(t-1)|y'(t-2)]' 
    This requires that models be transformed so that lagged endogeneous variables
    are inputs.  See Mittnik p1190.
    If blocks=NULL (the default) then at least 3 blocks are generated, and
    up to n+1, but the series is truncated if the blocks are effectively zero.
    This will affect the size of the Hankel matrix.
}
\seealso{
    \code{\link{SVDbalanceMittnik}}
}
\references{
    See references for \code{\link{MittnikReduction}}. 
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    markovParms(model)
}
%\keyword{DSE}
\keyword{ts}
\keyword{algebra}


\eof
\name{nseriesInput}
\alias{nseriesInput}
\alias{nseriesInput.default}
\alias{nseriesInput.TSdata}
\alias{nseriesInput.SS}
\alias{nseriesInput.ARMA}
\alias{nseriesInput.TSestModel}

\alias{nseriesOutput}
\alias{nseriesOutput.default}
\alias{nseriesOutput.TSdata}
\alias{nseriesOutput.SS}
\alias{nseriesOutput.ARMA}
\alias{nseriesOutput.TSestModel}

\title{Number of Series in in Input or Output}
\description{Number of input or output series in a TSdata object.}
\usage{
    nseriesInput(x)
    \method{nseriesInput}{default}(x)
    \method{nseriesInput}{TSdata}(x)
    \method{nseriesInput}{SS}(x)
    \method{nseriesInput}{ARMA}(x)
    \method{nseriesInput}{TSestModel}(x)
    
    nseriesOutput(x)
    \method{nseriesOutput}{default}(x)
    \method{nseriesOutput}{TSdata}(x)
    \method{nseriesOutput}{SS}(x)
    \method{nseriesOutput}{ARMA}(x)
    \method{nseriesOutput}{TSestModel}(x)
}
\arguments{
    \item{x}{Object of class TSdata, TSmodel or TSestModel.}
}
\value{
    An integer indicating the number of series.
}
\seealso{
    \code{\link{seriesNamesInput}}
    \code{\link{seriesNamesOutput}}
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
nseriesOutput(eg1.DSE.data) 
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{observability}
\alias{observability}
\alias{observability.ARMA}
\alias{observability.SS}
\alias{observability.TSestModel}

\title{Calculate Model Observability Matrix}
\description{
    Calculate the singular values of the observability matrix of a model.
}
\usage{
    observability(model)
    \method{observability}{ARMA}(model)
    \method{observability}{SS}(model)
    \method{observability}{TSestModel}(model)
}
\arguments{
    \item{model}{An object containing a TSmodel.}
}
\value{The singular values of the observability matrix.}
\details{
    If all singular values are significantly different from zero the
    model is observable.
}
\seealso{
    \code{\link{reachability}},
    \code{\link{stability}}
    \code{\link{McMillanDegree}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- toSS(estVARXls(eg1.DSE.data.diff))
    observability(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{percentChange}
\alias{percentChange}
\alias{percentChange.default}
\alias{percentChange.TSdata}
\alias{percentChange.list}
\alias{percentChange.TSestModel}
 
\title{Calculate percent change}
\description{Calculate the percent change relative to the data lag periods prior. 
}
\usage{
    percentChange(obj, ...)
    \method{percentChange}{default}(obj, base=NULL, lag=1, cumulate=FALSE, e=FALSE, ...)
    \method{percentChange}{TSdata}(obj, base=NULL, lag=1, cumulate=FALSE, e=FALSE, ...)
    \method{percentChange}{TSestModel}(obj, base=NULL, lag=1, cumulate=FALSE, e=FALSE, ...)
    }
\arguments{
    \item{obj}{
    An object of class TSdata or TSestModel, a time series matrix,
a matrix with columns corresponding to series (which are treated 
individually), or a list of one of these kinds of objects. (called m below)}
    \item{e}{
    If e is TRUE the exponent of the series is used (after cumulating 
if cumulate is TRUE). e can be
a logical vector with elements corresponding to columns of m.}
    \item{base}{
      If base is provided it is treated as the first period value 
      (that is, prior to differencing). It is prefixed to the m prior to 
      cumulating. It should be a vector of length dim(m)[2]. 
      (If e is TRUE then base should be log of the original data).}
    \item{lag}{integer indicating the number of periods relative to which the
      percent change should be calculated.}
    \item{cumulate}{logical indicating if the series should be cumulated before
      the percent change is calculated.}
    \item{...}{arguments passed to other methods.}
}
\value{
For an object of class TSdata the percent change calculation is done
with the output data and the result is an object of class TSdata (or a list of
objects of class TSdata).
For an object of class TSestModel the percent change calculation is done
with \code{estimates$pred} and the result is an object of class TSdata (or a 
list of objects of class TSdata).
}
\details{
If code{cumulate} is TRUE then the data is cumulated first. \code{cumulate} can be
a logical vector with elements corresponding to columns of m.
}
\seealso{
\code{\link{ytoypc}}
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
z <- percentChange(outputData(eg1.DSE.data))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{periods.TSdata}
\alias{periods.TSdata}
\alias{periods.TSestModel}
\alias{start.TSdata}
\alias{start.TSestModel}
\alias{end.TSdata}
\alias{end.TSestModel}
\alias{frequency.TSdata}
\alias{frequency.TSestModel}

\title{Specific Methods for tframed Data}
\description{See the generic function description.}
\usage{
    \method{periods}{TSdata}(x, ...)
    \method{periods}{TSestModel}(x)
    \method{start}{TSdata}(x, ...)
    \method{start}{TSestModel}(x, ...)
    \method{end}{TSdata}(x, ...)
    \method{end}{TSestModel}(x, ...)
    \method{frequency}{TSdata}(x, ...)
    \method{frequency}{TSestModel}(x, ...)
}
\arguments{
    \item{x}{a time series object.}
    \item{...}{(further arguments, currently disregarded).}
}

\seealso{
    \code{\link[tframe]{periods}}, 
    \code{\link[tframe]{start}},
    \code{\link[tframe]{end}},
    \code{\link[tframe]{frequency}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{periodsInput}
\alias{periodsInput}
\alias{periodsInput.TSdata}
\alias{periodsInput.TSestModel}

\alias{periodsOutput}
\alias{periodsOutput.TSdata}
\alias{periodsOutput.TSestModel}

\alias{startInput}
\alias{startInput.TSdata}
\alias{startInput.TSestModel}

\alias{startOutput}
\alias{startOutput.TSdata}
\alias{startOutput.TSestModel}

\alias{endInput}
\alias{endInput.TSdata}
\alias{endInput.TSestModel}

\alias{endOutput}
\alias{endOutput.TSdata}
\alias{endOutput.TSestModel}

\alias{frequencyInput}
\alias{frequencyInput.TSdata}
\alias{frequencyInput.TSestModel}

\alias{frequencyOutput}
\alias{frequencyOutput.TSdata}
\alias{frequencyOutput.TSestModel}

\title{TSdata Periods}
\description{Apply a method to the input or output data.}
\usage{
    periodsInput(x)
    \method{periodsInput}{TSdata}(x)
    \method{periodsInput}{TSestModel}(x)

    periodsOutput(x)
    \method{periodsOutput}{TSdata}(x)
    \method{periodsOutput}{TSestModel}(x)

    startInput(x)
    \method{startInput}{TSdata}(x)
    \method{startInput}{TSestModel}(x)

    startOutput(x)
    \method{startOutput}{TSdata}(x)
    \method{startOutput}{TSestModel}(x)

    endInput(x)
    \method{endInput}{TSdata}(x)
    \method{endInput}{TSestModel}(x)

    endOutput(x)
    \method{endOutput}{TSdata}(x)
    \method{endOutput}{TSestModel}(x)

    frequencyInput(x)
    \method{frequencyInput}{TSdata}(x)
    \method{frequencyInput}{TSestModel}(x)

    frequencyOutput(x)
    \method{frequencyOutput}{TSdata}(x)
    \method{frequencyOutput}{TSestModel}(x)

}
\arguments{
    \item{x}{An object containing TSdata.}
}
\value{Depends.}
\details{
    Apply a method to the input or output data so, for example,
    periodsInput(x) in theory does periods(inputData(x)), which 
    returns the number of periods in input data. The actual implementation
    may not do periods(inputData(x)). For example, with TSPADIdata 
    inputData(x) requires a database retrieval which may be fairly slow,
    while the number of periods may be available much more quickly.
}

\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    periodsOutput(eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{plot.roots}
\alias{plot.roots}
\title{Plot Model Roots}
\description{Calculate and plot roots of a model.}
\usage{
    \method{plot}{roots}(x, pch='*', fuzz=0, ...)}
\arguments{
    \item{x}{An object of class roots (a vector of complex (or real) 
values as returned by the function roots).}
    \item{pch}{character to be used for the plot (passed to plot.default).}
    \item{fuzz}{If non-zero then roots within fuzz distance are considered equal.}
    \item{...}{(further arguments, currently disregarded).}
}
\sideffects{The roots and a unit circle are plotted on the complex plane.}
\value{
The eigenvalues of the state transition matrix or the inverse of the roots of the
determinant of the AR polynomial are returned invisibly.
}
\seealso{
    \code{\link{addPlotRoots}}
    \code{\link{roots}}
    \code{\link{stability}}
    \code{\link{McMillanDegree}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    plot(roots(model))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{print.TSdata}
\alias{print.TSdata}

\title{Print Specific Methods}
\description{See the generic function description.}
\usage{
    \method{print}{TSdata}(x, ...)
    }
\arguments{
    \item{x}{An object of class TSdata.}
    \item{...}{arguments to be passed to other methods.}
}
\seealso{
    \code{\link{print}}
    \code{\link{summary}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{print.TSestModel}
\alias{print.TSestModel}
\alias{print.SS}
\alias{print.ARMA}

\title{Display TSmodel Arrays}
\description{
Display TSmodel arrays.
}
\usage{
    \method{print}{SS}(x, digits=options()$digits, latex=FALSE, ...)
    \method{print}{ARMA}(x, digits=options()$digits, latex=FALSE, L=TRUE, fuzz=1e-10, ...)
    \method{print}{TSestModel}(x, ...)
    }

\arguments{
    \item{x}{An object of class TSmodel or TSestModel.}
    \item{digits}{the number of significant digits}
    \item{L}{logical
      if TRUE then ARMA model arrays are displayed as a polynomial matrix 
      with L indicating lags. Otherwise, each lag in the array is 
      displayed as a matrix.}
    \item{latex}{logical. If TRUE additional context is added to make the output
      suitable for inclusion in a latex document.}
    \item{fuzz}{
      ARMA model polynomial elements with absolute value less than fuzz
      are not displayed (i.e.-as if they are zero)}
    \item{...}{arguments passed to other methods.}
}
\value{The object is returned invisibly.}
\sideffects{The model arrays are displayed.}
\note{BUG: digits cannot be controlled for some numbers (e.g.- 1.0 is printed
as 0.9999999999)
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
print(model)
print(model, digits=3)
print(model, digits=3, fuzz=0.001)
print(model, digits=3, fuzz=0.001, latex=TRUE)
}


\seealso{
    \code{\link{print}}
    \code{\link{summary}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{reachability}
\alias{reachability}
\alias{reachability.ARMA}
\alias{reachability.SS}
\alias{reachability.TSestModel}

\title{Calculate Model Reachability Matrix}
\description{
    Calculate the singular values of the reachability matrix of a model.
}
\usage{
    reachability(model)
    \method{reachability}{ARMA}(model)
    \method{reachability}{SS}(model)
    \method{reachability}{TSestModel}(model)
}
\arguments{
    \item{model}{An object containing TSmodel.}}
\value{The singular values of the reachability matrix.}
\details{

If all singular values are significantly different from zero the
model is controllable.
}
\seealso{
\code{\link{observability}},
\code{\link{stability}}
\code{\link{roots}}
\code{\link{McMillanDegree}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- toSS(estVARXls(eg1.DSE.data.diff))
reachability(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{residualStats}
\alias{residualStats}
\title{Calculate Residuals Statistics and Likelihood}
\description{Calculate the residuals statistics and likelihood of a residual.}
\usage{
    residualStats(pred, data, sampleT=nrow(pred), warn=TRUE)}
\arguments{
    \item{pred}{A matrix with columns representing time series.}
    \item{data}{A matrix with columns representing time series.}
    \item{sampleT}{An integer indicating the sample to use.}
    \item{warn}{If FALSE certain warnings are suppressed.}
}
\details{
    Residuals are calculated as
    pred[1:sampleT,,drop=FALSE] - data[1:sampleT,,drop=FALSE]
    and then statistics are calculated based on these residuals. If pred
    or data are NULL they are treated as zero.
}
\value{
    A list with elements like, cov, pred, and sampleT. pred and sampleT are
    as supplied (and are returned as this is a utility function called by
    other functions and it is convenient to pass them along). cov is the
    covariance of the residual and like is a vector of four elements 
    representing the total, constant, determinant and covariance 
    terms of the negative log likelihood function.
}
\seealso{
\code{\link{l}}
}
\examples{
    residualStats(matrix(rnorm(200), 100,2), NULL) # but typically used for a residual
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{roots}
\alias{roots}
\alias{roots.SS}
\alias{roots.ARMA}
\alias{roots.TSestModel}

\title{Calculate Model Roots}
\description{Calculate roots of a TSmodel.}
\usage{
    roots(obj, ...)
    \method{roots}{SS}(obj, fuzz=0, randomize=FALSE, ...)
    \method{roots}{ARMA}(obj, fuzz=0, randomize=FALSE, warn=TRUE, by.poly=FALSE, ...)
    \method{roots}{TSestModel}(obj, ...)
    }
\arguments{
    \item{obj}{An object of class TSmodel.}
    \item{fuzz}{If non-zero then roots within fuzz 
       distance are considered equal.}
    \item{randomize}{
       Randomly arrange complex pairs of roots so the one with the positive imaginary
       part is not always first (so random experiments are not biased).}
    \item{warn}{If FALSE then warnings about unit roots added for TREND are not printed.}
    \item{by.poly}{
       If TRUE then roots are calculated by expanding the determinant of the A
       polynomial.  Otherwise, they are calculated by converting to a state
       space representation and calculating the eigenvalues of F. This second
       method is preferable for speed, accuracy, and because of a limitation
       in the degree of a polynomial which can be handled by polyroot.}
    \item{...}{arguments passed to other methods.}
}
\value{

The eigenvalues of the state transition matrix or the inverse of the roots of the
determinant of the AR polynomial are returned.
}
\seealso{
\code{\link{stability}},
\code{\link{McMillanDegree}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
roots(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{scale.TSdata}
\alias{scale.TSdata}
\alias{scale.TSestModel}
\alias{scale.ARMA}
\alias{scale.innov}
\alias{scale.nonInnov}
\alias{checkScale}
\alias{checkScale.TSestModel}
\alias{checkScale.TSmodel}

\title{Scale Methods for TS objects}
\description{Scale data or a model by a given factor.}
\usage{
    \method{scale}{TSdata}(x, center=FALSE, scale=NULL)
    \method{scale}{TSestModel}(x, center=FALSE, scale=NULL)
    \method{scale}{ARMA}(x, center=FALSE, scale=NULL)
    \method{scale}{innov}(x, center=FALSE, scale=NULL)
    \method{scale}{nonInnov}(x, center=FALSE, scale=NULL)
    
    checkScale(x, scale)
    \method{checkScale}{TSestModel}(x, scale)
    \method{checkScale}{TSmodel}(x, scale)
    }
\arguments{
    \item{x}{TSdata, TSmodel or an object containing these.}
    \item{center}{to match generic arguments, not currently used.}
    \item{scale}{
    A list with two matrices or vectors, named input and output,
    giving the multiplication factor for inputs and outputs.
    Vectors are treated as diagonal matrices.
    \code{scale$input} can be NULL if no transformation is to be applied (or 
    the data or model has no input.)}
}

\value{
    The resulting data or model is different from
    the original in proportion to scale.  ie. if S and T are output and 
    input scaling matrices then 
         y'(t) = S y(t) where y' is the new output
         u'(t) = S u(t) where u' is the new input

    For models the result has inputs and outputs (and innovations) 
    which are scaled as if data scaling had been applied to them as above. 
    Thus if the input and output scales are diagonal matrices or 
    scalars the plot of the predictions and residuals for
    l(scale(model,scale=somescale), scale(data, scale=somescale))
    while have the same appearance as l(model, data) but will be 
    scaled differently.
}
\seealso{
    \code{\link[base]{scale}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    # This is a simple example. Usually scale would have something 
    # to do with the magnitude of the data.
    z <- scale(eg1.DSE.data.diff, 
        scale=list(input=rep(2, nseriesInput(eg1.DSE.data.diff)), 
                  output=rep(2,nseriesOutput(eg1.DSE.data.diff))))
    model <- estVARXls(eg1.DSE.data.diff)
    model <- scale(model, 
        scale=list(input=rep(2, nseriesInput(eg1.DSE.data.diff)), 
                  output=rep(2,nseriesOutput(eg1.DSE.data.diff))))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{seriesNames.TSdata}
\alias{seriesNames.TSdata}
\alias{seriesNames.TSmodel}
\alias{seriesNames.TSestModel}
\alias{seriesNames<-.TSdata}
\alias{seriesNames<-.TSmodel}
\alias{seriesNames<-.TSestModel}

\title{Series Names Specific Methods}
\description{See the generic function description.}
\usage{
    \method{seriesNames}{TSdata}(x)
    \method{seriesNames}{TSmodel}(x)
    \method{seriesNames}{TSestModel}(x)
    
    \method{seriesNames}{TSdata}(x) <- value
    \method{seriesNames}{TSmodel}(x) <- value
    \method{seriesNames}{TSestModel}(x) <- value
}
\arguments{
    \item{x}{an object from which series names can be extracted or
             to which series names are to be assigned.}
    \item{value}{series names to be assigned to data.}
}

\seealso{ \code{\link[tframe]{seriesNames}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{seriesNamesInput}
\alias{seriesNamesInput}
\alias{seriesNamesInput.TSdata}
\alias{seriesNamesInput.TSmodel}
\alias{seriesNamesInput.TSestModel}
\alias{seriesNamesInput<-}
\alias{seriesNamesInput<-.TSdata}
\alias{seriesNamesInput<-.TSmodel}
\alias{seriesNamesInput<-.TSestModel}


\alias{seriesNamesOutput}
\alias{seriesNamesOutput.TSdata}
\alias{seriesNamesOutput.TSmodel}
\alias{seriesNamesOutput.TSestModel}
\alias{seriesNamesOutput<-}
\alias{seriesNamesOutput<-.TSdata}
\alias{seriesNamesOutput<-.TSmodel}
\alias{seriesNamesOutput<-.TSestModel}

\title{TSdata Series Names}
\description{Extract or set names of input or output series in a TSdata object.}
\usage{
    seriesNamesInput(x)
    \method{seriesNamesInput}{TSdata}(x)
    \method{seriesNamesInput}{TSmodel}(x)
    \method{seriesNamesInput}{TSestModel}(x)

    seriesNamesOutput(x)
    \method{seriesNamesOutput}{TSdata}(x)
    \method{seriesNamesOutput}{TSmodel}(x)
    \method{seriesNamesOutput}{TSestModel}(x)

    seriesNamesInput(x)  <- value
    seriesNamesOutput(x) <- value
}
\arguments{
    \item{x}{Object of class TSdata, TSmodel or TSestModel.}
    \item{value}{value to be assigned to object.}
}
\value{
The first usages gives a vector of strings with the series names.
The second usages assigns a vector of strings to be the series names of data.
}
\seealso{
    \code{\link[tframe]{seriesNames}}
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
seriesNamesOutput(eg1.DSE.data) 
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{setArrays}
\alias{setArrays}
\alias{setArrays.ARMA}
\alias{setArrays.SS}
\alias{setArrays.TSestModel}

\title{Set TSmodel Array Information}
\description{
    Complete parameter array information based on parameter vector settings.
    This function is used internally and is not normally called by a user. 
}
\usage{
    setArrays(model, coefficients=NULL)
    \method{setArrays}{ARMA}(model, coefficients=NULL)
    \method{setArrays}{SS}(model, coefficients=NULL)
    \method{setArrays}{TSestModel}(model, coefficients=NULL)
    }
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{coefficients}{A vector of new values for the model coefficients
    (parameters).}
}
\value{A TSmodel object.}

%\keyword{DSE}
\keyword{internal}



\eof
\name{setTSmodelParameters}
\alias{setTSmodelParameters}
\alias{setTSmodelParameters.default}
\alias{setTSmodelParameters.ARMA}
\alias{setTSmodelParameters.SS}

\title{Set TSmodel Parameter Information}
\description{
    Complete parameter vector information based on parameter array settings.
    This function is used internally and is not normally called by a user. 
}
\usage{
    setTSmodelParameters(model, constants=NULL)
    \method{setTSmodelParameters}{default}(model, constants=NULL)
    \method{setTSmodelParameters}{ARMA}(model, constants=NULL)
    \method{setTSmodelParameters}{SS}(model, constants=NULL)
}
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{constants}{A list of logical arrays indicating TRUE for any model
    array entries that should be treated as constants.}
}
\value{An object of class 'TSmodel'.}
\seealso{

\code{\link{setArrays}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    model <- setTSmodelParameters(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{simulate}
\alias{simulate}
\alias{simulate.ARMA}
\alias{simulate.SS}
\alias{simulate.TSestModel}

\title{Simulate a TSmodel}
\description{Simulate a model to produce artificial data.}
\usage{
    simulate(model, ...)
    \method{simulate}{ARMA}(model, y0=NULL, input=NULL, input0=NULL,
        start=NULL, freq=NULL, sampleT=100, noise=NULL, sd=1, SIGMA=NULL,
        rng=NULL, noise.model=NULL, compiled=.DSECOMPILED, ...)
    \method{simulate}{SS}(model, input=NULL, 
        start=NULL, freq=NULL,sampleT=100, noise=NULL, sd=1, SIGMA=NULL,
	rng=NULL, compiled=.DSECOMPILED, ...)
    \method{simulate}{TSestModel}(model, input=inputData(model),
			sd=NULL, SIGMA=NULL, ...)
}
\arguments{
      \item{model}{An object of class TSmodel or TSestModel.}
    \item{input}{Data for the exogenous variable if specified in the model.}
    \item{sampleT}{The length of the sample to simulate.}
    \item{start}{start date for resulting data.}
    \item{freq}{freq for resulting data.}
    \item{y0, input0}{
Lagged values prior to t=1 for y and u, in reverse order so y0[1,] and
input0[1,]correspond to t=0. These arguments  are not implemented for state 
space models. If not specified initial values are set to zero. }
    \item{noise}{
Noise can be supplied. Otherwise it will be generated.
If supplied it should be a list as described below under returned value.}
    \item{SIGMA}{
The covariance of the noise process. If this is specified then sd is ignored.
A vector or scalar is treated as a diagonal matrix. For an object of class
TSestModel, if neither SIGMA nor sd are specified, then SIGMA is set to
the estimated covariance (\code{model$estimates$cov}).}
    \item{sd}{The standard deviation of the noise. This can be a vector.}
    \item{noise.model}{
A TSmodel to be used for generating noise (not yet supported by SS methods).}
    \item{rng}{The random number generator information needed to
regenerate a simulation.}
    \item{compiled}{
    Specifies the compiled version of the code should be used (instead of
    the S code version).}
    \item{...}{arguments passed to other methods.}
}

\value{
The value returned is an object of class TSdata which can be supplied as an
argument to estimation routines. (See TSdata). In addition to the usual
elements (see the description of a TSdata object)
there are some additional elements:
   model- the generating model,
   rng - the initial RNG and seed,
   version - the version of S used (random number generators may vary)
   SIGMA as specified
   sd    as specified
   noise - a list of e, w and w0 - the noise processes. w0 is w for t=0 in
      the state space model and prior lags in ARMA models. For VAR models B has
      no lags so w0 has no effect.
   state - the state variable for state space models.
}
\details{

A state space or ARMA model as described in TSmodel is simulated with pseudo
random noise (The default noise is a normally distributed processes. 
An object of class TSdata is
returned. This can be used as input to estimation algorithms. If start and 
freq are specified, or if input or \code{noise$w} (in that order)
have time series properties, these are given to the output. 
If \code{noise$w0} is a matrix (rather than a vector) for a state space model 
simulation (as it is for ARMA simulations) then it is set to a
vector of zeros. This provides compatability with VAR models (ARMA
models with no lags in B). In general ARMA and SS simulations will
not produce exactly the same results because it is impossible to
determine necessary transformation of initial conditions and w0.
The rng will be set first if it is specified. 
}
\seealso{
\code{\link[tframe]{makeTSnoise}},
\code{\link{TSmodel}},
\code{\link{TSdata}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <- simulate(model)}
%\keyword{DSE}
\keyword{ts}


\eof
\name{smoother}
\alias{smoother}
\title{Evaluate a smoother with a TSmodel}
\description{Evaluate a state space model.}
\usage{
    smoother(model, data, compiled=.DSECOMPILED)
}
\arguments{
    \item{model}{
    An object of class `TSestModel' or 'TSmodel' with a model of 
class `nonInnov' 'SS' 'TSmodel'. If filter informatin is not provided 
(i.e. in a TSestModel)
then smoother runs the Kalman filter (l.SS) first.}
    \item{data}{A TSdata object.}.
    \item{compiled}{
    If TRUE the compiled version of the code is used. Otherwise the S version is used.}
}
\value{
An object of class TSestModel with an additional element \code{smooth}. 
\code{smooth} is a list of \code{state}, the smoothed state, and 
\code{track}, the smoothed tracking 
error. The result will also contain the element \code{filter} with \code{state}
and \code{track} (which may or may not have been in the original arguement). 
}
\details{
Smoother takes the filtered state \code{estimates$state} and produces a smoothed 
estimate of the state (sometimes called a two sided filter).
}
\seealso{
\code{\link{l}},
\code{\link{l.SS}}
\code{\link{TSmodel}}
\code{\link{TSestModel.object}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
#smoother requires an non-innovations form model
model <- TSmodel(toSSChol(estVARXls(eg1.DSE.data.diff))) 
smoothed.model <- smoother(model, eg1.DSE.data.diff, compiled=FALSE)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{sourceInfo.TSdata}
\alias{sourceInfo.TSdata}
\alias{sourceInfo.TSestModel}
\alias{sourcedb.TSdata}
\alias{sourcedb.TSestModel}
\alias{sourceserver.TSdata}
\alias{sourceserver.TSestModel}
\alias{identifiers.TSdata}
\alias{identifiers.TSestModel}

\title{tfPADI Specific Methods}
\description{See the generic function description.}
\usage{
    \method{sourceInfo}{TSdata}(obj)
    \method{sourceInfo}{TSestModel}(obj)
    
    \method{sourcedb}{TSdata}(obj)
    \method{sourcedb}{TSestModel}(obj)
    
    \method{sourceserver}{TSdata}(obj)
    \method{sourceserver}{TSestModel}(obj)
    
    \method{identifiers}{TSdata}(obj)
    \method{identifiers}{TSestModel}(obj)    
}
\arguments{
    \item{obj}{An object of class TSdata or TSestModel.}
}

\seealso{ \code{\link[dsepadi]{sourceInfo}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{stability}
\alias{stability}
\alias{stability.ARMA}
\alias{stability.roots}
\alias{stability.TSmodel}
\alias{stability.TSestModel}

\title{Calculate Stability of a TSmodel}
\description{Calculate roots and their modulus and indicate stability.}
\usage{
    stability(obj, fuzz=1e-4, digits=8, verbose=TRUE)
    \method{stability}{ARMA}(obj, fuzz=1e-4, digits=8, verbose=TRUE)
    \method{stability}{roots}(obj, fuzz=1e-4, digits=8, verbose=TRUE)
    \method{stability}{TSmodel}(obj, fuzz=1e-4, digits=8, verbose=TRUE)
    \method{stability}{TSestModel}(obj, fuzz=1e-4, digits=8, verbose=TRUE)
    }
\arguments{
    \item{obj}{An object of class TSmodel.}
    \item{fuzz}{Roots within fuzz are considered equal.}
    \item{digits}{Printing precision.}
    \item{verbose}{Print roots and there moduli.}
}
\value{TRUE or FALSE if the model is stable or not stable.}
\sideffects{
The eigenvalues of the state transition matrix or the roots of the
determinant of the AR polynomial are printed if verbose is T.
}
\seealso{
\code{\link{McMillanDegree}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
stability(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{standardize}
\alias{standardize}
\title{standardize data}
\description{Convert data to mean zero, standard deviation one. This does not
remove cross correlations between series.}
\usage{standardize(ser)}
\arguments{
    \item{ser}{A vector time matrix.}
    }
\value{A time series matrix.}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- standardize(outputData(eg1.DSE.data.diff))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{sumSqerror}
\alias{sumSqerror}
\title{Calculate sum of squared prediction errors}
\description{

Calculate a weighted sum squared prediction errors for a parameterization.
}
\usage{
    sumSqerror(coefficients, model=NULL, data=NULL, error.weights=NULL)}
\arguments{
    \item{coefficients}{A vector of coefficients (parameters).}
    \item{model}{an object of class TSmodel which gives the structure 
      of the model for which coefficients are used. \code{coef(model)} should 
      be the same length as coefficients.}
    \item{data}{an object of class TSdata which gives the data with 
       which the model is to be evaluated.}
    \item{error.weights}{a vector of weights to be applied to the 
       squared prediction errors.}
}
\value{
The value of the sum squared errors for a prediction horizon given by the 
length of error.weights. Each period ahead is weighted by the corresponding 
weight in error.weights.
}
\details{
This function is primarily for use in parameter optimization,
which requires that an objective function be specified by a vector
of parameters.
}
\seealso{
\code{\link{l}}
\code{\link{l.SS}}
\code{\link{l.ARMA}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
sumSqerror(1e-10 + coef(model), model=TSmodel(model), 
        data=TSdata(model), error.weights=c(1,1,10))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{summary.TSdata}
\alias{summary.TSdata}
\alias{summary.SS}
\alias{summary.ARMA}
\alias{summary.TSestModel}

\alias{print.summary.TSdata}
\alias{print.summary.SS}
\alias{print.summary.ARMA}
\alias{print.summary.TSestModel}

\title{Specific Methods for Summary}
\description{See the generic function description.}
\usage{
    \method{summary}{TSdata}(object, ...)
    \method{summary}{SS}(object, ...)
    \method{summary}{ARMA}(object, ...)
    \method{summary}{TSestModel}(object, ...)
    \method{print}{summary.TSdata}(x, digits=options()$digits, ...)
    \method{print}{summary.SS}(x, digits=options()$digits, ...)
    \method{print}{summary.ARMA}(x, digits=options()$digits, ...)
    \method{print}{summary.TSestModel}(x, digits=options()$digits, ...)
    }
\arguments{
    \item{object}{an object to be summarized.}
    \item{x}{a summary object to be printed.}
    \item{digits}{number of significant digits to use for printing.}
    \item{...}{arguments passed to other methods.}
}

\seealso{
    \code{\link{print}}, 
    \code{\link{summary}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{testEqual.ARMA}
\alias{testEqual.ARMA}
\alias{testEqual.SS}
\alias{testEqual.TSdata}
\alias{testEqual.TSmodel}
\alias{testEqual.TSestModel}

\title{Specific Methods for Testing Equality}
\description{See the generic function description.}
\usage{
    \method{testEqual}{ARMA}(obj1, obj2, fuzz=0)
    \method{testEqual}{SS}(obj1, obj2, fuzz=0)
    \method{testEqual}{TSdata}(obj1, obj2, fuzz=1e-16)
    \method{testEqual}{TSmodel}(obj1, obj2, fuzz=0)
    \method{testEqual}{TSestModel}(obj1, obj2, fuzz=0)
    }
\arguments{
    \item{obj1}{see generic method.}
    \item{obj2}{see generic method.}
    \item{fuzz}{see generic method.}
}

\seealso{\code{\link[tframe]{testEqual}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.TSdata}
\alias{tfplot.TSdata}
\alias{tfplot.TSestModel}

\title{Tfplot Specific Methods}
\description{See the generic function description.}
\usage{
    \method{tfplot}{TSdata}(x, ..., 
        tf=NULL, start=tfstart(tf), end=tfend(tf), 
        select.inputs  = seq(length=nseriesInput(x)),
        select.outputs = seq(length=nseriesOutput(x)),
        Title=NULL, xlab=NULL, ylab=NULL, 
	graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{TSestModel}(x, ..., 
        tf=NULL, start=tfstart(tf), end=tfend(tf), 
        select.inputs=NULL, select.outputs=NULL,
        Title=NULL, xlab=NULL, ylab=NULL, 
        graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    }
\arguments{
    \item{x}{object to be plotted.}
    \item{...}{additional objects to be plotted.}
    \item{start}{start of plot.}
    \item{end}{end of plot.}
    \item{tf}{an alternate way to specify start and end of plot.}
    \item{select.inputs}{series to be plotted. (passed to selectSeries)}
    \item{select.outputs}{series to be plotted. (passed to selectSeries)}
    \item{Title}{string to use for plot title (passed to plot).}
    \item{xlab}{string to use for x label (passed to plot).}
    \item{ylab}{string to use for y label (passed to plot).}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{mar}{margins passed to plot. See par.)}
    \item{reset.screen}{logical indicating if the plot window should be cleared
       before starting. If this is not TRUE then mar values will have no effect.}
}

\seealso{ \code{\link[tframe]{tfplot}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{tframed.TSdata}
\alias{tframed.TSdata}
\alias{tframe<-.TSdata}
\alias{tfwindow.TSdata}
\alias{tbind.TSdata}
\alias{trimNA.TSdata}

\alias{window.TSdata}

\title{Specific Methods for tframed Data}
\description{See the generic function description.}
\usage{
    \method{tframed}{TSdata}(x, tf=NULL, names=NULL)
    \method{tframe}{TSdata}(x) <- value
    \method{tfwindow}{TSdata}(x, tf=NULL, start=tfstart(tf), end=tfend(tf), warn=TRUE)
    \method{tbind}{TSdata}(x, d2, ..., pad.start=TRUE, pad.end=TRUE, warn=TRUE)
    \method{trimNA}{TSdata}(x, startNAs=TRUE, endNAs=TRUE)
    \method{window}{TSdata}(x, start=NULL, end=NULL, tf=NULL, warn=TRUE, ...)
}
\arguments{
    \item{x}{See the generic function.}
    \item{tf}{a time frame. See the generic function.}
    \item{value}{a time frame to associate with x.}
    \item{names}{A list with elements input and output which are strings
         passed as names to the default method.}
    \item{start}{See the generic function.}
    \item{startNAs}{See the generic function.}
    \item{end}{See the generic function.}
    \item{endNAs}{See the generic function.}
    \item{d2}{See the generic function.}
    \item{pad.start}{See the generic function.}
    \item{pad.end}{See the generic function.}
    \item{warn}{logical indicating if some warning messages should be suppressed.}
    \item{...}{arguments passed to other functions.}
}

\seealso{
    \code{\link[tframe]{tframed}},
    \code{\link[tframe]{tfwindow}},
    \code{\link[tframe]{tbind}},
    \code{\link[tframe]{trimNA}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{toARMA}
\alias{toARMA}
\alias{toARMA.ARMA}
\alias{toARMA.SS}
\alias{toARMA.TSestModel}

\title{Convert to an ARMA Model}
\description{
Convert a state space model to an ARMA representation. The state is 
eliminated by a method which uses an equivalence that can be demonstrated
by the Cayley Hamilton theorem.
}
\usage{
    toARMA(model, ...)
    \method{toARMA}{ARMA}(model, ...)
    \method{toARMA}{SS}(model, fuzz=1e-10, ...)
    \method{toARMA}{TSestModel}(model, ...)
}
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{fuzz}{Parameters closer than fuzz to one or zero are set to 1.0
     or 0.0 respectively}
    \item{...}{arguments to be passed to other methods.}
}
\value{
An object of class 'ARMA' 'TSmodel' containing an ARMA model.
}
\references{See, for example, 
  M. Aoki(1990)\emph{State Space Modelling of Time Series}. 2d ed. rev. 
  and enl., Springer-Verlag}
\seealso{
\code{\link{toSS}}
\code{\link{fixConstants}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- toSS(estVARXls(eg1.DSE.data.diff))
model <- toARMA(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{toSS}
\alias{toSS}
\alias{toSS.ARMA}
\alias{toSS.SS}
\alias{toSS.TSestModel}
\alias{toSSaugment}
\alias{toSSaugment.ARMA}
\alias{toSSaugment.TSestModel}
\alias{toSSnested}
\alias{toSSnested.SS}
\alias{toSSnested.ARMA}
\alias{toSSnested.TSestModel}

\title{Convert to State Space Model}
\description{
Convert a model to state space form.
}
\usage{
    toSS(model, ...)
    \method{toSS}{ARMA}(model, ...)
    \method{toSS}{SS}(model, ...)
    \method{toSS}{TSestModel}(model, ...)
    
    toSSaugment(model, ...)
    \method{toSSaugment}{ARMA}(model, fuzz=1e-14, ...)
    \method{toSSaugment}{TSestModel}(model, ...)
    
    toSSnested(model, ...)
    \method{toSSnested}{ARMA}(model, n=NULL, Aoki=FALSE, ...)
    \method{toSSnested}{SS}(model, n=NULL, Aoki=FALSE, ...)
    \method{toSSnested}{TSestModel}(model, ...)
    
    }
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{n}{If n is specified then it is used as the state dimension when the 
      markov parameter conversion technique is required.}
    \item{Aoki}{logical indicating if Aoki's method (which does not work in
       general) should be tried.}
    \item{fuzz}{if the zero lag term of polynomials A and B are within fuzz of
       the identitity matrix then they are not inverted. (i.e. they are assumed
       to be identity.)}
    \item{...}{arguments to be passed to other methods.}
}
\value{
A state space model in an object of class 'SS' 'TSmodel'.
}
\details{

If the order of the AR polynomial equals or exceeds the MA 
polynomial (and the input polynomial) then the model is converted 
by state augmentation. Otherwise, it is converted by approximating 
the markov coefficients a la Mittnik.  (This may not always work
very well. Compare the results to check.)
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
model <- toSS(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{toSSChol}
\alias{toSSChol}
\alias{toSSChol.TSmodel}
\alias{toSSChol.TSestModel}

\title{Convert to Non-Innovation State Space Model}
\description{

Convert to a non-innovations state space representation using 
the given matrix as the measurement noise covariance. 
This function may not be working properly.
}
\usage{
    toSSChol(model, ...)
    \method{toSSChol}{TSmodel}(model, Om=diag(1,nseriesOutput(model)), ...)
    \method{toSSChol}{TSestModel}(model, Om=NULL, ...)
}
\arguments{
    \item{model}{An object of class TSmodel.}
    \item{Om}{
      a matrix to be used as the measurement noise covariance. If Om is
      not supplied and model is of class TSestModel then 
      \code{model$estimates$cov} is used. Otherwise, Om is set to the 
      identity matrix.}
    \item{...}{arguments to be passed to other methods.}
}
\value{
An object of class 'SS' 'TSmodel' containing a state space model which is
not in innovations form.
}
\details{
Convert to a non-innovations SS  representation using a Cholesky 
decomposition of Om as the coefficient matrix of the output noise.
}
\seealso{
\code{\link{toSSinnov}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    model <- toSSChol(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{toSSOform}
\alias{toSSOform}
\alias{toSSOform.TSmodel}
\alias{toSSOform.TSestModel}

\title{Convert to Oform}
\description{Convert a state space model to (observability?) form.}
\usage{
    toSSOform(model)
    \method{toSSOform}{TSmodel}(model)
    \method{toSSOform}{TSestModel}(model)
    }
\arguments{
    \item{model}{An object of class TSmodel.}
}
\value{
An object of class 'SS' 'TSmodel' containing a state space 
model in observability form (more or less).
 WARNING: This function does not work properly.}
\seealso{
    \code{\link{toSSinnov}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
%    model <- toSSOform(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{toSSinnov}
\alias{toSSinnov}
\title{Convert to State Space Innovations Model}
\description{Convert to a state space innovations representation.}
\usage{
    toSSinnov(model, ...)}
\arguments{
    \item{model}{an object of class TSmodel.}
    \item{...}{arguments passed to other methods.}
}
\value{
If the argument is a TSmodel then the result is an object of 
class 'SS' 'TSmodel' If the argument is a TSestModel then the converted model is
evaluated with the data an a TSestModel is returned. The TSmodel is an 
innovations state space representation.
}
\seealso{
\code{\link{toSS}},
\code{\link{toSSOform}}
\code{\link{toSSChol}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
model <- toSSinnov(model)
summary(model)

model2 <- SS(F=diag(1,3), H=matrix(c(1,0,0,1,0,0),2,3),
   Q=diag(0.5, 3, 3), R=diag(1.1, 2,2),
    description="test model", output.names=c("output 1", "output 2"))
model2 <- toSSinnov(model2)
summary(model2)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{ytoypc}
\alias{ytoypc}
\title{Convert to year to year percent change}
\description{Convert level data to year over year percent change. }
\usage{
    ytoypc(ser)}
\arguments{
    \item{ser}{A time series.}
}
\value{
A vector time series of the year over year percent change. This uses 
percentChange with lag=frequency(ser).
}
\seealso{
\code{\link{percentChange}}
}
\examples{
if(is.R()) data("eg1.DSE.data", package="dse1")
z <- ytoypc(outputData(eg1.DSE.data))
}
%\keyword{DSE}
\keyword{ts}


\eof
