\name{EstEval}
\alias{EstEval}
\alias{is.EstEval}

\title{Evaluate an estimation method}
\description{Evaluate an estimation method.}
\usage{
    EstEval(model, replications=100, rng=NULL, quiet=FALSE, 
                       simulation.args=NULL,
                       estimation=NULL, estimation.args=NULL, 
                       criterion ="coef", criterion.args =NULL, 
		       Spawn=if (exists(".SPAWN")) .SPAWN else FALSE)
    is.EstEval(obj)
}
\arguments{
    \item{model}{A TSmodel.} 
    \item{replications}{The number of simulations.}
    \item{rng}{The RNG and starting seed.}
    \item{quiet}{If TRUE then no information is printed during estimation.}
    \item{simulation.args}{A list of any arguments to pass to simulate.}
    \item{estimation}{A character string indicating the estimation routine to use.}
    \item{estimation.args}{A list of any arguments to pass to the estimation routine.}
    \item{criterion}{
    A function to apply to the results of estimation to extract 
    the information which is to be retained.}
    \item{criterion.args}{
    A list of any arguments to be passed to the criterion function.}
    \item{Spawn}{a logical indication if Splus For loops should be used.}
    \item{obj}{an object.}
}
\value{
A list with element \code{result} of length replications, each element containing the
results of criterion(estimation(simulate(model))). Other elements of
the list contain information from the supplied arguments.
}
\details{
estimation.args and criterion.args should be NULL if no args are needed.
If model is an object of class 'EstEval' or 'simulation'
then the model and the seed!!! are extracted so the evaluation will be
based on the same generated sample.
criterion can be { 'coef', 'roots', 'TSmodel', 'TSestModel'}
With the default (coef) or with TSmodel the other criteria can be reconstructed
(when the estimation method finds a known form for the model - which
is not always the case, for example with estBlackBox methods).
If criterion = 'roots' then criterion.args= list(verbose=FALSE) is advised.
}
\seealso{
\code{\link[dse1]{simulate}}
\code{\link{MonteCarloSimulations}}
\code{\link{distribution}}
\code{\link{forecastCovWRTtrue}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(TSdata(output=outputData(eg1.DSE.data.diff)))
    z <-  EstEval(model, 
       estimation="estVARXls", estimation.args=list(max.lag=2))
    tfplot(z)
    zz <-  EstEval(model, 
       estimation="estVARXls", estimation.args=list(max.lag=2),
       simulation.args=list(sampleT=50, sd=1.5))
    is.EstEval(z)
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{MonteCarloSimulations}
\alias{MonteCarloSimulations}
\alias{MonteCarloSimulations.default}
\alias{MonteCarloSimulations.TSmodel}
\alias{MonteCarloSimulations.TSestModel}
\alias{MonteCarloSimulations.EstEval}
\alias{MonteCarloSimulations.MonteCarloSimulations}
\alias{is.MonteCarloSimulations}

\title{Generate simulations}
\description{Run multiple simulations}
\usage{
    is.MonteCarloSimulations(obj)
    MonteCarloSimulations(model, simulation.args=NULL, 
           replications=100, rng=NULL, quiet =FALSE, ...)
    \method{MonteCarloSimulations}{default}(model, simulation.args = NULL, 
 		replications = 100, rng = NULL, quiet =FALSE, ...)
    \method{MonteCarloSimulations}{TSmodel}(model, simulation.args=NULL,
          replications=100, rng=NULL, quiet=FALSE, 
	  Spawn=if (exists(".SPAWN")) .SPAWN else FALSE, ...)
    \method{MonteCarloSimulations}{TSestModel}(model, simulation.args=NULL, 
           replications=100, rng=NULL, quiet=FALSE, ...)
    \method{MonteCarloSimulations}{EstEval}(model, simulation.args=NULL,
            replications=100, rng=getRNG(model),  quiet=FALSE, ...)
    \method{MonteCarloSimulations}{MonteCarloSimulations}(model, 
       simulation.args=NULL, replications=100, rng=getRNG(model),  quiet=FALSE, ...)
}
\arguments{
    \item{model}{an object from which a model can be extracted. The model must
         have an associated \code{simulation} method (e.g. a TSmodel).}
    \item{simulation.args,}{
       A list of arguments in addition to model which are passed to simulate.}
    \item{replications}{The number of simulations.}
    \item{rng}{The RNG and starting seed.}
    \item{quiet}{logical indicating if printing and many warning messages should
       be suppressed.}
    \item{Spawn}{If TRUE "For" loops are used in Splus.}
    \item{obj}{an object.}
    \item{...}{arguments passed to other methods.}
}
\value{A list of simulations.}
\details{
This function runs many simulations using \code{simulate}.
Often it not be necessary to do this since the seed can be used to
reproduce the sample and many functions for testing estimation methods, etc.,
will produce samples as they proceed. This function is useful for verification
and for looking at the stochastic properties of the output of a model.
If \code{model} is an object of class \code{EstEval} or 
\code{simulation}
then the model and the seed!!! are extracted so the same sample will be 
generated. The default method expects the result of \code{simulate(model)} to be
a matrix.
There is a \code{tfplot} method (time series plots of the simulations) and a
\code{distribution} method for the result. The latter plots kernel estimates 
of the distribution of the simulations at specified periods.
}
\seealso{
\code{\link[dse1]{simulate}}
\code{\link{EstEval}}
\code{\link{distribution}}
\code{\link{forecastCovWRTtrue}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <-  MonteCarloSimulations(model, simulation.args=list(sampleT=100))
tfplot(z)
distribution(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{TSdata.forecastCov}
\alias{TSdata.forecastCov}
\alias{TSestModel.coefEstEval}
\alias{TSmodel.forecastCov}
\alias{TSmodel.coefEstEval}

\title{TS Extractor Specific Methods}
\description{See the generic function description.}
\usage{
    \method{TSdata}{forecastCov}(data, ...)
    \method{TSestModel}{coefEstEval}(obj)
    \method{TSmodel}{forecastCov}(obj, select=1, ...)
    \method{TSmodel}{coefEstEval}(obj, ...)
}
\arguments{
    \item{data}{an object from which to extract the TSdata.}
    \item{obj}{an object from which to extract the TSmodel or TSestModel.}
    \item{select}{an integer indicating which of multiple models to extract.}
    \item{...}{arguments to be passed to other methods.}
}

\seealso{ 
    \code{\link[dse1]{TSdata}} 
    \code{\link[dse1]{TSestModel}} 
    \code{\link[dse1]{TSmodel}} 
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{coef.TSmodelEstEval}
\alias{coef.TSmodelEstEval}
\alias{coef.TSestModelEstEval}
\title{Specific Methods for coef}
\description{See the generic function description.}
\usage{
    \method{coef}{TSmodelEstEval}(object, criterion.args=NULL, ...)
    \method{coef}{TSestModelEstEval}(object, criterion.args=NULL, ...)
}
\arguments{
    \item{object}{an object (model) from which to extract
       coefficients(parameters).}
    \item{criterion.args}{arguments to be passed to this method when it is
         called by \code{EstEval}.}
   \item{...}{(further arguments, currently disregarded).}
}
\details{
    The methods ***.ee are intended mainly to be called from \code{EstEval} as
    criterion for evaluating an estimation method. See \code{coef}. }
\seealso{
    \code{\link{EstEval}} 
    \code{\link{coef}} }
%\keyword{DSE}
\keyword{ts}


\eof
\name{combine.forecastCov}
\alias{combine.forecastCov}
\alias{combine.forecastCovEstimatorsWRTdata}
\alias{combine.forecastCovEstimatorsWRTtrue} 

\title{Combine 2 Forecast Cov Objects}
\description{Combine 2 forecastCov type objects.}
\usage{
    \method{combine}{forecastCov}(e1, e2)
    \method{combine}{forecastCovEstimatorsWRTdata}(e1, e2)
    \method{combine}{forecastCovEstimatorsWRTtrue}(e1, e2)
}
\arguments{
    \item{e1, e2}{
    Objects as returned by functions which calculate forecast covariances.}
}
\value{
An object as returned by functions which calculate forecast covariances.
}
\details{
Functions which calculate forecast covariances return lists. Usually multiple 
estimation techniques or models will be combined together when the object is
first formed. However, it is sometimes useful to add results calculated later
without re-doing the initial object.
}
\seealso{
\code{\link[dse1]{combine}},
\code{\link{forecastCovEstimatorsWRTdata}},
\code{\link{forecastCovEstimatorsWRTtrue}}
\code{\link{forecastCov}}
}
\examples{#z <-  combine(obj1, obj2)}
%\keyword{DSE}
\keyword{ts}




\eof
\name{distribution.MonteCarloSimulations}
\alias{distribution.MonteCarloSimulations}
\title{Generate distribution plots of Monte Carlo simulations}
\description{Generate distribution plots of Monte Carlo simulations.}
\usage{
    \method{distribution}{MonteCarloSimulations}(obj,
        series=seq(dim(obj$simulations)[2]),
        x.sections=TRUE, periods=1:3, graphs.per.page=5, ...)
}
\arguments{
    \item{obj}{The result of MonteCarloSimulations.}
    \item{series}{The series which should be plotted. The default gives all series.}
    \item{x.sections}{If TRUE then kernel density estimates are plotted 
       for periods indicated by periods. If FALSE then a time series plots 
       of the mean and estimates 1  and 2 standard deviations from the 
       mean. Periods is ignored if x.sections is FALSE.}
    \item{periods}{
       The periods at which the distribution should be calculated and plotted.
       The default gives the first three.}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{...}{(further arguments, currently disregarded).}
}
\value{None}
\details{
Kernel estimates of the densities (series by series, not joint densities) are estimated using ksmooth (if available) or density (if available) to produces density plots.
Output graphics can be paused between pages by setting par(ask=TRUE).
}
\seealso{
\code{\link{tfplot.MonteCarloSimulations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <-  MonteCarloSimulations(model)
distribution(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{distribution}
\alias{distribution}
\alias{distribution.default}
\alias{distribution.TSdata}
\alias{distribution.coefEstEval}
\alias{distribution.rootsEstEval}
\title{Plot distribution of estimates}
\description{Plot distribution of estimates.}
\usage{
    distribution(obj, ...)
     \method{distribution}{TSdata}(obj, bandwidth=0.2, 
             select.inputs = seq(length= nseriesInput(obj)),
             select.outputs= seq(length=nseriesOutput(obj)), ...)
     \method{distribution}{default}(obj, bandwidth=0.2, series=NULL, ...)
     \method{distribution}{coefEstEval}(obj,  Sort=FALSE, bandwidth=0.2,
	graphs.per.page=5, ...)
     \method{distribution}{rootsEstEval}(obj, mod=TRUE, invert=FALSE, Sort=FALSE,
        bandwidth=0.2, select=NULL, ...)
}
\arguments{
    \item{obj}{an object as returned by EstEval.} 
    \item{Sort}{if Sort is true then sort is applied. This helps (a bit)
       with estimation methods like black.box which may not return parameters
       of the same length or in the same order.}
    \item{bandwidth}{passed to \code{density} or \code{ksmooth}.}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{series}{series to be plotted. (passed to selectSeries)}
    \item{select.inputs}{series to be plotted. (passed to selectSeries)}
    \item{select.outputs}{series to be plotted. (passed to selectSeries)}
    \item{...}{arguments passed to other methods.}
    \item{invert}{logical indicating if the inverse of roots should be plotted}
    \item{mod}{logical indicating if the modulus of roots should be plotted}
    \item{select}{integer vector indicating roots to be plotted. If select 
        is not NULL then roots are sorted by magnitude and only the 
	indicated roots are plotted. For example, select=c(1,2)  will plot only 
	the two largest roots.}
}
\value{None}
\details{
\code{ksmooth} is applied if available to get a smoothed 
estimate of the distribution of the estimates. If \code{ksmooth} is not
available then \code{density} is applied if it is available.
}
\seealso{
\code{\link{EstEval}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(TSdata(output=outputData(eg1.DSE.data.diff)), max.lag=2)
# now use this as the true model
z <-  EstEval(model, 
    estimation="estVARXls", estimation.args=list(max.lag=2))
distribution(z) 
tfplot(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estimateModels}
\alias{estimateModels}
\alias{is.estimatedModels}

\title{Estimate Models}
\description{Estimate models using given estimation method}
\usage{
    estimateModels(data, estimation.sample = NULL, trend =FALSE, quiet =FALSE,
                       estimation.methods = NULL)
    is.estimatedModels(obj)
}
\arguments{
    \item{data}{An object of class TSdata.}
    \item{estimation.methods}{
       A named list with the names indicating the estimation method and the
       value associated with the name is a list of arguments for each the
       method indicated. Its value should be NULL if no args are needed.}
    \item{estimation.sample}{
       An integer indicating the number of points in the sample to use for
       estimation. If it is NULL the whole sample is used.}
    \item{trend}{
       If trend is TRUE then a linear trend is calculated and returned as
       the element \code{trend.coef}.}
    \item{quiet}{
       If quiet is TRUE then most printing and some warning messages are
       suppressed.}
    \item{obj}{An object.}
}
\value{
Element \code{multi.model} in the result is a list of the same length as 
estimation.methods with resulting models as elements.
}
\details{
    Estimate models from data with estimation methods indicated by 
    estimation.methods. This is primarily a utility for other functions.
}
\seealso{

\code{\link{EstEval}},
\code{\link{outOfSample.forecastCovEstimatorsWRTdata}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  estimateModels(eg1.DSE.data.diff, estimation.methods= list(
        bft=list(verbose=FALSE), 
        estVARXar=list(max.lag=3)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{estimatorsHorizonForecastsWRTdata}
\alias{estimatorsHorizonForecastsWRTdata}
\title{Estimate models and forecast at given horizons}
\description{Estimate models and forecast at given horizons.}
\usage{
    estimatorsHorizonForecastsWRTdata(data, 
                       estimation.sample=.5, horizons=1:12,quiet=FALSE,
                       estimation.methods=NULL)
}
\arguments{
    \item{data}{A TSdata object.}
    \item{estimation.methods}{
    A list of estimation methods to use. (See estimateModels.)}
    \item{estimation.sample}{The portion  of the sample to use for estimation.}
    \item{horizons}{The horizons for which forecasts are to be produced.}
    \item{quiet}{If true no estimation information is printed.}
}
\value{A list of forecasts at different horizons as returned by horizonForecasts.
}
\details{
estimation.sample indicates the part of the data to use for estimation.
If estimation.sample is less than or equal 1.0 it is
used to indicate the portion of points to use for estimation.
Otherwise it should be an integer and is used to indicate the number
of points from the beginning of the sample to use for estimation. 
}
\seealso{
\code{\link{estimateModels}},
\code{\link{horizonForecasts}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  estimatorsHorizonForecastsWRTdata(eg1.DSE.data.diff, 
    estimation.methods=list(estVARXls=list(max.lag=3), 
                estVARXar=list(max.lag=3)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{excludeForecastCov}
\alias{excludeForecastCov}
\title{Filter Object to Remove Forecasts}
\description{Filter object to remove forecasts.}
\usage{
    excludeForecastCov(obj, exclude.series=NULL)
}
\arguments{
    \item{obj}{An object as returned by stripMine.}
    \item{exclude.series}{An indication of series to which should be excluded.}
}
\value{
The returned result is a forecastCov object like obj, but filtered to
remove any forecasts from models which depend on the series which are
indicated for exclusion.
}
\details{
Exclude results which depend on the indicated series from a 
(forecastCovEstimatorsWRTdata.subsets forecastCov) object.
}
\seealso{
\code{\link{minForecastCov}},
\code{\link{selectForecastCov}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- stripMine(eg1.DSE.data.diff, essential.data=c(1,2),
                   estimation.methods=list(estVARXls=list(max.lag=3)))
z <-  excludeForecastCov(z, exclude.series=3)
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{extractforecastCov}
\alias{extractforecastCov}
\alias{extractforecastCov.forecastCovEstimatorsWRTdata}
\alias{extractforecastCov.forecastCovEstimatorsFromModel}

\title{Extract Forecast Covariance}
\description{extract forecastCov from objects}
\usage{
    extractforecastCov(e, n)
    \method{extractforecastCov}{forecastCovEstimatorsWRTdata}(e, n)
    \method{extractforecastCov}{forecastCovEstimatorsFromModel}(e, n)
}
\arguments{
    \item{e}{A "forecastCovEstimatorsWRTdata", "forecastCov" object.}
    \item{n}{A vector on integers.}
}
\value{A forecastCov object.}
\details{
    Select a subset of models and their forecast covariances from a
    larger object.
}
\seealso{
    \code{\link{forecastCov}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{featherForecasts}
\alias{featherForecasts}
\alias{featherForecasts.TSdata}
\alias{featherForecasts.TSmodel}
\alias{featherForecasts.TSestModel}
\alias{is.featherForecasts}

\title{Multiple Horizon-Step Ahead Forecasts}
\description{Calculate multiple horizon-step ahead forecasts.}
\usage{
    featherForecasts(obj, ...)
    \method{featherForecasts}{TSestModel}(obj, data=NULL, ...)
    \method{featherForecasts}{TSdata}(obj, model, ...)
    \method{featherForecasts}{TSmodel}(obj, data, horizon=36,
             from.periods =NULL, ...)
    is.featherForecasts(obj)
}
\arguments{
    \item{obj}{an object of class TSmodel.}
    \item{data}{an object of class TSdata.}
    \item{model}{an object of class TSmodel.}
    \item{from.periods}{the starting points to use for forecasts.}
    \item{horizon}{the number of periods to forecast.}
    \item{...}{for a TSmodel additional arguments are passed to \code{l()}}
}
\value{
The result is a list of class featherForecasts with elements \code{model} (a 
TSestModel), \code{data}, \code{from.periods}, \code{featherForecasts}.
The element \code{featherForecasts} is a list with length(from.periods) elements, each of which
is a tframed matrix.
There is a plot method for this class.
}
\details{
Calculate multiple horizon-step ahead forecasts 
ie. use the samples indicated by from.periods to calculate forecasts for 
horizon periods.
Thus, for example, the result of
featherForecasts(model, data, from.periods=c(200,250,300))
would be forecasts for 1 through 36 steps ahead (the default),
starting at the 200th,250th, and 300th point of outputData(data).
This function assumes that inputData(data) (the exogenous 
variable) is as long as necessary for the most future forecast.
}
\seealso{
\code{\link{forecast}},
\code{\link{horizonForecasts}}
}
\examples{
if(is.R()) data("egJofF.1dec93.data", package="dse1")
model <- estVARXls(egJofF.1dec93.data)
pr <- featherForecasts(model, egJofF.1dec93.data)
tfplot(pr)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecast}
\alias{forecast}
\alias{forecast.TSdata}
\alias{forecast.TSmodel}
\alias{forecast.TSestModel}
\alias{is.forecast}

\title{Forecast Multiple Steps Ahead}
\description{Calculate forecasts multiple steps ahead.}
\usage{
    is.forecast(obj)
    forecast(obj, ...)
    \method{forecast}{TSmodel}(obj, data,  horizon=36,
       conditioning.inputs=NULL, 
       conditioning.inputs.forecasts=NULL, percent=NULL, ...)
    \method{forecast}{TSestModel}(obj, ...)
    \method{forecast}{TSdata}(obj, model, ...)
}
\arguments{
    \item{obj}{An object of a class for which a specific method is available.}
    \item{model}{An object of class TSmodel.}
    \item{data}{An object of class TSdata.}
    \item{conditioning.inputs}{
    A time series matrix or list of time series matrices to use as input variables.}
    \item{conditioning.inputs.forecasts}{
    A time series matrix or list of time series matrices to append to input
variables for the forecast periods.}
    \item{horizon }{The number of periods to forecast.}
    \item{percent}{
    A vector indication percentages of the last input to use for 
    forecast periods.}
    \item{...}{arguments passed to other methods.}
}
\value{
The result is an object of class forecast which is a list with 
elements \code{model}, \code{horizon}, \code{conditioning.inputs}, 
\code{percent}, \code{pred} and \code{forecast}.
The element \code{forecast} is a list with TSdata objects as elements, one for each element
in the list conditioning.inputs.
The element \code{pred} contains the one-step ahead forecasts for the preiods when output data
is available. 
There is a plot method for this class.
}
\details{
Calculate (multiple) forecasts from the end of data to a horizon determined
either from supplied input data or the argument horizon (more details below).
In  the case of a model with no inputs the horizon is determined by the
argument horizon.
In the case of models with inputs, on which the forecasts
are conditioned, the argument horizon is ignored (except when percent is
specified) and the actual horizon is determined by the inputs in the 
following way:
If inputs are not specified by optional arguments (as below) then the default
will be to use inputData(data). This will be the same as the function l() unless
inputData(data) is longer than
outputData(data) (after NAs are trimmed from each separately).
Otherwise, if conditioning.inputs is specified it is used for inputData(data).
It must be a time series matrix or a list of time series matrices each
of which is used in turn as inputData(data). The default above is the same as
forecast(model, trimNA(data), conditioning.inputs=trimNA(inputData(data)) )
Otherwise, if conditioning.inputs.forecasts is specified it is appended 
to inputData(data). It must be a time series  
matrix or a list of time series matrices each of which is 
appended to inputData(data) and the concatenation used as conditioning.inputs.
Both conditioning.inputs and conditioning.inputs.forecasts should not be
specified.
Otherwise, if percent is specified then conditioning.inputs.forecasts are set
to percent/100 times the value of input corresponding to the last period
of outputData(data) and used for horizon periods. percent can be a vector, 
in which case each value is applied in turn. ie c(90,100,110) would would 
give results for conditioning.input.forecasts 10 percent above and below 
the last value of input.
}
\seealso{
\code{\link{featherForecasts}},
\code{\link{horizonForecasts}}
}
\examples{
    if(is.R()) data("egJofF.1dec93.data", package="dse1")
    model <- estVARXls(window(egJofF.1dec93.data, end=c(1985,12)))
    pr <- forecast(model, conditioning.inputs=inputData(egJofF.1dec93.data))
    #tfplot(pr) Rbug 0.90.1
    is.forecast(pr)
} 
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecastCov}
\alias{forecastCov}
\alias{forecastCov.TSdata}
\alias{forecastCov.TSmodel}
\alias{forecastCov.TSestModel}
\alias{is.forecastCov}

\title{Forecast covariance for different models}
\description{Calculate the forecast covariance for different models.}
\usage{
    is.forecastCov(obj)
    forecastCov(obj, ..., data=NULL, horizons=1:12, discard.before=NULL,
       zero=FALSE, trend=FALSE, estimation.sample= NULL, compiled=.DSECOMPILED)
    \method{forecastCov}{TSmodel}(obj, ..., data=NULL,
       horizons=1:12, discard.before=NULL, 
       zero=FALSE, trend=FALSE, estimation.sample= periods(data), compiled=.DSECOMPILED)
    \method{forecastCov}{TSestModel}(obj, ..., data=TSdata(obj), 
       horizons=1:12, discard.before=NULL, zero=FALSE, trend=FALSE, 
       estimation.sample= periods(TSdata(obj)), compiled=.DSECOMPILED)
    \method{forecastCov}{TSdata}(obj, ..., data=NULL,
       horizons=1:12, discard.before=1,
       zero=FALSE, trend=FALSE, estimation.sample= NULL,
        compiled=.DSECOMPILED)
}
\arguments{
    \item{obj}{TSdata or one or more TSmodels or TSestModels}
    \item{data}{an object of class TSdata.}
    \item{discard.before}{
       period before which forecasts should be discarded when calculating covariance.}
    \item{horizons}{horizons for which forecast covariance should be calculated.}
    \item{zero}{if TRUE the covariance is calculated for a forecast of zero.}
    \item{trend}{if TRUE the covariance is calculated for a forecast of trend.}
    \item{estimation.sample}{portion of the sample to use for calculating the trend.}
    \item{compiled}{a logical indicating if compiled code should be
      used. (Usually true except for debugging.)}
    \item{...}{arguments passed to other methods.}
}
\value{
    A list with the forecast covariance for supplied models on the given
    sample. This is in the element \code{forecastCov} of the result. Other
    elements contain information in the arguments.
}
\details{
Calculate the forecast cov of obj relative to data. If obj is TSdata
then the output data is used as the forecast. For other classes of obj
TSmodel(obj) is used with data to produce a forecast. TSmodel() is also
applied to each element of ... to extract a model. All models should
work with data. If obj is a TSestModel and data is NULL then TSdata(obj)
is used as the data.
This is multiple applications of forecastCovSingle.TSmodel
discard.before is an integer indicating the number of points in the
beginning of forecasts to discard before calculating covariances.
If it is the default, NULL, then the default (minimumStartupLag) will
be used for each model and the default (1) will be used for trend and zero.
If zero  is TRUE then forecastCov is also calculated for a forecast of zero.
If trend is TRUE then forecastCov is also calculated for a forecast of a linear 
trend using data to estimation.sample.
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model1 <- estVARXar(eg1.DSE.data.diff)
    model2 <- estVARXls(eg1.DSE.data.diff)
    z <-  forecastCov(model1, model2, data=trimNA(eg1.DSE.data.diff))
    is.forecastCov(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecastCovCompiled}
\alias{forecastCovCompiled}
\alias{forecastCovCompiled.ARMA}
\alias{forecastCovCompiled.SS}
\alias{forecastCovCompiled.innov}
\alias{forecastCovCompiled.nonInnov}
\alias{forecastCovSingle.TSmodel}

\title{Forecast covariance for different models - internal}
\description{See forecastCov.}
\usage{
    forecastCovCompiled(model, data, horizons = 1:12,
         discard.before=minimumStartupLag(model))
    \method{forecastCovCompiled}{ARMA}(model, data, horizons = 1:12,
         discard.before=minimumStartupLag(model))
    \method{forecastCovCompiled}{SS}(model, data, horizons = 1:12,
         discard.before=minimumStartupLag(model))
    \method{forecastCovCompiled}{innov}(model, data, horizons = 1:12,
         discard.before=minimumStartupLag(model))
    \method{forecastCovCompiled}{nonInnov}(model, data, horizons = 1:12,
         discard.before=minimumStartupLag(model))
     forecastCovSingle.TSmodel( model, data=NULL, horizons=1:12, 
          discard.before=minimumStartupLag(model), compiled=.DSECOMPILED)
}
\arguments{
    \item{obj}{TSdata or one or more TSmodels or TSestModels}
    \item{data}{an object of class TSdata.}
    \item{discard.before}{
       period before which forecasts should be discarded when calculating covariance.}
    \item{horizons}{horizons for which forecast covariance should be calculated.}
    \item{zero}{if TRUE the covariance is calculated for a forecast of zero.}
    \item{trend}{if TRUE the covariance is calculated for a forecast of trend.}
    \item{estimation.sample}{portion of the sample to use for calculating the trend.}
    \item{compiled}{a logical indicating if compiled code should be
      used. (Usually true except for debugging.)}
    \item{...}{arguments passed to other methods.}
}
\value{
    A list with the forecast covariance for supplied models on the given
    sample. This is in the element \code{forecastCov} of the result. Other
    elements contain information in the arguments.
}
\details{
Not to be called by users. See \code{forecastCov}.
}

%\keyword{DSE}
\keyword{internal}


\eof
\name{forecastCovEstimatorsWRTdata}
\alias{forecastCovEstimatorsWRTdata}
\alias{is.forecastCovEstimatorsWRTdata}

\title{Calculate Forecast Cov of Estimators WRT Data}
\description{forecast covariance of estimated models with respect to a
 given sample}
\usage{
    forecastCovEstimatorsWRTdata(data, estimation.sample=NULL, 
                       compiled=.DSECOMPILED, discard.before=10,
                       horizons=1:12, zero=FALSE, trend=FALSE,quiet=FALSE,
                       estimation.methods=NULL)
    is.forecastCovEstimatorsWRTdata(obj)
}
\arguments{
    \item{data}{an object of class TSdata.}
    \item{estimation.methods}{a list as used by estimateModels.}
    \item{discard.before}{
       an integer indicating the number of points in the
       beginning of forecasts to discard for calculating covariances.}
    \item{zero}{if TRUE then forecastCov is also calculated for a 
       forecast of zero.}
    \item{trend}{
      if TRUE then forecastCov is also calculated for a forecast of a linear trend.}
    \item{estimation.sample}{
       an integer indicating the number of points in the
       sample to use for estimation. If it is NULL the whole sample is used.}
    \item{horizons}{horizons for which forecast covariance should be calculated.}
    \item{quiet}{if TRUE then estimation information is not printed.}
    \item{compiled}{a logical indicating if the compiled version of the code
       should be used. (FALSE would typically only be used for debugging.)}
    \item{obj}{an object.}
}
\value{
A list with the forecast covariance for supplied models on the given
sample. This is in the element \code{forecastCov} of the result. Other
elements contain information in the arguments. 
}
\details{
Calculate the forecasts cov of models estimated from data with estimation
methods indicated by estimation.methods  (see estimateModels).
estimation.sample is an integer indicating the number of points in the
sample to use for estimation. If it is NULL the whole sample is used.
}
\seealso{
\code{\link{outOfSample.forecastCovEstimatorsWRTdata}},
\code{\link{estimateModels}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- forecastCovEstimatorsWRTdata(eg1.DSE.data.diff, 
    estimation.methods=list(estVARXls=list(max.lag=4)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecastCovEstimatorsWRTtrue}
\alias{forecastCovEstimatorsWRTtrue}
\alias{is.forecastCovEstimatorsWRTtrue}

\title{Compare Forecasts Cov Relative to True Model Output}
\description{}
\usage{
    forecastCovEstimatorsWRTtrue(true.model, rng=NULL,
                       simulation.args=NULL,
                       est.replications = 2, pred.replications = 2,
                       discard.before = 10, horizons = 1:12, quiet =FALSE,
                       estimation.methods=NULL, compiled=.DSECOMPILED, 
                       Spawn=if (exists(".SPAWN")) .SPAWN else FALSE
		       )
    is.forecastCovEstimatorsWRTtrue(obj)
}
\arguments{
    \item{true.model}{An object of class TSmodel.}
    \item{estimation.methods}{A list as used by estimateModels.}
    \item{simulation.args}{an arguments to be passed to simulate.}
    \item{est.replications}{An arguments to be passed to simulate.}
    \item{pred.replications}{An arguments to be passed to simulate.}
    \item{discard.before}{
      An integer indicating the number of points in the
      beginning of forecasts to discard for calculating covariances.}
    \item{horizons}{Horizons for which forecast covariance should be calculated.}
    \item{rng}{If specified then it is used to set RNG.}
    \item{Spawn}{If TRUE then Splus For loops are used.}
    \item{quiet}{If TRUE then some messages are not printed.}
    \item{compiled}{a logical indicating if the compiled version of the code
       should be used. (FALSE would typically only be used for debugging.)}
    \item{obj}{an object.}
}
\value{
The returned results has element
  \code{forecastCov.true, forecastCov.zero, forecastCov.trend} containing 
     covariances averaged over estimation replications and simulation
     replications (forecasts will not change but simulated data will).
   \code{forecastCov} a list of the same length as estimation.methods with each
     element containing covariances averaged over estimation replications 
     and simulation replications.
   \code{estimatedModels} a list of length est.replications, with each elements as
     returned by estimateModels, thus each element has \code{multi.model} as a
     subelement containing models for different estimation techniques.  
     So, eg. \code{estimatedModels[[2]]$multi.model[[1]]}  in the result will
     be the model from the first estimation technique in the second replication.
}
\details{
Calculate the forecasts cov of models estimated from simulations of 
  true.model with estimation methods indicated by estimation.methods (see 
       estimateModels). 
This function makes multiple calls to forecastCovWRTtrue.
}
\seealso{
\code{\link{forecastCovWRTtrue}}
\code{\link{forecastCovEstimatorsWRTdata}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
true.model <- estVARXls(eg1.DSE.data.diff) # just to have a starting model
z <-  forecastCovEstimatorsWRTtrue(true.model, 
    estimation.methods=list(estVARXls=list(max.lag=4)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecastCovReductionsWRTtrue}
\alias{forecastCovReductionsWRTtrue}

\title{Forecast covariance for different models}
\description{Calculate the forecast covariance for different models.}
\usage{
   forecastCovReductionsWRTtrue(true.model, rng=NULL,
                       simulation.args=NULL,
                       est.replications=2, pred.replications=2,
                       discard.before=10, horizons=1:12,quiet=FALSE,
                       estimation.methods=NULL,
                       criteria=NULL, compiled=.DSECOMPILED, 
                       Spawn=if (exists(".SPAWN")) .SPAWN else FALSE)
}
\arguments{
    \item{true.model}{An object of class TSmodel or TSestModel.}
    \item{discard.before}{
       An integer indicating the number of points in the
       beginning of forecasts to discard for calculating covariances.}
    \item{est.replications}{an interger indicating the number of times
       simulation and estimation are repeated.}
    \item{pred.replications}{an argument passed to \code{forecastCovWRTtrue}.}
    \item{simulation.args}{
       A list of any arguments which should be passed to simulate
       in order to simulate the true model.}
    \item{horizons}{Horizons for which forecast covariance should be calculated.}
    \item{rng}{If specified then it is used to set RNG.}
    \item{Spawn}{If TRUE then Splus For loops are used.}
    \item{quiet}{If TRUE then some messages are not printed.}
    \item{estimation.methods}{a list as used by estimateModels.}
    \item{criteria}{a ...}
    \item{compiled}{a logical indicating if compiled code should be
      used. (Usually true except for debugging.)}
}
\value{
    A list ...
}
\details{
 Calculate the forecasts cov of reduced models estimated from simulations of
  true.model with an estimation method indicated by estimation.methods. 
   (estimation.methods is as in estimation.models BUT ONLY THE FIRST IS USED.)
  discard.before is an integer indicating 1+the number of points in the
  beginning of forecasts to discard for calculating forecast covariances.
  criteria can  be a vector of criteria as in informationTests,
   (eg c("taic", "tbic") in which case the "best" model for each criteria
   is accounted separately. (ie. it is added to the beginning of the list of
  estimated models)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecastCovWRTtrue}
\alias{forecastCovWRTtrue}
\alias{is.forecastCovWRTdata}

\title{Compare Forecasts to True Model Output}
\description{
    Generate forecasts and compare them against the output of a true model.}
\usage{
    forecastCovWRTtrue(models, true.model, 
        pred.replications=1, simulation.args=NULL, quiet=FALSE, rng=NULL, 
	compiled=.DSECOMPILED,
        horizons=1:12, discard.before=10, trend=NULL, zero=NULL, 
	Spawn=if (exists(".SPAWN")) .SPAWN else FALSE)
    is.forecastCovWRTdata(obj)
}
\arguments{
    \item{models}{A list of objects of class TSmodel.}
    \item{true.model}{An object of class TSmodel or TSestModel.}
    \item{discard.before}{
    An integer indicating the number of points in the
    beginning of forecasts to discard for calculating covariances.}
    \item{zero}{If TRUE then forecastCov is also calculated for a forecast of zero.}
    \item{trend}{If TRUE then forecastCov is also calculated for a forecast of a linear trend.}
    \item{pred.replications}{integer indicating the number of times simulated
       data is generated.}
    \item{simulation.args}{
    A list of any arguments which should be passed to simulate
    in order to simulate the true model.}
    \item{horizons}{Horizons for which forecast covariance should be calculated.}
    \item{rng}{If specified then it is used to set RNG.}
    \item{Spawn}{If TRUE then Splus For loops are used.}
    \item{quiet}{If TRUE then some messages are not printed.}
    \item{compiled}{a logical indicating if compiled code should be
      used. (Usually true except for debugging.)}
    \item{obj}{an object.}
}
\value{
A list with the forecast covariance for supplied models on samples
generated by the given true model. This is in the element \code{forecastCov}
of the result. Other elements contain information in the arguments.
}
\details{
The true model is used to generate
data and for each generated data set the forecasts of the 
models are evaluated against the simulated data.
If trend is not null it is treated as a model output (forecast) and
should be the same dimension as a simulation of the models with 
simulation.args. If zero is not null a zero forecast is also evaluated. If 
simulating the true model requires input data then a convenient way to do this
is for true.model to be a TSestModel. Otherwise, input data should be passed
in simulation.args
}
\seealso{
\code{\link{forecastCovEstimatorsWRTdata}}
\code{\link[dse1]{simulate}}
\code{\link{EstEval}}
\code{\link{distribution}}
\code{\link{MonteCarloSimulations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
true.model <- estVARXls(eg1.DSE.data.diff) # A starting model TSestModel
data <- simulate(true.model)
models <- list(TSmodel(estVARXar(data)),TSmodel(estVARXls(data)))
z <-  forecastCovWRTtrue( models, true.model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{forecasts}
\alias{forecasts}
\alias{forecasts.forecast}
\alias{forecasts.featherForecasts}
\alias{forecasts.horizonForecasts}
\title{Extract Forecasts}
\description{Extract forecasts from and object.}
\usage{
    forecasts(obj)
    \method{forecasts}{forecast}(obj)
    \method{forecasts}{featherForecasts}(obj)
    \method{forecasts}{horizonForecasts}(obj)
}
\arguments{
    \item{obj}{An object which contains forecasts.}
}
\value{The forecasts from an object which contains forecasts.}
\details{
    This generic method extracts the forecasts (only) from objects returned 
    by other methods that calculate forecasts. Usually the objects returned
    by the methods which calculate forecasts contain additional
    information which is not returned by this extractor. 
}
\seealso{
    \code{\link{forecast}}
}
\examples{
    if(is.R()) data("egJofF.1dec93.data", package="dse1")
    model <- estVARXls(window(egJofF.1dec93.data, end=c(1985,12)))
    pr <- forecast(model, conditioning.inputs=inputData(egJofF.1dec93.data))
    z <- forecasts(pr)
} 
%\keyword{DSE}
\keyword{ts}


\eof
\name{genMineData}
\alias{genMineData}
\alias{build.diagonal.model}
\alias{build.input.models}

\title{Generate Data}
\description{Generate data for Monte Carlo experiments}
\usage{
    genMineData(umodel, ymodel, uinput=NULL, sampleT=100, 
      unoise=NULL, usd=1,ynoise=NULL, ysd=1, rng=NULL)
    build.input.models(all.data, max.lag=NULL)
    build.diagonal.model(multi.models)
}
\arguments{
    \item{umodel}{Model for input data.}
    \item{ymodel}{Model for output data.}
    \item{sampleT}{Number of periods of data to generate.}
    \item{unoise}{Input noise.}
    \item{usd}{Standard deviationof input noise.}
    \item{ynoise}{Output noise.}
    \item{ysd}{Standard deviation of output noise.}
    \item{rng}{RNG setting.}
    \item{multi.models}{}
    \item{all.data}{}
    \item{max.lag}{}
    \item{uinput}{}
}
\value{A TSdata object.}
\details{
    This function generates test data using specified models.
    umodel is used to generate data input data and
    ymodel is used to generate data corresponding output data.
    The result of umodel is used as input to ymodel so the input dimension of
    ymodel should be the output dimension of umodel. Typically the ymodel would
    be degenerate in some of the input variables so the effective inputs 
    are a subset.
    If noise is NULL then an normal noise will be generated by simulate.
    This will be iid N(0,I).
    The RNG will be set first if it is specified. 
    If unoise or ynoise are specified they should be as expected by simulate
    for the specified umodel and ymodel.
    
    \code{genMineData} uses \code{build.input.models} which makes a list of univariate 
    models, one for each series in \code{inputData(data)} and 
    \code{build.diagonal.model} which builds one diagonal model from a list 
    of models returned  by \code{build.input.models}. It uses the AR part only.
}

\seealso{
    \code{\link[dse1]{simulate}}
}

\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    umodel <- build.diagonal.model(
            build.input.models(eg1.DSE.data.diff, max.lag=2))
    z  <- TSdata(output=outputData(eg1.DSE.data.diff), 
                 input = inputData(eg1.DSE.data.diff))
    ymodel <- TSmodel(estVARXls(z, max.lag=3))   
    sim.data <- genMineData(umodel, ymodel)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{generateSSmodel}
\alias{generateSSmodel}
\title{Randomly generate a state space model}
\description{Randomly generate a state space model.}
\usage{
    generateSSmodel(m,n,p, stable=FALSE)
}
\arguments{
    \item{n,m,p}{Input, state and output dimensions.}
    \item{stable}{TRUE or FALSE indicating if the model must be stable.}
}
\value{An SS TSmodel.}
\details{
Randomly generate a state space model. If stable is true then the
largest root will have magnitude less than 1.0.
}
\examples{
z <-  generateSSmodel(2,3,1)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{horizonForecasts}
\alias{horizonForecasts}
\alias{horizonForecasts.TSdata}
\alias{horizonForecasts.TSmodel}
\alias{horizonForecasts.TSestModel}
\alias{horizonForecasts.forecastCov}
\alias{is.horizonForecasts}

\title{Calculate forecasts at specified horizons}
\description{Calculate forecasts at specified horizons.}
\usage{
    is.horizonForecasts(obj)
    horizonForecasts(obj, ...)
    \method{horizonForecasts}{TSmodel}(obj, data, horizons=1:4,
	 discard.before=minimumStartupLag(obj), compiled=.DSECOMPILED, ...)
    \method{horizonForecasts}{TSestModel}(obj, data=NULL, ...)
    \method{horizonForecasts}{TSdata}(obj, model, ...)
    \method{horizonForecasts}{forecastCov}(obj,horizons=NULL, 
         discard.before=NULL, ...)
}
\arguments{
    \item{obj}{an object of class TSmodel, TSdata, or TSestModel.}
    \item{model}{an object of class TSmodel.}
    \item{data}{an object of class TSdata} 
    \item{horizons}{
    a vector of integers indicating the horizon at which forecasts should be 
    produced.}
    \item{discard.before}{period before which forecasts are not calculated.}
    \item{compiled}{if TRUE compiled code is called.}
    \item{...}{arguments passed to other methods.}
}
\value{
The result is a list of class horizonForecasts with elements model (a 
TSmodel), data, horizons, discard.before, and horizonForecasts.
horizonForecasts is an array with three dimension: 
\code{c(length(horizons), dim(model$data))}.
Projections are not calculated before discard.before or after
the end of outputData(data).
Each horizon is aligned so that horizonForecasts[h,t,] contains the forecast
for the data point outputData(data)[t,] (from horizon[h] periods prior).
}
\details{
Calculate multiple 'horizon'-step ahead forecasts 
ie. calculate forecasts but return only those indicated by horizons.
   Thus, for example, the result of
 horizonForecasts(model, data horizons=c(1,5))    
 would be the one-step ahead and five step ahead forecasts.
}
\seealso{
\code{\link{featherForecasts}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <-  horizonForecasts(model, eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{horizonForecastsCompiled}
\alias{horizonForecastsCompiled}
\alias{horizonForecastsCompiled.ARMA}
\alias{horizonForecastsCompiled.SS}

\title{Calculate forecasts at specified horizons}
\description{Calculate forecasts at specified horizons.}
\usage{
   horizonForecastsCompiled(obj, data, horizons=1:4,
	  discard.before=minimumStartupLag(obj))
   \method{horizonForecastsCompiled}{SS}(obj, data, horizons=1:4,
	 discard.before=minimumStartupLag(obj))
   \method{horizonForecastsCompiled}{ARMA}(obj, data, horizons=1:4,
	  discard.before=minimumStartupLag(obj))
}
\arguments{
    \item{obj}{see horizonForecasts.}
    \item{data}{see horizonForecasts.} 
    \item{horizons}{see horizonForecasts.}
    \item{discard.before}{see horizonForecasts.}
    \item{...}{see horizonForecasts.}
}
\value{See horizonForecasts.}
\details{Internal function not to be called by users. See horizonForecasts.}
\seealso{
\code{\link{horizonForecasts}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <-  horizonForecasts(model, eg1.DSE.data.diff)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{is.forecastCovEstimatorsWRTdata.subsets}
\alias{is.forecastCovEstimatorsWRTdata.subsets}
\title{Check Inheritance}
\description{Check inheritance.}
\usage{
    is.forecastCovEstimatorsWRTdata.subsets(obj)
}
\arguments{
    \item{obj}{Any object.}
}
\value{logical}
\details{
    This tests if an object isnherits 
    from \code{forecastCovEstimatorsWRTdata.subsets}. This type of object
    code be generated in different ways but the only current example
    is \code{stripMine}.
}
\seealso{
    \code{\link{stripMine}}
}
\
%\keyword{DSE}
\keyword{ts}


\eof
\name{minForecastCov}
\alias{minForecastCov}
\title{Minimum Forecast Cov Models}
\description{Extract the minimum forecastCov at each horizon}
\usage{
    minForecastCov(obj, series=1, verbose=TRUE)
}
\arguments{
    \item{obj}{An object as returned by stripMine.}
    \item{series}{
    An indicator of the series which are to be used as the bases for selection.}
    \item{verbose}{If true additional information is printed.}
}
\value{
The returned object is a vector indicating the element of forecastCov which
was the min at each horizon.
}
\details{
Select the min covariance (for series only!) at each horizon and
print.  The returned object is a vector indicating the element of
forecastCov which was the min at each horizon. It is suitable as an
argument to plot eg:  tfplot(obj, select.cov=minForecastCov(obj)) The
results of this plot are similar to the default results of
tfplot(selectForecastCov(obj)). Covariance information and information
about the horizon where the model is optimal are given.
}
\seealso{
\code{\link{selectForecastCov}},
\code{\link{excludeForecastCov}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- stripMine(eg1.DSE.data.diff, essential.data=c(1,2),
                   estimation.methods=list(estVARXls=list(max.lag=3)))
z <-  minForecastCov(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{mineStepwise}
\alias{mineStepwise}
\title{Mine Stepwise}
\description{mineStepwise}
\usage{
    mineStepwise(data, essential.data=1,
      method="efroymson", f.crit=2, intercept=TRUE,
      subtract.means=FALSE,  standardize=FALSE, 
      lags.in=6, lags.out=6, trend=FALSE, plot.=TRUE)
}
\arguments{
    \item{data}{TSdata}
    \item{essential.data}{An integer vector indication important data.}
    \item{method}{method to pass to stepwise.}
    \item{f.crit}{See details.}
    \item{intercept}{See details.}
    \item{subtract.means}{See details.}
    \item{standardize}{See details.}
    \item{lags.in}{See details.}
    \item{lags.out}{See details.}
    \item{trend}{See details.}
    \item{plot.}{See details.}
}
\value{x}
\details{This documentation is out of date.
    Data should be of class TSdata. essential.data must have length 1.
    standardize and subtract means ...
    The result is a list with the results of stepwise,..., and
    several vectors indicating information about the columns of the matrix
    passed to stepwise: 
    io.indicator indicating an input (FALSE) or output (TRUE) variable
    v.indicator  indicating which series
    lag.indicator indicating the lag
    s.input.indicator and s.output.indicator are logic matrices
    \code{length(stepwise$rss)} by m and p respectively indicating if
    a series is included for each element of rss.
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{minimumStartupLag}
\alias{minimumStartupLag}
\alias{minimumStartupLag.SS}
\alias{minimumStartupLag.ARMA}
\alias{minimumStartupLag.TSestModel}
\alias{startShift}

\title{Starting Periods Required}
\description{Number of Starting Periods Required for a Model}
\usage{
    minimumStartupLag(model)
    \method{minimumStartupLag}{SS}(model)
    \method{minimumStartupLag}{ARMA}(model)
    \method{minimumStartupLag}{TSestModel}(model)
    startShift(model,data, y0=NULL)
}
\arguments{
    \item{model}{A TSmodel or object containing a TSmodel.}
    \item{data}{A TSdata object.}
    \item{y0}{initial condition ...}
}
\value{An integer.}
\details{
    For many time series models several starting data points are 
    required before output from the model can be calculated (or makes
    sense). This generic function extracts or calculates that 
    number of periods.
}

\seealso{
    \code{\link[dse1]{TSmodel}}
}

\note{There is some redundancy between this and startShift which 
      should be cleaned up.}
%\keyword{DSE}
\keyword{ts}


\eof
\name{nseriesfeatherForecasts}
\alias{nseries.featherForecasts}

\title{Number of Series}
\description{Return the number of series.}
\usage{
    \method{nseries}{featherForecasts}(x)
    }

\arguments{
    \item{x}{A featherForecasts object.}
}
\value{An integer.}
\details{
    See the generic method.
}

\keyword{programming}
\keyword{utilities}
\keyword{ts}


\eof
\name{outOfSample.forecastCovEstimatorsWRTdata}
\alias{outOfSample.forecastCovEstimatorsWRTdata}
\title{Calculate Out-of-Sample Forecasts}
\description{Calculate out-of-sample forecasts.}
\usage{
    outOfSample.forecastCovEstimatorsWRTdata(data, zero=FALSE, trend=FALSE,
                       estimation.sample=.5, horizons=1:12,quiet=FALSE,
                       estimation.methods=NULL, compiled=.DSECOMPILED)
}
\arguments{
    \item{data}{an object of class TSdata.}
    \item{estimation.methods}{a list as used by estimateModels.}
    \item{zero}{if TRUE then forecastCov is also calculated for a forecast of zero.}
    \item{trend}{
    if TRUE then forecastCov is also calculated for a forecast of a linear trend.}
    \item{estimation.sample}{
    indicates the portion of the data to use for estimation.
If estimation.sample is an integer then it is used to indicate the number
of points in the sample to use for estimation. If it is a fracton it is
used to indicate the portion of points to use for estimation. The remainder
of the sample is used for evaluating forecasts.}
    \item{horizons}{horizons for which forecast covariance should be calculated.}
    \item{quiet}{if TRUE then estimation information is not printed.}
    \item{compiled}{a logical indicating if compiled code should be
      used. (Usually true except for debugging.)}
}
\value{An object as returned by forecastCovEstimatorsWRTdata.}
\details{
The data is slpit into a sub-sample used for estimation and another
sub-sample used for calculating the forecast covariance.
}
\seealso{
\code{\link{forecastCovEstimatorsWRTdata}},
\code{\link{forecastCovEstimatorsWRTtrue}},
\code{\link{estimateModels}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <-  outOfSample.forecastCovEstimatorsWRTdata(eg1.DSE.data.diff,
           estimation.methods=list(estVARXls=list(max.lag=4)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{permute}
\alias{permute}
\title{Permute}
\description{
    Return matrix with rows indicating all possible selections of
    elements from seq(M). 0 in the result indicates omit. 
    M is usually a positive integer. M=0 gives NULL.
    Neg. M give -permute(abs(M)).
}
\usage{
    permute(M)
}
\arguments{
    \item{M}{An integer.}
}
\value{A matrix.}
\examples{
    permute(4)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{phasePlots}
\alias{phasePlots}
\title{Calculate Phase Plots}
\description{Calculate phase plots}
\usage{
    phasePlots(data, max.lag=1,diff=FALSE)
}
\arguments{
    \item{data}{A matrix, time series matrix, or an object of class TSdata.}
    \item{max.lag}{The maximum number of shifts to plot}
    \item{diff}{If TRUE the data is plotted against the difference with lagged values.}
}
\value{None}
\sideffects{
A plot of (the phase space) the data against (differenced) lagged values is produced.
}
\details{
Non-linearities may show up as a non-linear surface, but this is
a projection so, for example, a spherical space would not show up.
Some sort of cross-section window would show this but require even more plots.
A good statistical test would be better!
}
\examples{
if(is.R()) data("egJofF.1dec93.data", package="dse1")
phasePlots(egJofF.1dec93.data)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{plot.mineStepwise}
\alias{plot.mineStepwise}
\title{Plot Mine Stepwise Object}
\description{plot.mineStepwise}
\usage{
    \method{plot}{mineStepwise}(x, ...)
}
\arguments{
    \item{x}{Object returned by mineStepwise.}
    \item{...}{(further arguments, currently disregarded).}
}
\value{None}
\sideffects{A plot}
\seealso{
    \code{\link{mineStepwise}}
}
%\keyword{DSE}
\keyword{ts}

\eof
\name{print.estimatedModels}
\alias{print.estimatedModels}
\alias{print.EstEval}
\alias{print.forecastCov}
\alias{print.forecastCovEstimatorsWRTdata.subsets}
\alias{print.forecastCovEstimatorsWRTtrue}
\alias{print.MonteCarloSimulations}

\title{Print Specific Methods}
\description{See the generic function description.}
\usage{
    \method{print}{estimatedModels}(x, digits=options()$digits, ...)
    \method{print}{EstEval}(x, digits=options()$digits, ...)
    \method{print}{forecastCov}(x, digits=options()$digits, ...)
    \method{print}{forecastCovEstimatorsWRTdata.subsets}(x, digits=options()$digits, ...)
    \method{print}{forecastCovEstimatorsWRTtrue}(x, digits=options()$digits, ...)
    \method{print}{MonteCarloSimulations}(x, digits=options()$digits, ...)
}
\arguments{
    \item{x}{an object to be printed.}
    \item{digits}{a non-null value is used to indicate the number of significant
       digits. If \code{digits} is \code{NULL} then the value of digits 
       specified by \code{options} is used.}
    \item{...}{(further arguments, currently disregarded).}
}
\seealso{
    \code{\link{print}}
    \code{\link{summary}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{roots.estimatedModels}
\alias{roots.estimatedModels}
\alias{roots.forecastCovEstimatorsWRTtrue}
\alias{roots.coefEstEval}
\alias{roots.rootsEstEval}
\alias{roots.TSestModelEstEval}
\alias{roots.TSmodelEstEval}

\title{Roots Specific Methods}
\description{See the generic function description.}
\usage{
    \method{roots}{estimatedModels}(obj, digits=options()$digits, mod =FALSE, ...)
    \method{roots}{forecastCovEstimatorsWRTtrue}(obj, digits=options()$digits,
       mod=FALSE, ...)
    \method{roots}{coefEstEval}(obj, criterion.args=NULL, ...)
    \method{roots}{rootsEstEval}(obj, ...)
    \method{roots}{TSestModelEstEval}(obj, criterion.args=NULL, ...)
    \method{roots}{TSmodelEstEval}(obj, criterion.args=list(randomize = TRUE), ...)
}
\arguments{
    \item{obj}{an object from which roots are to be extracted or calculated and
       printed.}
    \item{digits}{an integer indicating the number of significant digits to be
       printed (passed to the print method).}
    \item{mod}{if TRUE the modulus of the roots is calculated. Otherwise, a
       complex value may result.}
    \item{criterion.args}{arguments to be passed to this method when it is
         called by \code{EstEval}.}
    \item{...}{arguments to be passed to other methods.}
}
\details{
    The methods ***.ee are intended mainly to be called from EstEval as
    criterion for evaluating an estimation method.
}
\seealso{\code{\link[dse1]{roots}}
         \code{\link[dse1]{stability}}
         \code{\link{EstEval}}
	}

%\keyword{DSE}
\keyword{ts}


\eof
\name{selectForecastCov}
\alias{selectForecastCov}
\title{Select Forecast Covariances Meeting Criteria}
\description{Select forecast covariances meeting given criteria.}
\usage{
    selectForecastCov(obj, series=1, 
    select.cov.best=1,
    select.cov.bound=NULL,
    ranked.on.cov.bound=NULL,
    verbose=TRUE)
}
\arguments{
    \item{obj}{an object as returned by stripMine.}
    \item{series}{
    an indication of series to which the tests should be applied.}
    \item{select.cov.best}{the number of 'best' forecasts to select.}
    \item{select.cov.bound}{a bound to use as criteria for selection.}
    \item{ranked.on.cov.bound}{see details.}
    \item{verbose}{if verbose=TRUE then summary results are printed.}
}
\value{
The returned result is a forecastCov object like obj, but filtered
to remove models which do not meet criteria.
}
\details{
Select models with forecast covariance for \code{series} meeting criteria.
The default \code{select.cov.best=1} selects the best model at each horizon.
\code{select.cov.best=3} would select the best 3 models at each horizon.
If \code{select.cov.bound} is not \code{NULL} then  \code{select.cov.best}
 is ignored and
any model which is better than the bound at all horizons is selected.
\code{select.cov.bound} can be a vector of the same length 
as \code{series},
in which case corresponding elements are applied to the different series.
  Any model which is better than the bound at all horizons is selected.
 \code{ranked.on.cov.bound} is is used if it is not NULL and 
 \code{select.cov.bound} is
  \code{NULL}. In this case \code{select.cov.best} is ignored.
  \code{ranked.on.cov.bound} should be a positive integer. The forecast
  covariances are ranked by there maximum over the horizon and the
  lowest number up to \code{ranked.on.cov.bound} are selected. This amounts
  to adjusting the covariance bound to allow for the given number of
  models to be selected. If \code{series} is a vector the results are 
  the best up to the given number on any series!
 \code{select.cov.bound} can be a vector of the same length as 
 \code{series},
  in which case corresponding elements are applied to the different series.
 If \code{verbose=TRUE} then summary results are printed.
 The returned result is a \code{forecastCov} object like obj, but filtered
  to remove models which do not meet criteria.
}
\seealso{
\code{\link{minForecastCov}},
\code{\link{excludeForecastCov}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- stripMine(eg1.DSE.data.diff, essential.data=c(1,2),
                   estimation.methods=list(estVARXls=list(max.lag=3)))
z <-  selectForecastCov(z)
tfplot(selectForecastCov(z, select.cov.bound=20000))
tfplot(selectForecastCov(z, select.cov.best=1))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{seriesNamesInput.forecast}
\alias{seriesNamesInput.forecast}
\alias{seriesNamesInput.featherForecasts}
\alias{seriesNamesInput.MonteCarloSimulations}

\alias{seriesNamesOutput.forecast}
\alias{seriesNamesOutput.featherForecasts}
\alias{seriesNamesOutput.MonteCarloSimulations}

\title{TS Input and Output Specific Methods}
\description{See the generic function description.}
\usage{
    \method{seriesNamesInput}{forecast}(x)
    \method{seriesNamesInput}{featherForecasts}(x)
    \method{seriesNamesInput}{MonteCarloSimulations}(x)

    \method{seriesNamesOutput}{forecast}(x)
    \method{seriesNamesOutput}{featherForecasts}(x)
    \method{seriesNamesOutput}{MonteCarloSimulations}(x)
}
\arguments{
    \item{x}{an object from which to extract the names of the input or output
      series.}
}

%\keyword{DSE}
\keyword{ts}



\eof
\name{shockDecomposition}
\alias{shockDecomposition}
\title{Shock Decomposition}
\description{Graphs of the effect of shocks are plotted.}
\usage{
    shockDecomposition(model, horizon=30, shock=rep(1,horizon))
}
\arguments{
    \item{model}{An object of class TSmodel or TSestModel.}
    \item{horizon}{The number of periods for which to calculate the effect 
        of shocks.}
    \item{shock}{data to be used model output. See details.}
}
\value{None}
\sideffects{
Graphs of the effect of shocks are plotted. 
}
\details{All output data is set to zero and then each output in turn is switched to 
a value of shock (default 1.0) for all periods. 
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
shockDecomposition(model)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{stripMine}
\alias{stripMine}
\title{Select a Data Subset and Model}
\description{Select a data subset and model.}
\usage{
    stripMine(all.data, essential.data=1, 
                       estimation.sample=.5, 
                       discard.before=1, horizons=1:12,quiet=FALSE,
                       estimation.methods=NULL,
                       step.size=NULL)
}
\arguments{
    \item{all.data}{An object of class TSdata.}
    \item{essential.data}{A vector indicating the important series.}
    \item{estimation.sample}{The portion of the data to use for estimation.}
    \item{discard.before}{
    Period before which data should be disacrded when calculating the 
    forecast covariances.}
    \item{horizons}{Forecast horizons which should be considered.}
    \item{quiet}{If T then estimation information is not printed. quiet=TRUE may also have to be
    set in the arguments to estimation methods.}
    \item{estimation.methods}{
    A list indicating the model estimation method to use. The list should contain
    one element. The name of the element indicates the estimation method to use
    and the value of the element is a list of arguments to pass to the 
    estimation method.}
    \item{step.size}{
    An integer indicting how many dta subset/model steps should be attempted.
    This may be necessary to accommodate memory constraints on the system. 
    (see below.)}
}
\value{
The returned result contains a list (\code{forecastCov}) of the forecast
covariance on the essential data for the various models and data
subsets. It can be plotted with the generic function tfplot. Additional
information in the result comes from the function arguments.
}
\details{
Calculate the predictions cov for essential.data of models estimated 
 with estimation methods indicated by estimation.methods. 
 estimation.methods is a list with syntax similar to programs
  for comparing estimation methods (eg. estimateModels), BUT ONLY 
  THE FIRST element (estimation method) is considered.
Essential.data indicates the subset of output variables to included in all
 models. It should be a vector of the indices. All possible combinations of
 input series and other output series data are considered. If omitted, essential.data is taken as the 
 first output series.
 Only forecast covariances for essential data are returned.
 discard.before is an integer indicating 1+the number of points in the
 beginning of predictions to discard for calculating prediction covariances.
 estimation.sample indicates the portion of the data to use for estimation.
  If estimation.sample is an integer then it is used to indicate the number
  of points in the sample to use for estimation. If it is a fracton it is
  used to indicate the portion of points to use for estimation. The remainder
  of the sample is used for evaluating predictions.
  If step.size is NULL then all possible data permutations are attempted.
   Because S has a hard-coded limit in the number of synchronize calls this is
   not always possible (For loops call synchronize.) An error message:
     Error in synchronize(1): No room in database table
   If step.size is not NULL it should be a positive integer. In this case 
   variable permutions are divided up into
   steps of the given size. The result returned by the function can be used
   to continue from the last step:
    intermediate.result <- stripMine(data, ...)
      intermediate.result <- stripMine(intermediate.result)
      intermediate.result <- stripMine(intermediate.result)
      result <- stripMine(intermediate.result)
  This can be done either interactively or in a batch process, but cannot be
  done in a function because the database table is not cleared until the top
  level expression is complete.
  The class of an intermediate result is stripMine.intermediate.result and
  the class of the final result is
         c('forecastCovEstimatorsWRTdata.subsets', 'forecastCov')
  If the final result is used in a call to stripMine then it is just 
  returned, so extra calls do not cause errors and are very quick.
  This is useful when you are too lazy to calculate the exact number of steps.
}
\seealso{
\code{\link[dse1]{estBlackBox4}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
z <- stripMine(eg1.DSE.data.diff, 
    estimation.methods=list(bft=list(max.lag=2, verbose=FALSE)))
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{summary.estimatedModels}
\alias{summary.estimatedModels}
\alias{summary.TSestModelEstEval}
\alias{summary.TSmodelEstEval}
\alias{summary.EstEval}
\alias{summary.forecastCov}
\alias{summary.forecastCovEstimatorsWRTdata.subsets}
\alias{summary.forecastCovEstimatorsWRTtrue}
\alias{summary.MonteCarloSimulations}
\alias{summary.coefEstEval}
\alias{summary.rootsEstEval}

\alias{print.summary.estimatedModels}
\alias{print.summary.TSestModelEstEval}
\alias{print.summary.TSmodelEstEval}
\alias{print.summary.EstEval}
\alias{print.summary.forecastCov}
\alias{print.summary.forecastCovEstimatorsWRTdata.subsets}
\alias{print.summary.forecastCovEstimatorsWRTtrue}
\alias{print.summary.MonteCarloSimulations}
\alias{print.summary.coefEstEval}
\alias{print.summary.rootsEstEval}

\title{Summary Specific Methods}
\description{See the generic function description.}
\usage{
    \method{summary}{estimatedModels}(object, ...)
    \method{summary}{TSestModelEstEval}(object, ...)
    \method{summary}{TSmodelEstEval}(object, ...)
    \method{summary}{EstEval}(object, ...)
    \method{summary}{forecastCov}(object, horizons=object$horizons,
        series=seq(nseriesOutput(object$data)), ...)
    \method{summary}{forecastCovEstimatorsWRTdata.subsets}(object, ...)
    \method{summary}{forecastCovEstimatorsWRTtrue}(object,
        digits=options()$digits, ...)
    \method{summary}{MonteCarloSimulations}(object, series=NULL, periods=1:3, ...)
    \method{summary}{coefEstEval}(object, verbose=TRUE, ...)
    \method{summary}{rootsEstEval}(object, verbose=TRUE, ...)
    
    \method{print}{summary.estimatedModels}(x, digits=options()$digits, ...)
    \method{print}{summary.TSestModelEstEval}(x, digits=options()$digits, ...)
    \method{print}{summary.TSmodelEstEval}(x, digits=options()$digits, ...)
    \method{print}{summary.EstEval}(x, digits=options()$digits, ...)
    \method{print}{summary.forecastCov}(x, digits=options()$digits, ...)
    \method{print}{summary.forecastCovEstimatorsWRTdata.subsets}(x,
       digits=options()$digits, ...)
    \method{print}{summary.forecastCovEstimatorsWRTtrue}(x,
       digits=options()$digits, ...)
    \method{print}{summary.MonteCarloSimulations}(x, digits=options()$digits, ...)
    \method{print}{summary.coefEstEval}(x, digits=options()$digits, ...)
    \method{print}{summary.rootsEstEval}(x, digits=options()$digits, ...)
}
\arguments{
    \item{object}{an object for which a summary is to be printed.}
    \item{x}{an object for which a summary is to be printed.}
    \item{digits}{a non-null value is used to indicate the number of significant
       digits. If \code{digits} is \code{NULL} then the value of digits 
       specified by \code{options} is used.}
    \item{horizons}{optional integer vector indicating horizons at which the 
       summary should be calculated.}
    \item{series}{
       The series which should be plotted. The default NULL gives all series.}
    \item{periods}{optional integer vector indicating periods at which the 
       summary should be calculated.}
    \item{verbose}{logical indicating if a longer summary should be produced.}
    \item{...}{arguments passed to other methods.}
}

\seealso{
    \code{\link{summary}}
    \code{\link{print}}
}
%\keyword{DSE}
\keyword{ts}



\eof
\name{testEqual.estimatedModels}
\alias{testEqual.estimatedModels}
\alias{testEqual.EstEval}
\alias{testEqual.forecast}
\alias{testEqual.forecastCov}
\alias{testEqual.horizonForecasts}
\alias{testEqual.MonteCarloSimulations}

\title{Specific Methods for Testing Equality}
\description{See the generic function description.}
\usage{
    \method{testEqual}{estimatedModels}(obj1, obj2, fuzz = 0)
    \method{testEqual}{EstEval}(obj1, obj2, fuzz=0)
    \method{testEqual}{forecast}(obj1, obj2, fuzz=1e-14)
    \method{testEqual}{forecastCov}(obj1, obj2, fuzz=1e-14)
    \method{testEqual}{horizonForecasts}(obj1, obj2, fuzz=1e-14)
    \method{testEqual}{MonteCarloSimulations}(obj1, obj2, fuzz=1e-16)
}
\arguments{
    \item{obj1}{an object which is to be compared with the second object.}
    \item{obj2}{an object which is to be compared with the first object.}
    \item{fuzz}{tolerance for numerical comparisons. Values within fuzz will be
       considered equal.}
}

\seealso{ \code{\link[tframe]{testEqual}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.MonteCarloSimulations}
\alias{tfplot.MonteCarloSimulations}
\title{Generate plots of Monte Carlo simulations}
\description{Generate plots of Monte Carlo simulations.}
\usage{
    \method{tfplot}{MonteCarloSimulations}(x, 
    tf=tframe(x$simulations), start=tfstart(tf), end=tfend(tf),
    series=seq((dim(x$simulations)[2])), 
    select.simulations=seq(dim(x$simulations)[3]),
    graphs.per.page=5, mar=par()$mar)
}
\arguments{
    \item{x}{The result of MonteCarloSimulations.}
    \item{tf}{The time frame for plots. see \code{tfplot}.}
    \item{start}{The starting period for plots, taken from tf by default.}
    \item{end}{The ending period for plots, taken from tf by default.}
    \item{series}{
       The series which should be plotted. The default NULL gives all series.}
    \item{select.simulations}{
    Vector of integers indicating the simulations which should be plotted. 
    The default plots all simulations.}
    \item{graphs.per.page}{The number of graphs to put on a page.}
    \item{mar}{Plot margins (see \code{par}).}
}
\value{None}
\details{
This function produces plots of the simulated series.
Output graphics can be paused between pages by setting par(ask=TRUE).
}
\seealso{
\code{\link{distribution.MonteCarloSimulations}}
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model <- estVARXls(eg1.DSE.data.diff)
z <-  MonteCarloSimulations(model)
tfplot(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.TSdata.ee}
\alias{tfplot.TSmodelEstEval}
\alias{tfplot.TSestModelEstEval}
\alias{tfplot.featherForecasts}
\alias{tfplot.forecast}
\alias{tfplot.EstEval}
\alias{tfplot.horizonForecasts}
\alias{tfplot.multiModelHorizonForecasts}

\title{Specific Methods for tfplot}
\description{See the generic function description.}
\usage{
    \method{tfplot}{TSmodelEstEval}(x, graph.args=NULL,
                       criterion ="coef", criterion.args=NULL)
    \method{tfplot}{TSestModelEstEval}(x, graph.args=NULL,
                       criterion ="coef", criterion.args=NULL)
    \method{tfplot}{featherForecasts}(x, tf=NULL, start=tfstart(tf), end=tfend(tf), 
        series=seq(nseries(x)), 
        Title="Predictions (dotted) and actual data (solid)", 
        ylab=seriesNamesOutput(x),
        graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{forecast}(x, tf=NULL, start=tfstart(tf), end=tfend(tf),
        series = seq(length=nseriesOutput(x$data)),
	Title="Predictions (dotted) and actual data (solid)",
        ylab = seriesNamesOutput(x$data), 
	graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{EstEval}(x, tf=NULL, start=tfstart(tf), end=tfend(tf),
        truth= if(is.TSdata(x$truth)) outputData(x$truth) else x$truth,
        series = seq(length=nseries(truth)),
	Title="Estimated (and true) results",
        ylab = seriesNames(truth), remove.mean = FALSE,
	graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{horizonForecasts}(x, tf=NULL, start=tfstart(tf), end=tfend(tf),
         series=seq(length=nseriesOutput(x$data)),
	 Title="Predictions (dotted) and actual data (solid)", 
	 ylab=seriesNamesOutput(x$data), 
	 graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{multiModelHorizonForecasts}(x, 
         tf=NULL, start=tfstart(tf), end=tfend(tf), series=NULL)
}
\arguments{
    \item{x}{an object for which a tfplot is to be produced.}
    \item{tf}{see \code{tfplot}.}
    \item{start}{see \code{tfplot}.}
    \item{end}{see \code{tfplot}.}
    \item{truth}{true value which will be plotted along with estimates.}
    \item{Title}{string of characters to use for title.}
    \item{remove.mean}{logical indicating if means should be removed
          before plotting results.}
    \item{ylab}{vector of strings for y axis labelling.}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{reset.screen}{logical indicating if the plot window should be cleared
       before starting.}
    \item{series}{integer or string indicating
      the series which should be plotted.}
    \item{mar}{plot margins. See \code{par}.}
    \item{graph.args}{list of graphics arguments eventually passed 
       to \code{plot}. See \code{par}.}
    \item{criterion}{criterion which should be used to extract something from
         the object which will then be plotted. See \code{EstEval}.}
    \item{criterion.args}{arguments to be passed to code{criterion}.}
}

\seealso{
    \code{\link[tframe]{tfplot}}
    \code{\link{EstEval}} }

%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.coefEstEval}
\alias{tfplot.coefEstEval}

\title{Specific tfplot methods for coefEstEval (EstEval) objects}
\description{See the generic function description.}
\usage{
    \method{tfplot}{coefEstEval}(x, cumulate=TRUE, norm=FALSE, bounds=TRUE,
        invert=FALSE, Sort=FALSE, graphs.per.page = 5)
}
\arguments{
    \item{x}{an object for which a tfplot is to be produced.}
    \item{cumulate}{logical indicating if the cumulative average of roots should be
       plotted}
    \item{invert}{logical indicating if the inverse of roots should be
       plotted}
    \item{Sort}{logical indicating if the roots should be sorted.}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{norm}{logical indicating if the euclidean norm of roots should be
       plotted (square root of the sum of squared roots).}
    \item{bounds}{logical indicating if estimated one standard error bounds
       should be plotted around the lines for the true roots.}
}

\details{
 If cumulate is true the cumulative average is plotted.
 If norm is true the norm is used, each parameter is plotted.
 If invert is true the reciprical is used (before cumulating).
 If Sort is true then sort is applied (before ave). This is not usually
   recommended but of interest
   with estimation methods like black.box which may not return parameters
   of the same length or in the same order.
 Plotting the true lines only makes sense if truth is the same length as 
  result (and sometimes not even then). 
}
\seealso{
    \code{\link[tframe]{tfplot}}
    \code{\link{EstEval}}
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.forecastCov}
\alias{tfplot.forecastCov}
\alias{tfplot.forecastCovEstimatorsWRTdata}

\title{Plots of Forecast Variance}
\description{Generate plots of forecast variance calculated by forecastCov.}
\usage{
    \method{tfplot}{forecastCov}(x, ...,
       series = 1:dim(x$forecastCov[[1]])[2], 
       select.cov = 1:length(x$forecastCov), select.true =TRUE, 
       select.zero =TRUE, select.trend =TRUE, y.limit = NULL, line.labels =FALSE, 
       lty = NULL, Legend = NULL, Title = NULL,
       graphs.per.page = 5, mar=par()$mar, reset.screen=TRUE)
    \method{tfplot}{forecastCovEstimatorsWRTdata}(x, 
        series=1:dim(x$forecastCov[[1]])[2], 
        select.cov=1:length(x$forecastCov),
        select.zero=TRUE, select.trend=TRUE,
        graphs.per.page = 5, mar=par()$mar, reset.screen=TRUE, lty=NULL)
}
\arguments{
    \item{x}{The result of forecastCov.}
    \item{series}{integer or string indicating
      the series which should be plotted.}
    \item{select.cov}{logical indicating that
      for the case of multiple models select the covariance to be plotted.}
    \item{select.true}{logical indicating that results from the forecast of the 
      true model (if available) should be plotted.}
    \item{select.zero}{logical indicating that results from a forecast of 
      zero should be plotted.}
    \item{select.trend}{logical indicating that results from a forecast of 
      trend should be plotted.}
    \item{graphs.per.page}{The number of graphs to put on a page.}
    \item{mar}{plot margins (see \code{par}).}
    \item{reset.screen}{logical indicating if the plot window should be cleared
       before starting.}
    \item{lty}{see details.}
    \item{Legend}{optional legend passed to \code{legend}.}
    \item{Title}{optional legend passed to \code{title}.}
    \item{y.limit}{optional limit on the y scale. Covariance values larger
       than y.limit will not be shown.}
    \item{line.labels}{logical indicating line labels should be printed.}
    \item{...}{additional objects to be plotted.}
}
\value{None}
\details{
    This function produces plots of the variance at different horizons.
Output graphics can be paused between pages by setting par(ask=TRUE).
If lty is NULL (default) it is set to
     seq(length(select.cov) +select.true+select.zero+select.trend),
 and corrected if these are TRUE but not in the object.

}
\seealso{
    \code{\link{plot}}
}
\examples{
    if(is.R()) data("eg1.DSE.data.diff", package="dse1")
    model <- estVARXls(eg1.DSE.data.diff)
    z <- forecastCov(model, data=eg1.DSE.data.diff)
    tfplot(z)
}
%\keyword{DSE}
\keyword{ts}


\eof
\name{tfplot.rootsEstEval}
\alias{tfplot.rootsEstEval}
\alias{plot.rootsEstEval}

\title{Specific tfplot methods for rootsEstEval (EstEval) objects}
\description{See the generic function description.}
\usage{
    \method{tfplot}{rootsEstEval}(x, ...)
    \method{plot}{rootsEstEval}(x, complex.plane=TRUE, cumulate=TRUE, norm=FALSE,
       bounds=TRUE, transform=NULL, invert=FALSE, Sort=TRUE, ...)
}
\arguments{
    \item{x}{an object for which a tfplot is to be produced.}
    \item{complex.plane}{logical indicating if the plot should be on the complex
       plane.}
    \item{cumulate}{logical indicating if the cumulative average of roots should be
    plotted}
    \item{invert}{logical indicating if the inverse of roots should be
    plotted}
    \item{Sort}{logical indicating if the roots should be sorted.}
    \item{...}{arguments passed to other methods.}
    \item{norm}{logical indicating if the euclidean norm of roots should be
       plotted (square root of the sum of squared roots).}
    \item{bounds}{logical indicating if estimated one standard error bounds
       should be plotted around the lines for the true roots.}
    \item{transform}{an optional string indicating the name of a function which
       should be applied to the roots before plotting.}
}

\details{
If complex.plane is TRUE then all results are plotted on a complex plane and 
   the arguements cumulate and Sort do not apply. If complex.plane is FALSE 
   then a sequential plot of the real and imaginary parts is produced.
 If cumulate is true the cumulative average is plotted.
 If mod is true the modulus is used, otherwise real and imaginary are separated.
 if invert is true the reciprical is used (before cumulating).
 if Sort is true then sort is applied (before cumulate but after mod) by the Re part of the root.
   Some grouping is usually necessary since roots are not in an obvious order
   but sorting by the real part of the roots could be improved upon.
}
\seealso{
    \code{\link[tframe]{tfplot}}
    \code{\link{EstEval}}
}

%\keyword{DSE}
\keyword{ts}


\eof
\name{totalForecastCov}
\alias{totalForecastCov}
\title{Sum covariance of forecasts across all series}
\description{Sum covariance of forecasts across all series.}
\usage{
    totalForecastCov(obj, select=NULL)
}
\arguments{
    \item{obj}{An object  as returned by forecastCov.}
    \item{select}{
    Series to be select for summation. With the default all series are selected.}
}
\value{
An object similar to that returned by forecastCov, with the covariance
summed over all selected series.
}
\examples{
if(is.R()) data("eg1.DSE.data.diff", package="dse1")
model1 <- estVARXar(eg1.DSE.data.diff)
model2 <- estVARXls(eg1.DSE.data.diff)
z <-  totalForecastCov(forecastCov(model1, model2,
                         data=trimNA(eg1.DSE.data.diff)))
}
%\keyword{DSE}
\keyword{ts}


\eof
