\name{00Intro.tframe}
\alias{00Intro.tframe}
\title{Tframe}

\center{Generic Methods for Handling Time}

\description{Programs for implementing an object oriented approach to
    handling different time representations.}
\section{Introduction}{
The \emph{tframe} library facilitates an object oriented approach
to handling different time representations in S/R. It provides generic methods
by which code can be developed without too much dependence on the
representation of time (i.e. specific time series objects). This can make
most code very robust with respect to other (and future)
improved/different representations of time. However, something like putting
the time axis label on a plot will require a method which is specific
to the time representation. The methods also provide a
simple way to fix some inconsistency which have occured with the
mix of possible time representations.

The classes and methods associated with the implementation of rts, cts
and its in Splus and ts in R accomplish some of these objectives. 
For many time series programs the availability of a \emph{window} method 
provided by those classes is the main method which is needed. However, after
calculations are done the time attributes of the resulting objects are
often lost and the programmer must re-assign time attributes to the resulting
object if those are to be retained. This has typically been done with something
like tsp(), but these functions rely on a particular time representation and
would not always make sense for another representation. In order to address this,
the tframe methods attempt to further separate the time representation from 
the data to allow statements like

\code{tframe(x) <- tframe(y)}

to make the time frame of x the same as that of y, without the need to
worry about what time representation is used in y (e.g. ts, rts, cts,
its, ...). In this assignment x and y need not be too similar (one
might be a univariate series while the other is a matrix or an array or
list of spatial or panel time series data), as long as they are similar
in the time dimension. For the case where \code{tsp(x) <- tsp(y)} would 
make sense,
that is effectively what the above tframe assignment will do, and for 
existing code, most of the conversion to these more robust methods is
accomplished simply by changing "tsp" to "tframe" and nrow() for a time 
series matrix to periods().

The tframe assignment example above is accomplished by switching the 
dispatch so that it follows the classes of the tframe of y, rather than 
the classes of x as would normally be done for the above kind of assignment.
Doing this in a generic way allows for the possibility
of future classes of time representation. This is different from the
way that rts, cts, its and ts are implemented, in the sense that it is the
tframe of the data which is assigned a class indicating the
time representation, not the data object itself.

The most general (last) class of the tframe should be "tframe". 
The method "is.tframe"  checks if an object is a tframe, and 
the method "is.tframed" checks if an object has a tframe. 
In general, tframe methods act on the time frame (tframe) and
tframed methods act on data which is tframed.

More specific methods can be defined for any special time
representation (methods are defined in this library for tframes of class
c("default", "tframe") which in Splus are the old style tsp convention for time).
Also, there is a sketched implementaion for rts, cts,
its, and a class and methods style implementation of tsp called tf. The
tframe's specific classes are called rtstframe, ctstframe, and itstframe, 
to prevent confusion using inherit().
The R version includes an implementaion for the ts time series class with
tframe's specific class is tstframe.

When implementing a new time series, suppose it is called zzz, then the tframe
attibute of an object using this time frame should have class c("zzz", "tframe").
(Note zzz should be different from the class of the object itself.)
The most important methods which need to be
supported are tframe.zzz(), start.zzz(), end.zzz(), and periods.zzz().
While frequency.zzz() should not in theory be necessary, it makes 
porting code much easier. Other methods which should be supported are
time.zzz(), checktframeConsistent.zzz(), tfTruncate.zzz(), 
tfExpand.zzz(), 
earliestStartIndex.zzz(),
earliestEndIndex.zzz(), latestStartIndex.zzz(), and
latestEndIndex.zzz().

The method tfwindow is used in this library and is typically just the same as
window, but the new name has been used because of historical changes and bugs
in window, and in order to support the argument "warn" to suppress  
messages (when objects are windowed unnecessarily).

One implication of the desire to be able to use a statement like
tframe(x) <- tframe(y)
is that the tframe should not indicate which dim of the data is the
time dimension. In general this will have to be another attribute of
the data, but the current convention of using the first dimension for
matrix data and the length for vector data, makes it unnecessary to
specify.

Operations which should be possible on tframed data:

  In the time dimension
  
     - window (tfwindow), splice 
     
  In other dimensions
  
     -tbind  (like cbind) with shift to align the the tframe 
                 (and NA pad.start, pad.end, pad=TRUE/FALSE) 
         
     - [  ]   without losing the tframe. This is done with selectSeries().


The attribute "seriesNames" is also supported as a way to indicate the names
of series in an object. This overlaps with the use of "names" and 
"dimnames[[2]]" used previously for series names in S, but seems necessary in
order to have a more complete generic decompostion of the time dimension from
the other dimensions. 

Many of the functions in the library are not yet individually documented 
(and were it exists the documentation is a draft) however, the functions
are all very short and can be examined easily. 
The code in the tests subdirectory provides a short set of tests and may
serve as an example.
}

\keyword{programming}
\keyword{utilities}
\keyword{ts}

\author{Paul Gilbert}


\eof
\name{addDate}
\alias{addDate}
\title{Add Periods to a Date}
\description{
    Add periods to two element start date of given frequency to give a
    new date. NULL periods is treated as 0.
}
\usage{
    addDate(date, periods, freq)
}
\arguments{
    \item{date}{A two element date as used by tsp i.e c(year, period).}
    \item{periods}{A number of periods.}
    \item{freq}{The number of periods in a year.}
}
\value{A two element date.}

\seealso{
    \code{\link{tfExpand}}
}
\examples{
    addDate(c(1998,1), 20, 12)
}

\note{A useful utility not strictly part of tframe.}

\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}




\eof
\name{checktframeConsistent}
\alias{checktframeConsistent}
\alias{checktframeConsistent.default}

\title{Check for a Consistent tframe}
\description{
    Check if tframe and a time series are consistent with one another.}
\usage{
    checktframeConsistent(tf, x)
    \method{checktframeConsistent}{default}(tf, x)
    }

\arguments{
    \item{tf}{A tframe)}
    \item{x}{An object)}
}
\value{A logical scalar.}
\details{Check if the number of periods in the tframe corresponds to the
    number of observations in the time series.
}
\seealso{
    \code{\link{is.tframe}}
    \code{\link{periods}}
}

\examples{
    z <- ts(rnorm(100), start=c(1982,1), frequency=12)
    checktframeConsistent(tframe(z), rnorm(100))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{classed}
\alias{classed}

\title{Tframe Library Utilities}
\description{Some utilities used by other functions in the tframe library.}

\usage{
    classed(x, cls)
}
\arguments{
    \item{x}{An object.}
    \item{cls}{A vector of strings.}
}
\keyword{internal}


\eof
\name{earliestEnd}
\alias{earliestEnd}
\alias{earliestEndIndex}
\alias{earliestEndIndex.default}
\alias{earliestEndIndex.tframe}

\alias{earliestStart}
\alias{earliestStartIndex}
\alias{earliestStartIndex.default}
\alias{earliestStartIndex.tframe}

\alias{latestEnd}
\alias{latestEndIndex}
\alias{latestEndIndex.default}
\alias{latestEndIndex.tframe}

\alias{latestStart}
\alias{latestStartIndex}
\alias{latestStartIndex.default}
\alias{latestStartIndex.tframe}

\title{Start and End for Objects with Multiple Time Series}
\description{Return start or end date (or index of the object) from multiple
    time series objects.
}

\usage{
    earliestEnd(x, ...)
    earliestEndIndex(x, ...)
    \method{earliestEndIndex}{default}(x, ...)
    \method{earliestEndIndex}{tframe}(x, ...)

    earliestStart(x, ...)
    earliestStartIndex(x, ...)
    \method{earliestStartIndex}{default}(x, ...)
    \method{earliestStartIndex}{tframe}(x, ...)

    latestEnd(x, ...)
    latestEndIndex(x, ...)
    \method{latestEndIndex}{default}(x, ...)
    \method{latestEndIndex}{tframe}(x, ...)

    latestStart(x, ...)
    latestStartIndex(x, ...)
    \method{latestStartIndex}{default}(x, ...)
    \method{latestStartIndex}{tframe}(x, ...)

}

\arguments{
    \item{x}{A tframe or tframed object.}
    \item{...}{Additional tframe or tframed objects.}
}
\value{A date or index.}
\details{
    These functions calculate the start
    and end of each object in the argument and return a result by 
    comparing across objects. Thus, latestStart returns the start date of
    the object which starts latest and latestStartIndex returns the 
    corresponding index of the object in the argument list. 
}
\seealso{
    \code{\link{tframe}}
    \code{\link{tfwindow}}
    \code{\link{tfTruncate}}
    \code{\link{trimNA}}
}

\examples{
    t1<-ts(c(1,2,3,4,5), start=c(1991,1))
    t2<-ts(c(2,3,4,5,6,7,8), start=c(1992,1))
    t3<-ts(c(NA,2,3,4,5), start=c(1991,1))

    latestStart(t1,t2,t3)  # 1992 1 corresponding to the starting date of 
                            # the object which starts latest (t2)
    latestStart(t1,t3)     # both start in 1991 1 (NAs count as data)
    latestStart(tbind(t1,t2,t3)) # tbind gives a single object starting in 1991 1
    latestStart(t2, tbind(t1,t2,t3))

    latestStartIndex(t1,t2,t3)  # position of t2 in the argument list
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{freeze}
\alias{freeze}
\alias{freeze.default}
\title{Get fixed data snapshot}
\description{Generic method. See specific methods for details.}

\usage{
    freeze(data, ...)
    \method{freeze}{default}(data, ...)
}

\arguments{
    \item{data}{A data source description.}
    \item{...}{Additional arguments for specific methods.}
}

\details{
This function extracts data from 
a database (for example using the TS PADI programs are available 
at \url{www.bank-banque-canada.ca/pgilbert}). This method is generic.
The function logically belongs with the tfpadi functions in the dsepadi package
but the generic and default are included here since they are required in many
functions whereas the database interface has much more limited use.
Typically the argument data is a tfPADIdata or TSPADIdata object identifying 
the source of the data. See help for tfPADIdata and TSPADIdata. 
The default method usually just
returns its argument, so freeze has no effect. This way freeze
can be used to write functions which will take a snapshot from the database
when they execute or will work with an already fixed copy of data if that is
what is supplied. The default does allow for a character argument, in which case
it is used to construct a tfPADIdata object using server="ets", then freeze that
object. This allows for a simple syntax to grab a series from the database. The
server="ets" is for convenience at the Bank of Canada and another default server
might be more convenient elsewhere.
}

\seealso{ 
    \code{\link[dsepadi]{tfPADIdata}}
    \code{\link[dsepadi]{tfputpadi}}
    \code{\link[dse2]{freeze.tfPADIdata}}
    \code{\link[dsepadi]{freeze.TSPADIdata}}
    \code{\link[dse2]{TSPADIdata}}
}
\keyword{programming}
\keyword{ts}
\keyword{data}



\eof
\name{makeTSnoise}
\alias{makeTSnoise}
\title{Generate a random time series}
\description{
    Generate a random time series (matrix). This is a utility typically used in a time
    series model simulate method and not called directly by the user.}
\usage{
    makeTSnoise(sampleT,p,lags,noise=NULL, rng=NULL,
             SIGMA=NULL, sd=1, noise.model=NULL, noise.baseline=0,
             tf=NULL, start=NULL,frequency=NULL)
}
\arguments{
    \item{sampleT}{an integer indicating the number of periods.}
    \item{p}{an integer indicating the number of series.}
    \item{lags}{an integer indicating the number of periods prior to the sample
       (initial data w0) for which random numbers should be generated. This is
       useful in ARMA models.}
    \item{noise}{Noise can be supplied. Otherwise it will be generated.
        If supplied it should be a list as described below under returned value.}
    \item{SIGMA}{The covariance of the noise process. If this is specified then sd
       is ignored. A vector or scalar is treated as a diagonal matrix. For an object of class
       TSestModel, if neither SIGMA nor sd are specified, then SIGMA is set to
       the estimated covariance (\code{model$estimates$cov}).}
    \item{sd}{The standard deviation of the noise. This can be a vector.}
    \item{noise.model}{A TSmodel to be used for generating noise (not yet
       supported by SS methods).}
    \item{noise.baseline}{a constant or matrix to be added to noise. Alternately this
       can be a vector of length p, each value of which is treated as a constant to
       add to the coresponding noise series.}
    \item{rng}{The random number generator information needed to
       regenerate a simulation.}
    \item{tf}{a time frame to use for the generated matrix. (alternately
       use start and frequency)}
    \item{start}{a time start date to use for the generated matrix.}
    \item{frequency}{a time frequency to use for the generated matrix.}
}
\value{A time series matrix.}
\keyword{internal}

\eof
\name{nseries}
\alias{nseries}
\alias{nseries.default}

\title{Number of Series}
\description{Return the number of series.}
\usage{
    nseries(x)
    \method{nseries}{default}(x)
    }

\arguments{
    \item{x}{A time series object.}
}
\value{An integer.}
\details{
    Generic method to return the number of series.
}

\examples{
    nseries(tbind(rnorm(100,20,5)))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{selectSeries}
\alias{selectSeries}
\alias{selectSeries.default}
\alias{selectSeries.ts}

\title{Extract a Subset of Series}
\description{Extract a subset of series from a tframed object.}
\usage{
    selectSeries(x, series = seqN(nseries(x)))
    \method{selectSeries}{default}(x, series = seqN(nseries(x)))
    \method{selectSeries}{ts}(x, series = seqN(nseries(x)))
    }

\arguments{
    \item{x}{A tframed object.}
    \item{series}{The subset of series to retain.}
}
\value{A tframed object.}
\details{
    This is like [ , , drop=FALSE] but retains class, series name and tframe 
    information. It
    also provides a methods which works with multivariate series which 
    are not matrices (e.g. tfPADIdata).
}
\seealso{
    \code{\link{seriesNames}}
    \code{\link[dsepadi]{tfPADIdata}}
}
\examples{
    z <- selectSeries(matrix(rnorm(1000), 100,10), series=c(2, 5, 6))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{seqN}
\alias{seqN}

\title{Tframe Library Utilities}
\description{Some utilities used by other functions in the tframe library.}

\usage{
   seqN(N)
}

\arguments{
    \item{N}{NULL, 0, or a positive integer}
}
\keyword{internal}


\eof
\name{seriesNames}
\alias{seriesNames}
\alias{seriesNames.default}
\alias{seriesNames<-}
\alias{seriesNames<-.default}
\alias{seriesNames<-.ts}

\title{Names of Series in a time series object}
\description{Extract or set names of series in a time series object.}
\usage{
    seriesNames(x)
    \method{seriesNames}{default}(x)

    seriesNames(x) <- value 
    \method{seriesNames}{default}(x) <- value
    \method{seriesNames}{ts}(x) <- value
}
\arguments{
    \item{x}{a time series object.}
    \item{value}{names to be given to time series.}
}
\value{
The first usage returns a vector of strings with the series names. 
The assignment method makes \code{names} (a vector of strings) 
the series names of data.
}
\seealso{
    \code{\link{tframed}},
    \code{\link[dse1]{seriesNamesInput}},
    \code{\link[dse1]{seriesNamesOutput}}
}

\examples{
    z <- matrix(rnorm(100), 50,2)
    seriesNames(z) <- c("a", "b")
    seriesNames(z)
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{splice}
\alias{splice}
\alias{splice.default}
\title{Splice Time Series}
\description{
Splice together (in time dimension) two time series objects. This function can also be 
used to overlay obj1 on obj2 (obj1 takes precedence). The time windows do
not have to correspond.
}
\usage{
    splice(mat1,mat2, ...)
    \method{splice}{default}(mat1,mat2, ...)
    }
\arguments{
    \item{mat1}{A time series object.}
    \item{mat2}{A time series object.}
    \item{...}{arguments to be passed to other methods (not used by the default
       method).}
}
\value{A time series object}
\details{
Splice together two time series objects. The mat1 and mat2 objects should contain the 
same number of time series variables and be arranged in the same order.
(e.g. - the first column of mat1 is spliced to the first column of 
mat2, etc.).
If data  is provided in both mat1 and mat2 for a given period then mat1 takes 
priority. The frequencies should be the same.
}

\seealso{
    \code{\link{tfwindow}},
    \code{\link{trimNA}},
    \code{\link{tbind}}
}

\examples{
splice(ts(matrix(rnorm(24),24,1), start=c(1980,1), frequency=4),
       ts(matrix(rnorm(6),  6,1), start=c(1986,1), frequency=4))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{tbind}
\alias{tbind}
\alias{tbind.default}
\alias{tbind.ts}

\title{Bind Time Series}
\description{
Bind together (in non-time dimension) two time series objects. 
}
\usage{
    tbind(x, ..., pad.start=TRUE, pad.end=TRUE, warn=TRUE)
    \method{tbind}{default}(x, ..., pad.start=TRUE, pad.end=TRUE, warn=TRUE)
    \method{tbind}{ts}(x, ..., pad.start=TRUE, pad.end=TRUE, warn=TRUE)
    }
\arguments{
    \item{x}{A time series object.}
    \item{...}{Time series objects.}
    \item{pad.start}{Logical indicating if the start should be truncated or
        padded with NAs to align time.}
    \item{pad.end}{Logical indicating if the end should be truncated or
        padded with NAs to align time.}
    \item{warn}{Logical indicating if warnings should be issued.}
}
\value{A time series object}
\details{
Bind data as in cbind (or formerly tsmatrix) and align time dimension. 
The default action pads series with NA to time union. 
If pad.start and/or pad.end is FALSE and the intersection is empty then NULL
is returned and a warning is issued if warn=TRUE.
}

\seealso{
    \code{\link{tfwindow}},
    \code{\link{trimNA}},
    \code{\link{splice}}
}

\examples{
tbind(    ts(matrix(rnorm(24),24,1), start=c(1986,1), frequency=4),
    ts(matrix(rnorm(6),  6,1), start=c(1986,1), frequency=4))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{testEqual}
\alias{testEqual}
\alias{testEqual.default}
\alias{testEqual.array}
\alias{testEqual.list}
\alias{testEqual.matrix}
\alias{testEqual.numeric}

\title{Compare Two Objects}
\description{
Generic function to compare two objects.  The methods return a 
logical value, TRUE if the objects are the same type and value and FALSE otherwise.
The default compares array values but not attributes or class.
Some descriptive information in the objects may be ignored.
}
\usage{
    testEqual(obj1, obj2, fuzz = 0)
    \method{testEqual}{default}(obj1, obj2, fuzz = 1e-16)
    \method{testEqual}{array}(obj1, obj2, fuzz = 1e-16)
    \method{testEqual}{list}(obj1, obj2, fuzz = 1e-16)
    \method{testEqual}{matrix}(obj1, obj2, fuzz = 1e-16)
    \method{testEqual}{numeric}(obj1, obj2, fuzz = 1e-16)
    }
\arguments{
    \item{obj1, obj2}{Objects of the same class.}
    \item{fuzz}{Differences less than fuzz are ignored.}
}
\value{
TRUE or FALSE.
}
\details{
The functions for comparing numeric values used in the default method for this 
generic replacement.
}
\examples{
testEqual(matrix(1:10,10,2), array(1:10, c(10,2)))
testEqual(matrix(1:10,10,1),1:10)
}
\seealso{
    \code{\link{testEqualtframes}}
}
\keyword{programming}
\keyword{utilities}



\eof
\name{testEqualtframes}
\alias{testEqualtframes}
\alias{testEqualtframes.default}
\alias{testEqualtframes.stamped}

\title{Compare Two Time Frames}
\description{
Generic function to compare two time frames.  The methods return a 
logical value, TRUE if the time frames are the same type and value and FALSE otherwise.
}
\usage{
    testEqualtframes(tf1,tf2)
    \method{testEqualtframes}{default}(tf1,tf2)
    \method{testEqualtframes}{stamped}(tf1,tf2)
    }
\arguments{
    \item{tf1, tf2}{Time frames of the same class.}
}
\value{TRUE or FALSE}
\details{
Time frames are compared. Time frames need to be of the same class (although it would be nice if they did
not need to be).
}
\examples{
testEqualtframes(tframe(matrix(1:10,10,2)), tframe(array(1:10, c(10,2))))
}
\seealso{
    \code{\link{testEqual}}
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{tfExpand}
\alias{tfExpand}
\alias{tfExpand.default}
\alias{tfExpand.tframe}
\alias{tfTruncate}
\alias{tfTruncate.default}
\alias{tfTruncate.tframe}

\title{Expand a Tframe or Tframed Object.}
\description{Expand a tframe or tframed object.}

\usage{
    tfExpand(x, add.start = 0, add.end = 0)
    \method{tfExpand}{default}(x, add.start = 0, add.end = 0)
    \method{tfExpand}{tframe}(x, add.start = 0, add.end = 0)

    tfTruncate(x, start=NULL, end=NULL)
    \method{tfTruncate}{default}(x, start=NULL, end=NULL)
    \method{tfTruncate}{tframe}(x, start=NULL, end=NULL)
}
\arguments{
    \item{x}{A tframe or tframed object.}
    \item{start}{an integer indicating the position at which the new tframe is
        to start.}
    \item{end}{an integer indicating the position at which the new tframe is
        to end.}
    \item{add.start}{an integer indicating the number of periods on the
        beginning.}
    \item{add.end}{an integer indicating the number of periods on the end.}
 }
\value{A tframe or tframed object.}
\details{
    These methods are like tfwindow but use position indicators (rather 
    than dates) and work with a tframe or tframed data. Applied to a tframe
    they return an adjusted tframe. Applied to a tframed object
    they return an adjusted object with its adjusted tframe.They are low level
    utilities for other functions.
}
\seealso{
    \code{\link{tfwindow}}
    \code{\link{tframed}}
}

\examples{
    z <- ts(rnorm(100), start=c(1982,1), frequency=12)
    Dz <- tframed(diff(z), tfTruncate(tframe(z), start=2))
    tframe(Dz)
    IDz <- tframed(cumsum(c(0, Dz)), tfExpand(tframe(Dz), add.start=1))
    tframe(IDz)
    tframe(tfTruncate(z, start=5))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{tfdiff}

\alias{tfdiff}
\alias{tfdiff.default}
\alias{tfdiff.tframe}
\alias{diff.tframed}
%\alias{lag}
%\alias{lag.default}
\title{Time Series Differencing}
\description{Difference a tframed object.}
\usage{
    tfdiff(x, lag=1, differences=1)
    \method{tfdiff}{default}(x,lag=1, differences=1)
    \method{tfdiff}{tframe}(x,lag=1, differences=1)
    \method{diff}{tframed}(x,lag=1, differences=1, ...)
}
\arguments{
    \item{x}{a tframed object.}
    \item{lag}{difference calculated relative to lag periods previous.}
    \item{differences}{order of differencing.}
    \item{...}{arguments to be passed to other methods.}
}
\details{A time framed object is created by differencing the number of times
   indicated by differences at a lagged number of periods indicated by lag. The
   default is take the difference from data one period previous.
}
\seealso{
    \code{\link[base]{diff}},
    \code{\link[ts]{lag}}
}
\examples{
    z <- ts(rnorm(100), start=c(1982,1), frequency=12)
    tfstart(z)
    tfperiods(z)
    z <- tfdiff(z)
    tfstart(z)
    tfperiods(z)
}
\keyword{internal}



\eof
\name{tfplot}
\alias{tfplot}
\alias{tfplot.default}
\alias{tfOnePlot}

\title{Plot Tframed Objects}
\description{
    Plot tframe or tframed objects. 
}
\usage{
    tfplot(x, ...)

    \method{tfplot}{default}(x, ..., tf=tfspan(x, ...), start=tfstart(tf), end=tfend(tf),
       series=seq(nseries(x)), Title=NULL, xlab=NULL, ylab=seriesNames(x), 
       graphs.per.page=5, mar=par()$mar, reset.screen=TRUE)
    tfOnePlot(x, xlab=NULL, ylab=NULL, 
              tf=tframe(x), start=tfstart(tf), end=tfend(tf),...)
}

\arguments{
    \item{x}{a tframe or tframed object to plot.}
    \item{...}{any additional tframed objects for the same plot.}
    \item{start}{start of plot. (passed to tfwindow)}
    \item{end}{end of plot. (passed to tfwindow)}
    \item{tf}{a tframe or tframed object which can be used to specify start and
    end.}
    \item{series}{series to be plotted. (passed to selectSeries)}
    \item{Title}{string to use for plot title.}
    \item{xlab}{string to use for x label (passed to plot).}
    \item{ylab}{string to use for y label (passed to plot).}
    \item{graphs.per.page}{integer indicating number of graphs to place on a
       page.}
    \item{mar}{margins passed to plot. See par.)}
    \item{reset.screen}{logical indicating if the plot window should be cleared
       before starting. If this is not TRUE then mar values will have no effect.}
}
\value{None.}
\details{
    In many cases these are the same as plot methods, however, tfplot puts
    different series in the object \code{x} in different plot panels, whereas
    \code{plot} usually puts them in the same panel. For this reason,
    \code{tfplot} tends to work better when the scale of the different series
    are very different. If additional objects are supplied, then they should
    each have the same number of series as \code{x} and all corresponding series
    will be plotted in the same panel.
    
    \code{tfplot} provides an alternate generic
    mechanism for plotting time series data.
    New classes of time series may define
    there own \code{tfplot} (and \code{plot}) methods.
    
    The start and end arguments to tfplot 
    determine the start and end of the plot. The argument tf is an alternate way
    to specify the start and end. It is ignored if start and end are specified.
}
\sideffects{An object is plotted.}
\seealso{
    \code{\link{tfprint}}
    \code{\link{tframe}}
    \code{\link{tframed}}
    \code{\link{print}}
    \code{\link{plot}}
}
\examples{
    tfplot(ts(rnorm(100), start=c(1982,1), frequency=12))
    tfplot(ts(rnorm(100), start=c(1982,1), frequency=12), start=c(1985,6))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{hplot}



\eof
\name{tfprint}
\alias{tfprint}
\alias{tfprint.default}
\alias{tfprint.tframe}
\alias{print.tframe}

\title{Print Tframed Objects}
\description{
    Print tframe or tframed objects. 
}
\usage{
    tfprint(x, ...)
    \method{tfprint}{default}(x, ...)
    \method{tfprint}{tframe}(x, ...)
    \method{print}{tframe}(x, ...)
}

\arguments{
    \item{x}{a tframe or tframed object.}
    \item{...}{arguments to be passed to other methods.}
}
\value{tfprint methods return the object invisibly.}
\details{
    \code{tfprint} prints data in a tframed object while 
    \code{tframePrint} prints the tframe.
    In many cases these are the same as print methods. 
    However, tfprint tries to provide an alternate generic
    mechanism that is consistent with the tframe view of the data.
    This may not always be the preferred print method. 
    Also, new classes of time series may define
    there own print methods in ways which use a different logic
    from the tframe library. Thus tfprint provides a
    way to program functions which use methods consistent with the tframe
    library logic.
}
\sideffects{An object is printed.}
\seealso{
    \code{\link{tfplot}}
    \code{\link{tframe}}
    \code{\link{tframed}}
    \code{\link{print}}
    \code{\link{plot}}
}
\examples{
    tfprint(ts(rnorm(100)))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}



\eof
\name{tframe}
\alias{tframe}
\alias{tframe.default}
\alias{tframe.ts}
\alias{tframe<-}
\alias{tframe<-.default}
\alias{tframe<-.rts}
\alias{tframe<-.cts}
\alias{tframe<-.its}
\alias{tframe<-.ts}
\alias{tframed}
\alias{tframed.default}
\alias{is.tframe}
\alias{is.tframed}

\title{Extract or Set a tframe}
\description{Extract or set the tframe of an object.}

\usage{
    tframe(x)
    \method{tframe}{default}(x)
    \method{tframe}{ts}(x)

    tframe(x) <- value
    \method{tframe}{default}(x) <- value
    \method{tframe}{rts}(x) <- value
    \method{tframe}{cts}(x) <- value
    \method{tframe}{its}(x) <- value
    \method{tframe}{ts}(x) <- value

    tframed(x, tf=NULL, names = NULL)
    \method{tframed}{default}(x, tf = NULL, names = NULL)

    is.tframe(x)
    is.tframed(x)
}

\arguments{
    \item{x}{an object (to which a tframe is assigned in assignment methods).}
    \item{value}{a tframe.}
    \item{tf}{a tframe object or a tframed object from which a tframe is taken.}
    \item{names}{optional vector of strings to specify new series names.}
}
\value{Depends.}
\details{
    The first usage returns the tframe of a tframed object. The assignment
    methods and tframed set the tframe of an object. \code{is.tframe} returns 
    a logical.
    
    The pure tframe approach is to set a "tframe" attribute for an object. This
    attribute has a class which indicates the time framing which is used. The 
    the time frame information is often secondary, in the sense that it does not
    describe the object structure, but only provides some additional information
    which is useful for doing time based operations on the data, plotting, and
    printing the object. By putting this in 
    an attribute, the objects class can be used for indicating other
    information about the structure of the object.  For
    these pure tframe objects the default \code{tframe} and code{tframe<-} will
    often be adequate. The generic/method approach allows for special case (like
    TSdata where the tframe information is not an attribute of the object, but
    rather an attribute of the data matrices which are elements of the object).

    The generic/method approach also allows for (faking) tframe assignment and 
    extraction with classes like rts, ctc, its, ts, and others which may appear,
    that try to make the time description part of the object class. (Not 
    a "tframe" approach.) The problem is to extract real tframes and
    also fake these other classes and old style tsp objects so they look 
    like tfamed objects. Another approach would be to mutilate these objects 
    and force them really be tframed objects (to have a tframe attribute), 
    but that risks conflicting with other (non tframe) code which used 
    the objects. This faking is accomplished by specific methods of the classes,
    and for old style tsp objects it is built into the default.

    This \code{tframed} constructor is simply a shortcut for assigning the tframe
    (tframe(x) <- tf) and series names (seriesNames(x) <-  names) to an object,
    but never assigns NULL values, so the result is guaranteed to be a 
    code{tframed} object. It is like \code{ts} but enables the tframe library's  
    methods for handling time. If the \code{tf} argument is a 
    \code{tframed} object rather than a \code{tframe}, then the code{tframe} 
    is extracted and used. If the \code{names} argument is not mode "character"
    of appropriate length, then \code{seriesNames(names)} is used. These make it
    simple to assign the time frame and names of one object to another by
    \code{z <- tframed(x, tf=y, names=y)}.
    
    \code{is.tframed} returns TRUE if a
    \code{tframe()} can extract a tframe from the object. This is true for many
    objects which are not truly tframed (like ts objects), since \code{tframe()}
    tries fairly hard to build a tframe for the object. 
}
\seealso{
    \code{\link{tfstart}},
    \code{\link{tfend}},
    \code{\link{tffrequency}},
    \code{\link{tfperiods}},
    \code{\link{tftime}},
    \code{\link{tfdiff}}
}
\examples{
    z <- tframe(ts(rnorm(100), start=c(1982,1), frequency=12))
    is.tframe(z)
    zz <- tframed(matrix(rnorm(200), 100,2), tf=z)
    is.tframed(zz)
    zzz <- tframed(matrix(rnorm(200), 100,2), tf=zz)
    is.tframed(zzz)
    tframe(zzz)
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}


\eof
\name{tfspan}

\alias{tfspan}

\title{Time Span}
\description{Calculate Time Span of Objects.}

\usage{
    tfspan(x, ...)
}

\arguments{
    \item{x}{a tframe or a tframed object.}
    \item{...}{other tframes or tframed objects.}
}
\value{A tframe}
\details{
    Calculate the time frame from the earliest start to latest end of all
    arguments. 
}

\seealso{
    \code{\link{tframe}}, 
    \code{\link{tframed}}
    \code{\link{start}}
    \code{\link{end}}
    \code{\link{frequency}}
    \code{\link{periods}}
    \code{\link{time}}
}
\examples{
    z  <- ts(rnorm(100), start=c(1982,1), frequency=12)
    zz <- ts(rnorm(100), start=c(1992,1), frequency=12)
    tfspan(z, zz)
}

\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{tfstart}

\alias{tfstart}
\alias{tfstart.default}
\alias{tfstart.tstframe}
\alias{start.tframed}
\alias{start.tframe}

\alias{tfend}
\alias{tfend.default}
\alias{tfend.tstframe}
\alias{end.tframed}
\alias{end.tframe}

\alias{tffrequency}
\alias{tffrequency.default}
\alias{frequency.tframed}
\alias{frequency.tframe}

\alias{tfperiods}
\alias{tfperiods.default}
\alias{tfperiods.stamped}
\alias{periods}
\alias{periods.default}
\alias{periods.tframed}
\alias{periods.tframe}


\alias{tftime}
\alias{tftime.default}
\alias{time.tframed}
\alias{time.tframe}

\title{Extract Time Frame Information}
\description{Functions for extracting time frame information.}

\usage{
    \method{start}{tframed}(x, ...)
    \method{start}{tframe}(x, ...)
    tfstart(x)
    \method{tfstart}{default}(x)
    \method{tfstart}{tstframe}(x)

    \method{end}{tframed}(x, ...)
    \method{end}{tframe}(x, ...)
    tfend(x)
    \method{tfend}{default}(x)
    \method{tfend}{tstframe}(x)

    \method{frequency}{tframed}(x, ...)
    \method{frequency}{tframe}(x, ...)
    tffrequency(x)
    \method{tffrequency}{default}(x)

    periods(x)
    \method{periods}{default}(x)
    \method{periods}{tframed}(x)
    \method{periods}{tframe}(x)
    tfperiods(x)
    \method{tfperiods}{default}(x)
    \method{tfperiods}{stamped}(x)

    \method{time}{tframed}(x, ...)
    \method{time}{tframe}(x, ...)
    tftime(x)
    \method{tftime}{default}(x)
    \method{time}{tframed}(x, ...)
}

\arguments{
    \item{x}{a tframe or a tframed object.}
    \item{...}{arguments to be passed to other methods.}
}
\value{Depends}
\details{
    The methods start and end return the start or end date of a tframe or
    tframed object. Periods return the number of observations (time points).
    frequency returns the frequency of observation, 
    typically the number of observations in a year for economic data, but
    possibly something else in other contexts.  The concept of frequency is
    not very consistently defined for time series data, and the use of 
    the frequency method should probably be avoided where possible. In
    practice it seems rarely necessary, but the method makes porting of 
    older code much easier. 
}

\seealso{
    \code{\link{tframe}}, 
    \code{\link{tframed}}
    \code{\link[base]{start}}
    \code{\link[base]{end}}
    \code{\link[base]{frequency}}
    \code{\link[base]{periods}}
    \code{\link[base]{time}}
    \code{\link{lag}}
    \code{\link{diff}}
}
\examples{
    z <- ts(rnorm(100), start=c(1982,1), frequency=12)
    tfstart(z)
    z <- tframed(matrix(rnorm(200), 100,2), 
            tf=list(start=c(1982,1), frequency=12))
    tfend(z)
    periods(z)
    time(z)
}

\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}


\eof
\name{tfwindow}
\alias{tfwindow}
\alias{tfwindow.default}
\alias{tfwindow.ts}
\alias{tfwindow.tframe}

\title{Truncate a Time Series}
\description{Truncate a time series object to a time window.}
\usage{
    tfwindow(x, tf=NULL, start=tfstart(tf), end=tfend(tf), warn=TRUE)
    \method{tfwindow}{default}(x, tf=NULL, start=tfstart(tf), end=tfend(tf), warn=TRUE)
    \method{tfwindow}{ts}(x, tf=NULL, start=tfstart(tf), end=tfend(tf), warn=TRUE)
    \method{tfwindow}{tframe}(x, tf=NULL, start=tfstart(tf), end=tfend(tf), warn=TRUE)
    }

\arguments{
    \item{x}{A time series object.}
    \item{start}{A start date of a format compatible with the time series}
    \item{end}{An end date of a format compatible with the time series}
    \item{tf}{A tframe or tframed object}
    \item{warn}{A logical indicating if warning should be produced}
}
\value{
A time series object similar to x, but typically spanning a shorter time period.
}
\details{
If \code{start} or \code{end} are omitted and \code{tf} is specified then the
start or end is taken from the \code{tf} object.
For ts class objects this function calls window but makes more effort to 
preserve \code{seriesNames} if \code{x} has them. 
It also supports the optional argument \code{warn} to suppress
warning messages. Frequently it is convenient to write code which always 
truncates to a window without first checking if the data is already within 
the window. Since window produces a warning in this situation, the optional 
argument is frequently useful when tfwindow is used by other code.
In Splus \code{tfwindow} also corrects for some bugs in older versions 
of code{window}.

The method for windowing a tframe is a utility to be used by other programs and
would not typically be called by a user.
}
\examples{
  z <- ts(matrix(rnorm(24),24,1), start=c(1980,1), frequency=4)
  zz <- tfwindow(z, start=c(1982,2))
  zzz <- matrix(rnorm(24),24,1)
  tframe(zzz) <- tframe(z)
  tfwindow(zzz, tf=tframe(zz))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
\name{trimNA}
\alias{trimNA}
\alias{trimNA.default}

\title{Trim NAs from Time Series}
\description{Trim NAs from the start and end of a time series object.}
\usage{
    trimNA(x, startNAs=TRUE, endNAs=TRUE)
    \method{trimNA}{default}(x, startNAs=TRUE, endNAs=TRUE)
    }
\arguments{
    \item{x}{A time series matrix or an object of class TSdata.}
    \item{startNAs}{If FALSE then beginning NAs are not trimmed.}
    \item{endNAs}{If FALSE   then ending NAs are not trimmed.}
}
\value{A time series object which is windowed to the subset of 
 data which is not NAs (usually the available data).}
\details{
Trim NAs from the ends of a time series object.
Observations in a given period for all series are dropped if any one contains an NA.
}
\examples{
trimNA(ts(rbind(NA, matrix(1:20,10,2)), start=c(1980,1), frequency=12))
}
\keyword{programming}
\keyword{utilities}
\keyword{ts}
\keyword{chron}



\eof
