\name{ellipse}
\alias{ellipse}
\alias{ellipse.default}
\title{
Make an ellipse
}
\description{
A generic function returning an ellipse or other outline of a confidence region
for two parameters.  
}
\usage{
ellipse(x, \dots)
ellipse.default(x, scale = c(1, 1), centre = c(0, 0), level = 0.95, 
    t = sqrt(qchisq(level, 2)), which = c(1, 2), npoints = 100, \dots)
}
\arguments{
\item{x}{
An object. In the default method the parameter \code{x} should be a correlation between -1 and 1 or a
square positive definite matrix at least 2x2
in size. It will be treated as the correlation or covariance 
of a multivariate normal distribution.
}
\item{\dots}{
Descendant methods may require additional parameters.
}
\item{scale}{
If \code{x} is a correlation matrix, then the standard deviations of each
parameter can be given in the scale parameter.  This defaults to \code{c(1, 1)},
so no rescaling will be done.
}
\item{centre}{
The centre of the ellipse will be at this position.
}
\item{level}{
The confidence level of a pairwise confidence region.  The default is
0.95, for a 95\% region.  This is used to control the size of the ellipse
being plotted.  A vector of levels may be used.
}
\item{t}{
The size of the ellipse may also be controlled by specifying the value
of a t-statistic on its boundary.  This defaults to the appropriate
value for the confidence region.
}
\item{which}{
This parameter selects which pair of variables from the matrix will be
plotted.  The default is the first 2.
}
\item{npoints}{
The number of points used in the ellipse.  Default is 100.
}
}
\value{
An \code{npoints} x \code{2} matrix is returned with columns named according to the
row names of the matrix \code{x} (default \code{'x'} and \code{'y'}), suitable
for plotting.
}
\details{
The default method uses the 
\code{(cos(theta + d/2), cos(theta - d/2))} parametrization of an ellipse, where 
\code{cos(d)} is the correlation of the parameters.
}
\seealso{
\code{\link{ellipse.lm}}, \code{\link{ellipse.nls}}}, 
\code{\link{ellipse.profile}, \code{\link{ellipse.profile.nls}},
\code{\link{ellipse.arima0}},
\code{\link{plotcorr}}
}
\references{
Murdoch, D.J. and Chow, E.D. (1996). A graphical display of large 
correlation matrices. The American Statistician 50, 178-180. 
}
\examples{
# Plot an ellipse corresponding to a 95\% probability region for a
# bivariate normal distribution with mean 0, unit variances and 
# correlation 0.8.
plot(ellipse(0.8), type = 'l')
}
\keyword{dplot}

\eof
\name{ellipse.arima0}
\alias{ellipse.arima0}
\title{
Outline an approximate pairwise confidence region 
}
\description{
This function produces the ellipsoidal outline of an approximate 
pairwise confidence region for an ARIMA model fit.
}
\usage{
\method{ellipse}{arima0}(x, which = c(1, 2), level = 0.95, t = sqrt(qchisq(level, 2)), \dots)
}
\arguments{
\item{x}{
The first argument should be an \code{arima0} object, usually resulting from a call
to \code{arima0()}.
}
\item{which}{
Which selects the pair of parameters to be plotted.  The default is the
first two.
}
\item{level}{
The confidence level of the region.  Default 95\%.
}
\item{t}{
The t statistic on the boundary of the ellipse.
}
\item{\dots}{
Other \code{ellipse.default} parameters may also be used.
}}
\value{
A matrix with columns \code{x} and \code{y} to outline the confidence region.
}
\details{
The \code{summary} function is used to obtain the approximate covariance matrix of the
fitted parameters.
}
\seealso{
\code{\link{ellipse}}
}
\examples{
data(USAccDeaths)
fit <- arima0(USAccDeaths, order = c(0, 1, 1), seasonal = list(order = c(0, 1, 1)))
# Plot the approximate 95\% confidence region for the first two parameters
# of the model
plot(ellipse(fit), type = 'l')
points(fit$coef[1], fit$coef[2])
}
\keyword{dplot}
\keyword{ts}


\eof
\name{ellipse.glm}
\alias{ellipse.glm}
\title{
Outline an approximate pairwise confidence region 
}
\description{
This function produces the ellipsoidal outline of an approximate 
pairwise confidence region for a generalized linear model fit.
}
\usage{
\method{ellipse}{glm}(x, which = c(1, 2), level = 0.95, t, npoints = 100, 
        dispersion, \dots)
}
\arguments{
\item{x}{
The first argument should be a \code{glm} object, usually resulting from a call
to \code{glm()}.
}
\item{which}{
Which selects the pair of parameters to be plotted.  The default is the
first two.
}
\item{level}{
The confidence level of the region.  Default 95\%.
}
\item{t}{
The t statistic on the boundary of the ellipse.  For Binomial or Poisson
families, \code{sqrt(qchisq(level, 2))} is used; for other distributions,
\code{sqrt(2*qf(level, 2, df))} where \code{df} is the residual degrees of freedom.
}
\item{npoints}{
How many points to return in the ellipse.
}
\item{dispersion}{
The value of dispersion to use.  If specified, it is treated as fixed,
and the chi-square limits for \code{t} are used. If missing, it is 
taken from \code{summary(x)}.
}
\item{\dots}{
Other \code{ellipse.default} parameters may also be used.
}}
\value{
A matrix with columns named according to which to outline the confidence region.
}
\details{
The summary function is used to obtain the approximate covariance matrix of the
fitted parameters, the dispersion estimate, and the degrees of freedom.
}
\seealso{
\code{\link{ellipse.default}}
}
\examples{     
## Dobson (1990) Page 93: Randomized Controlled Trial :

     counts <- c(18,17,15,20,10,20,25,13,12)
     outcome <- gl(3,1,9)
     treatment <- gl(3,3)
     glm.D93 <- glm(counts ~ outcome + treatment, family=poisson())

# Plot an approximate 95 \% confidence region for the two Outcome parameters

	plot(ellipse(glm.D93, which = c(2,3)), type = 'l')
	points(glm.D93$coefficients[2], glm.D93$coefficients[3])
}
\keyword{dplot}
\keyword{regression}


\eof
\name{ellipse.lm}
\alias{ellipse.lm}
\title{
Outline a pairwise confidence region for a linear model fit.
}
\description{
This function produces the ellipsoidal outline of a pairwise confidence
region for a linear model fit.
}
\usage{
\method{ellipse}{lm}(x, which = c(1, 2), level = 0.95, 
    t = sqrt(2 * qf(level, 2, x$df.residual)), \dots)
}
\arguments{
\item{x}{
The first argument should be an \code{lm} object, usually resulting from a call
to \code{lm()}.
}
\item{which}{
Which selects the pair of parameters to be plotted.  The default is the
first two.
}
\item{level}{
The confidence level of the region.  Default 95\%.
}
\item{t}{
The t statistic on the boundary of the ellipse.
}
\item{\dots}{
Other \code{ellipse.default} parameters may also be used.
}}
\value{
A matrix with columns \code{x} and \code{y} to outline the confidence region.
}
\details{
The summary function is used to obtain the covariance matrix of the
fitted parameters.
}
\seealso{
\code{\link{ellipse.default}}
}
\examples{
# Plot the estimate and joint 90\% confidence region for the displacement and cylinder
# count linear coefficients in the mtcars dataset
data(mtcars)
fit <- lm(mpg ~ disp + cyl , mtcars)
plot(ellipse(fit, which = c('disp', 'cyl'), level = 0.90), type = 'l')
points(fit$coefficients['disp'], fit$coefficients['cyl'])
}
\keyword{dplot}
\keyword{regression}

\eof
\name{ellipse.nls}
\alias{ellipse.nls}
\title{
Outline an approximate pairwise confidence region 
}
\description{
This function produces the ellipsoidal outline of an approximate 
pairwise confidence region for a nonlinear model fit.
}
\usage{
\method{ellipse}{nls}(x, which = c(1, 2), level = 0.95, 
    t = sqrt(2 * qf(level, 2, s$df[2])), \dots)
}
\arguments{
\item{x}{
The first argument should be an \code{nls} object, usually resulting from a call
to \code{nls()}.
}
\item{which}{
Which selects the pair of parameters to be plotted.  The default is the
first two.
}
\item{level}{
The confidence level of the region.  Default 95\%.
}
\item{t}{
The t statistic on the boundary of the ellipse.
}
\item{\dots}{
Other \code{ellipse.default} parameters may also be used.
}}
\value{
A matrix with columns \code{x} and \code{y} to outline the confidence region.
}
\details{
The summary function is used to obtain the approximate covariance matrix of the
fitted parameters.
}
\seealso{
\code{\link{ellipse.default}}, \code{\link{ellipse.profile}}
}
\examples{
# Plot an approximate 95\% confidence region for the weight and displacement
# parameters in the Michaelis Menten model
data(Puromycin)
fit <- nls(rate ~ Vm*conc/(K + conc), data = Puromycin, subset = state=="treated", 
    start = list(K = 0.05, Vm = 200))
plot(ellipse(fit,which=c('Vm','K')), type = 'l')
params <- fit$m$getPars()
points(params['Vm'],params['K'])
}
\keyword{dplot}
\keyword{nonlinear}


\eof
\name{ellipse.profile}
\alias{ellipse.profile}
\title{
Pairwise profile sketch
}
\description{
This routine approximates a contour of a function based on the profile
of that function.
}
\usage{
\method{ellipse}{profile}(x, which = c(1, 2), level = 0.95, t = sqrt(qchisq(level, 2)), 
    npoints = 100, \dots)
}
\arguments{
\item{x}{
An object of class \code{\link{profile}}, e.g. from 
\link[MASS:confint]{profile.glm} in the MASS package.
}
\item{which}{
Which pair of parameters to use.
}
\item{level}{
The \code{ellipse.profile} function defaults assume that the profiled
function is -2 times the log likelihood of a regular model. 
With this assumption the \code{level} argument
specifies the confidence level for an asymptotic confidence region.
}
\item{t}{
The square root of the value to be contoured.
}
\item{npoints}{
How many points to use in the ellipse.
}
\item{\dots}{
Extra arguments are not used.
}
}
\value{
An \code{npoints} x \code{2} matrix with columns having the chosen parameter names, 
which approximates a contour of the function that was profiled.
}
\details{
This function uses the 4 point approximation to the contour as described
in Appendix 6 of Bates and Watts (1988).  It produces the exact contour for
quadratic surfaces, and good approximations for mild deviations from
quadratic.  If the surface is multimodal, the algorithm is likely to
produce nonsense.
}
\references{
Bates and Watts (1988) Nonlinear Regression Analysis \& its Applications
}
\seealso{
\code{\link{profile}}, \code{\link{ellipse.nls}}
}
\examples{
# Plot an approximate 95\% confidence region for the Puromycin
# parameters Vm and K, and overlay the ellipsoidal region

data(Puromycin)
Purboth <- nls(formula = rate ~ ((Vm + delV * (state == "treated"))
  * conc)/(K + conc), data = Puromycin,
  start = list(Vm = 160, delV = 40, K = 0.05))
Pur.prof <- profile(Purboth)
plot(ellipse(Pur.prof, which = c('Vm', 'K')), type = 'l')
lines(ellipse(Purboth, which = c('Vm', 'K')), lty = 2)
params <- Purboth$m$getPars()
points(params['Vm'],params['K'])
}
\keyword{dplot}
\keyword{models}

\eof
\name{ellipse.profile.glm}
\alias{ellipse.profile.glm}
\title{
Pairwise profile sketch	for GLM profiles
}
\description{
This routine approximates a pairwise confidence region for a glm
model.
}
\usage{
\method{ellipse}{profile.glm}(x, which = c(1, 2), level = 0.95, t, 
    npoints = 100, dispersion, \dots)
}
\arguments{
\item{x}{
An object of class \code{\link{profile.glm}}.
}
\item{which}{
Which pair of parameters to use.
}
\item{level}{
The \code{level} argument
specifies the confidence level for an asymptotic confidence region.
}
\item{t}{
The square root of the value to be contoured.  By default, this is \code{qchisq(level, 2)}
for models with fixed dispersion (i.e. binomial and Poisson), and \code{2 * qf(level, 2, df)} for other models,
where \code{df} is the residual degrees of freedom.
}
\item{npoints}{
How many points to use in the ellipse.
}
\item{dispersion}{
If specified, fixed dispersion is assumed, otherwise the dispersion is taken from the model.
}
\item{\dots}{
Extra parameters which are not used (for compatibility with the generic).
}
}
\value{
An \code{npoints} x \code{2} matrix with columns having the chosen parameter names, 
which approximates a contour of the function that was profiled.
}
\details{
This function uses the 4 point approximation to the contour as described
in Appendix 6 of Bates and Watts (1988).  It produces the exact contour for
quadratic surfaces, and good approximations for mild deviations from
quadratic.  If the surface is multimodal, the algorithm is likely to
produce nonsense.
}
\references{
Bates and Watts (1988) Nonlinear Regression Analysis \& its Applications
}
\seealso{
\code{\link{profile}}, \code{\link{glm}}, \code{\link{ellipse.glm}}
}
\examples{
## MASS has a pairs.profile function that conflicts with ours, so
## do a little trickery here
	 noMASS <- is.na(match('package:MASS', search()))
	 if (noMASS) require(MASS)

## Dobson (1990) Page 93: Randomized Controlled Trial :

     counts <- c(18,17,15,20,10,20,25,13,12)
     outcome <- gl(3,1,9)
     treatment <- gl(3,3)
     glm.D93 <- glm(counts ~ outcome + treatment, family=poisson())

##  Plot an approximate 95\% confidence region for the two outcome variables
     prof.D93 <- profile(glm.D93)
     plot(ellipse(prof.D93, which = 2:3), type = 'l')
     lines(ellipse(glm.D93, which = 2:3), lty = 2)
     params <- glm.D93$coefficients
     points(params[2],params[3])
     
##  Clean up our trickery
	 if (noMASS) detach('package:MASS')
	 
}
\keyword{dplot}
\keyword{models}

\eof
\name{ellipse.profile.nls}
\alias{ellipse.profile.nls}
\title{
Pairwise profile sketch
}
\description{
This routine approximates a pairwise confidence region for a nonlinear regression
model.
}
\usage{
\method{ellipse}{profile.nls}(x, which = c(1, 2), level = 0.95, t = sqrt(2 * qf(level, 2, attr(x, "summary")$df[2])), 
    npoints = 100, \dots)
}
\arguments{
\item{x}{
An object of class \code{\link{profile.nls}}.
}
\item{which}{
Which pair of parameters to use.
}
\item{level}{
The \code{level} argument
specifies the confidence level for an asymptotic confidence region.
}
\item{t}{
The square root of the value to be contoured.
}
\item{npoints}{
How many points to use in the ellipse.
}
\item{\dots}{
Extra parameters which are not used (for compatibility with the generic).
}
}
\value{
An \code{npoints} x \code{2} matrix with columns having the chosen parameter names, 
which approximates a contour of the function that was profiled.
}
\details{
This function uses the 4 point approximation to the contour as described
in Appendix 6 of Bates and Watts (1988).  It produces the exact contour for
quadratic surfaces, and good approximations for mild deviations from
quadratic.  If the surface is multimodal, the algorithm is likely to
produce nonsense.
}
\references{
Bates and Watts (1988) Nonlinear Regression Analysis \& its Applications
}
\seealso{
\code{\link{profile}}, \code{\link{ellipse.nls}}
}
\examples{
# Plot an approximate 95\% confidence region for the Puromycin
# parameters Vm and K, and overlay the ellipsoidal region
data(Puromycin)
Purboth <- nls(formula = rate ~ ((Vm + delV * (state == "treated"))
  * conc)/(K + conc), data = Puromycin,
  start = list(Vm = 160, delV = 40, K = 0.05))
Pur.prof <- profile(Purboth)
plot(ellipse(Pur.prof, which = c('Vm', 'K')), type = 'l')
lines(ellipse(Purboth, which = c('Vm', 'K')), lty = 2)
params <- Purboth$m$getPars()
points(params['Vm'],params['K'])
}
\keyword{dplot}
\keyword{models}

\eof
\name{pairs.profile}
\alias{pairs.profile}
\title{
Profile pairs
}
\description{
This function produces pairwise plots of profile traces, profile
sketches, and ellipse approximations to confidence intervals.
}
\usage{
\method{pairs}{profile}(x, labels = c(names(x), "Profile tau"), panel = lines, invert = TRUE, 
    plot.tau = TRUE, plot.trace = TRUE, plot.sketch = TRUE, 
	plot.ellipse = FALSE, level = 0.95, \dots)
}
\arguments{
\item{x}{
An object of class \code{profile}, generally the result of the \code{profile()}
function.
}
\item{labels}{
The labels to use for each variable.  These default to the variable
names.
}
\item{panel}{
The function to use to draw the sketch in each panel.
}
\item{invert}{
Whether to swap the axes so things look better.
}
\item{plot.tau}{
Whether to do the profile tau (profile t) plots.
}
\item{plot.trace}{
Whether to do the profile trace plots.
}
\item{plot.sketch}{
Whether to do the profile sketch plots.
}
\item{plot.ellipse}{
Whether to do the ellipse approximations.
}
\item{level}{
The nominal confidence level for the profile sketches and ellipses.
}
\item{\dots}{
Other plotting parameters.
}}
\section{Side Effects}{
Produces a plot on the current device for each pair of variables in the
profile object.
}
\details{
This function implements the plots used in Bates and Watts (1988) for
nonlinear regression diagnostics.
}
\references{
Bates and Watts (1988), Nonlinear Regression Analysis \& its
Applications.
}
\seealso{
\code{\link{profile}}, \code{\link{ellipse.profile}}, \code{\link{ellipse.nls}}
}
\examples{
 # Plot everything for the Puromycin data
 data(Puromycin)
 Purboth <- nls(formula = rate ~ ((Vm + delV * (state == "treated"))
   * conc)/(K + conc), data = Puromycin,
   start = list(Vm = 160, delV = 40, K = 0.05))
 Pur.prof <- profile(Purboth)
 pairs(Pur.prof, plot.ellipse = TRUE)
}
\keyword{dplot}
\keyword{regression}
\keyword{nonlinear}

\eof
\name{plotcorr}
\alias{plotcorr}
\title{
Plot correlation matrix ellipses
}
\description{
This function plots a correlation matrix using ellipse-shaped glyphs for
each entry.  The ellipse represents a level curve of the density of a
bivariate normal with the matching correlation.
}
\synopsis{
plotcorr(corr, outline = TRUE, col = 'grey', numbers = FALSE, bty = "n", axes = FALSE,
            xlab = "", ylab = "", asp = 1, cex.lab = par("cex.lab"), cex = 0.75*par("cex"),
			mar = 0.1 + c(2,2,4,2), ...) 
}
\usage{
plotcorr(corr, outline = TRUE, col = "grey", numbers = FALSE, \dots)
}
\arguments{
\item{corr}{
A matrix containing entries between \code{-1} and \code{1} to be plotted as
correlations.  
}
\item{outline}{
Whether the ellipses should be outlined in the default colour.
}
\item{col}{
Which colour to use to fill the ellipses.  
}
\item{numbers}{
Whether to plot numerical correlations in place of ellipses.  If
numbers is \code{TRUE}, then the correlations will be rounded to a single decimal
place and placed on the plot.
}
\item{\dots}{
Additional graphical parameters.
}}
\details{
The ellipses being plotted will be tangent to a unit character square,
with the shape chosen to match the required correlation.
}
\references{
Murdoch, D.J. and Chow, E.D. (1996). A graphical display of large 
correlation matrices. The American Statistician 50, 178-180. 
}
\seealso{
\code{\link{ellipse}}
}
\examples{
# Plot the correlation matrix for the mtcars data full model fit 
data(mtcars)
fit <- lm(mpg ~ ., mtcars)
plotcorr(summary(fit, correlation = TRUE)$correlation)
# Plot a second figure with numbers in place of the
# ellipses
plotcorr(summary(fit, correlation = TRUE)$correlation, numbers = TRUE)
}
\keyword{hplot}

\eof
