\name{BJnoint}
\alias{BJnoint}
\title{The Buckley-James censored regression estimator without intercept term}
\usage{
BJnoint(x, y, delta, beta0 = NA, maxiter=30, error = 0.00001)
}
\arguments{
    \item{x}{a matrix or vector containing the covariate, one row
 	 per observation.}
    \item{y}{a numeric vector of length N, censored responses. }
    \item{delta}{a vector of length N, delta=0/1 for censored/uncensored.}
    \item{beta0}{an optional vector for starting value of iteration.}
    \item{maxiter}{an optional integer to control iterations.}
    \item{error}{an optional positive value to control interations.}
}
\description{
    Compute the Buckley-James estimator in the regression
model 
\deqn{ y_i = beta x_i + epsilon_i } 
with right censored y.

}
\details{
If your model do have an intercept term, then you should use the function
\code{bj( )} in the Design library.

This function is included here mainly to produce the estimator value
that may provide some useful info with function \code{bjtest( )}.
For example you may want to test the beta value close to the
Buckley-James estimator. 

}
\value{
    A list with the following components:
    \item{beta}{the Buckley-James estimator.}
    \item{iteration}{number of iterations performed.}
}
\references{
    Buckley, J. and James, I. (1979).
    Biometrika 

    Owen, A. (1990). Empirical likelihood ratio confidence regions. 
    Ann. Statist. \bold{18} 90-120.
}
\author{ Mai Zhou. }
\examples{
x <- matrix(c(rnorm(50,mean=1), rnorm(50,mean=2)), ncol=2,nrow=50)
## Suppose now we wish to test Ho: 2mu(1)-mu(2)=0, then
y <- 2*x[,1]-x[,2]
xx <- c(28,-44,29,30,26,27,22,23,33,16,24,29,24,40,21,31,34,-2,25,19)
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{bjtest}
\alias{bjtest}
\title{Test the Buckley-James estimator by Empirical Likelihood}
\usage{
bjtest(y, d, x, beta)
}
\arguments{
    \item{y}{a vector of length N, containing the censored responses.}
    \item{d}{a vector (length N) of either 1's or 0's. 
              d=1 means y is uncensored;
	      d=0 means y is right censored. }
    \item{x}{a matrix of size N by q. }
    \item{beta}{a vector of length q. the value of the regression coefficient to 
          be tested in the model y = x beta + epsilon }
}
\description{
    Use the empirical likelihood ratio and Wilks theorem to test if the
    regression coefficient is equal to beta.

The log empirical likelihood been maximized is
\deqn{ \sum_{d=1} \log \Delta F(e_i) + \sum_{d=0} \log [1-F(e_i)];}
where \eqn{e_i} are the residuals.
}
\details{

The above likelihood should be understood as the likelihood of the 
error term, so in the regression model the error epsilon should be iid.

This version can handle the model where beta is a vector (of length q).

The estimation equation used when maximize the 
empirical likelihood is 
\deqn{ \sum d_i \Delta F(e_i) (x \cdot m[,i])/(n w_i) }
which was descrided in detail in the reference below.
}
\value{
    A list with the following components:
    \item{"-2LLR"}{the -2 loglikelihood ratio; have approximate chisq 
                  distribution under \eqn{H_o}.}
    \item{logel2}{the log empirical likelihood, under estimating equation.}
    \item{logel}{the log empirical likelihood of the Kaplan-Meier of e's.}
    \item{prob}{the probabilities that max the empirical likelihood 
               under estimating equation.}
}
\references{
    Buckley, J. and James, I. (1979). Linear regression with censored data.
      Biometrika,  \bold{66} 429-36.

    Zhou, M. and Li, G. (2004). Empirical likelihood analysis of the
       Buckley-James estimator. Tech. Report. 
}
\author{ Mai Zhou. }
\examples{
xx <- c(28,-44,29,30,26,27,22,23,33,16,24,29,24,40,21,31,34,-2,25,19)
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{bjtest1d}
\alias{bjtest1d}
\title{Test the Buckley-James estimator by Empirical Likelihood, 1-dim only}
\usage{
bjtest1d(y, d, x, beta)
}
\arguments{
    \item{y}{a vector of length N, containing the censored responses.}
    \item{d}{a vector of either 1's or 0's. d=1 means y is uncensored.
	d=0 means y is right censored. }
    \item{x}{a vector of length N, covariate.}
    \item{beta}{a number. the regression coefficient to 
          be tested in the model y = x beta + epsilon }
}
\description{
    Use the empirical likelihood ratio and Wilks theorem to test if the
    regression coefficient is equal to beta.

The log empirical likelihood been maximized is
\deqn{ \sum_{d=1} \log \Delta F(e_i) + \sum_{d=0} \log [1-F(e_i)] .}
}
\details{

In the above likelihood, \eqn{ e_i = y_i - x * beta } is the residuals.
 
}
\value{
    A list with the following components:
    \item{"-2LLR"}{the -2 loglikelihood ratio; have approximate chi square 
                  distribution under \eqn{H_o}.}
    \item{logel2}{the log empirical likelihood, under estimating equation.}
    \item{logel}{the log empirical likelihood of the Kaplan-Meier of e's.}
    \item{prob}{the probabilities that max the empirical likelihood 
               under estimating equation constraint.}
}
\references{
  Buckley, J. and James, I. (1979). Linear regression with censored data.
     Biometrika,  \bold{66} 429-36.

  Owen, A. (1990). Empirical likelihood ratio confidence regions. 
    Ann. Statist. \bold{18} 90-120.
}
\author{Mai Zhou.}
\examples{
xx <- c(28,-44,29,30,26,27,22,23,33,16,24,29,24,40,21,31,34,-2,25,19)
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{el.cen.EM}
\alias{el.cen.EM}
\title{Empirical likelihood ratio for mean 
with right, left or doubly censored data, by EM algorithm}
\usage{
el.cen.EM(x,d,fun=function(t){t},mu,maxit=25,error=1e-9,...)
}
\description{
This program uses EM algorithm to compute the maximized 
(wrt \eqn{p_i}) empirical
log likelihood function for right, left or doubly censored data with 
the MEAN constraint:
\deqn{ \sum_{d_i=1}  p_i f(x_i)  = \int f(t) dF(t) = \mu ~. }
Where \eqn{p_i = \Delta F(x_i)} is a probability,
\eqn{d_i} is the censoring indicator, 1(uncensored), 0(right censored),
2(left censored). 
It also returns those \eqn{p_i}. 

The empirical log likelihood been maximized is
\deqn{ \sum_{d_i=1} \log \Delta F(x_i) + \sum_{d_i=0} \log [1-F(x_i)] 
    + \sum_{d_i=2}  \log F(x_i) . } 
}
\arguments{
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector containing the censoring indicators, 
           1-uncensored; 0-right censored; 2-left censored.}
    \item{fun}{a continuous (weight) function used to calculate
         the mean as in \eqn{H_0}.
         \code{fun(t)} must be able to take a vector input \code{t}.
         Default to the identity function \eqn{f(t)=t}.} 
    \item{mu}{a real number used in the constraint, mean value of \eqn{f(X)}.}
    \item{maxit}{an optional integer, used to control maximum number of
             iterations. }
    \item{error}{an optional positive real number specifying the tolerance of
       iteration error. This is the bound of the
       \eqn{L_1} norm of the difference of two successive weights.}
   \item{...}{additional arguments, if any, to pass to fun.}
}
\value{
    A list with the following components:
    \item{loglik}{the maximized empirical log likelihood under the constraint.}
    \item{times}{locations of CDF that have positive mass.}
    \item{prob}{the jump size of CDF at those locations.}
    \item{"-2LLR"}{If available, it is Minus two times the 
                   Empirical Log Likelihood Ratio.
                   Should be approx. chi-square distributed under Ho.}
    \item{Pval}{The P-value of the test, using chi-square approximation.}
}
\details{

This implementation is all in R and have several for-loops in it. 
A faster version would use C to do the for-loop part.
But this version seems faster enough and is easier to port to Splus.

We return the log likelihood all the time. Sometimes, (for right censored
and no censor case) we also return the -2 log likelihood ratio.
In other cases, you have to plot a curve with many values of the 
parameter, mu, to
find out where is the place the log likelihood becomes maximum.
And from there you can get -2 log likelihood ratio between
the maximum location and your current parameter in Ho.

In order to get a proper distribution as NPMLE, we automatically
change the \eqn{d} for the largest observation to 1
(even if it is right censored), similar for the left censored, 
smallest observation.
\eqn{\mu} is a given constant. 
When the given constants \eqn{\mu} is too far
away from the NPMLE, there will be no distribution
satisfy the constraint.
In this case the computation will stop.
The -2 Log empirical likelihood ratio
should be infinite. 

The constant \code{mu} must be inside 
\eqn{( \min f(x_i) , \max f(x_i) ) }
for the computation to continue. 
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{mu} closer
to the NPMLE --- 
\deqn{ \sum_{d_i=1} p_i^0 f(x_i) } 
\eqn{p_i^0} taken to be the jumps of the NPMLE of CDF. 
Or use a different \code{fun}. 

}
\author{ Mai Zhou }
\references{
    Zhou, M. (2002). 
        Computing censored empirical likelihood ratio 
        by EM algorithm. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}

    Murphy, S. and van der Varrt (1997)
         Semiparametric likelihood ratio inference.
         \emph{Ann. Statist.} \bold{ 25}, 1471-1509.
}
\examples{
## example with tied observations
x <- c(1, 1.5, 2, 3, 4, 5, 6, 5, 4, 1, 2, 4.5)
d <- c(1,   1, 0, 1, 0, 1, 1, 1, 1, 0, 0,   1)
el.cen.EM(x,d,mu=3.5)
## we should get "-2LLR" = 1.2466....
myfun5 <- function(x, theta, eps) {
u <- (x-theta)*sqrt(5)/eps 
INDE <- (u < sqrt(5)) & (u > -sqrt(5)) 
u[u >= sqrt(5)] <- 0 
u[u <= -sqrt(5)] <- 1 
y <- 0.5 - (u - (u)^3/15)*3/(4*sqrt(5)) 
u[ INDE ] <- y[ INDE ] 
return(u)
}
el.cen.EM(x, d, fun=myfun5, mu=0.5, theta=3.5, eps=0.1)
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}


\eof
\name{el.cen.EM2}
\alias{el.cen.EM2}
\title{Empirical likelihood ratio test for a vector of means 
with right, left or doubly censored data, by EM algorithm}
\usage{
el.cen.EM2(x,d,xc=1:length(x),fun,mu,maxit=25,error=1e-9,...)
}
\description{
This function is similar to \code{el.cen.EM()}, but for multiple constraints.
In the input there is a vector of observations 
\eqn{x = (x_1, \cdots , x_n)} and a 
function \code{fun}. The function \code{fun} should return the (nxk) matrix
\deqn{
          ( f_1(x), f_2(x), \cdots, f_k (x) ) . 
}

Also, the ordering of the observations, when consider censoring or 
redistributing-to-the-right, 
is according to the value of \code{x}, not \code{fun(x)}. 
So the probability distribution is for values \code{x}.
This program uses EM algorithm to maximize 
(wrt \eqn{p_i}) empirical
log likelihood function for right, left or doubly censored data with 
the MEAN constraint:
\deqn{ j = 1,2, \cdots ,k ~~~~ 
   \sum_{d_i=1} p_i f_j(x_i) = \int f_j(t) dF(t) = \mu_j ~. }
Where \eqn{p_i = \Delta F(x_i)} is a probability,
\eqn{d_i} is the censoring indicator, 1(uncensored), 0(right censored),
2(left censored). 
It also returns those \eqn{p_i}. 
The log likelihood function is defined as
\deqn{ \sum_{d_i=1} \log \Delta F(x_i)  + \sum_{d_i=2} \log F(x_i) 
     + \sum_{d_i=0} \log [ 1-F(x_i)] ~.}
}
\arguments{
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector containing the censoring indicators, 
           1-uncensored; 0-right censored; 2-left censored.}
    \item{xc}{an optional vector of collapsing control values. 
              If xc[i] xc[j] has different values then two 
              (x,d) will not merge into one observation even 
              if they are identical. Default is to merge.}
    \item{fun}{a continuous (weight) function that returns a matrix. 
         The columns (=k) of the matrix is used to calculate
         the means as in \eqn{H_0}.
         \code{fun(t)} must be able to take a vector input \code{t}.} 
    \item{mu}{a vector of length k. Used in the constraint, 
                    as the mean of \eqn{f(X)}.}
    \item{maxit}{an optional integer, used to control maximum number of
             iterations. }
    \item{error}{an optional positive real number specifying the tolerance of
       iteration error. This is the bound of the
       \eqn{L_1} norm of the difference of two successive weights.}
    \item{...}{additional inputs to pass to \code{fun()}.}
}
\value{
    A list with the following components:
    \item{loglik}{the maximized empirical log likelihood under the constraints.}
    \item{times}{locations of CDF that have positive mass.}
    \item{prob}{the jump size of CDF at those locations.}
    \item{"-2LLR"}{If available, it is Minus two times the 
                   Empirical Log Likelihood Ratio.
                   Should be approx. chi-square distributed under Ho.}
    \item{Pval}{If available, the P-value of the test, 
               using chi-square approximation.}
}
\details{

This implementation is all in R and have several for-loops in it. 
A faster version would use C to do the for-loop part.
(but this version is easier to port to Splus, and seems faster enough). 

We return the log likelihood all the time. Sometimes, (for right censored
and no censor case) we also return the -2 log likelihood ratio.
In other cases, you have to plot a curve with many values of the 
parameter, mu, to
find out where the log likelihood becomes maximum.
And from there you can get -2 log likelihood ratio between
the maximum location and your current parameter in Ho.

In order to get a proper distribution as NPMLE, we automatically
change the \eqn{d} for the largest observation to 1
(even if it is right censored), similar for the left censored, 
smallest observation.
\eqn{\mu} is a given constant vector. 
When the given constants \eqn{\mu} is too far
away from the NPMLE, there will be no distribution
satisfy the constraint.
In this case the computation will stop.
The -2 Log empirical likelihood ratio
should be infinite. 

The constant vector \code{mu} must be inside 
\eqn{( \min f(x_i) , \max f(x_i) ) }
for the computation to continue. 
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{mu} closer
to the NPMLE --- 
\deqn{ \hat \mu _j = \sum_{d_i=1} p_i^0 f_j(x_i) } 
where \eqn{p_i^0} taken to be the jumps of the NPMLE of CDF. 
Or use a different \code{fun}. 

}
\author{ Mai Zhou }
\references{
    Zhou, M. (2002). 
        Computing censored empirical likelihood ratio 
        by EM algorithm. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}

}
\examples{
## censored regression with one right censored observation.
## we check the estimation equation, with the MLE inside myfun7. 
y <- c(3, 5.3, 6.4, 9.1, 14.1, 15.4, 18.1, 15.3, 14, 5.8, 7.3, 14.4)
x <- c(1, 1.5, 2,   3,   4,    5,    6,    5,    4,  1,   2,   4.5)
d <- c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0)
myfun7 <- function(y, xmat) {
temp1 <- y - ( 1.392885 +  2.845658 * xmat)
return( cbind( temp1, xmat*temp1) )
}
el.cen.EM2(y,d, xc=1:12, fun=myfun7, mu=c(0,0), xmat=x)
## we should get, Pval = 1 , as the MLE should.
## for different values of (a, b) inside myfun7, you get other Pval
rqfun1 <- function(y, xmat, beta, tau = 0.5) {
temp1 <- tau - (1-myfun55(y-beta*xmat))
return(xmat * temp1)
}
myfun55 <- function(x, eps=0.001){
u <- x*sqrt(5)/eps
INDE <- (u < sqrt(5)) & (u > -sqrt(5))
u[u >= sqrt(5)] <- 0
u[u <= -sqrt(5)] <- 1
y <- 0.5 - (u - (u)^3/15)*3/(4*sqrt(5))
u[ INDE ] <- y[ INDE ]
return(u)
}
el.cen.EM2(x=y,d=d,xc=1:12,fun=rqfun1,mu=0,xmat=x,beta=3.08,tau=0.44769875)
## default tau=0.5 
el.cen.EM2(x=y,d=d,xc=1:12,fun=rqfun1,mu=0,xmat=x,beta=3.0799107404)
}
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}


\eof
\name{el.cen.test}
\alias{el.cen.test}
\title{Empirical likelihood ratio for mean
with right censored data, by QP.}
\usage{
el.cen.test(x,d,fun=function(x){x},mu,error=1e-8,maxit=15)
}
\description{
This program computes the maximized (wrt \eqn{p_i}) empirical
log likelihood function for right censored data with 
the MEAN constraint:
\deqn{ \sum_i [ d_i p_i g(x_i) ] = \int g(t) dF(t) = \mu  } 
where \eqn{p_i = \Delta F(x_i)} is a probability,
\eqn{d_i} is the censoring indicator. 
The \eqn{d} for the largest observation is always taken to be 1.
It then computes the -2 log 
empirical likelihood ratio which should be approximately chi-square
distributed if the constraint is true.
Here \eqn{F(t)} is the (unknown) CDF; 
\eqn{g(t)} can be any given left
continuous function in \eqn{t}.
\eqn{\mu} is a given constant. 
The data must contain some right censored observations.
If there is no censoring or the only censoring is the largest 
observation, the code will stop and we should use 
\code{el.test} that is for uncensored data.  

The log likelihood been maximized is
\deqn{ \sum_{d_i=1} \log \Delta F(x_i) + \sum_{d_i=0} \log [ 1-F(x_i) ].}
}
\arguments{
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector containing the censoring indicators, 
           1-uncensor; 0-censor.}
    \item{fun}{a left continuous (weight) function used to calculate
         the mean as in \eqn{H_0}. 
         \code{fun(t)} must be able to take a vector input \code{t}. 
      Default to the identity function \eqn{f(t)=t}.}
    \item{mu}{a real number used in the constraint, sum to this value.}
    \item{error}{an optional positive real number specifying the tolerance of
       iteration error in the QP. This is the bound of the
       \eqn{L_1} norm of the difference of two successive weights.}
    \item{maxit}{an optional integer, used to control maximum number of
             iterations. }
}
\value{
    A list with the following components:
    \item{"-2LLR"}{The -2Log Likelihood ratio.}
    \item{xtimes}{the location of the hazard jumps.}
    \item{weights}{the jump size of CDF function at those locations.}
    \item{Pval}{P-value}
    \item{error}{the \eqn{L_1} norm between the last two \code{wts}.}
    \item{iteration}{number of iterations carried out}
}
\details{
When the given constants \eqn{\mu} is too far
away from the NPMLE, there will be no distribution
satisfy the constraint.
In this case the computation will stop.
The -2 Log empirical likelihood ratio
should be infinite. 

The constant \code{mu} must be inside 
\eqn{( \min f(x_i) , \max f(x_i) ) }
for the computation to continue. 
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{mu} closer
to the NPMLE, or use a different \code{fun}. 

This function uses sequential Quadratic Programming to find the
maximum. Unlike other functions in this package,
it can be slow for larger sample sizes.
It took about one minute for a sample of size 2000 with 20\% censoring
on a 1GHz, 256MB PC, about 19 seconds on a 3 GHz 512MB PC.
}
\author{ Mai Zhou, Kun Chen }
\references{
    Pan, X. and Zhou, M. (1999). Empirical likelihood ratio, one parameter
                         sub-family of distributions and censored data.
    \emph{J. Statist. Plann. Inference}. \bold{75}, 379-392.

    Chen, K. and Zhou, M. (2000). 
        Computing censored empirical likelihood ratio 
        using Quadratic Programming. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}
}
\examples{
el.cen.test(rexp(100), c(rep(0,25),rep(1,75)), mu=1.5)
## second example with tied observations
x <- c(1, 1.5, 2, 3, 4, 5, 6, 5, 4, 1, 2, 4.5)
d <- c(1,   1, 0, 1, 0, 1, 1, 1, 1, 0, 0,   1)
el.cen.test(x,d,mu=3.5)
# we should get  "-2LLR" = 1.246634  etc. 
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}


{  # This program compute the maximized (wrt w_i) loglikelihood 
   # function for right censored data with the MEAN constraint: 
   #                   sum[d_i * w_i * fun(x_i)]= mu        (star)
   # where w_i is the probability.
   # Inputs
   # x  : life time data vector, size n
   # d  : censoring indicator. length n vector. d=1 uncensor; d=0 censor.
   # fun : function to compute the mean, default is f(t)=t. 
   # mu : constrained mean value, a scalar. 
   # error : iteration stopping control standard. default=1e-8.
   # maxit : maximum number of iteration
   #
   # Output
   # -2LLR   : -2log maximized loglikelihood ratio (chisq under Ho)
   # weight  : weight under mean constraint
   # m       : the number of iterations
   #************************************************************
   # depends on Wdataclean2(), WKM() and solve3.QP() 
   ####################################################################
} 

\eof
\name{el.ltrc.EM}
\alias{el.ltrc.EM}
\title{Empirical likelihood ratio for mean 
with left truncated and right censored data, by EM algorithm}
\usage{
el.ltrc.EM(y,x,d,fun=function(t){t},mu,maxit=30,error=1e-9)
}
\description{
This program uses EM algorithm to compute the maximized 
(wrt \eqn{p_i}) empirical
log likelihood function for left truncated and right censored data with 
the MEAN constraint:
\deqn{ \sum_{d_i=1}  p_i f(x_i)  = \int f(t) dF(t) = \mu ~. }
Where \eqn{p_i = \Delta F(x_i)} is a probability,
\eqn{d_i} is the censoring indicator, 1(uncensored), 0(right censored). 
The \eqn{d} for the largest observation \eqn{x}, is always (automatically)
changed to 1.  \eqn{\mu} is a given constant. 
This function also returns those \eqn{p_i}. 

The log empirical likelihood function been maximized is
\deqn{\sum_{d_i=1} \log \frac{ \Delta F(x_i)}{1-F(y_i)}  + 
    \sum_{d_i=0} \log \frac{1-F(x_i)}{1-F(y_i)}.}
}
\arguments{
    \item{y}{an optional vector containing the observed truncation times.}
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector containing the censoring indicators, 
           1-uncensored; 0-right censored.}
    \item{fun}{a continuous (weight) function used to calculate
         the mean as in \eqn{H_0}.
         \code{fun(t)} must be able to take a vector input \code{t}.
         Default to the identity function \eqn{f(t)=t}.}
    \item{mu}{a real number used in the constraint, mean value of \eqn{f(X)}.}
    \item{error}{an optional positive real number specifying the tolerance of
       iteration error. This is the bound of the
       \eqn{L_1} norm of the difference of two successive weights.}
    \item{maxit}{an optional integer, used to control maximum number of
             iterations. }
}
\value{
    A list with the following components:
    \item{times}{locations of CDF that have positive mass.}
    \item{prob}{the probability of the constrained NPMLE of 
                CDF at those locations.}
    \item{"-2LLR"}{It is Minus two times the 
                   Empirical Log Likelihood Ratio.
                   Should be approximate chi-square distributed under Ho.}
}
\details{

We return the -2 log likelihood ratio, and the constrained
NPMLE of CDF.
The un-constrained NPMLE should be WJT or Lynden-Bell estimator.

When the given constants \eqn{\mu} is too far
away from the NPMLE, there will be no distribution
satisfy the constraint.
In this case the computation will stop.
The -2 Log empirical likelihood ratio
should be infinite. 

The constant \code{mu} must be inside 
\eqn{( \min f(x_i) , \max f(x_i) ) }
for the computation to continue. 
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{mu} closer
to the NPMLE --- 
\deqn{ \sum_{d_i=1} p_i^0 f(x_i) } 
\eqn{p_i^0} taken to be the jumps of the NPMLE of CDF. 
Or use a different \code{fun}. 

This implementation is all in R and have several for-loops in it. 
A faster version would use C to do the for-loop part.
(but this version is easier to port to Splus, and seems faster enough).
}
\author{ Mai Zhou }
\references{
    Zhou, M. (2002). 
        Computing censored and truncated empirical likelihood ratio 
        by EM algorithm. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}

    Turnbbull, B. (1976). The empirical distribution function with
   arbitrarily grouped, censored and truncated data. JRSS B 290-295.
}
\examples{
## example with tied observations
y <- c(0, 0, 0.5, 0, 1, 2, 2, 0, 0, 0, 0, 0 )
x <- c(1, 1.5, 2, 3, 4, 5, 6, 5, 4, 1, 2, 4.5)
d <- c(1,   1, 0, 1, 0, 1, 1, 1, 1, 0, 0,   1)
el.ltrc.EM(y,x,d,mu=3.5)
ypsy <- c(51, 58, 55, 28, 25, 48, 47, 25, 31, 30, 33, 43, 45, 35, 36)
xpsy <- c(52, 59, 57, 50, 57, 59, 61, 61, 62, 67, 68, 69, 69, 65, 76)
dpsy <- c(1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1 )
el.ltrc.EM(ypsy,xpsy,dpsy,mu=64)
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}


\eof
\name{el.test}
\alias{el.test}
\title{Empirical likelihood ratio test for the means, uncensored data}
\usage{
el.test(x, mu, lam, maxit=25, gradtol=1e-7, 
                 svdtol = 1e-9, itertrace=FALSE)
}
\arguments{
    \item{x}{a matrix or vector containing the data, one row
 	 per observation.}
    \item{mu}{a numeric vector (of length \code{ = ncol(x)})
           to be tested as the mean vector of \code{x}
		above, as \eqn{H_0}.}
    \item{lam}{an optional vector of length \code{ = length(mu)},
            the starting value of Lagrange
	 multipliers, will use \eqn{0} if missing.}
    \item{maxit}{an optional integer to control iteration when solve
	 constrained maximization.}
    \item{gradtol}{an optional real value for convergence test.}
    \item{svdtol}{an optional real value to detect singularity while
		solve equations.}
    \item{itertrace}{a logical value. If the iteration history 
	needs to be printed out.}
}
\description{
    Compute the empirical likelihood ratio with the
	mean vector fixed at mu.

The log empirical likelihood been maximized is
\deqn{ \sum_{i=1}^n \log \Delta F(x_i).}
}
\details{
If \code{mu} is in the interior of the convex hull of the
observations \code{x}, then \code{wts} should sum to \code{n}. 
If \code{mu} is outside
the convex hull then \code{wts} should sum to nearly zero, and 
\code{-2LLR}
will be a large positive number.  It should be infinity,
but for inferential purposes a very large number is
essentially equivalent.  If mu is on the boundary of the convex
hull then \code{wts} should sum to nearly k where k is the number of
observations within that face of the convex hull which contains mu.

When \code{mu} is interior to the convex hull, it is typical for
the algorithm to converge quadratically to the solution, perhaps
after a few iterations of searching to get near the solution.
When \code{mu} is outside or near the boundary of the convex hull, then
the solution involves a \code{lambda} of infinite norm.  The algorithm
tends to nearly double \code{lambda} at each iteration and the gradient
size then decreases roughly by half at each iteration.

The goal in writing the algorithm was to have it ``fail gracefully"
when \code{mu} is not inside the convex hull.  The user can 
either leave \code{-2LLR} ``large and positive" or can replace
it by infinity when the weights do not sum to nearly n.
}
\value{
    A list with the following components:
    \item{-2LLR}{the -2 loglikelihood ratio; approximate chisq distribution
                 under \eqn{H_o}.}
    \item{Pval}{the observed P-value by chi-square approximation.}
    \item{lambda}{the final value of Lagrange multiplier.}
    \item{grad}{the gradient at the maximum.}
    \item{hess}{the Hessian matrix.}
    \item{wts}{weights on the observations}
    \item{nits}{number of iteration performed}
}
\references{
    Owen, A. (1990). Empirical likelihood ratio confidence regions. 
    Ann. Statist. \bold{18} 90-120.
}
\author{ Original Splus code by Art Owen. Adapted to R by Mai Zhou. }
\examples{
x <- matrix(c(rnorm(50,mean=1), rnorm(50,mean=2)), ncol=2,nrow=50)
el.test(x, mu=c(1,2))
## Suppose now we wish to test Ho: 2mu(1)-mu(2)=0, then
y <- 2*x[,1]-x[,2]
el.test(y, mu=0)
xx <- c(28,-44,29,30,26,27,22,23,33,16,24,29,24,40,21,31,34,-2,25,19)
el.test(xx, mu=15)  #### -2LLR = 1.805702
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{el.test.wt}
\alias{el.test.wt}
\title{Weighted Empirical Likelihood ratio for mean, uncensored data}
\usage{
el.test.wt(x, wt, mu)
}
\description{
This program is similar to el.test except
it takes weights, and is for one dim mu.

The mean constraint:
\deqn{ \sum_{i=1}^n  p_i x_i  = \mu . }
where \eqn{p_i = \Delta F(x_i)} is a probability.
Plus the probability constraint: \eqn{ \sum p_i =1}.

The weighted log empirical likelihood been maximized is
\deqn{ \sum_{i=1}^n w_i \log p_i. } 
}
\arguments{
    \item{x}{a vector containing the observations.}
    \item{wt}{a vector containing the weights.}
    \item{mu}{a real number used in the constraint, weighted
                mean value of \eqn{f(X)}.}
}
\value{
    A list with the following components:
    \item{x}{the observations.}
    \item{wt}{the vector of weights.}
    \item{prob}{The probabilities that maximized the weighted 
                  empirical likelihood under mean constraint.}
}
\details{
This function used to be an internal function. 
It becomes external because others may find it useful.

The constant \code{mu} must be inside 
\eqn{( \min x_i , \max x_i ) }
for the computation to continue. 

}
\author{ Mai Zhou }
\references{
    Zhou, M. (2002). 
        Computing censored empirical likelihood ratio 
        by EM algorithm. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}
}
\examples{
## example with tied observations
x <- c(1, 1.5, 2, 3, 4, 5, 6, 5, 4, 1, 2, 4.5)
d <- c(1,   1, 0, 1, 0, 1, 1, 1, 1, 0, 0,   1)
el.cen.EM(x,d,mu=3.5)
## we should get "-2LLR" = 1.2466....
myfun5 <- function(x, theta, eps) {
u <- (x-theta)*sqrt(5)/eps 
INDE <- (u < sqrt(5)) & (u > -sqrt(5)) 
u[u >= sqrt(5)] <- 0 
u[u <= -sqrt(5)] <- 1 
y <- 0.5 - (u - (u)^3/15)*3/(4*sqrt(5)) 
u[ INDE ] <- y[ INDE ] 
return(u)
}
el.cen.EM(x, d, fun=myfun5, mu=0.5, theta=3.5, eps=0.1)
}
\keyword{nonparametric}
\keyword{htest}

\eof
\name{el.trun.test}
\alias{el.trun.test}
\title{Empirical likelihood ratio for mean with left truncated data}
\usage{
el.trun.test(y,x,fun=function(t){t},mu,maxit=20,error=1e-9)
}
\description{
This program uses EM algorithm to compute the maximized
(wrt \eqn{p_i}) empirical
log likelihood function for left truncated data with 
the MEAN constraint:
\deqn{ \sum  p_i f(x_i)  = \int f(t) dF(t) = \mu ~. }
Where \eqn{p_i = \Delta F(x_i)} is a probability.
\eqn{\mu} is a given constant. 
It also returns those \eqn{p_i} and the \eqn{p_i} without
constraint, the Lynden-Bell estimator.

The log likelihood been maximized is
\deqn{ \sum_{i=1}^n \log \frac{\Delta F(x_i)}{1-F(y_i)} .}
}
\arguments{
    \item{y}{a vector containing the left truncation times.}
    \item{x}{a vector containing the survival times. truncation means x>y.}
    \item{fun}{a continuous (weight) function used to calculate
         the mean as in \eqn{H_0}.
         \code{fun(t)} must be able to take a vector input \code{t}.
         Default to the identity function \eqn{f(t)=t}.}
    \item{mu}{a real number used in the constraint, mean value of \eqn{f(X)}.}
    \item{error}{an optional positive real number specifying the tolerance of
       iteration error. This is the bound of the
       \eqn{L_1} norm of the differnence of two successive weights.}
    \item{maxit}{an optional integer, used to control maximum number of
             iterations. }
}
\value{
    A list with the following components:
    \item{"-2LLR"}{the maximized empirical log likelihood ratio
                  under the constraint.}
    \item{NPMLE}{jumps of NPMLE of CDF at ordered x.}
    \item{NPMLEmu}{same jumps but for constrained NPMLE.}
}
\details{

This implementation is all in R and have several for-loops in it. 
A faster version would use C to do the for-loop part. But it seems
faster enough and is easier to port to Splus.

When the given constants \eqn{\mu} is too far
away from the NPMLE, there will be no distribution
satisfy the constraint.
In this case the computation will stop.
The -2 Log empirical likelihood ratio
should be infinite. 

The constant \code{mu} must be inside 
\eqn{( \min f(x_i) , \max f(x_i) ) }
for the computation to continue. 
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{mu} closer
to the NPMLE --- 
\deqn{ \sum_{d_i=1} p_i^0 f(x_i) } 
\eqn{p_i^0} taken to be the jumps of the NPMLE of CDF. 
Or use a different \code{fun}. 

}
\author{ Mai Zhou }
\references{
    Zhou, M. (2002). 
        Computing censored empirical likelihood ratio 
        by EM algorithm. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}

    Li, G. (1995)
        Nonparametric likelihood ratio estimation of probabilities
        for truncated data. \emph{JASA} 90, 997-1003.
    
    Turnbull (1976)
       The empirical distribution function with arbitrarily grouped, censored
and truncated data. \emph{JRSS} B 38, 290-295.

}
\examples{
## example with tied observations
vet <- c(30, 384, 4, 54, 13, 123, 97, 153, 59, 117, 16, 151, 22, 56, 21, 18,
         139, 20, 31, 52, 287, 18, 51, 122, 27, 54, 7, 63, 392, 10)
vetstart <- c(0,60,0,0,0,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0)
el.trun.test(vetstart, vet, mu=80, maxit=15)
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}


\eof
\name{emplik-internal}
\alias{Wdataclean2}
\alias{Wdataclean3}
\alias{Wdataclean5}
\alias{DnR}
\alias{logelr}
\alias{logwelr}
\alias{gradf}
\alias{llog}
\alias{llogp}
\alias{llogpp}
\alias{solve3.QP}
\alias{WKM}
\alias{el.test.wt2}
\alias{el.test.wt3}
\alias{LTRC}
\alias{iter}
\alias{redistF}
\title{Internal emplik functions}
\usage{
logelr(x,mu,lam)
logwelr(x,mu,wt,lam)
gradf(z,wt,lam)
llog(z, eps)
llogp(z, eps)
llogpp(z, eps)
Wdataclean2(z,d,wt=rep(1,length(z)))
Wdataclean3(z,d,zc=rep(1, length(z)),wt=rep(1,length(z)))
Wdataclean5(z,d,zc=rep(1, length(z)),wt=rep(1,length(z)),xmat)
DnR(x,d,w,y=rep(-Inf,length(x)))
solve3.QP(D, d, A, b, meq, factorized=FALSE)
WKM(x,d,zc=rep(1,length(d)),w=rep(1,length(d)))
LTRC(x,d,w=rep(1, length(d)),y=rep(-Inf, length(x)))
el.test.wt2(x,wt,mu,maxit,gradtol,Hessian,svdtol,itertrace)
el.test.wt3(x,wt,mu,maxit,gradtol,Hessian,svdtol,itertrace)
iter(x, y, delta, beta)
redistF(y, d, Fdist)
}
\description{ Internal emplik functions }
\details{
These are not intended to be called by the user. 

\code{Wdataclean2} and
\code{DnR} are used by the functions \code{emplikH1.test}, 
\code{emplikH2.test} and \code{emplikdisc.test}.  It is also used by 
\code{LTRC}.  

\code{logelr}, \code{llog}, \code{llogp} and \code{llogpp} 
are used by function \code{el.test}.

\code{Wdataclean2}, \code{WKM} and \code{solve3.QP} are
used by function \code{el.cen.test}. 

\code{Wdataclean2} is used by \code{el.cen.EM}.

\code{LTRC} is for Left Truncated and Right Censored data.

\code{el.test.wt2}, \code{el.test.wt3} are similar to \code{el.test.wt} 
but can take vector mean as constraint. \code{llog}, \code{llogp} and
\code{llogpp} are used by both functions. In addition \code{logwelr}
is used by \code{el.test.wt3}.

\code{iter} is for perform one iteration of EM in the Buckley-James
            censored regression estimation.
\code{redistF} is for redistribution, according to Fdist.
}
\keyword{internal}

\eof
\name{emplikH1.test}
\alias{emplikH1.test}
\title{Empirical likelihood for hazard with right censored, 
                 left truncated data}
\usage{
emplikH1.test(x, d, y= -Inf, theta, fun, tola=.Machine$double.eps^.25)
}
\description{Use empirical likelihood ratio and Wilks theorem to test 
the null hypothesis that 
\deqn{\int f(t) dH(t) = \theta }
with right censored data. Where \eqn{H(t)} is the unknown
cumulative hazard
function; \eqn{f(t)} can be any given left continuous function and
\eqn{\theta} a given constant. In fact, \eqn{f(t)} can even be data
dependent, just have to be `predictable'.
}
\arguments{
   \item{x}{a vector of the observed survival times.}
   \item{d}{a vector of the censoring indicators, 1-uncensor; 0-censor.}
   \item{y}{a vector of the observed left truncation times.}
   \item{theta}{a real number used in the \eqn{H_0} to set the hazard 
               to this value.}
   \item{fun}{a left continuous (weight) function used to calculate 
       the weighted hazard in \eqn{ H_0}. \code{fun} must be able 
       to take a vector input. See example below.}
    \item{tola}{an optional positive real number specifying the tolerance of
       iteration error in solve the non-linear equation needed in constrained
        maximization.}
}
\details{
This function is designed for the case where the 
true distributions are all continuous.
So there should be no tie in the data.

The log empirical likelihood used here is the `Poisson' version likelihood:
\deqn{  \sum_{i=1}^n \delta_i \log (dH(x_i))  - [ H(x_i) - H(y_i) ]  ~. } 
}

If there are ties in the data that are resulted from rounding,
you may break the tie by adding a tiny number to the tied
observation. If those are true ties (thus the true distribution is discrete)
we recommend use \code{emplikdisc.test()}

The constant \code{theta} must be inside the so called
feasible region for the computation to continue. This is similar to the
requirement that in testing the value of the mean, the value must be
inside the convex hull of the observations.
It is always true that the NPMLE values are feasible. So when the
computation stops, that means there is no hazard function satisfy
the constraint. You may try to move the \code{theta} value closer
to the NPMLE.  When the computation stops, the -2LLR should have value
infinite.
}
\value{
    A list with the following components:
    \item{times}{the location of the hazard jumps.}
    \item{wts}{the jump size of hazard function at those locations.}
    \item{lambda}{the Lagrange multiplier.}
    \item{"-2LLR"}{the -2Log Likelihood ratio.}
    \item{Pval}{P-value}
    \item{niters}{number of iterations used}
}
\author{ Mai Zhou } 
\references{
    Pan, X. and Zhou, M. (2002),
    ``Empirical likelihood in terms of hazard for censored data''. 
    \emph{Journal of Multivariate Analysis} \bold{80}, 166-188.
}
\examples{
fun <- function(x) { as.numeric(x <= 6.5) }
emplikH1.test( x=c(1,2,3,4,5), d=c(1,1,0,1,1), theta=2, fun=fun) 

fun2 <- function(x) {exp(-x)}  
emplikH1.test( x=c(1,2,3,4,5), d=c(1,1,0,1,1), theta=0.2, fun=fun2) 
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}

\eof
\name{emplikH2.test}
\alias{emplikH2.test}
\title{Empirical likelihood for weighted hazard with 
        right censored, left truncated data}
\usage{
emplikH2.test(x, d, y= -Inf, K, fun, tola=.Machine$double.eps^.25,...)
}
\description{Use empirical likelihood ratio and Wilks theorem to test 
the null hypothesis that 
\deqn{\int f(t, ... ) dH(t) = K }
with right censored, left truncated data, where \eqn{ H(t) } is the (unknown)
cumulative hazard function;
\eqn{ f(t, ... )} can be any given left continuous function in \eqn{t};
(of course the integral must be finite). 
}
\arguments{
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector of the censoring indicators, 1-uncensor; 0-censor.}
    \item{y}{a vector containing the left truncation times.}
    \item{K}{a real number used in the constraint, i.e. to set the
               weighted integral of hazard to this value.}
    \item{fun}{a left continuous (in \code{t}) weight function
              used to calculate
         the weighted hazard in \eqn{H_0}. \code{fun(t, ... )} must be
         able to take a vector input \code{t}.}
    \item{tola}{an optional positive real number specifying the tolerance of
       iteration error in solve the non-linear equation needed in constrained
        maximization.}
    \item{...}{additional parameter(s), if any, passing along to \code{fun}. 
            This allows an implicit function of \code{fun}.  }
}
\value{
    A list with the following components:
    \item{times}{the location of the hazard jumps.}
    \item{wts}{the jump size of hazard function at those locations.}
    \item{lambda}{the Lagrange multiplier.}
    \item{"-2LLR"}{the -2Log Likelihood ratio.}
    \item{Pval}{P-value}
    \item{niters}{number of iterations used}
}
\author{ Mai Zhou } 
\references{
    Pan and Zhou (2002),
    ``Empirical likelihood in terms of cumulative
      hazard for censored data''. 
    \emph{Journal of Multivariate Analysis} \bold{80}, 166-188.
}
\details{
This version works for implicit function \code{ f(t, ...)}.

This function is designed for continuous distributions.
Thus we do not expect tie in the observations. If you believe
the true underlying distribution is continuous but the
sample observations have tie due to rounding, then you might want
to add a small number to the observations to break tie.

The likelihood used here is the `Poisson' version of the likelihood
\deqn{
\prod_{i=1}^n ( dH(x_i) )^{\delta_i} \exp [-H(x_i)+H(y_i)] . 
}

For discrete distributions we recommend
use \code{emplikdisc.test()}.

Please note here the largest observed time is NOT autometically defined to be
uncensored. In the el.cen.EM(), it is (to make F a proper distribution always).

The constant \code{K} must be inside the so called
feasible region for the computation to continue. This is similar to the
requirement that in testing the value of the mean, the value must be
inside the convex hull of the observations.
It is always true that the NPMLE values are feasible. So when the
computation stops, that means there is no hazard function satisfy 
the constraint. You may try to move the \code{theta} and \code{K} closer
to the NPMLE.  When the computation stops, the -2LLR should have value
infinite.
}
\examples{
z1<-c(1,2,3,4,5)
d1<-c(1,1,0,1,1)
fun4 <- function(x, theta) { as.numeric(x <= theta) }
emplikH2.test(x=z1,d=d1, K=0.5, fun=fun4, theta=3.5)
#Next, test if H(3.5) = log(2) .
emplikH2.test(x=z1,d=d1, K=log(2), fun=fun4, theta=3.5)
#Next, try one sample log rank test
indi <- function(x,y){ as.numeric(x >= y) }
fun3 <- function(t,z){rowsum(outer(z,t,FUN="indi"),group=rep(1,length(z)))} 
emplikH2.test(x=z1, d=d1, K=sum(0.25* z1), fun=fun3, z=z1) 
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}

\eof
\name{emplikdisc.test}
\alias{emplikdisc.test}
\title{Empirical likelihood ratio for discrete hazard
with right censored, left truncated data}
\usage{
emplikdisc.test(x, d, y= -Inf, K, fun, tola=.Machine$double.eps^.25, theta)
}
\description{
Use empirical likelihood ratio and Wilks theorem to test 
the null hypothesis that
\deqn{ \sum_i[f(x_i, \theta) \log(1- dH(x_i))] = K   }
where \eqn{H(t)} is the (unknown) discrete cumulative
hazard function; \eqn{f(t,\theta)} can be any given left 
continuous function in \eqn{t}. 
\eqn{\theta} is the parameter of the function 
and \code{K} is a given constant. 
The data can be right censored and left truncated.

When the given constants \eqn{\theta} and/or \code{K} are too far
away from the NPMLE, there will be no hazard function satisfy this 
constraint and the minus 2Log empirical likelihood ratio
will be infinite. In this case the computation will stop.
}
\arguments{
    \item{x}{a vector containing the observed survival times.}
    \item{d}{a vector containing the censoring indicators, 
           1-uncensor; 0-censor.}
    \item{y}{optional vector containing the left truncation times.}
    \item{K}{a real number used in the constraint, sum to this value.}
    \item{fun}{a left continuous (weight) function used to calculate
         the weighted discrete hazard in \eqn{H_0}. 
         \code{fun(x, theta)} must be able to take a
       vector input \code{x}, and a parameter \code{theta}. }
    \item{tola}{an optional positive real number specifying the tolerance of
       iteration error in solve the non-linear equation needed in constrained 
        maximization.}
    \item{theta}{a given real number used as the parameter of the 
	      function \eqn{f}.}
}
\value{
    A list with the following components:
    \item{times}{the location of the hazard jumps.}
    \item{wts}{the jump size of hazard function at those locations.}
    \item{lambda}{the final value of the Lagrange multiplier.}
    \item{"-2LLR"}{The -2Log Likelihood ratio.}
    \item{Pval}{P-value}
    \item{niters}{number of iterations used}
}
\details{
The log likelihood been maximized is the binomial likelihood:
\deqn{ \sum D_i \log w_i  + (R_i-D_i) \log [1-w_i] }
where \eqn{w_i = \Delta H(t_i)} is the jump 
of the cumulative hazard function, \eqn{D_i} is the number of failures
observed at \eqn{t_i}, \eqn{R_i} is the number of subjects at risk at
time \eqn{t_i}.

For discrete distributions, the jump size of the cumulative hazard at
the last jump is always 1. We have to exclude this jump from the 
summation since \eqn{ \log( 1- dH(\cdot))} do not make sense.

The constants \code{theta} and \code{K} must be inside the so called
feasible region for the computation to continue. This is similar to the
requirement that in testing the value of the mean, the value must be
inside the convex hull of the observations.
It is always true that the NPMLE values are feasible. So when the
computation stops, try move the \code{theta} and \code{K} closer
to the NPMLE.  When the computation stops, the -2LLR should have value
infinite.

In case you do not need the \code{theta} in the definition of the
function \eqn{f}, you still need to formally define your \code{fun} function
with a \code{theta} input, just to match the arguments.
}
\author{ Mai Zhou }
\references{
    Fang, H. (2000). Binomial Empirical Likelihood Ratio Method in
	   Survival Analysis. Ph.D. Thesis, Univ. of Kentucky, Dept
 	   of Statistics.

    Zhou and Fang (2001). 
    ``Empirical likelihood ratio for 2 sample problem 
                       for censored data''. 
    \emph{Tech Report, Univ. of Kentucky, Dept of Statistics}

    Zhou, M. (2002). 
}
\examples{
fun4 <- function(x, theta) { as.numeric(x <= theta) }
x <- c(1, 2, 3, 4, 5, 6, 5, 4, 3, 4, 1, 2.4, 4.5)
d <- c(1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1)
# test if -H(4) = -0.7 
emplikdisc.test(x=x,d=d,K=-0.7,fun=fun4,theta=4)
# we should get "discrete.-2logemlikRatio" 0.1446316  etc....
y <- c(-2,-2, -2, 1.5, -1)
emplikdisc.test(x=x,d=d,y=y,K=-0.7,fun=fun4,theta=4)
}
\keyword{nonparametric}
\keyword{survival}
\keyword{htest}

\eof
