\name{boxcox}

\alias{rboxcox}
\alias{dboxcox}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{The Box-Cox Transformed Normal Distribution}

\description{Functions related with the Box-Cox family of transformations.
  Density and random generation for the Box-Cox transformed normal
  distribution with mean
  equal to \code{mean} and standard deviation equal to \code{sd}, \emph{in the normal scale}.
}

\usage{
rboxcox(n, lambda, lambda2 = NULL, mean = 0, sd = 1)

dboxcox(x, lambda, lambda2 = NULL, mean = 0, sd = 1)

}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{lambda}{numerical value(s) for the transformation parameter
    \eqn{\lambda}{lambda}.  }
  \item{lambda2}{logical or numerical value(s) of the additional transformation
    (see DETAILS below). Defaults to \code{NULL}.  }
  \item{n}{number of observations to be generated.  }
  \item{x}{a vector of quantiles (\code{dboxcox}) or an output of
    \code{boxcox.fit} (\code{print}, \code{plot}, \code{lines}).  }
  \item{mean}{a vector of mean values at the normal scale.  }
  \item{sd}{a vector of standard deviations at the normal scale.  }
}

\details{
  Denote \eqn{Y} the variable at the original scale and \eqn{Y'} the
  transformed variable. The Box-Cox transformation is defined by:
  
  \deqn{Y' = \left\{ \begin{array}{ll}
    log(Y)
    \mbox{ , if $\lambda = 0$} \cr
    \frac{Y^\lambda - 1}{\lambda} \mbox{ ,  otherwise}
    \end{array} \right.}{Y' = log(Y) if lambda = 0 , Y' = ((Y ^ lambda) - 1)/lambda otherwise }. 
  
  An additional shifting parameter \eqn{\lambda_2}{lambda2} can be
  included in which case the transformation is given by: 
  
  \deqn{Y' = \left\{
    \begin{array}{ll}
    log(Y + \lambda_2)
    \mbox{ , $\lambda = 0$ } \cr
    \frac{(Y + \lambda_2)^\lambda - 1}{\lambda} \mbox{ ,  otherwise}
    \end{array} \right.}{Y' = log(Y + lambda2) if lambda = 0 ,
    Y' = (((Y + lambda2) ^ lambda) - 1)/lambda otherwise }.

 The function \code{rboxcox} samples \eqn{Y'}{Y'} from the normal distribution using
 the function \code{\link{rnorm}} and backtransform the values according to the
 equations above to obtain values of \eqn{Y}{Y}.
 If necessary the back-transformation truncates the values such that
 \eqn{Y' \geq \frac{1}{\lambda}}{Y' >= -1/lambda} results in
 \eqn{Y = 0}{Y = 0} in the original scale.
 Increasing the value of the mean and/or reducing the variance might help to avoid truncation.
}

\value{
  The functions returns the following results:
  \item{rboxcox}{a vector of random deviates.  }
  \item{dboxcox}{a vector of densities.  }
}

\references{
  Box, G.E.P. and Cox, D.R.(1964) An analysis of transformations. JRSS B
  \bold{26}:211--246.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{The parameter estimation function \code{\link{boxcox.fit}},
  the function \code{\link[MASS]{boxcox}} in the package \pkg{MASS} and
  the function \code{\link[car]{boxcox}} in the package \pkg{car}.
}

\examples{
## Simulating data
simul <- rboxcox(100, lambda=0.5, mean=10, sd=2)
##
## Comparing models with different lambdas,
## zero  means and unit variances
curve(dboxcox(x, lambda=-1), 0, 8)
for(lambda in seq(-.5, 1.5, by=0.5))
  curve(dboxcox(x, lambda), 0, 8, add = TRUE)
}
\keyword{distribution}

\eof
\name{InvChisquare}

\alias{dinvchisq}
\alias{rinvchisq}

\title{The (Scaled) Inverse Chi-Squared Distribution}

\description{
  Density and random generation for the scaled inverse chi-squared
  (\eqn{\chi^2_{ScI}}{chi^2_{ScI}}) distribution with
  \code{df} degrees of freedom and optional non-centrality parameter
  \code{scale}.
}
\usage{
dinvchisq(x, df, scale, log = FALSE)
rinvchisq(n, df, scale = 1/df)
}

\arguments{
  \item{x}{vector of quantiles.}
  \item{n}{number of observations. If \code{length(n) > 1}, the length
    is taken to be the number required.}
  \item{df}{degrees of freedom.}
  \item{scale}{scale parameter.}
  \item{log}{logical; if TRUE, densities d are given as log(d).}
}

\value{
  \code{dinvchisq} gives the density and \code{rinvchisq}
  generates random deviates.
}

\details{
  The inverse chi-squared distribution with \code{df}\eqn{= n}
  degrees of freedom has density
  \deqn{f(x) = \frac{1}{{2}^{n/2} \Gamma (n/2)} {(1/x)}^{n/2+1} {e}^{-1/(2x)}}{%
    f(x) = 1 / (2^(n/2) Gamma(n/2))  (1/x)^(n/2-1) e^(-1/(2x))}
  for \eqn{x > 0}.
  The mean and variance are \eqn{\frac{1}{(n-2)}}{1/(n-2)} and
  \eqn{\frac{2}{(n-4)(n-2)^2}}{2/((n-4)(n-2)^2)}.

  The non-central chi-squared distribution with \code{df}\eqn{= n}
  degrees of freedom and non-centrality parameter \code{scale}
  \eqn{= S^2} has density
  \deqn{
    f(x) = \frac{{n/2}^{n/2}}{\Gamma (n/2)} S^n {(1/x)}^{n/2+1}
    {e}^{-(n S^2)/(2x)}}{%
    f(x) = ((n/2)^(n/2))/(\Gamma (n/2)) S^n (1/x)^((n/2)+1) e^(-(n
    S^2)/(2x))}
  }
  for \eqn{x \ge 0}.
  The first is a particular case of the latter for \eqn{\lambda = n/2}. 
}

\seealso{
  \code{\link{rchisq}} for the chi-squared distribution which
  is the basis for this function.
}
\examples{
set.seed(1234); rinvchisq(5, df=2)
set.seed(1234); 1/rchisq(5, df=2)

set.seed(1234); rinvchisq(5, df=2, scale=5)
set.seed(1234); 5*2/rchisq(5, df=2)

## inverse Chi-squared is a particular case
x <- 1:10
all.equal(dinvchisq(x, df=2), dinvchisq(x, df=2, scale=1/2))
}
\keyword{distribution}

\eof
\name{Ksat}
\alias{Ksat}
\docType{data}
\title{Saturated Hydraulic Conductivity}
\description{
  The data consists of 32 measurements of the saturated hydraulic
  conductivity of a soil. 
}
\usage{data(Ksat)}
\format{
  The object \code{ca20} is a list of the class \code{geodata}
  with the following elements:
  \describe{    
    \item{\code{coords}}{a matrix with the coordinates of the soil samples.  }
    \item{\code{data}}{measurements of the saturated hidraulic conductivity.  }
    \item{\code{borders}}{a data-frame with the coordinates of a polygon defining the
      borders of the area.  }
    }
}
%\details{
%  ~~ If necessary, more details than the __description__ above ~~
%}
\source{
  Data provided by Dr. Dcio Cruciani, ESALQ/USP, Brasil.
}
%\references{
%  ~~ possibly secondary sources and usages ~~
%}
\examples{
data(Ksat)
summary(Ksat)
plot(Ksat, border=borders)
}
\keyword{datasets}

\eof
\name{SIC}
\docType{data}

\alias{sic}
\alias{SIC}
\alias{sic.100}
\alias{sic.367}
\alias{sic.all}
\alias{sic.some}
\alias{sic.borders}

\title{Spatial Interpolation Comparison data}

\usage{data(SIC)}

\description{
  Data from the SIC-97 project: Spatial Interpolation Comparison.
}

\format{
  Four objects of the \code{\link{class}}  \code{"geodata"}:
  \code{sic.all}, \code{sic.100}, \code{sic.367}, \code{sic.some}.
  Each is a list with two components:
  
  \describe{
    \item{\code{coords}}{the coordinates of the data locations. The
      distance are given in kilometers.  }
    \item{\code{data}}{rainfall values. The unit is milimeters.  }
    \item{\code{altitude}}{elevation values. The unit is milimeters.  }
  }
  
  Additionally an matrix \code{sic.borders} with the borders of the country.
}

\source{
  Data from the project \emph{Spatial Interpolation Comparison 97};
  see \url{ftp://ftp.geog.uwo.ca/SIC97/}
}

\references{
  
  Christensen, O.F., Diggle, P.J. and Ribeiro Jr, P.J. (2001)  Analysing positive-valued spatial data: the transformed Gaussian model.  In Monestiez, P., Allard, D. and Froidevaux (eds), GeoENV III - Geostatistics for environmental applications. Quantitative Geology and Geostatistics, Kluwer Series, 11, 287--298.

}

\examples{
data(SIC)
points(sic.100, borders=sic.borders)
}

\keyword{datasets}
\keywords{spatial}

\eof
\name{as.geodata}
\alias{as.geodata}
\alias{is.geodata}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Converts an Object to the Class "geodata"}

\description{
  Converts a matrix or a data-frame to an object of the
  \code{\link{class}}  \code{"geodata"}.\cr
  Objects of the class \code{"geodata"} are lists with two obligatory
  components: \code{coords} and \code{data}.
  Optional components are allowed and a typical example is a vector or
  matrix with covariate(s) value(s).
}

\usage{
as.geodata(obj, coords.col = 1:2, data.col = 3, data.names = NULL, 
           covar.col = NULL, covar.names = "obj.names",
           realisations = NULL,
           na.action = c("ifany", "ifdata", "ifcovar", "none"),
           rep.data.action, rep.covar.action)

is.geodata(x)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{obj}{a matrix or data-frame where each line corresponds to one
    spatial location. It should contain values of 2D coordinates,
    data and, optionally, covariate(s) value(s) at the locations.
    It can also take an output of the function \code{\link{grf}}, see
    DETAILS below. }
  \item{coords.col}{a vector with the column numbers corresponding to the
    spatial coordinates.  }
  \item{data.col}{a scalar or vector with column number(s) corresponding
    to the data.  }
  \item{data.names}{optional. A string or vector of strings with names for
    the data columns. Only valid if there is more than one column of
    data. By default, takes the names from the original object.  }
  \item{covar.col}{optional. A scalar or vector
    of the column number(s) corresponding to the covariate(s).  }
  \item{covar.names}{optional. A string or vector of strings with the name(s) of
    the covariates. By default take the names from the original object.  }
  \item{realisations}{optional. A vector indicating the realisation
    number or a number indicating a column in \code{obj} with the
    realisation indicator variable.
    See \code{DETAILS} below.  }
  \item{na.action}{string defining action to be taken in the presence of
    \code{NA}'s. The default option \code{"ifany"} excludes all points
    for which there are \code{NA}'s in the data or covariates. The option \code{"ifdata"} excludes points
    for which there are \code{NA}'s in the data. The default option \code{"ifcovar"} excludes all points
    for which there are \code{NA}'s in the covariates. The option
    \code{"none"} do not exclude points.  }
  \item{rep.data.action}{a string or a function. Defines action to be taken when there is more than
    one data at the same location. The default option \code{"none"}
    keeps the repeated locations, if any. The option \code{"first"} retains
    only the first data recorded at each location. Alternativelly a
    function can be passed and it will be used. For instance if
    \code{mean} is provided, the function will compute and return the average of
    the data at coincident locations. The non-default
    options will eliminate the repeated locations.  }
  \item{rep.covar.action}{idem to \code{rep.data.locations}, to be applied to
    the covariates, if any. Defaults to
    the same option set for \code{rep.data.locations}. }
  \item{x}{an object which is tested for the class \code{geodata}.}
}

\details{
  Objects of the class \code{"geodata"} contain data for
  geostatistical analysis using the package \pkg{geoR}.
  Storing data in this format facilitates the usage of the functions in \pkg{geoR}.
  However, conversion of objects to this class is not obligatory
  to carry out the analysis.
  
  \code{NA}'s are not allowed in the coordinates. By default the
  respective rows will not be included in the output.

  \bold{Realisations}\cr
  Tipically geostatistical data correspond to a unique realisation of
  the spatial process.
  However, sometimes different "realisations" are possible.
  For instance, if data are collected in the same area at different
  points in time and independence between time points is assumed,
  each time can be considered a different "replicate" or "realisation"
  of the same process. The argument \code{realisations} takes a vector
  indication the replication number and can be passed to other \pkg{geoR}
  functions as, for instance, \code{\link{likfit}}.

  The data format is similar to the usual \code{geodata} format in
  \pkg{geoR}.
  Suppose there are realisations (times) \eqn{1, \ldots, J}{1, ..., J}
  and for each realisations \eqn{n_1, ..., n_j} observations are available.
  The coordinates for different realisations
  should be combined in a single \eqn{n \times 2}{nx2} object,
  where \eqn{n=n_1 + \ldots + n_J}{n=n_1 + ... + n_J}.
  Similarly for the data vector and covariates (if any).

  \bold{grf objects}\cr
  If an object of the class \code{grf} is provided the functions just
  extracts the elements \code{coords} and \code{data} of this object. 
}

\value{
  An object of the \code{\link{class}} \code{"geodata"} which is a list
  with two obligatory components (coords and data)
  and other optional components:
  \item{coords}{an \eqn{n \times 2}  matrix where \eqn{n} is the number
    of spatial locations.}
  \item{data}{a vector of length \eqn{n}, for the univariate case or,
    an \eqn{n \times v} matrix or data-frame for the multivariate case,
    where \eqn{v} is the number of variables.}
  \item{covariates}{a vector of length \eqn{n} or
    an \eqn{n \times p} matrix with covariate(s) values,
    where \eqn{p} is the number of covariates. Only returned if
    covariates are provided.  }
  \item{realisations}{a vector on size \eqn{n} with the replication
    number. Only returned if argument realisations is provided.  } 
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }
%
% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{read.geodata}} for reading data from an
  \emph{ASCII} file and \code{\link{list}} for general information on lists.
}

\examples{
\dontrun{
## converting the data-set "topo" from the package spatial (VR's bundle)
## to the geodata format:
require(spatial)
data(topo)
topo
topogeo <- as.geodata(topo)
names(topogeo)
topogeo
}
}
\keyword{spatial}
\keyword{classes}
\keyword{manip}

\eof
\name{boxcox.fit}

\alias{boxcox.fit}
\alias{boxcox.negloglik}
\alias{plot.boxcox.fit}
\alias{lines.boxcox.fit}
\alias{print.boxcox.fit}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Parameter Estimation for the Box-Cox Transformation}

\description{Parameter estimation and plotting of the results for the
  Box-Cox transformed normal distribution.
}

\usage{
boxcox.fit(data, xmat, lambda, lambda2 = NULL, add.to.data = 0, \dots)

print.boxcox.fit(x, \dots)

plot.boxcox.fit(x, hist = TRUE, data = eval(x$call$data), \dots)

lines.boxcox.fit(x, data = eval(x$call$data), \dots)

}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{data}{a vector with the data.  }
  \item{xmat}{a matrix with covariates values. Defaults to \code{rep(1,
      length(y))}.  } 
  \item{lambda}{numerical value(s) for the transformation parameter
    \eqn{\lambda}{lambda}. 
    Used as the initial value in the function for parameter estimation.
    If not provided default values are assumed. If multiple values are passed
    the one with highest likelihood is used as initial value. }
  \item{lambda2}{logical or numerical value(s) of the additional transformation
    (see DETAILS below). Defaults to \code{NULL}.
    If \code{TRUE} this parameter is also estimated and the initial
    value is set to the absolute value of the minimum data.
    A numerical value is provided it is used as the initial
    value. Multiple values are allowed as for \code{lambda}.  }
  \item{add.to.data}{a constant value to be added to the data.  }
  \item{x}{a list, typically an output of the function
    \code{boxcox.fit}.  }
  \item{hist}{logical indicating whether histograms should to be plotted.  }
  \item{\dots}{extra parameters to be passed to the minimization
    function \code{\link{optim}} (\code{boxcox.fit}), \code{\link{hist}}
    (\code{plot}) or \code{\link{curve}} (\code{lines}).  }
  
}

\value{
  The functions returns the following results:
  \item{boxcox.fit}{a list with estimated  parameters and results on
    the numerical minimization.  }
  \item{print.boxcox.fit}{print estimated parameters. No values
    returned.  }
  \item{plot.boxcox.fit}{plots histogram of the data (optional) and
    the model.  No values returned.  This function is only valid if
    covariates are not included in \code{boxcox.fit}. }
  \item{lines.boxcox.fit}{adds a line with the fitted model to the
    current plot. No values returned.  This function is only valid if
    covariates are not included in \code{boxcox.fit}.  }
}

\references{
  Box, G.E.P. and Cox, D.R.(1964) An analysis of transformations. JRSS B
  \bold{26}:211--246.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{rboxcox}} and  \code{\link{dboxcox}} for the
  expression and more on the
  Box-Cox transformation,
  the minimization function \code{\link{optim}},
  the function \code{\link[MASS]{boxcox}} in the package \pkg{MASS} and
  the function \code{\link[car]{boxcox}} in the package \pkg{car}.
}

\examples{
## Simulating data
simul <- rboxcox(100, lambda=0.5, mean=10, sd=2)
## Finding the ML estimates
ml <- boxcox.fit(simul)
ml
## Ploting histogram and fitted model
plot(ml)
##
## Comparing models with different lambdas,
## zero  means and unit variances
curve(dboxcox(x, lambda=-1), 0, 8)
for(lambda in seq(-.5, 1.5, by=0.5))
  curve(dboxcox(x, lambda), 0, 8, add = TRUE)
##
## Another example, now estimating lambda2
##
simul <- rboxcox(100, lambda=0.5, mean=10, sd=2)
ml <- boxcox.fit(simul, lambda2 = TRUE)
ml
plot(ml)
##
## An example with a regression model
##
if(require(MASS)){
  data(trees)
  boxcox.fit(data = trees[,3], xmat = trees[,1:2])
  }
}

\keyword{regression}
\keyword{models}
\keyword{hplot}


\eof
\name{boxcox.geodata}

\alias{boxcox.geodata}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Box-Cox transformation for geodata objects}
\description{
  Method for Box-Cox transformation for objects of the class
  \code{geodata} assuming the data are independent. 
  Computes and optionally plots profile log-likelihoods for the parameter of the Box-Cox simple power transformation \eqn{y^lambda}.
}
\usage{
boxcox.geodata(object, trend = "cte", ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{an object of the class geodata. See \code{\link{as.geodata}}. }
  \item{trend}{specifies the mean part of the model. See
    \code{\link{trend.spatial}} for further details.
    Defaults to \code{"cte"}.  }
  \item{\dots}{arguments to be passed for the function
    \code{\link[MASS]{boxcox}}.  }
}

\details{
  This is just a wrapper for the function \code{\link[MASS]{boxcox}}
  facilitating its usage with \code{geodata} objects.

  Notice this assume independent observations which is typically
  not the case for \code{geodata} objects.
}

\value{
  A list of the \code{lambda} vector and the computed profile log-likelihood vector, invisibly if the result is plotted.
}

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{\code{\link[MASS]{boxcox}}, \code{\link[MASS]{boxcox.fit}} for
  parameter estimation results for independent data and
  \code{\link[MASS]{likfit}} for parameter estimation
  within the geostatistical model.
}

\examples{
if(require(MASS)){
data(wolfcamp)
boxcox(wolfcamp)

data(ca20)
boxcox(ca20, trend = ~altitude)
}
}
\keyword{regression}
\keyword{models}
\keyword{hplot}



\eof
\name{ca20}
\alias{ca20}

\docType{data}

\title{Calcium content in soil samples}
\usage{data(ca20)}
\description{
  This data set contains the calcium content measured in soil samples
  taken from the 0-20cm layer at 178 locations within a certain study area divided in three
  sub-areas. The elevation at each location was also recorded.

  The first region is tipically flooded during the rain season and not
  used as an experimental area. The calcium levels would represent
  the natural content in the region. The second region has received
  fertilizers a while ago and is tipically occupied by rice fields.
  The third region has recieved fertilizers recently and is frequently
  used as an experimental area.
}

\format{
  The object \code{ca20} belongs to the class \code{geodata}
  and is a list
  with the following elements:
  
  \describe{    
    \item{\code{coords}}{a matrix with the coordinates of the soil samples.  }
    \item{\code{data}}{calcium content measured in \eqn{mmol_c/dm^3}.  }
    \item{\code{covariate}}{a data-frame with the covariates
      \describe{
	\item{\code{altitude}}{a vector with the elevation of each
	  sampling location, in meters (\eqn{m}).}
	\item{\code{area}}{a factor indicating the sub area to which the
	  locations belongs.}
      }
    }
    \item{\code{borders}}{a matrix with the coordinates defining the
      borders of the area.  }
    \item{\code{reg1}}{a matrix with the coordinates of the limits of
      the sub-area 1. }
    \item{\code{reg1}}{a matrix with the coordinates of the limits of
      the sub-area 2. }
    \item{\code{reg1}}{a matrix with the coordinates of the limits of
      the sub-area 3. }
  }
}

\source{
  The data was collected by researchers from PESAGRO and EMBRAPA-Solos, Rio
  de Janeiro, Brasil
  and provided by Dra. Maria Cristina Neves de Oliveira.

  Capeche, C.L. \emph{et al.} (1997) Caracterizao pedolgica da
  fazenda Angra - PESAGRO/RIO - Estao experimental de Campos (RJ).
  CD-ROM In: \emph{Congresso BRASILEIRO de Cincia do Solo} 26,
  Informao, globalizao, uso do solo; Rio de Janeiro,
  1997. trabalhos. Rio de Janeiro: Embrapa/SBCS, 1997.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\keyword{datasets}

\eof
\name{camg}
\alias{camg}
\docType{data}
\title{Calcium and magnesium content in soil samples}
\description{
  This data set contains the calcium content measured in soil samples
  taken from the 0-20cm layer at 178 locations within a certain study area divided in three
  sub-areas. The elevation at each location was also recorded.

  The first region is tipically flooded during the rain season and not
  used as an experimental area. The calcium levels would represent
  the natural content in the region. The second region has received
  fertilizers a while ago and is tipically occupied by rice fields.
  The third region has recieved fertilizers recently and is frequently
  used as an experimental area.
}
\usage{data(camg)}
\format{
  A data frame with 178 observations on the following 10 variables.
  \describe{
    \item{east}{east-west coordinates, in meters.}
    \item{north}{north-south coordinates, in meters.}
    \item{elevation}{elevation, in meters}
    \item{region}{a factor where numbers indicate different sub-regions
      within the area}
    \item{ca020}{calcium content in the 0-20cm soil layer, measured in \eqn{mmol_c/dm^3}.  }
    \item{mg020}{calcium content in the 0-20cm soil layer, measured in \eqn{mmol_c/dm^3}.  }
    \item{ctc020}{calcium content in the 0-20cm soil layer.  }
    \item{ca2040}{calcium content in the 20-40cm soil layer, measured in \eqn{mmol_c/dm^3}.  }
    \item{mg2040}{calcium content in the 20-40cm soil layer, measured in \eqn{mmol_c/dm^3}.  }
    \item{ctc2040}{calcium content in the 20-40cm soil layer.  }
  }
}
\details{
  More details about this data-set, including coordinates of the region
  and sub-region borders
  can be found in the data object \code{\link{ca20}}.
}
\source{
  The data was collected by researchers from PESAGRO and EMBRAPA-Solos, Rio
  de Janeiro, Brasil
  and provided by Dra. Maria Cristina Neves de Oliveira.

  Capeche, C.L. \emph{et al.} (1997) Caracterizao pedolgica da
  fazenda Angra - PESAGRO/RIO - Estao experimental de Campos (RJ).
  CD-ROM In: \emph{Congresso BRASILEIRO de Cincia do Solo} 26,
  Informao, globalizao, uso do solo; Rio de Janeiro,
  1997. trabalhos. Rio de Janeiro: Embrapa/SBCS, 1997.
}
%\references{
%  ~~ possibly secondary sources and usages ~~
%}
\examples{
data(camg)
plot(camg[-(1:2),])
mg20 <- as.geodata(camg, data.col=6)
plot(mg20)
points(mg20)
}
\keyword{datasets}

\eof
\name{cite.geoR}
\alias{cite.geoR}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Citing Package geoR in Publications}
\description{
  How to cite the package \pkg{geoR} in publications.
}
\usage{
cite.geoR()
}
%- maybe also 'usage' for other objects documented here.
\details{
  Execute function \code{cite.geoR()} for information on how to cite geoR in
     publications.
}
%\value{
%  ~Describe the value returned
%  If it is a LIST, use
%  \item{comp1 }{Description of 'comp1'}
%  \item{comp2 }{Description of 'comp2'}
%  ...
%}
%\references{ ~put references to the literature/web site here ~ }
%\author{ ~~who you are~~ }
%\note{ ~~further notes~~ }


%\seealso{ ~~objects to See Also as \code{\link{~~fun~~}}, ~~~ }
\examples{
cite.geoR()
}
\keyword{misc}% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{coords.aniso}

\alias{coords.aniso}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Geometric Anisotropy Correction}

\description{
  Transforms or back-transforms a set of coordinates according to
  the geometric anisotropy parameters.
}

\usage{
coords.aniso(coords, aniso.pars, reverse = FALSE)
}

\arguments{
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix with the coordinates
    to be transformed.  }
  \item{aniso.pars}{a vector with two elements, \eqn{\psi_A}{psiA} and
    \eqn{\psi_R}{psiR},  the \emph{anisotropy
      angle} and the \emph{anisotropy ratio}, respectively. Notice that the
    parameters must be provided in this order. 
    See section \code{DETAILS} below for more information on anisotropy parameters.  }
  \item{reverse}{logical. Defaults to \code{FALSE}. If \code{TRUE} the reverse
    transformation is performed.  }
}

\details{
  Geometric anisotropy is defined by two parameters:
  \describe{
    \item{Anisotropy angle}{defined here as the azimuth angle of the
    direction with greater spatial continuity, i.e. the angle between the
    \emph{y-axis} and the direction with the maximum range.  }
    
    \item{Anisotropy ratio}{defined here as the ratio between the ranges
    of the directions with greater and smaller continuity, i.e. the ratio
    between maximum and minimum ranges. Therefore, its value is always
    greater or equal to one.  }
  }
    
  If \code{reverse = FALSE} (the default) the
  coordinates are transformed  from the \emph{anisotropic space} to the \emph{isotropic
    space}.
  The transformation consists in multiplying the original
  coordinates by a rotation matrix \eqn{R}
  and a
  shrinking matrix \eqn{T}, as follows:
  \deqn{X_m = X  R T , }{X_m = X \%*\% R \%*\% T , }
  where \eqn{X_m} is a matrix with the modified coordinates (isotropic
  space) , \eqn{X} is a matrix with original coordinates (anisotropic
  space), \eqn{R} rotates coordinates according to the anisotropy angle
  \eqn{\psi_A}{psiA} and \eqn{T} shrinks the coordinates according to
  the anisotropy ratio \eqn{\psi_R}{psiR}.
  
  If \code{reverse = TRUE}, the back-transformation is performed, i.e.
  transforming the coordinates from the \emph{isotropic space} to the
  \emph{anisotropic  space} by computing:
  \deqn{X = X_m (R T)^{-1}}{X = X_m \%*\% solve( R \%*\% T ).}

}


\value{
  An \eqn{n \times 2}{n x 2} matrix with the transformed coordinates.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br} \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

 
%\seealso{ ~~objects to SEE ALSO as \code{\link{~~fun~~}}, ~~~ }

\examples{
op <- par(no.readonly = TRUE)
par(mfrow=c(3,2))
par(mar=c(2.5,0,0,0))
par(mgp=c(2,.5,0))
par(pty="s")
## Defining a set of coordinates
coords <- expand.grid(seq(-1, 1, l=3), seq(-1, 1, l=5))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coords[,1], coords[,2], 1:nrow(coords))
## Transforming coordinates according to some anisotropy parameters
coordsA <- coords.aniso(coords, aniso.pars=c(0, 2))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coordsA[,1], coordsA[,2], 1:nrow(coords))
##
coordsB <- coords.aniso(coords, aniso.pars=c(pi/2, 2))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coordsB[,1], coordsB[,2], 1:nrow(coords))
##
coordsC <- coords.aniso(coords, aniso.pars=c(pi/4, 2))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coordsC[,1], coordsC[,2], 1:nrow(coords))
##
coordsD <- coords.aniso(coords, aniso.pars=c(3*pi/4, 2))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coordsD[,1], coordsD[,2], 1:nrow(coords))
##
coordsE <- coords.aniso(coords, aniso.pars=c(0, 5))
plot(c(-1.5, 1.5), c(-1.5, 1.5), xlab="", ylab="", type="n")
text(coordsE[,1], coordsE[,2], 1:nrow(coords))
##
par(op)
}

\keyword{spatial}

\eof
\name{coords2coords}
\alias{coords2coords}
\alias{zoom.coords}
\alias{zoom.coords.default}
\alias{zoom.coords.geodata}
\alias{rect.coords}

\title{Operations on Coordinates}
\description{
  Functions for shifting, zooming and envolving rectangle of a set of coordinates.
}
\usage{
coords2coords(coords, xlim, ylim, xlim.ori, ylim.ori)

zoom.coords(x, \ldots)

zoom.coords.default(x, xzoom, yzoom, xlim.ori, ylim.ori, xoff=0, yoff=0, \ldots)

zoom.coords.geodata(x, \ldots)

rect.coords(coords, xzoom = 1, yzoom=xzoom, add.to.plot=TRUE,
            quiet = FALSE, \ldots)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{coords, x}{two column matrix or data-frame with coordinates.  }
  \item{xlim}{range of the new x-coordinates. }
  \item{ylim}{range of the new y-coordinates. }
  \item{xlim.ori}{optional. Range of the original x-coordinates, by
    default the range of the original x-coordinates.  }
  \item{ylim.ori}{optional. Range of the original y-coordinates, by
    default the range of the original y-coordinates.  }
  \item{xzoom}{scalar, expanding factor in the x-direction.  }
  \item{yzoom}{scalar, expanding factor in the y-direction.  }
  \item{xoff}{scalar, shift in the x-direction.  }
  \item{yoff}{scalar, shift in the y-direction.  }
  \item{add.to.plot}{logical, if \code{TRUE} the retangle is added to
    the current plot.  }
  \item{quiet}{logical, none is returned.  }
  \item{\ldots}{further arguments to be passed to \code{\link{rect}}.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  \item{coords2coords and zoom.coords}{return an object of the same type as given in the argument
  \code{coords} with the transformed coordinates.  }
  \item{rect.coords}{returns a matrix with the 4 coordinates of
  the rectangle defined by the coordinates.  }
}
%\references{ ~put references to the literature/web site here ~ }
%\author{ ~~who you are~~ }
\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{\code{\link{subarea}}, \code{\link{rect}}}

\examples{
foo <- matrix(c(4,6,6,4,2,2,4,4), nc=2)
foo1 <- zoom.coords(foo, 2)
foo1
foo2 <- coords2coords(foo, c(6,10), c(6,10))
foo2
plot(1:10, 1:10, type="n")
polygon(foo)
polygon(foo1, lty=2)
polygon(foo2, lwd=2)
arrows(foo[,1], foo[,2],foo1[,1],foo1[,2], lty=2)
arrows(foo[,1], foo[,2],foo2[,1],foo2[,2])
legend(1,10, c("foo", "foo1 (zoom.coords)", "foo2 (coords2coords)"), lty=c(1,2,1), lwd=c(1,1,2), cex=1.7)

## "zooming" part of The Gambia map
data(gambia)
gb <- gambia.borders/1000
gd <- gambia[,1:2]/1000
plot(gb, ty="l", asp=1, xlab="W-E (kilometres)", ylab="N-S (kilometres)")
points(gd, pch=19, cex=0.5)
r1b <- gb[gb[,1] < 420,]
rc1 <- rect.coords(r1b, lty=2)

r1bn <- zoom.coords(r1b, 1.8, xoff=90, yoff=-90)
rc2 <- rect.coords(r1bn, xz=1.05)
segments(rc1[c(1,3),1],rc1[c(1,3),2],rc2[c(1,3),1],rc2[c(1,3),2], lty=3)

lines(r1bn)
r1d <- gd[gd[,1] < 420,]
r1dn <- zoom.coords(r1d, 1.7, xlim.o=range(r1b[,1],na.rm=TRUE), ylim.o=range(r1b[,2],na.rm=TRUE), xoff=90, yoff=-90)
points(r1dn, pch=19, cex=0.5)
text(450,1340, "Western Region", cex=1.5)
}
\keyword{spatial}% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{cov.spatial}

\alias{cov.spatial}
\alias{cor.number}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Computes Value of the Covariance Function}

\description{
  Computes the covariances for pairs variables, given the separation
  distance of their locations.
  Options for different correlation functions are available.
  The results can be seen as a change of metric,
  from the \emph{Euclidean distances} to \emph{covariances}.
}

\usage{
cov.spatial(obj, cov.model= "matern",
            cov.pars=stop("no cov.pars argument provided"),
            kappa = 0.5)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{obj}{a numeric object (vector or matrix), typically with values
    of distances between pairs of spatial locations.  }
  \item{cov.model}{string indicating the type of the correlation
    function. Available choices are: "matern", "exponential", "gaussian",
            "spherical", "circular", "cubic", "wave",
            "power", "powered.exponential", "cauchy", 
            "gneiting", "gneiting.matern", "pure.nugget".
    See section \code{DETAILS} for available options and expressions of the correlation
    functions.  }
  \item{cov.pars}{a vector with 2 elements or an \eqn{ns \times 2}{ns x 2} matrix with
    the covariance parameters. The first element (if a vector) or first
    column (if a matrix) corresponds to the variance parameter \eqn{\sigma^2}{sigma^2}. The
    second element or column corresponds to the range parameter
    \eqn{\phi}{phi} of the correlation function.
    If a matrix is provided, each row corresponds to
    the parameters of one \emph{spatial structure} (see DETAILS below).
    }
  \item{kappa}{numerical value for the additional smoothness parameter of the
    correlation function.
    Only required by the following correlation
    functions: \code{"matern"}, \code{"powered.exponential"},
    \code{"cauchy"} and \code{"gneiting.matern"}.  }
}


\details{
  Covariance functions return the value of the covariance
  \eqn{C(h)} between a pair variables located at points separated by the
  distance \eqn{h}.
  The covariance function can be written as a product of a variance
  parameter \eqn{\sigma^2}{sigma^2} times a positive definite
  \emph{correlation function} \eqn{\rho(h)}{rho(h)}:
  \deqn{C(h) = \sigma^2 \rho(h).}{C(h) = sigma^2 * rho(h).}
  The expressions of the covariance functions available in \pkg{geoR}
  are given below. We recommend the \emph{LaTeX} (and/or the corresponding
  \emph{.dvi}, \emph{.pdf} or \emph{.ps}) version of this document for
  better visualization of the formulas.
  
  Denote \eqn{\phi}{phi} the basic parameter of the correlation
  function and name it the \emph{range parameter}.
  Some of the correlation functions will have an extra parameter
  \eqn{\kappa}{kappa}, the \emph{smoothness parameter}.
  \eqn{K_\kappa(x)} denotes the modified Bessel
  function of the third kind of order \eqn{\kappa}{kappa}. See
  documentation of the function \code{\link{besselK}} for further details.
  In the equations below the functions are valid for  \eqn{\phi>0}{phi >
    0} and \eqn{\kappa>0}{kappa > 0}, unless stated otherwise.
  
  \bold{exponential}\cr
  \deqn{\rho(h) = \exp(-\frac{h}{\phi})}{exp(-h/phi)}

  \bold{wave}\cr
  \deqn{\rho(h) = \frac{\phi}{h}\sin(\frac{h}{\phi})}{(phi/h) *
    sin(h/phi)} 

  \bold{matern}\cr
  \deqn{\rho(h) =
    \frac{1}{2^{\kappa-1}\Gamma(\kappa)}(\frac{h}{\phi})^\kappa
    K_{\kappa}(\frac{h}{\phi})}{rho(h) =
    (1/(2^(kappa-1) * \Gamma(kappa))) * ((h/phi)^kappa) * K_{kappa}(h/phi)} 
  
  \bold{gaussian}\cr
  \deqn{\rho(h) = \exp[(-\frac{h}{\phi})^2]}{exp(-(h/phi)^2)} 

  \bold{spherical}\cr
  \deqn{\rho(h) = \left\{ \begin{array}{ll}
    1 - 1.5\frac{h}{\phi} + 0.5(\frac{h}{\phi})^3
    \mbox{ , if $h$ < $\phi$} \cr
    0    \mbox{ ,  otherwise}
    \end{array} \right.}{rho(h) = 
    1 - 1.5 * (h/phi) + 0.5(h/phi)^3   if h < $phi$ , 
    0   otherwise}
    
  \bold{circular}\cr
  Let \eqn{\theta = \min(\frac{h}{\phi},1)}{theta = min(h/phi,1)} and
  \deqn{g(h)= 2\frac{(\theta\sqrt{1-\theta^2}+
      \sin^{-1}\sqrt{\theta})}{\pi}.}{gamma(h)= 2 * ((\theta\sqrt{1-\theta^2}+
      \sin^{-1}\sqrt{theta}))/pi.}
  Then, the circular model is given by:
  \deqn{\rho(h) = \left\{ \begin{array}{ll}
    1 - g(h)  \mbox{ , if $h < \phi$}\cr
    0    \mbox{ , otherwise}
    \end{array} \right.}{rho(h) =
    1 - gamma(h)  if h < phi , 0 otherwise}

  
  \bold{cubic}\cr
  \deqn{\rho(h) = \left\{ \begin{array}{ll}
    7(\frac{h}{\phi})^2-8.75(\frac{h}{\phi})^3+
    3.5(\frac{h}{\phi})^5-0.75(\frac{h}{\phi})^7 \mbox{ , if $h<\phi$} \cr
    0  \mbox{ , otherwise.}
    \end{array} \right.}{rho(h) = 
    7 * ((h/phi)^2) - 8.75 * ((h/phi)^3) +
    3.5 * ((h/phi)^5) - 0.75 * ((h/phi)^7) if h < phi
    , 0  otherwise.}
  
  \bold{power (and linear)}\cr
  The parameters of the this model
  \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi} can not be  
  interpreted as \emph{partial sill} and \emph{range}
  as for the other models.
  This model implies an unlimited dispersion and,
  therefore, has no sill and corresponds to a process which is only
  intrinsically stationary.
  The variogram function is given by:
  \deqn{\gamma(h) = \sigma^2 {h}^{\phi} \mbox{ , } 0 < \phi < 2,
    \sigma^2 > 0}{sigma^2
    * h^phi , 0 < phi < 2, sigma^2 > 0} 

  Since the corresponding process is not second order stationary the
  covariance and correlation functions are not defined.
  For internal calculations the \emph{geoR} functions uses "working" covariance and correlation
  functions computed as:
  \deqn{\rho_w(h) = \frac{C_w(h)}{max(\gamma(h))} =
    \frac{max(\gamma(h)) - \gamma(h)}{max(\gamma(h))}.}{rho_w(h) =
    C_w(h)/max(gamma(h)) = (max(gamma(h)) - gamma(h))/max(gamma(h)).}
  
  The \emph{linear} model corresponds to the particular case where
  \eqn{\phi = 1}{phi = 1}.
  
  \bold{powered.exponential}\cr
  \deqn{\rho(h) = \exp[-(\frac{h}{\phi})^\kappa] \mbox{ , } 0 < \kappa
    \leq 2}{rho(h) = exp[-(h/phi)^kappa] if  0 < kappa <= 2}

  
  \bold{cauchy} \cr
  \deqn{\rho(h) = [1+(\frac{h}{\phi})^2]^{-\kappa}}{rho(h) = [1+(h/phi)^2]^(-kappa)}

  
  \bold{gneiting}\cr
  Let \eqn{\theta = \min(\frac{h}{\phi},1)}{theta = min((h/phi),1)}.
  The Gneiting model is given by:
  \deqn{\rho(h) = (1+8\theta+25\theta^2+32\theta^3)(1-\theta)^8}{rho(h)
    = (1 + 8 * theta + 25 * (theta^2) + 32 * (theta^3)) * ((1 - theta)^8)}

  
  \bold{gneiting.matern}\cr
  Let \eqn{\alpha=\phi\kappa_2}{alpha=phi * kappa_2}, \eqn{\rho_m(\cdot)}{rho_m(.)} denotes the \eqn{\mbox{Mat\'{e}rn}}{Matern} model
  and \eqn{\rho_g(\cdot)}{rho_g(.)} the Gneiting model. Then the
  \eqn{\mbox{Gneiting-Mat\'{e}rn}}{Gneiting-Matern} is given by
  \deqn{\rho(h) = \rho_g(h|\phi=\alpha) \,
    \rho_m(h|\phi=\phi,\kappa=\kappa_1)}{rho(h) = rho_g(h | phi=alpha) *
    rho_m(h | phi=phi, kappa = kappa_1)}


  Models with several structures
  usually called \emph{nested models}
  in the geostatistical literature are also allowed.
  In this case the argument \code{cov.pars} takes a matrix and
  \code{cov.model} and \code{lambda} can either have length equal to
  the number of rows of this matrix or length 1.
  For the latter  cov.model and/or lambda are recycled, i.e. the same
  value is used for all structures.

}


\value{
  The function returns values of the covariances corresponding to  the
  given distances. 
  The type of output is the same as the type of the object provided in the
  argument \code{obj}, typically a vector, matrix or array.
}

\references{

  For a review on correlation functions:\cr
  Schlather, M. (1999) \emph{An introduction to positive definite functions and to unconditional
    simulation of random fields}. Technical report ST 99-10, Dept. of Maths and Statistics,
    Lancaster University. 

%    For \pkg{geoR} implementation: \cr   
%    Ribeiro, P.J. Jr. and Diggle, P.J. (1999) \emph{geoR: a geostatistical package/library for R/S-PLUS}. Tech. Report ST-99-09, Dept
%  Maths and Stats, Lancaster University.\cr
%  Available on-line at: 
%  \url{http://www.maths.lancs.ac.uk/~ribeiro/publications.html}.
  

  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{matern}} for computation of the
  \eqn{\mbox{Mat\'{e}rn}}{Matern} model, \code{\link{besselK}} for
  computation of the Bessel function  and
  \code{\link{varcov.spatial}} for computations related to the covariance matrix. }

\examples{
#
# Variogram models with the same "practical" range:
#
v.f <- function(x, ...){1-cov.spatial(x, ...)}
#
curve(v.f(x, cov.pars=c(1, .2)), from = 0, to = 1,
      xlab = "distance", ylab = expression(gamma(h)),
      main = "variograms with equivalent \"practical range\"")
curve(v.f(x, cov.pars = c(1, .6), cov.model = "sph"), 0, 1,
      add = TRUE, lty = 2)
curve(v.f(x, cov.pars = c(1, .6/sqrt(3)), cov.model = "gau"),
      0, 1, add = TRUE, lwd = 2)
legend(0.5,.3, c("exponential", "spherical", "gaussian"),
       lty=c(1,2,1), lwd=c(1,1,2))
#
# Matern models with equivalent "practical range"
# and varying smoothness parameter
#
curve(v.f(x, cov.pars = c(1, 0.25), kappa = 0.5),from = 0, to = 1,
      xlab = "distance", ylab = expression(gamma(h)), lty = 2,
      main = "models with equivalent \"practical\" range")
curve(v.f(x, cov.pars = c(1, 0.188), kappa = 1),from = 0, to = 1,
      add = TRUE)      
curve(v.f(x, cov.pars = c(1, 0.14), kappa = 2),from = 0, to = 1,
      add = TRUE, lwd=2, lty=2)      
curve(v.f(x, cov.pars = c(1, 0.117), kappa = 2),from = 0, to = 1,
      add = TRUE, lwd=2)      
legend(0.4,.4, c(expression(paste(kappa == 0.5, "  and  ",
       phi == 0.250)),
       expression(paste(kappa == 1, "  and  ", phi == 0.188)),
       expression(paste(kappa == 2, "  and  ", phi == 0.140)),
       expression(paste(kappa == 3, "  and  ", phi == 0.117))),
       lty=c(2,1,2,1), lwd=c(1,1,2,2))
# plotting a nested variogram model
curve(v.f(x, cov.pars = rbind(c(.4, .2), c(.6,.3)),
          cov.model = c("sph","exp")), 0, 1, ylab='nested model')
}

\keyword{spatial}%-- one or more ...
\keyword{models}%-- one or more ...






\eof
\name{dup.coords}
\alias{dup.coords}
\alias{dup.coords.default}
\alias{dup.coords.geodata}

\title{Locates duplicated coordinates}
\description{
  This funtions takes an object with 2-D coordinates and returns the
  positions of the duplicated coordinates.
}
\usage{
dup.coords(x)
dup.coords.default(x)
dup.coords.geodata(x)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{a two column numeric matrix or data frame.}
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  A list where each component is a vector with the positions  or the
  rownames, if different from \code{NULL}, of the duplicates coordinates. 
}
%\references{ ~put references to the literature/web site here ~ }
\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br} \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{\code{\link{as.geodata}}}
\examples{
## simulating data
foo <- grf(30, cov.p=c(1, .3)) 
## forcing some duplicated locations
foo$coords[4,] <- foo$coords[14,] <- foo$coords[24,] <- foo$coords[2,]
foo$coords[17,] <- foo$coords[23,] <- foo$coords[8,]
## locations the duplicated coordinates
dup.coords(foo)
}
\keyword{spatial}
\keyword{manip}
%\keyword{logic}

\eof
\name{elevation}
\docType{data}
\alias{elevation}
\title{Surface Elevations}
\description{
  Surface elevation data taken from Davis (1972).
  An onject of the class \code{geodata} with elevation values at 52 locations.
}
\usage{
data(topo)
}
\format{
  An object of the class \code{geodata} which is a list with the
  following elements:
  \describe{
    \item{\code{coords}}{x-y coordinates (multiples of 50 feet).}
    \item{\code{data}}{elevations (multiples of 10 feet).}
  }
}
\source{
  Davis, J.C. (1973) \emph{Statistics and Data Analysis in Geology.} Wiley.
}
\examples{
data(elevation)
summary(elevation)
plot(elevation)
}
\keyword{datasets}

\eof
\name{gambia}
\alias{gambia}
\alias{gambia.borders}
\alias{gambia.map}
\docType{data}
\title{Gambia Malaria Data}
\description{
  Malaria prevalence in children recorded at villages in The Gambia, Africa.
}
\usage{data(gambia)}
\format{
  Two objects are made available:
  \begin{enumerate}
  \item \code{gambia}\cr
  A data frame with 2035 observations on the following 8 variables.
  \describe{
    \item{x}{x-coordinate of the village (UTM).  }
    \item{y}{y-coordinate of the village (UTM).  }
    \item{pos}{presence (1) or absence (0) of malaria in a blood
    sample taken from the child.}
    \item{age}{age of the child, in days}
    \item{netuse}{indicator variable denoting whether (1) or not (0)
      the child regularly sleeps under a bed-net.  }
    \item{treated}{indicator variable denoting
     whether (1) or not (0) the bed-net is
      treated (coded 0 if netuse=0).}
    \item{green}{satellite-derived measure of the green-ness of vegetation
    in the immediate vicinity of the village (arbitrary units).  }
    \item{phc}{indicator variable denoting the presence (1) or
    absence (0) of a health center in the
      village.  }
  }
  \item \code{gambia.borders}\cr
  A data frame with 2 variables:
  \describe{
    \item{x}{x-coordinate of the country borders.  }
    \item{y}{y-coordinate of the country borders.  }
  }
 \end{enumerate}
}
%\details{
%  ~~ If necessary, more details than the __description__ above ~~
%}
%\source{
%  ~~ reference to a publication or URL from which the data were obtained ~~
%}
\references{
  Thomson, M., Connor, S., D Alessandro, U., Rowlingson, B., Diggle, P., Cresswell, M. & Greenwood, B.
(1999). Predicting malaria infection in Gambian children from satellite data and bednet use surveys: the
importance of spatial correlation in the interpretation of results. \emph{American Journal of Tropical
Medicine and Hygiene} 61: 2--8.

  Diggle, P., Moyeed, R., Rowlingson, B. & Thomson, M. (2002). Childhood
  malaria in The Gambia: a case-study in model-based geostatistics,
  \emph{Applied Statistics}.
}
\examples{
data(gambia)
plot(gambia.borders, type="l", asp=1)
points(gambia[,1:2], pch=19)
# a built-in function for a zoomed map
gambia.map()
# Building a "village-level" data frame
ind <- paste("x",gambia[,1], "y", gambia[,2], sep="")
village <- gambia[!duplicated(ind),c(1:2,7:8)]
village$prev <- as.vector(tapply(gambia$pos, ind, mean))
plot(village$green, village$prev)
}
\keyword{datasets}


\eof
\name{geoR-defunct}

\title{Defunct Functions in the Package geoR}

\alias{geoRdefunct}
\alias{olsfit}
\alias{wlsfit}
\alias{likfit.old}
\alias{likfit.nospatial}
\alias{loglik.spatial}
\alias{proflik.nug}
\alias{proflik.phi}
\alias{proflik.ftau}
\alias{distdiag}

\usage{
geoRdefunct()
}

\description{
  The functions listed here are no longer part of the package \pkg{geoR} as they are not needed (any more).
}

\details{
  
  The following functions are now defunct:
  
  \enumerate{
    
    \item{olsfit}{functionality incorporated by \code{\link{variofit}}.
      From \code{geoR_1.0-6}.  }
    
    \item{wlsfit}{functionality incorporated by \code{\link{variofit}}.
      From \code{geoR_1.0-6}.  }
    
    \item{likfit.old}{functionality incorporated by \code{\link{likfit}}.
      From \code{geoR_1.0-6}. The associated functions were also made
      defunct: \cr
      \code{likfit.nospatial}, \code{loglik.spatial}, \code{proflik.nug}, \code{proflik.phi}, \code{proflik.ftau}.  }

    \item{distdiag}{functionally is redundant with
      \code{\link[mva]{dist}}.  }
  }


}

\seealso{\code{\link{variofit}}}

\keyword{spatial}
\keyword{utilities}


\eof
\name{geoR-internal}

\alias{beta.sigmasq.post}
\alias{cond.sim}
\alias{print.betavar}
\alias{backtransform.moments}
\alias{BCtransform}
\alias{rMVnorm}
\alias{solve.geoR}
\alias{print.grf}
  
\title{geoR internal functions}

\description{
  These are functions internally called by other functions
  in the package \pkg{geoR} and not meant to be called by the user.
}

%\usage{
%ap()
%}
%%- maybe also `usage' for other objects documented here.
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\seealso{ ~~objects to SEE ALSO as \code{\link{~~fun~~}}, ~~~ }

\keyword{spatial}
\keyword{internal}
%\examples{}


\eof
\name{grf}

\alias{grf}
\alias{grf.aux2}
\alias{grf.aux1}
\alias{grfclass}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Simulation of Gaussian Random Fields}

\description{
  Generates simulations of Gaussian random fields for
  given covariance parameters.
}


\usage{
grf(n, grid = "irreg", nx, ny, xlims = c(0, 1), ylims = c(0, 1),
    nsim = 1, cov.model = "matern",
    cov.pars = stop("covariance parameters (sigmasq and phi) needed"), 
    kappa = 0.5, nugget = 0, lambda = 1, aniso.pars,
    method = c("cholesky", "svd", "eigen", "circular.embedding"),
    messages)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{n}{number of points (spatial locations) in each simulations.  }
  \item{grid}{optional. An \eqn{n \times 2}{n x 2} matrix with coordinates of the
    simulated data.  }
  \item{nx}{optional. Number of points in the X direction.  }
  \item{ny}{optional. Number of points in the Y direction.  }
  \item{xlims}{optional. Limits of the area in the X direction. Defaults
  to \eqn{[0,1]}.  }
  \item{ylims}{optional. Limits of the area in the Y direction.  Defaults
  to \eqn{[0,1]}.  }
  \item{nsim}{Number of simulations. Defaults to 1.}
  \item{cov.model}{correlation function. See \code{\link{cov.spatial}} for
    further details. Defaults to the 
    \emph{exponential} model.  }
  \item{cov.pars}{a vector with 2 elements or an \eqn{n \times 2}{n x 2}
    matrix with values of  the covariance parameters
    \eqn{\sigma^2}{sigma^2} (partial sill) and \eqn{\phi}{phi} (range
    parameter). If a vector, the elements are the values of
    \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}, respectively.
    If a matrix, corresponding to a model with several structures, the
    values of \eqn{\sigma^2}{sigma^2} 
    are in the first column and the values of \eqn{\phi}{phi} are in the second.  }
  \item{kappa}{additional smoothness parameter required only for the
    following correlation 
    functions: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"} 
     and  \code{"gneiting.matern"}. More details on the documentation for the
    function \code{\link{cov.spatial}}.  }
  \item{nugget}{the value of the nugget effect parameter \eqn{\tau^2}{tau^2}.}
  \item{lambda}{value of the Box-Cox transformation parameter. The value \eqn{\lambda
      = 1}{lambda = 1} corresponds to no transformation, the default.
    For any other value of \eqn{\lambda}{lambda} Gaussian data is
    simulated and then transformed.  }
  \item{aniso.pars}{geometric anisotropy parameters. By default an
    isotropic field is assumed and this argument is ignored.
    If a vector with 2 values is provided, with values for the
    anisotropy angle \eqn{\psi_A}{psi_A} (in
    radians) and 
    anisotropy ratio \eqn{\psi_A}{psi_A}, the coordinates
    are transformed, 
    the simulation is performed on the isotropic (transformed) space
    and then the coordinates are back-transformed such that the resulting
    field is anisotropic. Coordinates transformation is performed 
    by the function \code{\link{coords.aniso}}.  }
  \item{method}{simulation method. Defaults to the \emph{Cholesky}
    decomposition.  See section \code{DETAILS} below.  }
  \item{messages}{logical,  indicating
    whether or not status messages are printed on the screen (or output device)
    while the function is running. Defaults to \code{TRUE}.  } 
}


\details{
  For the methods \code{"cholesky"}, \code{"svd"} and \code{"eigen"} the
  simulation consists of multiplying a vector of standardized
  normal deviates by a square root of the covariance matrix.
  The square root of a matrix is not uniquely defined. The 
  three available methods differs in the way they compute the
  square root of the (positive definite) covariance matrix.

  For \code{method = "circular.embedding"} the algorithm implements
  the method described by Wood & Chan (1994) which is based on Fourier
  transforms.
  Only regular and equally spaced grids can be generated using this method.
  \cr
  The code for the \code{"circular.embedding"} method
  was provided by Martin Schlather, University of Bayreuth\cr
  (\url{http://btgyn8.geo.uni-bayreuth.de/~martin/}).
 
  \bold{WARNING:} The code for the \code{"circular.embedding"} method is
  no longer being maintained. Martin has released a
  package called \pkg{RandomFields} (available on \acronym{CRAN})
  for simulation of random fields.
  We strongly recommend the use of this package for simulations on fine
  grids with large number of locations. 

}


\value{
  A list with the components:
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix with the coordinates of the
    simulated data.  }
  \item{data}{a vector (if \code{nsim = 1}) or a matrix with the
    simulated values. For the latter each column corresponds to one
    simulation.  }
  \item{cov.model}{a string with the name of the correlation function.  }
  \item{nugget}{the value of the nugget parameter.}
  \item{cov.pars}{a vector with the values of \eqn{\sigma^2}{sigma^2}
    and \eqn{\phi}{phi}, respectively.  }
  \item{kappa}{value of the parameter \eqn{\kappa}{kappa}.  }
  \item{lambda}{value of the Box-Cox transformation parameter
    \eqn{\lambda}{lambda}.  }
  \item{aniso.pars}{a vector with values of the anisotropy parameters, if
    provided in the function call.  }
  \item{method}{a string with the name of the simulation method used.}
  \item{sim.dim}{a string "1d" or "2d" indicating the spatial dimension of the
    simulation.}
  \item{.Random.seed}{the random seed by the time the function was
    called.  }
  \item{messages}{messages produced by the function describing the
    simulation.  }
  \item{call}{the function call.  }
}


\references{
  Wood, A.T.A. and Chan, G. (1994) Simulation of stationary Gaussian
  process in \eqn{[0,1]^d}{[0,1]^d}.
  \emph{Journal of Computatinal and Graphical Statistics}, \bold{3}, 409--432.
  
  Schlather, M. (1999) \emph{Introduction to positive definite functions
    and to unconditional simulation of random fields}. Tech. Report
  ST--99--10, Dept Maths and Stats, Lancaster University.
  
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br},  \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


\seealso{\code{\link{plot.grf}} and \code{\link{image.grf}} for
  graphical output,
  \code{\link{coords.aniso}} for anisotropy coordinates transformation
  and, \code{\link{chol}},
  \code{\link{svd}} and \code{\link{eigen}} for methods of matrix decomposition.
}

\examples{
#
sim1 <- grf(100, cov.pars = c(1, .25))
# a display of simulated locations and values
points(sim1)   
# empirical and theoretical variograms
plot(sim1)
#
# a "smallish" simulation
sim2 <- grf(441, grid = "reg", cov.pars = c(1, .25)) 
image(sim2, xlab="X Coord", ylab="Y Coord")
#
# a "bigger" one
sim3 <- grf(40401, grid = "reg", cov.pars = c(10, .2), met = "circ") 
image(sim3, xlab="X Coord", ylab="Y Coord")
##
## 1-D simulations using the same seed and different noise/signal ratios
##
set.seed(234)
sim11 <- grf(100, ny=1, cov.pars=c(1, 0.25), nug=0)
set.seed(234)
sim12 <- grf(100, ny=1, cov.pars=c(0.75, 0.25), nug=0.25)
set.seed(234)
sim13 <- grf(100, ny=1, cov.pars=c(0.5, 0.25), nug=0.5)
##
par.ori <- par(no.readonly = TRUE)
par(mfrow=c(3,1), mar=c(3,3,.5,.5))
yl <- range(c(sim11$data, sim12$data, sim13$data))
image(sim11, type="l", ylim=yl)
image(sim12, type="l", ylim=yl)
image(sim13, type="l", ylim=yl)
par(par.ori)
}

\keyword{spatial}
\keyword{datagen}


\eof
\name{hist.krige.bayes}

\alias{hist.krige.bayes}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plots Sample from Posterior Distributions}

\description{
  Plots histograms and/or density estimation with samples
  from the posterior distribution of the model parameters
}

\usage{
\method{hist}{krige.bayes}(x, pars, density.est = TRUE, histogram = TRUE, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object of the class \code{krige.bayes}, with an output of
    the funtions \code{\link{krige.bayes}}. }
  \item{pars}{a vector with the names of one or more of the model parameters.
    Defaults to all model parameters. Setting to -1 excludes the intercept.  }
  \item{density.est}{logical indication whether a line with the density 
    estimation should be added to the plot.  }
  \item{histogram}{logical indicating whether the histogram is included in the 
    plot.  }
  \item{\dots}{further arguments for the plotting functions and or for the 
    density estimation.  }
}

%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  Produces a plot in the currently graphics device.\cr
  Returns a \code{\link{invisible}} list with the components:
  
  \item{histogram}{with the output of the function \code{\link{hist}} for 
    each parameter}
  \item{density.estimation}{with the output of the function 
    \code{\link{density}} for each parameter}
  
}


%\references{ ~put references to the literature/web site here ~ }

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{
  \code{\link{krige.bayes}}, \code{\link{hist}}, \code{\link{density}}.
}

\examples{
## See documentation for krige.bayes()
}

\keyword{spatial}
\keyword{dplot}

\eof
\name{hoef}
\docType{data}

\alias{hoef}

\title{Data for spatial analysis of experiments}
\usage{data(hoef)}
\description{
  The \code{hoef} data frame has 25 rows and 5 columns.\cr
  The data consists of a uniformity trial for which \emph{artificial}
  treatment effects were assign to the plots.
}

\format{
  This data frame contains the following columns:
  \describe{
    \item{x1}{x-coordinate of the plot.}
    \item{x2}{y-coordinate of the plot.}
    \item{dat}{the \emph{artificial} data.}
    \item{trat}{the treatment number.}
    \item{ut}{the data from the uniformity trial, without the treatment effect.}
  }
}

\details{
  The treatment effects assign to the plots are:
  \itemize{
    \item Treatment 1: \eqn{\tau_1 = 0}{tau1 = 0}
    \item Treatment 2: \eqn{\tau_2 = -3}{tau2 = -3}
    \item Treatment 3: \eqn{\tau_3 = -5}{tau3 = -5}
    \item Treatment 4: \eqn{\tau_4 = 6}{tau4 = 6}
    \item Treatment 5: \eqn{\tau_5 = 6}{tau5 = 6}
  }    
}

\source{
  Ver Hoef, J.M. & Cressie, N. (1993) Spatial statistics: analysis of field experiments.
  In Scheiner S.M. and Gurevitch, J. (Eds) \emph{Design and Analysis of
  Ecological Experiments}. Chapman and Hall. 
}
%\references{
%~~ possibly secondary sources and usages ~~
%}
\examples{
data(hoef)
hoef.geo <- as.geodata(hoef, covar.col=4)
summary(hoef)
summary(hoef.geo)
points(hoef.geo, cex.min=2, cex.max=2, pt.div="quintiles")
}
\keyword{datasets}

\eof
\name{image.grf}

\alias{image.grf}
\alias{persp.grf}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Image or Perspective Plot of Simulated Gaussian Random Field}

\description{
  Plots an image or perspective plot with a realization of a Gaussian
  random field, simulated using the function \code{\link{grf}}.  
}

\usage{
\method{image}{grf}(x, sim.number = 1, xlim, ylim, x.leg, y.leg, \dots)
\dots)

\method{persp}{grf}(x, sim.number = 1, \dots)
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{an object of the class \code{grf}, typically an output of the function
    \code{\link{grf}}.  }
  \item{sim.number}{simulation number.
    Indicates the number of the simulation top be plotted.
    Only valid if the object contains more than one simulation.
    Defaults to 1.  }
  \item{xlim, ylim} {the ranges to be encompassed by the x and y axes.  }
  \item{x.leg, y.leg}{limits for the legend in the horizontal and vertical directions.  }
%  \item{cex.leg}{controls the size of the numerical values in the legend.  }
%  \item{vertical}{logical, defining whether the legend is drawn in
%    horizontal (default) or vertical position.  }
  \item{\dots}{further arguments to be passed to the functions
    \code{\link{image}} or \code{\link{persp}}.  }
}

%\details{
%}

\value{
  An image or perspective plot is produced on the current graphics
  device.
  No values are returned.
}


\references{
  Further information about the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{grf}} for simulation of Gaussian random fields,
  \code{\link{image}} and \code{\link{persp}} for the generic plotting functions.}

\examples{
# generating 4 simulations of a Gaussian random field
sim <- grf(441, grid="reg", cov.pars=c(1, .25), nsim=4)
op <- par(no.readonly = TRUE)
par(mfrow=c(2,2), mar=c(3,3,1,1), mgp = c(2,1,0))
for (i in 1:4)
  image(sim, sim.n=i)
par(op)
}

\keyword{spatial}
\keyword{dplot}

\eof
\name{image.krige.bayes}

\alias{image.krige.bayes}
\alias{contour.krige.bayes}
\alias{persp.krige.bayes}
\alias{prepare.graph.krige.bayes}

\title{Plots Results of the Predictive Distribution}

\description{
  This function produces an image or perspective plot of a selected
  element
  of the predictive distribution
  returned by the function \code{\link{krige.bayes}}.
}

\usage{
\method{image}{krige.bayes}(x, locations, borders,
                  values.to.plot=c("mean", "variance",
                            "mean.simulations", "variance.simulations",
                            "quantiles", "probabilities", "simulation"),
                  number.col, coords.data, xlim, ylim, x.leg, y.leg,
                  messages, \dots)
\method{contour}{krige.bayes}(x, locations, borders, 
                  values.to.plot = c("mean", "variance",
                       "mean.simulations", "variance.simulations",
                       "quantiles", "probabilities", "simulation"),
                  number.col, coords.data, xlim, ylim,
                  x.leg, y.leg, messages, ...)
\method{persp}{krige.bayes}(x, locations, borders,
                  values.to.plot=c("mean", "variance",
                       "mean.simulations", "variance.simulations",
                       "quantiles", "probabilities", "simulation"),
                  number.col, messages, \dots)
}

\arguments{
  \item{x}{an object of the class \code{krige.bayes}, typically an
    output of the function \code{\link{krige.bayes}}.  }
  \item{locations}{an \eqn{n \times 2}{n x 2} matrix with the
    coordinates of the prediction locations, which should define a
    regular grid
    in order to be plotted by \code{\link{image}} or
    \code{\link{persp}}. By default does not need to be provided and
    evaluates the attribute "prediction.locations" from the input object.  }
  \item{borders}{an \eqn{n \times 2}{n x 2} matrix with the
    coordinates defining the borders of a region inside the grid defined
    by \code{locations}. Elements in the
    argument \code{values} are assigned to locations internal to the
    borders  and \code{NA}'s to the external ones.  }
  \item{values.to.plot}{select the element of the predictive
    distribution to be plotted. See DETAILS below.}
  \item{number.col}{Specifies the number of the column to be plotted.
    Only used if previous argument is set to one of \code{"quantiles"}, 
    \code{"probabilities"}  or \code{"simulation"}.  }
  \item{coords.data}{optional. If an \eqn{n \times 2}{n x 2}
    matrix with the data coordinates is
    provided, points indicating the data locations are included in the
    plot.  }
  \item{xlim, ylim} {the ranges to be encompassed by the x and y axes.  }
  \item{x.leg, y.leg}{limits for the legend in the horizontal and vertical directions.  }
%  \item{cex.leg}{controls the size of the numerical values in the legend.  }
%  \item{vertical}{logical, defining whether the legend is drawn in
%    horizontal (default) or vertical position.  }
  \item{messages}{logical, if TRUE status messages are printed while
    running the function.  }
  \item{\dots}{extra arguments to be passed to the plotting function
    \code{\link{image}} or \code{\link{persp}}.  }
}

\details{
  The function \code{\link{krige.bayes}} returns
  summaries and other results about the predictive distributions.
  The argument \code{values.to.plot} specifies which result will be
  plotted. It can be passed to the function in two different forms:
  \itemize{
    \item a vector with the object containing the values to be plotted, or
    \item one of the following options: \code{"moments.mean"},
    \code{"moments.variance"}, 
    \code{"mean.simulations"},
    \code{"variance.simulations"},
    \code{"quantiles"},
    \code{"probability"} or
    \code{"simulation"}.  }
  For the last three options, if the results are stored in matrices,
  a column number must be provided using the argument \code{number.col}.

  The documentation for the function \code{\link{krige.bayes}} provides
  further details about these options.
}

\value{
  An \code{\link{image}} or \code{\link{persp}} plot is produced on the
  current graphics device. No values are returned. 
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br},
  \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{krige.bayes}} for Bayesian Kriging computations and, \code{\link{image}} and  \code{\link{persp}} for the generic plotting functions.}

\examples{
#See examples in the documentation for the function krige.bayes().
}

\keyword{spatial}


\eof
\name{image.kriging}

\alias{image.kriging}
\alias{persp.kriging}
\alias{contour.kriging}
\alias{prepare.graph.kriging}
\alias{plot.1d}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Image or Perspective Plot with Kriging Results}

\description{
  Plots image or perspective plots with results of the kriging
  calculations.
}

\usage{
\method{image}{kriging}(x, locations, borders, values = x$predict,
              coords.data, xlim, ylim, x.leg, y.leg, \dots)
\method{contour}{kriging}(x, locations, borders, values = x$predict,
              coords.data, xlim, ylim, filled=FALSE, \dots)
\method{persp}{kriging}(x, locations, borders, values = x$predict, \dots)
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{an object of the  class \code{kriging}, typically
    with the output of the functions \code{\link{krige.conv}} or \code{\link{ksline}}.  }
  \item{locations}{an \eqn{n \times 2}{n x 2} matrix with the
    coordinates of the prediction locations, which should define a
    regular grid
    in order to be plotted by \code{\link{image}} or
    \code{\link{persp}}. By default does not need to be provided and
    evaluates the attribute "prediction.locations" from the input object. }
  \item{borders}{an \eqn{n \times 2}{n x 2} matrix with the
    coordinates defining the borders of a region inside the grid defined
    by \code{locations}. Elements in the
    argument \code{values} are assigned to locations internal to the
    borders and \code{NA}'s to the external ones.
  }
  \item{values}{a vector with values to be plotted. Defaults to \code{obj$predict}.  }
  \item{coords.data}{optional. If an \eqn{n \times 2}{n x 2}
    matrix with the data coordinates is
    provided, points indicating the data locations are included in the
    plot.  }
  \item{xlim, ylim} {the range to be encompassed by the x and y axes.  }
  \item{x.leg, y.leg}{limits for the legend in the horizontal and vertical directions.  }
  \item{filled}{logical. If \code{FALSE} the function
    \code{\link{contour}} is used otherwise
    \code{\link{filled.contour}}. Defaults to \code{FALSE}.  }
  \item{\dots}{further arguments to be passed to the functions
    \code{\link{image}}, \code{\link{contour}}, \code{\link{filled.contour}}, \code{\link{persp}} or
    \code{\link{legend.krige}}.
    For instance, the
    argument \code{zlim} can be used to set the the minimum and maximum
    `z' values for which colors should be plotted.
    See documentation for those function for
    possible arguments.
  }   
}

\details{
  \code{plot1d} and \code{prepare.graph.kriging} are auxiliary functions
  called by the others.
}

\value{
  An image or perspective plot is produced o the current graphics
  device.
  No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


%\note{ ~~further notes~~ }

 
\seealso{\code{\link{krige.conv}} and \code{\link{ksline}} for kriging
  calculations. Documentation for
  \code{\link{image}}, \code{\link{contour}}, \code{\link{filled.contour}} and  \code{\link{persp}} contain basic information
  on the plotting functions.}

\examples{
data(s100) 
loci <- expand.grid(seq(0,1,l=31), seq(0,1,l=31))
kc <- krige.conv(s100, loc=loci,
                 krige=krige.control(cov.pars=c(1, .25)))
image(kc)
contour(kc)
persp(kc, theta=20, phi=20)
#
# now dealing with borders
#
bor <- matrix(c(.4,.1,.3,.9,.9,.7,.9,.7,.3,.2,.5,.8),
              ncol=2)
# plotting just inside borders
image(kc, borders=bor)
contour(kc, borders=bor)
# kriging just inside borders
kc1 <- krige.conv(s100, loc=loci,
                 krige=krige.control(cov.pars=c(1, .25)),
                 borders=bor)
image(kc1, loc=loci)
contour(kc1, loc=loci)
# avoiding the borders to be drawn
image(kc1, loc=loci, borders=NULL)
contour(kc1, loc=loci, borders=NULL)

op <- par(no.readonly = TRUE)
par(mfrow=c(1,2), mar=c(3,3,0,0), mgp=c(1.5, .8,0))
image(kc)
image(kc, val=sqrt(kc$krige.var))

# notice different ways to add the legends and pass arguments:
image(kc, ylim=c(-0.2, 1), x.leg=c(0,1), y.leg=c(-0.2, -0.1))
image(kc, val=kc$krige.var, ylim=c(-0.2, 1))
legend.krige(y.leg=c(-0.2,-0.1), x.leg=c(0,1), val=sqrt(kc$krige.var))

image(kc, loc=loci, ylim=c(-0.2, 1), x.leg=c(0,1), y.leg=c(-0.2, -0.1), cex=1.5)
image(kc, loc=loci, ylim=c(-0.2, 1), x.leg=c(0,1), y.leg=c(-0.2, -0.1))
image(kc, loc=loci, ylim=c(-0.2, 1), x.leg=c(0,1), y.leg=c(-0.2, -0.1), offset.leg=0.5)

image(kc, loc=loci, xlim=c(0, 1.2))
legend.krige(x.leg=c(1.05,1.1), y.leg=c(0,1),kc$pred, vert=TRUE)
image(kc, loc=loci, xlim=c(0, 1.2))
legend.krige(x.leg=c(1.05,1.1), y.leg=c(0,1),kc$pred, vert=TRUE, off=1.5, cex=1.5)

par(op)
}

\keyword{spatial}
\keyword{dplot}

\eof
\name{isaaks}
\docType{data}

\alias{isaaks}

\title{Data from Isaaks and Srisvastava's book}
\description{
  Toy example used in the book \emph{An Introduction to Geostatistics} to
  illustrate the effects of different models
  and parameters in the kriging results when predicting at a given point.
}
\usage{data(isaaks)}
\format{
  An object of the class \code{geodata} which is a list with the elements:
  \describe{
    \item{coords}{coordinates of the data location. }
    \item{data}{the data vector. }
    \item{x0}{coordinate of the prediction point. }
  }
}
%\details{
%  ~~ If necessary, more details than the __description__ above ~~
%}
\source{
  Isaaks, E.H. & Srisvastava, R.M. (1989)
  An Introduction to Applied Geostatistics.
  Oxford University Press.
}

%\references{
%  ~~ possibly secondary sources and usages ~~
%}
\examples{
data(isaaks)
isaaks
summary(isaaks)
plot(isaaks$coords, asp=1, type="n")
text(isaaks$coords, as.character(isaaks$data))
points(isaaks$x0, pch="?", cex=2, col=2)
}
\keyword{datasets}

\eof
\name{krige.bayes}

\alias{krige.bayes}
\alias{model.control}
\alias{prior.control}
\alias{post2prior}
\alias{print.krige.bayes}
\alias{print.posterior.krige.bayes}

\title{Bayesian Analysis for Gaussian Geostatistical Models}

\description{
  The function \code{krige.bayes} performs Bayesian analysis of
  geostatistical data allowing specifications of
  different levels of uncertainty in the model parameters.\cr
  It returns results on the posterior distributions for the model
  parameters and on the predictive distributions for prediction
  locations (if provided).
}

\usage{
krige.bayes(geodata, coords = geodata$coords, data = geodata$data,
            locations = "no", borders = NULL, model, prior, output)

model.control(trend.d = "cte", trend.l = "cte", cov.model = "matern",
              kappa = 0.5, aniso.pars = NULL, lambda = 1)

prior.control(beta.prior = c("flat", "normal", "fixed"),
              beta = NULL, beta.var.std = NULL,
              sigmasq.prior = c("reciprocal", "uniform",
                                "sc.inv.chisq", "fixed"),
              sigmasq = NULL, df.sigmasq = NULL,
              phi.prior = c("uniform", "exponential","fixed",
                            "squared.reciprocal", "reciprocal"),
              phi = NULL, phi.discrete = NULL,
              tausq.rel.prior = c("fixed", "uniform"),
              tausq.rel = 0, tausq.rel.discrete = NULL)

post2prior(obj)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix where each row has the 2-D
    coordinates of the \eqn{n} data locations.
    By default it takes the
    component \code{coords} of the argument \code{geodata}, if provided.
  }
  \item{data}{a vector with \emph{n} data values. By default it takes the
    component \code{data} of the argument \code{geodata}, if provided.  }
  \item{locations}{an \eqn{N \times 2}{N x 2} matrix or data-frame with the 2-D
    coordinates of the \eqn{N} prediction locations.
    Defaults to \code{"no"} in which case
    the function returns only results on the posterior distributions of
    the model parameters.  }
  \item{borders}{optional. If a two column matrix defining a polygon is
    provided the prediction is performed only at locations inside this polygon. }
  \item{model}{a list defining the fixed components of the model.
    It can take an output to a call to \code{model.control} or
    a list with elements as for the arguments in \code{model.control}.
    Default values are assumed for arguments not provided.
    See section DETAILS below.  }
  \item{prior}{a list with the specification of priors for the model
    parameters.
    It can take an output to a call to \code{prior.control} or
    a list with elements as for the arguments in \code{prior.control}.
    Default values are assumed for arguments not provided.
    See section DETAILS below.  }
  \item{output}{a list specifying output options.
    It can take an output to a call to \code{output.control} or
    a list with elements as for the arguments in \code{output.control}.
    Default values are assumed for arguments not provided.
    See documentation for
    \code{\link{output.control}} for further details.  }
  
  \item{trend.d}{specifies the trend (covariates) values at the data
    locations. See documentation
    of \code{\link{trend.spatial}} for further details.
    Defaults to \code{"cte"}.  }  
  \item{trend.l}{specifies the trend (covariates) at the prediction
    locations. Must be of the same type as defined for \code{trend.d}.
    Only used if prediction locations are provided in the argument
    \code{locations}.  }
  \item{cov.model}{string indicating the name of the model for the correlation function. Further details in the
    documentation for \code{\link{cov.spatial}}. }
  
  \item{kappa}{additional smoothness parameter. Only used if the
    correlation function is one of: \code{"matern"},
    \code{"powered.exponential"}, \code{"cauchy"} or
    \code{"gneiting.matern"}. In the current implementation this
    parameter is always regarded as fixed during the Bayesian analysis.  }
  
  \item{aniso.pars}{fixed parameters for geometric anisotropy
    correction. If \code{aniso.pars = FALSE} no correction is made, otherwise
    a two elements vector with values for the anisotropy parameters
    must be provided. Anisotropy correction consists of a
    transformation of the data and prediction coordinates performed
    by the function \code{\link{coords.aniso}}.  }
  
  \item{lambda}{numerical value of the Box-Cox transformation parameter.
    The value \eqn{\lambda = 1}{lambda = 1} corresponds to
    no transformation. The Box-Cox parameter \eqn{\lambda}{lambda}
    is always regarded as fixed and
    data transformation is 
    performed before the analysis.
    Prediction results are back-transformed and
    returned is the same scale as for the original data.
    For \eqn{\lambda = 0}{lambda = 0} the log-transformation is performed.
    If \eqn{\lambda < 0}{lambda < 0} the mean predictor doesn't make sense
    (the resulting distribution has no expectation).  }
  
  \item{beta.prior}{prior distribution for the mean (vector)
    parameter \eqn{\beta}{beta}. The options are "flat" (default),
    "normal" or "fixed" (known mean).  }
  
  \item{beta}{mean hyperparameter for the distribution of the mean (vector) parameter        \eqn{\beta}{beta}. Only used if \code{beta.prior = "normal"} or
    \code{beta.prior = "fixed"}. For the later \code{beta} defines the value of
    the known mean.  }
  
  \item{beta.var.std}{standardised (co)variance hyperparameter(s)
    for the prior for the mean
    (vector) parameter \eqn{\beta}{beta}.
    The (co)variance matrix for\eqn{\beta}{beta} is given by the
    multiplication of this matrix by \eqn{\sigma^2}{sigma^2}.
    Only used if \code{beta.prior = "normal"}.  }
  
  \item{sigmasq.prior}{specifies the prior for the parameter
    \eqn{\sigma^2}{sigma^2}. If \code{"reciprocal"} (the default), the prior 
    \eqn{\frac{1}{\sigma^2}}{1/sigma^2} is used. Otherwise the
    parameter is regarded as fixed.  }
  
  \item{sigmasq}{fixed value of the sill parameter
    \eqn{\sigma^2}{sigma^2}. Only used if 
    \code{sigmasq.prior = FALSE}.  }
  \item{df.sigmasq}{numerical. Number of degrees of freedom for the
    prior for the parameter \eqn{\sigma^2}{sigma^2}. Only used if
    \code{sigmasq.prior = "sc.inv.chisq"}.  }
  \item{phi.prior}{prior distribution for the range parameter
    \eqn{\phi}{phi}.
    Options are: \code{"uniform"}, \code{"exponential"},
    \code{"reciprocal"} , \code{"squared.reciprocal"} and
    \code{"fixed"}.
    Alternativelly, a user defined discrete distribution can be
    specified. In this case the argument takes a vector of numerical
    values of probabilities with corresponding support points
    provided in the argument \code{phi.discrete}. \cr     
    If \code{"fixed"} the argument \eqn{\phi}{phi}
    must be provided and is regarded as fixed when performing predictions.\cr
    For the exponential prior the argument \code{phi} provides
    the value for the hyperparameter \eqn{\nu}{nu} which corresponds to the
    expected value for this distribution.
  }
  
  \item{phi}{fixed value of the range parameter \eqn{\phi}{phi}. Only needed if
    \code{phi.prior = "fixed"}.  }
  
  \item{phi.discrete}{support points of the discrete prior
    for the range parameter \eqn{\phi}{phi}. The default is a sequence
    of 51 values between 0 and 2 times the maximum distance between the
    data locations.}
  
  \item{tausq.rel.prior}{specifies a prior distribution for the
    relative nugget parameter
    \eqn{\frac{\tau^2}{\sigma^2}}{tau^2/sigma^2}.
    If \code{tausq.rel.prior = "fixed"} the relative nugget is
    considered known (fixed) with value given by the argument
    \code{tausq.rel}.
    If \code{tausq.rel.prior = "uniform"} a discrete uniform prior is used
    with support points given by the argument
    \code{tausq.rel.discrete}.
    Alternativelly, a user defined discrete distribution can be
    specified. In this case the argument takes the a vector of
    probabilities of a discrete distribution and the support points
    should be provided in the argument \code{tausq.rel.discrete}.      
  }
  
  \item{tausq.rel}{fixed value for the relative nugget parameter.
    Only used if
    \code{tausq.rel.prior = "fixed"}.  }
  
  \item{tausq.rel.discrete}{support points of the discrete prior
    for the relative nugget parameter \eqn{\frac{\tau^2}{\sigma^2}}{tau^2/sigma^2}.  }

  \item{obj}{an object of the class \code{krige.bayes} or
    \code{posterior.krige.bayes} with the output of a call to
    \code{krige.bayes}. The function \code{post2prior} takes the
    posterior distribution computed by one call to \code{krige.bayes}
    and prepares it to be used a a prior in a subsequent call. Notice
    that in this case the function \code{post2prior} is used instead
    of \code{prior.control}.  }
}

\details{
  \code{krige.bayes} is a generic function for Bayesian geostatistical 
  analysis of (transformed) Gaussian where predictions take into account the  parameter
  uncertainty.
  
  It can be set to run conventional kriging methods which
  use known parameters or \emph{plug-in} estimates. However, the
  functions \code{krige.conv} and \code{ksline} are preferable for
  prediction with fixed parameters.\cr

  \bold{PRIOR SPECIFICATION}
  
  The basis of the Bayesian algorithm is the discretisation of the prior
  distribution for the parameters \eqn{\phi}{phi} and \eqn{\tau^2_{rel}
    =\frac{\tau^2}{\sigma^2}}{tau_rel = tau/sigma}.
  The Tech. Report (see \code{References} below)
  provides details on the results used in the current implementation.\cr
  The expressions of the implemented priors for the parameter \eqn{\phi}{phi}
  are:
  \describe{
    \item{"uniform": }{\eqn{p(\phi) \propto 1}{p(phi) propto 1}. }
    \item{"exponential": }{\eqn{p(\phi) = \frac{1}{\nu}
	\exp(-\frac{1}{\nu} * \phi)}{(-1/nu) exp(- (1/nu) * phi)}.  }
    \item{"reciprocal": }{\eqn{p(\phi) \propto \frac{1}{\phi}}{p(\phi) propto 1/phi}.  }
    \item{"squared.reciprocal": }{\eqn{p(\phi) \propto
	\frac{1}{\phi^2}}{p(\phi) propto 1/phi^2}.  }
    \item{"fixed": }{fixed known or estimated value of \eqn{\phi}{phi}.  }
  }

  We remember that apart from those a \emph{user defined} prior can be specifyed  by
  entering a vector of probabilities for a discrete distribution 
  with suport points given by the argument \code{phi.discrete}.\cr

  
  \bold{CONTROL FUNCTIONS}
  
  The function call includes auxiliary control functions which allows
  the user to specify and/or change the specification of model
  components
  (using \code{model.control}), prior
  distributions (using \code{prior.control}) and 
  output options (using \code{output.control}).
  Default options are available in most of the cases.
  
  
}


\value{An object with \code{\link{class}} \code{"krige.bayes"} and
  \code{"kriging"}.
  The attribute \code{prediction.locations} containing the name of the
  object with the coordinates of the prediction locations (argument
  \code{locations}) is assigned to the object.
  Returns a list with the following components:
  
  \item{posterior}{results on on the posterior distribution of the
    model parameters. A list with the following possible components:\cr  }
  
  \itemize{   
    \item{beta}{summary information on the posterior distribution
      of the mean parameter \eqn{\beta}{beta}.  }
    \item{sigmasq}{summary information on the posterior distribution
      of the variance parameter \eqn{\sigma^2}{sigma^2} (partial sill).  }
    \item{phi}{summary information on the posterior distribution
      of the correlation parameter \eqn{\phi}{phi} (range parameter) .  }
    \item{tausq.rel}{summary information on the posterior distribution
      of the relative nugget variance parameter
      \eqn{\tau^2_{rel}}{tau^2_rel}.  }
    \item{joint.phi.tausq.rel}{information on discrete the joint
      distribution of these parameters.  }
    \item{sample}{a data.frame with a sample from the posterior
      distribution.
      Each column corresponds to one of the basic model parameters.  }
    \cr
    \cr
  }
  
  \cr
  \cr
  
  \item{predictive}{results on the predictive distribution at the
    prediction locations, if provided.
    A list with the following possible components:\cr  }
  
  \itemize{    
    \item{mean}{expected values.  }
    \item{variance}{expected variance.  }
    \item{distribution}{type of posterior distribution.  }
    \item{mean.simulations}{mean of the simulations at each locations.  }
    \item{variance.simulations}{variance of the simulations at each locations.  }
    \item{quantiles.simulations}{quantiles computed from the the simulations.  }
    \item{probabilities.simulations}{probabilities computed from the simulations.  }
    \item{simulations}{simulations from the predictive distribution.  }
  }
  
  \cr
  \cr
  
  \item{prior}{a list with information on the prior distribution
    and hyper-parameters of the model parameters (\eqn{\beta, \sigma^2,
      \phi, \tau^2_{rel}}{beta, sigma^2, phi, tau^2_rel}).
  }
  
  \item{model}{model specification as defined by \code{model.control}.  }
  
  \item{.Random.seed}{system random seed before running the function.
    Allows reproduction of results. If
    the \code{\link{.Random.seed}} is set to this value and the function is run
    again, it will produce exactly the same results.  }
  
  \item{max.dist}{maximum distance found between two data locations.  }
  \item{call}{the function call.  }
  
}


\references{
  Diggle, P.J. \& Ribeiro Jr, P.J. (2002) Bayesian inference in
  Gaussian model-based geostatistics. Geographical and Environmental
  Modelling, Vol. 6, No. 2, 129-146.
  
  The technical details about the  implementation of \code{krige.bayes} can be
  found at:
  \cr
  Ribeiro, P.J. Jr. and Diggle, P.J. (1999) \emph{Bayesian inference in
    Gaussian model-based geostatistics}. Tech. Report ST-99-08, Dept
  Maths and Stats, Lancaster University.\cr
  Available at: \cr
  \url{http://www.maths.lancs.ac.uk/~ribeiro/publications.html}

  Further information about \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.  

  For a extended list of examples of the usage
  see \url{http://www.est.ufpr.br/geoR/geoRdoc/examples.krige.bayes.R}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


\seealso{\code{\link{lines.variomodel.krige.bayes}},
  \code{\link{plot.krige.bayes}} for outputs related to the
  parameters in the model,
  \code{\link{image.krige.bayes}} and
  \code{\link{persp.krige.bayes}} for graphical output of
  prediction results.  
  \code{\link{krige.conv}} and
  \code{\link{ksline}} for conventional kriging methods.
}


\examples{
\dontrun{
# generating a simulated data-set
ex.data <- grf(50, cov.pars=c(10, .25))
#
# defining the grid of prediction locations:
ex.grid <- as.matrix(expand.grid(seq(0,1,l=21), seq(0,1,l=21)))
#
# computing posterior and predictive distributions
# (warning: the next command can be time demanding)
ex.bayes <- krige.bayes(ex.data, loc=ex.grid, prior =
                 prior.control(phi.discrete=seq(0, 2, l=21)))
#
# Prior and posterior for the parameter phi
plot(ex.bayes, type="h", tausq.rel = FALSE, col=c("red", "blue"))
#
# Plot histograms with samples from the posterior
par(mfrow=c(3,1))
hist(ex.bayes)
par(mfrow=c(1,1))

# Plotting empirical variograms and some Bayesian estimates:
# Empirical variogram
plot(variog(ex.data, max.dist = 1), ylim=c(0, 15))
# Since ex.data is a simulated data we can plot the line with the "true" model 
lines(ex.data)
# adding lines with summaries of the posterior of the binned variogram
lines(ex.bayes, summ = mean, lwd=1, lty=2)
lines(ex.bayes, summ = median, lwd=2, lty=2)
# adding line with summary of the posterior of the parameters
lines(ex.bayes, summary = "mode", post = "parameters")

# Plotting again the empirical variogram
plot(variog(ex.data, max.dist=1), ylim=c(0, 15))
# and adding lines with median and quantiles estimates
my.summary <- function(x){quantile(x, prob = c(0.05, 0.5, 0.95))}
lines(ex.bayes, summ = my.summary, ty="l", lty=c(2,1,2), col=1)

# Plotting some prediction results
op <- par(no.readonly = TRUE)
par(mfrow=c(2,2))
par(mar=c(3,3,1,1))
par(mgp = c(2,1,0))
image(ex.bayes, main="predicted values")
image(ex.bayes, val="variance", main="prediction variance")
image(ex.bayes, val= "simulation", number.col=1,
      main="a simulation from the \npredictive distribution")
image(ex.bayes, val= "simulation", number.col=2,
      main="another simulation from \nthe predictive distribution")
#
par(op)
}
##
## For a extended list of exemples of the usage of krige.bayes()
## see http://www.est.ufpr.br/geoR/geoRdoc/examples.krige.bayes.R
##

}
\keyword{spatial}
\keyword{models}


\eof
\name{krige.conv}

\alias{krige.conv}
\alias{krige.control}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Spatial Prediction -- Conventional Kriging}


\description{
  This function performs spatial prediction for fixed covariance
  parameters using global neighbourhood.
  
  Options available implement the following types of kriging:
  \emph{SK} (simple kriging), \emph{OK} (ordinary kriging),
  \emph{KTE} (external trend kriging) and \emph{UK} (universal kriging). 
  
}

\usage{
krige.conv(geodata, coords=geodata$coords, data=geodata$data,
            locations, borders = NULL, krige, output)

krige.control(type.krige = "ok", trend.d = "cte", trend.l = "cte",
            obj.model = NULL, beta, cov.model, cov.pars, kappa,
            nugget, micro.scale = 0, dist.epsilon = 1e-10, 
            aniso.pars, lambda)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. Typically an object of the \code{\link{class}}
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix or data-frame with the 2-D
    coordinates of the \eqn{n} data locations.
    By default it takes the
    component \code{coords} of the argument \code{geodata}, if provided.  }
  \item{data}{a vector with \emph{n} data values. By default it takes the
    component \code{data} of the argument \code{geodata}, if provided.  }
  \item{locations}{an \eqn{N \times 2}{N x 2} matrix or data-frame with the 2-D
    coordinates of the \eqn{N} prediction locations.  }
  \item{borders}{optional. If a two column matrix defining a polygon is
    provided the prediction is performed only at locations inside this polygon. }
  \item{krige}{a list defining the model components and the type of
    kriging. It can take an output to a call to \code{krige.control} or
    a list with elements as for the arguments in \code{krige.control}.
    Default values are assumed for arguments  or list elements not provided.
    See the description of arguments in `krige.control' below.  }
  \item{output}{a list specifying output options.
    It can take an output to a call to \code{output.control} or
    a list with elements as for the arguments in \code{output.control}.
    Default values are assumed for arguments not provided.
    See documentation for
    \code{\link{output.control}} for further details.  }
  \item{type.krige}{type of kriging to be performed. Options are
    \code{"SK", "OK"} corresponding to simple or ordinary
    kriging. Kriging with external trend and universal kriging can be
    defined setting \code{type.krige = "OK"} and specifying the
    trend model using the arguments \code{trend.d} and
    \code{trend.l}.  } 
  \item{trend.d}{specifies the trend (covariate) values at the data
    locations.
    See documentation of \code{\link{trend.spatial}} for
    further details.
    Defaults to \code{"cte"}.   }
  \item{trend.l}{specifies the trend (covariate) values at prediction
    locations. It must be of the same type as for \code{trend.d}.
    Only used if prediction locations are provided in the argument
    \code{locations}.  }
  \item{obj.model}{a list with the model parameters. Typically an
    output of \code{\link{likfit}} or
    \code{\link{variofit}}.  }
  \item{beta}{numerical value of the mean (vector) parameter.
    Only used if \code{type.krige="SK"}.}  
  \item{cov.model}{string indicating the name of the model for the
    correlation function. Further details  can be found in the
    documentation of the function
    \code{\link{cov.spatial}}.  }
  \item{cov.pars}{a 2 elements vector with values of the covariance parameters \eqn{\sigma^2}{sigma^2}
    (partial sill)  and \eqn{\phi}{phi} (range parameter),
    respectively.  }
 %  \item{cov.pars}{a vector with 2 elements or an \eqn{n \times 2}{n x
%      2} matrix with the covariance parameters \eqn{\sigma^2}{sigma^2}
%    (partial sill)  and \eqn{\phi}{phi} (range parameter). If a vector, the elements are the values of
%    \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}, respectively.
%    If a matrix, corresponding to a model with several structures, the
%    values of \eqn{\sigma^2}{sigma^2} 
%    are in the first column and the values of \eqn{\phi}{phi} are in the second.  }
  \item{kappa}{additional smoothness parameter required by the following correlation
    functions: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"} and
    \code{"gneiting.matern"}.   }
  \item{nugget}{the value of the nugget variance parameter \eqn{\tau^2}{tau^2}. Defaults to zero.  }
  \item{micro.scale}{micro-scale variance. If different from zero, the
    nugget variance is divided into 2 terms: \emph{micro-scale variance}
    and \emph{measurement error}. This affect the precision of the predictions. 
    Often in practice, these two variance components are indistinguishable but the
    distinction can be made here if justifiable. See the section
    \code{DETAILS} in the documentation of \code{\link{output.control}}. } 
  \item{dist.epsilon}{a numeric value. Locations which are separated
    by a distance less than this value
    are considered co-located.  }
  \item{aniso.pars}{parameters for geometric anisotropy
    correction. If \code{aniso.pars = FALSE} no correction is made, otherwise
    a two elements vector with values for the anisotropy parameters
    must be provided. Anisotropy correction consists of a
    transformation of the data and prediction coordinates performed
    by the function \code{\link{coords.aniso}}.  }
  \item{lambda}{numeric value of the Box-Cox transformation parameter.
    The value \eqn{\lambda = 1}{lambda = 1} corresponds to
    no transformation and \eqn{\lambda = 0}{lambda = 0} corresponds to
    the log-transformation. 
    Prediction results are back-transformed and
    returned is the same scale as for the original data.  }
}

\details{
  According to the arguments provided,  one of the following different
  types of kriging: \emph{SK}, \emph{OK}, \emph{UK} or  \emph{KTE}
  is performed.
  Defaults correspond to ordinary kriging.

}


\value{
  An object of the \code{\link{class}} \code{kriging}.
  The attribute \code{prediction.locations} containing the name of the
  object with the coordinates of the prediction locations (argument
  \code{locations}) is assigned to the object.
  Returns  a  list with the following components:
  
  \item{predict}{a vector with predicted values.  }
  \item{krige.var}{a vector with predicted variances.  }
  \item{beta.est}{estimates of the \eqn{\beta}{beta}, parameter
    implicit in kriging procedure. Not included if
    \code{type.krige = "SK"}.  }
  \item{simulations}{an \eqn{ni \times n.sim}{ni x n.sim} matrix where \eqn{ni} is the
    number of prediction locations. Each column corresponds to a
    conditional simulation of the predictive distribution.
    Only returned if \code{n.sim > 0}.  }
  \item{message}{messages about the type of prediction performed.  }
  \item{call}{the function call.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

%\note{}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{output.control}} sets output options,
\code{\link{image.kriging}} and \code{\link{persp.kriging}}
  for graphical output of the results,
  \code{\link{krige.bayes}} for Bayesian prediction and \code{\link{ksline}}
  for a different implementation of kriging allowing for moving
  neighborhood. For model fitting see \code{\link{likfit}} or \code{\link{variofit}}.
}


\examples{
\dontrun{
data(s100)
# Defining a prediction grid
loci <- expand.grid(seq(0,1,l=21), seq(0,1,l=21))
# predicting by ordinary kriging
kc <- krige.conv(s100, loc=loci,
                 krige=krige.control(cov.pars=c(1, .25)))
# mapping point estimates and variances
par.ori <- par(no.readonly = TRUE)
par(mfrow=c(1,2), mar=c(3.5,3.5,1,0), mgp=c(1.5,.5,0))
image(kc, main="kriging estimates")
image(kc, val=sqrt(kc$krige.var), main="kriging std. errors")
# Now setting the output to simulate from the predictive
# (obtaining conditional simulations),
# and to compute quantile and probability estimators
s.out <- output.control(n.predictive = 1000, quant=0.9, thres=2)
set.seed(123)
kc <- krige.conv(s100, loc = loci,
         krige = krige.control(cov.pars = c(1,.25)),
         output = s.out)
par(mfrow=c(2,2))
image(kc, val=kc$sim[,1], main="a cond. simul.")
image(kc, val=kc$sim[,1], main="another cond. simul.")
image(kc, val=(1 - kc$prob), main="Map of P(Y > 2)")
image(kc, val=kc$quant, main="Map of y s.t. P(Y < y) = 0.9")
par(par.ori)
}
}
\keyword{spatial}


\eof
\name{ksline}

\alias{ksline}
\alias{ksline.aux.1}

\title{Spatial Prediction -- Conventional Kriging}

\description{
  This function performs spatial prediction for given covariance
  parameters. Options implement
  the following kriging types:
  \emph{SK} (simple kriging), \emph{OK} (ordinary kriging),
  \emph{KTE} (external trend kriging) and \emph{UK} (universal kriging).
  
  The function \code{\link{krige.conv}} should be preferred, unless
  moving neighborhood is to be used.
}

\usage{
ksline(geodata, coords = geodata$coords, data = geodata$data,
       locations, borders = NULL, 
       cov.model = "matern",
       cov.pars=stop("covariance parameters (sigmasq and phi) needed"),
       kappa = 0.5, nugget = 0, micro.scale = 0,
       lambda = 1, m0 = "ok", nwin = "full",
       n.samples.backtransform = 500, trend = 1, d = 2,
       ktedata = NULL, ktelocations = NULL, aniso.pars = NULL,
       signal = FALSE, dist.epsilon = 1e-10, messages)
}


\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. Typically an object of the \code{\link{class}}
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix where each row has the 2-D
    coordinates of the \eqn{n} data locations.
    By default it takes the
    component \code{coords} of the argument \code{geodata}, if
    provided.  }
  \item{data}{a vector with \emph{n} data values. By default it takes the
    component \code{data} of the argument \code{geodata}, if provided.  }
  \item{locations}{an \eqn{N \times 2}{N x 2} matrix or data-frame with the 2-D
    coordinates of the \eqn{N} prediction locations. }
  \item{borders}{optional. If a two column matrix defining a polygon is
    provided the prediction is performed only at locations inside this polygon. }
  \item{cov.pars}{a vector with 2 elements or an \eqn{n \times 2}{n x
      2} matrix with the covariance parameters \eqn{\sigma^2}{sigma^2}
    (partial sill)  and \eqn{\phi}{phi} (range parameter). If a vector, the elements are the values of
    \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}, respectively.
    If a matrix, corresponding to a model with several structures, the
    values of \eqn{\sigma^2}{sigma^2} 
    are in the first column and the values of \eqn{\phi}{phi} are in the second.  }
  \item{nugget}{the value of the nugget variance parameter \eqn{\tau^2}{tau^2}. Defaults to zero.  }
  \item{micro.scale}{micro-scale variance. If different from zero, the
    nugget variance is divided into 2 terms: \emph{micro-scale variance}
    and \emph{measurement error}. This might affect the precision of
    the predictions. 
    In practice, these two variance components are usually indistinguishable but the
    distinction can be made here if justifiable.  } 
  \item{cov.model}{string indicating the name of the model for the
    correlation function. Further details  in the
    documentation for
    \code{\link{cov.spatial}}.  
    Defaults are equivalent to the \emph{exponential} model.  }
  \item{kappa}{additional smoothness parameter required by the following correlation
    functions: \code{"matern"}, 
    \code{"powered.exponential"},  
    \code{"cauchy"}  and  
     \code{ "gneiting.matern"}.   }
  \item{lambda}{numeric value of the Box-Cox transformation parameter.
    The value \eqn{\lambda = 1}{lambda = 1} corresponds to
    no transformation and \eqn{\lambda = 0}{lambda = 0} corresponds to
    the log-transformation. 
    Prediction results are back-transformed and
    returned is the same scale as for the original data.  }
  \item{m0}{The default value \code{"ok"} indicates that ordinary
    kriging will be performed. Other options are \code{"kt"} for kriging
    with a trend model (universal kriging) and \code{"kte"} for kriging
    with external trend (covariates). If a numeric value is provided it
    is assumed to be the value of a know mean and simple kriging is then
    performed.
    If \code{"av"} the arithmetic mean of the data is assumed to be the
    know mean for simple kriging algorithm.  }
  \item{nwin}{If \code{"full"} \emph{global neighborhood} is used
    i.e., all data values are used in the
    prediction of every prediction location.
    An integer number defines the \emph{moving neighborhood} algorithm.
    The number provided is used as  
    the number closest neighbors to be used for the prediction at each
    location.
    Defaults to \code{"full"}.  }
  \item{n.samples.backtransform}{number of samples used in the
    back-transformation. When transformations are used
    (specified by an argument \code{lambda}), back-transformations
    are usually performed by sampling from the predictive distribution and
    then back-transforming the sampled values. The exceptions are for
    \eqn{\lambda = 0}{lambda = 0} (log-transformation)  and
    \eqn{\lambda = 1}{lambda = 1} (no transformation).  }
  \item{trend}{only required if \code{m0 = "kt"} (universal kriging).
    Possible values are \eqn{1} or \eqn{2}, corresponding to a first or second degree
    polynomial trend on the coordinates, respectively.  }
  \item{d}{spatial dimension, \eqn{1} defines a prediction on a line, \eqn{2} on
    a plane (the default).}
  \item{ktedata}{only required if \code{m0 = "kte"}. A vector or matrix
    with the values of the
    external trend (covariates) at the data locations.  }
  \item{ktelocations}{only required if \code{m0 = "kte"}. A vector or
    matrix with the values of the
    external trend (covariates) at the prediction locations.  }
  \item{aniso.pars}{parameters for geometric anisotropy
    correction. If \code{aniso.pars = FALSE} no correction is made, otherwise
    a two elements vector with values for the anisotropy parameters
    must be provided. Anisotropy correction consists of a
    transformation of the data and prediction coordinates performed
    by the function \code{\link{coords.aniso}}.  }
  \item{signal}{logical. If \code{TRUE} the signal is predicted, otherwise the
    variable is predicted. If no transformation is performed the
    expectations are the same in both cases and the difference is only for
    values of the kriging variance, if the value of the nugget is
    different from zero.  }
  \item{dist.epsilon}{a numeric value. Points which are separated
    by a distance less than this value
    are considered co-located.  }
  \item{messages}{logical. Indicates
    whether or not status messages are printed on the screen (or other output device)
    while the function is running.}
}

%\details{
%}

\value{
  An object of the \code{\link{class}} \code{kriging} which is a  list
  with the following components:
  \item{predict}{the predicted values.  }
  \item{krige.var}{the kriging variances.  }
  \item{dif}{the difference between the predicted value and the global
    mean. Represents the contribution to the neighboring data to
    the prediction at each point.  }
  \item{summary}{values of the arithmetic and weighted mean of the data and standard
    deviations. The weighted mean corresponds to the estimated value of
    the global mean. }
  \item{ktrend}{the matrix with trend if \code{m0 = "kt"}
    (universal kriging).  }
  \item{ktetrend}{the matrix with trend if \code{m0 = "kte"}
    (external trend kriging).}
  \item{beta}{the value of the mean which is implicitly estimated for
    \code{m0 = "ok", "kte"} or \code{"kt"}.}
  \item{wofmean}{weight of mean. The predicted value is an
    weighted average between the global mean and the values at the
    neighboring locations.
    The value returned is the weight of the mean.  }
  \item{locations}{the coordinates of the prediction locations.  }
  \item{message}{status messages returned by the algorithm.  }
  \item{call}{the function call.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\note{
  This is a preliminary and inefficient function implementing kriging methods.
  For predictions using global neighborhood the function
  \code{\link{krige.conv}} should be used instead.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{krige.conv}} for a more efficient implementation of
  conventional kriging methods, \cr \code{\link{krige.bayes}} for Bayesian prediction.}


\examples{
data(s100)
loci <- expand.grid(seq(0,1,l=31), seq(0,1,l=31))
kc <- ksline(s100, loc=loci, cov.pars=c(1, .25))
par(mfrow=c(1,2))
image(kc, loc=loci, main="kriging estimates")
image(kc, loc=loci, val=sqrt(kc$krige.var),
              main="kriging std. errors")

}

\keyword{spatial}%-- one or more ...



\eof
\name{legend.krige}

\alias{legend.krige}

\title{Add a legend to a image with kriging results}

\description{
  This function allows adds a legend to an image plot generated by
  \code{\link{image.kriging}} or \code{\link{image.krige.bayes}}.
  It can be called internally by these functions or directly by the user.
}

\usage{
legend.krige(x.leg, y.leg, values, scale.vals,
             vertical = FALSE, offset.leg = 1, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x.leg}{limits for the legend in the \eqn{x} direction.  }
  \item{y.leg}{limits for the legend in the \eqn{y} direction.  }
  \item{values}{values plotted in the image.  }
  \item{scale.vals}{optional. Values to appear in the legend.
    If not provided the function \code{\link{pretty}} is used to define
    the values.  }
  \item{vertical}{If \code{TRUE} the legend is drawn in the vertical
    direction. Defaults to \code{FALSE}.  }
  \item{offset.leg}{numeric value controlling the distance between the
    legend text and the legend box. }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{text}}.  }
}

%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  A legend is added to the current plot. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{image.kriging}}, \code{\link{image.krige.bayes}}.  }

\examples{
# See examples in the documentation for image.kriging
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{aplot}% __ONLY ONE__ keyword per line

\eof
\name{likfit}

\alias{likfit}
\alias{likfit.limits}
\alias{negloglik.GRF}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Likelihood Based Parameter Estimation for Gaussian Random Fields}

\description{
  \emph{Maximum likelihood} (ML) or
  \emph{restricted maximum likelihood} (REML) parameter estimation for (transformed) Gaussian
  random fields.
}

\usage{
likfit(geodata, coords = geodata$coords, data = geodata$data,
       trend = "cte", ini.cov.pars, fix.nugget = FALSE, nugget = 0,
       fix.kappa = TRUE, kappa = 0.5, fix.lambda = TRUE, lambda = 1,
       fix.psiA = TRUE, psiA = 0, fix.psiR = TRUE, psiR = 1, 
       cov.model = "matern", realisations,
       method.lik = "ML", components = FALSE,
       nospatial = TRUE, limits = pars.limits(),
       print.pars = FALSE, messages, \dots)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. 
    Typically an object of the class \code{"geodata"}.
        If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix where each row has the 2-D
    coordinates of the \eqn{n} data locations.
    By default it takes the
    component \code{coords} of the argument \code{geodata}, if provided.
  }
  \item{data}{a vector with \emph{n} data values. By default it takes the
    component \code{data} of the argument \code{geodata}, if provided.  }
  \item{trend}{specifies the mean part of the model. See documentation
    of \code{\link{trend.spatial}} for further details.
    Defaults to \code{"cte"}.  }
  \item{ini.cov.pars}{initial values for the covariance parameters:
    \eqn{\sigma^2}{sigma^2} (partial sill) and \eqn{\phi}{phi} (range
    parameter).  Typically a vector with two components. However a
    matrix can be used to provide several initial values. See
    \code{DETAILS} below.}
  \item{fix.nugget}{logical, indicating whether the parameter
    \eqn{\tau^2}{tau^2} (nugget variance) should be regarded as fixed
    (\code{fix.nugget = TRUE}) or should be
    estimated (\code{fix.nugget = FALSE}). Defaults to
    \code{FALSE}.  }    
  \item{nugget}{value of the nugget parameter.
    Regarded as a fixed value if \code{fix.nugget = TRUE} otherwise
    as the initial value for the minimisation algorithm. Defaults to zero.  }
  \item{fix.kappa}{logical, indicating whether the extra parameter
    \eqn{\kappa}{kappa} should be regarded as fixed
    (\code{fix.kappa = TRUE}) or should be 
    estimated (\code{fix.kappa = FALSE}). Defaults to \code{TRUE}.  }    
  \item{kappa}{value of the extra parameter \eqn{\kappa}{kappa}.
    Regarded as a fixed value if \code{fix.kappa = TRUE}
    otherwise as the initial value for the minimisation algorithm. Defaults to
    \eqn{0.5}. This parameter is valid only if the covariance function is one
    of: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"} or
    \code{"gneiting.matern"}.  For more details on covariance functions
    see documentation for \code{\link{cov.spatial}}.  }
  \item{fix.lambda}{logical, indicating whether the Box-Cox transformation parameter
    \eqn{\lambda}{lambda} should be regarded as fixed
    (\code{fix.lambda = TRUE}) or should be
    be estimated (\code{fix.lambda = FALSE}). Defaults to \code{TRUE}.  }    
  \item{lambda}{value of the Box-Cox transformation parameter
    \eqn{\lambda}{lambda}.
    Regarded as a fixed value if \code{fix.lambda = TRUE} otherwise 
    as the initial value for the minimisation algorithm. Defaults to
    \eqn{1}. Two particular cases are \eqn{\lambda = 1}{lambda = 1}
    indicating no transformation and \eqn{\lambda = 0}{lambda
      = 0} indicating log-transformation.  }
  \item{fix.psiA}{logical, indicating whether the anisotropy angle parameter
    \eqn{\psi_R}{psi_R} should be regarded as fixed
    (\code{fix.psiA = TRUE}) or should 
    be estimated (\code{fix.psiA = FALSE}). Defaults to
    \code{TRUE}.  }    
  \item{psiA}{value (in radians) for the anisotropy angle parameter
    \eqn{\psi_A}{psi_A}. 
    Regarded as a fixed value if \code{fix.psiA = TRUE}
    otherwise  as the initial value for the minimisation algorithm.
    Defaults to \eqn{0}. See \code{\link{coords.aniso}} for further
    details on anisotropy correction.  }
  \item{fix.psiR}{logical, indicating whether the anisotropy ratio parameter
    \eqn{\psi_R}{psi_R} should be regarded as fixed
    (\code{fix.psiR = TRUE}) or should be estimated (\code{fix.psiR = FALSE}). Defaults to
    \code{TRUE}.  }    
  \item{psiR}{value, always greater than 1, for the anisotropy ratio  parameter
    \eqn{\psi_R}{psi_R}. 
    Regarded as a fixed value if \code{fix.psiR = TRUE}
    otherwise as the initial value for the minimisation algorithm.
    Defaults to \eqn{1}. See \code{\link{coords.aniso}} for further
    details on anisotropy correction.  }
  \item{cov.model}{a string specifying the model for the correlation
    function. For further details see documentation for \code{\link{cov.spatial}}.
    Defaults are equivalent to the \emph{exponential} model.  }
  \item{realisations}{optional. A vector indicating the number of replication
    for each datum. For further information see \code{DETAILS} below and
    documentation for \code{\link{as.geodata}}. }
  \item{method.lik}{options are \code{"ML"} for maximum likelihood and \code{"REML"} for
    restricted maximum likelihood. Defaults to \code{"ML"}.  }
  \item{components}{an \eqn{n \times 3}{n x 3} data-frame with fitted
    values for the three model components: trend, spatial and residuals.
    See the section \code{DETAILS} below for the model specification.  }
  \item{nospatial}{logical. If \code{TRUE} parameter estimates for the
    model without spatial component are included in the output.  }
  \item{limits}{values defining lower and upper limits for the model
    parameters used in the numerical minimisation.
    The auxiliary function \code{\link{pars.limits}} is called to set the limits.  }
  \item{print.pars}{logical. If \code{TRUE} the parameters and the value
  of the negative log-likelihood (up to a constant) are printed each
  time the function to be minimised is called.  }
  \item{messages}{logical. Indicates whether
    status messages should be printed on the screen (or output device)
    while the function is running.  }
  \item{\dots}{additional parameters to be passed to the minimisation
    function. Typically  arguments of the type  \code{control()} which controls the
    behavior of the minimisation algorithm. For further details see documentation 
    for the minimisation function \code{\link{optim}}.  }
}

\details{
  This function estimate the parameters of the Gaussian random field model,
  specified as:
  \deqn{Y(x) = \mu(x) + S(x) + e}{Y(x) = mu(x) + S(x) + e}
  where
  \itemize{
    \item \eqn{x} defines a spatial location. Typically Euclidean
    coordinates on a plane. 
    \item \eqn{Y} is the variable been observed.
    \item \eqn{\mu(x) = X\beta}{mu(x) = X \%*\% beta} is the mean component of
    the model (trend). 
    \item \eqn{S(x)} is a stationary Gaussian process with variance \eqn{\sigma^2}{sigma^2}
    (partial sill) and a correlation function parametrized in its
    simplest form by \eqn{\phi}{phi} (the range parameter). Possible extra parameters
    for the correlation function are the smoothness parameter
    \eqn{\kappa}{kappa} and the anisotropy parameters \eqn{\phi_R}{phi_R} and
    \eqn{\phi_A}{phi_A} (anisotropy ratio and angle, respectively).  
    \item \eqn{e} is the error term with variance parameter  \eqn{\tau^2}{tau^2}
    (nugget variance).
  }
  
  The additional parameter \eqn{\lambda}{lambda} allows for the Box-Cox
  transformation of the response variable.
  If used (i.e. if \eqn{\lambda \neq 1}) \eqn{Y(x)} above is replaced by \eqn{g(Y(x))}
  such that
  \deqn{g(Y(x)) = \frac{Y^\lambda(x) - 1}{\lambda}.}{g(Y(x)) = ((Y^lambda(x)) -
    1)/lambda .}

  Two particular cases are \eqn{\lambda = 1}{lambda = 1}
  which indicates no transformation and \eqn{\lambda = 0}{lambda = 0}
  indicating the log-transformation.
  
  In general parameter estimation is performed numerically using the \R
  function \code{\link{optim}} to minimise the
  negative log-likelihood computed by the function \code{negloglik.GRF}.
  If the nugget, anisotropy (\eqn{\psi_A, \psi_R}{psiA, psiR}),
  smoothness (\eqn{\kappa}{kappa}) and transformation (\eqn{\lambda}{lambda}) parameters
  are held fixed then the numerical minimisation can be reduced to
  one-dimension and the function \code{\link{optimize}} is used instead
  of \code{optim}. In this case initial values are irrelevant.
  
  Lower and upper limits for parameter values can be
  individually specified using the function \code{link{pars.limits}}.
  For example, including the following in the function call:\cr
  \code{limits = pars.limits(phi=c(0, 10), lambda=c(-2.5, 2.5))}, \cr
  will change the limits for the parameters \eqn{\phi}{phi} and \eqn{\lambda}{lambda}.
  Default values are used if the argument \code{limits} is not provided.  
  
  If the \code{fix.lambda = FALSE} and \code{nospatial = FALSE} the
  Box-Cox parameter for the model without the spatial component is
  obtained numerically, with log-likelihood computed by the function
  \code{boxcox.ns}.

  \bold{Multiple initial values} can be specified providing a \eqn{n
    \time 2}{n x 2} matrix for the argument \code{ini.cov.pars} and/or
  providing a vector for the values of the remaining model parameters.
  In this case the log-likelihood is computed for all combinations of
  the model parameters. The parameter set which maximises the
  value of the log-likelihood is then used to start the
  minimisation algorithm.

  The argument \bold{realisations} allows replicated data to be used.
  For instance, data collected at different times at least partially
  the same locations
  can be pooled together in the parameter estimation if independence is
  assumed between time points.
  The argument \code{realisations} takes a vector indicating the
  replication number (e.g. the times). The log-likelihoods are computed
  for each replication and added together.\cr
  \emph{Notice that this assumes independence among the replications}.
}


\value{
  An object of the classes \code{"likGRF"} and \code{"variomodel"}.\cr
  The function \code{\link{summary.likGRF}} is used to print a summary
  of the fitted model.\cr 
  The object is a list with the following components:
  \item{cov.model}{a string with the name of the correlation
    function.  }
  \item{nugget}{value of the nugget parameter \eqn{\tau^2}{tau^2}.
    This is an estimate if \code{fix.nugget = FALSE} otherwise, a fixed
    value.  }
  \item{cov.pars}{a vector with the estimates of the parameters
    \eqn{\sigma^2}{sigma^2} 
    and \eqn{\phi}{phi}, respectively.  }
  \item{kappa}{value of the smoothness parameter. Valid only if 
    the correlation function is one of: \code{"matern"},
    \code{"powered.exponential"}, \code{"cauchy"}
    or \code{"gneiting.matern"}.  }
  \item{beta}{estimate of mean parameter \eqn{\beta}{beta}.
    This can be a scalar or vector depending on the
    trend (covariates) specified in the model.  }
  \item{beta.var}{estimated variance (or covariance matrix) for the mean
    parameter \eqn{\beta}{beta}.  }
  \item{lambda}{values of the Box-Cox transformation parameter. A fixed value if
    \code{fix.lambda = TRUE} otherwise the estimate value.  }
  \item{aniso.pars}{fixed values or estimates of the anisotropy
    parameters, according to the function call.  }
  \item{method.lik}{estimation method used, \code{"ML"} (maximum likelihood)
    or \code{"REML"} (restricted maximum likelihood).  }
  \item{loglik}{the value of the maximized likelihood.  }
  \item{npars}{number of estimated parameters.  }
  \item{AIC}{value of the Akaike Information Criteria, \eqn{AIC=-2
      ln(L) + 2 p} where \eqn{L} is the maximised likelihood and p is
    the number of parameters in the model.  }
  \item{BIC}{value of the Bayesian Information Criteria,
    \eqn{BIC=-2ln(L) + p log(n)}, where \eqn{n} is the number of data,
      \eqn{L,p} as for AIC above.  }
  \item{parameters.summary}{a data-frame with all model parameters,
    their status (estimated or fixed) and values.  }
  \item{info.minimisation}{results returned by the minimisation function.}
  \item{max.dist}{maximum distance between 2 data points. This
    information relevant for other functions which use outputs from
    \code{likfit}.  }
  \item{trend}{the trend (covariates) matrix \eqn{X}.  }
  \item{log.jacobian}{numerical value of the logarithm of the Jacobian of
    the transformation.  }
  \item{nospatial}{estimates for the model without the spatial
    component. }
  \item{call}{the function call.  }
}

\references{
  Further information on the package
  \pkg{geoR}
  can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }
%
% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{summary.likGRF}} for summary of the results,
  \code{\link{plot.variogram}}, \code{\link{lines.variogram}} and
  \code{\link{lines.variomodel}} for graphical output,
  \code{\link{proflik}} for computing profile likelihoods,
  \code{\link{variofit}} and for other estimation methods, 
  and \code{\link{optim}} for the numerical minimisation function.
}

\examples{
\dontrun{
data(s100)
ml <- likfit(s100, ini=c(0.5, 0.5), fix.nug = TRUE)
ml
summary(ml)
reml <- likfit(s100, ini=c(0.5, 0.5), fix.nug = TRUE, met = "REML")
summary(reml)
plot(variog(s100))
lines(ml)
lines(reml, lty = 2)
}
}
\keyword{spatial}
\keyword{models}%-- one or more ...

\eof
\name{lines.variomodel.krige.bayes}

\alias{lines.variomodel.krige.bayes}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Adds a Bayesian Estimate of the Variogram to a Plot}

\description{
  Adds a Bayesian estimate of the variogram model to a plot typically with an empirical variogram.
  The estimate is a chosen summary (mean, mode or mean) of the
  posterior distribution returned by the function \code{\link{krige.bayes}}.
}

\usage{
\method{lines.variomodel}{krige.bayes}(x, summary.posterior, max.dist, uvec,
                 posterior = c("variogram", "parameters"),  \dots)
}


\arguments{
  \item{x}{an object of the class \code{krige.bayes}, typically an output
    of the function \code{\link{krige.bayes}}.  }
  \item{summary.posterior}{specify which summary of the posterior
    distribution should be used as the parameter estimate.
    Options are \code{"mean"}, \code{"median"} or
  \code{"mode"}. See \code{DETAILS} below. }
  \item{max.dist}{numerical, the maximum distance for the x-axis.  }
  \item{uvec}{a numerical vector with support points to compute the
    variogram values. Only used if \code{posterior = "variogram"}.
    Defaults to \code{seq(0, max.dist, length = 51)}.
  }
  \item{posterior}{indicates whether the the variogram line is
    based on the posterior of the variogram function (default)
    or the posterior of the model parameters.  }
  \item{\dots}{arguments passed to the functions \code{\link{lines}} or \code{\link{curve}}.  }
}

\details{
  The function \code{\link{krige.bayes}} returns samples from the
  posterior distribution of the parameters \eqn{(\sigma^2, \phi,
    \tau^{2}_{rel})}{(sigma^2, phi, tausq.rel)}.
  
  This function allows for two basic options to draw a line with a
  summary of the variogram function.
  \enumerate{
    \item[1.]{"variogram": }{for each sample of the parameters the variogram
      function is computed at the support points defined in the
      argument \code{uvec}. Then a function provided by the user in the
      argument \code{summary.posterior} is used to compute a summary of
      the values obtained at each support point.}
    \item[2.]{"parameters": }{in this case summaries of the posterior
      distribution of the model parameters as "plugged-in" in the
      variogram function.
      One of the options \code{"mode"} (default) ,\code{"median"} or \code{"mean"}
      can be provided in the argument \code{summary.posterior}.
      The option \code{mode}, uses the mode of \eqn{(\phi,
	\tau^{2}_{rel})}{(phi, tau^2.rel)} and the mode of
      of \eqn{\sigma^2}{sigma^2} conditional on the modes of the former parameters.
      For the options \code{mean} and \code{median} these summaries are
      computed from the samples of the posterior.
    }
  }
}

\value{
  A line with the estimated variogram plot is added to the plot in the
  current graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }


\seealso{\code{\link{lines.variomodel}}, \code{\link{krige.bayes}} and \code{\link{lines}}.}

\examples{
#See examples in the documentation of the function krige.bayes().
}

\keyword{spatial}
\keyword{aplot}


\eof
\name{lines.variogram}

\alias{lines.variogram}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Line with a Empirical Variogram}

\description{
  A sample (empirical) variogram computed using the
  function \code{\link{variog}} is added to the current plot.
}

\usage{
\method{lines}{variogram}(x, max.dist, type = "o",  scaled = FALSE,
         pts.range.cex, \dots)
}

\arguments{
  \item{x}{an object of the class \code{"variogram"}, typically an
    output from the function \code{\link{variog}}.  }
  \item{max.dist}{maximum distance for the x-axis. By default takes
    the maximum distance for which the sample variogram was computed.  }
  \item{type}{type of line for the empirical variogram. The default is
    \code{"o"} (dots and lines). See documentation for
    \code{\link{lines}} for further details.  }
  \item{scaled}{logical. If \code{TRUE} the variogram values are divided by the
    sample variance. This allows comparison between variograms of
    different variables.  }
  \item{pts.range.cex}{optional. A two elements vector with maximum and
    minimum values for the caracter expansion factor \code{cex}. If
    provided the point sizes in binned variogram are proportional to the
    number of pairs of points used to compute each bin.  } 
  \item{\dots}{other arguments to be passed to the function
    \code{\link{lines}}.  }
}

%\details{
%  Adds a sample variogram(s) to a plot.
%  Together with \code{\link{lines.variogram}} can be
%  used to compare sample variograms for different variables.
%}


\value{
  A line with the empirical variogram is added to the plot in the current
  graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{variog}}, \code{\link{lines.variogram}},
  \code{\link{lines.variomodel}}, \code{\link{variog.model.env}},
  \code{\link{variog.mc.env}}, \code{\link{plot.grf}}, \code{\link{lines}}.}

%\examples{
%  TO DO: ADD AN EXAMPLE WITH 2 VARIABLES AND SCALED VARIOGRAMS
%}

\keyword{spatial}
\keyword{aplot}



\eof
\name{lines.variogram.envelope}

\alias{lines.variogram.envelope}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Adds Envelopes Lines to a Variogram Plot}

\description{
  Variogram envelopes computed by \code{\link{variog.model.env}} or
  \code{\link{variog.mc.env}} are added to the current variogram plot.
}

\usage{
\method{lines}{variogram.envelope}(x, lty = 3, \dots)
}

\arguments{
  \item{x}{an object of the class \code{"variogram.envelope"},
    typically an output of the functions \code{\link{variog.model.env}} or
    \code{\link{variog.mc.env}}.  } 
  \item{lty}{line type. Defaults to 3.  }
  \item{\dots}{arguments to be passed to the function \code{\link{lines}}.  }
}

%\details{
%}

\value{
  Lines defining the variogram envelope
  are added to the plotin the current graphics device.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{variog}} for variogram computation,
  \code{\link{variog.model.env}} and \code{\link{variog.mc.env}} for
  computation of variogram envelopes, and \code{\link{lines}} for the
  generic function.  }

\examples{
if(is.R()) data(s100)
s100.vario <- variog(s100, max.dist = 1)
s100.ml <- likfit(s100, ini=c(.5, .5))
s100.mod.env <- variog.model.env(s100, obj.variog = s100.vario,
   model = s100.ml) 
s100.mc.env <- variog.mc.env(s100, obj.variog = s100.vario)
plot(s100.vario)
lines(s100.mod.env)
lines(s100.mc.env, lwd=2)

}

\keyword{spatial}
\keyword{aplot}



\eof
\name{lines.variomodel}

\alias{lines.variomodel}
\alias{lines.variomodel.default}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Adds a Line with a Variogram Model to a Variogram Plot}

\description{
  This function adds a line with a variogram model specifyed by the user
  to a current variogram plot.
  The variogram is specifyed either by passing a list with values for the variogram
  elements or using each argument in the function.
}


\usage{
lines.variomodel(x, \dots)
\method{lines.variomodel}{default}(x, cov.model, cov.pars, nugget, kappa,
                          max.dist, scaled = FALSE, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{a list with the values for the following components: \code{cov.model}, \code{cov.pars},
    \code{nugget}, \code{kappa} (not aways compulsory, see argument
    kappa below), \code{max.dist}.
    This argument is optional if the other arguments in the function
    are provided, otherwise is compulsory.
  }
  \item{cov.model}{a string with the type of the variogram function. See
    documentation of \code{\link{cov.spatial}} for further details.
  }
  \item{cov.pars}{a vector or matrix with the values for the sill
    (\eqn{\sigma^2}{sigmasq}) and range (\eqn{\phi}{phi}) parameters. }
  \item{nugget}{a scalar with the value of the nugget
    (\eqn{\tau^2}{tausq}) parameter.
  }
  \item{kappa}{a scalar with the value of the smoothness
    (\eqn{\kappa}{kappa}) parameters. Only required if \code{cov.model}
    is one of the following: \code{"matern"}, \code{"powered.exponential"},
    \code{"cauchy"} and \code{"gneiting.matern"}
  }
  \item{max.dist}{maximum distance (x-axis) to compute and draw the line
    representing the  variogram model.
    The default is the distance given by \code{obj$max.dist}.  }
  \item{scaled}{logical. If \code{TRUE} the total sill in the plot is
    equals to \eqn{1}.  }
  \item{\dots}{arguments to be passed to the function
    \code{\link{curve}}.  }
}

\details{
  Adds a line with a variogram model  to a plot.
  In conjuction with \code{\link{plot.variogram}} can be
  used for instance to compare sample variograms against fitted models returned by
  \code{\link{variofit}}  and/or \code{\link{likfit}}.
}

\value{
  A line with a variogram model is added to a plot on the current
  graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.}

%\note{ ~~further notes~~ }

%~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{lines.variomodel.krige.bayes}},
  \code{\link{lines.variomodel.grf}},
  \code{\link{lines.variomodel.variofit}},
  \code{\link{lines.variomodel.likGRF}},
  \code{\link{plot.variogram}}, \code{\link{lines.variogram}},
  \code{\link{variofit}}, \code{\link{likfit}}, \code{\link{curve}}. 
}

\examples{
data(s100)
# compute and plot empirical variogram
vario <- variog(s100, max.dist = 1)
plot(vario)
# estimate parameters
vario.wls <- variofit(vario, ini = c(1, .3), fix.nugget = TRUE)
# adds fitted model to the plot  
lines(vario.wls)
#
# Ploting different variogram models
plot(0:1, 0:1, type="n")
lines.variomodel(cov.model = "exp", cov.pars = c(.7, .25), nug = 0.3, max.dist = 1) 
# an alternative way to do this is:
my.model <- list(cov.model = "exp", cov.pars = c(.7, .25), nugget = 0.3,
max.dist = 1) 
lines.variomodel(my.model, lwd = 2)
# now adding another model
lines.variomodel(cov.m = "mat", cov.p = c(.7, .25), nug = 0.3,
                 max.dist = 1, kappa = 1, lty = 2) 
}
\keyword{spatial}
\keyword{aplot}


\eof
\name{lines.variomodel.grf}

\alias{lines.variomodel.grf}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Lines with True Variogram for Simulated Data}

\description{
  This functions adds to the graphics device a line with the theoretical
  (true) variogram used when generating simulations with
  the function \code{\link{grf}}.
}

\usage{
\method{lines.variomodel}{grf}(x, max.dist = max(dist(x$coords)),
length = 100, lwd = 2, \dots)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object from the class \code{grf} typically an output of
    the function \code{\link{grf}}.  }
  \item{max.dist}{maximum distance to compute and plot the true
    variogram. Defaults to the maximum distance between two data locations.  }
  \item{length}{number of points used to compute and draw the variogram line.}
  \item{lwd}{width of the line.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{curve}}.  }
}

%\details{
%
%}

\value{
  A line with the true variogram model is added to the current
  plot on the graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }


\seealso{\code{\link{lines.variomodel}},
  \code{\link{grf}}, \code{\link{plot.grf}}, \code{\link{curve}}.}

\examples{
sim <- grf(100, cov.pars=c(1, .25)) # simulates data
plot(variog(sim, max.dist=1))       # plot empirical variogram
lines(sim, max.dist=1)              # plot true variogram
}

\keyword{spatial}
\keyword{aplot}%-- one or more ...

\eof
\name{lines.variomodel.likGRF}

\alias{lines.variomodel.likGRF}

\title{Adds a Variogram Line to a Variogram Plot}

\description{
  This function adds a fitted variogram based on the estimates of the
  model parameters returned by the function \code{\link{likfit}}
  to a current variogram plot.
}


\usage{
\method{lines.variomodel}{likGRF}(x, max.dist, scaled = FALSE, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object of the class \code{likGRF}
    which is a list containing
    information about the fitted model parameters, typically an output
    of the function \code{\link{likfit}}.  }
  \item{max.dist}{maximum distance (x-axis) to compute and draw the line
    representing the  variogram model.
    The default is the distance given by \code{obj$max.dist}.  }
  \item{scaled}{logical. If \code{TRUE} the total sill in the plot is
    equals to \eqn{1}.  }
  \item{\dots}{arguments to be passed to the function
    \code{\link{curve}}.  }
}

\details{
  Adds variogram model(s)  to a plot.
  In conjuction with \code{\link{plot.variogram}} can be
  used to compare sample variograms against fitted models returned by
  \code{\link{likfit}}.
}

\value{
  A line with a variogram model is added to a plot on the current
  graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.}

%\note{ ~~further notes~~ }

%~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{lines.variomodel}},
  \code{\link{lines.variomodel.variofit}},
  \code{\link{plot.variogram}}, \code{\link{lines.variogram}},
  \code{\link{variofit}}, \code{\link{likfit}}, \code{\link{curve}}. 
}

\examples{
data(s100)
# compute and plot empirical variogram
vario <- variog(s100, max.dist = 1)
plot(vario)
# estimate parameters
vario.ml <- likfit(s100, ini = c(1, .3), fix.nugget = TRUE)
# adds fitted model to the plot  
lines(vario.ml)
}
\keyword{spatial}
\keyword{aplot}


\eof
\name{lines.variomodel.variofit}

\alias{lines.variomodel.variofit}

\title{Adds a Line with a Fitted Variogram Model to a Variogram Plot}

\description{
  This function adds a line with the variogram model
  fitted by the function
  \code{\link{variofit}} to a current variogram plot.
}

\usage{
\method{lines.variomodel}{variofit}(x, max.dist, scaled = FALSE, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object of the class \code{variofit}
    which is a list containing information about
    the fitted model parameters, typically an output of the function
    \code{\link{variofit}}.  }
  \item{max.dist}{maximum distance (x-axis) to compute and draw the line
    representing the variogram model.
    The default is the distance given by \code{x$max.dist}.  }
  \item{scaled}{logical. If \code{TRUE} the total sill in the plot is
    set to \eqn{1}.  }
  \item{\dots}{arguments to be passed to the function
    \code{\link{curve}}.  }
}

\details{
  Adds fitted variogram model to a plot.
  In conjuction with \code{\link{plot.variogram}} can be
  used to compare empirical variograms against fitted models returned by
  \code{\link{variofit}}.
}

\value{
  A line with a variogram model is added to a plot on the current
  graphics device. No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.}

%\note{ ~~further notes~~ }

%~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{lines.variomodel}},
  \code{\link{lines.variomodel.likGRF}},
  \code{\link{plot.variogram}}, \code{\link{lines.variogram}},
  \code{\link{variofit}}, \code{\link{likfit}}, \code{\link{curve}}. 
}

\examples{
data(s100)
# compute and plot empirical variogram
vario <- variog(s100, max.dist = 1)
plot(vario)
# estimate parameters
vario.wls <- variofit(vario, ini = c(1, .3), fix.nugget = TRUE)
# adds fitted model to the plot  
lines(vario.wls)
}
\keyword{spatial}
\keyword{aplot}


\eof
\name{locations.inside}

\alias{locations.inside}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{select prediction locations inside borders}

\description{
  Selects the prediction locations located inside a polygon defining
  borders of a region where prediction is aimed.
  Typically internally called by \code{\link{krige.bayes}}, \code{\link{krige.conv}}, \code{\link{ksline}}. 
}

\usage{
locations.inside(locations, borders)
}

\arguments{
  \item{locations}{a two columns matrix or dqata frame with coordinates
    of the prediction locations.  }
  \item{borders}{a two column matrix or data-frame with coordinates of a
    polygon defining the borders of the region.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  A two columns matrix with coordinates of points inside the borders.  
}

%\references{ ~put references to the literature/web site here ~ }
%\author{ ~~who you are~~ }
%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{}

%\examples{
%}
\keyword{spatial}% at least one, from doc/KEYWORDS

\eof
\name{loglik.GRF}

\alias{loglik.GRF}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Log-Likelihood for a Gaussian Random Field}

\description{
  This function computes the value of the log-likelihood for a
  realisation of a Gaussian random field.
}

\usage{
loglik.GRF(geodata, coords = geodata$coords, data = geodata$data,
           obj.model = NULL cov.model = "exp", cov.pars, nugget = 0,
           kappa = 0.5, lambda = 1, psiR = 1, psiA = 0,
           trend = "cte", method.lik = "ML", compute.dists = TRUE,
           realisations = NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. 
    Typically an object of the class \code{"geodata"} - a \pkg{geoR}
    data-set.
    If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix, each row containing Euclidean
    coordinates of the \emph{n} data locations. By default it takes the
    element \code{coords} of the argument \code{geodata}.  }
  \item{data}{a vector with data values. By default it takes the
    element \code{data} of the argument \code{geodata}.  }
  \item{obj.model}{a object of the class \code{variomodel} with a fitted
    model. Tipically an output of 
    \code{\link{likfit}} or \code{\link{variofit}}. }
  \item{cov.model}{a string specifying the model for the correlation
    function. For further details see 
    documentation for \code{\link{cov.spatial}}. }
  \item{cov.pars}{a vector with 2 elements with values of  the covariance parameters
    \eqn{\sigma^2}{sigma^2} (partial sill) and \eqn{\phi}{phi} (range
    parameter).  }
  \item{nugget}{value of the nugget parameter. Defaults to \eqn{0}.  }
  \item{kappa}{value of the smoothness parameter. Defaults to
    \eqn{0.5}.  }
  \item{lambda}{value of the Box-Cox tranformation parameter. Defaults
    to \eqn{1}.  }
  \item{psiR}{value of the anisotropy ratio parameter. Defaults to
    \eqn{1}, corresponding to isotropy.  }
  \item{psiA}{value (in radians) of the anisotropy rotation parameter. Defaults to zero.}
  \item{trend}{specifies the mean part of the model.
    The options are:
    \code{"cte"} (constant mean),
    \code{"1st"} (a first order polynomial
    on the coordinates), \code{"2nd"} (a second order polynomial
    on the coordinates), or a formula of the type \code{~X} where \code{X}
    is a matrix with the covariates (external trend). Defaults to \code{"cte"}.  }
  \item{method.lik}{options are \code{"ML"} for likelihood and \code{"REML"} for
    restricted likelihood. Defaults to \code{"ML"}.  }
  \item{compute.dists}{for internal use with other function. Don't
    change the default unless you know what you are doing.  }
    \item{realisations}{optional. A vector indicating replication number
    for each data. For more details see \code{\link{as.geodata}}. }
}

\details{
  The expression log-likelihood is:
  \deqn{l(\theta) = -\frac{n}{2} \log (2\pi) - \frac{1}{2} \log |\Sigma|
    - \frac{1}{2} (y -
  F\beta)' \Sigma^{-1} (y - F\beta),}{l(theta) = -(n/2) * log(2 * pi) - 0.5 * log|V| - 0.5 * (y - F b)' V^{-1} (y - F
    b),}
  where \eqn{n} is the size of the data vector \eqn{y}, \eqn{\beta}{b}
  is the mean (vector) parameter with dimention \eqn{p}, \eqn{\Sigma}{V} is the covariance
  matrix and \eqn{F} is the matrix with the values of the covariates (a
  vector of \eqn{1}'s if the mean is constant.

  The expression restricted log-likelihood is:
  \deqn{rl(\theta) =  -\frac{n-p}{2} \log (2\pi) + \frac{1}{2} \log |F' F|
    - \frac{1}{2} \log |\Sigma| - \frac{1}{2} \log |F' \Sigma F| - \frac{1}{2} (y -
  F\beta)' \Sigma^{-1} (y - F\beta).}{rl(\theta) =  -((n-p)/2) *
  log (2 * pi) + 0.5 * log |F'F| - 0.5 * log |V|  - 0.5 * log |F'VF| -
  0.5 * (y - Fb)' V^(-1) (y - Fb).}
  
}

\value{
  The numerical value of the log-likelihood.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br},  \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{likfit}} for likelihood-based parameter estimation.}

\examples{
data(s100)
loglik.GRF(s100, cov.pars=c(0.8, .25), nugget=0.2)
loglik.GRF(s100, cov.pars=c(0.8, .25), nugget=0.2, met="RML")

## Computing the likelihood of a model fitted by ML
s100.ml <- likfit(s100, ini=c(1, .5))
s100.ml
s100.ml$loglik
loglik.GRF(s100, obj=s100.ml)

## Computing the likelihood of a variogram fitted model
s100.v <- variog(s100, max.dist=1)
s100.vf <- variofit(s100.v, ini=c(1, .5))
s100.vf
loglik.GRF(s100, obj=s100.vf)
}

\keyword{spatial}

\eof
\name{matern}

\alias{matern}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Computer Values of the Matern Correlation Function}

\description{
  This function computes values of the \eqn{\mbox{Mat\'{e}rn}}{Matern}
  correlation function
  for given distances and parameters.
}

\usage{
matern(u, phi, kappa)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{u}{a vector, matrix or array with values of the distances
    between pairs of data locations.  }
  \item{phi}{value of the range parameter \eqn{\phi}{phi}.  }
  \item{kappa}{value of the smoothness parameter \eqn{\kappa}{kappa}.  }
}

\details{
  The \eqn{\mbox{Mat\'{e}rn}}{Matern} model is defined as:
  
  \deqn{\rho(u;\phi,\kappa) =\{2^{\kappa-1}
    \Gamma(\kappa)\}^{-1} (u/\phi)^\kappa
    K_\kappa(u/\phi)}{rho(u;phi,kappa) =(2^(kappa-1) Gamma(kappa))^(-1)
    (u/phi)^\kappa K_\kappa(u/\phi)}
  
  where \eqn{\phi}{phi}  and \eqn{\kappa}{kappa} are parameters and 
  \eqn{K_\kappa(\cdot)}{K_\kappa(\ldots)} denotes the modified Bessel function of the third
  kind of order \eqn{\kappa}{kappa}.
  The family is valid for \eqn{\phi>0}{phi > 0} and \eqn{\kappa>0}{kappa > 0}.
}


\value{
  A vector matrix or array, according to the argument \code{u}, with the
  values of the \eqn{\mbox{Mat\'{e}rn}}{Matern}
  correlation function for the given distances.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }


\seealso{\code{\link{cov.spatial}} for the correlation functions
  implemented in \pkg{geoR}, and \code{\link{besselK}} for computation
  of the Bessel functions.}

\examples{
#
# Models with fixed range and varying smoothness parameter
#
curve(matern(x, phi= 0.25, kappa = 0.5),from = 0, to = 1,
      xlab = "distance", ylab = expression(rho(h)), lty = 2,
      main=expression(paste("varying  ", kappa, "  and fixed  ", phi)))
curve(matern(x, phi= 0.25, kappa = 1),from = 0, to = 1, add = TRUE)
curve(matern(x, phi= 0.25, kappa = 2),from = 0, to = 1, add = TRUE,
      lwd = 2, lty=2)
curve(matern(x, phi= 0.25, kappa = 3),from = 0, to = 1, add = TRUE,
      lwd = 2)
legend(0.6,1, c(expression(kappa == 0.5), expression(kappa == 1),
    expression(kappa == 2), expression(kappa == 3)),
    lty=c(2,1,2,1), lwd=c(1,1,2,2))
#
# Correlations with equivalent "practical range"
# and varying smoothness parameter
#
curve(matern(x, phi = 0.25, kappa = 0.5),from = 0, to = 1,
      xlab = "distance", ylab = expression(gamma(h)), lty = 2,
      main = "models with equivalent \"practical\" range")
curve(matern(x, phi = 0.188, kappa = 1),from = 0, to = 1, add = TRUE)      
curve(matern(x, phi = 0.14, kappa = 2),from = 0, to = 1,
      add = TRUE, lwd=2, lty=2)      
curve(matern(x, phi = 0.117, kappa = 2),from = 0, to = 1,
      add = TRUE, lwd=2)      
legend(0.4,1, c(expression(paste(kappa == 0.5, "  and  ",
       phi == 0.250)),
       expression(paste(kappa == 1, "  and  ", phi == 0.188)),
       expression(paste(kappa == 2, "  and  ", phi == 0.140)),
       expression(paste(kappa == 3, "  and  ", phi == 0.117))),
       lty=c(2,1,2,1), lwd=c(1,1,2,2))
}

\keyword{spatial}%-- one or more ...


\eof
\name{nlmP}

\alias{nlmP}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Adapts nlm for Constraints in the Parameter Values}

\description{
  This function adapts the \R function \code{\link{nlm}} to allow for
constraints (upper and/or lower bounds) in the values of the parameters. 
}
\usage{
nlmP(objfunc, params, lower=rep(-Inf, length(params)),
     upper=rep(+Inf, length(params)), \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{objfunc}{the function to be minimized.  }
  \item{params}{starting values for the parameters.  }
  \item{lower}{lower bounds for the variables. Defaults to \eqn{-Inf}.  }
  \item{upper}{upper bounds for the variables. Defaults to \eqn{-Inf}.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{nlm}}.  }
}

\details{
    Constraints on the parameter values are internally imposed by using
  exponential, logarithmic, and logit transformation of the parameter
values.
}

\value{
  The output is the same as for the function \code{\link{nlm}}.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{
  Patrick E. Brown \email{p.brown@lancaster.ac.uk}.\cr
  Adapted and included in \pkg{geoR} by \cr
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}

}


%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{nlm}}, \code{\link{optim}}.  }

%\examples{
%}

\keyword{spatial}
\keyword{optimize}%-- one or more ...

\eof
\name{output.control}

\alias{output.control}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{ Defines output options for prediction functions }

\description{
  Auxiliary function defining output options for
  \code{\link{krige.bayes}} and \code{\link{krige.conv}}.
}

\usage{
output.control(n.posterior, n.predictive, moments, n.back.moments,
               simulations.predictive, mean.var, quantile,
               threshold, signal, messages)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{n.posterior}{number of samples to be taken from the posterior
    distribution. Defaults to 1000. }
  \item{n.predictive}{number of samples to be taken from the
    predictive distribution. Default equals to
    \code{n.posterior}.  }
  \item{moments}{logical. Indicates whether the moments of the
    predictive distribution are returned. If \code{lambda = 1}
    there is no transformation/back-transformation.
    If \code{lambda = 0} or \code{lambda = 0.5} the moments are
    back-transformed by analytical expressions. For other cases the
    back-transformation is done by simulation. Defaults to \code{TRUE}. }
  \item{n.back.moments}{number of sample to back-transform moments by
    simulation. Defaults to 1000.  }
  \item{simulations.predictive}{logical. Defines whether to draw simulations
    from the predictive distribution.
    Only considered if prediction
    locations are provided in the argument \code{locations} of the main functions.
    Defaults to \code{FALSE} but changed to
    \code{TRUE} if an integer greater then zero is provided in the
    argument \code{n.predictive} and/or simulations are required in
    order to compute quantities required by other arguments such as
    threshold, quantiles and some values of the transformation parameter.
  } 
  \item{mean.var}{logical (optional). Indicates whether mean and
    variances of the simulations of the predictive distributions are
    computed and returned.   } 
  \item{quantile}{a (optional) numeric vector.
    If provided indicates whether quantiles of the
    simulations from the
    predictive distributions are computed and returned.
    If a vector with numbers in the interval
    \eqn{[0,1]}{[0,1]} is provided, the output includes the object
    \code{quantiles}, which contains values of corresponding
    estimated quantiles. For
    example, if \code{quantile = c(0.25, 0.50, 0.75)} the function
    returns the quartiles of the distributions at each of the
    prediction locations.
    If \code{quantile = TRUE} default values equals to \code{c(0.025, 0.5,
      0.975)}, are assumed. 
    A measure of uncertainty of the predictions, analogous to the kriging
    standard error, can be computed by \eqn{(quantile0.975 - quantile0.025)/4}.
    Only used if prediction locations are provided in the argument
    \code{locations}.  }
  \item{threshold}{a (optional) numerical vector.
    If one or more values are provided, object named
    \code{probabilities} is included in the output.
    This object contains, for each
    prediction location, the probability that the variable is less
    than or equal than the threshold provided by the user.
    Defaults to \code{FALSE}. }
  \item{signal}{logical indicating whether the signal or the variable is
    to be predicted. Defaults to \code{NULL} and changed internally in
    the functions which call \code{output.control}. See \code{DETAILS}
    below.  }
  \item{messages}{logical. Indicates
    whether or not status messages are printed on the output device
    while the function is running. Defaults to \code{TRUE}. }

}

\details{

  \bold{SIGNAL}

  This function is typically called by the \pkg{geoR}'s prediction functions
  \code{\link{krige.bayes}} and \code{\link{krige.conv}}
  defining the output to be returned by these functions.

  The underlying model
  \deqn{Y(x) = \mu + S(x) + \epsilon}{Y(x) = mu + S(x) + e}
  assumes that observations \eqn{Y(x)}{Y(x)} are noisy
  versions of a \emph{signal} \eqn{S(x)}{S(x)} and
  \eqn{Var(\epsilon)=\tau^2}{Var(e)=tau^2} is the nugget variance.

  If \eqn{\tau^2 = 0}{tau^2 = 0} the \eqn{Y} and \eqn{S} are
  indistiguishable.\cr

  If \eqn{\tau^2 > 0}{tau^2 > 0} and regarded as measurement error the
  option \code{signal} defines whether the \eqn{S} (\code{signal =
    TRUE}) or the variable \eqn{Y} (\code{signal = FALSE}) is to be
  predicted.\cr
  For the latter the predictions will "honor" the data,
  i.e. at data locations predictions will coincide with the data.\cr
  For unsampled locations, when there is no transformation of the data,
  the predicted values will be the same
  regardless whether \code{signal = TRUE} or \code{FALSE} but the
  predictions variances will differ.

  By default \code{\link{krige.bayes}} sets \code{signal = TRUE}
  and \code{\link{krige.conv}} sets \code{signal = FALSE}.

  The function \code{\link{krige.conv}} has an argument
  \code{micro.scale}. If \eqn{micro.scale > 0} the error term is
  divided as \eqn{\epsilon = \epsilon_{ms} + \epsilon_{me}}{e = e_{ms} +
    e_{me}} and the nugget variance is divided into two terms: \emph{micro-scale variance}
  and \emph{measurement error}.\cr
  If \code{signal = TRUE} the term \eqn{\epsilon_{ms}}{e_{ms}} is
  regarded as part of the signal and consequently the \emph{micro-scale variance} is added to
  the prediction variance.\cr
  If \code{signal = FALSE} the total error variance \eqn{\tau^2}{tau^2}
  is added to the prediction variance.
}

\value{
  A list with processed arguments to be passed to the main function.
}
%\references{ ~put references to the literature/web site here ~ }

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}
%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{The prediction functions \code{\link{krige.bayes}} and  \code{\link{krige.conv}}.}

%\examples{
%}

\keyword{spatial}% at least one, from doc/KEYWORDS


\eof
\name{parana}
\alias{parana}
\alias{maijun}
\docType{data}
\title{Rainfall Data from Parana State, Brasil}
\usage{data(parana)}

\description{
  This data-set was used by Diggle and Ribeiro (2001) to illustrate the methods
  discussed in the paper. The data reported analysis was
  carried out using the package \pkg{geoR}. 
  
  The data refers to average rainfall over different years for the period May-June
  (dry-season). It was collected at 143 recording stations throughout \eqn{\mbox{Paran\'{a}}}{Parana} State,
  Brasil.
}

\format{
  The object \code{parana} of the class \code{geodata}, which is a list
  containing the following components:
  
  \describe{    
    \item{\code{coords}}{a matrix with the coordinates of the recording
      stations.  }
    \item{\code{data}}{a vector with the average recorded rainfall for the May-June period.  }
    \item{\code{borders}}{a matrix with the coordinates defining the borders of
      \eqn{\mbox{Paran\'{a}}}{Parana} state.  }
    \item{\code{loci.paper}}{a matrix with the coordinates of the four
      prediction locations discussed in the paper. }
  }
}

\source{
  The data were collected at several recording stations at
  \eqn{\mbox{Paran\'{a}}}{Parana} State, Brasil, belonging to the following companies:
  COPEL, IAPAR, DNAEE, SUREHMA  and INEMET.
  
  The data base was organized by Laura Regina Bernardes Kiihl (IAPAR,
  Instituto \eqn{\mbox{Agron\^{o}mico}}{Agronomico} do \eqn{\mbox{Paran\'{a}}}{Parana}, Londrina, Brasil)
  and the fraction of the data included in this data-set was
  provided by Jacinta Loudovico Zamboti (Universidade Estadual de
  Londrina, Brasil).
  The coordinates of the borders of \eqn{\mbox{Paran\'{a}}}{Parana} State were provided
  by \eqn{\mbox{Jo\~{a}o}}{Joao} Henrique Caviglione (IAPAR).
}

\references{
  Diggle, P.J. \& Ribeiro Jr, P.J. (2002) Bayesian inference in
  Gaussian model-based geostatistics. Geographical and Environmental
  Modelling, Vol. 6, No. 2, 129-146.
}

\examples{
data(parana)
summary(parana)
plot(parana, bor=borders)
}
\keyword{datasets}


\eof
\name{pars.limits}
\alias{pars.limits}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Set limits for the parameter values}
\description{
  The functions \code{\link{likfit}} and \code{\limits{variofit}} in the
  package \pkg{geoR}
}
\usage{
pars.limits(phi = c(lower = 0, upper = +Inf),
            sigmasq = c(lower = 0, upper = +Inf),
            nugget.rel = c(lower = 0, upper = +Inf),
            kappa = c(lower = 0, upper = +Inf),
            lambda = c(lower = -3, upper = 3),
            psiR = c(lower = 1, upper = +Inf),
            psiA = c(lower = 0, upper = 2 * pi),
            tausq.rel = nugget.rel)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{phi}{a two elements vector with limits for the parameter phi. Defaults to [0, +Inf]}
  \item{sigmasq}{idem for sigmasq. Defaults to [0, +Inf]}
  \item{nugget.rel}{idem for nugget.rel. Defaults to [0, +Inf]}
  \item{kappa}{idem for kappa. Defaults to [0, +Inf]}
  \item{lambda}{idem for lambda. Defaults to [-3, +3]. Only used in
    \code{\link{likfit}}. }
  \item{psiR}{idem for psiR. Defaults to [1, +Inf]. Only used in
    \code{\link{likfit}}. }
  \item{psiA}{idem for psiA. Defaults to [0, 2 pi]. Only used in
    \code{\link{likfit}}. }
  \item{tausq.rel}{idem for tausq.rel. Defaults to [0, +Inf]}
}
\details{
  Lower and upper limits for parameter values can be
  individually specified.
  For example, including the following in the function call in
  \code{likfit} or \code{variofit}:\cr
  \code{limits = pars.limits(phi=c(0, 10), lambda=c(-2.5, 2.5))}, \cr
  will change the limits for the parameters \eqn{\phi}{phi} and \eqn{\lambda}{lambda}.
  Default values are used if the argument \code{limits} is not provided.  
  }
\value{
  A list of a 2 elements vector with limits for each parameters
}
%\references{ ~put references to the literature/web site here ~ }
%\author{ ~~who you are~~ }
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{}
\examples{
pars.limits(phi=c(0,10))
pars.limits(phi=c(0,10), sigmasq=c(0, 100))
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{models}% __ONLY ONE__ keyword per line

\eof
\name{plot.geodata}

\alias{plot.geodata}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Exploratory Geostatistical Plots}

\description{
  This function produces a \eqn{2 \times 2}{2 x 2} display
  with the following plots: 
  the first indicates the spatial locations,
  the next two shows data against the \emph{X} and
  \emph{Y} coordinates and the last is an histogram of the data values or optionally,
  a 3-D plot with spatial locations and associated data values.
}

\usage{
\method{plot}{geodata}(x, coords=x$coords, data = x$data,
             borders = NULL, trend="cte", lambda = 1, col.data = 1,
             weights.divide = NULL, lowess = FALSE, scatter3d = FALSE, \dots)
}

%- maybe also `usage' for other objects do cumented here.

\arguments{
  \item{x}{a list containing elements \code{coords} and
    \code{data} described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{ an \eqn{n \times 2}{n x 2} matrix containing in each row Euclidean
    coordinates of the \emph{n} data locations. By default it takes the
    element \code{coords} of the argument \code{geodata}.  }
  \item{data}{a vector with data values. By default it takes the
    element \code{data} of the argument \code{geodata}.  }
  \item{borders}{If an \eqn{n \times 2}{nx2} matrix or data-frame with
    the borders of the area is provided, the borders are included in the
    first plot.  }
  \item{trend}{specifies the mean part of the model. The options are:
    \code{"cte"} (constant mean - default option), \code{"1st"} (a first order polynomial
    on the coordinates), \code{"2nd"} (a second order polynomial
    on the coordinates), or a formula of the type \code{~X} where \code{X}
    is a matrix with the covariates (external trend).
    If provided the trend is "removed" using the function
    \code{\link[base]{lm}} and the residuals are plotted.  }
  \item{lambda}{value of the Box-Cox transformation parameter. Two particular cases
    are \eqn{\lambda = 1}{lambda = 1} which corresponds to no
    transformation and  \eqn{\lambda = 0}{lambda = 0} corresponding to
    the log-transformation.  }
  \item{col.data}{indicates the column number for the data
    to be plotted. Only valid if more than one data-set is available
    i.e., if the argument \code{data} is a matrix.  }
  \item{weights.divide}{if a vector of weights with the same length as
    the data is provided each  data is
    divided by the corresponding element in this vector.
    Defaults to \code{NULL}.  }
  \item{lowess}{logical. Indicates whether the function
    \code{\link{lowess}} should be used in the plots of the data against
  the coordinates.  }
  \item{scatter3d}{logical. If \code{TRUE} the last plot is
    produced by
    \code{\link[scatterplot3d]{scatterplot3d}} showing
    a 3d plot with data locations and corresponding values.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{hist}} or
    \code{\link[scatterplot3d]{scatterplot3d}}.  }
}

%\details{
%  By default, this function requires the package
%  \pkg{scatterplot3d} in order to produce a 3-D plot with data
%  locations and coordinates. If this package is not available an
%  histogram of the data replaces the 3-D plot.
%}

\value{
  A plot is produced on the graphics device. No values are returned. 
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{points.geodata}}, \code{\link[scatterplot3d]{scatterplot3d}}. }

\examples{
require(geoR)
data(s100)
plot(s100)
plot(s100, scatter3d=TRUE)

data(ca20)                        
plot(ca20, bor=borders)          # original data
plot(ca20, trend=~altitude+area) # residuals from an external trend
plot(ca20, trend='1st')          # residuals from a polynomial trend

data(SIC)
plot(sic.100, bor=sic.borders)           # original data
plot(sic.100, bor=sic.borders, lambda=0) # logarithm of the data
}

\keyword{spatial}
\keyword{dplot}%-- one or more ...

\eof
\name{plot.grf}

\alias{plot.grf}

\title{Plots Variograms for Simulated Data}

\description{
  This function plots variograms for simulated geostatistical data
  generated by the function \code{\link{grf}}. 
}

\usage{
\method{plot}{grf}(x, model.line = TRUE, plot.locations = FALSE, \dots)
}


\arguments{
  \item{x}{an object of the class \code{grf}, typically an output of the function
    \code{\link{grf}}.  }
  \item{model.line}{logical. If \code{TRUE} the true variogram model is added to the
    plot with the sample variogram(s).  }
  \item{plot.locations}{logical. If \code{TRUE} a plot with data locations is
    also shown.  }
  \item{\dots}{further arguments to be passed to the functions
    \code{\link{variog}} and \code{\link{plot}}.  }
}

%\details{
%  Plot with sample variogram(s) for simulated data are produced.
%  True variogram model used to generate the simulations is also
%  plotted if \code{model = TRUE}. 
%  Data locations are plotted first if \code{grid = TRUE}.
%}

\value{
  A plot with the empirical variogram(s) is produced on the output device. No
  values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{grf}} for simulation of Gaussian random fields,
  \code{\link{plot.variogram}} for plotting empirical variogram,
  \code{\link{variog}} for computation of empirical variograms  and
  \code{\link{plot}} for the generic plotting function.
}


\examples{
op <- par(no.readonly = TRUE)
par(mfrow=c(2,1))
sim1 <- grf(100, cov.pars=c(10, .25))
# generates simulated data
plot(sim1, plot.locations = TRUE)
#
# plots the locations and the sample true variogram model
#
par(mfrow=c(1,1))
sim2 <- grf(100, cov.pars=c(10, .25), nsim=10)
# generates 10 simulated data
plot(sim1)
# plots sample variograms for all simulations with the true model
par(op)
}

\keyword{spatial}
\keyword{dplot}

\eof
\name{plot.krige.bayes}

\alias{plot.krige.bayes}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plots Prior and/or Posterior Distributions}

\description{
  Produces
  plots the priors and posteriors distribuitions for the paramters
  \code{phi} and \code{tausq.rel} based on results returned by
  \code{\link{krige.bayes}}.
}

\usage{
\method{plot}{krige.bayes}(x, phi.dist = TRUE, tausq.rel.dist = TRUE, add = FALSE,
                 type=c("bars", "h", "l", "b", "o", "p"), thin, \dots)

}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object of the class \code{krige.bayes}, with an output of
    the funtions \code{\link{krige.bayes}}. }
  \item{phi.dist}{logical indicating whether or not plot the
    distributions for this parameter.  }
  \item{tausq.rel.dist}{logical indicating whether or not plot the
    distributions for this parameter.  }
  \item{add}{logical. If \code{TRUE} plots is added to current one.  }
  \item{type}{indicates the type of plot. Option \code{"bars"} uses the
    function \code{\link[base]{barplot}} and the others uses
    \code{\link[base]{matplot}}. }
  \item{thin}{a numerical vector defining the thining for values of the
    parameters \code{phi} and \code{tausq.rel} respectively.
    This improves visualisation when there are many values
    in the discrete distribution of the parameters.  }
  \item{\dots}{further arguments for the plotting function.  }
}

%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  For \code{plot.krige.bayes} a plot is produced or added to the current
  graphics device. No values are returned.
  
}


%\references{ ~put references to the literature/web site here ~ }

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{
\code{\link{krige.bayes}}, \code{\link{barplot}}, \code{\link{matplot}}.
}

\examples{
## See documentation for krige.bayes
}

\keyword{spatial}
\keyword{dplot}
\keyword{aplot}

\eof
\name{plot.proflik}

\alias{plot.proflik}
\alias{proflik.plot.aux1}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plots Profile Likelihoods}

\description{
  This function produces plots of the profile likelihoods computed by
  the function \code{\link{proflik}}.
}

\usage{
\method{plot}{proflik}(x, pages = c("user", "one", "two"), uni.only, bi.only,
             type.bi = c("contour", "persp"), conf.int = c(0.90, 0.95),
             yaxis.lims = c("conf.int", "as.computed"),
             by.col = TRUE, log.scale = FALSE, use.splines = TRUE,
             par.mar.persp = c(0, 0, 0, 0), ask = FALSE, \dots)
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{an object of the class \code{proflik}, typically an
    output of the function \code{\link{proflik}}.  }
  \item{pages}{specify how the plots will be arranged in the
    graphics device. The default option, \code{"user"}, uses the current
    graphical parameters. The option \code{"one"} places all the
    profiles in the same page and the option \code{"two"} places the
    univariate profiles in one page and the bivariate profiles in
    a second page.  }
  \item{uni.only}{only 1-D profiles are plotted even if the object
    contains results about the 2-D profiles.  }
  \item{bi.only}{only 2-D profile are plotted even if the object
    contains results about the 1-D profiles.  }
  \item{type.bi}{Type of plot for the 2-D profiles. Options are
    \code{"contour"} for  contour plot
    (the default) and \code{"persp"} for perspective plot.  }
  \item{conf.int}{a vector with numbers in the interval \eqn{[0,1]}
    specifying levels of the (approximated) confidence
    intervals. Defaults corresponds to the levels  90\% and 95\%.  }
  \item{yaxis.lims}{defines the lower limits for the y-axis in the 1-D
    plots. If \code{"conf.int"} the limit is determined by the level of
    the confidence interval (the default) otherwise will be determined
    by the smallest computed value.  }
  \item{by.col}{logical, If \code{TRUE} the plots are arranged by columns in a multiple graphics
    device.  }
  \item{log.scale}{plots the x-axis in the logarithmic scale. Defaults to
    \code{FALSE}.  }
  \item{use.splines}{logical. If \code{TRUE} (the default) the function
    \code{\link{spline}} is used to interpolate between the points
    computed by \code{proflik}.  }
  \item{par.mar.persp}{graphical parameters to be used with
    \code{\link{persp}} plots. For more details see \code{\link{par}}.  }
  \item{ask}{logical. Defines whether or not the user is prompted before
    each plot is produced.  }
  \item{\dots}{additional arguments to be passed to the functions
    \code{\link{plot}}, \code{\link{contour}} and/or \code{\link{persp}}.  }
}

%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  Produces plots with the profile likelihoods on the current graphics device.
  No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{proflik}} for computation of the profile
  likelihoods. For the generic plotting functions see
  \code{\link{plot}},  \code{\link{contour}}, \code{\link{persp}}.
  See \code{\link{spline}} for interpolation.  }

\examples{
# see examples in the documentation for the function proflik()
}
\keyword{spatial}
\keyword{dplot}


\eof
\name{plot.variog4}
\alias{plot.variog4}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plot Directional Variograms}

\description{
  This function plot directional variograms computed by the function
  \code{\link{variog4}}. The omnidirectional variogram can be also included
  in the plot.
}

\usage{
\method{plot}{variog4}(x, omnidirectional=FALSE, same.plot=TRUE, legend = TRUE, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{an object of the class \code{variog4}, typically an output of the function \code{\link{variog4}}. }
  \item{omnidirectional}{logical. Indicates whether the omnidirectional
    variogram is included in the plot. }
  \item{same.plot}{logical. Indicates whether the directional variograms
    are plotted in the same or separated plots.  }
  \item{legend}{logical indicating whether legends are automatically
    included in the plots.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{plot}}. Typical arguments are \code{col}, \code{lty},
    \code{lwd}. For \code{same.plot = TRUE} the arguments are passed to
    the function 
    \code{\link{matplot}} which is used to produce the plot.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  A plot is produced on the output device. No values returned.
}

\references{
  Further information about the \pkg{geoR} package can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }


\seealso{\code{\link{variog4}} for variogram calculations and
  \code{\link{matplot}} for multiple lines plotting.  }

\examples{
if(is.R()) data(s100)
s100.v4 <- variog4(s100, max.dist=1)
# Plotting variograms for the four directions
plot(s100.v4)
# changing plot options
plot(s100.v4, lwd=2)
plot(s100.v4, lty=1, col=c("darkorange", "darkblue", "darkgreen","darkviolet"))
plot(s100.v4, lty=1, lwd=2)
# including the omnidirectional variogram
plot(s100.v4, omni=TRUE)
# variograms on different plots
plot(s100.v4, omni=TRUE, same=FALSE)
}
\keyword{spatial}
\keyword{dplot}


\eof
\name{plot.variogram}

\alias{plot.variogram}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plot Empirical Variogram}

\description{
  Plots sample (empirical) variogram computed using the
  function \code{\link{variog}}.
}

\usage{
\method{plot}{variogram}(x, max.dist, vario.col = "all", scaled = FALSE,
               var.lines = FALSE, envelope.obj = NULL,
               pts.range.cex, bin.cloud = FALSE,  \dots)
}


\arguments{
  \item{x}{an object of the class \code{"variogram"}, typically an
    output of the function \code{\link{variog}}.  }
  \item{max.dist}{maximum distance for the x-axis. The default is
    the maximum distance for which the sample variogram was computed.}
  \item{vario.col}{only used if \code{obj} has information on more than
    one empirical variogram. The default \code{"all"} indicates that
    variograms of all variables should be plotted.  Alternativelly a
    numerical vector can be used to select  variables. }
  \item{scaled}{If \code{TRUE} the variogram values are divided by the
    sample variance. This allows comparison of variograms 
    of variables measured in different scales.  }
  \item{var.lines}{If \code{TRUE} a horizontal line is drawn at the value
    of the variance of the data (if \code{scaled = F}) or at 1 (if
    \code{scaled = T}).  }
  \item{envelope.obj}{adds a variogram envelope computed by
    the function \code{\link{variog.model.env}} or
    \code{\link{variog.mc.env}}.  }
  \item{pts.range.cex}{optional. A two elements vector with maximum and
    minimum values for the caracter expansion factor \code{cex}. If
    provided the point sizes in binned variogram are proportional to the
    number of pairs of points used to compute each bin.  } 
  \item{bin.cloud}{logical. If \code{TRUE}
    and the sample variogram
    was computed with the option \code{keep.cloud = TRUE}, box-plots
    of values at each bin are plotted instead of the empirical
    variograms.  }
  \item{\dots}{other arguments to be passed to the function
    \code{\link{plot}} or \code{\link{matplot}}  }

}

\details{
  This function plots empirical  variograms.
  Toghether with  \code{\link{lines.variogram}}
  can be used to compare sample variograms of different variables
  and 
  to compare variogram models against the
  empirical variogram.
  
  It uses the function \code{\link{matplot}} when plotting variograms
  for more them one variable.
}

\value{
  Produces a plot with the sample variogram on the current graphics
  device.
  No values are returned.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}
}

%\note{ ~~further notes~~ }
%
% ~Make other sections like WARNING with \section{WARNING }{....} ~
%
\seealso{
  \code{\link{variog}} for variogram calculations,
  \code{\link{lines.variogram}} and \code{\link{lines.variomodel}} for
  adding lines to the current plot,
  \code{\link{variog.model.env}} and \code{\link{variog.mc.env}} for
  variogram envelops computation, \code{\link{matplot}} for multiple
  lines plot 
  and \code{\link{plot}} for generic plot function.
}

\examples{
op <- par(no.readonly = TRUE)
sim <- grf(100, cov.pars=c(1, .2)) # simulates data
vario <- variog(sim, max.dist=1)   # computes sample variogram
par(mfrow=c(2,2))
plot(vario)                     # the sample variogram
plot(vario, scaled = TRUE)      # the scaled sample variogram
plot(vario, max.dist = 1)       # limiting the maximum distance
plot(vario, pts.range = c(1,3)) # points sizes proportional to number of pairs
par(op)
}

\keyword{spatial}
\keyword{dplot}

\eof
\name{plot.xvalid}

\alias{plot.xvalid}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plot Cross-Validation Results}

\description{
  This function produces ten plots with the results produced by the
  cross-validation function \code{\link{xvalid}}.
}

\usage{
\method{plot}{xvalid}(x, coords, borders = NULL, ask = TRUE,
            error = TRUE, std.error = TRUE, data.predicted = TRUE,
            pp = TRUE, map = TRUE, histogram = TRUE,
            error.predicted = TRUE, error.data = TRUE, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{an object of the class \code{"xvalid"}, typically an output
    from the function \code{\link{xvalid}}.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} object containing
    coordinates of the (cross-)validation locations.  }
  \item{borders}{optional. Takes a two column matrix or data-frame with
    coordinates of the borders. If provided the borders are included in
    the errors maps.  }
  \item{ask}{logical. Defines whether or not the user is prompted before
    each plot is produced.}
  \item{error}{logical. Defines whether the plots for the errors
    (\eqn{error = data - predicted}) will
    be produced.  }
  \item{std.error}{logical. Defines whether the plots for the
    standardised errors will be produced.  }
  \item{data.predicted}{logical defining whether a plot of data versus
    predicted should be displayed. Defaults to \code{TRUE}.  }
  \item{pp}{logical defining whether a \emph{pp} plot
    should be displayed. Defaults to \code{TRUE}.  }
  \item{map}{logical defining whether a map of the errors
    should be displayed. Defaults to \code{TRUE}.  }
  \item{histogram}{logical defining whether a histogram of the errors
    should be displayed. Defaults to \code{TRUE}.  }
  \item{error.predicted}{logical defining whether a plot of errors versus
    predicted should be displayed. Defaults to \code{TRUE}.  }
  \item{error.data}{logical defining whether a plot of errors versus
    data should be displayed. Defaults to \code{TRUE}.  }
  \item{\dots}{other arguments to be passed to the function
    \code{\link{plot}}.  }

}

\details{
  The number of plots to be produced will depend on the input options.
  If the graphics device is set to just one plot (something equivalent
  to \option{par(mfcol=c(1,1))}) after each graphic being displayed the user will be prompt
  to press \kbd{<return>} to see the next graphic.

  Alternativaly the user can set the graphical parameter to have several
  plots in one page. With default options for the arguments the maximum
  number of plots (10) is produced and setting \option{par(mfcol=c(5,2))})
  will display them in the same page.

  The \dQuote{errors} for the plots are defined as
  \deqn{error = data - predicted}
  and the plots uses the color blue to indicate positive errors and
  red to indicate negative erros. 
}

\value{
  No value returned. Plots are produced on the current graphics device.  
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{xvalid}} for the cross-validation computations.  }

\examples{
if(is.R()) data(s100)
wls <- variofit(variog(s100, max.dist = 1), ini = c(.5, .5), fix.n = TRUE)
xvl <- xvalid(s100, model = wls)
#
op <- par(no.readonly = TRUE)
par(mfcol = c(3,2))
par(mar = c(3,3,0,1))
par(mgp = c(2,1,0))
plot(xvl, error = FALSE, ask = FALSE)
plot(xvl, std.err = FALSE, ask = FALSE)
par(op)
}

\keyword{spatial}%-- one or more ...
\keyword{dplot}

\eof
\name{points.geodata}

\alias{points.geodata}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Plots Spatial Locations and Data Values}

\description{
  This function produces a plot with 
  points indicating the  data locations. Arguments can control the
  points sizes, patterns and colors. These can be set to be proportional
  to data values, ranks  or quantiles. Alternatively, points can be added
  to the current  plot.
}

\usage{
points.geodata(x, coords=x$coords, data=x$data, data.col = 1, borders = NULL,
               pt.divide=c("data.proportional","rank.proportional",
                           "quintiles", "quartiles", "deciles", "equal"),
               lambda = 1, trend = "cte", weights.divide = NULL, cex.min,
               cex.max, pch.seq, col.seq, add.to.plot = FALSE,
               x.leg, y.leg, dig.leg = 2, 
               round.quantiles = FALSE, graph.pars = FALSE, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{a list containing elements \code{coords} and
    \code{data} described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix containing
    coordinates of the \eqn{n} data locations in each row.
    Defaults to \code{geodata$coords}.  }
  \item{data}{a vector or matrix with data values.
    If a matrix is provided each column is regarded as one variable or realization.
    Defaults to \code{geodata$data}.  }
  \item{data.col}{the number of the data column. Only used if
    \code{data} is a matrix with columns corresponding to different
    variables or simulations.  }
  \item{borders}{If an \eqn{n \times 2}{nx2}
    matrix or data-frame with the coordinates of the borders of the
    regions is provided, the borders are added to the plot.  }
  \item{pt.divide}{defines the division of the points in categories.
    See \code{DETAILS} below for the available options.
    Defaults to \code{pt.divide = "data.proportional"}.   }
  \item{trend}{specifies the mean part of the model. The options are:
    \code{"cte"} (constant mean - default option), \code{"1st"} (a first order polynomial
    on the coordinates), \code{"2nd"} (a second order polynomial
    on the coordinates), or a formula of the type \code{~X} where \code{X}
    is a matrix with the covariates (external trend).
    If provided the trend is "removed" using the function
    \code{\link[base]{lm}} and the residuals are plotted.  }
  \item{lambda}{value of the Box-Cox transformation parameter. Two particular cases
    are \eqn{\lambda = 1}{lambda = 1} which corresponds to no
    transformation and  \eqn{\lambda = 0}{lambda = 0} corresponding to
    the log-transformation.  }
  \item{weights.divide}{if a vector of weights with the same length as
    the data is provided each  data is
    divided by the corresponding element in this vector.
    Defaults to \code{NULL}.  }
  \item{cex.min}{minimum value for the graphical parameter
    \code{cex}. This value defines the size of the point corresponding the minimum
    of the data. Defaults to 0.5.  }
  \item{cex.max}{maximum value for the graphical parameter
    \code{cex}. This value defines the size of the point corresponding the maximum
    of the data. If \code{pt.divide = "equal"} it is used to set
    the value for the
    graphical parameter \code{cex}. Defaults to 1.5.  }
  \item{pch.seq}{number(s) defining the graphical parameter \code{pch}.  }
  \item{col.seq}{number(s) defining the colors in the graphical parameter
    \code{col}.  }
  \item{add.to.plot}{logical. If \code{TRUE} the points are added
    to the current plot otherwise a display is open. Defaults to \code{FALSE}.  }
  \item{x.leg, y.leg}{\code{x} and \code{y} location of the legend.  }
  \item{dig.leg}{integer indicating the precision to be used in the
    legend values. }
  \item{round.quantiles}{logical. Defines whether or not the values
    of the quantiles should be rounded. Defaults to \code{FALSE}.  }
  \item{graph.pars}{logical. If \code{TRUE} the graphics
    parameters used to produce the plots are returned.
    Defaults to \code{FALSE}.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{plot}}, if \code{add.to.plot = FALSE}; or to the function
    \code{\link{points}}, if \code{add.to.plot = TRUE}.  }
}

\details{
  The points can be devided in categories and have different sizes
  and/or colours according to the argument
  \code{pt.divide}. The options are:
  
  \describe{
    \item{"data.proportional"}{sizes proportional to the data values.  }
    \item{"rank.proportional"}{sizes proportional to the rank of the
      data.  }
    \item{"quintiles"}{five different sizes according to the
      quintiles of the data.  }
    \item{"quartiles"}{four different sizes according to the
      quartiles of the data.  } 
    \item{"deciles"}{ten different sizes according to the
      deciles of the data. }
    \item{"equal"}{all points with the same size.  }
    \item{a numerical vector with quantiles}{the values in the
      vector will be used by the function \code{\link{cut}} as break
      points to divide the data in classes. }
  }

  For cases where points have different sizes the arguments
  \code{cex.min} and \code{cex.max} set the minimum and the maximum
  point sizes. Additionally,
  \code{pch.seq} can set different patterns for the points and
  \code{col.seq} can be used to define colors.
  For example, different colors
  can be used for quartiles, quintiles and deciles while a sequence of
  gray tones (or a color sequence) can be used
  for point sizes proportional to the data or their ranks.
  For more details see the section \code{EXAMPLES}.
}

\value{
  A plot is created or points are added to the current graphics device.\cr
  By default no value is returned. However, if \code{graph.pars = TRUE}
  a list with graphical parameters used to produce the plot is returned.
  According to the input options, the list has
  some or all of the
  following components:

  \item{quantiles}{the values of the quantiles used to divide the data.  }
  \item{cex}{the values of the graphics expansion parameter \code{cex}.
  } 
  \item{col}{the values of the graphics color parameter \code{col}.  }
  \item{pch}{the values of the graphics pattern parameter \code{pch}.  }

}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ }

\seealso{\code{\link{plot.geodata}} for another display of the data and
  \code{\link{points}} and \code{\link{plot}} for information on the
  generic \R functions. The documentation of
  \code{\link{par}} provides details on graphical parameters.
  For  color schemes in \R see \code{\link{gray}} and
  \code{\link{rainbow}}.  }

\examples{
data(s100)
op <- par(no.readonly = TRUE)
par(mfrow=c(2,2), mar=c(3,3,1,1), mgp = c(2,1,0))
points(s100, xlab="Coord X", ylab="Coord Y")
points(s100, xlab="Coord X", ylab="Coord Y", pt.divide="rank.prop")
points(s100, xlab="Coord X", ylab="Coord Y", cex.max=1.7,
               col=gray(seq(1, 0.1, l=100)), pt.divide="equal")
points(s100, pt.divide="quintile", xlab="Coord X", ylab="Coord Y")
par(op)

data(ca20)
points(ca20, pt.div='quartile', x.leg=4900, y.leg=5850, bor=borders)
}

\keyword{spatial}
\keyword{dplot}
\keyword{aplot}
%-- one or more ...




\eof
\name{polygrid}

\alias{polygrid}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Coordinates of Points Inside a Polygon }

\description{
  This function builds a rectangular grid and extracts points which are
  inside of an internal polygonal region.
}

\usage{
polygrid(xgrid, ygrid, borders, vec.inout = FALSE)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{xgrid}{grid values in the \emph{x}-direction.  }
  \item{ygrid}{grid values in the \emph{y}-direction.  }
  \item{borders}{a matrix with polygon coordinates defining the
    borders of the region.  }
  \item{vec.inout}{logical. If \code{TRUE} a logical vector is included
    in the output indicating whether each point of the grid is inside
    the polygon.  Defaults to \code{FALSE}.  }
}

\details{
  This function requires the package \pkg{splancs}.

  The function works as follows:
  First it creates a grid using the \R function
  \code{\link[base]{expand.grid}} and then it uses the function
  \code{\link[splancs]{inout}} from the package \pkg{splancs} to extract the points
  of the grid which are inside the polygon.
  
  Within the package \pkg{geoR}
  this function is typically used to select points in a non-rectangular
  region to perform spatial prediction
  using \code{\link{krige.bayes}}, \code{\link{krige.conv}} or
  \code{\link{ksline}}. It is also useful to produce
  image or perspective plots of the prediction results.
}

\value{
  A list with components:
  \item{xypoly}{an \eqn{n \times 2}{n x 2} matrix with the coordinates of the points inside the polygon.  }
  \item{vec.inout}{logical, a vector indicating whether each point of
    the rectangular grid is inside the polygon. Only returned if \code{vec.inout = TRUE}.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{expand.grid}}, \code{\link[splancs]{inout}. }
}

\examples{
if(require(splancs)){
 poly <- matrix(c(.2, .8, .7, .1, .2, .1, .2, .7, .7, .1), ncol=2)
 plot(0:1, 0:1, type="n")
 lines(poly)
 poly.in <- polygrid(seq(0,1,l=11), seq(0,1,l=11), poly, vec=TRUE)
 points(poly.in$xy)
}
}
\keyword{spatial}%-- one or more ...

\eof
\name{proflik}

\alias{proflik}

\alias{proflik.aux0}
\alias{proflik.aux1}
\alias{proflik.aux10}
\alias{proflik.aux11}
\alias{proflik.aux1.1}
\alias{proflik.aux12}
\alias{proflik.aux13}
\alias{proflik.aux14}
\alias{proflik.aux15}
\alias{proflik.aux16}
\alias{proflik.aux17}
\alias{proflik.aux18}
\alias{proflik.aux19}
\alias{proflik.aux2}
\alias{proflik.aux20}
\alias{proflik.aux21}
\alias{proflik.aux21.1}
\alias{proflik.aux22}
\alias{proflik.aux23}
\alias{proflik.aux24}
\alias{proflik.aux27}
\alias{proflik.aux28}
\alias{proflik.aux30}
\alias{proflik.aux3}
\alias{proflik.aux31}
\alias{proflik.aux32}
\alias{proflik.aux33}
\alias{proflik.aux4}
\alias{proflik.aux5}
\alias{proflik.aux6}
\alias{proflik.aux7}
\alias{proflik.aux8}
\alias{proflik.aux9}
\alias{proflik.cov}
\alias{proflik.lambda}
\alias{proflik.main}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Computes Profile Likelihoods}

\description{
  Computes profile likelihoods for model parameters
  previously estimated using the function
  \code{\link{likfit}}. 
}

\usage{
proflik(obj.likfit, geodata, coords = geodata$coords,
        data = geodata$data, sill.values, range.values,
        nugget.values, nugget.rel.values, lambda.values, 
        sillrange.values = TRUE, sillnugget.values = TRUE,
        rangenugget.values = TRUE, sillnugget.rel.values = FALSE,
        rangenugget.rel.values = FALSE, silllambda.values = FALSE,
        rangelambda.values = TRUE,  nuggetlambda.values = FALSE,
        nugget.rellambda.values = FALSE,
        uni.only = TRUE, bi.only = FALSE, messages, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{obj.likfit}{an object of the class \code{likfit}, typically an
    output of the function \code{\link{likfit}}.}
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{ an \eqn{n \times 2}{n x 2} matrix containing in each row Euclidean
    coordinates of the \eqn{n} data locations. By default it takes the
    element \code{coords} of the argument \code{geodata}.  }
  \item{data}{a vector with data values. By default it takes the
    element \code{data} of the argument \code{geodata}.  }
  \item{sill.values}{set of values of the partial sill parameter
    \eqn{\sigma^2}{sigma^2} for which the profile likelihood will be
    computed.  }
  \item{range.values}{set of values of the range parameter
    \eqn{\phi}{phi} for which the profile likelihood will be computed.  }
  \item{nugget.values}{set of values of the nugget parameter
    \eqn{\tau^2}{tau^2} for which the profile likelihood will be
    computed. Only used if the model was fitted using the function
    \code{\link{likfit}} with the option \code{fix.nugget = FALSE}.  }
  \item{nugget.rel.values}{set of values of the relative nugget parameter
    \eqn{\tau_{R}^{2}}{tauR^2} for which the profile likelihood will be
    computed. Only used if the model was fitted using the function
    \code{\link{likfit}} with the option \code{fix.nugget = FALSE}.  }
  \item{lambda.values}{set of values of the Box-Cox transformation parameter
    \eqn{\lambda}{lambda} for which the profile likelihood will be
    computed. Only to be used if the model was fitted using the function
    \code{\link{likfit}} with the option \code{fix.lambda = FALSE}.  }
  \item{sillrange.values}{logical indicating
    whether or not the 2-D profile likelihood should be computed.
    Only valid if \code{uni.only = FALSE}.  }
  \item{sillnugget.values}{as above.  }
  \item{rangenugget.values}{as above.  }
  \item{sillnugget.rel.values}{as above.  }
  \item{rangenugget.rel.values}{as above.  }
  \item{silllambda.values}{as above.  }
  \item{rangelambda.values}{as above.  }
  \item{nuggetlambda.values}{as above.  }
  \item{nugget.rellambda.values}{as above.  }
  \item{uni.only}{as above.  }
  \item{bi.only}{as above.  }
  \item{messages}{logical. Indicates whether
    status messages should be printed on the screen (i.e. current output device)
    while the function is running.  }
%  \item{minimisation.function}{minimization function to be
%    used. Defaults to \code{\link{optim}}.  }
  \item{\dots}{additional parameters to be passed to the minimization
    function.  }
}

\details{
  The functions \code{proflik.*} are auxiliary functions used to
  compute the profile likelihoods. These functions are
  internally called by the
  minimization functions when estimating the model parameters.
}

\value{
  An object of the class \code{"proflik"} which is 
  a list. Each element contains values of a parameter (or a pair of
  parameters for 2-D profiles) and the
  corresponding value of the profile likelihood.
  The components of the output will vary according to the
  input options.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\note{
  \enumerate{
    \item Profile likelihoods for Gaussian Random Fields are usually
    uni-modal. 
    Unusual or jagged shapes can
    be due to the lack of the convergence in the numerical minimization
    for particular values of the parameter(s).  
    If this is the case it might be necessary to pass \code{control} arguments 
    to the minimization functions using the argument \dots.
    It's also advisable to try the different options for the
    \code{minimisation.function} argument.
    See documentation of the functions \code{\link{optim}} and/or
    \code{\link{nlm}} for further details.
    
    \item 2-D profiles can be computed by setting the argument
    \code{uni.only = FALSE}. However, before computing 2-D profiles be
    sure they are really necessary.
    Their computation can be time demanding since it 
    is performed on a grid determined by the
    cross-product of the values defining the 1-D profiles.
        
    \item There is no "default strategy" to find reasonable values for the
    x-axis.
    They must be found in a "try-and-error" exercise. It's recommended
    to use short sequences in the initial attempts.
    The \code{EXAMPLE} section below illustrates this.
  }

}

\seealso{\code{\link{plot.proflik}} for graphical output,
  \code{\link{likfit}} for the parameter estimation,
  \code{\link{optim}} and \code{\link{nlm}} for further details about
  the minimization functions.  }

\examples{
op <- par(no.readonly=TRUE)
data(s100)
ml <- likfit(s100, ini=c(.5, .5), fix.nug=TRUE)
## a first atempt to find reasonable values for the x-axis:
prof <- proflik(ml, s100, sill.values=seq(0.5, 1.5, l=4),
                range.val=seq(0.1, .5, l=4))
par(mfrow=c(1,2))
plot(prof)
## a nicer setting 
\dontrun{
prof <- proflik(ml, s100, sill.values=seq(0.45, 2, l=11),
                range.val=seq(0.1, .55, l=11))
plot(prof)
## to include 2-D profiles use:
## (commented because this is time demanding)
#prof <- proflik(ml, s100, sill.values=seq(0.45, 2, l=11),
#                range.val=seq(0.1, .55, l=11), uni.only=FALSE)
#par(mfrow=c(2,2))
#plot(prof, nlevels=16)
}
par(op)
}

\keyword{spatial}

\eof
\name{read.geodata}

\alias{read.geodata}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Reads and Converts Data to geoR Format}

\description{
  Reads data from a \emph{ASCII} file and converts it to an object of the
  \code{\link{class}} \code{geodata}, the standard data format for the
  \pkg{geoR} package.
}

\usage{
read.geodata(file, header = FALSE, coords.col = 1:2, data.col = 3,
             data.names = NULL, covar.col = NULL,
             covar.names = "header", realisations = NULL,
             na.action = c("ifany", "ifdata", "ifcovar", "none"),
             rep.data.action, rep.covar.action, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{file}{a string with the name of the \emph{ASCII} file.  }
  \item{header}{logical.
    Indicates whether the variables names should be
    read from the first line of the input file.   }
  \item{coords.col}{a vector with the numbers of the columns containing the
    coordinates.  }
  \item{data.col}{a scalar or vector with the number of the column(s)
    containing the data.  }
  \item{data.names}{a string or vector of strings with names for
    the data columns. Only valid if there is more than one column of
    data.
    By default the names in the original object are used.  }
  \item{covar.col}{optional. A scalar or vector with the number
    of the column(s) with the values of the covariate(s).  }
  \item{covar.names}{optional. A vector with the names of the
    the covariates. By default the names in the original object are
    used.  }
  \item{realisations}{optional. A vector indicating the replication
    number. For more details see documentation for
    \code{\link{as.geodata}}.  }
  \item{na.action}{a string. Defines action to be taken in the presence of
    \code{NA}'s. For more details see documentation for
    \code{\link{as.geodata}}.  }
  \item{rep.data.action}{a string or a function. Defines action to be taken when there is more than
    one data at the same location. For more details see documentation for
    \code{\link{as.geodata}}.  }
  \item{rep.covar.action}{a string or a function. Defines action to be taken when there is more than
    one covariate at the same location. For more details see documentation for
    \code{\link{as.geodata}}.  }
  \item{\dots}{further arguments to be passed to the function \code{\link{read.table}}.  }
}

\details{
  The function \code{\link{read.table}} is used to read the data from the
\emph{ASCII} file and then \code{\link{as.geodata}} is used to convert
to an object of the \code{\link{class}} \code{geodata}.
}

\value{
  An object of the \code{\link{class}} \code{geodata}.
  See documentation for the function \code{\link{as.geodata}} for
  further details.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }
%
% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{as.geodata}} to convert existing \R objects,
  \code{\link{read.table}}, the basic \R function used to  read \emph{ASCII} files,
  and \code{\link{list}} for detailed information about lists.}

%\examples{}
\keyword{spatial}
\keyword{manip}%-- one or more ...


\eof
\name{s100 and s121}

\docType{data}

\alias{s100}
\alias{s121}


\title{Simulated Data-Sets which Illustrate the Usage of the Package
  geoR  }

\usage{
data(s100)

data(s121)
}

\description{
  These two simulated data sets are the ones used in the Technical
  Report which describes the package \pkg{geoR}
  (see reference below). 
  These data-sets are used in several examples throughout the package
  documentation.
}


\format{
  Two objects of the \code{\link{class}} \code{geodata}. Both are lists
  with the following components:
  \describe{
    \item{\code{coords}}{the coordinates of data locations.  }
    \item{\code{data}}{the simulated data. Notice that for \code{s121}
      this a \eqn{121 \times 10}{101x10} matrix with 10 simulations.  }
    \item{\code{cov.model}}{the correlation model.  }
    \item{\code{nugget}}{the values of the nugget parameter.  }
    \item{\code{cov.pars}}{the covariance parameters.  }
    \item{\code{kappa}}{the value of the parameter \emph{kappa}.  }
    \item{\code{lambda}}{the value of the parameter \emph{lambda}.  }
  }
}

\references{
  Ribeiro Jr, P.J. and Diggle, P.J. (1999) geoS: A geostatistical
  library for S-PLUS. \emph{Technical report ST-99-09, Dept of Maths and
  Stats, Lancaster University}. 
  
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\examples{
data(s100)
plot(s100)
data(s121)
plot(s121, type="l")
}
\keyword{datasets}


\eof
\name{s256i}
\docType{data}

\alias{s256i}


\title{Simulated Data-Set which Illustrate the Usage of krige.bayes  }

\description{
  This is the simulated data-set used in the Technical
  Report which describes the implementation of the function \bold{krige.bayes}
  (see reference below). 
}

\usage{
data(s256i)
}

\format{
  Two objects of the \code{\link{class}} \code{geodata}. Both are lists
  with the following components:
  \describe{
    \item{\code{coords}}{the coordinates of data locations.  }
    \item{\code{data}}{the simulated data.  }
  }
}

\references{
  Ribeiro Jr, P.J. and Diggle, P.J. (1999)
  Bayesian inference in Gaussian model-based geostatistics.
  \emph{Technical report ST-99-08, Dept of Maths and
    Stats, Lancaster University}. 
  
  Further information about the \pkg{geoR} package can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\examples{
data(s256i)
points(s256i, pt.div="quintiles", cex.min=1, cex.max=1)
}
\keyword{datasets}


\eof
\name{sample.posterior}

\alias{sample.posterior}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Samples from the posterior distribution}

\description{
  Sample quadruples \eqn{(\beta, \sigma^2, \phi, \tau^2_{rel})}{(beta,
    sigma^2, phi, tau^2.rel)} from the posterior
  distribution returned by \code{\link{krige.bayes}}.
}

\usage{
sample.posterior(n, kb.obj)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{n}{number of samples}
  \item{kb.obj}{on object with an output of \code{\link{krige.bayes}}.  }
}

%\details{
%  
%}

\value{
  A \eqn{n \times 4}{n x 4} data-frame with samples from the posterior distribution of the model parameters.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{krige.bayes}} and \code{\link{sample.posterior}}.  }

%\examples{
%}

\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{distribution}% __ONLY ONE__ keyword per line

\eof
\name{sample.prior}

\alias{sample.prior}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Samples from the prior}

\description{
  Sample quadruples \eqn{(\beta, \sigma^2, \phi, \tau^2_{rel})}{(beta,
    sigma^2, phi, tau^2.rel)} from the prior distribution of parameters specifying a Gaussian
  random field. Typically the prior is specified as for when calling \code{\link{krige.bayes}}.
}

\usage{
sample.prior(n, kb.obj=NULL, prior=prior.control())
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{n}{number of samples}
  \item{kb.obj}{on object with an output of \code{\link{krige.bayes}}.  }
  \item{prior}{an call to \code{\link{prior.control}}. Not needed if
    \code{kb.obj} is provided.  }
}
%\details{
%  
%}

\value{
  A data-frame with samples from the model parameters.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{krige.bayes}} and \code{\link{sample.posterior}}.  }

\examples{
sample.prior(50, prior=prior.control(beta.prior = "normal", beta = .5, beta.var.std=0.1, sigmasq.prior="sc", sigmasq=1.2, df.sigmasq= 2, phi.prior="rec", phi.discrete = seq(0,2, l=21)))
}


\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{distribution}% __ONLY ONE__ keyword per line

\eof
\name{set.coords.lims}

\alias{set.coords.lims}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Sets Limits to Scale Plots}
\description{
  This is an function typically called by functions in the package
  \pkg{geoR} to set limits for the axis when plotting spatial data.
}

\usage{
set.coords.lims(coords, xlim, ylim)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{coords}{an \eqn{n \times 2}{nx2} matrix with coordinates.  }
  \item{xlim, ylim} {the ranges to be encompassed by the x and y axes.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  A \eqn{2 \times 2} matrix with limits for the axis.
}

%\references{ ~put references to the literature/web site here ~ }
\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}
%\note{ ~~further notes~~ }

 
%\seealso{ ~~objects to SEE ALSO as \code{\link{~~fun~~}}, ~~~ }

%\examples{}
\keyword{spatial}% at least one, from doc/KEYWORDS


\eof
\name{statistics.predictive}

\alias{statistics.predictive}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Summary statistics from predictive distributions}
\description{
  Computes summaries based on simulations of predictive distribution
  returned by
  \code{\link{krige.bayes}} and \code{\link{krige.conv}}.

}
\usage{
statistics.predictive(simuls, mean.var = TRUE, quantile, threshold)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{simuls}{object with simulations from the predictive distribution}
  \item{mean.var}{Logical. Indicates whether or not to compute mean and variances
    of the simulations at each location.  }
  \item{quantile}{defines quantile estimator. See
    documentation for \code{\link{output.control}} .  }
  \item{threshold}{defines probability estimator. See
    documentation for \code{\link{output.control}}.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  A list with one ore more of the following components.
  \item{mean }{mean at each prediction location.  }
  \item{variance}{variance at each prediction location.  }
  \item{quantiles}{quantiles, at each prediction location.}  
  \item{probabilities}{probabilities, at each prediction location, of been below
    the provided threshold.  }

}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~


%\examples{
%}
\keyword{spatial}% at least one, from doc/KEYWORDS

\eof
\name{subarea}
\alias{subarea}

\title{Selects a Subarea from a Geodata Object}
\description{
  
}
\usage{
subarea(geodata, xlim, ylim, \ldots)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{geodata}{an object of the class \code{geodata} as defined in \code{\link{as.geodata}}.  }
  \item{xlim}{optional, a vector with selected range of the x-coordinates. }
  \item{ylim}{optional, a vector with selected range of the y-coordinates. }
  \item{\ldots}{further arguments to be passed to \code{\link{zoom.coords}}.  }
}
\details{
  The function copies the original \code{geodata} object and
  selects values of \code{$coords}, \code{$data}, \code{$borders},
  \code{$covariate} and \code{$units.m} which lies within the selected
  sub-area.
  Remaining components of the geodata objects are untouched.
  
  If \code{xlim} and/or \code{ylim} are not provided the function
  prompts the user to click 2 points defining an retangle defining the 
  subarea on a existing plot.

}
\value{
  Returns an \code{geodata} object, subsetting of the original one provided.
}
%\references{ ~put references to the literature/web site here ~ }
\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{\code{\link{zoom.coords}}, \code{\link{locator}}}

\examples{
foo <- matrix(c(4,6,6,4,2,2,4,4), nc=2)
foo1 <- zoom.coords(foo, 2)
foo1
foo2 <- coords2coords(foo, c(6,10), c(6,10))
foo2
plot(1:10, 1:10, type="n")
polygon(foo)
polygon(foo1, lty=2)
polygon(foo2, lwd=2)
arrows(foo[,1], foo[,2],foo1[,1],foo1[,2], lty=2)
arrows(foo[,1], foo[,2],foo2[,1],foo2[,2])
legend(1,10, c("foo", "foo1 (zoom.coords)", "foo2 (coords2coords)"), lty=c(1,2,1), lwd=c(1,1,2), cex=1.7)

## "zooming" part of The Gambia map
data(gambia)
gb <- gambia.borders/1000
gd <- gambia[,1:2]/1000
plot(gb, ty="l", asp=1, xlab="W-E (kilometres)", ylab="N-S (kilometres)")
points(gd, pch=19, cex=0.5)
r1b <- gb[gb[,1] < 420,]
rc1 <- rect.coords(r1b, lty=2)

r1bn <- zoom.coords(r1b, 1.8, xoff=90, yoff=-90)
rc2 <- rect.coords(r1bn, xz=1.05)
segments(rc1[c(1,3),1],rc1[c(1,3),2],rc2[c(1,3),1],rc2[c(1,3),2], lty=3)

lines(r1bn)
r1d <- gd[gd[,1] < 420,]
r1dn <- zoom.coords(r1d, 1.7, xlim.o=range(r1b[,1],na.rm=TRUE), ylim.o=range(r1b[,2],na.rm=TRUE), xoff=90, yoff=-90)
points(r1dn, pch=19, cex=0.5)
text(450,1340, "Western Region", cex=1.5)

if(require(geoRglm)){
data(rongelap)
points(rongelap, bor=borders)
## zooming the western area
rongwest <- subarea(rongelap, xlim=c(-6300, -4800))
points(rongwest, bor=borders)
## now zooming in the same plot
points(rongelap, bor=borders)
rongwest.z <- zoom.coords(rongwest, xzoom=3, xoff=2000, yoff=3000)
points(rongwest.z, bor=borders, add=TRUE)
rect.coords(rongwest$sub, quiet=TRUE)
rect.coords(rongwest.z$sub, quiet=TRUE)
}
}
\keyword{spatial}% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{summary.geodata}

\alias{summary.geodata}
\alias{print.summary.geodata}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Summaries for geodata object}
\description{
  Sumarises each of the main elements of an object of the class \code{geodata}.
}
\usage{
summary.geodata(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{an object of the class \code{geodata}.  }
  \item{\dots}{further arguments to be passed to the function
    \code{\link{summary.default}}.  }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
  A list with components
  \item{coords.summary }{a matrix with minimum and maximum values for
    the coordinates.  }
  \item{borders.summary }{a matrix with minimum and maximum values for
    the coordinates. Only returned if there is an element \code{borders}
  in the \code{geodata} object.  }
  \item{data.summary }{summary statistics (min, max, quartiles and mean)
    for the data.  }
  \item{units.m.summary }{summary statistics (min, max, quartiles and mean)
    for the offset variable. Only returned if there is an element \code{units.m}
  in the \code{geodata} object. }
  \item{covariate.summary }{summary statistics (min, max, quartiles and mean)
    for the covariate(s). Only returned if there is an element \code{covariate}
    in the \code{geodata} object. }
}
\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }
%
% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{summary}}, \code{\link{as.geodata}}. }

\examples{
data(s100)
summary(s100)

data(ca20)
summary(ca20)
}
\keyword{univar}
\keyword{spatial}

\eof
\name{summary.likGRF}

\alias{summary.likGRF}
\alias{print.summary.likGRF}
\alias{print.likGRF}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Summarizes Parameter Estimation Results for Gaussian Random Fields}

\description{
  Summarizes results returned by the function \code{\link{likfit}}.\cr
  Functions are \emph{methods} for \code{\link[base]{summary}} and
  \code{\link[base]{print}} for the classes \code{likGRF} and \code{summary.likGRF}.
}

\usage{
\method{summary}{likGRF}(object, \dots)
\method{print}{likGRF}(x, digits = max(3, getOption("digits") - 3), \dots)
\method{print}{summary.likGRF}(x, digits = max(3, getOption("digits") - 3), \dots)
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{an object of \code{\link[base]{class}} \emph{likGRF}, typically
    a result of a call to \code{\link{likfit}}.  }
  \item{x}{an object of \code{\link[base]{class}} \emph{likGRF} or
    \code{\link[geoR]{class}} \emph{summary.likGRF}, typically
    resulting from a call to \code{\link{likfit}}.  }
  \item{digits}{the number of significant digits to use when printing. }
  \item{\dots}{extra arguments for \code{\link[base]{print}}.  }
}

\details{
  A detailed summary of a object of the class \code{likGRF} is produced by
  by \code{summary.likGRF} and printed by \code{print.summary.likGRF}.
  This includes model specification with values of fixed and estimated parameters. 
  A simplified summary of the parameter estimation is printed by
  \code{print.likGRF}. 
}

\value{
  \code{print.likGRF} prints the parameter estimates and the value of the
  maximized likelihood.\cr
  \code{summary.likGRF} returns a list with main results of a call to
  \code{\link{likfit}}.\cr
  \code{print.summary.likGRF} prints these results on the screen (or other
  output device) in a "nice" format.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{likfit}}, \code{\link[base]{print}},
  \code{\link[base]{summary}}.
}

\examples{
# See examples for the function likfit()
}

\keyword{spatial}
\keyword{print}


\eof
\name{summary.variomodel}

\alias{summary.variomodel}
\alias{print.summary.variomodel}
\alias{print.variomodel}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Summarize Results of Variogram Estimation}

\description{
  This function prints a summary of the parameter estimation results given
  by \code{\link{variofit}}.
}

\usage{
\method{summary}{variomodel}(object, \dots)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{object}{an object of the class \code{"variomodel"}
    typically an output of
    \code{\link{variofit}}.}
  \item{\dots}{other arguments to be passed to the function
    \code{\link{print}} or \code{\link{summary}}.  }

}

%\details{
%}

\value{
  Prints a summary of the estimation results on the screen or
  other output device. 
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{The functions \code{\link{variofit}} for
  variogram based estimation. For likelihood based parameter estimation see  \code{\link{likfit}}. }

\examples{
data(s100)
s100.vario <- variog(s100, max.dist=1)
wls <- variofit(s100.vario, ini=c(.5, .5), fix.nugget = TRUE)
wls
summary(wls)
}

\keyword{spatial}













\eof
\name{trend.spatial}

\alias{trend.spatial}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Builds the Trend Matrix}

\description{
  Builds the \emph{trend} matrix in accordance to a specification
  of the mean provided by the user. 
}

\usage{
trend.spatial(trend, geodata)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{trend}{specifies the mean part of the model.
    See \code{DETAILS} below.  }
  \item{geodata}{optional. An object of the class \code{geodata} as described in
    \code{\link{as.geodata}}. }
}

\details{
  The implicity model assumes that there is an underlying process
  with mean \eqn{\mu(x)}{mu(x)}, where \eqn{x = (x_1, x_2)}{x = (x1, x2)} denotes the coordinates
  of a spatial location.
  The argument \code{trend} defines the form of the mean and the
  following options are allowed:
  
  \itemize{
    \item{\code{"cte"}}{the mean is assumed to be constant over the region,
      in which case \eqn{\mu(x)= \mu}{mu(x) = mu}. This is the default
      option.  }
    \item{\code{"1st"}}{the mean is assumed to be a first order polynomial
      on the coordinates:
      \deqn{\mu(x)= \beta_0 + \beta_1 x_1 + \beta_2 x_2}{%
	mu(x) = beta0 + beta1*x1 + beta2*x2. }}
    \item{\code{"2nd"}}{the mean is assumed to be a second order polynomial
      on the coordinates:
      \deqn{\mu(x)= \beta_0 + \beta_1 x_1 + \beta_2 x_2 + \beta_3 (x_1)^2 +
	\beta_4 (x_2)^2 + \beta_5 x_1 *  x_2}{%
	\mu(x)= beta0 + beta1*x1 + beta2*x2 + beta3*(x1)^2 +
	beta4*(x2)^2 + beta5*x1*x2.}}
    \item{\code{~ model}}{a model specification. See
      \code{\link{formula}} for further details on how to specify
      a model in \R  using formulas. Notice that the model term before
      the \code{~} is not necessary.
      Typically used to include covariates
      (external trend) in the model.}
  }
  
  Denote by \eqn{x_1} and \eqn{x_2} the spatial coordinates.
  The following specifications are equivalent:
  \itemize{
    \item \code{trend = "1st"} and \code{trend = ~ x1 + x2}
    \item \code{trend = "2nd"} and \code{trend = ~ x1 + x2 + I(x1^2) + I(x2^2) + I(x1*x2)}
  }
  
  \bold{Search path for covariates}\cr
  Typically, functions in the package \pkg{geoR} which calls
  \code{trend.spatial} will have the arguments \code{geodata},
  \code{coords} and \code{data}. 
  
  When the trend is specifed as \code{trend = ~ model}
  the terms included in the model will be searched for in the following
  path sequence (in this order):
  \enumerate{
    \item as elements of the list \code{geodata}
    \item as columns in a data-frame \code{geodata$covariates}
    \item as columns in a data-frame \code{geodata$data}
    \item in the session search path
  }
}

\value{
  An object of the class \code{trend.spatial}
  which is an \eqn{n \times p}{n x p} \emph{trend}
  matrix, where \eqn{n}
  is the number of spatial
  locations and \eqn{p} is the number of mean parameters in the model. 
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\note{This is an auxiliary function typically called by
  other \pkg{geoR} functions.
}

\seealso{The section \code{DETAILS} in the documentation for
  \code{\link{likfit}} for more about the underlying model.
}

\examples{
data(SIC)
# a first order polynomial trend
trend.spatial("1st", sic.100)[1:5,]
# a second order polynomial trend
trend.spatial("2nd", sic.100)[1:5,]
# a trend with a covariate
trend.spatial(~altitude, sic.100)[1:5,]
# a first degree trend plus a covariate
trend.spatial(~coords+altitude, sic.100)[1:5,]
}

\keyword{spatial}%-- one or more ...


\eof
\name{varcov.spatial}

\alias{varcov.spatial}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Computes Covariance Matrix and Related Results}

\description{
  This function builds the covariance matrix for a set of spatial
  locations, given the covariance parameters.
  According to the input options  other results related to the covariance matrix
  (such as decompositions, determinants, inverse. etc)
  can also be returned.
}

\usage{
varcov.spatial(coords = NULL, dists.lowertri = NULL,
               cov.model = "matern", kappa = 0.5, nugget = 0,
               cov.pars = stop("no cov.pars argument"), 
               inv = FALSE, det = FALSE,
               func.inv = c("cholesky", "eigen", "svd", "solve"),
               scaled = FALSE,  only.decomposition = FALSE,
               sqrt.inv = FALSE, try.another.decomposition = TRUE,
               only.inv.lower.diag = FALSE)

}
	     
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix with the coordinates
    of the data locations. If not provided
    the argument \code{dists.lowertri} should be provided instead.}
  \item{dists.lowertri}{a vector with the lower triangle of the matrix
    of distances between pairs of data points. If not provided
    the argument \code{coords} should be provided instead.}
  \item{cov.model}{a string indicating the type of the correlation
    function. More details in the
    documentation for \code{\link{cov.spatial}}.
    Defaults are equivalent to the \emph{exponential} model.  }
  \item{kappa}{values of the additional smoothness parameter, only required by
    the following correlation
    functions: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"} and
    \code{"gneiting.matern"}.  }
  \item{nugget}{the value of the nugget parameter \eqn{\tau^2}{tau^2}.  }
  \item{cov.pars}{a vector with 2 elements or an \eqn{ns \times 2} matrix with
    the covariance parameters. The first element (if a vector) or first
    column (if a matrix) corresponds to the variance parameter \eqn{\sigma^2}{sigma^2}.
    second element or column corresponds to the correlation function parameter
    \eqn{\phi}{phi}. If a matrix is provided each row corresponds to
    the parameters of one \emph{spatial structure}. Models with several structures
    are also called \emph{nested models} in the geostatistical
    literature.  }
  \item{inv}{if \code{TRUE} the inverse of covariance
    matrix is returned. Defaults to \code{FALSE}.}
  \item{det}{if \code{TRUE} the  logarithmic of the square root of the
    determinant of the covariance
    matrix is returned. Defaults to \code{FALSE}.}
  \item{func.inv}{algorithm used for the decomposition and inversion of the covariance
    matrix. Options are \code{"chol"} for Cholesky decomposition,
    \code{"svd"} for singular value decomposition and \code{"eigen"} for
    eigenvalues/eigenvectors decomposition. Defaults to \code{"chol"}.}
  \item{scaled}{logical indicating whether the covariance matrix should
    be scaled. If \code{TRUE} the partial sill
    parameter \eqn{\sigma^2}{sigma^2} is set to 1. Defaults to \code{FALSE}.}
  \item{only.decomposition}{logical. If \code{TRUE} only the square root
    of the covariance matrix is
    returned. Defaults to \code{FALSE}.  }
  \item{sqrt.inv}{if \code{TRUE} the square root of the inverse of  covariance
    matrix is returned. Defaults to \code{FALSE}.}
  \item{try.another.decomposition}{logical. If \code{TRUE} and the argument
    \code{func.inv} is one of \code{"cholesky"}, \code{"svd"} or
    \code{"solve"}, the matrix decomposition or inversion is tested and,
    if it fails, the argument \code{func.inv} is re-set to \code{"eigen"}.  }
  \item{only.inv.lower.diag}{logical. If \code{TRUE} only the lower triangle and
    the diagonal of the inverse of the covariance matrix are
    returned. Defaults to \code{FALSE}.}
}


\details{
  The elements of the covariance matrix are computed by the function
  \code{\link{cov.spatial}}. Typically  this is an auxiliary function  called by other
  functions in the \pkg{geoR} package.
}


\value{
  The result is always list. The components will vary according to the input
  options. The possible components are:
  
  \item{varcov}{the covariance matrix.  }
  \item{sqrt.varcov}{a square root of the covariance matrix.  }
  \item{lower.inverse}{the lower triangle of the inverse of covariance
    matrix.  }
  \item{diag.inverse}{the diagonal of the inverse of covariance matrix.
  }
  \item{inverse}{the inverse of covariance matrix.  }
  \item{sqrt.inverse}{a square root of the inverse of covariance
    matrix.  } 
  \item{log.det.to.half}{the logarithmic of the square root of the
    determinant of the covariance matrix.   }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{cov.spatial}} for more information on the
  correlation functions; \code{\link{chol}}, \code{\link{solve}},
  \code{\link{svd}} and \code{\link{eigen}} for matrix inversion and/or decomposition.
}

%\examples{
%}

\keyword{spatial}%-- one or more ...


\eof
\name{variofit}

\alias{variofit}
\alias{loss.vario}

\title{ Variogram Based Parameter Estimation }
\description{
  Estimate covariance parameters by fitting a parametric model to a
  empirical variogram. Variograms models can be fitted by using weighted or ordinary least squares.
}
\usage{
variofit(vario, ini.cov.pars, cov.model = "matern",
         fix.nugget = FALSE, nugget = 0,
         fix.kappa = TRUE, kappa = 0.5,
         simul.number = NULL, max.dist = vario$max.dist,
         weights = c("npairs", "equal", "cressie"),
         minimisation.function,
         limits = pars.limits(), messages, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{vario}{an object of the class \code{"variogram"}, typically an output of the function
    \code{\link{variog}}. The object is a list with information about the
    empirical variogram.  }
  \item{ini.cov.pars}{initial values for the covariance parameters:
    \eqn{\sigma^2}{sigma^2} (partial sill) and \eqn{\phi}{phi} (range
    parameter). See \code{DETAILS} below.  }
  \item{cov.model}{a string with the name of the correlation
    function. For further details see documentation for
    \code{\link{cov.spatial}}. For the linear model use \code{cov.model
      = "linear"}.
    Defaults are equivalent to the \emph{exponential} model.  }
  \item{fix.nugget}{logical, indicating whether the parameter
    \eqn{\tau^2}{tau^2} (nugget variance) should be regarded as fixed
    (\code{fix.nugget = TRUE}) or should
    be estimated (\code{fix.nugget = FALSE}). Defaults to
    \code{FALSE}.  }    
  \item{nugget}{value for the nugget parameter.  Regarded as a
    fixed values if \code{fix.nugget = TRUE} or as a initial value for the
    minimization algorithm if \code{fix.nugget = FALSE}. 
    Defaults to zero.  }
  \item{fix.kappa}{logical, indicating whether the parameter
    \eqn{\kappa}{kappa} should be regarded as fixed or 
    be estimated. Defaults to \code{TRUE}.  }    
  \item{kappa}{value of the smoothness parameter.  Regarded as a
    fixed values if \code{fix.kappa = TRUE} or as a initial value for the
    minimization algorithm if \code{fix.kappa = FALSE}. Only required if
    one of the following correlation functions is used: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"}
    and \code{"gneiting.matern"}. Defaults to \eqn{0.5}.  }
  \item{simul.number}{number of simulation. To be used when the object passed to the
    argument \code{vario} has empirical variograms for more than one
    data-set (or simulation). Indicates to which one the model will be
    fitted.  }
  \item{max.dist}{maximum distance considered when fitting the
    variogram. Defaults to \code{vario$max.dist}.  }
  \item{weights}{type weights used in the loss function. See
    \code{DETAILS} below.  } 
  \item{limits}{values defining lower and upper limits for the model
    parameters used in the numerical minimisation.
    Only valid if \code{minimisation.function = "optim"}.
    The auxiliary function \code{\link{pars.limits}} is called to set the limits.  }
  \item{minimisation.function}{minimization function used to estimate
    the parameters. Options are \code{"optim"}, \code{"nlm"}.
    If \code{weights = "equal"} the option 
    \code{"nls"} is also valid and det as default.
    Otherwise defaults to \code{"optim"}.  }
  \item{messages}{logical. Indicates
    whether or not status messages are printed on the screen (or other output device)
    while the function is running.  }
  \item{\dots}{further parameters to be passed to the minimization
    function. Typically arguments of the type \code{control()} which controls the
    behavior of the minimization algorithm. See documentation for the 
    selected minimization function for further details.  }
}

\details{
  \bold{Initial values}
  
  The algorithms for minimization functions require initial values of
  the parameters.
  
  A unique initial value is used if a vector is provided in the argument
  \code{ini.cov.pars}. The elements are initial values for
  \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}, respectively.
  This vector is concatenated with the value of the
  argument \code{nugget} if \code{fix.nugget = FALSE} and \code{kappa}
  if \code{fix.kappa = TRUE}.
  
  Specification of multiple initial values is also possible.
  If this is the case, the function
  searches for the one which minimizes the loss function and uses this as
  the initial value for the minimization algorithm.
  Multiple initial values are specified  by providing a matrix in the
  argument
  \code{ini.cov.pars} and/or, vectors in the arguments 
  \code{nugget} and \code{kappa} (if included in the estimation).
  If \code{ini.cov.pars} is a matrix, the first column has values of
  \eqn{\sigma^2}{sigma^2} and the second has values of  \eqn{\phi}{phi}. 
  
  If \code{minimisation.function = "nls"} only the values of
  \eqn{\phi}{phi} and \eqn{\kappa}{kappa} (if this is included in the
  estimation) are used. Values for the remaning are not need by the algorithm.
  
  If \code{cov.model = "linear"} only the value of
  \eqn{\sigma^2}{sigma^2} is used. Values for the
  remaning are not need by this algorithm.
  
  If \code{cov.model = "pure.nugget"} no initial values are needed since
  no minimisation function is used.

  
  \bold{Weights}
  
  The different options for the argument \code{weights}
  are used to define the loss function to be minimised.
  The available options are as follows.
  
  \describe{
    \item{\code{"npairs"}}{indicates that the weights are given by the
      number of pairs in each bin. The loss function is:
      \deqn{LOSS(\theta) = \sum_k n_k [(\hat{\gamma}_k) -
	\gamma_k(\theta)]^2}{LOSS(theta) = sum_k n_k (hat(gamma) -
	gamma(theta))^2}
    }
    
    \item{\code{"cressie"}}{weights as suggested by Cressie (1985).
      \deqn{LOSS(\theta) = \sum_k n_k [\frac{\hat{\gamma}_k -
	  \gamma_k(\theta)}{\gamma_k(\theta)}]^2}{LOSS(theta) = \sum_k n_k [(hat(gamma_k) -
	  gamma_k(theta))/{gamma_k(theta)}]^2}
    }
    
    \item{\code{"equal"}}{equal values for the weights. For this case
      the estimation corresponds to the ordinary least squares variogram
      fitting.
      \deqn{LOSS(\theta) = \sum_k [(\hat{\gamma_k}) -
	\gamma_k(\theta)]^2}{LOSS(\theta) = \sum_k (\hat(\gamma) -
	\gamma(\theta))^2}
    }
    

    Where \eqn{\theta}{theta} is the vector with the variogram parameters
    and
    for each \eqn{k^{th}}{kth}-bin
    \eqn{n_k}{n_k} is the number of
    pairs, \eqn{(\hat{\gamma}_k)}{hat(gamma_k)} is the
    value of the empirical variogram and
    \eqn{\gamma_k(\theta)}{gamma_k(theta)}
    is the value of the theoretical variogram.
    
    See also Cressie (1993) and Barry, Crowder and Diggle (1997) for further discussions on 
    methods to estimate the variogram parameters.
  }
}

\value{
  An object of the \code{\link{class}} \code{"variomodel"} and \code{"variofit"} which is list with the following components:
  \item{nugget}{value of the nugget parameter. An estimated value if
    \code{fix.nugget = FALSE} or a fixed value if \code{fix.nugget = TRUE}.  }
  \item{cov.pars}{a two elements vector with estimated values of the covariance
    parameters \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi},  respectively.  }
  \item{cov.model}{a string with the name of the correlation function.  }
  \item{kappa}{fixed value of the smoothness parameter.  }
  \item{value}{minimized value of the loss function.  }
  \item{max.dist}{maximum distance considered in the variogram fitting.
  }
  \item{minimisation.function}{minimization function used.  }
  \item{weights}{a string indicating the
    type of weights used for the variogram fitting.  }
  \item{method}{a string indicating the
    type of variogram fitting method (OLS or WLS).  }
  \item{fix.kappa}{logical indicating whether the parameter \eqn{\kappa}{kappa} was
    fixed.  }
  \item{fix.nugget}{logical indicating whether the nugget parameter was
    fixed.  }
  \item{lambda}{transformation parameters inherith from the object
    provided in the argument \code{vario}.  }
  \item{message}{status messages returned by the function.  }  
  \item{call}{the function call.  }
}

\references{
  Barry, J.T., Crowder, M.J. and Diggle, P.J. (1997) Parametric
  estimation of the variogram. \emph{Tech. Report, Dept Maths & Stats,
    Lancaster University}.
  
  Cressie, N.A.C (1985) \emph{Mathematical Geology}. \bold{17}, 563-586. 

  Cressie, N.A.C (1993) \emph{Statistics for Spatial Data}. New York: Wiley.
  
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{
  Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{cov.spatial}} for a detailed description of the
  available correlation (variogram) functions,
  \code{\link{likfit}} for maximum
  and restricted maximum likelihood estimation,
  \code{\link{lines.variomodel}} for graphical output of the fitted
  model. For details on the minimization functions see \code{\link{optim}},
  \code{\link{nlm}} and \code{\link{nls}}.
}

\examples{
data(s100)
vario100 <- variog(s100, max.dist=1)
ini.vals <- expand.grid(seq(0,1,l=5), seq(0,1,l=5))
ols <- variofit(vario100, ini=ini.vals, fix.nug=TRUE, wei="equal")
summary(ols)
wls <- variofit(vario100, ini=ini.vals, fix.nug=TRUE)
summary(wls)
plot(vario100)
lines(wls)
lines(ols, lty=2)

\testonly{
vr <- variog(s100, max.dist=1)
## OLS#
o1 <- variofit(vr, ini = c(.5, .5), fix.nug=TRUE, wei = "equal")
o2 <- variofit(vr, ini = c(.5, .5), wei = "equal")
o3 <- variofit(vr, ini = c(.5, .5), fix.nug=TRUE,
      fix.kappa = FALSE, wei = "equal")
#o4 <- variofit(vr, ini = c(.5, .5), fix.kappa = FALSE, wei = "equal")
## WLS
w1 <- variofit(vr, ini = c(.5, .5), fix.nug=TRUE)
w2 <- variofit(vr, ini = c(.5, .5))
w3 <- variofit(vr, ini = c(.5, .5), fix.nug=TRUE, fix.kappa = FALSE)
w4 <- variofit(vr, ini = c(.5, .5), fix.kappa = FALSE)
}

}

\keyword{spatial}


\eof
\name{variog}

\alias{variog}
\alias{rfm.bin}
\alias{define.bins}

\concepts{variogram}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Compute Empirical Variograms}

\description{
  Computes sample (empirical) variograms with options for  the \emph{classical} or \emph{robust}
  estimators. Output can be returned as a \code{binned variogram}, a \code{
    variogram cloud} or a \code{smoothed variogram}. Data
  transformation (Box-Cox) is allowed.
  ``Trends'' can be specified and are fitted by ordinary least
  squares in which case the variograms are computed using the
  residuals.
}

\usage{
variog(geodata, coords = geodata$coords, data = geodata$data, 
       uvec = "default", breaks = "default",
       trend = "cte", lambda = 1,
       option = c("bin", "cloud", "smooth"),
       estimator.type = c("classical", "modulus"), 
       nugget.tolerance, max.dist, pairs.min = 2,
       bin.cloud = FALSE, direction = "omnidirectional", tolerance = pi/8,
       unit.angle = c("radians","degrees"), messages, \dots) 
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing element \code{coords}
    as described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set.
    If not provided the arguments
    \code{coords} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix containing
    coordinates of the \eqn{n} data locations in each row.
    Defaults to \code{geodata$coords}, if provided.}
  \item{data}{a vector or matrix with data values.
    If a matrix is provided, each column is regarded as one variable or realization.
    Defaults to \code{geodata$data}, if provided.}
  \item{uvec}{a vector with values used to define the variogram binning. Only
    used when \code{option = "bin"}.
    See \code{DETAILS} below for more details on how to specify the bins.
  }
  \item{breaks}{a vector with values to define the variogram binning. Only
    used when \code{option = "bin"}.
    See \code{DETAILS} below for more details on how to specify the bins.
  }
  \item{trend}{specifies the mean part of the model.
    See documentation of \code{\link{trend.spatial}} for further details.
    Defaults to \code{"cte"}.  }
  \item{lambda}{values of the Box-Cox transformation parameter.
    Defaults to \eqn{1} (no
    transformation). If another value is provided the variogram is
    computed after transforming the 
    data. A case of particular interest is \eqn{\lambda = 0}{lambda = 0}
    which corresponds to log-transformation.  }
  \item{option}{defines the output type: the options \code{"bin"} returns values of
    binned variogram, \code{"cloud"} returns the variogram cloud and
    \code{"smooth"} returns the kernel smoothed variogram.
    Defaults to \code{"bin"}.}
  \item{estimator.type}{\code{"classical"} computes the classical method of
    moments estimator.  \code{"modulus"} returns the variogram
    estimator suggested by Hawkins and Cressie (see Cressie, 1993, pg 75).
    Defaults to \code{"classical"}.  }
  \item{nugget.tolerance}{a numeric value. Points which are separated
    by a distance less than this value
    are considered co-located. Defaults to zero.  }
  \item{max.dist}{a numerical value defining the maximum distance for
    the variogram. Pairs of locations
    separated for distance larger than this value are ignored for the
    variogram calculation. If not provided defaults takes the maximum
    distance among all pairs of data locations.  }
  \item{pairs.min}{a integer number defining the minimum numbers of
    pairs for the bins.
    For \code{option = "bin"},
    bins with number of pairs smaller than this
    value are ignored. Defaults to \code{NULL}.  }
  \item{bin.cloud}{logical. If \code{TRUE} and
    \code{option = "bin"} the cloud values for each class are
    included in the output. Defaults to \code{FALSE}.  }
  \item{direction}{a numerical value for the directional (azimuth) angle. This
    used to specify directional variograms. Default defines the
    omnidirectional variogram. The value must be in the interval
    \eqn{[0, \pi]}{[0, pi]} radians (\eqn{[0, 180]} degrees).  }
  \item{tolerance}{numerical value for the tolerance angle, when
    computing directional variograms. The value must be in the interval
    \eqn{[0, \pi/2]}{[0, pi/2]} radians (\eqn{[0, 90]} degrees).  Defaults to \eqn{\pi/8}{pi/8}.  }
  \item{unit.angle}{defines the unit for the specification of angles in
    the two previous arguments. Options are \code{"radians"} and
    \code{"degrees"}, with default to \code{"radians"}. }
  \item{messages}{logical. Indicates whether
    status messages should be printed on the screen (or output device)
    while the function is running.  }
  \item{\dots}{arguments to be passed to the function \code{\link{ksmooth}}, if
    \code{option = "smooth"}.  }
}

\details{
  Variograms are widely used in geostatistical analysis for exploratory
  purposes, to estimate covariance parameters and/or to compare theoretical
  and fitted models against sample variograms.

  \bold{Estimators}\cr
  The two estimators currently implemented are:
  \itemize{
    \item \emph{classical} (method of moments) estimator:
    \deqn{\gamma(h) = \frac{1}{2N_h} \sum_{i=1}^{N_h}[Y(x_{(i+h)}) -
      Y(x_i)]^2}{gamma(h) = (1/2N_h) sum (Y(x_i+h) - Y(x_i))^2}

    \item Hawkins and Cressie's \emph{modulus} estimator
    \deqn{\gamma(h) = \frac{[\frac{1}{N_h}\sum_{i=1}^{N_h}|Y(x_{i+h}) -
	Y(x_i)|^{\frac{1}{2}}]^4}{0.914 + \frac{0.988}{N_h}}}{\gamma(h)
      = ([(1/N_h)\sum_{i=1}^{N_h} |Y(x_{i+h}) -
	Y(x_i)|^(1/2)]^4)/(0.914 + (0.988/N_h))}
      }

  \bold{Defining the bins}

  \emph{The defaults}\cr
  If arguments \code{breaks} and \code{uvec} are not provided, the
  binning is defined as follows:\cr
  \enumerate{
    \item read the argument \code{max.dist}. If not provided it is set
    to the maximum distance between the pairs of points.
    \item the center of the bins are initially defined by the sequence \code{u = seq(0,
      max.dist, l = 13)}.
    \item the interval spanned by each bin is given by the mid-points
    between the centers of the bins.
%    \item The center of the first bin is set to \code{u[1] = u[2]/4}.
  }

  If an vector is passed to the argument \code{breaks} its elements are
  taken as the limits of the bins (classes of distance) and the argument \code{uvec} is ignored.

  \emph{Variations on the default}\cr
  The default definition of the bins is different for some particular
  cases.
  \enumerate{
    \item if there are  coincident data locations
    the bins follows the default above but
    one more bin is added at the origin (distance zero) for
    the collocated points.
    
    \item if the argument \code{nugget.tolerance} is provided the
    separation distance between all pairs
    in the interval \eqn{[0, nugget.tolerance]} are set to zero.
    The first bin distance is set to zero (\code{u[1] = 0}).
    The remaining bins follows the default.

    \item if a scalar is provided to the argument \code{uvec} the
    default number of bins is defined by this number.
%    \code{u = seq(0, max.dist, l = uvec)}

    \item if a vector is provided to the argument \code{uvec},
    its elements are taken as central points of the bins.
  }
  
  %    The values of \code{uvec} defines the mid-points of the bins.\cr
  %    If \eqn{uvec[1] > 0} the first bin is: \eqn{0 < u <= uvec[2] - 0.5*(uvec[2] - uvec[1])}.\cr
  %    If \eqn{uvec[1] = 0} first bin is: \eqn{0 < u <= 0.5*uvec[1]} and
  %    \eqn{uvec[1]} is replaced by the midpoint of this interval.
  
}

\value{
  An object of the \code{\link{class}} \code{variogram} which is a
  list with the following components:
  \item{u }{a vector with distances.  }
  \item{v }{a vector with estimated variogram values at distances given
    in \code{u}.  }
  \item{n }{number of pairs in each bin, if
    \code{option = "bin"}.  }
  \item{sd }{standard deviation of the values in each bin.  }
  \item{bins.lim }{limits defining the interval spanned by each bin.  }
  \item{ind.bin }{a logical vector indicating whether the number of
    pairs in each bin is greater or equal to the value in the argument
    \code{pairs.min}.  }
  \item{var.mark }{variance of the data.  }
  \item{beta.ols }{parameters of the mean part of the model fitted by
    ordinary least squares.  }
  \item{output.type }{echoes the \code{option} argument.  }
  \item{max.dist}{maximum distance between pairs allowed in the
    variogram calculations.}
  \item{estimator.type }{echoes the type of estimator used.  }
  \item{n.data }{number of data.  }
  \item{lambda}{value of the transformation parameter.  }
  \item{trend}{trend specification.  }
  \item{nugget.tolerance}{value of the nugget tolerance argument.  }
  \item{direction }{direction for which the variogram was computed.  }
  \item{tolerance }{tolerance angle for directional variogram.  }
  \item{uvec }{lags provided in the function call.  }
  \item{call }{the function call.  }
}

\references{
  Cressie, N.A.C (1993) \emph{Statistics for Spatial Data}. New York: Wiley.
  
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{variog4}} for more on computation of
  directional variograms, 
  \code{\link{variog.model.env}} and \code{\link{variog.mc.env}} for
  variogram envelopes,
  \code{\link{variofit}}  for variogram based
  parameter estimation and
  \code{\link{plot.variogram}} for graphical output.
}

\examples{
# Loading data:
data(s100)
#
# computing variograms:
#
# binned variogram
vario.b <- variog(s100, max.dist=1)
# variogram cloud
vario.c <- variog(s100, max.dist=1, op="cloud")
#binned variogram and stores the cloud
vario.bc <- variog(s100, max.dist=1, bin.cloud=TRUE)
# smoothed variogram
vario.s <- variog(s100, max.dist=1, op="sm", band=0.2)
#
#
# plotting the variograms:
par(mfrow=c(2,2))
plot(vario.b, main="binned variogram") 
plot(vario.c, main="variogram cloud")
plot(vario.bc, bin.cloud=TRUE, main="clouds for binned variogram")  
plot(vario.s, main="smoothed variogram") 

# computing a directional variogram
vario.0 <- variog(s100, max.dist=1, dir=0, tol=pi/8)
plot(vario.b, type="l", lty=2)
lines(vario.0)
legend(0, 1.2, legend=c("omnidirectional", expression(0 * degree)), lty=c(2,1))
}

\keyword{spatial}
\keyword{smooth}
\keyword{robust}






\eof
\name{variog.mc.env}

\alias{variog.mc.env}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Envelops for Empirical Variograms Based on Permutation}

\description{
  Computes envelops for empirical variograms by
  permutation of the data values on the spatial locations.
}

\usage{
variog.mc.env(geodata, coords = geodata$coords, data = geodata$data,
              obj.variog, nsim = 99, save.sim = FALSE, messages) 
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. 
    Typically an object of the class \code{"geodata"} - a \pkg{geoR}
    data-set.
    If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix, each row containing Euclidean
    coordinates of the \emph{n} data locations. By default it takes the
    element \code{coords} of the argument \code{geodata}.  }
  \item{data}{a vector with the data values. By default it takes the
    element \code{data} of the argument \code{geodata}.  }
  \item{obj.variog}{an object of the class \code{"variogram"}, typically
    an output of the function \code{\link{variog}}.  }
  \item{nsim}{number of simulations used to compute the envelope. Defaults to
    99.  }
  \item{save.sim}{logical. Indicates whether or not the simulated data
    are included in the output. Defaults to \code{FALSE}.  }
  \item{messages}{logical. If \code{TRUE}, the default, status messages
    are printed while the function is running.  }
}

\details{
  The envelops are obtained by permutation.
  For each simulations data values are randomly allocated to the spatial locations.
  The empirical variogram is computed for each simulation
  using the same lags as for the variogram originally computed for the data.
  The envelops are computed by taking, at each lag, the maximum and minimum
  values of the variograms for the simulated data.
}

\value{
  An object of the \code{\link{class}} \code{"variogram.envelope"} which is a
  list with the following components:
  \item{u}{a vector with distances. }
  \item{v.lower}{a vector with the minimum variogram values at each
    distance in \code{u}.  }
  \item{v.upper}{a vector with the maximum variogram values at each
    distance in \code{u}.  }
  \item{simulations}{a matrix with simulated data.
    Only returned if \code{save.sim = TRUE}.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}


\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{variog.model.env}} for envelops computed by
  from a model specification, 
   \code{\link{variog}} for variogram calculations,
  \code{\link{plot.variogram}} and \code{\link{variog.mc.env}} for
  graphical output.
}

\examples{
if(is.R()) data(s100)
s100.vario <- variog(s100, max.dist=1)
s100.env <- variog.mc.env(s100, obj.var = s100.vario)
plot(s100.vario, envelope = s100.env)

}

\keyword{spatial}
\keyword{nonparametric}



\eof
\name{variog.model.env}

\alias{variog.model.env}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Envelops for Empirical Variograms Based on Model Parameters}

\description{
  Computes envelopes for a empirical variogram by simulating
  data for given model parameters.
}

\usage{
variog.model.env(geodata, coords = geodata$coords, obj.variog,
                 model.pars, nsim = 99, save.sim = FALSE, messages)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing element \code{coords}
    as described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set.
    If not provided the argument
    \code{coords} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix, each row containing
    Euclidean 
    coordinates of the \emph{n} data locations. By default it takes the
    element \code{coords} of the argument \code{geodata}.  }
  \item{obj.variog}{an object of the class \code{"variogram"}, typically
    an output of the function \code{\link{variog}}.  }
  \item{model.pars}{a list with model specification and parameter
    values. The input is typically an object of the class
    \code{variomodel} which is an output of \code{\link{likfit}},
    \code{\link{variofit}}.
    The required components of the list are:
    \itemize{
      \item \code{beta}, the mean parameter. Defaults to zero.
      \item \code{cov.model}, the covariance model. Defaults to
      "exponential".
      \item \code{cov.pars}, the covariance parameters
      \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}.
      \item \code{kappa}, the extra covariance parameters for some of the
      covariance models. Defaults to \eqn{0.5}.
      \item \code{nugget}, the error component variance. Defaults to zero.
      \item \code{estimator.type}, the type of variogram
      estimator. Options for \code{"classical"} and
      \code{"robust"}. Defaults to \code{obj.variog$estimator}.  }
  } 
  \item{nsim}{number of simulations used to compute the envelopes.
    Defaults to 99.  }
  \item{save.sim}{logical. Indicates whether or not the simulated data
    are included in the output. Defaults to \code{FALSE}.  }
  \item{messages}{logical. If \code{TRUE}, the default, status messages
    are printed while the function is running.  }
}

\details{
  The envelopes are computed assuming a (transformed) Gaussian random
  field model.
  Simulated values are generated at the data locations, given the model parameters.
  The empirical variogram is computed for each simulation
  using the same lags as for the original variogram of the data.
  The envelopes are computed by taking, at each lag, the maximum and minimum
  values of the variograms for the simulated data.
}

\value{
  An object of the \code{\link{class}} \code{"variogram.envelope"} which is a
  list with the components:
  \item{u}{a vector with distances. }
  \item{v.lower}{a vector with the minimum variogram values at each
    distance in \code{u}.  }
  \item{v.upper}{a vector with the maximum variogram values at each
    distance in \code{u}.  }
  \item{simulations}{a matrix with the simulated data.
    Only returned if \code{save.sim = TRUE}.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br},  \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

\seealso{\code{\link{variog.mc.env}} for envelops computed by
  using data permutation, 
  \code{\link{variog}} for variogram calculations,
  \code{\link{plot.variogram}} and \code{\link{variog.mc.env}} for
  graphical output. The functions
  \code{\link{likfit}}, \code{\link{variofit}}
  are used to estimate the model parameters.
}

\examples{
if(is.R()) data(s100)
s100.ml <- likfit(s100, ini = c(0.5, 0.5), fix.nugget = TRUE)
s100.vario <- variog(s100, max.dist = 1)
s100.env <- variog.model.env(s100, obj.v = s100.vario,
                             model.pars = s100.ml)
plot(s100.vario, env = s100.env)

}

\keyword{spatial}%-- one or more ...


\eof
\name{variog4}
\alias{variog4}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Computes Directional Variograms}
\description{
  Computes directional variograms for 4 directions provided by the user.
}
\usage{
variog4(geodata, coords = geodata$coords, data = geodata$data, 
        uvec = "default", breaks = "default", trend = "cte", lambda = 1,
        option = c("bin", "cloud", "smooth"),
        estimator.type = c("classical", "modulus"), 
        nugget.tolerance, max.dist, pairs.min = 2,
        bin.cloud = FALSE, direction = c(0, pi/4, pi/2, 3*pi/4), tolerance = pi/8,
        unit.angle = c("radians", "degrees"), messages, \dots)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{geodata}{a list containing element \code{coords}
    as described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set.
    If not provided the arguments
    \code{coords} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix containing
    coordinates of the \eqn{n} data locations in each row.
    Defaults to \code{geodata$coords}, if provided.}
  \item{data}{a vector or matrix with data values.
    If a matrix is provided, each column is regarded as one variable or realization.
    Defaults to \code{geodata$data}, if provided.}
  \item{uvec}{a vector with values to define the variogram binning. For
    further details see documentation for \code{\link{variog}}.  } 
  \item{breaks}{a vector with values to define the variogram binning. For
    further details see documentation for \code{\link{variog}}.  } 
  \item{trend}{specifies the mean part of the model.
    The options are:
    \code{"cte"} (constant mean),
    \code{"1st"} (a first order polynomial
    on the coordinates), \code{"2nd"} (a second order polynomial
    on the coordinates), or a formula of the type \code{~X} where \code{X}
    is a matrix with the covariates (external trend). Defaults to \code{"cte"}.  }
  \item{lambda}{values of the Box-Cox transformation parameter.
    Defaults to \eqn{1} (no
    transformation). If another value is provided the variogram is
    computed after transforming the 
    data. A case of particular interest is \eqn{\lambda = 0}{lambda = 0}
    which corresponds to log-transformation.  }
  \item{option}{defines the output type: the options \code{"bin"} returns values of
    binned variogram, \code{"cloud"} returns the variogram cloud and
    \code{"smooth"} returns the kernel smoothed variogram.
    Defaults to \code{"bin"}.}
  \item{estimator.type}{\code{"classical"} computes the classical method of
    moments estimator.  \code{"modulus"} returns the variogram
    estimator suggested by Hawkins and Cressie (see Cressie, 1993, pg 75).
    Defaults to \code{"classical"}.  }
  \item{nugget.tolerance}{a numeric value. Points which are separated
    by a distance less than this value
    are considered co-located. Defaults to zero.  }
  \item{max.dist}{a numerical value defining the maximum distance for
    the variogram. Pairs of locations
    separated for distance larger than this value are ignored for the
    variogram calculation. Defaults to the maximum distance among the pairs
    of data locations.  }
  \item{pairs.min}{a integer number defining the minimum numbers of
    pairs for the bins.
    For \code{option = "bin"},
    bins with number of pairs smaller than this
    value are ignored. Defaults to \code{NULL}.  }
  \item{bin.cloud}{logical. If \code{TRUE} and
    \code{option = "bin"} the cloud values for each class are
    included in the output. Defaults to \code{FALSE}.  }
  \item{direction}{a vector with values of 4 angles, indicating the
    directions for which the variograms will be computed. Default
    corresponds to \code{c(0, 45, 90, 135)} (degrees). }
  \item{tolerance}{numerical value for the tolerance angle, when
    computing directional variograms. The value must be in the interval
    \eqn{[0, 90]} degrees.  Defaults to \eqn{\pi/8}{pi/8}.  }
  \item{unit.angle}{defines the unit for the specification of angles in
    the two previous arguments. Options are \code{"degrees"} and \code{"radians"}. }
  \item{messages}{logical. Indicates whether
    status messages should be printed on the screen (or output device)
    while the function is running.  }
  \item{\dots}{arguments to be passed to the function \code{\link{ksmooth}}, if
    \code{option = "smooth"}.  }
}

%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}

\value{
  The output is an object of the class \code{variog4},
  a list with five components.
  The first four elements are estimated variograms for the directions
  provided and the last is the omnidirectional variogram. 
  Each individual component is an object of the class \code{variogram},
  an output of the function \code{\link{variog}}.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }


\seealso{\code{\link{variog}} for variogram calculations and
  \code{\link{plot.variog4}} for plotting results}

\examples{
data(s100)
var4 <- variog4(s100, max.dist=1)
plot(var4)
}
\keyword{spatial}%-- one or more ...

\eof
\name{wolfcamp}
\docType{data}

\alias{wolfcamp}
\alias{wolf}

\title{Wolfcamp Aquifer Data}
\usage{data(wolfcamp)}

\description{
  Piezometric head measurements taken at the Wolfcamp Aquifer, Texas,
  USA.
  See Cressie (1993, p.212--214) for description of the scientific
  problem and the data.
  Original data were converted to SI units: coordinates are given in
  kilometers and pressure heads to meters.
}

\format{
  An object of the \code{\link{class}}  \code{"geodata"}, which is list with two components:
  
  \describe{
    \item{\code{coords}}{the coordinates of the data locations. The
      distance are given in kilometers.  }
    \item{\code{data}}{values of the piezometric head. The unit is heads
      to meters.  }
  }
}

\source{
  Harper, W.V and Furr, J.M. (1986) Geostatistical analysis of
  potentiometric data in the Wolfcamp Aquifer of the Palo Duro Basin,
  Texas. \emph{Technical Report BMI/ONWI-587, Bettelle Memorial Institute,
    Columbus, OH}.   
}

\references{
  Cressie, N.A.C (1993) \emph{Statistics for Spatial Data}. New York: Wiley.
  
  Papritz, A. and Moyeed, R. (2001) Parameter uncertainty in spatial
  prediction: checking its importance by cross-validating the Wolfcamp
  and Rongelap data sets. \emph{GeoENV 2000: Geostatistical for
    Environmental Applications. Ed. P. Monestiez, D. Allard,
    R. Froidevaux}. Kluwer. 
}

\examples{
data(wolfcamp)
summary(wolfcamp)
plot(wolfcamp)
}

\keyword{datasets}



\eof
\name{wrappers}

\alias{wrappers}
%\alias{distdiag}
\alias{diffpairs}
\alias{loccoords}
\alias{diagquadraticformXAX}
\alias{bilinearformXAY}
\alias{corr.diaglowertri}

%- Also NEED an `\alias' for EACH other topic documented here.

\title{Wrappers for the C functions used in geoR}
\description{
  These functions are \emph{wrappers} for some (but not all)
  the C functions
  included in the \pkg{geoR} package.\cr
  Typically the C code is directly  called from the \pkg{geoR}
  functions but these functions allows independent calls.
}

\usage{
%distdiag(coords)
diffpairs(coords, data)
loccoords(coords, locations)
diagquadraticformXAX(X, lowerA, diagA) 
bilinearformXAY(X, lowerA, diagA, Y) 
corr.diaglowertri(coords, cov.model, phi, kappa)
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix with the data
    coordinates.  }
  \item{data}{an vector with the data values.  }
  \item{locations}{an \eqn{N \times 2}{N x 2} matrix with the
    coordinates of the prediction locations. }
  \item{lowerA}{a vector with the diagonal terms of the symmetric matrix 
    A. }
  \item{diagA}{a vector with the diagonal terms of the symmetric matrix A. }
  \item{X}{a matrix with conforming dimensions. }
  \item{Y}{a matrix with conforming dimensions. }
  \item{cov.model}{covariance model, see \code{\link{cov.spatial}} for options and
    more details.  }
  \item{phi}{numerical value of the correlation function parameter phi.}
  \item{kappa}{numerical value of the correlation function parameter kappa.}
}
%\details{
%}
\value{
  The outputs for the different functions are: 
  
    \item{loccoords}{returns a \eqn{n \times N}{n x N}  matrix with distances
      between data points and prediction locations.  }
%    \item{distdiag}{returns a vector with distances between data
%      locations, including the diagonal zero values.  }
    \item{diagquadraticformXAX}{returns a vector with the diagonal term of the
      quadratic form \eqn{X' A X}{t(X) \%*\% A \%*\% X}.  }
    \item{bilinearformXAY}{returns a vector or a matrix with the terms of the
      quadratic form \eqn{X' A Y}{t(X) \%*\% A \%*\% Y}.  }
    \item{corr.diaglowertri}{returns the lower triangle of the
      correlation matrix, including the diagonal.  }
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo Justiniano Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}


%\note{ ~~further notes~~ }


%\seealso{ ~~objects to SEE ALSO as \code{\link{~~fun~~}}, ~~~ }

%\examples{
%}

\keyword{spatial} 
\keyword{programming}
\keyword{interface}

\eof
\name{xvalid}

\alias{xvalid}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Cross-validation using kriging}

\description{
  This is a function to perform model validation.
  Options include \emph{leaving-one-out} cross-validation where.
  each data location is removed from the data set and the variable at
  this location is predicted using the remaining locations, for as given
  model. This can be
  done for all or some of the locations.
  Alternativelly, other validation locations which are not the same as the
  original data locations can be used.   
}

\usage{
xvalid(geodata, coords = geodata$coords, data = geodata$data,
       model, reestimate = FALSE, variog.obj = NULL,
       output.reestimate = FALSE, locations.xvalid = "all",
       data.xvalid = NULL, messages, \dots)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing element \code{coords}
    as described next. Typically an object of the class
    \code{"geodata"} - a \pkg{geoR} data-set.
    If not provided the arguments
    \code{coords} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} matrix containing
    coordinates of the \eqn{n} data locations in each row.
    Defaults to \code{geodata$coords}, if provided.}
  \item{data}{a vector or matrix with data values.
    If a matrix is provided, each column is regarded as one variable or realization.
    Defaults to \code{geodata$data}, if provided.}
  \item{model}{an object containing information on a fitted
    model. Typically an output of \code{\link{likfit}},
    \code{\link{variofit}}.  }
  \item{reestimate}{logical. Indicates whether or not the model
    parameters should be re-estimated for each point removed from the data-set. }
  \item{variog.obj}{on object with the empirical variogram, typically an
    output  of the function \code{\link{variog}}. Only used if
    \code{reestimate = TRUE} and the object passed to the argument
    \code{model} is the result of a variogram based estimation, i.e. if
    the model was fitted by \code{\link{variofit}}.  }
  \item{output.reestimate}{logical. Only valid if \code{reestimate =
      TRUE}. Specifies whether the re-estimated parameters are returned.  }
  \item{locations.xvalid}{there are three possible specifications for
    this argument: \code{"all"} indicates the \emph{leaving-on-out}
    method is used. The second possibility is to use only a sub-set of
    the data for cross-validation. For this case  a vector should be
    provided with numbers indicating at which locations the
    cross-validation should be performed. The third option is to perform
    validation on a different set of data. For this a matrix with the
    coordinates of the validation points should be provided and the the
    argument \code{locations.data} should also be provided.  }
  \item{data.xvalid}{data values at the validation locations.
    Only used if the validation locations are not the same or a
    subset of the original data coordinates.  }
  \item{messages}{logical. Indicates whether
    status messages should be printed on the screen (or output device)
    while the function is running.  }
  \item{\dots}{further arguments to the minimization functions used by
    \code{\link{likfit}}, \code{\link{variofit}}.  }
}

\details{
  The cross-validation uses the function \code{krige.conv} to predict
  at each location.
  
  For models fitted by \code{\link{variofit}} the
  parameters \eqn{\kappa}{kappa}, \eqn{\psi_A}{psiA}, \eqn{\psi_R}{psiR}
  and \eqn{\lambda}{lambda} are always regarded as fixed.
  
  See documentation of the function \code{\link{likfit}} for more
  details on the model and its parameters.
}

\value{
  An object of the \code{\link{class}} \code{"xvalid"} which is a list with the following components:
  \item{data}{the original data.  }
  \item{predicted}{the values predicted by cross-validation.  }
  \item{krige.var}{the cross-validation prediction variance.  }
  \item{error}{difference \code{data - predicted}.   }
  \item{std.error}{the errors divided by the square root of the
    prediction variances.  }
  \item{prob}{the cumulative probability at original value under a
    normal distribution with parameters given by the cross-validation results.  }
  
  If \code{reestimate = TRUE} and \code{output = TRUE} additional
  columns are added to the data-frame. Each column will contain the
  values of the re-estimated parameters.
}

\references{
  Further information on the package \pkg{geoR} can be found at:\cr
  \url{http://www.est.ufpr.br/geoR}.
}

\author{Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{plot.xvalid}} for plotting of the results, \code{\link{likfit}},
  \code{\link{variofit}} for parameter estimation and
  \code{\link{krige.conv}} for the kriging method used for predictions.
}


\examples{
data(s100)
#
# Maximum likelihood estimation
#
s100.ml <- likfit(s100, ini = c(.5, .5), fix.nug = TRUE)
#
# Weighted least squares estimation
#
s100.var <- variog(s100, max.dist = 1)
s100.wls <- variofit(s100.var, ini = c(.5, .5), fix.nug = TRUE)
#
# Now, performing cross-validation without reestimating the model
#
s100.xv.ml <- xvalid(s100, model = s100.ml)
s100.xv.wls <- xvalid(s100, model = s100.wls)
##
## Plotting results
##
par.ori <- par(no.readonly = TRUE)
##
par(mfcol=c(5,2), mar=c(2.3,2.3,.5,.5), mgp=c(1.3, .6, 0))
plot(s100.xv.ml)
par(mfcol=c(5,2))
plot(s100.xv.wls)
##
par(par.ori)
#
\testonly{
data(s100)
  vr <- variog(s100, max.dist=1)
  ## OLS#
  o1 <- variofit(vr, ini=c(.5, .5), fix.nug=TRUE, wei="equal")
  xvo1 <- xvalid(s100, model=o1, variog.obj=vr, loc=sample(1:100,5))
  o2 <- variofit(vr, ini=c(.5, .5), wei = "equal")
  xvo2 <- xvalid(s100, model=o2, variog.obj=vr, loc=sample(1:100,5))
  o3 <- variofit(vr, ini=c(.5, .5), fix.nug=TRUE, fix.kappa = FALSE, wei = "equal")
  xvo3 <- xvalid(s100, model=o3, variog.obj=vr, loc=sample(1:100,5))
  #o4 <- variofit(vr, ini=c(.5, .5), fix.kappa = FALSE, wei = "equal")
  #xvo4 <- xvalid(s100, model=o4, variog.obj=vr, loc=sample(1:100,5))
  ## WLS
  w1 <- variofit(vr, ini=c(.5, .5), fix.nug=TRUE)
  xvw1 <- xvalid(s100, model=w1, variog.obj=vr, loc=sample(1:100,5))
  w2 <- variofit(vr, ini=c(.5, .5))
  xvw2 <- xvalid(s100, model=w2, variog.obj=vr, loc=sample(1:100,5))
  w3 <- variofit(vr, ini=c(.5, .5), fix.nug=TRUE, fix.kappa = FALSE)
  xvw3 <- xvalid(s100, model=w3, variog.obj=vr, loc=sample(1:100,5))
  w4 <- variofit(vr, ini=c(.5, .5), fix.kappa = FALSE)
  xvw4 <- xvalid(s100, model=w4, variog.obj=vr, loc=sample(1:100,5))
  # ML
  m1 <- likfit(s100, ini=c(.5,.5), fix.nug=FALSE, cov.model="mat", kappa=1)
  xvm1 <- xvalid(s100, model=m1, loc=sample(1:100,5))
  ap <- grf(10, cov.pars=c(1, .25))
  xvm2 <- xvalid(s100, model=m1, locations.xvalid=ap$coords, data.xvalid=ap$data)
  bor <- s100$coords[chull(s100$coords),]
  par(mfcol=c(5,2),mar=c(3,3,1,0),mgp=c(2,.5,0))  
  plot(xvm2, borders=bor)
  # RML
  rm1 <- likfit(s100, ini=c(.5,.5), fix.nug=FALSE, met = "REML", cov.model="mat", kappa=1)
  xvrm1 <- xvalid(s100, model=m1, loc=sample(1:100,5))
}
}

\keyword{spatial}%-- one or more ...

\eof
