\name{DAG}
\alias{DAG}
\title{Directed acyclic graphs (DAGs)}
\description{
  A simple way to define a DAG by means of regression model
  formulae.
  }
\usage{
DAG(..., order = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{\dots}{a sequence of model formulae}
  \item{order}{logical, defaulting to \code{FALSE}. If \code{TRUE} the nodes of the
    DAG are permuted according to the topological order. If
    \code{FALSE} the nodes are in the order they first appear in the model
    formulae (from left to right).}
}
\details{
  The DAG is defined by a sequence of recursive regression models.
  Each regression is defined by a model formula.
  For each formula the response defines a node of the graph and
  the explanatory variables the parents of that node. If the
  regressions are not recursive the function returns an error message.

  Some authors prefer the terminology acyclic directed graphs (ADG).
}
\value{the adjacency matrix of the DAG, i.e. 
  a square Boolean matrix of order equal to the number of nodes of the
  graph and a one in position \eqn{(i,j)} if there is an arrow from
  \eqn{i} to \eqn{j} and zero otherwise.  The rownames of the adjacency
  matrix are the nodes of the DAG. 
  
  If \code{order = TRUE} the
  adjacency matrix is permuted to have  parents before  children.
  This can always be done (in more than one way) for DAGs. The resulting
  adjacency matrix is upper triangular. 
}
\references{
  Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
  Clarendon Press.
}
\author{G. M. Marchetti}
\note{The model formulae may contain interactions, but they are
  ignored in the graph.} 
\seealso{\code{\link{UG}}, \code{\link{topSort}}, \code{\link{edgeMatrix}}, \code{\link{fitDag}}}
\examples{
## A Markov chain
DAG(y ~ x, x ~ z, z ~ u)

## Another DAG
DAG(y ~ x + z + u, x ~ u, z ~ u)

## A DAG with an isolated node
DAG(v ~ v, y ~ x + z, z ~ w + u)

## There can be repetitions
DAG(y ~ x + u + v, y ~ z, u ~ v + z)

## Interactions are ignored
DAG(y ~ x*z + z*v, x ~ z)

## A cyclic graph returns an error!
\dontrun{DAG(y ~ x, x ~ z, z ~ y)}

## The order can be changed
DAG(y ~ z, y ~ x + u + v,  u ~ v + z)

## If you want to order the nodes (topological sort of the DAG)
DAG(y ~ z, y ~ x + u + v,  u ~ v + z, order=TRUE)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{In}
\alias{In}
\title{Indicator matrix}
\description{
  Finds the indicator matrix of the zeros of a matrix.
}
\usage{
In(A)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{A}{a matrix.}
}
\details{
The indicator matrix is a matrix of zeros and ones which has a zero
element iff the corresponding element of \code{A} is (exactly) zero. 
}
\value{a matrix of the same dimensions as \code{A}.
}
\references{
  Wermuth, N. \& Cox, D.R. (2003). Joint response graphs
  and separation induced by triangular systems. Submitted and available at
  \url{http://psystat.sowi.uni-mainz.de}.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{DAG}}, \code{\link{inducedCovGraph}}, \code{\link{inducedConGraph}}}
\examples{
## A simple way to find the overall induced coincentration graph
## The DAG on p. 198 of Cox & Wermuth (1996)
amat <- DAG(y1 ~ y2 + y3, y3 ~ y5, y4 ~ y5)
A <- edgeMatrix(amat)
In(crossprod(A))
}
\keyword{array}
\keyword{algebra}
\keyword{graphs}
\keyword{multivariate}

\eof
\name{InducedGraphs}
\alias{inducedCovGraph}
\alias{inducedConGraph}
\alias{InducedGraphs}
\title{Graphs induced by marginalization or conditioning}
\description{
  Functions to find the induced covariance or concentration graphs after
  conditioning on a set of variables and marginalizing over another set. 
}
\usage{
inducedCovGraph(amat, sel = rownames(amat), cond = NULL)
inducedConGraph(amat, sel = rownames(amat), cond = NULL)
}
\arguments{
  \item{amat}{a square Boolean matrix, the adjacency matrix of a directed
    acyclic graph. The names of rows and of the columns
    are the nodes of the DAG.}
  \item{sel}{a character vector representing a subset
    of selected variables. 
    The elements of the  vector must be a subset of
    the names of the nodes i.e. of \code{rownames(A)}. 
    By default \code{sel} is the set of the nodes of the DAG.
  }
  \item{cond}{a character vector representing the variables on which you
    want to condition. \code{cond} must be disjoint from \code{sel} and
    their union must be a subset of the set of nodes. The set difference
    between the set of nodes and the union of \code{sel} and
    \code{cond} are  the variables over which we marginalize.
    \code{cond} may be the null vector (the default), meaning that you want
    to condition on the empty set.
  }
}
\details{Given a directed acyclic graph representing a set of
  conditional independencies it is possible to obtain other
  graphs of conditional independence implied after marginalizing
  over and conditionig on sets of nodes. Two such graphs are
  the covariance graph and the concentration graph
  (Cox \& Wermuth, 1996, 2003).
}
\value{
  \code{inducedCovGraph} returns the adjacency matrix of the covariance
  graph of the variables in set \code{sel} given the variables
  in set \code{cond}, implied by the original directed acyclic graph
  with adjacency matrix \code{amat}. 
  
  \code{inducedConGraph} returns the adjacency matrix of the concentration
  graph of the variables in set \code{sel} given the variables
  in set \code{cond}, implied by the original directed acyclic graph
  with adjacency matrix \code{amat}.
  
  If \code{sel} is \code{NULL} the functions return 
  the null matrix.
  If \code{cond} is \code{NULL}, the conditioning set is empty and the
  functions return the overall induced covariance or concentration
  matrices of the selected variables.
}
\note{
  If you do not specify \code{sel} you cannot specify a non \code{NULL}
  value of \code{cond}.
  }
  \references{
    Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
      dependencies}. London: Chapman \& Hall.
    
    Wermuth, N. \& Cox, D.R. (2003). Joint response graphs
    and separation induced by triangular systems. Submitted and available at
    \url{http://psystat.sowi.uni-mainz.de}.
  }
\author{Giovanni M. Marchetti}
\seealso{\code{\link{DAG}}, \code{\link{UG}},\code{\link{isAcyclic}}}
\examples{
## Define a DAG
dag <- DAG(a ~ x, c ~ b+d, d~ x)
dag
## Induced covariance graph of a, b, d given the empty set.
inducedCovGraph(dag, sel=c("a", "b", "d"), cond=NULL)

## Induced concentration graph of a, b, c given x
inducedConGraph(dag, sel=c("a", "b", "c"), cond="x")

## Overall covariance graph
inducedCovGraph(dag)

## Overall concentration graph
inducedConGraph(dag)

## Induced covariance graph of x, b, d given c, x.
inducedCovGraph(dag, sel=c("a", "b", "d"), cond=c("c", "x"))

## Induced concentration graph of a, x, c given d, b.
inducedConGraph(dag, sel=c("a", "x", "c"), cond=c("d", "b"))

## The DAG on p. 198 of Cox & Wermuth (1996)
dag <- DAG(y1~ y2 + y3, y3 ~ y5, y4 ~ y5)

## Cf. figure 8.7 p. 203 in Cox & Wermuth (1996)
inducedCovGraph(dag, sel=c("y2", "y3", "y4", "y5"), cond="y1")
inducedCovGraph(dag, sel=c("y1", "y2", "y4", "y5"), cond="y3")
inducedCovGraph(dag, sel=c("y1", "y2", "y3", "y4"), cond="y5")

## Cf. figure 8.8 p. 203 in Cox & Wermuth (1996)
inducedConGraph(dag, sel=c("y2", "y3", "y4", "y5"), cond="y1")
inducedConGraph(dag, sel=c("y1", "y2", "y4", "y5"), cond="y3")
inducedConGraph(dag, sel=c("y1", "y2", "y3", "y4"), cond="y5")

## Cf. figure 8.9 p. 204 in Cox & Wermuth (1996)
inducedCovGraph(dag, sel=c("y2", "y3", "y4", "y5"), cond=NULL)
inducedCovGraph(dag, sel=c("y1", "y2", "y4", "y5"), cond=NULL)
inducedCovGraph(dag, sel=c("y1", "y2", "y3", "y4"), cond=NULL)

## Cf. figure 8.10 p. 204 in Cox & Wermuth (1996)
inducedConGraph(dag, sel=c("y2", "y3", "y4", "y5"), cond=NULL)
inducedConGraph(dag, sel=c("y1", "y2", "y4", "y5"), cond=NULL)
inducedConGraph(dag, sel=c("y1", "y2", "y3", "y4"), cond=NULL)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{Simple Graph Operations}
\alias{bd}
\alias{ch}
\alias{pa}
\title{Simple graph operations}
\description{
Finds the boundary, children, parents of a subset of nodes of a graph.
}
\usage{
bd(nn, amat)
ch(nn, amat)
pa(nn, amat)
}
\arguments{
  \item{nn}{a vector of nodes. It may either a numeric vector, or a
    character vector. If it is character vector must be a subset
    of the \code{rownames} of the edge matrix.}
  \item{amat}{a square matrix with dimnames specifying the adjacency matrix of the graph}
}
\details{For definitions of these operators see Lauritzen (1996).
}
\value{
  A vector, specifying the boundary or the children or the parents of nodes  \code{nn}
  in the graph.  This is a numeric or a character vector depending
  on the mode of \code{nn}.
}
\references{Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
Clarendon Press.}

\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}, \code{\link{DAG}}}
\examples{
## find boundary of a subset of nodes of a DAG
G <- DAG(y ~ x+b+a, b~a, x~a)
bd("b", G)
bd(c("b", "x"), G)
bd("x", G)
bd(c("x","b"), G)
## find boundary of a subset of nodes of an UG
G <- UG(~ y*x*z + z*h*v)
bd("z", G)
bd(c("y", "x"), G)
bd("v", G)
bd(c("x","v"), G)
## children of a subset of nodes of a DAG
G <- DAG(y ~ x+b+a, b~a, x~a)
ch("b", G)
ch(c("b", "x"), G)
ch("x", G)
ch(c("a","x"), G)
## parents of a subset of nodes of a DAG
pa("b", G)
pa(c("b", "x"), G)
pa("x", G)
pa(c("x","b"), G)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{UG}
\alias{UG}
\title{Defining an undirected graph (UG)}
\description{
  A simple way to define an undirected graph by means of a single
  model formula.
}
\usage{
UG(f)
}
\arguments{
  \item{f}{a single model formula without response}
}
\details{
  The undirected graph \eqn{G = (V, E)} is defined by a set of nodes
  \eqn{V} and a set of pairs \eqn{E}. The set of pairs is defined by
  the set of interactions in the formula. Interactions 
  define complete subgraphs (not necessarily maximal) of the UG.
  The best way is to specify interactions that match the cliques
  of the undirected graph. This is the standard way to define
  graphical models for contingency tables. Remember that some
  hierarchical models are not graphical, but they imply the same graph.

  The function returns the edge matrix of the graph, i.e. 
  a square Boolean matrix of order equal to the number of nodes of the
  graph and a one in position \eqn{(i,j)} if there is an arrow from
  \eqn{j} to \eqn{i} and zero otherwise. By default this matrix
  has ones along the main diagonal. For UGs this matrix is symmetric.
  The dimnames of the edge matrix are the nodes of the UG.
}
\value{
  a Boolean matrix with dimnames, 
  the adjacency matrix of the undirected graph.
}
\references{Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
Clarendon Press.}
\author{Giovanni M. Marchetti}
\note{} 
\seealso{\code{\link{fitConGraph}}, \code{\link{fitCovGraph}}},
\code{\link{DAG}}}
\examples{
## X independent of Y given Z
UG(~ X*Z + Y*Z)

# The saturated model
UG(~ X*Y*Z)

## The model without three-way interactions has the same graph
UG(~ X*Y + Y*Z + Z*X)
UG(~ (X + Y + Z)^2)

## Butterfly model defined from the cliques
UG(~ mec*vec*alg + alg*ana*sta)

## Some isolated nodes
UG(~x*y*z + a + b) 
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{adjMatrix}
\alias{adjMatrix}
\title{Adjacency matrix of a graph}
\description{
  Transforms the ``edge matrix'' of a graph into
  the adjacency matrix.
}
\usage{
adjMatrix(A)
}
\arguments{
  \item{A}{a square matrix representing the edge matrix of a graph.}
}
\details{
  Given the edge matrix \eqn{A} of a graph, this can be transformed into
  an adjacency matrix \eqn{E} with the formula \eqn{E = (A-I)^T}.
}
\value{
  \item{E}{the adjacency matrix of the graph.}
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{edgeMatrix}}}
\examples{
amat <- DAG(y ~ x+z, z~u+v)
E <- edgeMatrix(amat)
adjMatrix(E)
}
\keyword{array}
\keyword{algebra}
\keyword{graphs}
\keyword{multivariate}

\eof
\name{allEdges}
\alias{allEdges}
\title{All edges of a graph}
\description{
  Finds the set of edges of a graph. That is
  the set of undirected edges if the graph is undirected
  and the set of arrows if the graph is directed.
}
\usage{
allEdges(amat)
}
\arguments{
  \item{amat}{a square Boolean matrix, with dimnames, the adjacency matrix of
    a graph.}
}
\value{
  a matrix with two columns. Each row of the matrix is a pair of
  indices indicating an edge of the graph. If the graph is undirected,
  then only one of the pairs \eqn{(i,j), (j,i)} is reported.
}
\author{Giovanni M. Marchetti} 
\seealso{\code{\link{cycleMatrix}}}
\examples{
## A UG graph
allEdges(UG(~ y*v*k +v*k*d+y*d))

## A DAG
allEdges(DAG(u~h+o+p, h~o, o~p))
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{basiSet}
\alias{basiSet}
\title{Basis set of a DAG}
\description{
  Finds a basis set for the conditional independencies
  implied by a directed acyclic graph, that is a minimal set of
  independencies that imply all the other ones.
}
\usage{
basiSet(amat)
}
\arguments{
  \item{amat}{a square matrix with dimnames representing the adjacency matrix
    of a DAG.}
}
\details{
  Given a DAG and a pair of non adjacent nodes
  \eqn{(i,j)} such that \eqn{j} has higher causal order than \eqn{i},
  the set of independency statements \eqn{i} independent of
  \eqn{j} given the union of the parents of both \eqn{i} and \eqn{j}
  is a basis set (see Shipley, 2000). This basis set has the property
  to lead to independent test statistics.
}
\value{
  a list of vectors representing several conditional independence
  statements. Each vector contains the names of two non adjacent
  nodes followed by the names of nodes in the conditioning set (which
  may be empty).
}
\references{Shipley, B. (2000). A new inferential test
for path models based on directed acyclic graphs. \emph{Structural
  Equation Modeling}, 7(2), 206--218.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{shipley.test}}, \code{\link{dSep}}, \code{\link{DAG}}}
\examples{
## See Shipley (2000), Figure 2, p. 213
A <- DAG(x5~ x3+x4, x3~ x2, x4~x2, x2~ x1)
basiSet(A)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{bfs}
\alias{bfs}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Breadth first search}
\description{Breadth-first search of a connected undirected graph.
}
\usage{
bfs(amat, v = 1)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a symmetric matrix with dimnames specifying the adjacency
    matrix of the undirected graph}
  \item{v}{an integer, indicating the starting node of the
    search. Defaults to the first node.
  }
}
\details{
  Breadth-first search is a systematic method for exploring a graph.
  The algorithm is taken from Aho, Hopcroft \& Ullman (1983).
}
\value{
  \item{tree}{the edge matrix of the resulting spanning tree}
  \item{branches}{a matrix with two columns, giving the indices of the
    branches of the spanning tree}
  \item{chords}{a matrix with two columns, giving the indices of the
    chords of the spanning tree}
}
\references{
Aho, A.V., Hopcrtoft, J.E. \& Ullman, J.D. (1983). \emph{Data structures
  and algorithms.} Reading: Addison-Wesley.

  Thulasiraman, K. \& Swamy, M.N.S. (1992). \emph{Graphs: theory and
    algorithms}. New York: Wiley.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}, \code{\link{findPath}}, \code{\link{cycleMatrix}}}
\examples{
## Finding a spanning tree of the butterfly graph
bfs(UG(~ a*b*o + o*u*j))
## Starting from another node
bfs(UG(~ a*b*o + o*u*j), v=3)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{checkIdent}
\alias{checkIdent}
\title{Identifiability of a model with one latent variable}
\description{
  Checks  four sufficient conditions for
  identifiability of a Gaussian DAG model
  with one latent variable.
}
\usage{
checkIdent(amat, latent)
}
\arguments{
  \item{amat}{a square matrix with dimnames,
    representing the adjacency matrix of a DAG.}
  \item{latent}{an integer representing the latent variables among the
    nodes, or the name of the node.}
}
\details{
  Stanghellini and Wermuth (2003) give some
  sufficient conditions for checking if a Gaussian
  model that factorizes according to a DAG is identified
  when there is one hidden node over which we marginalize.
  Specifically, the function checks the conditions of
  Theorem 1, (i) and (ii) and of Theorem 2 (i) and (ii).
}
\value{
  a vector of length  four, indicating if the model is identified
  according to the conditions of theorems 1 and 2 in Stanghellini
  \& Wermuth (2003). The answer is \code{TRUE} if the  condition holds
  and thus the model is globally identified or \code{FALSE} if the
  condition fails, and thus we do not know if the model is identifiable.
}
\references{
  Stanghellini, E. \& Wermuth, N. (2003). On the
  identification of path-analysis models with one
  hidden variable. Submitted and available at
  \url{http://psystat.sowi.uni-mainz.de}.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{isGident}},  \code{\link{InducedGraphs}}}
\examples{
## See DAG in Figure 4 (a) in Stanghellini & Wermuth (2003)
d <- DAG(y1 ~ y3, y2 ~ y3 + y5, y3 ~ y4 + y5, y4 ~ y6)
checkIdent(d, "y3")  # Identifiable
checkIdent(d, "y4")  # Not identifiable?

## See DAG in Figure 5 (a) in Stanghellini & Wermuth (2003)
d <- DAG(y1 ~ y5+y4, y2 ~ y5+y4, y3 ~ y5+y4)
checkIdent(d, "y4")  # Identifiable
checkIdent(d, "y5")  # Identifiable

## A simple function to check identifiability for each node

is.ident <- function(amat){
### Check suff. conditions on each node of a DAG.
   p <- nrow(amat)
   ## Degrees of freedom
     df <- p*(p+1)/2 - p  - sum(amat==1) - p + 1
   if(df <= 0)
       warning(paste("The degrees of freedom are ", df))
    a <- rownames(amat)
    for(i in a) {
      b <- checkIdent(amat, latent=i)
      if(TRUE \%in\% b)
        cat("Node", i, names(b)[!is.na(b)], "\n")
      else
        cat("Unknown.\n")
    }
  }
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{cliques}
\alias{cliques}
\title{Cliques of an undirected graph}
\description{
Finds the cliques of an undirected graph.
}
\usage{
cliques(amat)
}
\arguments{
  \item{amat}{a square Boolean matrix with dimnames, representing the
  adjacency matrix of an undirected graph.}
}
\details{
  The cliques of a graph are the subsets of nodes which induce a
  maximally complete subgraph. Determining the cliques of a graph is
  NP-hard in general. 
}
\value{
  a list of vectors of nodes. 
}
\references{
  Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
  Clarendon Press.}
\author{Mathias Drton}
\seealso{\code{\link{UG}}, \code{\link{bd}}}
\examples{
u <- UG(~ a*b*c + c*d*e*g)
u
cliques(u)

graph22 <- UG(~x1*x2*x3+x3*x4*x5*x6+x5*x7*x8*x9*x10+x9*x11*x12*x13*x14+
x12*x15+x15*x16*x17*x18+x17*x1*x18+x1*x5*x7*x19*x20+x6*x20*x21+x22)

cliques(graph22)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{cmpGraph}
\alias{cmpGraph}
\title{The complementary graph}
\description{
Finds the complementary graph of an undirected graph.
}
\usage{
cmpGraph(amat)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{the adjacency matrix of an undirected graph}
}
\details{
  The complementary graph of an UG is the graph that has the same set
  of nodes and an undirected edge connecting \eqn{i} and \eqn{j}
  whenever there is not an \eqn{(i,j)} edge in the original UG.
}
\value{
the edge matrix of the complementary graph.
}
\references{Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
Clarendon Press.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}, \code{\link{DAG}}}
\examples{
## A chordless four-cycle
four <- UG(~ a*b + b*d + d*e + e*a)
four
cmpGraph(four)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{conComp}
\alias{conComp}
\title{Connectivity components}
\description{
Finds the connectivity components of a graph.
}
\usage{
conComp(amat)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a square matrix with dimnames, the adjacency matrix of
    a DAG or a UG.}
}
\value{
  an integer vector representing a partition of the set of nodes.
}
\references{Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
Clarendon Press.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}}
\examples{
## three connected components
conComp(UG(~a*c+c*d+e+g*o*u))
## a connected graph
conComp(UG(~ a*b+b*c+c*d+d*a))
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{correlations}
\alias{correlations}
\title{Marginal and partial correlations}
\description{
  Computes a correlation matrix with ones along the
  diagonal, marginal correlations in the lower triangle and
  partial correlations given all remaining variables in the
  upper triangle.
}
\usage{
correlations(x)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{a square symmetric matrix, a covariance matrix, or a
  data.frame for n observations and p variables.}
}
\value{
  a square correlation matrix with marginal correlations (lower
  triangle) and partial correlations (upper triangle).
}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
  dependencies}. London: Chapman \& Hall.}
\author{Giovanni M. Marchetti} 
\seealso{\code{\link{parcor}},  \code{\link{cor}}}
\examples{
## See Table 6.1 in Cox & Wermuth (1996)
data(glucose)
correlations(glucose)
}
\keyword{array}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{cycleMatrix}
\alias{cycleMatrix}
\title{Fundamental cycles}
\description{
Finds the matrix of fundamental cycles of a connected undirected graph. 
}
\usage{
cycleMatrix(amat)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a symmetric matrix with dimnames denoting the adjacency matrix
    of the undirected graph. The graph must be connected, otherwise
    the function returns an error message.}
}
\details{
  All the cycles in an UG can be obtained from combination (ring sum)
  of the set of fundamental cycles. The matrix of fundamental cycles
  is a Boolean matrix having as rows the fundamental cycles and as
  columns the edges of the graph. If an entry is one then the edge
  associated to that column belongs to the cycle associated to the row.
}
\value{
  a Boolean matrix of the fundamental cycles of the undirected graph.
  If   there is no cycle the function returns \code{NULL}.}
\references{
  Thulasiraman, K. \& Swamy, M.N.S. (1992). \emph{Graphs: theory and
    algorithms}. New York: Wiley.
}
\author{Giovanni M. Marchetti}
\note{This function is used by \code{isGident}. The row sum of the
  matrix gives the length of the cycles.}
\seealso{\code{\link{UG}}, \code{\link{findPath}},
  \code{\link{fundCycles}}, \code{\link{isGident}}, \code{\link{bfs}}}
  \examples{
## Three cycles
cycleMatrix(UG(~a*b*d+d*e+e*a*f))
## No cycle
 cycleMatrix(UG(~a*b))
## two cycles: the first is even and the second is odd
cm <- cycleMatrix(UG(~a*b+b*c+c*d+d*a+a*u*v))
apply(cm, 1, sum)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{dSep}
\alias{dSep}
\title{d-separation}
\description{
Determines if in a directed acyclic graph two set of nodes a d-separated
by a third set of nodes.
}
\usage{
dSep(amat, first, second, cond)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a Boolean matrix with dimnames, representing the adjacency matrix of a directed
    acyclic graph. The function does not check if this is the case. See
    the function \code{isAcyclic}. }
  \item{first}{a vector representing  a subset of nodes of the DAG.
    The vector should be a character vector of the
    names of the variables matching the names of the nodes in
    \code{rownames(A)}. It can be also a numeric vector of indices. }
  \item{second}{a vector representing  another subset of nodes of the DAG.
    The set \code{second} must be disjoint from \code{first}.
    The mode of \code{second} must match the mode of \code{first}.}
  \item{cond}{a vector representing a conditioning subset of nodes.
    The set \code{cond} must be disjoint from the other two sets and
    must share the same mode.
  }
}
\details{
 d-separation is a fundamental concept introduced by Pearl (1988). 
}
\value{
  a logical value. \code{TRUE} if \code{first} and \code{second} are d-separated
  by \code{cond}. 
}
\references{
  Pearl, J. (1988). \emph{Probabilistic reasoning in intelligent
    systems.} San Mateo: Morgan Kaufmann.
  
  Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
  Clarendon Press.
}
\author{Giovanni M. Marchetti} 
\seealso{\code{\link{DAG}}, \code{\link{shipley.test}},
  \code{\link{inducedCovGraph}}}
\examples{
## Conditioning on a transition node
dSep(DAG(y ~ x, x ~ z), first="y", second="z", cond = "x")
## Conditioning on a collision node (collider)
dSep(DAG(y ~ x, y ~ z), first="x", second="z", cond = "y")
## Conditioning on a source node
dSep(DAG(y ~ x, z ~ x), first="y", second="z", cond = "x")
## Marginal independence
dSep(DAG(y ~ x, y ~ z), first="x", second="z", cond = NULL)
## The DAG defined on p.~47 of Lauritzen (1996)
dag <- DAG(g ~ x, h ~ x+f, f ~ b, x ~ l+d, d ~ c, c ~ a, l ~ y, y ~ b)
dSep(dag, first="a", second="b", cond=c("x", "y"))
dSep(dag, first="a", second=c("b", "d"), cond=c("x", "y"))
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{drawGraph}
\alias{drawGraph}
\title{Drawing a graph with a simple point and click interface.}
\description{
  Draw a graph from its adjacency matrix representation.
}
\usage{
drawGraph(amat, coor = NULL, bid = TRUE, adjust = TRUE, alpha = 1, beta = 3, lwd = 1)
}
\arguments{
  \item{amat}{the adjacency matrix representation of the graph. This
  can be an undirected graph a directed acyclic graph or an ancestral graph.}
\item{coor}{an optional matrix of dimensions \eqn{p \times 2} where
  \eqn{p} is the number of vertices of the graph.If \code{coor=NULL}
  then the function chooses a default position for the nodes.}
  \item{bid}{a logical value indicating if the edges of a covariance 
    graph must be plotted as bidirected (\code{bid=TRUE})or as dashed
    lines (\code{bid=FALSE}).
  }
  \item{adjust}{a logical value. If \code{TRUE} the graph is plotted and
    the system waits until the mouse button is pressed (same behaviour
    of \code{locator} function.}  
  \item{alpha}{a positive value between controlling the
    distance from the end of the edges to the nodes of the graph.
  }
  \item{beta}{a positive value controlling the distance of the labels of
  the variables from the nodes.}
  \item{lwd}{line width of the edges.}
}
\details{
  The function is a very simple tool useful for displaying small graphs,
  with a rudimentary interface for moving nodes and edges of a given
  graph and  adjusting the final plot. For better displays use
  \pkg{Rgraphviz} package in Bioconductor project.
}
\value{
  The function plots the graph with a initial positioning of the nodes,
  as  specified by  \code{coor} and remains in a waiting state.
  The position of each node can be shifted
  by pointing and clicking (with the first mouse button) close to the node.
  When the mouse button is pressed the node which is closer to the
  selected point is moved to that position.
  Thus, one must be careful to click closer to
  the selected node than to any other node. 
  The nodes can be moved to any position by repeating the
  previous operation.  The adjustment process is terminated by pressing any mouse
  button other than the first.

  At the end of the process, the function returns invisibly the
  coordinates of the nodes. The coordinates may be used later to
  redisplay the graph.
}
\references{
  \pkg{Rgraphwiz}, \url{http://www.bioconductor.org}.
  
  GraphViz, Graph Visualization Project. AT\&T Research.
  \url{http://www.graphviz.org}.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}, \code{\link{DAG}}, \code{\link{makeAG}}}
\examples{
## A directed acyclic graph
d <- DAG(y1 ~ y2+y6, y2 ~ y3, y3 ~ y5+y6, y4 ~ y5+y6)
\dontrun{drawGraph(d)}

## An undirected graph
g <- UG(~giova*anto*armo + anto*arj*sara) 
\dontrun{drawGraph(d)}

## An ancestral graph
ag <- makeAG(ug=UG(~y0*y1), dag=DAG(y4~y2, y2~y1), bg=UG(~y2*y3+y3*y4))
\dontrun{drawGraph(ag)}
\dontrun{drawGraph(ag, bid=FALSE)}

## A more complex example with coordinates: the UNIX evolution
xy <-
structure(c(5, 15, 23, 25, 26, 17, 8, 6, 6, 7, 39, 33, 23, 49, 
19, 34, 13, 29, 50, 68, 70, 86, 89, 64, 81, 45, 64, 49, 64, 87, 
65, 65, 44, 37, 64, 68, 73, 85, 83, 95, 84, 0, 7, 15, 27, 44, 
37, 36, 20, 51, 65, 44, 64, 59, 73, 69, 78, 81, 90, 97, 89, 72, 
85, 74, 62, 68, 59, 52, 48, 43, 50, 34, 21, 18, 5, 1, 10, 2, 
11, 2, 1, 44), .Dim = c(41, 2), .Dimnames = list(NULL, c("x", 
"y")))
Unix <- DAG(
                SystemV.3 ~ SystemV.2,
                SystemV.2 ~ SystemV.0,
                SystemV.0 ~ TS4.0,
                TS4.0 ~ Unix.TS3.0 + Unix.TS.PP + CB.Unix.3,
                PDP11.SysV ~ CB.Unix.3,
                CB.Unix.3 ~ CB.Unix.2,
                CB.Unix.2 ~ CB.Unix.1,
                Unix.TS.PP ~ CB.Unix.3,
                Unix.TS3.0 ~ Unix.TS1.0 + PWB2.0 + USG3.0 + Interdata,
                USG3.0 ~ USG2.0,
                PWB2.0 ~ Interdata + PWB1.2,
                USG2.0 ~ USG1.0,
                CB.Unix.1 ~ USG1.0,
                PWB1.2 ~ PWB1.0,
                USG1.0 ~ PWB1.0,
                PWB1.0 ~ FifthEd,
                SixthEd ~ FifthEd,
                LSX ~ SixthEd,
                MiniUnix ~ SixthEd,
                Interdata ~ SixthEd,
                Wollongong ~ SixthEd,
                SeventhEd ~ Interdata,
                BSD1 ~ SixthEd,
                Xenix ~ SeventhEd,
                V32 ~ SeventhEd,
                Uniplus ~ SeventhEd,
                BSD3 ~ V32,
                BSD2 ~ BSD1,
                BSD4 ~ BSD3,
                BSD4.1 ~ BSD4,
                EigthEd ~ SeventhEd + BSD4.1,
                NinethEd ~ EigthEd,
                Ultrix32 ~ BSD4.2,
                BSD4.2 ~ BSD4.1,
                BSD4.3 ~ BSD4.2,
                BSD2.8 ~ BSD4.1 + BSD2,
                BSD2.9 ~ BSD2.8,
                Ultrix11 ~ BSD2.8 + V7M + SeventhEd,
                V7M ~ SeventhEd
                )
drawGraph(Unix, coor=xy, adjust=FALSE)
# dev.print(file="unix.fig", device=xfig) # Edit the graph with Xfig
}
\keyword{graphs}
\keyword{hplot}
\keyword{iplot}

\eof
\name{edgeMatrix}
\alias{edgeMatrix}
\title{Edge matrix of a graph}
\description{
  Transforms the adjacency matrix of a graph into
  an ``edge matrix''.
}
\usage{
edgeMatrix(E, inv=FALSE)
}
\arguments{
  \item{E}{a square matrix, representing the adjacency matrix of a
    graph.}
  \item{inv}{a logical value.}
}
\details{
  In some matrix computations for graph objects the adjacency matrix
  of the graph is transformed into an ``edge matrix''. Briefly,
  if \eqn{E} is the adjacency matrix of the
  graph, the edge matrix is \eqn{A = (E+I)^T=[a_{ij}]}. 
  Thus, \eqn{A} has ones along the diagonal
  and if the graph has no edge beteween nodes \eqn{i} and \eqn{j} the entries
  \eqn{a_{i,j}} and \eqn{a_{j,i}} are both zero.  
  If there is an arrow from \eqn{j} to \eqn{i} 
  \eqn{a_{i,j}=1} and  \eqn{a_{j,i} = 0}. If there is an undirected edge, both
  \eqn{a_{i,j}=a_{j,i}=1}.
}
\value{
  \item{A}{the edge matrix of the graph.
   If \code{TRUE} the nodes are sorted in
  inverted topological order and the edge matrix is upper triangular.}
}
\references{Wermuth, N. (2003). Analysing social science data with
  graphical Markov models. In: \emph{Highly Structured Stochastic
    Systems.} P. Green, N. Hjort \& T. Richardson (eds.),
  47--52. Oxford: Oxford University Press.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{adjMatrix}}}
\examples{
amat <- DAG(y ~ x+z, z~u+v)
amat
edgeMatrix(amat)
edgeMatrix(amat, inv=TRUE)
}
\keyword{array}
\keyword{algebra}
\keyword{graphs}
\keyword{multivariate}

\eof
\name{findPath}
\alias{findPath}
\title{Finding paths}
\description{
Finds one path between two nodes of a graph.
}
\usage{
findPath(amat, st, en, path = c())
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a square Boolean matrix with dimnames, the adjacency matrix of a graph.}
  \item{st}{an integer, the starting node.}
  \item{en}{an integer, the ending node.}
  \item{path}{a vector of integers, used in recursive calls. At the
    beginning is \code{NULL}. It should not be modified by the user.}
}
\value{
  a vector of integers, the sequence of nodes of a path,
  starting from \code{st} to \code{en}. In some graphs
  (spanning trees) there is only one path between two nodes.
}
\note{
  This function is not intended to be directly called by the user.
}
\references{Python Softftware Foundation (2003). Python Patterns ---
  Implementing Graphs. \url{http://www.python.org/docs/essays/graphs.html}.}
\author{Giovanni M. Marchetti, translating the original \pkg{Python}
  code (see references).}
\seealso{\code{\link{fundCycles}}}
\examples{
## A (single) path on a spanning tree
findPath(bfs(UG(~ a*b*c + b*d + d*e+ e*c))$tree, st=1, en=5)
}
\keyword{graphs}

\eof
\name{fitAncestralGraph}
\alias{fitAncestralGraph}
\title{Fitting of Gaussian Ancestral Graph Models}
\description{
Iterative conditional fitting of Gaussian Ancestral Graph Models.
}
\usage{
fitAncestralGraph(amat, S, n, tol = 1e-06)
}
\arguments{
  \item{amat}{a square matrix, representing the adjacency matrix of
    an ancestral graph. }
  \item{S}{a symmetric positive definite matrix with dimnames, the
    sample covariance matrix.}
  \item{n}{the sample size, a positive integer.}
  \item{tol}{a small positive number indicating the tolerance
    used in convergence checks.}
}
\details{
  Ancestral graph models were introduced by Richardson \&
  Spirtes (2002) as a class of graphical models
  whose global Markov property is closed under conditioning and
  marginalization. In the Gaussian case, the models can be parameterized
  using precision parameters, regression coefficients, and error
  covariances (compare Richardson \& Spirtes, 2002, Section 8). This
  function finds the MLE \eqn{\hat \Lambda}{\code{Lhat}} of the precision 
  parameters by fitting a concentration  
  graph model. The MLE \eqn{\hat B}{\code{Bhat}} of the regression coefficients and
  the MLE \eqn{\hat\Omega}{\code{Ohat}} of the error covariances are obtained by
  iterative conditional fitting (Drton \& Richardson, 2003a, b). The
  three sets of parameters are 
  combined to the MLE \eqn{\hat\Sigma}{\code{Shat}} of the covariance matrix by
  matrix multiplication: 
  \deqn{\hat\Sigma = \hat B^{-1}(\hat \Lambda+\hat\Omega)\hat
    B^{-T}.}{\code{Shat = solve(Bhat) \%*\% (Lhat+Ohat) \%*\% t(solve(Bhat))}.}
  Note that in Richardson \& Spirtes (2002), the matrices \eqn{\Lambda}{Lhat}
  and \eqn{\Omega}{Ohat} are defined as submatrices.
}
\value{
  \item{Shat}{the fitted covariance matrix.}
  \item{Lhat}{matrix of the fitted precisions associated with undirected
    edges and vertices that do not have an arrowhead pointing at them.}
  \item{Bhat}{matrix of the fitted regression coefficients
    associated to the directed edges.  Precisely said \code{Bhat}
    contains ones on the diagonal and the off-diagonal entry
    \eqn{(i,j)}{(i,j)} equals the \emph{negated} MLE of the regression
    coefficient for variable \eqn{j}{j} in the regression of variable
    \eqn{i}{i} on its parents. Note that this \eqn{(i,j)}{(i,j)} entry
    in \code{Bhat} corresponds to a directed edge \eqn{j \to i}{j -> i},
    and thus to a one as \eqn{(j,i)}{(j,i)} entry in the adjacency matrix.}
  \item{Ohat}{matrix of the error covariances and variances of the residuals 
    between regression equations associated with bidirected edges and
    vertices with an arrowhead pointing at them.}
  \item{dev}{the `deviance' of the model.}
  \item{df}{the degrees of freedom.}
  \item{it}{the iterations.}
}
\references{
  Drton, M. \& Richardson, T. S. (2003a). Iterative Conditional Fitting
  for Gaussian Ancestral Graph Models.  Department of Statistics,
  University of Washington, Technical Report 437, under preparation.
  (Compare also \url{http://www.math.auc.dk/gr/gr2003/material/drton.pdf})
 
  Drton, M. \& Richardson, T. S. (2003b). A new algorithm for
  maximum likelihood estimation in Gaussian graphical models for
  marginal independence. \emph{Proceedings
    of the Nineteenth Conference on Uncertainty in Artificial
    Intelligence}, 184--191.

  Richardson, T. Spirtes, P. (2002). Ancestral Graph Markov
  Models. \emph{Annals of Statistics}. 30, 4, 962--1030.
}
\author{Mathias Drton}
\seealso{\code{\link{fitCovGraph}}, \code{\link{icf}},
  \code{\link{makeAG}}, \code{\link{fitDag}}
  }
\examples{
## A covariance matrix
"S" <- structure(c(2.93, -1.7, 0.76, -0.06,
                  -1.7, 1.64, -0.78, 0.1,
                   0.76, -0.78, 1.66, -0.78,
                  -0.06, 0.1, -0.78, 0.81), .Dim = c(4,4),
                 .Dimnames = list(c("y", "x", "z", "u"), c("y", "x", "z", "u")))
## The following should give the same fit.   
## Fit an ancestral graph y -> x <-> z <- u
fitAncestralGraph(ag1 <- makeAG(dag=DAG(x~y,z~u), bg = UG(~x*z)), S, n=100)

## Fit an ancestral graph y <-> x <-> z <-> u
fitAncestralGraph(ag2 <- makeAG(bg= UG(~y*x+x*z+z*u)), S, n=100)

## Fit the same graph with fitCovGraph
fitCovGraph(ag2, S, n=100)    

## Another example for the mathematics marks data

data(marks)
S <- var(marks)
mag1 <- makeAG(bg=UG(~mechanics*vectors*algebra+algebra*analysis*statistics))
fitAncestralGraph(mag1, S, n=88)

mag2 <- makeAG(ug=UG(~mechanics*vectors+analysis*statistics),
               dag=DAG(algebra~mechanics+vectors+analysis+statistics))
fitAncestralGraph(mag2, S, n=88) # Same fit as above
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}








\eof
\name{fitConGraph}
\alias{fitConGraph}
\title{Fitting of Gaussian concentration graph models}
\description{
  Fits a concentration graph (a covariance selection model) 
  to a sample covariance matrix, assuming a Gaussian model.
}
\usage{
fitConGraph(amat, S, n, pri = FALSE, alg=2, tol = 1e-06)
}
\arguments{
  \item{amat}{a square Boolean matrix representing
    the adjacency matrix of the DAG}
  \item{S}{a symmetric positive definite matrix, the
    sample covariance matrix}
  \item{n}{an postive integer, the sample size}
  \item{pri}{a logical value. If \code{TRUE} a the value
    of the deviance at each iteration is printed.}
  \item{alg}{and integer specifying the algorithm: if \code{alg=1}
  then the covariance matrix is updated, if \code{alg=2} 
  the concentration matrix is updated. 
  }  
  \item{tol}{a small positive number indicating the tolerance
    used in convergence tests.
  }
}
\details{
  Algorithms for fitting Gaussian graphical models specified
  by undirected graphs are discussed in Speed \& Kiiveri (1986).
  This function is based on the iterative proportional
  fitting algorithm described on p. 184 of Whittaker (1990). 
}
\value{
\item{Shat}{the fitted covariance matrix.}
\item{dev}{the `deviance' of the model.}
\item{df}{the degrees of freedom.}
\item{it}{the iterations.}
}
\references{
    Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
      dependencies}. London: Chapman \& Hall.
      
  Speed, T.P. \& Kiiveri, H (1986). Gaussian
  Markov distributions over finite graphs. \emph{Annals of
    Statistics}, 14, 138--150.
  
  Whittaker, J. (1990). \emph{Graphical models in applied
    multivariate statistics}. Chichester: Wiley.
}
\author{Giovanni M. Marchetti} 
\seealso{\code{\link{UG}}, \code{\link{fitDag}},
  \code{\link{cliques}}, \code{\link{marks}}}
\examples{
## A model for the sample covariance matrix of the
## mathematics marks (Whittaker, 1990)
data(marks)
S <- cov(marks) * 87 / 88
## A butterfly concentration graph
fitConGraph(UG(~ mechanics*vectors*algebra + algebra*analysis*statistics), S , n = 88)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{fitCovGraph}
\alias{fitCovGraph}
\title{Fitting of Gaussian covariance graph models}
\description{
Fits a Gaussian covariance graph by maximum likelihood.
}
\usage{
fitCovGraph(amat, S, n, alg = "icf", dual.alg = 2, start.icf = NULL, tol = 1e-06)
}
\arguments{
  \item{amat}{A symmetric Booloean matrix with dimnames representing
    the adjacency matrix of the graph.}
  \item{S}{A symmetric positive definite matrix with dimnames, the
  sample covariance matrix}
  \item{n}{A positive integer, the sample size.}
  \item{alg}{A character string, the algorithm used.
    If  \code{alg="icf"} (the default) the algorithm is based on iterative
    conditional fitting (see Drton and Richardson, 2003). In this case
    the ML estimates are returned.
    If  \code{alg="dual"} the algorithm is based on the dual
    likelihood (see Kauermann, 1996). The fitted values are not
    true ML estimates. 
    }
    \item{dual.alg}{And integer equal to 1 or 2. It is used if
      \code{alg="dual"}. In this case a concentration graph model
      is fitted to the inverse of the sample covariance matrix, and
      \code{dual.alg} is passed to \code{fitConGraph} to
      specify the algorithm used in \code{fitConGraph}.
    }
  \item{start.icf}{A symmetric matrix used as starting value
    of the algorithm. If \code{start=NULL} the starting value
    is a diagonal matrix with diagonal entries equal to sample
    variances.} 
  \item{tol}{A small positive number indicating the tolerance
    used in convergence tests.}
}
\details{
  A covariance graph is an undirected graph  in which
  the variables associated to two non-adjacent nodes
  are marginally independent. The edges of these
  models are represented by bidirected edges
  (Drton \& Richardson, 2003) or by dashed lines
  (Cox \& Wermuth, 1996).

  By default, this function gives the ML estimates in the covariance 
  graph model, by a new iterative method (Drton \&
  Richardson, 2003).  If desired then estimates from a ``dual
  likelihood'' heuristic (Kauermann, 1996; Edwards, 2000, \S 7.4).
}
\value{
\item{Shat}{the fitted covariance matrix.}
\item{dev}{the `deviance' of the model.}
\item{df}{the degrees of freedom.}
\item{it}{the iterations.}
}
\references{
  Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
    dependencies}. London: Chapman \& Hall.

  Drton, M. \& Richardson, T. S. (2003). A new algorithm for
  maximum likelihood estimation in Gaussian graphical models for
  marginal independence. \emph{Proceedings
    of the Nineteenth Conference on Uncertainty in Artificial
    Intelligence}, 184--191.
  
  Kauermann, G. (1996). On a dualization of graphical
  Gaussian models. \emph{Scandinavian Journal of Statistics}.
  23, 105--116.
}

\author{Mathias Drton}
\seealso{\code{\link{fitConGraph}}, \code{\link{icf}}}
\examples{
## Correlations among four strategies to cope with stress for 
## 72 students. Cox & Wermuth (1996), p. 73.
##  Y = cognitive avoidance
##  X = vigilance
##  V = blunting
##  U = monitoring

R <- matrix(c(
   1.00, -0.20,  0.46,  0.01,
  -0.20,  1.00,  0.00,  0.47,
   0.46,  0.00,  1.00, -0.15,
   0.01,  0.47, -0.15,  1.00), 4, 4)
nam <- c("Y", "X", "V", "U") 
dimnames(R) <- list(nam, nam)

## A chordless 4-cycle covariance graph
gr <- UG(~ Y*X + X*U + U*V + V*Y)
fitCovGraph(gr, R, n=72)
fitCovGraph(gr, R, n=72, alg="dual")
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{fitDag}
\alias{fitDag}
\title{Fitting of Gaussian DAG models}
\description{
  Fits linear recursive regressions with independent residuals specified
  by a DAG.
}
\usage{
fitDag(amat, S, n)
}
\arguments{
  \item{amat}{a square matrix with dimnames representing
    the adjacency matrix of the DAG}
  \item{S}{a symmetric positive definite matrix, the
    sample covariance matrix}
  \item{n}{an integer > 0, the sample size}
}
\details{
  \code{fitDag} checks if the order of the nodes in adjacency matrix
  is the same of \code{S} and if not it reorders the adjacency matrix
  to match the order of the variables in \code{S}. The nodes
  of the adjacency matrix may form a subset of the variables in \code{S}.
}
\value{
  \item{Shat}{the fitted covariance matrix.}
  \item{Ahat}{a square matrix of the fitted regression coefficients. The entry
    \code{Ahat[i,j]} is minus the regression coefficient of variable
    \code{i} in the regression equation \code{j}. Thus there is
    a non zero partial regression coefficient \code{Ahat[i,j]}
    corresponding to each non zero value  \code{amat[j,i]} in
    the adjacency matrix.}
  \item{Dhat}{a vector containing the partial variances of each
    variable given the parents.}
  \item{dev}{the `deviance' of the model.}
  \item{df}{the degrees of freedom.}
}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
  dependencies}. London: Chapman \& Hall.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{DAG}}, \code{\link{swp}}.}
\examples{
dag <- DAG(y ~ x+u, x ~ z, z ~ u)
"S" <- structure(c(2.93, -1.7, 0.76, -0.06,
                   -1.7, 1.64, -0.78, 0.1,
                    0.76, -0.78, 1.66, -0.78,
                    -0.06, 0.1, -0.78, 0.81), .Dim = c(4,4),
         .Dimnames = list(c("y", "x", "z", "u"), c("y", "x", "z", "u")))
fitDag(dag, S, 200)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{fitDagLatent}
\alias{fitDagLatent}
\title{Fitting Gaussian DAG models with one latent variable}
\description{
Fits by maximum likelihood a Gaussian DAG model where one of the nodes
of the graph is latent and it is marginalised over.
}
\usage{
fitDagLatent(amat, Syy, n, latent, norm = 1, seed = 144,
             maxit = 9000, tol = 1e-06, pri = FALSE)
}
\arguments{
  \item{amat}{a square matrix with dimnames representing
    the adjacency matrix of the DAG.}
  \item{Syy}{a symmetric positive definite matrix,
    with dimnames, the sample covariance
    matrix of the observed variables.
    The set of the observed nodes of the graph must be a subset
    of the set of the names of the variables in \code{Syy}.}
  \item{n}{a positive integer, the sample size.}
  \item{latent}{the name of the latent variable.}
  \item{norm}{an integer, the kind of normalization of the latent
    variable.
    If \code{norm=1}, the latent is scaled to have unit variance. If
    \code{norm=2}, the latent is scaled to have unit partial variance
    given its parents.}
  \item{seed}{an integer, used by \code{set.seed} to specify a random
  starting point of the EM algorithm.}
\item{maxit}{an integer denoting the maximum number of iterations
  allowed for the EM algorithm. If the convergence criterion is not
  satisfied within maxit iterations the algorithms stops and a warning
  message is returned.}
\item{tol}{a small real value, denoting the tolerance used in testing convergence.}
\item{pri}{logical, if \code{pri=TRUE} then the value of the deviance at
  each iteration is printed.}
}
\details{
For the EM algorithm used see Kiivery (1987).  
}
\value{
  \item{Shat}{
    the fitted covariance matrix of all the variables
    including the latent one. The latent variable is the last.
    If \code{norm=1} then the variance of the latent variable
    is constrained to 1.
  }
  \item{Ahat}{
   a square matrix of the fitted regression coefficients. The entry
    \code{Ahat[i,j]} is minus the regression coefficient of variable
    \code{i} in the regression equation \code{j}. Thus there is
    a non zero partial regression coefficient \code{Ahat[i,j]}
    corresponding to each non zero value  \code{amat[j,i]} in
    the adjacency matrix.
  }
  \item{Dhat}{
    a vector containing the partial variances of each
    variable given the parents. If \code{norm=2} then the
    partial variance of the latent variable is constrained to 1. 
  }
  \item{dev}{
    the `deviance' of the model.
  }
  \item{df}{
    the degrees of freedom of the model.
  }
  \item{it}{
    the number of EM algorithm iterations at convergence.
  }
}
\references{
  Kiiveri,H. T. (1987). An incomplete data approach to the
  analysis of covariance structures. \emph{Psychometrika}, 52, 4,
  539--554.

  J\"oreskog, K.G. \& Goldberger, A.S. (1975). Estimation of a model
  with multiple indicators and multiple causes of a single latent
  variable.
  \emph{Journal of the American Statistical Association}, 10, 631--639.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{fitDag}}, \code{\link{checkIdent}}}
\examples{
## data from Joreskog and Goldberger (1975)
V <- matrix(c(1,     0.36,   0.21,  0.10,  0.156, 0.158,
              0.36,  1,      0.265, 0.284, 0.192, 0.324,
              0.210, 0.265,  1,     0.176, 0.136, 0.226,
              0.1,   0.284,  0.176, 1,     0.304, 0.305, 
              0.156, 0.192,  0.136, 0.304, 1,     0.344,
              0.158, 0.324,  0.226, 0.305, 0.344, 1),     6,6)
nod <- c("y1", "y2", "y3", "x1", "x2", "x3")
dimnames(V) <- list(nod,nod)
dag <- DAG(y1 ~ z, y2 ~ z, y3 ~ z, z ~ x1 + x2 + x3, x1~x2+x3, x2~x3) 
fitDagLatent(dag, V, n=530, latent="z", seed=4564)
fitDagLatent(dag, V, n=530, latent="z", norm=2, seed=145)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{fundCycles}
\alias{fundCycles}
\title{Fundamental cycles}
\description{
Finds the list of fundamental cycles of a connected undirected graph.
}
\usage{
fundCycles(amat)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{amat}{a symmetric matrix with dimnames denoting the adjacency matrix
    of the undirected graph. The graph must be connected, otherwise
    the function returns an error message.}
}
\details{
  All the cycles in an UG can be obtained from combination (ring sum)
  of the set of fundamental cycles. 
}
\value{
  a list of matrices with two columns. Every component of the list
  is associated to a cycle. The cycle is described by a
  \eqn{k \times 2} matrix whose rows are the edges of the cycle. If
  there is no cycle the function returns \code{NULL}.
}
\references{
  Thulasiraman, K. \& Swamy, M.N.S. (1992). \emph{Graphs: theory and
    algorithms}. New York: Wiley.
}
\author{Giovanni M. Marchetti}
\note{This function is used by \code{cycleMatrix} and \code{isGident}.}
\seealso{\code{\link{UG}},\code{\link{findPath}},
  \code{\link{cycleMatrix}}, \code{\link{isGident}},\code{\link{bfs}}}
  \examples{
## Three fundamental cycles
fundCycles(UG(~a*b*d + d*e + e*a*f))
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{ggm}
\alias{ggm}
\title{
The package `ggm': summary information
}
\description{
This package provides functions for defining and manipulating graphs
and for fitting graphical Gaussian models.
}
\section{Functions}{
The main functions can be classified as follows.
\itemize{
\item Functions for defining graphs (undirected, directed acyclic, 
ancestral graphs): \code{\link{UG}}, \code{\link{DAG}}, \code{\link{makeAG}};
\item Functions for doing graph operations (parents, boundary, 
cliques, connected components, fundamental cycles, d-separation):
\code{\link{pa}}, \code{\link{bd}}, \code{\link{cliques}},
\code{\link{conComp}}, \code{\link{fundCycles}}, \code{\link{dSep}};
\item Function for finding covariance and concentration graphs
induced by marginalization and conditioning:
\code{\link{inducedCovGraph}}, \code{\link{inducedConGraph}};
\item Functions for fitting by ML Gaussian DAGs, concentration graphs, 
covariance graphs and ancestral graphs:
\code{\link{fitDag}}, \code{\link{fitConGraph}},
\code{\link{fitCovGraph}}, \code{\link{fitAncestralGraph}};
\item Functions for testing several conditional independences:
\code{\link{shipley.test}};
\item Functions for checking global identification 
of DAG Gaussian  models with one 
latent variable (Stanghellini-Vicard's condition for concentration graphs, 
new sufficient conditions for DAGs):
\code{\link{isGident}}, \code{\link{checkIdent}};
\item Functions for fitting Gaussian DAG models with one latent variable:
\code{\link{fitDagLatent}}.
}
The package is intended as a contribution to the gR-project 
derscribed by Lauritzen (2002).
}
\section{Authors}{
Giovanni M. Marchetti, Dipartimento di Statistica ``G. Parenti''.
Universit\`a di Firenze, Italy; 
 
Mathias Drton,  Department of Statistics, University of Washington, USA.
}
\section{Acknowledgements}{
Many thanks to Fulvia Pennoni for testing some of 
the functions, and to Elena Stanghellini for discussion and
examples. Thanks also to Michael Perlman and Thomas Richardson.

Giovanni Marchetti has been supported by MIUR, Italy, under grant 
scheme PRIN 2002, and Mathias Drton has been supported by NSF grant 
DMS-9972008 and University of Washington RRF grant 65-3010.
}
\references{
Lauritzen, S. L. (2002). gRaphical Models in R.
\emph{R News}, 3(2)39.
}
\keyword{multivariate}
\keyword{models}
\keyword{graphs}
 

\eof
\name{glucose}
\alias{glucose}
\docType{data}
\title{Glucose control}
\description{
Data on glucose control of diabetes patients.  
}
\usage{data(glucose)}
\format{
  A data frame with 68 observations on the following 8 variables.
  \describe{
    \item{Y}{a numeric vector, Glucose control (glycosylated
    haemoglobin), values up to about 7 or 8 indicate good glucose control.}
    \item{X}{a numeric vector, a score for knowdledge about the illness.} 
    \item{Z}{a numeric vector, a score for fatalistic externality (mere
      chance determines what occurs).}
    \item{U}{a numeric vector, a score for social externaliy (powerful
      others are responsible).}
    \item{V}{a numeric vector, a score for internality (the patient is
    him or herself responsible).}
    \item{W}{a numeric vector, duration of the illness in years.}
    \item{A}{a factor, level of education, with levels \code{1}: at least
      13 years of formal schooling, \code{2}: less then 13 years.}
    \item{B}{a factor, gender with levels \code{1}: females, \code{2}: males.}
  }
}
\details{
  Data on 68 patients with fewer than 25 years of diabetes. They were
  collected at the University of Mainz to identify psychological
  and socio-economic variables possibly important for glucose control,
  when patients choose the appropriate dose of treatment depending
  on the level of blood glucose measured several times per day.  

  The variable   of primary interest is \code{Y}, glucose control, measured
  by  glycosylated haemoglobin. \code{X}, knowdledge about the illness,
  is a response of secondary interest. Variables \code{Z}, \code{U} and
  \code{V} measure patients' type of attribution, called fatalistic
  externality, social externality and internality. These are intermediate
  variables. Background   variables are \code{W}, the duration of the
  illness, \code{A} the duration of formal schooling and \code{A},
  gender. These are intrinsic variables.
}
\source{
Cox & Wermuth (1996), p. 229.
}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
  dependencies}. London: Chapman \& Hall.}
\examples{
data(glucose)
## See Cox & Wermuth (1996), Figure 6.3 p. 140 
coplot(Y ~ W | A, data=glucose) 
}
\keyword{datasets}

\eof
\name{icf}
\alias{icf}
\alias{icfmag}
\title{Iterative conditional fitting}
\description{
  Main algorithm for MLE fitting of Gaussian Covariance Graphs and
  Gaussian Ancestral models.
}
\usage{
icf(bi.graph, S, start = NULL, tol = 1e-06)
icfmag(mag, S, tol = 1e-06)
}
\arguments{
  \item{bi.graph}{a symmetric matrix with dimnames representing the adjacency matrix of an undirected graph.}
  \item{mag}{a square matrix representing the adjacency matrix of an
    ancestral graph (for example returned by \code{makeAG}).}
  \item{S}{a symmetric positive definite matrix, the
    sample covariance matrix. The order of the variables
  must be the same of the order of vertices in the adjacency matrix.}
  \item{start}{a symmetric matrix used as starting value
    of the algorithm. If \code{start=NULL} the starting value
    is a diagonal matrix.}
  \item{tol}{A small positive number indicating the tolerance
    used in convergence tests.}
}
\details{
  These functions are not intended to be called directly by the user.
}
\value{
  \item{Sigmahat}{the fitted covariance matrix.}
  \item{Bhat}{matrix of the fitted regression coefficients
    associated to the directed edges.}
  \item{Omegahat}{matrix of the partial covariances of the residuals
    between regression equations.}
  \item{iterations}{the number of iterations.}
}
\references{
  Drton, M. \& Richardson, T. S. (2003a). A new algorithm for
  maximum likelihood estimation in Gaussian graphical models for
  marginal independence. \emph{Proceedings
    of the Ninetheen Conference on Uncertainty in Artificial
    Intelligence}, 184--191.

  Drton, M. \& Richardson, T. S. (2003b). Iterative Conditional Fitting
  for Gaussian Ancestral Graph Models.  Department of Statistics,
  University of Washington, Technical Report 437, under preparation. 
}
\author{Mathias Drton}
\seealso{\code{\link{fitCovGraph}}, \code{\link{fitAncestralGraph}}}
\keyword{internal}

\eof
\name{isAcyclic}
\alias{isAcyclic}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Graph queries}
\description{
  Checks if a given graph is acyclic.
}
\usage{
isAcyclic(amat)
}
\arguments{
  \item{amat}{a square Boolean matrix with dimnames, the adjacency matrix
    of a graph.}
}
\value{
  a logical value, \code{TRUE} if the graph is acyclic and \code{FALSE}
  otherwise.
}
\references{
  Aho, A.V., Hopcroft, J.E. \& Ullman, J.D. (1983). \emph{Data structures
    and algorithms.} Reading: Addison-Wesley.
}
\author{Giovanni M. Marchetti} 
\examples{
## A cyclic graph
d <- matrix(0,3,3)
rownames(d) <- colnames(d) <- c("x", "y", "z")
d["x","y"] <- d["y", "z"] <- d["z", "x"] <- 1
## Test if the graph is acyclic
isAcyclic(d)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{isGident}
\alias{isGident}
\title{G-identifiability of an UG}
\description{
Tests if an undirected graph is G-identifiable. 
}
\usage{
isGident(amat)
}
\arguments{
  \item{amat}{a symmetric matrix with dimnames representing the adjacency
    matrix of an undirected graph}
}
\details{
  An undirected graph is said G-identifiable if every connected component of
  the complementary graph contains an odd cycle (Stanghellini and
  Wermuth, 2003). See also Tarantola and Vicard (2002).
}
\value{
  a logical value, \code{TRUE} if the graph is G-identifiable and
  \code{FALSE} if it is not.
}
\references{Stanghellini, E. \& Wermuth, N. (2003). On the
  identification of path-analysis models with one
  hidden variable. Submitted and available at
  \url{http://psystat.sowi.uni-mainz.de}.
  
  Stanghellini, E. (1997). Identification of a single-factor
  model using graphical Gaussian rules. \emph{Biometrika}, 84, 241--244.

  Tarantola, C. \& Vicard, P. (2002). Spanning trees and identifiability
  of a single-factor model. \emph{Statistical Methods \& Applications},
  11, 139--152.
  
  Vicard, P. (2000). On the identification of a single-factor model
  with correlated residuals. \emph{Biometrika}, 87, 199--205.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{UG}}, \code{\link{cmpGraph}}, \code{\link{cycleMatrix}}}
\examples{
## A not G-identifiable UG
G1 <- UG(~ a*b + u*v)
isGident(G1)
## G-identifiable UG
G2 <- UG(~ a + b + u*v)
isGident(G2)
## G-identifiable UG
G3 <- cmpGraph(UG(~a*b*c+x*y*z))
isGident(G3)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{makeAG}
\alias{makeAG}
\title{Ancestral Graphs}
\description{
  Defines an ancestral graph from the directed, undirected and
  undirected components and checks if the components are
  compatible with an ancestral graph.
}
\usage{
makeAG(dag = NULL, ug = NULL, bg = NULL)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{dag}{the adjacency matrix of a directed acyclic graph specifying
    the arrows of the ancestral graph.}
  \item{ug}{the adjacency matrix of an undirected graph specifying the
    lines of the ancestral graph.}
  \item{bg}{the adjacency matrix of an undirected graph specifying the
    bidirected edges of the ancestral graph.}
}
\details{
  An ancestral graph is a mixed graph with three types of edges: undirected,
  directed and bidirected edges. The following conditions must hold:
  (i) no undirected edge meets an arrowhead; (ii) no directed cycles;
  (iii) spouses cannot be ancestors. For details see Richardson \&
  Spirtes (2002).

  The function checks if, given the matrices
  of the undirected, directed and bidirected edges, the above three conditions
  are respected. If so, a resulting adjacency matrix \eqn{E=[e_{ij}} is
  returned, with the following convention. If \eqn{(i,j)} is a directed
  edge, then \eqn{e_{ij} = 1} and \eqn{e_{ji}=0}.
  If \eqn{(i,j)} is an undirected
  edge, then \eqn{e_{ij} = e_{ji} = 1}. Finally, if \eqn{(i,j)} is a bidirected
  edge, then \eqn{e_{ij} = e_{ji} = 2}.

  Note that the three adjacency matrices must have labels and
  may be defined using the functions \code{DAG} and \code{UG}.
}
\value{
  a square matrix obtained by combining the three graph components
  into an adjacency matrix of an ancestral graph.
  See the details for the coding of the adjacency matrix.
}
\references{Richardson, T. Spirtes, P. (2002). Ancestral Graph Markov
  Models. \emph{Annals of Statistics}. 30, 4, 962--1030.
}
\author{Giovanni M. Marchetti, Mathias Drton}
\seealso{\code{\link{UG}}, \code{\link{DAG}}}
\examples{
## Examples from Richardson and Spirtes (2002)
\dontrun{a1 <- makeAG(dag=DAG(a~b, b~d, d~c), bg=UG(~a*c))}  # Not an AG. (a2) p.969
a2 <- makeAG(dag=DAG(b ~ a, d~c), bg=UG(~a*c+c*b+b*d))           # Fig. 3 (b1) p.969
 a3 <- makeAG(ug = UG(~ a*c), dag=DAG(b ~ a, d~c), bg=UG(~ b*d)) # Fig. 3 (b2) p.969
 a5 <- makeAG(bg=UG(~alpha*beta+gamma*delta), dag=DAG(alpha~gamma,
delta~beta))  # Fig. 6 p. 973
## Another Example
 a4 <- makeAG(ug=UG(~y0*y1), dag=DAG(y4~y2, y2~y1), bg=UG(~y2*y3+y3*y4))
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{marks}
\alias{marks}
\docType{data}
\title{Mathematics marks}
\description{
Examination marks of 88 students in five subjects.
}
\usage{data(marks)}
\format{
  A data frame with 88 observations on the following 5 variables.
  \describe{
    \item{mechanics}{a numeric vector, mark in Mechanics}
    \item{vectors}{a numeric vector, mark in Vectors}
    \item{algebra}{a numeric vector, mark in Algebra}
    \item{analysis}{a numeric vector, mark in Analysis}
    \item{statistics}{a numeric vector, mark in Statistics }
  }
}
\details{
  Mechanics and Vectors were closed book examinations. Algebra, Analysis
  and Statistics were open book examinations.
}
\source{
  Mardia, K.V., Kent, J.T. \& Bibby, (1979). \emph{Multivariate
  analysis}. London: Academic Press.
}
\references{
  Whittaker, J. (1990). \emph{Graphical models in applied
    multivariate statistics}. Chichester: Wiley.
}
\examples{
data(marks)
pairs(marks)
}
\keyword{datasets}

\eof
\name{parcor}
\alias{parcor}
\title{Partial correlations}
\description{
  Finds the matrix of the partial correlations between pairs
  of variables given the rest.
}
\usage{
parcor(S)
}
\arguments{
  \item{S}{a symmetric positive definite matrix, representing a
    covariance matrix.}
}
\details{
  The algorithm computes \eqn{- \sigma^{rs}/(\sigma^{rr}
    \sigma^{ss})^{1/2}} where the \eqn{\sigma^{rs}} are concentrations,
    i.e. elements of the inverse covariance matrix.  
}
\value{
  A symmetric matrix with ones along the diagonal and in position
  \eqn{(r,s)} the partial correlation between variables \eqn{r}
  and \eqn{s} given all the remaining variables.}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
    dependencies}. London: Chapman \& Hall.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{var}}, \code{\link{cor}}, \code{\link{correlations}}}
\examples{
### Partial correlations for the mathematics marks data
data(marks)
S <- var(marks)
parcor(S)
}
\keyword{array}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}


\eof
\name{pcor}
\alias{pcor}
\title{Partial correlation}
\description{
  Computes the partial correlation between two variables given
  a set of other variables.
}
\usage{
pcor(u, S)
}
\arguments{
  \item{u}{a vector of integers of length > 1. The first two integers are
    the indices of variables the correlation of which must be
    computed. The rest of the vector is the conditioning set.
  }
  \item{S}{a symmetric positive definite matrix, a sample covariance
    matrix.}
}
\value{
  a scalar, the partial correlation matrix between variables \code{u[1]}
  and \code{u[2]} given \code{u[-c(1,2)]}.
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{cor}},  \code{\link{parcor}}, \code{\link{correlations}}}
\examples{
data(marks)
## The correlation between vectors and algebra given analysis and statistics
 pcor(c("vectors", "algebra", "analysis", "statistics"), var(marks))
## The same
pcor(c(2,3,4,5), var(marks))
## The correlation between vectors and algebra given statistics
 pcor(c("vectors", "algebra", "statistics"), var(marks))
## The marginal correlation between analysis and statistics 
pcor(c("analysis","statistics"), var(marks))
}
\keyword{models}
\keyword{multivariate}

\eof
\name{pcor.test}
\alias{pcor.test}
\title{Test for zero partial association}
\description{
  Test for conditional independence between two variables, given the
  other ones, assuming a multivariate normal distribution.
}
\usage{
pcor.test(r, q, n)
}
\arguments{
  \item{r}{a partial correlation coefficient, computed by \code{\link{pcor}}.}
  \item{q}{the number of variables in the conditioning set.}
  \item{n}{integer > 0, the sample size.}
}
\value{
  \item{tval}{The Student's t-test statistic.}
  \item{df}{The degrees of freedom}
  \item{pvalue}{The P-value, assuming a two-sided alternative.}
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{pcor}}, \code{\link{shipley.test}}}
\examples{
## Are 2,3 independent given 1?
data(marks)
pcor.test(pcor(c(2,3,1), var(marks)), 1, n=88)
}
\keyword{htest}
\keyword{multivariate}

\eof
\name{rcorr}
\alias{rcorr}
\title{Random correlation matrix}
\description{
Generates a random correlation matrix 
with the method of Marsaglia and Olkin (1984). 
}
\usage{
rcorr(d)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{d}{an integer > 0,  the order of the correlation matrix.}
}
\details{
  The algorithm uses \code{\link{rsphere}} to generate \eqn{d}
  vectors on a sphere in  \eqn{d}-space. If \eqn{Z}
  is a matrix with such vectors as rows, then the random
  correlation matrix is \eqn{ZZ'}.
}
\value{
a correlation matrix of order \code{d}.
}
\references{Marshall, G.\& Olkin, I. (1984).Generating correlation
  matrices. \emph{SIAM J. Sci. Stat. Comput.}, 5, 2, 470--475.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{rsphere}}}
\examples{
## A random correlation matrix of order 3
rcorr(3)
## A random correlation matrix of order 5
rcorr(5)
}
\keyword{distribution}
\keyword{multivariate}

\eof
\name{rnormDag}
\alias{rnormDag}
\title{Random sample from a decomposable Gaussian model}
\description{
  Generates a sample from a mean centered  multivariate normal
  distribution whose covariance matrix has a given triangular
  decomposition.
}
\usage{
rnormDag(n, A, Delta)
}
\arguments{
  \item{n}{an integer > 0, the sample size.}
  \item{A}{a square, upper triangular matrix with ones along the
    diagonal. It defines, together with \code{Delta},
    the concentration matrix (and also the covariance matrix)
    of the multivariate normal. The order of \code{A} is the
    number of components of the normal.}
  \item{Delta}{a numeric vector of length equal to the number of columns
    of \code{A}.}
}
\details{
  The value in position \eqn{(i,j)} of  \code{A} (with \eqn{i < j}) is
  a regression coefficient (with  sign changed) in the regression of
  variable \eqn{i} on variables \eqn{i+1, \dots, d}.

  The value in position \eqn{i} of \code{Delta} is the residual
  variance in the above regression.
  
}
\value{a matrix with \code{n} rows and \code{nrow(A)} columns,
  a sample from a multivariate normal distribution with mean zero
  and covariance matrix
  \code{S = solve(A) \%*\% diag(Delta) \%*\% t(solve(A))}.
}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
  dependencies}. London: Chapman \& Hall.}
\author{Giovanni M. Marchetti}

\seealso{\code{\link{triDec}}, \code{\link{fitDag}}}
\examples{
## Generate a sample of 100 observation from a multivariate normal
## The matrix of the path coefficients 
A <- matrix(
c(1, -2, -3,  0, 0,  0,  0,
  0,  1,  0, -4, 0,  0,  0,
  0,  0,  1,  2, 0,  0,  0,
  0,  0,  0,  1, 1, -5,  0,
  0,  0,  0,  0, 1,  0,  3,
  0,  0,  0,  0, 0,  1, -4,
  0,  0,  0,  0, 0,  0,  1), 7, 7, byrow=TRUE)
D <- rep(1, 7)
X <- rnormDag(100, A, D)

## The true covariance matrix
solve(A) \%*\% diag(D) \%*\% t(solve(A))

## Triangular decomposition of the sample covariance matrix
triDec(cov(X))$A
}
\keyword{distribution}
\keyword{multivariate}

\eof
\name{rsphere}
\alias{rsphere}
\title{Random vectors on a sphere}
\description{
  Generates a sample of points uniformly distributed
  on the surface of a sphere in d-space.
}
\usage{
rsphere(n, d)
}
\arguments{
  \item{n}{an integer, the sample size.}
  \item{d}{an integer, the dimension of the space. For example, a circle
    is defined in 2D-space, a sphere in 3D-space.}
}
\details{The algorithm is based on normalizing to length 1 each d-vector
   of a sample from a multivariate normal \eqn{N(0, I)}.
}
\value{a matrix of \code{n} rows and \code{d} columns.
}}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{rnorm}}, \code{\link{rcorr}}}
\examples{
## 100 points on circle
z <- rsphere(100,2)
plot(z)

## 100 points on a sphere
z <- rsphere(100, 3)
pairs(z)
}
\keyword{distribution}
\keyword{multivariate}



\eof
\name{shipley.test}
\alias{shipley.test}
\title{Test of all independencies implied by a given DAG}
\description{
  Computes a simultaneous test of all independence relationships
  implied by a given Gaussian model  defined according to
  a directed acyclic graph, based on the sample covariance matrix.
}
\usage{
shipley.test(amat, S, n)
}
\arguments{
  \item{amat}{a square Boolean matrix, of the same dimension as \code{S},
    representing the adjacency matrix of a DAG.}
  \item{S}{a symmetric positive definite matrix, the sample covariance matrix.}
  \item{n}{a positive integer, the sample size.}
}
}
\details{
  The test statistic is \eqn{C = -2 \sum \ln p_j} where \eqn{p_j} are the
  p-values of tests of conditional independence in the basis set
  computed by \code{basiSet(A)}. The p-values are independent
  uniform variables on \eqn{(0,1)} and the statistic has exactly a
  chi square distribution on \eqn{2k} degrees of freedom  where
  \eqn{k} is the number of elements of the basis set.
  Shipley (2002) calls this test Fisher's C test.
}
\value{
  \item{ctest}{Test statistic \eqn{C}.}
  \item{df}{Degrees of freedom.}
  \item{pvalue}{The P-value of the test, assuming a two-sided alternative.}
}
\references{Shipley, B. (2000). A new inferential test
for path models based on directed acyclic graphs. \emph{Structural
  Equation Modeling}, 7(2), 206--218.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{basiSet}}, \code{\link{pcor.test}}}
\examples{
## A decomposable model for the mathematics marks data
data(marks)
dag <- DAG(mechanics ~ vectors+algebra, vectors ~ algebra, statistics ~ algebra+analysis, analysis ~ algebra)
shipley.test(dag, cov(marks), n=88)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{swp}
\alias{swp}
\title{Sweep operator}
\description{
Sweeps a covariance matrix with respect to a subset of indices. 
}
\usage{
swp(V, b)
}
\arguments{
  \item{V}{a symmetric positive definite matrix, the covariance matrix.}
  \item{b}{a subset of indices of the columns of \code{V}.}
}
\details{
The sweep operator has been introduced by Beaton (1964) as a tool for
inverting symmetric matrices (see Dempster, 1969).
}
\value{
  a square matrix \code{U} of the same order as \code{V}. If \code{a} is
  the complement of \code{b}, then \code{U[a,b]} is the matrix of
  regression coefficients of \code{a} given \code{b} and \code{U[a,a]}
  is the corresponding covariance matrix of the residuals.

  If \code{b} is empty the function returns \code{V}.

  If \code{b} is the vector \code{1:nrow(V)} (or its permutation) then
  the function returns the opposite of the inverse of \code{V}.
}
\references{
  Beaton, A.E. (1964). \emph{The use of special matrix operators
    in statistical calculus}. Ed.D. thesis, Harvard
  University. Reprinted as Educational Testing Service Research Bulletin
  64-51. Princeton.
  
  Dempster, A.P. (1969). \emph{Elements of continuous multivariate
    analysis}.  Reading: Addison-Wesley.

}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{fitDag}}}
\examples{
## A very simple example
V <- matrix(c(10, 1, 1, 2), 2, 2)
swp(V, 2)
}
\keyword{array}
\keyword{algebra}
\keyword{models}
\keyword{multivariate}

\eof
\name{topSort}
\alias{topSort}
\alias{topOrder}
\title{Topological sort}
\description{
  \code{topOrder} returns the topological order of a directed
  acyclic graph (parents, before children). \code{topSort} permutates
  the adjacency matrix according to the topological order.
}
\usage{
topSort(amat)
topOrder(amat)
}
\arguments{
  \item{amat}{a square Boolean matrix with dimnames, representing the
    adjacency matrix of a directed acyclic graph.}
}
\details{
  The topological order needs not to be unique.
  After the permutation the adjacency matrix of the graph is
  upper triangular. The function is a translation of
  the Matlab function \code{topological_sort} in Toolbox
  \pkg{BNT} written by Kevin P. Murphy. 
}
\value{
  \code{topSort(amat)} returns a vector of integers representing
  the permutation of the nodes. \code{topSort(amat)} returns
  the adjacency matrix with rows and columns permutated.
}
\note{
  The order of the nodes defined by \code{DAG} is that of their first
  appearance in the model formulae (from left to right).
}
\references{
  Aho, A.V., Hopcrtoft, J.E. \& Ullman, J.D. (1983). \emph{Data structures
    and algorithms.} Reading: Addison-Wesley.
    
  Lauritzen, S. (1996). \emph{Graphical models}. Oxford:
  Clarendon Press.
}
\author{Kevin P. Murphy, Giovanni M. Marchetti} 
\seealso{\code{\link{DAG}}, \code{\link{isAcyclic}}}
\examples{
## A simple example
dag <- DAG(a ~ b, c ~ a + b, d ~ c + b)
dag
topOrder(dag)
topSort(dag)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{transClos}
\alias{transClos}
\title{Transitive closure of a graph}
\description{
  Computes the transitive closure of a graph (undirected or directed
  acyclic).}
\usage{
transClos(amat)
}
\arguments{
  \item{amat}{a Boolean matrix with dimnames representing the adjacency
    matrix
    of a graph.}
}
\details{
The transitive closure of a directed graph with adjacency matrix \eqn{A} is a
graph with adjacency matrix \eqn{A^*} such that \eqn{A^*_{i,j} = 1}
if there is a directed path from \eqn{i} to \eqn{j}. The transitive
closure of an undirected graph is defined similarly (by substituting
path to directed path).
}
\value{
  \item{A}{The adjacency matrix of the transitive closure.}
}
\references{
}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{DAG}}, \code{\link{UG}}}
\examples{
## Closure of a DAG
d <- DAG(y ~ x, x ~ z)
transClos(d)

## Closure of an UG
g <- UG(~ x*y*z+z*u+u*v)
transClos(g)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
\name{triDec}
\alias{triDec}
\title{Triangular decomposition of a covariance matrix}
\description{
  Decomposes a symmetric positive definite matrix with a variant of the
  Cholesky decomposition. 
}
\usage{
triDec(Sigma)
}
\arguments{
  \item{Sigma}{a symmetric positive definite matrix.}
}
\details{
  Any symmetric positive definite matrix \eqn{\Sigma}{Sigma}
  can be decomposed as
  \eqn{\Sigma = B \Delta B^T}{Sigma = B \%*\% Delta \%*\% t(B)}
  where \eqn{B} is upper triangular with ones
  along the main diagonal and \eqn{\Delta}{Delta} is diagonal. If
  \eqn{\Sigma}{Sigma} is a covariance
  matrix, the concentration matrix is \eqn{\Sigma^{-1} = A^T \Delta^{-1}
    A} where  \eqn{A = B^{-1}} is the matrix of the regression coefficients
  (with the sign changed) of a system of linear recursive regression
  equations  with independent residuals. In the equations each variable 
  \eqn{i} is regressed on the variables \eqn{i+1, \dots, d}.
  The elements on the diagonal of \eqn{\Delta} are the partial variances.}
}
\value{
  \item{A}{a square upper triangular matrix of the same order as
    \code{Sigma} with ones on the diagonal.}
  \item{B}{the inverse of \code{A}, another triangular matrix with unit diagonal.}
  \item{Delta}{a vector containing the diagonal values of \eqn{\Delta}.}
}
\references{Cox, D. R. \& Wermuth, N. (1996). \emph{Multivariate
  dependencies}. London: Chapman \& Hall.}
\author{Giovanni M. Marchetti}
\seealso{\code{\link{chol}}}
\examples{
## Triangular decomposition of a covariance matrix
B <- matrix(c(1,  -2, 0, 1,
              0,   1, 0, 1,
              0,   0, 1, 0,
              0,   0, 0, 1), 4, 4, byrow=TRUE)
B
D <- diag(c(3, 1, 2, 1))
S <- B \%*\% D \%*\% t(B)
triDec(S)
solve(B)
}
\keyword{array}
\keyword{algebra}
\keyword{models}
\keyword{multivariate}

\eof
\name{unmakeAG}
\alias{unmakeAG}
\title{Ancestral Graph components}
\description{
  Splits the adjacency matrix of an ancestral graph into
  undirected and bidirected components.
}
\usage{
unmakeAG(mag)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{mag}{a square matrix, with dimnames, representing an
  ancestral graph.}
}
\value{
  It is the inverse of \code{makeAG}. It returns the following
  components.
  \item{dag}{the adjacency matrix of the directed edges.}
  \item{ug}{the adjacency matrix of the undirected edges.}
  \item{bg}{the adjacency matrix of the bidirected edges.}
}
\author{Mathias Drton}
\seealso{\code{\link{makeAG}}}
\examples{
ag <- makeAG(ug=UG(~y0*y1), dag=DAG(y4~y2, y2~y1), bg=UG(~y2*y3+y3*y4))
unmakeAG(ag)
}
\keyword{graphs}
\keyword{models}
\keyword{multivariate}

\eof
