\name{Ilocos}
\alias{Ilocos}
\title{Income Metadata from Ilocos, Philippines}
\description{Income metadata from surveys conducted by the
Philippines' National Statistics Office.}
\usage{data(Ilocos)}
\format{A data frame with 632 observations of 8 variables.
\describe{
\item{income}{total income of household,}
\item{sex}{sex of household head (\code{"male"} or \code{"female"}),}
\item{family.size}{family size (sometimes averaged over two semesters),}
\item{urbanity}{factor with levels \code{"rural"} and \code{"urban"},}
\item{province}{factor indicating the particular province,}
\item{AP.income}{total household income during the APIS,}
\item{AP.family.size}{family size during APIS,}
\item{AP.weight}{APIS survey weight for each household.}}}

\details{The data contains household income and metadata in one
of the sixteen regions of the Philippines called Ilocos.
The data comes from two of the NSO's surveys:
the 1997 Family and Income and Expenditure Survey
and the 1998 Annual Poverty Indicators Survey (APIS).

Since the APIS only has a six month reference
period, the original data were rescaled using an adjustment factor
from the quarterly GDP figures that can be obtained from the
major sectors.}
\source{National Statistics Office, Philippines:
\url{http://www.census.gov.ph/}, where also the whole data set may be
obtained.}
\keyword{datasets}

\eof
\name{Lc}
\alias{Lc}
\title{Lorenz Curve}
\description{
 Computes the (empirical) ordinary and generalized Lorenz curve of a vector x}
\usage{
Lc(x, n = rep(1,length(x)), plot = FALSE)
}
\arguments{
 \item{x}{a vector containing non-negative elements.}
 \item{n}{a vector of frequencies, must be same length as \code{x}.}
 \item{plot}{logical. If TRUE the empirical Lorenz curve will be plotted.}
}
\details{\code{Lc(x)} computes the empirical ordinary Lorenz curve of \code{x}
  as well as the generalized Lorenz curve (= ordinary Lorenz curve *
  mean(x)). The result can be interpreted like this: \code{p}*100 percent
  have \code{L(p)}*100 percent of \code{x}.

  If \code{n} is changed to anything but the default \code{x} is
  interpreted as a vector of class means and \code{n} as a vector of
  class frequencies: in this case \code{Lc} will compute the minimal
  Lorenz curve (= no inequality within each group). A maximal curve can be
  computed with \code{\link{Lc.mehran}}.
}
\value{
  A list of class \code{"Lc"} with the following components:
\item{p}{vector of percentages}
\item{L}{vector with values of the ordinary Lorenz curve}
\item{L.general}{vector with values of the generalized Lorenz curve}
}
\references{B C Arnold: Majorization and the Lorenz Order: A Brief Introduction, 1987, Springer,

F A Cowell: Measurement of Inequality, 2000, in A B Atkinson / F Bourguignon (Eds): Handbook of Income Distribution, Amsterdam,

F A Cowell: Measuring Inequality, 1995 Prentice Hall/Harvester Wheatshef.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{plot.Lc}}, \code{\link{Lc.mehran}},
\code{\link{plot.theorLc}}}

\examples{
## Load and attach income (and metadata) set from Ilocos, Philippines
data(Ilocos)
attach(Ilocos)
## extract and rescale income for the provinces "Pangasinan" und "La Union"
income.p <- income[province=="Pangasinan"]/10000
income.u <- income[province=="La Union"]/10000

## compute the Lorenz curves
Lc.p <- Lc(income.p)
Lc.u <- Lc(income.u)
## it can be seen the the inequality in La Union is higher than in
## Pangasinan because the respective Lorenz curve takes smaller values.
plot(Lc.p)
lines(Lc.u, col=2)
## the picture becomes even clearer with generalized Lorenz curves
plot(Lc.p, general=TRUE)
lines(Lc.u, general=TRUE, col=2)
## inequality measures emphasize these results, e.g. Atkinson's measure
ineq(income.p, type="Atkinson")
ineq(income.u, type="Atkinson")
## or Theil's entropy measure
ineq(income.p, type="Theil", parameter=0)
ineq(income.u, type="Theil", parameter=0)



# income distribution of the USA in 1968 (in 10 classes)
# x vector of class means, n vector of class frequencies
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
n <- c(482, 825, 722, 690, 661, 760, 745, 2140, 1911, 1024)

# compute minimal Lorenz curve (= no inequality in each group)
Lc.min <- Lc(x, n=n)
# compute maximal Lorenz curve (limits of Mehran)
Lc.max <- Lc.mehran(x,n)
# plot both Lorenz curves in one plot
plot(Lc.min)
lines(Lc.max, col=4)

# add the theoretic Lorenz curve of a Lognormal-distribution with variance 0.78
lines(Lc.lognorm, parameter=0.78)
# add the theoretic Lorenz curve of a Dagum-distribution
lines(Lc.dagum, parameter=c(3.4,2.6))
}
\keyword{misc}


\eof
\name{Lc.mehran}
\alias{Lc.mehran}
\title{Mehran Bounds For Lorenz Curves}
\description{
 Computes the Mehran bounds for a Lorenz curve of grouped data}
\usage{
Lc.mehran(x,n)
}
\arguments{
 \item{x}{vector of class means.}
 \item{n}{vector of class frequencies.}
}
\value{An object of class \code{"Lc"}, but containing only
  \code{p} and \code{L}.}
\references{F Mehran: Bounds on the Gini Index Based on Observed Points of the Lorenz Curve, 1975, JASA 70, 64-66. }
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}, \code{\link{plot.Lc}}, \code{\link{plot.theorLc}}}

\examples{
# income distribution of the USA in 1968 (in 10 classes)
# x vector of class means, n vector of class frequencies
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
n <- c(482, 825, 722, 690, 661, 760, 745, 2140, 1911, 1024)

# compute minimal Lorenz curve (= no inequality in each group)
Lc.min <- Lc(x, n=n)
# compute maximal Lorenz curve (limits of Mehran)
Lc.max <- Lc.mehran(x,n)
# plot both Lorenz curves in one plot
plot(Lc.min)
lines(Lc.max, col=4)

# add the theoretic Lorenz curve of a Lognormal-distribution with variance 0.78
lines(Lc.lognorm, parameter=0.78)
# add the theoretic Lorenz curve of a Dagum-distribution
lines(Lc.dagum, parameter=c(3.4,2.6))
}
\keyword{misc}


\eof
\name{Pen}
\alias{Pen}
\title{Pen's Parade}
\description{
 plots Pen's Parade of a vector \code{x}}
\usage{
Pen(x, main="Pen's Parade", ylab=expression(x[(i)]/bar(x)),
    xlab=expression(i/n), col=4, lwd=2, las=1, ...)
}
\arguments{
\item{x}{a vector containing non-negative elements.}
 \item{main, ylab, xlab, col, lwd, las, ...}{high-level \code{\link{plot}} function parameters.}
}
\details{
Pen's Parade is basically the inverse distribution function (standardized by
\code{mean(x)}).}
\references{F A Cowell: Measurement of Inequality, 2000, in A B Atkinson / F Bourguignon (Eds): Handbook of Income Distribution, Amsterdam,

F A Cowell: Measuring Inequality, 1995 Prentice Hall/Harvester Wheatshef,

J Pen: Income Distribution, 1971, Harmondsworth: Allen Lane.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}, \code{\link{plot.Lc}}}

\examples{
# load and attach Philippine income data
data(Ilocos)
attach(Ilocos)
# plot Pen's Parade of income
Pen(income)
}
\keyword{hplot}


\eof
\name{conc}
\alias{conc}
\alias{Herfindahl}
\alias{Rosenbluth}
\title{Concentration Measures}
\description{
 computes the concentration within a vector according to the
specified concentration measure }
\usage{
conc(x, parameter = NULL, type = c("Herfindahl", "Rosenbluth"))

Herfindahl(x, parameter = 1)
Rosenbluth(x)
}
\arguments{
 \item{x}{a vector containing non-negative elements}
\item{parameter}{parameter of the concentration measure (if set to \code{NULL}
  the default parameter of the respective measure is used)}
\item{type}{character string giving the measure used to compute concentration.
  must be one of the strings in the default argument (the first character
  is sufficient). defaults to "Herfindahl".} }
\details{\code{conc} is just a wrapper for the concentration measures of
         \code{Herfindahl} and \code{Rosenbluth}
         (Hall / Tiedemann / Rosenbluth). If parameter is set to \code{NULL}
         the default from the respective function is used.
}
\value{
  the value of the concentration measure
}
\references{F A Cowell: Measurement of Inequality, 2000, in A B Atkinson / F Bourguignon (Eds): Handbook of Income Distribution, Amsterdam,

F A Cowell: Measuring Inequality, 1995 Prentice Hall/Harvester Wheatshef,

M Hall / N Tidemann: Measures of Concentration, 1967, JASA 62, 162-168.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{ineq}}, \code{\link{pov}}}

\examples{
# generate vector (of sales)
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
# compute Herfindahl coefficient with parameter 1
conc(x)
# compute coefficient of Hall/Tiedemann/Rosenbluth
conc(x, type="Rosenbluth")
}
\keyword{univar}


\eof
\name{ineq}
\alias{ineq}
\alias{Gini}
\alias{RS}
\alias{Atkinson}
\alias{Theil}
\alias{Kolm}
\alias{var.coeff}
\alias{entropy}
\title{Inequality Measures}
\description{
 computes the inequality within a vector according to the
specified inequality measure }
\usage{
ineq(x, parameter = NULL, type = c("Gini", "RS", "Atkinson", "Theil", "Kolm", "var",
     "square.var", "entropy"))

Gini(x)
RS(x)
Atkinson(x, parameter = 0.5)
Theil(x, parameter = 0)
Kolm(x, parameter = 1)
var.coeff(x, square = FALSE)
entropy(x, parameter = 0.5)
}
\arguments{
 \item{x}{a vector containing at least non-negative elements}
\item{parameter}{parameter of the inequality measure (if set to \code{NULL}
  the default parameter of the respective measure is used)}
\item{type}{character string giving the measure used to compute inequality.
  must be one of the strings in the default argument (the first character
  is sufficient). defaults to "Gini".}
\item{square}{logical. Argument of the function \code{var.coeff}, for details
  see below.}}
\details{\code{ineq} is just a wrapper for the inequality measures \code{Gini},
         \code{RS}, \code{Atkinson}, \code{Theil}, \code{Kolm},\code{var.coeff},
         \code{entropy}. If parameter is set to \code{NULL} the default from
         the respective function is used.

         \code{Gini} is the Gini coefficient, \code{RS} is the the Ricci-Schutz
         coefficient (also called Pietras measure), \code{Atkinson} gives
         Atkinsons measure and \code{Kolm} computes Kolms measure.

         If the parameter in \code{Theil} is 0 Theils entropy measure is
         computed, for every other value Theils second measure is
         computed.

         \code{ineq(x, type="var")} and \code{var.coeff(x)} respectively
         compute the coefficient of variation, while
         \code{ineq(x,type="square.var")} and \code{var.coeff(x, square=TRUE)}
         compute the squared coefficient of variation.

         \code{entropy} computes the generalized entropy, which is for
         parameter 1 equal to Theils entropy coefficient and for parameter
         0 equal to the second measure of Theil.
}
\value{
  the value of the inequality measure
}
\references{F A Cowell: Measurement of Inequality, 2000, in A B Atkinson / F Bourguignon (Eds): Handbook of Income Distribution, Amsterdam,

F A Cowell: Measuring Inequality, 1995 Prentice Hall/Harvester Wheatshef,

Marshall / Olkin: Inequalities: Theory of Majorization and Its
Applications, New York 1979 (Academic Press).}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{conc}}, \code{\link{pov}}}

\examples{
# generate vector (of incomes)
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
# compute Gini coefficient
ineq(x)
# compute Atkinson coefficient with parameter=0.5
ineq(x, parameter=0.5, type="Atkinson")
}
\keyword{univar}


\eof
\name{major}
\alias{major}
\title{Majorization}
\description{
 tests whether a vector \code{x} majorizes another vector \code{y}}
\usage{
major(x,y)
}
\arguments{
\item{x,y}{vectors containing non-negative elements (with same length and same mean)}
}
\value{
  logical. TRUE if x majorizes y (x >=(M) y), FALSE if not.
}
\details{even if \code{x} and \code{y} are comparable (i.e. have same length
and same mean) it is possible that neither x majorizes y nor y majorizes x.}
\references{Marshall / Olkin: Inequalities: Theory of Majorization and Its
Applications, New York 1979 (Academic Press)}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}}

\examples{
# generate vectors (of incomes)
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
y <- c(841, 2063, 2445, 3438, 4437, 5401, 6392, 8304, 11304, 21961)
# test whether x majorizes y (TRUE, because y is result of
# Pigou-Dalton-transfers)
major(x,y)
}
\keyword{logic}


\eof
\name{plot.Lc}
\alias{plot.Lc}
\alias{lines.Lc}
\title{Plot Lorenz Curve}
\description{
 plotting method for objects of class \code{"Lc"} (Lorenz curves)}
\usage{
\method{plot}{Lc}(x, general=FALSE, lwd=2, xlab="p", ylab="L(p)",
    main="Lorenz curve", las=1, ...) }
\arguments{
 \item{x}{an object of class \code{"Lc"}}
 \item{general}{logical. If TRUE the generalized Lorenz curve will be plotted}
 \item{lwd, xlab, ylab, main, las, ...}{high-level \code{\link{plot}} function parameters.}
}
\references{B C Arnold: Majorization and the Lorenz Order: A Brief Introduction, 1987, Springer,

F A Cowell: Measurement of Inequality, 2000, in A B Atkinson / F Bourguignon (Eds): Handbook of Income Distribution, Amsterdam,

F A Cowell: Measuring Inequality, 1995 Prentice Hall/Harvester Wheatshef.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}, \code{\link{Lc.mehran}},
\code{\link{plot.theorLc}}}

\examples{
## Load and attach income (and metadata) set from Ilocos, Philippines
data(Ilocos)
attach(Ilocos)
## extract and rescale income for the provinces "Pangasinan" und "La Union"
income.p <- income[province=="Pangasinan"]/10000
income.u <- income[province=="La Union"]/10000
## compute the Lorenz curves
Lc.p <- Lc(income.p)
Lc.u <- Lc(income.u)
## plot both Lorenz curves
plot(Lc.p)
lines(Lc.u, col=2)
}
\keyword{hplot}


\eof
\name{plot.theorLc}
\alias{plot.theorLc}
\alias{lines.theorLc}
\title{Plot Theoretical Lorenz Curves}
\description{
 Plotting method for objects of class \code{"theorLc"} (theoretical Lorenz
curves)}
\usage{
\method{plot}{theorLc}(x, parameter=NULL, xlab="p", ylab="L(p)", lwd=2, las=1, ...)
}
\arguments{
 \item{x}{an object of class \code{"theorLc"}}
 \item{parameter}{vector containing parameters of the distributions. If \code{x}
  was generated by the function \code{\link{theorLc}} the parameters are already
  fixed and have to be set to \code{NULL}.}
 \item{xlab, ylab, lwd, las, ...}{high-level \code{\link{plot}} function parameters.}
}
\references{C Dagum: Income Distribution Models, 1983, in: Johnson / Kotz (Eds): Encyclopedia of Statistical Sciences Vol.4, 27-34.

J B McDonald: Some generalized functions for the size distribution of income, 1984, Econometrica 52, 647-664.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}, \code{\link{plot.Lc}}}

\examples{
# income distribution of the USA in 1968 (in 10 classes)
# x vector of class means, n vector of class frequencies
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
n <- c(482, 825, 722, 690, 661, 760, 745, 2140, 1911, 1024)

# compute minimal Lorenz curve (= no inequality in each group)
Lc.min <- Lc(x, n=n)
# compute maximal Lorenz curve (limits of Mehran)
Lc.max <- Lc.mehran(x,n)
# plot both Lorenz curves in one plot
plot(Lc.min)
lines(Lc.max, col=4)

# add the theoretic Lorenz curve of a Lognormal-distribution with variance 0.78
lines(Lc.lognorm, parameter=0.78)
# add the theoretic Lorenz curve of a Dagum-distribution
lines(Lc.dagum, parameter=c(3.4,2.6))
}
\keyword{hplot}


\eof
\name{pov}
\alias{pov}
\alias{Watts}
\alias{Sen}
\alias{Foster}
\title{Poverty Measures}
\description{
 computes the poverty of an (income) vector according to the
specified poverty measure }
\usage{
pov(x, k, parameter = NULL,type = c("Watts", "Sen", "Foster"))

Watts(x,k)
Sen(x,k)
Foster(x,k,parameter=1)
}
\arguments{
 \item{x}{a vector containing at least non-negative elements}
 \item{k}{a constant giving the absolute poverty boundary}
 \item{parameter}{parameter of the poverty measure (if set to \code{NULL}
  the default parameter of the respective measure is used)}
 \item{type}{character string giving the measure used to compute poverty
    coefficient must be one of the strings in the default argument (the first
    character is sufficient). defaults to "Watts".}
}
\details{\code{pov} is just a wrapper for the poverty measures of
         \code{Watts}, \code{Sen} and \code{Foster}
         (Foster / Greer / Thorbecke). If parameter is set to \code{NULL} the
         default from the respective function is used.

         \code{Foster} gives for parameter 1 the head count ratio and for
         parameter 2 the poverty gap ratio.
}
\value{
  the value of the poverty measure
}
\references{Buhong Zheng: Aggregate Poverty Measures, 1997, Journal of Economic Surveys Vol.11 No.2, 123-162. 
 
J E Foster: On Economic Poverty: A Survey Of Aggregate Measures, 1984, Advances in Econometrics Vol. 3, 215-251. 
}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{ineq}}, \code{\link{conc}}}

\examples{
# generate vectors (of incomes)
x <- c(541, 1463, 2445, 3438, 4437, 5401, 6392, 8304, 11904, 22261)
y <- c(841, 2063, 2445, 3438, 4437, 5401, 6392, 8304, 11304, 21961)
# compute Watts index with poverty boundary 2000
pov(x, 2000)
pov(y, 2000)
# compute head count ratio with poverty boundary 2000
pov(x, 2000, parameter=1, type="Foster")
pov(y, 2000, parameter=1, type="Foster")}
\keyword{univar}


\eof
\name{theorLc}
\alias{theorLc}
\alias{Lc.dagum}
\alias{Lc.singh}
\alias{Lc.pareto}
\alias{Lc.lognorm}
\alias{Lc.exp}
\title{Theoretical Lorenz Curves}
\description{
 Theoretical Lorenz curves of income distributions}
\usage{
theorLc(type=c("Singh-Maddala","Dagum","lognorm","Pareto","exponential"), parameter=0)

Lc.dagum(p, parameter=c(2,2))
Lc.singh(p, parameter=c(2,2))
Lc.pareto(p, parameter=2)
Lc.lognorm(p, parameter=1)
Lc.exp(p)
}
\arguments{
 \item{type}{character string giving the income distribution.
   Must be one of the strings in the default argument (the first character
   is sufficient). Defaults to "Singh-Maddala".}
 \item{parameter}{vector containing parameter(s) of the distributions.}
 \item{p}{vector with elements from [0,1].}
}
\details{\code{Lc.dagum}, \code{Lc.singh}, \code{Lc.pareto}, \code{Lc.lognorm},
         \code{Lc.exp} are theoretical Lorenz curves of income distributions.
         They are functions of class \code{"theorLc"} with plot- and a lines-
         method, so that they can be added into an existing Lorenz curve plot.

         \code{theorLc} returns a function of class \code{"theorLc"}, that is a
         one of the above theoretical Lorenz curves with fixed parameters.

         \code{Lc.dagum} is the Lorenz curve of the Dagum distribution (2
         parameters), \code{Lc.singh} the one of the Singh-Maddala
         distribution (2 parameters), \code{Lc.pareto} the one of the Pareto
         distribution (1 parameter), \code{Lc.lognorm} the one of the Lognormal
         distribution (1 parameter) and \code{Lc.exp} the Lorenz curve of the
         exponential distribution (no parameter).
}
\value{A function of class \code{"theorLc"} or its value at \code{p}
respectively.}
\references{
C Dagum: Income Distribution Models, 1983, in: Johnson / Kotz (Eds): Encyclopedia of Statistical Sciences Vol.4, 27-34.

J B McDonald: Some generalized functions for the size distribution of income, 1984, Econometrica 52, 647-664.}
\author{Achim Zeileis \email{zeileis@ci.tuwien.ac.at}}

\seealso{\code{\link{Lc}}, \code{\link{plot.Lc}}, \code{\link{plot.theorLc}}}

\examples{
## Load and attach income (and metadata) set from Ilocos, Philippines
data(Ilocos)
attach(Ilocos)
## extract income for the province "Pangasinan"
income.p <- income[province=="Pangasinan"]

## plot empirical Lorenz curve and add theoretical Lorenz curve of
## a lognormal distribution with an estimate of the standard
## deviation parameter
Lc.p <- Lc(income.p)
plot(Lc.p)
lines(Lc.lognorm, parameter=sd(log(income.p)), col=4)



# vector of percentages
p <- (1:10)*0.1
# compute values of theoretic Lorenz curve of a Dagum-distribution
Lc.dagum(p, parameter=c(3.4,2.6))
# or
mydagum <- theorLc(type="Dagum", parameter=c(3.4,2.6))
mydagum(p)
}
\keyword{misc}


\eof
