\name{Hermitian.test}
\title{Test a Matrix for Conjugate Symmetry}
\usage{
is.Hermitian(x, tol)
Hermitian.test(x)
}
\alias{is.Hermitian}
\alias{Hermitian.test}
\description{
    \code{Hermitian.test} returns the maximum modulus of the difference
    between a matrix and the conjugate of its transpose.
    \code{is.Hermitian} checks if this value is less than a tolerance.
}
\arguments{
    \item{x}{
	a numeric or complex matrix.
    }
    \item{tol}{
	tolerance for the maximum modulus of the elements of the difference 
	elements in the lower triangle and the corresponding elements in
	the upper triangle.
    }
}
\value{
    \code{Hermitian.test} returns the maximum modulus of the difference
    between \code{x} and its conjugate transpose.
    (If \code{x} is not a matrix or is not square, \code{Inf} is returned.)
    \code{is.Hermitian} returns a logical value according to whether
    the value of \code{Hermitian.test(x)} is less than or equal to the
    tolerance \code{tol}.
}
\examples{
is.Hermitian(hilbert(6))  # returns TRUE
x <- Matrix(1:9,3,3)
x[row(x) > col(x)] <- t(x)[row(x) > col(x)]
is.Hermitian(x, tol = 0)  # returns TRUE
Hermitian.test(x)         # returns 0
}
\keyword{algebra}

\eof
\name{LowerTriangular.test}
\title{Test a Matrix for Triangularity}
\usage{
is.LowerTriangular(x, tol = 0)
is.UpperTriangular(x, tol = 0)
LowerTriangular.test(x)
UpperTriangular.test(x)
}
\alias{is.LowerTriangular}
\alias{is.UpperTriangular}
\alias{LowerTriangular.test}
\alias{UpperTriangular.test}
\description{
    \code{LowerTriangular.test} (\code{UpperTriangular.test}) returns
    the maximum modulus of the strict upper (lower) triangle of x.
    \code{is.LowerTriangular} and \code{is.UpperTriangular} check if
    this value is less than a tolerance.
}
\arguments{
    \item{x}{
	a numeric or complex matrix.
    }
    \item{tol}{
	tolerance for the maximum modulus of the elements in the upper
	(or lower) triangle of the matrix.
    }
}
\value{
    \code{LowerTriangular.test} (\code{UpperTriangular.test}) returns
    the maximum modulus of the elements in the strict upper (lower)
    triangle of the matrix \code{x}.
    
    \code{is.LowerTriangular} (\code{is.UpperTriangular}) returns a
    logical value according to whether the result of
    \code{LowerTriangular.test} (\code{UpperTriangular.test}) is less
    than or equal to \code{tol}.
%If \code{tol} is missing, this value is equivalent to 
%\code{inherits(x, "LowerTriangular")} or \code{inherits(x, "UpperTriangular")}. 
}
\examples{
x <- hilbert(9)
x[row(x) > col(x)] <- 0
is.UpperTriangular(x, tol = 0)
UpperTriangular.test(x)
}
\keyword{algebra}

\eof
\name{Matrix}
\title{Construct a Classed Matrix}
\usage{
Matrix(data=NA, nrow=1, ncol=1, byrow=FALSE, dimnames=NULL)
}
\alias{Matrix}
\alias{print.Matrix}
\alias{as.matrix.Matrix}
\description{
    Construct a Matrix of class \code{Matrix}.
}
\arguments{
  \item{data}{an optional numeric data vector or matrix.}
  \item{nrow}{the desired number of rows}
  \item{ncol}{the desired number of columns}
  \item{byrow}{logical. If \code{FALSE}
    (the default) the matrix is filled by columns, otherwise the matrix is
    filled by rows.}
  \item{dimnames}{A \code{\link{dimnames}} attribute for the matrix: a
    \code{list} of length 2.}
}
\value{
    Returns an \code{nrow} by \code{ncol} matrix of class \code{"Matrix"}. 
}
\details{
    If either of \code{nrow} or \code{ncol} is
    not given, an attempt is made to infer it from the length of
    \code{data} and the other parameter.

    Although it is sometime possible to mix unclassed matrices (created
    with \code{matrix}) with ones of class \code{"Matrix"}, it is much
    safer to always use carefully constructed ones of class
    \code{"Matrix"}.
}
\seealso{
    \code{\link{as.Matrix}}, \code{\link{Matrix.class}}, \code{\link{matrix}}.
}
\examples{
Matrix(0, 3, 2)             # a 3 by 2 matrix of zeros

Matrix(1:6, 3, 2)           # a 3 by 2 matrix
Matrix(1:6, nrow=3)   
Matrix(1:6, ncol=2)   
Matrix(1:9, nrow=3, dimnames =list(c("a", "b", "c"), c("A", "B", "C")))
}
\keyword{array}
\keyword{algebra}

\eof
\name{Matrix.class}
\title{Determine the subclass of a Matrix}
\usage{
Matrix.class(x, tol = 0, symmetry = TRUE, unit.diagonal = TRUE, 
             triangularity = c(TRUE, TRUE),
             orthogonality = c(TRUE, TRUE), 
             normality = c(TRUE, TRUE))
as.Matrix(x, tol = .Machine$double.eps)
}
\alias{as.Matrix}
\alias{Matrix.class}
\description{
    \code{Matrix.class} determines the class of a Matrix in the \code{"Matrix"}
    class structure.  \code{as.Matrix} assigns the class determined by
    \code{Matrix.class} to the object \code{x} and returns the classed object.
}
\arguments{
    \item{x}{
	a numeric or complex matrix.
    }
    \item{tol}{
	tolerance for the various tests. The default is 0.
    }
    \item{symmetry}{
	a logical value indicating whether or not to test for symmetry
	(conjugate symmetry in the complex case).
	The default is to test for symmetry.
    }
    \item{unit.diagonal}{
	a logical value indicating whether or not to test for a unit diagonal.
	The default is to test for a unit diagonal.
    }
    \item{triangularity}{
	a logical vector of length indicating whether or not to test
	lower and/or upper triangularity, respectively.
	The default is to test for both lower and upper triangularity.
    }
    \item{orthogonality}{
	a logical vector of length indicating whether or not to test row
	and/or column orthogonality, respectively.
	The default is to test for both row and column orthogonality.
    }
    \item{normality}{
	a logical vector of length indicating whether or not to see if the 
	rows and/or columns have norm one, respectively.
	The default is to test if both row and columns are normalized.
    }
}
\value{
    Returns a class for the matrix, relative to the \code{"Matrix"}
    class structure.
    The matrix will be of class \code{"Matrix"}, but there may also subclasses 
    depending on the options chosen and their results.
}
\author{Douglas Bates \email{bates@stat.wisc.edu},
    Saikat DebRoy \email{saikat@stat.wisc.edu}}
\seealso{
    \code{\link{is.Hermitian}}, \code{\link{is.LowerTriangular}},
    \code{\link{is.Orthonormal}}.
}
\examples{
x <- hilbert(9)
Matrix.class(x) # should be "Hermitian", "Matrix"
x <- Matrix( rnorm(9), 3, 3)
Matrix.class(x) # should be "Matrix"
x[row(x) < col(x)] <- 0
Matrix.class(x) # should be "LowerTriangular" "Matrix"
}
\keyword{array}
\keyword{algebra}

\eof
\name{Orthonormal.test}
\title{
    Check for Orthogonality or Orthonormality
}
\usage{
is.Orthonormal(x, tol = sqrt(.Machine$double.eps), byrow = FALSE)
is.ColOrthonormal(x, tol = sqrt(.Machine$double.eps))
is.RowOrthonormal(x, tol = sqrt(.Machine$double.eps))
Orthogonal.test(x, byrow = FALSE, normal = TRUE)
Orthonormal.test(x, byrow = FALSE)
}
\alias{is.Orthonormal}
\alias{is.ColOrthonormal}
\alias{is.RowOrthonormal}
\alias{Orthonormal.test}
\alias{Orthogonal.test}
\description{
    Check for column (or row) orthogonality (or orthonormality) in a
    matrix or return the numerical value used to perform this check.
}
\arguments{
    \item{x}{
	an numeric or complex matrix.
    }
    \item{tol}{
	tolerance in the test for orthonormality or orthogonality.
    }
    \item{byrow}{logical. If \code{FALSE} (the default) the columns are
	tested, otherwise the rows are tested.
    }
    \item{normal}{logical. If \code{TRUE} (the default) determine the
	maximum modulus of the deviation of the squared lengths of the
	columns (rows) from unity.  Otherwise, determine only the
	maximum modulus of the inner products of distinct columns (rows).
    }
}
\value{
    \code{Orthonormal.test} returns a numeric measure of the
    deviation of the columns (rows) of the matrix from orthogonality,
    when \code{normal} is \code{FALSE}, or orthonormality, when
    \code{normal} is \code{TRUE}.  This value is always at least the
    maximum modulus of the inner products of distinct columns (rows).
    When \code{normal} it \code{TRUE}, it is the maximum of this value
    and the maximum modulus of the deviation of the squared lengths of the
    columns (rows) from unity.  That is, when \code{normal} is
    \code{TRUE} the value the maximum modulus
    of the deviation of \code{crossprod(x)} from an identity matrix.
	
    \code{is.Orthonormal}, \code{is.ColOrthonormal} and
    \code{is.RowOrthonormal} return a logical value according to whether
    the columns (rows) of the matrix are orthonormal.  These simply
    compare a result from \code{Orthonormal.test} with appropriate
    options to \code{tol}.
%    If \code{tol} is missing from \code{is.Orthonormal}, this value is
%    equivalent to \code{inherits(x, "Orthonormal")}.
}
\seealso{
    \code{\link{.Machine}}.
}
\examples{
v <- rnorm(3)
v <- v / norm(v, 'F')
x <- diag(3) - 2 * (v \%*\% t(v))
is.Orthonormal(x)  # should be TRUE
is.Orthonormal(x, tol = .Machine$double.eps)
is.Orthonormal(x, tol = sqrt(.Machine$double.eps))
Orthonormal.test(x)
}
\keyword{algebra}

\eof
\name{SVD}
\title{Singular Value Decomposition of a Matrix}
\usage{
SVD(x, nu = min(dim(x)), nv = min(dim(x)))
}
\alias{SVD}
\arguments{
  \item{x}{a matrix whose SVD decomposition is to be computed.}
  \item{nu}{the number of left eigenvectors to be computed.
    This must be one of \code{0}, \code{nrow(x)} and \code{ncol(x)}.}
  \item{nv}{the number of right eigenvectors to be computed.
    This must be one of \code{0}, and \code{ncol(x)}.}
}
\description{
  Compute the singular-value decomposition of a rectangular matrix.
}
\details{
  \code{SVD} provides an interface to the Lapack routine DGESVD.
  The singular value decomposition plays an important role in many
  statistical techniques.
}
\value{
  The SVD decomposition of the matrix as computed by Lapack,
  \deqn{ \bold{X = U D V'},} where \eqn{\bold{U}} and \eqn{\bold{V}} are
  orthogonal, \eqn{\bold{V'}} means \emph{V transposed}, and
  \eqn{\bold{D}} is a diagonal matrix with the singular
  values \eqn{D_{ii}}{D[i,i]}.  Equivalently, \eqn{\bold{D = U' X V}},
  which is verified in the examples, below.

  The components in the returned value correspond directly
  to the values returned by DGESVD.
  \item{d}{a vector containing the singular values of \code{x}.}
  \item{u}{a matrix whose columns contain the left eigenvectors of \code{x}.}
  \item{vt}{a matrix whose \bold{rows} contain the right eigenvectors of
      \code{x}.  That is, the \code{vt} component is the transpose of
      \eqn{\bold{V}} in the above decomposition.  This is different from
      the LINPACK result which returns \eqn{\bold{V}}.}
}
\references{
}
\seealso{\code{\link{svd}}, \code{\link{eigen}}, \code{\link{qr}}.}
\examples{
str(X <- hilbert(9)[,1:6])
str(s <- SVD(X))
Eps <- 100 * .Machine$double.eps

D <- diag(s$d)
stopifnot(abs(X - s$u \%*\% D \%*\% s$vt) < Eps)#  X = U D V'
stopifnot(abs(D - t(s$u) \%*\% X \%*\% t(s$vt)) < Eps)#  D = U' X V

X <- cbind(1, 1:7)
str(s <- SVD(X)); D <- diag(s$d)
stopifnot(abs(X - s$u \%*\% D \%*\% s$vt) < Eps)#  X = U D V'
stopifnot(abs(D - t(s$u) \%*\% X \%*\% t(s$vt)) < Eps)#  D = U' X V
}
\keyword{algebra}
\keyword{array}

\eof
\name{det}
\title{Calculate the Determinant of a Matrix}
\alias{determinant}
\alias{determinant.Matrix}
\alias{determinant.UnitLowerTriangular}
\alias{determinant.LowerTriangular}
\alias{determinant.UnitUpperTriangular}
\alias{determinant.UpperTriangular}
\usage{
\method{determinant}{Matrix}(x, logarithm = TRUE, ...)
}
\description{
  \code{determinant} calculates the determinant (or its logarithm) of a matrix
}
\arguments{
  \item{x}{numeric matrix.}
  \item{logarithm}{logical.
    When \code{TRUE}, the default, the logarithm of the
    determinant is returned.
    When \code{FALSE}, the determinant of
    \code{x} is returned.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
 \code{determinant} return a list with two elements
  \itemize{
    \item{modulus}{a numeric value.  The modulus (absolute value) of
      the determinant or the logarithm of the modulus.  The value
      of the \code{logarithm} argument is included as an
      attribute.} 
    \item{sign}{a numeric value, which is \eqn{\pm 1}{+/- 1}
      according to whether the determinant is positive or negative.}
  }
}
\details{
  \code{determinant} is a generic function with special methods for different types
  of matrices or decompositions.
  Use \code{\link{methods}("determinant")} to list all the methods for the
  \code{determinant} generic.
  
}
\note{
  Often, computing the determinant is \emph{not} what you should be doing
  to solve a given problem.
}
\seealso{\code{\link[base]{det}}}
\examples{
h9 <- hilbert(9)
determinant(h9)
determinant(h9, log = FALSE)
}
\keyword{array}
\keyword{algebra}

\eof
\name{diagDet}
\alias{diagDet}
\title{Determinant of triangular matrices}
\description{
  Return the determinant (or log-determinant) of triangular matrices.
}
\usage{
diagDet(x, logarithm=TRUE, \dots)
}
\arguments{
  \item{x}{a numeric vector representing the diagonal of a triangular or
    a diagonal matrix.}
  \item{logarithm}{logical.
    When \code{TRUE}, the default, the logarithm of the
    determinant is returned.
    When \code{FALSE}, the determinant of
    \code{x} is returned.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  The determinant of a triangular matrix can be calculated from its
  diagonal elements only.
}
\value{
  An object of class \code{det} as a list with two elements
  \itemize{
    \item{modulus}{a numeric value.  The modulus (absolute value) of
      the determinant or the logarithm of the modulus.  The value
      of the \code{logarithm} argument is included as an
      attribute.} 
    \item{sign}{a numeric value, which is \eqn{\pm 1}{+/- 1}
      according to whether the determinant is positive or negative.}
  }
}
\note{
  Often, computing the determinant is \emph{not} what you should be doing
  to solve a given problem.
}
\seealso{\code{\link{det}}}
\examples{
diagDet(1:5)  # log of product
diagDet(1:5, logarithm = FALSE)
}
\keyword{array}
\keyword{algebra}

\eof
\name{eigen}
\title{Spectral Decomposition of a Matrix}
\usage{
%eigen(x, symmetric, only.values = TRUE)
\method{eigen}{Matrix}(x, vectors = TRUE, balance = FALSE, 
   rcond = FALSE, \dots)
}
\alias{eigen}
\alias{eigen.default}
\alias{eigen.Matrix}
\arguments{
    \item{x}{a matrix whose spectral decomposition is to be computed.}
    \item{vectors}{if \code{TRUE}, both the eigenvalues and eigenvectors
	are returned, otherwise only the eigenvalues are
	returned.}
    \item{balance}{logical.  Not used at present.}
    \item{rcond}{logical.  Not used at present.}
    \item{\dots}{further arguments passed to or from other methods.}
}
\description{
    \code{eigen} is a generic function.  The default method
    \code{eigen.default} computes eigenvalues and eigenvectors by
    providing an interface to the EISPACK routines \code{RS}, \code{RG},
    \code{CH} and \code{CG}.  \code{eigen.Matrix} provides an interface
    to the Lapack functions \code{DSYEV} and \code{DGEEV}.
}
\value{
  The spectral decomposition of \code{x} is returned
  as components of a list.
  \item{values}{a vector containing the \eqn{p} eigenvalues of \code{x},
    sorted in \emph{decreasing} order, according to \code{Mod(values)}
    if they are complex.}
  \item{vectors}{a \eqn{p\times p}{p * p} matrix whose columns contain the
    eigenvectors of \code{x}, or \code{NULL} if \code{only.values} is
    \code{TRUE}.}
}
\references{
  Smith, B. T, Boyle, J. M., Dongarra, J. J., Garbow, B. S., Ikebe,Y.,
  Klema, V., and  Moler, C. B. (1976).
  \emph{Matrix Eigensystems Routines -- EISPACK Guide}.
  Springer-Verlag Lecture Notes in Computer Science.

  Anderson, E., et al. (1999). \emph{LAPACK User's Guide,}
  3rd edition, SIAM, Philadelphia.
}
\seealso{
  \code{\link[base]{eigen}},
  \code{\link{svd}}, a generalization of \code{eigen}; \code{\link{qr}}, and
  \code{\link{chol}} for related decompositions.

  To compute the determinant of a matrix, the \code{\link{qr}}
  decomposition is much more efficient: \code{\link{det}}.
}
\examples{
eigen(cbind(c(1,-1),c(-1,1)))   # uses Eispack
eigen(cbind(c(1,-1),c(-1,1)), symmetric = FALSE)# Eispack (different algorithm).
eigen(as.Matrix(cbind(c(1,-1),c(-1,1))))  # uses Lapack

eigen(cbind(1,c(1,-1)), only.values = TRUE)
eigen(as.Matrix(cbind(1,c(1,-1))), vectors = FALSE)
eigen(cbind(-1,2:1)) # complex values
eigen(as.Matrix(cbind(-1,2:1))) # complex values from Lapack
eigen(print(cbind(c(0,1i), c(-1i,0))))# Hermite ==> real Eigen values
## 3 x 3:
eigen(cbind( 1,3:1,1:3))
eigen(as.Matrix(cbind( 1,3:1,1:3)))
eigen(cbind(-1,c(1:2,0),0:2)) # complex values
eigen(as.Matrix(cbind(-1,c(1:2,0),0:2))) # complex values

Meps <- .Machine$double.eps
m <- matrix(round(rnorm(25),3), 5,5)
sm <- m + t(m) #- symmetric matrix
eigen(as.Matrix(sm), vectors = FALSE) # ordered INcreasingly
em <- eigen(sm); V <- em$vect
print(lam <- em$values) # ordered DEcreasingly


stopifnot(
 abs(sm \%*\% V - V \%*\% diag(lam))          < 60*Meps,
 abs(sm       - V \%*\% diag(lam) \%*\% t(V)) < 60*Meps)

##------- Symmetric = FALSE:  -- different to above : ---

em <- eigen(sm, symmetric = FALSE); V2 <- em$vect
print(lam2 <- em$values) # ordered decreasingly in ABSolute value !
                         # and V2 is not normalized (where V is):
print(i <- rev(order(lam2)))
stopifnot(abs(1 - lam2[i] / lam) < 60 * Meps)

zapsmall(Diag <- t(V2) \%*\% V2) # orthogonal, but not normalized
print(norm2V <- apply(V2 * V2, 2, sum))
stopifnot( abs(1- norm2V / diag(Diag)) < 60*Meps)

V2n <- sweep(V2,2, STATS= sqrt(norm2V), FUN="/")## V2n are now Normalized EV
apply(V2n * V2n, 2, sum)
##[1] 1 1 1 1 1

## Both are now TRUE:
stopifnot(abs(sm \%*\% V2n - V2n \%*\% diag(lam2))            < 60*Meps,
          abs(sm         - V2n \%*\% diag(lam2) \%*\% t(V2n)) < 60*Meps)

## Re-ordered as with symmetric:
sV <- V2n[,i]
slam <- lam2[i]
all(abs(sm \%*\% sV -  sV \%*\% diag(slam))             < 60*Meps)
all(abs(sm        -  sV \%*\% diag(slam) \%*\% t(sV)) < 60*Meps)
## sV  *is* now equal to V  -- up to sign (+-) and rounding errors
all(abs(c(1 - abs(sV / V)))       <     1000*Meps) # TRUE (P ~ 0.95)
}
\keyword{algebra}
\keyword{array}

\eof
\name{expand}
\title{Expand a Decomposition into Factors}
\usage{
expand(x, \dots)
}
\alias{expand}
\alias{expand.default}
\description{
    Expands decompositions stored in compact form into factors.
}
\arguments{
    \item{x}{
	a matrix decomposition. 
    }
    \item{\dots}{further arguments passed to or from other methods.}
}
\value{
    the expanded decomposition.
}
\details{
    This is a generic function with special methods for different types
    of decompositions.
    Use \code{\link{methods}("expand")} to list all the methods for the
    \code{expand} generic.
}
\section{NOTE}{
    Factors for decompositions such as \code{lu} and \code{qr} can be
    stored in a compact form.
    The function \code{expand} allows all factors to be fully expanded. 
}
\seealso{
%    \code{\link{expand.lu.Matrix}}, \code{\link{expand.lu.Hermitian}},
%    \code{\link{expand.qr.Matrix}}, \code{\link{expand.eigen.Matrix}},
    \code{\link{facmul}}.
}
\examples{
library(Matrix)
x <- Matrix( 1:9, 3, 3)
expand(qr(x))
}
\keyword{algebra}


\eof
\name{facmul}
\title{Multiplication by Decomposition Factors}
\usage{
facmul(x, factor, y, transpose = FALSE, left = TRUE, \dots)
}
\alias{facmul}
\alias{facmul.default}
\description{
    Performs multiplication by factors for certain decompositions (and
    allows explicit formation of those factors).
}
\arguments{
    \item{x}{
	a matrix decomposition. 
	No missing values or IEEE special values are allowed.
    }
    \item{factor}{
	an indicator for selecting a particular factor for multiplication.
    }
    \item{y}{
	a matrix or vector to be multiplied by the factor or its transpose.
	No missing values or IEEE special values are allowed.
    }
    \item{transpose}{
	a logical value.  When \code{FALSE} (the default) the factor is
	applied.  When \code{TRUE} the transpose of the factor is applied.
    }
    \item{left}{
	a logical value.  When \code{TRUE} (the default) the factor is
	applied from the left.  When \code{FALSE} the factor is applied
	from the right.
    }
    \item{...}{
	the method for \code{"qr.Matrix"} has additional arguments.
    }
}
\value{
    the product of the selected factor (or its transpose) and \code{y}
}
\section{NOTE}{
    Factors for decompositions such as \code{lu} and \code{qr} can be
    stored in a compact form.
    The function \code{facmul} allows multiplication without explicit
    formation of the factors, saving both storage and operations.
}
\references{
Golub, G., and Van Loan, C. F. (1989).
\emph{Matrix Computations,}
2nd edition, Johns Hopkins, Baltimore.
}
%  \seealso{
%      \code{\link{facmul.lu.Matrix}}, \code{\link{facmul.lu.Hermitian}}, \code{\link{facmul.qr.Matrix}}, \code{\link{expand}}.
%  }
\examples{
library(Matrix)
x <- Matrix(rnorm(9), 3, 3)
\dontrun{
qrx <- qr(x)                      # QR factorization of x
y <- rnorm(3)
facmul( qr(x), factor = "Q", y)   # form Q y
}
}
\keyword{array}
\keyword{algebra}

\eof
\name{hilbert}
\alias{hilbert}
\title{Generate a Hilbert matrix}
\description{
    Generate the \code{n} by \code{n} symmetric Hilbert matrix.  Because
    these matrices are ill-conditioned for moderate to large \code{n},
    they are often used for testing numerical linear algebra code.
}
\usage{
hilbert(n)
}
\arguments{
  \item{n}{a positive integer.}
}
\value{
    the \code{n} by \code{n} symmetric Hilbert matrix.
}
\examples{
round(hilbert(6), 4)
}
\keyword{array}
\keyword{algebra}

\eof
\name{lu.Matrix}
\alias{lu.Matrix}
\title{
  Triangular (LU) Decomposition of a Matrix
}
\description{
  Computes the LU decomposition of a matrix.
}
\usage{
\method{lu}{Matrix}(x, norm.comp = T, \dots)
}
\arguments{
  \item{x}{
    numeric or complex matrix (does not have to be square). Missing values (NAs) 
    are not allowed.
  }
  \item{norm.comp}{
    a logical vector of length two telling whether or not to compute the 
    one and/or infinity norm of the matrix, respectively. These norms
    should be computed if \code{solve} is going to eventually be applied
    to the factorization with condition estimation. The infinity norm is
    needed for solves involving the underlying matrix, and the one norm
    is needed for solves involving its transpose. 
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  The LU decomposition of \code{x} is returned as an object of class
  \code{"lu.Matrix"}, which is a list with components \code{"l"}, a unit
  lower triangular matrix, \code{"u"}, an upper triangular matrix, and
  \code{"permutation"}, a row permutation matrix.  The one or infinity
  norm is included as an attribute, if requested.
  and/or infinity norms of the matrix if requested (used for condition
  estimation). 
}
\details{
  This function uses the Lapack subroutine DGETRF to form an LU
  factorization of a matrix X.
  The factorization has the form
  X = P * L * U
  where  P  is  a  permutation matrix, L is lower triangular
  with unit diagonal elements (lower trapezoidal if m >  n),
  and U is upper triangular (upper trapezoidal if m < n).
}
\references{
  Anderson, E., et al. (1994).
  \emph{LAPACK User's Guide,}
  2nd edition, SIAM, Philadelphia.
}
\examples{
x <- Matrix((1:16)^2, nrow = 4, ncol = 4)
lu(x)
}
\keyword{algebra}

\eof
\name{lu}
\title{Triangular Decomposition of a Square Matrix}
\usage{
lu(x, \dots)
}
\alias{lu}
\description{
    Computes triangular decompositions of matrices.
}
\arguments{
    \item{x}{
	a matrix. No missing values or IEEE special values are allowed.
    }
    \item{\dots}{further arguments passed to or from other methods.}
}
\value{
    a representation of a triangular decomposition of \code{x}.
}
\details{
    This is a generic function with special methods for different types
    of matrices.
    Use \code{\link{methods}("lu")} to list all the methods for the
    \code{lu} generic.
}
\references{
Golub, G., and Van Loan, C. F. (1989).
\emph{Matrix Computations,}
2nd edition, Johns Hopkins, Baltimore.
}
\seealso{\code{\link{qr}}, \code{\link{chol}}, \code{\link{lu.Matrix}}}
\examples{
x <- Matrix( rnorm(9), 3, 3)
lu(x)
}
\keyword{array}
\keyword{algebra}

\eof
\name{norm}
\alias{norm}
\alias{norm.default}
\alias{norm.Matrix}
\title{
    Norm of a Matrix
}
\usage{\method{norm}{Matrix}(x, type = "M", \dots) }
\description{
    Computes a matrix norm of \code{x} using Lapack.  The norm can be
    the one norm, the infinity norm, the Frobenius norm, or the maximum
    modulus among elements of a matrix, as determined by the value of
    \code{type}.
}
\arguments{
  \item{x}{
    a real or complex matrix.
  }
  \item{type}{
    A character indicating the type of norm desired.
    \code{"O"} or \code{"o"} or \code{"1"} specifies the one norm,
    (maximum absolute column sum);
    \code{"I"} or \code{"i"} specifies the infinity norm (maximum
    absolute row sum);
    \code{"F"} or \code{"f"} specifies the Frobenius norm (the
    Euclidean norm of \code{x} treated as if it were a vector);
    and \code{"M"} or \code{"m"} specifies the maximum modulus of
    all the elements in \code{x}. The default is \code{"M"}.
    
    Only the first character of \code{type[1]} is used.
  }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
    A numeric value of class \code{"norm"}, representing the quantity
    chosen according to \code{type}.
%    A copy of the call to \code{"norm"} is returned as an attribute.
}
\details{
    Based on the Lapack functions \code{dlange}, \code{dlansy}, \code{dlantr},
    \code{zlange},  \code{zlansy}, and \code{zlantr}.
}
\references{
    Anderson, E., et al. (1994).
    \emph{LAPACK User's Guide,}
    2nd edition, SIAM, Philadelphia.
}
%\seealso{\code{\link{norm.eigen.Matrix}}, \code{\link{norm.svd.Matrix}}.}
\examples{
x <- hilbert(9)
norm(x, "1")
norm(x, "I")
norm(x, "F")
norm(x, "M")
}
\keyword{algebra}

\eof
\name{rcond}
\title{Estimate the Reciprocal Condition Number}
\usage{
rcond(x, \dots)
}
\alias{rcond}
\alias{rcond.default}
\alias{rcond.Matrix}
\description{
    Estimate the reciprocal of the condition number of a matrix.
}
\arguments{
    \item{x}{
	a matrix. No missing values or IEEE special values are allowed.
    }
    \item{\dots}{further arguments passed to or from other methods.}
}
\value{
    An estimate of the reciprocal condition number of \code{x}.
}
\section{BACKGROUND}{
    The condition number of a matrix is the product of the matrix and
    the norm of its inverse (or pseudo-inverse if the matrix is not
    square).
    Since it can take on values between 1 and infinity, inclusive, it
    can be viewed as a measure of how close a matrix is to being rank
    deficient.
    It can also be viewed as a factor by which errors in solving linear
    systems with this matrix as coefficient matrix could be magnified.

    Condition numbers are usually estimated, since exact computation is
    costly in terms of floating-point operations.
    An (over) estimate of reciprocal condition number is given, since by
    doing so overflow is avoided.
    Matrices are well-conditioned if the reciprocal condition number is
    near 1 and ill-conditioned if it is near zero.
}
\references{
Golub, G., and Van Loan, C. F. (1989).
\emph{Matrix Computations,}
2nd edition, Johns Hopkins, Baltimore.
}
\examples{
x <- Matrix(rnorm(9), 3, 3)
rcond(x)
rcond(hilbert(9))  # should be about 9.1e-13
h9 <- hilbert(9)
class(h9) <- Matrix.class(h9)  # now Hermitian
rcond(h9) # slightly different answer - different algorithm
}
\keyword{array}
\keyword{algebra}

\eof
\name{schur}
\title{Schur Decomposition of a Matrix}
\usage{
\method{schur}{Matrix}(x, vectors=TRUE, \dots)
}
\alias{schur}
\alias{schur.Matrix}
\description{
    Computes the Schur decomposition and eigenvalues of a square matrix.
}
\arguments{
  \item{x}{
    numeric or complex square Matrix inheriting from class
    \code{"Matrix"}. Missing values (NAs) are not allowed.
  }
  \item{vectors}{logical.  When \code{TRUE} (the default), the Schur
    vectors are computed.
  }
  \item{\dots}{further arguments passed to or from other methods.}      
}
\value{
  An object of class \code{c("schur.Matrix", "decomp")} whose
  attributes include the eigenvalues, the Schur quasi-triangular form
  of the matrix, and the Schur vectors (if requested).
}
\details{
  Based on the Lapack functions \code{dgeesx}
}
\section{BACKGROUND}{
  If \code{A} is a square matrix, then \code{A = Q T t(Q)}, where
  \code{Q} is orthogonal, and \code{T} is upper quasi-triangular
  (nearly triangular with either 1 by 1 or 2 by 2 blocks on the
  diagonal).
  The eigenvalues of \code{A} are the same as those of \code{T},
  which are easy to compute. The Schur form is used most often for
  computing non-symmetric eigenvalue decompositions, and for computing
  functions of matrices such as matrix exponentials.
}
\references{
  Anderson, E., et al. (1994).
  \emph{LAPACK User's Guide,}
  2nd edition, SIAM, Philadelphia.
}
\examples{
schur(hilbert(9))              # Schur factorization (real eigenvalues)
A <- Matrix(rnorm( 9*9, sd = 100), nrow = 9)
schur.A <- schur(A)
mod.eig <- Mod(schur.A$values) # eigenvalue modulus
schur.A
}
\keyword{algebra}

\eof
\name{solve.Matrix}
\title{Solve a System of Equations}
\usage{
\method{solve}{Matrix}(a, b, tol = 0, transpose = FALSE, \dots)
}
\alias{solve.Matrix}
\arguments{
  \item{a}{a numeric matrix containing the coefficients of the linear
    system.}
  \item{b}{a numeric vector or matrix giving the right-hand side(s) of
    the linear system.  If omitted, \code{b} is taken to be an identity
    matrix and \code{solve} will return the inverse of \code{a}.}
  \item{tol}{the tolerance for the reciprocal condition estimate.  If
      \code{tol} is negative, no condition estimation is done.
      Otherwise the reciprocal one norm condition estimate is computed 
      and the solution is attempted only when it is greater than \code{tol}.}
  \item{transpose}{a logical value.  If \code{TRUE}, the system to be
      solved is defined by the transpose of \code{a}.  The default is
      \code{FALSE}, for which the system to be solved is defined by
       \code{a}.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
    This function solves the equation \code{a \%*\% x = b} or
    \code{t(a) \%*\% x = b} for \code{x}, where \code{b} can be either a
    vector or a matrix. 
}
\seealso{
  \code{\link{rcond.Matrix}}
}
\examples{
h8 <- hilbert(8); round(h8, 4)
solve(h8)
solve(h8, diag(8)) # similar result but not necessarily Hermitian
Hermitian.test(solve(h8))
Hermitian.test(solve(h8, diag(8)))
}
\keyword{algebra}

\eof
\name{unpack}
\title{Full Storage Representation of Packed Matrices}
\usage{
unpack(x, \dots)
}
\alias{unpack}
\alias{unpack.default}
\description{
    Expands decompositions stored in compact form into matrix factors.
}
\arguments{
    \item{x}{
	a matrix stored in packed form. 
    }
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
    A \code{Matrix} object containing the full-storage representation of \code{x}.
}
\details{
    This is a generic function with special methods for different types
    of packed matrices.
    Use \code{\link{methods}("unpack")} to list all the methods for the
    \code{unpack} generic.
}
\examples{
\dontrun{
x <- Diagonal( 1:3)
x
unpack(x)
}
}
\keyword{array}
\keyword{algebra}

\eof
