% file spatial/Kaver.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{Kaver}
\alias{Kaver}
\title{
Average K-functions from Simulations
}
\description{
Forms the average of a series of (usually simulated) K-functions.
}
\usage{
Kaver(fs, nsim, \dots)
}
\arguments{
\item{fs}{
full scale for K-fn
}
\item{nsim}{
number of simulations
}
\item{\dots}{
arguments to simulate one point process object
}}
\value{
list with components \code{x} and \code{y} of the average K-fn on L-scale.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{Kfn}}, \code{\link{Kenvl}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 40), type="b")
plot(Kfn(towns, 10), type="b", xlab="distance", ylab="L(t)")
for(i in 1:10) lines(Kfn(Psim(69), 10))
lims <- Kenvl(10,100,Psim(69))
lines(lims$x,lims$l, lty=2, col="green")
lines(lims$x,lims$u, lty=2, col="green")
lines(Kaver(10,25,Strauss(69,0.5,3.5)),  col="red")
}
\keyword{spatial}

\eof
% file spatial/Kenvl.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{Kenvl}
\alias{Kenvl}
\title{
Compute Envelope and Average of Simulations of K-fns
}
\description{
Computes envelope (upper and lower limits) and average of simulations of K-fns
}
\usage{
Kenvl(fs, nsim, \dots)
}
\arguments{
\item{fs}{
full scale for K-fn
}
\item{nsim}{
number of simulations
}
\item{\dots}{
arguments to produce one simulation
}}
\value{
list with components

\item{x}{
distances
}
\item{lower}{
min of K-fns
}
\item{upper}{
max of K-fns
}
\item{aver}{
average of K-fns
}}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{Kfn}}, \code{\link{Kaver}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 40), type="b")
plot(Kfn(towns, 10), type="b", xlab="distance", ylab="L(t)")
for(i in 1:10) lines(Kfn(Psim(69), 10))
lims <- Kenvl(10,100,Psim(69))
lines(lims$x,lims$l, lty=2, col="green")
lines(lims$x,lims$u, lty=2, col="green")
lines(Kaver(10,25,Strauss(69,0.5,3.5)), col="red")
}
\keyword{spatial}

\eof
% file spatial/Kfn.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{Kfn}
\alias{Kfn}
\title{
Compute K-fn of a Point Pattern
}
\description{
Actually computes L = sqrt(K/pi).
}
\usage{
Kfn(pp, fs, k=100)
}
\arguments{
\item{pp}{
a list such as a pp object, including components \code{x} and \code{y}
}
\item{fs}{
full scale of the plot
}
\item{k}{
number of regularly spaced distances in (0, \code{fs})
}}
\value{
A list with components

\item{x}{
vector of distances
}
\item{y}{
vector of L-fn values
}
\item{k}{
number of distances returned -- may be less than \code{k} if \code{fs} is too large
}
\item{dmin}{
minimum distance between pair of points
}
\item{lm}{
maximum deviation from L(t) = t
}}
\details{
relies on the domain D having been set by \code{ppinit} or \code{ppregion}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{ppinit}}, \code{\link{ppregion}}, \code{\link{Kaver}}, \code{\link{Kenvl}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 10), type="s", xlab="distance", ylab="L(t)")
}
\keyword{spatial}

\eof
% file spatial/Psim.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{Psim}
\alias{Psim}
\title{
Simulate Binomial Spatial Point Process
}
\description{
Simulate Binomial spatial point process.
}
\usage{
Psim(n)
}
\arguments{
\item{n}{
number of points
}}
\value{
list of vectors of \code{x} and \code{y} coordinates.
}
\details{
relies on the region being set by \code{ppinit} or \code{ppregion}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\section{Side Effects}{
uses the random number generator.
}
\seealso{
\code{\link{SSI}}, \code{\link{Strauss}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 10), type="s", xlab="distance", ylab="L(t)")
for(i in 1:10) lines(Kfn(Psim(69), 10))
}
\keyword{spatial}

\eof
% file spatial/SSI.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{SSI}
\alias{SSI}
\title{
Simulates Sequential Spatial Inhibition Point Process
}
\description{
Simulates SSI (sequential spatial inhibition) point process.
}
\usage{
SSI(n, r)
}
\arguments{
\item{n}{
number of points
}
\item{r}{
inhibition distance
}}
\value{
list of vectors of \code{x} and \code{y} coordinates
}
\details{
uses the region set by \code{ppinit} or \code{ppregion}.
}
\section{Side Effects}{
uses the random number generator.
}
\section{Warnings}{
  will never return if \code{r} is too large and it cannot place
  \code{n} points.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{Psim}}, \code{\link{Strauss}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty = "s")
plot(Kfn(towns, 10), type = "b", xlab = "distance", ylab = "L(t)")
lines(Kaver(10, 25, SSI(69, 1.2)))
}
\keyword{spatial}

\eof
% file spatial/Strauss.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{Strauss}
\alias{Strauss}
\title{
Simulates Strauss Spatial Point Process
}
\description{
Simulates Strauss spatial point process.
}
\usage{
Strauss(n, c=0, r)
}
\arguments{
\item{n}{
number of points
}
\item{c}{
parameter \code{c} in [0, 1]. \code{c} = 0 corresponds to complete inhibition at
distances up to \code{r}.
}
\item{r}{
inhibition distance
}}
\value{
list of vectors of x and y coordinates
}
\section{Side Effects}{
uses the random number generator
}
\details{
Uses spatial birth-and-death process for 4\code{n} steps, or for 40\code{n} steps
starting from a binomial pattern on the first call from an other function.
Uses the region set by \code{ppinit} or \code{ppregion}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{Psim}}, \code{\link{SSI}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 10), type="b", xlab="distance", ylab="L(t)")
lines(Kaver(10, 25, Strauss(69,0.5,3.5)))
}
\keyword{spatial}

\eof
\name{anova.trls}
\alias{anova.trls}
\alias{anovalist.trls}
\title{
Anova tables for fitted trend surface objects
}
\description{
Compute analysis of variance tables for one or more
fitted trend surface model objects; where \code{anova.trls} is
called with multiple objects, it passes on the arguments to
\code{anovalist.trls}.
}
\usage{
\method{anova}{trls}(object, \dots)
anovalist.trls(object, \dots)
}
\arguments{
\item{object}{
A fitted trend surface model object from \code{surf.ls}
}
\item{\dots}{
Further objects of the same kind
}}
\value{
\code{anova.trls} and \code{anovalist.trls} return objects corresponding
to their printed tabular output.
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.ls}}
}
\examples{
data(topo, package="MASS")
topo0 <- surf.ls(0, topo)
topo1 <- surf.ls(1, topo)
topo2 <- surf.ls(2, topo)
topo3 <- surf.ls(3, topo)
topo4 <- surf.ls(4, topo)
anova(topo0, topo1, topo2, topo3, topo4)
summary(topo4)
}
\keyword{spatial}

\eof
% file spatial/correlogram.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{correlogram}
\alias{correlogram}
\title{
Compute Spatial Correlograms
}
\description{
Compute spatial correlograms of spatial data or residuals.
}
\usage{
correlogram(krig, nint, plotit = TRUE,  \dots)
}
\arguments{
\item{krig}{
trend-surface or kriging object with columns \code{x}, \code{y}, and \code{z}
}
\item{nint}{
number of bins used
}
\item{plotit}{
logical for plotting
}
\item{\dots}{
parameters for the plot
}}
\value{
\code{x} and \code{y} coordinates of the correlogram, and \code{cnt}, the number of pairs
averaged per bin.
}
\section{Side Effects}{
Plots the correlogram if \code{plotit = TRUE}.
}
\details{
Divides range of data into \code{nint} bins, and computes the covariance for
pairs with separation in each bin, then divides by the variance.
Returns results for bins with 6 or more pairs.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{variogram}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.ls(2, topo)
correlogram(topo.kr, 25)
d <- seq(0, 7, 0.1)
lines(d, expcov(d, 0.7))
}
\keyword{spatial}

\eof
% file spatial/expcov.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{expcov}
\alias{expcov}
\alias{gaucov}
\alias{sphercov}
\title{
Spatial Covariance Functions
}
\description{
Spatial covariance functions for use with \code{surf.gls}.
}
\usage{
expcov(r, d, alpha = 0, se = 1)
gaucov(r, d, alpha = 0, se = 1)
sphercov(r, d, alpha = 0, se = 1, D = 2)
}
\arguments{
\item{r}{
vector of distances at which to evaluate the covariance
}
\item{d}{
range parameter
}
\item{alpha}{
proportion of nugget effect
}
\item{se}{
standard deviation at distance zero
}
\item{D}{
dimension of spheres.
}}
\value{
vector of covariance values.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.gls}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.ls(2, topo)
correlogram(topo.kr, 25)
d <- seq(0, 7, 0.1)
lines(d, expcov(d, 0.7))
}
\keyword{spatial}

\eof
% file spatial/ppgetregion.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{ppgetregion}
\alias{ppgetregion}
\title{
Get Domain for Spatial Point Pattern Analyses
}
\description{
Retrieves the rectangular domain \code{(xl, xu)}  x  \code{(yl, yu)} from the
underlying \code{C} code.
}
\usage{
ppgetregion()
}
\value{
A vector of length four with names \code{c("xl", "xu", "yl", "yu")}.
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{ppregion}}
}
\keyword{spatial}

\eof
% file spatial/ppinit.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{ppinit}
\alias{ppinit}
\title{
Read a Point Process Object from a File
}
\description{
Read a file in standard format and create a point process object.
}
\usage{
ppinit(file)
}
\arguments{
\item{file}{
string giving file name
}}
\value{
  class \code{"pp"} object with components \code{x}, \code{y},
  \code{xl}, \code{xu}, \code{yl}, \code{yu}
}
\details{
The file should contain

the number of points\cr
a header (ignored)\cr
xl xu yl yu scale\cr
x y (repeated n times)\cr
}
\section{Side Effects}{
Calls \code{ppregion} to set the domain.
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{ppregion}}
}
\examples{
towns <- ppinit("towns.dat")
par(pty="s")
plot(Kfn(towns, 10), type="b", xlab="distance", ylab="L(t)")
}
\keyword{spatial}

\eof
% file spatial/pplik.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{pplik}
\alias{pplik}
\title{
Pseudo-likelihood Estimation of a Strauss Spatial Point Process
}
\description{
Pseudo-likelihood estimation of a Strauss spatial point process.
}
\usage{
pplik(pp, R, ng=50, trace=FALSE)
}
\arguments{
\item{pp}{
a pp object
}
\item{R}{
the fixed parameter \code{R}
}
\item{ng}{
use a \code{ng}  x  \code{ng} grid with border \code{R} in the domain for numerical
integration.
}
\item{trace}{
logical? Should function evaluations be printed?
}}
\value{
estimate for \code{c} in the interval \code{[0, 1]}.
}
\references{
  Ripley, B. D. (1988) \emph{Statistical Inference for Spatial
    Processes.} Cambridge.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{Strauss}}
}
\examples{
pines <- ppinit("pines.dat")
pplik(pines, 0.7)
}
\keyword{spatial}

\eof
% file spatial/ppregion.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{ppregion}
\alias{ppregion}
\title{
Set Domain for Spatial Point Pattern Analyses
}
\description{
Sets the rectangular domain \code{(xl, xu)}  x  \code{(yl, yu)}.
}
\usage{
ppregion(xl = 0, xu = 1, yl = 0, yu = 1)
}
\arguments{
\item{xl}{
Either \code{xl} or a list containing components \code{xl}, \code{xu}, \code{yl}, \code{yu} (such as
a point-process object)
}
\item{xu}{
}
\item{yl}{
}
\item{yu}{
}}
\value{
none
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\section{Side Effects}{
initializes variables in the \code{C} subroutines.
}
\seealso{
\code{\link{ppinit}}, \code{\link{ppgetregion}}
}
\keyword{spatial}

\eof
\name{predict.trls}
\alias{predict.trls}
\title{
Predict method for trend surface fits
}
\description{
Predicted values based on trend surface model object
}
\usage{
\method{predict}{trls}(object, x, y, \dots)
}
\arguments{
\item{object}{
Fitted trend surface model object returned by \code{surf.ls}
}
\item{x}{
Vector of prediction location eastings (x coordinates)
}
\item{y}{
Vector of prediction location northings (y coordinates)
}
\item{\dots}{
further arguments passed to or from other methods.
}}
\value{
\code{predict.trls} produces a vector of predictions corresponding
to the prediction locations. To display the output with \code{image}
or \code{contour}, use \code{trmat} or convert the returned vector
to matrix form.
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.ls}}, \code{\link{trmat}}
}
\examples{
data(topo, package="MASS")
topo2 <- surf.ls(2, topo)
topo4 <- surf.ls(4, topo)
x <- c(1.78, 2.21)
y <- c(6.15, 6.15)
z2 <- predict(topo2, x, y)
z4 <- predict(topo4, x, y)
cat("2nd order predictions:", z2, "\\n4th order predictions:", z4, "\\n")
}
\keyword{spatial}

\eof
% file spatial/prmat.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{prmat}
\alias{prmat}
\title{
Evaluate Kriging Surface over a Grid
}
\description{
Evaluate Kriging surface over a grid.
}
\usage{
prmat(obj, xl, xu, yl, yu, n)
}
\arguments{
\item{obj}{
object returned by \code{surf.gls}
}
\item{xl}{
limits of the rectangle for grid
}
\item{xu}{
}
\item{yl}{
}
\item{yu}{
}
\item{n}{
use \code{n}  x  \code{n} grid within the rectangle
}}
\value{
list with components \code{x}, \code{y} and \code{z} suitable for \code{contour} and \code{image}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.gls}}, \code{\link{trmat}}, \code{\link{semat}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.gls(2, expcov, topo, d=0.7)
prsurf <- prmat(topo.kr, 0, 6.5, 0, 6.5, 50)
contour(prsurf, levels=seq(700, 925, 25))
}
\keyword{spatial}

\eof
% file spatial/semat.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{semat}
\alias{semat}
\title{
Evaluate Kriging Standard Error of Prediction over a Grid
}
\description{
Evaluate Kriging standard error of prediction over a grid.
}
\usage{
semat(obj, xl, xu, yl, yu, n, se)
}
\arguments{
\item{obj}{
object returned by \code{surf.gls}
}
\item{xl}{
limits of the rectangle for grid
}
\item{xu}{
}
\item{yl}{
}
\item{yu}{
}
\item{n}{
use \code{n}  x  \code{n} grid within the rectangle
}
\item{se}{
standard error at distance zero as a multiple of the supplied
covariance. Otherwise estimated, and it assumed that a correlation
function was supplied.
}}
\value{
list with components x, y and z suitable for \code{contour} and \code{image}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.gls}}, \code{\link{trmat}}, \code{\link{prmat}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.gls(2, expcov, topo, d=0.7)
prsurf <- prmat(topo.kr, 0, 6.5, 0, 6.5, 50)
contour(prsurf, levels=seq(700, 925, 25))
sesurf <- semat(topo.kr, 0, 6.5, 0, 6.5, 30)
contour(sesurf, levels=c(22,25))
}
\keyword{spatial}

\eof
% file spatial/surf.gls.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{surf.gls}
\alias{surf.gls}
\title{
Fits a Trend Surface by Generalized Least-squares
}
\description{
Fits a trend surface by generalized least-squares.
}
\usage{
surf.gls(np, covmod, x, y, z, nx = 1000, \dots)
}
\arguments{
\item{np}{
degree of polynomial surface
}
\item{covmod}{
function to evaluate covariance or correlation function
}
\item{x}{
x coordinates or a data frame with columns \code{x}, \code{y}, \code{z}
}
\item{y}{
y coordinates
}
\item{z}{
z coordinates. Will supersede \code{x$z}
}
\item{nx}{
Number of bins for table of the covariance. Increasing adds accuracy,
and increases size of the object.
}
\item{\dots}{
parameters for \code{covmod}
}}
\value{
list with components

\item{beta}{
the coefficients
}
\item{x}{
}
\item{y}{
}
\item{z}{
and others for internal use only.
}}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{trmat}}, \code{\link{surf.ls}}, \code{\link{prmat}}, \code{\link{semat}}, \code{\link{expcov}}, \code{\link{gaucov}}, \code{\link{sphercov}}
}
\examples{
library(MASS)  # for eqscplot
data(topo, package="MASS")
topo.kr <- surf.gls(2, expcov, topo, d=0.7)
trsurf <- trmat(topo.kr, 0, 6.5, 0, 6.5, 50)
eqscplot(trsurf, type = "n")
contour(trsurf, add = TRUE)

prsurf <- prmat(topo.kr, 0, 6.5, 0, 6.5, 50)
contour(prsurf, levels=seq(700, 925, 25))
sesurf <- semat(topo.kr, 0, 6.5, 0, 6.5, 30)
eqscplot(sesurf, type = "n")
contour(sesurf, levels = c(22, 25), add = TRUE)
}
\keyword{spatial}

\eof
% file spatial/surf.ls.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{surf.ls}
\alias{surf.ls}
\title{
Fits a Trend Surface by Least-squares
}
\description{
Fits a trend surface by least-squares.
}
\usage{
surf.ls(np, x, y, z)
}
\arguments{
\item{np}{
degree of polynomial surface
}
\item{x}{
x coordinates or a data frame with columns \code{x}, \code{y}, \code{z}
}
\item{y}{
y coordinates
}
\item{z}{
z coordinates. Will supersede \code{x$z}
}}
\value{
list with components

\item{beta}{
the coefficients
}
\item{x}{
}
\item{y}{
}
\item{z}{
and others for internal use only.
}}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{trmat}}, \code{\link{surf.gls}}
}
\examples{
library(MASS)  # for eqscplot
data(topo, package="MASS")
topo.kr <- surf.ls(2, topo)
trsurf <- trmat(topo.kr, 0, 6.5, 0, 6.5, 50)
eqscplot(trsurf, type = "n")
contour(trsurf, add = TRUE)
points(topo)

eqscplot(trsurf, type = "n")
contour(trsurf, add = TRUE)
plot(topo.kr, add = TRUE)
title(xlab= "Circle radius proportional to Cook's influence statistic")
}
\keyword{spatial}

\eof
\name{trls.influence}
\alias{trls.influence}
\alias{plot.trls}
\title{
  Regression diagnostics for trend surfaces
}
\description{
  This function provides the basic quantities which are used in
  forming a variety of diagnostics for checking the quality of
  regression fits for trend surfaces calculated by \code{surf.ls}.
}
\usage{
trls.influence(object)
\method{plot}{trls}(x, border = "red", col = NA, pch = 4, cex = 0.6,
     add = FALSE, div = 8, \dots)
}
\arguments{
  \item{object, x}{
    Fitted trend surface model from \code{surf.ls}
  }
  \item{div}{
    scaling factor for influence circle radii in \code{plot.trls}
  }
  \item{add}{
    add influence plot to existing graphics if \code{TRUE}
  }
  \item{border, col, pch, cex, \dots}{
    additional graphical parameters
  }
}
\value{
  \code{trls.influence} returns a list with components:
  
  \item{r}{
    raw residuals as given by \code{residuals.trls}
  }
  \item{hii}{
    diagonal elements of the Hat matrix
  }
  \item{stresid}{
    standardised residuals
  }
  \item{Di}{
    Cook's statistic
  }
}
\references{
  Unwin, D. J., Wrigley, N. (1987) Towards a general-theory of
  control point distribution effects in trend surface models.
  \emph{Computers and Geosciences,} \bold{13}, 351--355.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.ls}}, \code{\link{influence.measures}}, \code{\link{plot.lm}}
}
\examples{
library(MASS)  # for eqscplot
data(topo, package = "MASS")
topo2 <- surf.ls(2, topo)
infl.topo2 <- trls.influence(topo2)
(cand <- as.data.frame(infl.topo2)[abs(infl.topo2$stresid) > 1.5, ])
cand.xy <- topo[as.integer(rownames(cand)), c("x", "y")]
trsurf <- trmat(topo2, 0, 6.5, 0, 6.5, 50)
eqscplot(trsurf, type = "n")
contour(trsurf, add = TRUE, col = "grey")
plot(topo2, add = TRUE, div = 3)
points(cand.xy, pch = 16, col = "orange")
text(cand.xy, labels = rownames(cand.xy), pos = 4, offset = 0.5)
}
\keyword{spatial}

\eof
% file spatial/trmat.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{trmat}
\alias{trmat}
\title{
Evaluate Trend Surface over a Grid
}
\description{
Evaluate trend surface over a grid.
}
\usage{
trmat(obj, xl, xu, yl, yu, n)
}
\arguments{
\item{obj}{
object returned by \code{surf.ls} or \code{surf.gls}
}
\item{xl}{
limits of the rectangle for grid
}
\item{xu}{
}
\item{yl}{
}
\item{yu}{
}
\item{n}{
use \code{n}  x  \code{n} grid within the rectangle
}}
\value{
list with components \code{x}, \code{y} and \code{z} suitable for \code{contour} and \code{image}.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{surf.ls}}, \code{\link{surf.gls}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.ls(2, topo)
trsurf <- trmat(topo.kr, 0, 6.5, 0, 6.5, 50)
}
\keyword{spatial}

\eof
% file spatial/variogram.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{variogram}
\alias{variogram}
\title{
Compute Spatial Variogram
}
\description{
Compute spatial (semi-)variogram of spatial data or residuals.
}
\usage{
variogram(krig, nint, plotit = TRUE, \dots)
}
\arguments{
\item{krig}{
trend-surface or kriging object with columns \code{x}, \code{y}, and \code{z}
}
\item{nint}{
number of bins used
}
\item{plotit}{
logical for plotting
}
\item{\dots}{
parameters for the plot
}}
\value{
  \code{x} and \code{y} coordinates of the variogram and \code{cnt},
  the number of pairs averaged per bin.
}
\section{Side Effects}{
Plots the variogram if \code{plotit = TRUE}
}
\details{
Divides range of data into \code{nint} bins, and computes the average squared
difference for pairs with separation in each bin.  Returns results for
bins with 6 or more pairs.
}
\references{
  Ripley, B. D. (1981) \emph{Spatial Statistics.} Wiley.
  
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{correlogram}}
}
\examples{
data(topo, package="MASS")
topo.kr <- surf.ls(2, topo)
variogram(topo.kr, 25)
}
\keyword{spatial}

\eof
