\keyword{ts}
\name{itsAccessor}
\alias{itsAccessor}
\alias{dates}
\alias{names}
\alias{core}
\alias{dates<-}
\alias{names<-}
\alias{core<-}
\alias{dates,its-method}
\alias{names,its-method}
\alias{core,its-method}
\alias{dates<-,its,POSIXt-method}
\alias{names<-,its,character-method}
\alias{core<-,its,matrix-method}
\title{Accessor, Assignment Functions for Irregular Time-Series Objects}
\description{
  Accessor & assignment functions for objects of class \code{"its"}.
}
\usage{
dates(x)
names(x)
core(x)
dates(x) <- 
names(x) <-
core(x) <- 
}

\arguments{
  \item{x}{an object of class \code{"its"}}
}

\details{
\code{dates} an accessor function, to access the time-stamps of the object.

\code{names} an accessor function, to access the (column) names of the object.

\code{core} an accessor function, to access the numeric data of the object.

\code{dates} an assignment function to assign the time-stamps of the object.

\code{names} an assignment function to assign the (column) names of the object.

\code{core} an assignment function to assign the numeric data of the object.
}

\value{

  For \code{dates}  a vector of class \code{"POSIXct"}.
  For \code{names}  a vector of mode \code{"character"}.
  For \code{core}  a matrix of mode\code{"numeric"}.
  For \code{dates<-, names<-, and core<-} an object of class \code{"its"}.
}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}},
  \code{\link{itsJoin}},
  \code{\link{itsTimes}},
  \code{\link{itsSubset}},
  \code{\link{itsFin}},
  \code{\link{itsDisp}},
  \code{\link{itsInfo}},
  \code{\link{itsCumdif}},
  \code{\link{itsArith}}
}

\examples{
x <- newIts(1:10)
dates(x)
names(x)
core(x)
dates(x) <- seq.POSIXt(from=Sys.time()+11*24*60*60,
            to=Sys.time()+20*24*60*60,
            by="DSTday")
names(x) <- "B"
core(x) <- as.matrix(11:20)
}

\eof
\keyword{ts}
\keyword{arith}
\name{itsArith}
\alias{itsArith}
\alias{Arith,its,its-method}
\alias{+}

\title{Arithmetic Methods for Irregular Time-Series Objects}
\description{
  Arithmetic methods for objects of class \code{"its"}.
}

\usage{
     x + y
     x - y
     x * y
     x / y
     x ^ y
     x %% y
     x %/% y
}

\arguments{
  \item{x,y}{an object of class \code{"its"}}
}
\details{

The arithmetic operators return numeric vectors containing the result of the 
element-by-element operations.  The arguments must be conformable (same dimensions) 
and their time-stamps must match for each row.

Note that the matrix multiplication operator, %*%, is inherited from matrix,
and the result of a matrix multiplication is a matrix, not an \code{"its"}.
The resulting matrix can be converted back to its using the function its() 
- see examples.
}

\value{

The returned object is an object of class \code{"its"}, conformable with the
inputs, and with identical time-stamps.

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
}

\examples{
b <- newIts(1:30,ncol=3)
b+b-2*b
b/b
b\%*\%diag(ncol(b))
its(b\%*\%diag(ncol(b))) 
}

\eof
\keyword{ts}
\name{itsCumdif}
\alias{itsCumdif}
\alias{cumsum}
\alias{diff}
\alias{cumsum,its-method}
\alias{diff,its-method}

\title{Cumulation, Differencing for Irregular Time-Series Objects}
\description{
  Cumulation, differencing for objects of class \code{"its"}.
}
\section{Usage}{

cumsum(x)
diff(x,lag=1)

}
\section{Arguments}{
\describe{
  \item{x}{an object of class \code{"its"}}
  \item{lag}{an integer number of lags}
  }
}

\details{

\code{cumsum} and \code{diff} perform the cumulative sum and difference of each
column in turn.  In \code{diff}, the first \code{lag} rows are discarded.

}

\value{

An object of class \code{"its"}

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}

\examples{
b <- newIts(1:60,ncol=3)
cumsum(b)
diff(b,lag=2)
}

\eof
\keyword{ts}
\name{itsDef}

\alias{its}
\alias{its-class}
\alias{is.its}
\alias{as.its}
\alias{its.format}
\alias{newIts}

\title{Irregularly Spaced Time-Series}

\description{
  The function \code{its} is used to create irregular time-series
  objects from user-supplied data and time-stamps.
  \code{newIts} is used to create semi-regular time-series
  objects from user-supplied data, and rule for generating time-stamps.
  \code{as.its} coerces an object to an irregularly spaced
  time-series. \code{is.its} tests whether an object is an irregularly
  spaced time series.
  }

\usage{
its(x,dates=as.POSIXct(x=strptime(dimnames(x)[[1]],format=its.format())),
names=dimnames(x)[[2]],format=its.format(),...)

newIts(x=NA,start=format(Sys.time(),format=its.format()),
end,ncol=1,by="DSTday",extract=FALSE,format=its.format(),...)

as.its(x,...)

is.its(object)

its.format(formatDefault=NULL)
}

\arguments{
  \item{dates}{a vector of class \code{"POSIXct"}
    representing the time-stamps of the irregular time-series
    object. The elements of the numeric vector are construed as the
    number of seconds since the beginning of 1970, see \code{\link{POSIXct}}.}
  \item{start, end}{POSIXct or character representation of the start or end time-stamp, 
    if character, then the format is as specified by the argument \code{format}}
  \item{ncol}{number of columns of synthetic sequence of dates}
  \item{by}{time increment for synthetic sequence of dates, see \code{\link{seq.POSIXt}}}
  \item{extract}{logical flag: if TRUE, a subset of the synthetic sequence of dates is extracted, 
        see \code{\link{extractIts}}}
  \item{x}{a numeric matrix representing the values of the
    irregular time-series object.  In the case of coercion in as.its, the first
    column is taken to be the time-stamps, in seconds since the beginning
    of 1970, see \code{\link{POSIXct}}.}
  \item{object} { }
  \item{names}{a vector of mode character}
  \item{format}{a formatting string, see \code{\link{format.POSIXct}}, defaults to 
                \code{its.format()}}
  \item{formatDefault}{a formatting string, see \code{\link{format.POSIXct}},
    defaults to \code{"\%Y-\%m-\%d"} if \code{formatDefault} is not specified.}
  \item{\dots}{further arguments passed to or from other methods:
    for \code{its} passed to \code{\link{format.POSIXct}}.;
    for \code{as.its} passed to \code{\link{its}}.;
    for \code{newIts} passed to \code{\link{extractIts}}.}
    }

\details{
 The function \code{its} is used to create irregular time-series objects,
 which have (S4) class "its". An object of class "its" is a matrix with rows indexed by a time-stamp
 of class "POSIXct". Unlike objects of class "ts", it can be used to
 represent irregularly spaced time-series.  The object consists of a
 matrix, with a single slot, the named POSIX vector named "dates".
 An object of class "its" inherits matrix arithmetic methods.  The matrix
 has dimnames: dimnames[[1]] is populated with a text representation of
 "dates", using a format which is defined by the function its.format. These
 dates are not used in computations - all computations use the 
 POSIX representation.  The dates are required to be in ascending order.
 
 When matrix multiplication is applied to an "its", the result is of class
 matrix.  It is possible to restore the "its" class (see examples) - its() 
 is in this sense idempotent i.e. its(mat)==its(its(mat)).  Note however that
 the dates will be taken from dimnames[[1]], so the accuracy of this
 operation depends on the format of the dates.

 \code{newIts} is a utility for creating a new "its" using a series of 'semi-regular' 
 time-stamps, such as weekday, weekly, monthend etc.  Conceptually the date sequence
 generation has two parts. The first part is the generation of a sequence using
 \code{\link{seq.POSIXt}} - the arguments from, to, and by are passed to this function.  The second 
 part (which is optional, and applies only if extract=TRUE) is an extraction, 
 performed by \code{extractIts}. See \code{\link{extractIts}} for details of the arguments,
 which are passed via '...' .

 The function \code{its.format} assigns a private variable and returns its value. The 
 value of this default format persists in the session until reset.  The purpose of the
 function is one of convenience: to access and/or assign the default text format for dates 
 in the "its" package, and hence reduce the need to define the format repeatedly in a session.

}

\value{
For \code{its}, \code{newIts} and \code{as.its}, an object of class \code{"its"}, inheriting
from matrix, with a single slot named \code{'dates'}, which is a vector of class \code{POSIXct}

For \code{is.its}, a logical representing the result of a test for class membership

For \code{its.format}, a text representation of dates formatting to be used in the "its" package,
see \code{\link{format.POSIXct}}
}

\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}},
  \code{\link{itsJoin}},
  \code{\link{itsTimes}},
  \code{\link{itsSubset}},
  \code{\link{itsFin}},
  \code{\link{itsDisp}},
  \code{\link{itsInfo}},
  \code{\link{itsCumdif}},
  \code{\link{itsArith}}
  }

\examples{
its.format("\%Y-\%m-\%d")    #defines text format of dates read from dimnames
mat <- structure(1:6,dim=c(2,3),dimnames=list(c("2003-01-01","2003-01-04"),letters[1:3]))
its(mat)
its.format("\%Y\%m\%d")    #defines text format of dates written to dimnames
times <- as.POSIXct(strptime(c("1999-12-31 01:00:00","2000-01-01 02:00:00"),format="\%Y-\%m-\%d \%X"))
its(mat,times)
its.format("\%Y-\%m-\%d \%X")
its(mat,times)
is.its(its(mat,times))
its.format("\%Y\%m\%d \%X")   #defines text format of dates written to dimnames
as.its(mat)
its.format("\%a \%d \%b \%Y")
newIts(start="2003-09-30",end="2005-05-05",format="\%Y-\%m-\%d",period="month",find="last",extract=TRUE,weekday=TRUE)
newIts(start=ISOdate(2003,12,24,0),end=ISOdate(2004,1,10),extract=TRUE,weekday=TRUE)
}

\eof
\keyword{ts}
\name{itsDisp}
\alias{itsDisp}
\alias{plot}
\alias{print}
\alias{plot,its,missing-method}
\alias{print,its-method}

\title{Display Methods for Irregular Time-Series Objects}
\description{
  Display methods for objects of class \code{"its"}.
}

\section{Usage}{
plot(x,y,colvec=1:ncol(x),type="l",ltypvec=1,lwdvec=1,
leg=FALSE,lab=FALSE,yrange,format,at,interp=c("linear","none"),...)
print(x,...)
}

\section{Arguments}{
\describe{
  \item{x}{an object of class \code{"its"}}
  \item{y}{missing}
  \item{type}{plot type, "l" for line, "p" for point, "b" for both - see \code{\link{par}}.}
  \item{colvec,ltypvec,lwdvec}{vectors of colour codes, line types, and line widths: see \code{\link{par}}.}
  \item{leg}{logical flag to display legend; uses \code{labcurve} in package \code{Hmisc} for auto-position.}
  \item{lab}{logical flag to display labels; uses \code{labcurve} in package \code{Hmisc} for auto-position.}
  \item{yrange}{ordinate range for display.}
  \item{format}{a formatting string, see \code{\link{format.POSIXct}}, for the axis dates}
  \item{at}{a vector of POSIX dates defining tickmark locations on axis: see \code{\link{axis.POSIXct}}}
  \item{interp}{a flag, indicating for plot of type line, how NA vallues are interpolated}
  \item{\dots}{further arguments passed to or from other methods:
    for \code{plot} passed to \code{\link{plot}}.;
    for \code{print} passed to \code{\link{print.matrix}}}
    }
}
\details{

\code{plot} is the method for plotting irregular time-series objects.
All series are displayed on a single set of axes, by default using different
colours, linetypes, and widths.  If the vectors defining these attributes are 
short, they are cycled.  \code{lab} and \code{leg} are alternatives for labelling 
the curves - these are located using function \code{labcurve()} from package 
Hmisc on CRAN.

\code{print} is the method for printing irregular time-series objects.
The format for the dates is determined at the time the 'its' object is created.

}

\value{

The methods are called for their side-effects.

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}},
  \code{\link{itsJoin}},
  \code{\link{itsTimes}},
  \code{\link{itsSubset}},
  \code{\link{itsFin}},
  \code{\link{itsDisp}},
  \code{\link{itsInfo}},
  \code{\link{itsCumdif}},
  \code{\link{itsArith}}
}

\examples{
b <- newIts(1:60,ncol=3)
plot(b,colvec=c(1,7),lwdvec=1:2,ltypvec=1:2,lab=TRUE)
print(b)
as.its(matrix(1:60,20,3))
}

\eof
\keyword{ts}
\name{itsFile}
\alias{itsFile}
\alias{readcsvIts}
\alias{writecsvIts}
\title{File Operations for Irregular Time-Series Objects}
\description{
  File read and write operations for objects of class \code{"its"}.
}
\usage{
readcsvIts(filename,informat=its.format(),outformat=its.format(),tz="",
usetz=FALSE,header=TRUE,...)
writecsvIts(x,filename,format=its.format(),tz="",usetz=FALSE,col.names=NA,
sep=",",...)
}
\arguments{
  \item{filename}{filename}
  \item{x}{an object of class \code{"its"}}
  \item{format, informat, outformat, tz, usetz}{formatting related arguments, see \code{\link{format.POSIXct}}.}
  \item{header}{see \code{\link{read.csv}}}
  \item{col.names, sep}{see \code{\link{write.table}}}
  \item{\dots}{further arguments passed to or from other methods: for
    \code{readcsvIts} passed to \code{\link{read.csv}}; for
    \code{writecsvIts} passed to \code{\link{write.table}}}
}
\details{
  \code{readcsvIts} reads from a .csv file to a matrix.  The first column is assumed to
  contain dates in text format specified by informat, which can optionally be
  reformatted into the text format outformat.  Both of these formats default to the
  format specified by \code{\link{its.format}}.  To convert the matrix to an its, use 
  \code{\link{its}} (see example)

  \code{writecsvIts} write an irregular time-series object to a text file.
}
\value{

  For \code{readcsvIts} a matrix

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}
\examples{
\dontrun{
b <- newIts(1:30,ncol=3)
fname <- tempfile()
# To write an irregular time-series object to a file one might use
writecsvIts(b,filename=fname)
# To read an irregular time-series object from a file one might use
its(readcsvIts(filename=fname))
unlink(fname)
}
}

\eof
\keyword{ts}
\name{itsFin}
\alias{itsFin}
\alias{accrueIts}
\title{Financial Functions for Irregular Time-Series Objects}
\description{
  Financial functions for objects of class \code{"its"}.
}
\usage{
accrueIts(x,daysperyear=365)
}
\arguments{
  \item{x}{an object of class \code{"its"}}
  \item{daysperyear}{integer, representing days per year for accrual}
}
\details{
  \code{accrueIts} Accrues an annual rate stored in x, expressed as a
  decimal (not a percentage), based on the difference of the time-stamps.
  The accrued value is not cumulated, it is a per-period return.
}
\value{

  An object of class \code{"its"}

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}
\examples{
a <- matrix(c(seq(by=24*60*60,length=20),1:20,41:60),nrow=20,ncol=3)
b <- as.its(a)
accrueIts(b[which(weekdayIts(b)),]/100,daysperyear=365)
}

\eof
\keyword{ts}
\name{itsInfo}
\alias{itsInfo}
\alias{start}
\alias{end}
\alias{summary}
\alias{start,its-method}
\alias{end,its-method}
\alias{summary,its-method}

\title{Summary Information for Irregular Time-Series Objects}
\description{
  Summary information for objects of class \code{"its"}.
}
\section{Usage}{

start(x,format=its.format(),...)
end(x,format=its.format(),...)
summary(object)

}
\section{Arguments}{
\describe{
  \item{x, object}{an object of class \code{"its"}}
  \item{format}{a formatting string, see \code{\link{format.POSIXct}}}
  \item{\dots}{further arguments passed to or from other methods:
    for \code{start} and \code{end}, passed to \code{\link{format.POSIXct}.} }
    }
}
\details{

Summary returns the same statistics as summary for a matrix, plus the number of
NA and non-NA data points for each series, and the standard deviation.

Start and end return the dates corresponding to the first and last rows,
respectively.

}

\value{

For \code{start}, \code{end}, a formatted text representation of the first and last times
For \code{summary}, a table of summary statistics for each series
}

\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}

\examples{
b <- newIts(1:30,ncol=3)
start(b,format="\%y-\%m-\%d")
end(b)
summary(b)
}

\eof
\keyword{ts}
\name{itsJoin}
\alias{itsJoin}
\alias{union}
\alias{union,its,NULL-method}
\alias{union,NULL,its-method}
\alias{union,its,its-method}
\alias{intersect}
\alias{intersect,its,NULL-method}
\alias{intersect,NULL,its-method}
\alias{intersect,its,its-method}
\alias{alignedIts}
\alias{appendIts}
\alias{collapseIts}
\title{Join Functions for Irregular Time-Series Objects}
\description{
  Join functions for objects of class \code{"its"}.
}
\usage{
union(x,y)
intersect(x,y)
alignedIts(obj1,obj2,print=FALSE)
appendIts(obj1,obj2,but=TRUE,matchnames=TRUE)
collapseIts(x)
}

\arguments{
  \item{x, y}{an object of class \code{"its"} or NULL}
  \item{obj1,obj2}{object of class \code{"its"}}
  \item{print}{logical flag to display summary information}
  \item{but}{logical flag controls whether overlap is disallowed}
  \item{matchnames}{logical flag controls whether names must match}
}

\details{

\code{union} has a number of time values (rows) which is determined by the union
of the time-stamps of the two inputs.  The number of columns is the sum of the
number of columns of the two inputs.

\code{intersect} has a number of time values (rows) which is determined by the
intersect of the time-stamps of the two inputs.  The number of columns is the sum
of the number of columns of the two inputs.

\code{alignedIts} selects the rows from two inputs which have identical time-stamps.

\code{appendIts} appends one object to the other, removing overlapping data from the
later object, optionally checking that the column names match.  Overlapping data
must match.

\code{collapseIts} checks for columns with the same names, using dimnames(x)[[2]].
columns with the same names are required to have the same numeric values in each row, but are
allowed NAs.  The numeric data is combined, and the resulting object has unique
column names - this will in general result in a reduction in the number of columns.

}

\value{
For \code{union}, \code{intersect}, \code{appendIts} an object of class \code{"its"}.

For \code{alignedIts}, a list of two objects of class \code{"its"}
}

\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}},
  \code{\link{itsJoin}},
  \code{\link{itsTimes}},
  \code{\link{itsSubset}},
  \code{\link{itsFin}},
  \code{\link{itsDisp}},
  \code{\link{itsInfo}},
  \code{\link{itsCumdif}},
  \code{\link{itsArith}}
}

\examples{
its.format("\%Y-\%m-\%d")
b <- newIts(1:30,ncol=3,start="2003-01-01",end="2003-01-10")
union(b,diff(b))
intersect(b,diff(b))
alignedIts(b,diff(b))
b1 <- newIts(1:30,ncol=3,start="2003-01-11",end="2003-01-20")
appendIts(b,b1)
c <- union(b,b*NA)
c[2,4] <- 99
c[2,1] <- NA
print(c)
collapseIts(c)
}

\eof
\keyword{ts}
\name{itsLags}
\alias{itsLags}
\alias{lagIts}
\alias{lagdistIts}
\title{Lag Functions for Irregular Time-Series Objects}
\description{
  Lag functions for objects of class \code{"its"}.
}
\usage{
lagIts(x,k=1)
lagdistIts(x,kmin,kmax)
}
\arguments{
  \item{x}{an object of class \code{"its"}}
  \item{k, kmin, kmax}{integer lag; positive value mean earlier data is
    assigned to a later timestamp}
}
\details{
  \code{lagIts} returns an object with the same time-stamps, but with the data shifted.
  \code{lagdistIts} applies lagIts over a range of lags, and appends these columns
}
\value{

  An object of class \code{"its"}.

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}

\examples{
b <- newIts(1:10)
lagIts(b)
lagdistIts(b[,1],1,3)
}

\eof
\keyword{ts}
\name{itsSubset}
\alias{itsSubset}
\alias{rangeIts}
\alias{extractIts}
\alias{x[i,j]}
\alias{[}
\alias{[,its-method}
\alias{[<-,its,its-method}
\title{Range and Extract for Irregular Time-Series Objects}
\description{
  Range and extract for objects of class \code{"its"}.
}
\usage{
rangeIts(x,start=dates(x)[1],end=dates(x)[n],format=its.format(),...)
extractIts(x,weekday=FALSE,find=c("all","last","first"),period=c("week","month"),partials=TRUE,select)
x[i,j,dates]
}
\arguments{
  \item{x}{an object of class \code{"its"}}
  \item{start, end}{POSIXct or character representation of the start or end time-stamp, 
    if character, then the format is as specified by the argument \code{format}}
  \item{format}{format of \code{"start"} and \code{"end"} dates, see
        \code{\link{format.POSIXct}}.}
  \item{i,j}{indices specifying the parts to be extracted from the irregular
        time-series object}
  \item{dates}{POSIX dates specifying the timestamps of rows to be extracted from the irregular
        time-series object}  
  \item{\dots}{further arguments passed to \code{\link{as.POSIXct}}}
  \item{weekday}{logical, defines whether only weekdays are to be returned}
  \item{find}{to find the first, last, or all samples within each period}
  \item{period}{the period within which 'find' and/or 'select' operate}
  \item{partials}{defines whether the first (possibly incomplete) period is processed 
                for find=first, and whether the last is processed for find=last}
  \item{select}{an integer vector defining one or more days to select.  The integer 
                specifies \code{wday} (for period=week) or \code{mday} (for period=month).  See 
                \code{\link{as.POSIXlt}} for details}
}
\details{
  \code{rangeIts} selects a range of rows that fall between two times, specified
  in text format.

  \code{extractIts} selects a subset of rows that obey some sort of semi-regular rule
  such as monthends, weekdays, and so on.  The order of application is \code{weekday}, \code{find}, 
  then \code{select}.

  \code{x[i,j,dates]} extractor method for an irregular time-series: \code{i} and \code{dates}
  are mutually exclusive alternatives for specifying rows.  It proceeds as for a matrix, with the 
  exception that drop=FALSE is enforced, so the result always inherits from matrix.

}
\value{

  An object of class \code{"its"}.

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}},
  \code{\link{itsJoin}},
  \code{\link{itsTimes}},
  \code{\link{itsSubset}},
  \code{\link{itsFin}},
  \code{\link{itsDisp}},
  \code{\link{itsInfo}},
  \code{\link{itsCumdif}},
  \code{\link{itsArith}}
}

\examples{
its.format("\%Y-\%m-\%d")
b <- newIts(start="2003-01-01",end="2003-01-20")
rangeIts(b,start = "2003-01-05" ,end= "2003-01-15")
rangeIts(b,start = ISOdate(2003,1,5,hour=0) ,end= ISOdate(2003,1,15,hour=0))
b[1:3,]
b[,1]
b[,dates=ISOdate(2003,1,1,hour=0)]
its.format("\%a \%d \%b \%Y")
c <- newIts()
extractIts(c,weekday=TRUE,period="month",find="last")  #the last weekdays of the month in c
}

\eof
\keyword{ts}
\name{itsTimes}
\alias{itsTimes}
\alias{daysecondIts}
\alias{weekdayIts}
\title{Time Functions for Irregular Time-Series Objects}
\description{
  Time functions for objects of class \code{"its"}.
}
\usage{
daysecondIts(x,...)
weekdayIts(x,...)
}

\arguments{
  \item{x}{an object of class \code{"its"}}
  \item{\dots}{further arguments passed to \code{\link{as.POSIXlt}}}
}

\details{
  \code{daysecondIts} returns the number of seconds since midnight.

  \code{weekdayIts} tests which entries of an  irregular time-series object are
  time-stamped with weekdays.

}

\value{

  For \code{daysecondIts} a vector of decimal numbers representing the number of seconds.

  For \code{weekdayIts}  a vector of \code{"logical"} representing the test results
  for each time.

}
\author{Giles Heywood}

\seealso{
  \code{\link{ts}},
  \code{\link{POSIXct}},
  \code{\link{itsFile}},
  \code{\link{itsLags}}
  \code{\link{itsJoin}}
  \code{\link{itsTimes}}
  \code{\link{itsSubset}}
  \code{\link{itsFin}}
  \code{\link{itsDisp}}
  \code{\link{itsInfo}}
  \code{\link{itsCumdif}}
  \code{\link{itsArith}}
}

\examples{
\dontrun{
its.format("\%Y-\%m-\%d \%A")
b <- newIts()
print(b)
daysecondIts(b)
weekdayIts(b)
its.format("\%Y-\%m-\%d")
}
}

\eof
