% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Fhat}
\alias{Fhat}
\title{
F nearest neighbour distribution function
}
\description{
Calculates an estimate of the F nearest neighbour distribution function
}
\usage{
Fhat(pts1,pts2,s)
}
\arguments{
\item{pts1}{
A points data set
}
\item{pts2}{
A points data set
}
\item{s}{
A vector of distances at which to evaluate Fhat
}}
\value{
A vector of the same length as \code{s}, containing the value of Fhat at the 
distances in \code{s}.


}
\details{
The function \code{Fhat(pts1,pts2,s)} is defined as the proportion of members of a
point set \code{pts2} for which the distance to the nearest member of
another points set \code{pts1} is less than or equal to \code{s}.
}
\examples{
data(uganda)
plot(seq(20, 500, 20), Fhat(as.points(uganda), 
as.points(csr(uganda$poly, length(uganda$x))), seq(20, 500, 20)), 
type="l", xlab="distance", ylab="Estimated F")
plot(Ghat(as.points(uganda), seq(20, 500, 20)), Fhat(as.points(uganda), 
as.points(csr(uganda$poly, length(uganda$x))), seq(20, 500, 20)), 
type="l", xlab="Estimated G", ylab="Estimated F")
lines(c(0,1),c(0,1),lty=2)
}
\seealso{
\code{\link{nndistF}},\code{\link{Fzero}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Fzero}
\alias{Fzero}
\title{
Theoretical nearest neighbour distribution function
}
\description{
Calculate the theoretical nearest neighbour distribution function.
}
\usage{
Fzero(density,s)
}
\arguments{
\item{density}{
The density of the point pattern, i.e. the number of points per unit area.
}
\item{s}{
A vector of distances at which to evaluate Fzero
}}
\value{
A vector of the same length as \code{s}, containing the value of Fzero at the 
distances in \code{s}.


}
\details{
Fzero returns the nearest neighbour distribution for a homogeneous planar
Poisson process. In fortran notation, Fzero(s) is
\code{FZERO = 1-EXP(-PI*DENSITY*(S**2))}.
}
\examples{
data(uganda)
plot(Ghat(as.points(uganda), seq(20, 500, 20)), Fzero(pdense(as.points(uganda), 
uganda$poly), seq(20, 500, 20)), type="l", ylab="Theoretical G", 
xlab="Estimated G")
lines(c(0,1),c(0,1),lty=2)
}
\seealso{
\code{\link{Fhat}},\code{\link{Ghat}},\code{\link{pdense}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Ghat}
\alias{Ghat}
\title{
G nearest neighbour distribution function
}
\description{
Calculates an estimate of the G nearest neighbour distribution function.
}
\usage{
Ghat(pts,s)
}
\arguments{
\item{pts}{
A points data set
}
\item{s}{
A vector of distances at which to evaluate the G function
}}
\value{
A vector of the same length as \code{s}, containing the estimate of G at the 
distances in \code{s}.


}
\details{
The function \code{Ghat(pts,s)} is defined as the proportion of members of a
point set for which the distance to the nearest other member of the set
is less than or equal to \code{s}.
}
\examples{
data(uganda)
plot(seq(20, 500, 20), Ghat(as.points(uganda), seq(20, 500, 20)), 
type="l", xlab="distance", ylab="Estimated G")
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}
\seealso{\code{\link{Fhat}}, \code{\link{nndistG}}}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Kenv.csr}
\alias{Kenv.csr}
\title{
Envelope of Khat from simulations of complete spatial randomness
}
\description{
Compute envelope of Khat from simulations of complete spatial randomness.
}
\usage{
Kenv.csr(nptg,poly,nsim,s,quiet=FALSE)
}
\arguments{
\item{nptg}{
Number of points to generate in each simulation.
}
\item{poly}{
Polygon in which to generate the points.
}
\item{nsim}{
Number of simulations to do.
}
\item{s}{
Vector of distances at which to calculate the envelope.
}
\item{quiet}{
If FALSE, print a message after every simulation for progress monitoring. If
TRUE, print no messages.
}}
\value{
A list with two components, called \code{$upper} and \code{$lower}. Each
component is a vector like \code{s}. The two components contain the upper 
and lower bound of the Khat envelope. 


}
\examples{
data(cardiff)
UL.khat <- Kenv.csr(length(cardiff$x), cardiff$poly, nsim=29, seq(2,30,2))
plot(seq(2,30,2), sqrt(khat(as.points(cardiff), cardiff$poly, 
seq(2,30,2))/pi)-seq(2,30,2), type="l", xlab="Splancs - polygon boundary", 
ylab="Estimated L", ylim=c(-1,1.5))
lines(seq(2,30,2), sqrt(UL.khat$upper/pi)-seq(2,30,2), lty=2)
lines(seq(2,30,2), sqrt(UL.khat$lower/pi)-seq(2,30,2), lty=2)
}
\seealso{
\code{\link{csr}}, \code{\link{khat}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Kenv.label}
\alias{Kenv.label}
\title{
Envelope of K1hat-K2hat from random labelling of two point patterns
}
\description{
Compute envelope of K1hat-K2hat from random labelling of two point patterns
}
\usage{
Kenv.label(pts1,pts2,poly,nsim,s,quiet=FALSE)
}
\arguments{
\item{pts1}{
First point data set.
}
\item{pts2}{
Second point data set.
}
\item{poly}{
Polygon containing the points.
}
\item{nsim}{
Number of random labellings to do.
}
\item{s}{
Vector of distances at which to calculate the envelope.
}
\item{quiet}{
If FALSE, print a message after every simulation for progress monitoring. If
TRUE, print no messages.
}}
\value{
A list with two components, called \code{$upper} and \code{$lower}. Each
component is a vector like \code{s}.
}
\details{
The two point data sets are randomly labelled using \code{rlabel}, then
\code{Khat} is called to estimate the K-function for each resulting set
at the distances in \code{s}. The difference between these two estimates
is then calculated. 
The maximum and minimum values of this difference at each distance,over
the \code{nlab}
labellings is returned.


}
\examples{
data(okwhite)
data(okblack)
okpoly <- list(x=c(okwhite$x, okblack$x), y=c(okwhite$y, okblack$y))
K1.hat <- khat(as.points(okwhite), bbox(as.points(okpoly)), seq(5,80,5))
K2.hat <- khat(as.points(okblack), bbox(as.points(okpoly)), seq(5,80,5))
K.diff <- K1.hat-K2.hat
plot(seq(5,80,5), K.diff, xlab="distance", ylab=expression(hat(K)[1]-hat(K)[2]), 
ylim=c(-11000,7000), type="l", main="Simulation envelopes, random labelling")
env.lab <- Kenv.label(as.points(okwhite), as.points(okblack), 
bbox(as.points(okpoly)), nsim=29, s=seq(5,80,5))
lines(seq(5,80,5), env.lab$upper, lty=2)
lines(seq(5,80,5), env.lab$lower, lty=2)
}
\seealso{
\code{\link{rlabel}},\code{i\link{khat}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Giovanni Petris and Roger S. Bivand 2001
%

\name{Kenv.pcp}
\alias{Kenv.pcp}
\title{Calculate simulation envelope for a Poisson Cluster Process}
\description{
  This function computes the envelope of Khat from simulations of a Poisson Cluster Process for a given polygon
}
\usage{
Kenv.pcp(rho, m, s2, region.poly, larger.region=NULL, nsim, r, vectorise.loop=TRUE)
}
\arguments{
  \item{rho}{intensity of the parent process}
  \item{m}{average number of offsprings per parent}
  \item{s2}{variance of location of offsprings relative to their parent}
  \item{region.poly}{a polygon defining the region in which the process is to be generated}
  \item{larger.region}{a rectangle containing the region of interest given in the form (xl,xu,yl,yu), defaults to \code{sbox()} around region.poly}
  \item{nsim}{number of simulations required}
  \item{r}{vector of distances at which the K function has to be estimated}
  \item{vectorise.loop}{if TRUE, use new vectorised code, if FALSE, use loop as before}
}
\value{
  \item{ave}{mean of simulations}
  \item{upper}{upper bound of envelope}
  \item{lower}{lower bound of envelope}
}
\references{Diggle, P. J. (1983) \emph{Statistical analysis of spatial point
patterns}, London: Academic Press, pp. 55-57 and 78-81; Bailey, T. C. and
Gatrell, A. C. (1995) \emph{Interactive spatial data analysis}, Harlow:
Longman, pp. 106-109.}
\author{Giovanni Petris <GPetris@uark.edu>,  Roger.Bivand@nhh.no}

\seealso{\code{\link{pcp}}, \code{\link{pcp.sim}}, \code{\link{khat}}}

\examples{
data(cardiff)
polymap(cardiff$poly)
pointmap(as.points(cardiff), add=TRUE)
title("Locations of homes of 168 juvenile offenders")
pcp.fit <- pcp(as.points(cardiff), cardiff$poly, h0=30, n.int=30)
pcp.fit
m <- npts(as.points(cardiff))/(areapl(cardiff$poly)*pcp.fit$par[2])
r <- seq(2,30,by=2)
K.env <- Kenv.pcp(pcp.fit$par[2], m, pcp.fit$par[1], cardiff$poly,
           nsim=20, r=r)
L.env <- lapply(K.env, FUN=function(x) sqrt(x/pi)-r)
limits <- range(unlist(L.env))
plot(r, sqrt(khat(as.points(cardiff),cardiff$poly,r)/pi)-r, ylim=limits,
     main="L function with simulation envelopes and average", type="l",
     xlab="distance", ylab="")
lines(r, L.env$lower, lty=5)
lines(r, L.env$upper, lty=5)
lines(r, L.env$ave, lty=6)
abline(h=0)
}
\keyword{spatial}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{Kenv.tor}
\alias{Kenv.tor}
\title{
Envelope of K12hat from random toroidal shifts of two point patterns
}
\description{
Compute envelope of K12hat from random toroidal shifts of two point patterns.
}
\usage{
Kenv.tor(pts1,pts2,poly,nsim,s,quiet=FALSE)
}
\arguments{
\item{pts1}{
First point data set.
}
\item{pts2}{
Second point data set.
}
\item{poly}{
Polygon containing the points.
}
\item{nsim}{
Number of random toroidal shifts to do.
}
\item{s}{
Vector of distances at which to calculate the envelope.
}
\item{quiet}{
If FALSE, print a message after every simulation for progress monitoring. If
true, print no messages.
}}
\value{
A list with two components, called \code{$upper} and \code{$lower}. Each
component is a vector like \code{s}.
}
\details{
The second point data set is randomly shifted using \code{rtor.shift}
in the rectangle defined by \code{poly}. Then \code{k12hat} is called 
to compute K12hat for the two patterns. 
The upper and lower values of K12hat over the \code{ntor}
toroidal shifts are returned.


}
\examples{
data(okwhite)
data(okblack)
okpoly <- list(x=c(okwhite$x, okblack$x), y=c(okwhite$y, okblack$y))
plot(seq(5,80,5), sqrt(k12hat(as.points(okwhite), as.points(okblack), 
bbox(as.points(okpoly)), seq(5,80,5))/pi) - seq(5,80,5), xlab="distance", 
ylab=expression(hat(L)[12]), ylim=c(-35,35), type="l",
main="Simulation envelopes, random toroidal shifts")
env.ok <- Kenv.tor(as.points(okwhite), as.points(okblack), 
bbox(as.points(okpoly)), nsim=29, s=seq(5,80,5))
lines(seq(5,80,5), sqrt(env.ok$upper/pi)-seq(5,80,5), lty=2)
lines(seq(5,80,5), sqrt(env.ok$lower/pi)-seq(5,80,5), lty=2)
}
\seealso{
\code{\link{rtor.shift}},\code{\link{k12hat}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Based on Kenv.tor.Rd, Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 2003 by Roger S. Bivand, 
% examples copyright 2003 Stephen Eglen <stephen@inf.ed.ac.uk>
\name{Kenv.tor1}
\alias{Kenv.tor1}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Modified envelope of K12hat from random toroidal shifts of two point patterns}
\description{
  Modification of \code{Kenv.tor()} to allow the assignment of a p value to the
goodness of fit, following the method outlined in Peter Diggle's 1986
paper (J Neurosci methods 18:115-125) and in his 2002 book.
}
\usage{
Kenv.tor1(pts1, pts2, poly, nsim, s, quiet = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{pts1}{First point data set}
  \item{pts2}{Second point data set}
  \item{poly}{Polygon containing the points}
  \item{nsim}{Number of random toroidal shifts to do}
  \item{s}{Vector of distances at which to calculate the envelope}
  \item{quiet}{If FALSE, print a message after every simulation for progress monitoring. If TRUE, print no messages}
}

\value{
A list with components: \code{$upper}, \code{$lower}, \code{real}, \code{u}, \code{ksim}, and \code{rank}. The first three
components are vectors like \code{s}, the next two contain results passed back from the simulations, and the final is a one-element vector with the rank of the observed data set.
}

\author{Stephen Eglen <stephen@inf.ed.ac.uk>}

\seealso{\code{\link{Kenv.tor}}}
\examples{
data(amacrines)
ama.a <- rbind(amacrines.on, amacrines.off)
ama.bb <- bbox(as.points(ama.a))
ama.t <- seq(from = 0.002, to=.250, by=0.002)
nsim=999
plot(amacrines.on, asp=1, pch=19,
 main="Data set, match figure 1.4 of Diggle(2002)?")
points(amacrines.off, pch=1)
#
k12 <- k12hat(amacrines.on, amacrines.off, ama.bb, ama.t)
#
k11 <- khat(amacrines.on, ama.bb, ama.t)
k22 <- khat(amacrines.off, ama.bb, ama.t)
k00 <- khat(ama.a, ama.bb, ama.t)
theor <- pi * (ama.t^2)
#
plot(ama.t, k12-theor, ylim=c(min( c(k12, k11, k22, k00) - theor),
 max( c(k12, k11, k22, k00) - theor)),
 main="2nd order properties, match figure 4.8 of Diggle (2002)", type="l")
lines(ama.t, -theor)
lines(ama.t, k11-theor, lty=2)
lines(ama.t, k22-theor, lty=3)
lines(ama.t, k00-theor, lty=5)
#
k12.tor <- Kenv.tor(amacrines.on, amacrines.off, ama.bb,
 nsim, ama.t, quiet=TRUE)
plot(ama.t, k12-theor, type="l", main="Output from Kenv.tor")
lines(ama.t, k12.tor$upper-theor, type="l", col="red")
lines(ama.t, k12.tor$lower-theor, type="l", col="red")
#
k12.sims <- Kenv.tor1(amacrines.on, amacrines.off, ama.bb,
 nsim, ama.t, quiet=TRUE)
plot(ama.t, sqrt(k12.sims$real/pi), type="l", asp=1, bty="n",
 main=paste("K12 versus toroidal sims; rank ", k12.sims$rank, "of",
 length(k12.sims$u)))
lines(ama.t, sqrt(k12.sims$upper/pi), col="red")
lines(ama.t, sqrt(k12.sims$lower/pi), col="red")
}
\keyword{spatial}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{addpoints}
\alias{addpoints}
\title{
Add points interactively to a point data set
}
\description{
Add points interactively to a point data set.
}
\usage{
addpoints(pts,plot=FALSE,quiet=FALSE)
}
\arguments{
\item{pts}{
A points data set.
}
\item{plot}{
if true, plot the \code{pts} data, using \code{pointmap}. If false,
 or if \code{pts} is missing,
don't plot the data. 
}
\item{quiet}{
if true, don't print a prompt to enter points.
}}
\value{
A points data set consisting of \code{pts} and the points entered on the current 
graphics device.


}
\details{
The points entered are displayed on the current graphics device.
}
\seealso{
\code{\link{delpoints}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright 2003 by Roger S. Bivand

\name{amacrines}
\title{Amacrines on/off data set}
\usage{data(amacrines)}
\alias{amacrines.on}
\alias{amacrines.off}
\format{Two two-column matrices of points marked on and off}
\source{\url{http://www.maths.lancs.ac.uk/~diggle/pointpatterns/Datasets/}, Peter J. Diggle, Department of Mathematics and Statistics, Lancaster University, Lancaster LA1 4YF, UK: public-domain spatial point pattern data-sets. 
}
\description{Two two-column matrices of points marked on and off}
\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{areapl}
\alias{areapl}
\title{
Calculate area of polygon
}
\description{
Calculate area of polygon
}
\usage{
areapl(poly)
}
\arguments{
\item{poly}{
a polygon data set
}}
\value{
The area of the polygon is returned


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{as.points}
\alias{as.points}
\title{
Creates data in spatial point format
}
\description{
Creates data in spatial point format.
}
\usage{
as.points(...)
}
\arguments{
\item{...}{
any object(s), such as x and y vectors of the same length, or a list or data
frame containing x and y vectors. Valid options for \code{...} are:
a points object ; returns it unaltered;
a list with x and y elements of the same length --- returns a
points object with the x and y elements as the coordinates of the points;
two vectors of equal length ; returns a points object with the 
first vector as the x coordinates, the second vector as the y-coordinates.
}
}

\value{
\code{as.points} tries to return the argument(s) as a points object. }

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{bbox}
\alias{bbox}
\title{
Generate a bounding box round a point object
}
\description{
Generate a bounding box round a point object
}
\usage{
bbox(pts)
}
\arguments{
\item{pts}{
A points data set
}}
\value{
A points data set of four points giving the coordinates of the bounding box


}
\seealso{
\code{\link{sbox}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{bodmin}
\title{Bodmin Moors granite tors}
\usage{data(bodmin)}
\alias{bodmin}
\format{A list corresponding to a Venables and Ripley point object with 35 observations
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    area \tab	list	\tab bounding box with xl, xu, yl, yu\cr
    poly \tab array \tab polygon boundary with columns x and y\cr
  }
}
\source{Pinder and Witherick, 1977 - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of 35 granite tors on Bodmin Moor, taken from Infomap data set (northings multiplied by -1 to correspond to Figure 3.2, p. 82, Bailey and Gatrell.
}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand
 
\name{burkitt}
\title{Burkitt's lymphoma in Uganda}
\usage{data(burkitt)}
\alias{burkitt}
\alias{burbdy}
\alias{burpts}
\format{The data is provided as a data table:
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    t \tab	numeric	\tab day number starting at 1/1/1960 of onset\cr
    age \tab numeric \tab age of child patient\cr
  }
  as a points object \code{burpts} of \code{burkitt$x} and \code{burkitt$y}; and a point object of the area boundary \code{burbdy}.
}
\source{Williams, E. H. et al. 1978, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of cases of Burkitt's lymphoma in the Western Nile
district of Uganda 1960-1975. The time variable is recorded as the number of
days starting from an origin of 1 Jan 1960. The examples given below show
how the \code{chron()} function and derived time structures may be used to
analyse the data in the time dimension.

}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\examples{
data(burkitt)
try.obj <- try(library(chron))
if (!is.na(match("chron", try.obj))) {
burdates <- chron(burkitt$t, out.format="y-m-d",
	origin=c(day=1, month=1, year=1960))
froms <- seq.dates(chron("7/1/1960", origin=c(1,1,1960)),
	chron("7/1/1975", origin=c(1,1,1960)), "years")
tos <- seq.dates(chron("9/30/1960", origin=c(1,1,1960)),
	chron("9/30/1975", origin=c(1,1,1960)), "years")
burd <- density(burdates, bw=100)
plot(chron(burd$x, origin=c(day=1, month=1, year=1960)), burd$y,
	type="n", xaxp=c(chron("1/1/1960", origin=c(1,1,1960)),
	chron("1/1/1976", origin=c(1,1,1960)),16), xlab="time",
ylab="density")
for (i in 1:length(tos)) polygon(c(froms[i], tos[i], tos[i], froms[i]),
	c(par("usr")[3], par("usr")[3], par("usr")[4], par("usr")[4]), 
	col="grey", border=NULL)
lines(chron(burd$x, origin=c(day=1, month=1, year=1960)), burd$y,
	lwd=3, col="red")
title("Density plot of Burkitt's lymphoma in West Nile district, 3Q grey shaded")
op <- par(mfrow=c(3,5))
for (i in 1961:1975) {
	polymap(burbdy)
	pointmap(burpts[which(years(burdates) == i),], add=TRUE, pch=19)
	title(i)
}
par(op)
op <- par(mfrow=c(2,2))
for (i in 1:4) {
	polymap(burbdy)
	pointmap(burpts[which(unclass(quarters(burdates)) == i),], add=TRUE,
pch=19)
	title(paste(i, "Q", sep=""))
}
par(op)
op <- par(mfrow=c(3,4))
for (i in 1:12) {
	polymap(burbdy)
	pointmap(burpts[which(unclass(months(burdates)) == i),], add=TRUE, pch=19)
	title(levels(months(burdates, FALSE))[i])
}
par(op)
}

}

\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{cardiff}
\title{Locations of homes of juvenile offenders}
\usage{data(cardiff)}
\alias{cardiff}
\format{A list corresponding to a Venables and Ripley point object with 168 observations
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    area \tab	list	\tab bounding box with xl, xu, yl, yu\cr
    poly \tab array \tab polygon boundary with columns x and y\cr
  }
}
\source{Herbert, 1980, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of homes of 168 juvenile offenders on a Cardiff housing estate

}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{csr}
\alias{csr}
\title{
Generate completely spatially random points on a polygon
}
\description{
Generate completely spatially random points on a polygon.
}
\usage{
csr(poly,npoints)
}
\arguments{
\item{poly}{
A polygon data set.
}
\item{npoints}{
The number of points to generate.
}}
\value{
 A point data set consisting of \code{npoints} points distributed randomly,
i.e. as an independent random sample from the uniform distribution
in the polygon defined by \code{poly}.


}
\details{
\code{csr} generates points randomly in the bounding box of \code{poly,} then uses 
\code{pip} to extract those in the polygon. If the number of points remaining is
less than that required, \code{csr} generates some more points in the
bounding box until at least \code{npoints} remain inside the polygon. If too many
points are generated then the list of points is truncated.

Uses \code{runif()} to generate random numbers and so updates \code{.Random.seed},
the standard S random number generator seed.
}
\examples{
data(cardiff)
nsim <- 29
emp.Ghat <- Ghat(as.points(cardiff), seq(0,30,1))
av.Ghat <- numeric(length(emp.Ghat))
U.Ghat <- numeric(length(emp.Ghat))
L.Ghat <- numeric(length(emp.Ghat))
U.Ghat <- -99999
L.Ghat <- 99999
for(i in 1:nsim) {
S.Ghat <- Ghat(csr(cardiff$poly, length(cardiff$x)), seq(0,30,1))
av.Ghat <- av.Ghat + S.Ghat
L.Ghat <- pmin(S.Ghat, L.Ghat)
U.Ghat <- pmax(S.Ghat, U.Ghat)
}
av.Ghat <- av.Ghat/nsim
plot(av.Ghat, emp.Ghat, type="l", xlim=c(0,1), ylim=c(0,1), 
xlab="Simulated average G", ylab="Empirical G")
lines(c(0,1),c(0,1),lty=2)
lines(U.Ghat,emp.Ghat,lty=3)
lines(L.Ghat,emp.Ghat,lty=3)
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{delpoints}
\alias{delpoints}
\title{
Select points to delete from a points data set
}
\description{
Select points to delete from a points data set.
}
\usage{
delpoints(pts,add=FALSE)
}
\arguments{
\item{pts}{
a points data set
}
\item{add}{
if false, plot the points using \code{pointmap}.
}}
\value{
A points object containing the undeleted points.


}
\details{
Using the mouse, the user selects points on the current graphics device.
These points are marked on the plot as they are selected. The function
returns the remaining points as a points object.
If \code{add} is false the points are plotted on the current plot device.
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{dsquare}
\alias{dsquare}
\title{
Distance-squared from a number of points to a number of sources
}
\description{
Computes the distance-squared from a number of points to a number of sources.
}
\usage{
dsquare(pts, srcs, namepref="d")
}
\arguments{
\item{pts}{
A number of points representing the locations of cases and controls.
}
\item{srcs}{
A number of points representing source locations
}
\item{namepref}{
A prefix given to the name of the results.
}}
\value{
A data frame with the same number of  columns as \code{srcs}. The column names will 
be the value of \code{namepref} prefixing the numbers from 1 to the number
of sources.
}
\seealso{
\code{\link{tribble}}, \code{\link{triblik}}


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{gen}
\alias{gen}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{generate points in polygon}
\description{
  generates random points within a defined polygon, trying to reach \code{npoints}
points - used in \code{csr}.
}
\usage{
gen(poly, npoints)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{poly}{A polygon data set}
  \item{npoints}{The number of points to generate}
}

\value{
  returns a point object.
}
\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\keyword{spatial}

\seealso{\code{\link{csr}} }



\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{getpoly}
\alias{getpoly}
\title{
Draw a polygon on the current graphics device
}
\description{
Draw a polygon on the current graphics device
}
\usage{
getpoly(quiet=FALSE)
}
\arguments{
\item{quiet}{
if TRUE, don't prompt for input of a polygon.
}}
\value{
A polygon data set consisting of the points entered. The current coordinate 
system is used.


}
\details{
The system prompts the user to enter points on the current graphics
device using the mouse or other pointing device. The points are joined
on the screen with the current line symbol.
A polygon of the points entered is drawn on the current graphics device.
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{gridpts}
\alias{gridpts}
\title{
Generate a grid of points
}
\description{
Generate a grid of points
}
\usage{
gridpts(poly,npts,xs,ys)
}
\arguments{
\item{poly}{
polygon in which to generate the points
}
\item{npts}{
approximate number of points to generate
}
\item{xs,ys}{
grid spacing in x and y 


Either \code{npts} or \code{xs} and \code{ys} must be specified. If all three
are given then \code{xs} and \code{ys} are ignored. 
}}
\value{
A points object containing a grid of points inside the polygon. If \code{npts}
is specified, then a grid spacing \code{xs} and \code{ys} will be calculated to give
approximately \code{npts} in the polygon. If \code{xs} and \code{ys} are given then these
will be used to generate a number of points in the polygon.


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2001 by Roger S. Bivand

\name{inout}
\alias{inout}
\title{
Test points for inclusion in a polygon
}
\description{
Test points for inclusion in a polygon.
}
\usage{
inout(pts,poly,bound=NULL,quiet=TRUE)
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon data set
}
\item{bound}{If points fall exactly on polygon boundaries, the default NULL gives arbitrary assignments. If TRUE, then all points "on" boundaries are set as within the polygon, if FALSE, outside.}
\item{quiet}{Do not report which points are on boundary for non-NULL bound}
}
\value{
A vector of logical values. \code{TRUE} means the point was inside the
polygon, \code{FALSE} means the point was outside. Note that "inside" is an arbitrary concept for points "on" the polygon boundary.



}
\seealso{
\code{\link{inpip}},\code{\link{pip}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\examples{
data(uganda)
suganda <- sbox(uganda$poly)
ruganda <- csr(suganda, 1000)
polymap(suganda)
polymap(uganda$poly, add=TRUE)
def <- inout(ruganda, uganda$poly, bound=NULL)
pointmap(as.points(ruganda[def,1], ruganda[def,2]), add=TRUE, col="black")
pointmap(as.points(ruganda[!def,1], ruganda[!def,2]), add=TRUE, col="red")
tru <- inout(ruganda, uganda$poly, bound=TRUE, quiet=FALSE)
which(tru & !def)
ds1 <- as.points(expand.grid(x=seq(-1.5,1.5,0.5), y=seq(-1.5,1.5,0.5)))
ds1.poly <- ds1[chull(ds1),]
ds2 <- as.points(rnorm(300),rnorm(300))
plot(ds2, type="n", asp=1)
polymap(ds1.poly, add=TRUE, border="lightblue", col="lightblue", lwd=1)
points(ds2[inout(ds2,ds1.poly),],  col="green",  pch=20)
points(ds2[!inout(ds2,ds1.poly),], col="orange", pch=20)
points(ds1[inout(ds1,ds1.poly),],  col="black",  pch=20)
points(ds1[!inout(ds1,ds1.poly),], col="red",    pch=20)
plot(ds2, type="n", asp=1)
polymap(ds1.poly, add=TRUE, border="lightblue", col="lightblue", lwd=1)
points(ds2[inout(ds2,ds1.poly,bound=TRUE),],  col="green",  pch=20)
points(ds2[!inout(ds2,ds1.poly,bound=TRUE),], col="orange", pch=20)
points(ds1[inout(ds1,ds1.poly,bound=TRUE),],  col="black",  pch=20)
points(ds1[!inout(ds1,ds1.poly,bound=TRUE),], col="red",    pch=20)
plot(ds2, type="n", asp=1)
polymap(ds1.poly, add=TRUE, border="lightblue", col="lightblue", lwd=1)
points(ds2[inout(ds2,ds1.poly,bound=FALSE),],  col="green",  pch=20)
points(ds2[!inout(ds2,ds1.poly,bound=FALSE),], col="orange", pch=20)
points(ds1[inout(ds1,ds1.poly,bound=FALSE),],  col="black",  pch=20)
points(ds1[!inout(ds1,ds1.poly,bound=FALSE),], col="red",    pch=20)
}

\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2001 by Roger S. Bivand

\name{inpip}
\alias{inpip}
\title{
Select points inside a polygon
}
\description{
Select points inside a polygon
}
\usage{
inpip(pts,poly,bound=NULL,quiet=TRUE)
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon data set
}
\item{bound}{If points fall exactly on polygon boundaries, the default NULL gives arbitrary assignments. If TRUE, then all points "on" boundaries are set as within the polygon, if FALSE, outside.}
\item{quiet}{Do not report which points are on boundary for non-NULL bound}
}
\value{
\code{inpip} returns a vector of indices of the points in \code{pts} that are located
in the polygon. Note that "in" is an arbitrary concept for points "on" the polygon boundary.


}
\seealso{
\code{\link{pip}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{is.points}
\alias{is.points}
\title{
Point Objects
}
\description{
Tests for data in spatial point format.
}
\usage{
is.points(p)
}
\arguments{
\item{p}{
any object.
}
}
\value{
\code{is.points} returns TRUE if \code{p} is a points object, FALSE otherwise.
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{k12hat}
\alias{k12hat}
\title{
Bivariate K-function
}
\description{
Calculates an estimate of the bivariate K-function
}
\usage{
k12hat(pts1,pts2,poly,s)
}
\arguments{
\item{pts1,pts2}{
Two points data sets
}
\item{poly}{
A polygon containing the points
}
\item{s}{
A vector of distances at which to estimate the K12 function
}}
\value{
A vector like \code{s} containing the value of K12hat at the points in \code{s}.


}
\details{
The bivariate K function is defined as the expected number of points of
pattern 1
within a distance s of an arbitrary point of pattern 2, divided by the
overall density of the points in pattern 1.  To estimate this function,
the approximately unbiased estimator given by Lotwick and Silverman
(1982) is used.
}
\examples{
data(okwhite)
data(okblack)
okpoly <- list(x=c(okwhite$x, okblack$x), y=c(okwhite$y, okblack$y))
plot(seq(5,80,5), sqrt(k12hat(as.points(okwhite), as.points(okblack), 
bbox(as.points(okpoly)), seq(5,80,5))/pi) - seq(5,80,5), xlab="distance", 
ylab=expression(hat(L)[12]), ylim=c(-20,20), type="l")
}
\references{
Lotwick, H.W. and Silverman B.W. (1982) Methods for Analysing Spatial
Processes of Several types of Points.
\emph{J. R. Statist Soc} B44 406-13;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{kernel2d}
\alias{kernel2d}
\title{
Kernel smoothing of a point pattern
}
\description{
Perform kernel smoothing of a point pattern
}
\usage{
kernel2d(pts,poly,h0,nx=20,ny=20,kernel='quartic')
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon data set
}
\item{h0}{
The kernel width parameter
}
\item{nx}{
Number of points along the x-axis of the returned grid.
}
\item{ny}{
Number of points along the y-axis of the returned grid.
}
\item{kernel}{
Type of kernel function to use. Currently only the quartic kernel is implemented.
}}
\value{
A list with the following components:
\item{x}{List of x-coordinates at which the kernel function has been evaluated.}
\item{y}{List of y-coordinates at which the kernel function has been evaluated.}
\item{z}{A matrix of dimension \code{nx} by \code{ny} containing the value of
the kernel function.}
\item{h0, kernel}{containing the values input to \code{kernel2d}}
}
\details{
The kernel estimate, with a correction for edge effects, is computed for
a grid of points that span the
input polygon. The kernel function for points in the grid that are outside the polygon are returned
as NA's.
The output list is in a format that can be read into \code{image()} directly,
for display and superposition onto other plots.
}

\examples{
data(bodmin)
plot(bodmin$poly, asp=1, type="n")
image(kernel2d(as.points(bodmin), bodmin$poly, h0=2, nx=100, ny=100), 
add=TRUE, col=terrain.colors(20))
pointmap(as.points(bodmin), add=TRUE)
polymap(bodmin$poly, add=TRUE)
}

\references{
Berman M. and Diggle P.J. (1989) Estimating Weighted Integrals of the
Second-Order Intensity of Spatial Point Patterns.  \emph{J. R. Statist
Soc} B51 81-92; Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point
pattern analysis code in S-Plus.  Computers and Geosciences, 19, 627-655,
(Barry Rowlingson ); the original sources
can be accessed at: \url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See
also Bivand, R. and Gebhardt, A. 2000 Implementing functions for spatial
statistical analysis using the R language. Journal of Geographical
Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{kernel3d}
\alias{kernel3d}
\title{
Space-time kernel
}
\description{
Compute the space-time kernel
}
\usage{
kernel3d(pts, times, xgr, ygr, zgr, hxy, hz)
}
\arguments{
\item{pts}{
A matrix of event coodinates x,y.
}
\item{times}{
A vector of event times, t.
}
\item{xgr}{
The values of x at which to compute the kernel function.
}
\item{ygr}{
The values of y at which to compute the kernel function.
}
\item{zgr}{
The values of time at which to compute the kernel function.
}
\item{hxy}{
The quartic kernel width in the x and y direction.
}
\item{hz}{
The quartic kernel width in the temporal direction.
}}
\value{
A list is returned. Most of the components are just copies of the 
input parameters, except for the \code{$v} parameter.
This is a three dimensional array containing the kernel-smoothed
values. Its dimension is
\code{[length(xgr),length(ygr),length(tgr)]}. 
}
\seealso{
\code{\link{kerview}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\examples{
data(burkitt)
b3d <- kernel3d(burpts, burkitt$t, seq(250,350,10), seq(250, 400, 10),
  seq(365,5800,365), 30, 200)
brks <- quantile(b3d$v, seq(0,1,0.05))
cols <- heat.colors(length(brks)-1)
oldpar <- par(mfrow=c(3,5))
for (i in 1:15) image(seq(250,350,10), seq(250, 400, 10), b3d$v[,,i],
  asp=1, xlab="", ylab="", main=1960+i, breaks=brks, col=cols)
par(oldpar)
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{kernrat}
\alias{kernrat}
\title{
Ratio of two kernel smoothings
}
\description{
Return the ratio of two kernel smoothings
}
\usage{
kernrat(pts1,pts2,poly,h1,h2,nx=20,ny=20,kernel='quartic')
}
\arguments{
\item{pts1,pts2}{
Point data sets
}
\item{poly}{
A polygon data set
}
\item{h1,h2}{
The kernel width parameters, \code{h1} for \code{pts1}, and \code{h2} for \code{pts2}
}
\item{nx}{
Number of points along the x-axis of the returned grid.
}
\item{ny}{
Number of points along the y-axis of the returned grid.
}
\item{kernel}{
Type of kernel function to use. Currently only the quartic kernel is implemented.
}}
\value{
A list with the following components:
\item{x}{List of x-coordinates at which the kernel function has been evaluated.}
\item{y}{List of y-coordinates at which the kernel function has been evaluated.}
\item{z}{A matrix of dimension \code{nx} by \code{ny} containing the ratio of the kernel functions.}
\item{h}{A vector of length 2 containing \code{h1} and \code{h2}}
\item{kernel}{a character string containing the kernel name.}


}
\seealso{
\code{\link{kernel2d}}, \code{\link{mse2d}}
}
\references{
Berman M. and Diggle P.J. (1989) Estimating Weighted Integrals of the
Second-Order Intensity of Spatial Point Patterns.
\emph{J. R. Statist Soc} B51 81-92;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{kerview}
\alias{kerview}
\title{
A linked-window system for browsing space-time data
}
\description{
A linked-window system for browsing space-time data.
}
\usage{
kerview(pts, times, k3, map=TRUE, addimg=TRUE, ncol=1)
}
\arguments{
\item{pts}{
A matrix of event x,y coordinates.
}
\item{times}{
A vector of event times.
}
\item{k3}{
An object returned from \code{kernel3d}, the space-time kernel smoothing
function
}
\item{map}{
If false, don't plot the map display.
}
\item{addimg}{
If true, overwrite successive images in the image display, else
make a fresh image plot each time.
}
\item{ncol}{
Number of columns and rows for multiple images and maps.
}}
\details{
This function displays three linked views of the data. In the current
graphics window a temporal slice from the kernel smoothing is displayed. 
Another graphics device is started to display a map of the data that
contributed to that time-slice. A third graphics device shows a 
histogram of the times of the events. Clicking with the mouse
in this window with button 1 sets the time for the other
displays to the time on the x-axis of the histogram at the clicked point.


In this way the 3-dimensional kernel smoothed function can be browsed, 
and the corresponding map of the data compared.


}
\seealso{
\code{\link{kernel3d}}


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2002 by Roger S. Bivand

\name{khat}
\alias{khat}
\alias{print.khat}
\alias{plot.khat}
\title{
K-function
}
\description{
Calculates an estimate of the K-function
}
\usage{
khat(pts,poly,s,newstyle=FALSE)
print.khat(x, ...)
plot.khat(x, ...)
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon containing the points
}
\item{s}{
A vector of distances at which to calculate the K function
}
\item{newstyle}{if TRUE, the function returns a khat object}
\item{x}{a \code{khat} object}
\item{...}{other arguments passed to plot and print functions}
}
\value{
If \code{newstyle} is FALSE, 
a vector like \code{s} containing the value of K at the points in \code{s}.
else a \code{khat} object list with:
\item{khat}{the value of K at the points in \code{s}}
\item{counts}{integer matrix of counts of points within the vector of
 distances \code{s} for each point}
\item{khats}{matrix of values of K within the vector of distances \code{s}
 for each point}
\item{s}{\code{s}}
}
\details{
The K function is defined as the expected number of further points
within a distance s of an arbitrary point, divided by the
overall density of the points. 
In practice an edge-correction is required to avoid biasing the
estimation due to non-recording of points outside the polygon.

The newstyle argument and khat object were introduced in collaboration 
with Thomas de Cornulier <oedic@cebc.cnrs.fr> to permit the mapping of 
counts or khats for chosen distance values, as in 
\url{ftp://pbil.univ-lyon1.fr/pub/mac/ADE/ADE4/DocThemPDFUS/Thema81.pdf}, p.18.
}
\seealso{\code{\link{Kenv.csr}}}
\examples{
data(cardiff)
s <- seq(2,30,2)
plot(s, sqrt(khat(as.points(cardiff), cardiff$poly, s)/pi) - s,
 type="l", xlab="Splancs - polygon boundary", ylab="Estimated L",
 ylim=c(-1,1.5))
newstyle <- khat(as.points(cardiff), cardiff$poly, s, newstyle=TRUE)
str(newstyle)
newstyle
apply(newstyle$khats, 2, sum)
plot(newstyle)
}
\references{
Ripley, B.D. 1976 The second-order analysis of stationary point processes,
\emph{ J. Appl. Prob, 13} 255-266;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{khvc}
\alias{khvc}
\title{
Covariance matrix for the difference between two K-functions
}
\description{
Calculate the covariance matrix for the difference between two K-functions.
Also return the contribution to the variance for each of the two point
patterns,
}
\usage{
khvc(pts1, pts2, poly, s)
}
\arguments{
\item{pts1}{
An object containing the case locations.
}
\item{pts2}{
An object containing the control locations.
}
\item{poly}{
A polygon enclosing the locations in \code{pts1} and \code{pts2}
}
\item{s}{
A vector of distances at which the calculation is to be made.
}}
\value{
A list with four components:
\item{varmat}{The upper triangle of the covariance matrix.}
\item{k11}{The variance of Khat for the cases}
\item{k22}{The variance of Khat for the controls}
\item{k12}{The covariance of Khat for the cases and Khat for controls.}
}
\section{Note}{
Note that the diagonal of the covariance matrix is
 \code{$k11 - 2 * $k12 + $k22
}}
\seealso{
\code{\link{khat}}, \code{\link{khvmat}}, \code{\link{secal}}


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{khvmat}
\alias{khvmat}
\title{
Covariance matrix for the difference between two K-functions
}
\description{
Calculate the covariance matrix for the difference between two K-functions 
under random labelling of the corresponding two sets of points.
}
\usage{
khvmat(pts1, pts2, poly, s)
}
\arguments{
\item{pts1}{
An object containing the case locations.
}
\item{pts2}{
An object containing the control locations.
}
\item{poly}{
Polygon enclosing the points in pts1 and pts2.
}
\item{s}{
A vector of distances at which the calculation is to be made.
}}
\value{
A matrix containing the covariances, with the variances on the diagonal.
}

\seealso{
\code{\link{secal}}
}

\references{Diggle P.J and Chetwynd A.C (1991) Second order analysis of
spatial clustering Biometrics 47 1155-63;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.

\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{mpoint}
\alias{mpoint}
\title{
Overlay a number of point patterns
}
\description{
Overlay a number of point patterns.
}
\usage{
mpoint(\dots,cpch,add=FALSE,type="p")
}
\arguments{
\item{...}{
At least one argument consisting of a points data set must be specified.
}
\item{cpch}{
A vector of characters for plotting symbols
}
\item{add}{
if \code{add} is \code{TRUE} then overlay on an existing plot
}
\item{type}{
plot data as points if \code{type="p"}, lines if \code{type="l"}


}}
\details{
\code{mpoint} enables several point or polygon datasets to be overlayed. The plot
region is calculated so that all the specified datasets fit in the region.
The parameter \code{cpch} specifies the characters to use for each set of points. The
default \code{cpch} consists of the numbers 1 to 9 followed by the uppercase 
letters A to Z. If \code{cpch} is shorter than the number of point sets to
plot, then it is repeated.
}
\seealso{
\code{\link{pointmap}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{mse2d}
\alias{mse2d}
\title{
Mean Square Error for a Kernel Smoothing
}
\description{
Estimate the Mean Square Error for a Kernel Smoothing.
}
\usage{
mse2d(pts,poly,nsmse, range)
}
\arguments{
\item{pts}{
A set of points.
}
\item{poly}{
A polygon containng the points.
}
\item{nsmse}{
Number of steps of \code{h} at which to calculate the mean square error.
}
\item{range}{
Maximum value of \code{h} for calculating the mean square error.
}}
\value{
A list with two components, \code{$h} and \code{$mse}. These vectors store
corresponding values of the mean square error at values of the kernel
smoothing parameter, \code{h}. 
The value of \code{h} corresponding to the minimum value of \code{$mse}
can be passed to \code{kernel2d} as the optimum smoothing parameter.


}
\examples{
data(bodmin)
Mse2d <- mse2d(as.points(bodmin), bodmin$poly, nsmse=50, range=8)
plot(Mse2d$h[5:50],Mse2d$mse[5:50], type="l")
}
\seealso{
\code{\link{kernel2d}}


}

\references{
Berman M. \& Diggle P.J. (1989) Estimating Weighted Integrals of the
Second-Order Intensity of a Spatial Point Pattern. 
\emph{ J. R. Statist Soc} B 51 81--92;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{n2dist}
\alias{n2dist}
\title{
Nearest neighbours for two point patterns
}
\description{
Calculate nearest neighbours for two point patterns
}
\usage{
n2dist(pts1,pts2)
}
\arguments{
\item{pts1,pts2}{
 Point data sets
}}
\value{
Returns a list with components \code{$dists} and \code{$neighs}.
\code{$dists[i]} is the distance
of the nearest neighbour of point \code{pts1[i,]}  
in \code{pts2} and \code{$neighs[i]}
is the index in \code{pts2} of the point nearest to \code{pts1[i,]}.


}
\seealso{
\code{\link{nndistF}}, \code{\link{Fhat}}, \code{\link{Ghat}}, \code{\link{Fzero}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{nndistF}
\alias{nndistF}
\title{
Nearest neighbour distances as used by Fhat()
}
\description{
Calculate nearest neighbour distances as used by  \code{Fhat()}
}
\usage{
nndistF(pts1,pts2)
}
\arguments{
\item{pts1}{
A points data set
}
\item{pts2}{
A points data set
}}
\value{
The set of distances from each of the points in \code{pts2} to the nearest
point in \code{pts1} is returned as a vector.


}
\seealso{
\code{\link{nndistG}}, \code{\link{Fhat}}, \code{\link{Ghat}}, \code{\link{Fzero}}
}
\examples{
data(uganda)
boxplot(nndistF(as.points(uganda), as.points(csr(uganda$poly, length(uganda$x)))))
require(stepfun)
plot(ecdf(nndistF(as.points(uganda), 
as.points(csr(uganda$poly, length(uganda$x))))),
main="Fhat ecdf Uganda volcano data")
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{nndistG}
\alias{nndistG}
\title{
Nearest neighbour distances as used by Ghat()
}
\description{
Calculate nearest neighbour distances as used by \code{Ghat()}.
}
\usage{
nndistG(pts)
}
\arguments{
\item{pts}{
A points data set
}}
\value{
Returns a list with components \code{$dists} and \code{$neighs}.
\code{$dists[i]} is the distance
to the nearest neighbour of point \code{i} in \code{pts,} and \code{$neighs[i]}
is the index
of the neighbour of point \code{i}.


}
\examples{
data(uganda)
boxplot(nndistG(as.points(uganda))$dists)
require(stepfun)
plot(ecdf(nndistG(as.points(uganda))$dists))
}
\seealso{
\code{\link{nndistF}}, \code{\link{Fhat}}, \code{\link{Ghat}}, \code{\link{Fzero}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{npts}
\alias{npts}
\title{
Number of points in data set
}
\description{
return number of points in data set
}
\usage{
npts(pts)
}
\arguments{
\item{pts}{
A points data set
}}
\value{
The number of points in the data set.


}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{okblack}
\title{Oklahoma black offenders}
\usage{data(okblack)}
\alias{okblack}
\format{A list corresponding to a Venables and Ripley point object with 147 observations
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    area \tab	list	\tab bounding box with xl, xu, yl, yu\cr
    }
}
\source{Carter and Hill, 1979, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of theft from property offences committed by black offenders in Oklahoma City

}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{okwhite}
\title{Oklahoma white offenders}
\usage{data(okwhite)}
\alias{okwhite}
\format{A list corresponding to a Venables and Ripley point object with 104 observations
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    area \tab	list	\tab bounding box with xl, xu, yl, yu\cr
  }
}
\source{Carter and Hill, 1979, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of theft from property offences committed by white offenders in Oklahoma City

}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\keyword{datasets}

\eof
% Copyright Giovanni Petris and Roger S. Bivand 2001

\name{pcp}
\alias{pcp}
\title{Fit a Poisson cluster process}
\description{
  The function fits a Poisson cluster process to point data for a given
enclosing polygon and fit parameters
}
\usage{
pcp(point.data, poly.data, h0=NULL, expo=0.25, n.int=20)
}
\arguments{
  \item{point.data}{a points object}
  \item{poly.data}{a polygon enclosing the study region}
  \item{h0}{upper bound of integration in the criterion function}
  \item{expo}{exponent in the criterion function}
  \item{n.int}{number of intervals used to approximate the integral in the criterion function with a sum}
}
\value{
  The function returns an object as returned by \code{optim}, including:
  \item{par}{The best set of parameters s2 and rho found}
  \item{value}{The value of the fit corresponding to `par'}
  \item{convergence}{`0' indicates successful convergence}
}
\references{Diggle, P. J. (1983) \emph{Statistical analysis of spatial point
patterns}, London: Academic Press, pp. 55-57 and 78-81; Bailey, T. C. and
Gatrell, A. C. (1995) \emph{Interactive spatial data analysis}, Harlow:
Longman, pp. 106-109.}
\author{Giovanni Petris <GPetris@uark.edu>,  Roger.Bivand@nhh.no}

\seealso{\code{\link{optim}}, \code{\link{pcp.sim}}, \code{\link{Kenv.pcp}}, \code{\link{khat}}}

\examples{
data(cardiff)
polymap(cardiff$poly)
pointmap(as.points(cardiff), add=TRUE)
title("Locations of homes of 168 juvenile offenders")
pcp.fit <- pcp(as.points(cardiff), cardiff$poly, h0=30, n.int=30)
pcp.fit
}
\keyword{spatial}

\eof
% Copyright Giovanni Petris and Roger S. Bivand 2001

\name{pcp.sim}
\alias{pcp.sim}
\title{Generate a Poisson Cluster Process}
\description{
  The function generates a Poisson cluster process for a given polygon within
a larger bounding region and given process parameters
}
\usage{
pcp.sim(rho, m, s2, region.poly, larger.region=NULL, vectorise.loop=TRUE)
}
\arguments{
  \item{rho}{intensity of the parent process}
  \item{m}{average number of offsprings per parent}
  \item{s2}{variance of location of offsprings relative to their parent}
  \item{region.poly}{a polygon defining the region in which the process is to be generated}
  \item{larger.region}{a rectangle containing the region of interest given in the form (xl,xu,yl,yu), defaults to \code{sbox()} around region.poly}
  \item{vectorise.loop}{if TRUE, use new vectorised code, if FALSE, use loop as before}
}
\details{
  The function generates the parents in the larger bounding region, generates
their children also in the larger bounding region, and then returns those
inside the given polygon.}
\value{
  A point object with the simulated pattern
}
\references{Diggle, P. J. (1983) \emph{Statistical analysis of spatial point
patterns}, London: Academic Press, pp. 55-57 and 78-81; Bailey, T. C. and
Gatrell, A. C. (1995) \emph{Interactive spatial data analysis}, Harlow:
Longman, pp. 106-109.}
\author{Giovanni Petris <GPetris@uark.edu>,  Roger.Bivand@nhh.no}

\seealso{\code{\link{pcp}}, \code{\link{Kenv.pcp}}, \code{\link{khat}}}

\examples{
data(cardiff)
polymap(cardiff$poly)
pointmap(as.points(cardiff), add=TRUE)
title("Locations of homes of 168 juvenile offenders")
pcp.fit <- pcp(as.points(cardiff), cardiff$poly, h0=30, n.int=30)
pcp.fit
m <- npts(as.points(cardiff))/(areapl(cardiff$poly)*pcp.fit$par[2])
sims <- pcp.sim(pcp.fit$par[2], m, pcp.fit$par[1], cardiff$poly)
pointmap(as.points(sims), add=TRUE, col="red")
}
\keyword{spatial}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{pdense}
\alias{pdense}
\title{
Overall density for a point pattern
}
\description{
Calculate overall density for a point pattern.
}
\usage{
pdense(pts,poly)
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon data set
}}
\value{
The density of the points in the polygon. i.e. the number of points
per unit area.


}
\seealso{
\code{\link{Fzero}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{pip}
\alias{pip}
\title{
Points inside or outside a polygon
}
\description{
Return points inside or outside a polygon.
}
\usage{
pip(pts,poly,out=FALSE,bound=NULL,quiet=TRUE)
}
\arguments{
\item{pts}{
A points data set
}
\item{poly}{
A polygon data set
}
\item{out}{
If \code{out=TRUE}, return the points outside the polygon, else the points inside.
}
\item{bound}{If points fall exactly on polygon boundaries, the default NULL gives arbitrary assignments. If TRUE, then all points "on" boundaries are set as within the polygon, if FALSE, outside.}
\item{quiet}{Do not report which points are on boundary for non-NULL bound}
}
\value{
\code{pip} returns the points of \code{pts} that lie inside (or outside with
\code{out=TRUE})
the polygon \code{poly}. Compare this with \code{inpip}, which returns
the indices of
the points in the polygon, and \code{inout} which returns a logical vector
indicating whether points are inside or outside.


}
\details{
\code{pip} calls \code{inout}, then selects the appropriate sub-set of points.
}
\seealso{
\code{\link{inpip}}, \code{\link{inout}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{plt}
\alias{plt}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{bins nearest neighbour distances}
\description{
 bins nearest neighbour distances
}
\usage{
plt(data, value)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{data}{nearest neighbour distances}
  \item{value}{breaks for binning distances}
}

\value{
  binned values
}
\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\keyword{spatial}

\seealso{\code{\link{Fhat}}, \code{\link{Ghat}}}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{pointmap}
\alias{pointmap}
\title{
Graphics
}
\description{
Plots point and polygon data sets on the current graphics device.
}
\usage{
pointmap(pts,add=FALSE,axes=TRUE,xlab="",ylab="", asp,\dots)
}
\arguments{
\item{pts}{
a points data set.
}
\item{add}{
if FALSE, start a new plot. If TRUE, superimpose on current plot.
}
\item{axes}{
if true, display axes with labelling. If false, do not display any 
axes on the plot.
}
\item{xlab,ylab}{
Label strings for x and y axes.
}
\item{asp}{aspect parameter for plot}
\item{...}{
Graphical arguments may be entered, and these are passed to the
standard S \code{points} and \code{polygon} routines.


}}
\details{
The specified data set is plotted on the current graphics device, either
as points or polygons. For \code{polymap}, the last point in the data set
is drawn connected to the first point.
\code{pointmap} and \code{polymap} preserve
the aspect ratio in the data by using the \code{asp=1} plot argument.
Graphical parameters can also be supplied to these routines, and are passed 
through to \code{plot}. Some useful parameters include \code{pch} to change the plotting
character for points, \code{lty} to change the line type for polygons, and 
\code{type="n"} to set up axes for the plot without plotting anything.
}
\seealso{
 \code{\link{mpoint}}
}
\examples{
data(bodmin)
plot(bodmin$poly, asp=1, type="n")
pointmap(as.points(bodmin), add=TRUE)
polymap(bodmin$poly, add=TRUE)
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{polymap}
\alias{polymap}
\title{
Graphics
}
\description{
Plots point and polygon data sets on the current graphics device.
}
\usage{
polymap(poly,add=FALSE,xlab="",ylab="",axes=TRUE, asp,...)
}
\arguments{
\item{poly}{
a polygon.
}
\item{add}{
if FALSE, start a new plot. If TRUE, superimpose on current plot.
}
\item{xlab,ylab}{
Label strings for x and y axes.
}
\item{axes}{
if true, display axes with labelling. If false, do not display any 
axes on the plot.
}
\item{asp}{aspect parameter for plot}
\item{...}{
Graphical arguments may be entered, and these are passed to the
standard S \code{points} and \code{polygon} routines.


}}
\details{
The specified data set is plotted on the current graphics device, either
as points or polygons. For \code{polymap}, the last point in the data set
is drawn connected to the first point.
\code{pointmap} and \code{polymap} preserve
the aspect ratio in the data by using the \code{asp=1} plot argument.
Graphical parameters can also be supplied to these routines, and are passed 
through to \code{plot}. Some useful parameters include \code{pch} to change the plotting
character for points, \code{lty} to change the line type for polygons, and 
\code{type="n"} to just set up axes for the plot without plotting anything.
}
\seealso{
\code{\link{par}}, \code{\link{mpoint}}
}
\examples{
data(bodmin)
plot(bodmin$poly, asp=1, type="n")
pointmap(as.points(bodmin), add=TRUE)
polymap(bodmin$poly, add=TRUE)
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{print.ribfit}
\alias{print.ribfit}
\title{
Display the fit from tribble()
}
\description{
Display the fit from \code{tribble}
}
\usage{
print.ribfit(x, ...)
}
\arguments{
\item{x}{
An object returned from \code{tribble}
}
\item{...}{optional arguments to pass through to \code{print()}}
}
\details{
 The parameter estimates and log-likelihood for the raised incidence 
model are displayed. The likelihood ratio, D = 2*(L-Lo), is also given.
 This function is called whenever \code{print} operates on an object
with class \code{ribfit}.
}
\seealso{
\code{\link{tribble}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{ranpts}
\alias{ranpts}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{adjust number of random points in polygon}
\description{
  adjust number of random points in polygon
}
\usage{
ranpts(pts, poly, nprq)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{pts}{points object}
  \item{poly}{polygon object}
  \item{nprq}{required number of points}
}

\value{
  points object with required number of random points
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\keyword{spatial}

\seealso{\code{\link{csr}}}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{rlabel}
\alias{rlabel}
\title{
Randomly label two or more point sets
}
\description{
Randomly label two or more point sets.
}
\usage{
rlabel(\dots)
}
\arguments{
\item{...}{
Any number of points data sets
}}
\value{
A list of points data sets. There are as many elements in the list as 
arguments.


}
\details{
The output data sets are a random labelling of the input data sets, i.e. 
all the points in the input data sets are randomly assigned to the output 
sets. The number of points in each output set is the same as its corresponding
input set.
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{rtor.shift}
\alias{rtor.shift}
\title{
Random toroidal shift on a point data set
}
\description{
Perform a random toroidal shift on a point data set
}
\usage{
rtor.shift(pts,rect)
}
\arguments{
\item{pts}{
The point data set to shift
}
\item{rect}{
A rectangle defining the region for the toroidal map. If not given, 
the bounding box of \code{pts} is used.
}}
\value{
A point data set like \code{pts}, but after application of a random toroidal 
shift along the x and y axes. 
}
\details{
The planar region defined by \code{rect}
is assumed connected at its top and bottom edges, and at its left and
right sides. A random shift is applied to the 
points and the resulting set of points returned. 


}
\seealso{
\code{\link{tor.shift}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{sbox}
\alias{sbox}
\title{
Generate a box surrounding a point object
}
\description{
Generate a box surrounding a point object
}
\usage{
sbox(pts, xfrac = .1, yfrac = .1)
}
\arguments{
\item{pts}{
A points data set
}
\item{xfrac}{
The fraction of the width of the point pattern by which the box will
surround the point pattern to the left and right.
}
\item{yfrac}{
The fraction of the height of the point pattern by which the box will 
surround the point pattern to the top and bottom. 
}}
\value{
A points data set of four points giving the coordinates of the surrounding box


}
\seealso{
\code{\link{bbox}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{secal}
\alias{secal}
\title{
Standard errors for the difference between two K-functions
}
\description{
Calculate standard errors for the difference between two K-functions under
random labelling of the corresponding two sets of points.
}
\usage{
secal(pts1,pts2,poly,s)
}
\arguments{
\item{pts1,pts2}{
Two point data sets
}
\item{poly}{
Polygon enclosing the points in \code{pts1} and \code{pts2}
}
\item{s}{
A vector of distances at which to calculate the standard error.
}}
\value{
A vector like \code{s} containing the value of the standard error at each of the
distances in \code{s}


}
\details{
To compare two point patterns, one can calculate the difference between
their K-functions. The function \code{secal} gives the pointwise standard 
errors for the estimated differences, under the random labelling hypothesis.
}
\references{
Diggle P.J. and Chetwynd A.G. (1991) Second-order analysis of spatial 
clustering
\emph{Biometrics 47} 1155--63; 
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

\seealso{
\code{\link{khat}}
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{shift}
\alias{shift}
\title{
Shift a point data set
}
\description{
Shift a point data set
}
\usage{
shift(pts,xsh=0.0,ysh=0.0)
}
\arguments{
\item{pts}{
The point data set to shift
}
\item{xsh}{
Amount to shift along the x-axis
}
\item{ysh}{
Amount to shift along the y-axis
}}
\value{
A point data set like \code{pts}, but with \code{xsh} added to its
x-coordinates, and \code{ysh} added to its y-coordinates.


}
\seealso{
\code{\link{tor.shift}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
\name{southlancs}
\title{Cancer cases in Chorley-Ribble}
\usage{data(southlancs)}
\alias{southlancs}
\alias{southlancs.pts}
\alias{southlancs.cc}
\alias{old.incinerator}
\alias{southlancs.bdy}
\format{A data frame with 974 observations
  \tabular{rlll}{
    [,1]  \tab x  \tab numeric  \tab grid eastings (metres)\cr
    [,2]  \tab y   \tab numeric  \tab grid northings (metres)\cr
    [,3]  \tab cc  \tab numeric  \tab case/control, lung=0, larynx=1\cr
  }
}
\source{Diggle, Gatrell and Lovett, 1990, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of cases of cancer of lung and larynx in
Chorley-Ribble, Lancashire. The data set is split into a points object \code{southlancs.pts} and a
case/control 0/1 vector \code{southlancs.cc}. There are 917 controls and 57
cases in this data set - these numbers differ from 978 and 58 in Diggle (1990)
and Diggle and Rowlingson (1994). The data set also includes the approximate
location of an old incinerator \code{old.incinerator}, as well as
\code{southlancs.bdy}, the study area boundary.

}
\references{Bailey and Gatrell 1995, ch. 3; Diggle, P. (1990) A point process
modelling approach to raised incidence of a rare phenomenon in the viscinity
of a prespecified point. Journal of the Royal Statistical Society, A, 153,
349-362; Diggle, P. and Rowlingson, B. (1994) A conditional approach to point
process modelling of elevated risk. Journal of the Royal Statistical Society,
A, 157, 433-440.

}
\examples{
data(southlancs)
op <- par(mfrow=c(2,1))
pointmap(southlancs.pts[southlancs.cc == 0,])
pointmap(old.incinerator, add=TRUE, col="red", pch=19)
title("Lung cancer controls")
pointmap(southlancs.pts[southlancs.cc == 1,])
pointmap(old.incinerator, add=TRUE, col="red", pch=19)
title("Larynx cancer cases")
par(op)
polymap(southlancs.bdy,border="grey")
contour(kernel2d(southlancs.pts[southlancs.cc == 0,], 
	southlancs.bdy, h=500, nx=100, ny=100), nlevels=20, 
	add=TRUE,drawlabels=FALSE)
pointmap(southlancs.pts[southlancs.cc == 1,], add=TRUE, pch=19,
	 col="green")
pointmap(old.incinerator, add=TRUE, pch=19, col="red")
title(xlab="h=500, quartic kernel")
title("Density map of control, green case points, red old incinerator")

}
\keyword{datasets}

\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{splancs}
\alias{splancs}
\title{
Return version number and author information
}
\description{
Return version number and author information
}
\usage{
splancs()
}
\value{
The version string is returned. This is a number of the format x.yy,
where x is the major version number and yy is the minor version number.


}
%\section{Side Effects}{
%A message is printed giving copyright and author information, together with
%the date of release of the software. This information is useful in case of
%problems with Splancs.
%
%
%}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{spoints}
\alias{spoints}
\title{
Point Objects
}
\description{
Creates and tests for data in spatial point format.
}
\usage{
spoints(data,npoints)
}
\arguments{
\item{data}{
vector containing the data values for the points
in order (x1,y1),(x2,y2),\dots{}


}
\item{npoints}{
number of points to generate, if missing, set to length(data)/2.
}}
\value{
\code{spoints} returns an object suitable for use as a point data
object. If \code{npoints} is given, the vector data is either truncated
or repeated until sufficient data values are generated.
The returned object is a two-column matrix, where the first column stores the 
x-coordinate, and the second column stores the y-coordinate.


}
\seealso{
\code{\link{npts}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{stdiagn}
\alias{stdiagn}
\title{
Summary plots for clustering analysis
}
\description{
Produces some summary plots for clustering analysis
}
\usage{
stdiagn(pts, stkh, stse, stmc=0,Dzero=FALSE)
}
\arguments{
\item{pts}{
A set of points, as used in Splancs
}
\item{stkh}{
An object returned from \code{stkhat}
}
\item{stse}{
An object returned from \code{stsecal}
}
\item{stmc}{
An object returned from \code{stmctest}
}
\item{Dzero}{FALSE - default D plot, TRUE Dzero plot}
}
\details{
Four plots are produced on the current graphics device.
The first plot is simply a map of the data. The second is a perspective plot
of the difference between space-time K-function and the product of spatial
and temporal K-functions. The third plot is of the 
standardised residuals against the product of spatial and temporal
K-functions. If the Monte-Carlo data is given the fourth plot is a
a histogram of the test statistics, with the value for the data indicated
with a vertical line.
See Diggle, Chetwynd, Haggkvist, and Morris (1995) for details.
}
\seealso{
\code{\link{stkhat}}, \code{\link{stsecal}}, \code{\link{stvmat}}, \code{\link{stmctest}}
}

\references{
Diggle, P., Chetwynd, A., Haggkvist, R. and Morris, S. 1995 Second-order analysis of space-time clustering. Statistical Methods in Medical Research, 4, 124-136;Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow, pp. 122-125;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}
\examples{
example(stkhat)
example(stsecal)
example(stmctest)
stdiagn(burpts, bur1, bur1se, bur1mc)
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{stkhat}
\alias{stkhat}
\title{
Space-time K-functions
}
\description{
Compute the space-time K-functions
}
\usage{
stkhat(pts, times, poly, tlimits, s, tm)
}
\arguments{
\item{pts}{
A set of points as defined in Splancs
}
\item{times}{
A vector of times, the same length as the number of points in \code{pts}
}
\item{poly}{
A polygon enclosing the points
}
\item{tlimits}{
A vector of length 2 specifying the upper and lower temporal domain.
}
\item{s}{
A vector of spatial distances for the analysis.
}
\item{tm}{
A vector of times for the analysis
}}
\value{
A list with the following components is returned:
\item{s, t}{The spatial and temporal scales}
\item{ks}{The spatial K-function}
\item{kt}{The temporal K-function}
\item{kst}{The space-time K-function}
For details see Diggle, Chetwynd, Haggkvist, and Morris (1995)
}
\seealso{
\code{\link{stsecal}}, \code{\link{stvmat}}, \code{\link{stmctest}}, \code{\link{stdiagn}}
}

\references{
Diggle, P., Chetwynd, A., Haggkvist, R. and Morris, S. 1995 Second-order analysis of space-time clustering. Statistical Methods in Medical Research, 4, 124-136;Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow, pp. 122-125;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}
\examples{
data(burkitt)
bur1 <- stkhat(burpts, burkitt$t, burbdy, c(400, 5800),
  seq(1,40,2), seq(100, 1500, 100))
oldpar <- par(mfrow=c(2,1))
plot(bur1$s, bur1$ks, type="l", xlab="distance", ylab="Estimated K",
  main="spatial K function")
plot(bur1$t, bur1$kt, type="l", xlab="time", ylab="Estimated K",
  main="temporal K function")
par(oldpar)
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{stmctest}
\alias{stmctest}
\title{
Monte-Carlo test of space-time clustering
}
\description{
Perform a Monte-Carlo test of space-time clustering.
}
\usage{
stmctest(pts, times, poly, tlimits, s, tt, nsim, quiet=FALSE)
}
\arguments{
\item{pts}{
A set of points as used by Splancs.
}
\item{times}{
A vector of times, the same length as the number of points in \code{pts}.
}
\item{poly}{
A polygon enclosing the points.
}
\item{tlimits}{
A vector of length 2, specifying the upper and lower temporal domain.
}
\item{s}{
A vector of spatial distances for the analysis.
}
\item{tt}{
A vector of times for the analysis.
}
\item{nsim}{
The number of simulations to do.
}
\item{quiet}{
If \code{quiet=TRUE} then no output is produced, otherwise the function
prints the number of simulations completed so far, and also how the
test statistic for the data ranks with the simulations.
}}
\value{
A vector of length \code{nsim+1}. The first element is the test statistic
for the data, and the remaining elements are those for the simulations.
}
\details{
The function  uses a sum of residuals as a test statistic, 
randomly permutes the times of the set of points and recomputes the
test statistic for a number of simulations.
See Diggle, Chetwynd, Haggkvist and Morris (1995) for details.
}
\seealso{
\code{\link{stkhat}}, \code{\link{stsecal}}, \code{\link{stvmat}}, \code{\link{stdiagn}}
}

\references{
Diggle, P., Chetwynd, A., Haggkvist, R. and Morris, S. 1995 Second-order analysis of space-time clustering. Statistical Methods in Medical Research, 4, 124-136;Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow, pp. 122-125;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}
\examples{
example(stkhat)
bur1mc <- stmctest(burpts, burkitt$t, burbdy, c(400, 5800),
  seq(1,40,2), seq(100, 1500, 100), nsim=49, quiet=TRUE)
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{stsecal}
\alias{stsecal}
\title{
Standard error for space-time clustering
}
\description{
Computes the standard error for space-time clustering.
}
\usage{
stsecal(pts, times, poly, tlim, s, tm)
}
\arguments{
\item{pts}{
A set of points, as defined in Splancs.
}
\item{times}{
A vector of times, the same length as the number of points in \code{pts}
}
\item{poly}{
A polygon enclosing the points
}
\item{tlim}{
A vector of length 2 specifying the upper and lower temporal domain.
}
\item{s}{
A vector of spatial distances for the analysis
}
\item{tm}{
A vector of times for the analysis
}}
\value{
A matrix of dimension \code{[length(s),length(t)]} is returned. Element 
\code{[i,j]} is the standard error at \code{s[i],t[j]}.
See Diggle Chetwynd Haggkvist and Morris (1995) for details.
}
\seealso{
\code{\link{stkhat}}, \code{\link{stsecal}}, \code{\link{stvmat}}, \code{\link{stdiagn}}
}

\references{
Diggle, P., Chetwynd, A., Haggkvist, R. and Morris, S. 1995 Second-order analysis of space-time clustering. Statistical Methods in Medical Research, 4, 124-136;Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow, pp. 122-125;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}
\examples{
example(stkhat)
bur1se <- stsecal(burpts, burkitt$t, burbdy, c(400, 5800),
 seq(1,40,2), seq(100, 1500, 100))
}
% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{stvmat}
\alias{stvmat}
\title{
Variance matrix for space-time clustering
}
\description{
Compute the variance matrix for space-time clustering
}
\usage{
stvmat(pts, times, poly, tlim, s, tm)
}
\arguments{
\item{pts}{
A set of points.
}
\item{times}{
A vector of times, the same length as the number of points in \code{pts}
}
\item{poly}{
A polygon that encloses the points
}
\item{tlim}{
A vector of length 2 specifying the upper and lower temporal domain.
}
\item{s}{
A vector of spatial distances for the analysis
}
\item{tm}{
A vector of times for the analysis
}}
\value{
A four-dimensional matrix is returned. The covariance between space-time
\code{t1,s1} and \code{t2,s2} is given by the corresponding element \code{[t1,s1,t2,s2]}
For full details, see Diggle, Chetwynd, Haggkvist and Morris (1995)
}
\seealso{
\code{\link{stkhat}}, \code{\link{stsecal}}, \code{\link{stmctest}}, \code{\link{stdiagn}}
}

\references{
Diggle, P., Chetwynd, A., Haggkvist, R. and Morris, S. 1995 Second-order analysis of space-time clustering. Statistical Methods in Medical Research, 4, 124-136;
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{thin}
\alias{thin}
\title{
Randomly thin a point data set
}
\description{
Randomly thin a point data set.
}
\usage{
thin(pts,n)
}
\arguments{
\item{pts}{
a points data set.
}
\item{n}{
the number of points to return
}}
\value{
Returns a point data set consisting of \code{n} points selected randomly
from the set \code{pts}.
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{tor.shift}
\alias{tor.shift}
\title{
Toroidal shift on a point data set
}
\description{
Perform a toroidal shift on a point data set
}
\usage{
tor.shift(pts,xsh=0.0,ysh=0.0,rect)
}
\arguments{
\item{pts}{
The point data set to shift
}
\item{xsh}{
Amount to shift along the x-axis
}
\item{ysh}{
Amount to shift along the y-axis
}
\item{rect}{
A rectangle defining the region for the toroidal map. If not given, 
the bounding box of \code{pts} is used.
}}
\value{
A point data set like \code{pts}, but after application of a toroidal 
shift along the x and y axes. 
}
\details{
The planar region defined by \code{rect}
is assumed connected at its top and bottom edges, and at its left and
right sides. A shift of \code{xsh} and \code{ysh} is applied to the 
points and the resulting set of points returned. 


}
\seealso{
\code{\link{shift}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{tribble}
\alias{tribble}
\title{
Diggle-Rowlingson Raised Incidence Model
}
\description{
Fits the Diggle-Rowlingson Raised Incidence Model.
}
\usage{
tribble(ccflag, vars=NULL, alphas=NULL, betas=NULL, rho, 
 which=1:length(alphas), covars=NULL, thetas=NULL, 
 steps=NULL, reqmin=0.001, icount=50, hessian=NULL)
}
\arguments{
\item{ccflag}{
Case-control flag : a vector of ones and zeroes.
}
\item{vars}{
A matrix where \code{vars[i,j]} is the distance squared from point \code{i} 
to source \code{j}.
}
\item{alphas}{
Initial value of the \code{alpha} parameters.
}
\item{betas}{
Initial value of the \code{beta} parameters.
}
\item{rho}{
Initial value of the \code{rho} parameter.
}
\item{which}{
Defines the mapping from sources to parameters.
}
\item{covars}{
A matrix of covariates to be modelled as log-linear terms. The element 
\code{covars[i,j]} is the value of covariate \code{j} for case/control \code{i}.
}
\item{thetas}{
Initial values of covariate parameters.
}
\item{steps}{
Step sizes for the Nelder-Mead simplex algorithm.
}
\item{reqmin}{
Tolerance for simplex algorithm
}
\item{icount}{
Iteration count for simplex algorithm
}
\item{hessian}{by default NULL, any other value causes hessian to be computed and returned}
}
\value{
The return value is a list with many components, and class \code{ribfit}. 
\item{alphas}{A vector of the alpha parameters at the maximum}
\item{betas}{A vector of the beta values at the maximum}
\item{rho}{The value of rho at the maximum}
\item{logl}{The maximised log-likelihood}
\item{null.logl}{The null log-likelihood}
\item{call}{The function call to \code{tribble}}
For further information see Diggle and Rowlingson (1993).
}
\seealso{
\code{\link{triblik}}, \code{\link{dsquare}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{triblik}
\alias{triblik}
\title{
Log-likelihood for the Diggle-Rowlingson raised incidence model
}
\description{
Calculates the log-likelihood for the Diggle-Rowlingson raised incidence model.
}
\usage{
triblik(ccflag, vars=NULL, alphas=NULL, betas=NULL, rho, 
 which=1:length(alphas), covars=NULL, thetas=NULL)
}
\arguments{
\item{ccflag}{
Case-control flag : a vector of ones and zeroes.
}
\item{vars}{
A matrix where \code{vars[i,j]} is the distance squared from point \code{i} 
to source \code{j}.
}
\item{alphas}{
The \code{alpha} parameters.
}
\item{betas}{
The \code{beta} parameters.
}
\item{rho}{
The \code{rho} parameter.
}
\item{which}{
Defines the mapping from sources to parameters.
}
\item{covars}{
A matrix of covariates to be modelled as log-linear terms. The element 
\code{covars[i,j]} is the value of covariate \code{j} for case/control \code{i}.
}
\item{thetas}{
The covariate parameters.
}}
\value{
The log-likelihood for the given parameters and the given distances and
optional covariates is returned.
}
\seealso{
\code{\link{tribble}}, \code{\link{dsquare}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{uganda}
\title{Craters in Uganda}
\usage{data(uganda)}
\alias{uganda}
\format{A list corresponding to a Venables and Ripley point object with 120 observations
  \tabular{lll}{
    x  \tab numeric  \tab grid eastings\cr
    y   \tab numeric  \tab grid northings\cr
    area \tab	list	\tab bounding box with xl, xu, yl, yu\cr
    poly \tab array \tab polygon boundary with columns x and y\cr
  }
}
\source{Tinkler, 1971, - Bailey and Gatrell 1995, ch. 3.
}
\description{Locations of craters in a volcanic field in Uganda

}
\references{Bailey, T. C. and Gatrell, A. C. 1995, Interactive spatial data analysis. Longman, Harlow.

}
\keyword{datasets}


\eof
% Copyright Barry Rowlingson  and 
% Peter Diggle (c) 1991-3; http://www.maths.lancs.ac.uk/~rowlings/Splancs/
% R port: copyright 1998-2000 by Roger S. Bivand

\name{zoom}
\alias{zoom}
\title{
Interactively specify a region of a plot for expansion
}
\description{
Interactively specify a region of a plot for expansion
}
\usage{
zoom(quiet=FALSE,out=FALSE,...)
}
\arguments{
\item{quiet}{
If false, prompt the user to enter two coordinates. If true, say nothing.
}
\item{out}{
If true, expand the limits of the current plot by a factor of three, centred
on the current plot.
}
\item{...}{
Other arguments are passed through to pointmap.
}}
\value{
None


}
\details{
A prompt is optionally displayed, and the user selects two points
forming the diagonal of a rectangle. A new, empty plot is created that has its
axis limits set to the bounding square of the selected rectangle. 
If \code{out=TRUE}, no prompt is displayed, and a new blank plot is created with
its limits in x and y set to span an area three times the height and width
centred on the current centre.
}
\seealso{
\code{\link{pointmap}}
}

\references{
Rowlingson, B. and Diggle, P. 1993 Splancs: spatial point pattern analysis
code in S-Plus.  Computers and Geosciences, 19, 627-655;
the original sources can be accessed at:
\url{http://www.maths.lancs.ac.uk/~rowlings/Splancs/}. See also Bivand, R. and
Gebhardt, A. 2000 Implementing functions for spatial statistical analysis
using the R language. Journal of Geographical Systems, 2, 307-317.
}

% Converted by Sd2Rd version 0.3-3.
\keyword{spatial}


\eof
