\name{Mandible}
\alias{Mandible}
\title{Mandible Data}
\usage{data(Mandible)}
\description{
Mandible Data.
}
\format{
  Data from 167 fetuses, especially:
  \describe{
    \item{age}{gestational age in weeks.}
    \item{length}{mandible length in mm.}
  }
}
\source{
  The data was originally published by Chitty et al., 1993, and analysed in
Royston and Altman, 1994 (the data is given there). Only measurments with
\code{age <= 28} were used in this analysis.
}
\references{

  L. S. Chitty and S. Campbell and D. G. Altman (1993), 
  Measurement of the fetal mandible -- feasibility and
  construction of a centile chart., \emph{Prenatal Diagnosis}, 
  \bold{13}, 749--756.

  P. Royston and D. G. Altman (1994), Regression Using 
  Fractional Polynomials of Continuous Covariates: Parsimonious 
  Parametric Modelling. \emph{Applied Statistics}, \bold{43}, 429--453.

}

\examples{
data(Mandible)
lm(length ~ age, data=Mandible, subset=(age <= 28))

}
\keyword{datasets}

\eof
\name{bgtest}
\alias{bgtest}
\title{Breusch-Godfrey Test}
\description{
 \code{bgtest} performs the Breusch-Godfrey test for higher order
 serial correlation.
}
\usage{
bgtest(formula, order = 1, type = c("Chisq", "F"), data = list())
}
\arguments{
  \item{formula}{a symbolic description for the model to be tested.}
  \item{order}{integer. maximal order of serial correlation to be tested.}
  \item{type}{the type of test statistic to be returned. Either
    \code{"Chisq"} for the Chi-squared test statistic or
    \code{"F"} for the F test statistic.}
  \item{data}{an optional data frame containing the variables in the
    model. By default the variables are taken from the environment
    which \code{bgtest} is called from.}
  }
\details{
  Under \eqn{H_0} the test statistic is asymptotically Chi-squared with
  degrees of freedom as given in \code{parameter}.
  If \code{type} is set to \code{"F"} the function returns
  the exact F statistic which, under \eqn{H_0}, follows an \eqn{F}
  distribution with degrees of freedom as given in \code{parameter}.

  The starting values for the lagged residuals in the supplementary
  regression are chosen to be 0.
}
\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{parameter}{degrees of freedom.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\references{
  Johnston, J. (1984): \emph{Econometric Methods}, Third Edition, McGraw Hill
  Inc.

  Godfrey, L.G. (1978): `Testing Against General Autoregressive and
  Moving Average Error Models when the Regressors Include Lagged
  Dependent Variables', \emph{Econometrica}, 46, 1293-1302.

  Breusch, T.S. (1979): `Testing for Autocorrelation in Dynamic Linear
  Models', \emph{Australian Economic Papers}, 17, 334-355.
}
\author{David Mitchell <david.mitchell@dotars.gov.au>, Achim Zeileis}
\seealso{\code{\link{dwtest}}}
\examples{

     ## Generate a stationary and an AR(1) series
     x <- rep(c(1, -1), 50)

     y1 <- 1 + x + rnorm(100)

     ## Perform Breusch-Godfrey test for first order serial correlation:
     bgtest(y1 ~ x)
     ## or for fourth order serial correlation
     bgtest(y1 ~ x, order = 4)
     ## Compare with Durbin-Watson test results:
     dwtest(y1 ~ x)

     if(library(ts, logical = TRUE)) {
     y2 <- filter(y1, 0.5, method = "recursive")
     bgtest(y2 ~ x) }


}
\keyword{htest}

\eof
\name{bondyield}
\alias{bondyield}
\non_function{}
\title{ Bond Yield  }
\usage{data(bondyield)}
\description{
Bond Yield Data.
}
\format{
 A multivariate quarterly time series from 1961(1) to 1975(4) with variables
 \describe{
    \item{RAARUS}{difference of interest rate on government and corporate
bonds,}
    \item{MOOD}{measure of consumer sentiment,}
    \item{EPI}{index of employment pressure,}
    \item{EXP}{interest rate expectations,}
    \item{Y}{artifical time series based on RAARUS,}
    \item{K}{artifical time series based on RAARUS.}
  }
}
\source{
The data was originally studied by Cook and Hendershott (1978) and Yawitz and
Marshall (1981), the data set is given in Krmer and Sonnberger (1986). Below
we replicate a few examples given in their book. 
Some of these results differ more or
less seriously and are sometimes parameterized differently.
}
\references{

T.Q. Cook & P.H. Hendershott (1978),
The Impact of Taxes, Risk and Relative Security Supplies of Interest Rate
Differentials. \emph{The Journal of Finance} \bold{33}, 1173--1186

J.B. Yawitz & W. J. Marshall (1981),
Measuring the Effect of Callability on Bond Yields.
\emph{Journal of Money, Credit and Banking} \bold{13}, 60--71

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(bondyield)

## page 134, fit Cook-Hendershott OLS model and Yawitz-Marshall OLS model
## third and last line in Table 6.5

modelCH <- RAARUS ~ MOOD + EPI + EXP + RUS
lm(modelCH, data=bondyield)
dwtest(modelCH, data=bondyield)
## wrong sign of RUS coefficient

modelYM <- RAARUS ~ MOOD + Y + K
lm(modelYM, data=bondyield)
dwtest(modelYM, data=bondyield)
## coefficient of Y and K differ by factor 100


## page 135, fit test statistics in Table 6.6 b)
################################################

## Chow 1971(1)
if(require(strucchange, quietly = TRUE)) {
sctest(modelCH, point=c(1971,1), data=bondyield, type="Chow") }

## Breusch-Pagan
bptest(modelCH, data=bondyield, studentize=FALSE)
bptest(modelCH, data=bondyield)

## Fluctuation test
if(require(strucchange, quietly = TRUE)) {
sctest(modelCH, type="fluctuation", data=bondyield, rescale=FALSE)}

## RESET
reset(modelCH, data=bondyield)
reset(modelCH, power=2, type="regressor", data=bondyield)
reset(modelCH, type="princomp", data=bondyield)

## Harvey-Collier
harvtest(modelCH, order.by= ~ MOOD, data=bondyield)
harvtest(modelCH, order.by= ~ EPI, data=bondyield)
harvtest(modelCH, order.by= ~ EXP, data=bondyield)
harvtest(modelCH, order.by= ~ RUS, data=bondyield)

## Rainbow
raintest(modelCH, order.by = "mahalanobis", data=bondyield)


## page 136, fit test statistics in Table 6.6 d)
################################################

## Chow 1966(1)
if(require(strucchange, quietly = TRUE)) {
sctest(modelYM, point=c(1965,4), data=bondyield, type="Chow") }

## Fluctuation test
if(require(strucchange, quietly = TRUE)) {
sctest(modelYM, type="fluctuation", data=bondyield, rescale=FALSE) }

## RESET
reset(modelYM, data=bondyield)
reset(modelYM, power=2, type="regressor", data=bondyield)
reset(modelYM, type="princomp", data=bondyield)

## Harvey-Collier
harvtest(modelYM, order.by= ~ MOOD, data=bondyield)
harvtest(modelYM, order.by= ~ Y, data=bondyield)
harvtest(modelYM, order.by= ~ K, data=bondyield)

## Rainbow
raintest(modelYM, order.by = "mahalanobis", data=bondyield)
}
\keyword{datasets}

\eof
\name{bptest}
\alias{bptest}
\title{Breusch-Pagan Test}
\description{Performs the Breusch-Pagan test for heteroskedasticity.}
\usage{
bptest(formula, varformula = NULL, studentize = TRUE, data = list())
}
\arguments{
 \item{formula}{a symbolic description for the model to be tested.}
 \item{varformula}{a formula describing only the potential explanatory variables
   for the variance (no dependent variable needed). By default the same
   explanatory variables are taken as in the main regression model.}
 \item{studentize}{logical. If set to \code{TRUE} Koenker's studentized
   version of the test statistic will be used.}
 \item{data}{an optional data frame containing the variables in the model.
By default the variables are taken from the environment which \code{bptest} is
called from.}
}
\details{The Breusch-Pagan test fits a linear regression model to the residuals
of a linear regression model
(by default the same explanatory variables are taken as in the main regression
model) and rejects if too much of the variance
is explained by the additional explanatory variables.

Under \eqn{H_0} the test statistic of the Breusch-Pagan test follows a
chi-squared distribution with \code{parameter} (the number of regressors without
the constant in the model) degrees of freedom.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.
}

\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{parameter}{degrees of freedom.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}

\references{
T.S. Breusch & A.R. Pagan (1979),
A Simple Test for Heteroscedasticity and Random Coefficient Variation.
\emph{Econometrica} \bold{47}, 1287--1294

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica
}

\seealso{\code{\link{lm}}}

\examples{
## generate a regressor
x <- rep(c(-1,1), 50)
## generate heteroskedastic and homoskedastic disturbances
err1 <- rnorm(100, sd=rep(c(1,2), 50))
err2 <- rnorm(100)
## generate a linear relationship
y1 <- 1 + x + err1
y2 <- 1 + x + err2
## perform Breusch-Pagan test
bp <- bptest(y1 ~ x)
bp
## calculate critical value for 0.05 level
qchisq(0.95, bp$parameter)
bptest(y2 ~ x)
}
\keyword{htest}

\eof
\name{currencysubstitution}
\alias{currencysubstitution}
\non_function{}
\title{ Currency Substitution }
\usage{data(currencysubstitution)}
\description{
Currency Subsitution Data.
}
\format{
 A multivariate quarterly time series from 1960(4) to 1975(4) with variables
 \describe{
    \item{logCUS}{logarithm of the ratio of canadian holdings of Canadian
dollar balances and canadian holdings of U.S. dollar balances,}
    \item{Iu}{yield on U.S. Treasury bills,}
    \item{Ic}{yield on Canadian Treasury bills,}
    \item{logY}{logarithm of Canadian real gross national product.}
  }
}
\source{
The data was originally studied by Miles (1978), the data set is given in
Krmer and Sonnberger (1986). Below we replicate a few examples 
from their book.
Some of these results differ more or less seriously and are sometimes
parameterized differently.
}
\references{

M. Miles (1978),
Currency Substitution, Flexible Exchange Rates, and Monetary Independence.
\emph{American Economic Review} \bold{68}, 428--436

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(currencysubstitution)

## page 130, fit Miles OLS model and Bordo-Choudri OLS model
## third and last line in Table 6.3

modelMiles <- logCUS ~ log((1+Iu)/(1+Ic))
lm(modelMiles, data=currencysubstitution)
dwtest(modelMiles, data=currencysubstitution)

modelBordoChoudri <- logCUS ~ I(Iu-Ic) + Ic + logY
lm(modelBordoChoudri, data=currencysubstitution)
dwtest(modelBordoChoudri, data=currencysubstitution)


## page 131, fit test statistics in Table 6.4 b)
################################################

if(require(strucchange, quietly = TRUE)) {
## Fluctuation test
sctest(modelMiles, type="fluctuation", data=currencysubstitution,
 rescale=FALSE) }

## RESET
reset(modelMiles, data=currencysubstitution)
reset(modelMiles, power=2, type="regressor", data=currencysubstitution)
reset(modelMiles, type="princomp", data=currencysubstitution)

## Harvey-Collier
harvtest(modelMiles, order.by = ~log((1+Iu)/(1+Ic)), data=currencysubstitution)

## Rainbow
raintest(modelMiles, order.by = "mahalanobis", data=currencysubstitution)


## page 132, fit test statistics in Table 6.4 d)
################################################

if(require(strucchange, quietly = TRUE)) {
## Chow 1970(2)
sctest(modelBordoChoudri, point=c(1970,2), data=currencysubstitution,
 type="Chow") }

## Breusch-Pagan
bptest(modelBordoChoudri, data=currencysubstitution, studentize=FALSE)
bptest(modelBordoChoudri, data=currencysubstitution)

## RESET
reset(modelBordoChoudri, data=currencysubstitution)
reset(modelBordoChoudri, power=2, type="regressor", data=currencysubstitution)
reset(modelBordoChoudri, type="princomp", data=currencysubstitution)

## Harvey-Collier
harvtest(modelBordoChoudri, order.by = ~ I(Iu-Ic), data=currencysubstitution)
harvtest(modelBordoChoudri, order.by = ~ Ic, data=currencysubstitution)
harvtest(modelBordoChoudri, order.by = ~ logY, data=currencysubstitution)

## Rainbow
raintest(modelBordoChoudri, order.by = "mahalanobis", data=currencysubstitution)
}
\keyword{datasets}

\eof
\name{dwtest}
\alias{dwtest}
\title{Durbin-Watson Test}
\description{
 Performs the Durbin-Watson test for autocorrelation of disturbances.
}
\usage{
dwtest(formula, alternative = c("greater", "two.sided", "less"),
       iterations = 15, exact = NULL, tol = 1e-10, data = list())
}
\arguments{
 \item{formula}{a symbolic description for the model to be tested.}
 \item{alternative}{a character string specifying the alternative hypothesis.}
 \item{iterations}{an integer specifying the number of iterations
    when calculating the p-value with the "pan" algorithm.}
 \item{exact}{logical. If set to \code{FALSE} a normal approximation
   will be used to compute the p value, if \code{TRUE} the "pan"
   algorithm is used. The default is to use "pan" if the sample size
   is < 100.}
 \item{tol}{tolerance. Eigenvalues computed have to be greater than
            \code{tol} to be treated as non-zero.}
 \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which \code{dwtest}
    is called from.}
}
\details{The Durbin-Watson test has the null hypothesis that the autocorrelation
of the disturbances is 0; it can be tested against the alternative that it is
greater than, not equal to, or less than 0 respectively. This can be specified
by the \code{alternative} argument.

The null distribution of the Durbin-Watson test statistic is a linear
combination of chi-squared distributions. The p value is computed using a
Fortran version of the Applied Statistics Algorithm AS 153 by Farebrother
(1980, 1984). This algorithm is called "pan" or "gradsol". For large sample
sizes the algorithm might fail to compute the p value; in that case a warning
is printed and an approximate p value will be given; this p value is computed
using a normal approximation with mean and variance of the Durbin-Watson test
statistic.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.

For an overview on R and econometrics see Racine & Hyndman (2002).
}
\value{An object of class \code{"htest"} containing:
\itemize
\item{statistic}{the test statistic.}
\item{p.value}{the corresponding p-value.}
\item{method}{a character string with the method used.}
\item{data.name}{a character string with the data name.}
}
\references{

J. Durbin & G.S. Watson (1950),
Testing for Serial Correlation in Least Squares Regression I.
\emph{Biometrika} \bold{37}, 409--428.

J. Durbin & G.S. Watson (1951),
Testing for Serial Correlation in Least Squares Regression II.
\emph{Biometrika} \bold{38}, 159--178.

J. Durbin & G.S. Watson (1971),
Testing for Serial Correlation in Least Squares Regression III.
\emph{Biometrika} \bold{58}, 1--19.

R.W. Farebrother (1980),
Pan's Procedure for the Tail Probabilities of the
Durbin-Watson Statistic (Corr: 81V30 p189; AS R52: 84V33 p363- 366; AS
R53: 84V33 p366- 369).
\emph{Applied Statistics} \bold{29},  224--227.

R. W. Farebrother (1984),
[AS R53] A Remark on Algorithms AS 106 (77V26 p92-98), AS 153 (80V29 p224-227)
  and AS 155: The Distribution of a Linear Combination of $\chi^2$ Random
  Variables (80V29 p323-333)
\emph{Applied Statistics} \bold{33}, 366--369.

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica.

J. Racine & R. Hyndman (2002),
Using R To Teach Econometrics.
\emph{Journal of Applied Econometrics} \bold{17}, 175--189.

}

\seealso{\code{\link{lm}}}

\examples{

## generate two AR(1) error terms with parameter
## rho = 0 (white noise) and rho = 0.9 respectively
err1 <- rnorm(100)

## generate regressor and dependent variable
x <- rep(c(-1,1), 50)
y1 <- 1 + x + err1

## perform Durbin-Watson test
dwtest(y1 ~ x)

if(library(ts, logical = TRUE)) {
err2 <- filter(err1, 0.9, method="recursive")
y2 <- 1 + x + err2
dwtest(y2 ~ x) }

}
\keyword{htest}

\eof
\name{ftemp}
\alias{ftemp}
\title{Femal Temperature Data}
\usage{data(ftemp)}
\description{
Daily morning temperature of adult female (in degrees Celsius).
}
\format{
  Univariate daily time series of 60 observations starting from 1990-07-11.
}
\details{
The data gives the daily morning temperature of an adult woman
measured in degrees Celsius at about 6.30am each morning.

At the start of the period the woman was sick, hence the high temperature.
Then the usual monthly cycle can be seen. On the second cycle, the temperature
doesn't complete the downward part of the pattern due to a conception.
}

\source{
The data set is taken from the Time Series Data Library at

  \url{http://www-personal.buseco.monash.edu.au/~hyndman/TSDL/}

maintained by Rob Hyndman and Muhammad Akram.
}

\examples{
data(ftemp)
plot(ftemp)
y <- window(ftemp, start = 8, end = 60)
if(require(strucchange)) {
  bp <- breakpoints(y ~ 1)
  plot(bp)
  fm.seg <- lm(y ~ 0 + breakfactor(bp))
  plot(y)
  lines(8:60, fitted(fm.seg), col = 4)
  lines(confint(bp))
}

}
\keyword{datasets}

\eof
\name{gqtest}
\alias{gqtest}
\title{Goldfeld-Quandt Test}
\description{Goldfeld-Quandt test for heteroskedasticity.}
\usage{
gqtest(formula, point = 0.5, order.by = NULL, data = list())
}
\arguments{
 \item{formula}{a symbolic description for the model to be tested}
 \item{point}{numerics. If \code{point} is smaller than 1 it is
     interpreted as percentages of data, i.e. \code{n*point} is
     taken to be the (potential) breakpoint in the variances, if
     \code{n} is the number of observations in the model. If \code{point}
     is greater than 1 it is interpreted to be the index of the breakpoint.}
 \item{order.by}{formula. A formula with a single explanatory
    variable like \code{~ x}. Then the observations in the model
    are ordered by the size of \code{x}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g. a
    time series).}
 \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which \code{gqtest}
    is called from.}
}
\details{The Goldfeld-Quandt test compares the variances of two submodels
divided by a specified breakpoint and rejects if the variances differ.

Under \eqn{H_0} the test statistic of the Goldfeld-Quandt test follows an F
distribution with the degrees of freedom as given in \code{parameter}.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.
}
\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{parameter}{degrees of freedom.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}
\references{

S.M. Goldfeld & R.E. Quandt (1965),
Some Tests for Homoskedasticity.
\emph{Journal of the American Statistical Association} \bold{60}, 539--547

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}

\seealso{\code{\link{lm}}}

\examples{
## generate a regressor
x <- rep(c(-1,1), 50)
## generate heteroskedastic and homoskedastic disturbances
err1 <- c(rnorm(50, sd=1), rnorm(50, sd=2))
err2 <- rnorm(100)
## generate a linear relationship
y1 <- 1 + x + err1
y2 <- 1 + x + err2
## perform Goldfeld-Quandt test
gqtest(y1 ~ x)
gqtest(y2 ~ x)
}
\keyword{htest}

\eof
\name{growthofmoney}
\alias{growthofmoney}
\non_function{}
\title{ Growth of Money Supply }
\usage{data(growthofmoney)}
\description{
Growth of Money Supply Data.
}
\format{
 A multivariate quarterly time series from 1970(2) to 1974(4) with variables
 \describe{
    \item{TG1.TG0}{difference of current and preceding target for the growth
 rate of the
money supply,}
    \item{AG0.TG0}{difference of actual growth rate and target growth
rate for the preceding period.}
 }
}
\source{
The data was originally studied by Hetzel (1981), the data set is given in
Krmer and Sonnberger (1986). Below we replicate a few examples from their
 book.
Some of these results differ more or less seriously and are sometimes
parameterized differently.
}
\references{

R.L. Hetzel (1981),
The Federal Reserve System and Control of the Money Supply in the 1970's.
\emph{Journal of Money, Credit and Banking} \bold{13}, 31--43

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(growthofmoney)

## page 137, fit Hetzel OLS model
## first/second line in Table 6.7

modelHetzel <- TG1.TG0 ~ AG0.TG0
lm(modelHetzel, data=growthofmoney)
dwtest(modelHetzel, data=growthofmoney)


## page 135, fit test statistics in Table 6.8
#############################################

if(require(strucchange, quietly = TRUE)) {
## Chow 1974(1)
sctest(modelHetzel, point=c(1973,4), data=growthofmoney, type="Chow") }

## RESET
reset(modelHetzel, data=growthofmoney)
reset(modelHetzel, power=2, type="regressor", data=growthofmoney)
reset(modelHetzel, type="princomp", data=growthofmoney)

## Harvey-Collier
harvtest(modelHetzel, order.by= ~ AG0.TG0, data=growthofmoney)

## Rainbow
raintest(modelHetzel, order.by = "mahalanobis", data=growthofmoney)


## Identification of outliers
#############################

## Figure 6.1
plot(modelHetzel, data=growthofmoney)
abline(v=0)
abline(h=0)
abline(coef(lm(modelHetzel, data=growthofmoney)), col=2)

## Table 6.7, last line
growthofmoney2 <- as.data.frame(growthofmoney[-c(5:6),])
lm(modelHetzel, data=growthofmoney2)
dwtest(modelHetzel, data=growthofmoney2)
}
\keyword{datasets}

\eof
\name{harvtest}
\alias{harvtest}
\title{Harvey-Collier Test}
\description{Harvey-Collier test for linearity.}

\usage{
harvtest(formula, order.by = NULL, data = list())
}

\arguments{
 \item{formula}{a symbolic description for the model to be tested.}
 \item{order.by}{formula. A formula with a single explanatory
    variable like \code{~ x}. The observations in the model
    are ordered by the size of \code{x}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g. a
    time series).}
 \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which
    \code{harvtest} is called from.}
}

\details{
The Harvey-Collier test performs a t-test (with \code{parameter} degrees of
freedom) on the recursive residuals. If the true relationship is not linear but
convex or concave the mean of the recursive residuals should differ
from 0 significantly.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.
}

\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{parameter}{degrees of freedom.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}

\references{

A. Harvey & P. Collier (1977),
Testing for Functional Misspecification in Regression Analysis.
\emph{Journal of Econometrics} \bold{6}, 103--119

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}

\seealso{\code{\link{lm}}}

\examples{
# generate a regressor and dependent variable
x <- 1:50
y1 <- 1 + x + rnorm(50)
y2 <- y1 + 0.3*x^2

## perform Harvey-Collier test
harv <- harvtest(y1 ~ x)
harv
## calculate critical value vor 0.05 level
qt(0.95, harv$parameter)
harvtest(y2 ~ x)
}
\keyword{htest}

\eof
\name{hmctest}
\alias{hmctest}
\title{Harrison-McCabe test}
\description{Harrison-McCabe test for heteroskedasticity.}
\usage{
hmctest(formula, point = 0.5, order.by = NULL, simulate.p = TRUE, nsim = 1000,
  plot = FALSE, data = list())
}

\arguments{
 \item{formula}{a symbolic description for the model to be tested}
 \item{point}{numeric. If \code{point} is smaller than 1 it is
     interpreted as percentages of data, i.e. \code{n*point} is
     taken to be the (potential) breakpoint in the variances, if
     \code{n} is the number of observations in the model. If \code{point}
     is greater than 1 it is interpreted to be the index of the breakpoint.}
 \item{order.by}{formula. A formula with a single explanatory
    variable like \code{~ x}. The observations in the model
    are ordered by the size of \code{x}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g. a
    time series).}
  \item{simulate.p}{logical. If \code{TRUE} a p value will be assessed
    by simulation, otherwise the p value is \code{NA}.}
  \item{nsim}{integer. Determins how many runs are used to simulate the
    p value.}
  \item{plot}{logical. If \code{TRUE} the test statistic for all possible
    breakpoints is plotted.}
  \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which \code{hmctest}
    is called from.}
}

\details{The Harrison-McCabe test statistic is the fraction of the residual sum
 of
squares that relates to the fraction of the data before the breakpoint. Under
 \eqn{H_0}
the test statistic should be close to the size of this fraction, e.g. in the
 default
case close to 0.5. The null hypothesis is reject if the statistic is too small.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.}

\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the simulated p-value of the test.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}

\references{

M.J. Harrison & B.P.M McCabe (1979),
A Test for Heteroscedasticity based on Ordinary Least Squares Residuals.
\emph{Journal of the American Statistical Association} \bold{74}, 494--499

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}

\seealso{\code{\link{lm}}}

\examples{
## generate a regressor
x <- rep(c(-1,1), 50)
## generate heteroskedastic and homoskedastic disturbances
err1 <- c(rnorm(50, sd=1), rnorm(50, sd=2))
err2 <- rnorm(100)
## generate a linear relationship
y1 <- 1 + x + err1
y2 <- 1 + x + err2
## perform Harrison-McCabe test
hmctest(y1 ~ x)
hmctest(y2 ~ x)
}
\keyword{htest}

\eof
\name{jocci}
\alias{fyff}
\alias{gmdc}
\alias{ip}
\alias{jocci}
\alias{lhur}
\alias{pw561}
\title{U.S. Macroeconomic Time Series}
\usage{
data(fyff)
data(gmdc)
data(ip)
data(jocci)
data(lhur)
data(pw561)
}
\description{
Several macroeconomic time series from the U.S.
}
\format{
 All data sets are multivariate monthly time series from
 1959(8) to 1993(12) (except 1993(10) for \code{jocci}) with variables
 \describe{
    \item{y}{original time series,}
    \item{dy}{transformed times series (first differences
              or log first differences),}
    \item{dy1}{transformed series at lag 1,}
    \item{dy2}{transformed series at lag 2,}
    \item{dy3}{transformed series at lag 3,}
    \item{dy4}{transformed series at lag 4,}
    \item{dy5}{transformed series at lag 5,}
    \item{dy6}{transformed series at lag 6.}
  }
}

\details{
The description from Stock & Watson (1996) for the time series
(with the transformation used):
  \describe{
    \item{fyff}{interest rate (first differences),}
    \item{gmdc}{pce, impl pr defl: pce (1987 = 100) (log first differences),}
    \item{ip}{index of industrial production (log first differences),}
    \item{jocci}{department of commerce commodity price index (log first differences),}
    \item{lhur}{unemployment rate: all workers, 16 years & over (\%, sa) (first differences),}
    \item{pw561}{producer price index: crude petroleum (82 = 100, nsa) (log first differences).}
  }
Stock & Watson (1996) fitted an AR(6) model to all transformed time series.
}

\source{
Stock & Watson (1996) study the stability of 76 macroeconomic time series, which can be obtained
from Mark W. Watson's homepage \url{http://www.wws.princeton.edu/~mwatson/}.
}
\references{

J.H. Stock & M.W. Watson (1996),
Evidence on Structural Instability in Macroeconomic Time Series Relations.
\emph{Journal of Business & Economic Statistics} \bold{14}, 11--30.

}
\examples{
data(jocci)

dwtest(dy ~ 1, data = jocci)
bgtest(dy ~ 1, data = jocci)
ar6.model <- dy ~ dy1 + dy2 + dy3 + dy4 + dy5 +dy6
bgtest(ar6.model, data = jocci)

var.model <- ~ I(dy1^2) + I(dy2^2) + I(dy3^2) + I(dy4^2) + I(dy5^2) + I(dy6^2)
bptest(ar6.model, var.model, data = jocci)
}
\keyword{datasets}

\eof
\name{moneydemand}
\alias{moneydemand}
\non_function{}
\title{ Demand for Money  }
\usage{data(moneydemand)}
\description{
Money Demand Data.
}
\format{
 A multivariate yearly time series from 1879 to 1974 with variables
 \describe{
    \item{logM}{logarithm of quantity of money,}
    \item{logYp}{logarithm of real permanent income,}
    \item{Rs}{short term interest rate,}
    \item{Rm}{rate of return on money,}
    \item{Rl}{not documented in the sources,}
    \item{logSpp}{logarithm of an operational measure of the variability of
the rate of price changes.}
 }
}
\source{
The data was originally studied by Allen (1982), the data set is given in
Krmer and Sonnberger (1986). Below we replicate a few examples 
from the book.
Some of these results differ more or less seriously and are sometimes
parameterized differently.

}
\references{

S.D. Allen (1982),
Kleins's Price Variability Terms in the U.S. Demand for Money.
\emph{Journal of Money, Credit and Banking} \bold{14}, 525--530

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(moneydemand)
moneydemand <- window(moneydemand, start=1880, end=1972)

## page 125, fit Allen OLS model (and Durbin-Watson test),
## last line in Table 6.1

modelAllen <- logM ~ logYp + Rs + Rl + logSpp
lm(modelAllen, data = moneydemand)
dwtest(modelAllen, data = moneydemand)

## page 127, fit test statistics in Table 6.1 c)
################################################

## Breusch-Pagan
bptest(modelAllen, studentize = FALSE, data = moneydemand)
bptest(modelAllen, studentize = TRUE, data = moneydemand)

## RESET
reset(modelAllen, data = moneydemand)
reset(modelAllen, power = 2, type = "regressor", data = moneydemand)
reset(modelAllen, type = "princomp", data = moneydemand)

## Harvey-Collier tests (up to sign of the test statistic)
harvtest(modelAllen, order.by = ~logYp, data = moneydemand)
harvtest(modelAllen, order.by = ~Rs, data = moneydemand)
harvtest(modelAllen, order.by = ~Rl, data = moneydemand)
harvtest(modelAllen, order.by = ~logSpp, data = moneydemand)

## Rainbow test
raintest(modelAllen, order.by = "mahalanobis", data = moneydemand)


if(require(strucchange, quietly = TRUE)) {
## Chow (1913)
sctest(modelAllen, point=c(1913,1), data = moneydemand, type = "Chow") }

if(require(strucchange, quietly = TRUE)) {
## Fluctuation
sctest(modelAllen, type = "fluctuation", rescale = FALSE, data = moneydemand)}
}
\keyword{datasets}

\eof
\name{raintest}
\alias{raintest}
\title{Rainbow Test}

\description{Rainbow test for linearity.}

\usage{
raintest(formula, fraction = 0.5, order.by = NULL, center = NULL,
   data=list())
}

\arguments{
 \item{formula}{a symbolic description for the model to be tested.}
 \item{fraction}{numeric. The percentage of observations in the subset is
   determined by
   \code{fraction*n} if \code{n} is the number of observations
   in the model.}
 \item{order.by}{either a formula or a string. A formula with a single
    explanatory variable like \code{~ x}. The observations in the model
    are ordered by the size of \code{x}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g. a
    time series). If set to \code{"mahalanobis"} then the observations
    are ordered by their Mahalanobis distance of the data.}
 \item{center}{numeric. If \code{center} is smaller than 1 it is
     interpreted as percentages of data, i.e. the subset is chosen that
     \code{n*fraction} observations are around observation number
     \code{n*center}. If \code{center} is greater than 1 it is interpreted to be
     the index of the center of the subset. By default \code{center} is 0.5.

     If the Mahalanobis distance is chosen \code{center} is taken to be
     the mean regressor, but can be specified to be a \code{k}-dimensional
     vector if \code{k} is the number of regressors and should be in the
     range of the respective regressors.}
\item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which
    \code{raintest} is called from.}
}

\details{The basic idea of the Rainbow test is that even if the true
 relationship is
non-linear, a good linear fit can be achieved on a subsample in the "middle" of
the data. The null hypothesis is rejected whenever the overall fit is
significantly inferious to the fit of the subsample. The test statistic under
\eqn{H_0} follows an F distribution with \code{parameter} degrees of freedom.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.
}

\value{
 A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{parameter}{degrees of freedom.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{data.name}{a character string giving the name(s) of the data.}
}

\references{

J.M. Utts (1982),
The Rainbow Test for Lack of Fit in Regression.
\emph{Communications in Statistics - Theory and Methods} \bold{11},
1801--1815

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}

\seealso{\code{\link{lm}}}

\examples{
x <- c(1:30)
y <- x^2 + rnorm(30,0,2)
rain <- raintest(y ~ x)
rain
## critical value
qf(0.95, rain$parameter[1], rain$parameter[2])
}

\keyword{htest}

\eof
\name{reset}
\alias{reset}
\title{RESET Test}

\description{Ramsey's RESET test for functional relation.}

\usage{
reset(formula, power = 2:3, type = c("fitted", "regressor",
  "princomp"), data = list())
}

\arguments{
 \item{formula}{a symbolic description for the model to be tested.}
 \item{power}{integers. A vector of positive integers indicating
    the powers of the variables that should be included. By default
    it is tested for a quadratic or cubic influence of the fitted response.}
 \item{type}{a string indicating whether powers of the fitted response, the
    regressor variables (factors are left out) or the first principal
    component of the regressor matrix should be included in the
    extended model.}
 \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment which \code{reset}
    is called from.}
}

\details{
RESET test is popular means of diagnostic for correctness of functional
form. The basic assumption is that under the alternative, the model can be
written by the regression
\eqn{ y = X\beta + Z\gamma + u}{y=X * beta + Z * gamma}.
\code{Z} is generated by taking powers either of the fitted response, the
regressor variables or the first principal component of \code{X}. A standard
F-Test is then applied to determin whether these additional variables have
significant influence. The test statistic under \eqn{H_0} follows an F
distribution with \code{parameter} degrees of freedom.

Examples can not only be found on this page, but also on the help pages of the
data sets \code{\link{bondyield}}, \code{\link{currencysubstitution}},
\code{\link{growthofmoney}}, \code{\link{moneydemand}},
 \code{\link{unemployment}},
\code{\link{wages}}.
}

\value{An object of class \code{"htest"} containing:
\itemize
\item{statistic}{the test statistic.}
\item{p.value}{the corresponding p-value.}
\item{parameter}{degrees of freedom.}
\item{method}{a character string with the method used.}
\item{data.name}{a character string with the data name.}
}
\references{

J.B. Ramsey (1969),
Tests for Specification Error in Classical Linear Least Squares Regression
Analysis.
\emph{Journal of the Royal Statistical Society, Series B} \bold{31},
350--371

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}



\seealso{\code{\link{lm}}}

\examples{
x <- c(1:30)
y1 <- 1 + x + x^2 + rnorm(30)
y2 <- 1 + x + rnorm(30)
reset(y1 ~ x , power=2, type="regressor")
reset(y2 ~ x , power=2, type="regressor")
}

\keyword{htest}

\eof
\name{unemployment}
\alias{unemployment}
\non_function{}
\title{Unemployment Data}
\usage{data(unemployment)}
\description{
Unemployment Data.
}
\format{
 A multivariate yearly time series from 1890 to 1979 with variables
 \describe{
    \item{UN}{unemployment rate,}
    \item{m}{broad money supply,}
    \item{p}{implicit deflator of Gross National Product,}
    \item{G}{real purchases of goods and services,}
    \item{x}{real exports.}
  }

}
\source{
The data was originally studied by Rea (1983), the data set is given in
Krmer and Sonnberger (1986). Below we replicate a few examples 
from their book. Some of these results differ more or less seriously and are
 sometimes
parameterized differently.
}
\references{

J.D. Rea (1983),
The Explanatory Power of Alternative Theories of Inflation and Unemployment,
1895-1979.
\emph{Review of Economics and Statistics} \bold{65}, 183--195

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(unemployment)

## data transformation
myunemployment <- window(unemployment, start=1895, end=1956)
time <- 6:67

## page 144, fit Rea OLS model
## last line in Table 6.12

modelRea <- UN ~ log(m/p) + log(G) + log(x) + time
lm(modelRea, data = myunemployment)
## coefficients of logged variables differ by factor 100

## page 143, fit test statistics in table 6.11
##############################################

if(require(strucchange, quietly = TRUE)) {
## Chow 1941
sctest(modelRea, point=c(1940,1), data=myunemployment, type="Chow") }

## Breusch-Pagan
bptest(modelRea, data=myunemployment, studentize=FALSE)
bptest(modelRea, data=myunemployment)

## RESET (a)-(b)
reset(modelRea, data=myunemployment)
reset(modelRea, power=2, type="regressor", data=myunemployment)

## Harvey-Collier
harvtest(modelRea, order.by = ~ log(m/p), data=myunemployment)
harvtest(modelRea, order.by = ~ log(G), data=myunemployment)
harvtest(modelRea, order.by = ~ log(x), data=myunemployment)
harvtest(modelRea, data=myunemployment)

## Rainbow
raintest(modelRea, order.by = "mahalanobis", data=myunemployment)
}
\keyword{datasets}

\eof
\name{valueofstocks}
\alias{valueofstocks}
\non_function{}
\title{ Value of Stocks  }
\usage{data(valueofstocks)}
\description{
Value of Stocks Data
}
\format{
 A multivariate quarterly time series from 1960(1) to 1977(3) with variables
 \describe{
    \item{VST}{value of stocks,}
    \item{MB}{monetary base,}
    \item{RTPD}{dollar rent on producer durables,}
    \item{RTPS}{dollar rent on producer structures,}
    \item{XBC}{production capacity for business output.}
  }

}
\source{The data was originally studied by Woglom (1981), the data set is given
 in
Krmer and Sonnberger (1986).
}
\references{

G. Woglom (1981),
A Reexamination of the Role of Stocks in the Consumption Function and the
Transmission Mechanism.
\emph{Journal of Money, Credit and Banking} \bold{13}, 215--220

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model Under Test}. Heidelberg: Physica


}
\examples{
data(valueofstocks)
lm(log(VST) ~., data=valueofstocks)
}
\keyword{datasets}

\eof
\name{wages}
\alias{wages}
\non_function{}
\title{ Wages  }
\usage{data(wages)}
\description{
Wages Data.
}
\format{
 A multivariate yearly time series from 1960 to 1979 with variables
 \describe{
    \item{w}{wages,}
    \item{CPI}{consumer price index,}
    \item{u}{unemployment,}
    \item{mw}{minimum wage.}
  }

}
\source{The data was originally studied by Nicols (1983), the data set is given
in Krmer and Sonnberger (1986). Below we replicate a few examples from their
 book.
Some of these results differ more or less seriously and are sometimes
parameterized differently.
}
\references{

D.A. Nicols (1983),
Macroeconomic Determinants of Wage Adjustments in White Collar Occupations.
\emph{Review of Economics and Statistics} \bold{65}, 203--213

W. Krmer & H. Sonnberger (1986),
\emph{The Linear Regression Model under Test}. Heidelberg: Physica


}
\examples{
data(wages)

## data transformation to include lagged series
mywages <- cbind(wages, lag(wages[,2], k = -1), lag(wages[,2], k = -2))
colnames(mywages) <- c(colnames(wages), "CPI2", "CPI3")
mywages <- window(mywages, start=1962, end=1979)

## page 142, fit Nichols OLS model
## equation (6.10)

modelNichols <- w ~ CPI + CPI2 + CPI3 + u + mw
lm(modelNichols, data = mywages)

## page 143, fit test statistics in table 6.11
##############################################

if(require(strucchange, quietly = TRUE)) {
## Chow 1972
sctest(modelNichols, point=c(1971,1), data=mywages, type="Chow") }

## Breusch-Pagan
bptest(modelNichols, data=mywages, studentize=FALSE)
bptest(modelNichols, data=mywages)

## RESET (a)-(b)
reset(modelNichols, data=mywages)
reset(modelNichols, power=2, type="regressor", data=mywages)

## Harvey-Collier
harvtest(modelNichols, order.by = ~ CPI, data=mywages)
harvtest(modelNichols, order.by = ~ CPI2, data=mywages)
harvtest(modelNichols, order.by = ~ CPI3, data=mywages)
harvtest(modelNichols, order.by = ~ u, data=mywages)

## Rainbow
raintest(modelNichols, order.by = "mahalanobis", data=mywages)
}
\keyword{datasets}

\eof
