\name{lp}
\alias{lp}
\title{Linear and Integer Programming}
\description{Interface to lp\_solve linear/integer programming system}
\usage{
lp (direction = "min", objective.in, const.mat, const.dir, const.rhs,
	transpose.constraints = TRUE, int.vec)
}
\arguments{
\item{direction}{Character string giving direction of optimization:
"min" (default) or "max."}
\item{objective.in}{Numeric vector of coefficients of objective function}
\item{const.mat}{Matrix of numeric constraint coefficients, one row 
per constraint, one column per variable (unless transpose.constraints = 
FALSE; see below).}
\item{const.dir}{Vector of character strings giving the direction of 
the constraint: each value should be one of "<," "<=," "=," "==," ">," or ">=."}
\item{const.rhs}{Vector of numeric values for the right-hand sides of 
the constraints.}
\item{transpose.constraints}{By default each constraint occupies a row 
of const.mat, and that matrix needs to be transposed before being passed 
to the optimizing code.  For very large constraint matrices it may be wiser 
to construct the constraints in a matrix column-by-column. In that case set 
transpose.constraints to FALSE.}
\item{int.vec}{Numeric vector giving the indices of variables that are 
required to be integer. The length of this vector will therefore be the 
number of integer variables.}
}
\details{
This function calls the lp\_solve 4.0 solver. That system has many options not
supported here. The current version is maintained at
\url{ftp://ftp.es.ele.tue.nl/pub/lp_solve}
}
\value{
An lp object. See \code{link{lp.object}} for details.
}
\author{Sam Buttrey, \email{buttrey@nps.navy.mil}}
\seealso{\code{\link{lp.assign}}, \code{\link{lp.transport}}}
\examples{
#
# Set up problem: maximize
#   x1 + 9 x2 +   x3 subject to
#   x1 + 2 x2 + 3 x3  <= 9
# 3 x1 + 2 x2 + 2 x3 <= 15
#
f.obj <- c(1, 9, 3)
f.con <- matrix (c(1, 2, 3, 3, 2, 2), nrow=2, byrow=TRUE)
f.dir <- c("<=", "<=")
f.rhs <- c(9, 15)
#
# Now run.
#
lp ("max", f.obj, f.con, f.dir, f.rhs)
\dontrun{Success: the objective function is 40.5}
lp ("max", f.obj, f.con, f.dir, f.rhs)$solution
\dontrun{[1] 0.0 4.5 0.0}
#
# Run again, this time requiring that all three variables be integer
#
lp ("max", f.obj, f.con, f.dir, f.rhs, int.vec=1:3)
\dontrun{Success: the objective function is 37}
lp ("max", f.obj, f.con, f.dir, f.rhs, int.vec=1:3)$solution
\dontrun{[1] 1 4 0}
}
\keyword{optimize}

\eof
\name{lp.assign}
\alias{lp.assign}
\title{Integer Programming for the Assignment Problem}
\description{Interface to lp\_solve linear/integer programming
system specifically for solving assignment problems}
\usage{
lp.assign (cost.mat)
}
\arguments{
\item{cost.mat}{Matrix of costs: the ij-th element is the cost of 
assigning source i to destination j.}
}
\details{
This is a particular integer programming problem. All the decision variables
are assumed to be integers; each row has the constraint that its entries 
must add up to 1 (so that there is one 1 and the remaining entries are 0) 
and each column has the same constraint. This is assumed to be a
minimization problem.
}
\value{
An \code{\link{lp}} object. See documentation for details.
}
\author{Sam Buttrey, \email{buttrey@nps.navy.mil}}
\seealso{\code{\link{lp}}, \code{\link{lp.transport}}}
\examples{
assign.costs <- matrix (c(2, 7, 7, 2, 7, 7, 3, 2, 7, 2, 8, 10, 1, 9, 8, 2), 4, 4)
\dontrun{
> assign.costs
     [,1] [,2] [,3] [,4]
[1,]    2    7    7    1
[2,]    7    7    2    9
[3,]    7    3    8    8
[4,]    2    2   10    2
}
lp.assign (assign.costs)
\dontrun{Success: the objective function is 8}
lp.assign (assign.costs)$solution
\dontrun{
     [,1] [,2] [,3] [,4]
[1,]    0    0    0    1
[2,]    0    0    1    0
[3,]    0    1    0    0
[4,]    1    0    0    0
}
}
\keyword{optimize}



\eof
\name{lp.object}
\alias{lp.object}
\title{LP (linear programming) object}
\description{Structure of lp object}
\value{
An lp.object is a list containing the following elements:
\item{direction}{Optimization direction, as entered}
\item{x.count}{Number of variables in objective function}
\item{objective}{Vector of objective function coefficients, as entered}
\item{const.count}{Number of constraints entered}
\item{constraints}{Constraint matrix, as entered}
\item{int.count}{Number of integer variables}
\item{int.vec}{Vector of integer variables' indices, as entered}
\item{objval}{{Value of objective function at optimum}}
\item{solution}{Vector of optimal coefficients}
\item{status}{Numeric indicator: 0 = success, 2 = no feasible solution}
}
\author{Sam Buttrey, \email{buttrey@nps.navy.mil}}
\seealso{\code{\link{lp}}, \code{\link{lp.assign}}, \code{\link{lp.transport}}}
\keyword{optimize}

\eof
\name{lp.transport}
\alias{lp.transport}
\title{Integer Programming for the Transportation Problem}
\description{Interface to lp\_solve linear/integer programming
system specifically for solving transportation problems}
\usage{
lp.transport (cost.mat, row.signs, row.rhs, col.signs, col.rhs)
}
\arguments{
\item{cost.mat}{Matrix of costs; ij-th element is the cost of transporting 
one item from source i to destination j.}
\item{row.signs}{Vector of character strings giving the direction of the 
row constraints: each value should be one of "<," "<=," "=," "==," ">," 
or ">=."}
\item{row.rhs}{Vector of numeric values for the right-hand sides of the 
row constraints.}
\item{col.signs}{Vector of character strings giving the direction of the 
column constraints: each value should be one of "<," "<=," "=," "==," ">," 
or ">=."}
\item{col.rhs}{Vector of numeric values for the right-hand sides of the 
column constraints.}
}
\details{
This is a particular integer programming problem. All the decision variables
are assumed to be integers, and there is one constraint per row and one per
column (and no others). This is assumed to be a minimization problem.
}
\value{
An \code{\link{lp}} object. See documentation for details.
}
\references{Example problem from Bronson (1981), \emph{Operations Research},
Scahum's Outline Series, McGraw-Hill.}
\author{Sam Buttrey, \email{buttrey@nps.navy.mil}}
\seealso{\code{\link{lp.assign}}, \code{\link{lp.transport}}}

\examples{
#
# Transportation problem, Bronson, problem 9.1, p. 86
#
# Set up cost matrix
#
costs <- matrix (10000, 8, 5); costs[4,1] <- costs[-4,5] <- 0
costs[1,2] <- costs[2,3] <- costs[3,4] <- 7; costs[1,3] <- costs[2,4] <- 7.7
costs[5,1] <- costs[7,3] <- 8; costs[1,4] <- 8.4; costs[6,2] <- 9
costs[8,4] <- 10; costs[4,2:4] <- c(.7, 1.4, 2.1)
#
# Set up constraint signs and right-hand sides.
#
row.signs <- rep ("<", 8)
row.rhs <- c(200, 300, 350, 200, 100, 50, 100, 150)
col.signs <- rep (">", 5)
col.rhs <- c(250, 100, 400, 500, 200)
#
# Run
#
lp.transport (costs, row.signs, row.rhs, col.signs, col.rhs)
\dontrun{Success: the objective function is 7790}
lp.transport (costs, row.signs, row.rhs, col.signs, col.rhs)$solution
\dontrun{
     [,1] [,2] [,3] [,4] [,5]
[1,]    0  100    0  100    0
[2,]    0    0  300    0    0
[3,]    0    0    0  350    0
[4,]  200    0    0    0    0
[5,]   50    0    0    0   50
[6,]    0    0    0    0   50
[7,]    0    0  100    0    0
[8,]    0    0    0   50  100
}
}
\keyword{optimize}



\eof
\name{print.lp}
\alias{print.lp}
\title{Print an lp object}
\description{Print method for lp objects}
\usage{
print.lp (x, ...)
}
\arguments{
\item{x}{List with items named \code{objval} and \code{status}.
Normally this will have been called by \code{\link{lp}},
\code{\link{lp.assign}}, or \code{\link{lp.transport}}.}
\item{...}{Other arguments, all currently ignored}
}
\details{
This function prints the objective function value, together with the
word "Success" if the operation is successful, or an indication of the
error if not.}
\value{
None
}
\author{Sam Buttrey, \email{buttrey@nps.navy.mil}}
\seealso{\code{\link{lp}}, \code{\link{lp.assign}}, \code{\link{lp.transport}}}
\keyword{optimize}

\eof
