\name{Defaults.Mclust}
\alias{Defaults.Mclust}
\alias{.Mclust}
\title{
  List of values controlling defaults for some MCLUST functions.
}
\description{
  A named list of values including tolerances for singularity and
  convergence assessment, and an enumeration of models used as defaults
  in MCLUST functions. 
}
\value{
  A list with the following components:
  
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due
    to computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is the relative machine precision
    \code{.Machine\$double.eps}, which is approximately \$2e-16\$ on
    IEEE-compliant machines. 
  }
  \item{tol}{
    A vector of length two giving relative convergence tolerances for the 
    loglikelihood and for parameter convergence in the inner loop for models
    with iterative M-step ("VEI", "VEE", "VVE", "VEV"), respectively.
    The default is \code{c(1.e-5,1.e-5)}.
  }
  \item{itmax}{
    A vector of length two giving integer limits on the number of EM
    iterations and on the number of iterations in the inner loop for
    models with iterative M-step ("VEI", "VEE", "VVE", "VEV"),
    respectively. The default is \code{c(Inf,Inf)} allowing termination
    to be completely governed by \code{tol}.
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model. Default: \code{equalPro = FALSE}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered. Default: \code{warnSingular = TRUE}.
  }
  \item{emModelNames}{
    A vector of character strings indicating the models to be used for
    multivariate data in the functions such as \code{EMclust}
    and \code{mclustDAtrain} that involve multiple models. The default
    is all of the multivariate models available in MCLUST: \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation 
  }
  \item{hcModelName}{
    A vector of two character strings 
    giving the name of the model to be used in the hierarchical
    clustering phase for univariate and multivariate data, respectively,
    in \code{EMclust} and \code{EMclustN}. The default is
    \code{c("V","VVV")}, giving the unconstrained model in each case. 
  }
  \item{symbols}{A vector whose entries are either integers
    corresponding to graphics symbols or single characters for plotting
    for classifications. Classes are assigned symbols in the given
    order. 
  }
}
\details{
  A function \code{mclustOptions} is supplied for assigning values to
  the \code{.Mclust} list. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association}. 
  See \url{http://www.stat.washington.edu/tech.reports} (No. 380, 2000). 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/tech.reports}.
}
\seealso{
  \code{\link{mclustOptions}},
  \code{\link{EMclust}},
  \code{\link{mclustDAtrain}},
  \code{\link{em}},
  \code{\link{me}},
  \code{\link{estep}},
  \code{\link{mstep}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
odd <- seq(1, 2*n, 2)
train <- mclustDAtrain(x[odd, ], labels = xclass[odd]) ## training step
even <- odd + 1
test <- mclustDAtest(x[even, ], train) ## compute model densities

data(iris)
irisMatrix <- iris[,1:4]
irisClass <- iris[,5]

.Mclust
.Mclust <- mclustOptions(tol = 1.e-6, emModelNames = c("VII", "VVI", "VVV"))
.Mclust
irisBic <- EMclust(irisMatrix)
summary(irisBic, irisMatrix)
.Mclust <- mclustOptions() # restore defaults
.Mclust
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{EMclust}
\alias{EMclust}
\alias{print.EMclust}
\alias{plot.EMclust}
\title{
  BIC for Model-Based Clustering
}
\description{
  BIC for EM initialized by hierarchical clustering
  for parameterized Gaussian mixture models.
}
\usage{
EMclust(data, G, emModelNames, hcPairs, subset, eps, tol, itmax, equalPro,
        warnSingular, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{G}{
    An integer vector specifying the numbers of mixture components
    (clusters) for which the BIC is to be calculated. The default is
    \code{1:9}. 
  }
  \item{emModelNames}{
    A vector of character strings indicating the models to be fitted 
    in the EM phase of clustering. Possible models: \cr\cr    
    "E" for spherical, equal variance (one-dimensional) \cr
    "V" for spherical, variable variance (one-dimensional) \cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr
    The default is \code{.Mclust\$emModelNames}.
  }
  \item{hcPairs}{
    A matrix of merge pairs for hierarchical clustering such as produced
    by function \code{hc}. The default is to compute a hierarchical
    clustering tree by applying function \code{hc} with
    \code{modelName = .Mclust\$hcModelName[1]} to univariate data and
    \code{modelName = .Mclust\$hcModelName[2]} to multivariate data or a
    subset as indicated by the \code{subset} argument. The hierarchical
    clustering results are used as starting values for EM.  
  }
  \item{subset}{
    A logical or numeric vector specifying the indices of a subset of the data
    to be used in the initial hierarchical clustering phase.
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due
    to computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is \code{.Mclust\$eps}. 
  }
  \item{tol}{
    A scalar tolerance for relative convergence of the loglikelihood. 
    The default is \code{.Mclust\$tol}.
  }
  \item{itmax}{
    An integer limit on the number of EM iterations. 
    The default is \code{.Mclust\$itmax}.
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model. The default is \code{.Mclust\$equalPro}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{warnSingular=FALSE}.
  }
  \item{\dots }{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  Bayesian Information Criterion for the specified mixture models
  numbers of clusters. Auxiliary information returned as attributes.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611:631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{summary.EMclust}}, 
  \code{\link{EMclustN}}, 
  \code{\link{hc}},
  \code{\link{me}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

irisBic <- EMclust(irisMatrix)
irisBic
plot(irisBic)

irisBic <- EMclust(irisMatrix, subset = sample(1:nrow(irisMatrix), 100))
irisBic
plot(irisBic)
}
\keyword{cluster}
% docclass is function

\eof
\name{EMclustN}
\alias{EMclustN}
\alias{print.EMclustN}
\alias{plot.EMclustN}
\title{
  BIC for Model-Based Clustering with Poisson Noise
}
\description{
  BIC for EM initialized by hierarchical clustering
  for parameterized Gaussian mixture models with Poisson noise.
}
\usage{
EMclustN(data, G, emModelNames, noise, hcPairs, eps, tol, itmax, 
         equalPro, warnSingular=FALSE, Vinv, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{G}{
    An integer vector specifying the numbers of MVN (Gaussian) mixture
    components (clusters) for which the BIC is to be calculated. The
    default is \code{0:9} where \code{0} indicates only a noise
    component. 
  }
  \item{emModelNames}{
    A vector of character strings indicating the models to be fitted 
    in the EM phase of clustering. Possible models: \cr\cr    
    "E" for spherical, equal variance (one-dimensional) \cr
    "V" for spherical, variable variance (one-dimensional) \cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr    
    The default is \code{.Mclust\$emModelNames}.
  }
  \item{noise}{
    A logical or numeric vector indicating whether or not observations
    are initially estimated to noise in the data. If there is no noise
    \code{EMclust} should be use rather than \code{EMclustN}.
  }
  \item{hcPairs}{
    A matrix of merge pairs for hierarchical clustering such as produced
    by function \code{hc}. The default is to compute a hierarchical
    clustering tree by applying function \code{hc} with
    \code{modelName = .Mclust\$hcModelName[1]} to univariate data and
    \code{modelName = .Mclust\$hcModelName[2]} to multivariate data or a
    subset as indicated by the \code{subset} argument. The hierarchical
    clustering results are used as starting values for EM.  
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due to
    computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is \code{.Mclust\$eps}. 
  }
  \item{tol}{
    A scalar tolerance for relative convergence of the loglikelihood. 
    The default is \code{.Mclust\$tol}.
  }
  \item{itmax}{
    An integer limit on the number of EM iterations. 
    The default is \code{.Mclust\$itmax}.
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model. The default is \code{.Mclust\$equalPro}.
  }
  \item{Vinv}{
    An estimate of the reciprocal hypervolume of the data region.
    The default is determined by applying function 
    \code{hypvol} to the data.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{warnSingular=FALSE}.
  }
  \item{\dots }{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  Bayesian Information Criterion for the specified mixture models
  numbers of clusters. Auxiliary information returned as attributes.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{summary.EMclustN}}, 
  \code{\link{EMclust}}, 
  \code{\link{hc}},
  \code{\link{me}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

b <- apply( irisMatrix, 2, range)
n <- 450
set.seed(0)
poissonNoise <- apply(b, 2, function(x, n=n) 
                      runif(n, min = x[1]-0.1, max = x[2]+.1), n = n)
set.seed(0)
noiseInit <- sample(c(TRUE,FALSE),size=150+450,replace=TRUE,prob=c(3,1))
Bic <-  EMclustN(data=rbind(irisMatrix, poissonNoise), noise = noiseInit)
Bic
plot(Bic)
}
\keyword{cluster}
% docclass is function

\eof
\name{Mclust}
\alias{Mclust}
\alias{print.Mclust}
\title{
  Model-Based Clustering
}
\description{
  Clustering via EM initialized by hierarchical clustering for
  parameterized Gaussian mixture models. The number of clusters and the
  clustering model is chosen to maximize the BIC. 
}
\usage{
Mclust(data, minG, maxG)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{minG}{
    An integer vector specifying the minimum number of mixture components 
    (clusters) to be considered.
    The default is \code{1} component.
  }
  \item{maxG}{
    An integer vector specifying the maximum number of mixture components 
    (clusters) to be considered.
    The default is \code{9} components.
  }
}
\value{
  A list representing the best model (according to BIC) for the given range
  of numbers of clusters. The following components are included: 
  \item{BIC}{ 
    A matrix giving the BIC value for each model (rows) and number of
    clusters (columns).
  }
  \item{bic}{ 
    A scalar giving the optimal BIC value.
  }
  \item{modelName}{ 
    The MCLUST name for the best model according to BIC. 
  }
  \item{classification}{ 
    The classification corresponding to the optimal BIC value. 
  }
  \item{uncertainty}{ 
    The  uncertainty in the classification corresponding to the
    optimal BIC value.
  }
  \item{mu}{ 
    For multidimensional models, a matrix whose columns are the means
    of each group in the best model. For one-dimensional models, a
    vector whose entries are the means for each group in the best
    model. 
  }
  \item{sigma}{ 
    For multidimensional models, a three dimensional array in which
    \code{sigma[,,k]} gives the covariance for the \emph{k}th group in
    the best model. For one-dimensional models, either a scalar giving
    a common variance for the groups or a vector whose entries are the
    variances for each group in the best model.
  }
  \item{pro}{ 
    The mixing probabilities for each component in the best model.
  }
  \item{z}{ 
    A matrix whose \emph{[i,k]}th entry is the probability that
    observation \emph{i} belongs to the \emph{k} component in the
    model. The optimal classification is derived from this, chosing
    the class to be the one giving the maximum probability.
  }
  \item{loglik}{ 
    The log likelihood for the data under the best model.
  }
}
\section{Details}{
  The following models are compared in \code{Mclust}: \cr\cr 
  "E" for spherical, equal variance (one-dimensional) \cr
  "V" for spherical, variable variance (one-dimensional) \cr\cr
  "EII": spherical, equal volume \cr
  "VII": spherical, unequal volume \cr
  "EEI": diagonal, equal volume, equal shape \cr
  "VVI": diagonal, varying volume, varying shape \cr
  "EEE": ellipsoidal, equal volume, shape, and orientation \cr
  "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr
  \code{Mclust} is intended to combine \code{EMclust} and its
  \code{summary} in a simiplified one-step model-based clustering
  function. The latter provide more flexibility including choice of
  models. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
 
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{plot.Mclust}},
  \code{\link{EMclust}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]
irisMclust <- Mclust(irisMatrix)

\dontrun{plot(irisMclust,irisMatrix)}
}
\keyword{cluster}
% docclass is function

\eof
\name{bic}
\alias{bic}
\title{
  BIC for Parameterized MVN Mixture Models
}
\description{
  Compute the BIC (Bayesian Information Criterion) for parameterized
  mixture models  given the loglikelihood, the dimension of the data,
  and number of mixture components in the model. 
}
\usage{
bic(modelName, loglik, n, d, G, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model. Possible models: \cr \cr
    "E" for spherical, equal variance  (one-dimensional)  \cr
    "V" for spherical, variable variance (one-dimensional)  \cr
    "EII": spherical, equal volume  \cr
    "VII": spherical, unequal volume  \cr
    "EEI": diagonal, equal volume, equal shape  \cr
    "VEI": diagonal, varying volume, equal shape  \cr
    "EVI": diagonal, equal volume, varying shape  \cr
    "VVI": diagonal, varying volume, varying shape  \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation  \cr
    "EEV": ellipsoidal, equal volume and equal shape \cr
    "VEV": ellipsoidal, equal shape  \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation 
  }
  \item{loglik}{
    The loglikelihood for a data set with respect to the MVN mixture model
    specified in the \code{modelName} argument.
  }
  \item{n}{
    The number of observations in the data use to compute
    \code{loglik}.
  }
  \item{d}{
    The dimension of the data used to compute \code{loglik}.
  }
  \item{G}{
    The number of components in the MVN mixture model used to compute
    \code{loglik}.
  }
  \item{\dots}{
    Arguments for diagonal-specific methods, in particular
    \describe{
      \item{equalPro}{
	A logical variable indicating whether or not the components in the
	model are assumed to be present in equal proportion. The default is
	\code{.Mclust\$equalPro}. 
      }
      \item{noise}{
	A logical variable indicating whether or not the model includes and
	optional Poisson noise component. The default is to assume that the
	model does not include a noise component.
      }
    }
  }
}
\value{
  The BIC or Bayesian Information Criterion for the given input arguments.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611:631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{bicE}}, \ldots,
  \code{\link{bicVVV}},
  \code{\link{EMclust}},
  \code{\link{estep}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}.
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

n <- nrow(irisMatrix)
d <- ncol(irisMatrix)
G <- 3

emEst <- me(modelName="VVI",data=irisMatrix, unmap(irisClass))
names(emEst)

args(bic)
bic(modelName="VVI",loglik=emEst$loglik,n=n,d=d,G=G)
do.call("bic", emEst)    ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{bicE}
\alias{bicE}
\alias{bicV}
\alias{bicEII}
\alias{bicVII}
\alias{bicEEI}
\alias{bicVEI}
\alias{bicEVI}
\alias{bicVVI}
\alias{bicEEE}
\alias{bicEEV}
\alias{bicVEV}
\alias{bicVVV}
\title{
  BIC for a Parameterized MVN Mixture Model
}
\description{
  Compute the BIC (Bayesian Information Criterion) for a parameterized
  mixture model given the loglikelihood, the dimension of the data, 
  and number of mixture components in the model.
}
\usage{
bicE(loglik, n, G, equalPro, noise = FALSE, \dots)
bicV(loglik, n, G, equalPro, noise = FALSE, \dots)
bicEII(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicVII(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicEEI(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicVEI(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicEVI(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicVVI(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicEEE(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicEEV(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicVEV(loglik, n, d, G, equalPro, noise = FALSE, \dots)
bicVVV(loglik, n, d, G, equalPro, noise = FALSE, \dots)
}
\arguments{
  \item{loglik}{
    The loglikelihood for a data set with respect to the MVN mixture model.
  }
  \item{n}{
    The number of observations in the data used to compute
    \code{loglik}.
  }
  \item{d}{
    The dimension of the data used to compute \code{loglik}.
  }
  \item{G}{
    The number of components in the MVN mixture model used to compute
    \code{loglik}.
  }
  \item{equalPro}{
    A logical variable indicating whether or not the components in the
    model are assumed to be present in equal proportion. The default is
    \code{.Mclust\$equalPro}.
  }
  \item{noise}{
    A logical variable indicating whether or not the model includes and
    optional Poisson noise component. The default is to assume that the
    model does not include a noise component. 
  }
  \item{\dots}{
    Catch unused arguments from a \code{do.call} call.
  }
}
\value{
  The BIC or Bayesian Information Criterion for the MVN mixture
  model and data set corresponding to the input arguments.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611:631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{bic}},
  \code{\link{EMclust}},
  \code{\link{estepE}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

n <- nrow(irisMatrix)
d <- ncol(irisMatrix)
G <- 3

emEst <- meVVI(data=irisMatrix, unmap(irisClass))
names(emEst)

bicVVI(loglik=emEst$loglik, n=n, d=d, G=G)
do.call("bicVVI", emEst)  ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{bicEMtrain}
\alias{bicEMtrain}
\title{Select models in discriminant analysis using BIC}
\description{
For the ten available discriminant models the BIC is calulated. The
models for one-dimensional data are "E" and "V"; for higher dimensions
they are "EII", "VII", "EEI", "VEI", "EVI", "VVI", "EEE", "EEV", "VEV"
and "VVV". This function is much faster than \code{cv1EMtrain}.
}
\usage{
bicEMtrain(data, labels, modelNames)
}
\arguments{
  \item{data}{A data matrix}
  \item{labels}{Labels for each row in the data matrix}
  \item{modelNames}{Vector of model names that should be tested.}
}
\value{
  Returns a vector where each element is the BIC for the
  corresponding model.
}
\author{C. Fraley}
\seealso{\code{\link{cv1EMtrain}}}
\examples{
data(lansing)
odd <- seq(from=1, to=nrow(lansing), by=2)
round(bicEMtrain(lansing[odd,-3], labels=lansing[odd, 3]), 1)
}
\keyword{cluster}

\eof
\name{cdens}
\alias{cdens}
\title{
  Component Density for Parameterized MVN Mixture Models
}
\description{
  Computes component densities for observations in parameterized MVN mixture
  models.
}
\usage{
cdens(modelName, data, mu, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model. Possible models: \cr\cr
    "E" for spherical, equal variance (one-dimensional) \cr
    "V" for spherical, variable variance (one-dimensional) \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr
    For fitting a single Gaussian:\cr\cr    
    "X": one-dimensional \cr
    "XII": spherical \cr
    "XXI": diagonal \cr
    "XXX": ellipsoidal
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the components.
  }
  \item{\dots}{
    Arguments for model-specific functions. Specifically:
    \itemize{
      \item \code{logarithm}: A logical value indicating whether or not
      the logarithm of the component densities should be returned. The
      default is to return the component densities, obtained from the
      log component densities by exponentiation.
      \item An argument describing the variance (depends on the model):
      \describe{
	\item{sigmasq}{
	  for the one-dimensional models ("E", "V") and spherical models
	  ("EII", "VII"). This is either a vector whose \emph{k}th
	  component is the variance for the \emph{k}th component in the
	  mixture model ("V" and "VII"), or a scalar giving the common
	  variance for all components in the mixture model ("E" and
	  "EII").
	}
	\item{decomp}{
	  for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
	  ellipsoidal models ("EEV", "VEV"). This is a list with the following
	  components:
	  \describe{
	    \item{d}{
	      The dimension of the data.
	    }
	    \item{G}{
	      The number of components in the mixture model.
	    }
	    \item{scale}{
	      Either a \emph{G}-vector giving the scale of the covariance (the
	      \emph{d}th root of its determinant) for each component in the
	      mixture model, or a single numeric value if the scale is the
	      same for each component.
	    }
	    \item{shape}{
	      Either a \emph{G} by \emph{d} matrix in which the \emph{k}th
	      column is the shape of the covariance matrix (normalized to have
	      determinant 1) for the \emph{k}th component, or a
	      \emph{d}-vector giving a common shape for all components.
	    }
	    \item{orientation}{
	      Either a \emph{d} by \emph{d} by \emph{G} array whose
	      \code{[,,k]}th entry is the orthonomal matrix of eigenvectors of
	      the covariance matrix of the \emph{k}th component, or a \emph{d}
	      by \emph{d} orthonormal matrix if the mixture components have a
	      common orientation. The \code{orientation} component of
	      \code{decomp} can be omitted in spherical and diagonal models,
	      for which the principal components are parallel to the
	      coordinate axes so that the orientation matrix is the identity.
	    }
	  }
	}
	\item{Sigma}{
	  for the equal variance model "EEE". A \emph{d} by \emph{d}
	  matrix giving the common covariance for all components of the
	  mixture model.
	}
	\item{sigma}{
	  for the unconstrained variance model "VVV". A \emph{d} by
	  \emph{d} by \emph{G} matrix array whose \code{[,,k]}th entry
	  is the covariance matrix for the \emph{k}th component of the
	  mixture model.
	}
	The form of the variance specification is the same as for the
	output for the \code{em}, \code{me}, or \code{mstep} methods for
	the specified mixture model. 
      }
      \item \code{eps}:
      A scalar tolerance for deciding when to terminate computations
      due to computational singularity in covariances. Smaller values
      of \code{eps} allow computations to proceed nearer to
      singularity. The default is \code{.Mclust\$eps}. 
      
      For those models with iterative M-step ("VEI", "VEV"), two
      values can be entered for \code{eps}, in which case the second
      value is used for determining singularity in the M-step. 
      
      \item \code{warnSingular}:
      A logical value indicating whether or not a warning should be issued
      whenever a singularity is encountered.
      The default is \code{.Mclust\$warnSingular}.
    }      
  }
}
\value{
  A numeric matrix whose \code{[i,j]}th entry is the 
  density of observation \emph{i} in component \emph{j}. 
  The densities are not scaled by mixing proportions.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{cdensE}}, \ldots,
  \code{\link{cdensVVV}},
  \code{\link{dens}},
  \code{\link{EMclust}},
  \code{\link{mstep}},
  \code{\link{mclustDAtrain}},
  \code{\link{mclustDAtest}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
n <- 100 ## create artificial data

set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
clPairs(x, cl = xclass, sym = c("1","2")) ## display the data

set.seed(0)
I <- sample(1:(2*n)) ## random ordering of the data
x <- x[I, ]
xclass <- xclass[I]

odd <- seq(1, 2*n, by = 2)
oddBic <- EMclust(x[odd, ]) 
oddSumry <- summary(oddBic, x[odd, ]) ## best parameter estimates
names(oddSumry)

even <- odd + 1
temp <- cdens(modelName = oddSumry$modelName, data = x[even, ],
              mu = oddSumry$mu, decomp = oddSumry$decomp)
cbind(class = xclass[even], temp)

## alternative call 

temp <- do.call( "cdens", c(list(data = x[even, ]), oddSumry))
cbind(class = xclass[even], temp)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{cdensE}
\alias{cdensE}
\alias{cdensV}
\alias{cdensEII}
\alias{cdensVII}
\alias{cdensEEI}
\alias{cdensVEI}
\alias{cdensEVI}
\alias{cdensVVI}
\alias{cdensEEE}
\alias{cdensEEV}
\alias{cdensVEV}
\alias{cdensVVV}
\title{
  Component Density for a Parameterized MVN Mixture Model
}
\description{
  Computes component densities for points in a parameterized MVN mixture model.
}
\usage{
cdensE(data, mu, sigmasq, eps, warnSingular, logarithm = FALSE, \dots)
cdensV(data, mu, sigmasq, eps, warnSingular, logarithm = FALSE, \dots)
cdensEII(data, mu, sigmasq, eps, warnSingular, logarithm = FALSE, \dots)
cdensVII(data, mu, sigmasq, eps, warnSingular, logarithm = FALSE, \dots)
cdensEEI(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensVEI(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensEVI(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensVVI(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensEEE(data, mu, eps, warnSingular, logarithm = FALSE, \dots)
cdensEEV(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensVEV(data, mu, decomp, eps, warnSingular, logarithm = FALSE, \dots)
cdensVVV(data, mu, eps, warnSingular, logarithm = FALSE, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the
    means of the components.
  }
  \item{sigmasq}{
    for the one-dimensional models ("E", "V") and spherical
    models ("EII", "VII"). This is either a vector whose \emph{k}th
    component is the variance for the \emph{k}th component in the
    mixture model ("V" and "VII"), or a scalar giving the common
    variance for all components in the mixture model ("E" and "EII").
  }
  \item{decomp}{
    for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
    ellipsoidal models ("EEV", "VEV"). This is a list described in more
    detail in \code{\link{cdens}}.
  }
  \item{logarithm}{
    A logical value indicating whether or not the logarithm of the
    component densities should be returned.
    The default is to return the component densities,
    obtained from the log component densities by exponentiation.
  }
  \item{\dots}{
    An argument giving the variance that takes one of the following forms:
    \describe{
      \item{decomp}{for models "EII" and "VII"; see above.}
      \item{cholSigma}{see Sigma, for "EEE".}
      \item{Sigma}{ 
	for the equal variance model "EEE". A \emph{d} by \emph{d} matrix
	giving the common covariance for all components of the mixture
	model.
      }
      \item{cholsigma}{see sigma, for "VVV".}
      \item{sigma}{ 
	for the unconstrained variance model "VVV".
	A \emph{d} by \emph{d} by \emph{G} matrix array whose 
	\code{[,,k]}th entry is the covariance matrix for the
	\emph{k}th component of the mixture model.
      }
      
      The form of the variance specification is the same as for the output
      for the \code{em}, \code{me}, or \code{mstep} methods for the
      specified mixture model.
      
      Also used to catch unused arguments from a \code{do.call} call.
    }
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due
    to computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is \code{.Mclust\$eps}. 
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered. The default is
    \code{.Mclust\$warnSingular}. 
  }
}
\value{
  A numeric matrix whose \code{[i,j]}th 
  entry is the density of observation \emph{i} in component \emph{j}. 
  The densities are not scaled by mixing proportions.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{cdens}},
  \code{\link{dens}},
  \code{\link{EMclust}},
  \code{\link{mstep}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
n <- 100 ## create artificial data

set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
clPairs(x, cl = xclass, sym = c("1","2")) ## display the data

modelVII <- meVII(x, z = unmap(xclass))
modelVVI <- meVVI(x, z = unmap(xclass))
modelVVV <- meVVV(x, z = unmap(xclass))

names(modelVII)
args(cdensVII)
cdenVII <- cdensVII(data = x, mu = modelVII$mu, pro = modelVII$pro,
                    decomp = modelVII$decomp)
names(modelVVI)
args(cdensVVI)
cdenVVI <- cdensVII(data = x, mu = modelVVI$mu, pro = modelVVI$pro,
                    decomp = modelVVI$decomp)
names(modelVVV)
args(cdensVVV)
cdenVVV <- cdensVVV( data = x, mu = modelVVV$mu, pro = modelVVV$pro,
                     cholsigma = modelVVV$cholsigma)

cbind(class=xclass,VII=map(cdenVII),VVI=map(cdenVVI),VVV=map(cdenVVV))

## alternative call
 
cdenVII <- do.call("cdensVII", c(list(data = x), modelVII))
cdenVVI <- do.call("cdensVVI", c(list(data = x), modelVVI))
cdenVVV <- do.call("cdensVVV", c(list(data = x), modelVVV))

cbind(class=xclass,VII=map(cdenVII),VVI=map(cdenVVI),VVV=map(cdenVVV))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{chevron}
\alias{chevron}
\non_function{}
\title{Simulated minefield data}
\usage{data(chevron)}
\description{A two-dimensional data set of simulated minefield data
  (1104 observations).}
\references{C. Fraley and A.E. Raftery, \emph{Computer J.}, 41:578-588 (1998)}

\keyword{datasets}


\eof
\name{clPairs}
\alias{clPairs}
\title{
  Pairwise Scatter Plots showing Classification
}
\description{
  Creates a scatter plot for each pair of variables in given data.
  Observations in different classes are represented by different symbols.
}
\usage{
clPairs(data, classification, symbols, labels=dimnames(data)[[2]], 
        CEX=1, col, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{classification}{
    A numeric or character vector representing a classification of observations (rows) of \code{data}.
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to each
    unique class \code{classification}. Elements in \code{symbols}
    correspond to classes in order of appearance in the sequence of
    observations (the order used by the function
    \code{unique}). Default: If \emph{G} is the number of groups in the
    classification, the first \emph{G} symbols in
    \code{.Mclust\$symbols}, otherwise if \emph{G} is less than 27 then
    the first \emph{G} capital letters in the Roman alphabet. If no
    \code{classification} argument is given the default symbol is
    \code{"."}.
  }
  \item{labels}{
    A vector of character strings for labeling the variables. The default
    is to use the column dimension names of \code{data}.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. 
    The default value is 1.
  }
  \item{col}{
    Color vector to use. Default is one color per class. Splus default:
    all black.
  }
  \item{\dots}{
    Additional arguments to be passed to the graphics device.
  }
}
\section{Side Effects}{
  Scatter plots for each combination of variables in \code{data} are
  created on the current graphics device. Observations of different
  classifications are labeled with different symbols. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{pairs}},
  \code{\link{coordProj}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

clPairs(irisMatrix, cl=irisClass, symbols=as.character(1:3))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{classError}
\alias{classError}
\alias{classErrors}
\title{
  Classification error.
}
\description{
  Error for a given classification relative to a known truth. 
  Location of errors in a given classification relative to a known truth.
}
\usage{
classError(classification, truth)
}
\arguments{
  \item{classification}{
    A numeric or character vector of class labels.
  }
  \item{truth}{
    A numeric or character vector of class labels.
    Must have the same length as
    \code{classification}.
  }
  
}
\value{
  \code{classError} gives the fraction of elements 
  misclassified for \code{classification}
  relative to \code{truth}. 
  \code{classErrors} is a logical vector of the
  same length as \code{classification} and
  \code{truth} which gives the location
  of misclassified elements in \code{classification}
  relative to \code{truth}. 
  
}
\details{
  \code{classErrors} will only return one possibility
  if more than one mapping between classification and truth results in the
  minimum error.
}
\seealso{
  \code{\link{compareClass}},
  \code{\link{mapClass}},
  \code{\link{table}}
}
\examples{
a <- rep(1:3, 3)
a
b <- rep(c("A", "B", "C"), 3)
b
classError(a, b)
classErrors(a, b)

a <- sample(1:3, 9, replace = TRUE)
a
b <- sample(c("A", "B", "C"), 9, replace = TRUE)
b
classError(a, b)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{compareClass}
\alias{compareClass}
\title{
  Compare classifications.
}
\description{
  Compare classifications via the normalized 
  variation of information criterion.
}
\usage{
compareClass(a, b)
}
\arguments{
  \item{a}{
    A numeric or character vector of class labels.
  }
  \item{b}{
    A numeric or character vector of class labels.
    Must have the same length as
    \code{a}.
  }
  
}
\value{
  The variation of information criterion (Meila 2002) for \code{a} and
  \code{b} divided by the log of the length of the sequences so that it
  falls in \emph{[0,1]}.
}
\section{References}{
  Marina Meila (2002).
  Comparing clusterings.
  Technical Report no. 418, Department of Statistics, University
  of Washington.
  
  See \url{http://www.stat.washington.edu/www/research/reports}.
}
\seealso{
  \code{\link{mapClass}},
  \code{\link{classError}},
  \code{\link{table}}
}
\examples{
a <- rep(1:3, 3)
a
b <- rep(c("A", "B", "C"), 3)
b
compareClass(a, b)
a <- sample(1:3, 9, replace = TRUE)
a
b <- sample(c("A", "B", "C"), 9, replace = TRUE)
b
compareClass(a, b)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{coordProj}
\alias{coordProj}
\title{
  Coordinate projections of data in more than two dimensions modelled by an MVN
  mixture.
}
\description{
  Plots coordinate projections given data in more than two dimensions
  and parameters of an MVN mixture model for the data.
}
\usage{
coordProj(data, \dots, dimens = c(1, 2),
          type = c("classification","uncertainty","errors"), ask = TRUE,
          quantiles = c(0.75, 0.95), symbols, scale = FALSE,
          identify = FALSE, CEX = 1, PCH = ".", xlim, ylim)
}
\arguments{
  \item{data}{
    A numeric matrix or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{dimens}{
    A vector of length 2 giving the integer dimensions of the
    desired coordinate projections. The default is
    \code{c(1,2)}, in which the first
    dimension is plotted against the second.
  }
  \item{\dots}{
    One or more of the following:
    \describe{
      \item{classification}{
	A numeric or character vector representing a classification of
	observations (rows) of \code{data}.
      }
      \item{uncertainty}{
	A numeric vector of values in \emph{(0,1)} giving the
	uncertainty of each data point.
      }
      \item{z}{
	A matrix in which the \code{[i,k]}th entry gives the
	probability of observation \emph{i} belonging to the \emph{k}th class. 
	Used to compute \code{classification} and
	\code{uncertainty} if those arguments aren't available.
      }
      \item{truth}{
	A numeric or character vector giving a known
	classification of each data point.
	If \code{classification}
	or \code{z} is also present, 
	this is used for displaying classification errors.
      }
      \item{mu}{
	A matrix whose columns are the means of each group. 
      }
      \item{sigma}{
	A three dimensional array 
	in which \code{sigma[,,k]} gives the covariance
	for the \emph{k}th group.
      }
      \item{decomp}{
	A list with \code{scale}, \code{shape} and \code{orientation}
	components giving an alternative form for the covariance
	structure of the mixture model. 
      }
    }
  }
  \item{type}{
    Any subset of 
    \code{c("classification","uncertainty","errors")}. The function will
    produce the corresponding plot if it has been supplied sufficient
    information to do so. If more than one plot is possible then users
    will be asked to choose from a menu if \code{ask=TRUE}. 
  }
  \item{ask}{
    A logical variable indicating whether or not a menu should be
    produced when more than one plot is possible. The default is
    \code{ask=TRUE}. 
  }
  \item{quantiles}{
    A vector of length 2 giving quantiles used in plotting
    uncertainty. The smallest symbols correspond to the smallest
    quantile (lowest uncertainty), medium-sized (open) symbols to points
    falling between the given quantiles, and large (filled) symbols to
    those in the largest quantile (highest uncertainty). The default is
    \emph{(0.75,0.95)}. 
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to
    each unique class in \code{classification}. Elements in
    \code{symbols} correspond to classes in \code{classification} in
    sorted order. Default: If \emph{G} is the number of groups in the
    classification, the first \emph{G} symbols in
    \code{.Mclust\$symbols}, otherwise if \emph{G} is less than 27 then
    the first \emph{G} capital letters in the Roman alphabet. 
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and
    thus preserve the shape of the distribution.
    Default: \code{scale=FALSE} 
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. 
    The default value is 1.
  }
  \item{PCH}{
    An argument specifying the symbol to be used when a classificatiion
    has not been specified for the data. The default value is a small dot ".".
  }
  \item{xlim, ylim}{
    Arguments specifying bounds for the ordinate, abscissa of the plot.
    This may be useful for when comparing plots.
  }
}
\section{Side Effects}{
  Coordinate projections of the data, possibly showing location of the
  mixture components, classification, uncertainty, and/or classification
  errors. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{clPairs}},
  \code{\link{randProj}},
  \code{\link{mclust2Dplot}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstepVVV(irisMatrix, unmap(irisClass))

par(pty = "s", mfrow = c(1,2))
coordProj(irisMatrix,dimens=c(2,3), truth = irisClass, 
          mu = msEst$mu, sigma = msEst$sigma, z = msEst$z)
do.call("coordProj", c(list(data=irisMatrix, dimens=c(2,3), truth=irisClass),
                       msEst))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{cv1EMtrain}
\alias{cv1EMtrain}
\title{Select discriminant models using cross validation}
\description{
For the ten available discriminant models the leave-one-out cross
validation error is calulated. The models for one-dimensional data are
"E" and "V"; for higher dimensions they are "EII", "VII", "EEI",
"VEI", "EVI", "VVI", "EEE", "EEV", "VEV" and "VVV".
}
\usage{
cv1EMtrain(data, labels, modelNames)
}
\arguments{
  \item{data}{A data matrix}
  \item{labels}{Labels for each row in the data matrix}
  \item{modelNames}{Vector of model names that should be tested.}
}
\value{
  Returns a vector where each element is the error rate for the
  corresponding model.
}
\author{C. Fraley}
\seealso{\code{\link{bicEMtrain}}}
\examples{
data(lansing)
odd <- seq(from=1, to=nrow(lansing), by=2)
round(cv1EMtrain(data=lansing[odd,-3], labels=lansing[odd,3]), 3)

cv1Modd <- mstepEEV(data=lansing[odd,-3], z=unmap(lansing[odd,3]))
cv1Zodd <- do.call("estepEEV", c(cv1Modd, list(data=lansing[odd,-3])))$z
compareClass(map(cv1Zodd), lansing[odd,3])

even <- (1:nrow(lansing))[-odd]
cv1Zeven <- do.call("estepEEV", c(cv1Modd, list(data=lansing[even,-3])))$z
compareClass(map(cv1Zodd), lansing[odd,3])$error
}
\keyword{cluster}


\eof
\name{decomp2sigma}
\alias{decomp2sigma}
\title{
  Convert mixture component covariances to matrix form.
}
\description{
  Converts a set of covariances from a parameterization by 
  eigenvalue decomposition to representation as a 3-D array.
}
\usage{
decomp2sigma(d, G, scale, shape, orientation, \dots)
}
\arguments{
  \item{d}{
    The dimension of the data. 
  }
  \item{G}{
    The number of components in the mixture model. 
  }
  \item{scale}{
    Either a \emph{G}-vector giving the scale of the covariance (the
    \emph{d}th root of its determinant) for each component in the
    mixture model, or a single numeric value if the scale is the same
    for each component.  
  }
  \item{shape}{
    Either a \emph{G} by \emph{d} matrix in which the \emph{k}th column
    is the shape of the covariance matrix (normalized to have
    determinant 1) for the \emph{k}th component, or a \emph{d}-vector
    giving a common shape for all components. 
  }
  \item{orientation}{
    Either a \emph{d} by \emph{d} by \emph{G} array whose \code{[,,k]}th
    entry is the orthonomal matrix of eigenvectors of the covariance
    matrix of the \emph{k}th component, or a \emph{d} by \emph{d}
    orthonormal matrix if the mixture components have a common
    orientation. The \code{orientation} component of \code{decomp} can
    be omitted in spherical and diagonal models, for which the principal
    components are parallel to the coordinate axes so that the
    orientation matrix is the identity.  
  }
  \item{\dots}{
    Catch unused arguments from a \code{do.call} call.
  }
}
\value{
  A 3-D array whose \code{[,,k]}th component is the 
  covariance matrix of the \emph{k}th component in an MVN mixture model.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation, and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{sigma2decomp}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

meEst <- meVEV(irisMatrix, unmap(irisClass)) 
names(meEst)
meEst$decomp
meEst$sigma

dec <- meEst$decomp
decomp2sigma(d=dec$d, G=dec$G, shape=dec$shape, scale=dec$scale,
             orientation = dec$orientation)
do.call("decomp2sigma", meEst$decomp)  ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{dens}
\alias{dens}
\title{
  Density for Parameterized MVN Mixtures
}
\description{
  Computes densities of obseravations in parameterized MVN mixtures.
}
\usage{
dens(modelName, data, mu, logarithm, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model. Possible models: \cr\cr
    "E" for spherical, equal variance (one-dimensional) \cr
    "V" for spherical, variable variance (one-dimensional) \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation\cr 
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr
    For fitting a single Gaussian,\cr\cr
    "X": one-dimensional \cr
    "XII": spherical \cr
    "XXI": diagonal \cr
    "XXX": ellipsoidal
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the components.
  }
  \item{logarithm}{Return logarithm of the density, rather than the
    density itself. Default: FALSE}
  \item{\dots}{
    Other arguments, such as an argument describing the variance.
    See \code{\link{cdens}}.
  }
}
\value{
  A numeric vector whose \emph{i}th component is the density of observation 
  \emph{i} in the MVN mixture specified by \code{mu} and
  \code{\dots}.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{grid1}},
  \code{\link{cdens}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
n <- 100 ## create artificial data

set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
clPairs(x, cl = xclass, sym = c("1","2")) ## display the data

set.seed(0)
I <- sample(1:(2*n))
x <- x[I, ]
xclass <- xclass[I]

odd <- seq(1, 2*n, by = 2)
oddBic <- EMclust(x[odd, ]) 
oddSumry <- summary(oddBic, x[odd, ]) ## best parameter estimates
names(oddSumry)

oddDens <- dens(modelName = oddSumry$modelName, data = x,
   mu = oddSumry$mu, decomp = oddSumry$decomp, pro = oddSumry$pro)
oddDens <- do.call("dens", c(list(data = x), oddSumry))  ## alternative call

even <- odd + 1
evenBic <- EMclust(x[even, ]) 
evenSumry <- summary(evenBic, x[even, ]) ## best parameter estimates
evenDens <- do.call( "dens", c(list(data = x), evenSumry))

cbind(class = xclass, odd = oddDens, even = evenDens)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{density}
\alias{density}
\alias{print.density}
\title{Kernel Density Estimation}
\usage{
density(\dots, method, G)
}
\arguments{
  \item{\dots}{Arguments to the \code{density} function in the base
    package.}

  \item{method}{If equal to "mclust", \code{EMclust} is used to estimate
    the density.}

  \item{G}{The number of gaussians to consider in the model-based
    density estimation. Default: 1:9. Ignored if method is not equal to
    "mclust".}
}
\description{
  This is exaclty the same function as in the base package but for the
  \code{method} argument: if it is given and equals \code{"mclust"}, the
  \code{mclust} density estimation is used. Optionally, the number of
  gaussians to be considered can be given as well (\code{G}).}
}
\value{
  If \code{give.Rkern} is true, the number \eqn{R(K)}, otherwise
  an object with class \code{"density"} whose
  underlying structure is a list containing the following components.
  \item{x}{the \code{n} coordinates of the points where the density is
    estimated.}
  \item{y}{the estimated density values.}
  \item{bw}{the bandwidth used.}
  \item{N}{the sample size after elimination of missing values.}
  \item{call}{the call which produced the result.}
  \item{data.name}{the deparsed name of the \code{x} argument.}
  \item{has.na}{logical, for compatibility (always FALSE).}
}
\references{
  Fraley, C. and Raftery, A.E. (2002)
  MCLUST: software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report No. 415,
  Dept. of Statistics, University of Washington.
  
  Scott, D. W. (1992)
  \emph{Multivariate Density Estimation. Theory, Practice and Visualization}.
  New York: Wiley.

  Sheather, S. J. and Jones M. C. (1991)
  A reliable data-based bandwidth selection method for kernel density
  estimation.
  \emph{J. Roy. Statist. Soc.} \bold{B}, 683--690.

  Silverman, B. W. (1986)
  \emph{Density Estimation}.
  London: Chapman and Hall.

  Venables, W. N. and Ripley, B. D. (1999)
  \emph{Modern Applied Statistics with S-PLUS}.
  New York: Springer.
}
\seealso{
  \code{\link[base]{density}} (base package), \code{\link{bw.nrd}},
  \code{\link{plot.density}}, \code{\link{hist}}.
}
\examples{
plot(density(c(-20,rep(0,98),20)), xlim = c(-4,4))# IQR = 0

# The Old Faithful geyser data
data(faithful)
d <- density(faithful$eruptions, bw = "sj")
d
plot(d)
dmc <- density(faithful$eruptions, method="mclust")
plot(dmc, type = "n")
polygon(dmc, col = "wheat")
lines(d, col="red")

## Missing values:
x <- xx <- faithful$eruptions
x[i.out <- sample(length(x), 10)] <- NA
doRmc <- density(x=x, method="mclust", na.rm = TRUE)
lines(doRmc, col="blue")
doR <- density(x, bw = 0.15, na.rm = TRUE)
lines(doR, col = "green")
rug(x)
points(xx[i.out], rep(0.01, 10))

## function formals returns something different now the original
## density function is masked...
base.density <- if(exists("density", envir = NULL)) {
  get("density", envir = NULL)
} else
  stats::density
(kernels <- eval(formals(base.density)$kernel))

## show the kernels in the R parametrization
plot (density(0, bw = 1), xlab = "",
      main="R's density() kernels with bw = 1")
for(i in 2:length(kernels))
   lines(density(0, bw = 1, kern =  kernels[i]), col = i)
legend(1.5,.4, legend = kernels, col = seq(kernels),
       lty = 1, cex = .8, y.int = 1)

data(precip)
bw <- bw.SJ(precip) ## sensible automatic choice
plot(density(precip, bw = bw, n = 2^13))
lines(density(precip, G=2:5, method="mclust"), col="red")
rug(precip)
}
\keyword{cluster}
\keyword{distribution}
\keyword{smooth}

\eof
\name{diabetes}
\alias{diabetes}
\alias{diabetes.class}
\non_function{}
\title{Diabetes data}
\usage{data(diabetes)}
\description{
Diabetes data from Reaven and Miller. Number of objects: 145; 3
variables. Three classes.
}
\references{G.M. Reaven and R.G. Miller, \emph{Diabetologica} 16:17-24 (1979).
  }

\keyword{datasets}

\eof
\name{em}
\alias{em}
\title{
  EM algorithm starting with E-step for parameterized MVN mixture models.
}
\description{
  Implements the EM algorithm for parameterized MVN mixture models,
  starting with the expectation step.
}
\usage{
em(modelName, data, mu, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model: \cr\cr
    "E": equal variance  (one-dimensional) \cr
    "V": variable variance (one-dimensional) \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape\cr 
    "VEI": diagonal, varying volume, equal shape\cr 
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation\cr 
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the
    components.
  }
  \item{\dots}{
    Arguments for model-specific em functions. Specifically:
    \itemize{
      \item An argument describing the variance (depends on the model):
      \describe{
	\item{sigmasq}{
	  for the one-dimensional models ("E", "V") and spherical models
	  ("EII", "VII"). This is either a vector whose \emph{k}th
	  component is the variance for the \emph{k}th component in the
	  mixture model ("V" and "VII"), or a scalar giving the common
	  variance for all components in the mixture model ("E" and
	  "EII").
	}
	\item{decomp}{
	  for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
	  ellipsoidal models ("EEV", "VEV"). For a description,
	  see \code{\link{cdens}}.
	}
	\item{Sigma}{
	  for the equal variance model "EEE". A \emph{d} by \emph{d}
	  matrix giving the common covariance for all components of the
	  mixture model.
	}
	\item{sigma}{
	  for the unconstrained variance model "VVV". A \emph{d} by
	  \emph{d} by \emph{G} matrix array whose \code{[,,k]}th entry
	  is the covariance matrix for the \emph{k}th component of the
	  mixture model.
	}
	The form of the variance specification is the same as for the
	output for the \code{em}, \code{me}, or \code{mstep} methods for
	the specified mixture model. 
      }
      
      \item \code{pro}:
      Mixing proportions for the components of the mixture. There
      should one more mixing proportion than the number of MVN
      components if the mixture model includes a  Poisson noise term. 
      \item \code{eps}:
      A scalar tolerance for deciding when to terminate computations
      due to computational singularity in covariances. Smaller values
      of \code{eps} allow computations to proceed nearer to
      singularity. The default is \code{.Mclust\$eps}. 
      
      For those models with iterative M-step ("VEI", "VEV"), two
      values can be entered for \code{eps}, in which case the second
      value is used for determining singularity in the M-step. 

      \item \code{tol}:
      A scalar tolerance for relative convergence of the loglikelihood. 
      The default is \code{.Mclust\$tol}.
      
      For those models with iterative M-step ("VEI", "VEV"), two
      values can be entered for \code{tol}, in which case the second
      value governs parameter convergence in the M-step. 

      \item \code{itmax}:
      An integer limit on the number of EM iterations. 
      The default is \code{.Mclust\$itmax}.
      
      For those models with iterative M-step ("VEI", "VEV"), two
      values can be entered for \code{itmax}, in which case the second
      value is an upper limit on the number of iterations in the
      M-step. 

      \item \code{equalPro}:
      Logical variable indicating whether or not the mixing proportions are
      equal in the model. The default is \code{.Mclust\$equalPro}.
      
      \item \code{warnSingular}:
      A logical value indicating whether or not a warning should be issued
      whenever a singularity is encountered.
      The default is \code{.Mclust\$warnSingular}.
      
      \item \code{Vinv}:
      An estimate of the reciprocal hypervolume of the data region.
      The default is determined by applying function 
      \code{hypvol} to the data.
      Used only when \code{pro} includes an additional
      mixing proportion for a noise component.
    }
  }
}

\value{
  A list including the following components: 
  
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the conditional probability of
    the \emph{i}th observation belonging to the \emph{k}th component of the
    mixture.   
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{mu}{
    A matrix whose kth column is the
    mean of the \emph{k}th component of the mixture model.
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{
    A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{modelName}{
    A character string identifying the model (same as the input argument).
  }
  \item{Attributes:}{
    \itemize{
      \item \code{"info"}: Information on the iteration.
      \item \code{"warn"}: An appropriate warning if problems are
      encountered in the computations.
    }
  }
}
\details{
  This function can be used with an indirect or list call
  using \code{do.call}, allowing the output
  of e.g. \code{mstep} to be passed
  without the need to specify individual parameters as arguments.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{emE}}, \ldots,
  \code{\link{emVVV}},
  \code{\link{estep}},
  \code{\link{me}},
  \code{\link{mstep}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]
 
msEst <- mstep(modelName = "EEE", data = irisMatrix, 
               z = unmap(irisClass))
names(msEst)

em(modelName = msEst$modelName, data = irisMatrix,
   mu = msEst$mu, Sigma = msEst$Sigma, pro = msEst$pro)
do.call("em", c(list(data = irisMatrix), msEst))   ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{emE}
\alias{emE}
\alias{emV}
\alias{emEII}
\alias{emVII}
\alias{emEEI}
\alias{emVEI}
\alias{emEVI}
\alias{emVVI}
\alias{emEEE}
\alias{emEEV}
\alias{emVEV}
\alias{emVVV}
\title{
  EM algorithm starting with E-step for a parameterized MVN mixture model.
}
\description{
  Implements the EM algorithm for a parameterized MVN mixture model,
  starting with the expectation step.
}
\usage{
emE(data, mu, sigmasq, pro, eps, tol, itmax, equalPro, warnSingular,
    Vinv, \dots)
emV(data, mu, sigmasq, pro, eps, tol, itmax, equalPro, warnSingular,
    Vinv, \dots)
emEII(data, mu, sigmasq, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emVII(data, mu, sigmasq, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emEEI(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emVEI(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emEVI(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emVVI(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emEEE(data, mu, Sigma, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emEEV(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emVEV(data, mu, decomp, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
emVVV(data, mu, sigma, pro, eps, tol, itmax, equalPro, warnSingular,
      Vinv, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables.
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the
    components. 
  }
  \item{sigmasq}{
    for the one-dimensional models ("E", "V") and spherical
    models ("EII", "VII"). This is either a vector whose \emph{k}th
    component is the variance for the \emph{k}th component in the
    mixture model ("V" and "VII"), or a scalar giving the common
    variance for all components in the mixture model ("E" and "EII").
  }
  \item{decomp}{
    for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
    ellipsoidal models ("EEV", "VEV"). This is a list described in more
    detail in \code{\link{cdens}}.
  }
  \item{Sigma}{ 
    for the equal variance model "EEE". A \emph{d} by \emph{d} matrix
    giving the common covariance for all components of the mixture
    model.
  }
  \item{sigma}{ 
    for the unconstrained variance model "VVV".
    A \emph{d} by \emph{d} by \emph{G} matrix array whose 
    \code{[,,k]}th entry is the covariance matrix for the
    \emph{k}th component of the mixture model.
  }
  \item{\dots}{
    An argument giving the variance that takes one of the following forms:
    \describe{
      \item{decomp}{for models "VVV", "EII" and "VII"; see
	\code{\link{cdens}}.} 
      \item{cholSigma}{see Sigma, for "EEE".}
      \item{cholsigma}{see sigma, for "VVV".}
      \item{sigma}{see sigma, for "VVV".}
      \item{Sigma}{see Sigma, for "EEE".}

      The form of the variance specification is the same as for the output
      for the \code{em}, \code{me}, or \code{mstep} methods for the
      specified mixture model.

      Also used to catch unused arguments from a \code{do.call} call.
    }
  }
  \item{pro}{
    Mixing proportions for the components of the mixture. 
    There should one more mixing proportion than the number of MVN components
    if the mixture model includes a  Poisson noise term.
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due to
    computational singularity in covariances.  
    Smaller values of \code{eps} allow computations to
    proceed nearer to singularity. 
    The default is \code{.Mclust\$eps}.
  }
  \item{tol}{
    A scalar tolerance for relative convergence of the loglikelihood values. 
    The default is \code{.Mclust\$tol}.
  }
  \item{itmax}{
    An integer limit on the number of EM iterations. 
    The default is \code{.Mclust\$itmax}.
  }
  \item{equalPro}{
    A logical value indicating whether or not the components in the model are 
    present in equal proportions.
    The default is \code{.Mclust\$equalPro}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{.Mclust\$warnSingular}.
  }
  \item{Vinv}{
    An estimate of the reciprocal hypervolume of the data region.
    The default is determined by applying function 
    \code{hypvol} to the data.
    Used only when \code{pro} includes an additional
    mixing proportion for a noise component.
  }
}
\value{
  A list including the following components: 
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{mu}{
    A matrix whose kth column is the
    mean of the \emph{k}th component of the mixture model.
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{
    A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{modelName}{
    Character string identifying the model.
  }
  \item{Attributes:}{
    \itemize{
      \item \code{"info"}: Information on the iteration.
      \item \code{"warn"}: An appropriate warning if problems are
      encountered in the computations.
    }
  }
}

\details{
  This function can be used with an indirect or list call
  using \code{do.call}, allowing the output
  of e.g. \code{mstep} to be passed 
  without the need to specify individual parameters as arguments.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{em}},
  \code{\link{mstep}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}} 
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstepEEE(data = irisMatrix, z = unmap(irisClass))
names(msEst)

emEEE(data = irisMatrix, mu = msEst$mu, pro = msEst$pro, cholSigma = msEst$cholSigma)
do.call("emEEE", c(list(data=irisMatrix), msEst)) ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{estep}
\alias{estep}
\title{
  E-step for parameterized MVN mixture models.
}
\description{
  Implements the expectation step of EM algorithm for parameterized MVN
  mixture models. 
}
\usage{
estep(modelName, data, mu, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model: \cr\cr 
    "E": equal variance  (one-dimensional) \cr
    "V": variable variance (one-dimensional) \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape \cr
    "VEI": diagonal, varying volume, equal shape\cr 
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows correspond
    to observations and columns correspond to variables. 
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the
    components. 
  }
  \item{\dots}{
    Arguments for model-specific functions. Specifically:
    \itemize{
      \item An argument describing the variance (depends on the model):
      \describe{
	\item{sigmasq}{
	  for the one-dimensional models ("E", "V") and spherical models
	  ("EII", "VII"). This is either a vector whose \emph{k}th
	  component is the variance for the \emph{k}th component in the
	  mixture model ("V" and "VII"), or a scalar giving the common
	  variance for all components in the mixture model ("E" and
	  "EII").
	}
	\item{decomp}{
	  for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
	  ellipsoidal models ("EEV", "VEV"). This is a list described in
	  \code{\link{cdens}}.
	}
	\item{Sigma}{
	  for the equal variance model "EEE". A \emph{d} by \emph{d}
	  matrix giving the common covariance for all
	  components of the
	  mixture model.
	}
	\item{sigma}{
	  for the unconstrained variance model "VVV". A
	  \emph{d} by \emph{d} by \emph{G} matrix array whose
	  \code{[,,k]}th entry is the covariance matrix for
	  the \emph{k}th component of the mixture model. 
	}
	The form of the variance specification is the same as
	for the output for the \code{em}, \code{me}, or
	\code{mstep} methods for the specified mixture model. 
      }
      \item{pro}{
	Mixing proportions for the components of the
	mixture. There should one more mixing proportion than the
	number of MVN components if the mixture model includes a
	Poisson noise term. 
      }
      \item{eps}{
	A scalar tolerance for deciding when to terminate
	computations due to computational singularity in
	covariances. Smaller values of \code{eps} allow
	computations to proceed nearer to singularity. The default
	is \code{.Mclust\$eps}. 
      }
      \item{warnSingular}{
	A logical value indicating whether or not a warning
	should be issued whenever a singularity is
	encountered. The default is \code{.Mclust\$warnSingular}.  
      }
      \item{Vinv}{
	An estimate of the reciprocal hypervolume of the data region.
	The default is determined by applying function 
	\code{hypvol} to the data.
	Used only when \code{pro} includes an additional
	mixing proportion for a noise component.
      }
    }
  }
}
\value{
  A list including the following components: 
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the conditional probability
    of the \emph{i}th observation belonging to the \emph{k}th component
    of the mixture.   
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{modelName}{
    A character string identifying the model (same as the input argument).
  }
  \item{Attribute}{
    \itemize{
      \item \code{"warn"}: An appropriate warning if problems are
      encountered in the computations.
    }
  }
}
\details{
  This function can be used with an indirect or list call using
  \code{do.call}, allowing the output of e.g. \code{mstep} to be passed
  without the need to specify individual parameters as arguments. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{estepE}}, ...,
  \code{\link{estepVVV}},
  \code{\link{em}},
  \code{\link{mstep}},
  \code{\link{do.call}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstep(modelName = "EII", data = irisMatrix, 
               z = unmap(irisClass))
names(msEst)

estep(modelName = msEst$modelName, data = irisMatrix,
      mu = msEst$mu, sigmasq = msEst$sigmasq, pro = msEst$pro)           
do.call("estep", c(list(data = irisMatrix), msEst))   ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{estepE}
\alias{estepE}
\alias{estepV}
\alias{estepEII}
\alias{estepVII}
\alias{estepEEI}
\alias{estepVEI}
\alias{estepEVI}
\alias{estepVVI}
\alias{estepEEE}
\alias{estepEEV}
\alias{estepVEV}
\alias{estepVVV}
\title{
  E-step in the EM algorithm for a parameterized MVN mixture model.
}
\description{
  Implements the expectation step in the EM algorithm for a 
  parameterized MVN mixture model.
}
\usage{
estepE(data, mu, sigmasq, pro, eps, warnSingular, Vinv, \dots)
estepV(data, mu, sigmasq, pro, eps, warnSingular, Vinv, \dots)
estepEII(data, mu, sigmasq, pro, eps, warnSingular, Vinv, \dots)
estepVII(data, mu, sigmasq, pro, eps, warnSingular, Vinv, \dots)
estepEEI(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepVEI(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepEVI(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepVVI(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepEEE(data, mu, Sigma, pro, eps, warnSingular, Vinv, \dots)
estepEEV(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepVEV(data, mu, decomp, pro, eps, warnSingular, Vinv, \dots)
estepVVV(data, mu, sigma, pro, eps, warnSingular, Vinv, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the
    means of the components.
  }
  \item{sigmasq}{
    for the one-dimensional models ("E", "V") and spherical
    models ("EII", "VII"). This is either a vector whose \emph{k}th
    component is the variance for the \emph{k}th component in the
    mixture model ("V" and "VII"), or a scalar giving the common
    variance for all components in the mixture model ("E" and "EII").
  }
  \item{decomp}{
    for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
    ellipsoidal models ("EEV", "VEV"). This is a list described in more
    detail in \code{\link{cdens}}.
  }
  \item{sigma}{
    for the unconstrained variance model "VVV" or the equal variance
    model "EEE". A \emph{d} by \emph{d} by \emph{G} matrix array whose
    \code{[,,k]}th entry is the covariance matrix for the \emph{k}th
    component of the mixture model.
  }
  \item{Sigma}{
    for the equal variance model "EEE". A \emph{d} by \emph{d} matrix
    giving the common covariance for all components of the mixture
    model.
  }
  \item{pro}{
    Mixing proportions for the components of the mixture. There should
    one more mixing proportion than the number of MVN components if the
    mixture model includes a  Poisson noise term. 
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due
    to computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is \code{.Mclust\$eps}. 
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered. The default is
    \code{.Mclust\$warnSingular}. 
  }
  \item{Vinv}{
    An estimate of the reciprocal hypervolume of the data region. The
    default is determined by applying function \code{hypvol} to the
    data. Used only when \code{pro} includes an additional mixing
    proportion for a noise component. 
  }
  \item{\dots}{
    Other arguments to describe the variance, in particular
    \code{decomp}, \code{sigma} or \code{cholsigma} for model "VVV",
    \code{decomp} for models "VII" and "EII", and \code{Sigma} or
    \code{cholSigma} for model "EEE".  \code{Sigma} is an \emph{d} by
    \emph{d} matrix giving the common covariance for all components of
    the mixture model.

    Also used to catch unused arguments from a \code{do.call} call.
  }

  }
}
\value{
  A list including the following components: 
  
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{modelName}{
    Character string identifying the model.
  }
  \item{Attribute}{
    \itemize{
      \item \code{"warn"}: An appropriate warning if problems are
      encountered in the computations.
    }
  }
}
\details{
  This function can be used with an indirect or list call using
  \code{do.call}, allowing the output of e.g. \code{mstep} to be passed
  without the need to specify individual parameters as arguments. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{estep}},
  \code{\link{em}},
  \code{\link{mstep}},
  \code{\link{do.call}},
  \code{\link{mclustOptions}} 
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstepEII(data = irisMatrix, z = unmap(irisClass))
names(msEst)

estepEII(data = irisMatrix, mu = msEst$mu, pro = msEst$pro, 
         sigmasq = msEst$sigmasq)
do.call("estepEII", c(list(data=irisMatrix), msEst)) ## alternative call
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{grid1}
\alias{grid1}
\alias{grid2}
\title{Generate grid points}
\description{
Generate grid points in one or two dimensions.
}
\usage{
grid1(n, range = c(0, 1), edge = TRUE)
grid2(x, y)
}
\arguments{
  \item{n}{Number of grid points.}
  \item{range}{Range of grid points.}
  \item{edge}{Logical: include edges or not?}
  \item{x, y}{Vectors.}
}
\value{
The value returned is simple: \code{grid1} generates a
vector; \code{grid2} generates a matrix.
}
\author{C. Fraley}
\seealso{
  \code{\link{lansing}},
  \code{\link{dens}}
}
\examples{
data(lansing)
maples <- lansing[as.character(lansing[,"species"]) == "maple", -3]
maplesBIC <- EMclust(maples)
maplesModel <- summary(maplesBIC, maples)
x <- grid1(100, range=c(0,1))
y <- x
xyDens <- do.call("dens", c(list(data=grid2(x, y)), maplesModel))
xyDens <- matrix(xyDens, ncol=100)
contour(xyDens)
points(maples, cex=.2, col="red")

image(xyDens)
points(maples, cex=.5)
}
\keyword{cluster}


\eof
\name{hc}
\alias{hc}
\title{
  Model-based Hierarchical Clustering
}
\description{
  Agglomerative hierarchical clustering based on maximum likelihood criteria 
  for MVN mixture models parameterized by eigenvalue decomposition.
}
\usage{
hc(modelName, data, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model. Possible models: \cr\cr 
    "E" : equal variance  (one-dimensional) \cr
    "V" : spherical, variable variance (one-dimensional) \cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{\dots}{
    Arguments for the method-specific hc functions. See
    \code{\link{hcE}}.
  }
}
\value{
  A numeric two-column matrix in which the \emph{i}th row gives the 
  minimum index for observations in each of the two clusters merged at the
  \emph{i}th stage of agglomerative hierarchical clustering.
}
\details{
  Most models have memory usage of the order of the square of the
  number groups in the initial partition for fast execution.
  Some models, such as equal variance or \code{"EEE"},
  do not admit a fast algorithm under the usual agglomerative
  hierarchical clustering paradigm. 
  These use less memory but are much slower to execute.
}
\note{
  If \code{modelName = "E"} (univariate with equal variances) or
  \code{modelName = "EII"} (multivariate with equal spherical
  covariances), then the method is equivalent to Ward's method for
  hierarchical clustering.
}
\section{References}{
  J. D. Banfield and A. E. Raftery (1993).
  Model-based Gaussian and non-Gaussian Clustering.
  \emph{Biometrics 49:803-821}. 
  
  C. Fraley (1998).
  Algorithms for model-based Gaussian hierarchical clustering.
  \emph{SIAM Journal on Scientific Computing 20:270-281}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{hcE}},...,
  \code{\link{hcVVV}},
  \code{\link{hclass}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

hcTree <- hc(modelName = "VVV", data = irisMatrix)
cl <- hclass(hcTree,c(2,3))

par(pty = "s", mfrow = c(1,1))
clPairs(irisMatrix,cl=cl[,"2"])
clPairs(irisMatrix,cl=cl[,"3"])

par(mfrow = c(1,2))
dimens <- c(1,2)
coordProj(irisMatrix, classification=cl[,"2"], dimens=dimens)
coordProj(irisMatrix, classification=cl[,"3"], dimens=dimens)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{hcE}
\alias{hcE}
\alias{hcV}
\alias{hcEII}
\alias{hcVII}
\alias{hcEEE}
\alias{hcVVV}
\title{
  Model-based Hierarchical Clustering
}
\description{
  Agglomerative hierarchical clustering based on maximum likelihood
  for a MVN mixture model parameterized by eigenvalue decomposition.
}
\usage{
hcE(data, partition, minclus=1, \dots)
hcV(data, partition, minclus = 1, alpha = 1, \dots)
hcEII(data, partition, minclus = 1, \dots)
hcVII(data, partition, minclus = 1, alpha = 1, \dots)
hcEEE(data, partition, minclus = 1, \dots)
hcVVV(data, partition, minclus = 1, alpha = 1, beta = 1, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{partition}{
    A numeric or character vector representing a partition of
    observations (rows) of \code{data}. If provided, group merges will
    start with this partition. Otherwise, each observation is assumed to
    be in a cluster by itself at the start of agglomeration. 
  }
  \item{minclus}{
    A number indicating the number of clusters at which to stop the
    agglomeration. The default is to stop when all observations have been
    merged into a single cluster.
  }
  \item{alpha, beta}{
    Additional tuning parameters needed for initializatiion in some models. 
    For details, see Fraley 1998. The defaults provided are usually adequate.
  }
  \item{\dots}{
    Catch unused arguments from a \code{do.call} call.
  }
}
\value{
  A numeric two-column matrix in which the \emph{i}th row gives the minimum 
  index for observations in each of the two clusters merged at the
  \emph{i}th stage of agglomerative hierarchical clustering.
}
\details{
  Most models have memory usage of the order of the square of the
  number groups in the initial partition for fast execution.
  Some models, such as equal variance or \code{"EEE"},
  do not admit a fast algorithm under the usual agglomerative
  hierachical clustering paradigm. 
  These use less memory but are much slower to execute.
}
\section{References}{
  J. D. Banfield and A. E. Raftery (1993).
  Model-based Gaussian and non-Gaussian Clustering.
  \emph{Biometrics 49:803-821}. 
  
  C. Fraley (1998).
  Algorithms for model-based Gaussian hierarchical clustering.
  \emph{SIAM Journal on Scientific Computing 20:270-281}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{hc}},
  \code{\link{hclass}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

hcTree <- hcEII(data = irisMatrix)
cl <- hclass(hcTree,c(2,3))

par(pty = "s", mfrow = c(1,1))
clPairs(irisMatrix,cl=cl[,"2"])
clPairs(irisMatrix,cl=cl[,"3"])

par(mfrow = c(1,2))
dimens <- c(1,2)
coordProj(irisMatrix, classification=cl[,"2"], dimens=dimens)
coordProj(irisMatrix, classification=cl[,"3"], dimens=dimens)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{hclass}
\alias{hclass}
\title{
  Classifications from Hierarchical Agglomeration 
}
\description{
  Determines the classifications corresponding to different numbers of groups
  given merge pairs from hierarchical agglomeration.
}
\usage{
hclass(hcPairs, G)
}
\arguments{
  \item{hcPairs}{
    A numeric two-column matrix in which the \emph{i}th row gives the minimum 
    index for observations in each of the two clusters merged at the
    \emph{i}th stage of agglomerative hierarchical clustering.
  }
  \item{G}{
    An integer or vector of integers giving the number of clusters for which
    the corresponding classfications are wanted.
  }
}
\value{
  A matrix with \code{length(G)} columns, each column 
  corresponding to a classification. Columns are indexed by the character
  representation of the integers in \code{G}.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{hc}},
  \code{\link{hcE}}
}
\examples{
data(iris)
irisMatrix <- iris[,1:4]

hcTree <- hc(modelName="VVV", data = irisMatrix)
cl <- hclass(hcTree,c(2,3))

par(pty = "s", mfrow = c(1,1))
clPairs(irisMatrix,cl=cl[,"2"])
clPairs(irisMatrix,cl=cl[,"3"])
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{hypvol}
\alias{hypvol}
\title{
  Aproximate Hypervolume for Multivariate Data
}
\description{
  Computes a simple approximation to the hypervolume of a multivariate
  data set.
}
\usage{
hypvol(data, reciprocal=FALSE)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{reciprocal}{
    A logical variable indicating whether or not the reciprocal
    hypervolume is desired rather than the hypervolume itself. The
    default is to return the approximate hypervolume.  
  }
}
\value{
  Computes the hypervolume by two methods: simple variable bounds
  and principal components, and returns the minimum value.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611:631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
hypvol(irisMatrix)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{lansing}
\alias{lansing}
\non_function{}
\title{Maple trees in Lansing Woods}
\usage{data(lansing)}
\description{
The \code{lansing} data frame has 1217 rows and 3 columns.
The first two columns give the location, the third column the tree type.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{x}{a numeric vector}
    \item{y}{a numeric vector}
    \item{species}{a factor with levels \code{hickory} and \code{maple} }
  }
}
\source{
 D.J. Gerrard, Research Bulletin No. 20, Agricultural Experimental
 Station, Michigan State University, 1969.
}
\seealso{
  \code{\link{grid1}},
  \code{\link{dens}}
}
\examples{
data(lansing)
plot(lansing[,1:2], pch=as.integer(lansing[,3]),
     col=as.integer(lansing[,3]), main="Lansing Woods tree types")
}
\keyword{datasets}

\eof
\name{map}
\alias{map}
\title{
  Classification given Probabilities 
}
\description{
  Converts a matrix in which each row sums to \emph{1} 
  into the nearest matrix of \emph{(0,1)} indicator variables.
}
\usage{
map(z, warn=TRUE, \dots)
}
\arguments{
  \item{z}{
    A matrix (for example a matrix of conditional
    probabilities  in which each row sums to 1
    as produced by the E-step of the EM algorithm).
  }
  \item{warn}{
    A logical variable indicating whether or not a warning should be
    issued when there are some columns of \code{z} for which no row
    attains a maximum.
  }
  \item{\dots }{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  A integer vector with one entry for each row of z,
  in which the \emph{i}-th value is the column index at which the
  \emph{i}-th row of \code{z} attains a maximum.
  
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{unmap}},
  \code{\link{estep}},
  \code{\link{em}},
  \code{\link{me}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

emEst <- me(modelName = "VVV", data = irisMatrix, z = unmap(irisClass))

map(emEst$z)
}
\keyword{cluster}
% docclass is function

\eof
\name{mapClass}
\alias{mapClass}
\title{
  Correspondence between classifications.
}
\description{
  Best correspondence between classes given two vectors viewed
  as alternative classifications of the same object.
}
\usage{
mapClass(a, b)
}
\arguments{
  \item{a}{
    A numeric or character vector of class labels.
  }
  \item{b}{
    A numeric or character vector of class labels.
    Must have the same length as
    \code{a}.
  }
  
}
\value{
  A list with two named elements, 
  \code{aTOb} and 
  \code{bTOa} which are themselves lists. 
  The \code{aTOb} list has a component corresponding
  to each unique element of \code{a}, which gives
  the element or elements of \code{b} 
  that result in the closest class correspondence.
  
  The \code{bTOa} list has a component corresponding
  to each unique element of \code{b}, which gives
  the element or elements of \code{a} 
  that result in the closest class correspondence.
}
\seealso{
  \code{\link{mapClass}},
  \code{\link{classError}},
  \code{\link{table}}
}
\examples{
a <- rep(1:3, 3)
a
b <- rep(c("A", "B", "C"), 3)
b
mapClass(a, b)
a <- sample(1:3, 9, replace = TRUE)
a
b <- sample(c("A", "B", "C"), 9, replace = TRUE)
b
mapClass(a, b)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mclust-internal}
\title{Internal MCLUST functions}
\alias{[.mclustDAtest}
\alias{mvn2plot}
\alias{vecnorm}
\alias{traceW}
\alias{unchol}
\alias{shapeO}
\alias{orth2}
\alias{mclustProjControl}
\alias{mclust2DplotControl}
\alias{[.EMclust}
\alias{[.EMclustN}
\alias{nextPerm}
\alias{charconv}
\alias{estep2}
\description{
  Internal tools functions.
}
\details{
  These are not to be called by the user directly.
}
\keyword{internal}

\eof
\name{mclust1Dplot}
\alias{mclust1Dplot}
\title{
  Plot one-dimensional data modelled by an MVN mixture.
}
\description{
  Plot one-dimensional data given parameters of an MVN mixture model 
  for the data.
}
\usage{
mclust1Dplot(data, \dots,
             type = c("classification","uncertainty","density","errors"),
             ask = TRUE, symbols, grid = 100, identify = FALSE, CEX = 1, xlim) 
}
\arguments{
  \item{data}{
    A numeric vector of observations.
    Categorical variables are not allowed.
  }
  \item{\dots}{
    One or more of the following:
    \describe{
      \item{classification}{
	A numeric or character vector representing a classification of
	observations (rows) of \code{data}.
      }
      \item{uncertainty}{
	A numeric vector of values in \emph{(0,1)} giving the
	uncertainty of each data point.
      }
      \item{z}{
	A matrix in which the \emph{[i,k]}the entry gives the
	probability of observation \emph{i} belonging to the \emph{k}th class. 
	Used to compute \code{classification} and
	\code{uncertainty} if those arguments aren't available.
      }
      \item{truth}{
	A numeric or character vector giving a known
	classification of each data point.
	If \code{classification}
	or \code{z} is also present, this  
	is used for displaying classification errors.
      }
      \item{mu}{
	A vector whose entries are the means of each group. 
      }
      \item{sigma}{
	Either a vector whose entries are the variances for each group or a
	scalar giving a common variance for the groups.
      }
      \item{pro}{
	The vector of mixing proportions.
      }
    }
  }
  \item{type}{
    Any subset of
    \code{c("classification","uncertainty","density","errors")}.
    The function will produce the corresponding plot if
    it has been supplied sufficient information to do so. If more than
    one plot is possible then users will be asked to choose from a menu
    if \code{ask=TRUE}. 
  }
  \item{ask}{
    A logical variable indicating whether or not a menu should be produced
    when more than one plot is possible. The default is \code{ask=TRUE}.
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to
    each unique class \code{classification}. Elements in \code{symbols}
    correspond to classes in \code{classification} in order of
    appearance in the observations (the order used by the 
    function \code{unique}). The default is to use a single plotting
    symbol \emph{|}. Classes are delineated by showing them in separate
    lines above the whole of the data.
  }
  \item{grid}{
    Number of grid points to use.
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. 
    The default value is 1.
  }
  \item{xlim}{
    An argument specifying bounds of the plot.
    This may be useful for when comparing plots.
  }
}
\section{Side Effects}{
  One or more plots showing location of the mixture components, classification,
  uncertainty, density and/or classification errors. Points in the different
  classes are shown in separate lines above the whole of the data.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mclust2Dplot}},
  \code{\link{clPairs}},
  \code{\link{coordProj}},
  \code{\link{do.call}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
y <- c(rnorm(n,-5), rnorm(n,0), rnorm(n,5))
yclass <- c(rep(1,n), rep(2,n), rep(3,n))

yEMclust <- summary(EMclust(y),y)

mclust1Dplot(y, identify = TRUE, truth = yclass, z = yEMclust$z, ask=FALSE,
             mu = yEMclust$mu, sigma = yEMclust$sigma, pro = yEMclust$pro)

do.call("mclust1Dplot",
        c(list(data = y, identify = TRUE, truth = yclass, ask=FALSE), 
        yEMclust))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.


\eof
\name{mclust2Dplot}
\alias{mclust2Dplot}
\title{
  Plot two-dimensional data modelled by an MVN mixture.
}
\description{
  Plot two-dimensional data given parameters of an MVN mixture model 
  for the data.
}
\usage{
mclust2Dplot(data, \dots,
             type = c("classification","uncertainty","errors"), ask = TRUE,
             quantiles = c(0.75, 0.95), symbols, scale = FALSE,
             identify = FALSE, CEX = 1, PCH = ".", xlim, ylim,
             swapAxes = FALSE) 
}
\arguments{
  \item{data}{
    A numeric matrix or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables. 
    In this case the data are two dimensional, so there are two columns.
  }
  \item{\dots}{
    One or more of the following:
    \describe{
      \item{classification}{
	A numeric or character vector representing a classification of
	observations (rows) of \code{data}.
      }
      \item{uncertainty}{
	A numeric vector of values in \emph{(0,1)} giving the
	uncertainty of each data point.
      }
      \item{z}{
	A matrix in which the \emph{[i,k]}the entry gives the
	probability of observation \emph{i} belonging to the \emph{k}th class. 
	Used to compute \code{classification} and
	\code{uncertainty} if those arguments aren't available.
      }
      \item{truth}{
	A numeric or character vector giving a known
	classification of each data point.
	If \code{classification}
	or \code{z} is also present, this  
	is used for displaying classification errors.
      }
      \item{mu}{
	A matrix whose columns are the means of each group. 
      }
      \item{sigma}{
	A three dimensional array 
	in which \code{sigma[,,k]} gives the covariance
	for the \emph{k}th group.
      }
      \item{decomp}{
	A list with \code{scale},
	\code{shape} and 
	\code{orientation}
	components giving an alternative form for the covariance structure 
	of the mixture model.
      }
    }
  }
  \item{type}{
    Any subset of 
    \code{c("classification","uncertainty","errors")}. 
    The function will produce the corresponding plot if
    it has been supplied sufficient information to do so. If more than one
    plot is possible then users will be asked to choose from a menu
    if \code{ask=TRUE}.
  }
  \item{ask}{
    A logical variable indicating whether or not a menu should be produced
    when more than one plot is possible. 
    The default is \code{ask=TRUE}.
  }
  \item{quantiles}{
    A vector of length 2 giving quantiles used in plotting uncertainty.
    The smallest symbols correspond to the smallest quantile 
    (lowest uncertainty), medium-sized (open) symbols to points falling between 
    the given quantiles, and large (filled) symbols to those in the largest 
    quantile (highest uncertainty).
    The default is \emph{(0.75,0.95)}.
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to each
    unique class \code{classification}. 
    Elements in \code{symbols}
    correspond to classes in \code{classification}
    in order of appearance in the observations
    (the order used by the S-PLUS function 
    \code{unique}). 
    Default: If \emph{G} is the number of groups in the classification, the first
    \emph{G} symbols in \code{.Mclust\$symbols},
    otherwise if \emph{G} is less than 27 then the first \emph{G} capital
    letters in the Roman alphabet.
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and
    thus preserve the shape of the distribution.
    Default: \code{scale=FALSE} 
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. 
    The default value is 1.
  }
  \item{PCH}{
    An argument specifying the symbol to be used when a classificatiion
    has not been specified for the data. The default value is a small dot ".".
  }
  \item{xlim, ylim }{
    An argument specifying bounds for the ordinate, abscissa of the plot.
    This may be useful for when comparing plots.
  }
  \item{swapAxes}{
    A logical variable indicating whether or not the axes should be swapped
    for the plot.
  }
}
\section{Side Effects}{
  One or more plots showing location of the mixture components, classification,
  uncertainty, and/or classification errors.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{surfacePlot}},
  \code{\link{clPairs}},
  \code{\link{coordProj}},
  \code{\link{randProj}},
  \code{\link{spinProj}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))

xEMclust <- summary(EMclust(x),x)

mclust2Dplot(x, truth = xclass, z = xEMclust$z, ask=FALSE,
                mu = xEMclust$mu, sigma = xEMclust$sigma)

do.call("mclust2Dplot", c(list(data = x, truth = xclass, ask=FALSE), xEMclust))

}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mclustDA}
\alias{mclustDA}
\alias{print.mclustDA}
\title{
  MclustDA discriminant analysis.
}
\description{
  MclustDA training and testing.
}
\usage{
mclustDA(trainingData, labels, testData, G=1:6, verbose = FALSE)
}
\arguments{
  \item{trainingData}{
    
    A numeric vector, matrix, or data frame of training observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{labels}{
    A numeric or character vector assigning a class label to each 
    training observation.
  }
  \item{testData}{
    A numeric vector, matrix, or data frame of training observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{G}{
    An integer vector specifying the numbers of mixture components (clusters)
    to be considered for each class.
    Default: \code{1:6}.
  }
  \item{verbose}{
    A logical variable telling whether or not to print an indication that the
    function is in the training phase, which may take some time to complete.
  }
}
\value{
  A list with the following components: 
  \item{testClassification}{ 
    \code{mclustDA} classification of the test data.
  }
  \item{trainingClassification}{ 
    \code{mclustDA} classification of the training data.
  }
  \item{errorRate}{ 
    The error rate for the \code{mclustDA} classification 
    of the training data compared to the known labels,
  }
  \item{summary}{ 
    Gives the best model and number of clusters for each training class.
  }
  \item{models}{ 
    The mixture models used to fit the known classes.
  }
  \item{postProb}{ 
    A  matrix whose \emph{[i,k]}th entry is the probability that observation
    \emph{i} in the test data belongs to the \emph{k}th class.
  }
}
\section{Details}{
  The following models are compared in \code{Mclust}: \cr\cr
  "E" for spherical, equal variance (one-dimensional) \cr
  "V" for spherical, variable variance (one-dimensional) \cr\cr
  "EII": spherical, equal volume \cr
  "VII": spherical, unequal volume \cr
  "EEI": diagonal, equal volume, equal shape \cr
  "VVI": diagonal, varying volume, varying shape \cr
  "EEE": ellipsoidal, equal volume, shape, and orientation \cr
  "VVV": ellipsoidal, varying volume, shape, and orientation 
  
  \code{mclustDA} is a simplified function combining
  \code{mclustDAtrain} and \code{mclustDAtest} and their summaries.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{plot.mclustDA}}, 
  \code{\link{mclustDAtrain}}, 
  \code{\link{mclustDAtest}},
  \code{\link{compareClass}}, 
  \code{\link{classError}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))

\dontrun{
par(pty = "s")
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)
}

odd <- seq(from = 1, to = 2*n, by = 2)
even <- odd + 1
testMclustDA <- mclustDA(trainingData = x[odd, ], labels = xclass[odd], 
                         testData = x[even,])

clEven <- testMclustDA$testClassification ## classify training set
compareClass(clEven,xclass[even])
\dontrun{plot(testMclustDA, trainingData = x[odd, ], labels = xclass[odd], 
              testData = x[even,])}
}
\keyword{cluster}
% docclass is function

\eof
\name{mclustDAtest}
\alias{mclustDAtest}
\title{
  MclustDA Testing
}
\description{
  Testing phase for MclustDA discriminant analysis.
}
\usage{
mclustDAtest(data, models)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations
    to be classified.
  }
  \item{models}{
    A list of MCLUST-style models including parameters, usually the
    result of applying \code{mclustDAtrain} to some training data. 
  }
}
\value{
  A matrix in which the \code{[i,j]}th entry is the 
  density for that test observation \emph{i} in the model for class \emph{j}.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{summary.mclustDAtest}},
  \code{\link{mclustDAtrain}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
\dontrun{
par(pty = "s")
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)
}

odd <- seq(1, 2*n, 2)
train <- mclustDAtrain(x[odd, ], labels = xclass[odd]) ## training step
summary(train)

even <- odd + 1
test <- mclustDAtest(x[even, ], train) ## compute model densities
summary(test)$class ## classify training set
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mclustDAtrain}
\alias{mclustDAtrain}
\title{
  MclustDA Training
}
\description{
  Training phase for MclustDA discriminant analysis.
}
\usage{
mclustDAtrain(data, labels, G, emModelNames, eps, tol, itmax, 
              equalPro, warnSingular, verbose)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{labels}{
    A numeric or character vector assigning a class label to each observation.
  }
  \item{G}{
    An integer vector specifying the numbers of Gaussian mixture
    components (clusters) for which the BIC is to be calculated (the
    same specification is used for all classes). Default: \emph{1:9}.
  }
  \item{emModelNames}{
    A vector of character strings indicating the models to be fitted 
    in the EM phase of clustering. Possible models: \cr
    "E" for spherical, equal variance (one-dimensional)\cr 
    "V" for spherical, variable variance (one-dimensional)\cr 
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation \cr\cr
    The default is \code{.Mclust\$emModelNames}.
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due
    to computational singularity in covariances. Smaller values of
    \code{eps} allow computations to proceed nearer to singularity. The
    default is \code{.Mclust\$eps}. 
  }
  \item{tol}{
    A scalar tolerance for relative convergence of the loglikelihood.
    The default is \code{.Mclust\$tol}.
  }
  \item{itmax}{
    An integer limit on the number of EM iterations.
    The default is \code{.Mclust\$itmax}.
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model. The default is \code{.Mclust\$equalPro}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{warnSingular=FALSE}.
  }
  \item{verbose}{
    A logical value indicating whether or not to print the models and
    numbers of components for each class.
    Default:\code{verbose=TRUE}.
  }
}
\value{
  A list in which each element gives the optimal parameters for the model
  best fitting each class according to BIC.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{summary.mclustDAtrain}}, 
  \code{\link{mclustDAtest}}, 
  \code{\link{EMclust}}, 
  \code{\link{hc}},
  \code{\link{mclustOptions}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
par(pty = "s")
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
\dontrun{
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)}

odd <- seq(1, 2*n, 2)
train <- mclustDAtrain(x[odd, ], labels = xclass[odd]) ## training step
summary(train)

even <- odd + 1
test <- mclustDAtest(x[even, ], train) ## compute model densities
clEven <- summary(test)$class ## classify training set
compareClass(clEven,xclass[even])
}
\keyword{cluster}
% docclass is function

\eof
\name{mclustOptions}
\alias{mclustOptions}
\title{
  Set control values for use with MCLUST.
}
\description{
  Supplies a list of values including tolerances for singularity and
  convergence assessment, and an enumeration of models for use
  with \emph{MCLUST}.
}
\usage{
mclustOptions(eps, tol, itmax, equalPro, warnSingular, emModelNames,
              hcModelName, symbols) 
}
\arguments{
  \item{eps}{
    A scalar tolerance associated with deciding when to terminate
    computations due to computational singularity in
    covariances. Smaller values of \code{eps} allow computations to
    proceed nearer to singularity. The default is the relative machine
    precision \code{.Machine\$double.eps}, which is approximately
    \$2e-16\$ on IEEE-compliant machines. 
  }
  \item{tol}{
    A vector of length two giving relative convergence tolerances for the 
    loglikelihood and for parameter convergence in the inner loop for models
    with iterative M-step ("VEI", "VEE", "VVE", "VEV"), respectively.
    The default is \code{c(1.e-5,1.e-5)}.
  }
  \item{itmax}{
    A vector of length two giving integer limits on the number of EM
    iterations and on the number of iterations in the inner loop for
    models with iterative M-step ("VEI", "VEE", "VVE", "VEV"),
    respectively. The default is \code{c(Inf,Inf)} allowing termination
    to be completely governed by \code{tol}. 
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model. Default: \code{equalPro = FALSE}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered. The default is
    \code{warnSingular = TRUE}. 
  }
  \item{emModelNames}{
    A vector of character strings associated with multivariate models in
    MCLUST. The default includes strings encoding all of the
    multivariate models available:\cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape\cr 
    "VEI": diagonal, varying volume, equal shape\cr 
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation 
  }
  \item{hcModelName}{
    A vector of two character strings giving the name of the model to be
    used in the hierarchical clustering phase for univariate and
    multivariate data, respectively, in \code{EMclust} and
    \code{EMclustN}. The default is \code{c("V","VVV")}, giving the
    unconstrained model in each case. 
  }
  \item{symbols}{
    A vector whose entries are either integers corresponding to graphics
    symbols or single characters for plotting for
    classifications. Classes are assigned symbols in the given
    order. The default is
    \code{c(17,0,10,4,11,18,6,7,3,16,2,12,8,15,1,9,14,13,5)}. 
  }
}
\value{
  A named list in which the names are the names of the arguments
  and the values are the values supplied to the arguments.
}
\details{
  \code{mclustOptions} is provided for assigning values
  to the \code{.Mclust} list, which is used
  to supply default values to various functions in MCLUST.
  
  Calls to \code{mclustOptions} do not in themselves affect
  the outcome of computations.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{.Mclust}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

.Mclust
.Mclust <- mclustOptions(tol = 1.e-6, emModelNames = c("VII", "VVI", "VVV"))
.Mclust
irisBic <- EMclust(irisMatrix)
summary(irisBic, irisMatrix)
.Mclust <- mclustOptions() # restore default values
.Mclust
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{me}
\alias{me}
\title{
  EM algorithm starting with M-step for parameterized MVN mixture models.
}
\description{
  Implements the EM algorithm for parameterized MVN mixture models,
  starting with the maximization step.
}
\usage{
me(modelName, data, z, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model: \cr
    "E": equal variance  (one-dimensional) \cr
    "V": variable variance (one-dimensional)\cr 
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape\cr 
    "VEI": diagonal, varying volume, equal shape\cr 
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation 
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the ith observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{\dots}{
    Any number of the following:
    \itemize{
      \item{eps}{
	A scalar tolerance for deciding when to terminate computations due
	to computational singularity in covariances. Smaller values of
	\code{eps} allow computations to proceed nearer to singularity. The
	default is \code{.Mclust\$eps}. 
	
	For those models with iterative M-step ("VEI", "VEV"), two
	values can be entered for \code{eps}, in which case the second
	value is used for determining singularity in the M-step. 
      }
      \item{tol}{
	A scalar tolerance for relative convergence of the
	loglikelihood. The default is \code{.Mclust\$tol}.
	
	For those models with iterative M-step ("VEI", "VEV"), two values
	can be entered for \code{tol}, in which case the second value
	governs parameter convergence in the M-step.  
      }
      \item{itmax}{
	An integer limit on the number of EM iterations. The default is
	\code{.Mclust\$itmax}. For those models with iterative M-step
	("VEI", "VEV"), two values can be entered for \code{itmax}, in
	which case the second value is an upper limit on the number of
	iterations in the M-step. 
      }
      \item{equalPro}{
	Logical variable indicating whether or not the mixing proportions are
	equal in the model. The default is \code{.Mclust\$equalPro}.
      }
      \item{warnSingular}{
	A logical value indicating whether or not a warning should be issued
	whenever a singularity is encountered.
	The default is \code{.Mclust\$warnSingular}.
      }
      \item{noise}{
	A logical value indicating whether or not the model includes a
	Poisson noise component. The default assumes there is no noise
	component. 
      }
      \item{Vinv}{
	An estimate of the reciprocal hypervolume of the data region. The
	default is determined by applying function \code{hypvol} to the
	data. Used only when \code{noise = TRUE}. 
      }
    }
  }
}
\value{
  A list including the following components: 
  \item{mu}{
    A matrix whose kth column is the mean of the \emph{k}th component of
    the mixture model. 
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{
    A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{modelName}{
    A character string identifying the model (same as the input argument).
  }
  \item{Attributes:}{
    \itemize{
      \item{"info"}{
	Information on the iteration.
      }
      \item{"warn"}{
	An appropriate warning if problems are encountered in the computations.
      }
    }
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{meE}},...,
  \code{\link{meVVV}},
  \code{\link{em}},
  \code{\link{mstep}},
  \code{\link{estep}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]
 
me(modelName = "VVV", data = irisMatrix, z = unmap(irisClass))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.


\eof
\name{meE}
\alias{meE}
\alias{meV}
\alias{meEII}
\alias{meVII}
\alias{meEEI}
\alias{meVEI}
\alias{meEVI}
\alias{meVVI}
\alias{meEEE}
\alias{meEEV}
\alias{meVEV}
\alias{meVVV}
\title{
  EM algorithm starting with M-step for a parameterized MVN mixture model.
}
\description{
  Implements the EM algorithm for a parameterized MVN mixture model,
  starting with the maximization step.
}
\usage{
meE(data, z, eps, tol, itmax, equalPro, warnSingular, 
    noise = FALSE, Vinv)
meV(data, z, eps, tol, itmax, equalPro, warnSingular, 
    noise = FALSE, Vinv)
meEII(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meVII(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meEEI(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meVEI(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meEVI(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meVVI(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meEEE(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meEEV(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meVEV(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
meVVV(data, z, eps, tol, itmax, equalPro, warnSingular, 
      noise = FALSE, Vinv)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the ith observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due to
    computational singularity in covariances.  
    Smaller values of \code{eps} allows computations to
    proceed nearer to singularity. 
    The default is \code{.Mclust\$eps}.
  }
  \item{tol}{
    A scalar tolerance for relative convergence of the loglikelihood values. 
    The default is \code{.Mclust\$tol}.
  }
  \item{itmax}{
    An integer limit on the number of EM iterations. 
    The default is \code{.Mclust\$itmax}.
  }
  \item{equalPro}{
    Logical variable indicating whether or not the mixing proportions are
    equal in the model.
    The default is \code{.Mclust\$equalPro}.
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{.Mclust\$warnSingular}.
  }
  \item{noise}{
    A logical value indicating whether or not the model includes a Poisson
    noise component.
    The default assumes there is no noise component.
  }
  \item{Vinv}{
    An estimate of the reciprocal hypervolume of the data region.
    The default is determined by applying function 
    \code{hypvol} to the data.
    Used only when \code{noise = TRUE}.
  }
}
\value{
  A list including the following components: 
  \item{mu}{
    A matrix whose kth column is the mean of the \emph{k}th
    component of the mixture model.
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{ A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{z}{ 
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{modelName}{
    Character string identifying the model.
  }
  \item{loglik}{
    The logliklihood for the data in the mixture model. 
  }
  \item{Attributes:}{ 
    The return value also has the following attributes:
    \item \code{"info"}: Information on the iteration.
    \item \code{"warn"}: An appropriate warning if problems are
    encountered in the computations. 
    
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{em}},
  \code{\link{me}},
  \code{\link{estep}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

meVVV(data = irisMatrix, z = unmap(irisClass))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mstep}
\alias{mstep}
\title{
  M-step in the EM algorithm for parameterized MVN mixture models.
}
\description{
  Maximization step in the EM algorithm for parameterized MVN mixture models.
}
\usage{
mstep(modelName, data, z, \dots)
}
\arguments{
  \item{modelName}{
    A character string indicating the model: \cr\cr
    "E": equal variance  (one-dimensional) \cr
    "V": variable variance (one-dimensional) \cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume and shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume and shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the ith observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{\dots}{
    Any number of the following:
    \describe{
      \item{equalPro}{
	A logical value indicating whether or not the components in the
	model are present in equal proportions. The default is
	\code{.Mclust\$equalPro}.  
      }
      \item{noise}{
	A logical value indicating whether or not the model includes a Poisson
	noise component. The default assumes there is no noise component.
      }
      \item{eps}{
	A scalar tolerance for deciding when to terminate computations due
	to computational singularity in covariances. Smaller values of
	\code{eps} allows computations to proceed nearer to singularity. The
	default is \code{.Mclust\$eps}.  
	
	Not used for models "EII", "VII", "EEE", "VVV".
      }
      \item{tol}{
	For models with iterative M-step ("VEI", "VEE", "VVE", "VEV"), 
	a scalar tolerance for relative convergence of the parameters. 
	The default is \code{.Mclust\$tol}. 
      }
      \item{itmax}{
	For models with iterative M-step ("VEI", "VEE", "VVE", "VEV"), 
	an integer limit on the number of EM iterations. 
	The default is \code{.Mclust\$itmax}. 
      }
      \item{warnSingular}{
	A logical value indicating whether or not a warning should be issued
	whenever a singularity is encountered.
	The default is \code{.Mclust\$warnSingular}.
	
	Not used for models "EII", "VII", "EEE", "VVV".
      }
    }
  }
}
\value{
  A list including the following components: 
  
  \item{mu}{
    A matrix whose kth column is the
    mean of the \emph{k}th component of the mixture model.
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{
    A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{modelName}{
    A character string identifying the model (same as the input argument).
  }
  \item{Attributes:}{
    \item \code{"info"}: Information on the iteration.
    \item \code{"warn"}: An appropriate warning if problems are
    encountered in the computations. 
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mstepE}}, \ldots,
  \code{\link{mstepVVV}},
  \code{\link{me}},
  \code{\link{estep}},
  \code{\link{mclustOptions}}.
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]
 
mstep(modelName = "VII", data = irisMatrix, z = unmap(irisClass))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mstepE}
\alias{mstepE}
\alias{mstepV}
\alias{mstepEII}
\alias{mstepVII}
\alias{mstepEEI}
\alias{mstepVEI}
\alias{mstepEVI}
\alias{mstepVVI}
\alias{mstepEEE}
\alias{mstepEEV}
\alias{mstepVEV}
\alias{mstepVVV}
\title{
  M-step in the EM algorithm for a parameterized MVN mixture model.
}
\description{
  Maximization step in the EM algorithm for a parameterized MVN mixture model.
}
\usage{
mstepE(data, z, equalPro, noise = FALSE, \dots)
mstepV(data, z, equalPro, noise = FALSE, \dots)
mstepEII(data, z, equalPro, noise = FALSE, \dots)
mstepVII(data, z, equalPro, noise = FALSE, \dots)
mstepEEI(data, z, equalPro, noise = FALSE, eps, warnSingular, \dots)
mstepVEI(data, z, equalPro, noise = FALSE, eps, tol, itmax, warnSingular, \dots)
mstepEVI(data, z, equalPro, noise = FALSE, eps, warnSingular, \dots)
mstepVVI(data, z, equalPro, noise = FALSE, eps, warnSingular, \dots)
mstepEEE(data, z, equalPro, noise = FALSE, \dots)
mstepEEV(data, z, equalPro, noise = FALSE, eps, warnSingular, \dots)
mstepVVV(data, z, equalPro, noise = FALSE, \dots)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the ith observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{equalPro}{
    A logical value indicating whether or not the components in the model are 
    present in equal proportions.
    The default is \code{.Mclust\$equalPro}. 
  }
  \item{noise}{
    A logical value indicating whether or not the model includes a Poisson
    noise component.
    The default assumes there is no noise component.
  }
  \item{eps}{
    A scalar tolerance for deciding when to terminate computations due to
    computational singularity in covariances.
    Smaller values of \code{eps} allows computations to
    proceed nearer to singularity. 
    The default is \code{.Mclust\$eps}. 
    
    Not used for models "EII", "VII", "EEE", "VVV".
  }
  \item{tol}{
    For models with iterative M-step ("VEI", "VEE", "VVE", "VEV"), 
    a scalar tolerance for relative convergence of the parameters. 
    The default is \code{.Mclust\$tol}. 
  }
  \item{itmax}{
    For models with iterative M-step ("VEI", "VEE", "VVE", "VEV"), 
    an integer limit on the number of EM iterations. 
    The default is \code{.Mclust\$itmax}. 
  }
  \item{warnSingular}{
    A logical value indicating whether or not a warning should be issued
    whenever a singularity is encountered.
    The default is \code{.Mclust\$warnSingular}.
    
    Not used for models "EII", "VII", "EEE", "VVV".
  }
  \item{\dots }{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  A list including the following components: 
  
  \item{mu}{
    A matrix whose kth column is the
    mean of the \emph{k}th component of the mixture model.
  }
  \item{sigma}{
    For multidimensional models, a three dimensional array 
    in which the \code{[,,k]}th entry gives the
    the covariance for the \emph{k}th group in the best model. <br>
    For one-dimensional models, either a scalar giving a common variance for
    the groups or a vector whose entries are the variances for each group
    in the best model.
  }
  \item{pro}{
    A vector whose \emph{k}th component
    is the mixing proportion for the \emph{k}th component of the mixture model.
  }
  \item{z}{
    A matrix whose \code{[i,k]}th entry is the
    conditional probability of the \emph{i}th observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{modelName}{
    A character string identifying the model (same as the input argument).
  }
  \item{Attributes:}{
    \item \code{"info"} Information on the iteration.
    \item \code{"warn"} An appropriate warning if problems are
    encountered in the computations. 
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mstep}},
  \code{\link{me}},
  \code{\link{estep}},
  \code{\link{mclustOptions}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

mstepVII(data = irisMatrix, z = unmap(irisClass))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mvn}
\alias{mvn}
\title{
  Multivariate Normal Fit
}
\description{
  Computes the mean, covariance, and loglikelihood from fitting a single
  MVN or Gaussian to given data.
}
\usage{
mvn( modelName, data)
}
\arguments{
  \item{modelName}{
    A character string representing a model name. This can be either
    \code{"Spherical"}, \code{"Diagonal"}, or \code{"Ellipsoidal"} or an
    MCLUST-style model name: \cr 
    "E", "V", "X" (one-dimensional) \cr
    "EII", "VII", "XII" (spherical) \cr
    "EEI", "VEI", "EVI", "VVI", "XXI" (diagonal) \cr
    "EEE", "EEV", "VEV", "VVV", "XXX" (ellipsoidal) 
  }
  \item{data}{
    A numeric vector, matrix, or data frame of observations. Categorical
    variables are not allowed. If a matrix or data frame, rows
    correspond to observations and columns correspond to variables. 
  }
}
\value{
  A list of including the parameters of the Gaussian model best fitting the
  data, and the corresponding loglikelihood for the data under the model.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mvnX}},
  \code{\link{mvnXII}},
  \code{\link{mvnXXI}},
  \code{\link{mvnXXX}},
  \code{\link{mstep}}
}
\examples{
n <- 1000

set.seed(0)
x <- rnorm(n, mean = -1, sd = 2)
mvn(modelName = "X", x) 

mu <- c(-1, 0, 1)

set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% (2*diag(3)), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvn(modelName = "XII", x) 
mvn(modelName = "Spherical", x) 

set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% diag(1:3), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvn(modelName = "XXI", x)
mvn(modelName = "Diagonal", x)

Sigma <- matrix(c(9,-4,1,-4,9,4,1,4,9), 3, 3)
set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% chol(Sigma), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvn(modelName = "XXX", x) 
mvn(modelName = "Ellipsoidal", x) 
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{mvnX}
\alias{mvnX}
\alias{mvnXII}
\alias{mvnXXI}
\alias{mvnXXX}
\title{
  Multivariate Normal Fit
}
\description{
  Computes the mean, covariance, and loglikelihood from fitting a single
  MVN or Gaussian.
}
\usage{
mvnX(data)
mvnXII(data)
mvnXXI(data)
mvnXXX(data)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
}
\value{
  A list of including the parameters of the Gaussian model best fitting the
  data, and the corresponding loglikelihood for the data under the model.
}
\details{
  \code{mvnXII} computes the best fitting
  Gaussian with the covariance restricted to be a multiple of the identity. 
  \code{mvnXXI} computes the best fitting
  Gaussian with the covariance restricted to be diagonal. 
  \code{mvnXXX} computes the best fitting
  Gaussian with ellipsoidal (unrestricted) covariance. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mvn}},
  \code{\link{mstepE}}
}
\examples{
n <- 1000

set.seed(0)
x <- rnorm(n, mean = -1, sd = 2)
mvnX(x) 

mu <- c(-1, 0, 1)

set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% (2*diag(3)), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvnXII(x) 

set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% diag(1:3), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvnXXI(x)

Sigma <- matrix(c(9,-4,1,-4,9,4,1,4,9), 3, 3)
set.seed(0)
x <- sweep(matrix(rnorm(n*3), n, 3) \%*\% chol(Sigma), 
           MARGIN = 2, STATS = mu, FUN = "+")
mvnXXX(x) 
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{partconv}
\alias{partconv}

\title{Convert partitioning into numerical vector.}
\description{\code{partconv} converts a partitioning into a numerical
  vector. The second argument is used to force consecutive numbers
  (default) or not.
}
\usage{
partconv(x, consec=TRUE)
}

\arguments{
  \item{x}{Partitioning. Maybe numerical or not.}
  \item{consec}{Logical flag, whether or not to use consecutive class
    numbers.}
}
\value{
  Vector of class numbers.
}

\examples{
data(iris)
partconv(iris[,5])

cl <- sample(1:10, 25, replace=TRUE)
partconv(cl, consec=FALSE)
partconv(cl, consec=TRUE)
}

\keyword{cluster}

\eof
\name{partuniq}
\alias{partuniq}
\title{
  Classifies Data According to Unique Observations
}
\description{
  Gives a one-to-one mapping from unique observations to rows of a data matrix.
}
\usage{
partuniq(x)
}
\arguments{
  \item{x}{Matrix of observations.}
}
\value{
  A vector of length \code{nrow(x)} with integer entries. An observation
  \code{k} is assigned an integer \code{i} whenever observation \code{i}
  is the first row of \code{x} that is identical to observation \code{k}
  (note that \code{i <= k}).
}
\examples{
data(iris)
partuniq(as.matrix(iris[,1:4]))
}
\keyword{cluster}
% Converted by Sd2Rd version 0.3-2.



\eof
\name{plot.Mclust}
\alias{plot.Mclust}
\title{
  Plot Model-Based Clustering Results
}
\description{
  Plot model-based clustering results: BIC, classification, uncertainty and 
  (for one- and two-dimensional data) density.
}
\usage{
plot.Mclust(x, data, dimens = c(1, 2), scale = FALSE, \dots)
}
\arguments{
  \item{x}{
    Output from \code{Mclust}.
  }
  \item{data}{
    The data used to produce \code{x}.
  }
  \item{dimens}{
    An integer vector of length two specifying the dimensions for
    coordinate projections if the data is more than two-dimensional. The
    default is \code{c(1,2)} (the first two dimesions). 
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and thus preserve
    the shape of the distribution. Default: \code{scale=FALSE}  
  }
  \item{\dots}{
    Further arguments to the lower level plotting functions.
  }
  
}
\value{
  Plots selected via a menu including the following options: BIC values
  used for choosing the number of clusters For data in more than two
  dimensions, a pairs plot of the showing the classification, coordinate
  projections of the data, showing location of the mixture components,
  classification, and/or uncertainty. For one- and two- dimensional
  data, plots showing location of the mixture components,
  classification, uncertainty, and or density.  
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{Mclust}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisMclust <- Mclust(irisMatrix)

\dontrun{plot(irisMclust,irisMatrix)}
}
\keyword{cluster}
% docclass is function

\eof
\name{plot.mclustDA}
\alias{plot.mclustDA}
\title{
  Plotting method for MclustDA discriminant analysis.
}
\description{
  Plots training and test data, known training data classification,
  mclustDA test data classification, and/or training errors.
}
\usage{
plot.mclustDA(x, trainingData, labels, testData, dimens=c(1,2),
              scale = FALSE, identify=FALSE, \dots)
}
\arguments{
  \item{x}{
    The object produced by applying \code{mclustDA} with
    \code{trainingData} and classification \code{labels} to
    \code{testData}. 
  }
  \item{trainingData}{
    The numeric vector, matrix, or data frame of training observations
    used to obtain \code{x}.  
  }
  \item{labels}{
    The numeric or character vector assigning a class label to each
    training observation. 
  }
  \item{testData}{
    A numeric vector, matrix, or data frame of training
    observations. Categorical variables are not allowed. If a matrix or
    data frame, rows correspond to observations and columns correspond
    to variables. 
  }
  \item{dimens}{
    An integer vector of length two specifying the dimensions for coordinate
    projections if the data is more than two-dimensional.
    The default is \code{c(1,2)} (the first two dimesions).
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and thus preserve
    the shape of the distribution. Default: \code{scale=FALSE}  
  }
  \item{identify}{
    A logical variable indicating whether or not to print a title identifying
    the plot. Default: \code{identify=FALSE} 
  }
  \item{\dots}{
    Further arguments to the lower level plotting functions.
  }
}
\value{
  Plots selected via a menu including the following options: 
  training and test data, known training data classification,
  mclustDA test data classification, training errors.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mclustDA}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
\dontrun{
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)
}
odd <- seq(from = 1, to = 2*n, by = 2)
even <- odd + 1
testMclustDA <- mclustDA(trainingData = x[odd, ], labels = xclass[odd], 
                         testData = x[even,])

clEven <- testMclustDA$testClassification ## classify training set
compareClass(clEven,xclass[even])

\dontrun{
plot(testMclustDA, trainingData = x[odd, ], labels = xclass[odd], 
testData = x[even,])
}
} 
\keyword{cluster}
% docclass is function

\eof
\name{randProj}
\alias{randProj}
\title{
  Random projections for data in more than two dimensions modelled
  by an MVN mixture.
}
\description{
  Plots random projections given data in more than two dimensions
  and parameters of an MVN mixture model for the data.
}
\usage{
randProj(data, seeds = 0, \dots, 
         type = c("classification", "uncertainty", "errors"), ask = TRUE,
         quantiles = c(0.75,0.95), symbols, scale = FALSE, identify = FALSE, 
         CEX = 1, PCH = ".", xlim, ylim)
}
\arguments{
  \item{data}{
    A numeric matrix or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{seeds}{
    A vector of integers between 0 and 1000, specifying seeds for 
    the random projections. The default value is the single seed 0.
  }
  \item{\dots}{
    Any number of the following:
    \describe{
      \item{classification}{
	A numeric or character vector representing a classification of
	observations (rows) of \code{data}.
      }
      \item{uncertainty}{
	A numeric vector of values in \emph{(0,1)} giving the
	uncertainty of each data point.
      }
      \item{z}{
	A matrix in which the \emph{[i,k]}the entry gives the
	probability of observation \emph{i} belonging to the \emph{k}th class. 
	Used to compute \code{classification} and
	\code{uncertainty} if those arguments aren't available.
      }
      \item{truth}{
	A numeric or character vector giving a known classification of each
	data point. If \code{classification} or \code{z} is also present,
	this is used for displaying classification errors. 
      }
      \item{mu}{
	A matrix whose columns are the means of each group. 
      }
      \item{sigma}{
	A three dimensional array in which \code{sigma[,,k]} gives the
	covariance for the \emph{k}th group. 
      }
      \item{decomp}{
	A list with \code{scale}, \code{shape} and \code{orientation}
	components giving an alternative form for the covariance structure
	of the mixture model. 
      }
    }
  }
  \item{type}{
    Any subset of \code{c("classification","uncertainty","errors")}. The
    function will produce the corresponding plot if it has been supplied
    sufficient information to do so. If more than one plot is possible
    then users will be asked to choose from a menu if \code{ask=TRUE}.
  }
  \item{ask}{
    A logical variable indicating whether or not a menu should be produced
    when more than one plot is possible. 
    The default is \code{ask=TRUE}.
  }
  \item{quantiles}{
    A vector of length 2 giving quantiles used in plotting
    uncertainty. The smallest symbols correspond to the smallest
    quantile (lowest uncertainty), medium-sized (open) symbols to points
    falling between the given quantiles, and large (filled) symbols to
    those in the largest quantile (highest uncertainty). The default is
    \emph{(0.75,0.95)}. 
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to
    each unique class \code{classification}. Elements in \code{symbols}
    correspond to classes in \code{classification} in order of
    appearance in \code{classification} (the order used by the S-PLUS
    function \code{unique}). Default: If \emph{G} is the number of
    groups in the classification, the first \emph{G} symbols in
    \code{.Mclust\$symbols}, otherwise if \emph{G} is less than 27 then
    the first \emph{G} capital letters in the Roman alphabet. 
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and
    thus preserve the shape of the distribution.
    Default: \code{scale=FALSE} 
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. The default
    value is 1.
  }
  \item{PCH}{
    An argument specifying the symbol to be used when a classificatiion
    has not been specified for the data. The default value is a small dot ".".
  }
  \item{xlim, ylim }{
    Arguments specifying bounds for the ordinate, abscissa of the plot.
    This may be useful for when comparing plots.
  }
}
\value{
  Random projections of the data, possibly showing location of the mixture 
  components, classification, uncertainty, and classficaition errors.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{coordProj}},
  \code{\link{spinProj}},
  \code{\link{mclust2Dplot}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}},
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstepVVV(irisMatrix, unmap(irisClass))

par(pty = "s", mfrow = c(2,3))
randProj(irisMatrix, seeds = 0:5, truth=irisClass, 
         mu = msEst$mu, sigma = msEst$sigma, z = msEst$z)
do.call("randProj", c(list(data = irisMatrix, seeds = 0:5, truth=irisClass),
                           msEst))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{sigma2decomp}
\alias{sigma2decomp}
\title{
  Convert mixture component covariances to decomposition form.
}
\description{
  Converts a set of covariance matrices from representation as a 3-D array 
  to a parameterization by eigenvalue decomposition.
}
\usage{
sigma2decomp(sigma, G, tol, \dots)
}
\arguments{
  \item{sigma}{
    Either a 3-D array whose [,,k]th component is the covariance matrix for the
    kth component in an MVN mixture model, or a single covariance
    matrix in the case that all components have the same covariance.
  }
  \item{G}{
    The number of components in the mixture. When 
    \code{sigma} is a 3-D array, the number of components
    can be inferred from its dimensions.
  }
  \item{tol}{
    Tolerance for determining whether or not the covariances have equal volume,
    shape, and or orientation. The default is the square root of the relative
    machine precision, \code{sqrt(.Machine\$double.eps)}, 
    which is about \code{1.e-8}.
  }
  \item{\dots}{
    Catch unused arguments from a \code{do.call} call.
  }
}
\value{
  The covariance matrices for the mixture components in decomposition form,
  including the following components: 
  \item{d}{
    The dimension of the data. 
  }
  \item{G}{
    The number of components in the mixture model. 
  }
  \item{scale}{
    Either a \emph{G}-vector giving the scale of the covariance (the
    \emph{d}th root of its determinant) for each component in the
    mixture model, or a single numeric value if the scale is the same
    for each component.
  }
  \item{shape}{
    Either a \emph{G} by \emph{d} matrix in which the \emph{k}th
    column is the shape of the covariance matrix (normalized to have
    determinant 1) for the \emph{k}th component, or a \emph{d}-vector
    giving a common shape for all components. 
  }
  \item{orientation}{
    Either a \emph{d} by \emph{d} by \emph{G} array whose
    \code{[,,k]}th entry is the orthonomal matrix of eigenvectors of
    the covariance matrix of the \emph{k}th component, or a \emph{d}
    by \emph{d} orthonormal matrix if the mixture components have a
    common orientation. The \code{orientation} component of
    \code{decomp} can be omitted in spherical and diagonal models, for
    which the principal components are parallel to the coordinate axes
    so that the orientation matrix is the identity.  
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation, and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{decomp2sigma}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

meEst <- meEEE(irisMatrix, unmap(irisClass)) 
names(meEst)
meEst$sigma

sigma2decomp(meEst$sigma)
do.call("sigma2decomp", meEst)  ## alternative call
}
\keyword{}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{sim}
\alias{sim}
\title{
  Simulate from Parameterized MVN Mixture Models
}
\description{
  Simulate data from parameterized MVN mixture models.
}
\usage{
sim(modelName, mu, \dots, seed = 0)
}
\arguments{
  \item{modelName}{
    A character string indicating the model. Possible models: \cr\cr
    "E": equal variance  (one-dimensional) \cr
    "V": variable variance (one-dimensional) \cr\cr
    "EII": spherical, equal volume \cr
    "VII": spherical, unequal volume \cr
    "EEI": diagonal, equal volume, equal shape \cr
    "VEI": diagonal, varying volume, equal shape \cr
    "EVI": diagonal, equal volume, varying shape \cr
    "VVI": diagonal, varying volume, varying shape \cr
    "EEE": ellipsoidal, equal volume, shape, and orientation \cr
    "EEV": ellipsoidal, equal volume and equal shape\cr
    "VEV": ellipsoidal, equal shape \cr
    "VVV": ellipsoidal, varying volume, shape, and orientation 
  }
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the means of the components.
  }
  \item{\dots}{
    Arguments for model-specific functions. Specifically:
    \itemize{
      \item An argument describing the variance (depends on the model):
      \describe{
	\item{sigmasq}{
	  for the one-dimensional models ("E", "V") and spherical models
	  ("EII", "VII"). This is either a vector whose \emph{k}th
	  component is the variance for the \emph{k}th component in the
	  mixture model ("V" and "VII"), or a scalar giving the common
	  variance for all components in the mixture model ("E" and
	  "EII"). 
	}
	\item{decomp}{
	  for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
	  ellipsoidal models ("EEV", "VEV"). This is a list described in
	  \code{\link{cdens}}. 
	}
	\item{Sigma}{
	  for the equal variance model "EEE". A \emph{d} by \emph{d}
	  matrix giving the common covariance for all
	  components of the mixture model.
	}
	\item{sigma}{
	  for the unconstrained variance model "VVV". A
	  \emph{d} by \emph{d} by \emph{G} matrix array whose
	  \code{[,,k]}th entry is the covariance matrix for
	  the \emph{k}th component of the mixture model.
	}
	The form of the variance specification is the same as
	for the output for the \code{em}, \code{me}, or
	\code{mstep} methods for the specified mixture model.
      }
    }
    \item{pro}{
      Component mixing proportions. If missing, equal proportions are
      assumed. 
    }
    \item{n}{
      An integer specifying the number of data points to be simulated.
    }
  }
  \item{seed}{
    A integer between 0 and 1000, inclusive, for specifying a seed for 
    random class assignment. The default value is 0.
  }
}
\value{
  A data set consisting of n points simulated from the specified 
  MVN mixture model.
}
\details{
  This function can be used with an indirect or list call using
  \code{do.call}, allowing the output of e.g. \code{mstep}, \code{em},
  \code{me}, or \code{EMclust} to be passed directly without the need to
  specify individual parameters as arguments. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{simE}}, \ldots,
  \code{\link{simVVV}},
  \code{\link{EMclust}},
  \code{\link{mstep}},
  \code{\link{do.call}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

irisBic <- EMclust(irisMatrix)
irisSumry <- summary(irisBic,irisMatrix)
names(irisSumry)
irisSim <- sim(modelName = irisSumry$modelName, n = dim(irisMatrix)[1],
       mu = irisSumry$mu, decomp = irisSumry$decomp, pro = irisSumry$pro)
irisSim <- do.call("sim", irisSumry) ## alternative call

par(pty = "s", mfrow = c(1,2))
dimens <- c(1,2)
xlim <- range(rbind(irisMatrix,irisSim)[,dimens][,1])
ylim <- range(rbind(irisMatrix,irisSim)[,dimens][,2])

cl <- irisSumry$classification
coordProj(irisMatrix, par=irisSumry, classification=cl, dimens=dimens, 
          xlim=xlim, ylim=ylim)
cl <- attr(irisSim,"classification")
coordProj(irisSim, par=irisSumry, classification=cl, dimens=dimens, 
          xlim=xlim, ylim=ylim)

irisSumry3 <- summary(irisBic,irisMatrix, G=3)
irisSim3 <- do.call("sim", c(list(n = 500, seed = 1), irisSumry3))
clPairs(irisSim3, cl = attr(irisSim3,"classification"))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{simE}
\alias{simE}
\alias{simV}
\alias{simEII}
\alias{simVII}
\alias{simEEI}
\alias{simVEI}
\alias{simEVI}
\alias{simVVI}
\alias{simEEE}
\alias{simEEV}
\alias{simVEV}
\alias{simVVV}
\title{
  Simulate from a Parameterized MVN Mixture Model
}
\description{
  Simulate data from a parameterized MVN mixture model.
}
\usage{
simE(mu, sigmasq, pro, \dots, seed = 0)
simV(mu, sigmasq, pro, \dots, seed = 0)
simEII(mu, sigmasq, pro, \dots, seed = 0)
simVII(mu, sigmasq, pro, \dots, seed = 0)
simEEI(mu, decomp, pro, \dots, seed = 0)
simVEI(mu, decomp, pro, \dots, seed = 0)
simEVI(mu, decomp, pro, \dots, seed = 0)
simVVI(mu, decomp, pro, \dots, seed = 0)
simEEE(mu, pro, \dots, seed = 0)
simEEV(mu, decomp, pro, \dots, seed = 0)
simVEV(mu, decomp, pro, \dots, seed = 0)
simVVV mu, pro, \dots, seed = 0)
}
\arguments{
  \item{mu}{
    The mean for each component. If there is more than one component,
    \code{mu} is a matrix whose columns are the
    means of the components.
  }
  \item{sigmasq}{ 
    for the one-dimensional models ("E", "V") and spherical models
    ("EII", "VII"). This is either a vector whose \emph{k}th component
    is the variance for the \emph{k}th component in the mixture model
    ("V" and "VII"), or a scalar giving the common variance for all
    components in the mixture model ("E" and "EII"). 
  }
  \item{decomp}{ 
    for the diagonal models ("EEI", "VEI", "EVI", "VVI") and some
    ellipsoidal models ("EEV", "VEV"). This is a list described in
    \code{\link{cdens}}.
  }
  \item{pro}{
    Component mixing proportions. If missing, equal proportions are assumed.
  }
  \item{\dots}{
    \item Other terms describing variance:
    \describe{
      \item{Sigma}{ 
	for the equal variance model "EEE".
	A \emph{d} by \emph{d} matrix giving the common covariance for
	all components of the mixture model.
      }
      \item{sigma}{ 
	for the unconstrained variance model "VVV".
	A \emph{d} by \emph{d} by \emph{G} matrix array whose 
	\code{[,,k]}th entry is the covariance matrix for the
	\emph{k}th component of the mixture model.
      }
      The form of the variance specification is the same as for the output
      for the \code{em}, \code{me}, or \code{mstep} methods for the
      specified mixture model.  
    }
    \item{n}{
      An integer specifying the number of data points to be simulated.
    }
  }
  \item{seed}{
    A integer between 0 and 1000, inclusive, for specifying a seed for 
    random class assignment. The default value is 0.
  }
}
\value{
  A data set consisting of \code{n} points simulated from 
  the specified MVN mixture model.
}
\details{
  This function can be used with an indirect or list call using
  \code{do.call}, allowing the output of e.g. \code{mstep}, \code{em}
  \code{me}, or \code{EMclust}, to be passed directly without the need
  to specify individual parameters as arguments. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{sim}},
  \code{\link{EMclust}},
  \code{\link{mstepE}},
  \code{\link{do.call}}
}
\examples{
d <- 2
G <- 2
scale <- 1
shape <- c(1, 9)

O1 <- diag(2)
O2 <- diag(2)[,c(2,1)]
O <- array(cbind(O1,O2), c(2, 2, 2))
O

decomp <- list(d= d, G = G, scale = scale, shape = shape, orientation = O)
mu <- matrix(0, d, G) ## center at the origin
simdat <- simEEV(n=200, mu=mu, decomp=decomp, pro = c(1,1))

cl <- attr(simdat, "classification")
sigma <- array(apply(O, 3, function(x,y) crossprod(x*y), 
                 y = sqrt(scale*shape)), c(2,2,2))
paramList <- list(mu = mu, sigma = sigma)
coordProj( simdat, paramList = paramList, classification = cl)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{spinProj}
\alias{spinProj}
\title{
  Planar spin for random projections of data in more than two dimensions
  modelled by an MVN mixture.
}
\description{
  Plots random 2-D projections with suggessive rotations through a specified 
  angles given data in more than two dimensions 
  and parameters of an MVN mixture model.
}
\usage{
spinProj(data, \dots, angles, seed = 0, reflection = FALSE, 
         type = c("classification", "uncertainty", "errors"), 
         ask = TRUE, quantiles = c(0.75,0.95), symbols, scale = FALSE,
         identify = FALSE, CEX = 1, PCH = ".", xlim, ylim)
}
\arguments{
  \item{data}{
    A numeric matrix or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{\dots}{
    Any number of the following:
    \describe{
      \item{classification}{
	A numeric or character vector representing a classification of
	observations (rows) of \code{data}.
      }
      \item{uncertainty}{
	A numeric vector of values in \emph{(0,1)} giving the
	uncertainty of each data point.
      }
      \item{z}{
	A matrix in which the \emph{[i,k]}the entry gives the probability of
	observation \emph{i} belonging to the \emph{k}th class. Used to
	compute \code{classification} and \code{uncertainty} if those
	arguments aren't available. 
      }
      \item{truth}{
	A numeric or character vector giving a known classification of each
	data point. If \code{classification} or \code{z} is also present,
	this is used for displaying classification errors. 
      }
      \item{mu}{
	A matrix whose columns are the means of each group.
      }
      \item{sigma}{
	A three dimensional array in which \code{sigma[,,k]} gives the
	covariance for the \emph{k}th group. 
      }
      \item{decomp}{
	A list with \code{scale}, \code{shape} and \code{orientation}
	components giving an alternative form for the covariance structure
	of the mixture model. 
      }
    }
  }
  \item{angles}{
    The angles (in radians) through which successive projections should
    be rotated or reflected.
  }
  \item{seed}{
    A integer between 0 and 1000, inclusive, for specifying a seed for 
    generating the initial random projection. The default value is 0. 
    The seed/projection correspondence is the same as in 
    \code{randProj}.
  }
  \item{reflection}{
    A logical variable telling whether or not the data should be reflected or
    rotated through the given angles. The default is rotation.
  }
  \item{type}{
    Any subset of \code{c("classification","uncertainty","errors")}. The
    function will produce the corresponding plot if it has been supplied
    sufficient information to do so. If more than one plot is possible
    then users will be asked to choose from a menu if \code{ask=TRUE}. 
  }
  \item{ask}{
    A logical variable indicating whether or not a menu should be produced
    when more than one plot is possible. The default is \code{ask=TRUE}.
  }
  \item{quantiles}{
    A vector of length 2 giving quantiles used in plotting
    uncertainty. The smallest symbols correspond to the smallest
    quantile (lowest uncertainty), medium-sized (open) symbols to points
    falling between the given quantiles, and large (filled) symbols to
    those in the largest quantile (highest uncertainty). The default is
    \emph{(0.75,0.95)}. 
  }
  \item{symbols}{
    Either an integer or character vector assigning a plotting symbol to
    each unique class \code{classification}. Elements in \code{symbols}
    correspond to classes in \code{classification} in order of
    appearance in \code{classification} (the order used by the S-PLUS
    function \code{unique}). Default: If \emph{G} is the number of
    groups in the classification, the first \emph{G} symbols in
    \code{.Mclust\$symbols}, otherwise if \emph{G} is less than 27 then
    the first \emph{G} capital letters in the Roman alphabet. 
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and
    thus preserve the shape of the distribution.
    Default: \code{scale=FALSE} 
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{CEX}{
    An argument specifying the size of the plotting symbols. 
    The default value is 1.
  }
  \item{PCH}{
    An argument specifying the symbol to be used when a classificatiion
    has not been specified for the data. The default value is a small dot ".".
  }
  \item{xlim, ylim}{
    Arguments specifying bounds for the ordinate, abscissa of the plot.
    This may be useful for when comparing plots.
  }
}
\value{
  Rotations or reflections of a random projection of the data,
  possibly showing location of the mixture components, classification,
  uncertainty and/or classfication errors.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{coordProj}},
  \code{\link{randProj}},
  \code{\link{mclust2Dplot}},
  \code{\link{mclustOptions}},
  \code{\link{do.call}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]

msEst <- mstepVVV(irisMatrix, unmap(irisClass))

par(pty = "s", mfrow = c(2,2))
spinProj(irisMatrix, seed = 1, truth=irisClass,
         mu = msEst$mu, sigma = msEst$sigma, z = msEst$z)
do.call("spinProj", c(list(data = irisMatrix, seeds = 2, truth=irisClass),
                           msEst))
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{summary.EMclust}
\alias{summary.EMclust}
\alias{print.summary.EMclust}
\title{
  Summary function for EMclust
}
\description{
  Optimal model characteristics and classification
  for \code{EMclust} results.
}
\usage{
summary.EMclust(object, data, G, modelNames, \dots)
}
\arguments{
  \item{object}{
    An \code{"EMclust"} object, 
    which is the result of applying \code{EMclust} 
    to \code{data}.
  }
  \item{data}{
    The matrix or vector of observations used to generate `object'.
  }
  \item{G}{
    A vector of integers giving the numbers of mixture components (clusters)
    over which the
    summary is to take place (\code{as.character(G)}
    must be a subset of the column names of \code{object}).
    The default is to summarize over all of the numbers of mixture components 
    used in the original analysis.
  }
  \item{modelNames}{
    A vector of character strings denoting the models over which the
    summary is to take place (must be a subset of the row names of `object').
    The default is to summarize over all models used in the original
    analysis.
  }
  \item{\dots}{
    Not used. For generic/method consistency.
  }
}
\value{
  A list giving the optimal (according to BIC) parameters, 
  conditional probabilities \code{z}, and loglikelihood,
  together with the associated classification and its uncertainty.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{EMclust}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

irisBic <- EMclust(irisMatrix)
summary(irisBic, irisMatrix)
summary(irisBic, irisMatrix, G = 1:6, modelName = c("VII", "VVI", "VVV"))
}
\keyword{cluster}
% docclass is function

\eof
\name{summary.EMclustN}
\alias{summary.EMclustN}
\alias{print.summary.EMclustN}
\title{
  summary function for EMclustN
}
\description{
  Optimal model characteristics and classification
  for \code{EMclustN} results.
}
\usage{
summary.EMclustN(object, data, G, modelNames, \dots)
}
\arguments{
  \item{object}{
    An \code{"EMclustN"} object, whch is the result of a pplying
    \code{EMclustN} to \code{data} with an initail noise estimate.
  }
  \item{data}{
    The matrix or vector of observations used to generate `object'.
  }
  \item{G}{
    A vector of integers giving the numbers of mixture components (clusters)
    over which the
    summary is to take place (\code{as.character(G)}
    must be a subset of the column names of `object').
    The default is to summarize over all of the numbers of mixture components 
    used in the original analysis.
  }
  \item{modelNames}{
    A vector of character strings denoting the models over which the
    summary is to take place (must be a subset of the row names of `object').
    The default is to summarize over all models used in the original
    analysis.
  }
  \item{\dots}{
    Not used. For generic/method consistency.
  }
}
\value{
  A list giving the optimal (according to BIC) parameters, 
  conditional probabilities \code{z}, and loglikelihood,
  together with the associated classification and its uncertainty.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{EMclustN}}
}

\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

b <- apply( irisMatrix, 2, range)
n <- 450
set.seed(0)
poissonNoise <- apply(b, 2, function(x, n=n) 
                      runif(n, min = x[1]-0.1, max = x[2]+.1), n = n)
set.seed(0)
noiseInit <- sample(c(TRUE,FALSE),size=150+450,replace=TRUE,prob=c(3,1))
irisNoise <- rbind(irisMatrix, poissonNoise)

Bic <- EMclustN(data=irisNoise, noise = noiseInit)
summary(Bic, irisNoise)
summary(Bic, irisNoise, G = 0:6, modelName = c("VII", "VVI", "VVV"))
}
\keyword{cluster}
% docclass is function

\eof
\name{summary.Mclust}
\alias{summary.Mclust}
\title{Very brief summary of an Mclust object.}
\description{
  Function gives a brief summary of an Mclust object: the type of model
  that is picked and the number of clusters.
}
\usage{
summary.Mclust(object, \dots)
}
\arguments{
  \item{object}{The result of a call to
    function \code{Mclust}.}
  \item{\dots}{Not used.}
}
\keyword{cluster}

\eof
\name{summary.mclustDAtest}
\alias{summary.mclustDAtest}
\title{
  Classification and posterior probability from mclustDAtest.
}
\description{
  Classifications from \code{mclustDAtest} and the corresponding
  posterior probabilities. 
}
\usage{
summary.mclustDAtest(object, pro, \dots)
}
\arguments{
  \item{object}{
    The output of \code{mclustDAtest}.
  }
  \item{pro}{
    Prior probabilities for each class in the training data.
  }
  \item{\dots}{
    Not used. For generic/method consistency.
  }
}
\value{
  A list with the following two components: 
  
  \item{classfication}{
    The classification from \code{mclustDAtest} 
  }
  \item{z}{
    Matrix of posterior probabilities in which the \code{[i,j]}th entry
    is the probability of observation \emph{i} belonging to class
    \emph{j}.
  }
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}. 
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mclustDAtest}}
}
\examples{
set.seed(0)
n <- 100 ## create artificial data

x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
\dontrun{
par(pty = "s")
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)
}

odd <- seq(1, 2*n, 2)
train <- mclustDAtrain(x[odd, ], labels = xclass[odd]) ## training step
summary(train)

even <- seq(1, 2*n, 2)
test <- mclustDAtest(x[even, ], train) ## compute model densities
testSummary <- summary(test) ## classify training set

names(testSummary)
testSummary$class
testSummary$z
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{summary.mclustDAtrain}
\alias{summary.mclustDAtrain}
\title{
  Models and classifications from mclustDAtrain
}
\description{
  The models selected in \code{mclustDAtrain}
  and the corresponding classfications.
}
\usage{
summary.mclustDAtrain(object, \dots)
}
\arguments{
  \item{object}{
    The output of \code{mclustDAtrain}.
  }
  \item{\dots}{
    Not used. For generic/method consistency.
  }
}
\value{
  A list identifying the model selected by 
  \code{mclustDAtrain} for each
  class of training data and the corresponding classification.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington.
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mclustDAtrain}}
}
\examples{
set.seed(0)
n <- 100 ## create artificial data

x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))
\dontrun{
par(pty = "s")
mclust2Dplot(x, classification = xclass, type="classification", ask=FALSE)}

odd <- seq(1, 2*n, 2)
train <- mclustDAtrain(x[odd, ], labels = xclass[odd]) ## training step
summary(train)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{surfacePlot}
\alias{surfacePlot}
\title{
  Density or uncertainty surface for two dimensional mixtures.
}
\description{
  Plots a density or uncertainty surface 
  given data in more than two dimensions
  and parameters of an MVN mixture model for the data.
}
\usage{
surfacePlot(data, mu, pro, \dots, type = c("contour", "image", "persp"), 
        what = c("density", "uncertainty", "skip"), 
        transformation = c("none", "log", "sqrt"),          
        grid = 50, nlevels = 20, scale = FALSE, identify = FALSE,
        verbose = FALSE, xlim, ylim, swapAxes = FALSE)
}
\arguments{
  \item{data}{
    A numeric vector, matrix, or data frame of observations.
    Categorical variables are not allowed.
    If a matrix or data frame, rows correspond to observations and
    columns correspond to variables.
  }
  \item{mu}{
    A matrix whose columns are the means of each group. 
  }
  \item{pro}{
    A list with \code{scale}, \code{shape} and \code{orientation}
    components giving an alternative form for the covariance structure
    of the mixture model.
  }
  \item{\dots}{
    An argument specifying the covariance structure of the model. If
    used an indirect function call via \code{do.call} (see example
    below), it is usually not necessary to know the precise form for
    this argument. This argument usually take one of the following
    forms:
    \describe{
      \item{\code{sigma}}{ A three dimensional array in which
	\code{sigma[,,k]} gives the covariance for the \emph{k}th group.}
      \item{\code{decomp}}{ A list with \code{scale}, \code{shape} and
	\code{orientation} components giving an alternative form for the
	covariance structure of the mixture model.}
    }
  }
  \item{type}{
    Any subset of \code{c("contour","image","persp")} indicating the
    plot type. For more than one selection, users will be asked to
    choose from a menu.
  }
  \item{what}{
    Any subset of \code{c("density","uncertainty","skip")} indicating
    what to plot. For more than one selection, users will be asked to
    choose from a menu. The \code{"skip"} produces and empty plot, which
    may be useful if multiple plots are displayed simultaneously. 
  }
  \item{transformation}{
    Any subset of \code{c("none","log","sqrt")} indicating  a
    transformation to be applied to the surface values before
    plotting. For more than one selection, users will be asked to choose
    from a menu. 
  }
  \item{grid}{
    The number of grid points (evenly spaced on each axis). 
    The mixture density and uncertainty is computed at 
    \code{grid x grid} points to produce the surface plot.
    Default: \code{50}.
  }
  \item{nlevels}{
    The number of levels to use for a contour plot.
    Default: \code{20}.
  }
  \item{scale}{
    A logical variable indicating whether or not the two chosen
    dimensions should be plotted on the same scale, and
    thus preserve the shape of the distribution.
    Default: \code{scale=F} 
  }
  \item{identify}{
    A logical variable indicating whether or not to add a title to the plot
    identifying the dimensions used.
  }
  \item{verbose}{
    A logical variable telling whether or not to print an indication that
    the function is in the process of computing values at the grid points,
    which typically takes some time to complete.
  }
  \item{xlim, ylim }{
    An argument specifying bounds for the ordinate, abscissa of the plot.
    This may be useful for when comparing plots.
  }
  \item{swapAxes}{
    A logical variable indicating whether or not the axes should be swapped
    for the plot.
  }
}
\value{
  An invisible list with components x, y, and z in which x and y are the
  values used to define the grid and z is the transformed density or
  uncertainty at the grid points.
}
\section{Side Effects}{
  One or more plots showing location of the mixture components, classification,
  uncertainty, and/or classification errors.
}
\section{Details}{
  For an image plot, a color scheme may need to be selected on the display
  device in order to view the plot.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}.
  
  
  C. Fraley and A. E. Raftery (2002).
  MCLUST:Software for model-based clustering, density estimation and
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{mclust2Dplot}},
  \code{\link{do.call}}
}
\examples{
n <- 250 ## create artificial data
set.seed(0)
x <- rbind(matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9)),
           matrix(rnorm(n*2), n, 2) \%*\% diag(c(1,9))[,2:1])
xclass <- c(rep(1,n),rep(2,n))

xEMclust <- summary(EMclust(x),x)
surfacePlot(x, mu = xEMclust$mu, sigma = xEMclust$sigma, pro=xEMclust$pro,
            type = "contour", what = "density", transformation = "none")

\dontrun{do.call("surfacePlot", c(list(data = x), xEMclust))}
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
\name{uncerPlot}
\alias{uncerPlot}
\title{
  Uncertainty Plot for Model-Based Clustering
}
\description{
  Plots the uncertainty in converting a conditional probablility from EM
  to a classification in model-based clustering.
}
\usage{
uncerPlot(z, truth, \dots)
}
\arguments{
  \item{z}{
    A matrix whose \emph{[i,k]}th entry is the
    conditional probability of the ith observation belonging to
    the \emph{k}th component of the mixture.  
  }
  \item{truth}{
    A numeric or character vector giving the true classification of the data. 
  }
  \item{\dots }{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  A plot of the uncertainty profile of the data,
  with uncertainties in increasing order of magnitude.
  If \code{truth} is supplied and the number of
  classes is the same as the number of columns of 
  \code{z}, the uncertainty
  of the misclassified data is marked by vertical lines on the plot.
}
\details{
  When \code{truth} is provided and the number of classes is compatible
  with \code{z}, the function \code{compareClass} is used to to find best
  correspondence between classes in \code{truth} and \code{z}.
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}.
  See \url{http://www.stat.washington.edu/mclust}.
    
  C. Fraley and A. E. Raftery (2002b).
  MCLUST: Software for model-based clustering, density estimation and 
  discriminant analysis. 
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{EMclust}},
  \code{\link{em}},
  \code{\link{me}},
  \code{\link{mapClass}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])

irisBic <- EMclust(irisMatrix)
irisSumry3 <-  summary(irisBic, irisMatrix, G = 3)

uncerPlot(z = irisSumry3$z)
 
uncerPlot(z = irisSumry3$z, truth = rep(1:3, rep(50,3)))

do.call("uncerPlot", c(irisSumry3, list(truth = rep(1:3, rep(50,3)))))
}
\keyword{cluster}
% docclass is function

\eof
\name{unmap}
\alias{unmap}
\title{
  Indicator Variables given Classification
}
\description{
  Converts a classification into a matrix of indicator variables.
}
\usage{
  unmap(classification, noise, \dots)
}
\arguments{
  \item{classification}{
    A numeric or character vector. Typically the distinct entries of this
    vector would represent a classification of observations in a data set.
  }
  \item{noise}{
    A single numeric or character value used to indicate observations 
    corresponding to noise.
  }
  \item{\dots}{
    Provided to allow lists with elements other than the arguments can
    be passed in indirect or list calls with \code{do.call}.
  }
}
\value{
  An \emph{n} by \emph{m} matrix of \emph{(0,1)} indicator variables,
  where \emph{n} is the length of \code{classification} and \emph{m} is
  the number of unique values or symbols in
  \code{classification}. Columns are labeled by the unique values in
  \code{classification}, and the \code{[i,j]}th entry is \emph{1} if
  \code{classification[i]} is the \emph{j}th unique value or symbol in
  order of appearance in the \code{classification}. If a \code{noise}
  value of symbol is designated, the corresponding indicator variables
  are located in the last column of the matrix. 
}
\section{References}{
  C. Fraley and A. E. Raftery (2002a).
  Model-based clustering, discriminant analysis, and density estimation.
  \emph{Journal of the American Statistical Association 97:611-631}. 
  See \url{http://www.stat.washington.edu/mclust}. 
  
  
  C. Fraley and A. E. Raftery (2002b).
  MCLUST:Software for model-based clustering, density estimation and 
  discriminant analysis.
  Technical Report, Department of Statistics, University of Washington. 
  See \url{http://www.stat.washington.edu/mclust}.
}
\seealso{
  \code{\link{map}},
  \code{\link{estep}},
  \code{\link{me}}
}
\examples{
data(iris)
irisMatrix <- as.matrix(iris[,1:4])
irisClass <- iris[,5]
  
z <- unmap(irisClass)
z
  
emEst <- me(modelName = "VVV", data = irisMatrix, z = z)
emEst$z
  
map(emEst$z)
}
\keyword{cluster}
% docclass is function
% Converted by Sd2Rd version 1.21.

\eof
