\name{AIC.pgam}
\alias{AIC.pgam}
\title{AIC extraction}
\description{Method for approximate Akaike Information Criterion extraction.
}
\usage{
\method{AIC}{pgam}(object, k = 2, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{k}{default is 2 for AIC. If \eqn{k=\log\left(n\right)} then an approximation for BIC is obtained. Important to note that these are merely approximations.}
  \item{\dots}{further arguments passed to method}
}
\details{An approximate measure of parsimony of the Poisson-Gama Additive Models can be achieved by the expression
\deqn{AIC=\left(D\left(y;\hat\mu\right)+2gle\right)/\left(n-\tau\right)}
where \eqn{gle} is the number of degrees of freedom of the fitted model and \eqn{\tau} is the index of the first non-zero observation.
}
\value{The approximate AIC value of the fitted model.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica

Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{deviance.pgam}}, \code{\link{logLik.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

AIC(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{aihrio}
\alias{aihrio}
\docType{data}
\title{Sample dataset}
\description{This is a dataset for Poisson-Gamma Additive Models functions testing.
}
\usage{data(aihrio)}
\format{
A data frame with 365 observations on the following 33 variables.
  \describe{
    \item{DATE}{a factor with levels}
    \item{TIME}{a numeric vector}
    \item{ITRESP65}{a numeric vector}
    \item{ITCIRC65}{a numeric vector}
    \item{ITDPOC65}{a numeric vector}
    \item{ITPNM65}{a numeric vector}
    \item{ITAVC65}{a numeric vector}
    \item{ITIAM65}{a numeric vector}
    \item{ITDIC65}{a numeric vector}
    \item{ITTCA65}{a numeric vector}
    \item{ITRESP5}{a numeric vector}
    \item{ITPNEU5}{a numeric vector}
    \item{ITDPC5}{a numeric vector}
    \item{WEEK}{a numeric vector}
    \item{MON}{a numeric vector}
    \item{TUE}{a numeric vector}
    \item{WED}{a numeric vector}
    \item{THU}{a numeric vector}
    \item{FRI}{a numeric vector}
    \item{SAT}{a numeric vector}
    \item{SUN}{a numeric vector}
    \item{HOLIDAYS}{a numeric vector}
    \item{MONTH}{a numeric vector}
    \item{warm.season}{a numeric vector}
    \item{tmpmed}{a numeric vector}
    \item{tmpmin}{a numeric vector}
    \item{tmpmax}{a numeric vector}
    \item{wet}{a numeric vector}
    \item{rain}{a numeric vector}
    \item{rainy}{a numeric vector}
    \item{PM}{a numeric vector}
    \item{SO2}{a numeric vector}
    \item{CO}{a numeric vector}
  }
}
\details{This is a reduced dataset of those used to estimate possible effects of air pollution on hospital admissions outcomes in Universidade do Estado do Rio de Janeiro, Rio de Janeiro, Brasil.
}
\source{
Secretaria de Meio Ambiente da Cidade do Rio de Janeiro, Ministrio de Aeronutica and Datasus.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\keyword{datasets}

\eof
\name{backfitting}
\alias{backfitting}
\title{Backfitting algorithm}
\description{Fit the nonparametric part of the model via backfitting algorithm.
}
\usage{
backfitting(y, x, df, fx, smoother = "spline", w = rep(1, length(y)), eps = 0.001, maxit = 100, info = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{y}{dependent variable for fitting. In semiparametric models, this is the partial residuals of parametric fit}
  \item{x}{matrix of covariates}
  \item{df}{equivalent degrees of freedom}
  \item{fx}{if \code{FALSE} the smoothing parameter is chosen by cross-validation}
  \item{smoother}{string with the name of the smoother to be used}
  \item{w}{vector with the diagonal elements of the weight matrix. Default is a vector of \eqn{1} with the same length of \eqn{y}}
  \item{eps}{convergence control criterion}
  \item{maxit}{convergence control iterations}
  \item{info}{if \code{FALSE} only fitted values are returned. It it is faster during iterations}
}
\details{Backfitting algorithm estimates the approximating regression surface, working around the "curse of dimentionality".

More details soon enough.
}
\value{Fitted smooth curves and partial residuals.
}
\references{
Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica

Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{predict.pgam}}, \code{\link{pgam.smooth}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{coef.pgam}
\alias{coef.pgam}
\title{Coefficients extraction}
\description{Method for parametric coefficients extraction.
}
\usage{
\method{coef}{pgam}(object, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{\dots}{further arguments passed to method}
}
\details{
This function only retrieves the estimated coefficients from the model object returned by \code{pgam}.
}
\value{Vector of coefficients estimates of the model fitted.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{predict.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

coef(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{deviance.pgam}
\alias{deviance.pgam}
\title{Deviance extraction}
\description{Method for total deviance value extraction.}
\usage{
\method{deviance}{pgam}(object, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{\dots}{further arguments passed to method}
}
\details{
See \code{\link{predict.pgam}} for further information on deviance extration in Poisson-Gamma models.
}
\value{The sum of deviance components.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{pgam.likelihood}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

deviance(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{elapsedtime}
\alias{elapsedtime}
\title{Utility function}
\description{Gadget to compute the elapsed time of a process
}
\usage{
elapsedtime(st, et)
}
\arguments{
  \item{st}{start time}
  \item{et}{end time}
}
\details{Start and end times can be gotten with \code{\link{proc.time}}.
}
\value{String with the elapsed time in \code{hh:mm:ss} format.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{envelope}
\alias{envelope}
\title{Normal plot with simulated envelope of the residuals.
}
\description{A normal plot with simulated envelope of the residual is produced.
}
\usage{
envelope(object, type = "deviance", size = 0.95, rep = 19, optim.method=NULL, 
epsilon = 0.001, maxit = 100, plot = TRUE, verbose = FALSE, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{type}{type of residuals to be extracted. Default is \code{deviance}. Options are described in \code{\link{residuals.pgam}}}
  \item{size}{value giving the size of the envelope. Default is \code{.95} which is equivalent to a 95\% band}
  \item{rep}{number of replications for envelope construction. Default is \code{19}}, that is the smallest 95\% band that can be build
  \item{optim.method}{optimization method to be passed to \code{\link{pgam}} and therefore to \code{\link{optim}}}
  \item{epsilon}{convergence control to be passed to \code{\link{pgam}}}
  \item{maxit}{convergence control to be passed to \code{\link{pgam}}}
  \item{plot}{if \code{TRUE} a plot of the envelope is produced}
  \item{verbose}{if \code{TRUE} a sort of information is printed during the running time}
  \item{\dots}{further arguments to \code{\link{plot}} function}
}
\details{Sometimes the usual Q-Q plot shows an unsatisfactory pattern of the residuals of a model fitted and we are led to think that the model is badly specificated. The normal plot with simulated envelope indicates that under the distribution of the response variable the model is OK if only a few points fall off the envelope.

If \code{object} is of class \code{pgam} the envelope is estimated and optionally plotted, else if is of class \code{envelope} then it is only plotted.
}
\value{An object of class \code{envelope} holding the information needed to plot the envelope.
}
\references{
Atkinson, A. C. (1985) Plots, transformations and regression : an introduction to graphical methods of diagnostic regression analysis. Oxford Science Publications, Oxford.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{predict.pgam}}, \code{\link{residuals.pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{f}
\alias{f}
\title{Utility function}
\description{Generate the partition of design matrix regarded to the seasonal factor in its argument. Used in the model formula.
}
\usage{
f(factorvar)
}
\arguments{
  \item{factorvar}{variable with the seasonal levels}
}
\value{List containing data matrix of dummy variables, level names and seasonal periods.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is intended to be called from within a model formula.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{fitted.pgam}
\alias{fitted.pgam}
\title{Fitted values extraction}
\description{
Method for fitted values extraction.
}
\usage{
\method{fitted}{pgam}(object, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{\dots}{further arguments passed to method}
}
\details{
Actually, the fitted values are worked out by the function \code{predict.pgam}. Thus, this method is supposed to turn fitted values extraction easier. See \code{\link{predict.pgam}} for details on one-step ahead prediction.
}
\value{Vector of predicted values of the model fitted.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{predict.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

f <- fitted(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{fnz}
\alias{fnz}
\title{Utility function}
\description{Return the index of first non-zero observation of a variable.
}
\usage{
fnz(y)
}
\arguments{
  \item{y}{variable vector}
}
\value{The index of first non-zero value.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{framebuilder}
\alias{framebuilder}
\title{Utility function}
\description{Generate a data frame given a formula and a dataset.}
\usage{
framebuilder(formula, dataset)
}
\arguments{
  \item{formula}{model formula}
  \item{dataset}{model dataset}
}
\details{Actually, this function is a wrapper for \code{\link{model.frame}}.
}
\value{A data frame restricted to the model.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{g}
\alias{g}
\title{Utility function}
\description{Collect information to smooth the term in its argument. Used in the model formula.
}
\usage{
g(var, df, fx = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{var}{variable to be smoothed}
  \item{df}{equivalent degrees of freedom to be passed to the smoother}
  \item{fx}{if \code{FALSE}, \code{df} is ignored and smoothing parameter is chosen by cross-validation}
}
\details{This function only sets things up for model fitting. The smooth terms are actually fitted by \code{\link{pgam.smooth}}.
}
\value{List containing the same elements of its argument.
}
\references{
Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London

Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is intended to be called from within a model formula.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{intensity}
\alias{intensity}
\title{Utility function}
\description{Apply spectral decomposition to its argument.
}
\usage{
intensity(w, x)
}
\arguments{
  \item{w}{frequency values}
  \item{x}{time series}
}
\details{The function apllies the spectral decomposition to the time series according to the following expression
\deqn{\left(\left(\sum\left(x*cos\left(w*t\right)\right)\right)^{2}+\left(\sum\left(x*cos\left(w*t\right)\right)\right)^{2}\right)/n}
}
\value{Decomposed series.
}
\references{
Box, G., Jenkins, G., Reinsel, G. (1994) Time Series Analysis : Forecasting and Control. 3rd edition, Prentice Hall, New Jersey.

Diggle, P. J. (1989) Time Series : A Biostatistical Introduction. Oxford University Press, Oxford.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{periodogram}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{link}
\alias{link}
\title{Utility function}
\description{Apply the link function or its inverse to the argument.
}
\usage{
link(x, link = "log", inv = FALSE)
}
\arguments{
  \item{x}{vector containing the predictor}
  \item{link}{string with the name of the link function}
  \item{inv}{if \code{TRUE} its inverse is applied}
}
\details{This function is intended to port other link functions than \eqn{\log{\left(\right)}} to Poisson-Gamma Additive Models. For now, the only allowed value is \code{"log"}.
}
\value{Evaluated link function at \code{x} values.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

McCullagh, P., Nelder, J. A. (1989). Generalized Linear Models. Chapman and Hall, 2nd edition, London
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{logLik.pgam}
\alias{logLik.pgam}
\title{Loglik extraction}
\description{Method for loglik value extraction.
}
\usage{
\method{logLik}{pgam}(object, ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{\dots}{further arguments passed to method}
}
\details{
See \code{\link{pgam.likelihood}} for more information on log-likelihood evaluation in Poisson-Gamma models.
}
\value{The maximum value achieved by the likelihood optimization process.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{pgam.likelihood}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

logLik(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{lpnorm}
\alias{lpnorm}
\title{Utility function}
\description{Compute the Lp-norm of two sequencies.
}
\usage{
lpnorm(seq1, seq2 = 0, p = 0)
}
\arguments{
  \item{seq1}{first sequency}
  \item{seq2}{second sequency}
  \item{p}{L-space of the norm. \code{0} is infinity norm or max norm, \code{1} is the absolute value norm, \code{2} is L2 norm and so on}
}
\value{The computed norm value.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{periodogram}
\alias{periodogram}
\title{Raw Periodogram}
\description{A raw periodogram is returned and optionally plotted.}
\usage{
periodogram(y, rows = trunc(length(na.omit(y))/2-1), plot = TRUE, ...)
}
\arguments{
  \item{y}{time series}
  \item{rows}{number of rows to be returned. Default and largest is \eqn{n/2-1}, where \eqn{n} is the number of valid observations of the time series \eqn{y}}
  \item{plot}{if \code{TRUE} a raw periodogram is plotted}
  \item{\dots}{further arguments to \code{\link{plot}} function}
}
\details{
The raw periodogram is an estimator of the spectrum of a time series, it still is a good indicator of unresolved seasonality patterns in residuals of the fitted model. See \code{\link{intensity}} for frequencies extraction.

This function plots a fancy periodogram where the intensities of the angular frequencies are plotted resembling tiny lollipops.
}
\value{Periodogram ordered by intensity.
}
\references{
Box, G., Jenkins, G., Reinsel, G. (1994) Time Series Analysis : Forecasting and Control. 3rd edition, Prentice Hall, New Jersey.

Diggle, P. J. (1989) Time Series : A Biostatistical Introduction. Oxford University Press, Oxford.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{intensity}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam}
\alias{pgam}
\title{Poisson-Gamma Additive Models}
\description{Fit Poisson-Gamma Additive Models using the roughness penalty approach
}
\usage{
pgam(formula, dataset, omega = 0.8, beta = 0.1, offset = 1, digits = getOption("digits"),
maxit = 100, eps = 1e-06, lfn.scale=1, control = list(), optim.method = "L-BFGS-B", partial.resid = "response",
smoother = "spline", bkf.eps = 0.001, bkf.maxit = 100, se.estimation = "numerical", verbose = TRUE)
}
\arguments{
  \item{formula}{a model formula. See \code{\link{pgam.parser}} for details}
  \item{dataset}{a data set in the environment search path}
  \item{omega}{initial value for the discount factor}
  \item{beta}{vector of initial values for covariates coefficients. If a sigle value is supplied it is replicated to fill in the whole vector}
  \item{offset}{default is \eqn{1}. Other value can be supplied here}
  \item{digits}{number of decimal places for printing information out}
  \item{maxit}{convergence control iterations}
  \item{eps}{convergence control criterion}
  \item{lfn.scale}{scales the likelihood function and is passed to \code{control} in \code{\link{optim}}. Value must be positive to ensure maximization}
  \item{control}{convergence control of \code{\link{optim}}. See its help for details}
  \item{optim.method}{optimization method passed to \code{\link{optim}}. Different methods can lead to different results, so the user must attempt to the trade off between speed and robustness. For example, \code{BFGS} is faster but sensitive to starting values and \code{L-BFGS-B} is more robust but slower. See its help for details.}
  \item{partial.resid}{type of partial residual to be used if semiparametric fitting. See \code{\link{residuals.pgam}} for details}
  \item{smoother}{smoother to be used in backfitting. See \code{\link{pgam.smooth}} for details}
  \item{bkf.eps}{convergence control criterion for the backfitting algorithm}
  \item{bkf.maxit}{convergence control iterations for the backfitting algorithm}
  \item{se.estimation}{if \code{numerical} numerical standard error of parameters are returned. If \code{analytical} then analytical extraction of the standard errors is performed. By setting it to \code{none} standard error estimation is avoided}
  \item{verbose}{if \code{TRUE} information during estimation process is printed out}
}
\details{There are a lot of details to be written. It will be very soon.

Specific information can be obtained on functions help.

This algorithm fits fully parametric Poisson-Gamma model also.
}
\value{List containing an object of class \code{pgam}.
}
\references{
Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica

Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{predict.pgam}}, \code{\link{pgam.parser}}, \code{\link{residuals.pgam}}, \code{\link{backfitting}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

summary(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.filter}
\alias{pgam.filter}
\title{Estimation of the conditional distributions parameters of the level}
\description{
The priori and posteriori conditional distributions of the level is gamma and their parameters are estimated through this recursive filter. See \bold{Details} for a thorough description.
}
\usage{
pgam.filter(w, y, eta)
}
\arguments{
  \item{w}{running estimate of discount factor \eqn{\omega} of a Poisson-Gamma model}
  \item{y}{\eqn{n} length vector of the time series observations}
  \item{eta}{full linear or semiparametric predictor. Linear predictor is a trivial case of semiparameric model}
}
\details{
Consider \eqn{Y_{t-1}} a vector of observed values of a Poisson process untill the instant \eqn{t-1}. Conditional on that, \eqn{\mu_{t}} has gamma distribution with parameters given by
\deqn{a_{t|t-1}=\omega a_{t-1}}
\deqn{b_{t|t-1}=\omega b_{t-1}\exp\left(-\eta_{t}\right)}
Once \eqn{y_{t}} is known, the posteriori distribution of \eqn{\mu_{t}|Y_{t}} is also gamma with parameters given by
\deqn{a_{t}=\omega a_{t-1}+y_{t}}
\deqn{b_{t}=\omega b_{t-1}+\exp\left(\eta_{t}\right)}
with \eqn{t=\tau,\ldots,n}, where \eqn{\tau} is the index of the first non-zero observation of \eqn{y}.

Diffuse initialization of the filter is applied by setting \eqn{a_{0}=0} and \eqn{b_{0}=0}. A proper distribution of \eqn{\mu_{t}} is obtained at \eqn{t=\tau}, where \eqn{\tau} is the fisrt non-zero observation of the time series.
}
\value{A list containing the time varying parmeters of the priori and posteriori conditional distribution is returned.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Harvey, A. C. (1990) Forecasting, structural time series models and the Kalman Filter. Cambridge, New York

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.likelihood}}, \code{\link{pgam.fit}}, \code{\link{predict.pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.fit}
\alias{pgam.fit}
\title{One-step ahead prediction and variance}
\description{
Estimate one-step ahead expectation and variance of \eqn{y_{t}} conditional on observed time series until the instant \eqn{t-1}.
}
\usage{
pgam.fit(w, y, eta, partial.resid)
}
\arguments{
  \item{w}{estimate of discount factor \eqn{\omega} of a Poisson-Gamma model}
  \item{y}{observed time series which is the response variable of the model}
  \item{eta}{semiparametric predictor}
  \item{partial.resid}{type of partial residuals.}
}
\details{Partial residuals for semiparametric estimation is extracted. Those are regarded to the parametric partition fit of the model. Available types are \code{raw}, \code{pearson} and \code{deviance}. The type \code{raw} is prefered. Properties of other form of residuals not fully tested. Must be careful on choosing it.
See details in \code{\link{predict.pgam}} and \code{\link{residuals.pgam}}.
}
\value{
  \item{yhat}{vector of one-step ahead prediction}
  \item{resid }{vector partial residuals}
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Harvey, A. C. (1990) Forecasting, structural time series models and the Kalman Filter. Cambridge, New York

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{residuals.pgam}}, \code{\link{predict.pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.hes2se}
\alias{pgam.hes2se}
\title{Utility function}
\description{
Put hyperparameters hessian matrix in the form of omega and beta standard error.
}
\usage{
pgam.hes2se(hes, fperiod, se.estimation="numerical")
}
\arguments{
  \item{hes}{hessian matrix returned by \code{optim}}
  \item{fperiod}{vector containing as many seasonal factors as there are in model formula}
  \item{se.estimation}{indicate what method is used to extract the stardard errors}
}
\value{List containing the hyperparameters \eqn{omega} and \eqn{beta} standard errors.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.likelihood}
\alias{pgam.likelihood}
\title{Likelihood function to be maximized}
\description{
This is the log-likelihood function that is passed to \code{optim} for likelihood maximization.
}
\usage{
pgam.likelihood(par, y, x, offset, fperiod, env = parent.frame())
}
\arguments{
  \item{par}{vector of parameters to be optimized}
  \item{y}{observed time series which is the response variable of the model}
  \item{x}{observed explanatory variables for parametric fit}
  \item{offset}{model offset. Just like in GLM}
  \item{fperiod}{vector of seasonal factors to be passed to \code{pgam.par2psi}}
  \item{env}{the caller environment for log-likelihood value to be stored}
}
\details{
Log-likelihood function of hyperparameters \eqn{\omega} and \eqn{\beta} is given by
\deqn{\log L\left(\omega,\beta\right)=\sum_{t=\tau+1}^{n}{\log \Gamma\left(a_{t|t-1}+y_{t}\right)-\log y_{t}!-\cr
\log \Gamma\left(a_{t|t-1}\right)+a_{t|t-1}\log b_{t|t-1}-\left(a_{t|t-1}+y_{t}\right)\log \left(1+b_{t|t-1}\right)}}
where \eqn{a_{t|t-1}} and \eqn{b_{t|t-1}} are estimated as it is shown in \code{\link{pgam.filter}}.
}
\value{
List containing log-likelihood value, optimum linear predictor and the gamma parameters vectors.}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Harvey, A. C. (1990) Forecasting, structural time series models and the Kalman Filter. Cambridge, New York

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.filter}}, \code{\link{pgam.fit}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.par2psi}
\alias{pgam.par2psi}
\title{Utility function}
\description{
Put unconstrained optimized parameters back into omega and beta form.}
\usage{
pgam.par2psi(par, fperiod)
}
\arguments{
  \item{par}{vector of unconstrained parameters}
  \item{fperiod}{vector containing as many seasonal factors as there are in model formula}
}
\value{List containing the hyperparameters \eqn{omega} and \eqn{beta}.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
%
\name{pgam.parser}
\alias{pgam.parser}
\title{Read the model formula and split it into the	parametric and nonparametric partitions}
\description{
Read the model formula and split it into two new ones concerning the parametric and nonparametric partitions of the predictor.
}
\usage{
pgam.parser(formula, parent.level=1)
}
\arguments{
\item{formula}{object representing the model formula. \R standard for GLM models}
\item{parent.level}{an environment to be used as the parent of the environment created}
}
\details{
This function extracts all the information in the model formula. Most important, split the model into two parts regarding the parametric nature of the model.
A model can be specified as following:
\deqn{Y~f\left(sf_{r}\right)+V1+V2+V3+g\left(V4,df_{4}\right)+g\left(V5,df_{5}\right)}
where \eqn{sf_{r}} is a seasonal factor with period \eqn{r} and \eqn{df_{i}} is the degree of freedom of the smoother of the \emph{i}-th covariate. Actually, two new formulae will be created:
\deqn{~sf_{1}+\dots+sf_{r}+V1+V2+V3}
and
\deqn{~V4+V5}
These two formulae will be used to build the necessary datasets for model estimation. \emph{Dummy} variables reproducing the seasonal factors will be created also.

Models without explanatory variables must be specified as in the following formula
\deqn{Y~NULL}
}
\value{List containing the information needed for model fitting.
}
\author{
Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{f}}, \code{\link{g}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}



\eof
\name{pgam.psi2par}
\alias{pgam.psi2par}
\title{Utility function}
\description{
Put hyperparameters into unconstrained form for optimization.
}
\usage{
pgam.psi2par(w, beta, fperiod)
}
\arguments{
  \item{w}{discount factor of the Poisson-Gamma model}
  \item{beta}{explanatory variables coefficients}
  \item{fperiod}{vector containing as many seasonal factors as there are in model formula}
}
\value{Vector of unconstrained parameters.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.parser}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{pgam.smooth}
\alias{pgam.smooth}
\title{Smoothing of nonparametric terms}
\description{Interface for smoothing functions
}
\usage{
pgam.smooth(y, x, df, fx, smoother = "spline", w = rep(1, length(y)))
}
\arguments{
  \item{y}{dependent variable for fitting. In semiparametric models, this is the partial residuals of parametric fit}
  \item{x}{independent variable. Univariate fit only}
  \item{df}{equivalent degrees of freedom}
  \item{fx}{if \code{FALSE} the smoothing parameter is chosen by cross-validation}
  \item{smoother}{string with the name of the smoother to be used}
  \item{w}{vector with the diagonal elements of the weight matrix. Default is a vector of \eqn{1} with the same length of \eqn{y}}
}
\details{Although several smoothers can be used in semiparametric regression models, only natural cubic splines is intended to be used in Poisson-Gamma Additive Models due to its interesting mathematical properties.

Nowadays, this function interface the \code{\link[modreg]{smooth.spline}} in \code{modreg} library. It will become not dependent soon enough.
}
\value{
  \item{fitted}{smoothed values}
  \item{lev}{diagonal of the influence matrix}
  \item{df}{degrees of freedom}
}
\references{
Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London

Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\note{This function is not intended to be called directly.
}
\seealso{\code{\link{pgam}}, \code{\link{predict.pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{plot.pgam}
\alias{plot.pgam}
\title{Plot of estimated curves}
\description{Plot of the local level and, when semiparametric model is fitted, the estimated smooth terms.
}
\usage{
\method{plot}{pgam}(x, rug = TRUE, se = TRUE, at.once = FALSE, ...)
}
\arguments{
  \item{x}{object of class \code{pgam} holding the fitted model}
  \item{rug}{if \code{TRUE} a density rug is drawn on the bottom of the graphic}
  \item{se}{if \code{TRUE} error band is drawn around the fitted values}
  \item{at.once}{if \code{TRUE} each plot goes to a separate window, else the user is prompted to continue}
  \item{\dots}{further arguments passed to method}
}
\details{Error band of smooth terms is approximated.
}
\value{No value returned.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{pgam.likelihood}}}\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

plot(m,at.once=TRUE)

}

	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{predict.pgam}
\alias{predict.pgam}
\title{Prediction}
\description{Prediction and forecasting of the fitted model.}
\usage{
\method{predict}{pgam}(object, forecast = FALSE, k = 1, x = NULL, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{forecast}{if \code{TRUE} the function tries to forecast}
  \item{k}{steps for forecasting}
  \item{x}{covariate values for forecasting if the model has covariates. Must have the \eqn{k} rows and \eqn{p} columns}
  \item{\dots}{further arguments passed to method}
}
\details{It estimates predicted values, their variances, deviance components, generalized Pearson statistics components, local level, smoothed prediction and forecast.

Considering a Poisson process and a gamma priori, the predictive distribution of the model is negative binomial with parameters \eqn{a_{t|t-1}} and \eqn{b_{t|t-1}}. So, the conditional mean and variance are given by
\deqn{E\left(y_{t}|Y_{t-1}\right)=a_{t|t-1}/b_{t|t-1}}
and
\deqn{Var\left(y_{t}|Y_{t-1}\right)=a_{t|t-1}\left(1+b_{t|t-1}\right)/b_{t|t-1}^{2}}

Deviance components are estimated as follow
\deqn{D\left(y;\hat\mu\right)=2\sum_{t=\tau+1}^{n}{a_{t|t-1}\log \left(\frac{a_{t|t-1}}{y_{t}b_{t|t-1}}\right)-\left(a_{t|t-1}+y_{t}\right)\log \frac{\left(y_{t}+a_{t|t-1}\right)}{\left(1+b_{t|t-1}\right)y_{t}}}}

Generalized Pearson statistics has the form
\deqn{X^{2}=\sum_{t=\tau+1}^{n}\frac{\left(y_{t}b_{t|t-1}-a_{t|t-1}\right)^{2}} {a_{t|t-1}\left(1+b_{t|t-1}\right)}}

Approximate scale parameter is given by the expression
\deqn{\hat\phi=frac{X^{2}}{edf}}
where \eqn{edf} is the number o degrees of reedom of the fitted model.
}
\value{List with those described in \bold{Details}
}
\references{
Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London

Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica

Harvey, A. C. (1990) Forecasting, structural time series models and the Kalman Filter. Cambridge, New York

Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London

McCullagh, P., Nelder, J. A. (1989). Generalized Linear Models. Chapman and Hall, 2nd edition, London
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{residuals.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

p <- predict(m)$yhat
plot(ITRESP5)
lines(p)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{print.summary.pgam}
\alias{print.summary.pgam}
\title{Summary output}
\description{Print output of model information}
\usage{
\method{print.summary}{pgam}(x, digits, ...)
}
\arguments{
  \item{x}{object of class \code{summary.pgam} holding the fitted model information}
  \item{digits}{number of decimal places for output}
  \item{\dots}{further arguments passed to method}
}
\details{This function actually only prints out the information. 
}
\value{No value is returned.
}
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{predict.pgam}}}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{residuals.pgam}
\alias{residuals.pgam}
\title{Residuals extraction}
\description{
Method for residuals extraction.
}
\usage{
\method{residuals}{pgam}(object, type = "deviance", ...)
}
\arguments{
  \item{object}{object of class \code{pgam} holding the fitted model}
  \item{type}{type of residuals to be extracted. Default is \code{deviance}. Options are described in \bold{Details}}
  \item{\dots}{further arguments passed to method}
}
\details{
The types of residuals available and a brief description are the following:

\code{response}\cr
These are raw residuals of the form \eqn{r_{t}=y_{t}-E\left(y_{t}|Y_{t-1}\right)}.

\code{pearson}\cr
Pearson residuals are quite known and for this model they take the form \eqn{r_{t}=\left(y_{t}-E\left(y_{t}|Y_{t-1}\right)\right)/Var\left(y_{t}|Y_{t-1}\right)}.

\code{deviance}\cr
Deviance residuals are estimated by \eqn{r_{t}=sign\left(y_{t}-E\left(y_{t}|Y_{t-1}\right)\right)*sqrt\left(d_{t}\right)}, where \eqn{d_{t}} is the deviance contribution of the \emph{t}-th observation. See \code{\link{deviance.pgam}} for details on deviance component estimation.

\code{std_deviance}\cr
Same as deviance, but the deviance component is divided by \eqn{(1-h_{t})}, where \eqn{h_{t}} is the \emph{t}-th element of the diagonal of the pseudo hat matrix of the approximating linear model. So they turn into \eqn{r_{t}=sign\left(y_{t}-E\left(y_{t}|Y_{t-1}\right)\right)*sqrt\left(d_{t}/\left(1-h_{t}\right)\right)}.\cr
The element \eqn{h_{t}} has the form \eqn{h_{t}=\omega\exp\left(\eta_{t+1}\right)/\sum_{j=0}^{t-1}\omega^{j}\exp\left(\eta_{t-j}\right)}, where \eqn{\eta} is the predictor of the approximating linear model.

%\code{std_scl_deviance}\cr
%Just like the last one except for the dispersion parameter in its expression, so they have the form \eqn{r_{t}=sign\left(y_{t}-E\left(y_{t}|Y_{t-1}\right)\right)*sqrt\left(d_{t}/\phi*\left(1-h_{t}\right)\right)}, where \eqn{\phi} is the estimated dispersion parameter of the model. See \code{\link{summary.pgam}} for \eqn{\phi} estimation.

%\code{adj_deviance}\cr
%These are the deviance residuals multiplied by the coefficient of skewness estimated from the distribution. So, considering the negative binomial predictive distribution, they take the form \eqn{r_{t}=sign\left(y_{t}-E\left(y_{t}|Y_{t-1}\right)\right)*sqrt\left(d_{t}\right)*K}, where \eqn{K} is estimated by \eqn{K=\left(2-btt1\right)/sqrt\left(att1*\left(1-btt1\right)\right)}.
}
\value{Vector of residuals of the model fitted.
}
\references{
Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417

Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria

Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica

McCullagh, P., Nelder, J. A. (1989). Generalized Linear Models. Chapman and Hall, 2nd edition, London

Pierce, D. A., Schafer, D. W. (1986) Residuals in generalized linear models. Journal of the American Statistical Association, 81(396),977-986
} 
\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
}
\seealso{\code{\link{pgam}}, \code{\link{pgam.fit}}, \code{\link{predict.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

r <- resid(m,"pearson")
plot(r)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
\name{summary.pgam}
\alias{summary.pgam}
\title{Summary output}
	\description{Output of model information}
	\usage{
	\method{summary}{pgam}(object, ...)
	}
	\arguments{
	\item{object}{object of class \code{pgam} holding the fitted model}
	\item{\dots}{further arguments passed to method}
	}
	\details{Hypothesis tests of coefficients are based o \emph{t} distribution. Significance tests of smooth terms are approximate for model selection purpose only. Be very careful about the later.
	}
	\value{List containing all the information about the model fitted.
	}
	\references{
	Harvey, A. C., Fernandes, C. (1989) Time series models for count data or qualitative observations. Journal of Business and Economic Statistics, 7(4):407--417
	
	Campos, E. L., De Leon, A. C. M. P., Fernandes, C. A. C. (2003) Modelo Poisson-Gama para Sries Temporais de Dados de Contagem - Teoria e Aplicaes. 10a ESTE - Escola de Sries Temporais e Econometria
	
	Junger, W. L. (2004) Modelo Poisson-Gama Semi-Paramtrico: Uma Abordagem de Penalizao por Rugosidade. MSc Thesis. Rio de Janeiro, PUC-Rio, Departamento de Engenharia Eltrica
	
	Green, P. J., Silverman, B. W. (1994) Nonparametric Regression and Generalized Linear Models: a roughness penalty approach. Chapman and Hall, London
	
	Hastie, T. J., Tibshirani, R. J.(1990) Generalized Additive Models. Chapman and Hall, London
	
	McCullagh, P., Nelder, J. A. (1989). Generalized Linear Models. Chapman and Hall, 2nd edition, London
	
	Pierce, D. A., Schafer, D. W. (1986) Residuals in generalized linear models. Journal of the American Statistical Association, 81(396),977-986
	} 
	\author{Washington Leite Junger \cr \email{wjunger@ims.uerj.br}
	}
	\seealso{\code{\link{pgam}}, \code{\link{predict.pgam}}}
\examples{
library(pgam)
data(aihrio)
attach(aihrio)
form <- ITRESP5~f(WEEK)+HOLIDAYS+rain+PM+g(tmpmax,7)+g(wet,3)
m <- pgam(form,aihrio,omega=.8,beta=.01,maxit=1e2,eps=1e-4,optim.method="BFGS",partial.resid="response")

summary(m)

}
	\keyword{ts}
	\keyword{smooth}
	\keyword{regression}

\eof
