\name{addlogo-methods}
\docType{methods}
\title{Methods for Adding a Pixmap Logo to a Plot}
\alias{addlogo}
\alias{addlogo-methods}
\alias{addlogo,pixmap-method}
\description{
  This method allows the addition of a re-scaled pixmap to an existing
  plot, possibly as a logo, optionally preserving aspect.  It may be
  used interactively with \code{locator}, and is positioned in the
  coordinate system of the plot region.  Since the logo is displayed
  using \code{image}, it does not yet seem possible to use the function
  outside this region.
}
\usage{
  addlogo(x, \dots)
  \S4method{addlogo}{pixmap}(x, px, py, asp = NULL)
}
\arguments{
  \item{x}{an object of class \code{pixmap}}
  \item{px}{a vector of two x coordinates, or a list with two named
    elements x and y, such as that returned by \code{\link{locator}}.}
  \item{py}{if px is not a list, a vector of two y coordinates}
  \item{asp}{if omitted or NULL (default), output respects both y
    coordinates, if a number greater than zero, aspect is preserved in
    proportion to the difference between x coordinates, multiplied by asp,
    and only the first y coordinate is respected.}
  \item{\dots}{potentially further arguments passed to and from methods.}
}
\value{The same pixmap object with changed bounding box and cell resolution}
\author{Roger Bivand}
\examples{
x <- read.pnm(system.file("pictures/logo.ppm", package="pixmap")[1])
plot(x)
for (i in 1:7)
  addlogo(x, px=c(0, (101/77)*11), py=c((i-1)*11, i*11), asp=1)
}
\keyword{methods}


\eof
\name{channels-methods}
\docType{methods}
\title{Methods for Channel Manipulation}
\alias{addChannels}
\alias{getChannels}
\alias{addChannels-methods}
\alias{getChannels-methods}
\alias{addChannels,pixmapRGB-method}
\alias{getChannels,pixmapChannels-method}
\description{Functions for manipulation and extraction of colors from
  channel-based pixmap formats. Methods for generic \code{addChannels} adds the
  color channels of a colored pixmap and returns a grey version. Methods
  for generic \code{getChannels} return numeric matrices or arrays
  containing the specified channels.}
\usage{
  addChannels(object, coef = NULL)
  getChannels(object, colors = "all")
}
\arguments{
  \item{object}{Object of class \code{"pixmap"}.}
  \item{coef}{Coefficients for the color channels, a numeric vercot with
    as many elements as there are color channels in the pixmap. The
    default for RGB is \code{c(0.30, 0.59, 0.11)}, which makes a
    luminance-based conversion from color to grey.}
  \item{colors}{Character vector naming the color channels that shall be
    extracted. The default of \code{"all"} returns all channels
    simultaneously.}
}
\keyword{methods}
\author{Friedrich Leisch}
\examples{
  x <- pixmapRGB(rep(1:5, 3), nrow=4)
  plot(x)
  print(x)

  getChannels(x)
  getChannels(x, colors=c("red", "green"))

  y = addChannels(x)
  plot(y)
  print(y)

  ## extract only the red channel
  y = addChannels(x, coef=c(1,0,0))
  plot(y)
}

\eof
\name{pixmap-class}
\docType{class}
\alias{pixmap-class}
\alias{pixmapRGB-class}
\alias{pixmapGrey-class}
\alias{pixmapIndexed-class}
\alias{pixmapChannels-class}
\alias{coerce,pixmapGrey,pixmapIndexed-method}
\alias{coerce,pixmapGrey,pixmapRGB-method}
\alias{coerce,pixmapRGB,pixmapGrey-method}
\alias{coerce,pixmapRGB,pixmapIndexed-method}
\alias{coerce,pixmapIndexed,pixmapRGB-method}
\alias{coerce,ANY,pixmapGrey-method}
\alias{coerce,ANY,pixmapIndexed-method}
\title{Class Family "pixmap".}
\description{The family \code{"pixmap"} (``pixel maps'') of classes provides
  methods for creating,
  plotting and converting bitmapped images in currently three different
  formats: RGB (\code{"pixmapRGB"}), grey
  (\code{"pixmapGrey"})and indexed pixmaps (\code{"pixmapIndexed"}).}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("pixmap", ...)} or
  using the creator functions \code{\link{pixmap}} (similar for all child
  classes of name (\code{"pixmapXXX"}).
}
\section{Slots}{
  \describe{
    \item{\code{size}:}{Object of class \code{"integer"} and length 2
      (number of rows and columns).}
    \item{\code{cellres}:}{Object of class \code{"numeric"} and length
      2 specifying the cell resolution of each pixel in user coordinates.}
    \item{\code{bbox}:}{Object of class \code{"numeric"} and length 4,
      the coordinates of the bounding box (x bottom, y bottom, x top, y
      top).}
    \item{\code{channels}:}{A character vector naming the channel slots
      of the object (\code{NULL} for indexed pixmaps).}
    \item{\code{red}, \code{green}, \code{blue}:}{Only for class
      \code{"pixmapRGB"} with matrices specifying the red, green
      and blue
      channel of the picture.}
    \item{\code{grey}:}{Only for class
      \code{"pixmapGrey"}, a matrix specifying the grey intensity
      (0=black, 1=white) of the picture.}
    \item{\code{col}:}{Only for class
      \code{"pixmapGrey"}, a character vector with a map of color
      names.}
    \item{\code{index}:}{Only for class
      \code{"pixmapIndexed"}, an integer matrix with codes from the
      color map.
    }
  }
}
\section{Details}{
  Class \code{"pixmap"} specifies the basic geometry of a picture: the
  size in pixels together with information for an optional coordinate
  system, see \code{\link{pixmap}} for details.

  Grey and indexed pixmaps are besically matrices (contained in the
  \code{grey} or \code{index} slot, respectively). The element \code{[1,1]}
  corresponds to the upper left corner as usual. For grey pixmaps the
  elements must be between 0 (black) and 1 (white). Indexed pixmaps have
  integer elements, each giving the index number corresponding to the
  palette specified in slot \code{"col"}.
  Colors are given using the usual R
  color strings (either names like \code{"red"} or hex values like
  \code{"#FF0000"}). Alternatively, a function to create a color palette
  can be specified, see \code{\link{rainbow}} or
  \code{\link{heat.colors}} for examples.

  RGB pixmaps have three matrices for each of the three color
  channels. Elements of the matrices must be between 0 (=color off) and
  1 (=color at maximum intensity).

  Methods for coercion between all formats are available.

  Class \code{"pixmapChannels"} is a
  helper parent class currently containing classes \code{"pixmapRGB"}
  and \code{"pixmapGrey"}.
}
\author{Friedrich Leisch}
\seealso{\code{\link{pixmap}}}
\keyword{classes}

\eof
\name{pixmap}
\title{Pixmap Images}
\alias{pixmap}
\alias{pixmapRGB}
\alias{pixmapGrey}
\alias{pixmapIndexed}
\alias{plot,pixmap-method}
\alias{show,pixmap-method}
\alias{[,pixmap-method}

\usage{
pixmap(data=NULL, nrow=dim(data)[1], ncol=dim(data)[2],
       bbox=NULL, bbcent=FALSE, cellres=NULL)
pixmapRGB(data, ...)
pixmapGrey(data, ...)
pixmapIndexed(data, col, ...)
}
\description{
  The family \code{"pixmap"} (``pixel maps'') of classes provides
  methods for creating,
  plotting and converting bitmapped images in three different formats:
  RGB, grey and indexed pixmaps.}
\arguments{
  \item{data}{An optional data vector.}
  \item{nrow}{Vertical size of the image in pixels.}
  \item{ncol}{Horizontal size of the image in pixels.}
  \item{bbox}{Bounding box of the image, vector of length 4 of form \code{c(x1,
      y1, x2, y2)} with coordinates for the lower left corner and upper
    right corner.}
  \item{bbcent}{Logical, if \code{TRUE} the bounding box specifies the
    coordinates of the centers of the lower left and upper right pixels,
    default is the coordinates of the lower left and upper right corner
    of the image.}
  \item{cellres}{Numeric vector of length 1 or 2, specifies the
    resolution of pixels in horizontal and vertical direction. If only
    one value is given, resolution in both directions is identical.}
  \item{col}{Character vector of colors to use for indexed pictures, or
    a function like \code{\link{rainbow}} which can
    be used to create a palette.}
  \item{...}{Additional arguments passed to \code{pixmap()}.}
}
\details{
  If the \code{data} argument is 2- or 3-dimensional, \code{nrow} and
  \code{ncol} default to the first two dimensions of \code{data}, such
  that \code{pixmap} does the expected when given a matrix or an array.
  
  The arguments \code{bbox}, \code{bbcent} and \code{cellres} can be
  used to specify a coordinate system for the image. Note that together
  with \code{nrow} and \code{ncol} the coordinate system is
  overspecified, hence not all parameters must be specified, the rest is
  computed or set to sensible defaults.

  For \code{bbcent=FALSE} we have
  \code{cellres[1] = (bbox[3]-bbox[1])/ncol}
  and
  \code{cellres[2] = (bbox[4]-bbox[2])/nrow}, for  \code{bbcent=TRUE}
  we get
  \code{cellres[1] = (bbox[3]-bbox[1])/(ncol-1)}
  and
  \code{cellres[2] = (bbox[4]-bbox[2])/(nrow-1)}.
  
  The name \code{pixmap} was chosen because both \code{image} and
  \code{bitmap} are already used in R.
}
\author{Friedrich Leisch}
\seealso{\code{\link{pixmap-class}}, \code{\link{read.pnm}}}
\keyword{color}
\examples{
 ## A simple example
 x <- pixmapIndexed(rep(1:8, 9), nrow=6, col=terrain.colors(8))
 plot(x)

 ## The same with different colors, and passing the function instead of
 ## a color vector 
 x <- pixmapIndexed(rep(1:8, 9), nrow=6, col=rainbow)
 plot(x)
 plot(x, asp=.5, axes=TRUE)

 ## Read data from a file
 x <- read.pnm(system.file("pictures/logo.ppm", package="pixmap")[1])
 plot(x)

 ## Another example that math can be beautiful
 x <- seq(-3,3,length=100)
 z1 <- outer(x,x,function(x,y) abs(sin(x)*sin(y)))
 z2 <- outer(x,x,function(x,y) abs(sin(2*x)*sin(y)))
 z3 <- outer(x,x,function(x,y) abs(sin(x)*sin(2*y)))

 ## Notice that we specify a bounding box to get the correct
 ## coordinates on the axes. z1, z2 and z3 are used as red,
 ## green and blue channel, respectively.
 z <- pixmapRGB(c(z1,z2,z3), 100, 100, bbox=c(-1,-1,1,1))
 plot(z, axes=TRUE)

 ## look at a grey version
 plot(as(z, "pixmapGrey"))

 ## subsetting works as expected
 plot(z[1:20,])
 plot(z[,1:40])
 plot(z[1:20,10:40])
}


\eof
\name{pnm}
\title{Read/Write Portable Anymap Images}
\alias{write.pnm}
\alias{read.pnm}
\alias{read.pnmdata}
\alias{read.pnmhead}
\usage{
read.pnm(file, \dots)
write.pnm(object, file= NULL, forceplain = FALSE, type = NULL, maxval = 255)
}
\description{Reading and writing of bitmap images in PBM (black/white), PGM
  (grey) and PPM (color) format.}
\arguments{
  \item{file}{name of the pnm file.}
  \item{\dots}{further arguments passed to \code{\link{pixmap}} (like
    \code{bbox}).}
  \item{object}{an object of class \code{"pixmap"}.}
  \item{forceplain}{logical; if true, an ASCII pnm file is written.
    Default is to write a binary (raw) file.}
  \item{type}{one of \code{"pbm"}, \code{"pgm"} or \code{"ppm"}. Default
    is to use \code{"pgm"} for grey images and \code{"ppm"} for color
    images.}
  \item{maxval}{the maximum color-component value; the default is a
    colour depth of 8 bits, i.e., the integer 255.}
}
\details{
  \code{read.pnm} reads a pnm file and loads the image into an
  object of class \code{\link{pixmap}}.

  \code{write.pnm} writes an object of class \code{\link{pixmap}} to a
  pnm file, the \code{type} argument controls wheter the written image
  file is a black-and-white bitmap (pbm), grey (pgm) or color (ppm).

  \code{plot.pnm} plots a pnm object using the command
  \code{\link{image}}. The only difference is that the element \code{[1,1]} of
  \code{pnmobj} is plotted as the upper left corner (plain
  \code{\link{image}} would plot \code{[1,1]} as the lower left corner.
}
\value{
  \code{read.pnm} returns an object of class \code{\link{pixmapRGB}} for color
  pixmaps (ppm), and an object of class \code{\link{pixmapGrey}} for pbm
  and pgm. Note that the \emph{type} of file as determined by the first
  two bytes according to pnm standards is important, \emph{not the
    extension} of the file. In fact, the file name extension is
  completely ignored.
}
\author{Roger Bivand and Friedrich Leisch}
\seealso{\code{\link{pixmap}}}
\keyword{file}
\keyword{color}
\examples{
 x <- read.pnm(system.file("pictures/logo.ppm", package="pixmap")[1])
 plot(x)
 print(x)

 x <- read.pnm(system.file("pictures/logo.pgm", package="pixmap")[1])
 plot(x)

 x <- read.pnm(system.file("pictures/logo.pbm", package="pixmap")[1])
 plot(x)
}

\eof
