\name{NIR}
\alias{NIR}
\title{Multivariate dataset of NIR spectra}
\usage{data(NIR)}
\references{H. Swierenga et al., Chemom. Intell. Lab. Syst. 49 (1999) 1-17.}
\format{A dataframe with matrices Xtrain (21x268) and Xtest (7x268), and 
two vectors Ytrain and Ytest.}
\description{A training set consisting of 21 NIR spectra of
PET yarns, measured at 268 wavelengths, and 21 corresponding
densities. A training set of 7 samples is also provided.
Many thanks to Erik Swierenga...}
\keyword{datasets}

\eof
\name{kernelpls}
\alias{kernelpls}
\title{Kernel PLS (Dayal and MacGregor)}
\usage{kernelpls(X, Y, ncomp, newX)}
\description{This function should not be called directly, but through
  the generic \code{pls} function with the argument
  \code{method="kernel"} (default). Kernel PLS is particularly efficient 
  when the number of objects is (much) larger than the number of
  variables. The results are equal to the NIPALS algorithm. Several
  different forms of kernel PLS have been described in literature, e.g.
  by De Jong and Ter Braak, and two algorithms by Dayal and
  MacGregor. This function implements the
  fastest of the latter, not calculating the crossproduct matrix of
  X. In the Dyal & MacGregor paper, this is 'algorithm 1'.
  (\code{kernelpls}).
}
\arguments{
  \item{X}{a matrix of observations. \code{NA}s and \code{Inf}s are not
    allowed.} 
  \item{Y}{a vector or matrix of responses. \code{NA}s and \code{Inf}s
    are not allowed.} 
  \item{ncomp}{the number of latent variables to be used in the
    modelling. The default number of latent variables is the smallest of
    the number of objects or the number of variables in \code{X}.} 
  \item{newX}{optional new measurements: if present, predictions will be
    made for them.}
}
\value{A list containing the following
  components is returned:
  \item{B}{an array of regression coefficients for all items in
      \code{ncomp}. The dimensions of \code{B} are
      \code{c(nvar, npred, length(ncomp))} with \code{nvar} the number
      of \code{X} variables and \code{npred} the number of variables to be
      predicted in \code{Y}.}
    \item{XvarExpl}{Fraction of X-variance explained.}
    \item{YvarExpl}{Fraction of Y-variance explained (one column, even
      for multiple Y).
      }
  \item{Ypred}{predictions for \code{newX} (if asked).}
}
\seealso{
  \code{\link{pls}}
  \code{\link{simpls}}
  \code{\link{mvr}}
}
\examples{
data(NIR)
attach(NIR)
NIR.kernelpls <- mvr(Xtrain, Ytrain, 1:6, validation="CV", method="kernelPLS")
}
\references{
  S. de Jong and C.J.F. ter Braak, J. Chemometrics, 8 (1994) 169-174
  B.S. Dayal and J. MacGregor, J. Chemometrics, 11 (1997) 73-85
  }
\keyword{regression}

\eof
\name{mvr}
\alias{mvr}
\alias{pcr}
\alias{pls}
\title{Multivariate Regression}
\usage{
mvr(X, Y, ncomp,
    method=c("PCR", "SIMPLS", "kernelPLS"),
    validation=c("none","CV"), grpsize, niter)
pcr(\dots)
pls(\dots, method)
}
\description{Generic function for multivariate regression. PCR and two
  types of PLS (SIMPLS and kernel-PLS) are implemented. The functions
  \code{pcr} and \code{pls} are simple wrappers for \code{mvr}.
  Cross-validation can be used; a nmber of latent variables
  to retain is suggested.}
\arguments{
  \item{X}{a matrix of observations. \code{NA}s and \code{Inf}s are not
    allowed.} 
  \item{Y}{a vector or matrix of responses. \code{NA}s and \code{Inf}s
    are not allowed.} 
  \item{ncomp}{the numbers of latent variables to be assessed in the
    modelling. Default is from one to the rank of \code{X}.}
  \item{method}{the multivariate regression method to be used.}
  \item{validation}{validation method, either "none" or cross-validation
    ("CV").}
  \item{grpsize}{the group size for the "CV" validation. If not
    specified, this is determined by the \code{niter} parameter.}
  \item{niter}{the number of iterations in the
    cross-validation. Default: 10. If both \code{niter} and
    \code{grpsize} are specified, \code{niter} takes precedence.}
  \item{\dots}{arguments for \code{mvr}.}
}
\value{An object of class \code{mvr} is returned. This has the following
  components:
  \item{nobj}{the number of objects, viz. the number of rows in X and
    Y.}
  \item{nvar}{the number of independent variables, viz. the number of
    columns in X.}
  \item{npred}{the number of Y variables.}
  \item{Xmeans}{column means of original X data matrix.}
  \item{Y}{original Y data. Y is represented as a matrix, possibly with
    only one column.}
  \item{ncomp}{the number of latent variables considered in the PCR
    regression.}
  \item{method}{the regression method used.}
  \item{training}{a list with the following components:
    \item{B}{an array of regression coefficients for all items in
      \code{ncomp}. The dimensions of \code{B} are
      \code{c(nvar, npred, length(ncomp))} with \code{nvar} the number
      of \code{X} variables and \code{npred} the number of variables to be
      predicted in \code{Y}.
    }
    \item{Ypred}{predicted values for the training data.}
    \item{RMS}{Root-mean-square error of the training data for all items
      in \code{ncomp}.} 
    \item{R2}{coefficient of multiple determination for the training
      data.}
    \item{Xscores}{X-scores for objects.}
    \item{Xload}{X-loadings for objects.}
    \item{Yscores}{Y-scores for objects: only in case of a PLS model
      with more than one Y variable.}
    \item{Yload}{Y-loadings for objects: only in case of a PLS model
      with more than one Y variable.}
  }
  \item{validat}{a list with the following components:
    \item{Ypred}{predicted values for the validation stage.}
    \item{nLV}{number of latent variables suggested by
      cross-validation. The least complex model within one standard
      error of the best is chosen.
    }
    \item{niter}{the number of groups in the cross-validation. If this
      number equals the number of objects, this is leave-one-out
      validation.}
    \item{RMS}{Root-mean-square error from cross-validation for all items
      in \code{ncomp}.} 
    \item{RMS.sd}{standard error of the root-mean-square estimate.} 
    \item{R2}{coefficient of multiple determination from
      cross-validation.}
  }
}
\seealso{
  \code{\link{pcr.model}}, \code{\link{simpls}},
  \code{\link{kernelpls}}, \code{\link{plot.mvr}},
  \code{\link{summary.mvr}}}
}
\examples{
data(NIR)
attach(NIR)
NIR.pcr <- mvr(Xtrain, Ytrain, 1:6, validation="CV")
NIR.simpls <- mvr(Xtrain, Ytrain, 1:6, validation="CV", method="SIMPLS")
NIR.kernelpls <- mvr(Xtrain, Ytrain, 1:6, validation="CV", method="kernelPLS")

data(sensory)
Pn <- scale(sensory$Panel)
Ql <- scale(sensory$Quality)
sens.pcr <- mvr(Ql, Pn, 1:5)
sens.simpls <- mvr(Ql, Pn, 1:5, method="SIMPLS")
sens.kernelpls <- mvr(Ql, Pn, 1:5, method="kernelPLS")
}
\keyword{regression}

\eof
\name{pcr.model}
\alias{pcr.model}
\title{PCR model building}
\usage{pcr.model(X, Y, ncomp, newX)}
\description{Function does the actual model building in PCR
  regression. Y is regressed on Principal Component scores.
  This function should not be called directly: the \code{pcr} 
  function is meant for that.}
\arguments{
  \item{X}{a matrix of observations. \code{NA}s and \code{Inf}s are not
    allowed.} 
  \item{Y}{a matrix of responses. \code{NA}s and \code{Inf}s
    are not allowed.} 
  \item{ncomp}{the number of latent variables to be used in the
    modelling. The default number of latent variables is the smallest of
    the number of objects or the number of variables in \code{X}.} 
  \item{newX}{optional new measurements: if present, predictions will be
    made for them.}
}
\value{A list containing the following components is returned:
  \item{B}{an array of regression coefficients for all items in
      \code{ncomp}. The dimensions of \code{B} are
      \code{c(nvar, npred, length(ncomp))} with \code{nvar} the number
      of \code{X} variables and \code{npred} the number of variables to be
      predicted in \code{Y}.}
  \item{XvarExpl}{coeff. of determination (\code{X}).}
  \item{Ypred}{predictions for \code{newX} (if asked).}
}
\seealso{
  \code{\link{mvr}}
  \code{\link{pcr}}
}
\keyword{regression}

\eof
\name{plot.mvr}
\alias{plot.mvr}
\title{Plot a MVR object}
\usage{
plot.mvr(x,
         plottype=c("prediction", "validation", "coefficients",
                    "scores", "loadings"),
         nlv, which=1:2, \dots)
}
\description{Several different ways of plotting a \code{mvr} object are
  supported:
  \item{\code{type="validation"}}{plot of errors versus the number of
    latent variables. Standard errors are also shown, and the selected
    number of components is indicated, as suggested in Hastie et
    al. (2001).}
  \item{\code{type="prediction"}}{For each Y variable, a plot is
    produced of true versus predicted values. If validation data are
    available, both training and validation plots are shown.
    For each Y variable separate plots are made.
  }
  \item{\code{type="loadings"}}{Shows a plot of loadings for X, in the
    case of a PCR or PLS1 model, and in the case of a PLS2
    model shows a biplot of loadings for X and Y. Loadings for Y are
    indicated in red.}
  \item{\code{type="scores"}}{Shows a plot of scores for X, in the
    case of a PCR model, and in the case of a PLS model shows a
    biplot of scores for X and Y. Scores for Y are
    indicated in red. In some cases, it may be necessary to scale scores
    before a meaningful plot is obtained.}
  \item{\code{type="coefficients"}}{Shows regression vector for the
    selected number of latent variables.}
  }
\arguments{
  \item{x}{an MVR object.} 
  \item{plottype}{type of plot. (Wow!)}
  \item{which}{whether training data (1), validation data (2) or both
    (1:2) should be shown in the \code{prediction} mode of this
    function.}
  \item{nlv}{the number of latent variables to take into account.  For the
    \code{loadings} and \code{scores} plotting types, this should be a
    vector of length 2, defaulting to c(1, 2). In the
    \code{coefficients} plotting type, this may be a vector. Not
    relevant for the \code{validation} plotting type.}
  \item{\dots}{optional arguments for the plotting command.}
}
\section{References}{
  T. Hastie, R. Tibshirani and J. Friedman, \emph{The elements of
    statistical learning}, Springer (2001).
}
\examples{
data(NIR)
attach(NIR)
nir.pcr <- pcr(Xtrain,Ytrain,1:9, validat="CV")
\dontrun{plot(nir.pcr)}
plot(nir.pcr, "coefficients", nlv=3:6, type="l")
\dontrun{plot(nir.pcr, "prediction", nlv=3:6)}
plot(nir.pcr, "prediction", nlv=3:6, which=2)

data(sensory)
Pn <- scale(sensory$Panel)
Ql <- scale(sensory$Quality)
sens.simpls <- mvr(Ql, Pn, 1:5, method="SIMPLS")
plot(sens.simpls, "loadings")
plot(sens.simpls, "scores")
}
\keyword{regression}

\eof
\name{predict.mvr}
\alias{predict.mvr}
\title{MVR predictions}
\usage{predict.mvr(object, newX, nlv, \dots)}
\description{Prediction for MVR (PCR, PLS) models. New responses are
  predicted using a previously defined model and a new matrix of
  independent observations.}
\arguments{
  \item{object}{a MVR model} 
  \item{newX}{a matrix of new independent observations} 
  \item{nlv}{the number of latent variables (optional)}
  \item{\dots}{not used currently.}
}
\value{
  Predicted values are returned.
}
\seealso{
  \code{\link{plot.mvr}}
  \code{\link{summary.mvr}}
}
\examples{
data(NIR)
attach(NIR)
nir.mvr <- mvr(Xtrain, Ytrain, 2)
nir.predictions <- predict(nir.mvr, Xtest)
}
\keyword{regression}

\eof
\name{sensory}
\alias{sensory}
\title{Multivariate dataset of sensory data}
\usage{data(sensory)}
\references{B.G.M. Vandeginste et al., Handbook of Chemometrics and
  Qualimetrics: Part B, Elsevier (1998), Tables 35.1 and 35.4.}
\format{A dataframe with matrices Quality (16 x 5) and Panel (16 x 6).}
\description{Panel gives scores from a sensory panel on a set of 6
  attributes for 16 olive oil samples. Quality gives the results of 5
  physico-chemical quality parameters. The first five oils are Greek,
  the next five are Italian and the last six are Spanish. This data set
  illustrates the qualitative capabilities of PLS.}
\keyword{datasets}

\eof
\name{simpls}
\alias{simpls}
\title{De Jong's SIMPLS}
\usage{simpls(X, Y, ncomp, newX)}
\description{This function should not be called directly, but through
  the generic \code{pls} function with the argument
  \code{method="simpls"}. It is much faster than the NIPALS algorithm,
  especially when the number of X variables increases, but gives
  slightly different results in the case of multivariate Y. SIMPLS truly
  maximises the covariance criterion. According to De Jong, the standard
  PLS2 algorithms lie closer to ordinary least-squares regression where
  a precise fit is sought; SIMPLS lies closer to PCR with stable
  predictions.}
\arguments{
  \item{X}{a matrix of observations. \code{NA}s and \code{Inf}s are not
    allowed.} 
  \item{Y}{a vector or matrix of responses. \code{NA}s and \code{Inf}s
    are not allowed.} 
  \item{ncomp}{the number of latent variables to be used in the
    modelling. The default number of latent variables is the smallest of
    the number of objects or the number of variables in \code{X}.} 
  \item{newX}{optional new measurements: if present, predictions will be
    made for them.}
}
\value{A list containing the following
  components is returned:
  \item{B}{an array of regression coefficients for all items in
      \code{ncomp}. The dimensions of \code{B} are
      \code{c(nvar, npred, length(ncomp))} with \code{nvar} the number
      of \code{X} variables and \code{npred} the number of variables to be
      predicted in \code{Y}.}
    \item{XvarExpl}{Fraction of X-variance explained.}
    \item{YvarExpl}{Fraction of Y-variance explained (one column, even
      for multiple Y).
      }
  \item{Ypred}{predictions for \code{newX} (if asked).}
}
\seealso{
  \code{\link{kernelpls}}
  \code{\link{mvr}}
}
\examples{
data(NIR)
attach(NIR)
NIR.simpls <- mvr(Xtrain, Ytrain, 1:6, validation="CV", method="SIMPLS")
}
\references{S. de Jong, Chemometrics and Intelligent Laboratory
Systems, 18 (1993) 251-263.}
\keyword{regression}

\eof
\name{summary.mvr}
\alias{summary.mvr}
\title{Summary of a MVR object}
\usage{summary.mvr(object, what, digits=4, print.gap=3, ...)}
\description{The information in the MVR object is summarised: the
  function gives the dimensions of X and Y, the method used, and the
  number of latent variables considered. If \code{what} equals "all" or
  "training", the RMS and R2 values of the training set are given; if
  \code{what} equals "all" or "validation", RMS and R2 values for the
  cross-validation are given. In the latter case, a suggestion is made
  about the optimal number of latent variables, as suggested in Hastie
  et al. (2001).
}
\arguments{
  \item{object}{An MVR object.}
  \item{what}{one of "all", "training" or "validation". }
  \item{digits}{Number of digits, default is 4.}
  \item{print.gap}{gap between columns of the print table.}
  \item{\dots}{other arguments to \code{print.table}.}
}
\section{References}{
  T. Hastie, R. Tibshirani and J. Friedman, \emph{The elements of
    statistical learning}, Springer (2001).
}
\examples{
data(NIR)
attach(NIR)
nir.mvr <- mvr(Xtrain, Ytrain, 2:8, valid="CV", grps=1)
summary(nir.mvr)
}
\keyword{regression}

\eof
