\name{Math.polynomial}
\alias{Math.polynomial}
\title{Math Group Methods for Polynomials}
\description{
  Group method for functions in the Math group.
}
\usage{
Math.polynomial(x, \dots)
}
\arguments{
  \item{x}{an object of class \code{"polynomial"}.}
  \item{\dots}{further arguments to be passed to or from methods, such
    as \code{digits}.}
}
\details{
  Most math group functions are disallowed with polynomial arguments.
  The only exceptions are \code{\link{ceiling}}, \code{\link{floor}},
  \code{\link{round}}, \code{\link{trunc}}, and \code{\link{signif}}
  which may be used to transform the coefficients accordingly.
}
\value{
  A polynomial with transformed coefficients.
}
\seealso{
  \code{\link{Ops.polynomial}},
  \code{\link{Summary.polynomial}}.
}
\examples{
op <- options(digits=18)
p <- poly.from.values(1:4, (2:5)^2)
## 1 + 2.00000000000001*x + x^2
p <- round(p)
## 1 + 2*x + x^2
options(op)
}
\keyword{symbolmath}

\eof
\name{Ops.polynomial}
\alias{Ops.polynomial}
\title{Arithmetic Ops Group Methods for Polynomials}
\description{
  Allows arithmetic operators to be used for polynomial calculations,
  such as addition, multiplication, division, etc.
}
\usage{
Ops.polynomial(e1, e2)
}
\arguments{
  \item{e1}{an object of class \code{"polynomial"}.}
  \item{e2}{an object of class \code{"polynomial"}.}
}
\value{
  A polynomial got by performing the operation on the two arguments.
}
\seealso{
  \code{\link{Math.polynomial}},
  \code{\link{Summary.polynomial}}.
}
\examples{
p <- polynomial(c(1, 2, 1))
## 1 + 2*x + x^2
r <- poly.calc(-1 : 1)
## -1*x + x^3
(r - 2 * p)^2
## 4 + 20*x + 33*x^2 + 16*x^3 - 6*x^4 - 4*x^5 + x^6
}
\keyword{symbolmath}

\eof
\name{as.function.polynomial}
\alias{as.function.polynomial}
\title{Coerce a Polynomial to a Function}
\description{
  Takes a polynomial argument and constructs an \R function to evaluate
  it at arbitrary points.
}
\usage{
\method{as.function}{polynomial}(x, \dots)
}
\arguments{
  \item{x}{An object of class \code{"polynomial"}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  This is a method for the generic function \code{\link{as.function}}.

  The polynomial is evaluated within the function using the Horner
  scheme.

  Note that you can use the model-oriented \code{predict} method for
  polynomials for purpose of evaluation (without explicit coercion to
  a function), see the example below.}
}
\value{
  A function to evaluate the polynomial \code{p}.
}
\seealso{
  \code{\link{as.function}},
  \code{\link{predict.polynomial}}
}
\examples{
pr <- (poly.calc(-1:1) - 2 * polynomial(c(1, 2, 1)))^2
pr
## 4 + 20*x + 33*x^2 + 16*x^3 - 6*x^4 - 4*x^5 + x^6
prf <- as.function(pr)
prf
## function (x) 
## 4 + x * (20 + x * (33 + x * (16 + x * (-6 + x * (-4 + x * (1))))))
## <environment: 0x402440f0>
prf(-3:3)
##  1024 64  0 4 64 144 64
predict(pr, -3:3)
##  1024 64  0 4 64 144 64
}
\keyword{symbolmath}

\eof
\name{change.origin}
\alias{change.origin}
\title{Change Origin for a Polynomial}
\description{
  Calculate the coefficients of a polynomial relative to a new origin
  on the x axis.
}
\usage{
change.origin(p, o)
}
\arguments{
  \item{p}{an object of class \code{"polynomial}".} 
  \item{o}{a numeric scalar representing the new origin on the original
    scale.}
}
\details{
  Let \eqn{P(x) = \sum_i p_i x^i} be a given polynomial and consider
  writing \eqn{P(x) = sum_j q_j (x - o)^j}.  This function calculates
  the coefficients \eqn{q_j} and returns the result as a polynomial.
}
\value{
  A polynomial with coefficients relative to the re-located x axis.
}
\examples{
pr <- poly.calc(1:5)
pr
## -120 + 274*x - 225*x^2 + 85*x^3 - 15*x^4 + x^5
change.origin(pr, 3)
## 4*x - 5*x^3 + x^5
}
\keyword{symbolmath}

\eof
\name{deriv.polynomial}
\alias{deriv.polynomial}
\title{Differentiate a Polynomial}
\description{
  Calculates the derivative of a univariate polynomial.
}
\usage{
\method{deriv}{polynomial}(expr, \dots)
}
\arguments{
  \item{expr}{an object of class \code{"polynomial"}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  This is a method for the generic function \code{\link{deriv}}.
}
\value{
  Derivative of the polynomial.
}
\seealso{
  \code{\link{integral.polynomial}},
  \code{\link{deriv}}.
}
\examples{
pr <- poly.calc(1:5)
pr
## -120 + 274*x - 225*x^2 + 85*x^3 - 15*x^4 + x^5
deriv(pr)
##  274 - 450*x + 255*x^2 - 60*x^3 + 5*x^4
}
\keyword{symbolmath}

\eof
\name{integral.polynomial}
\alias{integral}
\alias{integral.polynomial}
\title{Integrate a Polynomial}
\description{
  Find the integral of a univariate polynomial.
}
\usage{
\method{integral}{polynomial}(expr, limits = NULL, ...)
}
\arguments{
  \item{expr}{an object of class \code{"polynomial"}.}
  \item{limits}{numeric vector of length 2 giving the integration
    limits.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
  If \code{limits} is not given, the integral of \code{p} from 0 to
  `x'.  Otherwise, the integral with the given integration limits.}
}
\seealso{
  \code{\link{deriv.polynomial}}
}
\examples{
p <- poly.calc(1:5)
p
## -120 + 274*x - 225*x^2 + 85*x^3 - 15*x^4 + x^5
deriv(p)
## 274 - 450*x + 255*x^2 - 60*x^3 + 5*x^4
integral(deriv(p)) - 120
## -120 + 274*x - 225*x^2 + 85*x^3 - 15*x^4 + x^5
}
\keyword{symbolmath}

\eof
\name{lines.polynomial}
\alias{lines.polynomial}
\title{Lines Method for Polynomials}
\description{
  Add a polynomial to an existing plot usually as a line plot.
}
\usage{
\method{lines}{polynomial}(x, len = 100, \dots)
}
\arguments{
  \item{x}{an object of class \code{"polynomial"}.}
  \item{len}{size of vector at which evaluations are to be made.}
  \item{\dots}{additional arguments as for the \code{\link{lines}}
    generic.}
}
\details{
  This is a method for the generic function \code{\link{lines}}.
    
  Lines representing the given polynomial are added to an existing
  plot.  Values outside the current plot region are not shown.
}
\seealso{
  \code{\link{lines}},
  \code{\link{points}},
  \code{\link{points.polynomial}},
  \code{\link{plot}},
  \code{\link{plot.polynomial}}.
}
\examples{
plot  (poly.calc(-1:5))
lines (poly.calc( 2:4), lty = 2)
points(poly.calc(-2:6), pch = 4)
}
\keyword{symbolmath}

\eof
\name{monic}
\alias{monic}
\title{Monic Polynomials}
\description{
  Convert a polynomial to monic form by dividing by the leading
  coefficient.
}
\usage{
monic(p)
}
\arguments{
  \item{p}{A polynomial.  A warning is issued if the polynomial is
    identically zero.}
}
\details{
  Similar in effect to \code{p/as.numeric(p[length(p)])} but with some
  safeguards against leading zero coefficients.
}
\value{
  A polynomial proportional to \code{p} with leading coefficient 1.
}
\seealso{
  \code{\link{change.origin}}
}
\keyword{symbolmath}

\eof
\name{plot.polynomial}
\alias{plot.polynomial}
\title{Plot Method for Polynomials}
\description{
  Plots polynomials, optionally allowing the ``interesting'' region to
  be automatically determined.
}
\usage{
\method{plot}{polynomial}(x, xlim = 0:1, ylim = range(Px), type = "l",
                len = 100, \dots)
}
\arguments{
  \item{x}{an object of class \code{"polynomial"}.}
  \item{xlim}{the range to be encompased by the x axis.}
  \item{ylim}{the range to be encompased by the y axis.}    
  \item{type}{as for \code{\link{plot}()}.}
  \item{len}{number of x points drawn.}
  \item{\dots}{additional arguments as for \code{\link{plot}}.}
}
\details{
  This is a method for the generic function \code{\link{plot}}.
    
  A plot of the polynomial is produced on the currently active device.
  Unless otherwise specified, the domain is chosen to enclose the real
  parts of all zeros, stationary points and zero itself.
}
\seealso{
  \code{\link{plot}},
  \code{\link{lines}},
  \code{\link{points}},
  \code{\link{lines.polynomial}},
  \code{\link{points.polynomial}}.
}
\examples{
plot(p <- poly.calc(-1:5))
}
\keyword{symbolmath}

\eof
\name{points.polynomial}
\alias{points.polynomial}
\title{Points Method for Polynomials}
\description{
  Add a polynomial to an existing plot usually as a point plot.
}
\usage{
\method{points}{polynomial}(x, length = 100, \dots)
}
\arguments{
  \item{x}{an object of class \code{"polynomial"}.}
  \item{length}{size of x vector at which evaluations are to be made.}
  \item{\dots}{additional arguments as for the points generic.}
}
\details{
  This is a method for the generic function \code{\link{points}}.

  Points representing the given polynomial are added to an existing
  plot.  Values outside the current plot region are not shown.
}
\seealso{
  \code{\link{plot}},
  \code{\link{lines}},
  \code{\link{points}},
  \code{\link{plot.polynomial}},
  \code{\link{lines.polynomial}}.
}
\examples{
plot(poly.calc(-1:5))
lines(poly.calc(2:4), lty=2)
points(poly.calc(-2:6), pch=4)
}
\keyword{symbolmath}

\eof
\name{poly.calc}
\alias{poly.calc}
\alias{poly.from.zeros}
\alias{poly.from.roots}
\alias{poly.from.values}
\title{
Calculate Polynomials from Zeros or Values
}
\description{
  Calculate either the monic polynomial with specified zeros, or the
  Lagrange interpolation polynomial through the (x,y) points.
}
\usage{
poly.calc(x, y, tol=sqrt(.Machine$double.eps), lab=dimnames(y)[[2]])
}
\arguments{
  \item{x}{numeric vector specifying either the zeros of the desired
    polynomial if this is the only non-missing argument, or the	x-values
    for Lagrange interpolation.}
  \item{y}{numeric vector or matrix specifying the y-values for the
    Lagrange interpolation polynomial.  If \code{y} is a matrix,
    \code{nrow(y)} must equal \code{length(x)}, and each column of
    \code{y} is used separately with \code{x}.}
  \item{tol}{An absolute value tolerance, below which coefficients are
    treated as zero.}
  \item{lab}{If \code{y} is a matrix, lab is used as the names vector
    for the list result.}
}
\details{
  If \code{y} is a matrix, the result is a list of polynomials using
  each column separately.

  If \code{x} only is given, repeated zeros are allowed.  If \code{x}
  and \code{y} are given, repeated values in the \code{x} vector must
  have identical \code{y} values associated with them (up to
  \code{tol}), otherwise the first y-value only is used and a warning
  is issued.
}
\value{
  Either a polynomial object, or a list of polynomials, as appropriate.
  In the latter case the object is of class \code{"polylist"}.
}
\seealso{
  \code{\link{polynomial}}
}
\examples{
poly.calc(rep(1,3))
## -1 + 3*x - 3*x^2 + x^3 
poly.calc(0:4, (0:4)^2 + 1)
## 1 + x^2 
poly.calc(0:4, cbind(0:4, (0:4)^2 + 1), lab = letters[1:2])
## List of polynomials:
## $a:
## x
##
## $b:
## 1 + x^2 
}
\keyword{symbolmath}

\eof
\name{poly.orth}
\alias{poly.orth}
\title{Construct Orthogonal Polynomials}
\description{
  Construct the orthogonal polynomials on a given vector, up to a
  specified degree.
}
\usage{
poly.orth(x, degree = length(unique(x)) - 1, norm = TRUE)
}
\arguments{
  \item{x}{a numeric vector of abscissae.  When evaluated at \code{x}
    the polynomials will generate an orthonormal set.}
  \item{degree}{maximum degree requried.  The default is one fewer than
    the number of distinct values in \code{x}, which is maximum
    possible.}
  \item{norm}{a logical indicating whether the polynomials should be
    normalized.}
}
\value{
  A list of class \code{"polylist"} of objects of class
  \code{"polynomial"} of degree 1, 2, \dots, \code{degree}.
}
\examples{
x <- rep(1:4, 1:4)		# x with repetitions for weighting
x
## [1] 1 2 2 3 3 3 4 4 4 4
polx <- poly.orth(x, 3)		# calculate orthogonal polynomials
polx
## List of polynomials:
## [[1]]
## 0.3162278 
## 
## [[2]]
## -0.9486833 + 0.3162278*x 
## 
## [[3]]
## 2.139203 - 1.863177*x + 0.3450328*x^2 
## 
## [[4]]
## -5.831564 + 8.80369*x - 3.803194*x^2 + 0.4930066*x^3 
v <- sapply(polx, predict, x)   # orthonormal basis
round(crossprod(v), 10)	        # check orthonormality
}
\keyword{symbolmath}

\eof
\name{polylist}
\alias{polylist}
\alias{deriv.polylist}
\alias{integral.polylist}
\alias{plot.polylist}
\alias{print.polylist}
\title{Lists of Polynomials}
\description{Create and manipulate lists of polynomials.}
\usage{polylist(\dots)}
\arguments{
  \item{\dots}{a list of \R objects.}
}
\details{
  \code{polylist} takes a list of arguments, tries to convert each into
  a polynomial (see \code{\link{polynomial}}), and sets the class of the
  list to \code{"polylist"}.

  This class has several useful methods, such as taking derivatives
  (\code{link{deriv}}) and antiderivatives (\code{\link{integral}}),
  printing and plotting.
}
\examples{
## Calculate orthogonal polynomials
pl <- poly.orth(rep(1:4, 1:4), 3)
pl
plot(pl)
deriv(pl)
integral(pl)
}
\keyword{symbolmath}

\eof
\name{polynomial}
\alias{polynomial}
\alias{as.polynomial}
\alias{is.polynomial}
\alias{as.character.polynomial}
\alias{print.polynomial}
\alias{coef.polynomial}
\title{Polynomials}
\description{
  Construct, coerce to, test for, and print polynomial objects.
}
\usage{
polynomial(coef = c(0, 1))
as.polynomial(p)
is.polynomial(p)

\method{as.character}{polynomial}(x, decreasing = FALSE, \dots)
\method{print}{polynomial}(x, digits = getOption("digits"), decreasing = FALSE, \dots)
}
\arguments{
  \item{coef}{numeric vector, giving the polynomial coefficients in
    \emph{in}creasing order.}
  \item{p}{an arbitrary \R object.}
  \item{x}{a \code{polynomial} object.}
  \item{decreasing}{a logical specifying the order of the terms;
    in increasing (default) or decreasing powers.}
  \item{digits}{the number of significant digits to use for printing.}
  \item{\dots}{potentially further arguments passed to and from other methods.}
}
\details{
  \code{polynomial} constructs a polynomial from its coefficients,
  i.e., \code{p[1:k]} specifies the polynomial
  \deqn{p_1 + p_2 x + p_3 x^2 + \dots + p_k x^{k-1}.}{p[1] + p[2]* x +
    p[3]* x^2 + ... + p[k]* x^(k-1).}
  Internally, polynomials are simply numeric coefficient vectors of
  class \code{"polynomial"}.  Several useful methods are available for
  this class, such as coercion to character (\code{as.character()}) and
  function (\code{\link{as.function.polynomial}}), extraction of
  the coefficients (\code{coef()}), printing (using \code{as.character})
  and plotting (\code{\link{plot.polynomial}}).

  \code{as.polynomial} tries to coerce its arguments to a polynomial.

  \code{is.polynomial} tests whether its argument is a polynomial (in
  the sense that it has class \code{"polynomial"}.
}
\examples{
polynomial(1:4)
p <- as.polynomial(c(1,0,3,0))
p
print(p, decreasing = TRUE)
stopifnot(coef(p) == c(1,0,3))

polynomial(c(2,rep(0,10),1))
}
\keyword{symbolmath}

\eof
\name{predict.polynomial}
\alias{predict.polynomial}
\title{Evaluate a Polynomial}
\usage{
\method{predict}{polynomial}(object, newdata, \dots)
}
\arguments{
  \item{object}{A polynomial object to be evaluated.}
  \item{newdata}{Argument at which evaluation is requested.  May be
    numeric or itself a polynomial}
  \item{\dots}{Not used by this method.}
}
\description{
  Evaluate a polynomial at a given numeric or polynomial argument.
}
\details{
  This is a method for the generic function \code{\link{predict}}.

  The polynomial is evaluated according to the Horner scheme for speed
  and numerical accuracy.
}
\value{
  Evaluated object of the same class as \code{newdata}.
}
\seealso{
  \code{\link{as.function.polynomial}}
}
\keyword{symbolmath}

\eof
\name{solve.polynomial}
\alias{solve.polynomial}
\title{Zeros of a Polynomial}
\description{
  Find the zeros, if any, of a given polynomial.
}
\usage{
\method{solve}{polynomial}(a, b, \dots)
}
\arguments{
  \item{a}{A polynomial object for which the zeros are required.}
  \item{b}{a numeric value specifying an additional intercept.  If
    given, the zeros of \code{a - b} are found.}
  \item{\dots}{Not used by this method.}
}
\value{
  A numeric vector, generally complex, of zeros.
}
\details{
  This is a method for the generic function \code{\link{solve}}.
    
  The zeros are found as the eigenvalues of the companion matrix,
  sorted according to their real parts.
}
\seealso{
  \code{\link{polyroot}},
  \code{\link{poly.calc}},
  \code{\link{summary.polynomial}}
}
\examples{
p <- polynomial(6:1)
p
## 6 + 5*x + 4*x^2 + 3*x^3 + 2*x^4 + x^5 
pz <- solve(p)
pz
## [1] -1.49180+0.0000i -0.80579-1.2229i -0.80579+1.2229i
## [4]  0.55169-1.2533i  0.55169+1.2533i
## To retrieve the original polynomial from the zeros:
poly.calc(pz)
## Warning: imaginary parts discarded in coercion
## 6 + 5*x + 4*x^2 + 3*x^3 + 2*x^4 + x^5
}
\keyword{symbolmath}

\eof
\name{summary.polynomial}
\alias{summary.polynomial}
\alias{print.summary.polynomial}
\title{Summary of a Polynomial}
\description{
  Summarize a polynomial by describing its ``key'' points.
}
\usage{
\method{summary}{polynomial}(object, \dots)
}
\arguments{
  \item{object}{an object of class \code{"polynomial"}.}
  \item{\dots}{Not used by this method.}
}
\value{
  A list of class \code{"summary.polynomial"} (which has its own
  \code{print} method) containing information on zeros, stationary and
  inflexion points.
}
\details{
  This is a method for the generic function \code{\link{summary}}.
}
\examples{
p <- polynomial(6:1)
p
## 6 + 5*x + 4*x^2 + 3*x^3 + 2*x^4 + x^5 
pz <- summary(p)
pz
## [1] -1.49180+0.0000i -0.80579-1.2229i -0.80579+1.2229i
## [4]  0.55169-1.2533i  0.55169+1.2533i
## To retrieve the original polynomial from the zeros:
poly.calc(pz)
## Warning: imaginary parts discarded in coercion
## 6 + 5*x + 4*x^2 + 3*x^3 + 2*x^4 + x^5
}
\keyword{symbolmath}

\eof
\name{zSummary.polynomial}
\alias{Summary.polynomial}
\title{Summary Group Methods for Polynomials}
\description{
  Allows summary functions to be used on polynomial arguments.
}
\usage{
Summary.polynomial(x, \dots)
}
\arguments{
  \item{x}{an object of class \code{"polynomial"}.}
  \item{\dots}{optional additional arguments.}
}
\details{
  Summary functions on polynomials do not make very much sense, so an
  error is returned.
}
\seealso{
  \code{\link{Math.polynomial}},
  \code{\link{Ops.polynomial}}
}
\keyword{symbolmath}

\eof
