\name{NNclean}
\alias{NNclean}
\alias{print.nnclean}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Nearest neighbor based clutter/noise detection}
\description{
  Detects if data points are noise or part of a cluster,
  based on a Poisson process model.
}
\usage{
NNclean(data, k, distances = NULL, edge.correct = FALSE, wrap = 0.1,
convergence = 0.001, plot=FALSE)

\method{print}{nnclean}(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{data}{numerical matrix or data frame.}
  \item{k}{integer. Nmber of considered nearest neighbors per point.}
  \item{distances}{distance matrix object of class \code{dist}. If
    specified, it is used instead of computing distances from the data.}
  \item{edge.correct}{logical. If \code{TRUE} and the data is
    two-dimensional, neighbors for points at the edges of the parent
    region of the noise Poisson process are determined after wrapping
    the region onto a toroid.}
  \item{wrap}{numerical. If \code{edge.correct=TRUE}, points in a
    strip of size \code{wrap*range} along the edge for each variable
    are candidates for
    being neighbors of points from the opposite.}
  \item{convergence}{numerical. Convergence criterion for EM-algorithm.}
  \item{plot}{logical. If \code{TRUE}, a histogram of the distance to
    kth nearest neighbor and fit is plotted.}
  \item{x}{object of class \code{nnclean}.}
  \item{...}{necessary for print methods.}
}
\details{
  The assumption is that the noise is distributed as a homogeneous
  Poisson process  on a certain region and the clusters are distributed
  as a homogeneous Poisson process with larger intensity on a
  subregion (disconnected in case of more than one cluster).
  The distances are then distributed according to a mixture of two
  transformed Gamma distributions, and this mixture is estimated via the
  EM-algorithm. The points are assigned to noise or cluster component
  by use of the estimated a posteriori probabilities. 
}
\value{
  \code{NNclean} returns a list of class \code{nnclean} with components
  \item{z}{0-1-vector of lenght of the number of data points. 1 means
    cluster, 0 means noise.}
  \item{probs}{vector of estimated a priori probabilities for each point
    to belong to the cluster component.}
  \item{k}{see above.}
  \item{lambda1}{intensity parameter of cluster component.}
  \item{lambda2}{intensity parameter of noise component.}
  \item{p}{estimated probability of cluster component.}
  \item{kthNND}{distance to kth nearest neighbor.}
}
\references{
Byers, S. and Raftery, A. E. (1998) Nearest-Neighbor Clutter
Removal for Estimating Features in Spatial Point Processes,
\emph{Journal of the American Statistical Association}, 93, 577-584.
}
\author{R-port by Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/},\cr
  original Splus package by S. Byers and A. E. Raftery.
}

\note{The software can be freely used for non-commercial purposes, and can
be freely distributed for non-commercial purposes only.} 

\examples{
library(mclust)
data(chevron)
nnc <-  NNclean(chevron[,2:3],15,plot=TRUE)
plot(chevron[,2:3],col=1+nnc$z)
}

\keyword{multivariate}% at least one, from doc/KEYWORDS
\keyword{cluster}% __ONLY ONE__ keyword per line
\eof
\name{autoconst}
\alias{autoconst}
\alias{autoreg}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Spatial autocorrelation parameter estimation}
\description{
  Monte Carlo estimation of the disjunction/spatial autocorrelation
  parameter \code{pd} for the simulation model used in
  \code{randpop.nb}, used for tests for clustering of presence-absence data.

  \code{autoconst} is the main function; \code{autoreg} performs the
  simulation and is executed within \code{autoconst}.
}
\usage{
autoconst(x, prange = c(0, 1), twostep = TRUE, step1 = 0.1,
step2 = 0.01, plot = TRUE, nperp = 4, ejprob = NA,
species.fixed = TRUE, pdfnb=FALSE)

autoreg(x, probs, ejprob, plot = TRUE, nperp = 4, species.fixed = TRUE,
pdfnb=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{object of class \code{prab} as generated by \code{prabinit}.
    Presence-absence data to be analyzed.}
  \item{prange}{numerical range vector, lower value not smaller than 0, larger
    value not larger than 1. Range where the parameter is to be found.}
  \item{twostep}{logical. If \code{TRUE}, a first estimation step is
    carried out in the whole \code{prange}, and then the final
    estimation is determined between the preliminary estimator
    \code{-5*step2} and {+5*step2}. Else, the first simulation
    determines the final estimator.}
  \item{step1}{numerical between 0 and 1. Interval length between
    subsequent choices of \code{pd} for the first simulation.}
  \item{step2}{numerical between 0 and 1. Interval length between
    subsequent choices of \code{pd} for the second simulation in case of
    \code{twostep=TRUE}.}
  \item{plot}{logical. If \code{TRUE}, a scatterplot of \code{pd}-values
    against resulting \code{ejprob} values (see below), with regression
    line and data value of \code{ejprob} is shown.}
  \item{nperp}{integer. Number of simulations per \code{pd}-value.}
  \item{ejprob}{numerical between 0 and 1. Observed disjunction
    probability for data \code{x}; if not specified in advance,
    it will be computed by \code{autoconst}.}
  \item{species.fixed}{logical. If \code{TRUE}, sizes of generated
    species match the species sizes in \code{x}, else they are generated
    from the empirical distribution of species sizes in \code{x}.}
  \item{probs}{vector of numericals between 0 and 1. \code{pd} values
    for the simulation.}
  \item{pdfnb}{logical. If \code{TRUE}, the probabilities of the regions
    are modified according to the number of neighboring regions in
    \code{randpop.nb}, see Hennig and Hausdorf (2002), p. 5.}
}
\details{
  The spatial autocorrelation parameter \code{pd}
  of the model for the generation of
  presence-absence data sets used by \code{randpop.nb} can be estimated
  by use of the observed disjuction probability \code{ejprob} which is
  the sum of
  all species' connectivity components minus the number of species
  divided by the number of "presence" entries minus the number of
  species. This is done by a simulation of artificial data sets with
  characteristics of \code{x} and different \code{pd}-values, governed
  by \code{prange, step1, step2} and \code{nperp}. \code{ejprob} is then
  calculated for all simulated populations. A linear regression of
  \code{ejprob} on \code{pd} is performed and the estimator of \code{pd}
  is determined by computing the inverse of the regression function for
  the \code{ejprob}-value of \code{x}. 
}
\value{
  \code{autoconst} produces the same list as \code{autoreg} with
  additional component \code{ejprob}. The components are 
  \item{pd}{(eventually) estimated parameter \code{pd}.}
  \item{coef}{(eventually) estimated regression coefficients.}
  \item{ejprob}{see above.}
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.

Hausdorf, B. and Hennig, C. (2003)  Biotic Element Analysis in
Biogeography. To appear in  \emph{Systematic Biology}.

Hausdorf, B. and Hennig, C. (2003) Nestedness of nerth-west European
land snail ranges as a consequence of differential immigration from
Pleistocene glacial refuges. \emph{Oecologia} 135, 102-109.

}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{randpop.nb}}, \code{\link{prabinit}}, \code{\link{con.comp}} 
}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(1234)
x <- prabinit(prabmatrix=kykladspecreg, neighborhood=nb)
ax <- autoconst(x,nperp=2)
}
\keyword{spatial}% at least one, from doc/KEYWORDS

\eof
\name{cluspop.nb}
\alias{cluspop.nb}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Simulation of presence-absence matrices (clustered)}
\description{
  Generates a simulated matrix where the rows are interpreted as regions
  and the columns as species, 1 means that a species is present in the
  region and 0 means that the species is absent. Species are generated
  in order to produce 2 clusters of species with similar ranges.
  Spatial autocorrelation of a species' presences is governed by
  the parameter \code{p.nb} and a list of neighbors for each region.
}
\usage{
cluspop.nb(neighbors, p.nb = 0.5, n.species, clus.specs, reg.group,
grouppf = 10, n.regions = length(neighbors),
vector.species = rep(1, n.species), pdf.regions = rep(1/n.regions,
n.regions), count = TRUE, pdfnb = FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{neighbors}{A list with a component for every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a list \code{numeric(0)}.}
  \item{p.nb}{numerical between 0 and 1. The probability that a new
    region is drawn from the non-neighborhood of the previous regions
    belonging to a species under generation. Note that for a given
    presence-absence matrix, this parameter can be estimated by
    \code{autoconst} (called \code{pd} there).}
  \item{n.species}{integer. Number of species.}
  \item{clus.specs}{integer not larger than \code{n.species}. Number of
    species restricted to one of the two groups of regions defined by
    \code{reg.group} (called "clustered species" because this leads to
    more similar species ranges).}
  \item{reg.group}{vector of pairwise distinct integers not larger than
    \code{n. regions}. Defines a group of regions to which a part of the
    \code{clus.specs} clustered species is restricted (more or less, see
    \code{grouppf}). The other clustered species are restricted to the
    complementary regions.}
  \item{grouppf}{numerical. The probability of the region of
    a clustered species to belong to the corresponding group of regions
    is up-weighted by factor \code{grouppf} compared to the generation
    of "non-clustered" species.}
  \item{n.regions}{integer. Number of regions.}
  \item{vector.species}{vector of integers. The sizes
    (i.e., numbers of regions)
    of the species are generated randomly from
    the empirical distribution of \code{vector.species}.}
  \item{pdf.regions}{numerical vector of length \code{n.species}. The
    entries must sum up to 1 and give probabilities for the regions to
    be drawn during the generation of a species. These probabilities are
    used conditional on the new region being a neighbor or a
    non-neighbor of the previous regions of the species, see
    \code{p.nb}, modified by \code{grouppf} for the clustered species.}
  \item{count}{logical. If \code{TRUE}, the number of the currently
    generated species is printed.}
  \item{pdfnb}{logical. If \code{TRUE}, the probabilities of the regions
    are modified according to the number of neighboring regions by
    dividing them relative to the others by min(1,number of neighbors).}
}
\details{
  The non-clustered species are generated as explained on the help page
  for \code{randpop.nb}. The general principle for the clustered species
  is the same, but with modified probabilities for the regions. For each
  clustered species, one of the two groups of regions is drawn,
  distributed according to the sum of its regions' probability given by
  \code{pdf.regions}. The first region of such a species is only drawn
  from the regions of this group.
}
\value{
  A 0-1-matrix, rows are regions, columns are species.
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.
}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{randpop.nb}},
  
  \code{\link{autoconst}} estimates \code{p.nb} from matrices of class
  \code{prab}. These are generated by \code{\link{prabinit}}.
}

\examples{
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(888)
cluspop.nb(nb, p.nb=0.1, n.species=10, clus.specs=9, reg.group=1:17,
vector.species=c(10))
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\eof
\name{comp.test}
\alias{comp.test}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Compare species clustering and species groups}
\description{
  Tests for independence between a clustering and another grouping of species.
  This is simply an interface to \code{chisq.test}.
}

\usage{
comp.test(cl,spg)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{cl}{a vector of integers. Clustering of species (may be taken
    from \code{prabclust}).}
  \item{spg}{a vector of integers of the same length, groups of species.}
}
\details{
  \code{chisq.test} with simulated p-value is used.
}
\value{
  Output of \code{chisq.test}.
}

\references{
Hausdorf, B. and Hennig, C. (2003)  Biotic Element Analysis in
Biogeography. To appear in  \emph{Systematic Biology}.
}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{chisq.test}}, \code{\link{prabclust}}.
}


\examples{
set.seed(1234)
g1 <- c(rep(1,34),rep(2,12),rep(3,15))
g2 <- sample(3,61,replace=TRUE)
comp.test(g1,g2)
}
\keyword{htest}% at least one, from doc/KEYWORDS

\eof
\name{con.comp}
\alias{con.comp}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Connectivity components of an undirected graph}
\description{
  Computes the connectivity components of an undirected graph from a
  matrix giving the edges.
}
\usage{
con.comp(comat)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{comat}{a symmetric logical or 0-1 matrix, where \code{comat[i,j]=TRUE}
    means that there is an edge between vertices \code{i} and
    \code{j}. The diagonal is ignored.}
}
\details{
  The "depth-first search" algorithm of Cormen, Leiserson and Rivest
  (1990, p. 477) is used.
}
\value{
  An integer vector, giving the number of the connectivity component for
  each vertice.
}
\references{
  Cormen, T. H., Leiserson, C. E. and Rivest, R. L. (1990), \emph{Introduction
    to Algorithms}, Cambridge: MIT Press.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}
}

\seealso{
  \code{\link{hclust}}, \code{\link{cutree}} for cutted single linkage
  trees (often equivalent).
}

\examples{
  set.seed(1000)
  x <- rnorm(20)
  m <- matrix(0,nrow=20,ncol=20)
  for(i in 1:20)
    for(j in 1:20)
      m[i,j] <- abs(x[i]-x[j])
  d <- m<0.2
  cc <- con.comp(d)
  max(cc) # number of connectivity components
  plot(x,cc)
  # The same should be produced by
  # cutree(hclust(as.dist(m),method="single"),h=0.2).
}
\keyword{array}% at least one, from doc/KEYWORDS
\keyword{cluster}% __ONLY ONE__ keyword per line


\eof
\name{con.regmat}
\alias{con.regmat}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Connected regions per species}
\description{
  Returns a vector of the numbers of connected regions per species for a
  presence-absence matrix.   
}
\usage{
con.regmat(regmat, neighbors, count = TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{regmat}{0-1-matrix. Columns are species, rows are regions.}
  \item{neighbors}{A list with a component for every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a list \code{numeric(0)}.}
  \item{count}{logical. If \code{TRUE}, the number of the currently
    processed species is printed.}
}
\details{
  Uses \code{con.comp}.
}
\value{
  Vector of numbers of connected regions per species.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\note{Designed for use in \code{prabtest}.}

\seealso{\code{\link{con.comp}}, \code{\link{prabtest}}}

\examples{
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(888) 
cp <- cluspop.nb(nb, p.nb=0.1, n.species=10, clus.specs=9,
                 reg.group=1:17,vector.species=c(10))
con.regmat(cp,nb)
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{cluster}% __ONLY ONE__ keyword per line
\eof
\name{distratio}
\alias{distratio}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Distance ratio test statistics for distance based clustering}
\description{
  Calculates the ratio between the \code{prop} smallest and largest
  distances of a distance matrix.
}
\usage{
distratio(distmat, prop = 0.25)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{distmat}{symmetric distance matrix.}
  \item{prop}{numerical. Proportion between 0 and 1.}
}

\details{
  Rounding is by \code{floor} for small and \code{ceiling} for large
  distances.
}

\value{
  A list with components
  \item{dr}{ratio of \code{prop} smallest to \code{prop} largest distances.}
  \item{lowmean}{mean of \code{prop} smallest distances.}
  \item{himean}{mean of \code{prop} smallest distances.}
  \item{prop}{see above.}
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabtest}}}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
j <- jaccard(t(kykladspecreg))
distratio(j)
}
\keyword{cluster}% at least one, from doc/KEYWORDS
% \keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{homogen.test}
\alias{homogen.test}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Classical distance-based test for homogeneity against clustering}
\description{
  Classical distance-based test for homogeneity against clustering. Test
  statistics is number of isolated vertices in the graph of smallest
  distances. The homogeneity model is a random graph model where \code{ne}
  edges are drawn from all possible edges.
}
\usage{
homogen.test(distmat, ne = ncol(distmat), testdist = "erdos")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{distmat}{numeric symmetric distance matrix.}
  \item{ne}{integer. Number of edges in the data graph, corresponding to
    smallest distances.}
  \item{testdist}{string. If \code{testdist="erdos"}, the test
    distribution is a Poisson asymptotic distibution as given by Erdos
    and Renyi (1960). If \code{testdist="ling"}, the test distribution
    is exact as given by Ling (1973), which needs much more computing time.}
}

\details{
  The "ling"-test is one-sided (rejection if the number of isolated vertices
  is too large), the "erdos"-test computes a one-sided as well as a
  two-sided p-value.
}

\value{
  A list with components
  \item{p}{p-value for one-sided test.}
  \item{p.twoside}{p-value for two-sided test, only if \code{testdist="erdos"}.}
  \item{iv}{number of isolated vertices in the data.}
  \item{lambda}{parameter of the Poisson test distribution, only if
    \code{testdist="erdos"}.}
  \item{distcut}{largest distance value for which an edge has been
    drawn.}
  \item{ne}{see above.}
}
\references{
  Erdos, P. and Renyi, A. (1960) On the evolution of random
  graphs. \emph{Publications of the Mathematical Institute of the
  Hungarian Academy of Sciences} 5, 17-61.

  Godehardt, E. and Horsch, A. (1995) Graph-Theoretic Models for Testing
  the Homogeneity of Data. In Gaul, W. and Pfeifer, D. (Eds.) \emph{From
  Data to Knowledge}, Springer, Berlin, 167-176.
  
  Ling, R. F. (1973) A probability theory of cluster
  analysis. \emph{Journal of the American Statistical Association} 68,
  159-164.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabtest}}}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
j <- jaccard(t(kykladspecreg))
homogen.test(j, testdist="erdos")
homogen.test(j, testdist="ling")
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{htest}% __ONLY ONE__ keyword per line







\eof
\name{incmatrix}
\alias{incmatrix}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Nestedness matrix}
\description{
  Computes species*species nestedness matrix and number of nestings
  (inclusions) from regions*species presence-absence matrix.
}
\usage{
incmatrix(regmat)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{regmat}{0-1-matrix. Columns are species, rows are regions.}
}

\value{
  A list with components
  \item{m}{0-1-matrix. \code{m[i,j]=1} means that the occupied region of
  species j is a subset (or equal) of the region of species i.}
  \item{ninc}{integer. Number of strict inclusions.}
  \item{neq}{integer. Number of region equalities between species (not
    including equality between species i and i).}
}
\references{
Hausdorf, B. and Hennig, C. (2003) Nestedness of nerth-west European
land snail ranges as a consequence of differential immigration from
Pleistocene glacial refuges. \emph{Oecologia} 135, 102-109.

}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{prabtest}}
}


\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
incmatrix(t(kykladspecreg))$ninc
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{array}

\eof
\name{jaccard}
\alias{jaccard}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Jaccard distance matrix}
\description{
  Computes Jaccard distances between the columns of a 0-1-matrix.
}
\usage{
jaccard(regmat)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{regmat}{0-1-matrix. Columns are species, rows are regions.}
}
\details{
  The Jaccard distance between two species is 1-(number of regions where
  both species are present)/(number of regions where at least one
  species is present).
}
\value{
  A symmetrical matrix of Jaccard distances.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{kulczynski}}
}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
jaccard(t(kykladspecreg))
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{spatial}% __ONLY ONE__ keyword per line
\eof
\name{kulczynski}
\alias{kulczynski}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Kulczynski distance matrix}
\description{
  Computes Kulczynski distances between the columns of a 0-1-matrix.
}
\usage{
kulczynski(regmat)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{regmat}{0-1-matrix. Columns are species, rows are regions.}
}
\details{
  The Kulczynski distance between two species is 1-(mean of (number of
  regions where
  both species are present)/(number of regions where species 1 is
  present) and (number of
  regions where
  both species are present)/(number of regions where species 2 is
  present)).
}
\value{
  A symmetrical matrix of Kulczynski distances.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{jaccard}}
}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
kulczynski(t(kykladspecreg))
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{spatial}% __ONLY ONE__ keyword per line

\eof
\name{kykladspecreg}
\alias{kykladspecreg}
% \non_function{}
\title{Snail presence-absence data from Aegean sea}
\description{
  0-1-matrix where rows are snail species and columns are islands in the
  Aegean sea. An entry of 1 means that the species is present in the
  region.     
}
\usage{data(kykladspecreg)}
\format{
  A 0-1 matrix with 80 rows and 34 columns.}
\source{
  Hausdorf, B. and Hennig, C. (2002) Biogeographical tests of the
  Importance of Vicariance in the Diversification of Mediterranean Land
  Snails. Submitted. 
}
\details{
  Reads from example data file \code{kykladspecreg.dat}.
  }

\seealso{
  \code{\link{nb}} provides neighborhood information about the 34
  islands.
  }
\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
}
\keyword{datasets}
% \keyword{spatial}

\eof
\name{lcomponent}
\alias{lcomponent}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Largest connectivity component}
\description{
  Computes the size of the largest connectivity component of the graph
  of \code{ncol(distmat)} vertices with edges defined by the smallest
  \code{ne} distances.
}
\usage{
lcomponent(distmat, ne = floor(3*ncol(distmat)/4))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{distmat}{symmetric distance matrix.}
  \item{ne}{integer.}
}

\value{
  list with components
  \item{lc}{size of the largest connectivity component.}
  \item{ne}{see above.}
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabtest}}}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
j <- jaccard(t(kykladspecreg))
lcomponent(j)
}
\keyword{cluster}% at least one, from doc/KEYWORDS




\eof
\name{nb}
\alias{nb}
% \non_function{}
\title{Neighborhood list for Aegean islands}
\description{
  List of neighboring islands for 34 Aegean islands.
}
\usage{data(nb)}
\format{
  List with 34 components, all being vetors of integers (or
  \code{numeric(0)} in case of no neighbors) indicating the neighboring
  islands.
}
\details{
  Reads from example data file \code{kyklneigh.txt}.
}

\source{
  Hausdorf, B. and Hennig, C. (2002) Biogeographical tests of the
  Importance of Vicariance in the Diversification of Mediterranean Land
  Snails. Submitted. 
  
}
\examples{
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
}
\keyword{datasets}

\eof
\name{nbtest}
\alias{nbtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Test of neighborhood list}
\description{
  Tests a list of neighboring regions for proper format. Neighborhood is
  tested for being symmetrical. Causes an error if tests fail.
}
\usage{
nbtest(nblist, n.regions=length(nblist))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{nblist}{A list with a component for
    every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a vector \code{numeric(0)}.}
  \item{n.regions}{Number of regions.}
}

\value{
  \code{invisible{TRUE}}.
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabinit}}.}

\examples{
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
nbtest(nb)
nb[[1]][1] <- 1
try(nbtest(nb))
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\eof
\name{nn}
\alias{nn}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Mean distance to kth nearest neighbor}
\description{
  Computes the mean of the distances from each point to its \code{ne}th
  nearest neighbor.
}
\usage{
nn(distmat, ne = 1)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{distmat}{symmetric distance matrix.}
  \item{ne}{integer.}
}
\value{
  numerical.
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabtest}}}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
j <- jaccard(t(kykladspecreg))
nn(j,4)
}
\keyword{cluster}% at least one, from doc/KEYWORDS

\eof
\name{pop.sim}
\alias{pop.sim}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{p-value simulation for presence-absence matrices clustering test}
\description{
  Parametric bootstrap simulation of the p-value of a test of a
  homogeneity hypothesis against clustering (or significant nestedness).
  Designed for use within
  \code{prabtest}. The null model is defined by \code{randpop.nb}.
}
\usage{
pop.sim(regmat, neighbors, h0c = 1, times = 200, dist = "kulczynski",
teststat = "isovertice", testc = NA, n.species = ncol(regmat),
specperreg = NA, regperspec = NA, species.fixed=FALSE, pdfnb=pdfnb)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{regmat}{0-1-matrix. Columns are species, rows are regions.}
  \item{neighbors}{A list with a component for every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a list \code{numeric(0)}.}
  \item{h0c}{numerical. Parameter \code{p.nb} for use in \code{randpop.nb}.}
  \item{times}{integer. Number of simulation runs.}
  \item{dist}{"kulczynski" or "jaccard", see \code{kulczynski} and
    \code{jaccard}.}
  \item{teststat}{"isovertice", "lcomponent", "distratio", "nn" or
    "inclusions". See
    the corresponding functions, \code{homogen.test} for "isovertice",
    \code{incmatrix} for "inclusions").}
  \item{testc}{numerical. Tuning constant for the test statistics.}
  \item{n.species}{integer. Number of species.}
  \item{specperreg}{vector of integers. Numbers of species per region
    (is calculated from the data by default).}
  \item{regperspec}{vector of integers. Number of regions per species
    (is calculated from the data by default).}
  \item{species.fixed}{logical. If \code{TRUE}, the sizes of the species
    are taken directly from \code{regmat}. Otherwise, they are drawn by
    random from the empirical distribution of the values from
    \code{regmat}.}
  \item{pdfnb}{logical. Probability correction in \code{randpop.nb}.}
}
\value{
  List with components
  \item{results}{vector of teststatistic values for the simulated matrices.}
  \item{p.above}{p-value if large test statistic leads to rejection.}
  \item{p.below}{p-value if small test statistic leads to rejection.}
  \item{datac}{test statistic value for the original data.}
  \item{testc}{see above.}
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.

Hausdorf, B. and Hennig, C. (2003)  Biotic Element Analysis in
Biogeography. To appear in  \emph{Systematic Biology}.

Hausdorf, B. and Hennig, C. (2003) Nestedness of nerth-west European
land snail ranges as a consequence of differential immigration from
Pleistocene glacial refuges. \emph{Oecologia} 135, 102-109.
}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}


\seealso{
  \code{\link{prabtest}}, \code{\link{randpop.nb}},
  \code{\link{jaccard}}, \code{\link{kulczynski}},
  \code{\link{homogen.test}}, \code{\link{lcomponent}},
  \code{\link{distratio}}, \code{\link{nn}},
  \code{\link{incmatrix}}.
}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(1234)
pop.sim(t(kykladspecreg), nb, times=20, h0c=0.35, teststat="nn", testc=3)
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{htest}% __ONLY ONE__ keyword per line
\eof
\name{prabclust}
\alias{prabclust}
\alias{print.prabclust}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Clustering of species ranges from presence-absence matrices}
\description{
  Clusters a presence-absence matrix object by calculating an MDS from
  the distances, and applying maximum likelihood Gaussian mixtures clustering
  with "noise" (package \code{mclust}) to the MDS points. The solution
  is plotted. A standard execution will be \cr
  \code{prabmatrix <- prabinit(file="path/prabmatrixfile",
    neighborhood="path/neighborhoodfile")}\cr
  \code{clust <- prabclust(prabmatrix)}\cr
  \code{print(clust)} \cr
  \bold{Note:} Data formats are described
  on the \code{prabinit} help page. You may also consider the example datasets
  \code{kykladspecreg.dat} and \code{nb.dat}. Take care of the
  parameter \code{rows.are.species} of \code{prabinit}.
}

\usage{
prabclust(prabobj, mdsmethod = "classical", mdsdim = 4, nnk =
ceiling(prabobj$n.species/40), nclus = 0:9, modelid = "noVVV")

\method{print}{prabclust}(x, bic=FALSE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{prabobj}{object of class \code{prab} as
    generated by \code{prabinit}. Presence-absence data to be analyzed.
  }
  \item{mdsmethod}{\code{"classical"}, \code{"kruskal"}, or
    \code{"sammon"}. The MDS method
    to transform the distances to data points. \code{"classical"} indicates
    metric MDS by function \code{cmdscale}, \code{"kruskal"} is
    non-metric MDS.}
  \item{mdsdim}{integer. Dimension of the MDS points.}
  \item{nnk}{integer. Number of nearest neighbors to determine the
    initial noise estimation by \code{NNclean}.}
  \item{nclus}{vector of integers. Numbers of clusters to perform the
    mixture estimation.}
  \item{modelid}{string. Model name for \code{EMclustN} (see the
    corresponding help page). Additionally, \code{"noVVV"} is possible, which
    fits all methods except \code{"VVV"}.}
  \item{x}{object of class \code{prabclust}. Output of
    \code{prabclust}.}
  \item{bic}{logical. If \code{TRUE}, information about the BIC
    criterion to choose the model is displayed.}
  \item{...}{necessary for summary method.}
}

\value{
  \code{print.prabclust} does not produce output. 
  \code{prabclust} generates an object of class \code{prabclust}. This is a
  list with components
  \item{clustering}{vector of integers indicating the cluster memberships of
    the species. Noise can be recognized by output component \code{symbols}.}
  \item{clustsummary}{output object of \code{summary.EMclustN}. A list
    giving the optimal (according to BIC) parameters, 
    conditional probabilities `z', and loglikelihood, together with
    the associated classification and its uncertainty.}
  \item{bicsummary}{output object of \code{EMclustN}. Bayesian Information
    Criterion for the specified mixture models and numbers of clusters. }
  \item{points}{numerical matrix. MDS configuration.}
  \item{nnk}{see above.}
  \item{mdsdim}{see above.}
  \item{mdsmethod}{see above.}
  \item{symbols}{vector of characters, similar to \code{clustering}, but
    indicating estimated noise and points belonging to
    one-point-components (which should be interpreted as some kind of
    noise as well) by \code{"N"}. } 
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.
}
\note{
  Note that we used \code{mdsmethod="kruskal"} in our publications, but
  we prefer the new default \code{mdsmethod="classical"} now, because we
  discovered some numerical instabilities of the
  \code{isoMDS}-implementation in connection with our distance matrices.
  }
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link[mclust]{EMclustN}}, \code{\link[mclust]{summary.EMclustN}},
  \code{\link{NNclean}}, \code{\link{cmdscale}},
  \code{\link{isoMDS}}, \code{\link{sammon}},
  \code{\link{prabinit}}.
}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(1234)
x <- prabinit(prabmatrix=kykladspecreg, neighborhood=nb)
# If you want to use your own ASCII data files, use
# x <- prabinit(file="path/prabmatrixfile",
# neighborhood="path/neighborhoodfile")
print(prabclust(x))
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{spatial}% __ONLY ONE__ keyword per line
\eof
\name{prabinit}
\alias{prabinit}
\alias{print.prab}
\alias{prab}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Presence-absence matrix initialization}
\description{
  \code{prabinit} converts a matrix into an object
  of class \code{prab} (presence-absence). The matrix may be read from a
  file or an R-object. It may be a 0-1 matrix or a matrix with
  non-negative entries from which the 0-1 matrix is to be generated.
  \code{print.prab} is a print method for such
  objects. 
}
\usage{
prabinit(file = NA, prabmatrix = NA, rows.are.species = TRUE,
neighborhood = "none", distance = "kulczynski", toprab = FALSE, toprabp
= 0.05, outc = 5.2)

\method{print}{prab}(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{file}{string. 0-1 matrix ASCII file (such as example dataset
    \code{kykladspecreg.dat}) from which the
    matrix is read by \code{read.table}.}
  \item{prabmatrix}{matrix with non-negative entries. Either \code{file}
    or \code{prabmatrix} should be \code{NA}.}
  \item{rows.are.species}{logical. If \code{TRUE}, rows are interpreted
    as species and columns are interpreted as regions. In this case,
    rows and columns are interchanged by \code{prabinit}.}
  \item{neighborhood}{A string or a list with a component for
    every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a vector \code{numeric(0)}. If \code{neighborhood}
    is a filename, it is attempted to read such a list from a file,
    where every row should correspond to one region (such as example
    dataset \code{nb.dat}). If
    \code{neighborhood="none"}, all neighborhoods are set to
    \code{numeric(0)}. The neighborhood is tested by \code{nbtest}.}
  \item{distance}{\code{"kulczinsky"}, \code{"jaccard"} or
    \code{"none"}. The distance measure
    between species to compute by \code{prabinit}.}
  \item{toprab}{logical. If \code{TRUE}, a presence-absence matrix is
    computed from the non-negative input matrix. "Absence", i.e., the
    entry 0, is chosen if the original entry is 0, or the original entry
    is both smaller than \code{toprabp} times the sum of entries in the
    corresponding region and log(original entry) is considered to be a
    lower outlier compared with the other entries of the corresponding
    species (see \code{outc}). "Presence", i.e., the entry 1, thus
    means that the original
    entry is non-negligible w.r.t. the species or w.r.t. the region.}
  \item{toprabp}{numerical between 0 and 1, see \code{toprab}.}
  \item{outc}{numerical. Tuning constant for the outlier identification
    associated with \code{toprab=TRUE}. An entry smaller than
    \code{outc*mad} times the median is considered as a lower outlier.}
  \item{x}{object of class \code{prab}.}
  \item{...}{necessary for print method.}
}

\value{
  An object of class \code{prab}, which is a list with components
  \item{distmat}{distance matrix between species.}
  \item{prab}{presence-absence matrix. Rows are regions, columns are
    species.}
  \item{nb}{neighborhood list, see above.}
  \item{regperspec}{vector of the number of regions occupied by a
    species.}
  \item{specperreg}{vector of the number of species present in a region.}
  \item{n.species}{number of species.}
  \item{n.regions}{number of regions.}
  \item{distance}{string denoting the chosen distance measure.}
  \item{spatial}{\code{TRUE}, if there is a specified neighborhood
    structure.}  
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{read.table}}, \code{\link{jaccard}},
\code{\link{kulczynski}}, \code{\link{nbtest}}}

\examples{
# If you want to use your own ASCII data files, use
# x <- prabinit(file="path/prabmatrixfile",
# neighborhood="path/neighborhoodfile")
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
prabinit(prabmatrix=kykladspecreg, neighborhood=nb)
}
\keyword{spatial}% at least one, from doc/KEYWORDS
\keyword{cluster}% __ONLY ONE__ keyword per line
\eof
\name{prabtest}
\alias{prabtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Parametric bootstrap test for clustering in presence-absence matrices}
\description{
  Parametric bootstrap test of a null model of i.i.d., but spatially
  autocorrelated species against clustering of the species' occupied
  areas (or alternatively nestedness). In spite of the lots of
  parameters, a standard execution will be \cr
  \code{prabmatrix <- prabinit(file="path/prabmatrixfile",
    neighborhood="path/neighborhoodfile")}\cr
  \code{test <- prabtest(prabmatrix)}\cr
  \code{summary(test)}\cr
  \bold{Note:} Data formats are described
  on the \code{prabinit} help page. You may also consider the example datasets
  \code{kykladspecreg.dat} and \code{nb.dat}. Take care of the
  parameter \code{rows.are.species} of \code{prabinit}.}


\usage{
prabtest(x, teststat = "distratio", tuning=switch(teststat,distratio=0.25,
lcomponent=floor(3*ncol(x$distmat)/4),
isovertice=ncol(x$distmat),nn=4,NA), times = 1000, pd = NULL,
prange = c(0, 1), nperp = 4, step = 0.1, twostep = TRUE, sf.sim = FALSE,
sf.const = sf.sim, pdfnb=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{an object of class \code{prab} (presence-absence data), as
    generated by \code{prabinit}.}
  \item{teststat}{string, indicating the test statistics. \code{"isovertice"}:
    number of isolated vertices in the graph of \code{tuning}
    smallest distances
    between species. \code{"lcomponent"}: size of largest connectivity
    component in this graph. \code{"distratio"}: ratio between \code{tuning}
    smallest and largest distances. \code{"nn"}: average distance of species to
    \code{tuning}th nearest neighbor. 
    \code{"inclusions"}: number of inclusions between areas of different
    species (tests for nestedness structure, not for clustering).}
  \item{tuning}{integer or (if \code{teststat="distratio"}) numerical
    between 0 and 1. Tuning constant for test statistics, see
    \code{teststat}.}
  \item{times}{integer. Number of simulation runs.}
  \item{pd}{numerical between 0 and 1. The probability that a new
    region is drawn from the non-neighborhood of the previous regions
    belonging to a species under generation. If \code{NA} (the default),
    \code{prabtest} estimates this by function
    \code{autoconst}. Otherwise the next four parameters have no effect.}
  \item{prange}{numerical range vector, lower value not smaller than 0, larger
    value not larger than 1. Range where \code{pd} is to be found. Used
    by function \code{autoconst}.}
  \item{nperp}{integer. Number of simulations per \code{pd}-value. Used
    by function \code{autoconst}.}
  \item{step}{numerical between 0 and 1. Interval length between
    subsequent choices of \code{pd} for the first simulation. Used
    by function \code{autoconst}.}
  \item{twostep}{logical. If \code{TRUE}, a first estimation step for
    \code{pd} is
    carried out in the whole \code{prange}, and then the final
    estimation is determined between the preliminary estimator
    \code{-5*step2} and {+5*step2}. Else, the first simulation
    determines the final estimator. Used
    by function \code{autoconst}.}
  \item{sf.sim}{logical. Indicates if the range sizes of the species
    are held fixed
    in the test simulation (\code{TRUE}) or generated from their empirical
    distribution in \code{x} (\code{FALSE}). See function \code{randpop.nb}.}
  \item{sf.const}{logical. Same as \code{sf.sim}, but for estimation of
    \code{pd} by \code{autoconst}.}
  \item{pdfnb}{logical. If \code{TRUE}, the probabilities of the regions
    are modified according to the number of neighboring regions in
    \code{randpop.nb}, see Hennig and Hausdorf (2002), p. 5. This is
    usually no improvement.}
}
\details{
  From the original data, the distribution of the
  range sizes of the species, the autocorrelation parameter \code{pd}
  (estimated by \code{autoconst}) and the distribution on the regions
  induced by the relative species numbers are taken. With these
  parameters, \code{times} populations according to the null model
  implemented in \code{randpop.nb} are generated and the test statistic
  is evaluated. The resulting p-value is number of simulated statistic
  values more extreme than than the value of the original data\code{+1}
  divided by \code{times+1}. "More extreme" means smaller for
  \code{"lcomponent"}, \code{"distratio"}, \code{"nn"}, larger for
  \code{"inclusions"}, and
  twice the smaller number between the original statistic value and the
  "border", i.e., a two-sided test for \code{"isovertice"}.
  If \code{pd=NA} was
  specified, a diagnostic plot
  for the estimation of \code{pd} is plotted by \code{autoconst}.
  For details see Hennig
  and Hausdorf (2002) and the help pages of the cited functions. 
}
\value{
  An object of class \code{prabtest}, which is a list with components
  \item{results}{vector of test statistic values for all simulated
    populations.}
  \item{datac}{test statistic value for the original data.'}
  \item{p.value}{the p-value.}
  \item{tuning}{see above.}
  \item{pd}{see above.}
  \item{reg}{regression coefficients from \code{autoconst}.}
  \item{teststat}{see above.}
  \item{distance}{the distance measure chosen, see \code{prabinit}.}
  \item{times}{see above.}
  \item{pdfnb}{see above.}
}

\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.

Hausdorf, B. and Hennig, C. (2003)  Biotic Element Analysis in
Biogeography. To appear in  \emph{Systematic Biology}.

Hausdorf, B. and Hennig, C. (2003) Nestedness of nerth-west European
land snail ranges as a consequence of differential immigration from
Pleistocene glacial refuges. \emph{Oecologia} 135, 102-109.
}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{prabinit}} generates objects of class \code{prab}.
  
  \code{\link{autoconst}} estimates \code{pd} from such objects.
  
  \code{\link{randpop.nb}} generates populations from the null model.
  An alternative model is given by \code{\link{cluspop.nb}}.

  Some more information on the test statistics is given in
  \code{\link{homogen.test}}, \code{\link{lcomponent}},
  \code{\link{distratio}}, \code{\link{nn}},
  \code{\link{incmatrix}}.

  The simulations are computed by \code{\link{pop.sim}}.

  Summary and print methods: \code{\link{summary.prabtest}}.
}


\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(1234)
x <- prabinit(prabmatrix=kykladspecreg, neighborhood=nb)
# If you want to use your own ASCII data files, use
# x <- prabinit(file="path/prabmatrixfile",
# neighborhood="path/neighborhoodfile")
prabtest(x, times=100, pd=0.35)
# These settings are chosen to make the example execution
# a bit faster; usually you will use prabtest(kprab).
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{spatial}% __ONLY ONE__ keyword per line
\eof
\name{randpop.nb}
\alias{randpop.nb}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Simulation of presence-absence matrices (non-clustered)}
\description{
  Generates a simulated matrix where the rows are interpreted as regions
  and the columns as species, 1 means that a species is present in the
  region and 0 means that the species is absent. Species are generated
  i.i.d.. Spatial autocorrelation of a species' presences is governed by
  the parameter \code{p.nb} and a list of neighbors for each region.
}
\usage{
randpop.nb(neighbors, p.nb = 0.5, n.species, n.regions =
length(neighbors), vector.species = rep(1, n.species),
species.fixed = FALSE, pdf.regions = rep(1/n.regions, n.regions),
count = TRUE, pdfnb = FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{neighbors}{A list with a component for every region. The
    components are vectors of integers indicating
    neighboring regions. A region without neighbors (e.g., an island)
    should be assigned a list \code{numeric(0)}.}
  \item{p.nb}{numerical between 0 and 1. The probability that a new
    region is drawn from the non-neighborhood of the previous regions
    belonging to a species under generation. Note that for a given
    presence-absence matrix, this parameter can be estimated by
    \code{autoconst} (called \code{pd} there).}
  \item{n.species}{integer. Number of species.}
  \item{n.regions}{integer. Number of regions.}
  \item{vector.species}{vector of integers. If
    \code{species.fixed=TRUE}, \code{vector.species} must have length
    \code{n.species} and gives the sizes (i.e., numbers of regions) of
    the species to generate. Else, the sizes are generated randomly from
    the empirical distribution of \code{vector.species}.}
  \item{species.fixed}{logical. See \code{vector.species}.}
  \item{pdf.regions}{numerical vector of length \code{n.species}. The
    entries must sum up to 1 and give probabilities for the regions to
    be drawn during the generation of a species. These probabilities are
    used conditional on the new region being a neighbor or a
    non-neighbor of the previous regions of the species, see
    \code{p.nb}.}
  \item{count}{logical. If \code{TRUE}, the number of the currently
    generated species is printed.}
  \item{pdfnb}{logical. If \code{TRUE}, the probabilities of the regions
    are modified according to the number of neighboring regions by
    dividing them relative to the others by min(1,number of neighbors).}
}
\details{
  The principle is that a single species with given size is generated
  one-by-one region. The first region is drawn according to
  \code{pdf.regions}. For all following regions, a neighbor or
  non-neighbor of the previous configuration is added (if possible),
  as explained in \code{pdf.regions}, \code{p.nb}.  
}
\value{
  A 0-1-matrix, rows are regions, columns are species.
}
\references{
Hennig, C. and Hausdorf, B. (2002) Distance-based parametric bootstrap
tests for clustering of species ranges, submitted,
\url{http://stat.ethz.ch/Research-Reports/110.html}.

Hausdorf, B. and Hennig, C. (2003)  Biotic Element Analysis in
Biogeography. To appear in  \emph{Systematic Biology}.

Hausdorf, B. and Hennig, C. (2003) Nestedness of nerth-west European
land snail ranges as a consequence of differential immigration from
Pleistocene glacial refuges. \emph{Oecologia} 135, 102-109.

}
\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{
  \code{\link{autoconst}} estimates \code{p.nb} from matrices of class
  \code{prab}. These are generated by \code{\link{prabinit}}.
  
  \code{\link{prabtest}} uses \code{randpop.nb} as a null model for
  tests of clustering. An alternative model is given by
  \code{\link{cluspop.nb}}.  
}

\examples{
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(2346)
randpop.nb(nb, p.nb=0.1, n.species=5, vector.species=c(1,10,20,30,34))
}
\keyword{spatial}% at least one, from doc/KEYWORDS

\eof
\name{summary.prabtest}
\alias{summary.prabtest}
\alias{print.summary.prabtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Print and summary method for prabtest}
\description{
  Useful compression of the results of \code{prabtest}.
}
\usage{
summary.prabtest(object, ...)

\method{print}{summary.prabtest}(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{object of class \code{prabtest}.}
  \item{x}{object of class \code{summary.prabtest}.}
  \item{\dots}{no meaning, necessary for print and summary methods.}
}

\value{
  \code{summary.prabtest} produces an object of class
  \code{summary.prabtest}, which is a list with components
  \item{rrange}{range of the simulation results (test statistic values)
    of \code{object}.}
  \item{rmean}{mean of the simulation results (test statistic values)
    of \code{object}.}
  \item{datac, p.value, pd, tuning, teststat, distance, times, pdfnb}{directly
    taken from \code{object}, see \code{prabtest}.}
}

\author{Christian Hennig
  \email{hennig@math.uni-hamburg.de}
  \url{http://www.math.uni-hamburg.de/home/hennig/}}

\seealso{\code{\link{prabtest}}}

\examples{
data(kykladspecreg)
# Note: If you do not use the installed package, replace this by
# kykladspecreg <- read.table("(path/)kykladspecreg.dat")
data(nb)
# Note: If you do not use the installed package, replace this by
# nb <- list()
# for (i in 1:34)
#   nb <- c(nb,list(scan(file="(path/)nb.dat",
#                   skip=i-1,nlines=1)))
set.seed(5678)
kprab <- prabinit(prabmatrix=kykladspecreg, neighborhood=nb)
kpt <- prabtest(kprab, times=50, pd=0.35, teststat="lcomponent")
# The parameters times and
# pd are only given to fasten the example execution.
summary(kpt)
}
\keyword{cluster}% at least one, from doc/KEYWORDS
\keyword{spatial}% __ONLY ONE__ keyword per line


\eof
