\name{ckappa}
\alias{ckappa}
\title{Cohen's Kappa for 2 raters}
\description{Computes Cohen's Kappa for agreement in the case of 2 raters. The diagnosis (the object of the rating) may have k possible values.}
\usage{ckappa(r)}
\arguments{
  \item{r}{n*2 matrix or dataframe, n subjects and 2 raters}}
\details{The function deals with the case where the two raters have not exactly the same scope of rating (some software associate an error with this situation).
Missing value are omitted.}
\value{A list with :
  \item{$table}{the 2*k table of raw data (first rater in rows, second rater in columns)}
  \item{$kappa}{Cohen's Kappa}}
\references{Cohen, J. (1960), A coefficient of agreement for nominal scales, Educational and Psychological measurements, 20, 37-46.}
\author{Bruno Falissard}
\examples{data(expsy)
ckappa(expsy[,c(12,14)])          # Cohen's kappa for binary diagnosis

library(boot)
ckappa.boot <- function(data,x) {ckappa(data[x,])[[2]]}
res <- boot(expsy[,c(12,14)],ckappa.boot,1000)
quantile(res$t,c(0.025,0.975))    # two-sided bootstrapped confidence interval of kappa
ckappa(expsy[,c(11,13)])          # Cohen's kappa for non binary diagnosis
}
\keyword{univar}


\eof
\name{cronbach}
\alias{cronbach}
\title{Cronbach's coefficient alpha}
\description{
Computes the Cronbach's reliability coefficient alpha. This coefficient may be applied to a series of items destinated to be aggregated in a single score. It estimates reliability in the framework of the domain sampling model.
}
\usage{
cronbach(v1)
}
\arguments{
  \item{v1}{n*p matrix or dataframe, n subjects and p items}
}
\details{
Missing value are omitted in a "listwise" way (all items are removed even if only one of them is missing).
}
\value{
A list with :
  \item{$sample.size}{Number of subjects under study}
  \item{$number.of.items}{Number of items of the scale or questionnaire}
  \item{$alpha}{alpha}
}
\references{Nunnaly, J.C., Bernstein, I.H. (1994), Psychometric Theory, 3rd edition, McGraw-Hill Series in Psychology.}
\author{Bruno Falissard}
\examples{
data(expsy)
cronbach(expsy[,1:10])  ## not good because item 2 is reversed (1 is high and 4 is low)
cronbach(cbind(expsy[,c(1,3:10)],-1*expsy[,2]))  ## better

datafile <- cbind(expsy[,c(1,3:10)],-1*expsy[,2])
library(boot)
cronbach.boot <- function(data,x) {cronbach(data[x,])[[3]]}
res <- boot(datafile,cronbach.boot,1000)
quantile(res$t,c(0.025,0.975))  ## two-sided bootstrapped confidence interval of Cronbach's alpha
}
\keyword{univar}

\eof
\name{expsy}
\alias{expsy}
\non_function{}
\title{Data set related to psychometry}
\usage{data(expsy)}
\description{
The \code{expsy}
data frame has 30 rows and 16 columns with missing data.
it1-it10 correspond to the rating of 30 patients with a 10 items scale.
r1, r2, r3 to the rating of item 1 by 3 different clinicians of the same 30 patients.
rb1, rb2, rb3 to the binary transformation of r1, r2, r3 (1 or 2 -> 0; and 3 or 4 -> 1) .
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{it1}{a numeric vector corresponding to item 1 (rated from 1:low to 4:high)}
    \item{it2}{a numeric vector corresponding to item 2 (rated from 1:high to 4:low)}
    \item{it3}{a numeric vector corresponding to item 3 (rated from 1:low to 4:high)}
    \item{it4}{a numeric vector corresponding to item 4 (rated from 1:low to 4:high)}
    \item{it5}{a numeric vector corresponding to item 5 (rated from 1:low to 4:high)}
    \item{it6}{a numeric vector corresponding to item 6 (rated from 1:low to 4:high)}
    \item{it7}{a numeric vector corresponding to item 7 (rated from 1:low to 4:high)}
    \item{it8}{a numeric vector corresponding to item 8 (rated from 1:low to 4:high)}
    \item{it9}{a numeric vector corresponding to item 9 (rated from 1:low to 4:high)}
    \item{it10}{a numeric vector corresponding to item 10 (rated from 1:low to 4:high)}
    \item{r1}{a numeric vector corresponding to item 1 rated by rater 1}
    \item{rb1}{binary transformation of r1}
    \item{r2}{a numeric vector corresponding to item 1 rated by rater 2}
    \item{rb2}{binary transformation of r2}
    \item{r3}{a numeric vector corresponding to item 1 rated by rater 3}
    \item{rb3}{binary transformation of r3}
  }
}
\source{
artificial data}
\examples{
data(expsy)
expsy[1:4,]
}
\keyword{datasets}


\eof
\name{fpca}
\alias{fpca}
\title{Focused Principal Components Analysis}
\description{
Graphical representation similar to a principal components analysis but adapted to data structured with dependent/independent variables
}
\usage{
fpca(datafile, y, x, cx=0.75, namesvar=attributes(datafile)$names, pvalues="No", partial="Yes", input="data", contraction="No", sample.size=1)
}
\arguments{
  \item{datafile}{name of datafile}
  \item{y}{column number of the dependent variable}
  \item{x}{column numbers of the independent (explanatory) variables}
  \item{cx}{size of the lettering (0.75 by default, 1 for bigger letters, 0.5 for smaller)}
  \item{namesvar}{label of variables (names of columns by default)}
  \item{pvalues}{vector of prespecified pvalues (pvalues="No" by default) (see below)}
  \item{partial}{partial="Yes" by default, corresponds to the original method (see below)}
  \item{input}{input="Cor" for a correlation matrix (input="data" by default)}
  \item{contraction}{change the aspect of the diagram, contraction="Yes" is convenient for large data set (contraction="No" by default)}
  \item{sample.size}{to be specified if input="Cor"}
}
\details{
This representation is close to a Principal Components Analysis (PCA).
Contrary to PCA, correlations between the dependent variable and the other variables are represented faithfully. The relationships between non dependent variables are interpreted like in a PCA: correlated variables are close or diametrically opposite (for negative correlations), independent variables make a right angle with the origin.
The focus on the dependent variable leads formally to a partialisation of the correlations between the non dependent variables by the dependent variable (see reference). To avoid this partialisation, the option partial="No" can be used.
It may be interesting to represent graphically the strength of association between the dependent variable and the other variables using p values coming from a model. A vector of pvalue may be specified in this case.
}
\value{
A plot.
}
\references{Falissard B, Focused Principal Components Analysis: looking at a correlation matrix with a particular interest in a given variable. Journal of Computational and Graphical Statistics (1999), 8(4): 906-912.}
\author{Bruno Falissard, Bill Morphey}
\examples{
data(sleep)
fpca(sleep,5,c(2:4,7:11)) 
## focused PCA of the duration of paradoxical sleep (dreams, 5th column)
## against constitutional variables in mammals (columns 2, 3, 4, 7, 8, 9, 10, 11).
## Variables inside the red cercle are significantly correlated
## to the dependent variable with p<0.05.
## Green variables are positively correlated to the dependent variable,
## yellow variables are negatively correlated.
## There are three clear clusters of independent variables. 
}
\keyword{multivariate}


\eof
\name{icc}
\alias{icc}
\title{Intraclass correlation coefficient (ICC)}
\description{
Computes the ICC of several series of measurements, for example in an interrater agreement study. Two types of ICC are proposed: consistency and agreement.
}
\usage{
icc(data)
}
\arguments{
  \item{data}{n*p matrix or dataframe, n subjects p raters}
}
\details{
Missing data are omitted in a listwise way. The "agreement" ICC is the ratio of the subject variance by the sum of the subject variance, the rater variance and the residual; it is generally prefered. The "consistency" version is the ratio of the subject variance by the sum of the subject variance and the residual; it may be of interest when estimating the reliability of pre/post variations in measurements.
}
\value{
A list with :
  \item{$nb.subjects}{number of subjects under study}
  \item{$nb.raters}{number of raters}
  \item{$subject.variance}{subject variance}
  \item{$rater.variance}{rater variance}
  \item{$residual}{residual variance}
  \item{$icc.consistency}{Intra class correlation coefficient, "consistency" version}
  \item{$icc.agreement}{Intra class correlation coefficient, "agreement" version}
}
\references{Shrout, P.E., Fleiss, J.L. (1979), Intraclass correlation: uses in assessing rater reliability, Psychological Bulletin, 86, 420-428.}
\author{Bruno Falissard}
\note{}
\examples{
data(expsy)
icc(expsy[,c(12,14,16)])

library(boot)
icc.boot <- function(data,x) {icc(data[x,])[[7]]}
res <- boot(expsy[,c(12,14,16)],icc.boot,1000)
quantile(res$t,c(0.025,0.975))  # two-sided bootstrapped confidence interval of icc (agreement)
}
\keyword{univar}


\eof
\name{lkappa}
\alias{lkappa}
\title{Light's kappa for n raters}
\description{
Computes Light's Kappa for agreement in the case of n raters. The diagnosis (the object of the rating) may have k possible values (ordered or not).
}
\usage{
lkappa(r, type="Cohen", weights="squared")
}
\arguments{
  \item{r}{m*n matrix, m subjects and n raters}
  \item{type}{type="Cohen" for a categorical diagnosis. If not, the diagnosis is supposed to be ordered}
  \item{weights}{weights="squared" for a weighted kappa with squared weights. If not, weigths are computed with absolute values.}
}
\details{
Light's Kappa is equal to the mean of the n(n-1)/2 kappas obtained from each pair of raters. Missing values are omitted locally when considering each pair of raters.
If type="Cohen" the diagnosis is considered as a categorical variable. If not, the diagnosis is considered as an ordered variable and weigthed kappa's are computed.
In this last situation, the type of weights that is used (squared or absolute values) is given by the variable weigths.
}
\value{
kappa (mean of the n(n-1)/2 kappas obtained from each pair of raters)
}
\references{
Conger, A.J. (1980), Integration and generalisation of Kappas for multiple raters, Psychological Bulletin, 88, 322-328.
}
\author{Bruno Falissard}
\examples{
data(expsy)
lkappa(expsy[,c(11,13,15)])       # Light's kappa for non binary diagnosis
lkappa(expsy[,c(12,14,16)])       # Light's kappa for binary diagnosis
lkappa(expsy[,c(11,13,15)], type="weighted")      # Light's kappa for non binary ordered diagnosis

library(boot)
lkappa.boot <- function(data,x) {lkappa(data[x,], type="weighted")}
res <- boot(expsy[,c(11,13,15)],lkappa.boot,1000)
quantile(res$t,c(0.025,0.975))       # Bootstrapped confidence interval of Light's kappa  
}
\keyword{univar}


\eof
\name{mdspca}
\alias{mdspca}
\title{Graphical representation of a correlation matrix using a Principal Component Analysis}
\description{
Similar to many routines, the interest is in the possible representation of both variables and subjects (and by the way categorical variables) with active and supplementary points. Missing data are omitted.
}
\usage{
mdspca(datafile, supvar="no", supsubj="no", namesupvar=colnames(supvar,do.NULL=FALSE), namesupsubj=colnames(supsubj, do.NULL=FALSE), dimx=1, dimy=2, cx=0.75)
}
\arguments{
  \item{datafile}{name of datafile}
  \item{supvar}{matrix corresponding to supplementary variables (if any), supvar="no" by default}
  \item{supsubj}{matrix corresponding to supplementary subjects (if any), supsubj="no" by default}
  \item{namesupvar}{names of the points corresponding to the supplementary variables}
  \item{namesupsubj}{names of the points corresponding to the supplementary subjects}
  \item{dimx}{rank of the component displayed on the x axis (1 by default)}
  \item{dimy}{rank of the component displayed on the y axis (2 by default)}
  \item{cx}{size of the lettering (0.75 by default, 1 for bigger letters, 0.5 for smaller)}
}
}
\value{
A diagram (two diagrams if supplementary subjects are used)
}
\author{Bruno Falissard}
\examples{
data(sleep)

mdspca(sleep[,c(2:5,7:11)])
## three consistent groups of variables, paradoxical sleep (dream)
## is negatively correlated with danger

mdspca(sleep[,c(2:5,7:11)],supvar=sleep[,6],namesupvar="Total.sleep",supsubj=sleep[,1],namesupsubj="",cx=0.5)
## Total.sleep is here a supplementary variable since it is deduced
## from Paradoxical.sleep and Slow.wave.sleep
## The variable Species is displayed in the subject plane,
## Rabbit and Cow have a high level of danger
}
\keyword{multivariate}


\eof
\name{scree.plot}
\alias{scree.plot}
\title{Screeplot of eigenvalues, simulated data are available}
\description{
Graphical representation of the eigenvalues of a correlation/covariance matrix.
Usefull to determine the dimensional structure of a set of variables.
Simulation are proposed to help the interpretation.
}
\usage{
scree.plot(namefile, title = "Scree Plot", type = "R", use = "complete.obs", simu = "F")
}
\arguments{
  \item{namefile}{dataset}
  \item{title}{Title}
  \item{type}{ type="R" to obtain the eigenvalues of the correlation matrix of dataset, type="V" for the covariance matrix, type="M" if the input data is directly the matrix, type="E" if the input data are directly the eigenvalues}
  \item{use}{omit missing values by default, use="P" to analyse the pairwise correlation/covariance matrix}
  \item{simu}{simu=p to add p screeplots of simulated random normal data (same number of patients and variables as in the original data set, same pattern of missing data if use="P")}
}
\details{
Simulations lead sometimes to underestimate the actual number of dimensions (as opposed to Kayser rule: eigen values superior to 1). Basically, simu=20 is enough.
}
\value{
a plot
}
\references{Falissard, B. (2001), Mesurer la subjectivit en sant, perspective mthodologique et statistique. Masson, Paris}
\author{Bruno Falissard}
\examples{
data(expsy)
scree.plot(expsy[,1:10],simu=20,use="P") #no obvious structure with such a small sample
}
\keyword{multivariate}


\eof
\name{sleep}
\alias{sleep}
\non_function{}
\title{ Ecological and Constitutional Data in Mammals }
\usage{data(sleep)}
\description{
Data from which conclusions were drawn in the article 
Mammals: Ecological and Constitutional Correlates,
by Allison, T. and Cicchetti, D. (1976)
Science, November 12, vol. 194, pp.732-734
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Species}{a factor with levels}
    \item{Body.weight}{a numeric vector, body weight in kg}
    \item{Brain.weight}{a numeric vector, Brain weight in g}
    \item{Slow.wave.sleep}{a numeric vector, nondreaming sleep (hrs/day)}
    \item{Paradoxical.sleep}{a numeric vector, dreaming sleep (hrs/day)}
    \item{Total.sleep}{a numeric vector, nondreaming + dreaming (hrs/day)}
    \item{Maximum.life.span}{a numeric vector (in years)}
    \item{Gestation.time}{a numeric vector (in days)}
    \item{Predation}{a numeric vector, Predation index (1 min - 5 max)}
    \item{Sleep.exposure}{a numeric vector, Sleep exposure index (1 min - 5 max)}
    \item{Danger}{a numeric vector, Overall danger index (1 min - 5 max)}
  }
}
\source{
http://lib.stat.cmu.edu/datasets/sleep
}
\references{
Mammals: Ecological and Constitutional Correlates, by Allison, T. and Cicchetti, D. (1976)
Science, November 12, vol. 194, pp.732-734
}
\examples{
data(sleep)
str(sleep)
}
\keyword{datasets}

\eof
\name{sphpca}
\alias{sphpca}
\title{Spherical Representation of a Correlation Matrix}
\description{
Graphical representation of a correlation matrix, similar to principal component analysis (PCA) but the mapping is on a sphere. The information is close to a 3d PCA, the picture is however easier to interpret since the variables are in fact on a 2d map.
}
\usage{
sphpca(datafile, h=0, v=0, f=0, cx=0.75, nbsphere=2, back=FALSE)
}
\arguments{
  \item{datafile}{name of datafile}
  \item{h}{rotation of the sphere on a horizontal plane (in degres)}
  \item{v}{rotation of the sphere on a vertical plane (in degres)}
  \item{f}{rotation of the sphere on a frontal plane (in degres)}
  \item{cx}{size of the lettering (0.75 by default, 1 for bigger letters, 0.5 for smaller)}
  \item{nbsphere}{two by default: front and back}
  \item{back}{"FALSE" by default: the back sphere is not seen through}
}
\details{
The sphere may be rotated to help in visualising most of variables on a same side (front for example).
By default, the back of the sphere (right plot) is not seen showing through. Computations are based
on a principal components approximation (see reference for details).
}
\value{
A plot
}
\references{Falissard B, A spherical representation of a correlation matrix, Journal of Classification (1996), 13:2, 267-280.}
\author{Bruno Falissard}
\examples{
data(sleep)
sphpca(sleep[,c(2:5,7:11)])
##spherical representation of ecological and constitutional correlates in mammals
}
\keyword{multivariate}


\eof
\name{wkappa}
\alias{wkappa}
\title{weighted Kappa for 2 raters}
\description{
Computes a weighted Kappa for agreement in the case of 2 raters. The diagnosis (the object of the rating) may have k possible ordered values.
}
\usage{
wkappa(r,weights="squared")
}
\arguments{
  \item{r}{n*2 matrix or dataframe, n subjects and 2 raters}
  \item{weights}{weights="squared" to obtain squared weights. If not, absolute weights are computed (see details)}
}
\details{
The diagnoses are ordered as follow: numbers < letters, letters and numbers ordered naturally.
For weigths="squared", weights are related to squared differences between rows and columns indices (in this situation wkappa is close to an icc).
For weights!="squared", weights are related to absolute values of differences between rows and columns indices.
The function deals with the case where the two raters have not exactly the same scope of rating (some software associate an error with this situation).
Missing value are omitted.
}
\value{
A list with :
  \item{$table}{the 2*k table of raw data (first rater in rows, second rater in columns)}
  \item{$weights}{"squared" or "absolute"}
  \item{$kappa}{Weighted Kappa}
}
\references{Cohen, J. Weighted kappa: nominal scale agreement with provision for scaled disagreement or partial credit. Psychological Bulletin 70 (1968): 213-220.
}
\author{Bruno Falissard}
\examples{
data(expsy)
wkappa(expsy[,c(11,13)])        # weighted kappa (squared weights)

library(boot)
wkappa.boot <- function(data,x) {wkappa(data[x,])[[3]]}
res <- boot(expsy[,c(11,13)],wkappa.boot,1000)
quantile(res$t,c(0.025,0.975))  # two-sided bootstrapped confidence interval of weighted kappa
}
\keyword{univar}


\eof
