\name{A starting point}
\alias{A starting point}
\title{Introductory comments}
\description{
  A brief introduction to the R/qtl package, with a walk-through of an
  analysis.  
}

\section{New to R and/or R/qtl?}{
  \itemize{
    \item In order to use the R/qtl package, you must type (within R)
    \code{library(qtl)}.  You may wish to include this in a
    \code{\link[base:Startup]{.First}} function or
    \code{\link[base:Startup]{.Rprofile}} file. 

    \item Documention and several tutorials are available
    at the R archive (\url{http://cran.r-project.org}).

    \item Use the \code{\link[base]{help.start}} function to start the
    html version of the R help.  In Windows or MacOS, you may wish to
    use \code{options(htmlhelp=TRUE)} to get easy access to the html
    version of the help files; this could be included in a
    \code{\link[base:Startup]{.First}} function or
    \code{\link[base:Startup]{.Rprofile}} file.

    \item Type \code{library(help=qtl)} to get a list of the functions
    in R/qtl.

    \item Use the \code{\link[base]{example}} function to run examples
    of the various functions in R/qtl.

    \item A tutorial on the use of R/qtl is distributed with the
    package, \file{rqtltour.pdf}.

    \item Download the latest version of R/qtl from
    \url{http://www.biostat.jhsph.edu/~kbroman/qtl}.
  }
}

\section{Walk-through of an analysis}{
Here we briefly describe the use of R/qtl to analyze an experimental
cross.  A more extensive tutorial on its use is distributed with the
package, \file{rqtltour.pdf}.

A difficult first step in the use of most data analysis software is the
import of data.  With R/qtl, one may import data in several different
formats by use of the function \code{\link[qtl]{read.cross}}.  The
internal data structure used by R/qtl is rather complicated, and is
described in the help file for \code{\link[qtl]{read.cross}}.  We won't
discuss data import any further here, except to say that the
comma-delimited format (\code{"csv"}) is recommended.  If you have
trouble importing data, send an email to Karl Broman,
\email{kbroman@jhsph.edu}, perhaps attaching examples of your data
files.  (Such data will be kept confidential.)

We consider the example data \code{\link[qtl]{hyper}}, an experiment on
hypertension in the mouse, kindly provided
by Bev Paigen and Gary Churchill.  Use the \code{\link[base]{data}}
function to load the data.

\code{data(hyper)}

The \code{\link[qtl]{hyper}} data set has class \code{cross}.  The
function \code{\link[qtl]{summary.cross}} gives some summary information
on the data, and checks the data for internal consistency.  A number
of other utility functions are available; hopefully these are
self-explanatory.

\code{summary(hyper)} \cr
\code{nind(hyper)} \cr
\code{nphe(hyper)} \cr
\code{nchr(hyper)} \cr
\code{nmar(hyper)} \cr
\code{totmar(hyper)}

The function \code{\link[qtl]{plot.cross}} gives a graphical summary of
the data; it calls \code{\link[qtl]{plot.missing}} (to plot a matrix
displaying missing genotypes) and \code{\link[qtl]{plot.map}} (to plot
the genetic maps), and also displays histograms or barplots of the
phenotypes.  The \code{\link[qtl]{plot.missing}} function can plot
individuals ordered by their phenotypes; you can see that for most
markers, only individuals with extreme phenotypes were genotyped.

\code{plot(hyper)} \cr
\code{plot.missing(hyper)} \cr
\code{plot.missing(hyper, reorder=TRUE)} \cr
\code{plot.map(hyper)}

Note that one marker (on chromosome 14) has no genotype data.  The
function \code{\link[qtl]{drop.nullmarkers}} removes such markers from
the data. 

\code{hyper <- drop.nullmarkers(hyper)} \cr
\code{totmar(hyper)}

The function \code{\link[qtl]{est.rf}} estimates the recombination
fraction between each pair of markers, and calculates a LOD score for
the test of \eqn{r} = 1/2.  This is useful for identifying markers that
are placed on the wrong chromosome.  Note that since, for these data,
many markers were typed only on recombinant individuals, the pairwise
recombination fractions show rather odd patterns.

\code{hyper <- est.rf(hyper)} \cr
\code{plot.rf(hyper)} \cr
\code{plot.rf(hyper, chr=c(1,4))}

To re-estimate the genetic map for an experimental cross, use the
function \code{\link[qtl]{est.map}}.  The function
\code{\link[qtl]{plot.map}}, in addition to plotting a single map, can
plot the comparison of two genetic maps (as long as they are composed of
the same numbers of chromosomes and markers per chromosome).  The
function \code{\link[qtl]{replace.map}} map be used to replace the
genetic map in a cross with a new one.

\code{newmap <- est.map(hyper, error.prob=0.01, trace=TRUE)} \cr
\code{plot.map(hyper, newmap)} \cr
\code{hyper <- replace.map(hyper, newmap)}

Before doing QTL analyses, a number of intermediate calculations may
need to be performed.  The function \code{\link[qtl]{calc.genoprob}}
calculates conditional genotype probabilities given the multipoint
marker data.  \code{\link[qtl]{sim.geno}} simulates sequences of
genotypes from their joint distribution, given the observed marker data.
\code{\link[qtl]{argmax.geno}} calculates the most likely sequence of
underlying genotypes, given the observed marker data.  

These three functions return a modified version of the input cross, with
the intermediate calculations included.  

\code{hyper <- calc.genoprob(hyper, step=2.5, error.prob=0.01)} \cr
\code{hyper <- sim.geno(hyper, step=2.5, n.draws=64, error.prob=0.01)} \cr
\code{hyper <- argmax.geno(hyper, error.prob=0.01)}

The function \code{\link[qtl]{calc.errorlod}} may be used to assist in
identifying possible genotyping errors; it calculates the error LOD
scores described by Lincoln and Lander (1992).  It requires the results
of \code{\link[qtl]{calc.genoprob}}, run with a matching value for the
assumed genotyping error probability, \code{error.prob}.

\code{hyper <- calc.errorlod(hyper, error.prob=0.01)} \cr
\code{plot.errorlod(hyper)} \cr
\code{top.errorlod(hyper)} \cr
\code{plot.errorlod(hyper, chr=c(4,11,16))}

The function \code{\link[qtl]{plot.geno}} may be used to inspect the
observed genotypes for a chromosome, with likely genotyping errors
flagged.

\code{plot.geno(hyper, chr=16, ind=71:90, min.sep=4)}

The function \code{\link[qtl]{scanone}} performs a genome scan with a
single QTL model.  By default, it performs standard interval mapping
(Lander and Botstein 1989): use of a normal model and the EM algorithm.
If one specifies \code{method="hk"}, Haley-Knott regression is performed
(Haley and Knott 1992).  These two methods require the results from
\code{\link[qtl]{calc.genoprob}}.

\code{out.em <- scanone(hyper)} \cr
\code{out.hk <- scanone(hyper, method="hk")}

If one specifies \code{method="imp"}, a genome scan is performed by the
multiple imputation method of Sen and Churchill (2001).  This method
requires the results from \code{\link[qtl]{sim.geno}}.

\code{out.imp <- scanone(hyper, method="imp")}

The output of \code{\link[qtl]{scanone}} is a data.frame with class
\code{scanone}.  The function \code{\link[qtl]{plot.scanone}} may be
used to plot the results, and may plot up to three sets of results
against each other, as long as they conform appropriately.

\code{plot(out.em)} \cr
\code{plot(out.hk, col="blue", add=TRUE)} \cr
\code{plot(out.imp, col="red", add=TRUE)} \cr
\code{plot(out.hk, out.imp, out.em, chr=c(1,4), lty=1, col=c("blue","red","black"))}

The function \code{\link[qtl]{summary.scanone}} may be used to list
information on the peak LOD for each chromosome for which the LOD
exceeds a specified threshold.

\code{summary(out.em)} \cr
\code{summary(out.em, 3)} \cr
\code{summary(out.hk, 3)} \cr
\code{summary(out.imp, 3)}

The function \code{\link[qtl]{max.scanone}} returns the maximum LOD
score, genome-wide.

\code{max(out.em)} \cr
\code{max(out.hk)} \cr
\code{max(out.imp)}

One may also use \code{\link[qtl]{scanone}} to perform a permutation
test to get a genome-wide LOD significance threshold.  This will take
some time.

\code{operm.hk <- scanone(hyper, method="hk", n.perm=100)} \cr
\code{quantile(operm.hk, 0.95)}

We should say at this point that the function
\code{\link[base]{save.image}} will save your workspace to disk.  You'll
wish you had used this if R crashes.

\code{save.image()}

The function \code{\link[qtl]{scantwo}} performs a two-dimensional
genome scan with a two-QTL model.  Methods \code{"em"}, \code{"hk"} and
\code{"imp"} are all available.  \code{\link[qtl]{scantwo}} is
considerably slower than \code{\link[qtl]{scanone}}, and can require a
great deal of memory.  Thus, you may wish to create a version of
\code{hyper} for a more coarse grid.

\code{hyper.coarse <- calc.genoprob(hyper, step=10, err=0.01)} \cr
\code{hyper.coarse <- sim.geno(hyper, step=10, n.draws=64, err=0.01)} \cr
\cr
\code{out2.hk <- scantwo(hyper.coarse, method="hk")} \cr
\code{out2.em <- scantwo(hyper.coarse)} \cr
\code{out2.imp <- scantwo(hyper.coarse, method="imp")}

The output is an object with class \code{scantwo}.  The function
\code{\link[qtl]{plot.scantwo}} may be used to plot the results.  The
upper triangle contains LOD scores for tests of epistasis, while the
lower triangle contains joint LOD scores.

\code{plot(out2.hk)} \cr
\code{plot(out2.em)} \cr
\code{plot(out2.imp)}

The function \code{\link[qtl]{summary.scantwo}} lists the interesting
aspects of the output.  You must provide three LOD thresholds: for the joint
LOD, epistasis LOD, and conditional, single-QTL LOD scores.  The locus
pairs giving the highest joint LOD for each pair of chromosomes are
inspected, and those whose LOD score exceed the joint LOD threshold and
for which either the interaction LOD exceeds its threshold or both the
conditional single-QTL LODs exceed their threshold, are printed.

\code{summary(out2.em, c(8, 3, 3))} \cr
\code{summary(out2.em, c(0, 1000, 4))} \cr
\code{summary(out2.em, c(0, 4, 1000))}

The function \code{\link[qtl]{max.scantwo}} returns the maximum joint
and interaction LODs for a two-dimensional genome scan.

\code{max(out2.em)}

Permutation tests may also performed with \code{\link[qtl]{scantwo}};
it may take a few days of CPU time.  The output is a matrix with two
columns: the maximum joint and epistasis LODs, across the
two-dimensional genome scan, for each simulation replicate.

\code{operm2 <- scantwo(hyper.coarse, method="hk", n.perm=100)} \cr
\code{apply(operm2, 2, quantile, 0.95)} \cr
\code{hist(operm.hk,breaks=20)}
}

\section{Citing R/qtl}{
  To cite R/qtl in publications, use the Broman et al. (2003) reference
  listed below.
}

\references{
  Broman, K. W., Wu, H., Sen, S. and Churchill, G. A. (2003) R/qtl: QTL
  mapping in experimental crosses.  \emph{Bioinformatics} \bold{19},
  889--890. 

  Haley, C. S. and Knott, S. A. (1992) A simple regression method for mapping
  quantitative trait loci in line crosses using flanking markers.
  \emph{Heredity} \bold{69}, 315--324.

  Lander, E. S. and Botstein, D. (1989) Mapping Mendelian factors underlying
  quantitative traits using RFLP linkage maps.  \emph{Genetics}
  \bold{121}, 185--199.

  Lincoln, S. E. and Lander, E. S. (1992) Systematic detection of
  errors in genetic linkage data.  \emph{Genomics} \bold{14}, 604--610.

  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.
}

\note{
  Also see the package BQTL (Bayesian QTL mapping toolkit) by
  Charles C. Berry.  His package gave me the idea to have this ``A
  starting point'' help page.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{models}

\eof
\name{argmax.geno}
\alias{argmax.geno}

\title{Reconstruct underlying genotypes}

\description{
  Uses the Viterbi algorithm to identify the most likely sequence
  of underlying genotypes, given the observed multipoint marker data,
  with possible allowance for genotyping errors.
}

\usage{
argmax.geno(cross, step=0, off.end=0, error.prob=0, 
            map.function=c("haldane","kosambi","c-f","morgan"))
}

\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{step}{Maximum distance (in cM) between positions at which the
   genotypes are reconstructed, though for \code{step = 0}, genotypes
   are reconstructed only at the marker locations.}
 \item{off.end}{Distance (in cM) past the terminal markers on each
   chromosome to which the genotype reconstructions will be carried.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi,
   Carter-Falconer or Morgan map function when converting genetic
   distances into recombination fractions.} 
}

\details{
  We use the Viterbi algorithm to calculate
  \eqn{\arg \max_v \Pr(g = v | O)}{arg max_v Pr(g = v | O)} where
  \eqn{g} is the underlying sequence of genotypes and \eqn{O} is the
  observed marker genotypes.

  This is done by calculating
  \eqn{\gamma_k(v_k) = \max_{v_1, \ldots, v_{k-1}} \Pr(g_1 = v_1,
    \ldots, g_k = v_k, O_1, \ldots, O_k)}{% 
    Q[k](v[k]) = max{v[1], \ldots, v[k-1]} Pr(g[1] = v[1],
    \ldots, g[k] = v[k], O[1], \ldots, O[k])}
  for \eqn{k = 1, \ldots, n} and then tracing back through the
  sequence.
}

\value{
  The input \code{cross} object is returned with a component,
  \code{argmax}, added to each component of \code{cross$geno}.
  \code{argmax} is a matrix of size [n.ind x n.pos], where n.pos is the
  number of positions at which the reconstructed genotypes were obtained,
  containing the most likely sequences of underlying genotypes.
  Attributes \code{"error.prob"}, \code{"step"}, and \code{"off.end"}
  are set to the values of the corresponding arguments, for later
  reference.
}

\section{Warning}{
  The Viterbi algorithm can behave badly when \code{step} is small but
  positive. One may observe quite different results for different values
  of \code{step}. 

  The problem is that, in the presence of data like \code{A----H},  the
  sequences \code{AAAAAA} and \code{HHHHHH} may be more likely than any
  one of the sequences \code{AAAAAH}, \code{AAAAHH}, \code{AAAHHH},
  \code{AAHHHH}, \code{AHHHHH}, \code{AAAAAH}.  The Viterbi algorithm
  produces a single ``most likely'' sequence of underlying genotypes.
}


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
fake.f2 <- argmax.geno(fake.f2, step=2, off.end=5, err=0.01)

data(fake.bc)
fake.bc <- argmax.geno(fake.bc, step=0, off.end=0)
}

\references{
  Lange, K. (1999) \emph{Numerical analysis for statisticians}.
  Springer-Verlag. Sec 23.3.

  Rabiner, L. R. (1989) A tutorial on hidden Markov models and selected
  applications in speech recognition.  \emph{Proceedings of the IEEE}
  \bold{77}, 257--286.
}
  

\seealso{ \code{\link[qtl]{sim.geno}}, \code{\link[qtl]{calc.genoprob}} }

\keyword{utilities}

\eof
\name{badorder}
\alias{badorder}
\non_function{}

\title{An intercross with misplaced markers}

\description{
  Simulated data for an intercross with some markers out of order.
}  

\usage{data(badorder)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}

\details{
  There are 250 F2 individuals typed at a total of 36 markers on four
  chromosome.  The data were simulated with QTLs at the center of
  chromosomes 1 and 2. 

  The order of several markers on chromosome 1 is incorrect.  Markers on
  chromosomes 2 and 3 are switched.
}

\examples{
data(badorder)
# estimate recombination fractions
badorder <- est.rf(badorder)
plot.rf(badorder)

# re-estimate map
newmap <- est.map(badorder)
plot.map(badorder,newmap)

# assess marker order on chr 1
\dontrun{rip3 <- ripple(badorder,1,window=3)
summary(rip3)}
}

\seealso{ \code{\link[qtl]{est.rf}}, \code{\link[qtl]{ripple}}, 
  \code{\link[qtl]{est.map}}, \code{\link[qtl]{sim.cross}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{datasets}

\eof
\name{bristle3}
\alias{bristle3}
\non_function{}

\title{Data on bristle number in Drosophila}

\description{ 
  Data from bristle number in chromosome 3 recombinant isogenic lines of
  \emph{Drosophila melanogaster} (fruit flies).
} 

\usage{data(bristle3)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}

\details{
  There are 66 chromosome 3 recombinant isogenic lines, derived from
  inbred lines that were selected for low (A) and high (B) abdominal
  bristle numbers.  A recombinant chromosome 3 was placed in an isogenic
  low background.

  There are eight phenotypes: the average and SD of the number of
  abdominal and sternopleural bristles in males and females for each
  line.

  Each line is typed at 29 genetic markers on chromosome 3.
}

\references{
  Long, A. D., Mullaney, S. L., Reid, L. A., Fry, J. D., Langley,
  C. H. and MacKay, T. F. C. (1995) High resolution mapping of genetic
  factors affecting abdominal bristle number in \emph{Drosophila
    melanogaster}.  \emph{Genetics} \bold{139}, 1273--1291.
}
  
%\source{
%}

\examples{
data(bristle3)
# Summaries
summary(bristle3)
plot(bristle3)

# genome scan for each of the average phenotypes
bristle3 <- calc.genoprob(bristle3, step=2)
out <- vector("list", 4)
for(i in 1:4) out[[i]] <- scanone(bristle3, pheno.col=2*i-1)

# Plot the results
    # maximum LOD score among four phenotypes    
ym <- max(sapply(out, function(a) max(a[,3])))
plot(out[[1]], ylim=c(0,ym))
for(i in 2:4)
  plot(out[[i]], add=TRUE, col=c("blue","red","green")[i-1])
}

\seealso{ \code{\link[qtl]{bristleX}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{fake.bc}}, \code{\link[qtl]{fake.f2}},
  \code{\link[qtl]{fake.4way}}, \code{\link[qtl]{hyper}} }

\keyword{datasets}

\eof
\name{bristleX}
\alias{bristleX}
\non_function{}

\title{Data on bristle number in Drosophila}

\description{ 
  Data from bristle number in chromosome X recombinant isogenic lines of
  \emph{Drosophila melanogaster} (fruit flies).
} 

\usage{data(bristleX)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}

\details{
  There are 92 chromosome X recombinant isogenic lines, derived from
  inbred lines that were selected for low (A) and high (B) abdominal
  bristle numbers.  A recombinant chromosome X was placed in an isogenic
  low background.

  There are eight phenotypes: the average and SD of the number of
  abdominal and sternopleural bristles in males and females for each
  line.

  Each line is typed at 17 genetic markers on chromosome 3.
}

\references{
  Long, A. D., Mullaney, S. L., Reid, L. A., Fry, J. D., Langley,
  C. H. and MacKay, T. F. C. (1995) High resolution mapping of genetic
  factors affecting abdominal bristle number in \emph{Drosophila
    melanogaster}.  \emph{Genetics} \bold{139}, 1273--1291.
}
  
%\source{
%}

\examples{
data(bristleX)
# Summaries
summary(bristleX)
plot(bristleX)

# genome scan for each of the average phenotypes
bristleX <- calc.genoprob(bristleX, step=2)
out <- vector("list", 4)
for(i in 1:4) out[[i]] <- scanone(bristleX, pheno.col=2*i-1)

# Plot the results
    # maximum LOD score among four phenotypes    
ym <- max(sapply(out, function(a) max(a[,3])))
plot(out[[1]], ylim=c(0,ym))
for(i in 2:4) plot(out[[i]], add=TRUE, col=c("blue","red","green")[i-1])
}

\seealso{ \code{\link[qtl]{bristleX}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{fake.bc}}, \code{\link[qtl]{fake.f2}},
  \code{\link[qtl]{fake.4way}}, \code{\link[qtl]{hyper}} }

\keyword{datasets}

\eof
\name{c.cross}
\alias{c.cross}

\title{Combine data for QTL experiments}

\description{
  Concatenate the data for multiple QTL experiments.
}

\usage{c.cross(\dots)}

\arguments{
 \item{\dots}{A set of objects of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.  These must all either be of the
   same cross type or be a combination of backcrosses and intercrosses.
   All crosses must have exactly the same genetic maps, including the
   same number of chromosomes, chromosome names, number of markers per
   chromosome, marker names, and marker orders.}
}

\value{The concatenated input, as a \code{cross} object. Additional
  columns are added to the phenotype data indicating which cross an
  individual comes from; another column indicates cross type (0=BC,
  1=intercross), if there are crosses of different types.  The crosses
  are not required to have exactly the same set of phenotypes;
  phenotypes with the same names are assumed to be the same.}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
junk <- fake.f2
junk <- c(fake.f2,junk)
}

\seealso{ \code{\link[qtl]{subset.cross}} }

\keyword{manip}

\eof
\name{calc.errorlod}
\alias{calc.errorlod}

\title{Identify likely genotyping errors}

\description{
  Calculates a LOD score for each genotype, measuring the evidence for
  genotyping errors. 
}

\usage{
calc.errorlod(cross, error.prob=0.01,
              map.function=c("haldane","kosambi","c-f","morgan"))
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype)}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi,
   Carter-Falconer, or Morgan map function when converting genetic
   distances into recombination fractions.}
}

\details{
  Uses the results of the function \code{\link[qtl]{calc.genoprob}}
  (conditional genotype probabilities given multipoint marker data), and
  then calculates, for each individual at each marker, a LOD score
  measuring the strength of evidence for a genotyping error, as
  described by Lincoln and Lander (1992). 

  If results of \code{\link[qtl]{calc.genoprob}} are not contained in
  the input \code{cross} object, it is called with arguments \code{step
  = off.end = 0}.  If it had been run, but with a different value of
  \code{error.prob} then specified in the argument, it is called again
  with the specified value, and using the available \code{"step"} and
  \code{"off.end"} attributes, and the only genotype probabilities are
  replaced.

  Note that values below 3 are generally not interesting.  Also note
  that if markers are extremely tightly linked, \emph{recombination
  events} can give large error LOD scores.  The error LOD scores should
  not be trusted blindly, but should be viewed as a tool for identifying
  genotypes to study further.  

  Use \code{\link[qtl]{top.errorlod}} to print all genotypes with error
  LOD scores above a specified threshold,
  \code{\link[qtl]{plot.errorlod}} to plot the error LOD scores for
  specified chromosomes, and \code{\link[qtl]{plot.geno}} to view the
  observed genotype data with likely errors flagged.
}

\value{
  The input \code{cross} object is returned with a component,
  \code{errorlod}, added to each component of \code{cross$geno}.  The
  \code{errorlod} component is a matrix of size (n.ind x n.mar).  An
  attribute \code{"error.prob"} is set to the value of the corresponding 
  argument, for later reference.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\references{
  Lincoln, S. E. and Lander, E. S. (1992) Systematic detection of
  errors in genetic linkage data.  \emph{Genomics} \bold{14}, 604--610.
}

\examples{
data(hyper)
# run calc.genoprob, then calc.errorlod; use same error.prob
hyper <- calc.genoprob(hyper,error.prob=0.01)
hyper <- calc.errorlod(hyper,error.prob=0.01)

# plot the error LOD scores; print those above a specified cutoff 
plot.errorlod(hyper)
plot.errorlod(hyper,chr=1)
top.errorlod(hyper,cutoff=4)

# plot genotype data, flagging genotypes with error LOD > cutoff
plot.geno(hyper, chr=1, ind=160:200, cutoff=7, min.sep=2)

# Note that if markers are extremely tightly linked,
#     any recombination event will have a high error LOD
plot.geno(hyper, chr=1, ind=101:118, min.sep=2)
hyper$geno[[1]]$map
}

\seealso{ \code{\link[qtl]{plot.errorlod}},
  \code{\link[qtl]{top.errorlod}} } 


\keyword{utilities}

\eof
\name{calc.genoprob}
\alias{calc.genoprob}

\title{Calculate conditional genotype probabilities}

\description{
  Uses the hidden Markov model technology to calculate the
  probabilities of the true underlying genotypes given the observed
  multipoint marker data, with possible allowance for genotyping
  errors.
}

\usage{
calc.genoprob(cross, step=0, off.end=0, error.prob=0, 
              map.function=c("haldane","kosambi","c-f","morgan"))
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{step}{Maximum distance (in cM) between positions at which the
   genotype probabilities are calculated, though for \code{step = 0},
   probabilities are calculated only at the marker locations.}
 \item{off.end}{Distance (in cM) past the terminal markers on each
   chromosome to which the genotype probability calculations will be
   carried.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi or
   Carter-Falconer map function when converting genetic distances into
   recombination fractions.} 
}

\details{
  Let \eqn{O_k}{O[k]} denote the observed marker genotype at position
  \eqn{k}, and \eqn{g_k}{g[k]} denote the corresponding true underlying
  genotype. 

  We use the forward-backward equations to calculate
  \eqn{\alpha_{kv} = \log Pr(O_1, \ldots, O_k, g_k = v)}{%
    a[k][v] = log Pr(O[1], \ldots, O[k], g[k] = v)}
  and 
  \eqn{\beta_{kv} = \log Pr(O_{k+1}, \ldots, O_n | g_k = v)}{%
    b[k][v] = log Pr(O[k+1], \ldots, O[n] | g[k] = v)}

  We then obtain
  \eqn{Pr(g_k | O_1, \ldots, O_n) = \exp(\alpha_{kv} + \beta_{kv}) / s}{%
    Pr(g[k] | O[1], \ldots, O[n] = exp(a[k][v] + b[k][v]) / s}
  where
  \eqn{s = \sum_v \exp(\alpha_{kv} + \beta_{kv})}{%
    s = sum_v exp(a[k][v] + b[k][v])}

  In the case of the 4-way cross, with a sex-specific map, we assume a
  constant ratio of female:male recombination rates within the
  inter-marker intervals.
}

\value{
  The input \code{cross} object is returned with a component,
  \code{prob}, added to each component of \code{cross$geno}.
  \code{prob} is an array of size [n.ind x n.pos x n.gen] where n.pos is
  the number of positions at which the probabilities were calculated and
  n.gen = 3 for an intercross, = 2 for a backcross, and = 4 for a 4-way
  cross.  Attributes \code{"error.prob"}, \code{"step"},
  \code{"off.end"}, and \code{"map.function"} are set to the values of
  the corresponding arguments, for later reference (especially by the
  function \code{\link[qtl]{calc.errorlod}}).
}

\references{
  Lange, K. (1999) \emph{Numerical analysis for statisticians}.
  Springer-Verlag. Sec 23.3.

  Rabiner, L. R. (1989) A tutorial on hidden Markov models and selected
  applications in speech recognition.  \emph{Proceedings of the IEEE}
  \bold{77}, 257--286.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2, step=2, off.end=5)

data(fake.bc)
fake.bc <- calc.genoprob(fake.bc, step=0, off.end=0, err=0.01)
}

\seealso{ \code{\link[qtl]{sim.geno}}, \code{\link[qtl]{argmax.geno}},
         \code{\link[qtl]{calc.errorlod}} }

\keyword{utilities}

\eof
\name{clean}
\alias{clean}

\title{Remove derived data}

\description{
  Remove any intermediate calculations from a cross object.
}

\usage{
clean(cross)
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
}

\value{
  The input object, with any intermediate calculations
  (such as is produced by \code{\link[qtl]{calc.genoprob}},
  \code{\link[qtl]{argmax.geno}} and \code{\link[qtl]{sim.geno}})
  removed.
}

\examples{
data(fake.f2)
names(fake.f2$geno)
fake.f2 <- calc.genoprob(fake.f2)
names(fake.f2$geno)
fake.f2 <- clean(fake.f2)
names(fake.f2$geno)
}

\seealso{ \code{\link[qtl]{drop.nullmarkers}},
  \code{\link[qtl]{drop.markers}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{manip}

\eof
\name{comparecrosses}
\alias{comparecrosses}
\title{Compare two cross objects.}

\description{
  Verify that two objects of class \code{cross} have identical classes,
  chromosomes, markers, genotypes, genetic maps, and phenotypes.
}

\usage{comparecrosses(cross1, cross2)}

\arguments{
 \item{cross1}{An object of class \code{cross} (must be an
   intercross). See \code{\link[qtl]{read.cross}} for details.} 
 \item{cross2}{An object of class \code{cross} (must be an
   intercross). See \code{\link[qtl]{read.cross}} for details.} 
}

\value{None.}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
comparecrosses(listeria, listeria)
}

\seealso{ \code{\link[qtl]{summary.cross}} }

\keyword{manip}

\eof
\name{convert2ss}
\alias{convert2ss}
\title{Convert an intercross into a form to allow sex-specific maps}

\description{
  Convert a standard F2 intercross into a form which allows sex-specific
  maps (class \code{f2ss}).
}

\usage{convert2ss(cross)}

\arguments{
 \item{cross}{An object of class \code{cross} (must be an
   intercross). See \code{\link[qtl]{read.cross}} for details.} 
}

\value{The input \code{cross} object, but with the genetic map replaced
  by a pair (i.e., male and female) maps, and with the class
  changed to \code{f2ss}.}

\section{Warning}{
  The use of sex-specific maps for intercrosses is experimental.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
listeria.ss <- convert2ss(listeria)
}

\seealso{ \code{\link[qtl]{est.map}}, \code{\link[qtl]{listeria}} }

\keyword{manip}

\eof
\name{drop.markers}
\alias{drop.markers}

\title{Drop a set of markers}

\description{
  Drop a vector of markers from the data matrices and genetic maps.
}

\usage{
drop.markers(cross, markers)
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{markers}{A character vector of marker names.}
}

\value{
  The input object, with any markers in the vector \code{markers} removed
  from the genotype data matrices, genetic maps, and, if applicable, any
  derived data (such as produced by \code{\link[qtl]{calc.genoprob}}).
  (It might be a good idea to re-derive such things after using this
  function.)
}

\examples{
data(listeria)
listeria2 <- drop.markers(listeria, c("D10M44","D1M3","D1M75"))
}

\seealso{ \code{\link[qtl]{drop.nullmarkers}}, \code{\link[qtl]{geno.table}},
  \code{\link[qtl]{clean}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{manip}

\eof
\name{drop.nullmarkers}
\alias{drop.nullmarkers}

\title{Drop markers without any genotype data}

\description{
  Drop markers, from the data matrices and genetic maps, that have no
  genotype data.
} 

\usage{
drop.nullmarkers(cross)
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
}

\value{
  The input object, with any markers lacking genotype data removed from
  the genotype data matrices, genetic maps, and, if applicable, any
  derived data (such as produced by \code{\link[qtl]{calc.genoprob}}).
  (It might be a good idea to re-derive such things after using this
  function.)
}

\examples{
# removes one marker from hyper
data(hyper)
hyper <- drop.nullmarkers(hyper)

# shouldn't do anything to listeria
data(listeria)
listeria <- drop.nullmarkers(listeria)
}

\seealso{ \code{\link[qtl]{drop.markers}}, \code{\link[qtl]{clean}},
  \code{\link[qtl]{geno.table}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{manip}

\eof
\name{effectplot}
\alias{effectplot}

\title{Plot phenotype means against genotypes at one or two markers.}

\description{
  Plot the phenotype means for each group defined by the genotypes at 
  one or two markers (or the values at a discrete covariate).
}

\usage{
effectplot(cross, pheno.col=1, mname1, mark1, geno1, mname2, mark2,
           geno2, main, ylim, add.legend=TRUE)
}

\arguments{
  \item{cross}{An object of class \code{cross}.}
  \item{pheno.col}{Column number in the phenotype matrix which to be
    drawn in the plot.}
  \item{mname1}{Name for the first marker.}
  \item{mark1}{Genotype data for the first marker.  If unspecified,
    genotypes will be taken from the data in the input cross object,
    using the name specified in \code{mname1}.}
  \item{geno1}{Names for the genotypes (or classes in a covariate).}
  \item{mname2}{Name for the second marker (optional).}
  \item{mark2}{Like \code{mark1} (optional).}
  \item{geno2}{Names for the genotypes (or classes in a covariate).}
  \item{main}{Optional figure title.}
  \item{ylim}{Y-axis limits.}
  \item{add.legend}{A logical value to indicate add legend or not.}
}

\value{
  A data.frame containing the phenotype means and standard deviations
  for each group.
}

\details{
  In the plot, the Y-axis is the phenotype. X-axis is the genotype for
  the first marker.  In the case of one marker, there will be one line
  in the figure. In the case of two markers, the x-axis is for different
  genotypes of the second marker, and the first marker is represented by
  lines in different colors.  Error bars are plotted at \eqn{\pm}{+/-} 1
  SE. 
}

\examples{
data(fake.f2)

# plot of genotype-specific phenotype means for 1 marker
mname <- find.marker(fake.f2, 1, 37) # marker D1M437
effectplot(fake.f2, pheno.col=1, mname1=mname)

# Plot of sex-specific phenotype means
sex <- fake.f2$pheno[,2]
effectplot(fake.f2, mname1="Sex", mark1=sex, geno1=c("F","M"))

# plot of genotype-specific phenotype means for 2 markers
mname1 <- find.marker(fake.f2, 1, 37) # marker D1M437
mname2 <- find.marker(fake.f2, 13, 24) # marker D13M254
effectplot(fake.f2, mname1=mname1, mname2=mname2)

# Plot of sex- and genotype-specific phenotype means 
mname <- find.marker(fake.f2, 13, 24) # marker D13M254
effectplot(fake.f2, mname1=mname, mname2="Sex",
           mark2=fake.f2$pheno$sex, geno2=c("F","M"))

# Same as above, switch role of sex and the marker
effectplot(fake.f2, mname1="Sex", mark1=fake.f2$pheno$sex,
           geno1=c("F","M"), mname2=mname)

# X chromosome marker
mname <- find.marker(fake.f2, "X", 14) # marker DXM66
effectplot(fake.f2, mname1=mname)

# Two markers, including one on the X
mname1 <- find.marker(fake.f2, 13, 24) # marker D13M254
mname2 <- find.marker(fake.f2, "X", 14) # marker DXM66
effectplot(fake.f2, mname1=mname1, mname2=mname2)
}

\author{Hao Wu, \email{hao@jax.org}; Karl W Broman, \email{kbroman@jhsph.edu} }

\seealso{ \code{\link[qtl]{plot.pxg}}, \code{\link[qtl]{find.marker}},
  \code{\link[qtl]{effectscan}}} 

\keyword{hplot}

\eof
\name{effectscan}
\alias{effectscan}

\title{Plot allelic effect across the whole genome}

\description{
  This function is used to plot allelic effect at all markers for
  selected chromosomes. For backcross, there will be only one line
  represents the additive effect. For F2, there will be two lines
  represent additive and dominance effects. 
}

\usage{
effectscan(cross, pheno.col=1, chr, ylim, gap=25,
           col=c("black","blue","red"), lty=c(1,2,3), lwd=2,
           mtick=c("line", "triangle"), main, add.legend=TRUE,
           \dots)
}

\arguments{
  \item{cross}{An object of class \code{cross}.}
  \item{pheno.col}{Column number in the phenotype matrix which to be
    drawn in the plot.}
  \item{chr}{Chromosome number to be drawn in the plot.}
  \item{ylim}{y-axis limit.}
  \item{gap}{Gap separating chromosomes (in cM).}
  \item{col}{Line colors.}
  \item{lty}{Line types.}
  \item{lwd}{Line widths.}
  \item{mtick}{Tick mark type for markers. "line" for line and
    "triangle" for upward triangles.}
  \item{main}{Figure title.}
  \item{add.legend}{A logical value to indicate add legend or not.}
  \item{\dots}{Passed to the function \code{\link[base]{plot}} when it
    is called.}
}
\value{None.}

\examples{
data(fake.f2)
# allelic effect on whole genome
effectscan(fake.f2, ylim=c(-6,3))
# on first 4 chromosomes, use triangle as marker ticks
effectscan(fake.f2, chr=1:4, ylim=c(-6,3), mtick="triangle")

}

\author{Hao Wu, \email{hao@jax.org}}

\seealso{ \code{\link[qtl]{effectplot}}, \code{\link[qtl]{plot.pxg}} }

\keyword{hplot}

\eof
\name{est.map}
\alias{est.map}

\title{Estimate genetic maps}

\description{
  Uses the Lander-Green algorithm (i.e., the hidden Markov model
  technology) to re-estimate the genetic map for an experimental cross.
}

\usage{
est.map(cross, error.prob=0,
        map.function=c("haldane","kosambi","c-f","morgan"), maxit=4000,
        tol=1e-4, sex.sp=TRUE, trace=FALSE)
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi,
   Carter-Falconer, or Morgan map function when converting genetic
   distances into recombination fractions.}
 \item{maxit}{Maximum number of EM iterations to perform.}
 \item{tol}{Tolerance for determining convergence.}
 \item{sex.sp}{Indicates whether to estimate sex-specific maps; this is 
 used only for the 4-way cross.}
 \item{trace}{Logical; indicates whether to print initial and final
   estimates of the recombination fractions for each chromosome.}
}

\value{
  A \code{map} object; a list whose components (corresponding to
  chromosomes) are either vectors of marker positions (in cM) or
  matrices with two rows of sex-specific marker positions.
  The maximized log likelihood for each chromosome is saved as an
  attribute named \code{loglik}.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
newmap <- est.map(fake.f2)
logliks <- sapply(newmap, attr, "loglik")
plot.map(fake.f2, newmap)
fake.f2 <- replace.map(fake.f2, newmap)
}

\references{
  Lange, K. (1999) \emph{Numerical analysis for statisticians}.
  Springer-Verlag. Sec 23.3.

  Rabiner, L. R. (1989) A tutorial on hidden Markov models and selected
  applications in speech recognition.  \emph{Proceedings of the IEEE}
  \bold{77}, 257--286.

  Lander, E. S. and Green, P. (1987) Construction of multilocus genetic linkage
  maps in humans.  \emph{Proc. Natl. Acad. Sci. USA} \bold{84}, 2363--2367.
}  

\seealso{ \code{\link[qtl]{plot.map}}, \code{\link[qtl]{replace.map}},
  \code{\link[qtl]{est.rf}} }

\keyword{utilities}

\eof
\name{est.rf}
\alias{est.rf}

\title{Estimate pairwise recombination fractions}

\description{
  Estimate the sex-averaged recombination fraction between all pairs of
  genetic markers.
}

\usage{
est.rf(cross, maxit=4000, tol=1e-4)
}

\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{maxit}{Maximum number of iterations for the EM algorithm (not
   used with backcrosses).}
 \item{tol}{Tolerance for determining convergence (not used with
   backcrosses).} 
}

\value{
  The input \code{cross} object is returned with a component, \code{rf},
  added.  This is a matrix of size (tot.mar x tot.mar).  The diagonal
  contains the number of typed meioses per marker, the lower triangle
  contains the estimated recombination fractions, and the upper triangle
  contains the LOD scores (testing rf = 0.5).
}

\details{
  For a backcross, one can simply count recombination events.  For an
  intercross or 4-way cross, a version of the EM algorithm must be used
  to estimate recombination fractions. (Since, for example, in an
  intercross individual that is heterozygous at two loci, it is not
  known whether it shows 0 or 2 recombinations.)  Note that, for the
  4-way cross, we estimate sex-averaged recombination fractions.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(badorder)
badorder <- est.rf(badorder)
plot.rf(badorder)
}

\seealso{ \code{\link[qtl]{plot.rf}}, \code{\link[qtl]{est.map}},
  \code{\link[qtl]{badorder}} }

\keyword{utilities}

\eof
\name{fake.4way}
\alias{fake.4way}
\non_function{}

\title{Simulated data for a 4-way cross}

\description{ 
  Simulated data for a phase-known 4-way cross, obtained using
  \code{\link[qtl]{sim.cross}}.
} 

\usage{data(fake.4way)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details.  
}

\details{
There are 250 individuals typed at 157 markers, including 8 on the X
chromosome.  

There are two phenotypes (including sex, for which 0=female and 1=male).
The quantitative phenotype is affected by three QTLs: two on chromosome
2 at positions 10 and 25 cM on the female genetic map, and one on
chromosome 7 at position 40 cM on the female map. 
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.4way)
plot(fake.4way)
summary(fake.4way)

# estimate recombination fractions
fake.4way <- est.rf(fake.4way)
plot.rf(fake.4way)

# estimate genetic maps
ssmap <- est.map(fake.4way, trace=TRUE)
samap <- est.map(fake.4way, sex.sp=FALSE, trace=TRUE)
plot(ssmap, samap)

# error lod scores
fake.4way <- calc.genoprob(fake.4way, err=0.01)
fake.4way <- calc.errorlod(fake.4way, err=0.01)
top.errorlod(fake.4way, cutoff=2.5)

# genome scan
fake.4way <- calc.genoprob(fake.4way, step=2.5)
out.hk <- scanone(fake.4way, method="hk")
out.em <- scanone(fake.4way, method="em")
plot(out.em,out.hk,chr=c(2,7))
}

\seealso{ \code{\link[qtl]{sim.cross}}, \code{\link[qtl]{fake.bc}},
  \code{\link[qtl]{fake.f2}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{hyper}},
  \code{\link[qtl]{bristle3}}, \code{\link[qtl]{bristleX}} }

\keyword{datasets}

\eof
\name{fake.bc}
\alias{fake.bc}
\non_function{}

\title{Simulated data for a backcross}

\description{ 
  Simulated data for a backcross, obtained using
  \code{\link[qtl]{sim.cross}}.   
} 

\usage{data(fake.bc)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}


\details{
  There are 400 backcross individuals typed at 91 markers and with two
  phenotype and two covariates (sex and age).  

  The two phenotype are due to four QTLs, with no epistasis.  There is
  one on chromosome 2 (at 30 cM), two on chromosome 5 (at 10 and 50 cM),
  and one on chromosome 10 (at 30 cM).  The QTL on chromosome 2 has an
  effect only in the males (sex=1); the two QTLs on chromosome 5 have
  effect in coupling for the first phenotype and in repulsion for the
  second phenotype. Age has an effect of increasing the phenotypes.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.bc)
summary(fake.bc)
plot(fake.bc)

# genome scans without covariates
fake.bc <- calc.genoprob(fake.bc, step=2.5)
out1.nocovar <- scanone(fake.bc, pheno.col=1)
out2.nocovar <- scanone(fake.bc, pheno.col=2)

# genome scans with covariates
ac <- fake.bc$pheno[,c("sex","age")]
ic <- fake.bc$pheno[,"sex"]
out1.covar <- scanone(fake.bc, pheno.col=1,
                      addcovar=ac, intcovar=ic)
out2.covar <- scanone(fake.bc, pheno.col=2,
                      addcovar=ac, intcovar=ic)

# summaries
summary(out1.nocovar, 3)
summary(out1.covar, 3)
summary(out2.nocovar, 3)
summary(out2.covar, 3)

# plots
plot(out1.covar,out1.nocovar,chr=c(2,5,10))
plot(out2.covar,out2.nocovar,chr=c(2,5,10))
}

\seealso{ \code{\link[qtl]{sim.cross}}, \code{\link[qtl]{fake.4way}},
  \code{\link[qtl]{fake.f2}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{hyper}},
  \code{\link[qtl]{bristle3}}, \code{\link[qtl]{bristleX}} }

\keyword{datasets}

\eof
\name{fake.f2}
\alias{fake.f2}
\non_function{}

\title{Simulated data for an F2 intercross}

\description{ 
  Simulated data for an F2 intercross, obtained using
  \code{\link[qtl]{sim.cross}}.   
} 

\usage{data(fake.f2)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}

\details{
There are 200 F2 individuals typed at 94 markers, including 3 on the X
chromosome.  There is one quantitative phenotype, along with an
indication of sex (0=female, 1=male) and the direction of the cross (pgm
= paternal grandmother, 0=A, meaning the cross was (AxB)x(AxB), and
1=B, meaning the cross was (AxB)x(BxA)).

Note that the X chromosome genotypes are coded in a special way (see
\code{\link[qtl]{read.cross}}).  For the individuals with pgm=0, sex=0,
1=AA and 2=AB; for individuals with pgm=0, sex=1, 1=A and 2=B
(hemizygous); for individuals with pgm=1, sex=0, 1=BB and 2=AB; for
individuals with pgm=1, sex=1, 1=A and 2=B.  \bold{This requires special
care!}

The data were simulated using an additive model with three QTLs on
chromosome 1 (at 30, 50 and 70 cM), one QTL on chromosome 13 (at 30
cM), and one QTL on the X chromosome (at 10 cM). 
}


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
summary(fake.f2)
plot(fake.f2)
}

\seealso{ \code{\link[qtl]{sim.cross}}, \code{\link[qtl]{fake.bc}},
  \code{\link[qtl]{fake.4way}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{hyper}},
  \code{\link[qtl]{bristle3}}, \code{\link[qtl]{bristleX}} } 

\keyword{datasets}

\eof
\name{fill.geno}
\alias{fill.geno}

\title{Fill holes in genotype data}

\description{
  Replace the genotype data for a cross with a version imputed either
  by simulation with \code{\link[qtl]{sim.geno}} or by the Viterbi
  algorithm with \code{\link[qtl]{argmax.geno}}.
}

\usage{
fill.geno(cross, method=c("imp","argmax"), error.prob=0,
          map.function=c("haldane","kosambi","c-f","morgan"))
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{method}{Indicates whether to impute using a single simulation
   replicate from \code{\link[qtl]{sim.geno}} or using the Viterbi
   algorithm, as implemented in \code{\link[qtl]{argmax.geno}}.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
 of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi or
   Carter-Falconer map function when converting genetic distances into
   recombination fractions.} 
}

\details{
  This function is written so that one may perform rough genome scans by
  marker regression without having to drop individuals with missing
  genotype data.  \bold{We must caution the user that little trust
  should be placed in the results}. 
}

\value{
  The input \code{cross} object with the genotype data replaced by an
  imputed version.  Any intermediate calculations (such as is produced
  by \code{\link[qtl]{calc.genoprob}}, \code{\link[qtl]{argmax.geno}}
  and \code{\link[qtl]{sim.geno}}) are removed. 
}

\examples{
data(hyper)
out.mr <- scantwo(fill.geno(hyper,method="argmax"), method="mr")
plot(out.mr)
}

\seealso{ \code{\link[qtl]{clean}}, \code{\link[qtl]{sim.geno}},
          \code{\link[qtl]{argmax.geno}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{utilities}

\eof
\name{find.marker}
\alias{find.marker}

\title{Find marker closest to a specified position}

\description{
  Find the genetic marker closest to a specified position on a
  chromosome. 
}

\usage{find.marker(cross, chr, pos)}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{A vector of chromosome identifiers, or a single such.}
 \item{pos}{A vector of cM positions.}
}

\value{
  A vector of marker names (of the same length as the input \code{pos}),
  corresponding to the markers nearest to the specified
  chromosomes/positions. 
}

\details{
  If the input \code{chr} has length one, it is expanded to the same
  length as the input \code{pos}.

  If multiple markers are exactly the same distance from the specified
  position, one is chosen at random from among those with the most
  genotype data.

  For a cross with sex-specific maps, the input positions are assumed to
  correspond to the female genetic map.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
find.marker(listeria, 5, 28)
find.marker(listeria, c(1, 5, 13), c(81, 28, 26))
}

\seealso{ \code{\link[qtl]{plot.pxg}} }

\keyword{manip}

\eof
\name{fitqtl}
\alias{fitqtl}

\title{Fit a multiple QTL model}

\description{
  Fits a user-specified multiple QTL model.  If specified, 
  a drop-one-term analysis will be performed.}
}

\usage{
fitqtl(pheno, qtl, covar=NULL, formula, method=c("imp"),
       dropone=TRUE)
}
\arguments{
 \item{pheno}{Phenotype data (a numeric vector).}
 \item{qtl}{An object of class \code{qtl}, as output from
   \code{\link[qtl]{makeqtl}}).} 
 \item{covar}{A data.frame of covariates}
 \item{formula}{An object of class \code{\link[base]{formula}}
   indicating the model to be fitted.  QTLs are referred to as
   \code{Q1}, \code{Q2}, etc.  Covariates are referred to by their names
   in the data frame \code{covar}.} 
 \item{method}{Indicates whether to use the EM algorithm or  
   imputation. (Only imputation is implemented at this point.)}
 \item{dropone}{If TRUE, do drop-one-term analysis.}
}

\details{
  In the drop-one-term analysis, for a given QTL/covariate model, all
  submodels will be analyzed.  For each term in the input formula, when
  it is dropped, all higher order terms that contain it will also be
  dropped.  The comparison between the new model and the full (input)
  model will be output.
}

\section{Value}{
  An object of class \code{fitqtl}. It may contains two fields:
  \enumerate{
    \item result.full is the ANOVA table as a matrix for the full model
    result. It contains the degree of freedom (df), Sum of squares (SS),
    mean square (MS), LOD score (LOD), percentage of variance explained
    (\%var) and P value (Pvalue).
    \item result.drop is a drop-one-term ANOVA table as a matrix. It
    contains degrees of freedom (df), Type III sum of squares (Type III
    SS), LOD score(LOD), percentage of variance explained (\%var), F
    statistics (F value),  and P values for chi square (Pvalue(chi2))
    and F distribution (Pvalue(F)).
    
    Note that the degree of freedom, Type III sum of squares, the LOD
    score and the percentage of variance explained are the values
    comparing the full to the sub-model with the term dropped. Also note
    that for imputation method, the percentage of variance explained, the
    the F values and the P values are approximations calculated from the
    LOD score.
  }
}

\references{
  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.
}

\author{Hao Wu, \email{hao@jax.org} }

\examples{
data(fake.f2)

# take out several QTLs and make QTL object
qc <- c(1, 8, 13)
qp <- c(26, 56, 28)
fake.f2 <- subset(fake.f2, chr=qc)
fake.f2 <- sim.geno(fake.f2, n.draws=64, step=2, err=0.001)
qtl <- makeqtl(fake.f2, qc, qp)

# fit model with 3 interacting QTLs interacting
# (performing a drop-one-term analysis)
lod <- fitqtl(fake.f2$pheno[,1], qtl, formula=y~Q1*Q2*Q3)
summary(lod)

# fit an additive QTL model
lod.add <- fitqtl(fake.f2$pheno[,1], qtl, formula=y~Q1+Q2+Q3)
summary(lod.add)

# fit the model including sex as an interacting covariate
Sex <- data.frame(Sex=fake.f2$pheno$sex)
lod.sex <- fitqtl(fake.f2$pheno[,1], qtl, formula=y~Q1*Q2*Q3*Sex, cov=Sex)
summary(lod.sex)

# fit the same with an additive model
lod.sex.add <- fitqtl(fake.f2$pheno[,1], qtl, formula=y~Q1+Q2+Q3+Sex, cov=Sex)
summary(lod.sex.add)
}

\seealso{ \code{\link[qtl]{summary.fitqtl}}, \code{\link[qtl]{makeqtl}},
\code{\link[qtl]{scanqtl}} }  

\keyword{models}

\eof
\name{geno.table}
\alias{geno.table}

\title{Create table of genotype distributions}

\description{
  Create table showing the observed numbers of individuals with each
  genotype at each marker, including P-values from a chi-square tests
  for Mendelian segregation.
}

\usage{
geno.table(cross)
}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
}

\value{
  A matrix containing, for each marker, the number of individuals with
  each possible genotype, as well as the number that were not typed.
  The last column gives P-values from chi-square tests of Mendelian
  segregation.  
}

\examples{
data(listeria)
geno.table(listeria)
}

\seealso{ \code{\link[qtl]{summary.cross}},
  \code{\link[qtl]{drop.markers}}, \code{\link[qtl]{drop.nullmarkers}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{utilities}

\eof
\name{hyper}
\alias{hyper}
\non_function{}

\title{Data on hypertension}

\description{ 
  Data from an experiment on hypertension in the mouse.
} 

\usage{data(hyper)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}  

\details{
There are 250 backcross individuals typed at 174 markers (actually one
contains only missing values), including 4 on the X chromosome, with one
phenotype. 

The phenotype is the blood pressure.  See the reference below.  Note
that, for most markers, genotypes are available on only the individuals
with extreme phenotypes.  At many markers, only recombinant individuals
were typed.
}


\references{
  Sugiyama, F., Churchill, G. A., Higgens, D. C., Johns, C.,
  Makaritsis, K. P., Gavras, H. and Paigen, B. (2001) Concordance of
  murine quantitative trait loci for salt-induced hypertension with rat
  and human loci. \emph{Genomics} \bold{71}, 70--77. 
}
  
\source{
  Bev Paigen and Gary Churchill (The Jackson Laboratory, Bar Harbor,
  Maine)
  \url{http://www.jax.org/research/churchill/datasets/qtl/qtlarchive}   
}

\examples{
data(hyper)
summary(hyper)
plot(hyper)

# Note the selective genotyping
plot.missing(hyper, reorder=TRUE)

# A marker on c14 has no data; remove it
hyper <- drop.nullmarkers(hyper)

# Genome scan with normal model
hyper <- calc.genoprob(hyper, step=2.5)
out <- scanone(hyper)
summary(out,3)

# Non-parametric genome scan
out.np <- scanone(hyper, model="np")
summary(out.np,3)
plot(out.np,out,chr=c(1,4),lty=1,col=c("blue","black"))

# Two-dimensional genome scan, using Haley-Knott regression
# Here we just do c1 and c4, and on a more coarse grid.
hyper.coarse <- calc.genoprob(hyper, step=5)
out2 <- scantwo(hyper.coarse, method="hk", chr=c(1,4))
plot(out2)
summary(out2,c(8,0,0))
}

\seealso{ \code{\link[qtl]{fake.bc}}, \code{\link[qtl]{fake.f2}},
  \code{\link[qtl]{fake.4way}}, \code{\link[qtl]{listeria}},
  \code{\link[qtl]{bristle3}}, \code{\link[qtl]{bristleX}} } 

\keyword{datasets}

\eof
\name{listeria}
\alias{listeria}
\non_function{}

\title{Data on Listeria monocytogenes susceptibility}

\description{ 
  Data from an experiment on susceptibility to \emph{Listeria
  monocytogenes} infection in the mouse.  
} 

\usage{data(listeria)}

\format{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}}
  for details. 
}

\details{
  There are 120 F2 individuals typed at 133 markers, including 2 on the
  X chromosome, with one phenotype.

  The phenotype is the survival time (in hours) following infection.
  Mice with phenotype 264 hours may be considered to have recovered
  from the infection. See the references below.  
}

\references{
  Boyartchuk, V. L., Broman, K. W., Mosher, R. E., D'Orazio
  S. E. F., Starnbach, M. N. and Dietrich, W. F. (2001) Multigenic
  control of \emph{Listeria monocytogenes} susceptibility in
  mice. \emph{Nature Genetics} \bold{27}, 259--260.
  
  Broman, K. W., Boyartchuk, V. L. and Dietrich, W. F. (2000) Mapping
  time-to-death quantitative trait loci in a mouse cross with high
  survival rates. Technical Report MS00-04, Department of Biostatistics,
  Johns Hopkins University, Baltimore, MD.
}
  
\source{
  Victor Boyartchuk and William Dietrich (Department of Genetics,
  Harvard Medical School and Howard Hughes Medical Institute)
  \url{http://genetics.med.harvard.edu/~bdlab}
}

\examples{
data(listeria)
# Summaries
summary(listeria)
plot(listeria)

# Take log of phenotype
listeria$pheno[,2] <- log2(listeria$pheno[,1])
names(listeria$pheno)[2] <- "logSurv"
plot(listeria)

# Genome scan with a two-part model, using log survival
listeria <- calc.genoprob(listeria, step=2)
out <- scanone(listeria, model="2part", method="em",
               upper=TRUE, pheno.col=2)

# Summary of the results
summary(out, 4)

# Plot LOD curves for interesting chromosomes
#     (The two-part model gives three LOD scores)
plot(out, out, out, chr=c(1,5,6,13,15), lodcolumn=3:5,
     lty=1, col=c("black","red","blue"))
}

\seealso{ \code{\link[qtl]{fake.bc}}, \code{\link[qtl]{fake.f2}},
  \code{\link[qtl]{fake.4way}}, \code{\link[qtl]{hyper}},
  \code{\link[qtl]{bristle3}}, \code{\link[qtl]{bristleX}} }

\keyword{datasets}

\eof
\name{lodint}
\alias{lodint}

\title{LOD support interval}

\description{
  Calculate a LOD support interval for a particular chromosomes.
}

\usage{lodint(results, chr, drop=1.5)}

\arguments{
 \item{results}{Output from \code{\link[qtl]{scanone}}.}
 \item{chr}{A chromosome ID.}
 \item{drop}{LOD units to drop to form the interval.}
}

\value{
  An object of class \code{scanone}, like the input, indicating the
  position with the maximum LOD, and indicating approximate endpoints
  for the LOD support interval.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
listeria <- subset(listeria,chr=c(5,13))
listeria <- calc.genoprob(listeria, step=0.5)
out <- scanone(listeria, model="np")
lodint(out, 5, 1.5)
lodint(out, 13, 1.5)
}

\seealso{ \code{\link[qtl]{scanone}} }

\keyword{manip}

\eof
\name{makeqtl}
\alias{makeqtl}

\title{Make a qtl object}

\description{
  This function takes a cross object and specified chromosome numbers
  and positions and pulls out the genotype probabilities and/or imputed
  genotypes at the nearest pseudomarkers, for later use by the function
  \code{\link[qtl]{fitqtl}}. 
}

\usage{
makeqtl(cross, chr, pos, qtl.name)
}

\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{Vector indicating the chromosome for each QTL.}
 \item{pos}{Vector (of same length as \code{chr}) indicating the
   positions on the chromosome to be taken. If there's no marker or
   pseudomarker at a position, genotypes for the nearest positions are
   taken.} 
 \item{qtl.name}{The user-specified name for each QTL, used in the
   drop-one-term ANOVA table in \code{\link[qtl]{fitqtl}}.
   If unspecified, the names will be of the form \code{"Chr1@10"} for a
   QTL on Chromsome 1 at 10 cM.} 
}

\details{
  This function will take out the genotype probabilities and imputed
  genotypes if they are present in the input \code{cross} object. If both
  fields are missing in the input object, the function will report an
  error. Before running this function, the user must have first run either
  \code{\link[qtl]{calc.geno}} or \code{\link[qtl]{sim.geno}}.
}

\value{
  An object of class \code{qtl} with the following elements:
  \item{geno}{Imputed genotypes.}
  \item{prob}{Genotype probabilities.}
  \item{chr}{Input vector of chromosome numbers.}
  \item{pos}{Input vector of chromosome positions.}
  \item{n.qtl}{Number of QTLs.}
  \item{n.ind}{Number of individuals.}
  \item{n.gen}{A vector indicating the number of genotypes for each QTL.}
}

\references{
  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.
}

\author{Hao Wu, \email{hao@jax.org} }

\examples{
data(fake.f2)

# take out several QTLs and make QTL object
qc <- c(1, 6, 13)
qp <- c(25.8, 33.6, 18.63)
fake.f2 <- subset(fake.f2, chr=qc)
fake.f2 <- sim.geno(fake.f2, n.draws=64, step=2, err=0.001)
qtl <- makeqtl(fake.f2, qc, qp)
}

\seealso{ \code{\link[qtl]{fitqtl}}, \code{\link[qtl]{calc.genoprob}},
  \code{\link[qtl]{sim.geno}} } 

\keyword{models}

\eof
\name{map10}
\alias{map10}
\non_function{}

\title{An example genetic map}

\description{ 
  A genetic map corresponding approximately to the mouse genome with
  a 10 cM marker spacing.
} 

\usage{data(map10)}

\format{
  An object of class \code{map}: a list whose components are vectors of
  marker locations.  This map approximates the mouse genome (see Rowe et
  al 1992), with 20 chromosomes (including the X 
  chromosome) and 148 markers at a 10 cM spacing.
}

\examples{
  data(map10)
  plot(map10)

  mycross <- sim.cross(map10, type="f2", n.ind=100)
}    

\seealso{ \code{\link[qtl]{sim.map}}, \code{\link[qtl]{plot.map}},
  \code{\link[qtl]{pull.map}} }

\keyword{datasets}

\eof
\name{max.scanone}
\alias{max.scanone}

\title{Maximum peak in genome scan}

\description{
  Print the row of the output from \code{\link[qtl]{scanone}} that
  corresponds to the maximum LOD, genome-wide.
}

\usage{
max.scanone(\dots, chr, na.rm=TRUE)
}

\arguments{
 \item{\dots}{An object of the form output by the function
   \code{\link[qtl]{scanone}}: a data.frame whose third column is the
   LOD score.  Note that while multiple objects may be input, only the
   first is considered.}
 \item{chr}{Numeric or character vector indicating chromosomes
   on which to find the maximum LOD score.}
 \item{na.rm}{A logical indicating whether missing values should be
   removed.}
}

\value{
  An object of class \code{summary.scanone}, to be printed by
  \code{print.summary.scanone}.  This is a data.frame with one row,
  corresponding to the maximum LOD peak either genome-wide or for the
  particular chromosome specified.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
listeria <- calc.genoprob(listeria, step=2.5)
out <- scanone(listeria, model="2part")
# Maximum peak for LOD(p,mu)
max(out)
max(out,chr=13)

# Maximum peak for LOD(p)
max(out[,-3])

# Maximum peak for LOD(mu)
max(out[,-(3:4)])
}

\seealso{ \code{\link[qtl]{scanone}}, \code{\link[qtl]{plot.scanone}},
          \code{\link[qtl]{summary.scanone}} }

\keyword{print}

\eof
\name{max.scantwo}
\alias{max.scantwo}

\title{Maximum peak in two-dimensional genome scan}

\description{
  Print the maximum joint LOD and interaction LODs in the results of
  \code{scantwo}. 
}

\usage{
max.scantwo(\dots, na.rm=TRUE)
}

\arguments{
 \item{\dots}{An object of class \code{scantwo}, the output of the
   function \code{\link[qtl]{scantwo}}.  Note that while multiple
   objects may be input, only the first is considered.}
 \item{na.rm}{A logical indicating whether missing values should be
   removed.}
}

\value{
  An object of class \code{summary.scantwo}, to be printed by
  \code{print.summary.scantwo}.  Pairs of loci that have the maximum
  joint and interaction LOD, in the two-dimensional scan.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} } 

\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2, step=5)
out.2dim <- scantwo(fake.f2, method="hk", chr=c(1,6,13))
max(out.2dim)
}

\seealso{ \code{\link[qtl]{scantwo}}, \code{\link[qtl]{plot.scantwo}},
          \code{\link[qtl]{summary.scantwo}} }

\keyword{print}

\eof
\name{nmissing}
\alias{nmissing}

\title{Number of missing genotypes}

\description{
  Count number of missing genotypes for each individual or each marker
  in a cross.
}

\usage{nmissing(cross, which=c("ind","mar"))}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{which}{Indicates whether to count missing genotypes for each
   individual or each marker.}
}

\value{
  A vector containing the number of missing genotypes for each
  individual or for each marker.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)

# plot number of missing genotypes for each individual
plot(nmissing(listeria))

# plot number of missing genotypes for each marker
plot(nmissing(listeria, which="mar"))
}

\seealso{ \code{\link[qtl]{summary.cross}},
          \code{\link[qtl]{nind}}, \code{\link[qtl]{totmar}} }

\keyword{manip}

\eof
\name{plot.cross}
\alias{plot.cross}

\title{Plot various features of a cross object}

\description{
  Plots grid of the missing genotypes, genetic map, and histograms or
  barplots of phenotypes for the data from an experimental cross.
}

\usage{
plot.cross(x, auto.layout=TRUE, \dots)
}

\arguments{
 \item{x}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{auto.layout}{If TRUE, \code{par(mfrow)} is set so that all plots
   fit within one figure.}
 \item{\dots}{Ignored at this point.}
}

\value{None.}

\details{
  Calls \code{\link[qtl]{plot.missing}}, \code{\link[qtl]{plot.map}} and
  \code{\link[base]{hist}} or \code{\link[base]{barplot}} to plot the
  missing genotypes, genetic map, and histograms or barplots of all
  phenotypes. 

  If \code{auto.format=TRUE}, \code{par(mfrow)} is used with
  \code{ceiling(sqrt(n.phe+2))} rows and the minimum number of columns
  so that all plots fit on the plotting device.

  Numeric phenotypes are displayed as histograms with approximately
  \eqn{\sqrt{n}+5}{sqrt(n)+5} bins.  Phenotypes that are factors are
  displayed as barplots by calling \code{barplot(table(phe))}.
}    

\examples{
data(fake.bc)
plot(fake.bc)
}

\seealso{ \code{\link[qtl]{plot.missing}}, \code{\link[qtl]{plot.map}}, 
  \code{\link[base]{hist}}, \code{\link[base]{barplot}},
  \code{\link[base]{table}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.errorlod}
\alias{plot.errorlod}

\title{Plot grid of error LOD values}

\description{
  Plot a grid of the LOD scores indicating which genotypes are
  likely to be in error.
}

\usage{
plot.errorlod(x, chr, ind, breaks=c(-1,2,3,4.5,Inf),
              col=c("white","gray85","hotpink","purple3"), \dots)
}

\arguments{
  \item{x}{An object of class \code{cross}.  See
    \code{\link[qtl]{read.cross}} for details.}
  \item{chr}{The chromosomes for which the error LOD scores should
    be plotted.}
  \item{ind}{Indicates the individuals for which the error LOD scores
    should be plotted.}
  \item{breaks}{A set of breakpoints for the colors; must give one more
    breakpoint than color.  Intervals are open on the left and closed on
    the right, except for the lowest interval.}
  \item{col}{A vector of colors to appear in the image.}
  \item{\dots}{Ignored at this point.}
}

\details{
  Uses \code{\link[base]{image}} to plot a grid with different shades
  of pixels to indicate which genotypes are likely to be in error.

  Darker pixels have higher error LOD scores:
  \eqn{LOD \le 2} in white;
  \eqn{2 < LOD \le 3}{2 < LOD <= 3} in gray;
  \eqn{3 < LOD \le 4.5}{3 < LOD <= 4.5} in pink;
  \eqn{LOD > 4.5} in purple.
}

\value{None.}

\examples{
data(hyper)
# run calc.genoprob, then calc.errorlod; use same error.prob
hyper <- calc.genoprob(hyper,error.prob=0.01)
hyper <- calc.errorlod(hyper,error.prob=0.01)

# plot the error LOD scores; print those above a specified cutoff 
plot.errorlod(hyper)
plot.errorlod(hyper,chr=1)
top.errorlod(hyper,cutoff=4)

# plot genotype data, flagging genotypes with error LOD > cutoff
plot.geno(hyper, chr=1, ind=160:200, cutoff=7, min.sep=2)

# Note that if markers are extremely tightly linked,
#     any recombination event will have a high error LOD
plot.geno(hyper, chr=1, ind=101:118, min.sep=2)
hyper$geno[[1]]$map
}

\seealso{ \code{\link[qtl]{calc.errorlod}},
  \code{\link[qtl]{top.errorlod}}, \code{\link[base]{image}} }

\references{
  Lincoln, S. E. and Lander, E. S. (1992) Systematic detection of
  errors in genetic linkage data.  \emph{Genomics} \bold{14}, 604--610.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.geno}
\alias{plot.geno}

\title{Plot observed genotypes, flagging likely errors}

\description{
  Plot the genotypes on a particular chromosome for a set of
  individuals, flagging likely errors.
}

\usage{
plot.geno(x, chr, ind, horizontal=FALSE, cutoff=3.5,
          min.sep=2, cex=1.2, \dots) 
}			    

\arguments{
  \item{x}{An object of class \code{cross}.  See
    \code{\link[qtl]{read.cross}} for details. }
  \item{chr}{Chromosome number to plot.}
  \item{ind}{Vector of individuals to plot.  If missing, all individuals
    are plotted.}
  \item{horizontal}{If TRUE, chromosomes are plotted horizontally.}
  \item{cutoff}{Genotypes with error LOD scores above this value are
    flagged as possible errors.}
  \item{min.sep}{Markers separated by less than this value (as a
    percent of the chromosome length) are pulled apart, so that they may
    be distinguished in the picture.}
  \item{cex}{Character expansion for the size of points in the plot.
    Larger numbers give larger points; see \code{\link[qtl]{par}}.}
  \item{\dots}{Ignored at this point.}
}

\details{
  Plots the genotypes for a set of individuals.  Likely errors are
  indicated by red squares.  In a backcross, genotypes AA and AB are
  indicated by white and black circles, respectively.  In an intercross,
  genotypes AA, AB and BB are indicated by white, gray, and black
  circles, respectively, and the partially missing genotypes ``not BB''
  (D in mapmaker) and ``not AA'' (C in mapmaker) are indicated by green
  and orange circles, respectively.
}

\value{None.}

\examples{
data(hyper)
# run calc.genoprob, then calc.errorlod; use same error.prob
hyper <- calc.genoprob(hyper,error.prob=0.01)
hyper <- calc.errorlod(hyper,error.prob=0.01)

# plot the error LOD scores; print those above a specified cutoff 
plot.errorlod(hyper)
plot.errorlod(hyper,chr=1)
top.errorlod(hyper,cutoff=4)

# plot genotype data, flagging genotypes with error LOD > cutoff
plot.geno(hyper, chr=1, ind=160:200, cutoff=7, min.sep=2)

# Note that if markers are extremely tightly linked,
#     any recombination event will have a high error LOD
plot.geno(hyper, chr=1, ind=101:118, min.sep=2)
hyper$geno[[1]]$map
}

\seealso{ \code{\link[qtl]{calc.errorlod}},
  \code{\link[qtl]{plot.errorlod}}, \code{\link[qtl]{top.errorlod}},
  \code{\link[base]{image}} }


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.info}
\alias{plot.info}

\title{Plot the proportion of missing genotype information}

\description{
  Plot a measure of the proportion of missing information in the
  genotype data.  
}

\usage{
plot.info(x, chr, method=c("both","entropy","variance"), \dots)
}

\arguments{
 \item{x}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{Vector specifying the chromosomes to plot.}
 \item{method}{Indicates whether to plot the entropy version of the
   information, the variance version, or both.}
 \item{\dots}{Passed to \code{\link[qtl]{plot.scanone}}.}
}

\details{
  The missing information is calculated using the multipoint genotype
  probabilities calculated with \code{\link[qtl]{calc.genoprob}}.

  The entropy version of the missing information: for a single
  individual at a single genomic position, we measure the missing
  information as \eqn{H = \sum_g p_g \log p_g / \log n}{H = sum p[g] log
  p[g] / log n}, where \eqn{p_g}{p[g]} is the probability of the
  genotype \eqn{g}, and \eqn{n} is the number of possible genotypes,
  defining \eqn{0 \log 0 = 0}{0 log 0 = 0}.  This takes values between 0
  and 1, assuming the value 1 when the genotypes (given the marker data)
  are equally likely and 0 when the genotypes are completely determined.
  We calculate the missing information at a particular position as the
  average of \eqn{H} across individuals.  For an intercross, we don't
  scale by \eqn{\log n} but by the entropy in the case of genotype
  probabilities (1/4, 1/2, 1/4). 
  
  The variance version of the missing information: we calculate the
  average, across individuals, of the variance of the genotype
  distribution (conditional on the observed marker data) at a particular
  locus, and scale by the maximum such variance.

  Calculations are done in C (for the sake of speed in the presence of
  little thought about programming efficiency) and the plot is created
  by a call to \code{\link[qtl]{plot.scanone}}.

  Note that \code{\link[qtl]{summary.scanone}} may be used to display
  the maximum missing information on each chromosome.
}  

\value{
  An object with class \code{scanone}: a data.frame with columns the
  chromosome IDs and cM positions followed by the entropy and/or
  variance version of the missing information.
}

\examples{
data(hyper)
hyper <- calc.genoprob(hyper, step=2.5, off.end=5)
plot.info(hyper,chr=c(1,4,6,7,15))

# save the results and view maximum missing info on each chr
info <- plot.info(hyper)
summary(info)
}

\seealso{ \code{\link[qtl]{plot.scanone}},
  \code{\link[qtl]{plot.missing}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}
\keyword{univar}

\eof
\name{plot.map}
\alias{plot.map}

\title{Plot genetic map}

\description{
  Plot genetic map of marker locations for all chromosomes.
}

\usage{plot.map(x, map2, horizontal=FALSE, \dots)}

\arguments{
  \item{x}{A list whose components are vectors of marker locations.  A
    \code{cross} object may be given instead, in which case the genetic
    map it contains is used.}
  \item{map2}{An optional second genetic map with the same number of
    chromosomes and markers as the first.  As with the first argument, a
    \code{cross} object may be given instead.  If this argument is
    given, a comparison of the two genetic maps is plotted.}
  \item{horizontal}{Specifies whether the chromosomes should be
    plotted horizontally.}
  \item{\dots}{Ignored at this point.}
}

\value{None.}

\details{
  Plots the genetic map for each chromosome, or a comparison of the
  genetic maps if two maps are given.

  For a sex-specific map, female and male maps are plotted against one
  another.  For two sex-specific maps, female maps and male maps are
  plotted against one another.
}

\examples{
data(fake.bc)
plot.map(fake.bc)
plot.map(fake.bc,horizontal=TRUE)
newmap <- est.map(fake.bc)
plot(newmap)
plot.map(fake.bc, newmap)
}

\seealso{ \code{\link[qtl]{est.map}}, \code{\link[qtl]{plot.cross}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.missing}
\alias{plot.missing}

\title{Plot grid of missing genotypes}

\description{
  Plot a grid showing which genotypes are missing.
}

\usage{
plot.missing(x, chr, reorder=FALSE, main="Missing genotypes", \dots)
}			    

\arguments{
  \item{x}{An object of class \code{cross}.  See
    \code{\link[qtl]{read.cross}} for details. }
  \item{chr}{Vector of chromosomes to plot.  If missing, plot all
    chromosomes.}
  \item{reorder}{Specify whether to reorder individuals according to
  their phenotypes.
  \tabular{ll}{
    FALSE \tab Don't reorder \cr
    TRUE  \tab Reorder according to the sum of the phenotypes \cr
    n \tab Reorder according to phenotype n}}
  \item{main}{Title to place on plot.}
  \item{\dots}{Ignored at this point.}
}

\details{
  Uses \code{\link[base]{image}} to plot a grid with black pixels where the
  genotypes are missing.  For intercross and 4way cross data, gray
  pixels are plotted for the partially missing genotypes (for example,
  ``not A'').
}

\value{None.}

\examples{
data(fake.f2)
plot.missing(fake.f2)
}

\seealso{ \code{\link[qtl]{plot.cross}},  \code{\link[base]{image}} }


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.pxg}
\alias{plot.pxg}

\title{Plot phenotypes versus marker genotypes.}

\description{
  Plot the phenotype values versus the genotypes at a marker.
}

\usage{
plot.pxg(x, marker, pheno.col=1, jitter=1, infer=TRUE, \dots)
}			    

\arguments{
  \item{x}{An object of class \code{cross}.  See
    \code{\link[qtl]{read.cross}} for details. }
  \item{marker}{A marker name (a character string).}
  \item{pheno.col}{Column number in the phenotype matrix which should be
   used as the phenotype.}
  \item{jitter}{A positive number indicating how much to spread out the
    points horizontally.  (Larger numbers correspond to greater spread.)}
  \item{infer}{If TRUE, missing genotypes are filled in with a single
    random imputation and plotted in red; if FALSE, only individuals
    typed at the specified marker are plotted.}
  \item{\dots}{Ignored at this point.}
}

\details{
  Plots the phenotype data against the genotypes at the specified
  marker.  If infer=TRUE, the genotypes of individuals that were not
  typed is inferred based the genotypes at linked markers via a single
  imputation from \code{\link[qtl]{sim.geno}}; these points are plotted
  in red.  For each genotype, the phenotypic mean is plotted, with error
  bars at \eqn{\pm}{+/-} 1 SE.
}

\value{
  Returns a list with two components.  The first component is a nominal p-value
  corresponding to a test of a difference between the genotype-specific
  phenotypic means.  The second component is a matrix with three
  columns: the genotypes at the marker, the phenotypes, and a 0/1
  indicator with 1's corresponding to inferred genotypes.
}

\examples{
data(listeria)
mymarker <- find.marker(listeria, 5, 28) # marker D5M357
plot.pxg(listeria, mymarker)

data(fake.f2)
mname <- find.marker(fake.f2, 1, 37) # marker D1M437
plot.pxg(fake.f2, mname)

mname <- find.marker(fake.f2, "X", 14) # marker DXM66
plot.pxg(fake.f2, mname)
}

\seealso{ \code{\link[qtl]{find.marker}}, \code{\link[qtl]{effectplot}} }


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.rf}
\alias{plot.rf}

\title{Plot recombination fractions}

\description{
  Plot a grid showing the recombination fractions for all pairs of
  markers, or of the LOD scores between pairs of markers.
}

\usage{
plot.rf(x, chr, which=c("both","lod","rf"), \dots)
}

\arguments{
  \item{x}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
  \item{chr}{Vector specifying which chromosomes to plot
  (optional)}
  \item{which}{Indicate whether to plot LOD scores, recombination
               fractions or both.}
  \item{\dots}{Ignored at this point.}
}

\value{None.}

\details{
  Uses \code{\link[base]{image}} to plot a grid showing the
  recombination fractions and/or LOD scores for all pairs of markers.
  (The LOD scores are for a test of \eqn{r = 1/2}.)
  If both are plotted, the recombination fractions are in the upper left
  triangle while the LOD scores are in the lower right triangle.
  Red corresponds to a large LOD or a small recombination fraction,
  while blue is the reverse.  Note that missing values appear in light
  gray. 

  Recombination fractions are transformed by \eqn{-4(\log_2{r}+1)}{%
    -4(log2(r)+1)} to make them on the same sort of scale as LOD
  scores.  Values of LOD or the transformed recombination fraction that
  are above 12 are set to 12.
}


\examples{
data(badorder)
badorder <- est.rf(badorder)
plot.rf(badorder)

# plot just chr 1
plot.rf(badorder,chr=1)

# plot just the recombination fractions
plot.rf(badorder,which="rf") 

# plot just the LOD scores, and just for chr 2 and 3
plot.rf(badorder,which="lod")
}

\seealso{ \code{\link[qtl]{est.rf}}, \code{\link[base]{image}},
  \code{\link[qtl]{badorder}}, \code{\link[qtl]{ripple}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.scanone}
\alias{plot.scanone}

\title{Plot LOD curves}

\description{
  Plot the LOD curve for a genome scan with a single-QTL model (the
  output of \code{\link[qtl]{scanone}}).
}

\usage{
plot.scanone(x, x2, x3, chr, lodcolumn=3, incl.markers=TRUE,
             xlim, ylim, lty=c(1,2,3), col="black", lwd=2, add=FALSE,
             gap=25, main, mtick = c("line", "triangle"), \dots)
}			    

\arguments{
  \item{x}{A data.frame with at least three columns.  The first
    three columns should be the following (any further columns are
    ignored):} 
  \tabular{ll}{
     \code{chr} \tab Chromosome ID \cr
     \code{pos} \tab Position on genetic map \cr
     \code{lod}  \tab LOD score (or other statistic)}
  \item{x2}{data.frame for a second genome scan (optional)}
  \item{x3}{data.frame for a third genome scan (optional)}
  \item{chr}{Vector specifying which chromosomes to plot. (The
    chromosomes must be specified by name.)}
  \item{lodcolumn}{An integer, or vector of 3 integers, indicating which
    columns of the input should be plotted (generally column 3).}
  \item{incl.markers}{Indicate whether to plot line segments at the
  marker locations.}
  \item{xlim}{Limits for x-axis (optional).}
  \item{ylim}{Limits for y-axis (optional).}
  \item{lty}{Line types; a vector of length 1 or 3.}
  \item{col}{Line colors; a vector of length 1 or 3.}
  \item{lwd}{Line widths; a vector of length 1 or 3.}
  \item{add}{If TRUE, add to a current plot.}
  \item{gap}{Gap separating chromosomes (in cM).}
  \item{main}{An optional title for the plot.}
  \item{mtick}{Tick mark type for markers (line segments or
    upward-pointing triangels).}
  \item{\dots}{Passed to the function \code{\link[base]{plot}} when it
    is called.}
}
\value{None.}

\details{
  This function allows you to plot the results of up to three genome
  scans against one another.  Such objects must conform with each
  other. 

  One may alternatively use the argument \code{add} to add the plot of
  an additional genome scan to the current figure, but some care is
  required.  
}

\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2,step=2.5)
out.mr <- scanone(fake.f2, method="mr")
out.em <- scanone(fake.f2, chr=c(1,13), method="em")
plot(out.mr)
plot(out.mr, out.em, chr=c(1,13))
plot(out.mr, out.em, , c(1,13), lty=1, col=c("violetred","black"))
out.hk <- scanone(fake.f2, chr=c(1,13), method="hk")
plot(out.hk, chr=c(1,13), add=TRUE, col="slateblue")
}

\seealso{ \code{\link[qtl]{scanone}},
  \code{\link[qtl]{summary.scanone}}, \code{\link[base]{par}},
  \code{\link[base]{colors}} }

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{hplot}

\eof
\name{plot.scantwo}
\alias{plot.scantwo}

\title{Plot LOD scores for a two-dimensional genome scan}

\description{
  Plot the results of a two-dimensional, two-QTL genome scan.
}

\usage{
plot.scantwo(x, chr, incl.markers=FALSE, zlim,
             lower=c("cond-int","cond-add","joint"), nodiag=TRUE, 
             contours=FALSE, main, zscale=TRUE, \dots)
}			    

\arguments{
  \item{x}{An object of class \code{"scantwo"}, as output by
    \code{link[qtl]{scantwo}}.} 
  \item{chr}{Vector specifying which chromosomes to plot.}
  \item{incl.markers}{If FALSE, plot LOD scores on an evenly
    spaced grid (not including the results at the markers).}
  \item{zlim}{A vector of length 2 (optional), indicating the z limits
    for the joint LOD and the epistasis LOD scores, respectively.  If
    missing, the maximum limits are used for each.}
  \item{lower}{Indicates which LOD scores should be plotted in the lower
    triangle.  See the details below.}
  \item{nodiag}{If TRUE, suppress the plot of the scanone output
    (which is normally along the diagonal.)}
  \item{contours}{If TRUE, add contours to the plot, using a call to
    \code{\link[base]{contour}}.}
  \item{main}{An optional title for the plot.}
  \item{zscale}{If TRUE, a color scale is plotted at the right.}
  \item{\dots}{Ignored at this point.}
}

\value{None.}

\details{
  Uses \code{\link[base]{image}} to plot a grid of LOD scores.  The
  upper triangle contains the epistasis LOD scores; the content in the
  lower triangle depends on the input argument \code{lower}.
  The diagonal contains either all zeros or the main effects LOD scores
  (from \code{\link[qtl]{scanone}}).

  The \code{\link[qtl]{scantwo}} function calculates, for each pair of
  putative QTLs, \eqn{(q[1],q[2])}{(q1,q2)}, the likelihood undering the
  null model \eqn{L[0]}{L0}, the likelihood under each of the single-QTL
  models, \eqn{L(q[1])}{L(q1)} and \eqn{L(q[2])}{L(q2)}, the likelihood
  under an additive QTL model, \eqn{L[a](q[1],q[2])}{La(q1,q2)}, and the
  likelihood under a full QTL model (including QTL-QTL interaction),
  \eqn{L[f](q[1],q[2])}{Lf(q1,q2)}.  
  
  The upper triangle in plot from this function contains the epistasis
  LOD scores, \eqn{LOD[i] = log[10] L[f](q[1],q[2]) - log[10] L[a](q[1],q[2])}{%
    LODi = log10 Lf(q1,q2) - log10 Lf(q1,q2)}.

  If \code{lower="joint"}, the lower triangle will contain the joint LOD
  scores, \eqn{LOD[j] = log[10] L[f](q[1],q[2]) - log[10] L0}{%
    LODi = log10 Lf(q1,q2) - log10 L0}.

  If \code{lower="cond-int"}, the lower triangle will contain LOD scores
  comparing the two-QTL joint likelihood to the best of the single-QTL models,
  \eqn{LOD[ci] = log[10] L[f](q[1],q[2]) - log[10] max(L(q[1]), L(q[2]))}{%
    LODci = log10 Lf(q1,q2) - log10 max(L(q1), L(q2))}.

  If \code{lower="cond-add"}, the lower triangle will contain LOD scores
  comparing the two-QTL additive likelihood to the best of the single-QTL models,
  \eqn{LOD[ca] = log[10] L[a](q[1],q[2]) - log[10] max(L(q[1]), L(q[2]))}{%
    LODca = log10 La(q1,q2) - log10 max(L(q1), L(q2))}.  
    
  If a color scale is plotted (\code{zscale=TRUE}), the axis on the
  left indicates the scale for the epistasis LOD scores (the upper
  triangle), while the axis on the right indicates the scale for the
  lower triangle.  Note that the axis labels get screwed up
  if you change the size of the figure window; you'll need to redo the
  plot. 

  Note that the numbers plotted with the contours (if
  \code{contours=TRUE}) correspond to the scale for the lower triangle,
  and are likely incorrect for the upper triangle.  It appears to be
  difficult to fix this.   
}


\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2, step=5)
out.2dim <- scantwo(fake.f2, method="hk", chr=c(1,8,13,14))
plot(out.2dim)
plot(out.2dim, nodiag=FALSE)
plot(out.2dim, lower="joint", nodiag=FALSE)
}

\seealso{ \code{\link[qtl]{scantwo}},
  \code{\link[qtl]{summary.scantwo}}, \code{\link[qtl]{plot.scanone}} }

\author{Hao Wu, \email{hao@jax.org}; Karl W Broman,
  \email{kbroman@jhsph.edu} } 

\keyword{hplot}

\eof
\name{pull.geno}
\alias{pull.geno}

\title{Pull out the genotype data from a cross}

\description{
  Pull out the genotype data from a cross object, as a single big matrix.
}

\usage{pull.geno(cross)}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
}

\value{
  A matrix of size n.ind x tot.mar.  The raw genotype data in the input
  cross object, with the chromosomes pasted together.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
dat <- pull.geno(listeria)

# image of the genotype data
image(1:ncol(dat),1:nrow(dat),t(dat),ylab="Individuals",xlab="Markers",
      col=c("red","yellow","blue","green","violet"))
abline(v=cumsum(c(0,nmar(listeria)))+0.5)
abline(h=nrow(dat)+0.5)
}

\seealso{ \code{\link[qtl]{pull.map}} }

\keyword{manip}

\eof
\name{pull.map}
\alias{pull.map}

\title{Pull out the genetic map from a cross}

\description{
  Pull out the map portion of a cross object.
}

\usage{pull.map(cross)}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
}

\value{
  The genetic map: a list with each component containing the marker
  positions (in cM) for a chromosome.  Each component has class
  \code{A} or \code{X} according to whether it is an autosome or the X
  chromosome.  The map itself is given class \code{map}.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
map <- pull.map(fake.f2)
plot(map)
}

\seealso{ \code{\link[qtl]{replace.map}}, \code{\link[qtl]{plot.map}},
  \code{\link[qtl]{subset.cross}} }

\keyword{manip}

\eof
\name{qtl-internal}
\alias{addqtl}
\alias{adjust.rf.ri}
\alias{calc.pairprob}
\alias{checkcovar}
\alias{convert.cross}
\alias{create.map}
\alias{discan}
\alias{dropqtl}
\alias{fixXdata}
\alias{getsex}
\alias{getgenonames}
\alias{imf.cf}
\alias{imf.h}
\alias{imf.k}
\alias{imf.m}
\alias{locatemarker}
\alias{makeSSmap}
\alias{mf.cf}
\alias{mf.h}
\alias{mf.k}
\alias{mf.m}
\alias{parseformula}
\alias{pull.chr}
\alias{read.cro.qtlcart}
\alias{read.cross.csv}
\alias{read.cross.gary}
\alias{read.cross.karl}
\alias{read.cross.mm}
\alias{read.cross.qtlcart}
\alias{read.cross.qtx}
\alias{read.map.qtlcart}
\alias{read.maps.mm}
\alias{ripple.perm1}
\alias{ripple.perm2}
\alias{ripple.perm.sub}
\alias{scanone.perm}
\alias{scantwo.perm}
\alias{sim.cross.4way}
\alias{sim.cross.bc}
\alias{sim.cross.f2}
\alias{print.summary.cross}
\alias{print.summary.fitqtl}
\alias{print.summary.ripple}
\alias{print.summary.scanone}
\alias{print.summary.scantwo}
\alias{replaceqtl}
\alias{vbscan}
\alias{write.cross.csv}
\alias{write.cross.gary}
\alias{write.cross.mm}
\alias{write.cross.qtlcart}

\title{Internal qtl functions}

\description{
  Internal qtl functions.
}

\usage{
addqtl(cross, qtl, add.chr, add.pos, add.name, map)
adjust.rf.ri(r, type=c("self","sib"), expand=TRUE)
calc.pairprob(cross, step=0, off.end=0, error.prob=0, 
              map.function=c("haldane","kosambi","c-f","morgan"))
checkcovar(cross, pheno.col, addcovar, intcovar)
convert.cross(cross)
create.map(map, step, off.end)
discan(cross, pheno.col=1, method=c("em","mr"),
       x.treatment, maxit=4000, tol=1e-4)
dropqtl(qtl, drop)
fixXdata(type=c("f2ss","f2","bc"), x.treatment=c("simple","full"),
         sexpgm, geno, prob, draws, pairprob)
getsex(cross)
getgenonames(type=c("f2","bc","f2ss","riself","risib","4way"),
             chrtype=c("A","X"), x.treatment=c("simple","full"),
             sexpgm)
imf.cf(r)
imf.h(r)
imf.k(r)
imf.m(r)
locatemarker(map, pos, chr, flag)
makeSSmap(cross)
mf.cf(d)
mf.h(d)
mf.k(d)
mf.m(d)
parseformula(formula, qtl.dimname, covar.dimname)
pull.chr(cross, chr)
read.cro.qtlcart(file)
read.cross.csv(dir, file, sep=",", na.strings=c("-","NA"),
               genotypes=c("A","H","B","D","C"),
               estimate.map=TRUE)
read.cross.gary(dir, genfile, mnamesfile, chridfile, phefile,
                pnamesfile, mapfile,estimate.map)
read.cross.karl(dir, genfile, mapfile, phefile)
read.cross.mm(dir, rawfile, mapfile, estimate.map=TRUE)
read.cross.qtlcart(dir, crofile, mapfile)
read.cross.qtx(dir, file, estimate.map=TRUE)
read.map.qtlcart(file)
read.maps.mm(mapsfile)
ripple.perm1(n)
ripple.perm2(n)
ripple.perm.sub(x, mat)
scanone.perm(cross, pheno.col=1, model=c("normal","binary","2part","np"),
             method=c("em","imp","hk","mr","mr-imp","mr-argmax"),
             addcovar=NULL, intcovar=NULL, weights=NULL,
             x.treatment=c("simple","full"), upper=FALSE, ties.random=FALSE,
             start=NULL, maxit=4000, tol=1e-4, n.perm=1000, trace=TRUE)
scantwo.perm(cross, pheno.col=1,
             method=c("em","imp","hk","mr","mr-imp","mr-argmax"),
             addcovar=NULL, intcovar=NULL, weights=NULL,
             x.treatment=c("simple","full"), incl.markers=FALSE,
             maxit=4000, tol=1e-4, trace=FALSE, n.perm=1000)
sim.cross.4way(map, model, n.ind, error.prob, missing.prob,
               partial.missing.prob, keep.errorind, map.function)
sim.cross.bc(map, model, n.ind, error.prob, missing.prob,
             keep.errorind, map.function)
sim.cross.f2(map, model, n.ind, error.prob, missing.prob,
             partial.missing.prob, keep.errorind, map.function)
print.summary.fitqtl(x, \dots)
print.summary.cross(x, \dots)
print.summary.ripple(x, \dots)
print.summary.scanone(x, \dots)
print.summary.scantwo(x, \dots)
replaceqtl(cross, qtl, replace, by.chr, by.pos, by.name, map)
vbscan(cross, pheno.col=1, x.treatment, upper=FALSE,
       method="em", maxit=4000, tol=1e-4)
write.cross.csv(cross, filestem="data", chr, digits=5)
write.cross.gary(cross, chr, digits)
write.cross.mm(cross, filestem="data", chr, digits=5)
write.cross.qtlcart(cross, filestem="data", chr)
}

\details{
  These are generally not to be called by the user.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\keyword{internal}

\eof
\name{read.cross}
\alias{read.cross}

\title{Read data for a QTL experiment}

\description{
  Data for a QTL experiment is read from a set of files and converted
  into an object of class \code{cross}.  The comma-delimited format
  (\code{csv}) is recommended.  All formats require chromosome
  assignments for the genetic markers, and assume that markers are in
  their correct order.
}

\usage{
read.cross(format=c("csv","mm","qtx","qtlcart","gary","karl"), dir="",
           file, genfile, mapfile, phefile, chridfile, mnamesfile,
           pnamesfile, sep=",", na.strings=c("-","NA"),
           genotypes=c("A","H","B","D","C"), estimate.map=TRUE)
}

\arguments{
  \item{format}{Specifies the format of the data.}
  \item{dir}{Directory in which the data files will be found.  In
    Windows, use forward slashes (\code{"/"}) or double backslashes
    (\code{"\\\\"}) to specify directory trees.}
  \item{file}{The main imput file for formats \code{csv} and \code{mm}.}
  \item{genfile}{File with genotype data (formats \code{karl} and
    \code{gary} only).}
  \item{mapfile}{File with marker position information (all formats
    except \code{csv}).}
  \item{phefile}{File with phenotype data (formats \code{karl} and
    \code{gary} only).}
  \item{chridfile}{File with chromosome ID for each marker (\code{gary}
    format only).} 
  \item{mnamesfile}{File with marker names (\code{gary} format only).}
  \item{pnamesfile}{File with phenotype names (\code{gary} format
    only).}
  \item{sep}{The field separator (\code{csv} format only).  This is
    generally \code{","}, but could be any other character (such as
    \code{"\\t"} for tab), as long as that character does not appear in
    any of the records.}  
  \item{na.strings}{A vector of strings which are to be interpreted as
    missing values (\code{csv} format only).  These are interpreted
    globally for the entire file, so missing value codes in phenotypes
    must not be valid genotypes, and vice versa.}
  \item{genotypes}{A vector of character strings specifying the genotype
    codes (\code{csv} format only).  Generally this is a vector of
    length 5, with the elements corresponding to AA, AB, BB, not AA 
    (i.e., AB or  BB), and not BB (ie, AB or BB).  \bold{Note}: Pay
    careful attention to the third and fourth of these; the order of
    these can be confusing!  These five character strings are converted
    to (1,2,3,5,4) in the \code{cross} object that is returned.} 
  \item{estimate.map}{For formats \code{csv}, \code{qtx}, \code{mm}, and
    \code{gary} only: if TRUE and marker positions are not included in
    the input files, the genetic map is estimated using the function 
    \code{\link[qtl]{est.map}}.}   
}

\value{
  An object of class \code{cross}, which is a list with two components:

  \item{geno}{This is a list with elements corresponding to
    chromosomes.  \code{names(geno)} contains the names of the
    chromsomes.  Each chromosome is itself a list, and is given class
    \code{A} or \code{X} according to whether it is autosomal
    or the X chromosome. \cr \cr
    
    There are two components for each chromosome: \code{data}, a matrix
    whose rows are individuals and whose columns are markers, and
    \code{map}, either a vector of marker positions (in cM) or a matrix
    of dim (\code{2 x n.mar}) where the rows correspond to marker
    positions in female and male genetic distance, respectively.  \cr \cr

    The genotype data for a backcross is coded as follows: NA = missing,
    1 = AA, 2 = AB.  \cr \cr

    For an F2 intercross, the coding is NA = missing, 1 = AA, 2 = AB, 3
    = BB, 4 = not BB (ie AA or AB; D in mapmaker/qtl), 5 = not AA (ie AB
    or BB; C in mapmaker/qtl). \cr \cr

    For a 4-way cross, the mother and father are assumed to have
    genotypes AB and CD, respectively.  The genotype data for the
    progeny is assumed to be phase-known, with the following coding
    scheme: NA = missing, 1 = AC, 2 = BC, 3 = AD, 4 = BD, 5 = A = AC or AD,
    6 = B = BC or BD, 7 = C = AC or BC, 8 = D = AD or BD, 9 = AC or BD,
    10 = AD or BC.
  }

  \item{pheno}{data.frame of size (\code{n.ind x n.phe}) containing the
    phenotypes.} 
}

\details{
  The available formats are comma-delimited (\code{csv}), Mapmaker
  (\code{mm}), Map Manager QTX (\code{qtx}), Gary Churchill's format
  (\code{gary}) and Karl Broman's format (\code{karl}).  The required
  files and their specification for each format appears below.  The
  comma-delimited format is recommended. Note that these formats work
  only for backcross and intercross data. 
}


\section{X chromosome}{
  \bold{The genotypes for the X chromosome require special care!}

  Any X chromosome genotype data should be coded like an
  autosome in a backcross, with genotypes A and H.

  The phenotype data should contain a column named "sex" which indicates
  the sex of each individual, either coded as 0=female and 1=male, or as
  a factor with levels female/male or f/m.  Case will be ignored both in
  the name and in the factor levels.  If no such phenotype column is
  included, it will be assumed that all individuals are of the same sex.

  In the case of an intercross, the phenotype data may also contain a
  column names "pgm" (for "paternal grandmother") indicating the
  direction of the cross.  It should be coded as 0/1 with 0 indicating
  the cross (AxB)x(AxB) or (BxA)x(AxB) and 1 indicating the cross
  (AxB)x(BxA) or (BxA)x(BxA).  If no such phenotype column is included,
  it will be assumed that all individuals come from the same direction
  of cross. 

  In a backcross, females should be coded 1=AA and 2=AB, while males
  should be coded 1=A and 2=B (hemizygous).

  In an intercross, males should be coded as 1=A and 2=B (hemizygous),
  which females should be coded as 1=AA and 2=AB for pgm=0, and 1=BB and
  2=AB for pgm=1.
}

\section{CSV format}{
  The input file is a text file with a specified field delimiter
  (\code{sep}) (a comma is recommended).

  The first line should contain the phenotype names followed by the
  marker names.  \bold{At least one phenotype must be included}; for
  example, include a numerical index for each individual.

  The second line should contain blanks in the phenotype columns,
  followed by chromosome identifiers for each marker in all other
  columns. If a chromosome has the identifier \code{X} or \code{x}, it
  is assumed to be the X chromosome; otherwise, it is assumed to be an
  autosome.

  An optional third line should contain blanks in the phenotype
  columns, followed by marker positions, in cM.

  Marker order is taken from the cM positions, if provided; otherwise,
  it is taken from the column order.

  Subsequent lines should give the data, with one line for each
  individual, and with phenotypes followed by genotypes.  If possible,
  phenotypes are made numeric; otherwise they are converted to factors.

  The cross is determined to be a backcross if only the first two elements
  of the \code{genotypes} string are found; otherwise, it is assumed to
  be an intercross.
}

\section{Mapmaker format}{
  This format requires two files.  The so-called rawfile, specified by
  the argument \code{file}, contains the genotype and phenotype
  data. Rows beginning with the symbol \code{#} are ignored.  The first
  line should be either \code{data type f2 intercross} or
  \code{data type f2 backcross}.  The second line should begin with
  three numbers indicating the numbers of individuals, markers and
  phenotypes in the file.  This line may include the word \code{symbols}
  followed by symbol assignments (see the documentation for mapmaker,
  and cross your fingers).  The rest of the lines give genotype data
  followed by phenotype data, with marker and phenotype names always
  beginning with the symbol \code{*}.
    
  A second file contains the genetic map information, specified with
  the argument \code{mapfile}.  (For the Mapmaker format, if
  \code{genfile} is specified but not \code{mapfile}, we assume that
  \code{genfile} is the file to use.)  The map file may be in
  one of two formats.  The function will determine which format of map
  file is presented.  

  The simplest format for the map file is not standard for the Mapmaker
  software, but is easy to create.  The file contains two or three
  columns separated by white space and with no header row.  The first
  column gives the chromosome assignments.  The second column gives the
  marker names, with markers listed in the order along the chromosomes.
  An optional third column lists the map positions of the markers.

  Another possible format for the map file is the \code{.maps}
  format, which is produced by Mapmaker.  The code for reading this
  format was written by Brian Yandell; I'm not really familiar with it
  myself.  

  Marker order is taken from the map file, either by the order they are
  presented or by the cM positions, if specified.  

  If a chromosome has the identifier \code{X} or \code{x}, it is assumed
  to be the X chromosome; otherwise, it is assumed to be an autosome.
}

\section{Map Manager QTX format}{
  This format requires a single file (that produced by the Map Manager
  QTX program).
}  

\section{QTL Cartographer format}{
  This format requires two files: the \code{.cro} and \code{.map} files
  for the QTL Cartographer (produced by the QTL Cartographer
  sub-program, Rmap and Rcross).

  Note that the QTL Cartographer cross types are converted as follows:
  RF1 to riself, RF2 to risib, RF0 (doubled haploids) to bc, B1 or B2 to
  bc, RF2 or SF2 to f2.
}  

\section{Gary format}{
  This format requires the six files.  All files have default names, and
  so the file names need not be specified if the default names are used. 

  \code{genfile} (default = \code{"geno.dat"}) contains the genotype
  data.  The file contains one line per individual, with genotypes for
  the set of markers separated by white space.  Missing values are
  coded as 9, and genotypes are coded as 0/1/2 for AA/AB/BB.

  \code{mapfile} (default = \code{"markerpos.txt"}) contains two
  columns with no header row: the marker names in the first column and
  their cM position in the second column.  If marker positions are not
  available, use \code{mapfile=TRUE}, and a dummy map will be inserted.
  
  \code{phefile} (default = \code{"pheno.dat"}) contains the phenotype
  data, with one row for each mouse and one column for each phenotype.
  There should be no header row, and missing values are coded as
  \code{"-"}. 

  \code{chridfile} (default = \code{"chrid.dat"}) contains the
  chromosome identifier for each marker.  

  \code{mnamesfile} (default = \code{"mnames.txt"}) contains the marker
  names.

  \code{pnamesfile} (default = \code{"pnames.txt"}) contains the names
  of the phenotypes.  If phenotype names file is not available, use
  \code{pnamesfile=NULL}; arbitrary phenotype names will then be
  assigned. 
}


\section{Karl format}{
  This format requires three files; all files have default names, and so
  need not be specified if the default name is used. 

  \code{genfile} (default = \code{"gen.txt"}) contains the genotype
  data.  The file contains one line per individual, with genotypes
  separated by white space.  Missing values are coded 0; genotypes are
  coded as 1/2/3/4/5 for AA/AB/BB/not BB/not AA.
    
  \code{mapfile} (default = \code{"map.txt"}) contains the map
  information, in the following complicated format: \cr \cr
    \code{n.chr} \cr
    \code{n.mar(1) rf(1,1) rf(1,2) \ldots rf(1,n.mar(1)-1)}\cr
    \code{mar.name(1,1)}\cr 
    \code{mar.name(1,2)}\cr 
    \code{\ldots}\cr 
    \code{mar.name(1,n.mar(1))}\cr 
    \code{n.mar(2)}\cr
    \code{\ldots}\cr 
    \code{etc.} \cr

  \code{phefile} (default = \code{"phe.txt"}) contains a matrix of
  phenotypes, with one individual per line.  The first line in the
  file should give the phenotype names.
}

\examples{
\dontrun{# comma-delimited format
dat1 <- read.cross("csv", dir="Mydata", file="mydata.csv")

# Mapmaker format
dat2 <- read.cross("mm", dir="Mydata", file="mydata.raw",
                   mapfile="mydata.map")

# Map Manager QTX format
dat3 <- read.cross("qtx", dir="Mydata", file="mydata.qtx")

# QTL Cartographer format
dat4 <- read.cross("qtlcart", dir="Mydata", file="qtlcart.cro",
                   mapfile="qtlcart.map")

# Gary format
dat5 <- read.cross("gary", dir="Mydata", genfile="geno.dat",
                   mapfile="markerpos.txt", phefile="pheno.dat",
                   chridfile="chrid.dat", mnamesfile="mnames.txt",
                   pnamesfile="pnames.txt")

# Karl format
dat6 <- read.cross("karl", dir="Mydata", genfile="gen.txt",
                   phefile="phe.txt", mapfile="map.txt")}
} 

\author{Karl W Broman, \email{kbroman@jhsph.edu}; Brian S. Yandell}

\seealso{ \code{\link[qtl]{write.cross}}, \code{\link[qtl]{sim.cross}};
  the \code{sampledata} directory in the package distribution contains
  sample data files in all formats except Gary's.
}

\keyword{IO}

\eof
\name{replace.map}
\alias{replace.map}

\title{Replace the genetic map of a cross}

\description{
  Replace the map portion of a cross object.
}

\usage{replace.map(cross, map)}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{map}{A list containing the new genetic map.  This must be the
   same length and with the same marker names as that contained in
   \code{cross}. }
}

\value{The input \code{cross} object with the genetic map replaced by
  the input \code{map}.}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
newmap <- est.map(fake.f2)
plot.map(fake.f2,newmap)
fake.f2 <- replace.map(fake.f2, newmap)
}

\seealso{ \code{\link[qtl]{pull.map}}, \code{\link[qtl]{est.map}} }

\keyword{manip}

\eof
\name{ripple}
\alias{ripple}

\title{Compare marker orders}

\description{
  Investigate different marker orders for a given chromosome, comparing
  all possible permutations of a sliding window of markers.
}

\usage{
ripple(cross, chr, window=4, method=c("countxo","likelihood"),
       error.prob=0, map.function=c("haldane","kosambi","c-f","morgan"),
       maxit=4000, tol=1e-4, sex.sp=TRUE)
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{The chromosome to investigate.  Only one chromosome is allowed.}
 \item{window}{Number of markers to include in the sliding window of
   permuted markers.  Larger numbers result in the comparison of a
   greater number of marker orders, but will require a considerable
   increase in computation time.}
 \item{method}{Indicates whether to compare orders by counting the
   number of obligate crossovers, or by a likelihood analysis.} 
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi,
   Carter-Falconer, or Morgan map function when converting genetic
   distances into recombination fractions.}
 \item{maxit}{Maximum number of EM iterations to perform.}
 \item{tol}{Tolerance for determining convergence.}
 \item{sex.sp}{Indicates whether to estimate sex-specific maps; this is 
   used only for the 4-way cross.}
}

\details{
  For \code{method="likelihood"}, calculations are done by first
  constructing a matrix of marker orders and then making repeated calls
  to the R function \code{\link[qtl]{est.map}}.  Of course, it would be
  faster to do everything within C, but this was a lot easier to code.

  For \code{method="countxt"}, calculations are done within C.  
}

\value{
  A matrix, given class \code{"ripple"}; the first set of columns are
  marker indices describing the order.  In the case of
  \code{method="countxo"}, the last column is the number of obligate
  crossovers for each particular order.  In the case of
  \code{method="likelihood"}, the last two columns are LOD scores (log
  base 10 likelihood ratios) comparing each order to the initial order
  and the estimated chromosome length for the given order.  Positive LOD
  scores indicate that the alternate order has more support than the
  original.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(badorder)
rip7c <- ripple(badorder, 1, 7)
summary(rip7c)

rip2l <- ripple(badorder, 1, 2, method="likelihood")
summary(rip2l)

badorder <- switch.order(badorder, 1, rip7c[2,])
}

\seealso{ \code{\link[qtl]{summary.ripple}}, \code{\link[qtl]{switch.order}},
  \code{\link[qtl]{est.map}}, \code{\link[qtl]{est.rf}} }

\keyword{utilities}

\eof
\name{scanone}
\alias{scanone}

\title{Genome scan with a single QTL model}

\description{
  Genome scan with a single QTL model, with possible allowance for
  covariates, using any of several possible models for the phenotype and
  any of several possible numerical methods.
}

\usage{
scanone(cross, chr, pheno.col=1, model=c("normal","binary","2part","np"),
        method=c("em","imp","hk","mr","mr-imp","mr-argmax"),
        addcovar=NULL, intcovar=NULL, weights=NULL,
        x.treatment=c("simple","full"), upper=FALSE, ties.random=FALSE,
        start=NULL, maxit=4000, tol=1e-4, n.perm, trace=TRUE)
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{Vector indicating the chromosomes for which LOD scores
   should be calculated.}
 \item{pheno.col}{Column number in the phenotype matrix which should be
   used as the phenotype.}
 \item{model}{The phenotypic model: the usual normal model, a model for
   binary traits, a two-part model or non-parametric.}
 \item{method}{Indicates whether to use the EM algorithm, 
   imputation, Haley-Knott regression, or marker regression.  Not all
   methods are available for all models.  Marker regression is performed
   either by dropping individuals with missing genotypes (\code{"mr"}),
   or by first filling in missing data using a single imputation
   (\code{"mr-imp"}) or by the Viterbi algorithm (\code{"mr-argmax"}).}
 \item{addcovar}{Additive covariates; allowed only for the normal model.}
 \item{intcovar}{Interactive covariates (interact with QTL genotype);
   allowed only for the normal model.}
 \item{weights}{Optional weights of individuals.  Should be either NULL
   or a vector of length n.ind containing positive weights.  Used only
   in the case \code{model="normal"}}
 \item{x.treatment}{How to treat the X-chromosome.  Simple: hemizygous
   males are equivalent to homozygous females.  Full: hemizygous males
   are treated differently from homozygous females.  Note that columns
   \code{"sex"} and \code{"pgm"} in the phenotype data are needed; see
   the section on the X chromosome in the help file for
   \code{\link[qtl]{read.cross}}.} 
 \item{upper}{Used only for the two-part model; if true, the
   ``undefined'' phenotype is the maximum observed phenotype; otherwise,
   it is the smallest observed phenotype.} 
 \item{ties.random}{Used only for the non-parametric ``model;'' if TRUE,
   ties in the phenotypes are ranked at random.  If FALSE, average ranks
   are used and a corrected LOD score is calculated.}
 \item{start}{Used only for the EM algorithm with the normal model and
   no covariates.  If \code{NULL}, use the usual starting values; if
   length 1, use random initial weights for EM; otherwise, this should
   be a vector of length n+1 (where n is the number of possible
   genotypes for the cross), giving the initial values for EM.}
 \item{maxit}{Maximum number of iterations in the EM algorithm; used
   only in interval mapping.}
 \item{tol}{Tolerance value for determining convergence in the EM
   algorithm; used only in interval mapping.}
 \item{n.perm}{If specified, a permutation test is performed rather than
   an analysis of the observed data.  This argument defines the number
   of permutation replicates.}
 \item{trace}{In the case \code{n.perm} is specified, display
   information about the progress of the permutation tests.}
}

\section{Models}{
  \bold{The normal model} is the standard model for QTL mapping.  The
  residual phenotypic variation is assumed to follow a normal
  distribution, and analysis is analogous to linear regression. 

  \bold{The binary model} is for the case of a binary phenotype, which
  must have values 0 and 1.  The proportions of 1's in the different
  genotype groups are compared.  Currently only methods \code{em} and
  \code{mr} are available for this model.

  \bold{The two-part model} is appropriate for the case of a spike in
  the phenotype distribution (for example, metastatic density when many
  individuals show no metastasis, or survival time following an
  infection when individuals may recover from the infection and fail to
  die).  The two-part model was described by Broman et al. (2000) and
  Boyartchuk et al. (2001).  Individuals with QTL genotype \eqn{g} have
  probability \eqn{p_g}{p[g]} of having an undefined phenotype (the
  spike), while if their phenotype is defined, it comes from a normal
  distribution with mean \eqn{\mu_g}{mu[g]} and common standard
  deviation \eqn{\sigma}{s}. Three LOD scores are calculated:
  LOD(\eqn{p,\mu}{p,mu}) is for the test of the hypothesis that \eqn{p_g
    = p}{p[g] = p} and \eqn{\mu_g = \mu}{mu[g] = mu}.  LOD(\eqn{p}) is
  for the test that \eqn{p_g = p}{p[g] = p} while the \eqn{\mu_g}{mu[g]}
  may vary. LOD(\eqn{\mu}{mu}) is for the test that \eqn{\mu_g =
    \mu}{mu[g] = mu} while the \eqn{p_g}{p[g]} may vary. 

  \bold{With the non-parametric ``model''}, an extension of the
  Kruskal-Wallis test is used; this is similar to the method described
  by Kruglyak and Lander (1995).  In the case of incomplete genotype
  information (such as at locations between genetic markers), the
  Kruskal-Wallis statistic is modified so that the rank for each
  individual is weighted by the genotype probabilities, analgous to
  Haley-Knott regression.  For this method, if the argument
  \code{ties.random} is TRUE, ties in the phenotypes are assigned random
  ranks; if it is FALSE, average ranks are used and a corrected LOD
  score is calculate.  Currently the \code{method} argument is ignored
  for this model.
}

\section{Methods}{
  \bold{\code{em}}: maximum likelihood is performed via the
  EM algorithm (Dempster et al. 1977), first used in this context by
  Lander and Botstein (1989).

  \bold{\code{imp}}: multiple imputation is used, as described by Sen
  and Churchill (2001).

  \bold{\code{hk}}: Haley-Knott regression is used (regression of the
  phenotypes on the multipoint QTL genotype probabilities), as described
  by Haley and Knott (1992).

  \bold{\code{mr}}: Marker regression is used.  Analysis is performed
  only at the genetic markers, and individuals with missing genotypes
  are discarded.
}

\section{Covariates}{
Covariates are allowed only for the normal model, in which case the
model is \eqn{y = \beta_q + A \gamma + Z \delta_q + \epsilon}{y = b[q] +
  A g + Z d[q] + e} where \emph{q} is the unknown QTL genotype, \emph{A}
is a matrix of additive covariates, and \emph{Z} is a matrix of
covariates that interact with the QTL genotype.  The columns of \emph{z}
are forced to be contained in the matrix \emph{A}.

The LOD score is calculated comparing the likelihood of the above
model to that of the null model \eqn{y = \mu + A \gamma + \epsilon}{y =
  m + A g + e}.

Covariates must be numeric matrices.  Individuals with any missing
covariates are discarded.  
}

\details{
  Use of the EM algorithm or Haley-Knott regression require that
  multipoint genotype probabilities are first calculated using
  \code{\link[qtl]{calc.genoprob}}.  The imputation method uses the
  results of \code{\link[qtl]{sim.geno}}.

  Individuals with missing phenotypes are dropped.

  In the case that \code{n.perm} is not missing, so that a permutation
  test is performed, the R function \code{scanone} is called repeatedly.
 
  See further details on the models, the methods and the use of
  covariates below.
}

\value{
  If \code{n.perm} is missing, the function returns a data.frame whose
  first two columns contain the chromosome IDs and cM positions.  The
  third column contains the LOD score.  In the case of the two-part
  model, the third column is LOD(\eqn{p,\mu}{p,mu}), while the fourth
  and fifth columns are LOD(\eqn{p}) and LOD(\eqn{\mu}{mu}).  In the
  case of no covariates, further columns specify the parameter
  estimates. The data frame is given class \code{"scanone"} and
  attributes  \code{"model"}, \code{"method"} and \code{"type"} (the
  latter is the type of cross analyzed). \cr 

  If \code{n.perm} is specified, the function returns either a vector of
  length \code{n.perm}, containing the maximum LOD scores, genome-wide,
  for the permutation replicates. In the case of the two-part model, the
  return value is a matrix of size \code{n.perm x 3}, with columns
  corresponding to the three different LOD scores.
}

\references{
  Boyartchuk V. L., Broman, K. W., Mosher, R. E., D'Orazio
  S. E. F., Starnbach, M. N. and Dietrich, W. F. (2001) Multigenic
  control of \emph{Listeria monocytogenes} susceptibility in
  mice. \emph{Nature Genetics} \bold{27}, 259--260.
  
  Broman, K. W., Boyartchuk, V. L. and Dietrich, W. F. (2000) Mapping
  time-to-death quantitative trait loci in a mouse cross with high
  survival rates. Technical Report MS00-04, Department of Biostatistics,
  Johns Hopkins University, Baltimore, MD.

  Churchill, G. A. and Doerge, R. W. (1994) Empirical threshold values for
  quantitative trait mapping.  \emph{Genetics} \bold{138}, 963--971.

  Dempster, A. P., Laird, N. M. and Rubin, D. B. (1977) Maximum
  likelihood from incomplete data via the EM algorithm.  \emph{J. Roy.
    Statist. Soc.} B, \bold{39}, 1--38.

  Haley, C. S. and Knott, S. A. (1992) A simple regression method for mapping
  quantitative trait loci in line crosses using flanking markers.
  \emph{Heredity} \bold{69}, 315--324.

  Kruklyak, L. and Lander, E. S. (1995) A nonparametric approach for
  mapping quantitative trait loci.  \emph{Genetics} \bold{139},
  1421--1428. 

  Lander, E. S. and Botstein, D. (1989) Mapping Mendelian factors underlying
  quantitative traits using RFLP linkage maps.  \emph{Genetics}
  \bold{121}, 185--199.

  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.

  Soller, M., Brody, T. and Genizi, A. (1976) On the power of experimental
  designs for the detection of linkage between marker loci and
  quantitative loci in crosses between inbred lines.
  \emph{Theor. Appl. Genet.} \bold{47}, 35--39. 
}

\author{Karl W Broman, \email{kbroman@jhsph.edu}; Hao Wu,
  \email{hao@jax.org} }

\examples{
###################
# Normal Model
###################
data(hyper)
# Genotype probabilities for EM and H-K
hyper <- calc.genoprob(hyper, step=2.5)
out.em <- scanone(hyper, method="em")
out.hk <- scanone(hyper, method="hk")

# Summarize results: peaks above 3
summary(out.em, 3)
summary(out.hk, 3)

# Plot the results
plot(out.hk, out.em)
plot(out.hk, out.em, chr=c(1,4), lty=1, col=c("blue","black"))

# Imputation; first need to run sim.geno
# Do just chromosomes 1 and 4, to save time
\dontrun{hyper.c1n4 <- sim.geno(subset(hyper, chr=c(1,4)),
                       step=2.5, n.draws=64)}
\testonly{hyper.c1n4 <- sim.geno(subset(hyper, chr=c(1,4)),
                       step=2.5, n.draws=16)}
out.imp <- scanone(hyper.c1n4, method="imp")
summary(out.imp, 3)

# Plot all three results
plot(out.imp, out.hk, out.em, chr=c(1,4), lty=1,
     col=c("red","blue","black"))

# Permutation tests
\dontrun{permo <- scanone(hyper, method="hk", n.perm=1000)
}\testonly{permo <- scanone(hyper, method="hk", n.perm=3)
}quantile(permo, 0.95)


###################
# Non-parametric
###################
out.np <- scanone(hyper, model="np")
summary(out.np, 3)

# Plot with previous results
plot(out.np, chr=c(1,4), lty=1, col="green")
plot(out.imp, out.hk, out.em, chr=c(1,4), lty=1,
     col=c("red","blue","black"), add=TRUE)

###################
# Two-part Model
###################
data(listeria)
listeria <- calc.genoprob(listeria,step=2.5)
out.2p <- scanone(listeria, model="2part", upper=TRUE)
summary(out.2p, 5)

# Plot all three LOD scores together
plot(out.2p, out.2p, out.2p, lodcolumn=c(4,5,3), lty=1, chr=c(1,5,13),
     col=c("red","blue","black"))

# Permutation test
\dontrun{permo <- scanone(listeria, model="2part", upper=TRUE,
                 n.perm=1000)
}\testonly{permo <- scanone(listeria, model="2part", upper=TRUE,
                 n.perm=3)
}apply(permo, 2, quantile, 0.95)

###################
# Binary model
###################
listeria <- subset(listeria, ind=!is.na(listeria$pheno[,1]))
listeria$pheno[,2] <- rep(0,nind(listeria))
listeria$pheno[listeria$pheno[,1]==264,2] <- 1
out.bin <- scanone(listeria, pheno.col=2, model="binary")
summary(out.bin, 3)

# Plot LOD for binary model with LOD(p) from 2-part model
plot(out.bin, out.2p, lodcolumn=c(3,4), lty=1, col=c("black", "red"),
     chr=c(1,5,13))

# Permutation test
\dontrun{permo <- scanone(listeria, pheno.col=2, model="binary",
                 n.perm=1000)
}\testonly{permo <- scanone(listeria, pheno.col=2, model="binary",
                 n.perm=3)
}quantile(permo, 0.95)

###################
# Covariates
###################
data(fake.bc)
plot(fake.bc)
fake.bc <- calc.genoprob(fake.bc, step=2.5)
# genome scans without covariates
out.nocovar <- scanone(fake.bc)
# genome scans with covariates
ac <- fake.bc$pheno[,c("sex","age")]
ic <- fake.bc$pheno[,"sex"]
out.covar <- scanone(fake.bc, pheno.col=1,
                     addcovar=ac, intcovar=ic)
summary(out.nocovar,3)
summary(out.covar,3)
plot(out.covar,out.nocovar,chr=c(2,5,10))
}

\seealso{ \code{\link[qtl]{plot.scanone}}, 
  \code{\link[qtl]{summary.scanone}}, \code{\link[qtl]{scantwo}},
  \code{\link[qtl]{calc.genoprob}}, \code{\link[qtl]{sim.geno}},
  \code{\link[qtl]{max.scanone}} } 

\keyword{models}

\eof
\name{scanqtl}
\alias{scanqtl}

\title{General QTL scan}

\description{
  Performs a multiple QTL scan for specified chromosomes and positions
  or intervals, with the possible inclusion of QTL-QTL interactions and/or
  covariates. 
}

\usage{
scanqtl(cross, pheno.col=1, chr, pos, covar=NULL, formula,
        method=c("imp"), incl.markers=FALSE)
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{pheno.col}{Column number in the phenotype matrix which should be
   used as the phenotype.}
 \item{chr}{Vector indicating the chromosome for each QTL.}
 \item{pos}{List indicating the positions or intervals on the chromosome
   to be scanned.  Each element should be either a single number (for a
   specific position) or a pair of numbers (for an interval).}
 \item{covar}{A data frame of covariates.}
 \item{formula}{An object of class \code{\link[base]{formula}}
   indicating the model to be fitted.  QTLs are indicated as \code{Q1},
   \code{Q2}, etc.  Covariates are indicated by their names in \code{covar}.}
 \item{method}{Indicates whether to use the EM algorithm or  
   imputation. (Only imputation is implemented at this point.)}
 \item{incl.markers}{If FALSE, do calculations only at points on an
   evenly spaced grid.}
}

\details{
  The formula is used to specified the model to be fitted. In the
  formula, user has to use \code{Q1}, \code{Q2}, etc., or \code{q1},
  \code{q2}, etc., to represent the QTLs, and the column names in the
  covariate data frame to represent the covariates.
    
  Only the interaction terms need to be specifed in the formula. The
  main effects of all input QTLs (as specified by chr and pos) and
  covariates (as specifed by covar) will be included by default. For
  example, if the formula is \code{y~Q1*Q2*Sex}, there are three
  elements in input chr and pos and Sex is one of the column names for
  input cov, the formula used in genome scan will be
  \code{y ~ Q1 + Q2 + Q3 + Sex + Q1:Q2 + Q1:Sex + Q2:Sex + Q1:Q2:Sex}. 

  The input pos is a list or vector to specify the position/range of the
  input chromosomes to be scanned. If it is a vector, it gives the
  specified position on the chromosomes. If it is a list, it will
  contain either the certain positions or a range on chromosomes. For
  example, the input chr is c(1, 6, 13). If pos is c(9.8, 34.0, 18.6),
  it means to scan chromosome 1 at 9.8cM, chromosome 6 at 34cM and 
  chromosome 13 at 18.6cM. If pos is list(c(5,15), c(30,36), 18), it
  means to scan chromosome 1 from 5cM to 15cM, chromosome 6 from 30cM to
  36cM, fix chromosome 13 at 18cM.
}
 
\value{
  An object of class \code{scanqtl}. It is a multi-dimensional
  array of LOD scores, with the number of dimension equal to the number
  of QTLs specifed.} 

\references{
  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.
}

\author{Hao Wu, \email{hao@jax.org} }

\examples{
data(fake.f2)

# take out several QTLs
qc <- c(1, 8, 13)
fake.f2 <- subset(fake.f2, chr=qc)

# imputate genotypes
fake.f2 <- sim.geno(fake.f2, n.draws=64, step=2, err=0.001)

# 2-dimensional genome scan with additive 3-QTL model
pos <- list(c(15,35), c(45,65), 28)
result <- scanqtl(fake.f2, pheno.col=1, chr=qc, pos=pos,
                  formula=y~Q1+Q2+Q3)

# image of the results
# chr locations
chr1 <- as.numeric(matrix(unlist(strsplit(colnames(result),"@")),
                   ncol=2,byrow=TRUE)[,2])
chr8 <- as.numeric(matrix(unlist(strsplit(rownames(result),"@")),
                   ncol=2,byrow=TRUE)[,2])
# image plot
image(chr1, chr8, t(result), las=1, col=rev(rainbow(256,start=0,end=2/3)))

# do the same, allowing the QTLs on chr 1 and 13 to interact
result2 <- scanqtl(fake.f2, pheno.col=1, chr=qc, pos=pos,
                   formula=y~Q1+Q2+Q3+Q1:Q3)
# image plot
image(chr1, chr8, t(result2), las=1, col=rev(rainbow(256,start=0,end=2/3)))
}

\seealso{ \code{\link[qtl]{fitqtl}}, \code{\link[qtl]{makeqtl}} }

\keyword{models}

\eof
\name{scantwo}
\alias{scantwo}

\title{Two-dimensional genome scan with a two-QTL model}

\description{
  Perform a two-dimensional genome scan with a two-QTL model, with
  possible allowance for covariates. \bold{Warning}: two-dimensional
  genome scans on a dense grid can take a great deal of computer time
  and memory. 
}

\usage{
scantwo(cross, chr, pheno.col=1,
        method=c("em","imp","hk","mr","mr-imp","mr-argmax"), 
        addcovar=NULL, intcovar=NULL, weights=NULL,
        x.treatment=c("simple","full"),
        run.scanone=TRUE, incl.markers=FALSE, maxit=4000, tol=1e-4,
        trace=TRUE, n.perm)
}

\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{Vector indicating the chromosomes for which LOD scores
   should be calculated.}
 \item{pheno.col}{Column number in the phenotype matrix which should be
   used as the phenotype.}
 \item{method}{Indicates whether to use standard interval mapping (ie
   the EM algorithm), imputation, Haley-Knott regression, or marker
   regression.  Marker regression is performed either by dropping
   individuals with missing genotypes (\code{"mr"}), or by first filling
   in missing data using a single imputation (\code{"mr-imp"}) or by the
   Viterbi algorithm (\code{"mr-argmax"}).}
 \item{addcovar}{Additive covariates.}
 \item{intcovar}{Interactive covariates (interact with QTL genotype).}
 \item{x.treatment}{How to treat the X-chromosome.  Simple: hemizygous
   males are equivalent to homozygous females.  Full: hemizygous males
   are treated differently from homozygous females.  Note that columns
   \code{"sex"} and \code{"pgm"} in the phenotype data are needed; see
   the section on the X chromosome in the help file for
   \code{\link[qtl]{read.cross}}.} 
 \item{weights}{Optional weights of individuals.  Should be either NULL
   or a vector of length n.ind containing positive weights.}
 \item{run.scanone}{If TRUE, run the function \code{\link[qtl]{scanone}}
   and place the results on the diagonal.}
 \item{incl.markers}{If FALSE, do calculations only at points on an
   evenly spaced grid.}
 \item{maxit}{Maximum number of iterations in the EM algorithm; used
   only with method \code{"em"}.}
 \item{tol}{Tolerance value for determining convergence in the EM
   algorithm; used only with method \code{"em"}.}
 \item{trace}{If TRUE, display information about the progress of
   calculations.  For method \code{"em"}, if \code{trace} is an integer
   above 1, further details on the progress of the algorithm will be
   displayed.}
 \item{n.perm}{If specified, a permutation test is performed rather than
   an analysis of the observed data.  This argument defines the number
   of permutation replicates.}
}

\details{
  The interval mapping (i.e. EM algorithm) and Haley-Knott regression
  methods require that multipoint genotype probabilities are first
  calculated using \code{\link[qtl]{calc.genoprob}}.  The imputation
  method uses the results of \code{\link[qtl]{sim.geno}}. 

  The method \code{em} is standard interval mapping by the EM algorithm
  (Dempster et al. 1977; Lander and Botstein 1989).
  Marker regression is simply linear regression of phenotypes on marker
  genotypes  (individuals with missing genotypes are discarded).
  Haley-Knott regression uses the regression of phenotypes on multipoint
  genotype probabilities.  The imputation method uses the pseudomarker
  algorithm described by Sen and Churchill (2001).
  
  Individuals with missing phenotypes are dropped.

  In the presence of covariates, the full model is 
  \deqn{y = \mu + \beta_{q1} + \beta_{q2} + \beta_{q1 \times q2} +
    A \gamma + Z \delta_{q1} + Z \delta_{q2} + Z \delta_{q1 \times q2} +
    \epsilon}{y = m + b[q1] + b[q2] + b[q1 x q2] + A g + Z d[q1] + Z
    d[q2] + Z d[q1 x q2] + e}
  where \emph{q1} and \emph{q2} are the unknown QTL genotypes at two
  locations, \emph{A} is a matrix of covariates, and \emph{Z} is a
  matrix of covariates that interact with QTL genotypes.  The columns of
  \emph{Z} are forced to be contained in the matrix \emph{A}.

  We calculate LOD scores testing comparing the full model to each of
  two alternatives.  The joint LOD score compares the full model to the
  following null model:
  \deqn{y = \mu + A \gamma + \epsilon}{y = m + A g + e}
  The epistasis LOD score compares the full model to the following
  additive model: 
  \deqn{y = \mu + \beta_{q1} + \beta_{q2} + A \gamma + Z \delta_{q1} + Z
    \delta_{q2} + \epsilon}{y = m + b[q1] + b[q2] + A g + Z d[q1] + Z
    d[q2] + e}

  In the case that \code{n.perm} is specified, the R function
  \code{scantwo} is called repeatedly.
}

\value{
  If \code{n.perm} is missing, the function returns a list with 
  class \code{"scantwo"} and containing two components.  The first
  component is a matrix of dimension [tot.pos x tot.pos] whose upper
  triangle contains the epistasis LOD scores and whose lower triangle
  contains the joint LOD scores.  If \code{run.scanone=TRUE}, the
  diagonal contains the results of \code{\link[qtl]{scanone}}. The
  second component of the output is a data.frame indicating the
  locations at which the two-QTL LOD scores were calculated.  The first
  column is the chromosome identifier, the second column is the position
  in cM, and the third column is a 1/0 indicator for ease in later
  pulling out only the equally spaced positions.

  If \code{n.perm} is specified, the function returns a matrix with two
  columns, containing the maximum joint and epistasis LOD scores, across
  a two-dimensional scan, for each of the permutation replicates.
}

\references{
  Churchill, G. A. and Doerge, R. W. (1994) Empirical threshold values for
  quantitative trait mapping.  \emph{Genetics} \bold{138}, 963--971.

  Dempster, A. P., Laird, N. M. and Rubin, D. B. (1977) Maximum
  likelihood from incomplete data via the EM algorithm.  \emph{J. Roy.
    Statist. Soc.} B, \bold{39}, 1--38.

  Haley, C. S. and Knott, S. A. (1992) A simple regression method for mapping
  quantitative trait loci in line crosses using flanking markers.
  \emph{Heredity} \bold{69}, 315--324.

  Lander, E. S. and Botstein, D. (1989) Mapping Mendelian factors underlying
  quantitative traits using RFLP linkage maps.  \emph{Genetics}
  \bold{121}, 185--199.

  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.

  Soller, M., Brody, T. and Genizi, A. (1976) On the power of experimental
  designs for the detection of linkage between marker loci and
  quantitative loci in crosses between inbred lines.
  \emph{Theor. Appl. Genet.} \bold{47}, 35--39. 
}

\author{Karl W Broman, \email{kbroman@jhsph.edu}; Hao Wu,
  \email{hao@jax.org} }

\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2, step=10)
out.2dim <- scantwo(fake.f2, method="hk")
plot(out.2dim)

\testonly{permo.2dim <- scantwo(fake.f2, method="hk", n.perm=3)
}\dontrun{permo.2dim <- scantwo(fake.f2, method="hk", n.perm=1000)
}apply(permo.2dim,2,quantile,0.95)

# covariates
data(fake.bc)
fake.bc <- calc.genoprob(fake.bc, step=10)
ac <- fake.bc$pheno[,c("sex","age")]
ic <- fake.bc$pheno[,"sex"]
out <- scantwo(fake.bc, method="hk", pheno.col=1,
               addcovar=ac, intcovar=ic)
plot(out)
}

\seealso{ \code{\link[qtl]{plot.scantwo}}, \code{\link[qtl]{summary.scantwo}},
  \code{\link[qtl]{scanone}}, \code{\link[qtl]{max.scantwo}} }

\keyword{models}

\eof
\name{sim.cross}
\alias{sim.cross}

\title{Simulate a QTL experiment}

\description{
  Simulates data for a QTL experiment using a model in which QTLs act additively.
}

\usage{
sim.cross(map, model=NULL, n.ind=100, type=c("f2", "bc", "4way"),
          error.prob=0, missing.prob=0, partial.missing.prob=0,
          keep.qtlgeno=TRUE, keep.errorind=TRUE,
	  map.function=c("haldane","kosambi","c-f","morgan"))
}

\arguments{
 \item{map}{A list whose components are vectors containing the marker
   locations on each of the chromosomes.}
 \item{model}{A matrix where each row corresponds to a
 different QTL, and gives the chromosome number, cM position and
 effects of the QTL.}
 \item{n.ind}{Number of individuals to simulate.}
 \item{type}{Indicates whether to simulate an intercross (\code{f2}),
   a backcross (\code{bc}), or a phase-known 4-way cross (\code{4way}).}
 \item{error.prob}{The genotyping error rate.}
 \item{missing.prob}{The rate of missing genotypes.}
 \item{partial.missing.prob}{When simulating an intercross or 4-way
   cross, this gives the rate at which markers will be incompletely
   informative (i.e., dominant or recessive).}
 \item{keep.qtlgeno}{If TRUE, genotypes for the simulated QTLs will be
   included in the output.}
 \item{keep.errorind}{If TRUE, and if \code{error.prob > 0}, the
   identity of genotyping errors will be included in the output.}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi, 
   Carter-Falconer, or Morgan map function when converting genetic
   distances into recombination fractions.}
}

\details{
  Meiosis is assumed to exhibit no crossover interference.  If one of
  the chromosomes has class \code{X}, it is assumed to be the X
  chromosome, and is assumed to be segregating in the cross.  Thus, in
  an intercross, it is segregating like a backcross chromosome.  In a
  4-way cross, a second phenotype, \code{sex}, will be generated.

  QTLs are assumed to act additively, and the residual phenotypic
  variation is assumed to be normally distributed with variance 1.

  For a backcross, the effect of a QTL is a single number corresponding
  to half the difference between the homozygote and the heterozygote.

  For an intercross, the effect of a QTL is a pair of numbers,
  (\eqn{a,d}), where \eqn{a} is the additive effect (half the difference
  between the homozygotes) and \eqn{d} is the dominance deviation (the
  difference between the heterozygote and the midpoint between the
  homozygotes).

  For a four-way cross, the effect of a QTL is a set of three numbers,
  (\eqn{a,b,c}), where, in the case of one QTL, the mean phenotype,
  conditional on the QTL genotyping being AC, BC, AD or BD, is  \eqn{a},
  \eqn{b}, \eqn{c} or 0, respectively.
}  

\value{
  An object of class \code{cross}.  See \code{\link[qtl]{read.cross}} for
  details. \cr 

  If \code{keep.qtlgeno} is TRUE, the cross object will contain a
  component \code{qtlgeno} which is a matrix containing the QTL
  genotypes (with complete data and no errors), coded as in the genotype
  data. \cr 

  If \code{keep.errorind} is TRUE and errors were simulated, each
  component of \code{geno} will each contain a matrix \code{errors},
  with 1's indicating simulated genotyping errors.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\seealso{ \code{\link[qtl]{sim.map}}, \code{\link[qtl]{read.cross}},
    \code{\link[qtl]{fake.f2}}, \code{\link[qtl]{fake.bc}}
    \code{\link[qtl]{fake.4way}} }

\examples{
# simulate a genetic map
map <- sim.map()

# simulate 250 intercross individuals with 2 QTLs
fake <- sim.cross(map, type="f2", n.ind=250,
                  model = rbind(c(1,45,1,1),c(5,20,0.5,-0.5)))
}
                    
\keyword{datagen}

\eof
\name{sim.geno}
\alias{sim.geno}

\title{Simulate genotypes given observed marker data}

\description{
  Uses the hidden Markov model technology to simulate from the joint
  distribution Pr(g | O) where g is the underlying genotype vector and O
  is the observed multipoint marker data, with possible allowance for
  genotyping errors.
}

\usage{
sim.geno(cross, n.draws=16, step=0, off.end=0, error.prob=0, 
         map.function=c("haldane","kosambi","c-f","morgan"))
}

\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{n.draws}{Number of simulation replicates to perform.}
 \item{step}{Maximum distance (in cM) between positions at which the
   simulated genotypes will be drawn, though for \code{step = 0},
   genotypes are drawn only at the marker locations.}
 \item{off.end}{Distance (in cM) past the terminal markers on each
   chromosome to which the genotype simulations will be carried.}
 \item{error.prob}{Assumed genotyping error rate used in the calculation
   of the penetrance Pr(observed genotype | true genotype).}
 \item{map.function}{Indicates whether to use the Haldane, Kosambi, 
   Carter-Falconer, or Morgan map function when converting genetic
   distances into recombination fractions.}
}

\details{
  After performing the forward-backward equations, we draw from
  \eqn{Pr(g_1 = v | O)}{Pr(g[1] = v | O)} and then \eqn{Pr(g_{k+1} = v |
    O, g_k = u)}{Pr(g[k+1] = v | O, g[k] = u)}. 

  In the case of the 4-way cross, with a sex-specific map, we assume a
  constant ratio of female:male recombination rates within the
  inter-marker intervals. 
}

\value{
  The input \code{cross} object is returned with a component,
  \code{draws}, added to each component of \code{cross$geno}.
  \code{draws} is array of size [n.ind x n.pos x n.draws] where n.pos is
  the number of positions at which the simulations were performed and
  n.draws is the number of replicates.  Attributes \code{"error.prob"},
  \code{"step"}, and \code{"off.end"} are set to the values of the
  corresponding arguments, for later reference.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
fake.f2 <- sim.geno(fake.f2, step=2, off.end=5)
\testonly{data(fake.bc)}
\testonly{fake.bc <- sim.geno(fake.bc, step=0, off.end=0)}
}

\seealso{ \code{\link[qtl]{calc.genoprob}},
  \code{\link[qtl]{argmax.geno}} } 

\keyword{utilities}

\eof
\name{sim.map}
\alias{sim.map}

\title{Simulate a genetic map}

\description{
  Simulate the positions of markers on a genetic map.
}

\usage{
sim.map(len=rep(100,20), n.mar=10, anchor.tel=TRUE,
        include.x=TRUE, sex.sp=FALSE, eq.spacing=FALSE)
}

\arguments{
 \item{len}{A vector specifying the chromosome lengths (in cM)}
 \item{n.mar}{A vector specifying the number of markers per chromosome.}
 \item{anchor.tel}{If true, markers at the two telomeres will always be
   included, so if \code{n.mar} = 1 or 2, we'll give just the two
   telomeric markers.} 
 \item{include.x}{Indicates whether the last chromosome should be
   considered the X chromosome.}
 \item{sex.sp}{Indicates whether to create sex-specific maps, in which
   case the output will be a vector of 2-row matrices, with rows
   corresponding to the maps for the two sexes.}
 \item{eq.spacing}{If TRUE, markers will be equally spaced.}
}

\details{
  Aside from the telomeric markers, marker positions are simulated as
  iid Uniform(\eqn{0,L}).  If \code{len} or \code{n.mar} has just one element,
  it is expanded to the length of the other argument.  If they both have
  just one element, only one chromosome is simulated.

  If \code{eq.spacing} is TRUE, markers are equally spaced between 0 and
  \eqn{L}.  If \code{anchor.tel} is FALSE, telomeric markers are not
  included.  
}

\value{
  A list of vectors, each specifying the locations of the markers.  Each
  component of the list is given class \code{A} or \code{X}, according
  to whether it is autosomal or the X chromosome.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
# simulate 4 autosomes, each with 10 markers
map <- sim.map(c(100,90,80,40), 10, include.x=FALSE)
plot.map(map)

# equally spaced markers
map2 <- sim.map(c(100,90,80,40), 10, include.x=FALSE, eq.spacing=TRUE)
plot(map2)
}

\seealso{ \code{\link[qtl]{sim.cross}}, \code{\link[qtl]{plot.map}},
  \code{\link[qtl]{replace.map}}, \code{\link[qtl]{pull.map}} }

\keyword{datagen}

\eof
\name{subset.cross}
\alias{subset.cross}

\title{Subsetting data for QTL experiment}

\description{
  Pull out a specified set of chromosomes and/or individuals from a
  \code{cross} object.
}

\usage{subset.cross(x, chr, ind, \dots)}

\arguments{
 \item{x}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{A vector specifying which chromosomes to keep or discard.
   This may be a logical vector, a numeric vector, or a vector of
   character strings.} 
 \item{ind}{A vector specifying which individuals to keep or
   discard. This may be a logical or numeric vector.}
 \item{\dots}{Ignored at this point.}
}

\value{The input \code{cross} object, but with only the specified subset
  of the data.}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
fake.f2.A <- subset(fake.f2, chr=c("5","13"))
fake.f2.B <- subset(fake.f2, ind = -c(1,5,10))
fake.f2.C <- subset(fake.f2, chr=1:5, ind=1:50)

data(listeria)
y <- listeria$pheno[,1]
listeriaB <- subset(listeria, ind = (!is.na(y) & y < 264))
}

\seealso{ \code{\link[qtl]{pull.map}}, \code{\link[qtl]{drop.markers}} }

\keyword{manip}

\eof
\name{summary.cross}
\alias{summary.cross}
\alias{nind}
\alias{nmar}
\alias{totmar}
\alias{nchr}
\alias{nphe}

\title{Print summary of QTL experiment}

\description{
  Print some summary information about a \code{cross} object.
}

\usage{
summary.cross(object, \dots)
nind(object)
nmar(object)
totmar(object)
nchr(object)
nphe(object)
}

\arguments{
 \item{object}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{\dots}{Ignored at this point.}
}

\value{
The six functions described here return the following values.
 
  \item{summary.cross}{An object of class \code{summary.cross}
    containing a variety of summary information about the cross (this is
    generally printed automatically.)}

  \item{nind}{The number of individuals in the cross.}

  \item{nmar}{A vector containing the number of markers on
  each chromosome.}

  \item{totmar}{The total number of markers.}

  \item{nchr}{The number of chromosomes.}

  \item{nphe}{The number of phenotypes.} 
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
summary(fake.f2)
nind(fake.f2)
nmar(fake.f2)
totmar(fake.f2)
nchr(fake.f2)
nphe(fake.f2)
}

\seealso{ \code{\link[qtl]{read.cross}}, \code{\link[qtl]{plot.cross}} }

\keyword{print}

\eof
\name{summary.fitqtl}
\alias{summary.fitqtl}

\title{Summary of fit of qtl model}

\description{
  To be done.
}

\usage{
summary.fitqtl(object, \dots)
}
\arguments{
 \item{object}{Output from \code{\link[qtl]{fitqtl}}.}
 \item{\dots}{Ignored at this point.}
}

\details{
  To be added.
}

\value{
  To be added.
}

\references{
  Sen, S. and Churchill, G. A. (2001) A statistical framework for quantitative
  trait mapping.  \emph{Genetics} \bold{159}, 371--387.
}

\author{Hao Wu, \email{hao@jax.org} }

\examples{
}

\seealso{ \code{\link[qtl]{fitqtl}}, \code{\link[qtl]{makeqtl}},
\code{\link[qtl]{scanqtl}} }  

\keyword{models}

\eof
\name{summary.ripple}
\alias{summary.ripple}

\title{Print summary of ripple results}

\description{
  Print marker orders, from the output of the function \code{ripple},
  for which the log10 likelihood relative to the initial order is above
  a specified cutoff.
}

\usage{
summary.ripple(object, lod.cutoff = -1, \dots)
}

\arguments{
 \item{object}{An object of class \code{ripple}, the output of the
   function \code{\link[qtl]{ripple}}.}
 \item{lod.cutoff}{Only marker orders with LOD score (relative to the
   initial order) above this cutoff will be displayed.  For output of
   \code{\link[qtl]{ripple}} in the case of minimization of the number
   of obligate crossovers, we double this argument and treat it as a
   cutoff for the number of obligate crossovers.}
 \item{\dots}{Ignored at this point.}
}

\value{
  An object of class \code{summary.ripple}, whose rows correspond to
  marker orders with likelihood within some cutoff of the initial order.
  If no marker order, other than the initial one, has likelihood within
  the specified range, the initial and next-best orders are returned.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{ % I don't run the following, since they are run in ripple.Rd
\dontrun{data(badorder)
rip7c <- ripple(badorder, 1, 7)
summary(rip7c)

rip2l <- ripple(badorder, 1, 2, method="likelihood")
summary(rip2l)

badorder <- switch.order(badorder, 1, rip7c[2,])
}}

\seealso{ \code{\link[qtl]{ripple}}, \code{\link[qtl]{est.map}},
  \code{\link[qtl]{est.rf}} }

\keyword{print}

\eof
\name{summary.scanone}
\alias{summary.scanone}

\title{Summarize the results of a genome scan}

\description{
  Print the rows of the output from \code{\link[qtl]{scanone}} that
  correspond to the maximum LOD for each chromosome, provided that they
  exceed a specified threshold.
}

\usage{
summary.scanone(object, threshold=0, \dots)
}

\arguments{
 \item{object}{An object of the form output by the function
   \code{\link[qtl]{scanone}}: a data.frame whose first column is the
   chromosome ID and third column is the LOD score.  All other
   columns are ignored.}
 \item{threshold}{Only peaks with LOD score above this value will be
   returned.}
 \item{\dots}{Ignored at this point.}
}

\value{
  An object of class \code{summary.scanone}, to be printed by
  \code{print.summary.scanone}.  This is a data.frame with one row per
  chromosome, corresponding to the maximum LOD scores.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(listeria)
listeria <- calc.genoprob(listeria, step=2.5)
out <- scanone(listeria, model="2part")
# All peaks
summary(out)

# Peaks with LOD(p,mu) > 5
summary(out, 5)
}

\seealso{ \code{\link[qtl]{scanone}}, \code{\link[qtl]{plot.scanone}},
          \code{\link[qtl]{max.scanone}} }

\keyword{print}

\eof
\name{summary.scantwo}
\alias{summary.scantwo}

\title{Summarize the results of a two-dimensional genome scan}

\description{
  Summarize the interesting aspects of the results of \code{scantwo}.
}

\usage{
summary.scantwo(object, thresholds, \dots)
}

\arguments{
 \item{object}{An object of class \code{scantwo}, the output of the
   function \code{\link[qtl]{scantwo}}.}
 \item{thresholds}{A vector of length three, giving LOD thresholds
   for the joint LOD, interaction LOD and single-QTL conditional LOD.}
 \item{\dots}{Ignored at this point.}
}

\value{
  An object of class \code{summary.scantwo}, to be printed by
  \code{print.summary.scantwo}.  Pairs of loci meeting the specified
  criteria are printed, with their joint LOD, interaction LOD, and the
  conditional LOD for each locus, along with single-point P-values
  calculated by the \eqn{\chi^2}{chi-square} approximation.  P-values
  are printed as \eqn{-\log_{10}(P)}{-log10(P)}. \cr

  If the input \code{scantwo} object does not include the results of
  \code{scanone}, the interaction and conditional LOD thresholds are
  ignored, and all pairs of loci for which the joint LOD exceeds its
  threshold are printed, though without their conditional LOD scores.
}

\details{
  For each pair of chromosomes, the pair of loci for which the joint LOD
  score is a maximum is considered.  The pair is printed only if its
  joint LOD score exceeds the joint threshold and either (a) the
  interaction LOD score exceeds its threshold or (b) both of the loci have
  conditional LOD scores that are above the conditional LOD threshold,
  where the conditional LOD score for locus \eqn{q_1}{q1}, LOD(\eqn{q_1
    | q_2}{q1 | q2}), is the \eqn{\log_{10}}{log10} likelihood ratio
  comparing the model with \eqn{q_1}{q1} and \eqn{q_2}{q2} acting
  additively to the model with \eqn{q_2}{q2} alone. 

  In the case the results of \code{\link[qtl]{scanone}} are not
  available, the maximum locus pair for each chromosome is printed
  whenever its joint LOD exceeds the joint LOD threshold.

  The criterion used in this summary is due to Gary Churchill and Saunak
  Sen, and deserves careful consideration and possible revision.
}

\author{Hao Wu, \email{hao@jax.org}; Karl W Broman,
  \email{kbroman@jhsph.edu} } 

\examples{
data(fake.f2)
fake.f2 <- calc.genoprob(fake.f2, step=5)
out.2dim <- scantwo(fake.f2, method="hk", chr=c(1,6,13))
# All pairs of loci
summary(out.2dim)
# Pairs meeting specified criteria
summary(out.2dim, c(7, 3, 3))
# Pairs with both conditional LODs > 2
summary(out.2dim,c(0,1000,2))
# Pairs with interaction LOD is above 3
summary(out.2dim,c(0,3,1000))
}

\seealso{ \code{\link[qtl]{scantwo}}, \code{\link[qtl]{plot.scantwo}},
          \code{\link[qtl]{max.scantwo}} }

\keyword{print}

\eof
\name{switch.order}
\alias{switch.order}
\title{Switch the order of markers on a chromosome}

\description{
  Switch the order of markers on a specified chromosome to a specified
  new order.
}

\usage{switch.order(cross, chr, order, error.prob=0,
                    map.function=c("haldane","kosambi","c-f","morgan"))}

\arguments{
 \item{cross}{An object of class \code{cross}.  See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{The chromosome for which the marker order is to be
   switched.}
 \item{order}{A vector of numeric indices defining the new marker
   order. The vector may have length two more than the number of
   markers, for easy in use with the output of the function
   \code{\link[qtl]{ripple}}.} 
 \item{error.prob}{Assumed genotyping error rate (passed to
   \code{\link[qtl]{est.map}}).} 
 \item{map.function}{Map function to be used (passed to
   \code{\link[qtl]{est.map}}).} 
}

\value{
  The input \code{cross} object, but with the marker order on the
  specified chromosome updated, and with any derived data removed
  (except for recombination fractions, if present, which are not
  removed);  the genetic map for the relevant chromosome is
  re-estimated. 
}


\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(fake.f2)
fake.f2 <- switch.order(fake.f2, 1, c(1,3,2,4:7))
}

\seealso{ \code{\link[qtl]{ripple}}, \code{\link[qtl]{clean}} }

\keyword{manip}

\eof
\name{top.errorlod}
\alias{top.errorlod}

\title{List genotypes with large error LOD scores}

\description{
  Prints those genotypes with error LOD scores above a specified
  cutoff.
}

\usage{
top.errorlod(cross, chr, cutoff=3, msg=TRUE)
}
\arguments{
 \item{cross}{An object of class \code{cross}. See
   \code{\link[qtl]{read.cross}} for details.}
 \item{chr}{A vector specifying for which chromosomes the error LOD
   scores should be inspected.}
 \item{cutoff}{Only those genotypes with error LOD scores above this
   cutoff will be listed.}
 \item{msg}{If TRUE, print a message if there are no apparent errors.}
}

\value{
  A data.frame with 4 columns, whose rows correspond to the genotypes
  that are possibly in error.  The four columns give the chromosome
  number, individual number, marker name, and error LOD score.
}

\author{Karl W Broman, \email{kbroman@jhsph.edu} }

\examples{
data(hyper)
# run calc.genoprob, then calc.errorlod; use same error.prob
hyper <- calc.genoprob(hyper,error.prob=0.01)
hyper <- calc.errorlod(hyper,error.prob=0.01)

# plot the error LOD scores; print those above a specified cutoff 
plot.errorlod(hyper)
plot.errorlod(hyper,chr=1)
top.errorlod(hyper,cutoff=4)

# plot genotype data, flagging genotypes with error LOD > cutoff
plot.geno(hyper, chr=1, ind=160:200, cutoff=7, min.sep=2)

# Note that if markers are extremely tightly linked,
#     any recombination event will have a high error LOD
plot.geno(hyper, chr=1, ind=101:118, min.sep=2)
hyper$geno[[1]]$map
}

\seealso{ \code{\link[qtl]{calc.errorlod}},
  \code{\link[qtl]{plot.errorlod}} }

\keyword{manip}

\eof
\name{write.cross}
\alias{write.cross}

\title{Write data for a QTL experiment to a file}

\description{
  Data for a QTL experiment is written to a file (or files).
}

\usage{
write.cross(cross, format=c("csv","mm","qtlcart","gary"),
            filestem="data", chr, digits=5)
}

\arguments{
  \item{cross}{An object of class \code{cross}. See
    \code{\link[qtl]{read.cross}} for details.}
  \item{format}{Specifies whether to write the data in comma-delimited,
    Mapmaker, QTL Cartographer, or Gary Churchill's format.}
  \item{filestem}{A character string giving the first part of the output
    file names (the bit before the dot).  In Windows, use forward
    slashes (\code{"/"}) or double backslashes (\code{"\\\\"}) to
    specify directory trees.} 
  \item{chr}{A vector specifying for which chromosomes genotype data
    should be written.}
  \item{digits}{Number of digits to which phenotype values should be
    rounded.}
}

%\value{
%  Data is written to a single file (comma-delimited format) or two files
%  (Mapmaker format).
%}

\details{
  Comma-delimited format: a single csv file is created in the format
  read by \code{\link[qtl]{read.cross}}.

  Mapmaker format: Data is written to two files.
  Suppose \code{filestem="file"}.  Then \code{"file.raw"} will contain
  the genotype and phenotype data, and \code{"file.prep"} will contain
  the necessary code for defining the chromosome assignments, marker
  order, and inter-marker distances.

  QTL Cartographer format: Data is written to two files.    Suppose
  \code{filestem="file"}.  Then \code{"file.cro"} will contain 
  the genotype and phenotype data, and \code{"file.map"} will contain
  the genetic map information.  Note that cross types are converted to
  QTL Cartographer cross types as follows: riself to RF1, risib to RF2,
  bc to B1 and f2 to RF2. 

  Gary's format: Data is written to six files. They are:
  \code{"geno.data"} - genotype data; 
  \code{"pheno.data"} - phenotype data; 
  \code{"chrid.dat"} - the chromosome identifier for each marker; 
  \code{"mnames.txt"} - the marker names; 
  \code{"markerpos.txt"} - the marker positions; 
  \code{"pnames.txt"} - the phenotype names
}


\examples{
\dontrun{data(fake.bc)

# comma-delimited format
write.cross(fake.bc, "csv", "Data/fakebc", c(1,5,13))

# Mapmaker format
write.cross(fake.bc, "mm", "Data/fakebc", c(1,5,13))

# QTL Cartographer format
write.cross(fake.bc, "qtlcart", "Data/fakebc", c(1,5,13))

# Gary's format
write.cross(fake.bc, "gary", c(1,5,13))} }

\author{Karl W Broman, \email{kbroman@jhsph.edu}; Hao Wu; Brian S. Yandell}

\seealso{ \code{\link[qtl]{read.cross}} }

\keyword{IO}

\eof
