\name{format4Bugs}
\alias{format4Bugs}
\title{ Format a Data List as BUGS Data Input}
\description{
  Transform a data list into a format which can be read in as data by BUGS.
}
\usage{
format4Bugs(dataList, digits = 5)
}
\arguments{
  \item{dataList}{ A list of data to be formatted.}
  \item{digits}{The desired number of significant digits.}
}
\value{
  A charactor vector of data readable by BUGS.
}
\author{ Jun Yan \email{jyan@stat.uiowa.edu}}
\examples{
dat <- list(a = runif(1), b=rnorm(2), c=matrix(rexp(4), 2, 2))
format4Bugs(dat)
}
\keyword{utilities}
\keyword{interface}

\eof
\name{rbugs-internal}
\alias{formatData}
\title{ Internal functions of rbugs}
\description{
  Internal functions of rbug which are not to be called by users directly.
}
\usage{
formatData(datalist)
}
\keyword{internal}

\eof
\name{genBugsScript}
\alias{genBugsScript}
\title{Generating Script File for BUGS}
\description{
  Generating a script file to run BUGS batch-mode.
}
\usage{
genBugsScript(paramSet, n.chains, n.iter, n.burnin, n.thin,
              model.file, data.file, inits.files,
              workingDir=NULL, bugsWorkingDir=getwd(),
              script, debug = FALSE, useWine = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{paramSet}{A character vector of the names of the
    parameter to be monitored.}
  \item{n.chains}{ The number of chains to be monitored.}
  \item{n.iter}{ The number of iterations of each chain.}
  \item{n.burnin}{ The length of the burn-in. }
  \item{n.thin}{ Thinning rate.}
  \item{model.file}{ A character string of the name of file which
    contains the model specification for BUGS. It should end with .txt
    to be recognizable by BUGS.}
  \item{data.file}{ A character string of the name of the data file. It
    should end with .txt to be recognizable by BUGS. }
  \item{inits.files}{ A vector of file names of initial values. They
    should end with .txt to be recognizable by BUGS.}
  \item{workingDir}{ A character string specifying the directory to
    store temporary files for running BUGS.}
  \item{bugsWorkingDir}{ A character string specifying the directory
    which is recognizable by windows, if using wine.}
  \item{script}{ A character string naming the files to print the script
    to. It must end with .txt to be recognizable by BUGS.}
  \item{debug}{ A logical value indicating whether or not closing the
    BUGS window after running the script.}
  \item{useWine}{ If TRUE, BUGS is used via wine.}
}
\value{
  None.
}
\details{
  This function only write a script to \file{script}, which uses the
  file names of \file{model.file}, \file{data.file}, and
  \file{inits.files}, without actually reading them. These file names
  should be correct when using \code{\link{runBugs}}.

  workingDir and bugsWorkingDir establish the translation between a
  directory native to the operating system and a directory recognizable
  by wine. They are the same if useWine == FALSE.
}
\author{ Jun Yan \email{jyan@stat.uiowa.edu}}
\seealso{ \code{\link{genDataFile}}, \code{\link{genInitsFile}},
  \code{\link{format4Bugs}}
}
\examples{
## no tested examples for mac-os.
\dontrun{
#ifdef unix
script.file <- paste(tempfile("script"), ".txt", sep="")
genBugsScript(paramSet=c("alpha", "theta"), n.chains=2, n.keep=1500,
              n.burnin=500, n.thin=1,
              model.file="/var/scratch/c/tmp/model.txt",
              data.file="/var/scratch/c/tmp/data.txt",
              inits.file=c("/var/scratch/c/tmp/init1.txt",
                           "/var/scratch/c/tmp/init2.txt"),
              workingDir="/var/scratch/c/tmp",
              bugsWorkingDir="c:/tmp",
              script=script.file,
              debug=FALSE, useWine=TRUE)
file.show(script.file)
unlink(script.file)
#endif
#ifdef windows
script.file <- paste(tempfile("script"), ".txt", sep="")
genBugsScript(paramSet=c("alpha", "theta"), n.chains=2, n.keep=1500,
              n.burnin=500, n.thin=1,
              model.file="c:/tmp/model.txt",
              data.file="c:/tmp/data.txt",
              inits.file=c("c:/tmp/init1.txt",
                           "c:/tmp/init2.txt"),
              workingDir="c:/tmp",
              script=script.file,
              debug=FALSE, useWine=TRUE)
file.show(script.file)
unlink(script.file)
#endif
}
}
\keyword{utilities}
\keyword{interface}

\eof
\name{genDataFile}
\alias{genDataFile}
\title{Generating the Data File for BUGS}
\description{
  Generating the data file which can be used in the script file for
  runing BUGS batch-mode.
}
\usage{
genDataFile(dataList, dataFile)
}
\arguments{
  \item{dataList}{A list of data that are needed by BUGS. Its elements
    must be numeric vector or matrices.}
  \item{dataFile}{ A character string naming the file to print to. It
    must end with .txt to be recoganizable by BUGS.}
}
\value{
  None.
}
\author{Jun Yan \email{jyan@stat.uiowa.edu}}
\seealso{ \code{\link{genInitsFile}}, \code{\link{genBugsScript}},
  \code{\link{format4Bugs}}
}
\examples{
dat <- list(a = runif(1), b=rnorm(2), c=matrix(rexp(4), 2, 2))
genDataFile(dat, "foo.txt")
file.show("foo.txt")
unlink("foo.txt")
}
\keyword{utilities}

\eof
\name{genInitsFile}
\alias{genInitsFile}
\title{Generating Initial Value Files for BUGS}
\description{
  Generating the files of initial values for MCMC which can be used in
  the script for running BUGS in batch-mode.
}
\usage{
genInitsFile(n.chains, inits, initsFileStem)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{n.chains}{The number of chains to run.}
  \item{inits}{A list of list of initial values or a function which
    returns a list of initial values. See details.}
  \item{initsFileStem}{A character string naming the files to print
    to. See details.}
}
\details{
  if \code{inits} is a list, it should have length \code{n.chains}, and
  each element of the list should be a list which contains one set of
  initial values. If \code{inits} is a function, it will be run
  \code{n.chains} times to generate a list of list of initial values.

  The \code{initsFileStem} is the stem of the file names. The resulted
  file names end with .txt. For example, if \code{n.chains} is 3 and
  \code{initsFileStem} is "init", then the file names will be init1.txt,
  init2.txt, and init3.txt.
}
\value{
  None.
}
\author{ Jun Yan \email{jyan@stat.uiowa.edu}}
\seealso{  \code{\link{genDataFile}}, \code{\link{genBugsScript}}}
\examples{
## when inits is a list:
inits <- list(list(alpha=1, beta=2), list(alpha=4, beta=4))
genInitsFile(2, inits, "foo.init")
file.show("foo.init1.txt")
file.show("foo.init2.txt")
## when inits is a function:
inits <- function() list(alpha=rnorm(2, sd=100),
                         beta=rgamma(1, 0.1, 0.001))
genInitsFile(2, inits, "foo.init")
file.show("foo.init1.txt")
file.show("foo.init2.txt")
unlink("foo.init1.txt")
unlink("foo.init2.txt")
}
\keyword{utilities}

\eof
\name{getBugsOutput}
\alias{getBugsOutput}
\title{ Collect the MCMC samples from BUGS}
\description{
  Collect the MCMC samples from BUGS.
}
\usage{
getBugsOutput(n.chains, workingDir)
}
\arguments{
  \item{n.chains}{ The number of chains BUGS had run.}
  \item{workingDir}{ A character string specifying the name of the
    directory where the output files are saved.}
}
\details{
  This function assumes that under \code{\workingDir}, there is a coda
  index file \file{codaIndex.txt}, and the coda output for
  \code{n.chains} chains are named as coda1.txt, coda2.txt, ...
}
\value{
  A list of matrix whose columns and rows contain the monitored
  parameters and the MCMC iterations, respectively.
}
\author{ Jun Yan \email{jyan@stat.uiowa.edu}}
\keyword{utilities}
\keyword{interface}

\eof
\name{pumps}
\alias{pumps}
\docType{data}
\title{ 10 Power Plant Pumps }
\description{
  10 Power Plant Pumps.
}
\usage{data(pumps)}
\format{
  A data frame with 10 observations on the following 2 variables.
  \describe{
    \item{t}{The length of the operation time.}
    \item{x}{The number of failures.}
  }
}
\examples{
data(pumps)
pumps.data <- list(t = pumps$t, x = pumps$x, N = nrow(pumps))
pumps.model <- file.path(.path.package("rbugs"), "bugs/model", "pumps.bug")
file.show(pumps.model)
pumps.inits <- file.path(.path.package("rbugs"), "bugs/inits", "pumps.txt")
file.show(pumps.inits)
inits <- list(dget(pumps.inits))
parameters <- c("theta", "alpha", "beta")

\dontrun{
#ifdef windows
## Define the full name of bugs in .Renviron.
## In .Renviron, add a line similar to this:
##  BUGS="c:/program files/winbugs14/winbugs14.exe"

pumps.sim <- rbugs(data = pumps.data, inits, parameters,
                   pumps.model, n.chains = 1, n.iter = 1000,
                   bugs="e:/program files/winbugs14/winbugs14.exe")
#endif
#ifdef unix
## The workingDir must be a directory which can be seen by wine
## The nominalWorkingDir gives the directory name recoganizable for
## Windows.
## The full name of the bugs and wine executables need to set.
## In .Renviron, add lines similar to these:
##   BUGS="c:/program files/winbugs14/winbugs14.exe"
##   WINE="/var/scratch/jyan/wine-20040408/wine" 

pumps.sim <- rbugs(data = pumps.data, inits, parameters, 
                   pumps.model, n.chains = 1, n.iter = 1000,
                   workingDir="/var/scratch/jyan/c/tmp",
                   bugsWorkingDir="c:/tmp",
                   useWine=TRUE)
#endif
}
}
\keyword{datasets}

\eof
\name{rbugs}
\alias{rbugs}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Run BUGS from R in a BATCH}
\description{
  Generate files (data, init, script) that are necessary to run BUGS,
  call BUGS through the OS, and collect the MCMC output.
}
\usage{
rbugs(data, inits, paramSet, model,
      n.chains = 1, n.iter = 2000, n.burnin = floor(n.iter/2),
      n.thin = max(1, floor(n.chains * (n.iter - n.burnin)/1000)),
      debug = FALSE,
      bugs = Sys.getenv("BUGS"),
      workingDir = NULL, 
      bugsWorkingDir = getwd(),
      useWine = FALSE, wine = Sys.getenv("WINE"), verbose = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{data}{ a list of data object to be used by BUGS }
  \item{inits}{ a list of list of initial values, or a function which
    returns a list of initial values }
  \item{paramSet}{ a vector of the names of the parameters to be
    monitored }
  \item{model}{ the file name of the model description }
  \item{n.chains}{ the number of chains to be monitored }
  \item{n.iter}{ the number of iteration of each chain }
  \item{n.burnin}{ the length of the burn-in }
  \item{n.thin}{ thinning rate }
  \item{debug}{ a logical value indicating whether or not closing the
    BUGS window }
  \item{bugs}{ the full name (including the path) of the BUGS executable }
  \item{workingDir}{ a directory to store all the intermediate files }
  \item{bugsWorkingDir}{ the nominal name of the working directory,
    which is the same as workingDir is useWine is "FALSE". If useWine is
    "TRUE", it should be a directory which is recognizable by the
    Windows system though wine. }
  \item{useWine}{ a logical value with "TRUE" meaning wine is used }
  \item{wine}{ if useWine==TRUE, the full name of wine }
  \item{verbose}{ if TRUE, print the log file from BUGS}
}
\value{
  A list of MCMC outputs with length \code{nChains}.
}
\examples{
## see examples in \code{\link{pumps}} and \code{\link{schools}}
}
\author{
  Jun Yan \email{jyan@stat.uiowa.edu}
}
\keyword{ interface }

\eof
\name{runBugs}
\alias{runBugs}
\title{ Execute a BUGS Srcipt from R }
\description{
  Create the system command to run BUGS and execute it.
}
\usage{
runBugs(bugs=Sys.getenv("BUGS"), script, n.chains, workingDir,
        useWine=FALSE, wine=Sys.getenv("WINE"), verbose = TRUE)
}
\arguments{
  \item{bugs}{ A character string specifying the full name (including
    path) of the BUGS executable. It must end with .txt to be
    recognizable by BUGS.}
  \item{script}{ A character string specifying the full name (including
    path) of the script to be run. It must end with .txt to be
    recognizable by BUGS.}
  \item{n.chains}{The number of MCMC chains.}
  \item{workingDir}{The working directory where the output files from
    BUGS are stored.}
  \item{useWine}{ A logical value indicating if wine is to be used.}
  \item{wine}{ A character string specifying the full name (including
    path) of the wine executable.}
  \item{verbose}{If TRUE, print the log file from BUGS.}
}
\value{
  None.
}
\author{ Jun Yan \email{jyan@stat.uiowa.edu}}
\seealso{  \code{\link{rbugs}}}
\keyword{interface}

\eof
\name{schools}
\alias{schools}
\docType{data}
\title{ 8 schools }
\description{
  8 schools analysis from section 5.5 of "Bayesian Data Analysis" by
  Andrew Gelman, John B. Carlin,  Hal S. Stern, and Donald B. Rubin.
}
\usage{data(schools)}
\format{
  A data frame with 8 observations on the following 3 variables.
  \describe{
    \item{school}{a factor with levels \code{A} \code{B} \code{C} \code{D} \code{E} \code{F} \code{G} \code{H}}
    \item{estimate}{a numeric vector}
    \item{sd}{a numeric vector}
  }
}
\source{
  Gelman, A., Carlin, J.B., Stern, H.S., Rubin, D.B. (2003):
  \emph{Bayesian Data Analysis}, 2nd edition, CRC Press.
}
\examples{
data(schools)
J <- nrow(schools)
y <- schools$estimate
y <- rnorm(length(y))
sigma.y <- schools$sd
schools.data <- list ("J", "y", "sigma.y") 
## schools.data <- list(J=J, y=y, sigma.y=sigma.y)
inits <- function() {list (theta=rnorm(J,0,100),
                           mu.theta=rnorm(1,0,100),
                           sigma.theta=runif(1,0,100))}
parameters <- c("theta", "mu.theta", "sigma.theta")
schools.bug <- file.path(.path.package("rbugs"), "bugs/model", "schools.bug")
file.show(schools.bug)
\dontrun{
## no tested examples for mac-os.
#ifdef unix
schools.sim <- rbugs(data=schools.data, inits, parameters,
                     schools.bug, n.chains=3, n.iter=1000,
                     workingDir="/var/scratch/jyan/c/tmp",
                     bugsWorkingDir="c:/tmp",
                     useWine=TRUE,
                     wine="/var/scratch/jyan/wine-20040408/wine",
                     debug=TRUE)
#endif
#ifdef windows
schools.sim <- rbugs(data=schools.data, inits, parameters,
                     schools.bug, n.chains=3, n.iter=1000,
                     bugs="e:/Program Files/WinBUGS14/WinBUGS14.exe",
                     workingDir="c:/tmp",
                     debug=TRUE)
                     
#endif
}
}
\keyword{datasets}

\eof
