\name{HanoiTower}
\alias{HanoiTower}
\alias{move.HanoiTower}
\alias{print.HanoiTower}
\alias{plot.HanoiTower}
\title{ application example for references }
\description{
  This is an example for using references in S (R/S+) with package \code{ref}.
  \code{HanoiTower} implements a recursive algorithm solving the Hanoi tower problem.
  It is implemented such that the recursion can be done either by passing the HanoiTower \emph{by reference} or \emph{by value} to the workhorse function \code{move.HanoiTower}.
  Furthermore you can choose whether recursion should use \code{\link{Recall}} or should directly call \code{move.HanoiTower}.
  As the HanoiTower object is not too big, it can be extended by some garbage MBytes, that will demonstrate the advantage of passing references instead of values.
  The deeper we recurse, the more memory we waist by passing values (and the more memory we save by passing references).
  Functions \code{move.HanoiTower} and \code{print.HanoiTower} are internal (not intended to be called by the user directly).
}
\usage{
  HanoiTower(n = 5
  , parameter.mode = c("reference", "value")[1]
  , recursion.mode = c("recall", "direct")[1]
  , garbage = 0
  , print = FALSE
  , plot = TRUE
  , sleep = 0
  )
}
\arguments{
  \item{n}{ number of slices }
  \item{parameter.mode}{ one of "reference" or "value" deciding how to pass the HanoiTower object }
  \item{recursion.mode}{ one of "recall" or "direct" deciding how to call recursively }
  \item{garbage}{ no. of bytes to add to the HanoiTower size }
  \item{print}{ TRUE print the HanoiTower changes }
  \item{plot}{ FALSE not to plot the HanoiTower changes }
  \item{sleep}{ no. of seconds to wait between HanoiTower changes for better monitoring of progress }
}
\details{
  The Hanoi Tower problem can be described as follows: you have n slices of increasing size placed on one of three locations a,b,c such that the biggest slice is at the bottom, the next biggest slice on top of it and so forth with the smallest slice as the top of the tower.
  Your task is to move all slices from one stick to the other, but you are only allowed to move one slice at a time and you may never put a bigger slice on top of a smaller one.
  The recursive solution is: to move n slices from a to c you just need to do three steps: move n-1 slices to b, move the biggest slice to c and move n-1 slices from b to c. If n equals 1, just move from a to c.
}
\value{
  invisible()
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link{ref}}, \code{\link[base]{Recall}} }

\examples{
    HanoiTower(n=2)

 \dontrun{
    # small memory examples
    HanoiTowerDemoBytes <- 0
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "reference"
    , recursion.mode  = "direct"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "reference"
    , recursion.mode  = "recall"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "value"
    , recursion.mode  = "direct"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "value"
    , recursion.mode  = "recall"
    , garbage = HanoiTowerDemoBytes
    )
    rm(HanoiTowerDemoBytes)

    # big memory examples
    HanoiTowerDemoBytes <- 100000
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "reference"
    , recursion.mode  = "direct"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "reference"
    , recursion.mode  = "recall"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "value"
    , recursion.mode  = "direct"
    , garbage = HanoiTowerDemoBytes
    )
    if (is.R())
      gc()
    HanoiTower(
      parameter.mode  = "value"
    , recursion.mode  = "recall"
    , garbage = HanoiTowerDemoBytes
    )
    rm(HanoiTowerDemoBytes)
  }
}
\keyword{ programming }

\eof
\name{as.ref}
\alias{as.ref}
\title{ coercing to reference }
\description{
  This function RETURNs a reference to its argument.
}
\usage{
as.ref(obj)
}
\arguments{
  \item{obj}{ an object existing in the current environment/frame }
}
\value{
  an object of class "ref"
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link{ref}}, \code{\link{deref}} }
\examples{
  v <- 1
  r <- as.ref(v)
  r
  deref(r)
}
\keyword{ programming }

\eof
\name{deref}
\alias{deref}
\alias{deref<-}
\title{ dereferencing references }
\description{
  This functions allow to access a referenced object. \code{deref(ref)} returns the object, and \code{deref(ref) <- value} assigns to the referenced object.
}
\usage{
deref(ref)
deref<-(ref, value)
#the following does not pass R CMD CHECK
#deref(ref) <- value
#deref(ref)[1] <- value  # subsetted assignment appears to be inefficent in S+.
}
\arguments{
  \item{ref}{ a reference as returned by \code{\link{ref}} or \code{\link{as.ref}} }
}
\details{
  \code{deref} and \code{deref<-} provide convenient access to objects in other environments/frames.
  In fact they are wrappers to \code{\link{get}} and \code{\link{assign}}.
  However, convenient does not neccessarily means efficient.
  If performance is an issue, the direct use of \code{\link{new.env}}, \code{\link{substitute}} and \code{\link{eval}} may give better results.
  See the examples below.
}
\value{
  \code{deref} returns the referenced object.
  \cr \code{"deref<-"} returns a reference to the modified object, see \code{\link{ref}}.
}
\references{ Writing R Extensions }
\author{ Jens Oehlschlgel }
\note{ Subsetted assignment appears to be inefficent in S+. Note the use of \code{\link{substitute}} in the examples. }
\seealso{ \code{\link{ref}}, \code{\link{as.ref}},  \code{\link[base]{get}},  \code{\link[base]{assign}},  \code{\link[base]{substitute}},  \code{\link[base]{eval}} }
\examples{
  # Simple usage example
  x <- cbind(1:5, 1:5)          # take some object
  rx <- as.ref(x)               # wrap it into a reference
  deref(rx)                     # read it through the reference
  deref(rx) <- rbind(1:5, 1:5)  # replace the object in the reference by another one
  deref(rx)[1, ]                # read part of the object
  deref(rx)[1, ] <- 5:1         # replace part of the object
  deref(rx)                     # see the change
  cat("For examples how to pass by references see the Performance test examples at the help pages\n")

 \dontrun{
  ## Performance test examples showing actually passing by reference
  # define test size
  nmatrix <- 1000   # matrix size of nmatrix by nmatrix
  nloop   <- 10     # you might want to use less loops in S+, you might want more in R versions before 1.8

  # Performance test using ref
  t1 <- function(){ # outer function
    m <- matrix(nrow=nmatrix, ncol=nmatrix)
    a <- as.ref(m)
      t2(a)
    m[1,1]
  }
  # subsetting deref is slower (by factor 75 slower since R 1.8 compared to previous versions, and much, much slower in S+) ...
  t2 <- function(ref){ # inner function
    cat("timing", timing.wrapper(
      for(i in 1:nloop)
        deref(ref)[1,1] <- i
    ), "\n")
  }
  if (is.R())gc()
  t1()
  # ... than using substitute
  t2 <- function(ref){
    obj <- as.name(ref$name)
    loc <- ref$loc
    cat("timing", timing.wrapper(
      for(i in 1:nloop)
        eval(substitute(x[1,1] <- i, list(x=obj, i=i)), loc)
    ), "\n")
  }
  if (is.R())gc()
  t1()


  # Performance test using Object (R only)
  # see Henrik Bengtsson package(oo)
  Object <- function(){
    this <- list(env.=new.env());
    class(this) <- "Object";
    this;
  }
  "$.Object" <- function(this, name){
    get(name, envir=unclass(this)$env.);
  }
  "$<-.Object" <- function(this, name, value){
    assign(name, value, envir=unclass(this)$env.);
    this;
  }
  # outer function
  t1 <- function(){
    o <- Object()
    o$m <- matrix(nrow=nmatrix, ncol=nmatrix)
      t2(o)
    o$m[1,1]
  }
  # subsetting o$m is slower ...
  t2 <- function(o){
    cat("timing", timing.wrapper(
      for(i in 1:nloop)
        o$m[1,1] <- i
    ), "\n")
  }
  if (is.R())gc()
  t1()
  # ... than using substitute
  t2 <- function(o){
    env <- unclass(o)$env.
    cat("timing", timing.wrapper(
      for(i in 1:nloop)
        eval(substitute(m[1,1] <- i, list(i=i)), env)
    ), "\n")
  }
  if (is.R())gc()
  t1()

  }
}
\keyword{ programming }

\eof
\name{is.ref}
\alias{is.ref}
\alias{exists.ref}
\title{ checking (for) references }
\description{
  \code{is.ref} checks whether an object inherits from class "ref". \cr
  \code{exists.ref} checks whether an referenced object exists.
}
\usage{
  is.ref(x)
  exists.ref(ref)
}
\arguments{
  \item{x}{ an object that might be a reference }
  \item{ref}{ a reference as returned from \code{\link{ref}} or \code{\link{as.ref}} }
}
\value{
  logical scalar
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link{ref}}, \code{\link[base]{exists}}, \code{\link[base]{inherits}}, \code{\link[base]{class}} }
\examples{
  v <- 1
  good.r <- as.ref(v)
  bad.r <- ref("NonExistingObject")
  is.ref(v)
  is.ref(good.r)
  is.ref(bad.r)
  exists.ref(good.r)
  exists.ref(bad.r)
}
\keyword{ programming }

\eof
\name{optimal.index}
\alias{optimal.index}
\alias{need.index}
\alias{posi.index}
\title{ creating standardized, memory optimized index for subsetting }
\description{
  Function \code{optimal.index} converts an index specification of type {logical, integer, -integer, character} into one of {integer, -integer} whatever is smaller.
  Function \code{need.index} returns TRUE if the index does represent a subset (and thus indexing is needed).
  Function \code{posi.index} returns positive integers representing the (sub)set.
}
\usage{
optimal.index(i, n=length(i.names), i.names = names(i), i.previous = NULL, strict = TRUE)
need.index(oi)
posi.index(oi)
}
\arguments{
  \item{i}{ the original one-dimensional index }
  \item{n}{ length of the indexed dimension  (potential iMax if i where integer), not necessary if names= is given }
  \item{i.names}{ if i is character then names= represents the names of the indexed dimension }
  \item{i.previous}{ if i.previous= is given, the returned index represents \code{x[i.previous][i] == x[optimal.index]} rather than \code{x[i] == x[optimal.index]} }
  \item{strict}{ set to FALSE to allow for NAs and duplicated index values, but see details }
  \item{oi}{ a return value of \code{optimal.index} }
}
\details{
  When strict=TRUE it is expected that i does not contain NAs and no duplicated index values. Then \code{ identical(x[i], x[optimal.index(i, n=length(x), i.names=names(x))$i]) == TRUE } . \cr
  When strict=FALSE i may contain NAs and/or duplicated index values. In this case length optimisation is not performed and optimal.index always returns positive integers.
}
\note{
  \code{need.index(NULL)} is defined and returns FALSE. This allows a function to have an optional parameter oi=NULL and to determine the need of subsetting in one reqest.
}
\value{
  \code{optimal.index} returns the index oi with attributes n=n and ni=length(x[optimal.index]) (which is n-length(i) when i is negative).
  \code{need.index} returns a logical scalar
  \code{posi.index}  returns a vector of positive integers (or integer(0))
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link{refdata}}
          \cr please ignore the following unpublished links: ids2index, shift.index, startstop2index
}
\examples{
  l <- letters
  names(l) <- letters
  stopifnot({i <- 1:3 ; identical(l[i], l[optimal.index(i, n=length(l))])})
  stopifnot({i <- -(4:26) ; identical(l[i], l[optimal.index(i, n=length(l))])})
  stopifnot({i <- c(rep(TRUE, 3), rep(FALSE, 23)) ; identical(l[i], l[optimal.index(i, n=length(l))])})
  stopifnot({i <- c("a", "b", "c"); identical(l[i], l[optimal.index(i, i.names=names(l))])})
  old.options <- options(show.error.messages=FALSE); stopifnot(inherits(try(optimal.index(c(1:3, 3), n=length(l))), "try-error")); options(old.options)
  stopifnot({i <- c(1:3, 3, NA);identical(l[i], l[optimal.index(i, n=length(l), strict=FALSE)])})
  stopifnot({i <- c(-(4:26), -26);identical(l[i], l[optimal.index(i, n=length(l), strict=FALSE)])})
  stopifnot({i <- c(rep(TRUE, 3), rep(FALSE, 23), TRUE, FALSE, NA);identical(l[i], l[optimal.index(i, n=length(l), strict=FALSE)])})
  stopifnot({i <- c("a", "b", "c", "a", NA);identical(l[i], l[optimal.index(i, i.names=names(l), strict=FALSE)])})
  rm(l)
}
\keyword{ utilities }
\keyword{ manip }

\eof
\name{ref}
\alias{ref}
\alias{print.ref}
\title{ creating references }
\description{
  Package \code{ref} implements references for S (R/S+).
  Function \code{ref} creates references.
  For a memory efficient wrapper to matrixes and data.frames which allows nested subsetting see \code{\link{refdata}}
}
\usage{
ref(name, loc = parent.frame())
}
\arguments{
  \item{name}{ name of an (existing) object to be referenced }
  \item{loc}{ location of the referenced object, i.e. an environment in R or a frame in S+ }
}
\details{
  In S (R/S+) paramters are passed by value and not by reference.
  When passing big objects, e.g. in recursive algorithms, this can quickly eat up memory.
  The functions of package \code{ref} allow to pass references in function calls.
  The implementation is purely S and should work in R and S+.
  Existence of the referenced object is not checked by function \code{ref}.
  Usually \code{\link{as.ref}} is more convenient and secure to use.
  There is also a print method for references.
}
\value{
  a list with
  \item{ name }{  name of the referenced object }
  \item{ loc }{ location of the referenced object, i.e. an environment in R or a frame in S+ }
  and class "ref"
}
\note{
 Using this type of references is fine for prototyping in a non-objectoriented programming style.
 For bigger projects and safer programming you should consider the approach suggested by Henrik Bengtsson
 at \url{http://www.maths.lth.se/help/R/ImplementingReferences} (announced to be released as package "oo" or "classes")
}
\section{WARNING}{
 Usually functions in S have no side-effects except for the main effect of returning something.
 Working with references circumvents this programming style and can have considerable side-effects.
 You are using it at your own risk.
}
\section{R 1.8 WARNING}{
 Changing parts of referenced objects has been slowed down by order of magnitudes since R version 1.8 (see performance test examples on the help page for \{code\link{deref}}). Hopefully the old performance can be restored in future versions.
}
\section{S+ WARNING}{
 Package ref should generally work under R and S+. However, when changing very small parts of referenced objects, using references under S+ might be inefficient (very slow with high temporary memory requirements).
}
\section{Historical remarks}{
  This package goes back to an idea submitted April 9th 1997 and code offered on August 17th 1997 on s-news.
  The idea of implementing references in S triggered an intense discussion on s-news. The status reached in 1997 can be summarized as follows:\cr
  \enumerate{
    \item{\bold{advantage}}{passing by reference can save memory compared to passing by value}
    \item{\bold{disadvantage}}{passing by reference is more dangerous than passing by value}
    \item{\bold{however}}{the implementation is purely in S, thus rather channels existing danger than adding new danger}
    \item{\bold{restriction}}{assigning to a subsetted part of a referenced object was inefficient in S+ (was S+ version 3)}
  }
  Due to the last restriction the code was never submitted as a mature library.
  Now in 2003 we have a stable version of R and astonishingly assigning to a subsetted part of a referenced object \emph{can} be implemented efficient.
  This shows what a great job the R core developers have done. In the current version the set of functions for references was dramatically simplified, the main differences to 1997 beeing the following:
  \enumerate{
    \item{\bold{no idempotence}}{ \code{\link{deref}} and  \code{\link{deref<-}} now are a simple function and no longer are methods. This decision was made due top performance reasons. As a consequence, \code{deref()} no longer is idempotent: one has to know whether an object is a reference. Function \code{\link{is.ref}} provides a test. }
    \item{\bold{no write protection}}{ The 1997 suggestion included a write protection attribute of references, allowing for read only references and allowing for references that could only be changed by functions that knwe the access code. Reasons for this: there is no need for readonly references (due to copy on modify) and oop provides better mechanisms for security. }
    \item{\bold{no static variables}}{ The suggestion made in 1997 did include an implementation of static variables realized as special cases of references with a naming convention which reduced the risc of name collisions in the 1997 practice of assigning to frame 0. Now R has namespaces and the oop approach of Henrik Bengtsson using environments is to be prefered over relatively global static objects. }
  }
}
\author{ Jens Oehlschlgel }
\seealso{
 \code{\link{as.ref}}, \code{\link{deref}}, \code{\link{deref<-}}, \code{\link{exists.ref}}, \code{\link{is.ref}}, \code{\link{print.ref}}, \code{\link{HanoiTower}}
}
\examples{
  v <- 1
  r <- ref("v")
  r
  deref(r)
  cat("For more examples see ?deref\n")
}
\keyword{ programming }

\eof
\name{refdata}
\alias{refdata}
\alias{[.refdata}
\alias{[<-.refdata}
\alias{[[.refdata}
\alias{[[<-.refdata}
\alias{$.refdata}
\alias{$<-.refdata}
\alias{dim.refdata}
\alias{dim<-.refdata}
\alias{dimnames.refdata}
\alias{dimnames<-.refdata}
\alias{print.refdata}
\title{ subsettable reference to matrix or data.frame }
\description{
  Function \code{refdata} creates objects of class refdata which behave not totally unlike matrices or data.frames but allow for much more memory efficient handling.
}
\usage{
# -- usage for R CMD CHECK, see below for human readable version -----------
refdata(x)
[.refdata(x, i = NULL, j = NULL, drop = FALSE, ref = FALSE)
[<-.refdata(x, i = NULL, j = NULL, ref = FALSE, value)
 \method{dim}{refdata}(x, ref = FALSE)
 \method{dim}{refdata}(x) <- value
 \method{dimnames}{refdata}(x, ref = FALSE)
 \method{dimnames}{refdata}(x, ref = FALSE) <- value

# -- most important usage for human beings (does not pass R CMD CHECK) -----
# rd <- refdata(x)                   # create reference
# rd[]                               # get all data
# rd[i, j]                           # get part of data
# rd[i, j, ref=TRUE]                 # get new reference on part of data
# rd[i, j] <- value                  # modify part of data (now rd is reference on local copy of the data)
# rd[i, j, ref=TRUE] <- value        # modify part of original data (respecting subsetting history)
# dim(rd)                            # dim of (subsetted) data
# dim(rd, ref=TRUE)                  # dim of original data
# dimnames(rd)                       # dimnames of (subsetted) data
# dimnames(rd, ref=TRUE)             # dimnames of original data
# dimnames(rd) <- value              # modify dimnames (now rd is reference on local copy of the data)
# dimnames(rd, ref=TRUE) <- value    # modify complete dimnames of original object (NOT respecting subsetting history)
}
\arguments{
  \item{x}{ a matrix or data.frame or any other 2-dimensional object that has operators "[" and "[<-" defined }
  \item{i}{ row index }
  \item{j}{ col index }
  \item{ref}{ FALSE by default. In subsetting: FALSE returns data, TRUE returns new refdata object. In assignments: FALSE modifies a local copy and returns a refdata object embedding it, TRUE modifies the original. }
  \item{drop}{ FALSE by default, i.e. returned data have always a dimension attribute. TRUE drops dimension in some cases, the exact result depends on whether a \code{\link{matrix}} or \code{\link{data.frame}} is embedded }
  \item{value}{ some value to be assigned }
}
\details{
  Refdata objects store 2D-data in one environment and index information in another environment. Derived refdata objects usually share the data environment but not the index environment. \cr
  The index information is stored in a standardized and memory efficient form generated by \code{\link{optimal.index}}. \cr
  Thus refdata objects can be copied and subsetted and even modified without duplicating the data in memory. \cr
  Empty square bracket subsetting (\code{rd[]}) returns the data, square bracket subsetting (\code{rd[i, j]}) returns subsets of the data as expected. \cr
  An additional argument (\code{rd[i, j, ref=TRUE]}) allows to get a reference that stores the subsetting indices. Such a reference behaves transparently as if a smaller matrix/data.frame would be stored and can be subsetted again recursively.
  With ref=TRUE indices are always interpreted as row/col indices, i.e. \code{x[i]} and \code{x[cbind(i, j)]} are undefined (and raise stop errors) \cr
  Standard square bracket assignment (\code{rd[i, j] <- value}) creates a reference to a locally modified copy of the (potentially subsetted) data. \cr
  An additional argument (\code{rd[i, j, ref=TRUE] <- value}) allows to modify the original data, properly recognizing the subsetting history. \cr
  A method \code{\link{dim}(refdata)} returns the dim of the (indexed) data, the dim of the original (non-indexed) data can be accessed using parameter \code{ref=TRUE}. Assignment to dim(refdata)<- is not possible.  but \code{dim(refdata)<-} cannot be assigned. \cr
  A \code{\link{dimnames}(refdata)} returns the dimnames of the (indexed) data resp. the original data using parameter \code{ref=TRUE}. Assignment is possible but not recommended, parameter \code{ref} decides whether the original data is modified or a copy is created. \cr
}
\note{
  The refdata code is currently R only (not implemented for S+). \cr
  Please note the following differences to matrices and dataframes: \cr
  \describe{
     \item{\code{x[]}}{you need to write \code{x[]} in order to get the data}
     \item{\code{drop=FALSE}}{by default drop=FALSE which gives consistent behaviour for matrices and data.frames. You can use the $- or [[-operator to extract single column vectors which are granted to be of a consistent data type. However, currently $ and [[ are only wrappers to [. They might be performance tuned in later versions.}
     \item{\code{x[i]}}{single index subsetting is not defined, use \code{x[][i]} instead, but beware of differences between matrices and dataframes}
     \item{\code{x[cbind()]}}{matrix index subsetting is not defined, use \code{x[][cbind(i, j)]} instead}
     \item{\code{ref=TRUE}}{parameter \code{ref} needs to be used sensibly to exploit the advantages of refdata objects}
  }
}
\value{
  an object of class refdata (appended to class attributes of data), which is an empty list with two attributes
  \item{dat}{the environment where the data x and its dimension dim is stored}
  \item{ind}{the environment where the indexes i, j and the effective subset size ni, nj is stored}
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link[base]{Extract}},  \code{\link[base]{matrix}},  \code{\link[base]{data.frame}}, \code{\link{optimal.index}}, \code{\link{ref}} }

\examples{

  ## Simple usage Example
  x <- cbind(1:5, 5:1)            # take a matrix or data frame
  rx <- refdata(x)                # wrap it into an refdata object
  rx                              # see the autoprinting
  rm(x)                           # delete original to save memory
  rx[]                            # extract all data
  rx[-1, ]                        # extract part of data
  rx2 <- rx[-1, , ref=TRUE]       # create refdata object referencing part of data (only index, no data is duplicated)
  rx2                             # compare autoprinting
  rx2[]                           # extract 'all' data
  rx2[-1, ]                       # extract part of (part of) data
  cat("for more examples look the help pages\n")

 \dontrun{
  # Memory saving demos
  square.matrix.size <- 1000
  recursion.depth.limit <- 10
  non.referenced.matrix <- matrix(1:(square.matrix.size*square.matrix.size), nrow=square.matrix.size, ncol=square.matrix.size)
  rownames(non.referenced.matrix) <- paste("a", seq(length=square.matrix.size), sep="")
  colnames(non.referenced.matrix) <- paste("b", seq(length=square.matrix.size), sep="")
  referenced.matrix <- refdata(non.referenced.matrix)
  recurse.nonref <- function(m, depth.limit=10){
    x <- m[1,1]   # need read access here to create local copy
    gc()
    cat("depth.limit=", depth.limit, "  memory.size=", memsize.wrapper(), "\n", sep="")
    if (depth.limit)
      Recall(m[-1, -1, drop=FALSE], depth.limit=depth.limit-1)
    invisible()
  }
  recurse.ref <- function(m, depth.limit=10){
    x <- m[1,1]   # read access, otherwise nothing happens
    gc()
    cat("depth.limit=", depth.limit, "  memory.size=",  memsize.wrapper(), "\n", sep="")
    if (depth.limit)
      Recall(m[-1, -1, ref=TRUE], depth.limit=depth.limit-1)
    invisible()
  }
  gc()
  memsize.wrapper()
  recurse.ref(referenced.matrix, recursion.depth.limit)
  gc()
   memsize.wrapper()
  recurse.nonref(non.referenced.matrix, recursion.depth.limit)
  gc()
   memsize.wrapper()
  rm(recurse.nonref, recurse.ref, non.referenced.matrix, referenced.matrix, square.matrix.size, recursion.depth.limit)
  }
  cat("for even more examples look at regression.test.refdata()\n")
  regression.test.refdata()  # testing correctness of refdata functionality
}
\keyword{ programming }
\keyword{ manip }

\eof
\name{regression.test.refdata}
\alias{regression.test.refdata}
\title{ regression test for refdata }
\description{
  This function checks a series of use cases.
}
\usage{
regression.test.refdata()
}
\details{
  raises a stop error if a problem is detected
}
\value{
  TRUE if successful
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link{refdata}}, \code{\link[base]{example}} }

\examples{
  regression.test.refdata()
}
\keyword{ internal }

\eof
\name{sleep.wrapper}
\alias{sleep.wrapper}
\alias{memsize.wrapper}
\alias{timing.wrapper}
\title{ wrapper to get some measures for all platforms }
\description{
  interrupts execution for specified no. of seconds
}
\usage{
sleep.wrapper(time)
}
\arguments{
  \item{time}{ no. of seconds }
}
\value{
  NULL
}
\author{ Jens Oehlschlgel }
\seealso{ \code{\link[base]{Sys.sleep}} }
\keyword{ internal }

\eof
