\name{cat.picture}
\alias{cat.picture}
\title{JPEG picture of a cat}
\description{
JPEG file of a cat.
}
\usage{data(cat)}
\format{JPEG}
\source{Original photograph taken by a member of rimage developer.}
\examples{
  x <- read.jpeg(system.file("data", "cat.jpg", package="rimage"))
  plot(x)
}
\seealso{\code{\link{read.jpeg}}}
\keyword{datasets}



\eof
\name{clipping}
\alias{clipping}
\title{Clipping image}
\description{
  This function returns the image which restricts pixel value from the
  specified lowest value to the specified highest value
  in the original image. This means that the pixels which have lower
  value than the given lowest (default: 0) are replaced to
  the lowest and the pixels have greater value than
  the given highest (default: 1) are replaced to the highest.
}
\usage{clipping(img, low=0, high=1)}
\arguments{
  \item{img}{target image}
  \item{low}{lowest value}
  \item{high}{highest value}
}
\value{
  Data of the same mode as 'img'
}
\examples{
  data(logo)
  op <- par(mfrow=c(2,2))
  plot(logo, main="Source Image")

  # the appearance of next one doesn't change because of normalization.
  plot(normalize(2*logo), main="Doubled pixel value with normalization")

  # the next one is saturated as expected
  plot(clipping(2*logo), main="Doubled pixel value with clipping")
}
\keyword{misc}



\eof
\name{equalize}
\alias{equalize}
\title{Make image having equalized histogram}
\description{
  This function make image having equalized histogram of original image.
}
\usage{equalize(img)}
\arguments{
  \item{img}{target image}
}
\value{
  Data of the same mode as 'img', in which minimum value is 0 and maximum
  value is 1.
}
\examples{
  op <- par(mfrow=c(2,2))
  data(logo)

  plot(logo, main="Original Logo")
  plot(equalize(logo), main="Equalized Logo")

  catpic <- rgb2grey(read.jpeg(system.file("data", "cat.jpg", package="rimage")))
  plot(catpic, main="Cat Image in Greyscale")
  plot(equalize(catpic), main="Equalized Cat Image")

  par(op)
}
\keyword{misc}

\eof
\name{fftImg}
\alias{fftImg}
\title{Compute FFT image}
\description{
  This function computates the power spectrum of a given image by FFT.
}
\usage{
fftImg(img)
}
\arguments{
  \item{img}{target image}
}
\value{
  an imagematrix
}
\references{
  FFTW (Fastest Fourier Transform in the West) \url{http://www.fftw.org}
}
\seealso{\code{\link{fftw}},\code{\link{imagematrix}}}
\examples{
  data(logo)
  plot(normalize(fftImg(logo)))
}
\keyword{misc}

\eof
\name{fftw}
\alias{fftw}
\title{Apply FFT to 2-Dimensional Data}
\description{
  This function applies FFT to 2-dimensional data (i.e. image) using
  fftw library.
}
\usage{
fftw(img, dir = -1, debug=FALSE)
}
\arguments{
  \item{img}{target image}
  \item{dir}{set -1 for normal transformation and 
    1 for inverse transformation}
  \item{debug}{set TRUE if you want to output debug message}
}
\value{
  a matrix of complex number
}
\references{
  FFTW (Fastest Fourier Transform in the West) \url{http://www.fftw.org}
}
\seealso{\code{\link{fftw}}}
\keyword{misc}

\eof
\name{highpass}
\alias{highpass}
\title{High pass filter for image}
\description{
  Computes a high-pass filtered image with dimensions of the given
  input image. the hp pasing distance is given via radius.
}
\usage{
highpass(img,radius)
}
\arguments{
  \item{img}{pixmap or image}
  \item{radius}{gives the blocking radius}
}
\value{
  an imagematrix
}
\seealso{\code{\link{lowpass}}}
\examples{
  data(logo)
  plot(normalize(highpass(logo)))
}
\keyword{misc}

\eof
\name{imageType}
\alias{imageType}
\title{Get information on color type of imagematrix}
\description{
  This function returns color type ("rgb" or "grey") of
  a given imagematrix.
}
\usage{imageType(x)}
\arguments{
  \item{x}{target image}
}
\value{
  "rgb" or "grey"
}
\examples{
  x <- read.jpeg(system.file("data", "cat.jpg", package="rimage"))
  cat("Image Type", imageType(x))

  x.grey <- rgb2grey(x)
  cat("Image Type", imageType(x.grey))
}
\keyword{misc}



\eof
\name{imagematrix}
\alias{imagematrix}
\title{Generate an imagematrix, i.e. primary data structure of rimage}
\description{
  This function makes an imagematrix object from a matrix. This data
  structure is primary data structure to represent image in rimage package.
}
\usage{imagematrix(mat, type=NULL,
         ncol=dim(mat)[1], nrow=dim(mat)[2], noclipping=FALSE)}
\arguments{
  \item{mat}{array, matrix or vector}
  \item{type}{"rgb" or "grey"}
  \item{ncol}{width of image}
  \item{nrow}{height of image}
  \item{noclipping}{TRUE if you disable automatic clipping. See details.}
}
\details{
  For grey scale image, matrix should be given in the form of 2
  dimensional matrix. First dimension is row, and second dimension is
  column.

  For rgb image, matrix should be given in the form of 3 dimensional
  array (row, column, channel). mat[,,1], mat[,,2], mat[,,3] are
  red plane, green plane and blue plane, respectively.

  You can omit 'type' specification if you give a proper array or
  matrix.
  Also, if you give a rgb image matrix and specify "grey" as type, the rgb
  image matrix is automatically converted to a grey scale image.

  This function automatically clips the pixel values which are
  less than 0 or greater than 1. If you want to disable this
  behavior, give 'noclipiing=TRUE'.

  The major difference between imagematrix and pixmap is 
  representation method. pixmap (>0.3) uses OOP class.
  On the other hand, rimage uses traditional S class.
  The advantage of traditional S class in representing image is that
  one can deal with the data structure as an ordinary matrix.

  The minor difference between imagematrix and pixmap is
  automatic data conversion behavior.
  pixmap normalizes a given matrix automatically if
  any element of the matrix is out of range between 0 and 1.
  On the other hand, imagematrix clips the matrix, which means
  that the pixels which have lower
  value than 0 are replaced to 0 and the pixels have
  greater value than 1 are replaced to 1.
}
\value{
  return an imagematrix object
}
\examples{
  p <- q <- seq(-1, 1, length=20)
  r <- 1 - outer(p^2, q^2, "+") / 2
  plot(imagematrix(r))
}
\seealso{\code{\link{plot.imagematrix}},\code{\link{print.imagematrix}},\code{\link{clipping}},\code{\link{normalize}}}
\keyword{misc}



\eof
\name{laplacian}
\alias{laplacian}
\title{Laplacian of image}
\description{
  Calculate 2nd derivative of image for edge extraction
}
\usage{
laplacian(img)
}
\arguments{
  \item{img}{target image}
}
\value{
  an object of pixmap class whose the size is as same as target
}
\examples{
  data(logo)
  plot(normalize(laplacian(logo)))
}
\keyword{misc}

\eof
\name{logo}
\alias{logo}
\title{R logo imagematrix}
\description{
The imagematrix object of R logo of the size 101x77.
}
\usage{data(logo)}
\format{imagematrix}
\examples{
  data(logo)
  plot(logo)
}
\keyword{datasets}



\eof
\name{lowpass}
\alias{lowpass}
\title{Low Pass Filter for Image}
\description{
  Computes a low-pass filtered image with dimensions of the given
  input image. the lp pasing distance is given via radius.
}
\usage{
lowpass(img,radius)
}
\arguments{
  \item{img}{pixmap or matrix}
  \item{radius}{gives the pass radius}
}
\value{
  an imagematrix
}
\seealso{\code{\link{highpass}}}
\examples{
  data(logo)
  plot(normalize(lowpass(logo)))
}
\keyword{misc}

\eof
\name{meanImg}
\alias{meanImg}
\title{Mean filter}
\description{
  This function applies mean filter among 9 neighbors to a given image.
}
\usage{meanImg(img)}
\arguments{
  \item{img}{target image}
}
\value{
  object of imagematrix class whose the size is as same as target
}
\examples{
  data(logo)
  plot(meanImg(logo))
}
\keyword{misc}

\eof
\name{Rank Filters}
\alias{minImg}
\alias{maxImg}
\title{Rank filters (minImg and maxImg)}
\description{
  These functions apply a rank filter among 3x3 neighbors to a given image.
}
\usage{
minImg(img)
maxImg(img)
}
\arguments{
  \item{img}{target image}
}
\details{
  In 'minImg', filter replaces a pixel in question with
  the minimum value among 3x3 neighbors.
  In 'maxImg', filter replaces a pixel in question with
  the maximum value among 3x3 neighbors.
}
\value{
  object of imagematrix class whose the size is as same as target
}
\examples{
  data(logo)
  op <- par(mfrow=c(2,2))
  plot(logo, main="Original Image")            # original 
  plot(minImg(logo), main="Minimum-Filtered Image")   # minimum filtered
  plot(maxImg(logo), main="Maximum-Filtered Image")   # maximum filtered
  par(op)
}

\keyword{misc}

\eof
\name{normalize}
\alias{normalize}
\title{Normalization for vector and matrix}
\description{
  This function normalizes image so that the minimum value is 0 and
  the maximum value is 1.
}
\usage{normalize(img)}
\arguments{
  \item{img}{target image}
}
\value{
  Data of the same mode as 'img', in which minimum value is 0 and maximum
  value is 1.
}
\examples{
  data(logo)
  plot(normalize(logo))
}
\keyword{misc}



\eof
\name{plot.imagematrix}
\alias{plot.imagematrix}
\title{Plotting an imagematrix object}
\description{
  This function outputs an imagematrix object as an image.
}
\usage{plot.imagematrix(x, ...)}
\arguments{
  \item{x}{target image}
  \item{...}{plotting options}
}
\examples{
  op <- par(mfrow=c(1,2))

  data(logo)
  plot(logo, main="plot(logo)")
  plot(logo^2, main="plot(logo^2)")

  par(op)
}
\seealso{\code{\link{imagematrix}}}
\keyword{misc}



\eof
\name{print.imagematrix}
\alias{print.imagematrix}
\title{Print information on a given imagematrix object}
\description{
  This function outputs information on a given imagematrix object.
}
\usage{print.imagematrix(x, ...)}
\arguments{
  \item{x}{target image}
  \item{...}{ignored (dummy)}
}
\examples{
  data(logo)
  print(logo)
}
\seealso{\code{\link{imagematrix}}}
\keyword{misc}



\eof
\name{read.jpeg}
\alias{read.jpeg}
\title{Read JPEG file}
\description{
  This function reads a jpeg image file and return an imagematrix object.
}
\usage{read.jpeg(filename)}
\arguments{
  \item{filename}{filename of JPEG image}
}
\value{
  return an imagematrix object
}
\examples{
  x <- read.jpeg(system.file("data", "cat.jpg", package="rimage"))
  plot(x)
}
\seealso{\code{\link{imagematrix}}}
\keyword{misc}



\eof
\name{rgb2grey}
\alias{rgb2grey}
\title{Convert color imagematrix to grey imagematrix}
\description{
  This function convert color imagematrix to grey imagematrix.
}
\usage{rgb2grey(img, coefs=c(0.30, 0.59, 0.11))}
\arguments{
  \item{img}{target image}
  \item{coefs}{coefficients for red plane, green plane, and blue plane.}
}
\value{
  grey imagematrix
}
\examples{
  x <- read.jpeg(system.file("data", "cat.jpg", package="rimage"))
  plot(rgb2grey(x))
}
\keyword{misc}



\eof
\name{sobel}
\alias{sobel}
\title{Sobel filter}
\description{
  This function calculates an image which sobel filter is applied.
  It utilizes a C routine for improving speed.
}
\usage{sobel(img)}
\arguments{
  \item{img}{a matrix representing target image}
}
\value{
  a matrix representing the image after sobel filter is applied 
}
\examples{
  data(logo)
  plot(normalize(sobel(logo)))
}
\seealso{\code{\link{sobel.h}},\code{\link{sobel.v}},\code{\link{imagematrix}}}
\keyword{misc}

\eof
\name{sobel.h}
\alias{sobel.h}
\title{sobel filter to extract horizontal edges}
\description{
  This function calculates an image which sobel filter is applied.
  It extracts horizontal edges only.  It is faster than sobel.h
  extremely because utilization of a C routine.
}
\usage{sobel.h(img)}
\arguments{
  \item{img}{a matrix representing target image}
}
\value{
  a matrix representing the image after horizontal sobel filter is applied 
}
\examples{
  data(logo)
  plot(normalize(sobel.h(logo)))
}
\seealso{\code{\link{sobel.v}},\code{\link{sobel}}}
\keyword{misc}

\eof
\name{sobel.v}
\alias{sobel.v}
\title{Sobel filter to extract vertical edges}
\description{
  This function calculates an image which sobel filter is applied.
  It extracts vertical edges only. It is faster than sobel.v
  extremely because utilization of a C routine.
}
\usage{sobel.v(img)}
\arguments{
  \item{img}{a matrix representing target image}
}
\value{
  a matrix representing the image after vertical sobel filter is applied 
}
\examples{
  data(logo)
  plot(normalize(sobel.v(logo)))
}
\seealso{\code{\link{sobel.h}},\code{\link{sobel}}}
\keyword{misc}

\eof
\name{thresholding}
\alias{thresholding}
\title{thresholding image}
\description{
  This function applies thresholding to an image. You can choose fixed
  threshold mode or discriminal analysis mode. In fixed threshold mode, you can
  simply specify threshold value. In discriminal analysis mode,
  threshold is determined automatically so that two clusters are
  seperated most clearly.
}
\usage{thresholding(img, mode="fixed", th=0.5)}
\arguments{
  \item{img}{target imagematrix image}
  \item{mode}{thresholding mode. You can specify "fixed" for fixed
    threshold mode or "da" for discriminal analysis mode".}
  \item{th}{threshold to be used if mode is "fixed", otherwise ignored}
}
\value{
  a pixmap image
}
\examples{
  data(logo)
  op <- par(mfrow=c(2,2))
  plot(logo, main="Original")
  plot(thresholding(logo, mode="fixed"), main="threshold=0.5")
  plot(thresholding(logo, mode="fixed", th=0.9), main="threshold=0.9")
  plot(thresholding(logo, mode="da"), main="auto threshold by discriminal analysis")
  par(op)
}
\keyword{misc}


\eof
