\name{estfun}
\alias{estfun}
\alias{estfun.lm}
\alias{estfun.glm}
\alias{estfun.rlm}
\title{Extract Estimating Functions}
\description{
 Generic function for extracting the estimating functions
 of a fitted model with methods for (generalized) linear models.
}
\usage{
estfun(x, \dots)
}
\arguments{
 \item{x}{a fitted model object of class \code{"lm"}, \code{"glm"}
   or \code{"rlm"}.}
 \item{\dots}{currently not used.}
}

\value{A matrix containing the estimating functions.}

\seealso{\code{\link{lm}}, \code{\link{glm}}, \code{\link[MASS]{rlm}}}

\examples{
x <- sin(1:10)
y <- rnorm(10)
fm <- lm(y ~ x)

estfun(fm)
residuals(fm) * cbind(1, x)
}

\keyword{regression}

\eof
\name{isoacf}
\alias{isoacf}
\alias{pava.blocks}
\title{Isotonic Autocorrelation Function}
\description{
 Autocorrelation function (forced to be decreasing by isotonic regression).
}

\usage{
isoacf(x, lagmax = NULL, weave1 = FALSE)
}

\arguments{
 \item{x}{numeric vector.}
 \item{lagmax}{numeric. The maximal lag of the autocorrelations.}
 \item{weave1}{logical. If set to \code{TRUE} \code{isoacf}
   uses the \code{acf.R} and \code{pava.blocks} function from the
   original \code{weave} package, otherwise R's own \code{acf} and
   \code{isoreg} functions are used.}
}

\details{
\code{isoacf} computes the autocorrelation function (ACF)
of \code{x} enforcing the ACF to be decreasing by isotonic regression.
See also Robertson et al. (1988).
}

\value{
\code{isoacf} returns a numeric vector containing the ACF.
}

\references{
  Lumley A & Heagerty P (1999),
  Weighted Empirical Adaptive Variance Estimators for Correlated Data Regression.
  \emph{Journal of the Royal Statistical Society B}, \bold{61},
  459--477.

  Robertson T, Wright FT, Dykstra RL (1988),
  \emph{Order Restricted Statistical Inference}.
  New York. Wiley.
}

\seealso{\code{\link{weave}}, \code{\link{weightsLumley}}}

\examples{
x <- filter(rnorm(100), 0.9, "recursive")
isoacf(x)
acf(x, plot = FALSE)$acf
}

\keyword{regression}
\keyword{ts}

\eof
\name{kweights}
\alias{kweights}
\title{Kernel Weights}
\description{
 Kernel weights for kernel-based heteroskedasticity
 and autocorrelation consistent (HAC) covariance matrix estimators
 as introduced by Andrews (1991).
}

\usage{
kweights(x, kernel = c("Truncated", "Bartlett", "Parzen",
  "Tukey-Hanning", "Quadratic Spectral"), normalize = FALSE)
}

\arguments{
 \item{x}{numeric.}
 \item{kernel}{a character specifying the kernel used. All kernels used
    are described in Andrews (1991).}
 \item{normalize}{logical. If set to \code{TRUE} the kernels are
   normalized as described in Andrews (1991).}    
}

\value{
Value of the kernel function at \code{x}.
}

\references{
  Andrews DWK (1991),
  Heteroskedasticity and Autocorrelation Consistent Covariance Matrix Estimation.
  \emph{Econometrica}, \bold{59},
  817--858.
}

\seealso{\code{\link{kernHAC}}, \code{\link{weightsAndrews}}}

\examples{
curve(kweights(x, kernel = "Quadratic", normalize = TRUE),
      from = 0, to = 3.2, xlab = "x", ylab = "k(x)")
curve(kweights(x, kernel = "Bartlett", normalize = TRUE),
      from = 0, to = 3.2, col = 2, add = TRUE)
curve(kweights(x, kernel = "Parzen", normalize = TRUE),
      from = 0, to = 3.2, col = 3, add = TRUE)
curve(kweights(x, kernel = "Tukey", normalize = TRUE),
      from = 0, to = 3.2, col = 4, add = TRUE)
curve(kweights(x, kernel = "Truncated", normalize = TRUE),
      from = 0, to = 3.2, col = 5, add = TRUE)
}

\keyword{regression}
\keyword{ts}

\eof
\name{vcovHAC}
\alias{vcovHAC}
\title{Heteroskedasticity and Autocorrelation Consistent (HAC) Covariance Matrix Estimation}
\description{
 Heteroskedasticity and autocorrelation consistent (HAC) estimation
 of the covariance matrix of the coefficient estimates in a (generalized)
 linear regression model.
}
\usage{
vcovHAC(x, order.by = NULL, prewhite = FALSE, weights = weightsAndrews,
  diagnostics = FALSE, sandwich = TRUE, data = list())
}
\arguments{
 \item{x}{a fitted model object of class \code{"lm"} or \code{"glm"}.}
 \item{order.by}{Either a vector \code{z} or a formula with a single explanatory
    variable like \code{~ z}. The observations in the model
    are ordered by the size of \code{z}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g., a
    time series).}
 \item{prewhite}{logical or integer. Should the estimating functions
    be prewhitened? If \code{TRUE} or greater than 0 a VAR model of
    order \code{as.integer(prewhite)} is fitted via \code{ar} with
    method \code{"ols"} and \code{demean = FALSE}.}
 \item{weights}{Either a vector of weights for the autocovariances or a
    function to compute these weights based on \code{x}, \code{order.by},
    \code{prewhite} and \code{data}. If \code{weights} is a function
    it has to take these arguments. See also details.}
 \item{diagnostics}{logical. Should additional model diagnostics be returned?
   See below for details.}
 \item{sandwich}{logical. Should the sandwich estimator be computed?
    If set to \code{FALSE} only the middle matrix is returned.}
 \item{data}{an optional data frame containing the variables in the \code{order.by} 
    model. By default the variables are taken from the environment which
    \code{vcovHAC} is called from.}
}

\details{
This function constructs weighted information sandwich variance estimators
for (generalized) linear models fitted to time series data. These are basically
constructed from weighted sums of autocovariances of the estimation functions
(as extracted by \code{\link{estfun}}). The crucial step is the specification
of weights: the user can either supply \code{vcovHAC} with some vector of 
weights or with a function that computes these weights adaptively (based on
the arguments \code{x}, \code{order.by}, \code{prewhite} and \code{data}). 
Two functions for adaptively choosing weights are implemented in
\code{\link{weightsAndrews}} implementing the results of Andrews (1991) and
in \code{\link{weightsLumley}} implementing the results of Lumley (1999).
The functions \code{\link{kernHAC}} and \code{\link{weave}} respectively
are to more convenient interfaces for \code{vcovHAC} with these functions.

Prewhitening based on VAR approximations is described as suggested in
Andrews & Monahan (1992).

The covariance matrix estimators have been improved by the addition of a bias correction and an
approximate denominator degrees of freedom for test and confidence interval
construction.
}

\value{A matrix containing the covariance matrix estimate. If \code{diagnostics}
was set to \code{TRUE} this has an attribute \code{"diagnostics} which is a list 
with
  \item{bias.correction}{multiplicative bias correction}
  \item{df}{Approximate denominator degrees of freedom}
}


\references{
  Andrews DWK (1991),
  Heteroskedasticity and Autocorrelation Consistent Covariance Matrix Estimation.
  \emph{Econometrica}, \bold{59},
  817--858.

  Andrews DWK & Monahan JC (1992),
  An Improved Heteroskedasticity and Autocorrelation Consistent Covariance Matrix Estimatior.
  \emph{Econometrica}, \bold{60},
  953--966.

  Lumley A & Heagerty P (1999),
  Weighted Empirical Adaptive Variance Estimators for Correlated Data Regression.
  \emph{Journal of the Royal Statistical Society B}, \bold{61},
  459--477.

  Newey WK & West KD (1987),
  A Simple, Positive Semi-Definite, Heteroskedasticity and Autocorrelation Consistent Covariance Matrix.
  \emph{Econometrica}, \bold{55},
  703--708.
}

\seealso{\code{\link{weightsLumley}}, \code{\link{weightsAndrews}},
         \code{\link{weave}}, \code{\link{kernHAC}}}

\examples{
x <- sin(1:100)
y <- 1 + x + rnorm(100)
fm <- lm(y ~ x)
vcovHAC(fm)
vcov(fm)
}

\keyword{regression}
\keyword{ts}

\eof
\name{vcovHC}
\alias{vcovHC}
\title{Heteroskedasticity-Consistent Covariance Matrix Estimation}
\description{
 Heteroskedasticity-consistent estimation of the covariance matrix of the
 coefficient estimates in a linear regression model.
}
\usage{
vcovHC(x, order.by = NULL, data = list(),
  type = c("HC3", "const", "HC", "HC0", "HC1", "HC2", "HC4"),
  omega = NULL)
}
\arguments{
 \item{x}{a fitted model object of class \code{"lm"}.}
 \item{order.by}{formula. A formula with a single explanatory
    variable like \code{~ x}. The observations in the model
    are ordered by the size of \code{x}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g. a
    time series).}
 \item{data}{an optional data frame containing the variables in the \code{order.by} 
    model. By default the variables are taken from the environment which
    \code{vcovHC} is called from.}
 \item{type}{a character string specifying the estimation type. For
     details see below.}
 \item{omega}{a function depending on the arguments \code{residuals}
     (the residuals of the linear model), \code{diaghat} (the diagonal 
     of the corresponding hat matrix) and \code{df} (the residual degrees of
     freedom). For details see below.}
}
\details{When \code{type = "const"} constant variances are assumed and
  and \code{covHC} gives the usual estimate of the covariance matrix of
  the coefficient estimates:

  \deqn{\hat \sigma^2 (X^\top X)^{-1}}{sigma^2 (X'X)^{-1}}

  All other methods do not assume constant variances and are suitable in case of
  heteroskedasticity. \code{"HC"} (or equivalently \code{"HC0"}) gives White's
  estimator, the other estimators are refinements of this. They are all of form
  
  \deqn{(X^\top X)^{-1} X^\top \Omega X (X^\top X)^{-1}}{(X'X)^{-1} X' Omega X (X'X)^{-1}}

  and differ in the choice of Omega. This is in all cases a diagonal matrix whose 
  elements are function of the residuals, the diagonal elements of the hat matrix and
  the residual degrees of freedom. For White's estimator
  
  \code{omega <- function(residuals, diaghat, df) residuals^2}
  
  Instead of specifying a \code{type} the argument \code{omega} can also be specified
  directly. For details see the references.
}
\value{A matrix containing the covariance matrix estimate.
}
\references{

Cribari-Neto F. (2004), Asymptotic inference under heteroskedasticity
of unknown form. \emph{Computational Statistics \& Data Analysis}
\bold{45}, 215-233.

MacKinnon J. G., White H. (1985),
Some heteroskedasticity-consistent
covariance matrix estimators with improved finite sample properties.
\emph{Journal of Econometrics} \bold{29}, 305-325.

White H. (1980), A heteroskedasticity-consistent covariance matrix and
a direct test for heteroskedasticity. \emph{Econometrica} \bold{48},
817-838.

}

\seealso{\code{\link{lm}}, \code{\link[car]{hccm}},
\code{\link[lmtest]{bptest}}, \code{\link[car]{ncv.test}}}

\examples{
## generate linear regression relationship
## with homoskedastic variances
x <- sin(1:100)
y <- 1 + x + rnorm(100)
## compute usual covariance matrix of coefficient estimates
fm <- lm(y ~ x)
vcovHC(fm, type="const")
vcov(fm)

sigma2 <- sum(residuals(lm(y~x))^2)/98
sigma2 * solve(crossprod(cbind(1,x)))
}

\keyword{regression}
\keyword{ts}

\eof
\name{weightsAndrews}
\alias{weightsAndrews}
\alias{bwAndrews}
\alias{kernHAC}
\title{Kernel-based HAC Covariance Matrix Estimation}
\description{
 A set of functions implementing a class of kernel-based heteroskedasticity
 and autocorrelation consistent (HAC) covariance matrix estimators
 as introduced by Andrews (1991).
}

\usage{
kernHAC(x, order.by = NULL, prewhite = 1, bw = NULL,
  kernel = c("Quadratic Spectral", "Truncated", "Bartlett", "Parzen", "Tukey-Hanning"),
  approx = c("AR(1)", "ARMA(1,1)"), diagnostics = FALSE, sandwich = TRUE, data = list(), \dots)

weightsAndrews(x, order.by = NULL, bw = NULL,
  kernel = c("Quadratic Spectral", "Truncated", "Bartlett", "Parzen", "Tukey-Hanning"),
  prewhite = 1, data = list(), \dots)

bwAndrews(x, order.by = NULL, kernel = c("Quadratic Spectral", "Truncated",
  "Bartlett", "Parzen", "Tukey-Hanning"), approx = c("AR(1)", "ARMA(1,1)"),
  weights = NULL, prewhite = 1, data = list())
}

\arguments{
 \item{x}{a fitted model object of class \code{"lm"} or \code{"glm"}.}
 \item{order.by}{Either a vector \code{z} or a formula with a single explanatory
    variable like \code{~ z}. The observations in the model
    are ordered by the size of \code{z}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g., a
    time series).}
 \item{prewhite}{logical or integer. Should the estimating functions
    be prewhitened? If \code{TRUE} or greater than 0 a VAR model of
    order \code{as.integer(prewhite)} is fitted via \code{ar} with
    method \code{"ols"} and \code{demean = FALSE}. The default is to
    use VAR(1) prewhitening.}
 \item{bw}{numeric. The bandwidth of the kernel (corresponds to the
    truncation lag). If set to \code{NULL} (the default) it is adaptively
    chosen by the function \code{bwAndrews}.}
 \item{kernel}{a character specifying the kernel used. All kernels used
    are described in Andrews (1991).}
 \item{approx}{a character specifying the approximation method if the
    bandwidth \code{bw} has to be chosen by \code{bwAndrews}.}    
 \item{diagnostics}{logical. Should additional model diagnostics be returned?
   See \code{\link{vcovHAC}} for details.}
 \item{sandwich}{logical. Should the sandwich estimator be computed?
    If set to \code{FALSE} only the middle matrix is returned.}
 \item{data}{an optional data frame containing the variables in the \code{order.by} 
    model. By default the variables are taken from the environment which
    the function is called from.}
 \item{\dots}{further arguments passed to \code{bwAndrews}.}
 \item{weights}{numeric. A vector of weights used for weighting the estimated
   coefficients of the approximation model (as specified by \code{approx}). By
   default all weights are 1 except that for the intercept term (if there is more than
   one variable).}
}

\details{\code{kernHAC} is a convenience interface to \code{\link{vcovHAC}} using 
\code{weightsAndrews}: first a weights function is defined and then \code{vcovHAC}
is called.

The kernel weights underlying \code{weightsAndrews}
are directly accessible via the function \code{\link{kweights}} and require
the specification of the bandwidth parameter \code{bw}. If this is not specified
it can be chosen adaptively by the function \code{bwAndrews} (except for the
\code{"Truncated"} kernel). The automatic bandwidth selection is based on
an approximation of the estimating functions by either AR(1) or ARMA(1,1) processes.
To aggregate the estimated parameters from these approximations a weighted sum
is used. The \code{weights} in this aggregation are by default all equal to 1
except that corresponding to the intercept term which is set to 0 (unless there
is no other variable in the model) making the covariance matrix scale invariant.

Further details can be found in Andrews (1991).

The estimator of Newey & West (1987) can be obtained using the \code{"Bartlett"}
kernel.
}

\value{
\code{kernHAC} returns the same type of object as \code{\link{vcovHAC}}
which is typically just the covariance matrix.

\code{weightsAndrews} returns a vector of weights.

\code{bwAndrews} returns the selected bandwidth parameter.
}


\references{
  Andrews DWK (1991),
  Heteroskedasticity and Autocorrelation Consistent Covariance Matrix Estimation.
  \emph{Econometrica}, \bold{59},
  817--858.

  Newey WK & West KD (1987),
  A Simple, Positive Semi-Definite, Heteroskedasticity and Autocorrelation Consistent Covariance Matrix.
  \emph{Econometrica}, \bold{55},
  703--708.
}

\seealso{\code{\link{vcovHAC}}, \code{\link{weightsLumley}},
         \code{\link{weave}}}

\examples{
curve(kweights(x, kernel = "Quadratic", normalize = TRUE),
      from = 0, to = 3.2, xlab = "x", ylab = "k(x)")
curve(kweights(x, kernel = "Bartlett", normalize = TRUE),
      from = 0, to = 3.2, col = 2, add = TRUE)
curve(kweights(x, kernel = "Parzen", normalize = TRUE),
      from = 0, to = 3.2, col = 3, add = TRUE)
curve(kweights(x, kernel = "Tukey", normalize = TRUE),
      from = 0, to = 3.2, col = 4, add = TRUE)
curve(kweights(x, kernel = "Truncated", normalize = TRUE),
      from = 0, to = 3.2, col = 5, add = TRUE)

x <- sin(1:100)
y <- 1 + x + rnorm(100)
fm <- lm(y ~ x)
kernHAC(fm)
vcov(fm)
}

\keyword{regression}
\keyword{ts}

\eof
\name{weightsLumley}
\alias{weightsLumley}
\alias{weave}
\title{Weighted Empirical Adaptive Variance Estimation}
\description{
 A set of functions implementing a class of kernel-based heteroskedasticity
 and autocorrelation consistent (HAC) covariance matrix estimators
 as introduced by Andrews (1991).
}

\usage{
weave(x, order.by = NULL, prewhite = FALSE, C = NULL,
  method = c("truncate", "smooth"), acf = isoacf, diagnostics = FALSE, 
  sandwich = TRUE, data = list(), \dots)

weightsLumley(x, order.by = NULL, C = NULL,
  method = c("truncate", "smooth"), acf = isoacf, data = list(), \dots)
}

\arguments{
 \item{x}{a fitted model object of class \code{"lm"} or \code{"glm"}.}
 \item{order.by}{Either a vector \code{z} or a formula with a single explanatory
    variable like \code{~ z}. The observations in the model
    are ordered by the size of \code{z}. If set to \code{NULL} (the
    default) the observations are assumed to be ordered (e.g., a
    time series).}
 \item{prewhite}{logical or integer. Should the estimating functions
    be prewhitened? If \code{TRUE} or greater than 0 a VAR model of
    order \code{as.integer(prewhite)} is fitted via \code{ar} with
    method \code{"ols"} and \code{demean = FALSE}.}
 \item{C}{numeric. The cutoff constant \code{C} is by default
    4 for method \code{"truncate"} und 1 for method \code{"smooth"}.}
 \item{method}{a character specifying the method used, see details.}
 \item{acf}{a function that computes the autocorrelation function of 
    a vector, by default \code{\link{isoacf}} is used.}
 \item{diagnostics}{logical. Should additional model diagnostics be returned?
   See \code{\link{vcovHAC}} for details.}
 \item{sandwich}{logical. Should the sandwich estimator be computed?
    If set to \code{FALSE} only the middle matrix is returned.}
 \item{data}{an optional data frame containing the variables in the \code{order.by} 
    model. By default the variables are taken from the environment which
    the function is called from.}
 \item{\dots}{currently not used.}
}

\details{
\code{weave} is a convenience interface to \code{\link{vcovHAC}} using 
\code{weightsLumley}: first a weights function is defined and then \code{vcovHAC}
is called.

Both weighting methods are based on some estimate of the autocorrelation
function \deqn{\rho}{r} (as computed by \code{acf}) of the residuals of
the model \code{x}. The weights for the \code{"truncate"} method are 

\deqn{I\{n \rho^2 > C\}}{I{n * r ** 2 > C}}

and the weights for the \code{"smooth"} method are

\deqn{\min\{1, C n \rho^2\}}{min{1, C * n * r ** 2}}

where n is the number of observations in the model an C is the truncation 
constant \code{C}.

Further details can be found in Lumley & Heagerty (1999).
}

\value{
\code{weave} returns the same type of object as \code{\link{vcovHAC}}
which is typically just the covariance matrix.

\code{weightsLumley} returns a vector of weights.
}


\references{
  Lumley A & Heagerty P (1999),
  Weighted Empirical Adaptive Variance Estimators for Correlated Data Regression.
  \emph{Journal of the Royal Statistical Society B}, \bold{61},
  459--477.
}

\seealso{\code{\link{vcovHAC}}, \code{\link{weightsAndrews}},
         \code{\link{kernHAC}}}

\examples{
x <- sin(1:100)
y <- 1 + x + rnorm(100)
fm <- lm(y ~ x)
weave(fm)
vcov(fm)
}

\keyword{regression}
\keyword{ts}

\eof
