\name{agglomblock}
\alias{agglomblock}
\title{Agglomerate Two Block-Components in SCA}
\description{
  Agglomerate the two block-components which are \emph{closest}
  according to the specified \code{cluster}ing method.
}
\usage{
agglomblock(S, P, cluster = c("median","single","complete"))
}
\arguments{
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
  \item{cluster}{character specifying the clustering method; default
    \code{"median"}, see \code{\link{sca}(*, cluster=)}.}
}
\value{
  New component matrix with one block component less.
}
\seealso{\code{\link{sca}}, also for references}
\examples{
}
\keyword{multivariate}

\eof
\name{allcrit}
\alias{allcrit}
\alias{sccrit}
\title{Simple Component Quality Criterion Computation}
\description{
  Compute simple component criterion for components \code{P} on
  cor.matrix \code{S} (cumulative), using \code{sccrit()}.

  Function \code{allcrit()} computes even more criteria, some derived
  from \code{sccrit()}.
}
\usage{
allcrit(S, P, criterion, sortP = TRUE)
 sccrit(S, P, criterion, sortP = TRUE)
}
\arguments{
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
  \item{criterion}{character string specifying the optimality criterion
    to be used in \code{sccrit()} for evaluating a system of simple
    components.  One of \code{"csv"} (corrected sum of variances) or
    \code{"blp"} (best linear predictor).}
  \item{sortP}{logical indicating if \code{P} should be sorted; if true,
    \code{sortmatrix(S,P)} used in lieu of \code{P}.}
}
%-\details{
%-  ~~ If necessary, more details than the __description__  above ~~
%-}
\value{
  \code{sccrit()} returns a numeric vector, the criterion computed
  (cumulatively).

  \code{allcrit()} returns a list with components \code{varpc},
  \code{varsc}, \code{cumpc}, \code{cumsc}, \code{opt}, \code{corsc},
  and \code{maxcor};
  see the description of the \code{allcrit} component in the return value
  of \code{\link{sca}()}.
}
\author{Valentin Rousson \email{rousson@ifspm.unizh.ch} and
  Martin Maechler \email{maechler@stat.math.ethz.ch}.}
\seealso{\code{\link{quickcrit}}, \code{\link{sca}}, also for references.}
\examples{
}
\keyword{multivariate}

\eof
\name{corcomp}
\alias{corcomp}
\alias{covcomp}
#ifdef Rd2Sd
\alias{cov2cor}%% << for R versions < 1.8 (and S-plus!)
#endif
\title{Covariance and Correlation Matrix of Components P on S}
\description{
  \code{covcomp} returns the variance-covariance matrix of the
  components P on S, and \code{corcomp} returns the correlation matrix.
}
\usage{
corcomp(S, P)
covcomp(S, P)
#ifdef Rd2Sd
cov2cor(V)%% << for R versions < 1.8 (and S-plus!)
#endif
}
\arguments{
  \item{S}{correlation/covariance matrix of the \eqn{p} original variables.}
  \item{P}{component matrix of dimension \eqn{p \times b}{p x b}.}
#ifdef Rd2Sd
  \item{V}{symmetric numeric matrix, usually positive definite such as a
    covariance matrix.}
#endif
}
%-\details{
%-  ~~ If necessary, more details than the __description__  above ~~
%-}
\value{
  a square \eqn{b \times b}{b x b} matrix.
}
\author{Valentin Rousson \email{rousson@ifspm.unizh.ch} and
  Martin Maechler \email{maechler@stat.math.ethz.ch}.}
\seealso{\code{\link{sca}}, also for references}
\examples{
data(USJudgeRatings)
S.jr <- cor(USJudgeRatings)
sca.jr <- sca(S.jr, b=4, inter=FALSE)
Vr <- covcomp(S.jr, P = sca.jr$simplemat)
Vr
Cr <- corcomp(S.jr, P = sca.jr$simplemat)
Cr
}
\keyword{multivariate}

\eof
\name{firstpcres}
\alias{firstpcres}
\title{First Principal Component of Residuals given Components}
\description{
  Return the first principal component of residuals of \code{S} given
  the components \code{P}.
}
\usage{
firstpcres(S, P)
}
\arguments{
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
}
\value{
  numeric vector; actually, the first eigenvector of
  \eqn{S - A' (A P)^{-1} A}{S - A' (A P)^(-1) A} where \eqn{A := P' S}.
}
\seealso{\code{\link{sca}}, also for references}
\examples{
}
\keyword{multivariate}

\eof
\name{hearlossC}
\alias{hearlossC}
\docType{data}
\title{Hearing Loss Correlation Data}
\description{
  The data consist of eight measurements of hearing loss taken on 100
  males, aged 39, who had no indication of hearing difficulties.  These
  measurements are decibel loss (in comparison to a reference standard)  at
  frequencies 500Hz, 1000Hz, 2000Hz and 4000Hz for the left and the right
  ear, respectively.
}
\usage{data(hearlossC)}
\format{
  Eight Variables, first the ones for ``Left'', than for the ``Right''.
  The frequences are abbreviated, e.g., \code{2k} for \emph{2000 Hz} or
   \code{5c} for \emph{500 Hz}.
   The variable names are (in this order)
  \code{"Left5c",  "Left1k",  "Left2k",  "Left4k",
    "Right5c", "Right1k", "Right2k", "Right4k"}.
}
% \details{
% ..
% }
\source{
  This is the correlation matrix of data described in Chapter 5 of
  Jackson (1991).
}
\references{
  Jackson, J.E. (1991)
  \emph{A User's Guide to Principal Components}.
  John Wiley, New York.
}
\examples{
data(hearlossC)
symnum(hearlossC)
}
\keyword{datasets}

\eof
\name{maxmatrix}
\alias{maxmatrix}
\title{Largest Element in Correlation Matrix}
\description{
 return position and value of the largest element of a correlation matrix R
 (without taking into account the diagonal elements)
}
\usage{
maxmatrix(R)
}
\arguments{
  \item{R}{a square symmetric numeric matrix}
}
\value{
  a list with components
  \item{row}{row index of maximum}
  \item{col}{col index of maximum}
  \item{val}{value of maximum, i.e. \code{val == R[row,col]}.}
}
\seealso{\code{\link{sca}}, also for references}
\examples{
data(reflexesC)
maxmatrix(reflexesC) # ->  0.98 at  [1, 2]
}
\keyword{multivariate}

\eof
\name{nextdiff}
\alias{nextdiff}
\title{Compute the Next Simple Difference-Component for SCA}
\description{
  Compute the next simple difference-component; this is an auxiliary
  function for \code{\link{sca}}.
}
\usage{
nextdiff(S, P, withinblock, criterion)
}
\arguments{
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
  \item{withinblock}{logical indicating whether any given
    difference-component should only involve variables belonging to the
    same block-component.}
  \item{criterion}{character string specifying the optimality criterion,
    see \code{\link{sccrit}} for details.}
}
\details{
  Uses \code{\link{firstpcres}(S,P)} and subsequently
  \code{\link{shrinkdiff}()}, the latter in a loop when
  \code{withinblock} is true.

  In order to ensure uniqueness, we ensure that the first (non zero)
  entry of the principal component is always \emph{positive}.
}
\value{
  a list with components
  \item{P}{the new component matrix, i.e. the input \code{P} with one
    new column appended.}
  \item{nextpc}{the next principal component with many entries set to 0.}
}
\author{Valentin Rousson \email{rousson@ifspm.unizh.ch} and
  Martin Maechler \email{maechler@stat.math.ethz.ch}.}
\seealso{\code{\link{shrinkdiff}}; \code{\link{sca}}, also for references}
\examples{
}
\keyword{multivariate}

\eof
\name{percent}
\alias{percent}
\title{Simple Formatting of Percentages}
\description{
  Returns strings of the same length as \code{p}, displaying the
  \code{100 * p} percentages.
}
\usage{
percent(p, d = 0, sep = " ")
}
\arguments{
  \item{p}{number(s) in \eqn{[0,1]} -- to be ``displayed'' as percentage(s).}
  \item{d}{number of digits after decimal point.}
  \item{sep}{separator to use before the final \code{"\%"}.}
}
\value{
  character vector of the same length as \code{p}.
}
\author{Martin Maechler}
\examples{
percent(0.25)
noquote(percent((1:10)/10))
(pc <- percent((1:10)/30, 1, sep=""))
noquote(pc)
}
\keyword{utilities}
\keyword{print}

\eof
\name{pitpropC}
\alias{pitpropC}
\docType{data}
\title{Pitprops Strength Correlation Data}
\description{
  This correlation matrix was published in Jeffers (1967) and was calculated
  from 180 observations. The 13 variables were used as explanatory variables
  in a regression problem which arised from a study on the strength of
  pitprops cut from home-grown timber.}
\usage{data(pitpropC)}
\format{
  Its a correlation matrix of 13 variables which have the following meaning:
  \tabular{rll}{
    [,1] \tab  TOPDIAM \tab  Top diameter of the prop in inches\cr
    [,2] \tab  LENGTH  \tab  Length of the prop in inches\cr
    [,3] \tab  MOIST   \tab  Moisture content of the prop, expressed as a percentage of
    				the dry weight\cr
    [,4] \tab  TESTSG  \tab  Specific gravity of the timber at the time of the test\cr
    [,5] \tab  OVENSG  \tab  Oven-dry specific gravity of the timber\cr
    [,6] \tab  RINGTOP \tab  Number of annual rings at the top of the prop\cr
    [,7] \tab  RINGBUT \tab  Number of annual rings at the base of the prop\cr
    [,8] \tab  BOWMAX  \tab  Maximum bow in inches\cr
    [,9] \tab  BOWDIST \tab  Distance of the point of maximum bow from the top of the
				prop in inches\cr
    [,10]\tab  WHORLS  \tab  Number of knot whorls\cr
    [,11]\tab  CLEAR   \tab  Length of clear prop from the top of the prop in inches\cr
    [,12]\tab  KNOTS   \tab  Average number of knots per whorl\cr
    [,13]\tab  DIAKNOT \tab  Average diameter of the knots in inches
  }% tab
}
\details{
  Jeffers (1967) replaced these 13 variables by their first six
  principal components.  As noted by Vines (2000), this is an example
  where simple structure has proven difficult to detect in the past.
}
\references{
  Jeffers, J.N.R. (1967)
  Two case studies in the application of principal components analysis.
  \emph{Appl. Statist.} \bold{16}, 225--236.

  Vines, S.K. (2000)
  Simple principal components.
  \emph{Appl. Statist.} \bold{49}, 441--451.
}
\examples{
data(pitpropC)
symnum(pitpropC)
}
\keyword{datasets}

\eof
\name{quickcrit}
\alias{quickcrit}
\title{Additional Contribution of New Component to the SC System}
\description{
  Compute the additional contribution of a new component to the simple
  component system P on S.
}
\usage{
quickcrit(newcomp, S, P, criterion)
}
\arguments{
  \item{newcomp}{numeric vector, typically the result of
    \code{\link{simpvector}()}.}
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
  \item{criterion}{character string specifying the optimality criterion,
    see \code{\link{sccrit}} for details.}
}
\value{
  ...
}
\seealso{\code{\link{sccrit}}; further \code{\link{sca}}, also for
  references.}
\examples{
}
\keyword{multivariate}

\eof
\name{reflexesC}
\alias{reflexesC}
\docType{data}
\title{Human Reflexes Correlation Data}
\description{
  This correlation matrix was published in Jolliffe (2002, p.58).  The data
  consist of measurements of strength of reflexes at ten sites of the body,
  taken for 143 individuals. The variables come in five pairs, corresponding
  to right and left measurements on triceps, biceps, wrists, knees and
  ankles, respectively.
}
\usage{data(reflexesC)}
\format{
  It's a \eqn{10 x 10} correlation matrix, i.e. symmetric, and
  diagonal \code{1}.
  The five pairs of variables are (in this order)
  \code{"triceps.R", "triceps.L", "biceps.R", "biceps.L",
    "wrist.R", "wrist.L", "knee.R", "knee.L",
    "ankle.R", "ankle.L"}.
}
\references{
  Jolliffe, I.T. (2002)
  Principal Component Analysis (2nd ed.).
  Springer, New York.
}
\examples{
data(reflexesC)
symnum(reflexesC)
}
\keyword{datasets}

\eof
\name{sca-internal}
\alias{isblockempty}
\alias{isdiffcomp}
\alias{iswithinblock}
\alias{eqmatrix}
\alias{normmatrix}
\alias{redrawMatrix}

\title{Internal Functions of Package sca}
\description{
  Internal function of the \file{sca} package.
}
\usage{
isblockempty(P)
isdiffcomp(x)
iswithinblock(d, P)
eqmatrix(A, B)
normmatrix(P)
redrawMatrix(phase, I, Imax, P, crit, nextpc, okcomp, message, warn)
}
\details{
  These are not to be called by the user.
}
\keyword{internal}

\eof
\name{sca}
\title{Simple Component Analysis -- Interactively}
\alias{sca}
\alias{print.simpcomp}
\description{
  A system of simple components calculated from a correlation (or
  variance-covariance) matrix is built (interactively if \code{interactive =
    TRUE}) following the methodology of Rousson and Gasser (2003).
}
\usage{
sca(S, b = if(interactive) 5, d = 0, qmin = if(interactive) 0 else 5,
    corblocks = if(interactive) 0 else 0.3,
    criterion = c("csv", "blp"), cluster = c("median","single","complete"),
    withinblock = TRUE, invertsigns = FALSE,
    interactive = dev.interactive())
print.simpcomp(x, ndec = 2, \dots)
}
\arguments{
  \item{S}{the correlation (or variance-covariance) matrix to be analyzed.}
  \item{b}{the number of block-components initially proposed.}
  \item{d}{the number of difference-components initially proposed.}
  \item{qmin}{if larger than zero, the number of difference-components
    is chosen such that the system contains at least \code{qmin}
    components (overriding argument \code{d}!).}
  \item{corblocks}{if larger than zero, the number of block-components
    is chosen such that correlations among them are all smaller than
    \code{corblocks} (overriding argument \code{b}).}
  \item{criterion}{character string specifying the optimality criterion
    to be used for evaluating a system of simple components.  One of
    \code{"csv"} (corrected sum of variances) or \code{"blp"} (best
    linear predictor); can be abbreviated.}
  \item{cluster}{character string specifying the clustering method to be
    used in the definition of the block-components.  One of
    \code{"single"} (single linkage), \code{"median"} (median linkage)
    or \code{"complete"} (complete linkage) can be abbreviated.}
  \item{withinblock}{a logical indicating whether any given
    difference-component should only involve variables belonging to the
    same block-component.}
  \item{invertsigns}{a logical indicating whether the sign of some
    variables should be inverted initially in order to avoid negative
    correlations.}
  \item{interactive}{a logical indicating whether the system of simple
    components should be built interactively.  If \code{interactive=FALSE}, an
    optimal system of simple components is automatically calculated
    without any intervention of the user (according to \code{b} or
    \code{corblocks}, and to \code{d} or \code{qmin}).

    By default, \code{interactive = \code{\link{dev.interactive}}()} (which is
    true if \code{\link{interactive}()} and \code{\link{.Device}} is an
    interactive graphics device).}

%% print.*()
  \item{x}{an object of class \code{sca}, typically the result of
    \code{\link{sca}(..)}.}
  \item{ndec}{number of decimals \emph{after} the dot, for the
    percentages printed.}
  \item{\dots}{further arguments, passed to and from methods.}
}
\details{
  When confronted with a large number \eqn{p} of variables measuring
  different aspects of a same theme, the practitionner may like to
  summarize the information into a limited number \eqn{q} of components.  A
  \emph{component} is a linear combination of the original variables, and
  the weights in this linear combination are called the \emph{loadings}.
  Thus, a system of components is defined by a \eqn{p} times \eqn{q} dimensional
  matrix of loadings.

  Among all systems of components, principal components (PCs) are
  optimal in many ways.  In particular, the first few PCs extract a
  maximum of the variability of the original variables and they are
  uncorrelated, such that the extracted information is organized in an
  optimal way: we may look at one PC after the other, separately,
  without taking into account the rest.

  Unfortunately PCs are often difficult to interpret. The goal of Simple
  Component Analysis is to replace (or to supplement) the optimal but
  non-interpretable PCs by suboptimal but interpretable \emph{simple
  components}. The proposal of Rousson and Gasser (2003) is to look for
  an optimal system of components, but only among the simple ones,
  according to some definition of optimality and simplicity. The outcome
  of their method is a simple matrix of loadings calculated from the
  correlation matrix \eqn{S} of the original variables.

  Simplicity is not a guarantee for interpretability (but it helps in
  this regard).  Thus, the user may wish to partly modify an optimal
  system of simple components in order to enhance
  interpretability.  While PCs are by definition 100\% optimal, the
  optimal system of simple components proposed by the procedure \code{sca}
  may be, say, 95\%, optimal, whereas the simple system altered by the
  user may be, say, 93\% optimal. It is ultimately to the user to decide
  if the gain in interpretability is worth the loss of optimality.

  The interactive procedure \code{sca} is intended to assist the user in
  his/her choice for an interptetable system of simple components. The
  algorithm consists of three distinct stages and proceeds in an
  interative way. At each step of the procedure, a simple matrix of
  loadings is displayed in a window. The user may alter this matrix by
  clicking on its entries, following the instructions given there.  If
  all the loadings of a component share the same sign, it is a
  ``block-component''.  If some loadings are positive and some loadings
  are negative, it is a ``difference-component''.  Block-components are
  arguably easier to interpret than
  difference-components. Unfortunately, PCs almost always contain only
  one block-component. In the procedure \code{sca}, the user may choose the
  number of block-components in the system, the rationale being to have
  as many block-components such that correlations among them are below
  some cut-off value (typically .3 or .4).

  Simple block-components should define a partition of the original
  variables. This is done in the first stage of the procedure \code{sca}. An
  agglomerative hierarchical clustering procedure is used there.

  The second stage of the procedure \code{sca} consists in the definition of
  simple difference-components.  Those are obtained as simplified
  versions of some appropriate ``residual components''. The idea is to
  retain the large loadings (in absolute value) of these residual
  components and to shrink to zero the small ones. For each
  difference-component, the interactive procedure \code{sca} displays the
  loadings of the corresponding residual component (at the right side of
  the window), such that the user may know which variables are
  especially important for the definition of this component.

  At the third stage of the interactive procedure \code{sca}, it is possible
  to remove some of the difference-components from the system.

  For many examples, it is possible to find a simple system which is 90\%
  or 95\% optimal, and where correlations between components are below 0.3
  or 0.4. When the structure in the correlation matrix is complicated, it
  might be advantageous to invert the sign of some of the variables in
  order to avoid as much as possible negative correlations. This can be
  done using the option `invertsigns=TRUE'.

  In principle, simple components can be calculated from a correlation
  matrix or from a variance-covariance matrix. However, the definition
  of simplicity used is not well adapted to the latter case, such that
  it will result in systems which are far from being 100\%
  optimal. Thus, it is advised to define simple components from a
  correlation matrix, not from a variance-covariance matrix.
}
\value{
  An object of class \code{simpcomp} which is basically as list with
  the following components:
  \item{simplemat}{an integer matrix defining a system of simple
    components.  The rows correspond to variables and the columns correspond to
    components.}
  \item{loadings}{loadings of simple components.  This is a normalized
    (by \code{\link{normmatrix}}) version of \code{simplemat}.}
  \item{allcrit}{a list containing the following components:
    \describe{
      \item{varpc}{a vector containing the percentage of total
	variability accounted by each of the the first \code{nblock + ndiff}
	principal components of \code{S}.}
      \item{varsc}{a vector containing the percentage of total
	variability accounted by each of the simple components defined by
	\code{simplemat}.}
      \item{cumpc}{the sum of varpc, indicating the percentage of total
	variability accounted by the first \code{nblock + ndiff} principal
	components of \code{S}.}
      \item{cumsc}{a score indicating the percentage of total
	variability accounted by the system of simple
	components. \code{cumsc} is calculated according to \code{criterion}.}
      \item{opt}{indicates the optimality of the system of simple
	components and is computed as \code{cumsc/cumpc}.}
      \item{corsc}{correlation matrix of the simple components defined
	by \code{simplemat}.}
      \item{maxcor}{a list with the following components:
	\describe{
	  \item{row}{label of the row of the maximum value in \code{corsc}.}
	  \item{col}{label of the column of the maximum value in \code{corsc}.}
	  \item{val}{maximum value in \code{corsc} (in absolute value).}
	}
      }
  }}% end `allcrit' list
  \item{nblock}{number of block-components in \code{simplemat}.}

  \item{ndiff}{number of difference-components in \code{simplemat}.}

  \item{criterion}{as above.}
  \item{cluster}{as above.}
  \item{withinblock}{as above.}
  \item{invertsigns}{as above}

  \item{vardata}{the correlation (or variance-covariance) matrix which
    was analyzed. In principle it should be equal to argument \code{S} above,
    except if it has been transformed in order to avoid negative
    correlations.}
}
\references{
  Rousson, V. and Gasser, Th. (2003)
  Simple Component Analysis. Submitted.

  Rousson, V. and Gasser, Th. (2003)
  \emph{Some Case Studies of Simple Component Analysis}.
  Manuscript.

  Gervini, D. and Rousson, V. (2003)
  \emph{Some Proposals for Evaluating Systems of Components in Dimension
    Reduction Problems}.
  Submitted.
}
\author{Valentin Rousson \email{rousson@ifspm.unizh.ch} and
  Martin Maechler \email{maechler@stat.math.ethz.ch}.}
\seealso{\code{\link[mva]{pca}} etc.}
\examples{
data(pitpropC)
sc.pitp <- sca(pitpropC, interactive=FALSE)
sc.pitp
## to see it's low-level components:
str(sc.pitp)

## Let `X' be a matrix containing some data set whose rows correspond to
## subjects and whose columns correspond to variables. For example:

%% MM: need a real matrix here!
library(MASS)
Sig <- function(p, rho) { r <- diag(p); r[col(r) != row(r)] <- rho; r}
rmvN <- function(n,p, rho)
        mvrnorm(n, mu=rep(0,p), Sigma= Sig(p, rho))
X <- cbind(rmvN(100, 3, 0.7),
           rmvN(100, 2, 0.9),
           rmvN(100, 4, 0.8))
%% end MM

## An optimal simple system with at least 5 components for the data in `X',
## where the number of block-components is such that correlations among
## them are all smaller than 0.4, can be automatically obtained as:

(r <- sca(cor(X), qmin=5, corblocks=0.4, interactive=FALSE))

## On the other hand, an optimal simple system with two block-components
## and two difference-components for the data in `X' can be automatically
## obtained as:

(r <- sca(cor(X), b=2, d=2, qmin=0, corblocks=0, interactive=FALSE))

## The resulting simple matrix is contained in `r$simplemat'.
## A matrix of scores for such simple components can then be obtained as:

(Z <- scale(X) \%*\% r$loadings)

## On the other hand, scores of simple components calculated from the
## variance-covariance matrix of `X' can be obtained as:

r <- sca(var(X), b=2, d=2, qmin=0, corblocks=0, interactive=FALSE)
Z <- scale(X, scale=FALSE) \%*\% r$loadings

## One can also use the program interactively as follows:

if(interactive()) {
  r <- sca(cor(X), corblocks=0.4, qmin=5, interactive = TRUE)

  ## Since the interactive part of the program is active here, the proposed
  ## system can then be  modified according to the user's wishes. The
  ## result of the procedure will be contained in `r'.
}%interactive

}
\keyword{multivariate}

\eof
\name{shrinkdiff}
\alias{shrinkdiff}
\title{Shrink Component Towards a Simple Difference-Component in SCA}
\description{
  Shrinks a (principal) component towards a simple difference-component
  in \code{\link{sca}}.
}
\usage{
shrinkdiff(zcomp, S, P, criterion)
}
\arguments{
  \item{zcomp}{a component vector to be \emph{simplified}.}
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
  \item{criterion}{character string specifying the optimality criterion,
    see \code{\link{sccrit}} for details.}
}
% \details{
%  ......
% }
\value{
  a list with the components
  \item{scompmax}{a new simple component vector, typically result of
    \code{\link{simpvector}}.}
  \item{critmax}{the (optimal) value of the criterion, achieved for
    \code{scompmax}.}
}
\seealso{\code{\link{sca}}, also for references}
% \examples{
% }
\keyword{multivariate}

\eof
\name{simpvector}
\alias{simpvector}
\title{Simplify a (Principal Component) Vector to a Simple Component}
\description{
  Simplifies the vector \code{x} to become a ``simple'' component vector
  (of the same size).
}
\usage{
simpvector(x)
}
\arguments{
  \item{x}{numeric vector of length \code{n}, say.}
}
%\details{
%  ... %%% FIXME
%}
\value{
  a ``simplified'' version of \code{x}, i.e. an integer vector of the
  same length and each entry with the same signs.
}
\seealso{\code{\link{sca}}, also for references}
\examples{
 x0 <- c(-2:3, 3:-1,0:3,1,1)
 cbind(x0, simpvector(x0)) # entries (-11, 0, 3)
}
\keyword{multivariate}

\eof
\name{sortmatrix}
\alias{sortmatrix}
\title{Sort Simple Component Matrix}
\description{
 Reorder the columns of a component matrix \code{P} by decreasing
 variances of components where the block-components come first, the
 difference components afterwards.
}
\usage{
sortmatrix(S, P)
}
\arguments{
  \item{S}{correlation/covariance matrix}
  \item{P}{component matrix}
}
\value{
  numeric matrix which is just \code{P} with columns reordered.
}
\seealso{\code{\link{sca}}, also for references}
\examples{
}
\keyword{multivariate}

\eof
