\name{T.Owen}
\alias{T.Owen}
\title{
Owen's function
}
\description{
Evaluates function \emph{T(h,a)} studied by D.B.Owen
}
\usage{
T.Owen(h, a, jmax=50, cut.point=6)
}
\arguments{
\item{h}{
a numerical vector. Missing values (\code{NA}s) and \code{Inf} are allowed.
}
\item{a}{
a numerical scalar. \code{Inf} is allowed.
}
\item{jmax}{
an integer scalar value which regulates the accuracy of the result.
See DETAILS below for explanation.
}
\item{cut.point}{
a scalar value which regulates the behaviour of the algorithm, as
explained by the details below.
}}
\value{
a numerical vector
}
\details{
If \code{a>1} and \code{0<h<=cut.point}, a series expansion is used,
truncated after \code{jmax} terms.
If \code{a>1} and \code{h>cut.point}, an asymptotic approximation is used.
In the other cases, various reflection properties of the function
are exploited. See the reference below for more information.
}
\section{Background}{
The function \emph{T(h,a)} is useful for the computation of the bivariate
normal distribution function and related quantities, including the 
distribution function of a skew-normal variate, \code{psn}.
See the reference below for more information on  \emph{T(h,a)}.
}
\references{
Owen, D. B. (1956).
Tables for computing bivariate normal probabilities.
\emph{Ann. Math. Statist.}
\bold{27}, 1075-1090.
}
\seealso{
\code{\link{pnorm2}}, \code{\link{psn}}
}
\examples{
owen <- T.Owen(1:10, 2)
}
\keyword{math}

\eof
\name{ais}
\alias{ais}
\title{Australian Institute of Sport data}
\usage{data(ais)}
\description{
  Data on 102 male and 100 female athletes collected at the Australian
  Institute of Sport, courtesy of Richard Telford and Ross Cunningham.
}

\format{
  A data frame with 202 observations on 13 variables.
  \tabular{rll}{
    [, 1] \tab sex  \tab sex \cr
    [, 2] \tab sport\tab sport \cr
    [, 3] \tab rcc  \tab red cell count \cr
    [, 4] \tab wcc  \tab white cell count \cr
    [, 5] \tab Hc   \tab Hematocrit \cr
    [, 6] \tab Hg   \tab Hemoglobin \cr
    [, 7] \tab Fe   \tab plasma ferritin concentration \cr
    [, 8] \tab bmi  \tab body mass index, weight/(height) \cr
    [, 9] \tab ssf  \tab sum of skin folds \cr
    [,10] \tab Bfat \tab body fat percentage \cr
    [,11] \tab lbm  \tab lean body mass \cr
    [,12] \tab Ht   \tab height (cm) \cr
    [,13] \tab Wt   \tab weight (Kg) \cr
  }
}
\source{
  Cook and Weisberg (1994),
  \emph{An Introduction to Regression Graphics}.                   
  John Wiley & Sons, New York.
}
\examples{
data(ais)
attach(ais)
pairs(ais[,c(3:4,10:13)], main = "AIS data")
plot(Wt~sport)
}
\keyword{datasets}

\eof
\name{cp.to.dp}
\alias{cp.to.dp}
\alias{dp.to.cp}
\title{
Conversion between equivalent parametrizations  
}
\description{
  Convert direct parameters (DP) to centred parameters (CP) of the
  one-dimensional skew-normal distribution and \emph{vice versa}
}
\usage{
cp.to.dp(param)
dp.to.cp(param)
}
\arguments{
\item{param}{
a vector of length at least three.
If \code{lenght(param)} is \code{m+2}, then the first \code{m} components refer
to the regression coefficients (or the location parameter, in
case \code{m} is 1), and the remaining two components refer to scale and
shape, respectively; their role is preserved across parametrizations.
}}
\value{
a vector of the same length of \code{param}, representing \code{param} in the
alternative parametrization; \code{cp.to.dp} converts centred to direct
parameters, \code{dp.to.cp} converts direct to centred parameters.
}
\details{For a description of the two parametrizations, 
see the reference below.
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{sn.mle}}, \code{\link{sn.em}}
}
\examples{
cp <- dp.to.cp(c(30,30,2,4))
dp <- cp.to.dp(cp)
}
\keyword{distribution}

\eof
\name{dmsn}
\alias{dmsn}
\alias{pmsn}
\alias{rmsn}
\title{
Multivariate skew-normal distribution
}
\description{
Probability density function, distribution function and random number 
generation for the multivariate skew-normal (MSN) distribution.
}
\usage{
dmsn(x, xi=rep(0,d), Omega, alpha)
pmsn(x, xi=rep(0,d), Omega, alpha, ...)
rmsn(n=1, xi=rep(0,d), Omega, alpha)
}
\arguments{
\item{x}{
for \code{dmsn}, this is either a vector of length \code{d} or a matrix 
with \code{d} columns, where \code{d} is \code{length(alpha)}, giving 
the coordinates of the point(s) where the density must be avaluated;
for \code{pmsn}, only a vector of length \code{d} is allowed.
}
\item{Omega}{
a postive-definite covariance matrix of dimension \code{(d,d)}.
}
\item{alpha}{
a numeric vector which regulates the shape of the density.
}
\item{xi}{
a numeric vector of lenght \code{d}, or a matrix with \code{d} columns,
representing the location parameter of the distribution.
If \code{xi} is a matrix, its dimensions must agree with those of \code{x}.
}
\item{n}{
a numeric value which represents the number of random vectors
to be drawn.
}
\item{...}{
additional parameters passed to \code{pmvnorm}
}}
\value{
A vector of density values (\code{dmsn}), or a single probability 
(\code{pmsn}) or a matrix of random  points (\code{rmsn}).
}
\section{Background}{
The multivariate skew-normal distribution is discussed by
Azzalini and Dalla Valle (1996); the \code{(Omega,alpha)} parametrization
adopted here is the one of Azzalini and Capitanio (1999).
}
\details{
  The positive-definiteness of \code{Omega} is not tested for
  efficiency reasons. Function
  \code{pmsn} requires  \code{pmvnorm} from \code{library(mvtnorm)};
  the accuracy of its computation can be controlled via use of \code{...}
}
\references{
Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dsn}}, \code{\link{msn.fit}}, \code{\link{dmst}},
\code{\link[mvtnorm]{pmvnorm}}
}
\examples{
x <- seq(-3,3,length=15)
xi <- c(0.5, -1)
Omega <- diag(2)
Omega[2,1] <- Omega[1,2] <- 0.5
alpha <- c(2,2)
pdf <- dmsn(cbind(x,2*x-1), xi, Omega, alpha)
rnd <- rmsn(10,  xi, Omega, alpha)
library(mvtnorm)                        # only once in the session
cdf <- pmsn(c(2,1), xi, Omega, alpha)
}
\keyword{distribution}
\keyword{multivariate}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{dmst}
\alias{dmst}
\alias{pmst}
\alias{rmst}
\title{
Multivariate skew-t distribution
}
\description{
Probability density function, distribution function and random number 
generation for the multivariate skew-t (MST) distribution.
}
\usage{
dmst(x, xi=rep(0,d), Omega, alpha, df=Inf)
pmst(x, xi=rep(0,d), Omega, alpha, df=Inf, ...)
rmst(n=1, xi=rep(0,d), Omega, alpha, df=Inf)
}
\arguments{
\item{x}{
for \code{dmsn}, this is either a vector of length \code{d} or a matrix 
with \code{d} columns, where \code{d} is \code{length(alpha)}, giving 
the coordinates of the point(s) where the density must be avaluated;
for \code{pmsn}, only a vector of length \code{d} is allowed.
}
\item{xi}{
a numeric vector of lenght \code{d}, or a matrix with \code{d} columns,
representing the location parameter of the distribution.
If \code{xi} is a matrix, its dimensions must agree with those of \code{x}.
}
\item{Omega}{
a positive-definite covariance matrix of dimension \code{(d,d)}.
}
\item{alpha}{
a numeric vector which regulates the shape of the density.
}
\item{df}{
degrees of freedom (scalar); default is \code{df=Inf} which corresponds 
to the multivariate skew-normal distribution.
}
\item{n}{
a numeric value which represents the number of random vectors
to be drawn.
}
\item{...}{
additional parameters passed to \code{pmvt}
}}
\value{
A vector of density values (\code{dmst}), or a matrix of random 
points (\code{rmst}).
}
\section{Background}{
The family of multivariate skew-t distributions is an extension of the 
multivariate Student's t family, via the introduction of a \code{shape} 
parameter which regulates skewness; when \code{shape=0}, the skew-t 
distribution reduces to the usual t distribution. 
When \code{df=Inf} the distribution reduces to the multivariate skew-normal 
one; see \code{dmsn}. See the reference below for additional information.
}
\details{
  The positive-definiteness of \code{Omega} is not tested for
  efficiency reasons. Function
  \code{pmst} requires \code{pmvt} from \code{library(mvtnorm)};
  the accuracy of its computation can be controlled via use of \code{...}
}
\references{
  Azzalini, A. and Capitanio, A. (2003).
  Distributions generated by perturbation of symmetry 
  with emphasis on a multivariate skew \emph{t} distribution.
  \emph{J.Roy. Statist. Soc. B} \bold{65}, 367--389.
}
\seealso{
\code{\link{dst}}, \code{\link{mst.fit}}, \code{\link{dmsn}},
\code{\link[mvtnorm]{pmvt}}
}
\examples{
x <- seq(-4,4,length=15)
xi <- c(0.5, -1)
Omega <- diag(2)
Omega[2,1] <- Omega[1,2] <- 0.5
alpha <- c(2,2)
pdf <- dmst(cbind(x,2*x-1), xi, Omega, alpha, df=5)
rnd <- rmst(10,  xi, Omega, alpha, 6)
library(mvtnorm)                # only once in the session
cdf <- pmst(c(2,1), xi, Omega, alpha, df=5)

}
\keyword{distribution}
\keyword{multivariate}


\eof
\name{dsn}
\alias{dsn}
\alias{psn}
\alias{qsn}
\alias{rsn}
\title{
Skew-Normal Distribution
}
\description{
Density function, distribution function, quantiles and random number
generation for the skew-normal (SN) distribution.
}
\usage{
dsn(x, location=0, scale=1, shape=0, log=FALSE)
psn(x, location=0, scale=1, shape=0, ...)
qsn(p, location=0, scale=1, shape=0, tol=1e-8, ...)
rsn(n=1, location=0, scale=1, shape=0)
}
\arguments{
\item{x}{
  vector of quantiles. Missing values (\code{NA}s) and \code{Inf}'s
  are allowed.
}
\item{p}{
  vector of probabilities. Missing values (\code{NA}s) are allowed.
}
\item{location}{
  vector of location parameters.
}
\item{scale}{
  vector of (positive) scale parameters.
}
\item{shape}{
  vector of shape parameters.
  With \code{psn} and \code{qsn}, it must be of length 1.
}
\item{n}{
  sample size.
}
\item{tol}{
  a scalar value which regulates the accuracy of the result of \code{qsn}.
}
\item{log}{
  logical flag used in \code{dsn} (default \code{FALSE}).
  When \code{TRUE}, the logarithm of the density values is returned.
}
\item{...}{
  additional parameters passed to \code{T.Owen}
}}
\value{
density (\code{dsn}), probability (\code{psn}),
quantile (\code{qsn}) or  random sample (\code{rsn})
from the skew-normal distribution with given \code{location}, \code{scale}
and \code{shape} parameters.
}
\section{Background}{
The family of skew-normal distributions is an extension of the normal
family, via the introdution of a \code{shape} parameter which regulates
skewness; when \code{shape=0}, the skew-normal distribution reduces to the
normal one.  The density of the SN distribution in the "standard" case
having \code{location=0} and \code{scale=1}  is
\code{2*dnorm(x)*pnorm(shape*x)}.
A multivariate version of the distribution exists.
See the references below for additional information.
}
\section{Details}{
\code{psn} and \code{qsn} make use of function \code{T.Owen}
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.


Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.
}
\seealso{
\code{\link{T.Owen}}, \code{\link{dmsn}}, \code{\link{dst}}
}
\examples{
pdf <- dsn(seq(-3,3,by=0.1), shape=3)
cdf <- psn(seq(-3,3,by=0.1), shape=3)
qu <- qsn(seq(0.1,0.9,by=0.1), shape=-2)
rn <- rsn(100, 5, 2, 5)
}
\keyword{distribution}

\eof
\name{dsn2.plot}
\alias{dsn2.plot}
\title{
Plot of Bivariate Skew-normal Density Function
}
\description{
Produces a contour plot of the density function of a bivariate 
skew-normal variate.
}
\usage{
dsn2.plot(x, y, xi, Omega, alpha,  ...)
}
\arguments{
\item{x}{
vector of values of the first component.
}
\item{y}{
vector of values of the second component.
}
\item{xi}{
a vector of length 2 containing the location parameter.
}
\item{Omega}{
a 2 by 2 matrix containing a covariance matrix.
}
\item{alpha}{
a vector of length 2 containing the shape parameter.
}
\item{...}{
additional parameters to be passed to \code{contour}.
}}
\value{
A list containing the original input parameters plus a matrix
containing the density function evaluated at the grid formed 
by the \code{x} and \code{y} values. 
}
\section{Side effects}{
A contour plot is produced on the graphical window.
}
\details{
\code{dsn2.plot} makes use of function \code{dmsn}.
}
\section{Background}{
The multivariate skew-normal distribution is discussed by
Azzalini and Dalla Valle (1996); the \code{(Omega,alpha)} parametrization
adopted here is the one of Azzalini and Capitanio (1999).
}
\references{
Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.

Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dmsn}}, \code{link{dsn2.plot}}
}
\examples{
x <- y <- seq(-5, 5, length=35)
dsn2.plot(x, y, c(-1,2), diag(c(1,2.5)), c(2,-3), df=5)
}
\keyword{distribution}
% Comment

\eof
\name{dst}
\alias{dst}
\alias{pst}
\alias{qst}
\alias{rst}
\title{
Skew-t Distribution
}
\description{
Density function, distribution function and random number
generation for the skew-t (ST) distribution.
}
\usage{
dst(x, location=0, scale=1, shape=0, df=Inf)
pst(x, location=0, scale=1, shape=0, df=Inf)
qst(p, location=0, scale=1, shape=0, df=Inf, tol=1e-8)
rst(n=1, location=0, scale=1, shape=0, df=Inf)
}
\arguments{
\item{x}{
vector of quantiles. Missing values (\code{NA}s) are allowed.
}
\item{p}{
  vector of probabililities
  }
\item{location}{
vector of location parameters.
}
\item{scale}{
vector of (positive) scale parameters.
}
\item{shape}{
vector of shape parameters. With \code{pst} and \code{qst}, 
it must be of length 1.
}
\item{df}{
degrees of freedom (scalar); default is \code{df=Inf} which corresponds 
to the skew-normal distribution.
}
\item{n}{
sample size.
}
\item{tol}{
a scalar value which regulates the accuracy of the result of \code{qsn}.
}}
\value{
Density (\code{dst}), probability (\code{pst}), quantiles (\code{qst}) 
and random sample (\code{rst}) from the skew-t distribution with given 
\code{location}, \code{scale}, \code{shape} and \code{df} parameters.
}
\section{Background}{
The family of skew-t distributions is an extension of the Student's t
family, via the introduction of a \code{shape} parameter which regulates
skewness; when \code{shape=0}, the skew-t distribution reduces to the
usual Student's t distribution. When \code{df=Inf}, it reduces to the 
skew-normal distribution. A multivariate version of the distribution exists.
See the reference below for additional information.
}
\references{
Azzalini, A. and Capitanio, A. (2003).
Distributions generated by perturbation of symmetry 
with emphasis on a multivariate skew-\emph{t} distribution.
\emph{J.Roy. Statist. Soc. B} 
\bold{65}, 367--389.
}
\seealso{
\code{\link{dmst}}, \code{\link{dsn}}, \code{\link{st.mle}}
}
\examples{
pdf <- dst(seq(-4,4,by=0.1), shape=3, df=5)
rnd <- rst(100, 5, 2, -5, 8)
q <- qst(c(0.25,0.5,0.75), shape=3, df=5)
pst(q, shape=3, df=5)  # must give back c(0.25,0.5,0.75)

}
\keyword{distribution}


\eof
\name{dst2.plot}
\alias{dst2.plot}
\title{
Plot of Bivariate Skew-t Density Function
}
\description{
Produces a contour plot of the density function of a bivariate 
skew-t variate.
}
\usage{
dst2.plot(x, y, xi, Omega, alpha, df, ...)
}
\arguments{
\item{x}{
vector of values of the first component.
}
\item{y}{
vector of values of the second component.
}
\item{xi}{
a vector of length 2 containing the location parameter.
}
\item{Omega}{
a 2 by 2 matrix containing a covariance matrix.
}
\item{alpha}{
a vector of length 2 containing the shape parameter.
}
\item{df}{
a positive number, representing the degrees of freedom .
}
\item{...}{
additional parameters to be passed to \code{contour}.
}}
\value{
A list containing the original input parameters plus a matrix
containing the density function evaluated at the grid formed 
by the \code{x} and \code{y} values. 
}
\section{Side effects}{
A contour plot is produced on the graphical window.
}
\details{
\code{dst2.plot} makes use of function \code{dmst}.
}
\section{Background}{
The family of multivariate skew-t distributions is an extension of the 
multivariate Student's t family, via the introduction of a \code{shape} 
parameter which regulates skewness; when \code{shape=0}, the skew-t 
distribution reduces to the usual t distribution. 
When \code{df=Inf} the distribution reduces to the multivariate skew-normal 
one; see \code{dmsn}. See the reference below for additional information.
}
\references{
Azzalini, A. and Capitanio, A. (2002).
  Distributions generated by perturbation of symmetry 
  with emphasis on a multivariate skew \emph{t} distribution.
  Submitted to \emph{J.Roy. Statist. Soc.}
}
\seealso{
\code{\link{dmsn}}, \code{\link{dsn2.plot}}
}
\examples{
x <- y <- seq(-5, 5, length=35)
dst2.plot(x, y, c(-1,2), diag(c(1,2.5)), c(2,-3), df=5)
}
\keyword{distribution}
% Comment

\eof
\name{frontier}
\alias{frontier}
\title{Simulated sample from a skew-normal distribution}
\usage{data(frontier)}
\description{
  A sample simulated from the SN(0,1,5) distribution having sample
  index of skewness in the admissible range (-0.9952719,0.9952719) but
  maximum likelihood estimate on the frontier of the paramer space.
}

\format{
 A vector of length 50.
}
\source{
  Generated by a run of \code{rsn(50,0,1,5)}.
}
\examples{
data(frontier)
a <- sn.2logL.profile(y=frontier)
a <- sn.2logL.profile(y=frontier, param.range=c(0.8,1.6,10,30),
        use.cp=FALSE, npts=11)
}
\keyword{datasets}

\eof
\name{gamma1.to.lambda}
\alias{gamma1.to.lambda}
\title{
Converts skewness to shape parameter of skew-normal distribution
}
\description{
For a given value of the index of skewness (standardized third
cumulant), the function finds the corresponding shape parameter
of a skew-normal distribution.
}
\usage{
gamma1.to.lambda(gamma1)
}
\arguments{
\item{gamma1}{
a numeric vector of indices of skewness.
}}
\value{
A numeric vector of the corresponding shape parameters.
}
\details{
Feasible values for input must have \code{abs(gamma1)<0.5*(4-pi)*(2/(pi-2))^1.5},
which is about 0.99527.
If some values of \code{gamma1} are not in the feasible region, a warning
message is issued, and \code{NA}s are returned.


See the reference below for the expression of the index of skewnnes
of a skew-normal distribution.
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.
}
\seealso{
\code{\link{dsn}}
}
\examples{
}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{msn.affine}
\alias{msn.affine}
\alias{mst.affine}
\title{
  Affine transformation a multivariate skew-normal or
  skew-t variable
}
\description{
  Computes the parameters of an affine transformation \emph{a+ A Y}
  of a multivariate skew-normal or skew-t variable \emph{Y}
}
\usage{
  msn.affine(dp, A, a=0, drop=TRUE)
  mst.affine(dp, A, a=0, drop=TRUE)
}
\arguments{
  \item{dp}{a list containg the pamaters of the variable being
    transformed; it must include components \code{xi}, \code{Omega},
    \code{alpha} as described for \code{dmsn}; for \code{mst.affine},
    also a component \code{df} is expected
  }
  \item{A}{a matrix with \code{ncol(A)} equal to \code{nrow(dp$Omega)}
  }
  \item{a}{a vector wiht \code{length(a)} equal to \code{nrow(dp$Omega)}
  }
  \item{drop}{a logical flag (default value is \code{TRUE}) operating when 
    \code{nrow(A)} equals 1. If these conditions are met, the output
    is provided in the form of parameters of a scalar distribution,
    \code{dsn} or \code{dst}, depending in the case.
  }
}
\value{
  A list containing the same components of the input parameter \code{dp}
  }
\section{Background}{
  For background information about the skew-normal and skew-t
  distributions, their parameters and the properties of affine
  transformations, see the references below.
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B} \bold{61}, 579--602.

Azzalini, A. and Capitanio, A. (2003).
Distributions generated by perturbation of symmetry 
with emphasis on a multivariate skew-\emph{t} distribution.
\emph{J.Roy. Statist. Soc. B} \bold{65}, 367--389.

Capitanio, A. \emph{et al.} (2003).
Graphical models for skew-normal variates.
\emph{Scand.\ J.\ Statist.} \bold{30}, 129--144.

}
\seealso{
\code{\link{dsn}}, \code{\link{dst}}, \code{\link{dmsn}}, \code{\link{dmst}}
}
\examples{
dp<- list(xi=c(1,1,2), Omega=toeplitz(1/1:3), alpha=c(3,-1,2))
A <- matrix(c(1,-1,1,3,0,-2), 2, 3, byrow=TRUE) 
dp1 <- msn.affine(dp, A, 1:2)
#
dp$df <- 5
dp2<-  mst.affine(dp,A[1,,drop=FALSE])
dp3<-  mst.affine(dp,A[1,,drop=FALSE], drop=FALSE)
if(zapsmall(dp2$scale^2 - dp3$Omega)) print("something wrong here!")
}
\keyword{multivariate}
\keyword{distribution}

\eof
\name{msn.cond.plot}
\alias{msn.cond.plot}
\title{
Plot of the density of a conditional skew-normal variate
}
\description{
Plot of the exact and of the approximate density function of
a multivariate skew-normal variate conditionally on the values
taken on by some components.
}
\usage{
msn.cond.plot(xi, Omega, alpha, fixed.comp, fixed.values, n=35)
}
\arguments{
\item{xi}{
a numeric vector of length \code{k}, say, giving the location parameter.
}
\item{Omega}{
a covariance matrix of dimension \code{(k,k)}.
}
\item{alpha}{
a numeric vector of length \code{k}, which regulates the shape of the density.
}
\item{fixed.comp}{
a vector containing a subset of \code{1:k} which selects the components
whose values are to be fixed; it must be of length \code{k-2}.
}
\item{fixed.values}{
a numeric vector of values taken on by the components \code{fixed.comp};
it must be of the same length of \code{fixed.comp}.
}
\item{n}{
an integer value which determines the grid size of the density
computations and plot.
}}
\value{
A list containing the following elements:

\item{cumulants}{
two lists as returned by \code{msn.conditional}.
}
\item{pdf}{
a list containing the coordinates \code{x} and \code{y} of the points where the 
densities have been evaluated, and the matrices \code{f.exact} and \code{f.fitted}
of the exact and fitted conditional densities.
}
\item{rel.error}{
summary statistics of relative and absolute error of the approximation.
}}
\section{Side Effects}{
A contour plot of the exact and approximate densities is produced
on a graphical device.
}
\details{
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{msn.conditional}}, \code{\link{dmsn}}
}
\examples{
Omega <- diag(3)+0.5*outer(rep(1,3),rep(1,3))
a<- msn.cond.plot(rep(0,3), Omega, 1:3, 3, -0.75)
}
\keyword{multivariate}
\keyword{distribution}

\eof
\name{msn.conditional}
\alias{msn.conditional}
\title{
Cumulants and distribution of a skew-normal variate after conditioning
}
\description{
Finds cumulants up to 3rd order of a multivariate skew-normal
distribution conditionally on the values taken on by some of
its components, and finds a multivariate skew-normal distribution
with the same cumulants.
}
\usage{
msn.conditional(xi, Omega, alpha, fixed.comp, fixed.values)
}
\arguments{
\item{xi}{
a numeric vector of length \code{k}, say, giving the location parameter.
}
\item{Omega}{
a covariance matrix of dimension \code{(k,k)}.
}
\item{alpha}{
a numeric vector of length \code{k}, which regulates the shape of the density.
}
\item{fixed.comp}{
a vector containing a subset of \code{1:k} which selects the components
whose values are to be fixed; it must be of length \code{k-2}.
}
\item{fixed.values}{
a numeric vector of values taken on by the components \code{fixed.comp};
it must be of the same length of \code{fixed.comp}.
}}
\value{
A list containing the following elements:

\item{cumulants}{
a list containing mean vector, variance matrix, and indices of 
skewness of the conditional distribution.
}
\item{fit}{
a list containing the parameters of the fitted skew-normal
distribution in the \code{(xi,Omega,alpha)} parametrization, plus
the vector \code{delta}.
}}
\details{
See the reference below for details and background.
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{msn.cond.plot}}, \code{\link{msn.marginal}}
}
\examples{
Omega <- diag(3)+0.5*outer(rep(1,3),rep(1,3))
a<- msn.conditional(rep(0,3), Omega, 1:3, 3, -0.75)
}
\keyword{multivariate}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{msn.fit}
\alias{msn.fit}
\title{
Fitting multivariate skew-normal distributions
}
\description{
Fits a multivariate skew-normal (MSN) distribution to data, or fits a
linear regression model with multivariate skew-normal errors,
using maximum likelihood estimation. The outcome is then displayed
in graphical form.
}
\usage{
msn.fit(X, y, freq, plot.it=TRUE, trace=FALSE,  \dots )
}
\arguments{
\item{y}{
a matrix or a vector.  If \code{y} is a matrix, its rows refer to
observations, and its columns to components of the multivariate
distribution. If \code{y} is a vector, it is converted to a one-column
matrix, and a scalar skew-normal distribution is fitted.
}
\item{X}{
a matrix of covariate values.
If missing, a one-column matrix of 1's is created; otherwise,
it must have the same number of rows of \code{y}.
}
\item{freq}{
a vector of weights.
If missing, a one-column matrix of 1's is created; otherwise
it must have the same number of rows of \code{y}.
}
\item{plot.it}{
logical value which controls the graphical output (default=TRUE);
see below for description.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{...}{
additional paramters passed to \code{msn.mle}; in practice, the \code{start}
parameter can be passed.
}}
\value{
A list containing the following components:

\item{call}{
a string containing the calling statement.
}
\item{dp}{
a list containing the direct parameters \code{beta}, \code{Omega}, \code{alpha}.
Here, \code{beta} is a matrix of regression coefficients with
\code{dim(beta)=c(nrow(X),ncol(y))}, \code{Omega} is a covariance matrix of
order \code{ncol(y)}, \code{alpha} is a vector of shape parameters of length
\code{ncol(y)}.
}
\item{logL}{
log-likelihood evaluated at \code{dp}.
}
\item{se}{
a list containing the components \code{beta}, \code{alpha}, \code{info}.
Here, \code{beta} and \code{alpha} are the standard errors for the
corresponding point estimates;
\code{info} is the observed information matrix for the working parameter,
as explained below.
}
\item{optim}{
the list returned by the optimizer \code{optim}; see the documentation
of this function for explanation of its components.
}
\item{test.normality}{
a list of with elements \code{test} and \code{p.value}, which are the value
of the likelihood ratio test statistic for normality (i.e. test that
all components of the shape parameter are 0), and the corresponging
p-value.
}}
\section{Side Effects}{
Graphical output is produced if \code{(plot.it \& missing(freq))=TRUE} and
a suitable device is active. Three plots are produced, and the programs
pauses between each two of them, waiting for the <Enter> key to be pressed.

The first plot uses the variable \code{y} if \code{X} is missing, otherwise
it uses the residuals from the regression.
The form of this plot depends on the value of \code{k=ncol(y)};
if \code{k=1}, an histogram is plotted with the fitted distribution
superimposed. If \code{k>1}, a matrix of scatterplots is produced, with
superimposed the corresponding bivariate densities of the fitted
distribution.

The second plot has two panels, each representing a QQ-plot of
Mahalanobis distances. The first of these refers to the fitting of a
multivariate normal distribution, a standard statistical procedure; 
the second panel gives the corresponding QQ-plot of suitable Mahalanobis 
distances for the multivariate skew-normal fit.

The third plot is similar to the previous one, except that PP-plots
are produced.
}
\details{
For computing the maximum likelihood estimates, \code{msn.fit}
invokes \code{msn.mle} which does the actual computational work;
then, \code{msn.fit} displays the results in graphical form.
The documentation of \code{msn.mle} gives details of the numerical
procedure for maximum likelihood estimation.


Although the function accepts a vector \code{y} as input, the use of
\code{sn.mle} is recommended in the scalar case.


}
\section{Background}{
The multivariate skew-normal distribution is discussed by
Azzalini and Dalla Valle (1996); the \code{(Omega,alpha)} parametrization
adopted here is the one of Azzalini and Capitanio (1999).
}
\references{
Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{msn.mle}}, \code{\link{mst.fit}}
}
\examples{
data(ais)
attach(ais)
# a simple-sample case
b <- msn.fit(y=cbind(Ht,Wt))
#
# a regression case:
a <- msn.fit(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-6))
#
# refine the previous outcome
a1 <- msn.fit(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-9), start=a$dp)
}
\keyword{distribution}
\keyword{regression}

\eof
\name{msn.marginal}
\alias{msn.marginal}
\title{
Marginal components of a multivariate skew-normal distribution
}
\description{
Computes the marginal distribution of a subset of components of a
multivariate skew-normal distribution.
}
\synopsis{
msn.marginal(xi=NULL, Omega=NULL, alpha=NULL, comp=1:d, dp=NULL)
}
\usage{
msn.marginal(xi, Omega, alpha, comp)
msn.marginal(dp=, comp)
}
\arguments{
\item{xi}{
  a numeric vector of length \code{d}, say, giving the location parameter.
}
\item{Omega}{
  a covariance matrix of dimension \code{(d,d)}.
}
\item{alpha}{
  a numeric vector of length \code{d}, which regulates the shape of the density.
}
\item{comp}{
a vector containing a subset of \code{1:d} selecting the components of the
marginal distribution. A permutation of \code{1:d} is allowed, and
the components of \code{comp} do not need to be sorted.
}
\item{dp}{
  a list containing the components \code{xi}, \code{Omega}, \code{alpha}
}}
\value{
A list containing  components \code{xi, Omega, alpha} with the 
parameters of the marginal distribution. If \code{length(comp)} is  equal to
\code{r}, say, then the new components are of size \code{r, (r,r), r}, 
respectively.
}
\background{
See Azzalini and Capitanio (1999) for background information.
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B} \bold{61}, 579--602.
}
\seealso{
\code{\link{dmsn}}, \code{\link{msn.conditional}}, \code{\link{msn.affine}}
}
\examples{
xi <- c(10,0,-30)
Omega <- 5*diag(3)+outer(1:3,1:3)
alpha <- c(1,-3,5)
marg31 <- msn.marginal(xi,Omega,alpha,c(3,1))
}
\keyword{multivariate}
\keyword{distribution}


\eof
\name{msn.mle}
\alias{msn.mle}
\title{
Maximum likelihood estimation for a multivariate skew-normal distribution
}
\description{
Fits a multivariate skew-normal (MSN) distribution to data, or fits a
linear regression model with multivariate skew-normal errors,
using maximum likelihood estimation.
}
\usage{
msn.mle(X, y, freq, start, trace=FALSE,  method="BFGS", control=list(iter.max=150) )
}
\arguments{
\item{y}{
a matrix or a vector.  If \code{y} is a matrix, rows refer to
observations, and columns to components of the multivariate
distribution. If \code{y} is a vector, it is converted to a one-column
matrix, and a scalar skew-normal distribution is fitted.
}
\item{X}{
a matrix of covariate values.
If missing, a one-column matrix of 1's is created; otherwise,
it must have the same number of rows of \code{y}.
}
\item{freq}{
a vector of weights.
If missing, a one-column matrix of 1's is created; otherwise
it must have the same number of rows of \code{y}.
}
\item{start}{
a list contaning the components \code{beta},\code{Omega}, \code{alpha},
of the type described below. The \code{dp} component of the returned
list from a previous call has the required format.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{F}.
}
\item{method}{
this parameter is just passed to the optimizer \code{optim}; 
see the documentation of this function for its usage. Default value is
\code{"BFGS"}.
}
\item{control}{
this parameter is passed to the optimizer \code{optim}; 
see the documentation of this function for its usage.
}}
\value{
A list containing the following components:

\item{call}{
a string containing the calling statement.
}
\item{dp}{
a list containing the direct parameters \code{beta}, \code{Omega}, \code{alpha}.
Here, \code{beta} is a matrix of regression coefficients with
\code{dim(beta)=c(nrow(X),ncol(y))}, \code{Omega} is a covariance matrix of
order \code{ncol(y)}, \code{alpha} is a vector of shape parameters of length
\code{ncol(y)}.
}
\item{se}{
a list containing the components \code{beta}, \code{alpha}, \code{info}.
Here, \code{beta} and \code{alpha} are the standard errors for the
corresponding point estimates;
\code{info} is the observed information matrix for the working parameter,
as explained below.
}
\item{optim}{
the list returned by the optimizer \code{optim}; see the documentation
of this function for explanation of its components.
}}
\details{
The parameter \code{freq} is intended for use with grouped data,
setting the values of \code{y} equal to the central values of the
cells; in this case the resulting estimate is an approximation
to the exact maximum likelihood estimate. If \code{freq} is not
set, exact maximum likelihood estimation is performed.


The working parameter used in the maximization stage is
\code{c(beta,alpha/omega)}, since a profile `deviance' \code{-2*loglikelihood}
for this parameter is actually used;
see Azzalini and Capitanio (1999) for details.
The optimizer \code{optim} is called, supplying the gradient of
the profile deviance.


The function can take a vector \code{y} as input; however the use of
\code{sn.mle} is recommended in the scalar case.

}
\section{Background}{
The multivariate skew-normal distribution is discussed by
Azzalini and Dalla Valle (1996); the \code{(Omega,alpha)} parametrization
adopted here is the one of Azzalini and Capitanio (1999).
}
\references{
Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dmsn}},\code{\link{sn.mle}},\code{\link{msn.fit}}, \code{\link{optim}}}
}
\examples{
data(ais)
attach(ais)
# a simple-sample case
b <- msn.mle(y=cbind(Ht,Wt))
#
# a regression case:
a <- msn.mle(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-6))
#
# refine the previous outcome
a1 <- msn.mle(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-9), start=a$dp)
}
\keyword{distribution}
\keyword{regression}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{msn.quantities}
\alias{msn.quantities}
\title{
Quantities related to the multivariate skew-normal distribution.
}
\description{
Computes mean vector, variance matrix and other relevant quantities
of a given multivariate skew-normal distribution.
}
\usage{
msn.quantities(xi, Omega, alpha)
}
\arguments{
\item{xi}{
numeric vector giving the location parameter, of length \code{k}, say.
Missing values are not allowed.
}
\item{Omega}{
a covariance matrix of size \code{k} by \code{k}.
Missing values are not allowed.
}
\item{alpha}{
numeric vector of shape parameter of length \code{k}.
Missing values are not allowed.
}}
\value{
A list containing the following components:

\item{xi}{
the input parameter \code{xi}.
}
\item{Omega}{
the input parameter  \code{Omega}.
}
\item{alpha}{
the input parameter  \code{alpha}.
}
\item{omega}{
vector of scale parameters.
}
\item{mean}{
numeric vector representing the mean value of the distribution.
}
\item{variance}{
variance matrix of the distribution.
}
\item{Omega.conv}{
concentration matrix associated to \code{Omega}, i.e. its inverse.
}
\item{Omega.cor}{
correlation matrix associated to \code{Omega}.
}
\item{Omega.pcor}{
partial correlations matrix associated to \code{Omega}.
}
\item{lambda}{
shape parameters of the marginal distributions, in two equivalent
forms.
}
\item{Psi}{
correlation matrix of the equivalent \code{(lambda,Psi)} parametrization.
}
\item{delta}{
the parameter \code{delta} which determines the shape of the marginal 
distributions.
}

\item{skewness}{
numeric vector with marginal indices of skewness (the standardised
third cumulant).
}}
\details{
The meaning of the parameters is explained in the references below,
especially Azzalini and Capitanio (1999).
}
\references{
Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dmsn}}
}
\examples{
Omega <- 5*diag(3)+outer(1:3,1:3)
msn.quantities(c(0,0,1), Omega, c(-2,2,3))
}
\keyword{multivariate}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{mst.fit}
\alias{mst.fit}
\title{
Fitting multivariate skew-t distributions
}
\description{
Fits a multivariate skew-t (MST) distribution to data, or fits a
linear regression model with multivariate skew-t errors,
using maximum likelihood estimation. The outcome is then displayed
in graphical form.
}
\usage{
mst.fit(X, y, freq, fixed.df=NA, start=NA, plot.it=TRUE, 
        trace=FALSE, ...)
}
\arguments{
\item{y}{
a matrix or a vector.  If \code{y} is a matrix, its rows refer to
observations, and its columns to components of the multivariate
distribution. If \code{y} is a vector, it is converted to a one-column
matrix, and a scalar skew-t distribution is fitted.
}
\item{X}{
a matrix of covariate values.
If missing, a one-column matrix of 1's is created; otherwise,
it must have the same number of rows of \code{y}.
}
\item{freq}{
a vector of weights.
If missing, a vector of 1's is created; otherwise
it must have the same number of rows of \code{y}.
}
\item{fixed.df}{
a scalar value contaning the degrees of freedom (df), if these must
be taked as fixed, or \code{NA} (deafult value) if df is a parameter
to be estimated.
}
\item{start}{
a list contaning the components \code{beta},\code{Omega}, \code{alpha},
\code{df} of the type described below. The \code{dp} component of the returned
list from a previous call has the required format.
}
\item{plot.it}{
logical value which controls the graphical output (default=TRUE);
see below for description.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{...}{
additional parameters passed to \code{mst.mle}; in practice, the \code{start}
parameter can be passed.
}}
\value{
A list containing the following components:

\item{call}{
a string containing the calling statement.
}
\item{dp}{
a list containing the direct parameters \code{beta}, \code{Omega}, \code{alpha},
code{df}.
Here, \code{beta} is a matrix of regression coefficients with
\code{dim(beta)=c(nrow(X),ncol(y))}, \code{Omega} is a covariance matrix of
order \code{ncol(y)}, \code{alpha} is a vector of shape parameters of length
\code{ncol(y)}, code{df} is a positive scalar.
}
\item{logL}{
log-likelihood evaluated at \code{dp}.
}
\item{se}{
a list containing the components \code{beta}, \code{alpha}, \code{info}.
Here, \code{beta} and \code{alpha} are the standard errors for the
corresponding point estimates;
\code{info} is the observed information matrix for the working parameter,
as explained below.
}
\item{optim}{
the list returned by the optimizer \code{optim}; see the documentation
of this function for explanation of its components.
}
\item{test.normality}{
a list with elements \code{test} and \code{p.value}, which are the value
of the likelihood ratio test statistic for normality (i.e. test that
all components of the shape parameter are 0), and the corresponding
p-value.
}}
\section{Side Effects}{
Graphical output is produced if \code{(plot.it \& missing(freq))=TRUE} and
a suitable device is active. Three plots are produced, and the programs
pauses between each two of them, waiting for the <Enter> key to be pressed.

The first plot uses the variable \code{y} if \code{X} is missing, otherwise
it uses the residuals from the regression.
The form of this plot depends on the value of \code{d=ncol(y)};
if \code{d=1}, an histogram is plotted with the fitted distribution
superimposed. If \code{d>1}, a matrix of scatterplots is produced, with
superimposed the corresponding bivariate densities of the fitted
distribution.

The second plot has two panels, each representing a QQ-plot of
Mahalanobis distances. The first of these refers to the fitting of a
multivariate normal distribution, a standard statistical procedure; 
the second panel gives the corresponding QQ-plot of suitable Mahalanobis 
distances for the multivariate skew-normal fit.

The third plot is similar to the previous one, except that PP-plots
are produced.
}
\details{
For computing the maximum likelihood estimates, \code{mst.fit}
invokes \code{mst.mle} which does the actual computational work;
then, \code{mst.fit} displays the results in graphical form.
The documentation of \code{mst.mle} gives details of the numerical
procedure for maximum likelihood estimation.

}
\section{Background}{
The family of multivariate skew-t distributions is an extension of the 
multivariate Student's t family, via the introduction of a \code{shape} 
parameter which regulates skewness; when \code{shape=0}, the skew-t 
distribution reduces to the usual t distribution. 
When \code{df=Inf} the distribution reduces to the multivariate skew-normal 
one; see \code{dmsn}. See the reference below for additional information.
}
\references{
Azzalini, A. and Capitanio, A. (2003).
  Distributions generated by perturbation of symmetry 
  with emphasis on a multivariate skew \emph{t} distribution.
 \emph{J.Roy. Statist. Soc. B} \bold{65}, 367--389.

}
\seealso{
\code{\link{msn.fit}}, \code{\link{mst.mle}}
}
\examples{
data(ais)
attach(ais)
# a simple-sample case
b <- mst.fit(y=cbind(Ht,Wt))
#
# a regression case:
a <- mst.fit(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-6))
#
# refine the previous outcome
a1 <- mst.fit(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-9), start=a$dp)
}
\keyword{distribution}
\keyword{regression}

\eof
\name{mst.mle}
\alias{mst.mle}
\alias{st.mle}
\title{
Maximum likelihood estimation for a (multivariate) skew-t distribution
}
\description{
Fits a skew-t (ST) or multivariate skew-t (MST) distribution to data, or 
fits a linear regression model with (multivariate) skew-t errors,
using maximum likelihood estimation.
}
\usage{
mst.mle(X, y, freq, start=NA, fixed.df=NA, trace=FALSE,  method="BFGS", 
       control=list(iter.max=150) )

st.mle(X, y, freq, start=NA, fixed.df=NA, trace=FALSE,  method="BFGS", 
       control=list(iter.max=150) )

}
\arguments{
\item{y}{
a matrix (for \code{mst.mle}) or a vector (for \code{st.mle}).  
If \code{y} is a matrix, rows refer to observations, and columns to 
components of the multivariate distribution. 
}
\item{X}{
  a matrix of covariate values.
  If missing, a one-column matrix of 1's is created; otherwise,
  it must have the same number of rows of \code{y}.
  If \code{X} is supplied, then it must include a column of 1's.
}
\item{freq}{
a vector of weights.
If missing, a one-column matrix of 1's is created; otherwise
it must have the same number of rows of \code{y}.
}
\item{start}{
a list contaning the components \code{beta},\code{Omega}, \code{alpha},
\code{df} of the type described below. The \code{dp} component of the returned
list from a previous call has the required format and it can be used as a new
\code{start}.
}
\item{fixed.df}{
a scalar value containing the degrees of freedom (df), if these must
be taked as fixed, or \code{NA} (deafult value) if df is a parameter
to be estimated.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{method}{
this parameter is just passed to the optimizer \code{optim}; 
see the documentation of this function for its usage. Default value is
\code{"BFGS"}.
}
\item{control}{
this parameter is passed to the optimizer \code{optim}; 
see the documentation of this function for its usage.
}}
\value{
A list containing the following components:

\item{call}{
a string containing the calling statement.
}
\item{dp}{
a list containing the direct parameters \code{beta}, \code{Omega}, \code{alpha}.
Here, \code{beta} is a matrix of regression coefficients with
\code{dim(beta)=c(nrow(X),ncol(y))}, \code{Omega} is a covariance matrix of
order \code{ncol(y)}, \code{alpha} is a vector of shape parameters of length
\code{ncol(y)}.
Notice that, if \code{st.mle} was called or equivalently \code{mst.mle}
was called with \code{y} a vector, then \code{Omega} represents the
\emph{square} of the scale parameter. 
}
\item{se}{
a list containing the components \code{beta}, \code{alpha}, \code{info}.
Here, \code{beta} and \code{alpha} are the standard errors for the
corresponding point estimates;
\code{info} is the observed information matrix for the working parameter,
as explained below.
}
\item{optim}{
the list returned by the optimizer \code{optim}; see the documentation
of this function for explanation of its components.
}}
\details{
If \code{y} is a vector and it is supplied to \code{mst.mle}, then
it is converted to a one-column matrix, and a scalar skew-t 
distribution is fitted. This is the mechanism used by \code{st.mle}
which is simply an interface to \code{mst.mle}.

The parameter \code{freq} is intended for use with grouped data,
setting the values of \code{y} equal to the central values of the
cells; in this case the resulting estimate is an approximation
to the exact maximum likelihood estimate. If \code{freq} is not
set, exact maximum likelihood estimation is performed.

% To fit a scalar skew-t distribution to grouped data by exact
% maximum likelihood estimation, use \code{st.mle.grouped}.
}
\section{Background}{
The family of multivariate skew-t distributions is an extension of the 
multivariate Student's t family, via the introduction of a \code{shape} 
parameter which regulates skewness; when \code{shape=0}, the skew-t 
distribution reduces to the usual t distribution. 
When \code{df=Inf} the distribution reduces to the multivariate skew-normal 
one; see \code{dmsn}. See the reference below for additional information.
}
\references{
Azzalini, A. and Capitanio, A. (2002).
  Distributions generated by perturbation of symmetry 
  with emphasis on a multivariate skew \emph{t} distribution.
  Submitted to \emph{J.Roy. Statist. Soc.}
}
\seealso{
\code{\link{dmst}},\code{\link{msn.mle}},\code{\link{mst.fit}}, \code{\link{optim}}
% \code{\link{sn.mle.grouped}}
}
\examples{
data(ais)
attach(ais)
M <- model.matrix(~lbm+sex)
b <- sn.mle(M, bmi)
# 
b <- mst.mle(y=cbind(Ht,Wt))
#
# a multivariate regression case:
a <- mst.mle(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-6))
#
# refine the previous outcome
a1 <- mst.mle(X=cbind(1,Ht,Wt), y=bmi, control=list(x.tol=1e-9), start=a$dp)
}
\keyword{distribution}
\keyword{regression}


\eof
\name{pnorm2}
\alias{pnorm2}
\title{
Bivariate normal integral
}
\description{
Computes the cumulative distribution function of a bivariate
normal distribution with standardized marginals.
}
\usage{
pnorm2(x, y, rho)
}
\arguments{
\item{x}{
a numerical value representing the first coordinate
}
\item{y}{
a numerical value representing the second coordinate
}
\item{rho}{
a numerical value representing the correlation;
it must be between -1 and 1.
}}
\value{
a numerical value with the required probability
}
\details{
The input parameters must be all scalars.


This function is based on function \code{T.Owen}.
}
\seealso{
\code{\link{T.Owen}}
}
\examples{
p <- pnorm2(1.2, 0.5, 0.67)
}
\keyword{distribution}
\keyword{multivariate}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{sn-internal}
\alias{msn.dev}
\alias{msn.dev.grad}
\alias{sn.dev}
\alias{sn.dev.gh}
\alias{msn.moment.fit}
\alias{num.deriv}
\alias{mst.dev}
\alias{mst.dev.grad}
\alias{st.dev.fixed}
\alias{sn.logL.grouped}
\alias{st.logL.grouped}
\alias{sn.SFscore}
\title{Internal sn functions}
\description{
 Internal  functions of package \emph{sn}
}
\usage{
msn.dev(param, X, y, freq, trace=FALSE)
msn.dev.grad(param, X, y, freq, trace=FALSE)
msn.moment.fit(y)
mst.dev(param, X, y, freq, fixed.df=NA, trace=FALSE)
mst.dev.grad(param, X, y, freq, fixed.df=NA, trace=FALSE)
num.deriv(coefficients, FUN, ...)
st.dev.fixed(free.param, X, y, freq, trace=FALSE, fixed.comp=NA, fixed.values=NA)
sn.dev(cp, X, y, trace=FALSE)
sn.dev.gh(cp, X, y, trace=FALSE, hessian=FALSE)
sn.logL.grouped(param, breaks, freq, trace=FALSE)
st.logL.grouped(param, breaks, freq, trace=FALSE)
sn.SFscore(shape, data, trace=FALSE)
}
\arguments{
\item{param,cp, coefficients, shape}{
a numeric vector of parameter values.
}

\item{X}{
a matrix of explanatory variables; must have \code{col(X)} equal to
\code{length(y)}. Missing values (\code{NA}) are not allowed.
If \code{X} is missing, a one-column matrix of 1's is created.
}
\item{y}{
a numeric vector or matrix. Missing values (\code{NA}s) are not allowed.
}
   
\item{freq}{
a vector of frequencies.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{free.param}{
a vector of suitably re-parametrized parameters, not to be kept fixed during 
iteration.
}
\item{fixed.comp}{
a vector containing the subset of the parameters for which the
profile log-likelihood function is required; it can be of length 1 or 2.
}
\item{fixed.values}{
a numeric vector of values or a matrix with two columns, giving the
range spanned by the selected parameters. 
}
\item{fixed.df}{
a scalar value contaning the degrees of freedom (df), if these must
be taked as fixed, or \code{NA} (deafult value) if df is a parameter
to be estimated.
}
\item{breaks}{
a vector contaning the cut points of the groups, given
in ascending order. The last value can be \code{Inf}, the
first one can be \code{-Inf}
}
}
\value{
  A numeric value or a vector or a list.
}

\details{
  These functions are not intended to be called directly by the user.
}
\keyword{internal}


\eof
\name{sn.2logL.profile}
\alias{sn.2logL.profile}
\title{
Profile twice loglikelihood for skew-normal models
}
\description{
Computation and plot of 1-dimensional and 2-dimensional profile
2*loglikelihood for skew-normal regression models.
}
\usage{
sn.2logL.profile(X=matrix(rep(1, n)), y,
    param.range=c(sqrt(var(y)) * c(2/3, 3/2), -0.95, 0.95),
    use.cp=TRUE, npts=31 \%/\% d, plot.it=TRUE, ...)
}
\arguments{
\item{y}{
a numeric vector. Missing values (\code{NA}s) are not allowed.
}
\item{X}{
a matrix of explanatory variables; must have \code{col(X)} equal to
\code{length(y). Missing values (`NA}s) are not allowed.
If \code{X} is missing, a one-comun matrix of 1's is created.
}
\item{param.range}{
a numeric vector of length either 2 or 4. If the length is 2,
the dimensional paramter \code{d} is set to 1, and
a 1-dimensional profile is computed and plotted, for the shape
or skewness parameter (depending on the parametrization adopted;
see below); in this case the two value represent the minimum
and maximum value for the range of the parameter.
If the length of \code{param.range} is 4, the first two values
determine the range of the scale parameter, the last two give
the range of the shape (or skewness) parameter; in this case, \code{d=2}.
}
\item{use.cp}{
logical value which selects the parametrization adopted.
If \code{use.cp=TRUE} (default value), the centred parametrization is used,
otherwise the direct parametrization is adopted.
}
\item{npts}{
number of points (in the scalar case) or grid size (in the
2-dimensional case).
}
\item{plot.it}{
logical value which determines if plotting takes place; default
is \code{T}.
}
\item{...}{
any additional parameter is passed to \code{sn.em}.
}}
\value{
A list containing the following components

\item{param1}{
vectors of the parameters values where the function has been
evaluated. If \code{d=2}, the second vector contains \code{NA}s.
}
\item{param.names}{
a character vector of two elements with the names of the \code{param1}
and \code{param2}.
}
\item{2logL}{
a vector or a matrix which represents twice the profile loglikelihood;
this is in the "relative" version, i.e. setting the maximum value to
be 0.
}
\item{maximum}{
a numeric value with the maximum which has been subtracted to
obtain the "relative" version of \code{2logL}.
}}
\section{Side Effects}{
If \code{plot.it=TRUE}, a plot of the profile twice relative loglikeliood is
produced on a graphical device.
}
\details{
Likelihood maximization is performed by \code{sn.em}.


See the reference below for explanation of the two possible
parametrizations.
}
\references{
Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{sn.em}}, \code{\link{sn.mle}}
}
\examples{
data(ais)
attach(ais)
a <- sn.2logL.profile(y=bmi)
a <- sn.2logL.profile(y=bmi, use.cp=FALSE, param.range=c(3,6,1,5))
a <- sn.2logL.profile(X=cbind(1,lbm), y=bmi, param.range=c(0,0.9),
        npts=50)
#
data(frontier)
a <- sn.2logL.profile(y=frontier, param.range=c(0.8,1.6,10,30),
        use.cp=FALSE, npts=11)
}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{sn.Einfo}
\alias{sn.Einfo}
\title{
Expected Fisher information for SN distribution parameters
}
\description{
  Computes expected Fisher information for parameters of simple
  sample having one-dimensional skew-normal (SN) distribution
  or regression model having SN errors, in the DP and CP parametrizations.
}
\usage{
sn.Einfo(dp=NULL, cp=NULL, n=1, x=NULL)
}
\arguments{
\item{dp, cp}{vector of direct or centred parameters, respectively;
one of the two vectors must be supplied, but not both. See below for
more details.
}
\item{n}{sample size; if this parameter is supplied, then \code{x} must
not be.
}
\item{x}{design matrix of the regression model; if this parameter is
supplied, then \code{n} must not be.
}}
\value{
a list containing the following components:
\item{dp, cp}{
DP and CP parameters; one of the two vectors is the one supplied on
input, the other one  matches the previous one in the alternative
parametrization
}

\item{info.dp, info.cp}{
matrices of Fisher expected information in the two parametrizations
}

\item{se.dp, se.cp}{
vectors of standard errors in the two parametrizations
}

\item{D}{
matrix of derivatives of DP parameters with respect to CP parameters
}
}
\section{Details}{
When \code{x} is not set, then a simple random sample is assumed and a
 matrix \code{x} with a single column of all 1's is constructed; in this
case, the supplied  vector \code{dp} or \code{cp} must have length 3.
If \code{x} is set, then the supplied vector of parameters must have
length \code{ncol(x)+2}.
}
\section{Background}{
  For the description of the DP and CP parametrizations and
  for the expression of the exptected Fisher information of the DP
  parametrizations, see Azzalini (1985). Further discussion is in
  Azzalini and Capitanio (1999).
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dsn}}, \code{\link{cp.to.dp}}, \code{\link{dp.to.cp}}
}
\examples{
info <- sn.Einfo(dp=c(0,1,5), n=3)
#
data(ais)
M <- model.matrix(~ais$"Ht")
mle <- sn.mle(X=M, y=ais$"Wt", plot.it=FALSE)
info <- sn.Einfo(cp=mle$cp, x=M)
}
\keyword{distribution}

\eof
\name{SN}
\alias{SN}
\title{
The library `sn': summary information
}
\description{
This library provides functions related to the skew-normal (SN) and
the skew-t (ST) probability distributions, both for the univariate 
and for the the multivariate case, including regression models.
}
\section{Functions}{
% The functions of the scalar case section are:
% \code{dsn}, \code{psn}, \code{qsn}, \code{rsn}, \code{T.Owen}, 
% \code{cp.to.dp}, \code{dp.to.cp}, \code{zeta},
% \code{gamma1.to.lambda}, \code{sn.cumulants}, \code{sn.em}, 
% \code{sn.2logL.profile}, \code{sn.mle}, \code{sn.dev}, 
% \code{sn.dev.gh}.
% 
% 
% The functions of the multivariate section are: \code{dmsn},
% \code{rmsn}, \code{plot.dsn2}, \code{msn.quantities},
% \code{msn.conditional}, \code{msn.marginal}, \code{plot.msn.cond},
% \code{msn.fit}, \code{msn.mle}, \code{msn.dev}, \code{msn.dev.grad},
% \code{msn.moment.fit}, \code{num.deriv}.

The library includes several functions, a number of which are intended
as services to other functions, not really to be called directly by the
users. The following ones are those more relevant for practical use. 

\emph{SN distribution:}
\code{\link{dsn}}, \code{\link{psn}}, \code{\link{qsn}},
\code{\link{rsn}}, \code{\link{cp.to.dp}}, \code{\link{dp.to.cp}},
\code{\link{sn.mle}}, \code{\link{sn.em}}, 
%\code{\link{sn.mle.grouped}} 
for the univariate case, and 
\code{\link{dmsn}}, \code{\link{pmsn}}, \code{\link{rmsn}}, 
\code{\link{msn.fit}}, \code{\link{msn.mle}}, \code{\link{dsn2.plot}} 
for the multivariate case. 


\emph{ST distribution:}
\code{\link{dst}}, \code{\link{pst}}, \code{\link{rst}}, \code{\link{st.mle}}, 
%\code{\link{st.mle.grouped}} 
for the univariate case,
and \code{\link{dmst}},\code{\link{pmst}}, \code{\link{rmst}}, 
\code{\link{mst.fit}}, \code{\link{mst.mle}}, \code{\link{dst2.plot}} 
for the multivariate case. 

It is suggested that a user starts by reading the  documentation of 
(some of) these.
}
\section{Requirements}{
R 1.0.1.  Two functions  require  \code{library(mvtnorm)}: 
\code{pmsn}, \code{pmst}.
}
\section{Version}{
The version level of the library is given by the  command 
\code{print(.sn.version)}.
The most recent version of the library can be obtained from
the WWW page: \url{http://azzalini.stat.unipd.it/SN}
which also contains other related material
}
\section{Manual}{
There is no manual  besides on-line documentation and its derived versions. 
The papers listed below provide background information. 
}
\section{Author}{
Adelchi Azzalini, Dipart. Scienze Statistiche, Universit di Padova, Italia.
Please send comments, error reports, etc. to \email{adelchi.azzalini@unipd.it}
}
\section{Acknowledgements}{The library has evolved through several 
versions, developed over some years.

For versions up to 0.20, the following people and institutions 
have contributed. Many thanks go to Antonella Capitanio for testing 
the procedures, and to Brian Ripley and Martin Maechler for useful 
advice on R.  The function \code{num.deriv} is based on a similar 
function written  by Monica Chiogna. 
This software and part of the associated theoretical work has been developed 
while the author was at the Nuffield College, Oxford, under the Jemolo 
Fellowship scheme; the generous support of the  college is gratefully 
acknowledged. 
Additional support for the development of the theoretical research work has 
been  provided by the "Consiglio Nazionale delle Ricerche" of Italy, grant 
no.97.01331.CT10.

Versions 0.21 to 0.30 have been supported by "MIUR", Italy, under grant 
scheme PRIN 2000. Again, thanks to Antonella Capitanio for additional
testing of the new version of the library.

}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171--178.


Azzalini, A. (1986).
Further results on a class of distributions which includes 
the normal ones.
\emph{Statistica}
\bold{46},199--208.


Azzalini, A. and Dalla Valle, A. (1996).
The multivariate skew-normal distribution.
\emph{Biometrika}
\bold{83}, 715--726.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy. Statist. Soc. B}
\bold{61}, 579--602.

Azzalini, A. and Capitanio, A. (2003).
Distributions generated by perturbation of symmetry 
with emphasis on a multivariate skew-\emph{t} distribution.
\emph{J.Roy. Statist. Soc. B}  \bold{65}, 367--389.

}
\section{Licence}{
This library and its documentation are usable under the terms of the 
"GNU General Public License", a copy of which is distributed with the 
package.
}
\keyword{multivariate}
\keyword{distribution}
\keyword{regression}

\eof
\name{sn.cumulants}
\alias{sn.cumulants}
\title{
Cumulants of the skew-normal distribution
}
\description{
Cumulants of the skew-normal distribution.
}
\usage{
sn.cumulants(shape=0, n=4)
}
\arguments{
\item{shape}{
a vector of shape parameter.
}
\item{n}{
a scalar integer.
}}
\value{
the cumulants up to order \code{n} of the skew-normal distribution
with \code{location=0}, \code{scale=1} and \code{shape} as selected.
}
\details{
The moment generating function (hence the cumulant generating function)
of the distribution is given in the refence below.
The computations method used is proved analytically up to \code{n=3} but it is
seen to behave correctly up to the order which was checked (\code{n=8}).
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.
}
\seealso{
\code{\link{dsn}},\code{\link{zeta}}
}
\examples{
cum <- sn.cumulants(c(0,1,2,5,10),4)
}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{sn.em}
\alias{sn.em}
\title{
Fitting Skew-normal variables using the EM algorithm
}
\description{
Fits a skew-normal (SN) distribution to data, or fits a linear regression
model with skew-normal errors, using the EM algorithm to locate the MLE
estimate. The estimation procedure can be global or it can fix some
components of the parameters vector.
}
\usage{
sn.em(X, y, fixed, p.eps=0.0001, l.eps=0.01, trace=FALSE, data=FALSE)
}
\arguments{
\item{y}{
a vector contaning the observed variable. This is the response
variable in case of linear regression.
}
\item{X}{
a matrix of explanatory variables. If \code{X} is missing, then a one-column
matrix of all 1's is created. If \code{X} is supplied, and an intercept term
is required, then it must include a column of 1's.
}
\item{fixed}{
a vector of length 3, indicating which components of the
parameter vector must be regarded as fixed. In \code{fixed=c(NA,NA,NA)},
which is the default setting, a global maximization is performed.
If the 3rd component is given a value, then maximization is performed
keeping that value fixed for the shape parameter. If the 3rd and 2nd
parameters are fixed, then the scale and the shape parameter are
kept fixed. No other patterns of the fixed values are allowed.
}
\item{p.eps}{
numerical value which regulates the parameter convergence tolerance.
}
\item{l.eps}{
numerical value which regulates the log-likelihood convergence tolerance.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{F}.
}
\item{data}{
logical value. If \code{data=TRUE}, the returned list includes the original
data. Default value is \code{data=FALSE}.
}}
\value{
a list with the following components:

\item{dp}{
a vector of the direct parameters, as explained in the references below.
}
\item{cp}{
a vector of the centred parameters, as explained in the references below.
}
\item{logL}{
the log-likelihood at congergence.
}
\item{data}{
optionally (if \code{data=TRUE}), a list containing \code{X} and \code{y,} as supplied
on input, and a vector of \code{residuals}, which should have an approximate
SN distribution with \code{location=0} and \code{scale=1}, in the direct
parametrization.
}}
\details{
The function works using the direct parametrization; on convergence,
the output is then given in both parametrizations.


This function is based on the EM algorithm; it is generally quite slow,
but it appears to be very robust.
See \code{sn.mle} for an alternative method, which also returns standard
errors.
}
\section{Background}{
Background information on the SN distribution is given by Azzalini (1985).
See  Azzalini and Capitanio (1999) for a more detailed discussion of
the direct and centred parametrizations.
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.
}
\seealso{
\code{\link{dsn}}, \code{\link{sn.mle}}, \code{\link{cp.to.dp}}
}
\examples{
data(ais)
attach(ais)
#
a<-sn.em(y=bmi)
#
a<-sn.em(X=cbind(1,lbm,lbm^2),y=bmi)
#
M<-model.matrix(~lbm+I(ais$sex))
b<-sn.em(M,bmi)
#
fit <- sn.em(y=bmi, fixed=c(NA, 2, 3), l.eps=0.001)
}
\keyword{regression}
\keyword{distribution}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{sn.mle}
\alias{sn.mle}
\title{
Maximum likelihood estimation for skew-normal models
}
\description{
Fits a skew-normal (SN) distribution to data, or fits a linear regression
model with skew-normal errors, using maximum likelihood estimation.
}
\usage{
sn.mle(X, y, cp, plot.it=TRUE, trace=FALSE, method="L-BFGS-B",
          control=list(iter.max=100, abs.tol=1e-5))
}
\arguments{
\item{y}{
a vector contaning the observed variable. This is the response
variable in case of linear regression.
Missing values (\code{NA}s) are not allowed.
}
\item{X}{
  a matrix of explanatory variables.
  If \code{X} is missing, then a one-column matrix of all 1's is created.
  If \code{X} is supplied, then it must include a column of 1's.
  Missing values (\code{NA}s) are not allowed.
}
\item{cp}{
a vector of initial values for the centred parameters,
with \code{length(cp)=ncol(X)+2}
}
\item{plot.it}{
logical value, If \code{plot.it=TRUE} (default),
a plot of the nonparametric estimate of variable \code{y} (or the residuals,
in the case of regression), and the parametric fit is superimposed.
See below for details.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{method}{
this parameter is just passed to the optimizer \code{optim}; see the 
documentation of this function for its usage. Default value is 
\code{"L-BFGS-B"}.}
\item{control}{
this parameter is just passed to the optimizer \code{optim}; 
see the documentation of this function for its usage.
}}
\value{
a list containing the following components:

\item{call}{
a string containing the calling statement
}
\item{cp}{
a vector of length \code{ncol(X)+2} with the centred parameters
}
\item{logL}{
the log-likelihood at convergence
}
\item{se}{
a vector of standard errors for the \code{cp} component
}
\item{info}{
the observed information matrix for the \code{cp} component
}
\item{optim}{
the list returned by the optimizer \code{optim}; see the documentation
of this function for explanation of its components.
}}
\section{Side Effects}{
If \code{plot.it=TRUE} and a graphical device is active, a plot is produced,
as described above.
}
\details{
The optimizer \code{optim} is used, supplying the gradient of the log-likelihood.
Convergence is generally fast and reliable, but inspection of
the returned \code{message} from \code{optim} is always appropriate.
In suspect cases, re-run the function changing the starting \code{cp}
vector.

If plotting operates, the function \code{sm.density} of the library \code{sm}
is searched; this library is associated with the book by Bowman and
Azzalini (1997).  If \code{sm.density} is not found, an histogram is plotted.

% To fit a skew-notmal distribution to grouped data by exact
% maximum likelihood estimation, use \code{sn.mle.grouped}.

}
\section{Background}{
Background information on the SN distribution is given by Azzalini (1985).
See  Azzalini and Capitanio (1999) for a more detailed discussion of
the centred parametrization.
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.}
\bold{12}, 171-178.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B}
\bold{61}, 579--602.


Bowman, A.W. and Azzalini, A. (1997).
\emph{Applied Smoothing Techniques for Data Analysis:}
\emph{the Kernel Approach with S-Plus Illustrations.}
Oxford University Press, Oxford.
}
\seealso{
\code{\link{dsn}}, \code{\link{sn.em}}, \code{\link{msn.mle}},
\code{\link{optim}}, \code{\link{sn.mmle}}, \code{\link{sn.mle.grouped}}
}
\examples{
data(ais)
attach(ais)
a<-sn.mle(y=bmi)
#
a<-sn.mle(X=cbind(1,lbm),y=bmi)
#
b<-sn.mle(X=model.matrix(~lbm+sex), y=bmi)
}
\keyword{regression}
\keyword{distribution}

\eof
\name{sn.mle.grouped}
\alias{sn.mle.grouped}
\alias{st.mle.grouped}
\title{
Maximum likelihood estimation of SN and ST distribution for grouped data 
}
\description{
Fits a skew-normal (SN) and skew-t (ST) distribution to grouped data 
using exact maximum likelihood estimation
}
\usage{
sn.mle.grouped(breaks, freq, trace=FALSE, start=NA)
st.mle.grouped(breaks, freq, trace=FALSE, start=NA)
}
\arguments{
\item{breaks}{
a vector contaning the cut points of the groups, given
in ascending order. The last value can be \code{Inf}, the
first one can be \code{-Inf}}
\item{freq}{
a vector containing the observed frequencies corresponding to
the intervals defined by \code{breaks}; it is required
that \code{length(freq)=length(breaks)-1}
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}
}
\item{start}{
vector of length  with initial values for the woking parameters:
location, log(scale), shape and (for the ST case) log(df).
}}
\value{
a list containing the following components:

\item{call}{
  a string containing the calling statement
}
\item{dp}{
  for the SN case, a vector of length 3 containing the location, scale
  and shape parameter; for the ST case, there is an extra parameter, the
  degress of freedom
}
\item{end}{
  a vector of length 3 or 4 containing the working parameters; this
  vector can be used as a \code{start} for a new call to the function
}
\item{opt}{
  the list returned by the optimizer \code{optim}; see the documentation
  of this function for explanation of its components.
}}
\details{
  The optimizer \code{optim} is used, supplying  the log-likelihood
  function for grouped data, namely the  multinomial expression whose
  probabilities are assigned by the SN or ST distribution to the given
  intervals.
}
\section{Background}{
Background information on the SN distribution is given by Azzalini (1985);
see also Azzalini and Capitanio (1999). For the ST distribution, see
Azzalini and Capitanio (2003).
}
\references{
Azzalini, A. (1985).
A class of distributions which includes the normal ones.
\emph{Scand. J. Statist.} \bold{12}, 171-178.


Azzalini, A. and Capitanio, A. (1999).
Statistical applications of the multivariate skew-normal distribution.
\emph{J.Roy.Statist.Soc. B} \bold{61}, 579--602.

Azzalini, A. and Capitanio, A. (2003).
Distributions generated by perturbation of symmetry 
with emphasis on a multivariate skew-\emph{t} distribution.
\emph{J.Roy. Statist. Soc. B} \bold{65}, 367--389.

}
\seealso{
\code{\link{dsn}},  \code{\link{dst}}, \code{\link{optim}}, \code{\link{sn.mle}}
}
\examples{
data(ais)
attach(ais)
breaks<- c(130,160, seq(170, 190, by=2.5), 200, 230)
f <- cut(Ht[sex=="female"], breaks = breaks)
freq <- tabulate(f, length(levels(f)))
b1 <- sn.mle.grouped(breaks, freq)
b2 <- st.mle.grouped(breaks, freq, start=c(b1$end,log(5)))
print(b2$dp)
#
us.income <- c(0,seq(from=0.2, to=1.8, by=0.1), 2.0, 2.5, 5.0, Inf)
mid <- (us.income[-1]+us.income[-length(us.income)])/2
mid[length(mid)] <- 6.5
cum.freq<- c(1.78, 3.25, 5.56, 8.16, 11.12, 14.21, 17.54, 20.78, 24.00,
             27.52, 30.77, 34.21, 37.56, 40.70, 44.41, 47.85, 51.22, 
             57.60, 72.12, 96.40, 100) / 100
freq<- round(diff(c(0,cum.freq*34660)))
a <- st.mle.grouped(breaks=log(us.income), freq, trace=TRUE,
        start=c(1.2, log(0.9), -2.1, log(20)))
print(a$dp)
}
}
\keyword{distribution}


\eof
\name{sn.mmle}
\alias{sn.mmle}
\title{
Modified maximum likelihood estimation for skew-normal models
}
\description{
  Fits a one-dimensional skew-normal (SN) distribution to data, or fits
  a  linear regression model with skew-normal errors, using a
  modification  of maximum likelihood estimation.
}
\usage{
sn.mmle(X, y,  plot.it=TRUE, trace=FALSE, ...)
}
\arguments{
\item{y}{
  a vector contaning the observed variable. This is the response
  variable in case of linear regression.
  Missing values (\code{NA}s) are not allowed.
}
\item{X}{
  a matrix of explanatory variables.
  If \code{X} is missing, then a one-column matrix of all 1's is created.
  If \code{X} has only one column, then it is assumed to be made of 1's.
  Missing values (\code{NA}s) are not allowed.
}
\item{plot.it}{
  logical value, If \code{plot.it=TRUE} (default),
  a plot of the nonparametric estimate of variable \code{y} (or the residuals,
  in the case of regression), and the parametric fit is superimposed.
  See below for details.
}
\item{trace}{
  logical value which controls printing of the algorithm convergence.
  If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{...}{
  Additional arguments passed to \code{sn.mle}
}
}
\value{
a list containing the following components:

\item{call}{
a string containing the calling statement
}
\item{dp}{
a vector of length \code{ncol(X)+2} with estimates of the direct parameters
}
\item{se}{
a vector of standard errors for the parameters
}
\item{Einfo}{
the expected Fisher information matrix for the parameters
}
}
}
\details{
If plotting operates, the function \code{sm.density} of the library \code{sm}
is searched.  If \code{sm.density} is not found, an histogram is plotted.
}
\section{Background}{
  Maximum likelihood estimation for SN models can lead to estimates
  of the shape parameters on the frontier (that is \code{Inf} for the
  DP parameters); see  Azzalini and Capitanio (1999) for a discussion
  of this aspect.
  To avoid this situation, an alternative estimation criterion is the
  method of Sartori-Firth, which involves first regular maximum estimation
  and subsequent re-estimation of the shape parameter using a modified
  score function; see the references below for a full discussion.
  The effect of this modification is "negligible" for large sample size,
  but it avoids estimates of the frontier of the parameter space.
}
\references{

Azzalini, A. and Capitanio, A. (1999).
  Statistical applications of the multivariate skew-normal distribution.
  \emph{J.Roy.Statist.Soc. B} \bold{61}, 579--602.

Firth, D.  (1993).
  Bias reduction of maximum likelihood estimates.
  \emph{Biometrika}  \bold{80}, 27--38. (Corr: 95V82 p.667).

Sartori, N. (2003).
  Bias prevention of maximum  likelihood estimates: skew normal and skew t
  distributions. Working paper 2003-1,
  Department of Statistical Sciences, University of Padua. Submitted.
}
\seealso{
\code{\link{sn.mle}}, \code{\link{sn.Einfo}}
}
\examples{
data(ais)
attach(ais)
a <-  sn.mmle(y=bmi)
#
M <- model.matrix(~lbm+sex)
b <- sn.mmle(M,bmi)
}
\keyword{regression}
\keyword{distribution}


\eof
\name{st.2logL.profile}
\alias{st.2logL.profile}
\title{
Profile twice loglikelihood for skew-t models
}
\description{
Computation and plot of 1-dimensional and 2-dimensional profile
(negative) twice log-likelihood function for skew-t regression models.
}
\usage{
st.2logL.profile(X=matrix(rep(1, n)), y, freq, trace=FALSE,
          fixed.comp = c(ncol(X)+2, ncol(X)+3), 
          fixed.values = cbind(c(-4,4), log(c(1,25))),
          npts=30/length(fixed.comp), plot.it=TRUE, ...)
}
\arguments{
\item{X}{
a matrix of explanatory variables; must have \code{col(X)} equal to
\code{length(y)}. Missing values (\code{NA}) are not allowed.
If \code{X} is missing, a one-column matrix of 1's is created.
}
\item{y}{
a numeric vector. Missing values (\code{NA}s) are not allowed.
}
\item{freq}{
a vector of weights.
If missing, a vector of 1's is created; otherwise
it must have the same number of rows of \code{y}.
}
\item{trace}{
logical value which controls printing of the algorithm convergence.
If \code{trace=TRUE}, details are printed. Default value is \code{FALSE}.
}
\item{fixed.comp}{
a vector containing the subset of the parameters for which the
profile log-likelihood function is required; it can be of length 1 or 2.
The set of components of the parameters are \code{beta, log(omega), 
alpha, log(df)}, where \code{beta} represents the regression parameters
and has \code{ncol(X)} components.
}
\item{fixed.values}{
a numeric vector of values or a matrix with two columns, giving the
range spanned by the selected parameters. 
}
\item{npts}{
number of points on each parameter interval for which the function
must be evaluated.
}
\item{plot.it}{
logical value; if \code{plot.it=TRUE} (default value) a graphical display
is produced.
}
\item{...}{
graphical parameter passed to \code{plot} or \code{contour},
depending on whether \code{lenght(fixed.comp)} is 1 or 2.
}}
\value{
A list containing the following components:
\item{call}{
the matched call.
}
\item{param1}{
vector of values of the first parameters values where the function 
has been evaluated. 
}
\item{param2}{
vectors of the parameters values where the function has been evaluated. 
If \code{length(fixed.comp)=1}, the second vector contains \code{NA}s.
}
\item{deviance}{
a vector or a matrix which represents twice the negative relative 
profile loglikelihood; this is in the "relative" version, i.e.
setting the maximum value to be 0.
}
\item{max.logL}{
a numeric value with the maximum which has been added to
obtain the "relative" version of \code{deviance}.
}
\item{best}{
a list with the output of \code{optim} at the best evaluated point,
that is the one with higher log-likelihood. 
}}
\section{Side Effects}{
If \code{plot.it=TRUE}, a plot of the profile twice relative loglikeliood is
produced on a graphical device. 
If \code{length(fixed.comp)=1}, a plot of the `deviance' is produced
as a function of the chosen parameter component.
If \code{length(fixed.comp)=2}, a contour plot is produced whose
contour lines correspond to confidence regions of approximate
probability levels \code{c(0.25, 0.5,  0.75, 0.90,0.95, 0.99)}.
}
\section{Background}{
The family of skew-t distributions is an extension of the Student's t
family, via the introduction of a \code{shape} parameter which regulates
skewness; when \code{shape=0}, the skew-t distribution reduces to the
usual t distribution. A multivariate version of the distribution exists.
See the reference below for additional information.
}
\references{
Azzalini, A. and Capitanio, A. (2002).
  Distributions generated by perturbation of symmetry 
  with emphasis on a multivariate skew \emph{t} distribution.
  Submitted to \emph{J.Roy. Statist. Soc.}
}
\seealso{
\code{\link{st.mle}}, \code{\link{sn.2logL.profile}}
}
\examples{
data(ais,package)
attach(ais)
a <- st.2logL.profile(y=bmi, xlab="alpha", ylab="log(df)")
a <- st.2logL.profile(y=bmi, fixed.comp=4, fixed.values=log(c(1,25)))
a <- st.2logL.profile(X=cbind(1,lbm), y=bmi, fixed.comp=5,  
         fixed.values=log(c(5,25)), xlab="log(df)", npts=50)
a <- st.2logL.profile(X=cbind(1,Ht), y=Wt, fixed.comp=c(4,5),
         fixed.values=cbind(c(-1,5), log(c(2,25))),
         xlab="alpha", ylab="log(df)")
}
\keyword{distribution}

\eof
\name{st.cumulants}
\alias{st.cumulants}
\title{
Cumulants of the skew-t distribution
}
\description{
Cumulants of the skew-t distribution.
}
\usage{
st.cumulants(shape=0, df=Inf, n=4)
}
\arguments{
\item{shape}{
shape parameter (scalar).
}
\item{df}{
degrees of freedom (scalar); default is \code{df=Inf} which corresponds 
to the skew-normal distribution.
}
\item{n}{
a scalar integer, from 1 to 4.
}}
\value{
the cumulants up to order \code{n} of the skew-t distribution
with the selected parameters.
}
\details{
Expressions of the moments and other details on the skew-t distribution are 
given in the refence below.
}
\references{
Azzalini, A. and Capitanio, A. (2003).
Distributions generated by perturbation of symmetry 
with emphasis on a multivariate skew-\emph{t} distribution.
\emph{J.Roy. Statist. Soc. B} 
\bold{65}, 367--389.
}
\seealso{
\code{\link{sn.cumulants}},\code{\link{dst}}
}
\examples{
cum <- st.cumulants(shape=3, df=5)
}
\keyword{distribution}


\eof
\name{zeta}
\alias{zeta}
\title{
Function `log(2*pnorm(x))' and its derivatives
}
\description{
  The function \code{log(2*(pnorm(x))} and its derivatives up to order 4.
}
\usage{
zeta(k, x)
}
\arguments{
\item{k}{
  an integer scalar between 0 and 4.
}
\item{x}{
  a vector. Missing values (\code{NA}s)  and \code{Inf}s are allowed
}}
\value{
  a vector giving the \code{k}-th order derivative evaluated at \code{x}
}
\details{
  For \code{k} between 0 and 4, the derivative of  order \code{k}
  of \code{log(2*pnorm(x))} is  evaluated, where the derivative of
  order \code{k=0} refers to  the function itself.
  If \code{k} is not integer, it is converted to integer and a warning
  message is generated.
  If \code{k<0} or \code{k>4},  \code{NULL} is returned.
  This function is used by \code{sn.dev} and \code{msn.dev}, among others.
}
\seealso{
\code{\link{sn.mle}}, \code{\link{msn.mle}}
}
\examples{
y <- zeta(2,seq(-20,20,by=0.5))
#
for(k in 0:4) curve(zeta(k,x), from=-1, to=5, col = k+2, add = k > 0)
legend(3, -0.5, legend=as.character(0:4), col=2:6, lty=1)
}
\keyword{math}

\eof
