\name{Ops.Sample}
\alias{Ops.Sample}
\alias{sum.Sample}
\alias{prod.Sample}
\title{ Basic Operations for Sample objects }
\description{
These functions apply the basic operations pointwise to the waveforms of the samples.
}
\usage{s1 + s2
s1 - s2
s1 * s2
const * s
s  * const
s  / const
sum.Sample(s1, s2, ...)
prod.Sample(s1, s2, ...)
}
\arguments{
  \item{s1, s2, \dots}{ Sample objects, or strings giving the name of wav files. }
  \item{const}{ a double. }
}
\details{
The sum of two sample objects corresponds to the sound when both samples are played at a time. The product of two samples causes an effect called ringmodulation, but it can also be used to add some vibrato to a sound (see the examples).

If the samples have different sample parameters (bits, rate and channels), the system uses the command \code{\link{fitSampleParameters}} to adjust them before the waveforms are combined.

Be careful to make sure that the resulting waveform does not exceed the interval [-1,1] when it is played or saved to disk, otherwise you will lose information and hear cracks in the sound. To avoid this, you can use \code{const * s} or the \code{\link{normalize}} command.

In \code{prod} and \code{sum} also the names of wavefiles can be used. Other forms like \code{s1+s2} do not accept filenames as arguments. If the first argument \code{s1} is a filename, the explicit forms \code{sum.Sample} and \code{prod.Sample} must be used.
}

\value{
  a Sample object. 
}

\author{ Matthias Heymann }

\seealso{ \code{\link{normalize}}, \code{\link{center}} }

\examples{\dontrun{s1 <- Sine(440,1)
s2 <- Sine(220,1)
play((s1+s2)/2) # both samples at a time

play(Sine(440,1)*Sine(5,1)) # vibrato}
}
\keyword{ manip }
\eof
\name{Sample}
\alias{Sample}
\alias{as.Sample}
\alias{is.Sample}
\title{ Sample Objects }
\description{
  \code{as.Sample} creates a Sample object from a given numeric matrix.

  \code{is.Sample} tests if its argument is a Sample object or the name of a wav file.
}
\usage{as.Sample(sound, rate, bits)
is.Sample(s, argname="'s' ")
}
\arguments{
  \item{sound}{ a \code{channels(s)} x \code{sampleLength(s)} matrix or a vector of doubles describing the waveform(s) of the sample. }
  \item{rate}{ the sampling rate (number of samples per second). }
  \item{bits}{ the sample quality (the number of bits per sample), 8 or 16. }
  \item{s}{ an R object to be tested.}
  \item{argname}{ a string giving the name of the object that is tested. It is used for creating an error message. }
}

\details{ The rows of the matrix represent the channels of the sample: If \code{sound} is a vector or a matrix with only one row,
\code{as.Sample} will return a mono sample; if \code{sound} is a matrix with two rows, \code{as.Sample} returns a stereo sample, where the left and the right channel are represented by the first and the second row, respectively.

\code{sound} can contain any real number, but when the Sample object is played or saved to disk, [-1,1] is regarded as the range of the sample, and any values outside this interval will cause cracks in the sound.

A Sample object's waveform can exceed this interval during calculations. It is the task of the programmer to take care about the range of the waveform before saving or playing the sample, for example by using the \code{\link{normalize}} command.

Internally, the sound is saved as a matrix with doubles, independent of the \code{bits} parameter that is only used when the Sample object is played or saved to disk.

The \code{is.Sample} command is used by several other routines that allow both Sample objects and filenames as arguments.
}

\value{
  For \code{as.Sample} a Sample object, that is a list with the components \code{$sound}, \code{$rate} and \code{$bits}.

  \code{is.Sample} returns a list with the entries
  \item{test }{a logical indicating whether or not \code{s} is a Sample object or the name of a valid wav file.}
  \item{error }{a string with one of the messages "Filename must have the extension .wav.", "File not found.", "No read permission for this file.", or "Argument "+ argname + "must be a Sample object or the name of a wav file." If \code{test=TRUE}, this list entry doesn't exist.}
}

\author{ Matthias Heymann }

\seealso{ \code{\link{stereo}} for creating a stereo Sample object from two mono Sample objects, \code{\link{loadSample}} for loading a wav file and turning it into a Sample object, \code{\link{saveSample}} for saving a Sample object as a wav file, \code{\link{sound}}, \code{\link{bits}}, \code{\link{rate}}, \code{\link{channels}}, \code{\link{sampleLength}} and \code{\link{duration}} for access to the basic parameters of a Sample object.
}

\examples{\dontrun{waveLeft <- 2*((seq(0,80,length=88200)\%\%1^2)-.5)
s <- as.Sample(waveLeft,44100,16)
play(s) # a mono sample

waveRight <- waveLeft[88200:1]
s <- as.Sample(rbind(waveLeft,waveRight),44100,16)
play(s) # a stereo Sample

# How to use is.Sample to allow both a Sample object and a filename
# as an argument:
s <- anyargument
sampletest <- is.Sample(s, argname="s ")
if (!sampletest$test) stop(sampletest$error) #no valid argument
t <- loadSample(s,filecheck=FALSE)
# If s is Sample object, loadSample will return it immediately.
# If s is a string, the Sample object will be loaded from disk.
# No check for existance of the file will be performed since this
# was already tested in is.Sample.
#
# Now s is a Sample object, continue with routine.
}
}

\keyword{ classes }

\eof
\name{Sine}
\alias{Sine}
\alias{Sawtooth}
\alias{Square}
\alias{Silence}
\alias{Noise}
\title{ Create Sample Objects for the Basic waveforms }
\description{
  Create a Sample object with a sine, sawtooth, or square waveform, silence, or noise.
}
\usage{Sine(freq, dur, rate=44100, bits=16, channels=1)
Sawtooth(freq, dur, rate=44100, bits=16, channels=1, reverse=FALSE)
Square(freq, dur, rate=44100, bits=16, channels=1, upPerc=50)
Silence(dur, rate=8000, bits=8, channels=1)
Noise(dur, rate=44100, bits=16, channels=1)
}
\arguments{
  \item{freq}{ the frequency (a double). }
  \item{dur}{ the duration in seconds (a double). }
  \item{rate}{ the sampling rate, an integer between 1000 and 48000. }
  \item{bits}{ the sampling quality in bits per sample, 8 or 16. }
  \item{channels}{ 1 for mono, or 2 for stereo. }
  \item{reverse}{ logical. If \code{TRUE}, the waveform will be mirrored vertically. }
  \item{upPerc}{ a number between 0 and 100 giving the percentage of the waveform with value +1. }
}
\details{
 If \code{channels=2}, left and right side of the sample will be the same for \code{Sine}, \code{Sawtooth} and \code{Square}. For \code{Noise}, both sides will be generated separately, using \code{runif}.
}
\value{
  a Sample object.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{as.Sample}}, \code{\link{loadSample}}, \code{\link{nullSample}} }

\examples{\dontrun{s1 <- Sine(440,1)
play(s1)

s2 <- Sawtooth(440,1)
play(s2)

play(Noise(1))}
}
\keyword{ sysdata }

\eof
\name{WavPlayer}
\alias{WavPlayer}
\alias{findWavPlayer}
\alias{setWavPlayer}
\title{ Set or Get the Command for Playing WAV Files }
\description{
  \code{findWavPlayer} returns the most common system commands on your OS for playing wav files.

  \code{WavPlayer} returns the command that is currently used by \code{\link{play}}.

  \code{setWavPlayer} is used to define the command that is used by \code{\link{play}}.
}
\usage{findWavPlayer()
WavPlayer()
setWavPlayer(command=NULL)
}
\arguments{
  \item{command}{ a vector of character strings giving the command to be used as \code{"command wavfile.wav"}. If it contains more than one string, the commands are tested one after the other, and the first one that works properly will be used for future calls of the \code{\link{play}} command. If \code{command=NULL},the command \code{findWavPlayer()} is used to determine the standard commands for your system. }
}
\details{ The \code{\link{play}} command makes a system call of the form \code{"}command\code{ }wavfile\code{.wav"}, where 'command' is the string returned by \code{WavPlayer()}.

The default commands are \code{'mplay32 /play'} (calling the Windows media player) for Win32-systems and \code{'play'} and \code{'playwave'} for Linux systems. Other commands will be added in future versions of this package.

While the Windows Media player is included in the standard Windows installation, playwave might have to be installed manually. Under RedHat Linux playwave is part of the SDL\_mixer package. To download it, go to \url{http://www.libsdl.org/projects/SDL_mixer}.

But any other program that provides a system call of the above form to play wav files is also fine. Please report additional play commands to the author (send an email to \email{mail@MatthiasHeymann.de}) so that they can be recognized automatically in future versions of this package.

\code{setWavPlayer} is called directly after loading the package.

When \code{setWavPlayer} is called, it tries to play an empty wav file, using the new command(s). If it fails, no changes are made.
}
\value{
  \code{WavPlayer} returns the wav play command that is currently used, or \code{NULL}, if none is selected yet.

  \code{findWavPlayer} returns the default commands for your system, or \code{NULL}, if no command is known for your system.
}
\author{ Matthias Heymann}

\seealso{ \code{\link{play}} for playing Sample objects or wav files. }

\examples{\dontrun{setWavPlayer("playwave")
# tries to set the command "playwave wavfile.wav" as the 
# preference for playing wav files with the play command.
# If successful,
WavPlayer()
# returns the string "playwave" afterwards.}
}
\keyword{ sysdata }
\eof
\name{appendSample}
\alias{appendSample}
\title{ Append Samples }
\description{
  Append two or more Sample objects or wav files.
}
\usage{appendSample(s1, s2, ...)
}
\arguments{
  \item{s1, s2, \dots}{ Sample objects, or the names of wav files. }
}
\details{
If the samples have different sample parameters (bits, rate and channels), the command \code{\link{fitSampleParameters}} is called to adjust them before the samples are appended.
}
\value{
  a Sample object with the samples played one after the other.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{cutSampleEnds}} to avoid cracks between two appended samples,

\code{\link{sum.Sample}} for playing several samples at a time. }

\examples{\dontrun{s1 <- Sine(440,1)
s2 <- Sine(550,1)
s3 <- Sine(660,1)
s4 <- Sine(880,1)
play(appendSample(s1,s2,s3,s4))}
}
\keyword{ manip }

\eof
\name{bits}
\alias{bits}
\alias{bits<-}
\alias{setBits}
\title{ Bits per Sample }
\description{
  Get or set the \code{bits} parameter (the sampling quality) of a Sample object or a wav file.
}
\usage{bits(s)
bits(s) <- b
setBits(s,b)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{b}{ the number of bits per sample, 8 or 16. }
}
\details{
The replacement form can be used to reset the sampling quality of a Sample object, that is the number of bits per sample (8 or 16). Here, filenames are not accepted.
}
\value{
  For \code{bits}, the bits parameter (number of bits per sample) of the Sample object, 8 or 16.

  For \code{setBits}, a Sample object with the new \code{bits} parameter. 
}
\author{ Matthias Heymann }
\note{ Changing the sampling quality of a Sample object does not affect its actual data but only its \code{$bits} flag. The sampling quality is only used when a Sample object is played or saved to disk. Internally, \R always uses doubles for the waveform.

An 8 bit sample needs only half the disk space but has less quality.
}

\seealso{ \code{\link{fitSampleParameters}} }

\examples{\dontrun{s <- Sine(20000,1,rate=44100,bits=16)
play(s)
print(s)
bits(s) <- 8
play(s)  # now worse quality
print(s) # but less disk space
play(setBits(s,16)) # now better quality again, since waveform data was not changed.}
}
\keyword{ attribute }

\eof
\name{center}
\alias{center}
\title{ Center a Sample Object. }
\description{
This function adds a constant to a Sample object's waveform, so that its mean gets zero. This makes sense especially for sonification purposes, when (in general non-centered) data is transformed into sound.
}
\usage{center(s)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
}
\details{
For a stereo Sample object, both channels are treated separately.
}
\value{
  a Sample object with zero as the mean of each channel's waveform.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{normalize}} }

\examples{\dontrun{x <- seq(0,50*pi,length=10000)
waveform <- (sin(x))^2 + .6*cos(x/2)^2
s <- as.Sample(waveform,44100,16)
plot(s) # nice idea, but wrong range for a sample
play(s) # sounds ugly, too
s <- center(s)
plot(s)  # now zero is the mean
play(s)  # sounds good, but too quiet
s <- normalize(s)
plot(s)  # this looks like a perfect sample!
play(s)  # e voila!}
}
\keyword{ manip }
\eof
\name{channels}
\alias{channels}
\alias{channels<-}
\alias{setChannels}
\title{ Number of Channels of a Sample Object }
\description{
  Get or set the number of channels of a Sample object or a wav file.
}
\usage{channels(s)
channels(s) <- c
setChannels(s,c)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{c}{ 1 for mono, or 2 for stereo. }
}
\details{
The replacement form can be used to reset the number of channels of a Sample object (here, filenames are not accepted).

If a mono sample is transformed into a stereo sample, each channel of the stereo sample equals the waveform of the mono sample. If a stereo Sample is transformed to a mono sample, \code{(left(s)+right(s))/2} is returned.
}
\value{
  For \code{channels}, the number of channels of the sample (1 for mono, 2 for stereo).

  For \code{setChannels}, a Sample object with the new \code{channels} parameter. 
}
\author{ Matthias Heymann }

\seealso{ \code{\link{fitSampleParameters}} }

\examples{\dontrun{s <- stereo(Sine(440,1),Sine(220,1))
channels(s)  # 2
play(s)
channels(s) <- 1  # now a mono sample
play(s)}
}
\keyword{ attribute } 

\eof
\name{cutSample}
\alias{cutSample}
\alias{"[.Sample"}
\title{ Cut Sample Objects }
\description{
  Cut a part out of a Sample object.
}
\usage{cutSample(s, start, end)
s[v]
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{start}{ the start position in seconds. }
  \item{end}{ the end position in seconds. }
  \item{v}{ a vector of integers giving the numbers of the columns in the waveform matrix to be used. }
}
\details{
Only the intersection of [\code{start},\code{end}] with [0,\code{duration(s)}] is returned. Similarly, in the second form the intersection of \code{v} with \code{1:sampleLength(s)} is returned.
}
\value{
  the specified part of the given sample as a new Sample object.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{sound}} for direct access to the waveform matrix,

\code{\link{cutSampleEnds}} and \code{\link{noSilence}} for special cutoff techniques. }

\examples{\dontrun{s <- appendSample(Sine(330,1),Sine(440,1))
play(cutSample(s,.8,1.8))
play(s[(44100*.8):(44100*1.8)])  # the same}
}
\keyword{ manip }

\eof
\name{cutSampleEnds}
\alias{cutSampleEnds}
\title{ Prepare Sample Object for appendSample }
\description{
  Prepare a Sample object or a wav file for usage of \code{\link{appendSample}} to avoid cracks between two appended samples.
}
\usage{cutSampleEnds(s)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
}
\details{
  At the beginning of the sample, all values in the waveform until the first transition from negative to positive values are dropped, at the end everything after the last transition from negative to positive values is dropped.

Currently, only channel 1 is used to determine which parts to drop. Hence in stereo samples there can still be some cracks in the right channel.
}

\value{
  a Sample object.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{cutSample}}, \code{\link{appendSample}} }

\examples{\dontrun{s1 <- Sine(440,.01)
s2 <- Sine(550,.01)
s3 <- Sine(660,.01)
s4 <- Sine(880,.01)
l <- list(s1,s2,s3,s4)
#  first without cutSampleEnds:
s <- nullSample()
for (i in 1:99) {
  s <- appendSample(s,l[[i\%\%4+1]])
}
play(s) # ugly cracks
# now with cutSampleEnds:
s <- nullSample()
for (i in 1:99) {
  s <- appendSample(s,cutSampleEnds(l[[i\%\%4+1]]))
}
play(s) # no cracks,

# This is how it works:
# The waveform is not smooth between s1 and s2:
plot(appendSample(s1,s2))
# This is because s1 just ends somewhere at y=0.6:
plot(s1)
# Let's cut off the last positive part of it:
plot(cutSampleEnds(s1))
# A similar cuttoff would be made at the beginning
# of the sample (if it was necessary).
# Now the two samples fit perfectly (the cut is at x=400):
plot(appendSample(cutSampleEnds(s1),cutSampleEnds(s2)))}
}

\keyword{ manip }

\eof
\name{duration}
\alias{duration}
\alias{duration<-}
\alias{setDuration}
\title{ Duration of a Sample Object }
\description{
  Get or set the duration (in seconds) of a Sample object or a wav file.
}
\usage{duration(s)
duration(s) <- d
setDuration(s,d)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{d}{ a double giving the duration in seconds. }
}
\details{
The replacement form can be used to reset the duration of the Sample object (here, filenames are not accepted).

If a Sample object is shortened, extra values are discarded. When a Sample object is lengthened, it is padded out to its new length with zeros (silence).
}
\value{
  For \code{duration}, the duration of the sample in seconds.

  For \code{setDuration}, a Sample object with the new duration. 
}
\author{ Matthias Heymann }

\seealso{ \code{\link{sampleLength}} }

\examples{\dontrun{s <- Sine(440,3)
duration(s)  # 3
duration(s) <-.5  # sample is now .5 sec long
play(setDuration(s,1)) # plays a .5 sec sine wave and then .5 sec silence}
}
\keyword{ attribute }

\eof
\name{fitSampleParameters}
\alias{fitSampleParameters}
\title{ Adjust Parameters of Two Sample Objects. }
\description{
  Adjust the parameters sampling rate, channels and bits/sample of two Sample objects.
}
\usage{fitSampleParameters(s1, s2)
}
\arguments{
  \item{s1,s2}{ a Sample object, or strings giving the name of a wav file. }
}
\details{ The commands \code{\link{rate}}, \code{\link{channels}} and \code{\link{bits}} are used to transform copies of \code{s1} and \code{s2} to samples with the same parameters rate, channels and bits. Always the parameter with the better quality is chosen for the returned samples, that is the higher sampling rate, the larger number of channels and the larger number of bits per sample.
}
\value{
  a list containing the two transformed Samples as components.
}
\author{ Matthias Heymann }

\note{ This routine is called before certain commands such as \code{\link{sum.Sample}} or \code{\link{appendSample}} are applied to Sample objects with different parameters. }

\seealso{ \code{\link{rate}}, \code{\link{channels}}, \code{\link{bits}} }

\examples{\dontrun{s1 <- Sine(440,1,rate=22050,channels=1,bits=16)
s2 <- Sawtooth(440,1,rate=44100,channels=2,bits=8)
play(s1)
play(s2)
l  <- fitSampleParameters(s1,s2)
t1 <- l[[1]]
t2 <- l[[2]]
print(t1)
print(t2)  # both samples have the same parameters now
play(t1)
play(t2)   # none of the samples sounds different now,
           # since only parameters with higher quality were chosen}
}
\keyword{ manip }

\eof
\name{left}
\alias{left}
\alias{right}
\title{ Extract one Channel from a Stereo Sample }
\description{
  Extract either the left or the right channel of a stereo Sample object or a stereo wav file.
}
\usage{left(s)
right(s)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
}
\details{
If \code{s} is a mono sample, it will be returned as it is.
}
\value{
  a Sample object containing the left or the right channel of \code{s}.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{stereo}} for creating a stereo Sample object from two mono samples. }

\examples{\dontrun{sLeft <- Sine(440,1)
sRight <- Sine(220,1)
s <- stereo(sLeft,sRight)
play(s)
play(left(s))  # only the left channel
play(right(s)) # only the right channel}
}
\keyword{ manip }

\eof
\name{loadSample}
\alias{loadSample}
\title{ Load a WAV File from Disk }
\description{
  Load a wav file from disk and create a Sample object.
}
\usage{loadSample(filename, filecheck=TRUE)
}
\arguments{
  \item{filename}{ a string giving the path and the name of the wav file. }
  \item{filecheck}{ logical. If FALSE, no check for existance and read permission of the file will be performed. }
}
\details{
All kinds of wav files are supported: mono / stereo, 8 / 16 bits per sample, 1000 to 48000 samples/second.
}
\value{
  the Sample object that is equivalent to the wav file.
}
\author{ Matthias Heymann }

\note{ \code{filename} can also be a Sample object. In this case, the same object will be returned immediately. This can be useful when writing functions that accept both Sample objects and the names of a wav file as an argument. See \code{\link{is.Sample}} for an example. }

\seealso{  \code{\link{saveSample}}, \code{\link{as.Sample}} }

\examples{\dontrun{s <- loadSample("soundfile.wav")
play(s)}}

\keyword{ file }
\eof
\name{mirror}
\alias{mirror}
\title{ Mirror a Stereo Sample }
\description{
  Interchange the left and the right channel of a stereo Sample object or a stereo wav file.
}
\usage{mirror(s)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
}
\details{
If \code{s} is a mono sample, it will be returned as it is.
}
\value{
  a Sample object, with the left and the right channel of \code{s} interchanged.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{panorama}} for a command with more parameters, \code{\link{left}} and \code{\link{right}} for access to single channels of a sample. }

\examples{\dontrun{s <- stereo(Sine(440,1),Sine(220,1))
play(s)  # higher tone is on the left
play(mirror(s))  # now higher tone is on the right}
}
\keyword{ manip }

\eof
\name{noSilence}
\alias{noSilence}
\title{ Cut Off Silence from a Sample Object }
\description{
  Cut off silence or low noise at the beginning and/or at the end of a Sample object or a wav file.
}
\usage{noSilence(s, level=0, start=TRUE, end=TRUE)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{level}{ non-negative numeric. Absolute values in the waveform matrix smaller than or equal to this value are regarded as silence. }
  \item{start}{ logical. If \code{TRUE}, silence at the beginning of the sample will be cut off. }
  \item{end}{ logical. If \code{TRUE}, silence at the end of the sample will be cut off. }
}
\details{
For stereo samples, it is checked if the values of both channels are silence before the silence is cut off.
}
\value{
  a Sample object without those parts at the start and at the end of the original sample that are below the specified noise level.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{cutSample}} }

\examples{\dontrun{s <- Sine(440,5)
sound(s) <- sound(s)*matrix(seq(1,0,length=5*44100),nrow=1)
sampleLength(s)
play(s) # fade out
s <- noSilence(s,level=.05)
sampleLength(s)  # s is shorter now
play(s)    # although you don't hear that the end is missing}
}
\keyword{ manip }

\eof
\name{normalize}
\alias{normalize}
\title{ Rescale the Range of a Sample to [-1,1] }
\description{
  Multiply the waveform of a Sample object or a wav file with a positive constant so that the maximum absolut value becomes 1, or any other specified constant.

Use this command before saving or playing a Sample object to avoid cracks in the sound caused by parts in the waveform that exceed the range [-1,1]. 
}
\usage{normalize(s, level=1)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{level}{ a number between 0 and 1 specifying the desired maximum absolute value of the waveform. }
}
\value{
  a Sample object with the specified maximum absolut value of the waveform.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{Ops.Sample}}, \code{\link{center}} }

\examples{\dontrun{s <- .6*Sine(440,1)
plot(s)
plot(normalize(s)) # now it uses the full range
play(s)
play(normalize(s)) # this one is louder}
}
\keyword{ manip }

\eof
\name{nullSample}
\alias{nullSample}
\title{ The NULL Sample Object }
\description{
  Create a Sample object whose waveform has length 1 and value 0. Often useful to initialize loops.
}
\usage{nullSample(rate=44100, bits=16, channels=1)
}
\arguments{
  \item{rate}{ the sampling rate, between 1000 and 48000. }
  \item{bits}{ the sample quality (number of bits per sample), 8 or 16. }
  \item{channels}{ 1 for mono, or 2 for stereo. }
}
\value{
  a Sample object.
}
\author{ Matthias Heymann }

\note{ Future versions may use a special NULLSample flag instead of using a sample of length 1. }

\seealso{ \code{\link{Silence}}}

\examples{\dontrun{scale <- 2^(seq(0,1,length=13))[c(1,3,5,6,8,10,12,13)]
base <- 440
s <- nullSample()
for (f in scale)
  s <- appendSample(s,Sine(f*base,1))
play(s)}
}
\keyword{ sysdata }
\eof
\name{panorama}
\alias{panorama}
\title{ Narrow the Panorama of a Stereo Sample }
\description{
  Narrow the panorama of a stereo Sample object or of a stereo wav file.
}
\usage{panorama(s, pan)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{pan}{ a number between -50 and 50 giving the width of the panorama. }
}
\details{
If \code{abs(pan)<50}, mixtures of the two channels of \code{s} are used for the left and the right channel of the returned Sample object, so that they appear closer to the center. For \code{pan=0}, both sounds are completely in the center.

If \code{pan<0}, the left and the right channel are interchanged afterwards.
}
\value{
  a Sample object with the transformed panorama.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{mirror}} for \code{pan=-50}, \code{\link{left}} and \code{\link{right}} for access to single channels of the sample. }

\examples{\dontrun{s <- stereo(Sine(440,1),Sine(330,1))
play(s)
play(panorama(s,30))  # now right and left tones are closer to the center
play(panorama(s,10))  # now even closer
play(panorama(s,0))   # now both at the center, the same as setChannels(s,1)
play(panorama(s,-30)) # again wider, but both sides switched
play(panorama(s,-50)) # the same as mirror(s)}
}
\keyword{ manip }

\eof
\name{pitch}
\alias{pitch}
\title{ Pitch a Sample Object }
\description{
  Change the pitch of a Sample object or a wav file.
}
\usage{pitch(s, semitones)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{semitones}{ a double giving the number of semitones to be pitched. }
}
\details{
Pitching +12 semitones (+1 octave) means to double the frequencies. Negative values of \code{semitones} are used to lower the frequencies.

Note that this transformation changes the actual data of the sample. Since pitching a sample is equivalent to playing it at a different speed, the length of the Sample object will also change.
}

\value{
  a Sample object.
}

\author{ Matthias Heymann }

\note{ Future versions of this command may use a different algorithm to increase the quality of the returned sample. }

\examples{\dontrun{s <- Sine(440,1)
# Now play it 12 semitones = 1 octave deeper,
# that is half the frequencies and twice the length,
# or played at half speed.
play(pitch(s,-12)) # is the same as...
play(Sine(220,2))}
}
\keyword{ manip }

\eof
\name{play}
\alias{play}
\alias{play.Sample}
\alias{play.default}
\title{Play a Sample Object or a WAV File}
\description{
  Play a Sample object or a wav file, using the wav file play command returned by \code{\link{WavPlayer}}, or any other specified command.
}
\usage{play(s, stay=FALSE, command=WavPlayer())
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{stay}{ logical. If TRUE, the Windows Media Player won't be closed after playing the sample. }
  \item{command}{ a character string giving the system command to be used for playing the sample. }
}
\details{
If \code{s} is a Sample object, it will be saved into a temporary folder before it is played. The temporary file will only be deleted afterwards if \code{stay=FALSE}.
}

\author{ Matthias Heymann }

\examples{\dontrun{s <- Sine(440,1)
play(s)}
}
\keyword{ IO }

\eof
\name{plot.Sample}
\alias{plot.Sample}
\title{ Plot a Sample Object }
\description{
  Plot the waveform of a Sample object or a wav file.
}
\usage{\method{plot}{Sample}(x,xlab="sample #",ylab=NULL,\dots)
}
\arguments{
  \item{x}{ a Sample object, or a string giving the name of a wav file. If \code{x} is a string, the explicit form \code{plot.Sample} must be used. }
  \item{xlab}{ the character string giving the label for the x-axis.}
  \item{ylab}{ For mono Sample objects as usual. For stereo Sample objects, \code{ylab} can be a vector of two strings to distinguish the y-labels for the left and the right channel. If \code{ylab=NULL}, the presets are used, that is \code{"waveform"} for mono samples and \code{c("left","right")} for stereo samples.}
  \item{\dots}{ further graphical parameters. }
}
\author{ Matthias Heymann }

\note{ Use \code{plot(s[interval])} to plot parts of \code{s} only (see examples).

If the range of the graph exceeds [-1,1], you can use the \code{\link{normalize}} command before plotting to get a better view of the waveform. (Then you should also call this function to avoid cracks in the sound before you save or play it the next time.) }

\seealso{ \code{\link{print.Sample}} }

\examples{\dontrun{s <- Sine(440,1) + .4*Sine(1000,1)
plot(s[1:1000])
play(s)
s <- normalize(s)
plot(s[1:1000])  # now the range of the waveform is in [-1,1]
play(s)  # no cracks!}
}
\keyword{ hplot }

\eof
\name{print.Sample}
\alias{print.Sample}
\title{ Print a Sample Object }
\description{
  Display the basic information about a Sample object or a wav file.
}
\usage{print.Sample(x,\dots)
}
\arguments{
  \item{x}{ a Sample object, or a string giving the name of a wav file. }
  \item{\dots}{ further parameters, not used at the moment. }
}
\author{ Matthias Heymann }

\seealso{ \code{\link{plot.Sample}} for plotting the waveform of a sample. }

\examples{\dontrun{s <- Sine(440,1)
print(s)}
}

\keyword{ print }

\eof
\name{rate}
\alias{rate}
\alias{rate<-}
\alias{setRate}
\title{ The Sampling Rate }
\description{
  Get or set the sampling rate (number of samples per second) of a Sample object or a wav file.
}
\usage{rate(s)
rate(s) <- r
setRate(s,r)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{r}{ an integer between 1000 and 48000 giving the sampling rate. }
}
\details{
The replacement form can be used to reset the sampling rate. Here, filenames are not accepted.

Note that changing the sampling rate of a Sample object affects the waveform of the sample.
}
\value{
  For \code{rate}, the sampling rate (number of samples per second) of the sample.

  For \code{setRate}, a Sample object with the new sampling rate.
}
\author{ Matthias Heymann }

\note{ Common sampling rates are between 8000 and 44100 (CD quality). The sampling rate of DAT recorders is 48000. Not every rate is guaranteed to be supported by every wav file player.

Future versions may use a different algorithm for sampling rate conversion to achieve a better sound quality for the returned sample.
}

\seealso{ \code{\link{fitSampleParameters}}, \code{\link{pitch}} }

\examples{\dontrun{s <- Sine(440,1,rate=44100)
rate(s)  # 44100
play(s)
print(s)
rate(s) <- 8000
play(s)  # s has worse quality now (noise and additional high frequencies)
print(s) # but uses less memory}
}
\keyword{ attribute }

\eof
\name{reverse}
\alias{reverse}
\title{ Play a Sample Object Backwards }
\description{
  Returns the Sample object (or wav file) played backwards.
}
\usage{reverse(s)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
}
\value{
  a Sample object with the same parameters but with the sound played backwards.
}
\author{ Matthias Heymann }

\examples{\dontrun{waveform <- 2*((seq(0,80,length=88200)\%\%1^2)-.5)
s <- as.Sample(waveform,44100,16)
play(s)
play(reverse(s)) # now played backwards}
}
\keyword{ manip }

\eof
\name{sampleLength}
\alias{sampleLength}
\alias{sampleLength<-}
\alias{sampleLength<-.Sample}
\alias{setSampleLength}
\title{ Length of a Sample Object }
\description{
  Get or set the length (number of columns in the waveform matrix) of a Sample object or a wav file.
}
\usage{sampleLength(s)
sampleLength(s) <- l
setSampleLength(s,l)
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{l}{ an integer giving the sample length (number of columns in the waveform matrix). }
}
\details{
The replacement form can be used to reset the sample length (here, filenames are not accepted).

If a Sample object is shortened, extra values are discarded. When a Sample object is lengthened, it is padded out to its new length with zeros (silence).
}
\value{
  For \code{sampleLength}, the number of columns in the waveform matrix of the sample.

  For \code{setSampleLength}, a Sample object with the new length.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{duration}} }

\examples{\dontrun{s <- Sine(440,3,rate=44100,bits=16,channels=2)
sampleLength(s)  # 132300 samples ( = 3 sec * 44100 samples/sec )
sampleLength(s) <- 22050  # sample is now .5 sec long
play(setSampleLength(s,44100)) # plays a .5 sec sine wave and then .5 sec silence}
}
\keyword{ attribute }

\eof
\name{saveSample}
\alias{saveSample}
\title{ Save a Sample Object as a WAV File }
\description{
  Save a Sample object to disk as a wav file.
}
\usage{saveSample(s, filename, overwrite=FALSE)
}
\arguments{
  \item{s}{ a Sample object. }
  \item{filename}{ a string giving the path and the name of the destination file. }
  \item{overwrite}{ logical. If \code{FALSE} and \code{filename} already exists, an error will be reported. Otherwise the old file will be deleted. }
}
\author{ Matthias Heymann }

\seealso{ \code{\link{loadSample}} }

\examples{\dontrun{s <- Sine(440,1)
saveSample(s,"sine.wav")}
}
\keyword{ file }

\eof
\name{sound}
\alias{sound}
\alias{sound<-}
\title{ The Waveform Matrix of a Sample Object }
\description{
  Get or set the waveform matrix of a Sample object or a wav file.
}
\usage{sound(s)
sound(s) <- waveform
}
\arguments{
  \item{s}{ a Sample object, or a string giving the name of a wav file. }
  \item{waveform}{ a \code{channels(s)} x \code{sampleLength(s)} matrix of doubles. }
}
\details{
The replacement form can be used to reset the waveform of a sample object. Here, filenames are not accepted for code{s}.

The matrix can have one (for mono samples) or two rows (for stereo samples), where in the latter case the first row corresponds to the left and the second row to the right channel.

It contains the waveform(s) of the Sample object as sequence(s) of numbers between -1 and 1. \code{waveform} can contain arbitrary real numbers, but when the Sample object is played or saved to disk, [-1,1] is regarded as the native range of the sample, and any values outside this interval will cause cracks in the sound.

The waveform of a Sample object might exceed this interval during calculations. It is the task of the programmer to take care about the range of the waveform before saving or playing the sample, for example by using the \code{\link{normalize}} function.
}
\value{
  the waveform matrix of the sample.
}
\author{ Matthias Heymann }

\seealso{\code{\link{as.Sample}}}

\examples{\dontrun{s <- Sine(440,1,channels=2)  # stereo sine wave
sound(s)[2,] <- sound(s)[2,]*seq(1,0,length=sampleLength(s))
play(s)  # right channel fades to zero}
}
\keyword{ attribute }

\eof
\name{stereo}
\alias{stereo}
\title{ Create a Stereo Sample Object from Two Mono Samples }
\description{
  Create a stereo Sample object, given the two channels as Sample objects or wav files.
}
\usage{stereo(sLeft, sRight, pan=50)
}
\arguments{
  \item{sLeft}{ a Sample object or a string giving the name of a wav file. Used for the left channel. }
  \item{sRight}{ a Sample object or a string giving the name of a wav file. Used for the right channel. }
  \item{pan}{ a number between -50 and 50 describing the distance between the two sound sources.
}
}
\details{
If \code{abs(pan)<50}, mixtures of the two sources are used for the left and the right channel so that they appear closer to the center. For \code{pan=0}, both sounds are at the center. If \code{pan<0}, left and right channel are interchanged afterwards.

If the samples have different sample parameters (bits, rate and channels), the command code{\link{fitSampleParameters}} is called to adjust them before the two samples are combined.
}

\value{
  a stereo Sample object.
}
\author{ Matthias Heymann }

\seealso{ \code{\link{left}}, \code{\link{right}}, \code{\link{as.Sample}}, \code{\link{panorama}} }

\examples{\dontrun{sLeft <- Sine(440,1)
sRight <- Sine(220,1)
s <- stereo(sLeft,sRight)
play(s) }
}
\keyword{ manip }
\eof
