
% 	$Id: correlation.systemfit.Rd,v 1.3 2003/11/10 20:19:04 hamannj Exp $	

\name{correlation.systemfit}
\alias{correlation.systemfit}

\title{Correlation between Predictions from Equation i and j}

\description{
  \code{correlation} returns a vector of the correlations
  between the preditions of two equations in a set of equations. The
  correlation between the predictions is defined as,

  \deqn{
    r_{ijk} = \frac{x'_{ik}C_{ij}x_{jk}}{\sqrt{(x'_{ik}C_{ii}x_{ik})(x'_{jk}C_{jj}x_{jk})}}
  }

  where \eqn{r_{ijk}} is the correlation between the predicted values of
  equation i and j and  \eqn{C_{ij}} is the cross-equation variance-covariance
  matrix between equations i and j.

}

\usage{
correlation.systemfit( results, eqni, eqnj )
}

\arguments{
  \item{results}{an object of type systemfit.system.}
  \item{eqni}{index for equation i}
  \item{eqnj}{index for equation j}
}

\value{
  \code{correlation} returns a vector of the correlations between the
  predicted values in equation i and equation j.
}

\references{    

  Greene, W. H. (1993)
  \emph{Econometric Analysis, Second Edition}, Macmillan.

  Hasenauer, H; Monserud, R and T. Gregoire. (1998)
  Using Simultansous Regression Techniques with Individual-Tree Growth Models.
  \emph{Forest Science}. 44(1):87-95

  Kmenta, J. (1997)
  \emph{Elements of Econometrics, Second Edition}, University of
  Michigan Publishing

}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
inst <- ~ d + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform 2SLS on each of the equations in the system
fit2sls <- systemfit( "2SLS", system, labels, inst)
print( fit2sls )
print( fit2sls$rcov )

## perform the 3SLS
fit3sls <- systemfit( "3SLS", system, labels, inst )
print( fit3sls )
print( "covariance of residuals used for estimation (from 2sls)" )
print( fit3sls$rcovest )
print( "covariance of residuals" )
print( fit3sls$rcov )

## examine the correlation between the predicted values
## of suppy and demand by plotting the correlation over
## the value of q
r12 <- correlation.systemfit( fit3sls, 1, 2 )
plot( q, r12, main="correlation between predictions from supply and demand" )
}

\keyword{models}

\eof

% 	$Id: hausman.systemfit.Rd,v 1.3 2003/11/10 20:19:04 hamannj Exp $	

\name{hausman.systemfit}
\alias{hausman.systemfit}

\title{Hausman's Test}

\description{
  \code{hausman} returns the Hausman's statistic for specification.

  \deqn{q'(V_1 - V_0)q}

  where $V_1$ and $V_0$ are the covb values from a \code{twostage} or
  \code{threestage} object and q is the difference in the b vectors from
  the \code{twostage} or \code{threestage} objects.

}

\usage{
hausman.systemfit( li.results, fi.results )
}

\arguments{
  \item{li.results}{the limited information results object
    (\code{twostage}) }
  \item{fi.results}{the full information results object
    (\code{threestage}) }
}

\value{
  \code{hausman.systemfit} returns the value of the test statistic.
}

\references{

  Greene, W. H. (1993)
  \emph{Econometric Analysis, Second Edition}, Macmillan.

  Hausman, J. A. (1978)
  Specification Tests in Econometrics. \emph{Economtrica}. 46:1251-1271.

  Kmenta, J. (1997)
  \emph{Elements of Econometrics, Second Edition}, University of
  Michigan Publishing

}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}}

\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
inst <- ~ d + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform the estimation and report the results for the whoel system
fit2sls <- systemfit( "2SLS", system, labels, inst)
fit3sls <- systemfit( "3SLS", system, labels, inst)

## perform the hausman test on the first equation
h <- hausman.systemfit( fit2sls, fit3sls )
pval <- pchisq( h, dim( fit3sls$bcov )[1] )

print( h )
print( pval )

}
\keyword{models}




\eof

% 	$Id: kmenta.Rd,v 1.2 2002/11/30 01:56:28 hamannj Exp $	

\keyword{datasets}
\name{kmenta}
\alias{kmenta}
\non_function{}
\title{Partly Artificial Data on the U. S. Economy}
\usage{data(kmenta)}
\description{
These are partly contrived data from Kmenta (1986), constructed
to illustrate estimation of a simultaneous-equation model.

The \code{kmenta} data frame has 20 rows and 5 columns.
}
\format{
This data frame contains the following columns:
\describe{
\item{q}{
food consumption per capita.
}
\item{p}{
ratio of food prices to general consumer prices.
}
\item{d}{
disposable income in constant dollars.
}
\item{f}{
ratio of preceding year's prices received by farmers
to general consumer prices.
}
\item{a}{
time in years.
}
}
}

\details{
 The exogenous variables \code{d}, \code{f}, and \code{a} are based on
 real data; the endogenous variables \code{p} and \code{q} were generated
 by simulation.
}
\source{
 Kmenta, J. (1986)
 \emph{Elements of Econometrics, Second Edition}, Macmillan.
}
\examples{
data(kmenta)
}
\keyword{datasets}



\eof

% 	$Id: lrtest.systemfit.Rd,v 1.2 2003/08/12 20:27:48 hamannj Exp $	

\name{lrtest.systemfit}
\alias{lrtest.systemfit}

\title{Likelihood Ratio test for Equation Systems}

\description{
  Perform a Likelihood Ratio test to test linear restrictions
  when estimating an equation system.
}

\usage{
lrtest.systemfit( resultc, resultu )
}

\arguments{
  \item{resultc}{an object of type systemfit.system that
    contains the results of the restricted estimation.}
  \item{resultu}{an object of type systemfit.system that
    contains the results of the unconstrained estimation.}
}

\details{
  At the moment only the LR test for SUR models is implemented.

  This test calculates the likelihood ratio value by calculating
  the estimated residual covariance matrix \eqn{\hat{\Sigma}} by
  the maximum likelihood formula
  \eqn{\hat{\sigma}_{ij} = (\hat{e}_i' \hat{e}_j) / T}
  (no matter which formula is used in the estimation).
}

\value{
  \code{lrtest.sur.systemfit} returns a three values:

  \item{lr}{the empirical likelihood ratio value.}
  \item{p}{the p-value of a \eqn{\chi^2}-test.}
  \item{df}{the degrees of freedom of the test (= number of restrictions).}
}

\references{

  Greene, W. H. (2002)
  \emph{Econometric Analysis, Fifth Edition}, Prentice Hall.
}

\author{Arne Henningsen (\url{http://www.arne-henningsen.de})}

\seealso{\code{\link{systemfit}}}

\examples{
\dontrun{library( systemfit )}
data( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## unconstrained SUR estimation
fitsur <- systemfit("SUR", system, labels, data=kmenta )
print( fitsur )

## SUR estimation with 2 restrictions
Rrestr <- matrix(0,2,7)
qrestr <- matrix(0,2,1)
Rrestr[1,3] <-  1
Rrestr[1,7] <- -1
Rrestr[2,2] <- -1
Rrestr[2,5] <-  1
qrestr[2,1] <-  0.5
fitsur2 <- systemfit("SUR", system, labels, data=kmenta,
                      R.restr=Rrestr, q.restr=qrestr )
print( fitsur2 )

## perform LR-test
lrtest.systemfit( fitsur2, fitsur )
}

\keyword{models}




\eof

% 	$Id: nlsystemfit.Rd,v 1.2 2004/02/04 02:34:27 hamannj Exp $	


\name{nlsystemfit}
\alias{nlsystemfit}

\title{Nonlinear Equation System Estimation}

\description{
  Fits a set of structural nonlinear equations using Ordinary Least
  Squares (OLS), Seemingly Unrelated Regression (SUR),
  Two-Stage Least Squares (2SLS), Three-Stage Least Squares (3SLS).
}


\usage{ nlsystemfit( method="OLS", eqns, startvals,
                     eqnlabels=c(as.character(1:length(eqns))), inst=NULL,
                     data=list(), solvtol=.Machine$double.eps, pl=0,
                     maxiter=1000 )
}
		   

\arguments{
  \item{method}{the estimation method, one of "OLS", "SUR", "2SLS",
    "3SLS".}
  \item{eqns}{a list of structural equations to be estimated.}
  \item{startvals}{a list of starting parameter values for the minimization.}
  \item{eqnlabels}{an optional list of character vectors of names
    for the equation labels.}
  \item{inst}{one-sided model formula specifying instrumental variables
    or a list of one-sided model formulas if different instruments should
    be used for the different equations (only needed for 2SLS, 3SLS and
    GMM estimations).}
  \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment from which
    nlsystemfit is called.}
  \item{pl}{print.level argument that is passed to the \code{\link{nlm}} function. }
  \item{solvtol}{tolerance level used in the step tolerance and gradient
    tolerances arguments for \code{\link{nlm}} and the tolerance for
    detecting linear dependencies in the columns of X in the
    \code{\link{qr}} function calls.}
  \item{maxiter}{the maximum number of iterations for the \code{\link{nlm}} function.}
}

\details{

  The nlsystemfit function relies on \code{\link{nlm}} to perform the
  minimization of the objective functions and the \code{\link{qr}} set
  of functions.
  
  A system of nonlinear equations can be written as:
  
  \deqn{\epsilon_{t} = q( y_t, x_t, \theta )}
  \deqn{z_{t} = Z( x_t )}
  
  where \eqn{\epsilon_{t}} are the residuals from the y observations and
  the function evaluated at the parameter estimates. 
  
  The objective functions for the methods are:
  
  \tabular{lccc}{
    %    \hline
    Method \tab Instruments \tab Objective Function \tab Covariance of
    \eqn{\theta}\cr %\hline  
    OLS \tab no \tab \eqn{r'r} \tab \eqn{(X(diag(S)^{-1}\bigotimes
      I)X)^{-1}}\cr %\hline
    SUR \tab no \tab \eqn{r'(diag(S)_{OLS}^{-1}\bigotimes I)r} \tab
    \eqn{(X(S^{-1}\bigotimes I)X)^{-1}}\cr %\hline
    2SLS \tab yes \tab \eqn{r'(I \bigotimes W)r} \tab
    \eqn{(X(diag(S)^{-1}\bigotimes I)X)^{-1}}\cr %\hline 
    3SLS \tab yes \tab \eqn{r'(S_{2SLS}^{-1} \bigotimes W)r} \tab
    \eqn{(X(diag(S)^{-1}\bigotimes W)X)^{-1}} %\hline
  }
  
  where, r is a column vector for the residuals for each equation, S is
  variance-covariance matrix between the equations
  (\eqn{\hat{\sigma}_{ij} = (\hat{e}_i' \hat{e}_j) /
    \sqrt{(T - k_i)*(T - k_j)}}), X is matrix of the
  partial derivates with respect to the parameters, W is a matrix of the
  instrument variables \eqn{Z(Z'Z)^{-1}Z}, Z is a matrix of the
  instrument variables, and I is an nxn identity matrix.
  
  The SUR and 3SLS methods requires two solutions. The first solution
  for the SUR is an OLS solution to obtain the variance-covariance
  matrix. The 3SLS uses the variance-covatiance from a 2SLS solution,
  then fits all the equations simultaneously.  
  
  The user should be aware that the function is \bold{VERY} sensative to
  the starting values and the nlm function may not converge. The nlm
  function will be called with the \code{typsize} argument set the
  absolute values of the starting values for the OLS and 2SLS
  methods. For the SUR and 3SLS methods, the \code{typsize} argument is
  set to the absolute values of the resulting OLS and 2SLS parameter
  estimates from the nlm result structre. In addition, the starting
  values for the SUR and 3SLS methods are obtained from the OLS and 2SLS
  parameter estimates to shorten the number of iterations. The number of
  iterations reported in the summary are only those used in the last
  call to nlm, thus the number of iterations in the OLS portion of the
  SUR fit and the 2SLS portion of the 3SLS fit are not included. 
}

\value{
  \code{nlsystemfit} returns a list of the class \code{nlsystemfit.system} and
  contains all results that belong to the whole system.
  This list contains one special object: "eq". It is a list and contains
  one object for each estimated equation. These objects are of the class
  \code{nlsystemfit.equation} and contain the results that belong only to the
  regarding equation.

  The objects of the class \code{nlsystemfit.system} and
  \code{nlsystemfit.equation} have the following components (the elements of
  the latter are marked with an asterisk (\eqn{*})):

  \item{eq}{a list object that contains a list object for each equation.}
  \item{method}{estimation method.}
  \item{resids}{an \eqn{n \times g} matrix of the residuals.}
  \item{g}{number of equations.}
  \item{n}{total number of observations.}
  \item{k}{total number of coefficients.}
  \item{b}{vector of all estimated coefficients.}
  \item{se}{estimated standard errors of \code{b}.}
  \item{t}{t values for \code{b}.}
  \item{p}{p values for \code{b}.}
  \item{bcov}{estimated covariance matrix of \code{b}.}
  \item{rcov}{estimated residual covariance matrix.}
  \item{drcov}{determinant of \code{rcov}.}
  \item{rcovest}{residual covariance matrix used for estimation (only SUR and 3SLS).}
  \item{rcor}{estimated residual correlation matrix.}
  \item{nlmest}{results from the nlm function call} 
  \item{solvetol}{tolerance level when inverting a matrix or calculating a determinant.}

  ## elements of the class nlsystemfit.eq
  \item{eq}{a list that contains the results that belong to the individual equations.}
  \item{eqnlabel*}{the equation label of the ith equation (from the labels list).}
  \item{formula*}{model formula of the ith equation.}
  \item{n*}{number of observations of the ith equation.}
  \item{k*}{number of coefficients/regressors in the ith equation.}
  \item{df*}{degrees of freedom of the ith equation.}
  \item{b*}{estimated coefficients of the ith equation.}
  \item{se*}{estimated standard errors of \code{b}.}
  \item{t*}{t values for \code{b}.}
  \item{p*}{p values for \code{b}.}
  \item{covb*}{estimated covariance matrix of \code{b}.}
  \item{predicted*}{vector of predicted values of the ith equation.}
  \item{residuals*}{vector of residuals of the ith equation.}
  \item{ssr*}{sum of squared residuals of the ith equation.}
  \item{mse*}{estimated variance of the residuals (mean of squared errors) of the ith equation.}
  \item{s2*}{estimated variance of the residuals (\eqn{\hat{\sigma}^2}) of the ith equation.}
  \item{rmse*}{estimated standard error of the residulas (square root of mse) of the ith equation.}
  \item{s*}{estimated standard error of the residuals (\eqn{\hat{\sigma}}) of the ith equation.}
  \item{r2*}{R-squared (coefficient of determination).}
  \item{adjr2*}{adjusted R-squared value.}
}

\references{

  Gallant, R. H. (1987)
  \emph{Nonlinear Equation Estimation}, John Wiley and Sons, 610 pp.

  SAS Institute (1999)
  \emph{SAS/ETS User's Guide, Version 8}, Cary NC: SAS Institute 1546 pp.

}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com} }

\seealso{\code{\link{systemfit}}, \code{\link{nlm}}, and \code{\link{qr}}}

\examples{
library( systemfit )
data( ppine )

hg.formula <- hg ~ exp( h0 + h1*log(tht) + h2*tht^2 + h3*elev + h4*cr)
dg.formula <- dg ~ exp( d0 + d1*log(dbh) + d2*hg + d3*cr + d4*ba  )
labels <- list( "height.growth", "diameter.growth" )
inst <- ~ tht + dbh + elev + cr + ba
start.values <- c(h0=-0.5, h1=0.5, h2=-0.001, h3=0.0001, h4=0.08,
                  d0=-0.5, d1=0.009, d2=0.25, d3=0.005, d4=-0.02 )
model <- list( hg.formula, dg.formula )

model.ols <- nlsystemfit( "OLS", model, start.values, data=ppine, eqnlabels=labels )
print( model.ols )

model.3sls <- nlsystemfit( "3SLS", model, start.values, data=ppine,
                                    eqnlabels=labels, inst=inst )
print( model.3sls )
}

\keyword{models}
\keyword{regression}
\keyword{nonlinear}




\eof

% 	$Id: ppine.Rd,v 1.1 2004/02/04 02:26:04 hamannj Exp $	


\keyword{datasets}
\name{ppine}
\alias{ppine}
\non_function{}
\title{Tree Growth Data for Ponderosa Pine}
\usage{data(ppine)}
\description{
A subset of tree growth observations from a Ponderosa pine growth database.

The \code{ppine} data frame has 166 rows and 8 columns.
}

\format{
  This data frame contains the following columns:
  \describe{
    \item{elev}{
      Altitude of the plot, in feet above mean sea level.
    }
    \item{smi}{
      Summer moisture index is the ratio of growing season heating
      degree days to growing season precipitation.
    }
    \item{dbh}{
      Diameter of the tree at breast height (4.5 feet).
    }
    \item{tht}{
      Total stem height for the tree.
    }
    \item{cr}{
      Crown ratio code. The scale is from 1 to 9 where a crown class of
      one represents a crown ratio between 0 and 15 percent. A crown
      ratio code of 2 represents a crown ratio value between 16 and
      25\%,...,8=76-85\%, 9 >=85\%.
    }
    \item{ba}{
      Plot basal area at the beginning of the growth period.
    }
    \item{dg}{
      Five-year diameter increment.
    }
    \item{hg}{
      Five-year height increment.
    }
  }
}

\details{
 The exogenous variables are \code{elev}, \code{smi}, \code{dbh},
 \code{tht}, \code{cr}, and \code{ba}; the endogenous variables
 \code{dg} and \code{hg}. There are no lagged  variables in the dataset
 and the observations are for a single remeasurement.

 The data was provided by the USDA Forest Service Intermountain
 Research Station.

 
%  The data was provided by the USDA Forest Service Intermountain
%  Research Station from installations on USDA Forest Service, Spokane Indian
%  Reservation and the Nez Perce Indian Reservation lands.
}

\source{
  William R. Wykoff \email{wwykoff@fs.fed.us} \emph{Rocky Mountain Research Station, 1221 South Main Street, Moscow, ID 83843}
}


\examples{
data(ppine)
}

\keyword{datasets}



\eof

% 	$Id: prediction.systemfit.Rd,v 1.3 2003/11/18 08:54:16 henningsena Exp $	

\name{prediction.systemfit}
\alias{prediction.systemfit}

\title{Predictions from Equation System Estimation}

\description{
   Returns the predicted values, their standard errors and the
   confidence limits of prediction.
}

\usage{
prediction.systemfit( object, data=object$data, alpha=0.05 )
}

\arguments{
  \item{object}{an object of type systemfit.system.}
  \item{data}{a dataframe containing the exogenous variables.}
  \item{alpha}{level of significance.}
}

\value{
  \code{prediction.systemfit} returns a list of objects for each
  equation that contains the following components:

  \item{predicted}{predicted values.}
  \item{se.prediction}{standard error of the predicted values.}
  \item{prediction.limits}{lower and upper (1-alpha) confidence limits of prediction.}
}

\references{

  Greene, W. H. (1993)
  \emph{Econometric Analysis, Second Edition}, Macmillan.

  Kmenta, J. (1997)
  \emph{Elements of Econometrics, Second Edition}, University of
  Michigan Publishing.
}

\author{Arne Henningsen (\url{http://www.arne-henningsen.de})}


}

\seealso{\code{\link{systemfit}}}

\examples{
\dontrun{library( systemfit )}
data( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## OLS estimation
fitols <- systemfit("OLS", system, labels, data=kmenta )
print( fitols )

## calculate predicted values and limits
pred <- prediction.systemfit( fitols )
print( pred[[2]]$prediction.limits )
}

\keyword{models}




\eof

% 	$Id: print.nlsystemfit.equation.Rd,v 1.3 2004/02/04 02:34:27 hamannj Exp $	

\name{print.nlsystemfit.equation}
\alias{print.nlsystemfit.equation}
\title{print.nlsystemfit.equation}

\description{
  This function prints a summary of one equation of the estimated equation system.
}

\usage{
  print.nlsystemfit.equation(x, digits=6,...)
}

\arguments{
  \item{x}{an object of type \code{nlsystemfit.equation}.}
  \item{digits}{number of digits to print.}
  \item{...}{not used by user.}
}

\value{
  \code{print.nlsystemfit.equation} returns nothing.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{nlsystemfit}}, \code{\link{print.nlsystemfit.system}} and
  \code{\link{summary.nlsystemfit.equation}}}


\examples{
library( systemfit )
data( ppine )

hg.formula <- hg ~ exp( h0 + h1*log(tht) + h2*tht^2 + h3*elev + h4*cr)
dg.formula <- dg ~ exp( d0 + d1*log(dbh) + d2*hg + d3*cr + d4*ba  )
labels <- list( "height.growth", "diameter.growth" )
inst <- ~ tht + dbh + elev + cr + ba
start.values <- c(h0=-0.5, h1=0.5, h2=-0.001, h3=0.0001, h4=0.08,
                  d0=-0.5, d1=0.009, d2=0.25, d3=0.005, d4=-0.02 )
model <- list( hg.formula, dg.formula )

model.ols <- nlsystemfit( "OLS", model, start.values, data=ppine, eqnlabels=labels )
print( model.ols )

model.3sls <- nlsystemfit( "3SLS", model, start.values, data=ppine,
                                    eqnlabels=labels, inst=inst )
print( model.3sls )
}

\keyword{models}
\keyword{regression}
\keyword{nonlinear}

\eof

% 	$Id: print.nlsystemfit.system.Rd,v 1.2 2004/02/04 02:34:27 hamannj Exp $	

\name{print.nlsystemfit.system}
\alias{print.nlsystemfit.system}
\title{print.nlsystemfit.system}

\description{
  This function prints a summary of the estimated equation system.
}

\usage{
print.nlsystemfit.system(x, digits=6,...)
}

\arguments{
  \item{x}{an object of type \code{nlsystemfit.system}.}
  \item{digits}{number of digits to print.}
  \item{...}{not used by user.}
}

\value{
  \code{print.nlsystemfit.system} returns nothing.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{nlsystemfit}},
  \code{\link{print.nlsystemfit.equation}},
  \code{\link{summary.nlsystemfit.system}} and
  }

\examples{
library( systemfit )
data( ppine )

hg.formula <- hg ~ exp( h0 + h1*log(tht) + h2*tht^2 + h3*elev + h4*cr)
dg.formula <- dg ~ exp( d0 + d1*log(dbh) + d2*hg + d3*cr + d4*ba  )
labels <- list( "height.growth", "diameter.growth" )
inst <- ~ tht + dbh + elev + cr + ba
start.values <- c(h0=-0.5, h1=0.5, h2=-0.001, h3=0.0001, h4=0.08,
                  d0=-0.5, d1=0.009, d2=0.25, d3=0.005, d4=-0.02 )
model <- list( hg.formula, dg.formula )

model.ols <- nlsystemfit( "OLS", model, start.values, data=ppine, eqnlabels=labels )
print( model.ols )

model.3sls <- nlsystemfit( "3SLS", model, start.values, data=ppine,
                                    eqnlabels=labels, inst=inst )
print( model.3sls )
}

\keyword{models}
\keyword{regression}
\keyword{nonlinear}


\eof

% 	$Id: print.systemfit.equation.Rd,v 1.3 2003/11/10 20:19:04 hamannj Exp $	

\name{print.systemfit.equation}
\alias{print.systemfit.equation}
\title{print.systemfit.equation}

\description{
  This function prints a summary of one equation of the estimated equation system.
}

\usage{
  print.systemfit.equation(x, digits=6,...)
}

\arguments{
  \item{x}{an object of type \code{systemfit.equation}.}
  \item{digits}{number of digits to print.}
  \item{...}{not used by user.}
}

\value{
  \code{print.systemfit.equation} returns nothing.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}, \code{\link{print.systemfit.system}} and
  \code{\link{summary.systemfit.equation}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform OLS on each of the equations in the system
fitols <- systemfit( "OLS", system, labels )

## print the results
print( fitols$eq[[1]] )
}

\keyword{models}

\eof

% 	$Id: print.systemfit.system.Rd,v 1.6 2003/11/10 20:19:05 hamannj Exp $	

\name{print.systemfit.system}
\alias{print.systemfit.system}
\title{print.systemfit.system}

\description{
  This function prints a summary of the estimated equation system.
}

\usage{
print.systemfit.system(x, digits=6,...)
}

\arguments{
  \item{x}{an object of type \code{systemfit.system}.}
  \item{digits}{number of digits to print.}
  \item{...}{not used by user.}
}

\value{
  \code{print.systemfit.system} returns nothing.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}, \code{\link{print.systemfit.equation}}
  and \code{\link{summary.systemfit.system}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform OLS on each of the equations in the system
fitols <- systemfit( "OLS", system, labels )

## print the results
print( fitols )


}
\keyword{models}




\eof

% 	$Id: se.ratio.systemfit.Rd,v 1.3 2003/11/10 20:19:05 hamannj Exp $	


\name{se.ratio.systemfit}
\alias{se.ratio.systemfit}

\title{Ratio of the Standard Errors}

\description{
  \code{se.ratio.systemfit} returns a vector of the ratios of the
  standard errors of the predictions for two equations.
}

\usage{
se.ratio.systemfit( resultsi, resultsj, eqni )
}

\arguments{
  \item{resultsi}{an object of type systemfit.system.}
  \item{resultsj}{an object of type systemfit.system.}
  \item{eqni}{index for equation to obtain the ratio of standard errors.}
}

\value{
  \code{se.ratio} returns a vector of the standard errors of the ratios
  for the predictions between the predicted values in equation i and
  equation j.

}

\references{    

  Hasenauer, H; Monserud, R and T. Gregoire. (1998)
  Using Simultaneous Regression Techniques with Individual-Tree Growth Models.
  \emph{Forest Science}. 44(1):87-95

}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}} and \code{\link{correlation.systemfit}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
inst <- ~ d + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform 2SLS on each of the equations in the system
fit2sls <- systemfit( "2SLS", system, labels, inst )
fit3sls <- systemfit( "3SLS", system, labels, inst )

## print the results from the fits
print( fit2sls )
print( fit3sls )
print( "covariance of residuals used for estimation (from 2sls)" )
print( fit3sls$rcovest )
print( "covariance of residuals" )
print( fit3sls$rcov )

## examine the improvement of 3SLS over 2SLS by computing
## the ratio of the standard errors of the estimates
improve.ratio <- se.ratio.systemfit( fit2sls, fit3sls, 2 )
print( "summary values for the ratio in the std. err. for the predictions" )
print( summary( improve.ratio ) )
}
\keyword{models}




\eof

% 	$Id: summary.nlsystemfit.equation.Rd,v 1.2 2004/02/04 02:34:27 hamannj Exp $	

\name{summary.nlsystemfit.equation}
\alias{summary.nlsystemfit.equation}
\title{summary.nlsystemfit.equation}

\description{
  This function returns a summary of one equation of the estimated equation system.
}

\usage{
summary.nlsystemfit.equation(object,...)
}

\arguments{
  \item{object}{an object of type \code{nlsystemfit.equation}.}
  \item{...}{not used by user.}
}

\value{
  \code{summary.nlsystemfit.equation} returns an object of type \code{nlsystemfit.equation}.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{nlsystemfit}},
  \code{\link{summary.nlsystemfit.system}}, and 
  \code{\link{print.nlsystemfit.equation}} }

\examples{
library( systemfit )
data( ppine )

hg.formula <- hg ~ exp( h0 + h1*log(tht) + h2*tht^2 + h3*elev + h4*cr)
dg.formula <- dg ~ exp( d0 + d1*log(dbh) + d2*hg + d3*cr + d4*ba  )
labels <- list( "height.growth", "diameter.growth" )
inst <- ~ tht + dbh + elev + cr + ba
start.values <- c(h0=-0.5, h1=0.5, h2=-0.001, h3=0.0001, h4=0.08,
                  d0=-0.5, d1=0.009, d2=0.25, d3=0.005, d4=-0.02 )
model <- list( hg.formula, dg.formula )

model.ols <- nlsystemfit( "OLS", model, start.values, data=ppine, eqnlabels=labels )
print( model.ols )

model.3sls <- nlsystemfit( "3SLS", model, start.values, data=ppine,
                                    eqnlabels=labels, inst=inst )
print( model.3sls )
}



\keyword{models}
\keyword{regression}
\keyword{nonlinear}

\eof

% 	$Id: summary.nlsystemfit.system.Rd,v 1.2 2004/02/04 02:34:28 hamannj Exp $	

\name{summary.nlsystemfit.system}
\alias{summary.nlsystemfit.system}
\title{summary.nlsystemfit.system}

\description{
  This function returns a summary of the system of equations.
}

\usage{
summary.nlsystemfit.system(object,...)
}

\arguments{
  \item{object}{an object of type \code{nlsystemfit.system}.}
  \item{...}{not used by user.}
}

\value{
  \code{summary.nlsystemfit.system} returns an object of type \code{nlsystemfit.system}.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{nlsystemfit}}, \code{\link{print.nlsystemfit.system}} and
  \code{\link{print.nlsystemfit.system}}}

\examples{
library( systemfit )
data( ppine )

hg.formula <- hg ~ exp( h0 + h1*log(tht) + h2*tht^2 + h3*elev + h4*cr)
dg.formula <- dg ~ exp( d0 + d1*log(dbh) + d2*hg + d3*cr + d4*ba  )
labels <- list( "height.growth", "diameter.growth" )
inst <- ~ tht + dbh + elev + cr + ba
start.values <- c(h0=-0.5, h1=0.5, h2=-0.001, h3=0.0001, h4=0.08,
                  d0=-0.5, d1=0.009, d2=0.25, d3=0.005, d4=-0.02 )
model <- list( hg.formula, dg.formula )

model.ols <- nlsystemfit( "OLS", model, start.values, data=ppine, eqnlabels=labels )
print( model.ols )

model.3sls <- nlsystemfit( "3SLS", model, start.values, data=ppine,
                                    eqnlabels=labels, inst=inst )
print( model.3sls )
}

\keyword{models}
\keyword{regression}
\keyword{nonlinear}




\eof

% 	$Id: summary.systemfit.equation.Rd,v 1.3 2003/11/10 20:19:05 hamannj Exp $	

\name{summary.systemfit.equation}
\alias{summary.systemfit.equation}
\title{summary.systemfit.equation}

\description{
  This function returns a summary of one equation of the estimated equation system.
}

\usage{
summary.systemfit.equation(object,...)
}

\arguments{
  \item{object}{an object of type \code{systemfit.equation}.}
  \item{...}{not used by user.}
}

\value{
  \code{summary.systemfit.equation} returns an object of type \code{systemfit.equation}.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}, \code{\link{summary.systemfit.system}} and
  \code{\link{print.systemfit.equation}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
inst <- ~ d + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform OLS on each of the equations in the system
fitols <- systemfit( "OLS", system, labels )

## print the results
summary.systemfit.equation( fitols$eq[[1]] )
}

\keyword{models}

\eof
\name{summary.systemfit.system}
\alias{summary.systemfit.system}
\title{summary.systemfit.system}

\description{
  This function returns a summary of the system of equations.
}

\usage{
summary.systemfit.system(object,...)
}

\arguments{
  \item{object}{an object of type \code{systemfit.system}.}
  \item{...}{not used by user.}
}

\value{
  \code{summary.systemfit.system} returns an object of type \code{systemfit.system}.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}}

\seealso{\code{\link{systemfit}}, \code{\link{print.systemfit.system}} and
  \code{\link{print.systemfit.system}}}


\examples{
\dontrun{library( systemfit )}

data( kmenta )
attach( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
inst <- ~ d + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## perform OLS on each of the equations in the system
fitols <- systemfit( "OLS", system, labels )

## print the results
summary.systemfit.system( fitols )
}

\keyword{models}




\eof

% 	$Id: systemfit-internal.Rd,v 1.2 2004/02/04 02:34:28 hamannj Exp $	

\name{systemfit-internal}
\alias{knls}

\title{Internal systemfit functions}
\description{
  Internal systemfit functions
}
\usage{
knls( theta, eqns, data, fitmethod="OLS", parmnames, instr=NULL, S=NULL )
}

\details{
  These functions are no the called by the user.
  The knls is the driver function that is passed to the nlm call in
  \code{\link{nlsystemfit}}. 
}

%\seealso{\code{\link{nlm}},\code{\link{qr}}, and \code{\link{systemfit}} }


\keyword{internal}

\eof

% 	$Id: systemfit.Rd,v 1.7 2004/01/30 20:03:43 hamannj Exp $	


\name{systemfit}
\alias{systemfit}

\title{Equation System Estimation}

\description{
  Fits a set of linear structural equations using Ordinary Least
  Squares (OLS), Weighted Least Squares (WLS), Seemingly Unrelated Regression (SUR),
  Two-Stage Least Squares (2SLS), Weighted Two-Stage Least Squares (W2SLS)
  or Three-Stage Least Squares (3SLS).
}

\usage{
systemfit( method, eqns, eqnlabels=c(as.character(1:length(eqns))),
           inst=NULL, data=list(), R.restr=NULL,
           q.restr=matrix(0,max(nrow(R.restr),0),1),
           TX=NULL, maxiter=1, tol=1e-5,
           rcovformula=1, formula3sls="GLS",
           probdfsys=!(is.null(R.restr) & is.null(TX)),
           single.eq.sigma=(is.null(R.restr) & is.null(TX)),
           solvetol=.Machine$double.eps )
}

\arguments{
  \item{method}{the estimation method, one of "OLS", "WLS", "SUR",
    "2SLS", "W2SLS" or "3SLS".}
  \item{eqns}{a list of structural equations to be estimated; a regression
    constant is implied if not explicitly omitted.}
  \item{eqnlabels}{an optional list of character vectors of names
    for the equation labels.}
  \item{inst}{one-sided model formula specifying instrumental variables
    or a list of one-sided model formulas if different instruments should
    be used for the different equations (only needed for 2SLS, W2SLS and
    3SLS estimations).}
  \item{data}{an optional data frame containing the variables in the model.
    By default the variables are taken from the environment from which
    systemfit is called.}
  \item{R.restr}{an optional j x k matrix to impose linear
    restrictions on the parameters by \code{R.restr} * \eqn{b} = \code{q.restr}
    (j = number of restrictions, k = number of all parameters,
    \eqn{b} = vector of all parameters).}
  \item{q.restr}{an optional j x 1 matrix to impose linear
    restrictions (see \code{R.restr}); default is a j x 1 matrix
    that contains only zeros.}
  \item{TX}{an optional matrix to transform the regressor matrix and,
    hence, also the coefficient vector (see details).}
  \item{maxiter}{maximum number of iterations for WLS, SUR, W2SLS and
    3SLS estimations.}
  \item{tol}{tolerance level indicating when to stop the iteration (only
    WLS, SUR, W2SLS and 3SLS estimations).}
  \item{rcovformula}{formula to calculate the estimated residual covariance
    matrix (see details).}
  \item{formula3sls}{formula for calculating the 3SLS estimator,
    one of "GLS", "IV", "GMM", "Schmidt" or "EViews" (see details).}
  \item{probdfsys}{use the degrees of freedom of the whole system
    (in place of the degrees of freedom of the single equation)
    to calculate prob values for the t-test of individual parameters.}
  \item{single.eq.sigma}{use different \eqn{\sigma^2}s for each
    single equation to calculate the covariance matrix and the
    standard errors of the coefficients (only OLS and 2SLS).}
  \item{solvetol}{tolerance level for detecting linear dependencies
    when inverting a matrix or calculating a determinant (see
    \code{\link{solve}} and \code{\link{det}}).}
}

\details{
   The matrix \code{TX} transforms the regressor matrix (\eqn{X}) by
   \eqn{X^{*} = X *} \code{TX}. Thus, the vector of coefficients is now
   \eqn{b =} \code{TX} \eqn{\cdot b^{*}} , where \eqn{b} is the original (stacked) vector
   of all coefficients and \eqn{b^{*}} is the new coefficient vector that is
   estimated instead. Thus, the elements of vector \eqn{b} are
   \eqn{b_i = \sum_j TX_{ij} \cdot b^{*}_j} \cr
   The \code{TX} matrix can be used to change the order of the
   coefficients and also to restrict coefficients (if \code{TX} has less
   columns than it has rows). However restricting coefficients
   by the \code{TX} matrix is less powerfull and flexible than the
   restriction by providing the \code{R.restr} matrix and the
   \code{q.restr} vector. The advantage of restricting the coefficients
   by the \code{TX} matrix is that the matrix that is inverted for
   estimation gets smaller by this procedure, while it gets larger
   if the restrictions are imposed by \code{R.restr} and \code{q.restr}.

   If iterated (WLS, SUR, W2SLS or 3SLS estimation with \code{maxit}>1),
     the convergence criterion is \cr
   \eqn{\sqrt{\sum_i (b_{i,g} - b_{i,g-1})^2 \left/ \sum_i b_{i,g-1}^2 \right.}}
   < \code{tol} (\eqn{b_{i,g}} is the ith coefficient of the gth iteration step).

   The formula to calculate the estimated covariance matrix of the residuals
   (\eqn{\hat{\Sigma}}) can be one of the following (see Judge et al., 1985, p. 469): \cr
   if rcovformula=0: \eqn{\hat{\sigma}_{ij} = (\hat{e}_i' \hat{e}_j) / T}; \cr
   if rcovformula=1: \eqn{\hat{\sigma}_{ij} = (\hat{e}_i' \hat{e}_j) /
       \sqrt{(T - k_i)*(T - k_j)}}; \cr
   if rcovformula=2: \eqn{\hat{\sigma}_{ij} = (\hat{e}_i' \hat{e}_j) / (T - k_i - k_j
   + tr[(X_i'X_i)^{-1}X_i'X_j(X_j'X_j)^{-1}X_j'X_i]}. \cr
   If \eqn{k_i = k_j}, formula 1 and 2 are equal and yield an unbiased estimator for the
   residual covariance matrix.
   If \eqn{k_i \neq k_j}, only formula 2 yields an unbiased estimator for the residual
   covariance matrix, but it is not neccessarily positive semidefinit and
   its inverse is \bold{not} an unbiased estimator for the inverse of the residual
   covariance matrix. Thus, it is doubtful whether formula 2 is really superior
   to formula 1 (see Theil, 1971, p. 322).

   The formulas to calculate the 3SLS estimator lead to identical results
   if the same instruments are used in all equations. If different instruments
   are used in the different equations, only the GMM-3SLS estimator ("GMM")
   and the 3SLS estimator proposed by Schmidt (1990) ("Schmidt") are consistent,
   whereas "GMM" is efficient relative to "Schmidt" (see Schmidt, 1990).
}

\value{
  \code{systemfit} returns a list of the class \code{systemfit.system} and
  contains all results that belong to the whole system.
  This list contains one special object: "eq". It is a list and contains
  one object for each estimated equation. These objects are of the class
  \code{systemfit.equation} and contain the results that belong only to the
  regarding equation.

  The objects of the class \code{systemfit.system} and
  \code{systemfit.equation} have the following components (the elements of
  the latter are marked with an asterisk (\eqn{*})):

  \item{method}{estimation method.}
  \item{g}{number of equations.}
  \item{n}{total number of observations.}
  \item{k}{total number of coefficients.}
  \item{ki}{total number of linear independent coefficients.}
  \item{iter}{number of iteration steps.}
  \item{b}{vector of all estimated coefficients.}
  \item{bt}{coefficient vector transformed by \code{TX}.}
  \item{se}{estimated standard errors of \code{b}.}
  \item{t}{t values for \code{b}.}
  \item{p}{p values for \code{b}.}
  \item{bcov}{estimated covariance matrix of \code{b}.}
  \item{btcov}{covariance matrix of \code{bt}.}
  \item{rcov}{estimated residual covariance matrix.}
  \item{drcov}{determinant of \code{rcov}.}
  \item{rcovest}{residual covariance matrix used for estimation (only SUR and 3SLS).}
  \item{rcor}{estimated residual correlation matrix.}
  \item{olsr2}{System OLS R-squared value.}
  \item{mcelr2}{McElroys R-squared value for the system (only SUR and 3SLS).}
  \item{y}{vector of all (stacked) endogenous variables}
  \item{x}{matrix of all (diagonally stacked) regressors}
  \item{h}{matrix of all (diagonally stacked) instrumental variables (only 2SLS and 3SLS)}
  \item{data}{data frame of the whole system (including instruments)}
  \item{R.restr}{the restriction matrix.}
  \item{q.restr}{the restriction vector.}
  \item{TX}{matrix used to transform the regressor matrix.}
  \item{maxiter}{maximum number of iterations.}
  \item{tol}{tolerance level indicating when to stop the iteration}
  \item{rcovformula}{formula to calculate the estimated residual covariance
    matrix}
  \item{formula3sls}{formula for calculating the 3SLS estimator.}
  \item{probdfsys}{system degrees of freedom to calculate prob values?.}
  \item{single.eq.sigma}{different \eqn{\sigma^2}s for each single equation?.}
  \item{solvetol}{tolerance level when inverting a matrix or calculating a determinant.}

  ## elements of the class systemfit.eq
  \item{eq}{a list that contains the results that belong to the individual equations.}
  \item{eqnlabel*}{the equation label of the ith equation (from the labels list).}
  \item{formula*}{model formula of the ith equation.}
  \item{inst*}{instruments of the ith equation (only 2SLS and 3SLS).}
  \item{n*}{number of observations of the ith equation.}
  \item{k*}{number of coefficients/regressors in the ith equation (including the constant).}
  \item{ki*}{number of linear independent coefficients in the ith equation (including the constant
              differs from \code{k} only if there are restrictions that are not cross-equation).}
  \item{df*}{degrees of freedom of the ith equation.}

  \item{b*}{estimated coefficients of the ith equation.}
  \item{se*}{estimated standard errors of \code{b}.}
  \item{t*}{t values for \code{b}.}
  \item{p*}{p values for \code{b}.}
  \item{covb*}{estimated covariance matrix of \code{b}.}

  \item{y*}{vector of endogenous variable (response values) of the ith equation.}
  \item{x*}{matrix of regressors (model matrix) of the ith equation.}
  \item{h*}{matrix of instrumental variables of the ith equation (only 2SLS and 3SLS).}
  \item{data*}{data frame (including instruments) of the ith equation.}
  \item{predicted*}{vector of predicted values of the ith equation.}
  \item{residuals*}{vector of residuals of the ith equation.}

  \item{ssr*}{sum of squared residuals of the ith equation.}
  \item{mse*}{estimated variance of the residuals (mean of squared errors) of the ith equation.}
  \item{s2*}{estimated variance of the residuals (\eqn{\hat{\sigma}^2}) of the ith equation.}
  \item{rmse*}{estimated standard error of the residulas (square root of mse) of the ith equation.}
  \item{s*}{estimated standard error of the residuals (\eqn{\hat{\sigma}}) of the ith equation.}

  \item{r2*}{R-squared (coefficient of determination).}
  \item{adjr2*}{adjusted R-squared value.}
}

\references{

  Greene, W. H. (1993)
  \emph{Econometric Analysis, Second Edition}, Macmillan.

  Greene, W. H. (2002)
  \emph{Econometric Analysis, Fifth Edition}, Prentice Hall.

  Judge, George G.; W. E. Griffiths; R. Carter Hill; Helmut Ltkepohl and Tsoung-Chao Lee (1985)
  \emph{The Theory and Practice of Econometrics, Second Edition}, Wiley.

  Kmenta, J. (1997)
  \emph{Elements of Econometrics, Second Edition}, University of
  Michigan Publishing.

  Schmidt, P. (1990)
  \emph{Three-Stage Least Squares with different Instruments for different equations},
  Journal of Econometrics 43, p. 389-394.

  Theil, H. (1971)
  \emph{Principles of Econometrics}, Wiley, New York.
}

\author{Jeff D. Hamann \email{jeff.hamann@forestinformatics.com}
  Arne Henningsen (\url{http://www.arne-henningsen.de}) }
}

\seealso{\code{\link{lm}} and \code{\link{nlsystemfit}}}

\examples{
\dontrun{library( systemfit )}
data( kmenta )
demand <- q ~ p + d
supply <- q ~ p + f + a
labels <- list( "demand", "supply" )
system <- list( demand, supply )

## OLS estimation
fitols <- systemfit("OLS", system, labels, data=kmenta )
print( fitols )

## OLS estimation with 2 restrictions
Rrestr <- matrix(0,2,7)
qrestr <- matrix(0,2,1)
Rrestr[1,3] <-  1
Rrestr[1,7] <- -1
Rrestr[2,2] <- -1
Rrestr[2,5] <-  1
qrestr[2,1] <-  0.5
fitols2 <- systemfit("OLS", system, labels, data=kmenta,
                      R.restr=Rrestr, q.restr=qrestr )
print( fitols2 )

## iterated SUR estimation
fitsur <- systemfit("SUR", system, labels, data=kmenta, maxit=100 )
print( fitsur )

## 2SLS estimation
inst <- ~ d + f + a
fit2sls <- systemfit( "2SLS", system, labels, inst, kmenta )
print( fit2sls )

## 2SLS estimation with different instruments in each equation
inst1 <- ~ d + f
inst2 <- ~ d + f + a
instlist <- list( inst1, inst2 )
fit2sls2 <- systemfit( "2SLS", system, labels, instlist, kmenta )
print( fit2sls2 )

## 3SLS estimation with GMM-3SLS formula
inst <- ~ d + f + a
fit3sls <- systemfit( "3SLS", system, labels, inst, kmenta, formula3sls="GMM" )
print( fit3sls )

}

\keyword{models}
\keyword{regression}




\eof
