\name{ablrtest}
\alias{ablrtest}
\title{Likelihood ratio test for restrictions on alpha and beta}
\description{
  This function estimates a restricted VAR, where the restrictions are
  base upon \eqn{\bold{\alpha}}, \emph{i.e.} the loading vectors and
  \eqn{\bold{\beta}}, \emph{i.e} the matrix of cointegration vectors. The test
  statistic is distributed as \eqn{\chi^2} with \eqn{(p-m)r + (p-s)r} degrees of
  freedom, with \eqn{m} equal to the columns of the restricting matrix
  \eqn{\bold{A}}, \eqn{s} equal to the columns of the restricting matrix
  \eqn{\bold{H}} and \eqn{p} the order of the VAR.
}
\usage{
ablrtest(z, H, A, r)
}
\arguments{
  \item{z}{An object of class `ca.jo'.}
  \item{H}{The \eqn{(p \times s)} matrix containing the restrictions on
    \eqn{\bold{\beta}}.}
  \item{A}{The \eqn{(p \times m)} matrix containing the restrictions on
    \eqn{\bold{\alpha}}.}
  \item{r}{The count of cointegrating relationships; \cr
    inferred from \code{summary(`ca.jo-object')}.}
}
\details{
  The restricted \eqn{\bold{\alpha}} matrix, as well as \eqn{\bold{\beta}} is
  normalised with respect to the first variable.
}
\value{
  An object of class \code{`cajo.test'}.
}
\references{

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}}, \code{\link{alrtest}}, \code{\link{blrtest}},
  \code{\link{cajo.test-class}}, \code{\link{ca.jo-class}} and
  \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
HD1 <- matrix(c(1, -1, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 1), c(5,3))
DA <- matrix(c(1,0,0,0, 0, 1, 0, 0, 0, 0, 0, 1), c(4,3))
summary(ablrtest(sjd.vecm, H=HD1, A=DA, r=1))
}
\keyword{regression}

\eof
\name{alphaols}
\alias{alphaols}
\title{OLS regression of VECM weighting matrix}
\description{
  This functions estimates the \eqn{\bold{\alpha}} matrix of a VECM.
  The following OLS regression of the R-form of the VECM is hereby
  utilised:
  \deqn{\bold{R}_{0t} = \bold{\alpha}\bold{\beta}\prime \bold{R}_{kt} +
    \bold{\varepsilon}_t \qquad t=1, \dots, T}
}
\usage{
alphaols(z, reg.number = NULL)
}
\arguments{
  \item{z}{An object of class `ca.jo'.}
  \item{reg.number}{The number of the equation in the R-form that should
    be estimated or if set to \code{NULL} (the default), all equations
    within the R-form are estimated.}
}
\details{
  The cointegrating relations, \emph{i.e.} \eqn{\bold{R}_{kt}\prime
    \bold{\beta}} are calculated by using \code{z@RK} and \code{z@V}.
}
\value{
  Returns an object of class \code{`lm'}.
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}}, \code{\link[base]{lm}}, \code{\link{ca.jo-class}}
  and \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm1 <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
summary(alphaols(sjd.vecm1))
summary(alphaols(sjd.vecm1, reg.number=1))
}
\keyword{regression}

\eof
\name{alrtest}
\alias{alrtest}
\title{Likelihood ratio test for restrictions on alpha}
\description{
  This function estimates a restricted VAR, where the restrictions are
  base upon \eqn{\bold{\alpha}}, \emph{i.e.} the loading vectors. The test
  statistic is distributed as \eqn{\chi^2} with \eqn{r(p-m)} degrees of
  freedom, with \eqn{m} equal to the columns of the restricting matrix
  \eqn{\bold{A}}.
}
\usage{
alrtest(z, A, r)
}
\arguments{
  \item{z}{An object of class `ca.jo'.}
  \item{A}{The \eqn{(p \times m)} matrix containing the restrictions on
    \eqn{\bold{\alpha}}.}
  \item{r}{The count of cointegration relationships; \cr
    inferred from \code{summary(`ca.jo-object')}.}
}
\details{
  The orthogonal matrix to \eqn{\bold{A}} can be accessed as
  \code{object@B}. The restricted \eqn{\bold{\alpha}} matrix is
  normalised with respect to the first variable.
}
\value{
  An object of class \code{`cajo.test'}.
}
\references{

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.
}   
\seealso{
  \code{\link{ca.jo}}, \code{\link{blrtest}}, \code{\link{ablrtest}},
  \code{\link{cajo.test-class}}, \code{\link{ca.jo-class}} and
  \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
DA <- matrix(c(1,0,0,0), c(4,1))
summary(alrtest(sjd.vecm, A=DA, r=1))
}
\keyword{regression}

\eof
\name{blrtest}
\alias{blrtest}
\title{Likelihood ratio test for restrictions on beta}
\description{
  This function estimates a restricted VAR, where the restrictions are
  base upon \eqn{\bold{\beta}}, \emph{i.e.} the cointegration vectors. The test
  statistic is distributed as \eqn{\chi^2} with \eqn{r(p-s)} degrees of
  freedom, with \eqn{s} equal to the columns of the restricting matrix
  \eqn{\bold{H}}.
}
\usage{
blrtest(z, H, r)
}
\arguments{
  \item{z}{An object of class `ca.jo'.}
  \item{H}{The \eqn{(p \times s)} matrix containing the restrictions on
    \eqn{\bold{\beta}}.}
  \item{r}{The count of cointegrating relationships; \cr
    inferred from \code{summary(`ca.jo-object')}.}
}
\details{
  Please note, that in the case of nested hypothesis, the reported
  p-value should be adjusted to \eqn{r(s1-s2)} (see Johansen, S. and
  K. Juselius (1990)).
}
\value{
  An object of class \code{`cajo.test'}.
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}}, \code{\link{alrtest}}, \code{\link{ablrtest}},
  \code{\link{cajo.test-class}}, \code{\link{ca.jo-class}} and
  \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
HD0 <- matrix(c(-1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1), c(5,4))
summary(blrtest(sjd.vecm, H=HD0, r=1))
}
\keyword{regression}

\eof
\name{ca.jo-class}
\docType{class}
\alias{ca.jo-class}

\title{Representation of class `ca.jo'}
\description{
  This class contains the relevant information by applying the Johansen
  procedure to a matrix of time series data.
}
\section{Slots}{
  \describe{
    \item{\code{x}:}{Object of class \code{"ANY"}: A data matrix, or an
      object that can be coerced to it.}
    \item{\code{Z0}:}{Object of class \code{"matrix"}: The matrix of the
      differenced series.}
    \item{\code{Z1}:}{Object of class \code{"matrix"}: The regressor
      matrix, except for the lagged variables in levels.}
    \item{\code{ZK}:}{Object of class \code{"matrix"}: The matrix of the
      lagged variables in levels.}
    \item{\code{type}:}{Object of class \code{"character"}: The type of the
      test, either \code{"trace"} or \code{"eigen"}.}
    \item{\code{model}:}{Object of class \code{"character"}: The model
      description in prose, with respect to the inclusion of a linear
      trend.}
    \item{\code{const}:}{Object of class \code{"logic"}: If TRUE, no
      linear trend is included and \emph{vice versa}.}
    \item{\code{lag}:}{Object of class \code{"integer"}: The lag order
      for the variables in levels.}
    \item{\code{P}:}{Object of class \code{"integer"}: The count of
      variables.}
    \item{\code{season}:}{Object of class \code{"ANY"}: The frequency of
      the data, if seasonal dummies should be included, otherwise NULL.}
    \item{\code{dumvar}:}{Object of class \code{"ANY"}: A matrix
      containing dummy variables. The row dimension must be equal to
      \code{x}, otherwise NULL.}
    \item{\code{cval}:}{Object of class \code{"ANY"}: The critical
      values of the test at the 1\%, 5\% and 10\% level of significance.}
    \item{\code{teststat}:}{Object of class \code{"ANY"}: The values
      of the test statistics.}
    \item{\code{lambda}:}{Object of class \code{"vector"}: The eigenvalues.}
    \item{\code{Vorg}:}{Object of class \code{"matrix"}: The matrix of
      eigenvectors, such that \eqn{\hat V'S_{kk}\hat V = I}.}
    \item{\code{V}:}{Object of class \code{"matrix"}: The matrix of
      eigenvectors, normalised with respect to the first variable.}
    \item{\code{W}:}{Object of class \code{"matrix"}: The matrix of
      loading weights.}
    \item{\code{PI}:}{Object of class \code{"matrix"}: The coeffcient
      matrix of the lagged variables in levels.}
    \item{\code{DELTA}:}{Object of class \code{"matrix"}: The
      variance/covarinace matrix of \code{V}.}
    \item{\code{GAMMA}:}{Object of class \code{"matrix"}: The
      coeffecient matrix of \code{Z1}.}
    \item{\code{R0}:}{Object of class \code{"matrix"}: The matrix of
      residuals from the regressions in differences.}
    \item{\code{RK}:}{Object of class \code{"matrix"}: The matrix of
      residuals from the regression in lagged levels.}
    \item{\code{bp}:}{Object of class \code{"ANY"}: Potential break
      point, only set if function \code{cajolst} is called, otherwise
      \code{NA}.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `Johansen-Procedure'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ca.jo")} at the R prompt for a
  complete list of methods which are available for this class.

  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but critical values, eigenvectors
      and loading matrix added.}
    \item{\code{plot}:}{The series of the VAR and their potential
      cointegration relations.}
  }
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}}, \code{\link{plotres}} and \code{\link{urca-class}}.
}
\keyword{classes}

\eof
\name{ca.jo}
\alias{ca.jo}
\title{Johansen Procedure for VAR}
\description{
  Conducts the Johansen procedure on a given data set. The
  \code{"trace"} or \code{"eigen"} statistics are reported and the
  matrix of eigenvectors as well as the loading matrix.
}
\usage{
ca.jo(x, type = c("eigen", "trace"), constant = FALSE, K = 2,
spec=c("longrun", "transitory"), season = NULL, dumvar = NULL)
}
\arguments{
  \item{x}{Data matrix to be investigated for cointegration.}
  \item{type}{The test to be conducted, either \code{`eigen'} or
    \code{`trace'}.}
  \item{constant}{If TRUE, no linear trend is specified.}
  \item{K}{The lag order of the series (levels) in the VAR.}
  \item{spec}{Determines the specification of the VECM, see details below.}
  \item{season}{If seasonal dummies should be included, the data
    frequency must be set accordingly, \emph{i.e} `4' for quarterly
    data.}
  \item{dumvar}{If dummy variables should be included, a matrix with
    row dimension equal to \code{x} can be provided.} 
}
\details{

  Given a general VAR of the form:

  \deqn{\bold{X}_t = \bold{\Pi}_1 \bold{X}_{t-1} + \dots + \bold{\Pi}_k
    \bold{X}_{t-k} + \bold{\mu} + \bold{\Phi D}_t + \bold{\varepsilon}_t
    , \quad (t = 1, \dots, T),}

  the following two specifications of a VECM exist:

  \deqn{\Delta \bold{X}_t = \bold{\Gamma}_1 \Delta \bold{X}_{t-1} +
    \dots + \bold{\Gamma}_{k-1} \Delta \bold{X}_{t-k+1} + \bold{\Pi
    X}_{t-k} + \bold{\mu} + \bold{\Phi D}_t + \bold{\varepsilon}_t}

  where

  \deqn{\bold{\Gamma}_i = - (\bold{I} - \bold{\Pi}_1 - \dots -
    \bold{\Pi}_i), \quad (i = 1, \dots , k-1),}

  and

  \deqn{\bold{\Pi} = -(\bold{I} - \bold{\Pi}_1 - \dots - \bold{\Pi}_k)}

  The \eqn{\bold{\Gamma}_i} matrices contain the cumulative long-run
  impacts, hence if \code{spec="longrun"} is choosen, the above VECM is
  estimated.

  The other VECM specification is of the form:

  \deqn{\Delta \bold{X}_t = \bold{\Gamma}_1 \Delta \bold{X}_{t-1} +
    \dots + \bold{\Gamma}_{k-1} \Delta \bold{X}_{t-k+1} + \bold{\Pi
    X}_{t-1} + \bold{\mu} + \bold{\Phi D}_t + \bold{\varepsilon}_t}

  where

  \deqn{\bold{\Gamma}_i = - (\bold{\Pi}_{i+1} + \dots + \bold{\Pi}_k),
    \quad(i = 1, \dots , k-1),}

  and

  \deqn{\bold{\Pi} = -(\bold{I} - \bold{\Pi}_1 - \dots - \bold{\Pi}_k).}

  The \eqn{\bold{\Pi}} matrix is the same as in the first specification.
  However, the \eqn{\bold{\Gamma}_i} matrices now differ, in the sense
  that they measure transitory effects, hence by setting
  \code{spec="transitory"} the second VECM form is estimated. Please note
  that inferences drawn on \eqn{\bold{\Pi}} will be the same, regardless
  which specification is choosen and that the explanatory power is the
  same, too. 

  If \code{"season"} is not NULL, centered seasonal dummy variables are
  included. 

  If \code{"dumvar"} is not NULL, a matrix of dummy variables is included
  in the VECM. Please note, that the number of rows of the matrix
  containing the dummy variables must be equal to the row number of
  \code{x}.

  Critical values are only reported for systems with less than
  five variables.
  
}
\value{
  An object of class \code{`ca.jo'}.
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  
  \code{\link{plotres}}, \code{\link{alrtest}}, \code{\link{ablrtest}},
  \code{\link{blrtest}}, \code{\link{cajolst}}, \code{\link{cajools}},
  \code{\link{lttest}}, \code{\link{ca.jo-class}} and \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
summary(sjd.vecm)
#
data(finland)
sjf <- finland
sjf.vecm <- ca.jo(sjf, constant=FALSE, type="eigen", K=2,
spec="longrun", season=4)
summary(sjf.vecm)
}
\keyword{regression}

\eof
\name{ca.po-class}
\docType{class}
\alias{ca.po-class}

\title{Representation of class `ca.po'}
\description{
  This class contains the relevant information by applying the Phillips
  \& Ouliaris cointegration test to a data matrix.
}
\section{Slots}{
  \describe{
    \item{\code{z}:}{Object of class \code{"ANY"}: A data matrix, or an
      object that can be coerced to it.}
    \item{\code{type}:}{Object of class \code{"character"}: The type of
      the test, either the \code{"Pu"}-test or the normalisation
	invariant \code{"Pz"}-test.}
    \item{\code{model}:}{Object of class \code{"character"}: Determines
      how the series should be detrended.}
    \item{\code{lag}:}{Object of class \code{"integer"}: The lags used
      for variance/covariance correction.}
    \item{\code{cval}:}{Object of class \code{"matrix"}: The critical
      values of the test at the 1\%, 5\% and 10\% level of significance.}
    \item{\code{res}:}{Object of class \code{"matrix"}: The residuals of
      the the cointegration regression(s).}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: The value
      of the test statistic.}
    \item{\code{testreg}:}{Object of class \code{"ANY"}: The summary
      output of the cointegration regression(s).}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `Phillips \& Ouliaris'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ca.po")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but critical value and summary of
      test regression(s) added.}
    \item{\code{plot}:}{Residual plot(s) and their acfs' and pacfs'.}
  }
}
\references{
  Phillips, P.C.B. and Ouliaris, S. (1990), Asymptotic Properties of
  Residual Based Tests for Cointegration, \emph{Econometrica},
  \bold{Vol. 58, No. 1}, 165--193.
}
  
\seealso{
  \code{\link{ca.po}} and \code{\link{urca-class}}. 
}
\keyword{classes}

\eof
\name{ca.po}
\alias{ca.po}
\title{Phillips \& Ouliaris Cointegration Test}
\description{
  Performs the Phillips \& Ouliaris \code{"Pu"} and \code{"Pz"}
  cointegration test.
}
\usage{
ca.po(z, demean = c("none", "constant", "trend"),
      lag = c("short", "long"), type = c("Pu", "Pz"), tol = NULL)
}
\arguments{
  \item{z}{Data matrix to be investigated for cointegration.}
  \item{demean}{The method for detrending the series, either
    \code{"none"}, \code{"constant"} or \code{"trend"}.}
  \item{lag}{Either a short or long lag number used for
    variance/covariance correction.}
  \item{type}{The test type, either \code{"Pu"} or \code{"Pz"}.}
  \item{tol}{Numeric, this argument is passed to \code{solve()} in
    \code{ca.po()}.}
}
\details{
  The test \code{"Pz"}, compared to the test \code{"Pu"}, has the
  advantage that it is invariant to the normalization of the
  cointegration vector, \emph{i.e.} it does not matter which variable
  is one the left hand side of the equation. In case convergence
  problems are encountered by matrix inversion, one can pass a higher
  tolerance level \emph{via} \code{"tol=..."} to the \code{solve()}-function.
}
\value{
  An object of class \code{`ca.po'}.
}
\references{
  Phillips, P.C.B. and Ouliaris, S. (1990), Asymptotic Properties of
  Residual Based Tests for Cointegration, \emph{Econometrica},
  \bold{Vol. 58, No. 1}, 165--193.
}
  
\seealso{\code{\link{ca.po-class}}}
\examples{
data(ecb)
m3.real <- ecb[,"m3"]/ecb[,"gdp.defl"]
gdp.real <- ecb[,"gdp.nom"]/ecb[,"gdp.defl"]
rl <- ecb[,"rl"]
ecb.data <- cbind(m3.real, gdp.real, rl)
m3d.po <- ca.po(ecb.data, type="Pz")
summary(m3d.po)
}
\keyword{regression}

\eof
\name{cajo.test-class}
\docType{class}
\alias{cajo.test-class}
\title{Representation of class `cajo.test'}
\description{
  This class contains the relevant information by estimating and testing
  a VAR under linear restrictions on \eqn{\bold{\alpha}} and
  \eqn{\bold{\beta}}.
}
\section{Slots}{
  \describe{
    \item{\code{Z0}:}{Object of class \code{"matrix"}: The matrix of the
      differenced series.}
    \item{\code{Z1}:}{Object of class \code{"matrix"}: The regressor
      matrix, except for the lagged variables in levels.}
    \item{\code{ZK}:}{Object of class \code{"matrix"}: The matrix of the
      lagged variables in levels.}
    \item{\code{const}:}{Object of class \code{"logic"}: If TRUE, no
      linear trend is included and \emph{vice versa}.}
    \item{\code{H}:}{Object of class \code{"ANY"}: The matrix
      containing the restrictions placed upon \eqn{\bold{\beta}}.}
    \item{\code{A}:}{Object of class \code{"ANY"}: The matrix
      containing the restrictions placed upon \eqn{\bold{\alpha}}.}
    \item{\code{B}:}{Object of class \code{"ANY"}: The matrix
      orthogonal to matrix \eqn{\bold{A}}.}
    \item{\code{type}:}{Object of class \code{"character"}: The test type.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: The value
      of the test statistic.}
    \item{\code{pval}:}{Object of class \code{"vector"}: The p-value and
      the degrees of freedom.}
    \item{\code{lambda}:}{Object of class \code{"vector"}: The
      eigenvalues of the restricted model.}
    \item{\code{Vorg}:}{Object of class \code{"matrix"}: The matrix of
      eigenvectors, such that \eqn{\hat V_{\dots}'(H'S_{\dots}H)\hat
	V_{\dots} = I}.}
    \item{\code{V}:}{Object of class \code{"matrix"}: The matrix of the
      restricted eigenvectors, normalised with respect to the first variable.}
    \item{\code{W}:}{Object of class \code{"matrix"}: The matrix of the
      corresponding loading weights.}
    \item{\code{PI}:}{Object of class \code{"matrix"}: The coefficient
      matrix of the lagged variables in levels.}
    \item{\code{DELTA}:}{Object of class \code{"ANY"}: The
      variance/covarinace matrix of \eqn{\bold{V}}.}
    \item{\code{DELTA.bb}:}{Object of class \code{"ANY"}: The
      variance/covarinace matrix of the marginal factor
      \eqn{\bold{B}'\bold{R}_{0t}}.}
    \item{\code{DELTA.ab}:}{Object of class \code{"ANY"}: The
      variance/covarinace matrix of the conditional distribution of
      \eqn{\bold{A}'\bold{R}_{0t}} and \eqn{\bold{R}_{kt}}.}
    \item{\code{DELTA.aa.b}:}{Object of class \code{"ANY"}: The
      variance/covarinace matrix of the restricted loading matrix.}
    \item{\code{GAMMA}:}{Object of class \code{"matrix"}: The
      coefficient matrix of \eqn{\bold{Z1}}.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `Johansen-Procedure'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="cajo.test")} at the R prompt for a
  complete list of methods which are available for this class.

  Useful methods include
  \describe{
    \item{\code{show}:}{test-statistic.}
    \item{\code{summary}:}{like show, but p-value of test statistic,
      restricted eigenvectors, loading matrix and restriction matrices
      \eqn{\bold{H}} and \eqn{\bold{A}}, where applicable, added.}
    }
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ablrtest}}, \code{\link{alrtest}}, \code{\link{blrtest}},
  \code{\link{ca.jo}}, \code{\link{ca.jo-class}} and \code{\link{urca-class}}.
}
\keyword{classes}

\eof
\name{cajolst}
\alias{cajolst}
\title{Testing Cointegrating Rank with Level Shift at Unknown time}
\description{
  The function \code{cajolst} implements the procedure by Luetkepohl
  \emph{et al.} to test for the cointegration rank of a VAR process with
  a level shift at an unknown time.
}
\usage{
cajolst(x, trend = TRUE, K = 2, season = NULL)
}
\arguments{
  \item{x}{Data matrix to be investigated for cointegration.}
  \item{trend}{A linear trend is included in the auxiliary regressions
    for data adjustment (default is \code{TRUE}).}
  \item{K}{The lag order of the series (levels) in the VAR, must be at
    least equal to \eqn{K = 2}.}
  \item{season}{If seasonal dummies should be included, the data
    frequency must be set accordingly, \emph{i.e} `4' for quarterly data.}
}
\details{
  Note, that the slot \code{"x"} of the returned object contains the
  adjusted data series, that is, a matrix adjusted for the temptative
  break point, and if applicable, a linear trend and/or seasonal
  effects. The VECM is then estimated and tested for cointegration rank
  subject to the adjusted matrix. The break point is contained in the
  slot \code{"bp"}. Please note, that the \emph{transitory} VECM
  specification is estimated and that only the trace test is
  available. The critical values are taken from Trenkler, Carsten (2003).
}
\value{
  Returns an object of class \code{`ca.jo'}.
}
\references{
  
  L\"utkepohl, H., Saikkonen, P. and Trenkler, C. (2004), Testing for the
  Cointegrating Rank of a VAR Process with Level Shift at Unknown Time,
  \emph{Econometrica}, \bold{Vol. 72, No. 2}, 647--662.

  Trenkler, Carsten (2003), A new set of critical values for systems
  cointegration tests with a prior adjustment for deterministic terms,
  \emph{Economics Bulletin}, \bold{Vol. 3, No. 11}, 1--9. 
  
}
\seealso{
  
  \code{\link{plotres}}, \code{\link{alrtest}}, \code{\link{ablrtest}},
  \code{\link{blrtest}}, \code{\link{ca.jo}}, \code{\link{cajools}},
  \code{\link{lttest}}, \code{\link{ca.jo-class}} and \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.lst <- cajolst(sjd, trend=TRUE, K=2, season=4)
summary(sjd.lst)
}
\keyword{regression}

\eof
\name{cajools}
\alias{cajools}
\title{OLS regression of VECM}
\description{
  This function returns the OLS regressions of an unrestricted VECM,
  \emph{i.e.} it returns an object of class `lm'. The user can provide a
  certain number of which equation in the VECM should be estimated and
  reported, or if \code{"reg.number=NULL"} each equation in the VECM
  will be estimated and its results are reported.
}
\usage{
cajools(z, reg.number = NULL)
}
\arguments{
  \item{z}{An object of class `ca.jo' or `cajo.test'.}
  \item{reg.number}{The number of the equation in the VECM that should
    be estimated or if set to \code{NULL} (the default), all equations
    within the VECM are estimated.}
}
\value{
  Returns an object of class \code{`lm'}.
}
\references{

  Johansen, S. (1988), Statistical Analysis of Cointegration Vectors,
  \emph{Journal of Economic Dynamics and Control}, \bold{12}, 231--254.

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}}, \code{\link[base]{lm}}, \code{\link{ca.jo-class}}
  and \code{\link{urca-class}}. 
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm1 <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
sjd.vecm2 <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="transitory",
season=4)
sjd.vecm.ols1 <- cajools(sjd.vecm1)
sjd.vecm.ols2 <- cajools(sjd.vecm2)
summary(sjd.vecm.ols1)
summary(sjd.vecm.ols2)
}
\keyword{regression}

\eof
\name{denmark}
\alias{denmark}
\docType{data}
\title{Data set for Denmark, Johansen \& Juselius (1990)}
\description{
  This data set contains the series used by S. Johansen and K. Juselius
  for estimating a money demand function of Denmark.
}
\usage{data(denmark)}
\format{
  A data frame with 55 observations on the following 6 variables.

    \tabular{rl}{
    \code{period} \tab Time index from 1974:Q1 until 1987:Q3. \cr 
    \code{LRM} \tab Logarithm of real money, M2. \cr
    \code{LRY} \tab Logarithm of real income. \cr
    \code{LPY} \tab Logarithm of price deflator. \cr
    \code{IBO} \tab Bond rate. \cr
    \code{IDE} \tab Bank deposit rate. \cr
    }
}
\source{
  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.
  
}
\references{
  \url{http://www.math.ku.dk/~sjo/}
}
\keyword{datasets}

\eof
\name{ecb}
\alias{ecb}
\docType{data}
\title{Macroeconomic data of the Euro Zone}
\description{
  This data set contains some macroeconomic figures of the Euro Zone in
  order to estimate an exemplary money demand function.
}
\usage{data(ecb)}
\format{
  A data frame containing five series.

  \tabular{rl}{
    \code{period} \tab Time index from Q31997 until Q42003.\cr  
    \code{gdp.defl} \tab Gross Domestic Product Deflator,\cr 
    \tab [Index 2000=100, seasonally adjusted] \cr
    \code{gdp.nom} \tab Nominal Gross Domestic Product, \cr 
    \tab [Current prices, EUR billions, seasonally adjusted] \cr 
    \code{m3} \tab Monetary Aggregate M3, \cr
    \tab [outstanding amount at end of quarter, EUR billions, seasonally
    adjusted] \cr 
    \code{rl} \tab Benchmark Government Bond yield with a maturity of 10
    years, \cr
    \tab [percentages per annum, average of last quarter's month]\cr 
    }
  }
}
\source{
  European Central Bank, Monthly Bulletins, Frankfurt am Main, Germany.
}
\references{
 \url{http://www.ecb.int}
}
\keyword{datasets}

\eof
\name{finland}
\alias{finland}
\docType{data}
\title{Data set for Finland, Johansen \& Juseliues (1990)}
\description{
  This data set contains the series used by S. Johansen and K. Juselius
  for estimating a money demand function of Finland.
}
\usage{data(finland)}
\format{
  A data frame with 106 observations on the following 4 variables,
  ranging from 1958:Q2 until 1984:Q3.

    \tabular{rl}{
    \code{lrm1} \tab Logarithm of real money, M1. \cr
    \code{lny} \tab Logarithm of real income. \cr
    \code{lnmr} \tab Marginal rate of interest. \cr
    \code{difp} \tab Inflation rate. \cr
    }
}
\source{
  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.
  
}
\references{
  \url{http://www.math.ku.dk/~sjo/}
}
\keyword{datasets}

\eof
\name{lttest}
\alias{lttest}
\title{Likelihood ratio test for no linear trend in VAR}
\description{
  Conducts a likelihood ratio test for no inclusion of a linear trend in a
  VAR. That is, the Null hypothesis is for not including a linear trend
  and is assigned as 'H2*(r)'. The test statistic is distributed as
  \eqn{\chi^2} square with \eqn{(p-r)} degrees of freedom.
}
\usage{
lttest(z, r)
}
\arguments{
  \item{z}{An object of class `ca.jo'.}
  \item{r}{The count of cointegrating relationships.}
}
\details{
  The count of cointegrating relations should be given as integer and
  should be in the interval \eqn{1 \leq r < P}.
}
\value{
  \item{lttest}{Matrix containing the value of the test statistic and its
    p-value.}
}
\references{

  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.

  Johansen, S. (1991), Estimation and Hypothesis Testing of
  Cointegration Vectors in Gaussian Vector Autoregressive Models,
  \emph{Econometrica}, \bold{Vol. 59, No. 6}, 1551--1580.

}
\seealso{
  \code{\link{ca.jo}} and \code{\link{ca.jo-class}}.
}
\examples{
data(denmark)
sjd <- as.matrix(denmark[, c("LRM", "LRY", "IBO", "IDE")])
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
lttest(sjd.vecm, r=1)
#
data(finland)
sjf <- as.matrix(finland)
sjf.vecm <- ca.jo(sjf, constant=FALSE, type="eigen", K=2,
spec="longrun", season=4)
lttest(sjf.vecm, r=3)
}
\keyword{regression}

\eof
\name{npext}
\docType{data}
\alias{npext}
\title{Nelson \& Plosser extended data set}
\description{
  This data set contains the fourteen U.S. economic time series used by
  Schotman \& Dijk. All series are transformed by taking logarithms
  except for the bond yield. The sample period ends in 1988.
}
\usage{data(npext)}
\format{
  A data frame containing fourteen series.
		
  \tabular{rl}{
    \code{year} \tab Time index from 1860 until 1988. \cr 
    \code{realgnp} \tab Real GNP, [Billions of 1958 Dollars], \cr
    \tab [1909 -- 1988] \cr
    \code{nomgnp} \tab Nominal GNP, \cr
    \tab [Millions of Current Dollars], [1909 -- 1988] \cr
    \code{gnpperca} \tab Real Per Capita GNP, \cr
    \tab [1958 Dollars], [1909 -- 1988] \cr
    \code{indprod} \tab Industrial Production Index, \cr
    \tab [1967 = 100], [1860 -- 1988] \cr
    \code{employmt} \tab Total Employment, \cr
    \tab [Thousands], [1890 -- 1988] \cr
    \code{unemploy} \tab Total Unemployment Rate, \cr
    \tab [Percent], [1890 -- 1988] \cr
    \code{gnpdefl} \tab GNP Deflator, \cr
    \tab [1958 = 100], [1889 -- 1988] \cr
    \code{cpi} \tab Consumer Price Index, \cr
    \tab [1967 = 100], [1860 -- 1988] \cr
    \code{wages} \tab Nominal Wages \cr
    \tab (Average annual earnings per full-time employee in manufacturing), \cr
    \tab [current Dollars], [1900 -- 1988] \cr
    \code{realwag} \tab Real Wages, \cr
    \tab [Nominal wages/CPI], [1900 -- 1988] \cr
    \code{M} \tab Money Stock (M2), \cr
    \tab [Billions of Dollars, annual averages], [1889 -- 1988] \cr
    \code{velocity} \tab Velocity of Money, \cr
    \tab [1869 -- 1988] \cr
    \code{interest} \tab Bond Yield (Basic Yields of 30-year
       corporate bonds), \cr
    \tab [Percent per annum], [1900 -- 1988] \cr
    \code{sp500} \tab Stock Prices, \cr
    \tab [Index; 1941 -- 43 = 100], [1871 -- 1988] \cr
    }
}
\source{
  Schotman, P.C. and van Dijk, H.K. (1991), On Bayesian Routes to Unit
  Roots, \emph{Journal of Applied Econometrics}, \bold{6}, 387--401.

  Koop, G. and Steel, M.F.J. (1994), A Decision-Theoretic Analysis of
  the Unit-Root Hypothesis using Mixtures of Elliptical Models,
  \emph{Journal of Business and Economic Statistics}, \bold{12}, 95--107. 
}
\references{\url{http://www.amstat.org/publications/jbes/}}
\keyword{datasets}

\eof
\name{nporg}
\docType{data}
\alias{nporg}
\title{Nelson \& Plosser original data set}
\description{
  This data set contains the fourteen U.S. economic time series used by
  Nelson \& Plosser in their seminal paper.
}
\usage{data(nporg)}
\format{
  A data frame containing fourteen series.

  \tabular{rl}{
    \code{year} \tab Time index from 1860 until 1970. \cr 
    \code{gnp.r} \tab Real GNP, \cr
    \tab [Billions of 1958 Dollars], [1909 -- 1970] \cr
    \code{gnp.n} \tab Nominal GNP, \cr
    \tab [Millions of Current Dollars], [1909 -- 1970] \cr
    \code{gnp.pc} \tab Real Per Capita GNP, \cr
    \tab [1958 Dollars], [1909 -- 1970] \cr
    \code{ip} \tab Industrial Production Index, \cr
    \tab [1967 = 100], [1860 -- 1970] \cr
    \code{emp} \tab Total Employment, \cr
    \tab [Thousands], [1890 -- 1970] \cr
    \code{ur} \tab Total Unemployment Rate, \cr
    \tab [Percent], [1890 -- 1970] \cr
    \code{gnp.p} \tab GNP Deflator, \cr
    \tab [1958 = 100], [1889 -- 1970] \cr
    \code{cpi} \tab Consumer Price Index, \cr
    \tab [1967 = 100], [1860 -- 1970] \cr
    \code{wg.n} \tab Nominal Wages \cr
    \tab (Average annual earnings per full-time employee in manufacturing), \cr
    \tab [current Dollars], [1900 -- 1970] \cr
    \code{wg.r} \tab Real Wages, \cr
    \tab [Nominal wages/CPI], [1900 -- 1970] \cr
    \code{M} \tab Money Stock (M2), \cr
    \tab [Billions of Dollars, annual averages], [1889 -- 1970] \cr
    \code{vel} \tab Velocity of Money, \cr
    \tab [1869 -- 1970] \cr
    \code{bnd} \tab Bond Yield (Basic Yields of 30-year
        corporate bonds), \cr
    \tab [Percent per annum], [1900 -- 1970] \cr
    \code{sp} \tab Stock Prices, \cr
    \tab [Index; 1941 -- 43 = 100], [1871 -- 1970] \cr
    }
}
\source{
  Nelson, C.R. and Plosser, C.I. (1982), Trends and Random Walks in
  Macroeconomic Time Series, \emph{Journal of Monetary Economics},
  \bold{10}, 139--162.
}
\references{
  \url{http://korora.econ.yale.edu/phillips/index.htm}
}
\keyword{datasets}

\eof
\name{plot-methods}
\docType{methods}
\alias{plot-methods}
\alias{plot,ur.ers,missing-method}
\alias{plot,ur.kpss,missing-method}
\alias{plot,ca.jo,missing-method}
\alias{plot,ca.po,missing-method}
\alias{plot,ur.pp,missing-method}
\alias{plot,ur.sp,missing-method}
\alias{plot,ur.za,missing-method}
\title{Methods for Function plot in Package `urca'}
\description{
  Plot methods for objects belonging to classes set in package
  \code{`urca'}. Depending on the unit root/cointegration test a
  suitable graphical presentation is selected.
  }
\section{Methods}{\describe{

\item{x = "ur.ers", y = "missing"}{Diagram of fit of the Elliott,
  Rothenberg \& Stock unit root test of type \code{"DF-GLS"} with
  residual plot and their acfs' and pacfs'.}
\item{x = "ur.kpss", y = "missing"}{Residual plot and their acfs' and
  pacfs' of the KPSS test.}
\item{x = "ca.jo", y = "missing"}{Time series plots and associated
  cointegration relations for the Johansen procedure.}
\item{x = "ca.po", y = "missing"}{Residual plot and their acfs' and
  pacfs' of the cointegration regression(s) for the Phillips \& Ouliaris
  test.}
\item{x = "ur.pp", y = "missing"}{Diagram of fit of the Phillips \&
  Perron unit root test, residual plot and their acfs' and pacfs'.}
\item{x = "ur.sp", y = "missing"}{Diagram of fit of the Schmidt \&
  Phillips unit root test, residual plot and their acfs' and pacfs'.}
\item{x = "ur.za", y = "missing"}{Plot of recursive t-statistics
  as outcome of Zivot \& Andrews unit root test.}
}}
\seealso{
  \code{\link{ur.ers-class}}, \code{\link{ur.kpss-class}},
  \code{\link{ca.jo-class}}, \code{\link{ca.po-class}},
  \code{\link{ur.pp-class}}, \code{\link{ur.sp-class}} and
  \code{\link{ur.za-class}}.}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
gnp.l <- log(gnp)
#
ers.gnp <- ur.ers(gnp, type="DF-GLS", model="trend", lag.max=4)
plot(ers.gnp)
#
kpss.gnp <- ur.kpss(gnp.l, type="tau", lags="short")
plot(kpss.gnp)
#
pp.gnp <- ur.pp(gnp, type="Z-tau", model="trend", lags="short")
plot(pp.gnp)
#
sp.gnp <- ur.sp(gnp, type="tau", pol.deg=1, signif=0.01)
plot(sp.gnp)
#
za.gnp <- ur.za(gnp, model="both", lag=2)
plot(za.gnp)
#
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, season=4)
plot(sjd.vecm)
}
\keyword{methods}


\eof
\name{plotres}
\alias{plotres}
\title{Graphical inspection of VECM residuals}
\description{
  The function \code{plotres} should be used for graphical inspection
  of the VAR residuals, \emph{i.e.} the estimated specification as
  elaborated in the `Details' section of \code{ca.jo}. It displays the
  residuals for each equation within a VAR and their acf's and pacf's.
}
\usage{
plotres(x)
}
\arguments{
  \item{x}{Object of class `ca.jo'.}
}
\references{
  Johansen, S. and Juselius, K. (1990), Maximum Likelihood Estimation and
  Inference on Cointegration -- with Applications to the Demand for
  Money, \emph{Oxford Bulletin of Economics and Statistics}, \bold{52,
    2}, 169--210.
  
}
\seealso{
  \code{\link{ca.jo}} and \code{\link{ca.jo-class}}.
}
\examples{
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, spec="longrun",
season=4)
plotres(sjd.vecm)
}
\keyword{regression}

\eof
\name{show-methods}
\docType{methods}
\alias{show-methods}
\alias{show,ur.ers-method}
\alias{show,ur.kpss-method}
\alias{show,ca.jo-method}
\alias{show,ca.po-method}
\alias{show,cajo.test-method}
\alias{show,ur.pp-method}
\alias{show,ur.sp-method}
\alias{show,ur.za-method}
\title{Methods for Function show in Package `urca'}
\description{
  Displays the outcome of the unit root/cointegration tests.
}
\section{Methods}{\describe{

\item{object = "ca.jo"}{Displays the test statistic of the Johansen procedure.}
\item{object = "cajo.test"}{Displays the test statistic of a restricted
  VAR with respect to \eqn{\bold{\alpha}} and/or \eqn{\bold{\beta}}.}
\item{object = "ca.po"}{Displays the test statistic of the Phillips \&
  Ouliaris cointegration test.}
\item{object = "ur.ers"}{Displays the test statistic of the Elliott,
  Rothenberg \& Stock unit root test.}
\item{object = "ur.kpss"}{Displays the test statistic of the Kwiatkowski
  \emph{et al.} unit root test.}
\item{object = "ur.pp"}{Displays the test statistic of the Phillips \&
  Perron unit root test.}
\item{object = "ur.sp"}{Displays the test statistic of the Schmidt \&
  Phillips unit root test.}
\item{object = "ur.za"}{Displays the test statistic of the Zivot \&
  Andrews unit root test.}
}}
\seealso{
  \code{\link{ca.jo-class}}, \code{\link{cajo.test-class}},
  \code{\link{ca.po-class}}, \code{\link{ur.ers-class}},
  \code{\link{ur.kpss-class}}, \code{\link{ur.pp-class}},
  \code{\link{ur.sp-class}} and \code{\link{ur.za-class}}.
}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
gnp.l <- log(gnp)
#
ers.gnp <- ur.ers(gnp, type="DF-GLS", model="trend", lag.max=4)
show(ers.gnp)
#
kpss.gnp <- ur.kpss(gnp.l, type="tau", lags="short")
show(kpss.gnp)
#
pp.gnp <- ur.pp(gnp, type="Z-tau", model="trend", lags="short")
show(pp.gnp)
#
sp.gnp <- ur.sp(gnp, type="tau", pol.deg=1, signif=0.01)
show(sp.gnp)
#
za.gnp <- ur.za(gnp, model="both", lag=2)
show(za.gnp)
#
data(denmark)
sjd <- denmark[, c("LRM", "LRY", "IBO", "IDE")]
sjd.vecm <- ca.jo(sjd, constant=TRUE, type="eigen", K=2, season=4)
show(sjd.vecm)
#
HD0 <- matrix(c(-1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1), c(5,4))
show(blrtest(sjd.vecm, H=HD0, r=1))
}
\keyword{methods}

\eof
\name{show.urca}
\alias{show.urca}
\title{Function to show objects of classes for unit root tests}
\description{
  The function \code{show.urca} is called within the defined methods
  for classes \code{ca.jo}, \code{cajo.test}, \code{ca.po}, \code{ur.ers}, \code{ur.kpss}, \code{ur.po},
  \code{ur.pp}, \code{ur.sp} and \code{ur.za}.
}
\usage{
show.urca(object)
}
\arguments{
  \item{object}{Object of class contained in \code{`urca'}.}
}
\details{
  This function is called by method \code{show}.
}
\value{
  The Name and test statistic of a unit root/cointegration test.
}
\keyword{methods}

\eof
\name{summary-methods}
\docType{methods}
\alias{summary-methods}
\alias{summary,ur.ers-method}
\alias{summary,ur.kpss-method}
\alias{summary,ca.jo-method}
\alias{summary,cajo.test-method}
\alias{summary,ca.po-method}
\alias{summary,ur.pp-method}
\alias{summary,ur.sp-method}
\alias{summary,ur.za-method}
\title{Methods for Function summary in Package `urca'}
\description{
  Summarises the outcome of unit root/cointegration tests.
}
\section{Methods}{\describe{
    \item{object = "ur.ers"}{The test type, its statistic and the
      critical values for the Elliott, Rothenberg \& Stock test are
      returned. In case of test \code{"DF-GLS"} the summary output
      of the test regression is provided, too.}
    \item{object = "ur.kpss"}{The test statistic, the critical value as
      well as the test type and the number of lags used for error
      correction for the Kwiatkowski \emph{et al.} unit root test is
      returned.}
    \item{object = "ca.jo"}{The \code{"trace"} or \code{"eigen"} statistic,
      the critical values as well as the eigenvalues, eigenvectors and
      the loading matrix of the Johansen procedure are reported.}
    \item{object = "cajo.test"}{The test statistic of a restricted VAR
      with respect to \eqn{\bold{\alpha}} and/or \eqn{\bold{\beta}} with
      p-value and degrees of freedom is reported. Furthermore, the
      restriction matrix(ces), the eigenvalues and eigenvectors as well
      as the loading matrix are returned.}  
    \item{object = "ca.po"}{The \code{"Pz"} or \code{"Pu"} statistic,
      the critical values as well as the summary output of the test
      regression for the Phillips \& Ouliaris cointegration test.}
    \item{object = "ur.pp"}{The Z statistic, the critical values as
      well as the summary output of the test regression for the Phillips
      \& Perron test, as well as the test statistics for the
      coefficients of the deterministic part is returned.}
    \item{object = "ur.sp"}{The test statistic, the critical value as
      well as the summary output of the test regression for the Schmidt
      \& Phillips test is returned.}
    \item{object = "ur.za"}{The test statistic, the critical values as
      well as the summary output of the test regression for the Zivot \&
      Andrews test is returned.}
}}
\seealso{
  \code{\link{ur.ers-class}}, \code{\link{ur.kpss-class}},
  \code{\link{ca.jo-class}}, \code{\link{cajo.test-class}},
  \code{\link{ca.po-class}}, \code{\link{ur.pp-class}},
  \code{\link{ur.sp-class}} and \code{\link{ur.za-class}}.}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
gnp.l <- log(gnp)
#
ers.gnp <- ur.ers(gnp, type="DF-GLS", model="trend", lag.max=4)
summary(ers.gnp)
#
kpss.gnp <- ur.kpss(gnp.l, type="tau", lags="short")
summary(kpss.gnp)
#
pp.gnp <- ur.pp(gnp, type="Z-tau", model="trend", lags="short")
summary(pp.gnp)
#
sp.gnp <- ur.sp(gnp, type="tau", pol.deg=1, signif=0.01)
summary(sp.gnp)
#
za.gnp <- ur.za(gnp, model="both", lag=2)
summary(za.gnp)
#
data(finland)
sjf <- finland
sjf.vecm <- ca.jo(sjf, constant=FALSE, type="eigen", K=2, season=4)
summary(sjf.vecm)
#
HF0 <- matrix(c(-1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1), c(4,3))
summary(blrtest(sjf.vecm, H=HF0, r=3))
}
\keyword{methods}

\eof
\name{ur.ers-class}
\docType{class}
\alias{ur.ers-class}
\title{Representation of class `ur.ers'}
\description{
  This class contains the relevant information by applying the Elliott,
  Rothenberg \& Stock unit root test.
}
\section{Slots}{
  \describe{
    \item{\code{y}:}{Object of class \code{"vector"}: The time series to
      be tested.}
    \item{\code{yd}:}{Object of class \code{"vector"}: The detrended
      time series.}
    \item{\code{type}:}{Object of class \code{"character"}: Test type,
      either \code{"DF-GLS"} (default), or \code{"P-test"}.}
    \item{\code{model}:}{Object of class \code{"character"}: The
      deterministic model used for detrending, either intercept only, or
      intercept with linear trend.}
    \item{\code{lag}:}{Object of class \code{"integer"}: The number of
      lags used in the test/auxiliary regression.}
    \item{\code{cval}:}{Object of class \code{"matrix"}: The critical
      values of the test at the 1\%, 5\% and 10\% level of significance.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: The value
      of the test statistic.}
    \item{\code{testreg}:}{Object of class \code{"ANY"}: The test
      regression, only set for \code{"DF-GLS"}.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `Elliott, Rothenberg \& Stock'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ur.ers")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but test type, test regression (\code{type="DF-GLS"}) and critical values added.}
    \item{\code{plot}:}{Diagram of fit, residual plot and their acfs'
      and pacfs' for \code{type="DF-GLS"}.}
  }

}
\references{
  Elliott, G., Rothenberg, T.J. and Stock, J.H. (1996), Efficient Tests
  for an Autoregressive Unit Root, \emph{Econometrica},
  \bold{Vol. 64, No. 4}, 813--836.

  MacKinnon, J.G. (1991), Critical Values for Cointegration Tests,
  \emph{Long-Run Economic Relationships}, eds. R.F. Engle and
  C.W.J. Granger, London, Oxford, 267--276.

  Download possible at: \url{http://www.econ.ucsd.edu/papers/files/90-4.pdf}.
}
\seealso{
  \code{\link{ur.ers}} and \code{\link{urca-class}}.
}
\keyword{classes}

\eof
\name{ur.ers}
\alias{ur.ers}
\title{Elliott, Rothenberg \& Stock Unit Root Test}
\description{
  Performs the Elliott, Rothenberg \& Stock unit root test. 
}
\usage{
ur.ers(y, type = c("DF-GLS", "P-test"), model = c("constant", "trend"),
       lag.max = 4)
}
\arguments{
  \item{y}{Vector to be tested for a unit root.}
  \item{type}{Test type, either \code{"DF-GLS"} (default), or \code{"P-test"}.}
  \item{model}{The deterministic model used for detrending.}
  \item{lag.max}{The maximum numbers of lags used for testing of a
    decent lag truncation for the \code{"P-test"} (\code{BIC} used),
    or the maximum number of lagged differences to be included in the test
    regression for \code{"DF-GLS"}.}
}
\details{
  To improve the power of the unit root test, Elliot, Rothenberg \& Stock
  proposed a local to unity detrending of the time series. ERS developed
  a feasible point optimal test, \code{"P-test"}, which takes serial
  correlation of the error term into account. The second test type is
  the \code{"DF-GLS"} test, which is an ADF-type test applied to the
  detrended data without intercept. Critical values for this test are
  taken from MacKinnon in case of \code{model="constant"} and else from
  Table 1 of Elliot, Rothenberg \& Stock.
}
\value{
  An object of class \code{"ur.ers"}.
}

\references{
  Elliott, G., Rothenberg, T.J. and Stock, J.H. (1996), Efficient Tests
  for an Autoregressive Unit Root, \emph{Econometrica},
  \bold{Vol. 64, No. 4}, 813--836.

  MacKinnon, J.G. (1991), Critical Values for Cointegration Tests,
  \emph{Long-Run Economic Relationships}, eds. R.F. Engle and
  C.W.J. Granger, London, Oxford, 267--276.

  Download possible at: \url{http://www.econ.ucsd.edu/papers/files/90-4.pdf}.
}

\seealso{\code{\link{ur.ers-class}}}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
ers.gnp <- ur.ers(gnp, type="DF-GLS", model="const", lag.max=4)
summary(ers.gnp)
}
\keyword{regression}

\eof
\name{ur.kpss-class}
\docType{class}
\alias{ur.kpss-class}

\title{Representation of class `ur.kpss'}
\description{
  This class contains the relevant information by applying the
  Kwiatkowski, Phillips, Schmidt \& Shin unit root test to a time series.
}
\section{Slots}{
  \describe{
    \item{\code{y}:}{Object of class \code{"vector"}: The time series to
      be tested.}
    \item{\code{type}:}{Object of class \code{"character"}: Test type,
      \code{"mu"} or \code{"tau"} depending on the deterministic part.}
    \item{\code{lag}:}{Object of class \code{"integer"}: Number of lags
      for error term correction.}
    \item{\code{cval}:}{Object of class \code{"matrix"}: Critical value
      of test.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: Value of
      test statistic.}
    \item{\code{res}:}{Object of class \code{"vector"}: Residuals of
      test regression.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `KPSS'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ur.kpss")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but critical values, lags and test
      type added.}
    \item{\code{plot}:}{Residual plot and their acfs' and pacfs'.}
  }

}
\references{
  Kwiatkowski, D., Phillips, P.C.B., Schmidt, P. and Shin, Y., (1992),
  Testing the Null Hypothesis of Stationarity Against the Alternative of
  a Unit Root: How Sure Are We That Economic Time Series Have a Unit
  Root?, \emph{Journal of Econometrics}, \bold{54}, 159--178.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{
  \code{\link{ur.kpss}} and \code{\link{urca-class}}.
}
\keyword{classes}

\eof
\name{ur.kpss}
\alias{ur.kpss}
\title{Kwiatkowski et al. Unit Root Test}
\description{
  Performs the KPSS unit root test, where the Null hypothesis is
  stationarity. The test types specify as deterministic component either
  a constant \code{"mu"} or a constant with linear trend \code{"tau"}.
}
\usage{
ur.kpss(y, type = c("mu", "tau"), lags = c("short", "long", "nil"),
        use.lag = NULL)
}
\arguments{
  \item{y}{Vector to be tested for a unit root.}
  \item{type}{Type of deterministic part.}
  \item{lags}{Maximum number of lags used for error term correction.}
  \item{use.lag}{User specified number of lags.}
}
\details{
  \code{lags="short"} sets the number of lags to \eqn{\root 4 \of {4
      \times (n/100)}},
  whereas \code{lags="long"} sets the number of lags to \eqn{\root 4 \of
    {12 \times (n/100)}}. If \code{lags="nil"} is choosen, then no error
  correction is made. Furthermore, one can specify a different number of
  maximum lags by setting \code{use.lag} accordingly.
}
\value{
  An object of class \code{`ur.kpss'}.
}
\references{
  Kwiatkowski, D., Phillips, P.C.B., Schmidt, P. and Shin, Y., (1992),
  Testing the Null Hypothesis of Stationarity Against the Alternative of
  a Unit Root: How Sure Are We That Economic Time Series Have a Unit
  Root?, \emph{Journal of Econometrics}, \bold{54}, 159--178.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{\code{\link{ur.kpss-class}}}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
gnp.l <- log(gnp)
kpss.gnp <- ur.kpss(gnp.l, type="tau", lags="short")
summary(kpss.gnp)
}
\keyword{regression}

\eof
\name{ur.pp-class}
\docType{class}
\alias{ur.pp-class}
\title{Representation of class `ur.pp'}
\description{
  This class contains the relevant information by applying the Phillips
  \& Perron unit root test to a time series.
}
\section{Slots}{
  \describe{
    \item{\code{y}:}{Object of class \code{"vector"}: The time series to
      be tested.}
    \item{\code{type}:}{Object of class \code{"character"}: Test type of
      Z statistic, either \code{"Z-alpha"} or \code{"Z-tau"}.}
    \item{\code{model}:}{Object of class \code{"character"}: The type of
      the deterministic part, either \code{"constant"} or
      \code{"trend"}. The latter includes a constant term, too.}
    \item{\code{lag}:}{Object of class \code{"integer"}: Number of lags
      for error correction.}
    \item{\code{cval}:}{Object of class \code{"matrix"}: Critical values
      at the 1\%, 5\% and 10\% level of significance.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: Value of
      the test statistic.}
    \item{\code{testreg}:}{Object of class \code{"ANY"}: The summary
      output of the test regression.}
    \item{\code{auxstat}:}{Object of class \code{"matrix"}: Test
      statistic(s) of the deterministic part.}
    \item{\code{res}:}{Object of class \code{"vector"}: The residuals of
      the test regression.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e} `Phillips-Perron'.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ur.sp")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but critical value and summary of
      test regression added.}
    \item{\code{plot}:}{Diagram of fit plot, residual plot and their
      acfs' and pacfs'.}
  }
}
\references{
  Phillips, P.C.B. and Perron, P. (1988), Testing for a unit root in
  time series regression, \emph{Biometrika},  \bold{75(2)}, 335--346.

  MacKinnon, J.G. (1991), Critical Values for Cointegration Tests,
  \emph{Long-Run Economic Relationships}, eds. R.F. Engle and
  C.W.J. Granger, London, Oxford, 267--276.
  
  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)' and
  \url{http://www.econ.ucsd.edu/papers/files/90-4.pdf}.
}
\seealso{
  \code{\link{ur.pp}} and \code{\link{urca-class}}
}
\keyword{classes}

\eof
\name{ur.pp}
\alias{ur.pp}
\title{Phillips \& Perron Unit Root Test}
\description{
  Performs the Phillips \& Perron unit root test. Beside the Z
  statistics Z-alpha and Z-tau, the Z statistics for the deterministic
  part of the test regression are computed, too.
}
\usage{
ur.pp(x, type = c("Z-alpha", "Z-tau"), model = c("constant", "trend"),
      lags = c("short", "long"), use.lag = NULL)
}
\arguments{
  \item{x}{Vector to be tested for a unit root.}
  \item{type}{Test type, either \code{"Z-alpha"} or \code{"Z-tau"}.}
  \item{model}{Determines the deterministic part in the test regression.}
  \item{lags}{Lags used for correction of error term.}
  \item{use.lag}{Use of a different lag number, specified by the user.}
}
\details{
  The function \code{ur.pp()} computes the Phillips \& Perron test. For
  correction of the error term a Bartlett window is used. 
}
\value{
  An object of class \code{`ur.pp'}.
}
\references{
  Phillips, P.C.B. and Perron, P. (1988), Testing for a unit root in
  time series regression, \emph{Biometrika},  \bold{75(2)}, 335--346.

  MacKinnon, J.G. (1991), Critical Values for Cointegration Tests,
  \emph{Long-Run Economic Relationships}, eds. R.F. Engle and
  C.W.J. Granger, London, Oxford, 267--276.
  
  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)' and
  \url{http://www.econ.ucsd.edu/papers/files/90-4.pdf}.
}
\seealso{\code{\link{ur.pp-class}}.}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
pp.gnp <- ur.pp(gnp, type="Z-tau", model="trend", lags="short")
summary(pp.gnp)
}
\keyword{regression}

\eof
\name{ur.sp-class}
\docType{class}
\alias{ur.sp-class}

\title{Representation of class `ur.sp'}
\description{
  This class contains the relevant information by applying the Schmidt
  \& Phillips unit root test to a time series.
}
}
\section{Slots}{
  \describe{
    \item{\code{y}:}{Object of class \code{"vector"}: The time series to
      be tested.}
    \item{\code{type}:}{Object of class \code{"character"}: Test type,
      \code{"rho"} or \code{"tau"} test statistic.}
    \item{\code{polynomial}:}{Object of class \code{"integer"}:
      Deterministic trend specification}
    \item{\code{signif}:}{Object of class \code{"numeric"}: Critical values.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: Value of
      the test statistic.}
    \item{\code{cval}:}{Object of class \code{"numeric"}: The critical
      values, depending on \code{"signif"}, \code{"polynomial"} and the
      sample size.}
    \item{\code{res}:}{Object of class \code{"vector"}: The residuals of
      the test regression.}
    \item{\code{testreg}:}{Object of class \code{"ANY"}: The summary
      output of the test regression.}
    \item{\code{test.name}:}{Object of class \code{"character"}: The
      name of the test, \emph{i.e.} `"Schmidt \& Phillips'.}
  }
}
\section{Extends}{
Class \code{`"urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ur.sp")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic.}
    \item{\code{summary}:}{like show, but critical value and summary of
      test regression added.}
    \item{\code{plot}:}{Diagram of fit plot, residual plot and their
      acfs' and pacfs'.}
  }
}
\references{
  Schmidt, P. and Phillips, P.C.B. (1992), LM Test for a Unit Root in
  the Presence of Deterministic Trends, \emph{Oxford Bulletin of Economics and
  Statistics}, \bold{54(3)}, 257--287.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{
  \code{\link{ur.sp}} and \code{\link{urca-class}}. 
}
\keyword{classes}

\eof
\name{ur.sp}
\alias{ur.sp}
\title{Schmidt \& Phillips Unit Root Test}
\description{
  Performs the Schmidt \& Phillips unit root test, where under the Null
  and Alternative Hypothesis the coefficients of the deterministic
  variables are included.
}
\usage{
ur.sp(y, type = c("tau", "rho"), pol.deg = c(1, 2, 3, 4),
      signif = c(0.01, 0.05, 0.1))
}
\arguments{
  \item{y}{Vector to be tested for a unit root.}
  \item{type}{Test type, either \code{tau} or \code{rho} test.}
  \item{pol.deg}{Degree of polynomial in the test regression.}
  \item{signif}{Significance level for the critical value of the test statistic.}
}
\details{
  Under the Null and the Alternative hypothesis the coefficients of the
  deterministic part of the test regression are included. Two test types
  are available: the \code{rho}-test and the \code{tau}-test.
  Both test are extracted from the LM principle.
}
\value{
  An object of class \code{"ur.sp"}.
}
\references{
  Schmidt, P. and Phillips, P.C.B. (1992), LM Test for a Unit Root in
  the Presence of Deterministic Trends, \emph{Oxford Bulletin of Economics and
  Statistics}, \bold{54(3)}, 257--287.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{\code{\link{ur.sp-class}}}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
sp.gnp <- ur.sp(gnp, type="tau", pol.deg=1, signif=0.01)
summary(sp.gnp)
}
\keyword{regression}

\eof
\name{ur.za-class}
\docType{class}
\alias{ur.za-class}

\title{Representation of class `ur.za'}
\description{
  This class contains the relevant information by applying the Zivot \& Andrews
  unit root test to a time series.}
\section{Slots}{
  \describe{
    \item{\code{y}:}{Object of class \code{"vector"}: The time series to
      be tested.}
    \item{\code{model}:}{Object of class \code{"character"}: The model
      to be used, \emph{i.e.} intercept, trend or both}
    \item{\code{lag}:}{Object of class \code{"integer"}: The highest
      number of lags to include in the test regression.}
    \item{\code{teststat}:}{Object of class \code{"numeric"}: The t-statistic.}
    \item{\code{cval}:}{Object of class \code{"vector"}: Critical values
      at the 1\%, 5\% and 10\% level of significance.}
    \item{\code{bpoint}:}{Object of class \code{"integer"}: The
      potential break point.}
    \item{\code{tstats}:}{Object of class \code{"vector"} The
      t-statistics of the rolling regression.}
    \item{\code{res}:}{Object of class \code{"vector"} The residuals of
      the test regression.}
    \item{\code{test.name}:}{Object of class \code{"character"} The name
      of the test, \emph{i.e.} `Zivot \& Andrews'.}
    \item{\code{testreg}:}{Object of class \code{"ANY"} The summary
      output of the test regression.}
  }
}
\section{Extends}{
Class \code{`urca'}, directly.
}
\section{Methods}{
  Type \code{showMethods(classes="ur.za")} at the R prompt for a
  complete list of methods which are available for this class.
  
  Useful methods include
  \describe{
    \item{\code{show}:}{test statistic and critical values.}
    \item{\code{summary}:}{like show, but summary of test regression
      added.}
    \item{\code{plot}:}{plot of recursive t-statistics.}
  }
}
\references{
  Zivot, E. and Andrews, Donald W.K. (1992), Further Evidence on the
  Great Crash, the Oil-Price Shock, and the Unit-Root Hypothesis,
  \emph{Journal of Business \& Economic Statistics}, \bold{10(3)},
  251--270.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{
  \code{\link{ur.za}} and \code{\link{urca-class}}. 
}
\keyword{classes}

\eof
\name{ur.za}
\alias{ur.za}
\title{Zivot \& Andrews Unit Root Test}
\description{
  Performs the Zivot \& Andrews unit root test, which allows a
  break at an unknown point in either the intercept, the linear
  trend or in both.
}
\usage{
ur.za(y, model = c("intercept", "trend", "both"), lag)
}
\arguments{
  \item{y}{Vector to be tested for a unit root.}
  \item{model}{Specification if the potential break occured in either the intercept, the linear trend or in both.}
  \item{lag}{The highest number of lagged endogenous differenced variables to be included in the test regression}
}
\details{
  This test is based upon the recursive estimation of a test
  regression. The test statistic is defined as the minimum t-statistic
  of the coeffcient of the lagged endogenous variable. 
}
\value{
  An object of class \code{`ur.za'}.
}
\references{
  Zivot, E. and Andrews, Donald W.K. (1992), Further Evidence on the
  Great Crash, the Oil-Price Shock, and the Unit-Root Hypothesis,
  \emph{Journal of Business \& Economic Statistics}, \bold{10(3)},
  251--270.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{\code{\link{ur.za-class}}}
\examples{
data(nporg)
gnp <- na.omit(nporg[, "gnp.r"])
za.gnp <- ur.za(gnp, model="both", lag=2)
summary(za.gnp)
}
\keyword{regression}

\eof
\name{urca-class}
\docType{class}
\alias{urca-class}

\title{Class `urca'. Parent of classes in package `urca'}
\description{
  This class is the parent class of the specific classes
  designed holding the test specific information of the unit
  root/cointegration tests.
}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("urca", ...)},
but most often the slot \code{test.name} is set by calling one of the
unit root/cointegration functions, \emph{e.g} \code{ur.za}.
}
\section{Slots}{
  \describe{
    \item{\code{test.name}:}{Object of class \code{"character"}. The
      name of the unit root/cointegration test.}
  }
}
\section{Methods}{
No methods defined with class `urca'.
}
\seealso{
  \code{\link{ur.ers-class}}, \code{\link{ur.kpss-class}},
  \code{\link{ca.jo-class}}, \code{\link{ca.po-class}},
  \code{\link{ur.pp-class}}, \code{\link{ur.sp-class}}
  and \code{\link{ur.za-class}}.
}
\keyword{classes}

\eof
\name{.spcv}
\alias{.spcv}
\title{Critical values for Schmidt \& Phillips Unit Root Test}
\description{
  This function is an internal function and is called by
  \code{ur.sp}. It computes the critical value of the Schmidt \&
  Phillips test, given a level of significance, the polynomial degree of
  the test regression, the test type and the sample size.
}
\usage{
.spcv(obs, type = c("tau", "rho"), pol.deg = c(1, 2, 3, 4),
      signif = c(0.01, 0.025, 0.05, 0.1))
}
\arguments{
  \item{obs}{The sample size.}
  \item{type}{The test type.}
  \item{pol.deg}{The polynomial degree.}
  \item{signif}{The significance level.}
}
\value{
  The critical value of the test.
}
\references{
  Schmidt, P. and Phillips, P.C.B. (1992), LM Test for a Unit Root in
  the Presence of Deterministic Trends, \emph{Oxford Bulletin of Economics and
  Statistics}, \bold{54(3)}, 257--287.

  Download possible at: \url{http://cowles.econ.yale.edu/}, see rubric
  'Discussion Papers (CFDPs)'.
}
\seealso{\code{\link{ur.sp}}.}
\keyword{regression}

\eof
