\name{accessC}
\alias{accessC}
\title{Get Smoothed Data from Wavelet Structure}
\description{
  The smoothed and original data from a wavelet decomposition structure
  (returned from \code{\link{wd}}) are packed into a single vector in that
  structure.  This function extracts the data corresponding to a
  particular resolution level.
}
\usage{
accessC(wd.obj, level = wd.obj$nlevels, boundary=FALSE)
}
\arguments{
  \item{wd.obj}{
    wavelet decomposition structure from which you wish to extract the
    smoothed or original data if the structure is from a wavelet decomposition,
    or the reconstructed data if the structure is from a wavelet reconstruction.
  }
  \item{level}{
    the level that you wish to extract.  By default, this is the level with
    most detail (in the case of structures from a decomposition this is the
    original data, in the case of structures from a reconstruction this is the
    top-level reconstruction).
  }
  \item{boundary}{logical;
    if \code{TRUE} then all of the boundary correction values
    will be returned as well (note: the length of the returned vector
    may not be a power of 2).\cr
    If \code{boundary} is false, then just the coefficients will be returned.

    If the decomposition (or reconstruction)
    was done with periodic boundary conditions, this option has no effect.}
}
\value{A vector of the extracted data.
}
\details{
The wd (wr) function produces a wavelet decomposition (reconstruction)
structure.

For decomposition, the top level contains the original data, and
subsequent lower levels contain the successively smoothed data.
So if there are \eqn{2^m} original data points, there will be m+1 levels
indexed 0,1,\dots{},m. So

>  \code{accessC(wd.obj, level=m)}

pulls out the original data, as does

>  \code{accessC(wd.obj)}

To get hold of lower levels just specify the level that you're interested
in, e.g.

>  \code{accessC(wd.obj, level=2)}

gets hold of the second level.

For reconstruction, the top level contains the ultimate step in the
Mallat pyramid reconstruction algorithm, lower levels are intermediate
steps.

The need for this function is a consequence of the pyramidal structure
of Mallat's algorithm and the memory efficiency gain achieved by
storing the pyramid as a linear vector. AccessC obtains information about
where the smoothed data appears from the fl.dbase component of
wd.obj, in particular the array \code{fl.dbase$first.last.c} which
gives a complete specification of index numbers and offsets for
\code{wd.obj$C}.

Note that this and the \code{\link{accessD}} function only work with
objects of class \code{wd}, see \code{\link{wd.object}}.

Further note that this function only gets information from 'wd' class
objects. To put coefficients etc. into \code{wd} structures you have to
use the "putC" function.
}
\seealso{For background information, \code{\link{wr}} and
  \code{\link{wd}}.  Further, \code{\link{accessD}},
  \code{\link{filter.select}}, \code{\link{plot.wd}},
  \code{\link{threshold}}, \code{\link{putC}}, \code{\link{putD}}.}
}
\examples{
## Get the 3rd level of smoothed data from a decomposition
accessC(wd(rnorm(2^7)), level=3)

example(wd)
str(accessC(wds))
## Plot the time series from a reconstruction
plot.ts(accessC(wr(wds, return.obj = TRUE)))
}
\keyword{manip}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{accessD}
\alias{accessD}
\title{Get wavelet expansion coefficients from wavelet structure.}
\description{
  The coefficients from a wavelet expansion in a wavelet decomposition
  structure (returned from \code{\link{wd}} or \code{\link{wr}}) are
  packed into a single vector in that structure.
  This function extracts the coefficients corresponding to a particular
  resolution level.
}
\usage{
accessD(wd.obj, level, boundary=FALSE)
}
\arguments{
\item{wd.obj}{Wavelet decomposition structure from which you wish to
  extract the expansion coefficients.}
\item{level}{The level that you wish to extract. If the "original" data has
  \eqn{2^m} data points then there are m possible levels that you could want
  to access, indexed by 0,1,\dots{},(m-1).}
\item{boundary}{
  If this argument is TRUE then all of the boundary correction values
  will be returned as well (note: the length of the returned vector
  may not be a power of 2). If boundary is false, then just the
  coefficients will be returned. If the decomposition (or reconstruction)
  was done using periodic boundary handling then this option has no
  effect.}
}
\value{
  A vector of the extracted coefficients.
}
\details{
  The wd (wr) function produces a \bold{w}avelet \bold{d}ecomposition
  (\bold{r}econstruction) structure.

  The need for this function is a consequence of the pyramidal structure
  of Mallat's algorithm and the memory efficiency gain achieved by
  storing the pyramid as a linear vector.  AccessD obtains information about
  where the coefficients appear from the fl.dbase component of
  wd.obj, in particular the array \code{fl.dbase$first.last.d} which
  gives a complete specification of index numbers and offsets for
  \code{wd.obj$D}.

  Note that this function and accessC only work on objects of class
  \code{wd}. Also, you have to use \code{\link{putD}} to put wavelet
  coefficients into a wd object.
}
\seealso{
  \code{\link{wr}} and \code{\link{wd}} for background information;
  \code{\link{accessC}}, \code{\link{filter.select}},
  \code{\link{threshold}}, \code{\link{putC}}, \code{\link{putD}}.}
}
\examples{
example(wd)

## Get the 3rd level coefficients of a decomposition
accessD(wds, level=3)

## Do a qqnorm plot to assess the normality of some coefficients
qqnorm(accessD(wds, level=8))
}
\keyword{manip}


\eof
\name{compress}
\alias{compress}
\title{Compression - Generic function (wavelet)}
\usage{
compress(x, \dots)
}
\arguments{
  \item{x}{an \R object.}
  \item{\dots}{methods may have additional arguments.}
}
\description{
  Compresses \code{x} (typically by removing zeroes).
  This is the generic function.
}
\details{Functions with names beginning in \code{compress.} will be
  methods for this function,

  See their individual help pages, e.g.,
  \code{\link{compress.imwd}}, \code{\link{compress.default}}, for operation.
}
\value{a compressed version of the supplied argument.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{\code{\link{threshold}}, \code{\link{uncompress}}.}
\keyword{manip}
\keyword{utilities}

\eof
\name{compress.default}
\alias{compress.default}
\title{Compress a (Wavelet) Vector}
\description{
  Compression of a vector by removal of zero elements.
}
\usage{
\method{compress}{default}(x, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{numeric vector to compress}
  \item{verbose}{logical; if true then report on compression activity.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
  An object of class \code{compressed} if compression took place, otherwise a
  an object of class \code{uncompressed}
}
\details{
Images are large objects.  Thresholded 2d wavelet objects (imwd) are also
large, but many of their elements are zero.  Compress takes a vector,
decides whether compression is necessary and if it is makes an object
of class \code{compressed} containing the nonzero elements and their position
in the original vector.

The decision whether to compress the vector or not depends on two
things, first the number of non-zero elements in the vector (r, say),
and second the length of the vector (n, say). Since the position and value
of the non-zero elements is stored we will need to store 2r values
for the non-zero elements. So compression takes place if 2r < n.

This function is a method for the generic function
\code{compress()} for class \code{default}.
It can be invoked by calling \code{compress(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{compress.default(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\section{BUGS}{
Sometimes the compressed object can be larger than the original.
This usually only happens for small objects, so it doesn't really
matter.
}
\seealso{
  \code{\link{compressed.object}}, \code{\link{uncompressed.object}};
  \code{\link[base]{rle}} for another kind of compression.
}
}
\examples{
#
# Compress a vector with lots of zeroes
#
str(compress(c(rep(0,100),99)))
## List of 3
##  $ position       : int 101
##  $ values         : num 99
##  $ original.length: int 101
##  - attr(*, "class")= chr "compressed"

## Try to compress a vector with not many zeroes
compress(1:10)
## (uncompressed)
}
\keyword{manip}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{compress.imwd}
\alias{compress.imwd}
\alias{uncompress.imwdc}
\title{Compression and Decompression for `imwd' Objects}
\description{
  Compresses or decompresses a (thresholded) imwd object
  (\code{\link{imwd.object}}) by removing or padding with zero elements.
}
\usage{
\method{compress}{imwd}(x, verbose = getOption("verbose"), \dots)
\method{uncompress}{imwdc}(x, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{class \code{imwd} object to compress or class \code{imwdc} to
    decompress.}
  \item{verbose}{logical; if true then report on compression activity.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
  An object of class "imwdc" representing the compressed imwd object,
  see \code{\link{imwdc.object}}.
}
\details{
Thresholded imwd objects are usually very large and contain many zero
elements. This function compresses these objects into smaller "imwdc"
objects by using the \code{\link{compress.default}} function and
removing the zeroes.

\code{compress.imwd} is a method for the generic function
\code{\link{compress}} for class \code{imwd}.
The user shouldn't need to use this function directly.

\code{uncompress.imwd} uncompresses a \code{imwdc} object back into a
\code{imwd} one.  It is a method for the generic function
\code{uncompress()} for class \code{imwdc}.
It can be invoked by calling \code{uncompress(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{uncompress.imwdc(x)} regardless of the class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{
  \code{\link{compress.default}}, \code{\link{uncompress.default}},
  \code{\link{threshold}},
  \code{\link{imwd.object}},\code{\link{imwdc.object}}.
}
\examples{
%% add examples!!
}
\keyword{manip}


\eof
\name{compressed.object}
\alias{compressed.object}
\title{Compressed (Wavelet) Object}
\description{
  These are objects of class \code{"compressed"}.
  They are lists with components representing a vector, see below.
}
\section{GENERATION}{
By \code{\link{compress.default}}
}
\section{METHODS}{
The \code{"compressed"} class of objects has methods for the following generic
functions:
\code{uncompress.default}
}
\value{
The following components must be included in a legitimate
\code{compressed} object.
\item{position}{ a vector containing the position of the non-zero
  elements in the original vector.}
\item{values}{a vector, corresponding position-by-position to the
  position vector, containing the values of the original vector.}
\item{original.length}{the original length of the uncompressed vector.}
}
\details{
The \code{\link{compress.default}} function tries to compress a vector.
If it can it produces an object of class \code{compressed}, otherwise an
object of class \code{uncompressed}.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{
\code{\link{compress}}.
}
\keyword{classes}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{dof}
\alias{dof}
\title{Degrees of Freedom of Wavelet}
\usage{
dof(wd)
}
\description{Compute degrees of freedom for wavelet thresholding model.}
\arguments{
  \item{wd}{The wavelet object}
}
\value{The number of degrees of freedom.
}
\details{
  The wavelet thresholding is a non-linear transform and \code{dof} returns
  the approximate number of degrees of freedom.
}
\references{see \code{\link{wd}} for a list.
}
\seealso{
\code{\link{threshold}}
}
\examples{
example(wd)
threshy <- threshold(wds)

# Compare degrees of freedom
dof(wds)	# 1024
dof(threshy)    # about 23
}
\keyword{models}


\eof
\name{draw}
\alias{draw}
\title{Draw Wavelets - Generic function}
\description{
  Draw a picture of a wavelet.  This is a generic function.
  There are methods for 1D and 2D wavelets, see below.
}
\usage{
draw(x, \dots)
}
\arguments{
  \item{x}{an \R object.}
  \item{\dots}{methods may have additional arguments.
    In addition, the high-level graphics control arguments
    described under \code{\link{par}} and the arguments to
    \code{\link{title}} may be supplied to this function.
  }
}
\section{Side Effects}{
  a plot is created on the current graphics device.
}
\seealso{\code{\link{draw.default}}, \code{\link{draw.wd}},
  \code{\link{draw.imwd}},\code{\link{draw.imwdc}} for examples and
  \code{\link{wd.object}} for background.
}
\keyword{hplot}
% Converted by Sd2Rd version 0.3-3.

\eof
% $Id: draw.default.Rd,v 1.6 2001/12/17 07:27:09 maechler Exp maechler $
\name{draw.default}
\alias{draw.default}
\title{Draw a Picture of a Wavelet}
\description{
  Draws pictures of basic (Daubechies') wavelets in the wavethresh package.
}
\usage{
\method{draw}{default}(x = filter.number, filter.number = 2,
     family = c("DaubExPhase", "DaubLeAsymm"),
     resolution = 1024, verbose = FALSE, plot = TRUE,
     main = paste("Wavelet Picture", if(enhance) " (Enhanced)", sep=""),
     sub = zwd$filter$name, xlab = "x", ylab = expression(psi(x)), dimension = 1,
     twodplot = persp, enhance = TRUE, efactor = 0.05, \dots)
}
\arguments{
  \item{x, filter.number}{integer; number of the filter in the wavelet
    family specified.  The range of possible numbers depends on the family.}
  \item{family}{character, the family of wavelets to use, defaults to
    the first of the list shown above.}
  \item{resolution}{integer specifying the resolution to use in drawing
    pictures.  A higher resolution gives a better quality picture but
    will take longer to draw (especially in two dimensions).}
  \item{verbose}{logical, if \code{TRUE}, report the progress of drawing.}
  \item{plot}{logical, if true then plot the
    appropriate wavelet on the current active graphics device, otherwise
    the information about the wavelet is returned.}
  \item{main, sub}{main and sub-title for the plot, see \code{\link{title}}.}
  \item{xlab, ylab}{labels for the x and y axis.}
  \item{dimension}{
    If this is 1, then a one-dimensional version of the wavelet is drawn,
    if this is 2, then a two-d picture is drawn with the \code{twodplot} function.
  }
  \item{twodplot}{function such as \code{\link{persp}} that can do
    something interesting with a matrix.}
  \item{enhance}{logical; with Daubechies' wavelets their effective
    support is much less than their actual support.
    If \code{enhance} is true, the function tries to draw
    the wavelet on its effective support.}
  \item{efactor}{numeric, defining the effective support.
    Define z0 to be efactor times the maximum absolute value of the
    wavelet w(x). Define the set \eqn{A = \{ x: |w(x)| > z0 \}}, and the
    effective support is the smallest interval (in 1D, square in 2D)
    containing A.
  }
  \item{\dots}{further arguments to \code{plot} or \code{twodplot}.}
}
\value{
  If \code{plot} is false then no actual drawing is done, however, a list is
  returned with \code{dimension + 1} components:
  If \code{dimension} is one, the list has components x and y, representing the
  domain of definition (or effective support) of the wavelet and the
  value of the wavelet at x.
  \cr
  If \code{dimension == 2}, the list has three components x,y and z.
}
\section{Side Effects}{
  a picture of a wavelet is drawn on the currently active graphics device
  if plot=TRUE.
}
\details{
For Daubechies' compactly supported wavelets there is no known closed form
expression. However, it is possible to make use of the inverse wavelet
transform to draw pictures of the wavelets. The idea is to set
all but one wavelet coefficient equal to zero and that one coefficient
equal to one in a wavelet expansion.
Then the inverse transform (reconstruction) is applied to the expansion
and a picture of one wavelet is produced.

A method similar to the one we present here is presented in Daubechies (1992)
in Section~6.5 on the cascade algorithm.

The principle is simple, but the implementation to get good pictures is
surprisingly tricky. Ideally you want to put in the non-zero coefficient
at the lowest resolution level as possible, this will give you as much
detail as possible. However, the support of some of the large-scale
wavelets is greater than the union of the supports of all the high-resolution
small-scale wavelets and so it may not be possible to draw a complete wavelet.

This function analyses the supports of the wavelets at different levels
and finds a coefficient to set to one by choosing the wavelet at the lowest
possible resolution and whose support is within the union of supports of
the highest resolution wavelets. If there is
a choice of such wavelets, then the middle-most one is selected.
}
\section{RELEASE}{
  Release 2.2
  Copyright Guy Nason 1993
}
\seealso{\code{\link{draw}}, \code{\link{draw.wd}} etc.
}
\examples{
op <- par(mfrow=c(5,2), oma = c(0,0, 4, 0),
          mgp = c(1.2, .8, 0), mar = .1 + c(4,4, .5,1))
for(fn in 1:10) {
   draw.default(filter.number= fn, col = "blue", main = NULL, xlab= "")
   abline(h=0,v=0, lty=3, lwd=.5, col = "gray")
}
mtext(paste("draw.default(*, family = '",formals(draw.default)$family[[2]],"')"),
      side = 3, line = 1, outer = TRUE,
      cex = par("cex.main"), font = par("font.main"))
par(op)

# Draw a 2-dimensional Daubechies least asymmetric wavelet
draw.default(filter.number=6, family="DaubLeAsymm", dim=2, resolution=128)
}
\keyword{hplot}
% Converted by Sd2Rd version 0.3-3.

\eof
%% $Id: draw.wd.Rd,v 1.8 2001/12/17 07:27:05 maechler Exp maechler $
\name{draw.wd}
\alias{draw.wd}
\alias{draw.imwd}
\alias{draw.imwdc}
\title{Draw Wavelet Corresponding to Wavelet Object}
\description{
  Draws picture of the wavelet associated with (1D or 2D) wavelet decomposition
  object \code{wd}, as opposed to drawing \code{wd} itself (which is
  achieved by the corresponding \code{\link{plot}} method, such as
  \code{\link{plot.wd}}).
}
\usage{
\method{draw}{wd}(x, \dots)
%% not using \method for these: text would become confusing:
draw.imwd (x, resolution = 128, \dots)
draw.imwdc(x, resolution = 128, \dots)
}
\arguments{
  \item{x}{Object of class \code{wd}, \code{imwd}, or \code{imwdc},
    typically from a wavelet decomposition using
    the \code{\link{wd}}, \code{\link{imwd}}, or
    \code{\link{threshold}} function.}
  \item{resolution}{integer specifying the resolution to use in drawing.
    A higher resolution gives a better quality picture but will take
    longer to draw (especially in two dimensions).}
  \item{\dots}{further arguments to \code{\link{draw.default}}.}
}
\details{
  Sometimes it is more convenient to use this function rather than
  \code{\link{draw.default}}, since you want a picture of the wavelet
  that did your decomposition.
}
\seealso{\code{\link{draw.default}}, \code{\link{wd}}.
}
\examples{
example(wd)
# Draw a picture of the wavelets that were used
draw(wds); abline(h=0,v=0, lty=3, lwd=.5)

example(imwd)
# Draw a picture of the 2D wavelet that used
draw(imwdL)
}
\keyword{hplot}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{filter.select}
\alias{filter.select}
\title{Wavelet Filter Coefficients}
\usage{
filter.select(filter.number,
              family = c("DaubExPhase", "DaubLeAsymm"), constant = 1)
}
\description{
  This function stores the filter coefficients necessary for doing a discrete
  wavelet transform (and its inverse).
}
\arguments{
  \item{filter.number}{integer, selecting the desired filter; takes a
    value dependent upon the family that you select.}
  \item{family}{character string, selecting the type of wavelet.}
  \item{constant}{numeric; is applied as a multiplier to all the
    coefficients.  It can be a vector, and so you can adapt the filter
    coefficients to be whatever you want. (This is feature of negative
    utility.)}
}
\value{
  A list is returned with four components describing the filter:
  \item{H}{A vector containing the filter coefficients.}
  \item{name}{A character string containing the name of the filter.}
  \item{family}{A character string containing the family of the filter.}
  \item{filter.number}{The filter number used to select the filter.}
}
\details{
  This function contains three types of filter.  Two types can be
  selected with family set to \code{"DaubExPhase"} (as per default),
  these wavelets are the Haar wavelet (selected by
  \code{filter.number=1} within this family) and Daubechies ``extremal
  phase'' wavelets selected by filter.numbers ranging from 2 to 10 as
  proposed in Daubechies (1988).  Setting family to \code{"DaubLeAsymm"}
  gives you Daubechies least asymmetric wavelets also from Daubechies
  (1988), but here the filter number ranges from 4 to 10.

  For Daubechies wavelets, \code{filter.number} corresponds to the
  \eqn{N} of that paper, the wavelets become more regular as the filter.number
  increases, but they are all of compact support.

  This function is currently used by \code{\link{wr}} and \code{\link{wd}}
  in decomposing and reconstructing, however you may wish to look at the
  coefficients.
}
\section{NOTE}{
  The filter coefficients should always sum to
  \eqn{\sqrt{2}}{sqrt(2)}. This is a useful check on their validity.
}
\seealso{
  \code{\link{wd}}, \code{\link{wr}}, for background information;
  \code{\link{accessC}}, \code{\link{accessD}},
  \code{\link{imwd}}, \code{\link{imwr}}, \code{\link{threshold}}.
}
\examples{
## look at the filter coefficients for N=2 :
str(f2 <- filter.select(2))
##- List of 4
##-  $ H            : num [1:4]  0.483  0.837  0.224 -0.129
##-  $ name         : chr "Daub cmpct on ext. phase N=2"
##-  $ family       : chr "DaubExPhase"
##-  $ filter.number: num 2

all.equal(sum(f2 $H), sqrt(2))# TRUE
}
\keyword{math}
\keyword{smooth}


\eof
\name{first.last}
\alias{first.last}
\title{Build a first/last database for wavelet transforms}
\description{
This function is not intended for user use, but is used by various
functions involved in computing and displaying wavelet transforms.
}
\usage{
first.last(LengthH, DataLength, bc = c("periodic","symmetric"))
}
\arguments{
  \item{LengthH}{length of the filter used to produce a wavelet decomposition.}
  \item{DataLength}{length of the data before transforming; must be a
    power of 2, say \eqn{2^m}.}
  \item{bc}{character string, determining how the boundaries of the the
    function are to be handled; one of \code{"periodic"} (default) or
    \code{"symmetric"}.}
}
\value{
A first/last database structure, a list with the
following components:

\item{first.last.c}{
  A (m+1) x 3 matrix.  The first column specifies the real index of the first
  coefficient of the smoothed data at a level, the 2nd column is the
  real index of the last coefficient, the last column specifies the offset
  of the first smoothed datum at that level.  The offset is used by the
  C code to work out where the beginning of the sequence is within a packed
  vector of the pyramid structure.  The first and 2nd columns can be used
  to work out how many numbers there are at a level.

  If \code{bc="periodic"} then
  the pyramid is a true power of 2 pyramid, that is it starts with a
  power of 2, and the next level is half of the previous.
  If \code{bc="symmetric"} then the pyramid is nearly exactly a power of 2, but
  not quite, see the Details section for why this is so.
}
\item{ntotal}{The total number of smoothed data/original data points.}
\item{first.last.d}{
  A mx3 matrix. As for first.last.c but for the wavelet coefficients packed
  as the D component of a wavelet structure.
}
\item{ntotal.d}{The total number of wavelet coefficients.}
}
\details{
  Suppose you begin with \eqn{2^m=2048} coefficients.  At the next level you
  would expect 1024 smoothed data coefficients, and 1024 wavelet coefficients,
  and if \code{bc="periodic"} this is indeed what happens.
  However, if \code{bc="symmetric"} you actually need more than 1024 (as
  the wavelets extend over the edges). The first/last database keeps track
  of where all these "extras" appear and also where they are located in
  the packed vectors C and D of pyramidal coefficients within wavelet
  structures.

  For example, given a first.last.c row of \cr
  \code{-2 3 20}\cr

  The actual coefficients would be\cr
  \eqn{c_{-2}, c_{-1}, c_{0}, c_{1}, c_{2}, c_{3}}{%
    c_\{-2\}, c_\{-1\}, c_\{0\}, c_\{1\}, c_\{2\}, c_\{3\} \
    (in LaTeX notation where _\{x\} denotes subscript x)}.

  In other words, there are 6 coefficients, starting at -2 and ending at 3,
  and the first of these (\eqn{c_{-2}}{c\{-2\}}) appears at an offset of
  20 from the beginning of the \code{* $ C} component vector of the
  wavelet structure.

  You can ``do'' first.last in your head for periodic boundary handling,
  but for more general boundary treatments (e.g. symmetric) \code{first.last}
  is indispensable.
}
\section{RELEASE}{
  Release 2.2
  Copyright Guy Nason 1993
}
\references{
  The numbers in first/last databases were worked out from inequalities
  derived from:

  Daubechies, I. (1988).
  \emph{Orthonormal bases of compactly supported wavelets};
  Communications on Pure and Applied Mathematics, \bold{41}, 909-996.
}
\seealso{
\code{\link{wr}}, \code{\link{accessC}}, \code{\link{accessD}},
\code{\link{filter.select}}, \code{\link{threshold}}, \code{\link{wd}},
\code{\link{imwd}}, \code{\link{imwr}}.
}
\section{BUGS}{
None, I hope. However, with hindsight, I should have implemented the
periodic version first.  The symmetric boundary stuff confused a lot
of people (including me)!
}
\examples{
## If you're twisted then you may just want to look at one of these.

first.last(length(filter.select(2)), 64)
}
\keyword{misc}
\keyword{utilities}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{imwd}
\alias{imwd}
\title{2D Discrete Wavelet Transform (Image W. Decomposition)}
\description{
  This function performs the decomposition stage of Mallat's pyramid algorithm
  i.e. the discrete wavelet transform for images.
}
\usage{
imwd(image, filter.number=2,
     bc = c("periodic", "symmetric"), verbose = getOption("verbose"))
}
\arguments{
\item{image}{
  numeric square matrix containing the image.  The number of rows in the
  image must be a power of 2.  Since the matrix is square, this is also
  the number of columns.}
\item{filter.number}{
  integer; the filter that you wish to use to decompose the
  function. The filters are obtained from the
  \code{\link{filter.select}} function and are the compactly supported
  orthonormal wavelets as described in Daubechies, I. % <<<< ??
}
\item{bc}{
boundary treatment. The periodic (default) treatment causes the decomposition
to act as if the function you are trying to decompose is periodic (on it's
own interval). The other option is symmetric, which used to be the
default. This causes the decomposition to act as if the function extended
by symmetric reflection at each end.
}
\item{verbose}{logical; if true then informative messages are printed
  whilst the computations are performed.}
}
\value{
  An object of class \code{imwd}, a list containing the wavelet coefficients
  (see \code{\link{imwd.object}}).
}
\details{
The 2D algorithm is essentially the application of many 1D filters.
First, the columns are attacked with the smoothing (H) and bandpass
(G) filters, and the rows of each of these resultant images are
attacked again with each of G and H, this results in 4 images.
Three of them, GG, GH, and HG correspond to the highest resolution
wavelet coefficients. The HH image is a smoothed version of the
original and can be further attacked in exactly the same way as the
original image to obtain GG(HH), GH(HH), and HG(HH), the wavelet
coefficients at the second highest resolution level and HH(HH)
the twice-smoothed image, which then goes on to be further attacked.

After each attack the dimension of the images is halved.  After many
attacks you will obtain four real numbers, one of which correspond to
the image smoothed many times.

Exact details of the algorithm are to be found in Mallat 1989.
}
\seealso{\code{\link{wd}}, for background information;
  \code{\link{imwr}} for reconstruction, \code{\link{plot.imwd}},
  \code{\link{draw.imwd}}.
}
\examples{
# Do a decomposition of an image
#
data(lennon)
imwdL <- imwd(lennon)
# Look at the coefficients --> example(plot.imwd)
}
\keyword{smooth}
\keyword{nonlinear}


\eof
\name{imwd.object}
\alias{imwd.object}
\title{2d Wavelet Decomposition Object}
\description{
  These are objects of class \code{"imwd"}.
  They represent a decomposition of a 2D function with respect
  to a 2D wavelet basis.
}
\section{GENERATION}{
This class of objects is typically returned from \code{\link{imwd}(.)}
to represent a wavelet decomposition of an image (or other 2D function).
}
\section{METHODS}{
The \code{"imwd"} class of objects has methods for the following generic
functions:
\code{plot}, \code{threshold}, \code{summary}, \code{print}, \code{draw}, \code{imwr}, \code{compress}.
}
\value{
The following components must be included in a legitimate \code{imwd} object.
\item{nlevels}{
  number of levels in wavelet decomposition. If you raise 2 to the power
  of nlevels then you get the dimension of the image that you originally
  started with.
}
\item{fl.dbase}{
  The first last database associated with the decomposition.  For images,
  this list is not very useful as each level's components is stored as
  a list component, rather than being packaged up in a single vector
  as in the 1D case.  Nevertheless the internals still need to know
  about fl.dbase to get the computations correct.
}
\item{filter}{
  A filter object as returned by "filter.select".  This component records
  the filter used in the decomposition.  The reconstruction routines
  use this component to find out what filter to use in reconstruction.
}
\item{wNLx}{
  The object will probably contain many components with names of
  this form.  These are all the wavelet coefficients of the decomposition.
  In "wNLx" the "N" refers to the level number and the "x" refers to
  the direction of the coefficients with "1" being horizontal,
  "2" being vertical and "3" being diagonal.  Note that the levels
  should be in numerically decreasing order, so if nlevels is 5, then
  there will be w5L1, w5L2, w5L3 first, then down to w1L1, w1L2, and
  w1L3.  Note that these coefficients store their data according to
  the first.last database \code{fl.dbase$first.last.d}, so refer to
  them using this, see \code{\link{first.last}}.
}
\item{w0Lconstant}{
  This is the coefficient of the bottom level scaling function coefficient.
  So for example, if you used Haar wavelets this would be the sample
  mean of the data (scaled by some factor depending on the number
  of levels, nlevels).
}
\item{bc}{This component details how the boundaries were treated in the
  decomposition.}
}
\details{
  In previous releases the original image was stored as the "original"
  component of a imwd object. This is not done now as the resulting
  objects were excessively large.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{\code{\link{imwd}} for examples;
  \code{\link{compress}}, \code{\link{draw}}.
}
\keyword{classes}
\keyword{methods}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{imwdc.object}
\alias{imwdc.object}
\title{Compressed 2D wavelet decomposition object}
\description{
  These are objects of class \code{"imwdc"}.
  They represent a compressed decomposition of a 2D function with respect
  to a 2D wavelet basis.
}
\section{GENERATION}{
This class of objects is returned from the \code{\link{threshold}} and
\code{\link{compress}} functions
to represent a wavelet decomposition of a image (or other 2D function).
}
\section{METHODS}{
The \code{"imwdc"} class of objects has methods for the following generic
functions:
\code{plot}, \code{threshold}, \code{summary}, \code{print}, \code{draw}, \code{imwr}, \code{uncompress}.
}
\value{
  The imwdc object is the same as a imwd object (see
  \code{\link{imwd.object}}), except that the \code{wNLx} and
  \code{w0Lconstant} components have been compressed using
  \code{\link{compress.default}}.
}
\seealso{\code{\link{imwd.object}}, \code{\link{uncompress}}.
}
\keyword{classes}
\keyword{methods}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{imwr}
\alias{imwr}
\title{2D Inverse Discrete Wavelet Transform (Image W. Reconstruction)}
\description{
  Generic function for 2-dimensional wavelet reconstruction.
}
\usage{
imwr(imwd, \dots)
}
\arguments{
  \item{imwd}{a wavelet decomposition object.}
  \item{\dots}{methods may have additional arguments.}
}
\seealso{the principal method, \code{\link{imwr.imwd}},
  and \code{\link{imwr.imwdc}}.
}
\keyword{nonlinear}
\keyword{smooth}

\eof
%%% $Id: imwr.imwd.Rd,v 1.6 2001/12/17 07:27:15 maechler Exp maechler $
\name{imwr.imwd}
\alias{imwr.imwd}
\alias{imwr.imwdc}%- both!
\title{2D Inverse Discrete Wavelet Transform (Image W. Reconstruction)}
\description{
  These functions perform the reconstruction stage of Mallat's
  pyramid algorithm, i.e. the inverse discrete wavelet transform for images.
}
\usage{
\method{imwr}{imwd}(imwd,  bc=imwd$bc, verbose = getOption("verbose"), \dots)
\method{imwr}{imwdc}(imwd, bc=imwd$bc, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{imwd}{
    object of class \code{imwd} or \code{imwdc} respectively; typically
    returned by \code{\link{imwd}} and \code{\link{threshold.imwd}}.
  }
  \item{bc}{character, specifying the boundary handling. It is best left
    to be the boundary handling specified by default.%that in the supplied argument.
  }
  \item{verbose}{logical; if true then informative messages are printed
    detailing the computations to be performed.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
A matrix, of dimension determined by the original data set supplied
to the initial decomposition (more precisely, determined by the nlevels
component of the imwd.object). This matrix is the highest resolution
level of the reconstruction. If a \code{imwd} (decomposition) is followed
immediately by a \code{imwr} (reconstruction) then the returned matrix
will be exactly the same as the original image.
}
\details{
Details of the algorithm are to be found in Mallat (1989).
As for "imwd" the algorithm works by applying many 1D reconstruction
algorithms to the coefficients. The filters used are those
described in Daubechies (1988).

This function is a method for the generic function
\code{imwr()} for class \code{imwd}.
It can be invoked by calling \code{imwr(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{imwr.imwd(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{see \code{\link{wd}} for a list.
}
\seealso{
\code{\link{imwd}}, \code{\link{plot}}, \code{\link{threshold}}
}
\examples{
example(imwd)
# Look at the error
summary( abs(c(imwr(imwdL) - lennon)))#around 1e-9

## Threshold after decomposing an image -- automagically compresses:
(tdi <- threshold(imwdL))

## Now reconstruct;  imwr calling imwr.imwdc directly
filled.contour(answer <- imwr(tdi))
}
\keyword{nonlinear}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{lennon}
\alias{lennon}
\title{Image of John Lennon}
\description{
  The \code{lennon} matrix has 256 rows and 256 columns and
  integer values in \code{0:192} which are \emph{inverted} image gray
  scales, i.e., 0 means white and 192 is dark.
  There are a few dark speckles on the image.

  John Lennon (19xx-19xx) was \emph{the} composer/leader of the
  legendary Beatles.
}
\source{
  From Guy Nason, \email{G.P.Nason@bristol.ac.uk}
}
\author{\R format and 8bit packing by Martin Maechler}
\examples{
data(lennon)
str(lennon)

tlennon <- table(lennon)
plot(names(tlennon), sqrt(tlennon), type = "h",
     ylab = "tlennon __ sqrt scaled", yaxt = "n",
     main = "Gray value distribution of `Lennon'")
atl <- pretty(tlennon, 8)
axis(2, at=sqrt(atl), labels = formatC(atl,wid=1), las=2)

image(lennon, zlim = c(0, 192), col = gray(127:0/128))# white to dark
image(lennon, zlim = c(-10, 240), col = gray(127:0/128))
}
\keyword{datasets}

\eof
\name{lt.to.name}
\alias{lt.to.name}
\title{Convert notations (wavelet)}
\description{
  Convert level/notation into \code{imwd.object} component names.

  Not for user use!
}
\usage{
lt.to.name(level, type)
}
\arguments{
  \item{level}{the level of the decomposition.}
  \item{type}{the type, one of CD, DC or DD, reflecting the application
    of filters g,h in combination.}
}
\value{
A string in the imwd.object wavelet coefficient name format.
}
\keyword{utilities}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{pack8bit}
\alias{pack8bit}
\alias{unpack8bit}
\title{Packing of 8-bit Integers}
\description{
  In image processing, often pictures are stored as long array of values
  in \code{0:255}.  For (file) storage, these can be packed into 32 bit
  integers.  These functions are for packing and unpacking.
}
\usage{
pack8bit(v8)
unpack8bit(v)
}
\arguments{
  \item{v8}{integer vector with 8-bit elements in 0..255.}
  \item{v}{integer vector (at least 32 bits).}
}
\details{
  If the length of \code{v8} is \emph{not} a multiple of 4,  \code{v8}
  is effectively padded with one to three \code{0}s.

  The implementation is straightforward and short; look at the function
  definitions.
}
\value{
  \code{pack8bit(v8)} returns an integer vector of length
  \code{ceiling(length(v8) / 4)}.

  \code{unpack8bit(v)} returns a vector of values from \code{0:255} of
  length \code{4 * length(v)}.
}
\author{Martin Maechler}
\seealso{\code{\link{compress}} etc for wavelet compression.}
\examples{
pack8bit(0:11)
(pack8bit(0:11) == pack8bit(0:9))# both need 3 32-bit ints; only 1st is =
## BUG:
all(250:255 == unpack8bit(pack8bit(250:255)))
stopifnot(0:255 == unpack8bit(pack8bit(0:255)))
}
\keyword{utilities}


\eof
\name{plot.imwd}
\alias{plot.imwd}
\title{Plot Method for an `imwd' object}
\description{
  Images Wavelet Coefficients of a imwd class object
}
\usage{
\method{plot}{imwd}(x, scaling="by.level", co.type= c("abs", "mabs", "none"),
     plot.type = c("mallat", "cols"), arrangement = c(3,3),
     plot = exists("image",mode="function"), transform = FALSE,
     tfunction = sqrt, col = heat.colors(32), \dots)
}
\arguments{
  \item{x}{object of class \code{imwd} containing a wavelet decomposition of
    an image.}
  \item{scaling}{character string, determining the scaling applied to
    each sub-image.  The values can be
    \code{"by.level"} which means each image is scaled independently to the whole
    dynamic range of the image, otherwise the whole image is scaled as a whole
    (as in previous versions).  This argument only takes effect when the
    \code{plot.type} is \code{"mallat"}.
  }
  \item{co.type}{character string, specifying a transform to be applied to
    the coefficients before display. \cr
    If \code{co.type=="abs"}, then the absolute values of the coefficients
    will be displayed,\cr
    if \code{co.type=="mabs"}, then the negative of the absolute values
    will be displayed.  These two arguments ensure that large coefficients,
    whether positive or negative, will look different to small coefficients.\cr
    If co.type=="none", then no transforming will take place (as in previous
    releases).}
  \item{plot}{logical; if \code{TRUE}, \code{\link{image}} is used to
    display the result.}
  \item{plot.type}{
    If plot.type=="mallat", then the image within image type of plot as
    exemplified within Mallat (1989) will be produced. If plot.type=="rows",
    then the individual level/orientation plots are produced in an array
    according to the arrangement argument.
  }
  \item{arrangement}{
    Determines the parameter to pass to mfrow to set up the display for
    multiple plots per page. The default, c(3,3) specifies 3 rows and 3 columns.
  }
  \item{transform}{logical; if \code{TRUE}, then the function given by
    argument \code{tfunction} is applied.
  }
  \item{tfunction}{a (vectorizing) function; a transform to apply to the
    coefficients if \code{transform} is true.}
  \item{col}{vector of colors to use for \code{\link{image}}.}
  \item{\dots}{potential further graphics parameters.}
}
\value{
  The matrix of wavelet coefficients packed in Mallat form,
  returned \code{\link{invisible}} if \code{plot} is true.
}
\section{Side Effects}{
  An image of the wavelet coefficients is produced if \code{plot} is true.
}
\details{
If plot.type=="mallat" then
the picture produced is the same as the one described by
Mallat 1989. After a decomposition there will be exactly the same
number of coefficients as there were pixels in the original image.
This function is a good way of plotting all the coefficients, but it means
that you can't really see some of the lower resolution images very clearly.

If plot.type=="rows", then each sub-image of the decomposition is plotted
at the same size so you can see the lower resolution images. By default
the arrangement of each of the sub-images is on a 3x3 array, so three levels
can fit onto one plot. If you are using a screen device then it may be
desirable to turn on the "ask before you plot" option with dev.ask(),
since the coefficients plot may run to a few pages with the "rows"
plot.type.

It is not always easy to see exactly what's going on, so there are
a few arguments that try to manipulate the (sub)image(s). The scaling
argument only works when the plot.type=="mallat" is in force. If scaling
is by.level then each subimage is scaled independently, if not then
the whole image is scaled at once. The co.type works for both plot
types and causes absolute values, negative absolute values or just
the coefficients to be plotted - this is useful for enhancing various
effects. The most flexible transformation is provided when transform==TRUE,
then the function tfunction is applied to all the coefficients and can
produce some useful contrast enhancing effects.

At each stage of the decomposition we decompose an image into
four images, each of which are half the size of the original
(see "imwd" for what these are). Three of the images correspond
to wavelet coefficients in the horizontal, vertical and diagonal
directions and these form the bottom-right, top-left and
top-right sections of the displayed image. The fourth image
is further decomposed into another four subimages. Three of which
get put into the bottom-right, top-left, and
top-right sections OF THE BOTTOM-LEFT part of the previous image
and so on and so on. It is impossible to explain this properly
in a simple fashion, so the only way to really understand what
is going on is to do it yourself and see what happens.

This function is a method for the generic function
\code{plot()} for class \code{imwd}.
It can be invoked by calling \code{plot(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{plot.imwd(x)} regardless of the
class of the object.
}
\seealso{\code{\link{imwd}, for references and background.}
}
\examples{
example(imwd)
# See the wavelet coefficient in Mallat's form
c.gray <- gray(127:0 / 128)
plot(imwdL, col = c.gray)
plot(imwdL, col = c.gray, scaling = "none")
plot(imwdL, col = c.gray, scaling = "none", co.type = "none")
plot(imwdL, col = c.gray, plot.type = "cols")
}
\keyword{hplot}
\keyword{smooth}

\eof
\name{plot.imwdc}
\alias{plot.imwdc}
\title{Plot Method for an `imwdc' object}
\description{
  Images (visualizes) wavelet coefficients of an \code{imwdc} class object.
}
\usage{
\method{plot}{imwdc}(x, verbose = getOption("verbose"),\dots)
}
\arguments{
  \item{x}{object of class "imwdc", maybe from a thresholding using
    \code{\link{threshold}()}.}
  \item{verbose}{logical, passed to \code{\link{uncompress}}.}
  \item{\dots}{other arguments to \code{\link{plot.imwd}}.}
}
\section{Side Effects}{
A plot of image coefficients is performed
This function is a method for the generic function
\code{plot()} for class \code{imwdc}.
It can be invoked by calling \code{plot(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{plot.imwdc(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{
  \code{\link{plot.imwd}}, \code{\link{plot}}
}
\examples{
example(imwd)
# Look at the error
## Threshold after decomposing an image -- automagically compresses:
summary(tdi <- threshold(imwdL))
plot(tdi) # bug?
}
\keyword{hplot}
% Converted by Sd2Rd version 0.3-3.

\eof
%%% $Id: plot.wd.Rd,v 1.10 2003/11/11 10:44:58 maechler Exp $
\name{plot.wd}
\alias{plot.wd}
\title{Plot Method for a `wd' object}
\description{Plot wavelet coefficients of an object of class \code{"wd"}.
}
\usage{
\method{plot}{wd}(x, xlabels, first.level = 1,
     main = "Wavelet Decomposition Coefficients", sub = x$filter$name,
     xlab = "Translate", ylab = "Resolution Level",
     scaling="by.level", rhlab = FALSE,
     col = par("fg"), lty = par("lty"), lwd = par("lwd"), \dots)
}
\arguments{
  \item{x}{object of class \code{wd}, containing a wavelet decomposition of
    a function.}
  \item{xlabels}{if supplied, this argument should be a vector containing
    the x-axis for the plot. For example, if you are trying to regress y
    on x, then you might want to put "x" in as the x-axis. Otherwise, the
    translates will be plotted.}
  \item{first.level}{integer, determining how many of the low resolution
    levels are plotted.  The default, \code{first.level=1} means that 1
    coefficient is plotted.}
  \item{main, sub, xlab, ylab}{main and sub-title, x- and y- axis label of plot}
  \item{scaling}{type of scaling applied to levels within the plot.
    Either \code{"by.level"} or \code{"global"}.}
  \item{rhlab}{logical; determines whether the scale factors applied to
    each level before plotting are printed as the right hand axis.}
  \item{col, lty, lwd}{graphical parameters, passed to \code{\link{segments}}.}
  \item{\dots}{other arguments to be supplied to \code{plot.wd}, see
    \code{\link{plot}}.}
}
\value{
  Axis labels to the right of the picture.  These values are the maximum
  of the absolute value of the coefficients at that resolution level.
  They are returned because they are sometimes hard to read on the plot.
}
\section{Side Effects}{
  A plot of the wavelet coefficients at each resolution level is produced.
}
\details{
The picture produced is similar to those in Donoho and Johnstone
1992. Wavelet coefficients for each resolution level are plotted
one above the other, with the high resolution coefficients at the
bottom, and the low resolution at the top. The coefficients are
plotted using the "segment" function, with a large positive coefficient
being plotted above an imaginary horizontal centre line, and a large
negative coefficient plotted below it. The position of a coefficient
along a line is indicative of the wavelet basis function's translate
number.

The resolution levels are labelled on the left-hand side axis,
and the maximum values of the absolute values of the coefficients
for the particular level form the right-hand side axis.

The levels of coefficients can be scaled in two ways. If you are
not interested in comparing the relative scales of coefficients
from different levels, then the default "scaling" option, "by.level"
is what you need. This computes the maximum of the absolute value
of the coefficients at a particular level and scales the so that
the fit nicely onto the plot. For this option, each level is scaled
DIFFERENTLY. To obtain a uniform scale for all the levels specify
the "global" option to the "scaling" argument. This will allow you
to make inter-level comparisons.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{\code{\link{wd}} and \code{\link{wd.object}}
}
\examples{
example(wd)

plot(wds, rhlab = TRUE) # plotting the wavelet coefficients
}
\keyword{hplot}
\keyword{smooth}


\eof
%%% $Id: print.imwd.Rd,v 1.7 2001/12/17 07:27:12 maechler Exp maechler $
\name{print.imwd}
\alias{print.imwd}
\alias{print.imwdc}
\title{Print an `imwd' or `imwdc' object}
\description{
  Prints out information about the object and then calls \code{summary()}
  on it.
}
\usage{
\method{print}{imwd}(x, \dots)
%not \method{} -- the two lines would look silly:
print.imwdc(x, \dots)
}
\arguments{
  \item{x}{an object of class \code{imwd} or \code{imwdc}, respectively.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  Contributed in release 2.2 by Martin Maechler
}
\seealso{\code{\link{imwd}}, \code{\link{imwd.object}},
  \code{\link{imwdc.object}}.
}
\keyword{utilities}


\eof
\name{print.wd}
\alias{print.wd}
\title{Print Method for a `wd' object.}
\description{
  Prints out information about an wd object and then calls \code{summary()}
  on it.
}
\usage{
\method{print}{wd}(x, \dots)
}
\arguments{
  \item{x}{an object of class \code{wd}}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\section{Side Effects}{
  Information about the \code{wd} object is printed out.
}
\seealso{\code{\link{wd}}, \code{\link{wd.object}}.
}
\keyword{utilities}


\eof
\name{putC}
\alias{putC}
\title{Put Smoothed Data Into Wavelet Structure}
}
\description{
  Makes a copy of the wd object, replaces some smoothed data in
  the copy, and then returns it.
}
\usage{
putC(wd, level, v, boundary=FALSE)
}
\arguments{
  \item{wd}{object of class \code{wd} that is to be copied and have
    smoothed data replaced.}
  \item{level}{integer; the level at which the replacement is to take place.}
  \item{v}{the replacement data which should be of the correct length.}
  \item{boundary}{logical; if \code{FALSE} then only the "real" data is
    replaced (and it is easy to predict the required length of \code{v}).
    \cr
    If boundary is \code{TRUE}, you can replace the boundary values at a
    particular level as well (but it is hard to predict the required
    length of \code{v}, and the information has to be obtained from the
    \code{first.last} database component of \code{wd}).

    \code{boundary} has no meaning if \code{wd} was obtained with the
    periodic boundary handling method (\code{bc}).}
}
\value{A \code{wd} class object containing the replaced data, see
  \code{\link{wd.object}}.
}
\details{
The function \code{\link{accessC}} obtains the smoothed data for a particular
level. This function, \code{putC} replaces data at a particular level and
returns a modified wd object reflecting the change.

This function is probably not particularly useful, but it is present
for completeness.  It is required because of the pyramidal nature of
the smoothed data points being packed into a vector.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{
\code{\link{wd.object}}, \code{\link{accessC}}
}
\examples{
example(wd)
## Put the numbers 1:64 into level 6
summary(newds <- putC(wds, level=6, v=1:64, boundary=FALSE))
#
# If you look at the C component of new, you will see that
# some numbers have changed at the appropriate position.
all.equal(wds,newds)
##>[1] "Component C: ... difference: 2.1912"
}
\keyword{manip}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{putD}
\alias{putD}
\title{Put Wavelet Coefficients Into Wavelet Structure}
\description{
  Makes a copy of the wd object, replaces some wavelet coefficients
  into the copy, and then returns the copy.
}
\usage{
putD(wd, level, v, boundary=FALSE)
}
\arguments{
  \item{wd}{
    object of class \code{wd} that is to be copied and have wavelet
    coefficients replaced.}
  \item{level}{integer; the level at which the replacement is to take place.}
  \item{v}{the replacement coefficients which should be of the correct length.}
  \item{boundary}{logical; if \code{FALSE} then only the "real" coefficients
    are replaced (and it is easy to predict the required length of
    \code{v}, just the correct power of 2).
    \cr
    If boundary is \code{TRUE}, then you can replace the boundary
    coefficients as well (but it is hard to predict the required length
    of \code{v}, and the information has to be extracted from the
    \code{first.last} database component of \code{wd}).

    \code{boundary} has no meaning if \code{wd} was obtained with the
    periodic boundary handling method (\code{bc}).}
}
\value{A \code{wd} class object containing the replaced coefficients, see
  \code{\link{wd.object}}.
}
\details{
The function \code{\link{accessD}}obtains the wavelet coefficients for a
particular level.  This function, \code{putD} replaces coefficients at a
particular level and returns a modified wd object reflecting the change.

As opposed to the utility of \code{\link{putC}}, the \code{putD}
function is actually quite useful.  It is fun to replace coefficients,
since then you can dream up your own functions, get pictures of the
wavelets etc. etc.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{
\code{\link{wd.object}}, \code{\link{accessD}}, \code{\link{draw}}
}
\examples{
example(wd)

## Set all the wavelet coefficients to zero
for(i in 0:(wds$nlevels - 1))
  wds <- putC(wds, level=i, v=rep(0,2^i))

## and now see what you've done
plot(wds)
matplot(x, cbind(wr(wd(y)), wr(wds)), type = 'l')
}
\keyword{manip}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{summary.imwd}
\alias{summary.imwd}
\alias{summary.imwdc}
\title{Summary Methods for `imwd' and `imwdc'  Object}
\usage{
\method{summary}{imwd}(object, \dots)
%not \method{} -- the two lines would look silly:
summary.imwdc(object, \dots)
}
\arguments{
  \item{object}{
    The image wavelet decomposition of class "imwd" or "imwdc", respectively.
    This is assumed to be the result of some image wavelet decomposition.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\description{
  The number of levels, the dimension of the original image,
  name of the filter and the type of boundary handling are printed.
}
\section{RELEASE}{
  Release 2.2, Copyright Guy Nason 1993
}
\seealso{
\code{\link{summary}}, \code{\link{imwd.object}}, \code{\link{imwdc.object}}.
}
\examples{
example(imwd)
summary(imwdL)
}
\keyword{nonlinear}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{summary.wd}
\alias{summary.wd}
\title{Summary Method for `wd' (1D Wavelet) Object}
\usage{
summary.wd(object, \dots)
}
\arguments{
  \item{object}{
    The wavelet decomposition of class \code{"wd"}.  This is assumed to
    be the result of some wavelet decomposition.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\description{
  The number of levels, the length of the original data sequence,
  name of the filter and type of boundary handling are printed.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{\code{\link{wd.object}}, \code{\link{summary}}.
}
\examples{
summary(wd(1:512)) # silly
}
\keyword{nonlinear}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{support}
\alias{support}
\title{Compute support of wavelet}
\description{
  Computes the support of a wavelet
}
\usage{
support(filter.number = 2, family = c("DaubExPhase", "DaubLeAsymm"),
        m = 0, n = 0)
}
\arguments{
  \item{filter.number}{integer; the number within the wavelet family whose
    support you wish to compute.}
  \item{family}{character string giving the family of wavelets that should
    be used.}
  \item{m}{the dilation number.}
  \item{n}{the translation number.}
}
\value{
  A list containing the support of the wavelets.  The list contains
  the following components:
  \item{lh}{The left-hand end of the interval of the support of the wavelet.}
  \item{rh}{The right-hand end}
  \item{psi.lh}{The left-hand end of the support of the mother wavelet}
  \item{psi.rh}{The right-hand end}
  \item{phi.lh}{The left-hand end of the support of the scale function
    (father wavelet)}
  \item{phi.rh}{The right-hand end}
}
\details{
To draw a wavelet it is important to know it's support.  \code{support}
provides this information. If a new family of wavelets is added then their
support needs to be determined and this function modified.
This function needn't be called by the user in normal use.
If the wavelet's aren't compactly supported then the support will not
be a simple closed interval!
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\section{BUGS}{
As the example shows below, when m=0 and n=0 the lh and rh don't show
the mother wavelet's support, but the wavelet above the mother wavelet.
The calling functions allow for this.
}
\seealso{\code{\link{wr}}, \code{\link{draw}}.
}
\examples{
str(support())
## List of 6
##  $ lh    : num -2
##  $ rh    : num 4
##  $ psi.lh: num -1
##  $ psi.rh: num 2
##  $ phi.lh: num 0
##  $ phi.rh: num 3
}
\keyword{dplot}
% Converted by Sd2Rd version 0.3-3.

\eof
\name{threshold}
\alias{threshold}
\title{Thresholding Wavelets - Generic function}
\description{Generic function for soft or hard thresholding of wavelets.
}
\usage{
threshold(x, \dots)
}
\arguments{
  \item{x}{an \R object.}
  \item{\dots}{methods may have additional arguments.}
}
\value{
thresholding is applied to the object and a thresholded version is
returned.
}
\references{see \code{\link{wd}} for a list.
}
\seealso{
\code{\link{threshold.wd}}, \code{\link{threshold.imwd}} and
\code{\link{threshold.imwdc}} for the wavelet methods and \bold{examples}.

\code{\link{wd}}, \code{\link{wr}}, \code{\link{imwd}}, etc.
}
\keyword{smooth}
\keyword{nonlinear}
% Converted by Sd2Rd version 0.3-3.

\eof
%%% $Id: threshold.imwd.Rd,v 1.7 2003/11/11 10:41:04 maechler Exp $
\name{threshold.imwd}
\alias{threshold.imwd}
\title{Threshold an `imwd' Object (2D Wavelet)}
\description{
  Applies hard or soft thresholding to wavelet decomposition object
  of class \code{imwd}.
}
\usage{
\method{threshold}{imwd}(x, levels=3:(x$nlevels-1), type = c("hard", "soft"),
      policy = c("universal", "manual", "probability"),
      by.level = FALSE, value = 0, dev = var,
      verbose = getOption("verbose"), return.threshold = FALSE,
      compression = TRUE, \dots)
}
\arguments{
  \item{x}{object of class \code{"imwd"},  typically  from a wavelet
    decomposition using the \code{\link{imwd}} function.}
  \item{levels}{integer vector determining which levels are thresholded in
    the decomposition.}
  \item{type}{character, determining the type of thresholding; either
    "hard" or "soft".}
  \item{policy}{character indicating the threshold to use,
    can be "universal", "manual", or "probability".}
  \item{by.level}{logical; if \code{FALSE} then a global threshold is
    applied to \emph{all} the levels specified by \code{levels},
    otherwise a threshold is computed and applied separately to each level.
  }
  \item{value}{numeric user-supplied threshold for the \code{"manual"}
    \code{policy}, or the user-supplied quantile level for the
    \code{"probability"} policy.}
  \item{dev}{deviance function.  The default \code{var} is to use the
    variance, but you can insert your own measure of deviance.  }
  \item{verbose}{logical, if true then \code{threshold()} spurts informative
    messages at you.}
  \item{return.threshold}{logical, if true then the actual threshold is
    returned, otherwise the thresholded object is returned.}
  \item{compression}{logical; if true the thresholded object is
    compressed and then returned, otherwise it is returned unaltered.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{
  If \code{compression} is true then an object of class \code{"imwdc"}
  that has been thresholded and compressed,
  otherwise an uncompressed, but thresholded object of class \code{"imwd"}.
}
\details{
Thresholding modifies the coefficients within a \code{imwd} wavelet
decomposition object.  The modification can be performed either
with a "hard" or "soft" thresholding selected by the \code{type} argument.

%% ---> ./threshold.wd.Rd
See the ``Details'' section of \code{\link{threshold.wd}} for an
explanation of thresholding types and policies.

%% imwd specific :
The thresholding process forces many coefficients to zero. From release 2.2
onwards the thresholded \code{imwd} object is compressed by \code{\link{compress}}
and returned as a much smaller \code{imwdc} object.  An \code{imwdc} object is
easily converted into an \code{imwd} object by \code{\link{uncompress}}, but
all relevant functions will handle the \code{imwdc} object.

Note that the coefficients for the horizontal, diagonal and vertical
detail at a particular level are treated as one. In future releases
we may add the capability to treat the details differently, although
this would increase the complexity of the argument specification.

%% FIXME: "Everyone" should know the following :
%% =====  Don't say this in all the *.Rd files, rather "see \link{...}"...
This function is a method for the generic function
\code{threshold()} for class \code{imwd}.
It can be invoked by calling \code{threshold(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{threshold.imwd(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{see \code{\link{wd}} for a list.
}
\section{BUGS}{
There should be an optimal policy as well, although universal comes
close.
}
\seealso{For examples, see \code{\link{imwr.imwd}}.
  \code{\link{imwr}}, \code{\link{imwd}}, \code{\link{compress}},
  \code{\link{uncompress}}, \code{\link{imwd.object}},
  \code{\link{imwdc.object}}
}
\keyword{nonlinear}
\keyword{smooth}

\eof
%%% $Id: threshold.imwdc.Rd,v 1.6 2001/12/17 08:18:30 maechler Exp maechler $
\name{threshold.imwdc}
\alias{threshold.imwdc}
\title{Threshold an `imwdc' Object (2D Wavelet)}
\description{
  Applies hard or soft thresholding to wavelet decomposition object
  of class \code{imwdc}.
}
\usage{
\method{threshold}{imwdc}(x, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{object of class \code{imwdc}, maybe after a previous thresholding.}
  \item{verbose}{print information messages about what's going on}
  \item{\dots}{other arguments to threshold}
}
\value{
  An object of class \code{"imwdc"}, containing the thresholded object.
}
\details{
This function thresholds just like \code{\link{threshold.imwd}}, except that the
input is an "imwdc" object, not a "imwd" one, which suggests that
the object has already been thresholded (because \code{\link{threshold.imwd}}
returns an "imwdc" object).  Because the object is likely to have
been thresholded a warning message stating this is printed.

However, it is entirely possible that you would wish to impose a higher
threshold on an already thresholded object, and so this function does
just this.

%% FIXME: "Everyone" should know the following :
%% =====  Don't say this in all the *.Rd files, rather "see \link{...}"...
This function is a method for the generic function
\code{\link{threshold}()} for class \code{imwdc}.
It can be invoked by calling \code{threshold(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{threshold.imwdc(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{see \code{\link{wd}} for a list.
}
\section{BUGS}{
  There should be an optimal policy as well, although universal comes
  close.
}
\seealso{For examples, see \code{\link{imwr.imwd}}.
  \code{\link{imwr}}, \code{\link{imwd}}, \code{\link{compress}},
  \code{\link{uncompress}}, \code{\link{imwd.object}},
  \code{\link{imwdc.object}}
}
\keyword{nonlinear}
\keyword{smooth}


\eof
%%% $Id: threshold.wd.Rd,v 1.7 2003/11/11 10:39:32 maechler Exp $
\name{threshold.wd}
\alias{threshold.wd}
\title{Threshold a `wd' Object (1D Wavelet)}
\description{
  Applies hard or soft thresholding with different policies to a wavelet
  decomposition object.
}
\usage{
\method{threshold}{wd}(x, levels=3:(x$nlevels-1), type = c("hard", "soft"),
	policy = c("universal", "manual", "probability"), by.level = FALSE,
	value = 0, dev = var, boundary = FALSE,
	verbose = getOption("verbose"), return.threshold = FALSE, \dots)
}
\arguments{
  \item{x}{Object of class \code{"wd"}, typically from a wavelet decomposition
    using the \code{\link{wd}} function.}
  \item{levels}{integer vector determining which levels are thresholded in
    the decomposition.}
  \item{type}{character, determining the type of thresholding; either
    "hard" or "soft".}
  \item{policy}{character indicating the threshold to use,
    can be "universal", "manual", or "probability".}
  \item{by.level}{logical; if \code{FALSE} then a global threshold is
    applied to \emph{all} the levels specified by \code{levels},
    otherwise a threshold is computed and applied separately to each level.
  }
  \item{value}{numeric user-supplied threshold for the \code{"manual"}
    \code{policy}, or the user-supplied quantile level for the
    \code{"probability"} policy.}
  \item{dev}{deviance function.  The default \code{var} is to use the
    variance, but you can insert your own measure of deviance.  }
  \item{boundary}{logical, if true then the boundary correction values
    are included for thresholding, otherwise not.}
  \item{verbose}{logical, if true then \code{threshold()} spurts informative
    messages at you.}
  \item{return.threshold}{logical, if true then the actual threshold is
    returned, otherwise the thresholded object is returned.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{An object of class "wd" that has been thresholded.
}
\details{%
%%__FIXME___ These are VERY wordy --- and almost identical to those in
%%           ./threshold.imwd.Rd (and probably even other places) !!
%%
Thresholding modifies the coefficients within a \code{wd} wavelet
decomposition object. The modification can be performed either
with a "hard" or "soft" thresholding selected by the \code{type} argument.

Hard thresholding simply compares a coefficient with a threshold.  If it
is larger in absolute magnitude it is left alone, if it is smaller it is
set to zero.
The "soft threshold" formula is
\deqn{\mbox{soft}(w) = \mbox{sgn}(w) \max(\left|w\right| - t, 0)}{%
             soft(w) = sgn(w)*max(|w| - t, 0)}
where \eqn{w} is the wavelet coefficient, \eqn{t} is the threshold and
\eqn{\mbox{sgn}(w)}{sgn(w)} is the sign of \eqn{w}.  Soft thresholding
causes \eqn{w} to be replaced by \eqn{soft(w)}.

There are many ways that the threshold can be computed; we term this the
"policy".  A universal policy computes a threshold based on Donoho and
Johnstone's "universal thresholds".  The threshold is
sqrt(2*log(n))*noise, where noise is computed as sqrt(dev(w)), i.e. some
measure of the variability of the coefficients, and n is the number of
data points (or number of wavelet coefficients).  By default "dev" is
"var", so the noise level is estimated using the sample standard
deviation. You can use any other such estimate by writing your own
function and supplying it as the "dev" argument. For example you could
create the function \code{myvar} by
\cr \code{myvar <- function(d) mad(d)^2}

This computes the square of the mean absolute deviation of the data.
It is squared because "dev" should be a measure on the variance scale,
not the standard deviation (you know what I mean).
If you make the "by.levels" argument T, then a separate threshold is
computed for each level in the "levels" vector. This means that
the variance is estimated separately for each level.

The "manual" policy is simple. You supply a threshold value ("value")
and hard or soft thresholding is performed using that value.
The "value" argument is a vector. If it is of length 1 then
it is replicated to be the same length as the "levels" vector,
otherwise it is repeated as many times as necessary to be the
length vector's length. In this way, different thresholds can be
given for different levels. Note that the "by.level" argument has
no effect with this policy.

The "probability" policy works as follows. All coefficients that
are smaller than the "value"th quantile of the coefficients are
set to zero. If "by.level" is false, then the quantile is computed
for all coefficients in the levels specified by the "levels" vector;
if "by.level" is true, then each level's quantile is estimated separately.

This function is a method for the generic function
\code{threshold()} for class \code{wd}.
It can be invoked by calling \code{threshold(x)} for an
object \code{x} of the appropriate class, or directly by
calling \code{threshold.wd(x)} regardless of the
class of the object.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{see \code{\link{wd}} for a list.}
\section{BUGS}{
  There should be an optimal policy as well, although universal comes close.
}
\seealso{\code{\link{wr}}, \code{\link{wd}}}
\examples{
# "Standard" example:
example(wd)
#
# Threshold it
#
(thH.wds <- threshold(wds))
thS.wds <- threshold(wds, type="soft")
#
# Reconstruct from the thresholded coefficients
#
str(    trecH <- wr(thH.wds))
summary(trecS <- wr(thS.wds))
#
# Plot the data, the true & reconstructed functions
#
plot(x,y, col = "gray", cex = .6) # the data
lines(x, fx, col="blue", lwd=1)# true function
lines(x, trecH, col= "orchid", lty=2, lwd=2)
lines(x, trecS, col= "tomato", lty=4, lwd=2)
legend(0,5.6, c("true", paste(c("hard","soft")," thresh.")),
       col = c("blue","orchid","tomato"), lwd= c(1,2,2), lty = c(1,2,4))
}
\keyword{nonlinear}
\keyword{smooth}

\eof
\name{uncompress}
\alias{uncompress}
\title{Decompression - Generic Function (Wavelet)}
\description{
  Decompresses \code{x}  by padding with zeroes
  This is the generic function.
}
\usage{
uncompress(x, \dots)
}
\arguments{
  \item{x}{an \R object.}
  \item{\dots}{methods may have additional arguments.}
}
\value{an uncompressed version of the supplied argument.}
\details{Functions with names beginning in \code{uncompress.} will be
  methods for this function,

  See their individual help pages, e.g.,
  \code{\link{uncompress.imwdc}}, \code{\link{uncompress.default}},
  for operation.
}
\seealso{\code{\link{compress}}.}
\keyword{manip}
\keyword{utilities}


\eof
\name{uncompress.default}
\alias{uncompress.default}
\title{Decompress a Compressed or Uncompressed Object}
\description{
  The \code{\link{compress}} function compresses a sparse vector by
  removing zeroes.  This function performs the inverse transformation.
}
\usage{
\method{uncompress}{default}(x, verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{The \code{compressed} or \code{uncompressed} object to
    uncompress.}
  \item{verbose}{logical; if true then informative messages are printed.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\value{The uncompressed vector.
}
\details{
See \code{\link{compress.default}} to see what this function does to vectors.
The \code{uncompress.default} function takes objects that have been output
from \code{compress.default(.)} and restores them to their original values.
If the input to \code{compress.default} was a sparse vector, then an object
of class \code{compressed} would be returned.  This class of object
(see \code{\link{compressed.object}}) is a list containing the position
and values of all the non-zero elements of the original vector. The
\code{uncompress.default} function reconstitutes the original vector.

Sometimes compression is not worthwhile (i.e. the original vector is not
sparse enough). In this case \code{compress.default()} returns the original
vector in a list with class \code{uncompressed}. The \code{uncompress.default()}
function's task in this case is simple: only return the vector part of
\code{uncompressed}.
}
\seealso{\code{\link{compress}}, \code{\link{compressed.object}}.}
\examples{
## Compress a sparse vector and look at it
str(cv <- compress( vv <- c(rep(0,100),564) ))

## Uncompress the vector, (uncompress.default is used)
str(ucv <- uncompress(cv))
all(ucv == vv)

## a bit less sparse:
vv[sample(seq(vv), 5)] <- 1:5
str(cv <- compress(vv))
all(vv == uncompress(cv))
}
\keyword{manip}


\eof
\name{uncompressed.object}
\alias{uncompressed.object}
\title{Uncompressed (Wavelet) Object}
\description{
  These are objects of class \code{"uncompressed"}.
  They are lists with components representing a vector, see below.

  Typically, these are generated by \code{\link{compress.default}}.
}
\value{
  The following components must be included in a legitimate
  \code{uncompressed} object.
  \item{vector}{the original uncompressed vector}
}
\seealso{\code{\link{compressed.object}} for details; \code{\link{compress}}.
}
\keyword{classes}
% Converted by Sd2Rd version 0.3-3.

\eof
%%% $Id: wd.Rd,v 1.7 2003/11/11 10:36:43 maechler Exp $
\name{wd}
\alias{wd}
\title{Discrete Wavelet Transform (Decomposition).}
\description{
  This function performs the decomposition stage of Mallat's pyramid
  algorithm (Mallat 1989), i.e., the discrete wavelet transform.  The
  actual transform is performed by some C code, this is dynamically
  linked into \R.
}
\usage{
wd(data, filter.number = 2, family = c("DaubExPhase", "DaubLeAsymm"),
   bc = c("periodic", "symmetric"), verbose = getOption("verbose"))
}
\arguments{
  \item{data}{a vector containing the data you wish to decompose. The
    length of this vector must be a power of 2.}
  \item{filter.number}{
    This selects the wavelet that you want to use in the decomposition.
    By default this is 2, the Daubechies orthonormal compactly supported
    wavelet N=2 (Daubechies (1988)), extremal phase family.
  }
  \item{family}{
    specifies the family of wavelets that you want to use. The options are
    "DaubExPhase" and "DaubLeAsymm". The "DaubExPhase" wavelets were provided
    with previous versions of wavethresh.
  }
  \item{bc}{character, specifying the boundary handling.
    If \code{bc=="periodic"} the default, then the
    function you decompose is assumed to be periodic on its interval of
    definition,\cr
    if \code{bc == "symmetric"}, the function beyond its boundaries is
    assumed to be a symmetric reflection of the function in the
    boundary.
  }
  \item{verbose}{logical, controlling the printing of `informative'
    messages whilst the computations progress.
    turned off by default.
  }
}
\value{
  An object of class \code{wd}, see \code{\link{wd.object}} for details.
  Basically, this object is a list with the following components
  \item{C}{
    Vector of sets of successively smoothed data. The pyramid structure
    of Mallat is stacked so that it fits into a vector.
    \code{\link{accessC}} should be used to extract these.}
  \item{D}{
    Vector of sets of wavelet coefficients at different resolution levels,
    stacking Mallat's pyramid structure into a vector.
    The function \code{\link{accessD}} should be used to extract the
    coefficients for a particular resolution level.}
  \item{nlevels}{
    The number of resolution levels, depending on the length of the data
    vector. If \code{length(data) = 2^m}, then there will be m
    resolution levels.}
  \item{fl.dbase}{The first/last database associated with this
    decomposition, see \code{\link{wd.object}} and
    \code{\link{first.last}} for details.}
  \item{filter}{A list containing information about the filter}
  \item{bc}{How the boundaries were handled.}
}
\details{
The code implements Mallat's pyramid algorithm (Mallat 1989).
Essentially it works like this: you start off with some data cm,
which is a real vector of length \eqn{2^m}, say.

Then from this you obtain two vectors of length \eqn{2^{m-1}}{2^(m-1)}.
One of these is a set of smoothed data, c(m-1), say.  This looks like a
smoothed version of cm. The other is a vector, d(m-1), say.
%
This corresponds to the detail removed in smoothing cm to c(m-1).  More
precisely, they are the coefficients of the wavelet expansion
corresponding to the highest resolution wavelets in the expansion.
Similarly, c(m-2) and d(m-2) are obtained from c(m-1), etc. until you
reach c0 and d0.

All levels of smoothed data are stacked into a single vector for memory
efficiency and ease of transport across the SPlus-C interface.

The smoothing is performed directly by convolution with the wavelet filter
(\code{filter.select(n)$H}, essentially low-pass filtering), and then dyadic
decimation (selecting every other datum, see Vaidyanathan (1990)).
The detail extraction is performed by the mirror filter of H, which
we call G and is a bandpass filter.
G and H are also known quadrature mirror filters.

There are now two methods of handling "boundary problems". If you know
that your function is periodic (on it's interval) then use the
bc="periodic" option, if you think that the function is symmetric
reflection about each boundary then use bc="symmetric".
If you don't know then it is wise to experiment with both methods,
in any case, if you don't have very much data don't infer too much about
your decomposition! If you have loads of data then don't infer too much
about the boundaries. It can be easier to interpret the wavelet coefficients
from a bc="periodic" decomposition, so that is now the default.
Numerical Recipes implements some of the wavelets code, in particular
we have compared our code to "wt1" and "daub4" on page 595. We are
pleased to announce that our code gives the same answers!
The only difference that you might notice is that one of the coefficients,
at the beginning or end of the decomposition,
always appears in the "wrong" place. This is not so, when you assume
periodic boundaries you can imagine the function defined on a circle
and you can basically place the coefficient at the beginning or the
end (because there is no beginning or end, as it were).
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{
Any book on wavelets, especially

Chui, C. K. (1992)  \emph{An Introduction to Wavelets};
Academic Press, London.

Daubechies, I. (1988)
\emph{Orthonormal bases of compactly supported wavelets};
Communications on Pure and Applied Mathematics, \bold{41}, 909-996.

Daubechies, I. (1992)
\emph{Ten Lectures on Wavelets};
CBMS-NSF Regional Conference Series in Applied Mathematics.

Donoho, D and Johnstone, I. (1994)
\emph{Ideal Spatial Adaptation by Wavelet Shrinkage};
Biometrika; \bold{81}, 425-455.

Mallat, S. G. (1989)
\emph{A theory for multiresolution signal decomposition: the wavelet
  representation};
IEEE Transactions on Pattern Analysis and Machine Intelligence,
\bold{11}, No. 7, 674-693.

Vaidyanathan, P. P. (1990)
\emph{Multirate digital filters, filter banks, polyphase networks and
  applications: A tutorial};
Proceedings of the IEEE, \bold{78}, No.~1, 56-93.
}
\seealso{\code{\link{wr}}, \code{\link{threshold}}, \code{\link{plot.wd}},
  \code{\link{accessC}}, \code{\link{accessD}},
  \code{\link{filter.select}}.
}
\examples{
## Example from Nason's 1993 report
f.ssl <- function(x) sin(x) + sin(2*x) + log(1+x)
m <- 10 ; n <- 2^m
x <- seq(0, 10*pi, length = n)
fx <- f.ssl(x)
y <- fx + rnorm(n, s= .3)

## Decompose the test data
summary(wds <- wd(y))
}
\keyword{math}
\keyword{smooth}
\keyword{nonlinear}

\eof
\name{wd.object}
\alias{wd.object}
\title{Wavelet decomposition object (1D)}
\description{
These are objects of class "wd"
They represent a decomposition of a function with respect to
a wavelet basis.
}
\section{GENERATION}{
This class of objects is returned from the \code{wd} function
to represent a wavelet decomposition of a function.
Other functions also return a \code{wd.object}
}
\section{METHODS}{
The \code{"wd"} class of objects has methods for the following generic
functions:
\code{plot}, \code{threshold}, \code{summary}, \code{print}, \code{draw}.
}
\value{
  The following components must be included in a legitimate \code{wd} object.
  \item{C}{
    a vector containing each level's smoothed data.
    The wavelet transform works by applying both a smoothing filter
    and a bandpass filter to the previous level's smoothed data,
    the top level containing data at the highest resolution level.
    This the ``pyramid structure of Mallat''.
    Each of these levels are stored one after the other in this vector.
    The matrix \code{fl.dbase$first.last.c} determines exactly where each
    level is stored in the vector.\cr
    Usually, \code{\link{accessC}} should be used to extract \code{C}
    components.
  }
  \item{D}{wavelet coefficients.  If you were to write down the discrete
    wavelet transform of a function, then these D would be the
    coefficients of the wavelet basis functions.  Like the C, they are
    also formed in a pyramidal manner, but stored in a linear array.
    The storage details are to be found in \code{fl.dbase$first.last.d}.\cr
    Usually, \code{\link{accessD}} should be used to extract \code{D}
    components.
  }
  \item{nlevels}{
    The number of resolution levels in the pyramidal decomposition that produces
    the coefficients.  Therefore, \code{2^nlevels}\eqn{\equiv 2^m}{= 2^m} is the
    number of data points used in the decomposition.
    This means there will be \eqn{m} levels of wavelet coefficients (indexed
    \eqn{0,1,2,\dots, m-1}), and \eqn{m+1} levels of smoothed data
    (indexed \eqn{0,1,2,\dots{},m}).
  }
  \item{fl.dbase}{The first/last database associated with this decomposition.
    This is a list consisting of 2 integers, and 2 matrices. The matrices
    detail how the coefficients are stored in the C and D components
    of the \code{wd.object}.

    In the decomposition `extra' coefficients are generated that help take
    care of the boundary effects, this database lists where these start and
    finish, so the "true" data can be extracted.
    See \code{\link{first.last}} for more information.
  }
  \item{filter}{a list containing the details of the filter that did the
    decomposition}
  \item{bc}{how the boundaries were handled}
}
\details{
To retain your sanity the C and D coefficients should be extracted
by the \code{\link{accessC}} and \code{\link{accessD}} functions and put using
the \code{\link{putC}} and \code{\link{putD}} functions, rather than
by the \code{$} operator.

Mind you, if you want to muck about with coefficients directly,
then you'll have to do it yourself by working out what the \code{fl.dbase}
list means.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\seealso{\code{\link{wd}} for examples and background.
}
\keyword{classes}
\keyword{methods}
\keyword{smooth}
% Converted by Sd2Rd version 0.3-3.

\eof
%%% $Id: wr.Rd,v 1.7 2003/11/11 10:34:54 maechler Exp $
\name{wr}
\title{Discrete wavelet transform (reconstruction).}
\alias{wr}
\usage{
wr(wd, start.level = 0, verbose = getOption("verbose"),
   bc = wd$bc, return.object = FALSE,
   filter.number = wd$filter$filter.number, family = wd$filter$family)
}
\description{
  This function performs the reconstruction stage of Mallat's pyramid algorithm,
  i.e., the discrete inverse wavelet transform.
}
\arguments{
  \item{wd}{A wavelet decomposition object as returned by \code{\link{wd}},
    see \code{\link{wd.object}}.}
  \item{start.level}{integer; the level at which to start reconstruction.
    This is usually the first (level 0).}
  \item{bc, filter.number, family}{by default part of the \code{wd}
    object, but can specified differently by the ``knowing''.}
  \item{verbose}{logical, controlling the printing of ``informative''
    messages whilst the computations progress.  Such messages are
    generally annoying so it is turned off by default.}
  \item{return.object}{logical;
    If this is FALSE then the top level of the reconstruction is returned
    (this is the reconstructed function at the highest resolution).
    Otherwise if it is T the whole wd reconstructed object is returned.
  }
}
\value{
  Either a vector containing the top level reconstruction or
  an object of class \code{wd} containing the results of the reconstruction,
  details to be found in the documentation for \code{\link{wd.object}}.
}
\details{
The code implements Mallat's pyramid algorithm (Mallat 1989).
In the reconstruction the quadrature mirror filters G and H are
supplied with \eqn{c_0}{c0} and \eqn{d_0, d_1, \dots, d_{m-1}}{d0, d1,
  .., d(m-1)} (the wavelet coefficients) and rebuild
\eqn{c_1, \dots, c_m}{c1, .., cm}.

If \code{wd} was obtained directly from \code{\link{wd}()} then the
original function can be reconstructued \bold{exactly} as \eqn{c_m}{cm}
and can be sought with \code{accessC(wd.object, level=wd.object$levels)}.

Usually, the \code{wd} object has been modified in some way, for example,
some coefficients set to zero by threshold. \code{Wr} then reconstructs
the function with that set of wavelet coefficients.
}
\section{RELEASE}{
Release 2.2
Copyright Guy Nason 1993
}
\references{see \code{\link{wd}} for a list.}
\seealso{\code{\link{wd}}, \code{\link{accessC}}, \code{\link{accessD}},
  \code{\link{filter.select}}, \code{\link{threshold}}.
}
\examples{
# Decompose and then exactly reconstruct test.data
example(wd)#-> wds has wd() ressult
rec.wds <- wr(wds)
rec.wds.obj <- wr(wds, return.object = TRUE)
rec.wds2 <- accessC(rec.wds.obj, level=rec.wds.obj$nlevels)
all(rec.wds == rec.wds2)# since wr() internally uses accessC()

# Look at accuracy of reconstruction
summary(abs(rec.wds - y)) #~ 10^-11

# Reconstruct a hard.thresholded object, look at the wavelet coefficients
summary(thr.wds <- wr(threshold(wds, type="hard") ))
}
\keyword{math}
\keyword{smooth}
\keyword{nonlinear}

\eof
\name{wvrelease}
\alias{wvrelease}
\title{
Identify version of wavelet software.
}
\description{
Prints a message identifying the version of the wavelet software.
}
\usage{
wvrelease(do.cat = interactive())
}
\arguments{
  \item{do.cat}{logical; if true, \code{cat(.)} a few lines to the
    console, otherwise just invisibly return.}
}
\value{
  a list list with components
  \item{major}{the major release number}
  \item{R.minor}{the R port version (of the major release)}
}
\section{Side Effects}{
  Prints a message identifying the version of the wavelet software.
}
\details{
When attaching the wavelet software directory to your own data directory
it is useful to put a "wvrelease()" function call just after the
attach. This ensures that you have attached to the correct directory
and that you are using the most up to date software.
(This function is mainly useful to those of you that have received
earlier versions of this software).
}
\examples{
str(r <- wvrelease())
}
\keyword{smooth}
\keyword{nonlinear}


\eof
