\name{bpy.colors}
\alias{bpy.colors}
\title{ blue-pink-yellow color scheme that prints well as grey tone }
\description{ Create a vector of `n' ``contiguous'' colors.  }
\usage{ bpy.colors(n = 100, cutoff.tails = 0.1) }
\arguments{
\item{n}{number of colors (>= 1) to be in the palette}
\item{cutoff.tails}{tail fraction to be cut off. This palette runs
from black to white if \code{cutoff.tails} is 0; by cutting off 
the tails, it runs from blue to yellow, which looks nicer. }
}
\value{ A character vector, `cv', of color names.  This can be used
either to create a user-defined color palette for subsequent graphics
by `palette(cv)', a `col=' specification in graphics functions or in
`par'. }
\author{ unknown }
\references{
\url{http://www.ihe.uni-karlsruhe.de/mitarbeiter/vonhagen/palette.en.html};
gnuplot has this color map 
}
\note{ This color map prints well on black-and-white printers. }
\seealso{\link{rainbow}, \link{cm.colors}}
\examples{
bpy.colors(10)
p <- expand.grid(x=1:30,y=1:30)
p$z <- p$x + p$y
image(p, col = bpy.colors(100))
# require(lattice)
# trellis.par.set("regions", list(col=bpy.colors())) # make default
}
\keyword{color}

\eof
\name{bubble}
\alias{bubble}
\title{
Create a bubble plot of spatial data
}
\description{
Create a bubble plot of spatial data, with options for bicolour
residual plots (xyplot wrapper)
}
\usage{
bubble(data, xcol = 1, ycol = 2, zcol = 3, fill = TRUE, maxsize = 3,
    do.sqrt = TRUE, pch, col = c(2,3), key.entries = quantile(data[,zcol]),
    main, identify = FALSE, labels = row.names(data), ...)
}
\arguments{
\item{data}{ data frame from which x- and y-coordinate and z-variable are taken}
\item{xcol}{ x-coordinate column number or (quoted) name }
\item{ycol}{ y-coordinate column number or (quoted) name }
\item{zcol}{ z-variable column number or (quoted) name }
\item{fill}{ logical; if TRUE, filled circles are plotted (pch = 16), 
else open circles (pch = 1); the pch argument overrides this }
\item{maxsize}{ \code{cex} value for largest circle }
\item{do.sqrt}{ logical; if TRUE the plotting symbol area (sqrt(diameter)) 
is proportional to the value of the z-variable; if FALSE, the symbol size
(diameter) is proportional to the z-variable }
\item{pch}{ plotting character }
\item{col}{ colours to be used; numeric vector of size two: first value
is for negative values, second for positive values. }
\item{key.entries}{ the values that will be plotted in the key; by default 
the five quantiles min, q.25, median q.75, max }
\item{main}{ character; plot title }
\item{identify}{ logical; if true, regular \code{plot} is called instead of
\code{xyplot}, and followed by a call to \code{identify()}. }
\item{labels}{labels argument, passed to \code{plot} when \code{identify}
is TRUE}
\item{...}{ arguments, passed to \code{xyplot}, or \code{plot} if 
identification is required. }
}
\value{
returns (or plots) the bubble plot; if \code{identify} is TRUE, returns
the indexes (row numbers, not the row.names shown) of identified points.
}
\references{ }
\author{ Edzer J. Pebesma }
\seealso{\code{xyplot}, \link{mapasp}, \link{identify}}
\examples{
data(meuse)
bubble(meuse, max = 2.5, main = "cadmium concentrations (ppm)", 
    key.entries = c(.5,1,2,4,8,16))
bubble(meuse, "x", "y", "zinc", main = "zinc concentrations (ppm)",
    key.entries =  100 * 2^(0:4))
}
\keyword{dplot}

\eof
\name{fit.lmc}
\alias{fit.lmc}
\title{ Fit a Linear Model of Coregionalization to a Multivariable Sample Variogram }
\description{
Fit a Linear Model of Coregionalization to a Multivariable Sample Variogram;
in case of a single variogram model (i.e., no nugget) this is equivalent to
Intrinsic Correlation
}
\usage{
fit.lmc(v, g, model, fit.ranges = FALSE, fit.lmc = !fit.ranges, 
correct.diagonal = 1.0, ...) 
}
\arguments{
\item{v}{ multivariable sample variogram, output of \link{variogram} }
\item{g}{ gstat object, output of \link{gstat} }
\item{model}{ variogram model, output of \link{vgm}; if supplied 
this value is used as initial value for each fit }
\item{fit.ranges}{ logical; determines whether the range coefficients
(excluding that of the nugget component) should be fitted; or logical
vector: determines for each range parameter of the variogram model
whether it should be fitted or fixed. }
\item{fit.lmc}{ logical; if TRUE, each coefficient matrices of partial
sills is guaranteed to be positive definite }
\item{correct.diagonal}{ multiplicative correction factor to be applied
to partial sills of direct variograms only; the default value, 1.0, does
not correct. If you encounter problems with singular covariance matrices
during cokriging or cosimulation, you may want to try to increase this
to e.g. 1.01 }
\item{...}{ parameters that get passed to \link{fit.variogram} }
}
\value{
returns an object of class \code{gstat}, with fitted variograms;
}
\references{ \url{http://www.gstat.org/} }
\author{ Edzer J. Pebesma }
\note{ This function does not use the iterative procedure proposed by
M. Goulard and M. Voltz (Math. Geol., 24(3): 269-286; reproduced in
Goovaerts' 1997 book) but uses simply two steps: first, each variogram
model is fitted to a direct or cross variogram; next each of the partial
sill coefficient matrices is approached by its in least squares sense
closest positive definite matrices (by setting any negative eigenvalues
to zero). 

The argument \code{correct.diagonal} was introduced by experience: by
zeroing the negative eigenvalues for fitting positive definite partial
sill matrices, apparently still perfect correlation may result, leading
to singular cokriging/cosimulation matrices. If someone knows of a more
elegant way to get around this, please let me know.
}
\seealso{ \link{variogram}, \link{vgm}, \link{fit.variogram},
\code{demo(cokriging)} }
\examples{ 
}
\keyword{models}

\eof
\name{fit.variogram}
\alias{fit.variogram}
\title{ Fit a Variogram Model to a Sample Variogram }
\description{
Fit ranges and/or sills from a simple or nested variogram model 
to a sample variogram }
\usage{
fit.variogram(object, model, fit.sills = TRUE, fit.ranges = TRUE,
	fit.method = 7, print.SSE = FALSE, debug.level = 1)
}
\arguments{
\item{object}{ sample variogram, output of \link{variogram} }
\item{model}{ variogram model, output of \link{vgm} }
\item{fit.sills}{ logical; determines whether the partial sill
coefficients (including nugget variance) should be fitted; or logical
vector: determines for each partial sill parameter whether it should be
fitted or fixed.  }
\item{fit.ranges}{ logical; determines whether the range coefficients
(excluding that of the nugget component) should be fitted; or logical
vector: determines for each range parameter whether it should be fitted
or fixed.  }
\item{fit.method}{ fitting method, used by gstat. The default method uses
weights $N_h/h^2$ with $N_h$ the number of point pairs and $h$ the
distance. This criterion is not supported by theory, but by practice. 
For other values of \code{fit.method}, see table 4.2 in the gstat
manual. }
\item{print.SSE}{ logical; if TRUE, print the (weighted) sum of squared
errors of the fitted model }
\item{debug.level}{ integer; set gstat internal debug level }
}
\value{
returns a fitted variogram model (of class \code{variogram.model}).
This is a data.frame with a logical attribute "singular" that
indicates whether the non-linear fit converged, or ended in a 
singularity.
}
\references{ \url{http://www.gstat.org/} 

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.
}
\author{ Edzer J. Pebesma }
\seealso{
\link{variogram},
\link{vgm}
}
\examples{
data(meuse)
vgm1 <- variogram(log(zinc)~1, ~x+y, meuse)
fit.variogram(vgm1, vgm(1,"Sph",300,1))
}

\keyword{models}

\eof
\name{fit.variogram.reml}
\alias{fit.variogram.reml}
\title{ REML Fit Direct Variogram Partial Sills to Data }
\description{
Fit Variogram Sills to Data, using REML (only for direct variograms;
not for cross variograms)
}
\usage{
fit.variogram.reml(formula, locations, data, model, debug.level = 1, set, degree = 0)
}
\arguments{
\item{formula}{formula defining the response vector and (possible) 
regressors; in case of absence of regressors, use e.g. \code{z~1}}
\item{locations}{ spatial data locations; a formula with the
coordinate variables in the right hand (dependent variable) side. }
\item{data}{data frame where the names in formula and locations 
are to be found}
\item{model}{variogram model to be fitted, output of \code{vgm}}
\item{debug.level}{debug level; set to 65 to see the iteration trace and
log likelyhood}
\item{set}{additional options that can be set; use \code{set=list(iter=100)}
to set the max. number of iterations to 100.  }
\item{degree}{order of trend surface in the location, between 0 and 3}
}

\value{ an object of class "variogram.model"; see \link{fit.variogram} }

\references{ 
Christensen, R. Linear models for multivariate, Time Series,
and Spatial Data, Springer, NY, 1991. 

Kitanidis, P., Minimum-Variance Quadratic Estimation of Covariances of
Regionalized Variables, Mathematical Geology 17 (2), 195--208, 1985 }

\author{ Edzer J. Pebesma }
\note{
This implementation only uses REML fitting of sill parameters. For each
iteration, an $n \times n$ matrix is inverted, with $n$ the number of
observations, so for large data sets this method becomes rather, ehm,
demanding. I guess there is much more to likelyhood variogram fitting in 
package \code{geoR}, and probably also in \code{nlme}.
}
\seealso{
\link{fit.variogram},
}
\examples{
data(meuse)
fit.variogram.reml(log(zinc)~1, ~x+y, meuse, model = vgm(1, "Sph", 900,1))
}
\keyword{models}

\eof
\name{fulmar}
\alias{fulmar}
\title{Fulmaris glacialis data}
\description{
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{year}{year of measurement: 1998 or 1999}
	\item{x}{x-coordinate in UTM31} 
	\item{y}{y-coordinate in UTM31}
	\item{depth}{sea water depth, in m}
	\item{coast}{distance to coast, in m}
	\item{fulmar}{observed density (number of birds per square km)}
  }
}
\usage{
data(fulmar)
}
\authors{ Dutch National Institute for Coastal and Marine Management (RIKZ)}
\references{ 
}
\note{ }
\seealso{\link{ncp.grid}}
\keyword{datasets}
\examples{
data(fulmar)
summary(fulmar)
}

\eof
\name{get.contr}
\alias{get.contr}
\title{ Calculate contrasts from multivariable predictions }
\description{ Given multivariable predictions and prediction (co)variances,
calculate contrasts and their (co)variance }
\usage{ get.contr(data, gstat.object, X, ids = names(gstat.object$data)) }
\arguments{
\item{data}{data frame, output of \link{predict.gstat} }
\item{gstat.object}{object of class \code{gstat}, used to
extract ids; may be missing if \code{ids} is used }
\item{X}{ contrast vector or matrix; the number of variables in
\code{gstat.object} should equal the number of elements in \code{X}
if \code{X} is a vector, or the number of rows in \code{X} if \code{X}
is a matrix. }
\item{ids}{ character vector with (selection of) id names, present in data }
}

\details{ 
From data, we can extract the $(n \times 1)$ vector with multivariable
predictions, say $y$, and its $(n \times n)$ covariance matrix $V$. Given
a contrast matrix in $X$, this function computes the contrast vector is
$C=X'y$ and $Var(C)=X'V X$.
}

\value{
a data frame containing for each row in \code{data} the generalized
least squares estimates (named beta.1, beta.2, ...), their
variances (named var.beta.1, var.beta.2, ...) and covariances
(named cov.beta.1.2, cov.beta.1.3, ...)
}

\references{ 
\url{http://www.gstat.org/}
}
\author{ Edzer J. Pebesma }
\note{  }

\seealso{\link{predict.gstat}}

\examples{
}

\keyword{ models }

\eof
\name{gstat-internal}
\alias{vgm.panel.xyplot}
\alias{xvgm.panel.xyplot}
\alias{vgm.dir.panel.xyplot}
\alias{load.variogram.model}
\alias{gstat.formula}
\alias{gstat.formula.predict}
\alias{gstat.debug}
\alias{gstat.set}
\alias{cross.name}
\alias{gstat.load.set}
\alias{panel.pointPairs}

\title{Gstat Internal Functions}
\description{gstat internal functions}
\author{Edzer J. Pebesma}
\note{these functions should not be called by users directly}
\keyword{internal}

\eof
\name{gstat}
\alias{gstat}
\alias{print.gstat}
\alias{"[.gstat"}
\title{ Create gstat objects, or subset it }
\description{
Function that creates gstat objects; objects that hold all the information
necessary for univariate or multivariate geostatistical prediction
(simple, ordinary or universal (co)kriging), or its conditional or
unconditional Gaussian or indicator simulation equivalents. Multivariate
gstat object can be subsetted. }
\usage{
gstat(g, id, formula, locations, data, model = NULL, beta, nmax = Inf,
	nmin = 0, maxdist = Inf, dummy = FALSE, set, fill.all = FALSE, 
	fill.cross = TRUE, variance = "identity", weights = NULL, merge, 
	degree = 0)
print.gstat(x, ...)
}
\arguments{
  \item{g}{ gstat object to append to; if missing, a new gstat object
  is created }
  \item{id}{ identifier of new variable; if missing, \code{varn} is used with
    \code{n} the number for this variable. If a cross variogram is entered,
    \code{id} should be a vector with the two \code{id} values , e.g. 
    \code{c("zn", "cd")}, further only supplying arguments \code{g} 
    and \code{model}. Note that it is not advisable to use expressions, such
	as \code{log(zinc)}, as identifiers }
  \item{formula}{ formula that defines the dependent variable as a linear
  model of independent variables; suppose the dependent variable has name
  \code{z}, for ordinary and simple kriging use the formula \code{z~1};
  for simple kriging also define \code{beta} (see below); for universal
  kriging, suppose \code{z} is linearly dependent on \code{x} and \code{y},
  use the formula \code{z~x+y}}
  \item{locations}{ formula with only independent variables that define the
  spatial data locations (coordinates), e.g. \code{~x+y}; if \code{data}
  is of class \code{spatial.data.frame}, this argument may be ignored, as 
  it can be derived from the data }
  \item{data}{ data frame; contains the dependent variable, independent
  variables, and locations. }
  \item{model}{ variogram model for this \code{id}; defined by a call to 
  \link{vgm}; see argument \code{id} to see how cross variograms are entered }
  \item{beta}{ only for simple kriging (and simulation based on simple
  kriging); vector with the trend coefficients (including intercept);
  if no independent variables are defined the model only contains an
  intercept and this should be the simple kriging mean }
  \item{nmax}{ for local kriging: the number of nearest observations that
  should be used for a kriging prediction or simulation, where nearest
  is defined in terms of the space of the spatial locations }
  \item{nmin}{ for local kriging: if the number of nearest observations
  within distance \code{maxdist} is less than \code{nmin}, a missing 
  value will be generated; see maxdist }
  \item{maxdist}{ for local kriging: only observations within a distance
  of \code{maxdist} from the prediction location are used for prediction
  or simulation; if combined with \code{nmax}, both criteria apply }
  \item{dummy}{ logical; if TRUE, consider this data as a dummy variable
  (only necessary for unconditional simulation) }
  \item{set}{ named list with optional parameters to be passed to
  gstat (only \code{set} commands of gstat are allowed, and not all of
  them may be relevant; see the manual for gstat stand-alone, URL below ) }
  \item{x}{ gstat object to print }
  \item{fill.all}{ logical; if TRUE, fill all of the direct variogram and,
  depending on the value of \code{fill.cross} also all cross
  variogram model slots in \code{g} with the given variogram model }
  \item{fill.cross}{ logical; if TRUE, fill all of the cross variograms, if
  FALSE fill only all direct variogram model slots in \code{g} with the 
  given variogram model (only if \code{fill.all} is used)}
  \item{variance}{ character; variance function to transform to non-stationary
  covariances; "identity" does not transform, other options are "mu" (Poisson)
  and "mu(1-mu)" (binomial) }
  \item{weights}{ numeric vector; if present, covariates are present,
  and variograms are missing weights are passed to OLS prediction routines;
  if variograms are given, weights should be 1/variance, where variance 
  specifies location-specific measurement error as in Delhomme, J.P.
  Kriging in the hydrosciences.  Advances in Water Resources,
  1(5):251-266, 1978; see also the section Kriging with known measurement
  errors in the gstat user's manual, URL see below.  }
  \item{merge}{ either character vector of length 2, indicating two ids 
  that share a common mean; the more general gstat merging of any two
  coefficients across variables is obtained when a list is passed, with
  each element a character vector of length 4, in the form 
  \code{c("id1", 1,"id2", 2)}. This merges the first parameter 
  for variable \code{id1} to the second of variable \code{id2}.}
  \item{degree}{order of trend surface in the location, between 0 and 3}
  \item{...}{ arguments that are passed to the printing of variogram
  models only}
}
\details{ to print the full contents of the object \code{g} returned,
use \code{as.list(g)} or \code{print.default(g)} }
\value{
an object of class \code{gstat}, which inherits from \code{list}.
Its components are: 
 \item{data}{list; each element is a list with the \code{formula}, 
  \code{locations}, \code{data}, \code{nvars}, \code{beta}, etc., for a 
  variable}
 \item{model}{list; each element contains a variogram model; names are
  those of the elements of \code{data}; cross variograms have names of
  the pairs of data elements, separated by a \code{.} (e.g.: 
  \code{var1.var2}})
 \item{set}{list; named list, corresponding to set \code{name}=\code{value};
 gstat commands (look up the set command in the gstat manual for a full list)}
}
\references{ \url{http://www.gstat.org/} 

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.
}
\author{ Edzer J. Pebesma }
\note{ 
The function currently copies the data objects into the gstat object, so
this may become a large object. I would like to copy only the name of the
data frame, but could not get this to work. Any help is appreciated.  

Subsetting (see examples) is done using the \code{id}'s of the variables,
or using numeric subsets. Subsetted gstat objects only contain cross
variograms if (i) the original gstat object contained them and (ii) the
order of the subset indexes increases, numerically, or given the order
they have in the gstat object.

The merge item may seem obscure. Still, for colocated cokriging, it is
needed. See texts by Goovaerts, Wackernagel, Chiles and Delfiner, or
look for standardised ordinary kriging in the 1992 Deutsch and Journel
or Isaaks and Srivastava. In these cases, two variables share a common
mean parameter. Gstat generalises this case: any two variables may share
any of the regression coefficients; allowing for instance analysis of
covariance models, when variograms left out (see e.g. R. Christensen's
``Plane answers'' book on linear models. The tests directory of the
package contains examples in file merge.R.

}

\seealso{ \link{predict.gstat}, \link{krige} }

\examples{
data(meuse)
# let's do some manual fitting of two direct variograms and a cross variogram
g <- gstat(id = "ln.zinc", formula = log(zinc)~1, locations = ~x+y, 
	data = meuse)
g <- gstat(g, id = "ln.lead", formula = log(lead)~1, locations = ~x+y, 
	data = meuse)
# examine variograms and cross variogram:
plot(variogram(g))
# enter direct variograms:
g <- gstat(g, id = "ln.zinc", model = vgm(.55, "Sph", 900, .05))
g <- gstat(g, id = "ln.lead", model = vgm(.55, "Sph", 900, .05))
# enter cross variogram:
g <- gstat(g, id = c("ln.zinc", "ln.lead"), model = vgm(.47, "Sph", 900, .03))
# examine fit:
plot(variogram(g), model = g$model, main = "models fitted by eye")
# see also demo(cokriging) for a more efficient approach
g["ln.zinc"]
g["ln.lead"]
g[c("ln.zinc", "ln.lead")]
g[1]
g[2]

# Inverse distance interpolation with inverse distance power set to .5:
# (kriging variants need a variogram model to be specified)
data(meuse)
data(meuse.grid)
meuse.gstat <- gstat(id = "zinc", formula = zinc ~ 1, locations = ~ x + y,
	data = meuse, nmax = 7, set = list(idp = .5))
meuse.gstat
z <- predict(meuse.gstat, meuse.grid)
levelplot(zinc.pred~x+y, z, aspect = mapasp(z))
# see demo(cokriging) and demo(examples) for further examples, 
# and the manuals for predict.gstat and image
}
\keyword{ models }

\eof
\name{image}
\alias{image.data.frame}
\alias{image}
\alias{xyz2img}
\title{
Image Gridded Coordinates in Data Frame
}
\description{
Image gridded data, held in a data frame, keeping the
right aspect ratio for axes, and the right cell shape
}
\usage{
image.data.frame(x, zcol = 3, xcol = 1, ycol = 2, ...)
xyz2img(xyz, zcol = 3, xcol = 1, ycol = 2)
}
\arguments{
\item{x}{ data frame (or matrix) with x-coordinate,
y-coordinate, and z-coordinate in its columns }
\item{zcol}{ column number of z-variable }
\item{xcol}{ column number of x-coordinate }
\item{ycol}{ column number of y-coordinate }
\item{...}{ arguments, passed to image.default }
\item{xyz}{data frame (same as \code{x})}
}
\value{
\link{image.data.frame} plots an image from gridded data, organized
in arbritrary order, in a data frame. It uses \link{xyz2img} and
\link{image.default} for this. In the S-Plus version, \link{xyz2img}
tries to make an image object with a size such that it will plot with
an equal aspect ratio; for the R version, image.data.frame uses the
\code{asp=1} argument to guarantee this.

\link{xyz2img} returns a list with components: \code{z}, a matrix
containing the z-values; \code{x}, the increasing coordinates of the
rows of \code{z}; \code{y}, the increasing coordinates of the columns
of \code{z}. This list is suitable input to \link{image.default}.
}
\note{
I wrote this function before I found out about \code{levelplot},
a Lattice/Trellis function that lets you control the aspect ratio by
the \code{aspect} argument, and that automatically draws a legend, and
therefore I now prefer levelplot over \code{image}. Plotting points
on a levelplots is probably done with providing a panel function and
using \code{lpoints}.

(for S-Plus only -- ) it is hard (if not impossible) to get exactly right
cell shapes (e.g., square for a square grid) without altering the size of
the plotting region, but this function tries hard to do so by extending
the image to plot in either x- or y-direction.  The larger the grid, the
better the approximation. Geographically correct images can be obtained
by modifiying \code{par("pin")}. Read the examples, image a 2 x 2 grid,
and play with \code{par("pin")} if you want to learn more about this.
}
\references{ }
\author{ Edzer J. Pebesma }
\seealso{ }
\examples{
data(meuse)
data(meuse.grid)
g <- gstat(formula=log(zinc)~1,locations=~x+y,data=meuse,model=vgm(1,"Exp",300))
x <- predict(g, meuse.grid)
image(x, 4, main="kriging variance and data points")
points(meuse$x, meuse$y, pch = "+")
# non-square cell test:
image(x[((x$y - 20) \%\% 80) == 0,], main = "40 x 80 cells")
image(x[((x$x - 20) \%\% 80) == 0,], main = "80 x 40 cells")
# the following works for square cells only:
oldpin <- par("pin")
ratio <- length(unique(x$x))/length(unique(x$y))
par(pin = c(oldpin[2]*ratio,oldpin[2]))
image(x, main="Exactly square cells, using par(pin)")
par(pin = oldpin)
levelplot(var1.var~x+y, x, aspect = mapasp(x), main = "kriging variance")
}

\keyword{dplot}


\eof
\name{krige}
\alias{krige}
\title{ Simple, Ordinary or Universal, global or local, Point or Block Kriging }
\description{
Function for simple, ordinary or universal kriging (sometimes called
external drift kriging), kriging in a local neighbourhood, point kriging
or kriging of block mean values (rectangular or irregular blocks), and
conditional (Gaussian or indicator) simulation equivalents for all kriging
varieties.
}
\usage{ krige(formula, locations, data, newdata, model, beta, nmax = Inf,
nmin = 0, maxdist = Inf, block, nsim = 0, indicators = FALSE, 
na.action = na.pass, ...) }
\arguments{
 \item{formula}{ formula that defines the dependent variable as a linear
  model of independent variables; suppose the dependent variable has name
  \code{z}, for ordinary and simple kriging use the formula \code{z~1};
  for simple kriging also define \code{beta} (see below); for universal
  kriging, suppose \code{z} is linearly dependent on \code{x} and \code{y},
  use the formula \code{z~x+y}}
 \item{locations}{ formula with only independent variables that define the
  spatial data locations (coordinates), e.g. \code{~x+y}; if \code{data}
  is of class \code{spatial.data.frame}, this argument may be ignored, as
  it can be derived from the data }
 \item{data}{ data frame; should contain the dependent variable, independent
  variables, and coordinates. }
 \item{newdata}{ data frame with prediction/simulation locations; should 
  contain columns with the independent variables (if present) and the
  coordinates with names as defined in \code{locations} }
 \item{model}{ variogram model of dependent variable (or its residuals), 
  defined by a call to \link{vgm} or \link{fit.variogram}}
 \item{beta}{ only for simple kriging (and simulation based on simple
  kriging); vector with the trend coefficients (including intercept);
  if no independent variables are defined the model only contains an
  intercept and this should be the simple kriging mean }
 \item{nmax}{ for local kriging: the number of nearest observations that
  should be used for a kriging prediction or simulation, where nearest
  is defined in terms of the space of the spatial locations. By default,
  all observations are used }
 \item{nmin}{ for local kriging: if the number of nearest observations
  within distance \code{maxdist} is less than \code{nmin}, a missing 
  value will be generated; see maxdist }
 \item{maxdist}{ for local kriging: only observations within a distance
  of \code{maxdist} from the prediction location are used for prediction
  or simulation; if combined with \code{nmax}, both criteria apply }
 \item{block}{ block size; a vector with 1, 2 or 3 values containing
  the size of a rectangular in x-, y- and z-dimension respectively
  (0 if not set), or a data frame with 1, 2 or 3 columns, containing
  the points that discretize the block in the x-, y- and z-dimension
  to define irregular blocks relative to (0,0) or (0,0,0)---see also the details 
  section of \link{predict.gstat}. By default, predictions or simulations 
  refer to the support of the data values. }
 \item{nsim}{ integer; if set to a non-zero value, conditional simulation
  is used instead of kriging interpolation. For this, sequential Gaussian
  or indicator simulation is used (depending on the value of 
  \code{indicators}), following a single random path through the data.  }
 \item{indicators}{ logical, only relevant if \code{nsim} is non-zero; if
  TRUE, use indicator simulation; else use Gaussian simulation }
 \item{na.action}{ function determining what should be done with missing
  values in 'newdata'.  The default is to predict 'NA'.  Missing values 
  in coordinates and predictors are both dealt with. }
 \item{\dots}{ other arguments that will be passed to \link{gstat}}
}
\details{
This function is a simple wrapper function around \link{gstat} and
\link{predict.gstat} for univariate kriging prediction and conditional
simulation methods available in gstat. For multivariate prediction or
simulation, or for other interpolation methods provided by gstat (such as
inverse distance weighted interpolation or trend surface interpolation)
use the functions \link{gstat} and \link{predict.gstat} directly.

For further details, see \link{predict.gstat}.
}

\value{
a data frame containing the coordinates of \code{newdata}, and columns
of prediction and prediction variance (in case of kriging) or the
\code{abs(nsim)} columns of the conditional Gaussian or indicator
simulations }

\references{ N.A.C. Cressie, 1993, Statistics for Spatial Data,
Wiley. 

\url{http://www.gstat.org/}

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.
}
\author{ Edzer J. Pebesma }
\note{  
Daniel G. Krige is a South African scientist who was a mining engineer
when he first used generalised least squares prediction with spatial
covariances in the 50's. George Matheron coined the term \code{kriging}
in the 60's for the action of doing this, although very similar approaches
had been taken in the field of meteorology. Beside being Krige's name,
I consider "krige" to be to "kriging" what "predict" is to "prediction".
}

\seealso{ \link{gstat}, \link{predict.gstat} }

\examples{
data(meuse)
data(meuse.grid)
m <- vgm(.59, "Sph", 874, .04)
# ordinary kriging:
x <- krige(log(zinc)~1, ~x+y, model = m, data = meuse, newd = meuse.grid)
levelplot(var1.pred~x+y, x, aspect = mapasp(x),
	main = "ordinary kriging predictions")
levelplot(var1.var~x+y, x, aspect = mapasp(x),
	main = "ordinary kriging variance")
# simple kriging:
x <- krige(log(zinc)~1, ~x+y, model = m, data = meuse, newdata = meuse.grid, 
	beta=5.9)
# residual variogram:
m <- vgm(.4, "Sph", 954, .06)
# universal block kriging:
x <- krige(log(zinc)~x+y, ~x+y, model = m, data = meuse, newdata = 
	meuse.grid, block = c(40,40))
levelplot(var1.pred~x+y, x, aspect = mapasp(x),
	main = "universal kriging predictions")
levelplot(var1.var~x+y, x, aspect = mapasp(x),
	main = "universal kriging variance")
}
\keyword{ models }

\eof
\name{krige.cv}
\alias{krige.cv}
\alias{gstat.cv}
\title{ (co)kriging cross validation, n-fold or leave-one-out }
\description{ Cross validation functions for simple, ordinary or universal
point (co)kriging, kriging in a local neighbourhood.  }
\usage{
gstat.cv(object, nfold, remove.all = FALSE, verbose = FALSE, 
	all.residuals = FALSE, ...)
krige.cv(formula, locations, data, model = NULL, beta = NULL, nmax = Inf, 
	nmin = 0, maxdist = Inf, nfold = nrow(data), verbose = FALSE, ...)
}
\arguments{
  \item{object}{ object of class gstat; see function \link{gstat}}
  \item{nfold}{ apply n-fold cross validation; if \code{nfold} is set
  to \code{nrow(data)} (the default), leave-one-out cross validation is
  done; if set to e.g. 5, five-fold cross validation is done }
  \item{remove.all}{ logical; if TRUE, remove observations at cross validation
  locations not only for the first, but for all subsequent variables as well }
  \item{verbose}{ logical; if TRUE, progress is printed }
  \item{all.residuals}{ logical; if TRUE, residuals for all variables are
  returned instead of for the first variable only}
  \item{\dots}{ other arguments that will be passed to \link{predict.gstat}
  in case of \code{gstat.cv}, or to \link{gstat} in case of \code{krige.cv}}
  \item{formula}{ formula that defines the dependent variable as a linear
  model of independent variables; suppose the dependent variable has name
  \code{z}, for ordinary and simple kriging use the formula \code{z~1};
  for simple kriging also define \code{beta} (see below); for universal
  kriging, suppose \code{z} is linearly dependent on \code{x} and \code{y},
  use the formula \code{z~x+y}}
  \item{locations}{ formula with only independent variables that define the
  spatial data locations (coordinates), e.g. \code{~x+y}; if \code{data}
  is of class \code{spatial.data.frame}, this argument may be ignored, as
  it can be derived from the data }
  \item{data}{ data frame; should contain the dependent variable, independent
  variables, and coordinates. }
  \item{model}{ variogram model of dependent variable (or its residuals), 
  defined by a call to \link{vgm} or \link{fit.variogram}}
  \item{beta}{ only for simple kriging (and simulation based on simple
  kriging); vector with the trend coefficients (including intercept);
  if no independent variables are defined the model only contains an
  intercept and this should be the simple kriging mean }
  \item{nmax}{ for local kriging: the number of nearest observations that
  should be used for a kriging prediction or simulation, where nearest
  is defined in terms of the space of the spatial locations. By default,
  all observations are used }
  \item{nmin}{ for local kriging: if the number of nearest observations
  within distance \code{maxdist} is less than \code{nmin}, a missing 
  value will be generated; see maxdist }
  \item{maxdist}{ for local kriging: only observations within a distance
  of \code{maxdist} from the prediction location are used for prediction
  or simulation; if combined with \code{nmax}, both criteria apply }
}

\details{ Leave-one-out cross validation (LOOCV) visits a data point, and
predicts the value at that location by leaving out the observed value,
and proceeds with the next data point. (The observed value is left
out because kriging would otherwise predict the value itself.) N-fold
cross validation makes a partitions the data set in N parts. For all
observation in a part, predictions are made based on the remaining N-1
parts; this is repeated for each of the N parts. N-fold cross validation
may be faster than LOOCV. }

\value{ data frame containing the coordinates of \code{data} or those
of the first variable in \code{object}, and columns of prediction and
prediction variance of cross validated data points, observed values,
residuals, zscore (residual divided by kriging standard error), and fold.

If \code{all.residuals} is true, a data frame with residuals for all
variables is returned, without coordinates. }

\references{ \url{http://www.gstat.org/} }
\author{ Edzer J. Pebesma }
\note{ Leave-one-out cross validation seems to be much faster in plain
(stand-alone) gstat, apparently quite a bit of the effort is spent moving 
data around from R to gstat. }

\seealso{ \link{krige}, \link{gstat}, \link{predict.gstat} }

\examples{
data(meuse)
m <- vgm(.59, "Sph", 874, .04)
# five-fold cross validation:
x <- krige.cv(log(zinc)~1, ~x+y, model = m, data = meuse, nmax = 40, nfold=5)
bubble(x, z = "residual", main = "log(zinc): 5-fold CV residuals")
}
\keyword{ models }

\eof
\name{makegrid}
\alias{makegrid}
\title{ make regular grid with square cells and round numbers }
\description{ make regular grid with square cells and round numbers }
\usage{
makegrid(x, y, n=10000, nsig=2, margin=1.05, cell.size)
}
\arguments{
\item{x}{ x-coordinate }
\item{y}{ y-coordinate }
\item{n}{ approximate number of cells in grid }
\item{nsig}{number of significant digits to which cell size and origin are rounded}
\item{margin}{margin around the x and y coordinate limits}
\item{cell.size}{cell size; if missing, a reasonable, and rounded, estimate is made}
}
\value{
data frame with the following elements:
\item{x}{ x-coordinates }
\item{y}{ y-coordinate } 
}
\examples{
data(meuse)
grd <- makegrid(meuse$x, meuse$y, 1000)
diff(grd$x)
diff(grd$y)
summary(grd)
grd <- makegrid(meuse$x, meuse$y, cell.size = 40)
diff(grd$x)
diff(grd$y)
summary(grd)
}
\keyword{dplot}

\eof
\name{map.to.lev}
\alias{map.to.lev}
\title{ rearrange data frame for plotting with levelplot }
\description{ rearrange data frame for plotting with levelplot }
\usage{
map.to.lev(data, xcol = 1, ycol = 2, zcol = c(3, 4), ns = names(data)[zcol])
}
\arguments{
\item{data}{ data frame, e.g. output from \link{krige} or \link{predict.gstat} }
\item{xcol}{ x-coordinate column number }
\item{ycol}{ y-coordinate column number }
\item{zcol}{ z-coordinate column number range }
\item{ns}{names of the set of z-columns to be viewed}
}
\value{
data frame with the following elements:
\item{x}{ x-coordinate for each row}
\item{y}{ y-coordinate for each row} 
\item{z}{ column vector with each of the elements in columns \code{zcol}
of \code{data} stacked } 
\item{name}{ factor; name of each of the stacked \code{z} columns } 
}
\seealso{ \code{levelplot}, \link{image.data.frame}, \link{krige};
for examples see \link{predict.gstat} }
\keyword{dplot}

\eof
\name{mapasp}
\alias{mapasp}
\title{ Calculate plot aspect ratio for geographic maps }
\description{ Calculate plot aspect ratio for geographic maps }
\usage{
mapasp(data, x = data$x, y = data$y)
}
\arguments{
\item{data}{ data frame }
\item{x}{ x-coordinates }
\item{y}{ y-coordinates }
}
\value{
	diff(range(y))/diff(range(x))
}
\seealso{ \link{image.data.frame}, \link{krige}}

\keyword{dplot}

\eof
\name{meuse}
\alias{meuse}
\title{Meuse river data set}
\description{
This data set gives locations and top soil heavy metal concentrations
(ppm), along with a number of soil and landscape variables, collected in
a flood plain of the river Meuse, near the village Stein. Heavy metal
concentrations are bulk sampled from an area of approximately 15 m
x 15 m.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{x}{a numeric vector; x-coordinate (m)
	in RDM (Dutch topographical map coordinates) }
    \item{y}{a numeric vector; y-coordinate (m)
	in RDM (Dutch topographical map coordinates)}
	\item{cadmium}{topsoil cadmium concentration, ppm.; note that zero cadmium
	values in the original data set have been shifted to 0.2 (half the lowest 
	non-zero value) }
	\item{copper}{topsoil copper concentration, ppm. }
	\item{lead}{topsoil lead concentration, ppm. }
	\item{zinc}{topsoil zinc concentration, ppm. } 
	\item{elev}{relative elevation}
	\item{dist}{distance to river Meuse; obtained from the nearest cell in
	\link{meuse.grid}, which in turn was derived by a spread (spatial distance)
	GIS operation, therefore it is accurate up to 20 metres; 
	normalized $[0,1]$ }
	\item{om}{organic matter, as percentage }
	\item{ffreq}{flooding frequency class}
	\item{soil}{soil type}
	\item{lime}{lime class}
	\item{landuse}{landuse class}
	\item{dist.m}{distance to river Meuse (metres), as obtained during
	the field survey}
  }
}
\usage{
data(meuse)
}
\authors{ The actual field data were collected by Ruud van Rijn and
Mathieu Rikken }
\references{ 
P.A. Burrough, R.A. McDonnell, 1998. Principles of Geographical Information
Systems. Oxford University Press.

\url{http:/www.gstat.org/}
}
\note{ \code{Sample} refers to original sample number (9 of the original 164
samples were discarded) }
\seealso{\link{meuse.all}}
\keyword{datasets}
\examples{
data(meuse)
summary(meuse)
}

\eof
\name{meuse.all}
\alias{meuse.all}
\title{Meuse river data set -- original, full data set}
\description{
This data set gives locations and top soil heavy metal concentrations
(ppm), along with a number of soil and landscape variables, collected in
a flood plain of the river Meuse, near the village Stein. Heavy metal
concentrations are bulk sampled from an area of approximately 15 m
x 15 m.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{sample}{sample number}
    \item{x}{a numeric vector; x-coordinate (m)
	in RDM (Dutch topographical map coordinates) }
    \item{y}{a numeric vector; y-coordinate (m)
	in RDM (Dutch topographical map coordinates)}
	\item{cadmium}{topsoil cadmium concentration, ppm.; note that zero cadmium
	values in the original data set have been shifted to 0.2 (half the lowest 
	non-zero value) }
	\item{copper}{topsoil copper concentration, ppm. }
	\item{lead}{topsoil lead concentration, ppm. }
	\item{zinc}{topsoil zinc concentration, ppm. } 
	\item{elev}{relative elevation}
	\item{om}{organic matter, as percentage }
	\item{ffreq}{flooding frequency class}
	\item{soil}{soil type}
	\item{lime}{lime class}
	\item{landuse}{landuse class}
	\item{dist.m}{distance to river Meuse (metres), as obtained during
	the field survey}
	\item{in.pit}{logical; indicates whether this is a sample taken in
	a pit}
	\item{in.meuse155}{logical; indicates whether the sample is part of
	the \code{meuse} (i.e., filtered) data set; in addition to the samples
	in a pit, an sample (139) with outlying zinc content was removed }
	\item{in.BMcD}{logical; indicates whether the sample is used as part
	of the subset of 98 points in the various interpolation examples of
	Burrough \& McDonnell}
  } 
} 
\usage{
data(meuse.all)
}
\authors{ The actual field data were collected by Ruud van Rijn and
Mathieu Rikken }
\references{ 
P.A. Burrough, R.A. McDonnell, 1998. Principles of Geographical Information
Systems. Oxford University Press.

\url{http:/www.gstat.org/}
}
\note{ \code{sample} refers to original sample number.  Eight samples
were left out because they were not indicative for the metal content of
the soil. They were taken in an old pit. One sample contains an outlying
zinc value, which was also discarded for the meuse (155) data set.
}
\seealso{\link{meuse}}
\keyword{datasets}
\examples{
data(meuse.all)
summary(meuse.all)
}

\eof
\name{meuse.alt}
\alias{meuse.alt}
\title{Meuse river altitude data set}
\description{
This data set gives a point set with altitudes, digitized from the
1:10,000 topographical map of the Netherlands.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{x}{a numeric vector; x-coordinate (m) in RDM (Dutch topographical map coordinates) }
    \item{y}{a numeric vector; y-coordinate (m) in RDM (Dutch topographical map coordinates)}
	\item{alt}{altitude in m. above NAP (Dutch zero for sea level)}
  }
}
\usage{
data(meuse.alt)
}
\references{ \url{http:/www.gstat.org/} }
\seealso{\link{meuse}}
\keyword{datasets}
\examples{
data(meuse.alt)
xyplot(y~x, meuse.alt)
}

\eof
\name{meuse.grid}
\alias{meuse.grid}
\non_function{}
\title{ Prediction Grid for Meuse Data Set }
\usage{data(meuse.grid)}
\description{
The \code{meuse.grid} data frame has 3103 rows and 2 columns;
a grid with 40 m x 40 m spacing that covers the Meuse Study area
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{x}{a numeric vector; x-coordinate (see \link{meuse}) }
    \item{y}{a numeric vector; y-coordinate (see \link{meuse}) }
	\item{dist}{distance to the Meuse river; obtained by a spread
	(spatial distance) GIS operation, from border of river; normalized to
	$[0,1]$ }
	\item{ffreq}{flood frequency class}
	\item{part.a}{arbitrary division of the area in two areas, a and b}
	\item{part.b}{see \code{part.a}}
	\item{soil}{soil class (questionable whether this is from a real soil map)}
  }
}
\details{
\code{x} and \code{y} are in RDM, the Dutch topographical map
coordinate system. Roger Bivand projected this to UTM in the
R-Grass interface package.
}
\source{
\url{http://www.gstat.org/}
}
\references{
See the \link{meuse} documentation
}
\examples{
data(meuse.grid)
library(MASS)
eqscplot(meuse.grid,pch="+")
}
\keyword{datasets}

\eof
\name{ncp.grid}
\alias{ncp.grid}
\title{Grid for the NCP, the Dutch part of the North Sea}
\description{
}
\format{
  This data frame contains the following columns:
  \describe{
   \item{x}{x-coordinate, UTM31}
   \item{y}{y-coordinate, UTM31}
   \item{depth}{sea water depth, m}
   \item{coast}{distance to coast, m}
   \item{area}{identifier for administrative sub-areas}
  }
}
\usage{
data(ncp.grid)
}
\authors{Dutch National Institute for Coastal and Marine Management (RIKZ)}
\references{ 
}
\note{ }
\seealso{\link{fulmar}}
\keyword{datasets}
\examples{
data(ncp.grid)
summary(ncp.grid)
}

\eof
\name{ossfim}
\alias{ossfim}
\title{ Kriging standard errors as function of grid spacing and block size}
\description{ Calculate, for a given variogram model, ordinary block
kriging standard errors as a function of sampling spaces and block sizes }
\usage{
ossfim(spacings = 1:5, block.sizes = 1:5, model, nmax = 25, debug = 0)
}
\arguments{
\item{spacings}{range of grid (data) spacings to be used}
\item{block.sizes}{ range of block sizes to be used}
\item{model}{variogram model, output of \code{vgm}}
\item{nmax}{set the kriging neighbourhood size}
\item{debug}{debug level; set to 32 to see a lot of output}
}

\value{ data frame with columns \code{spacing} (the grid spacing),
\code{block.size} (the block size), and \code{kriging.se} (block kriging
standard error) }

\references{ 
Burrough, P.A., R.A. McDonnell (1999) Principles of Geographical
Information Systems. Oxford University Press (e.g., figure 10.11 on
page 261)

Burgess, T.M., R. Webster, A.B. McBratney (1981) Optimal interpolation and
isarithmic mapping of soil properties. IV Sampling strategy.  The journal
of soil science 32(4), 643-660.

McBratney, A.B., R. Webster (1981) The design of optimal sampling schemes
for local estimation and mapping of regionalized variables: 2 program
and examples. Computers and Geosciences 7: 335-365.

read more on a simplified, web-based version on
\url{http://www.gstat.org/ossfim.html}
}

\author{ Edzer J. Pebesma }
\note{ The idea is old, simple, but still of value. If you want to map
a variable with a given accuracy, you will have to sample it. Suppose
the variogram of the variable is known. Given a regular sampling scheme,
the kriging standard error decreases when either (i) the data spacing
is smaller, or (ii) predictions are made for larger blocks. This function
helps quantifying this relationship. Ossfim probably refers to ``optimal
sampling scheme for isarithmic mapping''.
}

\seealso{
\link{krige}
}
\examples{
x <- ossfim(1:15,1:15, model = vgm(1,"Exp",15))
levelplot(kriging.se~spacing+block.size, x, 
  main = "Ossfim results, variogram 1 Exp(15)")
# if you wonder about the decrease in the upper left corner of the graph,
# try the above with nmax set to 100, or perhaps 200.
}
\keyword{models}

\eof


\name{oxford}
\alias{oxford}
\title{Oxford soil samples}
\description{
Data: 126 soil augerings on a 100 x 100m square grid, with 6 columns
and 21 rows. Grid is oriented with long axis North-north-west to
South-south-east Origin of grid is South-south-east point, 100m outside
grid.

Original data are part of a soil survey carried out by P A. Burrough in
1967. The survey area is located on the chalk downlands on the Berkshire
Downs in Oxfordshire, UK. Three soil profile units were recognised on
the shallow Rendzina soils; these are Ia - very shallow, grey calcareous
soils less than 40cm deep over chalk;  Ct - shallow to moderately deep,
grey-brown calcareous soils  on calcareous colluvium, and Cr: deep,
moderately acid, red-brown clayey soils.  These soil profile classes
were registered at every augering.

In addition, an independent landscape soil map was made by interpolating
soil boundaries between these soil types, using information from the
changes in landform.  Because the soil varies over short distances,
this field mapping caused some soil borings to receive a different
classification from the classification based on the point data.

Also registered at each auger point were the site elevation (m), the
depth to solid chalk rock (in cm) and the depth to lime in cm. Also,
the percent clay content, the Munsell colour components of VALUE and
CHROMA , and the lime content of the soil (as tested using HCl)  were
recorded for the top two soil layers (0-20cm and 20-40cm).

Samples of topsoil taken as a bulk sample within a circle of radius 2.5m
around each sample point were used for the laboratory determination of Mg
(ppm), OM1 \%, CEC as mequ/100g air dry soil, pH, P as ppm and K (ppm).

}
\format{
This data frame contains the following columns:
\describe{
\item{PROFILE}{profile number}
\item{XCOORD}{x-coordinate, field, non-projected}
\item{YCOORD}{y-coordinate, field, non-projected}
\item{ELEV}{elevation, m.}
\item{PROFCLASS}{soil class, obtained by classifying the soil profile at the
sample site }
\item{MAPCLASS}{soil class, obtained by looking up the site location in the soil map}
\item{VAL1}{Munsell colour component VALUE, 0-20 cm}
\item{CHR1}{Munsell colour component CHROMA, 20-40 cm}
\item{LIME1}{Lime content (tested using HCl), 0-20 cm}
\item{VAL2}{Munsell colour component VALUE, 0-20 cm}
\item{CHR2}{Munsell colour component CHROMA, 20-40 cm}
\item{LIME2}{Lime content (tested using HCl), 20-40 cm}
\item{DEPTHCM}{soil depth, cm }
\item{DEP2LIME}{depth to lime, cm }
\item{PCLAY1}{percentage clay, 0-20 cm}
\item{PCLAY2}{percentage clay, 20-40 cm}
\item{MG1}{Magnesium content (ppm), 0-20 cm}
\item{OM1}{organic matter (\%), 0-20 cm}
\item{CEC1}{CES as mequ/100g air dry soil, 0-20 cm}
\item{PH1}{pH, 0-20 cm}
\item{PHOS1}{Phosphorous, 0-20 cm, ppm}
\item{POT1}{K (potassium), 0-20 cm, ppm}
}
}
\usage{
data(oxford)
}
\authors{ P.A. Burrough }
\references{ 
P.A. Burrough, R.A. McDonnell, 1998. Principles of Geographical Information
Systems. Oxford University Press.
}
\note{ \code{oxford.jpg}, in the gstat package data directory,
shows an image of the soil map for the region }
\keyword{datasets}
\examples{
data(oxford)
summary(oxford)
}

\eof
\name{plot.gstatVariogram}
\alias{plot.gstatVariogram}
\alias{plot.variogramMap}
\title{
Plot a Sample Variogram
}
\description{
Creates a variogram plot
}
\usage{
plot.gstatVariogram(x, model = NULL, ylim, xlim, xlab = "distance", 
	ylab = "semivariance", multipanel = TRUE, plot.numbers = FALSE, 
	scales, ids = x$id, group.id = TRUE, skip, layout, ...)
plot.variogramMap(x, np = FALSE, skip, ...)
}
\arguments{
\item{x}{ object of class "gstatVariogram", obtained from the function 
\link{variogram}, possibly containing directional or cross variograms }
\item{model}{ in case of a single variogram: a variogram model, as 
obtained from \link{vgm} or \link{fit.variogram}, to be drawn as 
a line in the variogram plot; in case of a set of variograms and
cross variograms: a list with variogram models }
\item{ylim}{ numeric vector of length 2, limits of the y-axis}
\item{xlim}{ numeric vector of length 2, limits of the x-axis}
\item{xlab}{ x-axis label }
\item{ylab}{ y-axis label }
\item{multipanel}{ logical; if TRUE, directional variograms are plotted in
different panels, if FALSE, directional variograms are plotted in the same
graph, using color, colored lines and symbols to distinguish them }
\item{plot.numbers}{ logical; if TRUE, plot number of point pairs next to
each plotted semivariance symbol }
\item{scales}{ optional argument that will be passed to \code{xyplot} in
case of the plotting of variograms and cross variograms }
\item{ids}{ ids of the data variables and variable pairs }
\item{group.id}{ logical; control for directional multivariate variograms:
if TRUE, panels divide direction and colors indicate variables
(ids), if FALSE panels divide variables/variable pairs and colors
indicate direction}
\item{skip}{ logical; can be used to arrange panels, see \code{xyplot}}
\item{layout}{ integer vector; can be used to set panel layout: c(ncol,nrow) }
\item{np}{ logical; if TRUE, plot number of point pairs, if FALSE plot
semivariances }
\item{\dots}{ any arguments that will be passed to the panel plotting functions
(such as \code{auto.key} in examples below) }
}
\value{
returns (or plots) the variogram plot
}
\references{ \url{http://www.gstat.org}}
\author{ Edzer J. Pebesma }
\note{ currently, plotting models and/or point pair numbers is not
supported when a variogram is both directional and multivariable; also,
three-dimensional directional variograms will probably not be displayed
correctly. }
\seealso{
\link{variogram},
\link{fit.variogram}, 
\link{vgm}
\link{variogram.line},
}
\examples{
data(meuse)
vgm1 <- variogram(log(zinc)~1, ~x+y, meuse)
plot(vgm1)
model.1 <- fit.variogram(vgm1,vgm(1,"Sph",300,1))
plot(vgm1, model=model.1)
plot(vgm1, plot.numbers = TRUE, pch = "+")
vgm2 <- variogram(log(zinc)~1, ~x+y, meuse, alpha=c(0,45,90,135))
plot(vgm2)
# the following demonstrates plotting of directional models:
model.2 <- vgm(.59,"Sph",926,.06,anis=c(0,0.3))
plot(vgm2, model=model.2)
 
g = gstat(id="zinc < 200", form=I(zinc<200)~1,loc=~x+y,data=meuse)
g = gstat(g, id="zinc < 400", form=I(zinc<400)~1,loc=~x+y,data=meuse)
g = gstat(g, id="zinc < 800", form=I(zinc<800)~1,loc=~x+y,data=meuse)
# calculate multivariable, directional variogram:
v = variogram(g, alpha=c(0,45,90,135))
plot(v, group.id = FALSE, auto.key = TRUE) # id and id pairs panels
plot(v, group.id = TRUE, auto.key = TRUE)  # direction panels

#if (require(sp)) {
#  plot(variogram(g, cutoff=1000, width=100, map=TRUE),
#  	main = "(cross) semivariance maps")
#  plot(variogram(g, cutoff=1000, width=100, map=TRUE), np=TRUE,
#  	main = "number of point pairs")
#}
}

\keyword{dplot}

\eof
\name{plot.pointPairs}
\alias{plot.pointPairs}
\title{
Plot a point pairs, identified from a variogram cloud
}
\description{
Plot a point pairs, identified from a variogram cloud
}
\usage{
plot.pointPairs(x, data, xcol = data$x, ycol = data$y, xlab = "x coordinate", 
ylab = "y coordinate", col.line = 2, line.pch = 0, ...) 
}
\arguments{
\item{x}{ object of class "pointPairs", obtained from the function 
\link{plot.variogramCloud}, containing point pair indices }
\item{data}{ data frame to which the indices refer (from which the
variogram cloud was calculated) }
\item{xcol}{ numeric vector with x-coordinates of data }
\item{ycol}{ numeric vector with y-coordinates of data }
\item{xlab}{ x-axis label }
\item{ylab}{ y-axis label }
\item{col.line}{ color for lines connecting points }
\item{line.pch}{ if non-zero, symbols are also plotted at the middle of
line segments, to mark lines too short to be visible on the plot;
the color used is \code{col.line}; the value passed to this argument
will be used as plotting symbol (pch) }
\item{...}{ arguments, further passed to \code{xyplot}}
}
\value{
plots the data locations, with lines connecting the point pairs identified
(and refered to by indices in) x
}
\references{ \url{http://www.gstat.org}}
\author{ Edzer J. Pebesma }
\seealso{
\link{plot.variogramCloud}
}
\examples{
### The following requires interaction, and is therefore outcommented
#data(meuse)
#vgm1 <- variogram(log(zinc)~1, ~x+y, meuse, cloud = TRUE)
#pp <- plot(vgm1, id = TRUE)
### Identify the point pairs
#plot(pp, data = meuse) # meuse has x and y as coordinates
}
\keyword{dplot}

\eof
\name{plot.variogramCloud}
\alias{plot.variogramCloud}
\title{
Plot and Identify Data Pairs on Sample Variogram Cloud
}
\description{
Plot a sample variogram cloud, possibly with identification of
individual point pairs
}
\usage{
plot.variogramCloud(x, identify = FALSE, digitize = FALSE, xlim, ylim, xlab, ylab, 
	keep = FALSE, ...)
}
\arguments{
\item{x}{ object of class \code{variogramCloud}}
\item{identify}{ logical; if TRUE, the plot allows identification of
a series of individual point pairs that correspond to individual variogram
cloud points (use left mouse button to select; right mouse button ends) }
\item{digitize}{ logical; if TRUE, select point pairs by digitizing a
region with the mouse (left mouse button adds a point, right mouse button
ends) }
\item{xlim}{ limits of x-axis }
\item{ylim}{ limits of y-axis }
\item{xlab}{ x axis label }
\item{ylab}{ y axis label }
\item{keep}{ logical; if TRUE and \code{identify} is TRUE, the labels
identified and their position are kept and glued to object x, which is
returned. Subsequent calls to plot this object will now have the labels
shown, e.g. to plot to hardcopy }
\item{...}{ parameters that are passed through to \link{plot.gstatVariogram}
(in case of identify = FALSE) or to plot (in case of identify = TRUE) }
}
\value{
If \code{identify} or \code{digitize} is TRUE, a data frame of class
\code{pointPairs} with in its rows the point pairs identified (pairs of
row numbers in the original data set); if identify is F, a plot of the
variogram cloud, which uses \link{plot.gstatVariogram}

If in addition to \code{identify}, \code{keep} is also TRUE, an object
of class \code{variogramCloud} is returned, having attached to it attributes
"sel" and "text", which will be used in subsequent calls to plot.variogramCloud
with \code{identify} set to FALSE, to plot the text previously identified.

If in addition to \code{digitize}, \code{keep} is also TRUE, an object of
class \code{variogramCloud} is returned, having attached to it attribute
"poly", which will be used in subsequent calls to plot.variogramCloud
with \code{digitize} set to FALSE, to plot the digitized line.

In both of the \code{keep = TRUE} cases, the attribute \code{ppairs} of
class \code{pointPairs} is present, containing the point pairs identified.
}
\references{ 
\url{http://www.gstat.org/}
}
\author{ Edzer J. Pebesma }
\seealso{
\link{variogram},
\link{plot.gstatVariogram},
\link{plot.pointPairs},
\link{identify},
\link{locator}
}
\examples{
data(meuse)
plot(variogram(log(zinc)~1, loc=~x+y, data=meuse, cloud=TRUE))
## commands that require interaction:
# x <- variogram(log(zinc)~1, loc=~x+y, data=meuse, cloud=TRUE)
# plot(plot(x, idendify = TRUE), meuse)
# plot(plot(x, digitize = TRUE), meuse)
}

\keyword{dplot}

\eof
\name{point.in.polygon}
\alias{point.in.polygon}
\title{ do point(s) fall in a given polygon? }
\description{
verifies for one or more points whether they fall in a given polygon
}
\usage{
point.in.polygon(point.x, point.y, pol.x, pol.y)
}
\arguments{
\item{point.x}{numerical array of x-coordinates of points}
\item{point.y}{numerical array of y-coordinates of points}
\item{pol.x}{numerical array of x-coordinates of polygon}
\item{pol.y}{numerical array of y-coordinates of polygon}
}
\value{ logical array; FALSE if a point is strictly exterior to the
polygon, TRUE if not (point is strictly interior to polygon, point is
a vertex of polygon, or point lies on the relative interior of an edge
of polygon) }
\references{ 
Uses the C function InPoly(), in gstat file polygon.c;
InPoly is Copyright 1998 by Joseph O'Rourke.  It may be freely
redistributed in its entirety provided that this copyright notice is
not removed.
}
\examples{
# open polygon:
point.in.polygon(1:10,1:10,c(3,4,4,3),c(3,3,4,4))
# closed polygon:
point.in.polygon(1:10,1:10,c(3,4,4,3,3),c(3,3,4,4,3))
}
\keyword{models}

\eof
\name{predict.gstat}
\alias{predict.gstat}
\title{ Multivariable Geostatistical Prediction and Simulation }
\description{ The function provides the following prediction methods:
simple, ordinary, and universal kriging, simple, ordinary, and universal
cokriging, point- or block-kriging, and conditional simulation equivalents
for each of the kriging methods.  }
\usage{
predict.gstat(object, newdata, block = numeric(0), nsim = 0, 
	indicators = FALSE, BLUE = FALSE, debug.level = 1, mask, 
	na.action = na.pass, ...)
}
\arguments{
\item{object}{ object of class \code{gstat}, see \link{gstat} and
  \link{krige}}
\item{newdata}{ data frame with prediction/simulation locations; should 
  contain columns with the independent variables (if present) and the
  coordinates with names as defined in \code{locations} }
\item{block}{ block size; a vector with 1, 2 or 3 values containing
  the size of a rectangular in x-, y- and z-dimension respectively
  (0 if not set), or a data frame with 1, 2 or 3 columns, containing
  the points that discretize the block in the x-, y- and z-dimension
  to define irregular blocks relative to (0,0) or (0,0,0)---see also
  the details section below. By default, predictions
  or simulations refer to the support of the data values. }
\item{nsim}{ integer; if set to a non-zero value, conditional simulation
  is used instead of kriging interpolation. For this, sequential Gaussian
  or indicator simulation is used (depending on the value of 
  \code{indicators}), following a single random path through the data.  }
\item{indicators}{ logical; only relevant if \code{nsim} is non-zero; if
 TRUE, use indicator simulation, else use Gaussian simulation }
\item{BLUE}{ logical; if TRUE return the BLUE trend estimates only, 
if FALSE return the BLUP predictions (kriging) }
\item{debug.level}{ integer; set gstat internal debug level, see below
for useful values. If set to -1 (or any negative value), a progress counter
is printed }
\item{mask}{ not supported anymore -- use na.action; 
logical or numerical vector; pattern with valid values
in newdata (marked as TRUE, non-zero, or non-NA); if mask is specified,
the returned data frame will have the same number and order of rows 
in newdata, and masked rows will be filled with NA's. }
\item{na.action}{ function determining what should be done with missing
values in 'newdata'.  The default is to predict 'NA'.  Missing values 
in coordinates and predictors are both dealt with. }
\item{...}{ ignored (but necessary for the S3 generic/method consistency) }
}

\details{ When a non-stationary (i.e., non-constant) mean is used, both
for simulation and prediction purposes the variogram model defined should
be that of the residual process, and not that of the raw observations.

For irregular block kriging, coordinates should discretize the area
relative to 0, (0,0) or (0,0,0); the coordinates in newdata should give
the centroids around which the block should be located. So, suppose the
block is discretized by points (3,3) (3,5) (5,5) and (5,3), we should
pass point (4,4) in newdata and pass points (-1,-1) (-1,1) (1,1) (1,-1)
to the block argument. Although passing the uncentered block and (0,0)
as newdata may work for global neighbourhoods, neighbourhood selection
is always done relative to the centroid values in newdata.

The algorithm used by gstat for simulation random fields is the
sequential simulation algorithm. This algorithm scales well to large or
very large fields (e.g., more than $10^6$ nodes). Its power lies in using
only data and simulated values in a local neighbourhood to approximate the
conditional distribution at that location, see \code{nmax} in \link{krige}
and \link{gstat}. The larger \code{nmax}, the better the approximation,
the smaller \code{nmax}, the faster the simulation process. For selecting
the nearest \code{nmax} data or previously simulated points, gstat uses
a bucket PR quadtree neighbourhood search algorithm; see the reference
below.

For sequential Gaussian or indicator simulations, a random path through
the simulation locations is taken, which is usually done for sequential
simulations. The reason for this is that the local approximation of the
conditional distribution, using only the \code{nmax} neareast observed
(or simulated) values may cause spurious correlations when a regular
path would be followed. Following a single path through the locations,
gstat reuses the expensive results (neighbourhood selection and solution
to the kriging equations) for each of the subsequent simulations when
multiple realisations are requested.  You may expect a considerable speed
gain in simulating 1000 fields in a single call to \link{predict.gstat},
compared to 1000 calls, each for simulating a single field.

The random number generator used for generating simulations is the native
random number generator of the environment (R, S); fixing randomness by
setting the random number seed with \code{set.seed()} works.

When mean coefficient are not supplied, they are generated as well
from their conditional distribution (assuming multivariate normal,
using the generalized least squares BLUE estimate and its estimation
covariance); for a reference to the algorithm used see Abrahamsen and
Benth, Math. Geol. 33(6), page 742 and leave out all constraints.

Memory requirements for sequential simulation: let n be the product of
the number of variables, the number of simulation locations, and the
number of simulations required in a single call.  the gstat C function
\code{gstat_predict} requires a table of size n * 12 bytes to pass the
simulations back to R, before it can free n * 4 bytes. Hopefully, R does
not have to duplicate the remaining n * 8 bytes when the coordinates
are added as columns, and when the resulting matrix is coerced to
a \code{data.frame}.

Useful values for \code{debug.level}: 0: suppres any output except
warning and error messages; 1: normal output (default): short data report,
program action and mode, program progress in \%, total execution time;
2: print the value of all global variables, all files read and written,
and include source file name and line number in error messages; 4: print
OLS and WLS fit diagnostics; 8: print all data after reading them; 16:
print the neighbourhood selection for each prediction location; 32:
print (generalised) covariance matrices, design matrices, solutions,
kriging weights, etc.; 64: print variogram fit diagnostics (number
of iterations and variogram model in each iteration step) and order
relation violations (indicator kriging values before and after order
relation correction); 512: print block (or area) discretization data
for each prediction location. To combine settings, sum their respective
values. Negative values for \code{debug.level} are equal to positive,
but cause the progress counter to work.

}

\value{
a data frame containing the coordinates of \code{newdata}, and columns
of prediction and prediction variance (in case of kriging) or the
columns of the conditional Gaussian or indicator simulations 
}

\references{ 
N.A.C. Cressie, 1993, Statistics for Spatial Data, Wiley. 

\url{http://www.gstat.org/}

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.

For bucket PR quadtrees, excellent demos are found at
\url{http://www.cs.umd.edu/~brabec/quadtree/index.html}
}
\author{ Edzer J. Pebesma }
\note{  }

\seealso{\link{gstat}, \link{krige}}

\examples{
# generate 5 conditional simulations
data(meuse)
data(meuse.grid)
v <- variogram(log(zinc)~1,~x+y, meuse)
m <- fit.variogram(v, vgm(1, "Sph", 300, 1))
plot(v, model = m)
set.seed(131)
sim <- krige(formula = log(zinc)~1, locations = ~x+y, model = m, data 
	= meuse, newdata = meuse.grid, nmax = 15, beta = 5.9, nsim = 5)
# show all 5 simulation, using map.to.lev to rearrange sim:
if (require(sp) == FALSE) {
levelplot(z~x+y|name, map.to.lev(sim, z=c(3:7)), aspect = mapasp(sim))

# calculate generalised least squares residuals w.r.t. constant trend:
g <- gstat(id = "log.zinc", formula = log(zinc)~1, locations = ~x+y, 
     model = m, data = meuse)
blue0 <- predict(g, newdata = meuse, BLUE = TRUE)
blue0$blue.res <- log(meuse$zinc) - blue0$log.zinc.pred
bubble(blue0, zcol = "blue.res", main = "GLS residuals w.r.t. constant")

# calculate generalised least squares residuals w.r.t. linear trend:
m <- fit.variogram(variogram(log(zinc)~sqrt(dist.m),~x+y, meuse), 
	vgm(1, "Sph", 300, 1))
g <- gstat(id = "log.zinc", formula = log(zinc)~sqrt(dist.m), 
	locations = ~x+y, model = m, data = meuse)
blue1 <- predict(g, newdata = meuse, BLUE = TRUE)
blue1$blue.res <- log(meuse$zinc) - blue1$log.zinc.pred
bubble(blue1, zcol = "blue.res", 
	main = "GLS residuals w.r.t. linear trend")

# unconditional simulation on a 100 x 100 grid
xy <- expand.grid(1:100, 1:100)
names(xy) <- c("x","y")
g.dummy <- gstat(formula = z~1, locations = ~x+y, dummy = TRUE, beta = 0,
	model = vgm(1,"Exp",15), nmax = 20)
yy <- predict(g.dummy, newdata = xy, nsim = 4)
# show one realisation:
levelplot(sim1~x+y, yy, aspect = mapasp(yy))
# show all four:
levelplot(z~x+y|name, map.to.lev(yy, z=c(3:6)), aspect = mapasp(yy))
}

}

\keyword{ models }

\eof
\name{select.spatial}
\alias{select.spatial}
\title{ select points spatially }
\description{
select a number of points by digitizing the area they fall in
}
\usage{
select.spatial(x = data$x, y = data$y, data, pch = "+", n = 512)
}
\arguments{
\item{x}{numerical array of x-coordinates of points}
\item{y}{numerical array of y-coordinates of points}
\item{data}{optional; data frame containing variables x and y}
\item{pch}{plotting character to be used for points}
\item{n}{default number of points to locate, can be overridden manually}
}
\value{ array with indexes (row numbers) of points inside
the polygon digitized }
\seealso{\link{point.in.polygon}, \link{locator}}
\examples{
data(meuse)
## the following command requires user interaction: left mouse
## selects points, right mouse ends digitizing
# select.spatial(data=meuse)
}
\keyword{models}

\eof
\name{show.vgms}
\alias{show.vgms}
\title{
Plot Variogram Model Functions
}
\description{
Creates a trellis plot for a range of variogram models, possibly with nugget;
and optionally a set of Matern models with varying smoothness.
}
\usage{
show.vgms(min = 1e-12 * max, max = 3, n = 50, sill = 1, range = 1,
    models = as.character(vgm()[c(1:16)]), nugget = 0, kappa.range = 0.5,
	plot = TRUE)
}
\arguments{
\item{min}{ numeric; start distance value for semivariance calculation 
beyond the first point at exactly zero  }
\item{max}{ numeric; maximum distance for semivariance calculation 
and plotting }
\item{n}{ integer; number of points to calculate distance values }
\item{sill}{ numeric; (partial) sill of the variogram model }
\item{range}{ numeric; range of the variogram model }
\item{models}{ character; variogram models to be plotted }
\item{nugget}{ numeric; nugget component for variogram models }
\item{kappa.range}{ numeric; if this is a vector with more than one
element, only a range of Matern models is plotted with these kappa 
values }
\item{plot}{ logical; if TRUE, a plot is returned with the models
specified; if FALSE, the data prepared for this plot is returned }
}
\value{
returns a (Trellis) plot of the variogram models requested; see
examples. I do currently have strong doubts about the ``correctness''
of the ``Hol'' model. The ``Spl'' model does seem to need a very
large range value (larger than the study area?) to be of some value.

If plot is FALSE, a data frame with the data prepared to plot
is being returned.
}
\references{ \url{http://www.gstat.org}}
\author{ Edzer J. Pebesma }
\note{ the \code{min} argument is supplied because the variogram
function may be discontinuous at distance zero, surely when a positive
nugget is present. }
\seealso{
\link{vgm}, \link{variogram.line},
}
\examples{
show.vgms()
show.vgms(models = c("Exp", "Mat", "Gau"), nugget = 0.1)
# show a set of Matern models with different smoothness:
show.vgms(kappa.range = c(.1, .2, .5, 1, 2, 5, 10), max = 10)
}

\keyword{dplot}

\eof
\name{sic2004}
\alias{sic2004}
\alias{sic.train}
\alias{sic.pred}
\alias{sic.grid}
\alias{sic.test}
\alias{sic.valid}
\title{ Spatial Interpolation Comparison 2004 data set: Natural Ambient Radioactivity }
\description{
The text below is copied from
\url{http://www.ai-geostats.org/events/sic2004/index.htm}, subsection
Data.

The variable used in the SIC 2004 exercise is natural ambient
radioactivity measured in Germany. The data, provided kindly by the
German Federal Office for Radiation Protection (BfS), are gamma dose rates
reported by means of the national automatic monitoring network (IMIS).

In the frame of SIC2004,  a rectangular area was used to select 1008
monitoring stations (from a total of around 2000 stations). For these
1008 stations, 11 days of measurements have been randomly selected during
the last 12 months and the average daily dose rates calculated for each
day. Hence, we ended up having 11 data sets.

Prior information (sic.train): 10 data sets of 200 points that are
identical for what concerns the locations of the monitoring stations
have been prepared. These locations have been randomly selected (see
Figure 1). These data sets differ only by their Z values since each set
corresponds to 1 day of measurement made during the last 14 months. No
information will be provided on the date of measurement.  These 10 data
sets (10 days of measurements) can be used as prior information to tune
the parameters of the mapping algorithms. No other information will be
provided about these sets. Participants are free of course to gather
more information about the variable in the literature and so on.

The 200 monitoring stations above were randomly taken from a larger set
of 1008 stations. The remaining 808 monitoring stations have a topology
given in sic.pred.  Participants to SIC2004 will have to estimate the
values of the variable taken at these 808 locations.  

The SIC2004 data (sic.test, variable first):
The exercise consists in using 200 measurements made on a 11th day (THE
data of the exercise) to estimate the values observed at the remaining
808 locations (hence the question marks as symbols in the maps shown
in Figure 3). These measurements will be provided only during two weeks
(15th of September until 1st of October 2004) on a web page restricted
to the participants. The true values observed at these 808 locations
will be released only at the end of the exercise to allow participants
to write their manuscripts (sic.valid).

In addition, a joker data set was released (sic.test, variable second),
which contains an anomaly. The anomaly was generated by a simulation
model, and does not represent measured levels.

}

\format{
  The data frames contain the following columns:
  \describe{
   \item{record}{this integer value is the number (unique value) of
   the monitoring station chosen by us.}
   \item{x}{X-coordinate of the monitoring station indicated in meters}
   \item{y}{Y-coordinate of the monitoring station indicated in meters}
   \item{day01}{mean gamma dose rate measured during 24 hours, at day01. Units
are nanoSieverts/hour}
   \item{day02}{same, for day 02}
   \item{day03}{...}
   \item{day04}{...}
   \item{day05}{...}
   \item{day06}{...}
   \item{day07}{...}
   \item{day08}{...}
   \item{day09}{...}
   \item{day10}{...}
   \item{first}{ the data observed at the 11-th day}
   \item{second}{ the joker data set, containing an anomaly not present
   in the training data}
  }
}
\note{
the data set sic.grid provides a set of points on a regular grid (almost
10000 points) covering the area; this is convenient for interpolation.

The coordinates have been projected around a point located in the South
West of Germany. 
}

\usage{
data(sic2004) # 
}
\authors{ 
Data: the German Federal Office for Radiation Protection (BfS),
\url{http://www.bfs.de/}, data provided by Gregoire Dubois, R compilation
by Edzer J. Pebesma.  }

\references{ 

\url{http:/www.ai-geostats.org/}

\url{http://www.ai-geostats.org/events/sic2004/index.htm}
}

\keyword{datasets}
\examples{
data(sic2004) 
# FIGURE 1. Locations of the 200 monitoring stations for the 11 data sets. 
# The values taken by the variable are known.
plot(y~x,sic.train,pch=1,col="red", asp=1)

# FIGURE 2. Locations of the 808 remaining monitoring stations at which 
# the values of the variable must be estimated.
plot(y~x,sic.pred,pch="?", asp=1, cex=.8) # Figure 2

# FIGURE 3. Locations of the 1008 monitoring stations (exhaustive data sets). 
# Red circles are used to estimate values located at the questions marks
plot(y~x,sic.train,pch=1,col="red", asp=1)
points(y~x, sic.pred, pch="?", cex=.8)

}

\eof
\name{variogram}
\alias{variogram}
\alias{variogram.gstat}
\alias{variogram.formula}
\alias{variogram.default}
\alias{print.gstatVariogram}
\alias{print.variogramCloud}
\title{ Calculate Sample or Residual Variogram or Variogram Cloud }
\description{
Calculates the sample variogram from data, or in case of a linear model
is given, for the residuals, with options for directional, robust,
and pooled variogram, and for irregular distance intervals.  
}
\synopsis{ variogram(object, ...) }
\usage{
variogram.formula(object, ...)
variogram.gstat(formula, locations, data, ...)
variogram.default(y, locations, X, cutoff, width = cutoff/15, alpha =
	0, beta = 0, tol.hor = 90/length(alpha), tol.ver =
	90/length(beta), cressie = FALSE, dX = numeric(0), boundaries =
	numeric(0), cloud = FALSE, trend.beta = NULL, debug.level = 1,
	cross = TRUE, grid, map = FALSE, ...)

print.gstatVariogram(v, ...)
print.variogramCloud(v, ...)
}
\arguments{
\item{object}{object of class \code{gstat}; in this form, direct
and cross (residual) variograms are calculated for all variables and
variable pairs defined in \code{object}}
\item{formula}{formula defining the response vector and (possible) 
regressors, in case of absence of regressors, use e.g. \code{z~1}}
\item{data}{data frame where the names in formula are to be found}
\item{locations}{ spatial data locations.  For variogram.formula: a
formula with only the coordinate variables in the right hand (explanatory
variable) side e.g. \code{~x+y}; see examples.

For variogram.default: list with coordinate matrices, each with the number
of rows matching that of corresponding vectors in y; the number of columns
should match the number of spatial dimensions spanned by the data (1
(x), 2 (x,y) or 3 (x,y,z)).  }
\item{...}{any other arguments that will be passed to 
\link{variogram.default}}
\item{y}{ list with for each variable the vector with responses }
\item{X}{ (optional) list with for each variable the matrix with
regressors/covariates; the number of rows should match that of the
correspoding element in y, the number of columns equals the number of
regressors (including intercept) }
\item{cutoff}{ spatial separation distance up to which point pairs
are included in semivariance estimates }
\item{width}{ the width of subsequent distance intervals into which
data point pairs are grouped for semivariance estimates }
\item{alpha}{ direction in  plane (x,y), in positive degrees clockwise
from positive y (North): alpha=0 for direction North (increasing y),
alpha=90 for direction East (increasing x); optional a vector of
directions in (x,y) }
\item{beta}{ direction in z, in positive degrees up from the (x,y) plane; }
optional a vector of directions
\item{tol.hor}{ horizontal tolerance angle in degrees }
\item{tol.ver}{ vertical tolerance angle in degrees }
\item{cressie}{ logical; if TRUE, use Cressie's robust variogram estimate;
if FALSE use the classical method of moments variogram estimate }
\item{dX}{ include a pair of data points $y(s_1),y(s_2)$ taken at
locations $s_1$ and $s_2$ for sample variogram calculation only when
$||x(s_1)-x(s_2)|| < dX$ with and $x(s_i)$ the vector with regressors at
location $s_i$, and $||.||$ the 2-norm.  This allows pooled estimation of
within-strata variograms (use a factor variable as regressor, and dX=0.5),
or variograms of (near-)replicates in a linear model (addressing point
pairs having similar values for regressors variables) }
\item{boundaries}{ numerical vector with distance interval boundaries; 
values should be strictly increasing }
\item{cloud}{ logical; if TRUE, calculate the semivariogram cloud }
\item{trend.beta}{vector with trend coefficients, in case they are
known. By default, trend coefficients are estimated from the data.}
\item{debug.level}{ integer; set gstat internal debug level }
\item{cross}{ logical; if FALSE, no cross variograms are calculated
when object is of class \code{gstat} and has more than one variable }
\item{v}{ object of class \code{variogram} or \code{variogramCloud}
to be printed}
\item{grid}{ grid parameters, if data are gridded }
\item{map}{ logical; if TRUE, and \code{cutoff} and \code{width}
are given, a variogram map is returned. This requires package
sp. Alternatively, a map can be passed, of class SpatialDataFrameGrid
(see sp docs) }
% \item{...}{arguments that get passed to \code{print}, e.g. digits}
}
\value{ 
If map is TRUE (or a map is passed), a grid map is returned containing
the (cross) variogram map(s). See package sp.

In other cases, an object of class "gstatVariogram" with the 
following fields:
\item{np}{the number of point pairs for this estimate; 
in case of a \code{variogramCloud} see below}
\item{dist}{the average distance of all point pairs considered
for this estimate}
\item{gamma}{the actual sample variogram estimate}
\item{dir.hor}{the horizontal direction}
\item{dir.ver}{the vertical direction}
\item{id}{the combined id pair}
If cloud is TRUE: an object of class \code{variogramCloud}, with the field
\code{np} encoding the numbers of the point pair that contributed to a
variogram cloud estimate, as follows. The first point is found by the
integer division of np by $2^{16}$, the second point by the remainder
of that division. \link{print.variogramCloud} shows no \code{np} field,
but does show in addition:
\item{left}{for variogramCloud: data id (row number) of one of 
the data pair}
\item{right}{for variogramCloud: data id (row number) of the other 
data in the pair}

In the past, gstat returned an object of class "variogram"; however,
this resulted in confusions for users of the package geoR: the geoR
variog function also returns objects of class "variogram", incompatible
to those returned by this function. That's why I changed the class name.

}

\references{ 
Cressie, N.A.C., 1993, Statistics for Spatial Data, Wiley.

\url{http://www.gstat.org/}

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.
}
\author{ Edzer J. Pebesma }
\note{
}
\seealso{
\link{print.gstatVariogram},
\link{plot.gstatVariogram},
\link{plot.variogramCloud};
for variogram models: \link{vgm},
to fit a variogram model to a sample variogram: 
\link{fit.variogram}
}
\examples{
data(meuse)
# no trend:
variogram(log(zinc)~1, loc=~x+y, meuse)
# residual variogram w.r.t. a linear trend:
variogram(log(zinc)~x+y, loc=~x+y, meuse)
# directional variogram:
variogram(log(zinc)~x+y, loc=~x+y, meuse, alpha=c(0,45,90,135))
}

\keyword{models}

\eof
\name{variogram.line}
\alias{variogram.line}
\title{ Semivariance Values For a Given Variogram Model }
\description{ Generates a semivariance values given a variogram model }
\usage{
variogram.line(object, maxdist, n = 200, min = 1.0e-6 * maxdist, 
	dir = c(1,0,0), ...)
}
\arguments{
\item{object}{ variogram model for which we want semivariance function values }
\item{maxdist}{ maximum distance for which we want semivariance values }
\item{n}{ number of points }
\item{min}{ minimum distance; a value slightly larger than zero is usually
used to avoid the discontinuity at distance zero if a nugget component is
present }
\item{dir}{ direction vector: unit length vector pointing the direction in 
x (East-West), y (North-South) and z (Up-Down) }
\item{...}{ignored}
}
\value{
a data frame of dimension (\code{n} x 2), with columns distance and gamma
}
\note{this function is used to plot a variogram model}
\author{ Edzer J. Pebesma }
\seealso{ \link{plot.gstatVariogram}}

\examples{
variogram.line(vgm(5, "Exp", 10, 5), 10, 10)
# anisotropic variogram, plotted in E-W direction:
variogram.line(vgm(1, "Sph", 10, anis=c(0,0.5)), 10, 10)
# anisotropic variogram, plotted in N-S direction:
variogram.line(vgm(1, "Sph", 10, anis=c(0,0.5)), 10, 10, dir=c(0,1,0))
}

\keyword{models}

\eof
\name{vgm}
\alias{vgm}
\alias{print.variogramModel}
\title{
Generate, or Add to Variogram Model
}
\description{
Generates a variogram model, or adds to an existing model. 
\code{print.variogramModel} prints the essence of a variogram  model.
}
\usage{
vgm(psill, model, range, nugget, add.to, anis, kappa = 0.5)
print.variogramModel(x, ...)
}
\arguments{
\item{psill}{ (partial) sill of the variogram model component }
\item{model}{ model type, e.g. "Exp", "Sph", "Gau", "Mat". Calling vgm() without
a model argument returns the list with available models. }
\item{range}{ range of the variogram model component }
\item{kappa}{ smoothness parameter for the Matern class of variogram 
models }
\item{nugget}{ nugget component of the variogram (this basically adds a nugget
compontent to the model) }
\item{add.to}{ a variogram model to which we want to add a component }
\item{anis}{ anisotropy parameters: see notes below }
\item{x}{ a variogram model to print }
\item{...}{ arguments that will be passed to \code{print}, e.g. \code{digits}
(see examples) }
}
\value{
an object of class \code{variogramModel}, which extends data.frame Called
without a model argument returns a character list with available models.
}
\references{ 
Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers \& Geosciences, 30: 683-691.
}
\author{ Edzer J. Pebesma }
\note{ 
Geometric anisotropy can be modelled for each individual simple model
by giving two or five anisotropy parameters, two for two-dimensional
and five for three-dimensional data. In any case, the range defined
is the range in the direction of the strongest correlation, or the
major range. Anisotropy parameters define which direction this is (the
main axis), and how much shorter the range is in (the) direction(s)
perpendicular to this main axis.

In two dimensions, two parameters define an anisotropy ellipse, say
\code{anis = c(45, 0.5)}. The first parameter, \code{30}, refers to
the main axis direction: it is the angle for the principal direction
of continuity (measured in degrees, clockwise from positive Y, North).
The second parameter, \code{0.5}, is the anisotropy ratio, the ratio
of the minor range to the major range (a value between 0 and 1).  So,
in our example, if the range in the major direction (North-East) is 100,
the range in the minor direction (South-East) is 50.

In three dimensions, five values should be given in the form \code{anis
= c(p,q,r,s,t)}.  Now, $p$ is the angle for the principal direction of
continuity (measured in degrees, clockwise from Y, in direction of X),
$q$ is the dip angle for the principal direction of continuity (measured
in positive degrees up from horizontal), $r$ is the third rotation angle
to rotate the two minor directions around the principal direction defined
by $p$ and $q$. A positive angle acts counter-clockwise while looking
in the principal direction. Anisotropy ratios $s$ and $t$ are the ratios
between the major range and each of the two minor ranges. 

(Note that \code{anis = c(p,s)} is equivalent to \code{anis = c(p,0,0,s,1)}.)

The implementation in gstat for 2D and 3D anisotropy was taken from
the gslib (probably 1992) code.  I have seen a paper where it is argued
that the 3D anisotropy code implemented in gslib (en then also in gstat)
is in error, but I have not corrected anything afterwards.
}
\seealso{
\link{show.vgms} to view the available models,
\link{fit.variogram},
\link{variogram.line},
\link{variogram} for the sample variogram.
}
\examples{
vgm(10, "Exp", 300)
x <- vgm(10, "Exp", 300)
vgm(10, "Nug", 0)
vgm(10, "Exp", 300, 4.5)
vgm(10, "Mat", 300, 4.5, kappa = 0.7)
vgm( 5, "Exp", 300, add.to = vgm(5, "Exp", 60, nugget = 2.5))
vgm(10, "Exp", 300, anis = c(30, 0.5))
vgm(10, "Exp", 300, anis = c(30, 10, 0, 0.5, 0.3))
# Matern variogram model:
vgm(1, "Mat", 1, kappa=.3)
x <- vgm(0.39527463, "Sph", 953.8942, nugget = 0.06105141)
x
print(x, digits = 3);
# to see all components, do
print.data.frame(x)
}

\keyword{models}

\eof
\name{zerodist}
\alias{zerodist}
\title{ find point pairs with equal spatial coordinates }
\description{ find point pairs with equal spatial coordinates }
\usage{ zerodist(x, y, z, zero = 0.0) }
\arguments{
\item{x}{ vector with x-coordinate }
\item{y}{ vector with y-coordinate (may be missing) }
\item{z}{ vector with z-coordinate (may be missing) }
\item{zero}{ value to be compared to for establishing when a distance
is considered zero (default 0.0) }
}
\value{pairs of row numbers with identical coordinates, numeric(0) if no
such pairs are found; if zero is set to a positive value, the distance
between point pairs is returned in the third column. }
\note{ Duplicate observations sharing identical spatial locations
result in singular covariance matrices in kriging situations.
This function may help identifying spatial duplications, so they
can be removed.  A matrix with all pair-wise distances is calculated,
so if x, y and z are large this function is slow 
}
\examples{
if (require(sp) == FALSE) {
data(meuse)
# pick 10 rows
n <- 10
ran10 <- sample(nrow(meuse), size = n, replace = TRUE)
meusedup <- rbind(meuse, meuse[ran10, ])
zerodist(meusedup$x, meusedup$y)
zd <- zerodist(meusedup$x, meusedup$y)
sum(abs(zd[1:n,1] - sort(ran10))) # 0!
# remove the duplicate rows:
meusedup2 <- meusedup[-zd[,2], ]
# find point pairs within 500 m distance of each other:
zerodist(meuse$x, meuse$y, 500)
}
}
\keyword{dplot}

\eof
